# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['synology_dsm',
 'synology_dsm.api',
 'synology_dsm.api.core',
 'synology_dsm.api.download_station',
 'synology_dsm.api.dsm',
 'synology_dsm.api.storage',
 'synology_dsm.api.surveillance_station']

package_data = \
{'': ['*']}

install_requires = \
['requests>=2.24.0,<3.0.0', 'urllib3>=1.25.10,<2.0.0']

entry_points = \
{'console_scripts': ['synologydsm-api = synology_dsm.__main__:main']}

setup_kwargs = {
    'name': 'synologydsm-api',
    'version': '1.0.2',
    'description': 'Python API for communication with Synology DSM',
    'long_description': '===========================\nPython API for Synology DSM\n===========================\n\n.. image:: https://github.com/hacf-fr/synologydsm-api/workflows/Tests/badge.svg\n    :target: https://github.com/hacf-fr/synologydsm-api/actions?query=workflow%3ATests+branch%3Amaster\n\n.. image:: https://img.shields.io/pypi/v/synologydsm-api.svg\n    :alt: Library version\n    :target: https://pypi.org/project/synologydsm-api\n\n.. image:: https://img.shields.io/pypi/pyversions/synologydsm-api.svg\n    :alt: Supported versions\n    :target: https://pypi.org/project/synologydsm-api\n\n.. image:: https://pepy.tech/badge/synologydsm-api\n    :alt: Downloads\n    :target: https://pypi.org/project/synologydsm-api\n\n.. image:: https://img.shields.io/badge/code%20style-black-000000.svg\n    :alt: Formated with Black\n    :target: https://github.com/psf/black\n\n\nInstallation\n============\n\n.. code-block:: bash\n\n    [sudo] pip install synologydsm-api\n\n\nUsage\n=====\n\nYou can import the module as `synology_dsm`.\n\n\nConstructor\n-----------\n\n.. code-block:: python\n\n    SynologyDSM(\n        dsm_ip,\n        dsm_port,\n        username,\n        password,\n        use_https=False,\n        verify_ssl=False,\n        timeout=None,\n        device_token=None,\n        debugmode=False,\n    )\n\n``device_token`` should be added when using a two-step authentication account, otherwise DSM will ask to login with a One Time Password (OTP) and requests will fail (see the login section for more details).\n\nDefault ``timeout`` is 10 seconds.\n\n\nLogin\n------\n\nThe library automatically login at first request, but you better use the ``login()`` function separately to authenticate.\n\nIt will return a boolean if it successed or faild to authenticate to DSM.\n\nIf your account need a two-step authentication (2SA), ``login()`` will raise ``SynologyDSMLogin2SARequiredException``.\nCall the function again with a One Time Password (OTP) as parameter, like ``login("123456")`` (better to be a string to handle first zero).\nStore the ``device_token`` property so that you do not need to reconnect with password the next time you open a new ``SynologyDSM`` session.\n\n\nCode exemple\n------------\n\nEvery API has an ``update()`` function that is needed to get the first data, then the data is cached and updated at the next ``update()`` call.\n\nThe ``SynologyDSM`` class can also ``update()`` all APIs at once.\n\n.. code-block:: python\n\n    from synology_dsm import SynologyDSM\n\n    print("Creating Valid API")\n    api = SynologyDSM("<IP/DNS>", "<port>", "<username>", "<password>")\n\n    print("=== Information ===")\n    api.information.update()\n    print("Model:           " + str(api.information.model))\n    print("RAM:             " + str(api.information.ram) + " MB")\n    print("Serial number:   " + str(api.information.serial))\n    print("Temperature:     " + str(api.information.temperature) + " °C")\n    print("Temp. warning:   " + str(api.information.temperature_warn))\n    print("Uptime:          " + str(api.information.uptime))\n    print("Full DSM version:" + str(api.information.version_string))\n    print("--")\n\n    print("=== Utilisation ===")\n    api.utilisation.update()\n    print("CPU Load:        " + str(api.utilisation.cpu_total_load) + " %")\n    print("Memory Use:      " + str(api.utilisation.memory_real_usage) + " %")\n    print("Net Up:          " + str(api.utilisation.network_up()))\n    print("Net Down:        " + str(api.utilisation.network_down()))\n    print("--")\n\n    print("=== Storage ===")\n    api.storage.update()\n    for volume_id in api.storage.volumes_ids:\n        print("ID:          " + str(volume_id))\n        print("Status:      " + str(api.storage.volume_status(volume_id)))\n        print("% Used:      " + str(api.storage.volume_percentage_used(volume_id)) + " %")\n        print("--")\n\n    for disk_id in api.storage.disks_ids:\n        print("ID:          " + str(disk_id))\n        print("Name:        " + str(api.storage.disk_name(disk_id)))\n        print("S-Status:    " + str(api.storage.disk_smart_status(disk_id)))\n        print("Status:      " + str(api.storage.disk_status(disk_id)))\n        print("Temp:        " + str(api.storage.disk_temp(disk_id)))\n        print("--")\n\n    print("=== Shared Folders ===")\n    api.share.update()\n    for share_uuid in api.share.shares_uuids:\n        print("Share name:        " + str(api.share.share_name(share_uuid)))\n        print("Share path:        " + str(api.share.share_path(share_uuid)))\n        print("Space used:        " + str(api.share.share_size(share_uuid, human_readable=True)))\n        print("Recycle Bin Enabled: " + str(api.share.share_recycle_bin(share_uuid)))\n        print("--")\n\n\nDownload Station usage\n--------------------------\n\n.. code-block:: python\n\n    from synology_dsm import SynologyDSM\n\n    api = SynologyDSM("<IP/DNS>", "<port>", "<username>", "<password>")\n\n    if "SYNO.DownloadStation.Info" in api.apis:\n\n        api.download_station.get_info()\n        api.download_station.get_config()\n\n        # The download list will be updated after each of the following functions:\n        # You should have the right on the (default) directory that the download will be saved, or you will get a 403 or 406 error\n        api.download_station.create("http://commondatastorage.googleapis.com/gtv-videos-bucket/sample/BigBuckBunny.mp4")\n        api.download_station.pause("dbid_1")\n        # Like the other function, you can eather pass a str or a list\n        api.download_station.resume(["dbid_1", "dbid_2"])\n        api.download_station.delete("dbid_3")\n\n        # Manual update\n        api.download_station.update()\n\n\nSurveillance Station usage\n--------------------------\n\n.. code-block:: python\n\n    from synology_dsm import SynologyDSM\n\n    api = SynologyDSM("<IP/DNS>", "<port>", "<username>", "<password>")\n    surveillance = api.surveillance_station\n    surveillance.update() # First update is required\n\n    # Returns a list of cached cameras available\n    cameras = surveillance.get_all_cameras()\n\n    # Assuming there\'s at least one camera, get the first camera_id\n    camera_id = cameras[0].camera_id\n\n    # Returns cached camera object by camera_id\n    camera = surveillance.get_camera(camera_id)\n\n    # Returns cached motion detection enabled\n    motion_setting = camera.is_motion_detection_enabled\n\n    # Return bytes of camera image\n    surveillance.get_camera_image(camera_id)\n\n    # Updates all cameras/motion settings and cahce them\n    surveillance.update()\n\n    # Gets Home Mode status\n    home_mode_status =  surveillance.get_home_mode_status()\n\n    # Sets home mode - true is on, false is off\n    surveillance.set_home_mode(True)\n\n\nSystem usage\n--------------------------\n\n.. code-block:: python\n\n    from synology_dsm import SynologyDSM\n\n    api = SynologyDSM("<IP/DNS>", "<port>", "<username>", "<password>")\n    system = api.system\n\n    # Reboot NAS\n    system.reboot()\n\n    # Shutdown NAS\n    system.shutdown()\n\n    # Manual update system information\n    system.update()\n\n    # Get CPU information\n    system.cpu_clock_speed\n    system.cpu_cores\n    system.cpu_family\n    system.cpu_series\n\n    # Get NTP settings\n    system.enabled_ntp\n    system.ntp_server\n\n    # Get system information\n    system.firmware_ver\n    system.model\n    system.ram_size\n    system.serial\n    system.sys_temp\n    system.time\n    system.time_zone\n    system.time_zone_desc\n    system.up_time\n\n    # Get list of all connected USB devices\n    system.usb_dev\n\n\nUpgrade usage\n--------------------------\n\n.. code-block:: python\n\n    from synology_dsm import SynologyDSM\n\n    api = SynologyDSM("<IP/DNS>", "<port>", "<username>", "<password>")\n    upgrade = api.upgrade\n\n    # Manual update upgrade information\n    upgrade.update()\n\n    # check if DSM update is available\n    if upgrade.update_available:\n        do something ...\n\n    # get available version string (return None if no update available)\n    upgrade.available_version\n\n    # get need of reboot (return None if no update available)\n    upgrade.reboot_needed\n\n    # get need of service restarts (return None if no update available)\n    upgrade.service_restarts\n\n\nCredits / Special Thanks\n========================\n- https://github.com/florianeinfalt\n- https://github.com/tchellomello\n- https://github.com/Quentame   (Multiple API addition & tests)\n- https://github.com/aaska      (DSM 5 tests)\n- https://github.com/chemelli74 (2SA tests)\n- https://github.com/snjoetw    (Surveillance Station library)\n- https://github.com/shenxn     (Surveillance Station tests)\n- https://github.com/Gestas     (Shared Folders)\n\nFound Synology API "documentation" on this repo : https://github.com/kwent/syno/tree/master/definitions\n\n\nOfficial references\n===================\n\n- `Calendar API documentation (2015-2019) <https://global.download.synology.com/download/Document/Software/DeveloperGuide/Package/Calendar/2.4/enu/Synology_Calendar_API_Guide_enu.pdf>`_\n\n- `Download Station API documentation (2012-2014) <https://global.download.synology.com/download/Document/Software/DeveloperGuide/Package/DownloadStation/All/enu/Synology_Download_Station_Web_API.pdf>`_\n\n- `File Station API documentation (2013-2019) <https://global.download.synology.com/download/Document/Software/DeveloperGuide/Package/FileStation/All/enu/Synology_File_Station_API_Guide.pdf>`_\n\n- `Surveillance Station API documentation (2012-2020) <https://global.download.synology.com/download/Document/Software/DeveloperGuide/Package/SurveillanceStation/All/enu/Surveillance_Station_Web_API.pdf>`_\n\n- `Virtual Machine Manager API documentation (2015-2019) <https://global.download.synology.com/download/Document/Software/DeveloperGuide/Package/Virtualization/All/enu/Synology_Virtual_Machine_Manager_API_Guide.pdf>`_\n',
    'author': 'Quentin POLLET (Quentame)',
    'author_email': None,
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/hacf-fr/synologydsm-api',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7.0,<4.0.0',
}


setup(**setup_kwargs)
