"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const kinesis = require("aws-cdk-lib/aws-kinesis");
const assertions_1 = require("aws-cdk-lib/assertions");
// --------------------------------------------------------------
// Test construct properties
// --------------------------------------------------------------
test('Test construct properties', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const pattern = new lib_1.ApiGatewayToKinesisStreams(stack, 'api-gateway-kinesis', {});
    expect(pattern.apiGateway !== null);
    expect(pattern.apiGatewayRole !== null);
    expect(pattern.apiGatewayCloudWatchRole !== null);
    expect(pattern.apiGatewayLogGroup !== null);
    expect(pattern.kinesisStream !== null);
    expect(pattern.cloudwatchAlarms !== null);
});
// --------------------------------------------------------------
// Test deployment w/ overwritten properties
// --------------------------------------------------------------
test('Test deployment w/ overwritten properties', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToKinesisStreams(stack, 'api-gateway-kinesis', {
        apiGatewayProps: {
            restApiName: 'my-api',
            deployOptions: {
                methodOptions: {
                    '/*/*': {
                        throttlingRateLimit: 100,
                        throttlingBurstLimit: 25
                    }
                }
            }
        },
        kinesisStreamProps: {
            shardCount: 1,
            streamName: 'my-stream'
        },
        putRecordRequestTemplate: `{ "Data": "$util.base64Encode($input.json('$.foo'))", "PartitionKey": "$input.path('$.bar')" }`,
        putRecordRequestModel: { schema: {} },
        putRecordsRequestTemplate: `{ "Records": [ #foreach($elem in $input.path('$.records')) { "Data": "$util.base64Encode($elem.foo)", "PartitionKey": "$elem.bar"}#if($foreach.hasNext),#end #end ] }`,
        putRecordsRequestModel: { schema: {} }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Stage', {
        MethodSettings: [
            {
                DataTraceEnabled: false,
                HttpMethod: '*',
                LoggingLevel: 'INFO',
                ResourcePath: '/*'
            },
            {
                HttpMethod: '*',
                ResourcePath: '/*',
                ThrottlingRateLimit: 100,
                ThrottlingBurstLimit: 25
            }
        ]
    });
    template.hasResourceProperties('AWS::Kinesis::Stream', {
        ShardCount: 1,
        Name: 'my-stream'
    });
    // Test for Cloudwatch Alarms
    template.resourceCountIs('AWS::CloudWatch::Alarm', 2);
});
// --------------------------------------------------------------
// Test deployment w/ existing stream without default cloudwatch alarms
// --------------------------------------------------------------
test('Test deployment w/ existing stream', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const construct = new lib_1.ApiGatewayToKinesisStreams(stack, 'api-gateway-kinesis', {
        apiGatewayProps: {},
        existingStreamObj: new kinesis.Stream(stack, 'ExistingStream', {
            shardCount: 5,
            retentionPeriod: aws_cdk_lib_1.Duration.days(4)
        }),
        createCloudWatchAlarms: false
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Kinesis::Stream', {
        ShardCount: 5,
        RetentionPeriodHours: 96
    });
    expect(construct.cloudwatchAlarms == null);
    // Since createCloudWatchAlars is set to false, no Alarm should exist
    template.resourceCountIs('AWS::CloudWatch::Alarm', 0);
});
test('Construct accepts additional PutRecord request templates', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToKinesisStreams(stack, 'api-gateway-kinesis-streams ', {
        additionalPutRecordRequestTemplates: {
            'text/plain': 'custom-template'
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: 'POST',
        Integration: {
            RequestTemplates: {
                'text/plain': 'custom-template'
            }
        }
    });
});
test('Construct accepts additional PutRecords request templates', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToKinesisStreams(stack, 'api-gateway-kinesis-streams ', {
        additionalPutRecordsRequestTemplates: {
            'text/plain': 'custom-template'
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: 'POST',
        Integration: {
            RequestTemplates: {
                'text/plain': 'custom-template'
            }
        }
    });
});
test('Construct uses default integration responses', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToKinesisStreams(stack, 'api-gateway-kinesis-streams ', {});
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: 'POST',
        Integration: {
            IntegrationResponses: [
                {
                    StatusCode: '200'
                },
                {
                    ResponseTemplates: {
                        'text/html': 'Error'
                    },
                    SelectionPattern: '500',
                    StatusCode: '500'
                }
            ]
        }
    });
});
test('Construct uses custom putRecordIntegrationResponses property', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToKinesisStreams(stack, 'api-gateway-kinesis-streams ', {
        putRecordIntegrationResponses: [
            {
                statusCode: '200',
                responseTemplates: {
                    'text/html': 'OK'
                }
            }
        ]
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: 'POST',
        Integration: {
            IntegrationResponses: [
                {
                    ResponseTemplates: {
                        'text/html': 'OK'
                    },
                    StatusCode: '200'
                }
            ]
        }
    });
});
test('Construct uses custom putRecordsIntegrationResponses property', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.ApiGatewayToKinesisStreams(stack, 'api-gateway-kinesis-streams ', {
        putRecordsIntegrationResponses: [
            {
                statusCode: '200',
                responseTemplates: {
                    'text/html': 'OK'
                }
            }
        ]
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ApiGateway::Method', {
        HttpMethod: 'POST',
        Integration: {
            IntegrationResponses: [
                {
                    ResponseTemplates: {
                        'text/html': 'OK'
                    },
                    StatusCode: '200'
                }
            ]
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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