"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_sqs_1 = require("@aws-cdk/aws-sqs");
const core_1 = require("@aws-cdk/core");
/**
 * The base class for QueueProcessingEc2Service and QueueProcessingFargateService services.
 */
class QueueProcessingServiceBase extends core_1.Construct {
    /**
     * Constructs a new instance of the QueueProcessingServiceBase class.
     */
    constructor(scope, id, props) {
        super(scope, id);
        if (props.cluster && props.vpc) {
            throw new Error('You can only specify either vpc or cluster. Alternatively, you can leave both blank');
        }
        this.cluster = props.cluster || this.getDefaultCluster(this, props.vpc);
        // Create the SQS queue and it's corresponding DLQ if one is not provided
        if (props.queue) {
            this.sqsQueue = props.queue;
        }
        else {
            this.deadLetterQueue = new aws_sqs_1.Queue(this, 'EcsProcessingDeadLetterQueue', {
                retentionPeriod: props.retentionPeriod || core_1.Duration.days(14)
            });
            this.sqsQueue = new aws_sqs_1.Queue(this, 'EcsProcessingQueue', {
                deadLetterQueue: {
                    queue: this.deadLetterQueue,
                    maxReceiveCount: props.maxReceiveCount || 3
                }
            });
            new core_1.CfnOutput(this, 'SQSDeadLetterQueue', { value: this.deadLetterQueue.queueName });
            new core_1.CfnOutput(this, 'SQSDeadLetterQueueArn', { value: this.deadLetterQueue.queueArn });
        }
        // Setup autoscaling scaling intervals
        const defaultScalingSteps = [{ upper: 0, change: -1 }, { lower: 100, change: +1 }, { lower: 500, change: +5 }];
        this.scalingSteps = props.scalingSteps !== undefined ? props.scalingSteps : defaultScalingSteps;
        // Create log driver if logging is enabled
        const enableLogging = props.enableLogging !== undefined ? props.enableLogging : true;
        this.logDriver = props.logDriver !== undefined
            ? props.logDriver
            : enableLogging
                ? this.createAWSLogDriver(this.node.id)
                : undefined;
        // Add the queue name to environment variables
        this.environment = { ...(props.environment || {}), QUEUE_NAME: this.sqsQueue.queueName };
        this.secrets = props.secrets;
        // Determine the desired task count (minimum) and maximum scaling capacity
        this.desiredCount = props.desiredTaskCount !== undefined ? props.desiredTaskCount : 1;
        this.maxCapacity = props.maxScalingCapacity || (2 * this.desiredCount);
        if (!this.desiredCount && !this.maxCapacity) {
            throw new Error('maxScalingCapacity must be set and greater than 0 if desiredCount is 0');
        }
        new core_1.CfnOutput(this, 'SQSQueue', { value: this.sqsQueue.queueName });
        new core_1.CfnOutput(this, 'SQSQueueArn', { value: this.sqsQueue.queueArn });
    }
    /**
     * Configure autoscaling based off of CPU utilization as well as the number of messages visible in the SQS queue
     *
     * @param service the ECS/Fargate service for which to apply the autoscaling rules to
     */
    configureAutoscalingForService(service) {
        const scalingTarget = service.autoScaleTaskCount({ maxCapacity: this.maxCapacity, minCapacity: this.desiredCount });
        scalingTarget.scaleOnCpuUtilization('CpuScaling', {
            targetUtilizationPercent: 50,
        });
        scalingTarget.scaleOnMetric('QueueMessagesVisibleScaling', {
            metric: this.sqsQueue.metricApproximateNumberOfMessagesVisible(),
            scalingSteps: this.scalingSteps
        });
    }
    /**
     * Grant SQS permissions to an ECS service.
     * @param service the ECS/Fargate service to which to grant SQS permissions
     */
    grantPermissionsToService(service) {
        this.sqsQueue.grantConsumeMessages(service.taskDefinition.taskRole);
    }
    /**
     * Returns the default cluster.
     */
    getDefaultCluster(scope, vpc) {
        // magic string to avoid collision with user-defined constructs
        const DEFAULT_CLUSTER_ID = `EcsDefaultClusterMnL3mNNYN${vpc ? vpc.node.id : ''}`;
        const stack = core_1.Stack.of(scope);
        return stack.node.tryFindChild(DEFAULT_CLUSTER_ID) || new aws_ecs_1.Cluster(stack, DEFAULT_CLUSTER_ID, { vpc });
    }
    /**
     * Create an AWS Log Driver with the provided streamPrefix
     *
     * @param prefix the Cloudwatch logging prefix
     */
    createAWSLogDriver(prefix) {
        return new aws_ecs_1.AwsLogDriver({ streamPrefix: prefix });
    }
}
exports.QueueProcessingServiceBase = QueueProcessingServiceBase;
//# sourceMappingURL=data:application/json;base64,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