import corner
import jax.numpy as jnp
import jax.random as random
import numpy as np
import numpyro
import numpyro.distributions as dist
from numpyro.infer import MCMC, NUTS
from prettyPlot.plotting import *


def read_data(
    filename: str, time_ind: int, conc_ind: int, ind_start=0
) -> tuple[np.ndarray, np.ndarray]:
    """
    Read observation data
    """
    tmp = np.loadtxt(filename)
    return tmp[ind_start:, time_ind], tmp[ind_start:, conc_ind]


def c_model(
    theta: list, t: np.ndarray | jnp.ndarray, t0: float, c0: float
) -> np.ndarray | jnp.ndarray:
    """
    Concentration vs time function involving kLa
    """
    cstar, kla = theta
    return (cstar - c0) * (1.0 - jnp.exp(-kla * (t - t0))) + c0


def bayes_step(
    time_obs: np.ndarray | jnp.ndarray,
    concentration_obs: np.ndarray | jnp.ndarray,
):
    """
    Rule for MCMC sampling
    """
    # define parameters (incl. prior ranges)
    cstar = numpyro.sample("cstar", dist.Uniform(1e-8, 1000))
    kla = numpyro.sample("kla", dist.Uniform(1e-8, 1000))
    sigma = numpyro.sample("sigma", dist.Uniform(1e-8, 1))

    # initial values
    t0 = time_obs[0]
    c0 = concentration_obs[0]

    # concentration predictions
    c_pred = c_model([cstar, kla], t=time_obs, t0=t0, c0=c0)

    # likelihood uncertainty
    std_c = jnp.ones(c_pred.shape[0]) * sigma

    # MCMC sampling with multivariate normal
    numpyro.sample("obs", dist.Normal(c_pred, std_c), obs=concentration_obs)


def hmc_samples_2_np_samples(hmc_samples: dict) -> np.ndarray:
    """
    go from hmc samples generated by numpyro to numpy
    """
    labels = list(hmc_samples.keys())
    nsamples = len(hmc_samples[labels[0]])
    nparams = len(labels)
    np_hmc_samples = np.zeros((nsamples, nparams))
    labels_np = ["cstar", "kla", "sigma"]

    for ilabel, label in enumerate(labels):
        if label == "cstar":
            nplabel = labels_np.index("cstar")
        if label == "kla":
            nplabel = labels_np.index("kla")
        if label == "sigma":
            nplabel = labels_np.index("sigma")
        np_hmc_samples[:, nplabel] = np.array(hmc_samples[label])

    return np_hmc_samples


def np_samples_2_pred(
    np_hmc_samples: np.ndarray,
    time_obs: np.ndarray,
    concentration_obs: np.ndarray,
) -> dict:
    """
    go from parameter samples to predictions
    """
    # Uncertainty propagation
    nsamples = np_hmc_samples.shape[0]
    realization = []
    for i in range(nsamples):
        c_pred = c_model(
            np_hmc_samples[i, :-1], time_obs, time_obs[0], concentration_obs[0]
        )
        realization.append(c_pred)
    realization = np.array(realization)

    mean_real = np.mean(realization, axis=0)
    p90_real = np.percentile(realization, 90, axis=0)
    p10_real = np.percentile(realization, 10, axis=0)

    return {"mean_pred": mean_real, "p10_pred": p10_real, "p90_pred": p90_real}


def read_kla_cstar(hmc_samples: dict) -> dict:
    """
    go from hmc samples generated by numpyro to relevant arrays
    """
    np_hmc_samples = hmc_samples_2_np_samples(hmc_samples)
    return {
        "kla_samples": np_hmc_samples[:, 1],
        "cstar_samples": np_hmc_samples[:, 0],
    }


def compute_accuracy(
    hmc_samples: dict, time_obs: np.ndarray, concentration_obs: np.ndarray
) -> float:
    """
    Compute how well the sampled parameter explain the observed data
    """
    np_hmc_samples = hmc_samples_2_np_samples(hmc_samples)
    pred_dict = np_samples_2_pred(np_hmc_samples, time_obs, concentration_obs)
    mean_real = pred_dict["mean_pred"]

    return np.mean(
        abs(mean_real - concentration_obs)
        / np.clip(concentration_obs, a_min=1e-12, a_max=None)
    )


def check_data_size(time_obs: np.ndarray, concentration_obs: np.ndarray):
    """
    Sanity checks
    """
    assert time_obs.shape == concentration_obs.shape
    assert len(time_obs) > 5
    assert len(time_obs.shape) == 1


def compute_kla(
    filename: str,
    time_ind: int,
    conc_ind: int,
    max_chop: int | None = None,
    bootstrap: bool = False,
) -> dict:
    """
    Main loop to compute kla
    """
    num_warmup = 4000
    num_samples = 1000
    rng_key = random.PRNGKey(0)
    rng_key, rng_key_ = random.split(rng_key)

    acc = []
    kla = []
    cstar = []
    kla_err = []
    cstar_err = []
    ind = []
    data_t_tmp, data_c_tmp = read_data(filename, time_ind, conc_ind, 0)
    check_data_size(data_t_tmp, data_c_tmp)

    # Find where to start in the timeseries
    for ind_start in range(len(data_t_tmp) - 5):
        if max_chop is not None:
            max_chop -= 1
        print(f"Chopping index = {ind_start}")
        data_t, data_c = read_data(filename, time_ind, conc_ind, ind_start)

        # Hamiltonian Monte Carlo (HMC) with no u turn sampling (NUTS)
        kernel = NUTS(bayes_step, target_accept_prob=0.9)
        mcmc = MCMC(
            kernel,
            num_warmup=num_warmup,
            num_samples=num_samples,
            progress_bar=False,
        )
        mcmc.run(
            rng_key_,
            time_obs=jnp.array(data_t),
            concentration_obs=jnp.array(data_c),
        )
        # mcmc.print_summary()

        # Read samples
        hmc_samples = mcmc.get_samples()
        samp_dict = read_kla_cstar(hmc_samples)
        kla_samples = samp_dict["kla_samples"]
        cstar_samples = samp_dict["cstar_samples"]

        accuracy = compute_accuracy(hmc_samples, data_t, data_c)

        ind.append(ind_start)
        acc.append(accuracy)
        kla.append(np.mean(kla_samples))
        cstar.append(np.mean(cstar_samples))
        kla_err.append(np.std(kla_samples))
        cstar_err.append(np.std(cstar_samples))

        # post_cal(hmc_samples, data_t, data_c)
        # If accuracy has significantly improved, we have chopped enough
        if len(acc) >= 3 and abs(acc[-1] - acc[-2]) < 0.1 * acc[0]:
            break
        if max_chop == 0:
            break

    # bootstrap
    bootstrapped = False
    if bootstrap and len(data_t_tmp[ind[-1] :]) > 10:
        kla_boot = []
        kla_err_boot = []
        cstar_boot = []
        cstar_err_boot = []
        bootstrapped = True
        print("BOOTSTRAP")

        for i in range(4):
            print(f"\t scenario {i}")
            if i == 0:
                data_t = data_t_tmp[ind_start + 1 :]
                data_c = data_c_tmp[ind_start + 1 :]
            elif i == 1:
                data_t = data_t_tmp[ind_start + 2 :]
                data_c = data_c_tmp[ind_start + 2 :]
            elif i == 2:
                data_t = data_t_tmp[ind_start:-1]
                data_c = data_c_tmp[ind_start:-1]
            elif i == 3:
                data_t = data_t_tmp[ind_start:-2]
                data_c = data_c_tmp[ind_start:-2]

            # Hamiltonian Monte Carlo (HMC) with no u turn sampling (NUTS)
            kernel = NUTS(bayes_step, target_accept_prob=0.9)
            mcmc = MCMC(
                kernel,
                num_warmup=num_warmup,
                num_samples=num_samples,
                progress_bar=False,
            )
            mcmc.run(
                rng_key_,
                time_obs=jnp.array(data_t),
                concentration_obs=jnp.array(data_c),
            )
            # mcmc.print_summary()

            # Read samples
            hmc_samples = mcmc.get_samples()
            samp_dict = read_kla_cstar(hmc_samples)
            kla_samples = samp_dict["kla_samples"]
            cstar_samples = samp_dict["cstar_samples"]

        kla_boot.append(np.mean(kla_samples))
        kla_err_boot.append(np.std(kla_samples))
        cstar_boot.append(np.mean(cstar_samples))
        cstar_err_boot.append(np.std(cstar_samples))

    else:
        kla_boot = [kla[-1]]
        kla_err_boot = [kla_err[-1]]
        cstar_boot = [cstar[-1]]
        cstar_err_boot = [cstar_err[-1]]

    return {
        "filename": filename,
        "time_ind": time_ind,
        "conc_ind": conc_ind,
        "kla": np.mean(np.array(kla_boot)),
        "kla_err": np.mean(np.array(kla_err_boot)),
        "cstar": np.mean(np.array(cstar_boot)),
        "cstar_err": np.mean(np.array(cstar_err_boot)),
        "kla_noboot": np.mean(np.array([kla[-1]])),
        "kla_err_noboot": np.mean(np.array([kla_err[-1]])),
        "cstar_noboot": np.mean(np.array([cstar[-1]])),
        "cstar_err_noboot": np.mean(np.array([cstar_err[-1]])),
        "bootstrapped": bootstrapped,
    }


def post_cal(
    hmc_samples: dict, time_obs: np.ndarray, concentration_obs: np.ndarray
) -> None:
    """
    Plot MCMC results
    """
    np_hmc_samples = hmc_samples_2_np_samples(hmc_samples)
    labels_np = ["cstar", "kla", "sigma"]
    nsamples = np_hmc_samples.shape[0]
    labels = list(hmc_samples.keys())
    nparams = len(labels)

    # Distribution of the parameter samples
    fig = corner.corner(np_hmc_samples, labels=labels_np)

    # Convergence
    fig, axes = plt.subplots(nparams, sharex=True)
    for i in range(nparams):
        ax = axes[i]
        ax.plot(np_hmc_samples[:, i], "k", alpha=0.3, rasterized=True)
        ax.set_ylabel(labels[i])

    # Uncertainty propagation
    pred_dict = np_samples_2_pred(np_hmc_samples, time_obs, concentration_obs)
    mean_real = pred_dict["mean_pred"]
    p10_real = pred_dict["p10_pred"]
    p90_real = pred_dict["p90_pred"]

    fig = plt.figure()
    plt.plot(
        time_obs, mean_real, color="k", linewidth=3, label="mean predictions"
    )
    plt.plot(
        time_obs,
        p90_real,
        "--",
        color="k",
        linewidth=3,
        label="10th and 90th percentile",
    )
    plt.plot(time_obs, p10_real, "--", color="k", linewidth=3)
    plt.plot(
        time_obs,
        concentration_obs,
        "o",
        color="k",
        markersize=7,
        label="Data observed",
    )
    pretty_labels("t", "c[mol/m3]", 14)
    pretty_legend()
    plt.show()


def print_res_dict(res_dict: dict) -> None:
    """
    Log to screen
    """
    file = res_dict["filename"]
    t_ind = res_dict["time_ind"]
    c_ind = res_dict["conc_ind"]
    kla = res_dict["kla"]
    kla_err = res_dict["kla_err"]
    cs = res_dict["cstar"]
    cs_err = res_dict["cstar_err"]

    kla_nb = res_dict["kla_noboot"]
    kla_err_nb = res_dict["kla_err_noboot"]
    cs_nb = res_dict["cstar_noboot"]
    cs_err_nb = res_dict["cstar_err_noboot"]

    bs = res_dict["bootstrapped"]

    print(f"For {file} with time index: {t_ind}, concentration index: {c_ind}")
    if bs:
        print(f"\tkla = {kla:.4g} +/- {kla_err:.4g}")
        print(f"\tcstar = {cs:.4g} +/- {cs_err:.4g}")
        print(f"Without data bootstrap")
    print(f"\tkla = {kla_nb:.4g} +/- {kla_err_nb:.4g}")
    print(f"\tcstar = {cs_nb:.4g} +/- {cs_err_nb:.4g}")


if __name__ == "__main__":
    res_dict = compute_kla("data_kla/volume_avg.dat", time_ind=0, conc_ind=1)
    print_res_dict(res_dict)
