"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProfilingGroup = void 0;
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const codeguruprofiler_generated_1 = require("./codeguruprofiler.generated");
class ProfilingGroupBase extends core_1.Resource {
    /**
     * Grant access to publish profiling information to the Profiling Group to the given identity.
     *
     * This will grant the following permissions:
     *
     *  - codeguru-profiler:ConfigureAgent
     *  - codeguru-profiler:PostAgentProfile
     *
     * @param grantee Principal to grant publish rights to
     */
    grantPublish(grantee) {
        // https://docs.aws.amazon.com/codeguru/latest/profiler-ug/security-iam.html#security-iam-access-control
        return aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions: ['codeguru-profiler:ConfigureAgent', 'codeguru-profiler:PostAgentProfile'],
            resourceArns: [this.profilingGroupArn],
        });
    }
    /**
     * Grant access to read profiling information from the Profiling Group to the given identity.
     *
     * This will grant the following permissions:
     *
     *  - codeguru-profiler:GetProfile
     *  - codeguru-profiler:DescribeProfilingGroup
     *
     * @param grantee Principal to grant read rights to
     */
    grantRead(grantee) {
        // https://docs.aws.amazon.com/codeguru/latest/profiler-ug/security-iam.html#security-iam-access-control
        return aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions: ['codeguru-profiler:GetProfile', 'codeguru-profiler:DescribeProfilingGroup'],
            resourceArns: [this.profilingGroupArn],
        });
    }
}
/**
 * A new Profiling Group.
 */
class ProfilingGroup extends ProfilingGroupBase {
    constructor(scope, id, props = {}) {
        var _a;
        super(scope, id, {
            physicalName: (_a = props.profilingGroupName) !== null && _a !== void 0 ? _a : core_1.Lazy.stringValue({ produce: () => this.generateUniqueId() }),
        });
        const profilingGroup = new codeguruprofiler_generated_1.CfnProfilingGroup(this, 'ProfilingGroup', {
            profilingGroupName: this.physicalName,
        });
        this.profilingGroupName = this.getResourceNameAttribute(profilingGroup.ref);
        this.profilingGroupArn = this.getResourceArnAttribute(profilingGroup.attrArn, {
            service: 'codeguru-profiler',
            resource: 'profilingGroup',
            resourceName: this.physicalName,
        });
    }
    /**
     * Import an existing Profiling Group provided a Profiling Group Name.
     *
     * @param scope The parent creating construct
     * @param id The construct's name
     * @param profilingGroupName Profiling Group Name
     */
    static fromProfilingGroupName(scope, id, profilingGroupName) {
        const stack = core_1.Stack.of(scope);
        return this.fromProfilingGroupArn(scope, id, stack.formatArn({
            service: 'codeguru-profiler',
            resource: 'profilingGroup',
            resourceName: profilingGroupName,
        }));
    }
    /**
     * Import an existing Profiling Group provided an ARN.
     *
     * @param scope The parent creating construct
     * @param id The construct's name
     * @param profilingGroupArn Profiling Group ARN
     */
    static fromProfilingGroupArn(scope, id, profilingGroupArn) {
        class Import extends ProfilingGroupBase {
            constructor() {
                super(...arguments);
                this.profilingGroupName = core_1.Stack.of(scope).parseArn(profilingGroupArn).resource;
                this.profilingGroupArn = profilingGroupArn;
            }
        }
        return new Import(scope, id);
    }
    generateUniqueId() {
        const name = this.node.uniqueId;
        if (name.length > 240) {
            return name.substring(0, 120) + name.substring(name.length - 120);
        }
        return name;
    }
}
exports.ProfilingGroup = ProfilingGroup;
//# sourceMappingURL=data:application/json;base64,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