# -*- coding: utf-8 -*-
"""
    test_party

"""
import unittest

import trytond.tests.test_tryton
from trytond.transaction import Transaction
from trytond.tests.test_tryton import DB_NAME, USER, CONTEXT

from test_prestashop import get_objectified_xml, BaseTestCase


class TestProduct(BaseTestCase):
    """Test Product > Template/variant integration
    """

    def test_0010_product_template_import(self):
        """Test Product Template import
        """
        with Transaction().start(DB_NAME, USER, context=CONTEXT) as txn:
            # Call method to setup defaults
            self.setup_defaults()

            with txn.set_context(
                current_channel=self.channel.id, ps_test=True,
            ):
                self.setup_channels()

                self.assertEqual(len(self.ProductTemplate.search([])), 1)
                self.assertEqual(len(self.Product.search([])), 1)
                self.assertEqual(len(self.ChannelListing.search([
                    ('channel', '=', self.channel.id)
                ])), 0)

                product_data = get_objectified_xml('products', 1)
                template = self.Product.create_from(
                    self.channel, product_data
                ).template

                self.assertEqual(len(self.ProductTemplate.search([])), 2)
                self.assertEqual(len(self.Product.search([])), 2)
                self.assertEqual(len(self.ChannelListing.search([
                    ('channel', '=', self.channel.id)
                ])), 1)

                # Product name should be in english and french
                with txn.set_context(language='en_US'):
                    template = self.ProductTemplate(template.id)
                    self.assertEqual(
                        template.name, 'iPod Nano'
                    )
                with txn.set_context(language='fr_FR'):
                    template = self.ProductTemplate(template.id)
                    self.assertEqual(
                        template.name, 'iPod Nano French'
                    )

                # Product description should be in english only
                with txn.set_context(language='en_US'):
                    product_desc_en = self.Product(
                        template.products[0].id
                    ).description
                with txn.set_context(language='fr_FR'):
                    product_desc_fr = self.Product(
                        template.products[0].id
                    ).description
                self.assertEqual(product_desc_en, product_desc_fr)

                # Nothing should be created under site_alt
                self.assertEqual(len(self.ChannelListing.search([
                    ('channel', '=', self.alt_channel.id)
                ])), 0)

                # Get template using prestashop data
                self.assertEqual(
                    template.id,
                    self.Product.create_from(
                        self.channel, product_data
                    ).template.id
                )

    def test_0020_product_import(self):
        """Test Product import
        """
        with Transaction().start(DB_NAME, USER, context=CONTEXT):
            # Call method to setup defaults
            self.setup_defaults()

            with Transaction().set_context(
                current_channel=self.channel.id, ps_test=True,
            ):
                self.setup_channels()

                self.assertEqual(len(self.ProductTemplate.search([])), 1)
                self.assertEqual(len(self.Product.search([])), 1)
                self.assertEqual(len(self.ChannelListing.search([
                    ('channel', '=', self.channel.id)
                ])), 0)

                product = self.Product.create_from(
                    self.channel, get_objectified_xml('combinations', 1)
                )
                # This should create a template and two variants where one
                # is created by template and other by this combination
                self.assertEqual(len(self.ProductTemplate.search([])), 2)
                self.assertEqual(len(self.Product.search([])), 3)
                self.assertEqual(len(self.ChannelListing.search([
                    ('channel', '=', self.channel.id)
                ])), 1)

                # Try importing the same product again, it should NOT create a
                # new one.
                self.Product.create_from(
                    self.channel, get_objectified_xml('combinations', 1)
                )
                self.assertEqual(len(self.Product.search([])), 3)

                # Test getting product using prestashop data
                self.assertEqual(
                    product.id,
                    self.Product.create_from(
                        self.channel, get_objectified_xml('combinations', 1)
                    ).id
                )

                # Test getting product using prestashop ID
                # Nothing should be created under site_alt
                self.assertEqual(len(self.ChannelListing.search([
                    ('channel', '=', self.alt_channel.id)
                ])), 0)


def suite():
    "Prestashop Product test suite"
    suite = trytond.tests.test_tryton.suite()
    suite.addTests(
        unittest.TestLoader().loadTestsFromTestCase(TestProduct)
    )
    return suite


if __name__ == '__main__':
    unittest.TextTestRunner(verbosity=2).run(suite())
