"""Classes for records with connectivity and bookkeeping information.

.. codeauthor::  Todd Dolinsky
.. codeauthor::  Yong Huang
.. codeauthor::  Nathan Baker
"""
import logging
from .general import BaseRecord


_LOGGER = logging.getLogger(__name__)


class Connection(BaseRecord):
    """CONECT class

    The CONECT records specify connectivity between atoms for which
    coordinates are supplied. The connectivity is described using the atom
    serial number as found in the entry. CONECT records are mandatory for
    HET groups (excluding water) and for other bonds not specified in the
    standard residue connectivity table which involve atoms in standard
    residues (see Appendix 4 for the list of standard residues). These
    records are generated by the PDB.

    +---------+-------------+----------+--------------------------------------+
    | COLUMNS | DATA TYPE   | FIELD    | DEFINITION                           |
    +=========+=============+==========+======================================+
    | 1-6     | Record name | "CONECT" |                                      |
    +---------+-------------+----------+--------------------------------------+
    | 7-11    | Integer     | serial   | Atom  serial number                  |
    +---------+-------------+----------+--------------------------------------+
    | 12-16   | Integer     | serial   | Serial number of bonded atom         |
    +---------+-------------+----------+--------------------------------------+
    | 17-21   | Integer     | serial   | Serial  number of bonded atom        |
    +---------+-------------+----------+--------------------------------------+
    | 22-26   | Integer     | serial   | Serial number of bonded atom         |
    +---------+-------------+----------+--------------------------------------+
    | 27-31   | Integer     | serial   | Serial number of bonded atom         |
    +---------+-------------+----------+--------------------------------------+
    """

    def __init__(self):
        super().__init__()
        self.serial = []

    def parse_line(self, line):
        """Parse PDB-format line.

        :param str line:  line to parse
        """
        super().parse_line(line)
        if line is None:
            line = ""
        istart = 6
        iend = 11
        while True:
            try:
                self.serial.append(int(line[istart:iend].strip()))
                istart += 5
                iend += 5
            except ValueError:
                break

    def __str__(self):
        string = "CONECT"
        for serial in self.serial:
            string += f"{serial:5}"
        return string.strip()


class Master(BaseRecord):
    """MASTER class

    The MASTER record is a control record for bookkeeping.
    It lists the number of lines in the coordinate entry or file for selected
    record types.

    +---------+-------------+-----------+-------------------------------------+
    | COLUMNS | DATA TYPE   | FIELD     | DEFINITION                          |
    +=========+=============+===========+=====================================+
    | 1-6     | Record name | "MASTER"  |                                     |
    +---------+-------------+-----------+-------------------------------------+
    | 11-15   | Integer     | numRemark | Number of REMARK records            |
    +---------+-------------+-----------+-------------------------------------+
    | 16-20   | Integer     | "0"       |                                     |
    +---------+-------------+-----------+-------------------------------------+
    | 21-25   | Integer     | numHet    | Number of HET records               |
    +---------+-------------+-----------+-------------------------------------+
    | 26-30   | Integer     | numHelix  | Number of HELIX records             |
    +---------+-------------+-----------+-------------------------------------+
    | 31-35   | Integer     | numSheet  | Number of SHEET records             |
    +---------+-------------+-----------+-------------------------------------+
    | 36-40   | Integer     | numTurn   | deprecated                          |
    +---------+-------------+-----------+-------------------------------------+
    | 41-45   | Integer     | numSite   | Number of SITE records              |
    +---------+-------------+-----------+-------------------------------------+
    | 46-50   | Integer     | numXform  | Number of coordinate transformation |
    |         |             |           | records  (ORIGX+SCALE+MTRIX)        |
    +---------+-------------+-----------+-------------------------------------+
    | 51-55   | Integer     | numCoord  | Number of atomic coordinate records |
    |         |             |           | records (ATOM+HETATM)               |
    +---------+-------------+-----------+-------------------------------------+
    | 56-60   | Integer     | numTer    | Number of TER records               |
    +---------+-------------+-----------+-------------------------------------+
    | 61-65   | Integer     | numConect | Number of CONECT records            |
    +---------+-------------+-----------+-------------------------------------+
    | 66-70   | Integer     | numSeq    | Number of SEQRES records            |
    +---------+-------------+-----------+-------------------------------------+
    """

    def __init__(self):
        super().__init__()
        self.num_remark = None
        self.num_het = None
        self.num_helix = None
        self.num_sheet = None
        self.num_turn = None
        self.num_site = None
        self.num_xform = None
        self.num_coord = None
        self.num_ter = None
        self.num_conect = None
        self.num_seq = None

    def parse_line(self, line):
        """Parse a PDB-format line

        :param str line:  line to parse
        """
        super().parse_line(line)
        self.num_remark = int(line[10:15].strip())
        self.num_het = int(line[20:25].strip())
        self.num_helix = int(line[25:30].strip())
        self.num_sheet = int(line[30:35].strip())
        self.num_turn = int(line[35:40].strip())
        self.num_site = int(line[40:45].strip())
        self.num_xform = int(line[45:50].strip())
        self.num_coord = int(line[50:55].strip())
        self.num_ter = int(line[55:60].strip())
        self.num_conect = int(line[60:65].strip())
        self.num_seq = int(line[65:70].strip())

    def __str__(self):
        return (
            f"MASTER    {self.num_remark:5}    0{self.num_het:5}"
            f"{self.num_helix:5}{self.num_sheet:5}{self.num_turn:5}"
            f"{self.num_site:5}{self.num_xform:5}{self.num_coord:5}"
            f"{self.num_ter:5}{self.num_conect:5}{self.num_seq:5}"
        ).strip()
