import os
from collections.abc import Iterable
from pathlib import Path


def find_project_root_by_folder(start: Path, folder_names: Iterable[str]) -> Path | None:
    """Find the project root by looking for specific folder names in parent directories."""
    current = start.resolve()
    for parent in [current, *list(current.parents)]:
        for folder in folder_names:
            if (parent / folder).is_dir():
                return parent
    return None


class Paths:
    """Central place for all important paths."""

    # user project root (to be set dynamically from CLI)
    _user_root: Path | None = None

    # ------------------ SET/GET ------------------

    @classmethod
    def set_user_root(cls, path: Path) -> None:
        """Set the user project root path."""
        cls._user_root = path.resolve()

    @classmethod
    def user_root(cls) -> Path:
        """Get the user project root path."""
        if cls._user_root is not None:
            return cls._user_root

        env_root = os.getenv("PYTEST_TUI_RUNNER_ROOT")
        if env_root:
            return Path(env_root).resolve()

        raise RuntimeError("User project root is not set yet.")

    # ------------------ USER-DEPENDENT PATHS ------------------

    @classmethod
    def config(cls) -> Path:
        """Path to the user configuration file."""
        return cls.user_root() / "pytest_tui_runner" / "default.yaml"

    @classmethod
    def log_dir(cls) -> Path:
        """Directory for log files."""
        return cls.user_root() / "pytest_tui_runner" / "logs"

    @classmethod
    def state_file(cls) -> Path:
        """Path to the file where the state of widgets is stored."""
        return cls.user_root() / "pytest_tui_runner" / "widgets_state.json"

    @classmethod
    def log_file(cls) -> Path:
        """Path to the main log file."""
        return cls.log_dir() / "app.log"

    @classmethod
    def log_config_file(cls) -> Path:
        """Path to the log configuration file."""
        return cls.log_dir() / "config.yaml"

    @classmethod
    def pytest_report(cls) -> Path:
        """Path to the JSON report generated by pytest-json-report plugin."""
        return cls.user_root() / "pytest_tui_runner" / "results" / "pytest_report.json"
