## BASIC PYTHON LIBRARIES
import os
from os import path as p 
from shutil import copy, rmtree

## drMD LIBRARIES
from ExaminationRoom import drLogger, drClusterizer
from UtilitiesCloset import drSelector

## PDB // DATAFRAME UTILS
from pdbUtils import pdbUtils

## CLEAN CODE
from typing import List, Dict, Union, Any, Optional
from os import PathLike
from UtilitiesCloset.drCustomClasses import FilePath, DirectoryPath
######################################################################################################
def clean_up_handler(batchConfig: dict) -> None:
    """
    Handler for performing post simulation operations based on the batch configuration.

    Args:
        batchConfig (dict): The batch configuration dictionary.

    Returns:
        None
    """

    if not "postSimulationInfo" in batchConfig:
        return

    ## run clustering if instructed to in config file
    cluster_handler(batchConfig)
    ## collect endpoint PDB files if instructed to in config file
    endpoint_handler(batchConfig)
    ## delete unwanted directories as instructed to in config file
    directory_cleanup_handler(batchConfig)

    
######################################################################################################
def cluster_handler(batchConfig: Dict) -> None:
    """
    Function to perform clustering on the trajectories generated by the MD simulations.
    Will only run clustering if told to in the config file

    Args:
        batchConfig (dict): The batch configuration dictionary.

    Returns:
        None
    
    """
    ## unpack batchConfig
    postSimulationInfo: Dict = batchConfig["postSimulationInfo"]
    pathInfo: Dict = batchConfig["pathInfo"]
    ## skip no clusterInfo key in postSimulationInfo exists
    if not "clusterInfo" in postSimulationInfo:
        return
    ## let user know what's going on
    drLogger.log_info(f"-->{' '*4}Performing clustering on MD trajectories...", True )
    ## get clusterInfo dict
    clusterInfo: Dict = postSimulationInfo["clusterInfo"]
    ## run clustering
    allClusterPdbs: List[FilePath] = drClusterizer.clustering_manager(pathInfo, clusterInfo)

    ## postprocess cluster PDB files
    ## remove atoms from cluster PDBS files as instructed
    removeAtoms = clusterInfo.get("removeAtoms", False)
    if removeAtoms:
        remove_atoms_from_pdbs(allClusterPdbs, removeAtoms)
    ## collect clustered PDB files and put them in one directory per step
    if "collate" in clusterInfo:
        if clusterInfo["collate"]:
            collate_pdbs(allClusterPdbs, pathInfo)
######################################################################################################
def endpoint_handler(batchConfig: Dict) -> None:
    """
    Function to collect endpoint PDB files from the trajectories generated by the MD simulations.
    Will only run if told to in the config file

    Args:
        batchConfig (dict): The batch configuration dictionary.

    Returns:
        None
    """
    ## unpack batchConfig
    postSimulationInfo: Dict = batchConfig["postSimulationInfo"]
    pathInfo: Dict = batchConfig["pathInfo"]
    endpointInfo = postSimulationInfo.get("endPointInfo", False)
    ## skip no endpointInfo key in postSimulationInfo exists
    if not endpointInfo:
        return
    ## get endpoint PDB files
    endpointPdbs: List[Union[PathLike, str]] = get_endpoint_pdbs(endpointInfo, pathInfo)
    ## postprocess endpoint PDB files
    ## remove atoms from cluster PDBS files as instructed
    removeAtoms = endpointInfo.get("removeAtoms", False)
    if removeAtoms:
        remove_atoms_from_pdbs(endpointPdbs, removeAtoms)
    ## collect endpoint PDB files and put them in one directory per step
    collate_pdbs(endpointPdbs, pathInfo)
######################################################################################################
def directory_cleanup_handler(batchConfig: dict) -> None:
    """
    Deletes unwanted directories after the simulation is complete

    Args:
        batchConfig (dict): The batch configuration dictionary.

    Returns:
        None
    
    """
    ## unpack batchConfig
    postSimulationInfo = batchConfig["postSimulationInfo"]
    ## skip no directoryCleanUpInfo key in postSimulationInfo exists
    if not "directoryCleanUpInfo" in postSimulationInfo:
        return
    ## get directoryCleanUpInfo dict
    directoryCleanupInfo: Dict = postSimulationInfo["directoryCleanUpInfo"]

    ## if instructed, delete all simulation directories 
    ## NOTE that this means only collated directories will be retained
    ## NOTE this will save a lot of storage, but you will loose all of your trajectory files
    if  "removeAllSimulationDirs" in directoryCleanupInfo:
        if directoryCleanupInfo["removeAllSimulationDirs"]:
            remove_siulation_directories(batchConfig)
            return
    ## remove any specified step directories
    ## NOTE this can be useful to delete prep / equiibriation directories to save storage space
    if "stepsToRemove" in directoryCleanupInfo:
        stepsToRemove = directoryCleanupInfo["stepsToRemove"]
        remove_step_directories(batchConfig, stepsToRemove)

######################################################################################################
def remove_step_directories(batchConfig: dict, stepsToRemove: list) -> None:
    """
    Removes any specified step directories after the simulation is complete

    Args:
        batchConfig (dict): The batch configuration dictionary.

    Returns:
        None
    """
    ## let user know what we are doing
    drLogger.log_info(f"-->{' '*4}Removing specified simulation directories", True)

    ## unpack batchConfig
    inputDir: DirectoryPath= batchConfig["pathInfo"]["inputDir"]
    outDir: DirectoryPath = batchConfig["pathInfo"]["outputDir"]

    ## create list of directories to remove 
    dirsToRemove: List[DirectoryPath] = [p.join(outDir,p.splitext(file)[0], stepName) 
                                        for file in os.listdir(inputDir) 
                                        if p.splitext(file)[1] == ".pdb" 
                                        for stepName in stepsToRemove]
    
    ## remove directories
    for dir in dirsToRemove:
        rmtree(dir)
######################################################################################################
def remove_siulation_directories(batchConfig: dict) -> None:
    """
    Remove all simulation directories after the simulation is complete
    USE WITH CAUTION!

    Args:
        batchConfig (dict): The batch configuration dictionary.

    Returns:
        None
    """
    ## let user know what we are doing
    drLogger.log_info(f"-->{' '*4}Removing all simulation directories", True)

    ## unpack batchConfig
    inputDir = batchConfig["pathInfo"]["inputDir"]
    outDir = batchConfig["pathInfo"]["outputDir"]
    ## create list of directories to remove
    dirsToRemove = [p.join(outDir,p.splitext(file)[0]) for file in os.listdir(inputDir) if p.splitext(file)[1] == ".pdb"]

    ## remove directories
    for dir in dirsToRemove:
        rmtree(dir)

######################################################################################################
def collate_pdbs(pdbFiles: List[FilePath], pathInfo: Dict) -> None:
    """
    Moves pdb files in a list into per-step directories, outside of individual simulation directories

    Args:
        pdbFiles (List[FilePath]): A list of pdb files to collate
        pathInfo (Dict): The path information dictionary.

    Returns:
        None
    """
    ## let user know what we are doing
    drLogger.log_info(f"-->{' '*4}Collating {len(pdbFiles)} PDB files into per-step directories", True)

    ## for each pdb file, move it to its per-step directory
    for pdbFile in pdbFiles:
        stepName = p.basename(p.dirname(pdbFile))
        stepCollateDir = p.join(pathInfo["outputDir"], "00_collated_pdbs", stepName)
        os.makedirs(stepCollateDir,exist_ok=True)
        copy(pdbFile, stepCollateDir)


######################################################################################################
def get_endpoint_pdbs(endPointInfo: Dict, pathInfo: Dict) -> List[FilePath]:
    """
    Collect the endpoint PDB files from specified steps for all runs in the batch

    Args:
        endPointInfo (Dict): The endpoint information dictionary.
        pathInfo (Dict): The path information dictionary.

    Returns:
        List[FilePath]: A list of endpoint PDB files
    
    """
    ## let user know what we are doing
    drLogger.log_info(f"-->{' '*4}Getting endpoint PDB files", True)

    ## unpack pathInfo
    outDir: DirectoryPath = pathInfo["outputDir"]
    ## create list of directories that are made by drMD - we don't want to look inside these
    notRunDirs: List[str] = ["00_configs", "01_ligand_parameters", "00_collated_pdbs", "00_drMD_logs"]
    ## create a list of all run directories
    runDirs: List[DirectoryPath] = [p.join(outDir, dir) for dir in os.listdir(outDir) if not dir in notRunDirs]
    ## create a list of step directories that we want to look for endpoint pdb files in 
    stepDirs: List[DirectoryPath] = [p.join(runDir,stepDir) for runDir in runDirs for stepDir in endPointInfo["stepNames"]]
    ## create a list of endpoint pdb files
    endpointPdbs: List[FilePath] = [p.join(stepDir,stepPdb) for stepDir in stepDirs 
                                    for stepPdb in os.listdir(stepDir)
                                    if p.splitext(stepPdb)[1] == ".pdb"]
    return endpointPdbs
######################################################################################################
def remove_atoms_from_pdbs(
    pdbFiles: List[FilePath],
    removeAtoms: List[Dict]) -> None:  
    """
    Removes atoms from a PDB file based on a selection
    This functoin is used to remove atoms from cluster and endpoint PDB files
    Args:
        pdbFiles (List[Union[PathLike, str]]): The PDB files to remove atoms from
        removeAtomsSelections (List[Dict]): The atom selections to remove

    Returns:       
        None
    """

    drLogger.log_info(f"-->{' '*4}Removing atoms from clustering pdbs...", True)

    removeAtomsSelections = [sele["selection"] for sele in removeAtoms]

    ## for each pdb file in input list
    for pdbFile in pdbFiles:
        ## convert to a dataframe
        pdbDf = pdbUtils.pdb2df(pdbFile)
        indexesToRemove = []
        for removeAtomsSelection in removeAtomsSelections:
            indexesToRemove.extend(drSelector.get_atom_indexes(removeAtomsSelection, pdbFile))
        droppedDf = pdbDf.drop(indexesToRemove)
        pdbUtils.df2pdb(droppedDf, pdbFile)
######################################################################################################

