"""
    This module implements the Report class for create_db_report.
"""
import os
import shutil
import warnings
import webbrowser
from pathlib import Path
from typing import Optional
from ...utils import is_notebook


class Report:
    """
    This class creates a customized Report object for the create_db_report function
    """

    def __init__(self, database_name: str, report: str) -> None:
        self.database_name = database_name
        self.report = report

    def show(self) -> None:
        """
        Open the report in notebook.
        """
        if is_notebook():
            warnings.warn(
                "Report generated by create_db_report currently does not support output in notebook,"
                " consider using report.show_browser() to view in browser.",
            )

    def save(self, path: Optional[str] = None) -> None:
        """
        Save report to current working directory.

        Parameters
        ----------
        path: Optional[str], default Path.cwd()
            The path to where the report will be saved.
        """
        output_folder_name = f"{self.database_name}_report"
        source = os.path.realpath(os.path.join(os.path.dirname(__file__), "layout"))
        if path:
            destination = Path(path).expanduser()
        else:
            path = str(Path.cwd())
            destination = Path(path).expanduser()
        destination = os.path.join(destination, output_folder_name)
        if os.path.exists(destination):
            shutil.rmtree(destination)
        shutil.copytree(source, destination)
        print(f"Report has been saved to {destination}!")

    def show_browser(self) -> None:
        """
        Open the report in the browser.
        """
        webbrowser.open(f"file://{self.report}", new=2)
