from .elements import Element, ElementCollection, Property
from typing import Dict, Iterable, FrozenSet, List, Union, Mapping


class ForEach(Element):
    """The class for representing loops of elements."""

    def __init__(self, name: str, content: Iterable[Element]):
        """
        Args:
            name (str): The name for this element (Cloud Office Print tag).
            content (Iterable[Element]): An iterable containing the elements for this loop element.
        """
        super().__init__(name)
        self._content = list(content)
        # if self._tags should be overwritten in a subclass of this one, remember to do so after calling super().__init__
        self._tags = {
            "{#" + name + "}",
            "{/" + name + "}"
        }

    @property
    def content(self) -> List[Element]:
        """Get the elements in this loop object.

        Returns:
            List[Element]: the elements in this loop object
        """
        return self._content

    @content.setter
    def content(self, value: Iterable[Element]):
        """Setter for the content of this loop object

        Args:
            value (Iterable[Element]): an iterable consisting of elements
        """
        self._content = value

    @property
    def available_tags(self) -> FrozenSet[str]:
        result = self._tags

        for element in self.content:
            result |= element.available_tags

        return frozenset(result)

    @property
    def as_dict(self) -> Dict:
        return {
            self.name: [element.as_dict for element in self.content]
        }


class Labels(ForEach):
    """Cloud Office Print also provides a way to print labels in Word documents.
    To do so you can create a document with labels by going to Mailings options and then to Labels.
    Fill in the tags in the address field and choose the type of label in the Label option.
    A document can then be generated by clicking New document.
    Currently however if labels are getting printed then we expect the document only containing labels and no other information,
    and that the tag keys are not used more than once."""

    def __init__(self, name: str, content: Iterable[Element]):
        """
        Args:
            name (str): The name for this element (Cloud Office Print tag).
            content (Iterable[Element]): An iterable containing the elements for this loop element.
        """
        super().__init__(name, content)
        self._tags = {
            "{-" + name + "}"
        }


class ForEachSlide(ForEach):
    """Loop where a slide will be repeated for each element of the loop. Only supported in PowerPoint templates."""

    def __init__(self, name: str, content: Iterable[Element]):
        """
        Args:
            name (str): The name for this element (Cloud Office Print tag).
            content (Iterable[Element]): An iterable containing the elements for this loop element.
        """
        super().__init__(name, content)
        self._tags = {
            "{!" + name + "}"
        }


class ForEachSheet(ForEach):
    """Loop where a sheet will be repeated for each element of the loop. Only supported in Excel templates."""

    def __init__(self, name: str, content: Union[Iterable[Element], Mapping[str, Element]]):
        """
        Args:
            name (str): The name for this element (Cloud Office Print tag).
            content (Union[Iterable[Element], Mapping[str, Element]]): An iterable or mapping containing the elements for this loop element.
        """
        # when content is a mapping, it means "sheet name": content for that sheet
        # when it's just an iterable, don't add sheet names (or users can add a Property manually)
        if isinstance(content, Mapping):
            new_content = []
            for sheetname, sheetcontent in content.items():
                # we need to add the additional sheet_name property,
                # so we should convert the Element to an ElementCollection if needed
                if not isinstance(sheetcontent, ElementCollection):
                    sheetcontent = ElementCollection.element_to_element_collection(
                        sheetcontent)
                # adding the new property containing sheet_name
                sheetcontent.add(Property("sheet_name", sheetname))
                new_content.append(sheetcontent)
            content = new_content
        # at this point, content is always Iterable[Element]

        super().__init__(name, content)
        self._tags = {
            "{!" + name + "}"
        }


class ForEachInline(ForEach):
    """Horizontal table looping for Word, Excel and CSV templates.
    Note: this tag can be used to repeat only one row in Word. In Excel this works like a normal loop tag and
    repeats the cells defined by the rectangular boundary of the starting and closing tag."""

    def __init__(self, name: str, content: Iterable[Element]):
        """
        Args:
            name (str): The name for this element (Cloud Office Print tag).
            content (Iterable[Element]): An iterable containing the elements for this loop element.
        """
        super().__init__(name, content)
        self._tags = {
            "{:" + name + "}",
            "{/" + name + "}"
        }


# These are the same, but they may not be forever
# and combining them into one class breaks consistency
ForEachHorizontal = ForEachInline


class ForEachTableRow(ForEach):
    """This tag will merge the cells of the loop defined by the tag over the amount of elements rows. 
    Only supported in PowerPoint templates."""

    def __init__(self, name: str, content: Iterable[Element]):
        """
        Args:
            name (str): The name for this element (Cloud Office Print tag).
            content (Iterable[Element]): An iterable containing the elements for this loop element.
        """
        super().__init__(name, content)
        self._tags = {
            "{=" + name + "}",
            "{/" + name + "}"
        }
