from collections.abc import Sequence

from engineer.registry import build_from_cfg
from ..registry import PIPELINES

@PIPELINES.register_module
class Compose(object):
    """Make data pre-processing pipeline customizable

    Args:
        transforms (list[dict | callable class]):
            Transform Operations with configs in pipelines
    """

    def __init__(self, transforms):

        assert isinstance(transforms, Sequence)
        self.transforms = []
        for transform in transforms:
            if isinstance(transform, dict):
                transform = build_from_cfg(transform, PIPELINES)
                self.transforms.append(transform)
            elif callable(transform):
                self.transforms.append(transform)
            else:
                raise TypeError('transform must be callable or a dict')

    def __call__(self, data):
        for t in self.transforms:
            data = t(data)
            if data is None:
                return None
        return data

    def __repr__(self):
        format_string = self.__class__.__name__ + '('
        for t in self.transforms:
            format_string += '\n'
            format_string += '    {0}'.format(t)
        format_string += '\n)'
        return format_string
