###############################################

# X3D Package for Python x3d.py

# generator:  X3duomToX3dPythonPackage.xslt
# X3DUOM:     X3dUnifiedObjectModel-4.0.xml
# Python X3D: https://www.web3d.org/x3d/stylesheets/python/python.html
"""
The x3d.py Python X3D Package supports programmers with Python interfaces and objects for standards-based X3D programming, all as open source.

This work is part of the X3D Python Scene Access Interface Library (X3DPSAIL).
"""

_DEBUG = True       # options True False
    
###############################################

# SimpleType Enumerations

ACCESSTYPECHOICES = (
    # strict set of allowed values follow, no other values are valid
    'initializeOnly', # A field with accessType initializeOnly can be initialized, but cannot send or receive events.
    'inputOnly', # A field with accessType inputOnly cannot be initialized or included in a scene file, but can receive input event values via a ROUTE.
    'outputOnly', # A field with accessType outputOnly cannot be initialized or included in a scene file, but can send output event values via a ROUTE.
    'inputOutput' # A field with accessType inputOutput can be initialized, and can also send or receive events.
)
def assertValidAccessType(fieldName, value):
    """
    Utility function to assert type validity of accessTypeChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if  isinstance(value,SFString) or isinstance(value,str):
        if str(value) in ACCESSTYPECHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + value + ' does not match allowed enumerations in ACCESSTYPECHOICES=' + str(ACCESSTYPECHOICES))
    if  isinstance(value,MFString):
        if MFString(value).XML() in ACCESSTYPECHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + MFString(value).XML() + ' does not match allowed enumerations in ACCESSTYPECHOICES=' + str(ACCESSTYPECHOICES))

APPLIEDPARAMETERSCHOICES = (
    # note these MFString values use XML syntax
    # strict set of allowed values follow, no other values are valid
    '"BOUNCE"', # The bounce field value is used.
    '"USER_FRICTION"', # The system will normally calculate the friction direction vector that is perpendicular to the contact normal. This setting indicates that the user-supplied value in this contact should be used.
    '"FRICTION_COEFFICIENT-2"', # Apply frictionCoefficients values
    '"ERROR_REDUCTION"', # Apply softnessErrorCorrection value
    '"CONSTANT_FORCE"', # Apply softnessConstantForceMix value
    '"SPEED-1"', # Apply first component of surfaceSpeed array
    '"SPEED-2"', # Apply second component of surfaceSpeed array
    '"SLIP-1"', # Apply first component of slipFactors array
    '"SLIP-2"' # Apply second component of slipFactors array
)
def assertValidAppliedParameters(fieldName, value):
    """
    Utility function to assert type validity of appliedParametersChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if  isinstance(value,SFString) or isinstance(value,str):
        if str(value) in APPLIEDPARAMETERSCHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + value + ' does not match allowed enumerations in APPLIEDPARAMETERSCHOICES=' + str(APPLIEDPARAMETERSCHOICES))
    if  isinstance(value,MFString):
        if MFString(value).XML() in APPLIEDPARAMETERSCHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + MFString(value).XML() + ' does not match allowed enumerations in APPLIEDPARAMETERSCHOICES=' + str(APPLIEDPARAMETERSCHOICES))

CLOSURETYPECHOICES = (
    # strict set of allowed values follow, no other values are valid
    'PIE', # Connects arc endpoints to center, forming a pie wedge
    'CHORD' # Connects arc endpoints directly to each other, as in chord on a circle
)
def assertValidClosureType(fieldName, value):
    """
    Utility function to assert type validity of closureTypeChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if  isinstance(value,SFString) or isinstance(value,str):
        if str(value) in CLOSURETYPECHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + value + ' does not match allowed enumerations in CLOSURETYPECHOICES=' + str(CLOSURETYPECHOICES))
    if  isinstance(value,MFString):
        if MFString(value).XML() in CLOSURETYPECHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + MFString(value).XML() + ' does not match allowed enumerations in CLOSURETYPECHOICES=' + str(CLOSURETYPECHOICES))

COLORMODECHOICES = (
    # strict set of allowed values follow, no other values are valid
    'POINT_COLOR', # RGB color of Material or Color, with alpha of texture
    'TEXTURE_COLOR', # Texture color only
    'TEXTURE_AND_POINT_COLOR' # Combined RGB color of texture with Material or Color, with alpha of texture
)
def assertValidColorMode(fieldName, value):
    """
    Utility function to assert type validity of colorModeChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if  isinstance(value,SFString) or isinstance(value,str):
        if str(value) in COLORMODECHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + value + ' does not match allowed enumerations in COLORMODECHOICES=' + str(COLORMODECHOICES))
    if  isinstance(value,MFString):
        if MFString(value).XML() in COLORMODECHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + MFString(value).XML() + ' does not match allowed enumerations in COLORMODECHOICES=' + str(COLORMODECHOICES))

COMPONENTNAMECHOICES = (
    # strict set of allowed values follow, no other values are valid
    'Core', # The Core component supplies the base functionality for the X3D run-time system, including the abstract base node type, field types, the event model, and routing.
    'CADGeometry', # The CADGeometry component is provided for Computer-Aided Design (CAD) nodes.
    'CubeMapTexturing', # The Cube Map Environmental Texturing component describes how additional texturing effects are defined to produce environmental effects such as reflections from objects.
    'DIS', # The Distributed Interactive Simulation (DIS) component provides networked interoperability with the IEEE DIS protocol for sharing state and conducting real-time platform-level simulations across multiple host computers.
    'EnvironmentalEffects', # Nodes in the Environmental effects component support the creation of realistic environmental effects such as panoramic backgrounds and fog.
    'EnvironmentalSensor', # The Environment Sensor nodes emit events indicating activity in the scene environment, usually based on interactions between the viewer and the world.
    'EventUtilities', # The Event Utility nodes provide the capability to filter, trigger, convert, or sequence numerous event-types for common interactive applications without the use of a Script node.
    'Followers', # The Follower nodes (Chasers and Dampers) support dynamic creation of smooth parameter transitions at run time.
    'Geometry2D', # The Geometry2D component defines how two-dimensional geometry is specified and what shapes are available.
    'Geometry3D', # The Geometry3D component describes how three-dimensional geometry is specified and defines ElevationGrid, Extrusion, IndexedFaceSet, and most primitive geometry nodes (Box, Cone, Cylinder, Sphere).
    'Geospatial', # The Geospatial component defines how to associate real-world locations in an X3D scene and specifies nodes particularly tuned for geospatial applications.
    'Grouping', # The Grouping component describes how nodes are organized into groups to establish a transformation hierarchy for the X3D scene graph.
    'HAnim', # The Humanoid Animation (HAnim) component for X3D defines node bindings and other details for implementing ISO/IEC 19774, the HAnim International Specification. Original name was H-Anim for X3D versions 3.0 through 3.3, both enumeration values HAnim and H-Anim are allowed to pass validation.
    'H-Anim', # Legacy enumeration H-Anim for X3D versions 3.0-3.3 provides backwards compatibility with Humanoid Animation (HAnim) version 1, preferred form of enumeration value is HAnim.
    'Interpolation', # Interpolator nodes provide keyframe-based animation capability.
    'KeyDeviceSensor', # The Key Device Sensor defines how keyboard keystrokes are inserted into an X3D world.
    'Layering', # The Layering component describes how to layer a set of subscene layers into a composite scene.
    'Layout', # The Layout component defines how to precisely position content in a scene in relation to the rendered results, especially for integrating 2D content with 3D content.
    'Lighting', # The Lighting component specifies how light sources are defined and positioned, as well as how lights effect the rendered image.
    'Navigation', # The Navigation component specifies how a user can effectively and intuitively move through and around a 3D scene.
    'Networking', # The Networking component defines node types and other features used to access file-based and streaming resources on the World Wide Web.
    'NURBS', # The NURBS component describes Non-uniform Rational B-Spline (NURBS) geometry and interpolation nodes.
    'ParticleSystems', # The Particle Systems component specifies how to model particles and their interactions through the application of basic physics principles to affect motion.
    'Picking', # The Picking component provides the ability to test for arbitrary object collision and provide basic capabilities to detecting object intersections and interactions.
    'PointingDeviceSensor', # Pointing device sensor nodes detect pointing events from user-interface devices, defining activities such as a user selecting a piece of geometry.
    'Rendering', # The Rendering component includes fundamental rendering primitives such as TriangleSet and PointSet nodes, as well as geometric properties nodes that define how coordinate indices, colors, normals and texture coordinates are specified.
    'RigidBodyPhysics', # The Rigid Body Physics component describes how to model rigid bodies and their interactions through the application of basic physics principles to effect motion.
    'Scripting', # The Script component describes how Script nodes are used to effect changes in X3D worlds.
    'Shaders', # The programmable shaders component describes how programmable shaders are specified and how they affect the visual appearance of geometry.
    'Shape', # The Shape component defines nodes for associating geometry with their visible properties and the scene environment.
    'Sound', # The Sound component defines how sound is delivered to an X3D world as well as how sounds are accessed.
    'Text', # The Text component defines how text strings are rendered in an X3D scene.
    'Texturing', # The Texturing component specifies how 2D texture images are defined and then positioned on associated geometry.
    'Texturing3D', # The Texturing3D component specifies how 3D volumetric textures describe surface properties as data points in a volume of space, rather than a flat surface.
    'Time', # The Time component defines how time is sensed, computed and associated with events in an X3D scene.
    'VolumeRendering' # The Volume Rendering component provides the ability to specify and render volumetric data sets.
)
def assertValidComponentName(fieldName, value):
    """
    Utility function to assert type validity of componentNameChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if  isinstance(value,SFString) or isinstance(value,str):
        if str(value) in COMPONENTNAMECHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + value + ' does not match allowed enumerations in COMPONENTNAMECHOICES=' + str(COMPONENTNAMECHOICES))
    if  isinstance(value,MFString):
        if MFString(value).XML() in COMPONENTNAMECHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + MFString(value).XML() + ' does not match allowed enumerations in COMPONENTNAMECHOICES=' + str(COMPONENTNAMECHOICES))

FIELDTYPECHOICES = (
    # strict set of allowed values follow, no other values are valid
    'SFBool', # Single Field (singleton) Boolean
    'MFBool', # Multiple Field (list) Boolean
    'SFColor', # Single Field (singleton) color value, red-green-blue
    'MFColor', # Multiple Field (list) color value, red-green-blue
    'SFColorRGBA', # Single Field (singleton) color value, red-green-blue alpha (opacity)
    'MFColorRGBA', # Multiple Field (list) color value, red-green-blue alpha (opacity)
    'SFDouble', # Single Field (singleton) double-precision (64-bit) float
    'MFDouble', # Multiple Field (list) 2-tuple double-precision (64-bit) float vector
    'SFFloat', # Single Field (singleton) single-precision (32-bit) float
    'MFFloat', # Multiple Field (list) single-precision (32-bit) float vector
    'SFImage', # Single Field (singleton) image value
    'MFImage', # Multiple Field (list) image values
    'SFInt32', # Single Field (singleton) 32-bit integer
    'MFInt32', # Multiple Field (list) 32-bit integer
    'SFNode', # Single Field (singleton) node
    'MFNode', # Multiple Field (list) nodes
    'SFRotation', # Single Field (singleton) rotation value using 3-tuple axis, radian angle
    'MFRotation', # Multiple Field (list) rotation values using 3-tuple axis, radian angle
    'SFString', # Single Field (singleton) string value
    'MFString', # Multiple Field (list) SFString array
    'SFTime', # Single Field (singleton) time value in seconds
    'MFTime', # Multiple Field (list) time array in seconds
    'SFVec2d', # Single Field (singleton) 2-tuple double-precision float vector
    'MFVec2d', # Multiple Field (list) 2-tuple double-precision float vectors
    'SFVec2f', # Single Field (singleton) 2-tuple single-precision float vector
    'MFVec2f', # Multiple Field (list) 2-tuple single-precision float vectors
    'SFVec3d', # Single Field (singleton) 3-tuple double-precision float vector
    'MFVec3d', # Multiple Field (list) 3-tuple double-precision float vectors
    'SFVec3f', # Single Field (singleton) 3-tuple single-precision float vector
    'MFVec3f', # Multiple Field (list) 3-tuple single-precision float vectors
    'SFVec4d', # Single Field (singleton) 4-tuple double-precision float vector
    'MFVec4d', # Multiple Field (list) 4-tuple double-precision float vectors
    'SFVec4f', # Single Field (singleton) 4-tuple single-precision float vector
    'MFVec4f', # Multiple Field (list) 4-tuple single-precision float vectors
    'SFMatrix3d', # Single Field (singleton) 3×3 matrix of double-precision floating point numbers
    'MFMatrix3d', # Multiple Field (list) 3×3 matrices of double-precision floating point numbers
    'SFMatrix3f', # Single Field (singleton) 3×3 matrix of single-precision floating point numbers
    'MFMatrix3f', # Multiple Field (list) 3×3 matrices of double-precision floating point numbers
    'SFMatrix4d', # Single Field (singleton) 4×4 matrix of double-precision floating point numbers
    'MFMatrix4d', # Multiple Field (list) 4×4 matric3w of double-precision floating point numbers
    'SFMatrix4f', # Single Field (singleton) 4×4 matrix of single-precision floating point numbers
    'MFMatrix4f' # Multiple Field (list) 4×4 matrices of single-precision floating point numbers
)
def assertValidFieldType(fieldName, value):
    """
    Utility function to assert type validity of fieldTypeChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if  isinstance(value,SFString) or isinstance(value,str):
        if str(value) in FIELDTYPECHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + value + ' does not match allowed enumerations in FIELDTYPECHOICES=' + str(FIELDTYPECHOICES))
    if  isinstance(value,MFString):
        if MFString(value).XML() in FIELDTYPECHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + MFString(value).XML() + ' does not match allowed enumerations in FIELDTYPECHOICES=' + str(FIELDTYPECHOICES))

FOGTYPECHOICES = (
    # strict set of allowed values follow, no other values are valid
    'LINEAR', # linear blending as a function of distance
    'EXPONENTIAL' # exponential blending as a function of distance
)
def assertValidFogType(fieldName, value):
    """
    Utility function to assert type validity of fogTypeChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if  isinstance(value,SFString) or isinstance(value,str):
        if str(value) in FOGTYPECHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + value + ' does not match allowed enumerations in FOGTYPECHOICES=' + str(FOGTYPECHOICES))
    if  isinstance(value,MFString):
        if MFString(value).XML() in FOGTYPECHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + MFString(value).XML() + ' does not match allowed enumerations in FOGTYPECHOICES=' + str(FOGTYPECHOICES))

FONTFAMILYVALUES = (
    # specification-defined values follow, other values are also allowed
    '"SANS"', # default font family for sans-serif font such as Helvetica
    '"SERIF"', # default font family for serif font such as Times-Roman
    '"TYPEWRITER"' # default font family for a fixed-pitch font such as Courier
)

FONTSTYLECHOICES = (
    # strict set of allowed values follow, no other values are valid
    'PLAIN', # default plain type
    'BOLD', # boldface type
    'ITALIC', # italic type
    'BOLDITALIC' # bold and italic type
)
def assertValidFontStyle(fieldName, value):
    """
    Utility function to assert type validity of fontStyleChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if  isinstance(value,SFString) or isinstance(value,str):
        if str(value) in FONTSTYLECHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + value + ' does not match allowed enumerations in FONTSTYLECHOICES=' + str(FONTSTYLECHOICES))
    if  isinstance(value,MFString):
        if MFString(value).XML() in FONTSTYLECHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + MFString(value).XML() + ' does not match allowed enumerations in FONTSTYLECHOICES=' + str(FONTSTYLECHOICES))

FORCEOUTPUTVALUES = (
    # specification-defined values follow, other values are also allowed
    '"ALL"', # all forceOutput fields computed
    '"NONE"' # no forceOutput fields computed
)

GENERATEDCUBEMAPTEXTUREUPDATECHOICES = (
    # strict set of allowed values follow, no other values are valid
    'NONE', # no further texture updates are rendered
    'NEXT_FRAME_ONLY', # render texture once at end of frame
    'ALWAYS' # texture to be rendered every frame
)
def assertValidGeneratedCubeMapTextureUpdate(fieldName, value):
    """
    Utility function to assert type validity of generatedCubeMapTextureUpdateChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if  isinstance(value,SFString) or isinstance(value,str):
        if str(value) in GENERATEDCUBEMAPTEXTUREUPDATECHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + value + ' does not match allowed enumerations in GENERATEDCUBEMAPTEXTUREUPDATECHOICES=' + str(GENERATEDCUBEMAPTEXTUREUPDATECHOICES))
    if  isinstance(value,MFString):
        if MFString(value).XML() in GENERATEDCUBEMAPTEXTUREUPDATECHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + MFString(value).XML() + ' does not match allowed enumerations in GENERATEDCUBEMAPTEXTUREUPDATECHOICES=' + str(GENERATEDCUBEMAPTEXTUREUPDATECHOICES))

GEOMETADATAKEYVALUES = (
    # specification-defined values follow, other values are also allowed
    'title',
    'description',
    'coordinateSystem',
    'horizontalDatum',
    'verticalDatum',
    'ellipsoid',
    'extent',
    'resolution',
    'originator',
    'copyright',
    'date',
    'metadataFormat',
    'dataUrl',
    'dataFormat'
)

GEOSYSTEMEARTHELLIPSOIDVALUES = (
    # specification-defined values follow, other values are also allowed
    'AM', # Modified Airy
    'AN', # Australian National
    'BN', # Bessel 1841 (Namibia)
    'BR', # Bessel 1841 (Ethiopia Indonesia ...)
    'CC', # Clarke 1866
    'CD', # Clarke 1880
    'EA', # Everest (India 1830)
    'EB', # Everest (Sabah & Sarawak)
    'EC', # Everest (India 1956)
    'ED', # Everest (W. Malaysia 1969)
    'EE', # Everest (W. Malaysia & Singapore 1948)
    'EF', # Everest (Pakistan)
    'FA', # Modified Fischer 1960
    'HE', # Helmert 1906
    'HO', # Hough 1960
    'ID', # Indonesia 1974
    'IN', # International 1924
    'KA', # Krassovsky 1940
    'RF', # Geodetic Reference System 1980 (GRS 80)
    'SA', # South American 1969
    'WD', # WGS 72
    'WE', # WGS 84
    'WGS84', # WGS84 geoid
    'Zn', # Zone number (1..60) (only used with UTM)
    'S' # Southern hemisphere (only used with UTM)
)

GEOSYSTEMSPATIALREFERENCEFRAMEVALUES = (
    # specification-defined values follow, other values are also allowed
    'GD', # Geodetic spatial reference frame (latitude/longitude), to be followed by ellipsoid
    'UTM', # Universal Transverse Mercator. One further required argument must be supplied for UTM in order to specify the zone number (1..60) with optional suffix of "S" may be appended in order to specify that the coordinates are in the southern hemisphere. Optional arguments can follow.
    'GC', # Earth-fixed Geocentric with respect to the WGS84 ellipsoid. No additional arguments are supported.
    'GDC', # Synonymous to GD, but may be subject to future deprecation
    'GCC' # Synonymous to GC, but may be subject to future deprecation
)

HANIMFEATUREPOINTNAMEVALUES = (
    # specification-defined values follow, other values are also allowed
    'skull_vertex', # CAESAR 2003 skull_vertex matches ISO 7250-1 part 5.22 Vertex (top of head). No corresponding landmark provided in CAESAR 2018.
    'glabella', # glabella is between the eyebrows and above the nose
    'sellion', # osseocartilaginous junction of the nasal dorsum
    'l_infraorbitale', # Left Infraorbitale foramen is opening in maxillary bone of skull located below the infraorbital margin of the orbit.
    'l_tragion', # notch just above the tragus of the ear
    'l_gonion', # Left Gonion is midpoint of mandibular angle of the jaw.
    'r_infraorbitale', # Right Infraorbitale foramen is opening in maxillary bone of skull located below the infraorbital margin of the orbit.
    'r_tragion', # notch just above the tragus of the ear
    'r_gonion', # Right Gonion is midpoint of the mandibular angle of the jaw.
    'supramenton', # center point above tip of chin
    'cervicale',
    'adams_apple',
    'suprasternale', # Suprasternale
    'substernale',
    'l_clavicle',
    'l_acromion',
    'l_axilla_proximal', # Left Axilla Proximal (Anterior)
    'l_axilla_distal', # Left Axilla Distal (Posterior)
    'l_axilla_posterior_folds',
    'r_clavicle',
    'r_acromion',
    'r_axilla_proximal', # Right Axilla Proximal (Anterior)
    'r_axilla_distal', # Right Axilla Distal (Posterior)
    'r_axilla_posterior_folds', # Right Posterior Axillary Folds
    'spine_1_middle_back',
    'spine_2_lower_back',
    'waist_preferred_anterior',
    'waist_preferred_posterior',
    'l_rib10',
    'l_thelion',
    'r_rib10',
    'r_thelion',
    'l_asis',
    'l_iliocristale',
    'l_psis',
    'r_asis',
    'r_iliocristale',
    'r_psis',
    'crotch',
    'l_femoral_lateral_epicondyle',
    'l_femoral_medial_epicondyle',
    'l_suprapatella',
    'l_trochanterion',
    'r_femoral_lateral_epicondyle',
    'r_femoral_medial_epicondyle',
    'r_suprapatella',
    'r_trochanterion',
    'l_tibiale',
    'l_medial_malleolus',
    'l_lateral_malleolus',
    'l_sphyrion',
    'r_tibiale',
    'r_medial_malleolus',
    'r_lateral_malleolus',
    'r_sphyrion',
    'l_metatarsal_phalanx_1',
    'l_metatarsal_phalanx_5',
    'l_dactylion',
    'l_calcaneus_posterior',
    'r_metatarsal_phalanx_1',
    'r_metatarsal_phalanx_5',
    'r_dactylion',
    'r_calcaneus_posterior',
    'l_humeral_lateral_epicondyle',
    'l_humeral_medial_epicondyle',
    'l_olecranon',
    'r_humeral_lateral_epicondyle',
    'r_humeral_medial_epicondyle',
    'r_olecranon',
    'l_radiale',
    'l_ulnar_styloid',
    'l_radial_styloid',
    'r_radiale',
    'r_ulnar_styloid',
    'r_radial_styloid',
    'l_metacarpal_phalanx_2',
    'l_metacarpal_phalanx_3',
    'l_metacarpal_phalanx_5',
    'r_metacarpal_phalanx_2',
    'r_metacarpal_phalanx_3',
    'r_metacarpal_phalanx_5',
    'nuchale',
    'l_neck_base',
    'r_neck_base',
    'navel',
    'l_ectocanthus',
    'r_ectocanthus',
    'menton',
    'mesosternale',
    'opisthocranion',
    'l_knee_crease',
    'r_knee_crease',
    'rear_center_midsagittal_plane',
    'buttocks_standing_wall_contact_point',
    'l_chest_midsagittal_plane',
    'r_chest_midsagittal_plane',
    'l_bideltoid',
    'r_bideltoid',
    'l_carpal_distal_phalanx_1',
    'l_carpal_distal_phalanx_2',
    'l_carpal_distal_phalanx_3',
    'l_carpal_distal_phalanx_4',
    'l_carpal_distal_phalanx_5',
    'r_carpal_distal_phalanx_1',
    'r_carpal_distal_phalanx_2',
    'r_carpal_distal_phalanx_3',
    'r_carpal_distal_phalanx_4',
    'r_carpal_distal_phalanx_5',
    'l_tarsal_distal_phalanx_1',
    'l_tarsal_distal_phalanx_2',
    'l_tarsal_distal_phalanx_3',
    'l_tarsal_distal_phalanx_4',
    'l_tarsal_distal_phalanx_5',
    'r_tarsal_distal_phalanx_1',
    'r_tarsal_distal_phalanx_2',
    'r_tarsal_distal_phalanx_3',
    'r_tarsal_distal_phalanx_4',
    'r_tarsal_distal_phalanx_5'
)

HANIMHUMANOIDINFOKEYVALUES = (
    # specification-defined values follow, other values are also allowed
    'authorName',
    'authorEmail',
    'copyright',
    'creationDate',
    'usageRestrictions',
    'humanoidVersion',
    'age',
    'gender',
    'height',
    'weight'
)

HANIMJOINTNAMEVALUES = (
    # specification-defined values follow, other values are also allowed
    'humanoid_root',
    'sacroiliac',
    'l_hip',
    'l_knee',
    'l_talocrural',
    'l_talocalcaneonavicular',
    'l_cuneonavicular_1',
    'l_tarsometatarsal_1',
    'l_metatarsophalangeal_1',
    'l_tarsal_interphalangeal_1',
    'l_cuneonavicular_2',
    'l_tarsometatarsal_2',
    'l_metatarsophalangeal_2',
    'l_tarsal_proximal_interphalangeal_2',
    'l_tarsal_distal_interphalangeal_2',
    'l_cuneonavicular_3',
    'l_tarsometatarsal_3',
    'l_metatarsophalangeal_3',
    'l_tarsal_proximal_interphalangeal_3',
    'l_tarsal_distal_interphalangeal_3',
    'l_calcaneocuboid',
    'l_transversetarsal',
    'l_tarsometatarsal_4',
    'l_metatarsophalangeal_4',
    'l_tarsal_proximal_interphalangeal_4',
    'l_tarsal_distal_interphalangeal_4',
    'l_tarsometatarsal_5',
    'l_metatarsophalangeal_5',
    'l_tarsal_proximal_interphalangeal_5',
    'l_tarsal_distal_interphalangeal_5',
    'r_hip',
    'r_knee',
    'r_talocrural',
    'r_talocalcaneonavicular',
    'r_cuneonavicular_1',
    'r_tarsometatarsal_1',
    'r_metatarsophalangeal_1',
    'r_tarsal_interphalangeal_1',
    'r_cuneonavicular_2',
    'r_tarsometatarsal_2',
    'r_metatarsophalangeal_2',
    'r_tarsal_proximal_interphalangeal_2',
    'r_tarsal_distal_interphalangeal_2',
    'r_cuneonavicular_3',
    'r_tarsometatarsal_3',
    'r_metatarsophalangeal_3',
    'r_tarsal_proximal_interphalangeal_3',
    'r_tarsal_distal_interphalangeal_3',
    'r_calcaneocuboid',
    'r_transversetarsal',
    'r_tarsometatarsal_4',
    'r_metatarsophalangeal_4',
    'r_tarsal_proximal_interphalangeal_4',
    'r_tarsal_distal_interphalangeal_4',
    'r_tarsometatarsal_5',
    'r_metatarsophalangeal_5',
    'r_tarsal_proximal_interphalangeal_5',
    'r_tarsal_distal_interphalangeal_5',
    'vl5',
    'vl4',
    'vl3',
    'vl2',
    'vl1',
    'vt12',
    'vt11',
    'vt10',
    'vt9',
    'vt8',
    'vt7',
    'vt6',
    'vt5',
    'vt4',
    'vt3',
    'vt2',
    'vt1',
    'vc7',
    'vc6',
    'vc5',
    'vc4',
    'vc3',
    'vc2',
    'vc1',
    'skullbase',
    'l_eyelid_joint',
    'r_eyelid_joint',
    'l_eyeball_joint',
    'r_eyeball_joint',
    'l_eyebrow_joint',
    'r_eyebrow_joint',
    'temporomandibular',
    'l_sternoclavicular',
    'l_acromioclavicular',
    'l_shoulder',
    'l_elbow',
    'l_radiocarpal',
    'l_midcarpal_1',
    'l_carpometacarpal_1',
    'l_metacarpophalangeal_1',
    'l_carpal_interphalangeal_1',
    'l_midcarpal_2',
    'l_carpometacarpal_2',
    'l_metacarpophalangeal_2',
    'l_carpal_proximal_interphalangeal_2',
    'l_carpal_distal_interphalangeal_2',
    'l_midcarpal_3',
    'l_carpometacarpal_3',
    'l_metacarpophalangeal_3',
    'l_carpal_proximal_interphalangeal_3',
    'l_carpal_distal_interphalangeal_3',
    'l_midcarpal_4_5',
    'l_carpometacarpal_4',
    'l_metacarpophalangeal_4',
    'l_carpal_proximal_interphalangeal_4',
    'l_carpal_distal_interphalangeal_4',
    'l_carpometacarpal_5',
    'l_metacarpophalangeal_5',
    'l_carpal_proximal_interphalangeal_5',
    'l_carpal_distal_interphalangeal_5',
    'r_sternoclavicular',
    'r_acromioclavicular',
    'r_shoulder',
    'r_elbow',
    'r_radiocarpal',
    'r_midcarpal_1',
    'r_carpometacarpal_1',
    'r_metacarpophalangeal_1',
    'r_carpal_interphalangeal_1',
    'r_midcarpal_2',
    'r_carpometacarpal_2',
    'r_metacarpophalangeal_2',
    'r_carpal_proximal_interphalangeal_2',
    'r_carpal_distal_interphalangeal_2',
    'r_midcarpal_3',
    'r_carpometacarpal_3',
    'r_metacarpophalangeal_3',
    'r_carpal_proximal_interphalangeal_3',
    'r_carpal_distal_interphalangeal_3',
    'r_midcarpal_4_5',
    'r_carpometacarpal_4',
    'r_metacarpophalangeal_4',
    'r_carpal_proximal_interphalangeal_4',
    'r_carpal_distal_interphalangeal_4',
    'r_carpometacarpal_5',
    'r_metacarpophalangeal_5',
    'r_carpal_proximal_interphalangeal_5',
    'r_carpal_distal_interphalangeal_5'
)

HANIMSEGMENTNAMEVALUES = (
    # specification-defined values follow, other values are also allowed
    'sacrum',
    'pelvis',
    'l_thigh',
    'l_calf',
    'l_talus',
    'l_navicular',
    'l_cuneiform_1',
    'l_metatarsal_1',
    'l_tarsal_proximal_phalanx_1',
    'l_tarsal_distal_phalanx_1',
    'l_cuneiform_2',
    'l_metatarsal_2',
    'l_tarsal_proximal_phalanx_2',
    'l_tarsal_middle_phalanx_2',
    'l_tarsal_distal_phalanx_2',
    'l_cuneiform_3',
    'l_metatarsal_3',
    'l_tarsal_proximal_phalanx_3',
    'l_tarsal_middle_phalanx_3',
    'l_tarsal_distal_phalanx_3',
    'l_calcaneus',
    'l_cuboid',
    'l_metatarsal_4',
    'l_tarsal_proximal_phalanx_4',
    'l_tarsal_middle_phalanx_4',
    'l_tarsal_distal_phalanx_4',
    'l_metatarsal_5',
    'l_tarsal_proximal_phalanx_5',
    'l_tarsal_middle_phalanx_5',
    'l_tarsal_distal_phalanx_5',
    'r_thigh',
    'r_calf',
    'r_talus',
    'r_navicular',
    'r_cuneiform_1',
    'r_metatarsal_1',
    'r_tarsal_proximal_phalanx_1',
    'r_tarsal_distal_phalanx_1',
    'r_cuneiform_2',
    'r_metatarsal_2',
    'r_tarsal_proximal_phalanx_2',
    'r_tarsal_middle_phalanx_2',
    'r_tarsal_distal_phalanx_2',
    'r_cuneiform_3',
    'r_metatarsal_3',
    'r_tarsal_proximal_phalanx_3',
    'r_tarsal_middle_phalanx_3',
    'r_tarsal_distal_phalanx_3',
    'r_calcaneus',
    'r_cuboid',
    'r_metatarsal_4',
    'r_tarsal_proximal_phalanx_4',
    'r_tarsal_middle_phalanx_4',
    'r_tarsal_distal_phalanx_4',
    'r_metatarsal_5',
    'r_tarsal_proximal_phalanx_5',
    'r_tarsal_middle_phalanx_5',
    'r_tarsal_distal_phalanx_5',
    'l5',
    'l4',
    'l3',
    'l2',
    'l1',
    't12',
    't11',
    't10',
    't9',
    't8',
    't7',
    't6',
    't5',
    't4',
    't3',
    't2',
    't1',
    'c7',
    'c6',
    'c5',
    'c4',
    'c3',
    'c2',
    'c1',
    'skull',
    'l_eyelid',
    'r_eyelid',
    'l_eyeball',
    'r_eyeball',
    'l_eyebrow',
    'r_eyebrow',
    'jaw',
    'l_clavicle',
    'l_scapula',
    'l_upperarm',
    'l_forearm',
    'l_carpal',
    'l_trapezium',
    'l_metacarpal_1',
    'l_carpal_proximal_phalanx_1',
    'l_carpal_distal_phalanx_1',
    'l_trapezoid',
    'l_metacarpal_2',
    'l_carpal_proximal_phalanx_2',
    'l_carpal_middle_phalanx_2',
    'l_carpal_distal_phalanx_2',
    'l_capitate',
    'l_metacarpal_3',
    'l_carpal_proximal_phalanx_3',
    'l_carpal_middle_phalanx_3',
    'l_carpal_distal_phalanx_3',
    'l_hamate',
    'l_metacarpal_4',
    'l_carpal_proximal_phalanx_4',
    'l_carpal_middle_phalanx_4',
    'l_carpal_distal_phalanx_4',
    'l_metacarpal_5',
    'l_carpal_proximal_phalanx_5',
    'l_carpal_middle_phalanx_5',
    'l_carpal_distal_phalanx_5',
    'r_clavicle',
    'r_scapula',
    'r_upperarm',
    'r_forearm',
    'r_carpal',
    'r_trapezium',
    'r_metacarpal_1',
    'r_carpal_proximal_phalanx_1',
    'r_carpal_distal_phalanx_1',
    'r_trapezoid',
    'r_metacarpal_2',
    'r_carpal_proximal_phalanx_2',
    'r_carpal_middle_phalanx_2',
    'r_carpal_distal_phalanx_2',
    'r_capitate',
    'r_metacarpal_3',
    'r_carpal_proximal_phalanx_3',
    'r_carpal_middle_phalanx_3',
    'r_carpal_distal_phalanx_3',
    'r_hamate',
    'r_metacarpal_4',
    'r_carpal_proximal_phalanx_4',
    'r_carpal_middle_phalanx_4',
    'r_carpal_distal_phalanx_4',
    'r_metacarpal_5',
    'r_carpal_proximal_phalanx_5',
    'r_carpal_middle_phalanx_5',
    'r_carpal_distal_phalanx_5'
)

HANIMVERSIONCHOICES = (
    # strict set of allowed values follow, no other values are valid
    '1.0', # International standard HAnim 19774 version 1 approved by ISO in 2006. Note that HAnim version 2.0 has more capabilties, while version 1.0 includes several small incompatibilities. Since it was never formally approved, version 1.1 is no longer an allowed value.
    '2.0' # Revised standard HAnim 19774 version 2 parts 1 and 2 were approved by ISO in Novemer 2019, published by Web3D Consortium May 2020.
)
def assertValidHanimVersion(fieldName, value):
    """
    Utility function to assert type validity of hanimVersionChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if  isinstance(value,SFString) or isinstance(value,str):
        if str(value) in HANIMVERSIONCHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + value + ' does not match allowed enumerations in HANIMVERSIONCHOICES=' + str(HANIMVERSIONCHOICES))
    if  isinstance(value,MFString):
        if MFString(value).XML() in HANIMVERSIONCHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + MFString(value).XML() + ' does not match allowed enumerations in HANIMVERSIONCHOICES=' + str(HANIMVERSIONCHOICES))

INTERSECTIONTYPEVALUES = (
    # specification-defined values follow, other values are also allowed
    'BOUNDS', # TODO undefined in X3D specification
    'GEOMETRY' # TODO undefined in X3D specification
)

JUSTIFYCHOICES = (
    # note these MFString values use XML syntax
    # strict set of allowed values follow, no other values are valid
    '"MIDDLE"',
    '"MIDDLE" "BEGIN"',
    '"MIDDLE" "END"',
    '"MIDDLE" "FIRST"',
    '"MIDDLE" "MIDDLE"',
    '"BEGIN"',
    '"BEGIN" "BEGIN"',
    '"BEGIN" "END"',
    '"BEGIN" "FIRST"',
    '"BEGIN" "MIDDLE"',
    '"END"',
    '"END" "BEGIN"',
    '"END" "END"',
    '"END" "FIRST"',
    '"END" "MIDDLE"',
    '"FIRST"',
    '"FIRST" "BEGIN"',
    '"FIRST" "END"',
    '"FIRST" "FIRST"',
    '"FIRST" "MIDDLE"'
)
def assertValidJustify(fieldName, value):
    """
    Utility function to assert type validity of justifyChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if  isinstance(value,SFString) or isinstance(value,str):
        if str(value) in JUSTIFYCHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + value + ' does not match allowed enumerations in JUSTIFYCHOICES=' + str(JUSTIFYCHOICES))
    if  isinstance(value,MFString):
        if MFString(value).XML() in JUSTIFYCHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + MFString(value).XML() + ' does not match allowed enumerations in JUSTIFYCHOICES=' + str(JUSTIFYCHOICES))

LAYOUTALIGNCHOICES = (
    # note these MFString values use XML syntax
    # strict set of allowed values follow, no other values are valid
    '"LEFT" "BOTTOM"',
    '"LEFT" "CENTER"',
    '"LEFT" "TOP"',
    '"CENTER" "BOTTOM"',
    '"CENTER" "CENTER"',
    '"CENTER" "TOP"',
    '"RIGHT" "BOTTOM"',
    '"RIGHT" "CENTER"',
    '"RIGHT" "TOP"'
)
def assertValidLayoutAlign(fieldName, value):
    """
    Utility function to assert type validity of layoutAlignChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if  isinstance(value,SFString) or isinstance(value,str):
        if str(value) in LAYOUTALIGNCHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + value + ' does not match allowed enumerations in LAYOUTALIGNCHOICES=' + str(LAYOUTALIGNCHOICES))
    if  isinstance(value,MFString):
        if MFString(value).XML() in LAYOUTALIGNCHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + MFString(value).XML() + ' does not match allowed enumerations in LAYOUTALIGNCHOICES=' + str(LAYOUTALIGNCHOICES))

LAYOUTSCALEMODECHOICES = (
    # note these MFString values use XML syntax
    # strict set of allowed values follow, no other values are valid
    '"NONE" "NONE"',
    '"NONE" "FRACTION"',
    '"NONE" "STRETCH"',
    '"NONE" "PIXEL"',
    '"FRACTION" "NONE"',
    '"FRACTION" "FRACTION"',
    '"FRACTION" "STRETCH"',
    '"FRACTION" "PIXEL"',
    '"STRETCH" "NONE"',
    '"STRETCH" "FRACTION"',
    '"STRETCH" "STRETCH"',
    '"STRETCH" "PIXEL"',
    '"PIXEL" "NONE"',
    '"PIXEL" "FRACTION"',
    '"PIXEL" "STRETCH"',
    '"PIXEL" "PIXEL"'
)
def assertValidLayoutScaleMode(fieldName, value):
    """
    Utility function to assert type validity of layoutScaleModeChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if  isinstance(value,SFString) or isinstance(value,str):
        if str(value) in LAYOUTSCALEMODECHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + value + ' does not match allowed enumerations in LAYOUTSCALEMODECHOICES=' + str(LAYOUTSCALEMODECHOICES))
    if  isinstance(value,MFString):
        if MFString(value).XML() in LAYOUTSCALEMODECHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + MFString(value).XML() + ' does not match allowed enumerations in LAYOUTSCALEMODECHOICES=' + str(LAYOUTSCALEMODECHOICES))

LAYOUTUNITSCHOICES = (
    # note these MFString values use XML syntax
    # strict set of allowed values follow, no other values are valid
    '"WORLD" "WORLD"',
    '"WORLD" "FRACTION"',
    '"WORLD" "PIXEL"',
    '"FRACTION" "WORLD"',
    '"FRACTION" "FRACTION"',
    '"FRACTION" "PIXEL"',
    '"PIXEL" "WORLD"',
    '"PIXEL" "FRACTION"',
    '"PIXEL" "PIXEL"'
)
def assertValidLayoutUnits(fieldName, value):
    """
    Utility function to assert type validity of layoutUnitsChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if  isinstance(value,SFString) or isinstance(value,str):
        if str(value) in LAYOUTUNITSCHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + value + ' does not match allowed enumerations in LAYOUTUNITSCHOICES=' + str(LAYOUTUNITSCHOICES))
    if  isinstance(value,MFString):
        if MFString(value).XML() in LAYOUTUNITSCHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + MFString(value).XML() + ' does not match allowed enumerations in LAYOUTUNITSCHOICES=' + str(LAYOUTUNITSCHOICES))

METADIRECTIONCHOICES = (
    # strict set of allowed values follow, no other values are valid
    'rtl', # right-to-left
    'ltr' # left-to-right
)
def assertValidMetaDirection(fieldName, value):
    """
    Utility function to assert type validity of metaDirectionChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if  isinstance(value,SFString) or isinstance(value,str):
        if str(value) in METADIRECTIONCHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + value + ' does not match allowed enumerations in METADIRECTIONCHOICES=' + str(METADIRECTIONCHOICES))
    if  isinstance(value,MFString):
        if MFString(value).XML() in METADIRECTIONCHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + MFString(value).XML() + ' does not match allowed enumerations in METADIRECTIONCHOICES=' + str(METADIRECTIONCHOICES))

METANAMEVALUES = (
    # specification-defined values follow, other values are also allowed
    'accessRights', # permission required to access resource or security status
    'author', # name of individual author
    'contributor', # name of individual contributing to this resource
    'created', # date of initial version
    'creator', # name of original author
    'description', # summary overview describing this resource
    'disclaimer', # statement of denial or disavowal regarding potential claims or responsiblity
    'drawing', # name or reference link to a supporting drawing or sketch file
    'error', # information about an error (or known problem) that can prevent proper operation
    'generator', # authoring tool or translation tool
    'hint', # user hint about resource features or operation
    'identifier', # url address or unique Uniform Resource Identifier (URI) for resource
    'Image', # name or reference link to supporting image file
    'info', # additional info of interest
    'information', # additional information of interest
    'isVersionOf', # Related resource of which the described resource is a version, edition, or adaptation.
    'keywords', # comma-separated tokens, each of which is a keyword of interest
    'license', # content or software license
    'mediator', # entity that mediates access to resource and for whom resource is intended or useful
    'modified', # date of modified version
    'movie', # name or reference link to supporting movie file (note that Dublin Core term is MovingImage)
    'MovingImage', # name or reference link to supporting movie
    'original', # name or reference link to original file or resource
    'photo', # name or reference link to supporting photo file (note that Dublin Core term is Image)
    'photograph', # name or reference link to supporting photograph file (note that Dublin Core term is Image)
    'publisher', # entity responsible for making the resource available
    'reference', # name or reference link to supporting reference
    'requires', # prerequisites for operation or viewing
    'rights', # intellectual property rights (IPR)
    'robots', # search engine and web-spider guidance value: noindex to block page indexing, nofollow to block following links
    'Sound', # name or reference link to supporting sound file
    'source', # related resource from which the described resource is derived
    'specificationSection', # title of relevant specification section
    'specificationUrl', # url for relevant specification section
    'subject', # search-index subject keywords, key phrases, or classification codes
    'Text', # resource consisting primarily of words for reading
    'title', # file name for this resource
    'TODO', # action item "to do" that still needs to be performed
    'translator', # name of person performing translation from another format or language
    'translated', # date of translation from another format or language
    'version', # current version number or ID of this resource
    'warning' # warning information about a known problem that impedes proper operation
)

MULTITEXTUREFUNCTIONVALUES = (
    # specification-defined values follow, other values are also allowed
    '"COMPLEMENT"', # Invert argument x as (1 - x)
    '"ALPHAREPLICATE"', # Replicate alpha information to all color channels before operation completes.
    '""' # No function is applied - empty SFString is allowed value within MFString array
)

MULTITEXTUREMODEVALUES = (
    # specification-defined values follow, other values are also allowed
    '"ADD"',
    '"ADDSIGNED"',
    '"ADDSIGNED2X"',
    '"ADDSMOOTH"',
    '"BLENDCURRENTALPHA"',
    '"BLENDDIFFUSEALPHA"',
    '"BLENDFACTORALPHA"',
    '"BLENDTEXTUREALPHA"',
    '"DOTPRODUCT3"',
    '"MODULATE"',
    '"MODULATE2X"',
    '"MODULATE4X"',
    '"MODULATEALPHA_ADDCOLOR"',
    '"MODULATEINVALPHA_ADDCOLOR"',
    '"MODULATEINVCOLOR_ADDALPHA"',
    '"OFF"',
    '"REPLACE"',
    '"SELECTARG1"',
    '"SELECTARG2"',
    '"SUBTRACT"'
)

MULTITEXTURESOURCEVALUES = (
    # specification-defined values follow, other values are also allowed
    '"DIFFUSE"',
    '"FACTOR"',
    '"SPECULAR"',
    '""'
)

NAVIGATIONTRANSITIONTYPEVALUES = (
    # specification-defined values follow, other values are also allowed
    '"TELEPORT"', # immediate transition
    '"LINEAR"', # transition may proceed directly through intervening objects
    '"ANIMATE"' # rowser-specific transition
)

NAVIGATIONTYPEVALUES = (
    # specification-defined values follow, other values are also allowed
    '"ANY"', # browser can offer any type for user to choose
    '"WALK"', # free navigation, avatar remains on ground, collision detection
    '"EXAMINE"', # view an individual object by rotating view about center
    '"FLY"', # free navigation, collision detection
    '"LOOKAT"', # navigate to particular object
    '"NONE"', # disables all navigation interfaces
    '"EXPLORE"' # consistent keystroke navigation for both geospatial and Cartesian modes
)

NETWORKMODECHOICES = (
    # strict set of allowed values follow, no other values are valid
    'standAlone', # ignore network but still respond to events in local scene
    'networkReader', # listen to network and read PDU packets at readInterval, act as remotely linked copy of entity
    'networkWriter' # send PDU packets to network at writeInterval, act as master entity
)
def assertValidNetworkMode(fieldName, value):
    """
    Utility function to assert type validity of networkModeChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if  isinstance(value,SFString) or isinstance(value,str):
        if str(value) in NETWORKMODECHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + value + ' does not match allowed enumerations in NETWORKMODECHOICES=' + str(NETWORKMODECHOICES))
    if  isinstance(value,MFString):
        if MFString(value).XML() in NETWORKMODECHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + MFString(value).XML() + ' does not match allowed enumerations in NETWORKMODECHOICES=' + str(NETWORKMODECHOICES))

PARTICLESYSTEMGEOMETRYTYPEVALUES = (
    # specification-defined values follow, other values are also allowed
    'LINE', # line is drawn along current velocity vector of particle
    'POINT', # point geometry is rendered at particle position
    'QUAD', # quad geometry is rendered at particle position facing direction traveled
    'SPRITE', # quad geometry is rendered at particle position facing screen
    'TRIANGLE', # pair of triangles creating quad geometry is rendered at particle position facing direction traveled
    'GEOMETRY' # geometry field is used for rendering each particle
)

PHASEFUNCTIONVALUES = (
    # specification-defined values follow, other values are also allowed
    'Henyey-Greenstein', # Henyey-Greenstein phase function for scattering model
    'NONE' # no scattering
)

PICKABLEOBJECTTYPEVALUES = (
    # specification-defined values follow, other values are also allowed
    '"ALL"', # each node is available for picking
    '"NONE"', # no node is available for picking
    '"TERRAIN"' # TERRAIN is an example value
)

PICKSENSORMATCHCRITERIONCHOICES = (
    # strict set of allowed values follow, no other values are valid
    'MATCH_ANY', # any match of objectType values is acceptable
    'MATCH_EVERY', # every objectType value in X3DPickSensorNode and X3DPickableObject shall match
    'MATCH_ONLY_ONE' # one and only one objectType value can match
)
def assertValidPickSensorMatchCriterion(fieldName, value):
    """
    Utility function to assert type validity of pickSensorMatchCriterionChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if  isinstance(value,SFString) or isinstance(value,str):
        if str(value) in PICKSENSORMATCHCRITERIONCHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + value + ' does not match allowed enumerations in PICKSENSORMATCHCRITERIONCHOICES=' + str(PICKSENSORMATCHCRITERIONCHOICES))
    if  isinstance(value,MFString):
        if MFString(value).XML() in PICKSENSORMATCHCRITERIONCHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + MFString(value).XML() + ' does not match allowed enumerations in PICKSENSORMATCHCRITERIONCHOICES=' + str(PICKSENSORMATCHCRITERIONCHOICES))

PICKSENSORSORTORDERVALUES = (
    # specification-defined values follow, other values are also allowed
    'ANY', # any single object that can satisfy picking conditions
    'CLOSEST', # return closest object by distance that satisfies conditions of this pick sensor
    'ALL', # every object that satisfies picking conditions for this pick sensor is returned
    'ALL_SORTED' # every object that satisfies picking conditions for this pick sensor is returned, in sorted order
)

PROFILENAMECHOICES = (
    # strict set of allowed values follow, no other values are valid
    'Core', # Core Profile includes no nodes and is provided as the basis for custom componentization. Allowed X3D statements for all profiles are: connect ExternProtoDeclare EXPORT field fieldValue IMPORT IS ProtoBody ProtoDeclare ProtoInterface ProtoInstance ROUTE X3D. Allowed X3D nodes for this profile are: MetadataBoolean MetadataDouble MetadataFloat MetadataInteger MetadataSet MetadataString.
    'Interchange', # Interchange Profile equals the minimum subset of nodes needed to display lightweight compelling content. Allowed X3D nodes for this profile are: Appearance Background Box Color ColorInterpolator ColorRGBA Cone Coordinate CoordinateInterpolator Cylinder DirectionalLight Group ImageTexture IndexedFaceSet IndexedLineSet IndexedTriangleFanSet IndexedTriangleSet IndexedTriangleStripSet LineSet Material MetadataBoolean MetadataDouble MetadataFloat MetadataInteger MetadataSet MetadataString MultiTexture MultiTextureCoordinate MultiTextureTransform NavigationInfo Normal NormalInterpolator OrientationInterpolator PixelTexture PointSet PositionInterpolator ScalarInterpolator Shape Sphere TextureCoordinate TextureCoordinateGenerator TextureTransform TimeSensor Transform TriangleFanSet TriangleSet TriangleStripSet Viewpoint WorldInfo.
    'CADInterchange', # CADInterchange Profile adds support for CADGeometry component nodes to Interchange Profile. Allowed X3D nodes for this profile are: Anchor Appearance CADAssembly CADFace CADLayer CADPart Billboard Collision Color ColorRGBA Coordinate DirectionalLight FragmentShader Group ImageTexture IndexedLineSet IndexedQuadSet IndexedTriangleFanSet IndexedTriangleSet IndexedTriangleStripSet Inline LineProperties LineSet LOD Material MetadataBoolean MetadataDouble MetadataFloat MetadataInteger MetadataSet MetadataString MultiShader MultiTexture MultiTextureCoordinate MultiTextureTransform NavigationInfo Normal PixelTexture PointSet QuadSet Shader ShaderAppearance Shape TextureCoordinate TextureCoordinateGenerator TextureTransform Transform TriangleFanSet TriangleSet TriangleStripSet Viewpoint VertexShader WorldInfo.
    'Interactive', # Interactive Profile adds interaction nodes (Anchor, KeySensor) to the minimum subset of nodes needed to display lightweight compelling content. Allowed X3D nodes for this profile are: Anchor Appearance Background BooleanFilter BooleanSequencer BooleanToggle BooleanTrigger Box Color ColorInterpolator ColorRGBA Cone Coordinate CoordinateInterpolator Cylinder CylinderSensor DirectionalLight ElevationGrid Group ImageTexture IndexedFaceSet IndexedLineSet IndexedTriangleFanSet IndexedTriangleSet IndexedTriangleStripSet Inline IntegerSequencer IntegerTrigger KeySensor LineSet Material MetadataBoolean MetadataDouble MetadataFloat MetadataInteger MetadataSet MetadataString MultiTexture MultiTextureCoordinate MultiTextureTransform NavigationInfo Normal NormalInterpolator OrientationInterpolator IndexedTriangleStripSet Inline IntegerSequencer IntegerTrigger KeySensor LineSet Material MetadataBoolean MetadataDouble MetadataFloat MetadataInteger MetadataSet MetadataString MultiTexture MultiTextureCoordinate MultiTextureTransform NavigationInfo Normal NormalInterpolator OrientationInterpolator PixelTexture PlaneSensor PointLight PointSet PositionInterpolator ProximitySensor ScalarInterpolator Shape Sphere SphereSensor SpotLight StringSensor Switch TextureCoordinate TextureCoordinateGenerator TextureTransform TimeSensor TimeTrigger TouchSensor Transform TriangleFanSet TriangleSet TriangleStripSet Viewpoint VisibilitySensor WorldInfo.
    'Immersive', # Immersive Profile equals all of the nodes in the VRML97 Specification, plus various X3D node additions including KeySensor, StringSensor and Scene. Allowed X3D nodes for this profile are: Anchor Appearance AudioClip Background Billboard BooleanFilter BooleanSequencer BooleanToggle BooleanTrigger Box Collision Color ColorInterpolator ColorRGBA Cone Coordinate CoordinateInterpolator Cylinder CylinderSensor DirectionalLight ElevationGrid Extrusion Fog FontStyle Group ImageTexture IndexedFaceSet IndexedLineSet IndexedTriangleFan IndexedTriangleSet IndexedTriangleStripSet Inline IntegerSequencer IntegerTrigger KeySensor LineProperties LineSet LoadSensor LOD Material MetadataBoolean MetadataDouble MetadataFloat MetadataInteger MetadataSet MetadataString MovieTexture MultiTexture MultiTextureCoordinate MultiTextureTransform NavigationInfo Normal NormalInterpolator OrientationInterpolator PixelTexture PlaneSensor PointLight PointSet Polyline2D Polypoint2D PositionInterpolator ProximitySensor Rectangle2D ScalarInterpolator Script Shape Sound Sphere SphereSensor SpotLight StringSensor Switch Text TextureCoordinate TextureCoordinateGenerator TextureTransform TimeSensor TimeTrigger TouchSensor TriangleFanSet TriangleSet TriangleSet2D TriangleStripSet Transform Viewpoint VisibilitySensor WorldInfo.
    'MedicalInterchange', # The MedicalInterchange profile adds support for VolumeRendering component to Interchange profile. Allowed X3D nodes for this profile are: Anchor Arc2D ArcClose2D Appearance Background Billboard BlendedVolumeStyle BooleanFilter BooleanSequencer BooleanToggle BooleanTrigger BoundaryEnhancementVolumeStyle Box CartoonVolumeStyle Circle2D ClipPlane Collision Color ColorInterpolator ColorRGBA ComposedVolumeStyle CompositeTexture3D Cone Coordinate CoordinateDouble CoordinateInterpolator Cylinder DirectionalLight Disk2D EdgeEnhancementVolumeStyle FillProperties FontStyle Group ImageTexture ImageTexture3D IndexedFaceSet IndexedLineSet IndexedTriangleFanSet IndexedTriangleSet IndexedTriangleStripSet Inline IntegerSequencer IntegerTrigger IsoSurfaceVolumeData LineProperties LineSet LOD Material MetadataBoolean MetadataDouble MetadataFloat MetadataInteger MetadataSet MetadataString MultiTexture MultiTextureCoordinate MultiTextureTransform NavigationInfo Normal NormalInterpolator OctTree OpacityMapVolumeStyle OrientationInterpolator OrthoViewpoint PixelTexture PixelTexture3D PointSet Polyline2D Polypoint2D PositionInterpolator ProjectionVolumeStyle Rectangle2D ScalarInterpolator SegmentedVolumeData ShadedVolumeStyle Shape SilhouetteEnhancementVolumeStyle Sphere StaticGroup Switch Text TextureCoordinate TextureCoordinate3D TextureCoordinate4D TextureCoordinateGenerator TextureMatrixTransform TextureProperties TextureTransform TextureTransform3D TimeSensor TimeTrigger ToneMappedVolumeStyle Transform TriangleFanSet TriangleSet TriangleStripSet Viewpoint ViewpointGroup VolumeData WorldInfo.
    'MPEG4Interactive', # MPEGInteractive Profile defines base interoperability with MPEG4 standards to a small subset of nodes needed to display lightweight compelling content. Allowed X3D nodes for this profile are: Anchor Appearance Background Box Color ColorInterpolator ColorRGBA Cone Coordinate CoordinateInterpolator Cylinder CylinderSensor DirectionalLight ElevationGrid Group ImageTexture IndexedFaceSet IndexedLineSet Inline LineSet Material MetadataBoolean MetadataDouble MetadataFloat MetadataInteger MetadataSet MetadataString NavigationInfo NormalInterpolator OrientationInterpolator PixelTexture PlaneSensor PointLight PointSet PositionInterpolator ProximitySensor ScalarInterpolator Shape Sphere SphereSensor SpotLight Switch TextureCoordinate TextureTransform TimeSensor TouchSensor Transform Viewpoint WorldInfo.
    'Full' # The Full Profile corresponds to all Immersive X3D nodes plus all approved/implemented extensions. All X3D nodes and statements are allowed in this profile.
)
def assertValidProfileName(fieldName, value):
    """
    Utility function to assert type validity of profileNameChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if  isinstance(value,SFString) or isinstance(value,str):
        if str(value) in PROFILENAMECHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + value + ' does not match allowed enumerations in PROFILENAMECHOICES=' + str(PROFILENAMECHOICES))
    if  isinstance(value,MFString):
        if MFString(value).XML() in PROFILENAMECHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + MFString(value).XML() + ' does not match allowed enumerations in PROFILENAMECHOICES=' + str(PROFILENAMECHOICES))

PROJECTIONVOLUMESTYLETYPECHOICES = (
    # strict set of allowed values follow, no other values are valid
    'MAX', # Maximum Intensity Projection (MIP) or Least MIP (LMIP) algorithm is used to generate output color
    'MIN', # Minimum Intensity Projection algorithm is used to generate output color
    'AVERAGE' # All voxels along ray are averaged to generate output color
)
def assertValidProjectionVolumeStyleType(fieldName, value):
    """
    Utility function to assert type validity of projectionVolumeStyleTypeChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if  isinstance(value,SFString) or isinstance(value,str):
        if str(value) in PROJECTIONVOLUMESTYLETYPECHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + value + ' does not match allowed enumerations in PROJECTIONVOLUMESTYLETYPECHOICES=' + str(PROJECTIONVOLUMESTYLETYPECHOICES))
    if  isinstance(value,MFString):
        if MFString(value).XML() in PROJECTIONVOLUMESTYLETYPECHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + MFString(value).XML() + ' does not match allowed enumerations in PROJECTIONVOLUMESTYLETYPECHOICES=' + str(PROJECTIONVOLUMESTYLETYPECHOICES))

SHADERLANGUAGEVALUES = (
    # specification-defined values follow, other values are also allowed
    'Cg', # nVidia Cg shading language
    'GLSL', # OpenGL shading language (GLSL)
    'HLSL' # Microsoft High Level Shading Language (HLSL)
)

SHADERPARTTYPEVALUES = (
    # specification-defined values follow, other values are also allowed
    'VERTEX', # vertex shader
    'FRAGMENT' # fragment shader
)

TEXTUREBOUNDARYMODECHOICES = (
    # strict set of allowed values follow, no other values are valid
    'CLAMP', # Clamp texture coordinates to range [0,1]
    'CLAMP_TO_EDGE', # Clamp texture coordinates such that a border texel is never sampled
    'CLAMP_TO_BOUNDARY', # Clamp texture coordinates such that texture samples are border texels for fragments
    'MIRRORED_REPEAT', # Texture coordinates are mirrored and then clamped as in CLAMP_TO_EDGE
    'REPEAT' # Repeat a texture across the fragment
)
def assertValidTextureBoundaryMode(fieldName, value):
    """
    Utility function to assert type validity of textureBoundaryModeChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if  isinstance(value,SFString) or isinstance(value,str):
        if str(value) in TEXTUREBOUNDARYMODECHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + value + ' does not match allowed enumerations in TEXTUREBOUNDARYMODECHOICES=' + str(TEXTUREBOUNDARYMODECHOICES))
    if  isinstance(value,MFString):
        if MFString(value).XML() in TEXTUREBOUNDARYMODECHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + MFString(value).XML() + ' does not match allowed enumerations in TEXTUREBOUNDARYMODECHOICES=' + str(TEXTUREBOUNDARYMODECHOICES))

TEXTURECOMPRESSIONMODECHOICES = (
    # strict set of allowed values follow, no other values are valid
    'DEFAULT', # browser-specified default compression mode
    'FASTEST', # fastest method available
    'HIGH', # greatest amount of compression
    'LOW', # least amount of compression
    'MEDIUM', # moderate amount of compressions
    'NICEST' # highest quality method available
)
def assertValidTextureCompressionMode(fieldName, value):
    """
    Utility function to assert type validity of textureCompressionModeChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if  isinstance(value,SFString) or isinstance(value,str):
        if str(value) in TEXTURECOMPRESSIONMODECHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + value + ' does not match allowed enumerations in TEXTURECOMPRESSIONMODECHOICES=' + str(TEXTURECOMPRESSIONMODECHOICES))
    if  isinstance(value,MFString):
        if MFString(value).XML() in TEXTURECOMPRESSIONMODECHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + MFString(value).XML() + ' does not match allowed enumerations in TEXTURECOMPRESSIONMODECHOICES=' + str(TEXTURECOMPRESSIONMODECHOICES))

TEXTURECOORDINATEGENERATORMODECHOICES = (
    # strict set of allowed values follow, no other values are valid
    'SPHERE', # Creates texture coordinates for a spherical environment
    'CAMERASPACENORMAL', # Use vertex normal, transformed to camera space, as input texture coordinates
    'CAMERASPACEPOSITION', # Use vertex position, transformed to camera space, as input texture coordinates
    'CAMERASPACEREFLECTIONVECTOR', # Use reflection vector, transformed to camera space, as input texture coordinates
    'SPHERE-LOCAL', # Sphere mapping but in local coordinates
    'COORD', # Use vertex coordinates
    'COORD-EYE', # Use vertex coordinates transformed to camera space
    'NOISE', # Apply Perlin solid noise function on vertex coordinates
    'NOISE-EYE', # Apply Perlin solid noise function on vertex coordinates transformed to camera space
    'SPHERE-REFLECT', # similar to CAMERASPACEREFLECTIONVECTOR with optional index of refraction
    'SPHERE-REFLECT-LOCAL' # Similar to SPHERE-REFLECT transformed to camera space
)
def assertValidTextureCoordinateGeneratorMode(fieldName, value):
    """
    Utility function to assert type validity of textureCoordinateGeneratorModeChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if  isinstance(value,SFString) or isinstance(value,str):
        if str(value) in TEXTURECOORDINATEGENERATORMODECHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + value + ' does not match allowed enumerations in TEXTURECOORDINATEGENERATORMODECHOICES=' + str(TEXTURECOORDINATEGENERATORMODECHOICES))
    if  isinstance(value,MFString):
        if MFString(value).XML() in TEXTURECOORDINATEGENERATORMODECHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + MFString(value).XML() + ' does not match allowed enumerations in TEXTURECOORDINATEGENERATORMODECHOICES=' + str(TEXTURECOORDINATEGENERATORMODECHOICES))

TEXTUREMAGNIFICATIONMODECHOICES = (
    # strict set of allowed values follow, no other values are valid
    'AVG_PIXEL', # weighted average of four texture elements closest to center of pixel being textured
    'DEFAULT', # browser-specified default magnification mode
    'FASTEST', # fastest method available
    'NEAREST_PIXEL', # texture element nearest to the center of pixel being textured
    'NICEST' # highest quality method available
)
def assertValidTextureMagnificationMode(fieldName, value):
    """
    Utility function to assert type validity of textureMagnificationModeChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if  isinstance(value,SFString) or isinstance(value,str):
        if str(value) in TEXTUREMAGNIFICATIONMODECHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + value + ' does not match allowed enumerations in TEXTUREMAGNIFICATIONMODECHOICES=' + str(TEXTUREMAGNIFICATIONMODECHOICES))
    if  isinstance(value,MFString):
        if MFString(value).XML() in TEXTUREMAGNIFICATIONMODECHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + MFString(value).XML() + ' does not match allowed enumerations in TEXTUREMAGNIFICATIONMODECHOICES=' + str(TEXTUREMAGNIFICATIONMODECHOICES))

TEXTUREMINIFICATIONMODECHOICES = (
    # strict set of allowed values follow, no other values are valid
    'AVG_PIXEL', # weighted average of four texture elements closest to center of pixel being textured
    'AVG_PIXEL_AVG_MIPMAP', # tri-linear mipmap filtering
    'AVG_PIXEL_NEAREST_MIPMAP', # choose mipmap that most closely matches size of pixel being textured, use weighted average of four texture elements closest to center of pixel
    'DEFAULT', # browser-specified default minification mode
    'FASTEST', # fastest method available, use mipmaps if possible
    'NEAREST_PIXEL', # texture element nearest to center of pixel being textured
    'NEAREST_PIXEL_AVG_MIPMAP', # texture element nearest to center of pixel being textured, use average of two nearest mipmaps
    'NEAREST_PIXEL_NEAREST_MIPMAP', # texture element nearest to center of pixel being textured, use nearest mipmap
    'NICEST' # highest quality method available
)
def assertValidTextureMinificationMode(fieldName, value):
    """
    Utility function to assert type validity of textureMinificationModeChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if  isinstance(value,SFString) or isinstance(value,str):
        if str(value) in TEXTUREMINIFICATIONMODECHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + value + ' does not match allowed enumerations in TEXTUREMINIFICATIONMODECHOICES=' + str(TEXTUREMINIFICATIONMODECHOICES))
    if  isinstance(value,MFString):
        if MFString(value).XML() in TEXTUREMINIFICATIONMODECHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + MFString(value).XML() + ' does not match allowed enumerations in TEXTUREMINIFICATIONMODECHOICES=' + str(TEXTUREMINIFICATIONMODECHOICES))

UNITCATEGORYCHOICES = (
    # strict set of allowed values follow, no other values are valid
    'angle', # angle default is radians
    'force', # force default is newtons
    'length', # length default is meters
    'mass' # mass default is kilograms
)
def assertValidUnitCategory(fieldName, value):
    """
    Utility function to assert type validity of unitCategoryChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if  isinstance(value,SFString) or isinstance(value,str):
        if str(value) in UNITCATEGORYCHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + value + ' does not match allowed enumerations in UNITCATEGORYCHOICES=' + str(UNITCATEGORYCHOICES))
    if  isinstance(value,MFString):
        if MFString(value).XML() in UNITCATEGORYCHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + MFString(value).XML() + ' does not match allowed enumerations in UNITCATEGORYCHOICES=' + str(UNITCATEGORYCHOICES))

VOLUMERENDERINGWEIGHTFUNCTIONCHOICES = (
    # strict set of allowed values follow, no other values are valid
    'CONSTANT', # Use weightConstant1
    'ALPHA1', # Use O_v
    'ALPHA2', # Use O_blend
    'ONE_MINUS_ALPHA1', # Use 1 - O_v
    'ONE_MINUS_ALPHA2', # Use 1 - O_blend
    'TABLE' # Use table lookup value
)
def assertValidVolumeRenderingWeightFunction(fieldName, value):
    """
    Utility function to assert type validity of volumeRenderingWeightFunctionChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if  isinstance(value,SFString) or isinstance(value,str):
        if str(value) in VOLUMERENDERINGWEIGHTFUNCTIONCHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + value + ' does not match allowed enumerations in VOLUMERENDERINGWEIGHTFUNCTIONCHOICES=' + str(VOLUMERENDERINGWEIGHTFUNCTIONCHOICES))
    if  isinstance(value,MFString):
        if MFString(value).XML() in VOLUMERENDERINGWEIGHTFUNCTIONCHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + MFString(value).XML() + ' does not match allowed enumerations in VOLUMERENDERINGWEIGHTFUNCTIONCHOICES=' + str(VOLUMERENDERINGWEIGHTFUNCTIONCHOICES))

X3DVERSIONCHOICES = (
    # strict set of allowed values follow, no other values are valid
    '3.0', # X3D version 3.0 approved by ISO in 2004.
    '3.1', # X3D version 3.1 Amendment 1 approved by ISO in 2005. Backwards compatibility maintained with version 3.0.
    '3.2', # X3D version 3.2 Amendment 2 approved by ISO in 2007. Backwards compatibility maintained with versions 3.0 and 3.1.
    '3.3', # X3D version 3.3 approved by ISO in 2013 as International Standard (IS). Backwards compatibility maintained with versions 3.0, 3.1 and 3.2.
    '4.0' # X3D version 4.0 under final development by Web3D Consortium. Backwards compatibility maintained with versions 3.0, 3.1, 3.2 and 3.3.
)
def assertValidX3dVersion(fieldName, value):
    """
    Utility function to assert type validity of x3dVersionChoices value, otherwise raise X3DTypeError with diagnostic message.
    Note MFString enumeration values are provided in XML syntax, so check validity accordingly.
    """
    if  not value:
        return True # no failure on empty defaults
    if  isinstance(value,SFString) or isinstance(value,str):
        if str(value) in X3DVERSIONCHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + value + ' does not match allowed enumerations in X3DVERSIONCHOICES=' + str(X3DVERSIONCHOICES))
    if  isinstance(value,MFString):
        if MFString(value).XML() in X3DVERSIONCHOICES:
            return True
        raise X3DTypeError(fieldName + ' value=' + MFString(value).XML() + ' does not match allowed enumerations in X3DVERSIONCHOICES=' + str(X3DVERSIONCHOICES))

###############################################

# Utility Functions

def metaDiagnostics(self, headElement=None):
    """
    Utility function to return any meta info, hint, warning, error, TODO values in this model.
    """
    if  headElement is None:
        headElement = self
    if  isinstance(headElement, X3D):
        headElement = headElement.head
    # print('type(headElement)=' + str(type(headElement)), flush=True) # diagnostic
    if  isinstance(headElement, head):
        result = "meta "
        for each in headElement.children:
            if isinstance(each, meta) and each.name in ('info', 'hint', 'warning', 'error', 'TODO'):
                result += each.name.strip() + ': ' + each.content.strip()
                if  result.endswith('.') or result.endswith(','):
                    result += ' '
                else:
                    result += ', '
        if  result.strip() != "meta":
            return result.rstrip(', ').strip()
    return ''

###############################################

# Field Validation Functions

# Type-specific functions to check for valid values, throw exception if illegal value is found

def fixBoolean(value, default=None):
    """
    Utility function to convert boolean to corresponding Python value.
    """
    # if _DEBUG: print('fixBoolean(value=' + str(value) + ', default=' + str(default) + ')', flush=True)
    if  value is None:
        return default
    if isinstance(value, list) and len(value) == 1:
        # if _DEBUG: print('fixBoolean downcasting by resetting singleton list value=' + str(value) + ' as value=' + str(value[0]), flush=True)
        value = value[0]
    elif isinstance(value, SFBool):
        return value.value # dereference
    elif isinstance(value, MFBool) and len(value) == 1:
        return value.value[0] # dereference
    if value in ('true', 'True'):
        return True
    if value in ('false', 'False'):
        return False
    if isinstance(value, bool):
        return value
    if isinstance(value, list):
        index = 0
        result = value
        for each in value:
#           print('each=' + str(each), flush=True) # debug
            if  each in ('true', 'True'):
                result[index] = True
            elif each in ('false', 'False'):
                result[index] = False
            while isinstance(each, list) and len(each) == 1:
                # if _DEBUG: print('fixBoolean downcasting by extracting singleton list value[' + str(index) + ']=' + str(each) + ' as each[0]=' + str(each[0]), flush=True)
                result[index] = each[0]
            if not isinstance(result[index], bool):
                # print(flush=True)
                raise X3DTypeError('fixBoolean(value=' + str(value) + ') MFBool value[' + str(index) + ']=' + str(each) + ', result[' + str(index) + ']=' + result[index] + ' with type=' + str(type(value)) + ' is not a valid Python bool expression')
            index += 1
        # if _DEBUG: print('...fixBoolean result=' + str(result), flush=True)
        return result
    # print(flush=True)
    raise X3DTypeError('fixBoolean(value=' + str(value) + ') with type=' + str(type(value)) + ') is not a valid Python bool')

def isPositive(value):
    """
    Utility function to confirm positive value(s) greater than or equal to zero.
    """
    if isinstance(value, list) and any(isinstance(x, tuple) for x in value):
        for each in value:
            for element in each:
                if element <= 0:
                    return False
        return True
    if isinstance(value, (list, tuple)):
        for each in value:
            if each <= 0:
                return False
        return True
    if isinstance(value, (int, float)):
        return value > 0

def assertPositive(fieldName, value):
    """
    Utility function to raise X3DTypeError if not isPositive(value).
    """
    # if _DEBUG: print('* debug: assertPositive(' + str(fieldName) + ', ' + str(value) + ')', flush=True);
    assert isPositive(value), fieldName + '=' + str(value) + ' fails assertPositive requirements: value(s) greater than or equal to zero'

def isNonNegative(value):
    """
    Utility function to confirm nonnegative value(s) greater than or equal to zero.
    """
    if isinstance(value, list) and any(isinstance(x, tuple) for x in value):
        for each in value:
            for element in each:
                if element < 0:
                    return False
        return True
    if isinstance(value, (list, tuple)):
        # if _DEBUG: print('isNonNegative: ', value, flush=True);
        for each in value:
            if each < 0:
                return False
        return True
    if isinstance(value, (int, float)):
        return value >= 0

def assertNonNegative(fieldName, value):
    """
    Utility function to raise X3DTypeError if not isNonNegative(value).
    """
    # if _DEBUG: print('* debug: assertNonNegative(' + str(fieldName) + ', ' + str(value) + ')', flush=True);
    assert isNonNegative(value), fieldName + '=' + str(value) + ' fails assertNonNegative requirements: value(s) greater than or equal to zero'

def isZeroToOne(value):
    """
    Utility function to confirm value(s) in range [0..1]
    """
    # if True or _DEBUG: print('* debug: isZeroToOne(' + str(value) + ')', flush=True);
    if isinstance(value, (list, tuple)):
        for each in value:
            if isinstance(each, (list, tuple)):
                for another in each:
                    if not 0 <= another <= 1:
                        return False
            elif not 0 <= each <= 1:
                return False
        return True
    if isinstance(value, (int, float)):
        return 0 <= value <= 1

def assertZeroToOne(fieldName, value):
    """
    Utility function to raise X3DTypeError if not isZeroToOne(value)
    """
    # if _DEBUG: print('* debug: assertZeroToOne(' + str(fieldName) + ', ' + str(value) + ')', flush=True);
    assert isZeroToOne(value), fieldName + '=' + str(value) + ' fails assertZeroToOne requirements: value(s) in range [0..1]'

def isLessThanEquals(value, maximum):
    """
    Utility function to confirm value(s) less than or equal to maximum.
    """
    # if True or _DEBUG: print('* debug: isLessThanEquals(' + str(value) + ')', flush=True);
    if isinstance(value, list) and any(isinstance(x, tuple) for x in value):
        for each in value:
            for element in each:
                if element > maximum:
                    return False
        return True
    if isinstance(value, (list, tuple)):
        for each in value:
            if each > maximum:
                return False
        return True
    if isinstance(value, (int, float)):
        return value <= maximum

def assertLessThanEquals(fieldName, value, maximum):
    """
    Utility function to raise X3DTypeError if not isLessThanEquals(value)
    """
    # if _DEBUG: print('* debug: assertLessThanEquals(' + str(fieldName) + ', ' + str(value) + ')', flush=True);
    assert isLessThanEquals(value, maximum), fieldName + '=' + str(value) + ' fails assertLessThanEquals maximum=' + str(maximum)

def isLessThan(value, maximum):
    """
    Utility function to confirm value(s) less than maximum.
    """
    # if True or _DEBUG: print('* debug: isLessThan(' + str(value) + ')', flush=True);
    if isinstance(value, list) and any(isinstance(x, tuple) for x in value):
        for each in value:
            for element in each:
                if element >= maximum:
                    return False
        return True
    if isinstance(value, (list, tuple)):
        for each in value:
            if each >= maximum:
                return False
        return True
    if isinstance(value, (int, float)):
        return value < maximum

def assertLessThan(fieldName, value, maximum):
    """
    Utility function to raise X3DTypeError if not isLessThan(value)
    """
    # if _DEBUG: print('* debug: assertLessThan(' + str(fieldName) + ', ' + str(value) + ')', flush=True);
    assert isLessThan(value, maximum), fieldName + '=' + str(value) + ' fails assertLessThan maximum=' + str(maximum)

######
def isGreaterThanEquals(value, minimum):
    """
    Utility function to confirm value(s) less than or equal to minimum.
    """
    # if True or _DEBUG: print('* debug: isGreaterThanEquals(' + str(value) + ')', flush=True);
    if isinstance(value, list) and any(isinstance(x, tuple) for x in value):
        for each in value:
            for element in each:
                if element < minimum:
                    return False
        return True
    if isinstance(value, (list, tuple)):
        for each in value:
            if each < minimum:
                return False
        return True
    if isinstance(value, (int, float)):
        return value >= minimum

def assertGreaterThanEquals(fieldName, value, minimum):
    """
    Utility function to raise X3DTypeError if not isGreaterThanEquals(value)
    """
    # if _DEBUG: print('* debug: assertGreaterThanEquals(' + str(fieldName) + ', ' + str(value) + ')', flush=True);
    assert isGreaterThanEquals(value, minimum), fieldName + '=' + str(value) + ' fails assertGreaterThanEquals minimum=' + str(minimum)

def isGreaterThan(value, minimum):
    """
    Utility function to confirm value(s) less than minimum.
    """
    # if True or _DEBUG: print('* debug: isGreaterThan(' + str(value) + ')', flush=True);
    if isinstance(value, list) and any(isinstance(x, tuple) for x in value):
        for each in value:
            for element in each:
                if element <= minimum:
                    return False
        return True
    if isinstance(value, (list, tuple)):
        for each in value:
            if each <= minimum:
                return False
        return True
    if isinstance(value, (int, float)):
        return value > minimum

def assertGreaterThan(fieldName, value, minimum):
    """
    Utility function to raise X3DTypeError if not isGreaterThan(value)
    """
    # if _DEBUG: print('* debug: assertGreaterThan(' + str(fieldName) + ', ' + str(value) + ')', flush=True);
    assert isGreaterThan(value, minimum), fieldName + '=' + str(value) + ' fails assertGreaterThan minimum=' + str(minimum)

def isBoundingBox(value):
    """
    Utility function to confirm legal X3D bounding box value of (-1 -1 -1) or nonnegative triple.
    """
    if value is None:
        return None
    # if True or _DEBUG: print('* debug: isBoundingBox(' + str(value) + ')', 'isinstance(value, tuple)=' + str(isinstance(value, tuple)), 'len(value)=' + str(len(value)), flush=True);
    if isinstance(value, (list, tuple)):
        if len(value) != 3:
            return False
        if value[0] == -1 and value[1] == -1 and value[2] == -1:
            return True
        return isNonNegative(value) # legal bounding box tuple

def assertBoundingBox(fieldName, value):
    """
    Utility function to raise X3DTypeError if not isBoundingBox(value)
    """
    # if True or _DEBUG: print('* debug: assertBoundingBox(' + str(fieldName) + ', ' + str(value) + ')', flush=True);
    assert isBoundingBox(value), fieldName + '=' + str(value) + ' fails assertBoundingBox requirements: must be (-1, -1, -1) or non-negative 3-tuple'

def isValidSFBool(value):
    """
    Utility function to determine type validity of a SFBool value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFBool) and not isinstance(value, MFBool):
            # if _DEBUG: print('SFBool type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFBool)=' + str(isinstance(value, SFBool)), flush=True)
            return False # type mismatch!
    if isinstance(value, SFBool):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, MFBool) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
        return True
    if not isinstance(value, bool):
        return False
    return True

def assertValidSFBool(value):
    """
    Utility function to assert type validity of a SFBool value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFBool) and not isinstance(value, MFBool):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFBool')
    if isinstance(value, SFBool):
        value = value.value # dereference value from this base type
    if isinstance(value, MFBool) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
    if not isinstance(value, bool):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid bool value (True or False) for SFBool')
    if not isValidSFBool(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python bool value (True or False) for SFBool')
    return True

def isValidMFBool(value):
    """
    Utility function to determine type validity of a MFBool value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFBool) and not isinstance(value, MFBool):
            # if _DEBUG: print('MFBool type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFBool)=' + str(isinstance(value, MFBool)), flush=True)
            return False # type mismatch!
    if isinstance(value, MFBool):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, SFBool):
        value = [value.value] # dereference value from this SF type, convert to list #1
        return True
    if not isinstance(value, list):
        return False
    for each in value:
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFBool):
            each = each.value # dereference
        if not isinstance(each, bool):
            return False
    return True

def assertValidMFBool(value):
    """
    Utility function to assert type validity of a MFBool value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFBool) and not isinstance(value, MFBool):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFBool')
    if isinstance(value, MFBool):
        value = value.value # dereference value from this base type
    if isinstance(value, SFBool) or not isinstance(value, list):
        value = [value.value] # dereference value from this SF type, convert to list #2
    if not isinstance(value, list):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFBool')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    for each in value:
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFBool):
            each = each.value # dereference
        if not isinstance(each, bool):
            # print(flush=True)
            raise X3DTypeError('MFBool list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid bool')
    if not isValidMFBool(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFBool')
    return True

def isValidSFColor(value):
    """
    Utility function to determine type validity of a SFColor value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFColor) and not isinstance(value, MFColor):
            # if _DEBUG: print('SFColor type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFColor)=' + str(isinstance(value, SFColor)), flush=True)
            return False # type mismatch!
    if isinstance(value, SFColor):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, MFColor) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
        return True
    if not isinstance(value, tuple):
        return False
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFColor):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            return False
        if (each < 0) or (each > 1):
            return False
    if tupleCount != 3:
        return False
    if not isZeroToOne(value):
        return False
    return True

def assertValidSFColor(value):
    """
    Utility function to assert type validity of a SFColor value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFColor) and not isinstance(value, MFColor):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFColor')
    if isinstance(value, SFColor):
        value = value.value # dereference value from this base type
    if isinstance(value, MFColor) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
    if not isinstance(value, tuple):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFColor')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFColor):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            # print(flush=True)
            raise X3DTypeError('SFColor list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
        if (each < 0) or (each > 1):
            # print(flush=True)
            raise X3DTypeError('SFColor' + str(value)[:100] + ' has value ' + str(each) + ' with type=' + str(type(value)) + ' is  out of range [0..1] and is not a valid SFColor')
    if tupleCount != 3:
        # print(flush=True)
        raise X3DTypeError('SFColor ' + str(value)[:100] + ', type=' + str(type(value)) + ' has ' + str(tupleCount) + ' elements instead of 3')
    assertZeroToOne('SFColor', value)
    if not isValidSFColor(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFColor')
    return True

def isValidMFColor(value):
    """
    Utility function to determine type validity of a MFColor value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFColor) and not isinstance(value, MFColor):
            # if _DEBUG: print('MFColor type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFColor)=' + str(isinstance(value, MFColor)), flush=True)
            return False # type mismatch!
    if isinstance(value, MFColor):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, SFColor):
        value = [value.value] # dereference value from this SF type, convert to list #1
        return True
    if not isinstance(value, list):
        return False
    _index = 0
    for each in value:
        _index += 1
        if len(each) % MFColor().TUPLE_SIZE() != 0:
            # if _DEBUG:
            print('* isValidMFColor tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFColor().TUPLE_SIZE()=' + str(MFColor().TUPLE_SIZE()) + ' for value=' + str(value), flush=True)
            return False
        for element in each:
            if not isinstance(element, float) and not isinstance(element, int):
                return False
            if (element < 0) or (element > 1):
                return False
    if not isZeroToOne(value):
        return False
    return True

def assertValidMFColor(value):
    """
    Utility function to assert type validity of a MFColor value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFColor) and not isinstance(value, MFColor):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFColor')
    if isinstance(value, MFColor):
        value = value.value # dereference value from this base type
    if isinstance(value, SFColor) or not isinstance(value, list):
        value = [value.value] # dereference value from this SF type, convert to list #2
    if not isinstance(value, list):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFColor')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #2
    if isinstance(value, list):
        _index = 0
        for each in value:
            if len(each) % MFColor().TUPLE_SIZE() != 0:
                # print(flush=True)
                raise X3DValueError('MFColor tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFColor().TUPLE_SIZE()=' + str(MFColor().TUPLE_SIZE()) + ' for value=' + str(value)[:100])
#            if not isinstance(each, (tuple, SFColor)):
#                # print(flush=True)
#                raise X3DTypeError('MFColor element #' + str(_index) + ' with value ' + str(each) + ', type=' + str(type(each)) + ' is not a valid tuple')
            _index += 1
            if isinstance(each, tuple):
                for element in each:
                    if not isinstance(element, float) and not isinstance(element, int):
                        # print(flush=True)
                        raise X3DTypeError('MFColor element #' + str(_index) + ' tuple ' + str(each) + ' has value=' + str(element) + ', type=' + str(type(element)) + ' that is not a valid float')
                    if (element < 0) or (element > 1):
                        # print(flush=True)
                        raise X3DTypeError('MFColor' + str(value)[:100] + ' has value ' + str(element) + ' with type=' + str(type(value)) + ' out of range [0..1] and is not a valid MFColor')
    assertZeroToOne('MFColor', value)
    if not isValidMFColor(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFColor')
    return True

def isValidSFColorRGBA(value):
    """
    Utility function to determine type validity of a SFColorRGBA value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFColorRGBA) and not isinstance(value, MFColorRGBA):
            # if _DEBUG: print('SFColorRGBA type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFColorRGBA)=' + str(isinstance(value, SFColorRGBA)), flush=True)
            return False # type mismatch!
    if isinstance(value, SFColorRGBA):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, MFColorRGBA) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
        return True
    if not isinstance(value, tuple):
        return False
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFColorRGBA):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            return False
        if (each < 0) or (each > 1):
            return False
    if tupleCount != 4:
        return False
    if not isZeroToOne(value):
        return False
    return True

def assertValidSFColorRGBA(value):
    """
    Utility function to assert type validity of a SFColorRGBA value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFColorRGBA) and not isinstance(value, MFColorRGBA):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFColorRGBA')
    if isinstance(value, SFColorRGBA):
        value = value.value # dereference value from this base type
    if isinstance(value, MFColorRGBA) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
    if not isinstance(value, tuple):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFColorRGBA')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFColorRGBA):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            # print(flush=True)
            raise X3DTypeError('SFColorRGBA list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
        if (each < 0) or (each > 1):
            # print(flush=True)
            raise X3DTypeError('SFColorRGBA' + str(value)[:100] + ' has value ' + str(each) + ' with type=' + str(type(value)) + ' is  out of range [0..1] and is not a valid SFColorRGBA')
    if tupleCount != 4:
        # print(flush=True)
        raise X3DTypeError('SFColorRGBA ' + str(value)[:100] + ', type=' + str(type(value)) + ' has ' + str(tupleCount) + ' elements instead of 4')
    assertZeroToOne('SFColorRGBA', value)
    if not isValidSFColorRGBA(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFColorRGBA')
    return True

def isValidMFColorRGBA(value):
    """
    Utility function to determine type validity of a MFColorRGBA value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFColorRGBA) and not isinstance(value, MFColorRGBA):
            # if _DEBUG: print('MFColorRGBA type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFColorRGBA)=' + str(isinstance(value, MFColorRGBA)), flush=True)
            return False # type mismatch!
    if isinstance(value, MFColorRGBA):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, SFColorRGBA):
        value = [value.value] # dereference value from this SF type, convert to list #1
        return True
    if not isinstance(value, list):
        return False
    _index = 0
    for each in value:
        _index += 1
        if len(each) % MFColorRGBA().TUPLE_SIZE() != 0:
            # if _DEBUG:
            print('* isValidMFColorRGBA tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFColorRGBA().TUPLE_SIZE()=' + str(MFColorRGBA().TUPLE_SIZE()) + ' for value=' + str(value), flush=True)
            return False
        for element in each:
            if not isinstance(element, float) and not isinstance(element, int):
                return False
            if (element < 0) or (element > 1):
                return False
    if not isZeroToOne(value):
        return False
    return True

def assertValidMFColorRGBA(value):
    """
    Utility function to assert type validity of a MFColorRGBA value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFColorRGBA) and not isinstance(value, MFColorRGBA):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFColorRGBA')
    if isinstance(value, MFColorRGBA):
        value = value.value # dereference value from this base type
    if isinstance(value, SFColorRGBA) or not isinstance(value, list):
        value = [value.value] # dereference value from this SF type, convert to list #2
    if not isinstance(value, list):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFColorRGBA')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #2
    if isinstance(value, list):
        _index = 0
        for each in value:
            if len(each) % MFColorRGBA().TUPLE_SIZE() != 0:
                # print(flush=True)
                raise X3DValueError('MFColorRGBA tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFColorRGBA().TUPLE_SIZE()=' + str(MFColorRGBA().TUPLE_SIZE()) + ' for value=' + str(value)[:100])
#            if not isinstance(each, (tuple, SFColorRGBA)):
#                # print(flush=True)
#                raise X3DTypeError('MFColorRGBA element #' + str(_index) + ' with value ' + str(each) + ', type=' + str(type(each)) + ' is not a valid tuple')
            _index += 1
            if isinstance(each, tuple):
                for element in each:
                    if not isinstance(element, float) and not isinstance(element, int):
                        # print(flush=True)
                        raise X3DTypeError('MFColorRGBA element #' + str(_index) + ' tuple ' + str(each) + ' has value=' + str(element) + ', type=' + str(type(element)) + ' that is not a valid float')
                    if (element < 0) or (element > 1):
                        # print(flush=True)
                        raise X3DTypeError('MFColorRGBA' + str(value)[:100] + ' has value ' + str(element) + ' with type=' + str(type(value)) + ' out of range [0..1] and is not a valid MFColorRGBA')
    assertZeroToOne('MFColorRGBA', value)
    if not isValidMFColorRGBA(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFColorRGBA')
    return True

def isValidSFDouble(value):
    """
    Utility function to determine type validity of a SFDouble value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFDouble) and not isinstance(value, MFDouble):
            # if _DEBUG: print('SFDouble type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFDouble)=' + str(isinstance(value, SFDouble)), flush=True)
            return False # type mismatch!
    if isinstance(value, SFDouble):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, MFDouble) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
        return True
    if not isinstance(value, float) and not isinstance(value, int):
        return False
    return True

def assertValidSFDouble(value):
    """
    Utility function to assert type validity of a SFDouble value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFDouble) and not isinstance(value, MFDouble):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFDouble')
    if isinstance(value, SFDouble):
        value = value.value # dereference value from this base type
    if isinstance(value, MFDouble) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
    if not isinstance(value, float) and not isinstance(value, int):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid float value for SFDouble')
    if not isValidSFDouble(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python float value for SFDouble')
    return True

def isValidMFDouble(value):
    """
    Utility function to determine type validity of a MFDouble value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFDouble) and not isinstance(value, MFDouble):
            # if _DEBUG: print('MFDouble type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFDouble)=' + str(isinstance(value, MFDouble)), flush=True)
            return False # type mismatch!
    if isinstance(value, MFDouble):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, SFDouble):
        value = [value.value] # dereference value from this SF type, convert to list #1
        return True
    if not isinstance(value, list):
        return False
    for each in value:
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFDouble):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            return False
    return True

def assertValidMFDouble(value):
    """
    Utility function to assert type validity of a MFDouble value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFDouble) and not isinstance(value, MFDouble):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFDouble')
    if isinstance(value, MFDouble):
        value = value.value # dereference value from this base type
    if isinstance(value, SFDouble) or not isinstance(value, list):
        value = [value.value] # dereference value from this SF type, convert to list #2
    if not isinstance(value, list):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFDouble')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    for each in value:
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFDouble):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            # print(flush=True)
            raise X3DTypeError('MFDouble list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    if not isValidMFDouble(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFDouble')
    return True

def isValidSFFloat(value):
    """
    Utility function to determine type validity of a SFFloat value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFFloat) and not isinstance(value, MFFloat):
            # if _DEBUG: print('SFFloat type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFFloat)=' + str(isinstance(value, SFFloat)), flush=True)
            return False # type mismatch!
    if isinstance(value, SFFloat):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, MFFloat) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
        return True
    if not isinstance(value, float) and not isinstance(value, int):
        return False
    return True

def assertValidSFFloat(value):
    """
    Utility function to assert type validity of a SFFloat value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFFloat) and not isinstance(value, MFFloat):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFFloat')
    if isinstance(value, SFFloat):
        value = value.value # dereference value from this base type
    if isinstance(value, MFFloat) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
    if not isinstance(value, float) and not isinstance(value, int):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid float value for SFFloat')
    if not isValidSFFloat(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python float value for SFFloat')
    return True

def isValidMFFloat(value):
    """
    Utility function to determine type validity of a MFFloat value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFFloat) and not isinstance(value, MFFloat):
            # if _DEBUG: print('MFFloat type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFFloat)=' + str(isinstance(value, MFFloat)), flush=True)
            return False # type mismatch!
    if isinstance(value, MFFloat):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, SFFloat):
        value = [value.value] # dereference value from this SF type, convert to list #1
        return True
    if not isinstance(value, list):
        return False
    for each in value:
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFFloat):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            return False
    return True

def assertValidMFFloat(value):
    """
    Utility function to assert type validity of a MFFloat value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFFloat) and not isinstance(value, MFFloat):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFFloat')
    if isinstance(value, MFFloat):
        value = value.value # dereference value from this base type
    if isinstance(value, SFFloat) or not isinstance(value, list):
        value = [value.value] # dereference value from this SF type, convert to list #2
    if not isinstance(value, list):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFFloat')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    for each in value:
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFFloat):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            # print(flush=True)
            raise X3DTypeError('MFFloat list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    if not isValidMFFloat(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFFloat')
    return True

def isValidSFImage(value):
    """
    Utility function to determine type validity of a SFImage value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFImage) and not isinstance(value, MFImage):
            # if _DEBUG: print('SFImage type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFImage)=' + str(isinstance(value, SFImage)), flush=True)
            return False # type mismatch!
    if isinstance(value, SFImage):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, MFImage) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
        return True
    if not isinstance(value, list):
        return False
    for each in value:
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFImage):
            each = each.value # dereference
        if not isinstance(each, int):
            return False
    if len(value) > 0: # SFImage list length checks
        if 0 < len(value) < 3:
            return False # SFImage list must start with width, height and number of components (0..4)
        width = value[0]
        height = value[1]
        numberComponents = value[2]
        if  len(value) != (width * height) + 3: # assumes each value in array has all component values in single integer
            print('SFImage array length of ' + str(len(value)) + ' does not equal (width=' + str(width)+ ' * height=' + str(height)+ ') + 3) = ' + str(width * height * numberComponents + 3) + ' (numberComponents=' + numberComponents + ')', flush=True)
            return False # SFImage has invalid list length
    return True

def assertValidSFImage(value):
    """
    Utility function to assert type validity of a SFImage value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFImage) and not isinstance(value, MFImage):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFImage')
    if isinstance(value, SFImage):
        value = value.value # dereference value from this base type
    if isinstance(value, MFImage) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
    if not isinstance(value, list):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for SFImage')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    for each in value:
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFImage):
            each = each.value # dereference
        if not isinstance(each, int):
            # print(flush=True)
            raise X3DTypeError('SFImage list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid int')
    if not isValidSFImage(value):
        # print(flush=True)
        diagnostic = ''
        if len(value) > 0: # SFImage list length checks
            if 0 < len(value) < 3:
                diagnostic = 'SFImage list must start with width, height and number of components (0..4)'
            else:
                width = value[0]
                height = value[1]
                numberComponents = value[2]
                diagnostic = ' array length of ' + str(len(value)) + ' does not equal (width=' + str(width)+ ' * height=' + str(height)+ ') + 3) = ' + str(width * height + 3)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for SFImage' + diagnostic)
    return True

def isValidMFImage(value):
    """
    Utility function to determine type validity of a MFImage value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFImage) and not isinstance(value, MFImage):
            # if _DEBUG: print('MFImage type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFImage)=' + str(isinstance(value, MFImage)), flush=True)
            return False # type mismatch!
    if isinstance(value, MFImage):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, SFImage):
        value = [value.value] # dereference value from this SF type, convert to list #1
        return True
    if not isinstance(value, list):
        return False
    for each in value:
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFImage):
            each = each.value # dereference
        if not isinstance(each, int):
            return False
    return True

def assertValidMFImage(value):
    """
    Utility function to assert type validity of a MFImage value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFImage) and not isinstance(value, MFImage):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFImage')
    if isinstance(value, MFImage):
        value = value.value # dereference value from this base type
    if isinstance(value, SFImage) or not isinstance(value, list):
        value = [value.value] # dereference value from this SF type, convert to list #2
    if not isinstance(value, list):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFImage')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    for each in value:
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFImage):
            each = each.value # dereference
        if not isinstance(each, int):
            # print(flush=True)
            raise X3DTypeError('MFImage list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid int')
    if not isValidMFImage(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFImage')
    return True

def isValidSFInt32(value):
    """
    Utility function to determine type validity of a SFInt32 value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFInt32) and not isinstance(value, MFInt32):
            # if _DEBUG: print('SFInt32 type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFInt32)=' + str(isinstance(value, SFInt32)), flush=True)
            return False # type mismatch!
    if isinstance(value, SFInt32):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, MFInt32) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
        return True
    if not isinstance(value, int):
        return False
    return True

def assertValidSFInt32(value):
    """
    Utility function to assert type validity of a SFInt32 value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFInt32) and not isinstance(value, MFInt32):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFInt32')
    if isinstance(value, SFInt32):
        value = value.value # dereference value from this base type
    if isinstance(value, MFInt32) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
    if not isinstance(value, int):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid int value for SFInt32')
    if not isValidSFInt32(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python int value for SFInt32')
    return True

def isValidMFInt32(value):
    """
    Utility function to determine type validity of a MFInt32 value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFInt32) and not isinstance(value, MFInt32):
            # if _DEBUG: print('MFInt32 type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFInt32)=' + str(isinstance(value, MFInt32)), flush=True)
            return False # type mismatch!
    if isinstance(value, MFInt32):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, SFInt32):
        value = [value.value] # dereference value from this SF type, convert to list #1
        return True
    if not isinstance(value, list):
        return False
    for each in value:
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFInt32):
            each = each.value # dereference
        if not isinstance(each, int):
            return False
    return True

def assertValidMFInt32(value):
    """
    Utility function to assert type validity of a MFInt32 value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFInt32) and not isinstance(value, MFInt32):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFInt32')
    if isinstance(value, MFInt32):
        value = value.value # dereference value from this base type
    if isinstance(value, SFInt32) or not isinstance(value, list):
        value = [value.value] # dereference value from this SF type, convert to list #2
    if not isinstance(value, list):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFInt32')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    for each in value:
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFInt32):
            each = each.value # dereference
        if not isinstance(each, int):
            # print(flush=True)
            raise X3DTypeError('MFInt32 list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid int')
    if not isValidMFInt32(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFInt32')
    return True

def isValidSFMatrix3d(value):
    """
    Utility function to determine type validity of a SFMatrix3d value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFMatrix3d) and not isinstance(value, MFMatrix3d):
            # if _DEBUG: print('SFMatrix3d type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFMatrix3d)=' + str(isinstance(value, SFMatrix3d)), flush=True)
            return False # type mismatch!
    if isinstance(value, SFMatrix3d):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, MFMatrix3d) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
        return True
    if not isinstance(value, tuple):
        return False
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFMatrix3d):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            return False
    if tupleCount != 9:
        return False
    return True

def assertValidSFMatrix3d(value):
    """
    Utility function to assert type validity of a SFMatrix3d value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFMatrix3d) and not isinstance(value, MFMatrix3d):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFMatrix3d')
    if isinstance(value, SFMatrix3d):
        value = value.value # dereference value from this base type
    if isinstance(value, MFMatrix3d) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
    if not isinstance(value, tuple):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFMatrix3d')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFMatrix3d):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            # print(flush=True)
            raise X3DTypeError('SFMatrix3d list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    if tupleCount != 9:
        # print(flush=True)
        raise X3DTypeError('SFMatrix3d ' + str(value)[:100] + ', type=' + str(type(value)) + ' has ' + str(tupleCount) + ' elements instead of 9')
    if not isValidSFMatrix3d(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFMatrix3d')
    return True

def isValidMFMatrix3d(value):
    """
    Utility function to determine type validity of a MFMatrix3d value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFMatrix3d) and not isinstance(value, MFMatrix3d):
            # if _DEBUG: print('MFMatrix3d type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFMatrix3d)=' + str(isinstance(value, MFMatrix3d)), flush=True)
            return False # type mismatch!
    if isinstance(value, MFMatrix3d):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, SFMatrix3d):
        value = [value.value] # dereference value from this SF type, convert to list #1
        return True
    if not isinstance(value, list):
        return False
    _index = 0
    for each in value:
        _index += 1
        if len(each) % MFMatrix3d().TUPLE_SIZE() != 0:
            # if _DEBUG:
            print('* isValidMFMatrix3d tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFMatrix3d().TUPLE_SIZE()=' + str(MFMatrix3d().TUPLE_SIZE()) + ' for value=' + str(value), flush=True)
            return False
        for element in each:
            if not isinstance(element, float) and not isinstance(element, int):
                return False
    return True

def assertValidMFMatrix3d(value):
    """
    Utility function to assert type validity of a MFMatrix3d value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFMatrix3d) and not isinstance(value, MFMatrix3d):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFMatrix3d')
    if isinstance(value, MFMatrix3d):
        value = value.value # dereference value from this base type
    if isinstance(value, SFMatrix3d) or not isinstance(value, list):
        value = [value.value] # dereference value from this SF type, convert to list #2
    if not isinstance(value, list):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFMatrix3d')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #2
    if isinstance(value, list):
        _index = 0
        for each in value:
            if len(each) % MFMatrix3d().TUPLE_SIZE() != 0:
                # print(flush=True)
                raise X3DValueError('MFMatrix3d tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFMatrix3d().TUPLE_SIZE()=' + str(MFMatrix3d().TUPLE_SIZE()) + ' for value=' + str(value)[:100])
#            if not isinstance(each, (tuple, SFMatrix3d)):
#                # print(flush=True)
#                raise X3DTypeError('MFMatrix3d element #' + str(_index) + ' with value ' + str(each) + ', type=' + str(type(each)) + ' is not a valid tuple')
            _index += 1
            if isinstance(each, tuple):
                for element in each:
                    if not isinstance(element, float) and not isinstance(element, int):
                        # print(flush=True)
                        raise X3DTypeError('MFMatrix3d element #' + str(_index) + ' tuple ' + str(each) + ' has value=' + str(element) + ', type=' + str(type(element)) + ' that is not a valid float')
    if not isValidMFMatrix3d(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFMatrix3d')
    return True

def isValidSFMatrix3f(value):
    """
    Utility function to determine type validity of a SFMatrix3f value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFMatrix3f) and not isinstance(value, MFMatrix3f):
            # if _DEBUG: print('SFMatrix3f type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFMatrix3f)=' + str(isinstance(value, SFMatrix3f)), flush=True)
            return False # type mismatch!
    if isinstance(value, SFMatrix3f):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, MFMatrix3f) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
        return True
    if not isinstance(value, tuple):
        return False
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFMatrix3f):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            return False
    if tupleCount != 9:
        return False
    return True

def assertValidSFMatrix3f(value):
    """
    Utility function to assert type validity of a SFMatrix3f value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFMatrix3f) and not isinstance(value, MFMatrix3f):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFMatrix3f')
    if isinstance(value, SFMatrix3f):
        value = value.value # dereference value from this base type
    if isinstance(value, MFMatrix3f) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
    if not isinstance(value, tuple):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFMatrix3f')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFMatrix3f):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            # print(flush=True)
            raise X3DTypeError('SFMatrix3f list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    if tupleCount != 9:
        # print(flush=True)
        raise X3DTypeError('SFMatrix3f ' + str(value)[:100] + ', type=' + str(type(value)) + ' has ' + str(tupleCount) + ' elements instead of 9')
    if not isValidSFMatrix3f(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFMatrix3f')
    return True

def isValidMFMatrix3f(value):
    """
    Utility function to determine type validity of a MFMatrix3f value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFMatrix3f) and not isinstance(value, MFMatrix3f):
            # if _DEBUG: print('MFMatrix3f type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFMatrix3f)=' + str(isinstance(value, MFMatrix3f)), flush=True)
            return False # type mismatch!
    if isinstance(value, MFMatrix3f):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, SFMatrix3f):
        value = [value.value] # dereference value from this SF type, convert to list #1
        return True
    if not isinstance(value, list):
        return False
    _index = 0
    for each in value:
        _index += 1
        if len(each) % MFMatrix3f().TUPLE_SIZE() != 0:
            # if _DEBUG:
            print('* isValidMFMatrix3f tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFMatrix3f().TUPLE_SIZE()=' + str(MFMatrix3f().TUPLE_SIZE()) + ' for value=' + str(value), flush=True)
            return False
        for element in each:
            if not isinstance(element, float) and not isinstance(element, int):
                return False
    return True

def assertValidMFMatrix3f(value):
    """
    Utility function to assert type validity of a MFMatrix3f value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFMatrix3f) and not isinstance(value, MFMatrix3f):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFMatrix3f')
    if isinstance(value, MFMatrix3f):
        value = value.value # dereference value from this base type
    if isinstance(value, SFMatrix3f) or not isinstance(value, list):
        value = [value.value] # dereference value from this SF type, convert to list #2
    if not isinstance(value, list):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFMatrix3f')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #2
    if isinstance(value, list):
        _index = 0
        for each in value:
            if len(each) % MFMatrix3f().TUPLE_SIZE() != 0:
                # print(flush=True)
                raise X3DValueError('MFMatrix3f tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFMatrix3f().TUPLE_SIZE()=' + str(MFMatrix3f().TUPLE_SIZE()) + ' for value=' + str(value)[:100])
#            if not isinstance(each, (tuple, SFMatrix3f)):
#                # print(flush=True)
#                raise X3DTypeError('MFMatrix3f element #' + str(_index) + ' with value ' + str(each) + ', type=' + str(type(each)) + ' is not a valid tuple')
            _index += 1
            if isinstance(each, tuple):
                for element in each:
                    if not isinstance(element, float) and not isinstance(element, int):
                        # print(flush=True)
                        raise X3DTypeError('MFMatrix3f element #' + str(_index) + ' tuple ' + str(each) + ' has value=' + str(element) + ', type=' + str(type(element)) + ' that is not a valid float')
    if not isValidMFMatrix3f(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFMatrix3f')
    return True

def isValidSFMatrix4d(value):
    """
    Utility function to determine type validity of a SFMatrix4d value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFMatrix4d) and not isinstance(value, MFMatrix4d):
            # if _DEBUG: print('SFMatrix4d type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFMatrix4d)=' + str(isinstance(value, SFMatrix4d)), flush=True)
            return False # type mismatch!
    if isinstance(value, SFMatrix4d):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, MFMatrix4d) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
        return True
    if not isinstance(value, tuple):
        return False
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFMatrix4d):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            return False
    if tupleCount != 16:
        return False
    return True

def assertValidSFMatrix4d(value):
    """
    Utility function to assert type validity of a SFMatrix4d value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFMatrix4d) and not isinstance(value, MFMatrix4d):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFMatrix4d')
    if isinstance(value, SFMatrix4d):
        value = value.value # dereference value from this base type
    if isinstance(value, MFMatrix4d) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
    if not isinstance(value, tuple):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFMatrix4d')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFMatrix4d):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            # print(flush=True)
            raise X3DTypeError('SFMatrix4d list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    if tupleCount != 16:
        # print(flush=True)
        raise X3DTypeError('SFMatrix4d ' + str(value)[:100] + ', type=' + str(type(value)) + ' has ' + str(tupleCount) + ' elements instead of 16')
    if not isValidSFMatrix4d(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFMatrix4d')
    return True

def isValidMFMatrix4d(value):
    """
    Utility function to determine type validity of a MFMatrix4d value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFMatrix4d) and not isinstance(value, MFMatrix4d):
            # if _DEBUG: print('MFMatrix4d type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFMatrix4d)=' + str(isinstance(value, MFMatrix4d)), flush=True)
            return False # type mismatch!
    if isinstance(value, MFMatrix4d):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, SFMatrix4d):
        value = [value.value] # dereference value from this SF type, convert to list #1
        return True
    if not isinstance(value, list):
        return False
    _index = 0
    for each in value:
        _index += 1
        if len(each) % MFMatrix4d().TUPLE_SIZE() != 0:
            # if _DEBUG:
            print('* isValidMFMatrix4d tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFMatrix4d().TUPLE_SIZE()=' + str(MFMatrix4d().TUPLE_SIZE()) + ' for value=' + str(value), flush=True)
            return False
        for element in each:
            if not isinstance(element, float) and not isinstance(element, int):
                return False
    return True

def assertValidMFMatrix4d(value):
    """
    Utility function to assert type validity of a MFMatrix4d value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFMatrix4d) and not isinstance(value, MFMatrix4d):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFMatrix4d')
    if isinstance(value, MFMatrix4d):
        value = value.value # dereference value from this base type
    if isinstance(value, SFMatrix4d) or not isinstance(value, list):
        value = [value.value] # dereference value from this SF type, convert to list #2
    if not isinstance(value, list):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFMatrix4d')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #2
    if isinstance(value, list):
        _index = 0
        for each in value:
            if len(each) % MFMatrix4d().TUPLE_SIZE() != 0:
                # print(flush=True)
                raise X3DValueError('MFMatrix4d tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFMatrix4d().TUPLE_SIZE()=' + str(MFMatrix4d().TUPLE_SIZE()) + ' for value=' + str(value)[:100])
#            if not isinstance(each, (tuple, SFMatrix4d)):
#                # print(flush=True)
#                raise X3DTypeError('MFMatrix4d element #' + str(_index) + ' with value ' + str(each) + ', type=' + str(type(each)) + ' is not a valid tuple')
            _index += 1
            if isinstance(each, tuple):
                for element in each:
                    if not isinstance(element, float) and not isinstance(element, int):
                        # print(flush=True)
                        raise X3DTypeError('MFMatrix4d element #' + str(_index) + ' tuple ' + str(each) + ' has value=' + str(element) + ', type=' + str(type(element)) + ' that is not a valid float')
    if not isValidMFMatrix4d(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFMatrix4d')
    return True

def isValidSFMatrix4f(value):
    """
    Utility function to determine type validity of a SFMatrix4f value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFMatrix4f) and not isinstance(value, MFMatrix4f):
            # if _DEBUG: print('SFMatrix4f type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFMatrix4f)=' + str(isinstance(value, SFMatrix4f)), flush=True)
            return False # type mismatch!
    if isinstance(value, SFMatrix4f):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, MFMatrix4f) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
        return True
    if not isinstance(value, tuple):
        return False
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFMatrix4f):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            return False
    if tupleCount != 16:
        return False
    return True

def assertValidSFMatrix4f(value):
    """
    Utility function to assert type validity of a SFMatrix4f value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFMatrix4f) and not isinstance(value, MFMatrix4f):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFMatrix4f')
    if isinstance(value, SFMatrix4f):
        value = value.value # dereference value from this base type
    if isinstance(value, MFMatrix4f) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
    if not isinstance(value, tuple):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFMatrix4f')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFMatrix4f):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            # print(flush=True)
            raise X3DTypeError('SFMatrix4f list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    if tupleCount != 16:
        # print(flush=True)
        raise X3DTypeError('SFMatrix4f ' + str(value)[:100] + ', type=' + str(type(value)) + ' has ' + str(tupleCount) + ' elements instead of 16')
    if not isValidSFMatrix4f(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFMatrix4f')
    return True

def isValidMFMatrix4f(value):
    """
    Utility function to determine type validity of a MFMatrix4f value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFMatrix4f) and not isinstance(value, MFMatrix4f):
            # if _DEBUG: print('MFMatrix4f type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFMatrix4f)=' + str(isinstance(value, MFMatrix4f)), flush=True)
            return False # type mismatch!
    if isinstance(value, MFMatrix4f):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, SFMatrix4f):
        value = [value.value] # dereference value from this SF type, convert to list #1
        return True
    if not isinstance(value, list):
        return False
    _index = 0
    for each in value:
        _index += 1
        if len(each) % MFMatrix4f().TUPLE_SIZE() != 0:
            # if _DEBUG:
            print('* isValidMFMatrix4f tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFMatrix4f().TUPLE_SIZE()=' + str(MFMatrix4f().TUPLE_SIZE()) + ' for value=' + str(value), flush=True)
            return False
        for element in each:
            if not isinstance(element, float) and not isinstance(element, int):
                return False
    return True

def assertValidMFMatrix4f(value):
    """
    Utility function to assert type validity of a MFMatrix4f value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFMatrix4f) and not isinstance(value, MFMatrix4f):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFMatrix4f')
    if isinstance(value, MFMatrix4f):
        value = value.value # dereference value from this base type
    if isinstance(value, SFMatrix4f) or not isinstance(value, list):
        value = [value.value] # dereference value from this SF type, convert to list #2
    if not isinstance(value, list):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFMatrix4f')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #2
    if isinstance(value, list):
        _index = 0
        for each in value:
            if len(each) % MFMatrix4f().TUPLE_SIZE() != 0:
                # print(flush=True)
                raise X3DValueError('MFMatrix4f tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFMatrix4f().TUPLE_SIZE()=' + str(MFMatrix4f().TUPLE_SIZE()) + ' for value=' + str(value)[:100])
#            if not isinstance(each, (tuple, SFMatrix4f)):
#                # print(flush=True)
#                raise X3DTypeError('MFMatrix4f element #' + str(_index) + ' with value ' + str(each) + ', type=' + str(type(each)) + ' is not a valid tuple')
            _index += 1
            if isinstance(each, tuple):
                for element in each:
                    if not isinstance(element, float) and not isinstance(element, int):
                        # print(flush=True)
                        raise X3DTypeError('MFMatrix4f element #' + str(_index) + ' tuple ' + str(each) + ' has value=' + str(element) + ', type=' + str(type(element)) + ' that is not a valid float')
    if not isValidMFMatrix4f(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFMatrix4f')
    return True

def isValidSFNode(value):
    """
    Utility function to determine type validity of a SFNode value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFNode) and not isinstance(value, MFNode):
            # if _DEBUG: print('SFNode type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFNode)=' + str(isinstance(value, SFNode)), flush=True)
            return False # type mismatch!
    if isinstance(value, SFNode):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, MFNode) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
        return True
    if not isinstance(value, object):
        return False
    return True

def assertValidSFNode(value):
    """
    Utility function to assert type validity of a SFNode value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFNode) and not isinstance(value, MFNode):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFNode')
    if isinstance(value, SFNode):
        value = value.value # dereference value from this base type
    if isinstance(value, MFNode) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
    if not isinstance(value, object):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid object value for SFNode')
    if not isValidSFNode(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python object value for SFNode')
    return True

def isValidMFNode(value):
    """
    Utility function to determine type validity of a MFNode value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFNode) and not isinstance(value, MFNode):
            # if _DEBUG: print('MFNode type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFNode)=' + str(isinstance(value, MFNode)), flush=True)
            return False # type mismatch!
    if isinstance(value, MFNode):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, SFNode):
        value = [value.value] # dereference value from this SF type, convert to list #1
        return True
    for each in value:
        if not isinstance(each, (_X3DNode, _X3DStatement)):
            return False
    return True

def assertValidMFNode(value):
    """
    Utility function to assert type validity of a MFNode value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFNode) and not isinstance(value, MFNode):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFNode')
    if isinstance(value, MFNode):
        value = value.value # dereference value from this base type
    if isinstance(value, SFNode) or not isinstance(value, list):
        value = [value.value] # dereference value from this SF type, convert to list #2
    for each in value:
        if not isinstance(each, _X3DNode) and not isinstance(each, _X3DStatement):
            # print(flush=True)
            raise X3DTypeError(str(value)[:100] + ' element ' + str(each) + ', type=' + str(type(each)) + ' is not a valid _X3DNode or _X3DStatement for MFNode')
    if not isValidMFNode(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid _X3DNode or _X3DStatement for MFNode')
    return True

def isValidSFRotation(value):
    """
    Utility function to determine type validity of a SFRotation value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFRotation) and not isinstance(value, MFRotation):
            # if _DEBUG: print('SFRotation type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFRotation)=' + str(isinstance(value, SFRotation)), flush=True)
            return False # type mismatch!
    if isinstance(value, SFRotation):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, MFRotation) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
        return True
    if not isinstance(value, tuple):
        return False
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFRotation):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            return False
    if tupleCount != 4:
        return False
    return True

def assertValidSFRotation(value):
    """
    Utility function to assert type validity of a SFRotation value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFRotation) and not isinstance(value, MFRotation):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFRotation')
    if isinstance(value, SFRotation):
        value = value.value # dereference value from this base type
    if isinstance(value, MFRotation) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
    if not isinstance(value, tuple):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFRotation')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFRotation):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            # print(flush=True)
            raise X3DTypeError('SFRotation list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    if tupleCount != 4:
        # print(flush=True)
        raise X3DTypeError('SFRotation ' + str(value)[:100] + ', type=' + str(type(value)) + ' has ' + str(tupleCount) + ' elements instead of 4')
    if not isValidSFRotation(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFRotation')
    return True

def isValidMFRotation(value):
    """
    Utility function to determine type validity of a MFRotation value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFRotation) and not isinstance(value, MFRotation):
            # if _DEBUG: print('MFRotation type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFRotation)=' + str(isinstance(value, MFRotation)), flush=True)
            return False # type mismatch!
    if isinstance(value, MFRotation):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, SFRotation):
        value = [value.value] # dereference value from this SF type, convert to list #1
        return True
    if not isinstance(value, list):
        return False
    _index = 0
    for each in value:
        _index += 1
        if len(each) % MFRotation().TUPLE_SIZE() != 0:
            # if _DEBUG:
            print('* isValidMFRotation tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFRotation().TUPLE_SIZE()=' + str(MFRotation().TUPLE_SIZE()) + ' for value=' + str(value), flush=True)
            return False
        for element in each:
            if not isinstance(element, float) and not isinstance(element, int):
                return False
    return True

def assertValidMFRotation(value):
    """
    Utility function to assert type validity of a MFRotation value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFRotation) and not isinstance(value, MFRotation):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFRotation')
    if isinstance(value, MFRotation):
        value = value.value # dereference value from this base type
    if isinstance(value, SFRotation) or not isinstance(value, list):
        value = [value.value] # dereference value from this SF type, convert to list #2
    if not isinstance(value, list):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFRotation')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #2
    if isinstance(value, list):
        _index = 0
        for each in value:
            if len(each) % MFRotation().TUPLE_SIZE() != 0:
                # print(flush=True)
                raise X3DValueError('MFRotation tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFRotation().TUPLE_SIZE()=' + str(MFRotation().TUPLE_SIZE()) + ' for value=' + str(value)[:100])
#            if not isinstance(each, (tuple, SFRotation)):
#                # print(flush=True)
#                raise X3DTypeError('MFRotation element #' + str(_index) + ' with value ' + str(each) + ', type=' + str(type(each)) + ' is not a valid tuple')
            _index += 1
            if isinstance(each, tuple):
                for element in each:
                    if not isinstance(element, float) and not isinstance(element, int):
                        # print(flush=True)
                        raise X3DTypeError('MFRotation element #' + str(_index) + ' tuple ' + str(each) + ' has value=' + str(element) + ', type=' + str(type(element)) + ' that is not a valid float')
    if not isValidMFRotation(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFRotation')
    return True

def isValidSFString(value):
    """
    Utility function to determine type validity of a SFString value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFString) and not isinstance(value, MFString):
            # if _DEBUG: print('SFString type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFString)=' + str(isinstance(value, SFString)), flush=True)
            return False # type mismatch!
    if isinstance(value, SFString):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, MFString) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
        return True
    if not isinstance(value, str):
        return False
    return True

def assertValidSFString(value):
    """
    Utility function to assert type validity of a SFString value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFString) and not isinstance(value, MFString):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFString')
    if isinstance(value, SFString):
        value = value.value # dereference value from this base type
    if isinstance(value, MFString) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
    if not isinstance(value, str):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid str value for SFString')
    if not isValidSFString(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python str value for SFString')
    return True

def isValidMFString(value):
    """
    Utility function to determine type validity of a MFString value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFString) and not isinstance(value, MFString):
            # if _DEBUG: print('MFString type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFString)=' + str(isinstance(value, MFString)), flush=True)
            return False # type mismatch!
    if isinstance(value, MFString):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, SFString):
        value = [value.value] # dereference value from this SF type, convert to list #1
        return True
    if not isinstance(value, list):
        return False
    for each in value:
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFString):
            each = each.value # dereference
        if not isinstance(each, str):
            return False
    return True

def assertValidMFString(value):
    """
    Utility function to assert type validity of a MFString value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFString) and not isinstance(value, MFString):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFString')
    if isinstance(value, MFString):
        value = value.value # dereference value from this base type
    if isinstance(value, SFString) or not isinstance(value, list):
        value = [value.value] # dereference value from this SF type, convert to list #2
    if not isinstance(value, list):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFString')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    for each in value:
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFString):
            each = each.value # dereference
        if not isinstance(each, str):
            # print(flush=True)
            raise X3DTypeError('MFString list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid str')
    if not isValidMFString(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFString')
    return True

def isValidSFTime(value):
    """
    Utility function to determine type validity of a SFTime value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFTime) and not isinstance(value, MFTime):
            # if _DEBUG: print('SFTime type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFTime)=' + str(isinstance(value, SFTime)), flush=True)
            return False # type mismatch!
    if isinstance(value, SFTime):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, MFTime) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
        return True
    if not isinstance(value, float) and not isinstance(value, int):
        return False
    return True

def assertValidSFTime(value):
    """
    Utility function to assert type validity of a SFTime value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFTime) and not isinstance(value, MFTime):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFTime')
    if isinstance(value, SFTime):
        value = value.value # dereference value from this base type
    if isinstance(value, MFTime) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
    if not isinstance(value, float) and not isinstance(value, int):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid float value for SFTime')
    if not isValidSFTime(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python float value for SFTime')
    return True

def isValidMFTime(value):
    """
    Utility function to determine type validity of a MFTime value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFTime) and not isinstance(value, MFTime):
            # if _DEBUG: print('MFTime type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFTime)=' + str(isinstance(value, MFTime)), flush=True)
            return False # type mismatch!
    if isinstance(value, MFTime):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, SFTime):
        value = [value.value] # dereference value from this SF type, convert to list #1
        return True
    if not isinstance(value, list):
        return False
    for each in value:
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFTime):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            return False
    return True

def assertValidMFTime(value):
    """
    Utility function to assert type validity of a MFTime value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFTime) and not isinstance(value, MFTime):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFTime')
    if isinstance(value, MFTime):
        value = value.value # dereference value from this base type
    if isinstance(value, SFTime) or not isinstance(value, list):
        value = [value.value] # dereference value from this SF type, convert to list #2
    if not isinstance(value, list):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFTime')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    for each in value:
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFTime):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            # print(flush=True)
            raise X3DTypeError('MFTime list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    if not isValidMFTime(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFTime')
    return True

def isValidSFVec2d(value):
    """
    Utility function to determine type validity of a SFVec2d value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFVec2d) and not isinstance(value, MFVec2d):
            # if _DEBUG: print('SFVec2d type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFVec2d)=' + str(isinstance(value, SFVec2d)), flush=True)
            return False # type mismatch!
    if isinstance(value, SFVec2d):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, MFVec2d) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
        return True
    if not isinstance(value, tuple):
        return False
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFVec2d):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            return False
    if tupleCount != 2:
        return False
    return True

def assertValidSFVec2d(value):
    """
    Utility function to assert type validity of a SFVec2d value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFVec2d) and not isinstance(value, MFVec2d):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFVec2d')
    if isinstance(value, SFVec2d):
        value = value.value # dereference value from this base type
    if isinstance(value, MFVec2d) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
    if not isinstance(value, tuple):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFVec2d')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFVec2d):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            # print(flush=True)
            raise X3DTypeError('SFVec2d list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    if tupleCount != 2:
        # print(flush=True)
        raise X3DTypeError('SFVec2d ' + str(value)[:100] + ', type=' + str(type(value)) + ' has ' + str(tupleCount) + ' elements instead of 2')
    if not isValidSFVec2d(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFVec2d')
    return True

def isValidMFVec2d(value):
    """
    Utility function to determine type validity of a MFVec2d value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFVec2d) and not isinstance(value, MFVec2d):
            # if _DEBUG: print('MFVec2d type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFVec2d)=' + str(isinstance(value, MFVec2d)), flush=True)
            return False # type mismatch!
    if isinstance(value, MFVec2d):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, SFVec2d):
        value = [value.value] # dereference value from this SF type, convert to list #1
        return True
    if not isinstance(value, list):
        return False
    _index = 0
    for each in value:
        _index += 1
        if len(each) % MFVec2d().TUPLE_SIZE() != 0:
            # if _DEBUG:
            print('* isValidMFVec2d tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFVec2d().TUPLE_SIZE()=' + str(MFVec2d().TUPLE_SIZE()) + ' for value=' + str(value), flush=True)
            return False
        for element in each:
            if not isinstance(element, float) and not isinstance(element, int):
                return False
    return True

def assertValidMFVec2d(value):
    """
    Utility function to assert type validity of a MFVec2d value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFVec2d) and not isinstance(value, MFVec2d):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFVec2d')
    if isinstance(value, MFVec2d):
        value = value.value # dereference value from this base type
    if isinstance(value, SFVec2d) or not isinstance(value, list):
        value = [value.value] # dereference value from this SF type, convert to list #2
    if not isinstance(value, list):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFVec2d')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #2
    if isinstance(value, list):
        _index = 0
        for each in value:
            if len(each) % MFVec2d().TUPLE_SIZE() != 0:
                # print(flush=True)
                raise X3DValueError('MFVec2d tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFVec2d().TUPLE_SIZE()=' + str(MFVec2d().TUPLE_SIZE()) + ' for value=' + str(value)[:100])
#            if not isinstance(each, (tuple, SFVec2d)):
#                # print(flush=True)
#                raise X3DTypeError('MFVec2d element #' + str(_index) + ' with value ' + str(each) + ', type=' + str(type(each)) + ' is not a valid tuple')
            _index += 1
            if isinstance(each, tuple):
                for element in each:
                    if not isinstance(element, float) and not isinstance(element, int):
                        # print(flush=True)
                        raise X3DTypeError('MFVec2d element #' + str(_index) + ' tuple ' + str(each) + ' has value=' + str(element) + ', type=' + str(type(element)) + ' that is not a valid float')
    if not isValidMFVec2d(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFVec2d')
    return True

def isValidSFVec2f(value):
    """
    Utility function to determine type validity of a SFVec2f value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFVec2f) and not isinstance(value, MFVec2f):
            # if _DEBUG: print('SFVec2f type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFVec2f)=' + str(isinstance(value, SFVec2f)), flush=True)
            return False # type mismatch!
    if isinstance(value, SFVec2f):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, MFVec2f) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
        return True
    if not isinstance(value, tuple):
        return False
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFVec2f):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            return False
    if tupleCount != 2:
        return False
    return True

def assertValidSFVec2f(value):
    """
    Utility function to assert type validity of a SFVec2f value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFVec2f) and not isinstance(value, MFVec2f):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFVec2f')
    if isinstance(value, SFVec2f):
        value = value.value # dereference value from this base type
    if isinstance(value, MFVec2f) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
    if not isinstance(value, tuple):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFVec2f')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFVec2f):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            # print(flush=True)
            raise X3DTypeError('SFVec2f list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    if tupleCount != 2:
        # print(flush=True)
        raise X3DTypeError('SFVec2f ' + str(value)[:100] + ', type=' + str(type(value)) + ' has ' + str(tupleCount) + ' elements instead of 2')
    if not isValidSFVec2f(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFVec2f')
    return True

def isValidMFVec2f(value):
    """
    Utility function to determine type validity of a MFVec2f value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFVec2f) and not isinstance(value, MFVec2f):
            # if _DEBUG: print('MFVec2f type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFVec2f)=' + str(isinstance(value, MFVec2f)), flush=True)
            return False # type mismatch!
    if isinstance(value, MFVec2f):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, SFVec2f):
        value = [value.value] # dereference value from this SF type, convert to list #1
        return True
    if not isinstance(value, list):
        return False
    _index = 0
    for each in value:
        _index += 1
        if len(each) % MFVec2f().TUPLE_SIZE() != 0:
            # if _DEBUG:
            print('* isValidMFVec2f tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFVec2f().TUPLE_SIZE()=' + str(MFVec2f().TUPLE_SIZE()) + ' for value=' + str(value), flush=True)
            return False
        for element in each:
            if not isinstance(element, float) and not isinstance(element, int):
                return False
    return True

def assertValidMFVec2f(value):
    """
    Utility function to assert type validity of a MFVec2f value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFVec2f) and not isinstance(value, MFVec2f):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFVec2f')
    if isinstance(value, MFVec2f):
        value = value.value # dereference value from this base type
    if isinstance(value, SFVec2f) or not isinstance(value, list):
        value = [value.value] # dereference value from this SF type, convert to list #2
    if not isinstance(value, list):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFVec2f')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #2
    if isinstance(value, list):
        _index = 0
        for each in value:
            if len(each) % MFVec2f().TUPLE_SIZE() != 0:
                # print(flush=True)
                raise X3DValueError('MFVec2f tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFVec2f().TUPLE_SIZE()=' + str(MFVec2f().TUPLE_SIZE()) + ' for value=' + str(value)[:100])
#            if not isinstance(each, (tuple, SFVec2f)):
#                # print(flush=True)
#                raise X3DTypeError('MFVec2f element #' + str(_index) + ' with value ' + str(each) + ', type=' + str(type(each)) + ' is not a valid tuple')
            _index += 1
            if isinstance(each, tuple):
                for element in each:
                    if not isinstance(element, float) and not isinstance(element, int):
                        # print(flush=True)
                        raise X3DTypeError('MFVec2f element #' + str(_index) + ' tuple ' + str(each) + ' has value=' + str(element) + ', type=' + str(type(element)) + ' that is not a valid float')
    if not isValidMFVec2f(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFVec2f')
    return True

def isValidSFVec3d(value):
    """
    Utility function to determine type validity of a SFVec3d value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFVec3d) and not isinstance(value, MFVec3d):
            # if _DEBUG: print('SFVec3d type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFVec3d)=' + str(isinstance(value, SFVec3d)), flush=True)
            return False # type mismatch!
    if isinstance(value, SFVec3d):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, MFVec3d) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
        return True
    if not isinstance(value, tuple):
        return False
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFVec3d):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            return False
    if tupleCount != 3:
        return False
    return True

def assertValidSFVec3d(value):
    """
    Utility function to assert type validity of a SFVec3d value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFVec3d) and not isinstance(value, MFVec3d):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFVec3d')
    if isinstance(value, SFVec3d):
        value = value.value # dereference value from this base type
    if isinstance(value, MFVec3d) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
    if not isinstance(value, tuple):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFVec3d')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFVec3d):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            # print(flush=True)
            raise X3DTypeError('SFVec3d list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    if tupleCount != 3:
        # print(flush=True)
        raise X3DTypeError('SFVec3d ' + str(value)[:100] + ', type=' + str(type(value)) + ' has ' + str(tupleCount) + ' elements instead of 3')
    if not isValidSFVec3d(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFVec3d')
    return True

def isValidMFVec3d(value):
    """
    Utility function to determine type validity of a MFVec3d value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFVec3d) and not isinstance(value, MFVec3d):
            # if _DEBUG: print('MFVec3d type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFVec3d)=' + str(isinstance(value, MFVec3d)), flush=True)
            return False # type mismatch!
    if isinstance(value, MFVec3d):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, SFVec3d):
        value = [value.value] # dereference value from this SF type, convert to list #1
        return True
    if not isinstance(value, list):
        return False
    _index = 0
    for each in value:
        _index += 1
        if len(each) % MFVec3d().TUPLE_SIZE() != 0:
            # if _DEBUG:
            print('* isValidMFVec3d tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFVec3d().TUPLE_SIZE()=' + str(MFVec3d().TUPLE_SIZE()) + ' for value=' + str(value), flush=True)
            return False
        for element in each:
            if not isinstance(element, float) and not isinstance(element, int):
                return False
    return True

def assertValidMFVec3d(value):
    """
    Utility function to assert type validity of a MFVec3d value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFVec3d) and not isinstance(value, MFVec3d):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFVec3d')
    if isinstance(value, MFVec3d):
        value = value.value # dereference value from this base type
    if isinstance(value, SFVec3d) or not isinstance(value, list):
        value = [value.value] # dereference value from this SF type, convert to list #2
    if not isinstance(value, list):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFVec3d')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #2
    if isinstance(value, list):
        _index = 0
        for each in value:
            if len(each) % MFVec3d().TUPLE_SIZE() != 0:
                # print(flush=True)
                raise X3DValueError('MFVec3d tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFVec3d().TUPLE_SIZE()=' + str(MFVec3d().TUPLE_SIZE()) + ' for value=' + str(value)[:100])
#            if not isinstance(each, (tuple, SFVec3d)):
#                # print(flush=True)
#                raise X3DTypeError('MFVec3d element #' + str(_index) + ' with value ' + str(each) + ', type=' + str(type(each)) + ' is not a valid tuple')
            _index += 1
            if isinstance(each, tuple):
                for element in each:
                    if not isinstance(element, float) and not isinstance(element, int):
                        # print(flush=True)
                        raise X3DTypeError('MFVec3d element #' + str(_index) + ' tuple ' + str(each) + ' has value=' + str(element) + ', type=' + str(type(element)) + ' that is not a valid float')
    if not isValidMFVec3d(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFVec3d')
    return True

def isValidSFVec3f(value):
    """
    Utility function to determine type validity of a SFVec3f value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFVec3f) and not isinstance(value, MFVec3f):
            # if _DEBUG: print('SFVec3f type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFVec3f)=' + str(isinstance(value, SFVec3f)), flush=True)
            return False # type mismatch!
    if isinstance(value, SFVec3f):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, MFVec3f) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
        return True
    if not isinstance(value, tuple):
        return False
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFVec3f):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            return False
    if tupleCount != 3:
        return False
    return True

def assertValidSFVec3f(value):
    """
    Utility function to assert type validity of a SFVec3f value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFVec3f) and not isinstance(value, MFVec3f):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFVec3f')
    if isinstance(value, SFVec3f):
        value = value.value # dereference value from this base type
    if isinstance(value, MFVec3f) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
    if not isinstance(value, tuple):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFVec3f')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFVec3f):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            # print(flush=True)
            raise X3DTypeError('SFVec3f list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    if tupleCount != 3:
        # print(flush=True)
        raise X3DTypeError('SFVec3f ' + str(value)[:100] + ', type=' + str(type(value)) + ' has ' + str(tupleCount) + ' elements instead of 3')
    if not isValidSFVec3f(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFVec3f')
    return True

def isValidMFVec3f(value):
    """
    Utility function to determine type validity of a MFVec3f value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFVec3f) and not isinstance(value, MFVec3f):
            # if _DEBUG: print('MFVec3f type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFVec3f)=' + str(isinstance(value, MFVec3f)), flush=True)
            return False # type mismatch!
    if isinstance(value, MFVec3f):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, SFVec3f):
        value = [value.value] # dereference value from this SF type, convert to list #1
        return True
    if not isinstance(value, list):
        return False
    _index = 0
    for each in value:
        _index += 1
        if len(each) % MFVec3f().TUPLE_SIZE() != 0:
            # if _DEBUG:
            print('* isValidMFVec3f tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFVec3f().TUPLE_SIZE()=' + str(MFVec3f().TUPLE_SIZE()) + ' for value=' + str(value), flush=True)
            return False
        for element in each:
            if not isinstance(element, float) and not isinstance(element, int):
                return False
    return True

def assertValidMFVec3f(value):
    """
    Utility function to assert type validity of a MFVec3f value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFVec3f) and not isinstance(value, MFVec3f):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFVec3f')
    if isinstance(value, MFVec3f):
        value = value.value # dereference value from this base type
    if isinstance(value, SFVec3f) or not isinstance(value, list):
        value = [value.value] # dereference value from this SF type, convert to list #2
    if not isinstance(value, list):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFVec3f')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #2
    if isinstance(value, list):
        _index = 0
        for each in value:
            if len(each) % MFVec3f().TUPLE_SIZE() != 0:
                # print(flush=True)
                raise X3DValueError('MFVec3f tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFVec3f().TUPLE_SIZE()=' + str(MFVec3f().TUPLE_SIZE()) + ' for value=' + str(value)[:100])
#            if not isinstance(each, (tuple, SFVec3f)):
#                # print(flush=True)
#                raise X3DTypeError('MFVec3f element #' + str(_index) + ' with value ' + str(each) + ', type=' + str(type(each)) + ' is not a valid tuple')
            _index += 1
            if isinstance(each, tuple):
                for element in each:
                    if not isinstance(element, float) and not isinstance(element, int):
                        # print(flush=True)
                        raise X3DTypeError('MFVec3f element #' + str(_index) + ' tuple ' + str(each) + ' has value=' + str(element) + ', type=' + str(type(element)) + ' that is not a valid float')
    if not isValidMFVec3f(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFVec3f')
    return True

def isValidSFVec4d(value):
    """
    Utility function to determine type validity of a SFVec4d value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFVec4d) and not isinstance(value, MFVec4d):
            # if _DEBUG: print('SFVec4d type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFVec4d)=' + str(isinstance(value, SFVec4d)), flush=True)
            return False # type mismatch!
    if isinstance(value, SFVec4d):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, MFVec4d) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
        return True
    if not isinstance(value, tuple):
        return False
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFVec4d):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            return False
    if tupleCount != 4:
        return False
    return True

def assertValidSFVec4d(value):
    """
    Utility function to assert type validity of a SFVec4d value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFVec4d) and not isinstance(value, MFVec4d):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFVec4d')
    if isinstance(value, SFVec4d):
        value = value.value # dereference value from this base type
    if isinstance(value, MFVec4d) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
    if not isinstance(value, tuple):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFVec4d')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFVec4d):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            # print(flush=True)
            raise X3DTypeError('SFVec4d list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    if tupleCount != 4:
        # print(flush=True)
        raise X3DTypeError('SFVec4d ' + str(value)[:100] + ', type=' + str(type(value)) + ' has ' + str(tupleCount) + ' elements instead of 4')
    if not isValidSFVec4d(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFVec4d')
    return True

def isValidMFVec4d(value):
    """
    Utility function to determine type validity of a MFVec4d value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFVec4d) and not isinstance(value, MFVec4d):
            # if _DEBUG: print('MFVec4d type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFVec4d)=' + str(isinstance(value, MFVec4d)), flush=True)
            return False # type mismatch!
    if isinstance(value, MFVec4d):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, SFVec4d):
        value = [value.value] # dereference value from this SF type, convert to list #1
        return True
    if not isinstance(value, list):
        return False
    _index = 0
    for each in value:
        _index += 1
        if len(each) % MFVec4d().TUPLE_SIZE() != 0:
            # if _DEBUG:
            print('* isValidMFVec4d tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFVec4d().TUPLE_SIZE()=' + str(MFVec4d().TUPLE_SIZE()) + ' for value=' + str(value), flush=True)
            return False
        for element in each:
            if not isinstance(element, float) and not isinstance(element, int):
                return False
    return True

def assertValidMFVec4d(value):
    """
    Utility function to assert type validity of a MFVec4d value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFVec4d) and not isinstance(value, MFVec4d):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFVec4d')
    if isinstance(value, MFVec4d):
        value = value.value # dereference value from this base type
    if isinstance(value, SFVec4d) or not isinstance(value, list):
        value = [value.value] # dereference value from this SF type, convert to list #2
    if not isinstance(value, list):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFVec4d')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #2
    if isinstance(value, list):
        _index = 0
        for each in value:
            if len(each) % MFVec4d().TUPLE_SIZE() != 0:
                # print(flush=True)
                raise X3DValueError('MFVec4d tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFVec4d().TUPLE_SIZE()=' + str(MFVec4d().TUPLE_SIZE()) + ' for value=' + str(value)[:100])
#            if not isinstance(each, (tuple, SFVec4d)):
#                # print(flush=True)
#                raise X3DTypeError('MFVec4d element #' + str(_index) + ' with value ' + str(each) + ', type=' + str(type(each)) + ' is not a valid tuple')
            _index += 1
            if isinstance(each, tuple):
                for element in each:
                    if not isinstance(element, float) and not isinstance(element, int):
                        # print(flush=True)
                        raise X3DTypeError('MFVec4d element #' + str(_index) + ' tuple ' + str(each) + ' has value=' + str(element) + ', type=' + str(type(element)) + ' that is not a valid float')
    if not isValidMFVec4d(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFVec4d')
    return True

def isValidSFVec4f(value):
    """
    Utility function to determine type validity of a SFVec4f value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFVec4f) and not isinstance(value, MFVec4f):
            # if _DEBUG: print('SFVec4f type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, SFVec4f)=' + str(isinstance(value, SFVec4f)), flush=True)
            return False # type mismatch!
    if isinstance(value, SFVec4f):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, MFVec4f) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
        return True
    if not isinstance(value, tuple):
        return False
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFVec4f):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            return False
    if tupleCount != 4:
        return False
    return True

def assertValidSFVec4f(value):
    """
    Utility function to assert type validity of a SFVec4f value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFVec4f) and not isinstance(value, MFVec4f):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a SFVec4f')
    if isinstance(value, SFVec4f):
        value = value.value # dereference value from this base type
    if isinstance(value, MFVec4f) and len(value) == 1:
        value = value.value[0] # dereference value from this MF type
    if not isinstance(value, tuple):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFVec4f')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #1
    tupleCount = 0
    for each in value:
        tupleCount += 1
        while isinstance(each, list) and len(each) == 1:
            each = each[0] # dereference
        if isinstance(each, SFVec4f):
            each = each.value # dereference
        if not isinstance(each, float) and not isinstance(each, int):
            # print(flush=True)
            raise X3DTypeError('SFVec4f list has contained value=' + str(each) + ' with type=' + str(type(each)) + ' which is not a valid float')
    if tupleCount != 4:
        # print(flush=True)
        raise X3DTypeError('SFVec4f ' + str(value)[:100] + ', type=' + str(type(value)) + ' has ' + str(tupleCount) + ' elements instead of 4')
    if not isValidSFVec4f(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python tuple for SFVec4f')
    return True

def isValidMFVec4f(value):
    """
    Utility function to determine type validity of a MFVec4f value.
    """
    if isinstance(value, _X3DField):
        if not isinstance(value, SFVec4f) and not isinstance(value, MFVec4f):
            # if _DEBUG: print('MFVec4f type mismatch diagnostic: value=' + str(value)[:100] + ' has type=' + str(type(value)) + ', isinstance(value, MFVec4f)=' + str(isinstance(value, MFVec4f)), flush=True)
            return False # type mismatch!
    if isinstance(value, MFVec4f):
        value = value.value # dereference value from base type
        return True
    if isinstance(value, SFVec4f):
        value = [value.value] # dereference value from this SF type, convert to list #1
        return True
    if not isinstance(value, list):
        return False
    _index = 0
    for each in value:
        _index += 1
        if len(each) % MFVec4f().TUPLE_SIZE() != 0:
            # if _DEBUG:
            print('* isValidMFVec4f tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFVec4f().TUPLE_SIZE()=' + str(MFVec4f().TUPLE_SIZE()) + ' for value=' + str(value), flush=True)
            return False
        for element in each:
            if not isinstance(element, float) and not isinstance(element, int):
                return False
    return True

def assertValidMFVec4f(value):
    """
    Utility function to assert type validity of a MFVec4f value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* debug value.__class__=' + str(value.__class__) + ', issubclass(value.__class__, _X3DField)=' + str(issubclass(value.__class__, _X3DField)) + ', super(value.__class__)=' + str(super(value.__class__)), flush=True)
    if isinstance(value, _X3DField) and not isinstance(value, SFVec4f) and not isinstance(value, MFVec4f):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ' has type ' + str(type(value)) + ' and is not a MFVec4f')
    if isinstance(value, MFVec4f):
        value = value.value # dereference value from this base type
    if isinstance(value, SFVec4f) or not isinstance(value, list):
        value = [value.value] # dereference value from this SF type, convert to list #2
    if not isinstance(value, list):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFVec4f')
    # perform duplicative tests prior to isValid call in order to provide better assertion diagnostics #2
    if isinstance(value, list):
        _index = 0
        for each in value:
            if len(each) % MFVec4f().TUPLE_SIZE() != 0:
                # print(flush=True)
                raise X3DValueError('MFVec4f tuple ' + str(each) + ' has length ' + str(len(each)) + ' which is not a multiple of MFVec4f().TUPLE_SIZE()=' + str(MFVec4f().TUPLE_SIZE()) + ' for value=' + str(value)[:100])
#            if not isinstance(each, (tuple, SFVec4f)):
#                # print(flush=True)
#                raise X3DTypeError('MFVec4f element #' + str(_index) + ' with value ' + str(each) + ', type=' + str(type(each)) + ' is not a valid tuple')
            _index += 1
            if isinstance(each, tuple):
                for element in each:
                    if not isinstance(element, float) and not isinstance(element, int):
                        # print(flush=True)
                        raise X3DTypeError('MFVec4f element #' + str(_index) + ' tuple ' + str(each) + ' has value=' + str(element) + ', type=' + str(type(element)) + ' that is not a valid float')
    if not isValidMFVec4f(value):
        # print(flush=True)
        raise X3DTypeError(str(value)[:100] + ', type=' + str(type(value)) + ' is not a valid Python list for MFVec4f')
    return True


def assertValidFieldInitializationValue(name, fieldType, value, parent=''):
    """
    Utility function to assert fieldType validity of a field initialization value, otherwise raise X3DTypeError with diagnostic message.
    """
    # if _DEBUG: print('* assertValidFieldInitializationValue name=' + str(name) + ', fieldType=' + str(fieldType) + ', value=' + str(value)[:100] + ', parent=' + parent, flush=True)
    # note that ExternProtoDeclare field definitions do not have any value
    if name is None:
        print('* assertValidFieldInitializationValue improper invocation: name=' + str(name) + ', fieldType=' + str(fieldType) + ', value=' + str(value)[:100] + ', parent=' + parent + ', ignored', flush=True)
        return # ignore
    if value is None or (not(fieldType == bool) and not value):
        return # ignore
    if fieldType == 'SFString':
        assertValidSFString(value)
    elif fieldType == 'MFString':
        assertValidMFString(value)
    elif (fieldType == 'SFBool') or (fieldType == bool) or isinstance(value, bool):
        assertValidSFBool(value)
    elif fieldType == 'MFBool':
        assertValidMFBool(value)
    elif (fieldType == 'SFInt32') or (fieldType == int) or isinstance(value, int):
        assertValidSFInt32(value)
    elif fieldType == 'MFInt32':
        assertValidMFInt32(value)
    elif (fieldType == 'SFFloat') or (fieldType == float) or isinstance(value, float):
        assertValidSFFloat(value)
    elif fieldType == 'MFFloat':
        assertValidMFFloat(value)
    elif fieldType == 'SFDouble':
        assertValidSFDouble(value)
    elif fieldType == 'MFDouble':
        assertValidMFDouble(value)
    elif fieldType == 'SFTime':
        assertValidSFTime(value)
    elif fieldType == 'MFTime':
        assertValidMFTime(value)
    elif fieldType == 'SFColor':
        assertValidSFColor(value)
    elif fieldType == 'MFColorRGBA':
        assertValidMFColorRGBA(value)
    elif fieldType == 'SFRotation':
        assertValidSFRotation(value)
    elif fieldType == 'MFRotation':
        assertValidMFRotation(value)
    elif fieldType == 'SFImage':
        assertValidSFImage(value)
    elif fieldType == 'MFImage':
        assertValidMFImage(value)
    elif fieldType == 'SFNode':
        assertValidSFNode(value)
    elif fieldType == 'MFNode':
        assertValidMFNode(value)
    elif fieldType == 'SFVec2f':
        assertValidSFVec2f(value)
    elif fieldType == 'MFVec2f':
        assertValidMFVec2f(value)
    elif fieldType == 'SFVec3f':
        assertValidSFVec3f(value)
    elif fieldType == 'MFVec3f':
        assertValidMFVec3f(value)
    elif fieldType == 'SFVec4f':
        assertValidSFVec4f(value)
    elif fieldType == 'MFVec4f':
        assertValidMFVec4f(value)
    elif fieldType == 'SFVec2d':
        assertValidSFVec2d(value)
    elif fieldType == 'MFVec2d':
        assertValidMFVec2d(value)
    elif fieldType == 'SFVec3d':
        assertValidSFVec3d(value)
    elif fieldType == 'MFVec3d':
        assertValidMFVec3d(value)
    elif fieldType == 'SFVec4d':
        assertValidSFVec4d(value)
    elif fieldType == 'MFVec4d':
        assertValidMFVec4d(value)
    elif fieldType == 'SFMatrix3d':
        assertValidSFMatrix3f(value)
    elif fieldType == 'MFMatrix3f':
        assertValidMFMatrix3f(value)
    elif fieldType == 'SFMatrix4f':
        assertValidSFMatrix4f(value)
    elif fieldType == 'MFMatrix4f':
        assertValidMFMatrix4f(value)
    elif fieldType == 'SFMatrix3d':
        assertValidSFMatrix3d(value)
    elif fieldType == 'MFMatrix3d':
        assertValidMFMatrix3d(value)
    elif fieldType == 'SFMatrix4d':
        assertValidSFMatrix4d(value)
    elif fieldType == 'MFMatrix4d':
        assertValidMFMatrix4d(value)
    elif (fieldType == str)   or isinstance(value, str):
        assertValidSFString(value)
    elif str(parent) == 'fieldValue':
        return True # TODO check further if possible
    elif (fieldType == list) or isinstance(value, list):
        try:
            if isinstance(value[0], tuple):
                print('*** assertValidFieldInitializationValue TODO validate list fieldType: name=' + str(name) + ', passed fieldType=' + str(fieldType) + ', fieldType(value)=' + str(fieldType(value)) + ', value=' + str(value)[:100] + ', parent=' + parent, flush=True)
                return True # TODO check further
            initialListItemType = fieldType(value[0])
            # https://stackoverflow.com/questions/522563/accessing-the-index-in-for-loops/28072982#28072982
            # https://stackoverflow.com/questions/1952464/in-python-how-do-i-determine-if-an-object-is-iterable
            for index, each in enumerate(value):
                assertValidFieldInitializationValue(name + '[' + str(index) + ']', initialListItemType, value[index], parent)
        except TypeError:
            return True # TODO check further if possible
    elif (fieldType == tuple) or isinstance(value, tuple):
        print('*** assertValidFieldInitializationValue TODO validate tuple fieldType: name=' + str(name) + ', passed fieldType=' + str(fieldType) + ', fieldType(value)=' + str(fieldType(value)) + ', value=' + str(value)[:100] + ', parent=' + parent, flush=True)
        return True # TODO check further if possible
#       initialListItemType = fieldType(value[0])
#       for index, each in enumerate(value):
#           assertValidFieldInitializationValue(name + '[' + str(index) + '], fieldType(value[index])', value[index], parent)
    else:
        print('*** assertValidFieldInitializationValue unknown fieldType: name=' + str(name) + ', passed fieldType=' + str(fieldType) + ', fieldType(value)=' + str(fieldType(value)) + ', value=' + str(value)[:100] + ', parent=' + parent, flush=True)
###############################################

class _X3DField(object):
    """
    All X3D fields implement _X3DField abstract type.
    """
###    value = None
    def NAME(self):
        return '_X3DField'
    def SPECIFICATION_URL(self):
        return 'https://www.web3d.org/documents/specifications/19775-1/V3.3/Part01/fieldsDef.html'
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#FieldTypesTable'
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide value of this field type. """
        return self.__value
    def __repl__(self):
        # if _DEBUG: print('* debug: type(self.value)=' + str(type(self.value)), flush=True)
        if isinstance(self.value, (SFString, str)):
            return "'" + self.value + "'"
        if  isinstance(self.value, tuple) and 'SF' in str(type(self)): # avoid X3DTypeError if value is not iterable
            result = '('
            for each in self.value:
                result += str(each) + ', '
                # if _DEBUG: print('* _X3DField debug: str(each)=' + str(each), flush=True)
            return result.rstrip(', ') + ')'
        if  isinstance(self.value, list) and 'MF' in str(type(self)): # avoid X3DTypeError if value is not iterable
            # isinstance(self.value, MFNode): not working, what got passed in was not an MFNode object apparently
            result = '['
            for each in self.value:
                result += str(each) + ', '
                # if _DEBUG: print('* _X3DField debug: str(each)=' + str(each), flush=True)
            return result.rstrip(', ') + ']'
        return str(self.value)
    def __str__(self):
        return self.__repl__()

def isX3DField(value):
    """
    Determine whether object is an instance of _X3DField.
    """
    return isinstance(value, _X3DField)

# Access Types

class AccessType(_X3DField):
    """
    accessType determines whether a field corresponds to event input, event output, or persistent state information. Events are strictly typed values with a corresponding timestamp. ROUTE connections must match accessType between source field and target field.

    initializeOnly: can be initialized, but cannot send or receive events. This is usually the case for fields that are considered too computationally expensive to change at run time.
    inputOutput: can be initialized, and can also send or receive events during run-time operations.
    inputOnly: cannot be initialized or included in a scene file, but can receive input event values via a ROUTE during run-time operations.
    outputOnly: cannot be initialized or included in a scene file, but can send output event values via a ROUTE during run-time operations.
    """
    SPECIFICATION_URL = 'https://www.web3d.org/documents/specifications/19775-1/V3.3/Part01/concepts.html#FieldSemantics'
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#accessType'
    # string constants listing each allowed accessType
    initializeOnly = 'initializeOnly'
    inputOutput = 'inputOutput'
    inputOnly = 'inputOnly'
    outputOnly = 'outputOnly'


# Field Types

class FieldType(_X3DField):
    """
    The X3D Architecture specification of field types classify the possible values for a field.
    Each field in each node (i.e. each XML attribute) has a strictly defined data type.
    Multiple data types are provided for boolean, integer, floating-point and string values.
    X3D is a strongly typed language, meaning that all data must strictly conform to these data types in order for a scene to be correct.
    """
    SPECIFICATION_URL = 'https://www.web3d.org/documents/specifications/19775-1/V3.3/Part01/fieldsDef.html'
    TOOLTIP_URL = 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#type'
    # string constants listing each allowed type
    SFBool = 'SFBool'
    MFBool = 'MFBool'
    SFColor = 'SFColor'
    MFColor = 'MFColor'
    SFColorRGBA = 'SFColorRGBA'
    MFColorRGBA = 'MFColorRGBA'
    SFDouble = 'SFDouble'
    MFDouble = 'MFDouble'
    SFFloat = 'SFFloat'
    MFFloat = 'MFFloat'
    SFImage = 'SFImage'
    MFImage = 'MFImage'
    SFInt32 = 'SFInt32'
    MFInt32 = 'MFInt32'
    SFMatrix3d = 'SFMatrix3d'
    MFMatrix3d = 'MFMatrix3d'
    SFMatrix3f = 'SFMatrix3f'
    MFMatrix3f = 'MFMatrix3f'
    SFMatrix4d = 'SFMatrix4d'
    MFMatrix4d = 'MFMatrix4d'
    SFMatrix4f = 'SFMatrix4f'
    MFMatrix4f = 'MFMatrix4f'
    SFNode = 'SFNode'
    MFNode = 'MFNode'
    SFRotation = 'SFRotation'
    MFRotation = 'MFRotation'
    SFString = 'SFString'
    MFString = 'MFString'
    SFTime = 'SFTime'
    MFTime = 'MFTime'
    SFVec2d = 'SFVec2d'
    MFVec2d = 'MFVec2d'
    SFVec2f = 'SFVec2f'
    MFVec2f = 'MFVec2f'
    SFVec3d = 'SFVec3d'
    MFVec3d = 'MFVec3d'
    SFVec3f = 'SFVec3f'
    MFVec3f = 'MFVec3f'
    SFVec4d = 'SFVec4d'
    MFVec4d = 'MFVec4d'
    SFVec4f = 'SFVec4f'
    MFVec4f = 'MFVec4f'

class SFBool(_X3DField):
    """
    Field type Python Boolean values are capitalized as True or False. SFBool is a logical type with possible values (true|false) to match the XML boolean type. Hint: XML boolean values are lower case (true|false) in order to maintain compatibility with HTML and other XML documents.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'SFBool'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFBool'
    def DEFAULT_VALUE(self):
        return True
    def ARRAY_TYPE(self):
        return False
    def TUPLE_SIZE(self):
        return 1
    def REGEX_XML(self):
        return r'\s*(true|false)\s*' # (includes lower-case true, false)
    # - - - - - - - - - -
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        value = fixBoolean(value, default=SFBool.DEFAULT_VALUE(self))
        if  value is None:
            value = self.DEFAULT_VALUE()
        elif isinstance(value, MFBool) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        assertValidSFBool(value)
        self.__value = value
    def __bool__(self):
        return len(self.__value) > 0
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower()
    def VRML(self):
        """ Provide VRML value for this field type. """
        return str(self.__value).upper()

class MFBool(_X3DField):
    """
    Field type Python Boolean values are capitalized as True or False. MFBool is an array of boolean values. Type MFBool was previously undefined in the VRML97 Specification, but nevertheless needed for event utilities and scripting. Example use: MFBool is useful for defining a series of behavior states using a BooleanSequencer prototype. Hint: XML boolean values are lower case (true|false) in order to maintain compatibility with HTML and other XML documents.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'MFBool'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFBool'
    def DEFAULT_VALUE(self):
        return [] # use empty list object, don't keep resetting a mutable python DEFAULT_VALUE
    def ARRAY_TYPE(self):
        return True
    def TUPLE_SIZE(self):
        return 1
    def REGEX_XML(self):
        return r'\s*((true|false)\s*,?\s*)*' # (includes lower-case true, false)
    # - - - - - - - - - -
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        value = fixBoolean(value, default=SFBool.DEFAULT_VALUE(self))
        if  value is None:
            value = self.DEFAULT_VALUE()
        elif not isinstance(value, list) and isValidSFBool(value):
            value = [value]
        assertValidMFBool(value)
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            if isValidSFBool(value):
                if isinstance(value, SFBool):
                    value = value.value # dereference
                self.__value.append(value)
            elif isValidMFBool(value):
                for each in value:
                    while isinstance(each, list) and len(each) == 1:
                        each = each[0] # dereference
                    if isinstance(each, SFBool):
                        each = each.value # dereference
                    self.__value.append(each)
            else:
                assertValidMFBool(value) # report type failure
    def __bool__(self):
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace(',','').replace('[','').replace(']','')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return str(self.__value).upper().replace(',','').replace('[','').replace(']','')

class SFColor(_X3DField):
    """
    Field type SFColor specifies one RGB (red-green-blue) color triple, where each color value is an RGB triple of floating point numbers in range [0,1]. The default value of an uninitialized SFColor field is (0 0 0). Warning: comma characters within singleton values do not pass strict XML validation.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'SFColor'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFColor'
    def DEFAULT_VALUE(self):
        return (0, 0, 0)
    def ARRAY_TYPE(self):
        return False
    def TUPLE_SIZE(self):
        return 3
    def REGEX_XML(self):
        return r'(\s)*(([+]?((0(\.[0-9]*)?|\.[0-9]+)|1(\.0*)?)([Ee][+-]?[0-9]+)?)(\s)+){2}([+]?((0(\.[0-9]*)?|\.[0-9]+)|1(\.0*)?)([Ee][+-]?[0-9]+)?)(\s)*' # (includes lower-case true, false)
    # - - - - - - - - - -
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE()
        elif isinstance(value, list):
            for each in value: # check that elements are not tuples
                if isinstance(each, tuple):
                    break
            else: # no tuples found, create 3-tuples
                value = [(x, y, z) for x, y, z in value]
        elif isinstance(value, MFColor) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        assertValidSFColor(value)
        self.__value = value
    def __bool__(self):
        return len(self.__value) > 0
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace(',','').replace('(','').replace(')','').replace('[','').replace(']','')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return str(self.__value).lower().replace(',','').replace('(','').replace(')','').replace('[','').replace(']','')

class MFColor(_X3DField):
    """
    Field type MFColor specifies zero or more SFColor RGB triples, where each color value is an RGB triple of floating point numbers in range [0,1]. The default value of an uninitialized MFColor field is the empty list. Individual SFColor array values are optionally separated by commas in XML syntax.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'MFColor'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFColor'
    def DEFAULT_VALUE(self):
        return [] # use empty list object, don't keep resetting a mutable python DEFAULT_VALUE
    def ARRAY_TYPE(self):
        return True
    def TUPLE_SIZE(self):
        return 3
    def REGEX_XML(self):
        return r'(\s)*((([+-]?((0(\.[0-9]*)?|\.[0-9]+)|1(\.0*)?)([Ee][+-]?[0-9]+)?)(\s)+){2}([+-]?((0(\.[0-9]*)?|\.[0-9]+)|1(\.0*)?)([Ee][+-]?[0-9]+)?)\s*,?\s*)*' # (includes lower-case true, false)
    # - - - - - - - - - -
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE()
        elif isinstance(value, list):
            for each in value: # check that elements are not tuples
                if isinstance(each, tuple):
                    break
            else: # no tuples found, create 3-tuples
                value = [(x, y, z) for x, y, z in value]
        elif not isinstance(value, list) and isValidSFColor(value):
            value = [value]
        assertValidMFColor(value)
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            if isValidSFColor(value):
                if isinstance(value, SFColor):
                    value = value.value # dereference
                self.__value.append(value)
            elif isValidMFColor(value):
                for each in value:
                    while isinstance(each, list) and len(each) == 1:
                        each = each[0] # dereference
                    if isinstance(each, SFColor):
                        each = each.value # dereference
                    self.__value.append(each)
            else:
                assertValidMFColor(value) # report type failure
    def __bool__(self):
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace('(','').replace(')','').replace(',','').replace('[','').replace(']','')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return '[' + str(self.__value).lower().replace('(','').replace(')','').replace(',','').replace('[','').replace(']','') + ']'

class SFColorRGBA(_X3DField):
    """
    Field type SFColorRGBA specifies one RGBA (red-green-blue-alpha) color 4-tuple, where each color value is an RGBA 4-tuple of floating point numbers in range [0,1]. Alpha (opacity) values = (1 - transparency). The default value of an uninitialized SFColorRGBA field is (0 0 0 0). Warning: comma characters within singleton values do not pass strict XML validation.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'SFColorRGBA'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFColorRGBA'
    def DEFAULT_VALUE(self):
        return (0, 0, 0, 0)
    def ARRAY_TYPE(self):
        return False
    def TUPLE_SIZE(self):
        return 4
    def REGEX_XML(self):
        return r'(\s)*(([+-]?((0(\.[0-9]*)?|\.[0-9]+)|1(\.0*)?)([Ee][+-]?[0-9]+)?)(\s)+){3}([+-]?((0(\.[0-9]*)?|\.[0-9]+)|1(\.0*)?)([Ee][+-]?[0-9]+)?)(\s)*' # (includes lower-case true, false)
    # - - - - - - - - - -
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE()
        elif isinstance(value, list):
            for each in value: # check that elements are not tuples
                if isinstance(each, tuple):
                    break
            else: # no tuples found, create 4-tuples
                value = [(x, y, z, w) for x, y, z, w in value]
        elif isinstance(value, MFColorRGBA) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        assertValidSFColorRGBA(value)
        self.__value = value
    def __bool__(self):
        return len(self.__value) > 0
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace(',','').replace('(','').replace(')','').replace('[','').replace(']','')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return str(self.__value).lower().replace(',','').replace('(','').replace(')','').replace('[','').replace(']','')

class MFColorRGBA(_X3DField):
    """
    Field type MFColorRGBA specifies zero or more SFColorRGBA 4-tuples, where each color value is an RGBA 4-tuple of floating point numbers in range [0,1]. Alpha (opacity) values = (1 - transparency). The default value of an uninitialized MFColor field is the empty list. Individual SFColorRGBA array values are optionally separated by commas in XML syntax.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'MFColorRGBA'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFColorRGBA'
    def DEFAULT_VALUE(self):
        return [] # use empty list object, don't keep resetting a mutable python DEFAULT_VALUE
    def ARRAY_TYPE(self):
        return True
    def TUPLE_SIZE(self):
        return 4
    def REGEX_XML(self):
        return r'(\s)*((([+-]?((0(\.[0-9]*)?|\.[0-9]+)|1(\.0*)?)([Ee][+-]?[0-9]+)?)(\s)+){3}([+-]?((0(\.[0-9]*)?|\.[0-9]+)|1(\.0*)?)([Ee][+-]?[0-9]+)?)\s*,?\s*)*' # (includes lower-case true, false)
    # - - - - - - - - - -
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE()
        elif isinstance(value, list):
            for each in value: # check that elements are not tuples
                if isinstance(each, tuple):
                    break
            else: # no tuples found, create 4-tuples
                value = [(x, y, z, w) for x, y, z, w in value]
        elif not isinstance(value, list) and isValidSFColorRGBA(value):
            value = [value]
        assertValidMFColorRGBA(value)
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            if isValidSFColorRGBA(value):
                if isinstance(value, SFColorRGBA):
                    value = value.value # dereference
                self.__value.append(value)
            elif isValidMFColorRGBA(value):
                for each in value:
                    while isinstance(each, list) and len(each) == 1:
                        each = each[0] # dereference
                    if isinstance(each, SFColorRGBA):
                        each = each.value # dereference
                    self.__value.append(each)
            else:
                assertValidMFColorRGBA(value) # report type failure
    def __bool__(self):
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace('(','').replace(')','').replace(',','').replace('[','').replace(']','')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return '[' + str(self.__value).lower().replace('(','').replace(')','').replace(',','').replace('[','').replace(']','') + ']'

class SFDouble(_X3DField):
    """
    Field type SFDouble is a double-precision floating-point type. Array values are optionally separated by commas in XML syntax. See GeoVRML 1.0 Recommended Practice, Section 2.3, Limitations of Single Precision for rationale.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'SFDouble'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFDouble'
    def DEFAULT_VALUE(self):
        return 0.0
    def ARRAY_TYPE(self):
        return False
    def TUPLE_SIZE(self):
        return 1
    def REGEX_XML(self):
        return r'(\s)*([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)(\s)*' # (includes lower-case true, false)
    # - - - - - - - - - -
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE()
        elif isinstance(value, MFDouble) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        assertValidSFDouble(value)
        self.__value = value
    def __bool__(self):
        return len(self.__value) > 0
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace(',','').replace('(','').replace(')','').replace('[','').replace(']','')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return str(self.__value).lower().replace(',','').replace('(','').replace(')','').replace('[','').replace(']','')

class MFDouble(_X3DField):
    """
    Field type MFDouble is an array of Double values, meaning a double-precision floating-point array type. See GeoVRML 1.0 Recommended Practice, Section 2.3, Limitations of Single Precision for rationale. Array values are optionally separated by commas in XML syntax.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'MFDouble'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFDouble'
    def DEFAULT_VALUE(self):
        return [] # use empty list object, don't keep resetting a mutable python DEFAULT_VALUE
    def ARRAY_TYPE(self):
        return True
    def TUPLE_SIZE(self):
        return 1
    def REGEX_XML(self):
        return r'(\s)*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*' # (includes lower-case true, false)
    # - - - - - - - - - -
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE()
        elif not isinstance(value, list) and isValidSFDouble(value):
            value = [value]
        assertValidMFDouble(value)
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            if isValidSFDouble(value):
                if isinstance(value, SFDouble):
                    value = value.value # dereference
                self.__value.append(value)
            elif isValidMFDouble(value):
                for each in value:
                    while isinstance(each, list) and len(each) == 1:
                        each = each[0] # dereference
                    if isinstance(each, SFDouble):
                        each = each.value # dereference
                    self.__value.append(each)
            else:
                assertValidMFDouble(value) # report type failure
    def __bool__(self):
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace('(','').replace(')','').replace(',','').replace('[','').replace(']','')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return '[' + str(self.__value).lower().replace('(','').replace(')','').replace(',','').replace('[','').replace(']','') + ']'

class SFFloat(_X3DField):
    """
    Field type SFFloat is a single-precision floating-point type.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'SFFloat'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFFloat'
    def DEFAULT_VALUE(self):
        return 0.0
    def ARRAY_TYPE(self):
        return False
    def TUPLE_SIZE(self):
        return 1
    def REGEX_XML(self):
        return r'(\s)*([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)(\s)*' # (includes lower-case true, false)
    # - - - - - - - - - -
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE()
        elif isinstance(value, MFFloat) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        assertValidSFFloat(value)
        self.__value = value
    def __bool__(self):
        return len(self.__value) > 0
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace(',','').replace('(','').replace(')','').replace('[','').replace(']','')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return str(self.__value).lower().replace(',','').replace('(','').replace(')','').replace('[','').replace(']','')

class MFFloat(_X3DField):
    """
    Field type MFFloat is an array of SFFloat values, meaning a single-precision floating-point array type. Array values are optionally separated by commas in XML syntax.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'MFFloat'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFFloat'
    def DEFAULT_VALUE(self):
        return [] # use empty list object, don't keep resetting a mutable python DEFAULT_VALUE
    def ARRAY_TYPE(self):
        return True
    def TUPLE_SIZE(self):
        return 1
    def REGEX_XML(self):
        return r'(\s)*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*' # (includes lower-case true, false)
    # - - - - - - - - - -
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE()
        elif not isinstance(value, list) and isValidSFFloat(value):
            value = [value]
        assertValidMFFloat(value)
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            if isValidSFFloat(value):
                if isinstance(value, SFFloat):
                    value = value.value # dereference
                self.__value.append(value)
            elif isValidMFFloat(value):
                for each in value:
                    while isinstance(each, list) and len(each) == 1:
                        each = each[0] # dereference
                    if isinstance(each, SFFloat):
                        each = each.value # dereference
                    self.__value.append(each)
            else:
                assertValidMFFloat(value) # report type failure
    def __bool__(self):
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace('(','').replace(')','').replace(',','').replace('[','').replace(']','')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return '[' + str(self.__value).lower().replace('(','').replace(')','').replace(',','').replace('[','').replace(']','') + ']'

class SFImage(_X3DField):
    """
    Field type SFImage specifies a single uncompressed 2-dimensional pixel image. SFImage fields contain three integers representing the width, height and number of components in the image, followed by (width x height) hexadecimal or integer values representing the pixels in the image.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'SFImage'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFImage'
    def DEFAULT_VALUE(self):
        return [0, 0, 0]
    def ARRAY_TYPE(self):
        return False
    def TUPLE_SIZE(self):
        return 1
    def REGEX_XML(self):
        return r'\s*([+]?(0|[1-9][0-9]*)([Ee][+]?[0-9]+)?\s+){2}[+]?[0-4](\s+(0x[0-9a-fA-F]{1,16}|[+]?(0|[1-9][0-9]*)([Ee][+]?[0-9]+)?))*\s*' # (includes lower-case true, false)
    # - - - - - - - - - -
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE()
        elif isinstance(value, MFImage) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        assertValidSFImage(value)
        self.__value = value
    def __bool__(self):
        return len(self.__value) > 0
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace(',','').replace('(','').replace(')','').replace('[','').replace(']','')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return str(self.__value).lower().replace(',','').replace('(','').replace(')','').replace('[','').replace(']','')

class MFImage(_X3DField):
    """
    Field type MFImage is an array of SFImage values.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'MFImage'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFImage'
    def DEFAULT_VALUE(self):
        return [] # use empty list object, don't keep resetting a mutable python DEFAULT_VALUE
    def ARRAY_TYPE(self):
        return True
    def TUPLE_SIZE(self):
        return 1
    def REGEX_XML(self):
        return r'\s*(([+]?(0|[1-9][0-9]*)([Ee][+]?[0-9]+)?\s+){2}[+]?[0-4](\s+(0x[0-9a-fA-F]{1,16}|[+]?(0|[1-9][0-9]*)([Ee][+]?[0-9]+)?))*\s*,?\s*)*' # (includes lower-case true, false)
    # - - - - - - - - - -
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE()
        elif not isinstance(value, list) and isValidSFImage(value):
            value = [value]
        assertValidMFImage(value)
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            if isValidSFImage(value):
                if isinstance(value, SFImage):
                    value = value.value # dereference
                self.__value.append(value)
            elif isValidMFImage(value):
                for each in value:
                    while isinstance(each, list) and len(each) == 1:
                        each = each[0] # dereference
                    if isinstance(each, SFImage):
                        each = each.value # dereference
                    self.__value.append(each)
            else:
                assertValidMFImage(value) # report type failure
    def __bool__(self):
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace('(','').replace(')','').replace(',','').replace('[','').replace(']','')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return '[' + str(self.__value).lower().replace('(','').replace(')','').replace(',','').replace('[','').replace(']','') + ']'

class SFInt32(_X3DField):
    """
    Field type SFInt32 specifies one 32-bit signed integer.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'SFInt32'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFInt32'
    def DEFAULT_VALUE(self):
        return 0
    def ARRAY_TYPE(self):
        return False
    def TUPLE_SIZE(self):
        return 1
    def REGEX_XML(self):
        return r'(\s)*[+-]?(0|[1-9][0-9]*)([Ee][+-]?[0-9]+)?(\s)*' # (includes lower-case true, false)
    # - - - - - - - - - -
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE()
        elif isinstance(value, MFInt32) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        assertValidSFInt32(value)
        self.__value = value
    def __bool__(self):
        return len(self.__value) > 0
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace(',','').replace('(','').replace(')','').replace('[','').replace(']','')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return str(self.__value).lower().replace(',','').replace('(','').replace(')','').replace('[','').replace(']','')

class MFInt32(_X3DField):
    """
    Field type MFInt32 defines an array of 32-bit signed integers. Array values are optionally separated by commas in XML syntax.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'MFInt32'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFInt32'
    def DEFAULT_VALUE(self):
        return [] # use empty list object, don't keep resetting a mutable python DEFAULT_VALUE
    def ARRAY_TYPE(self):
        return True
    def TUPLE_SIZE(self):
        return 1
    def REGEX_XML(self):
        return r'(\s)*([+-]?(0|[1-9][0-9]*)([Ee][+-]?[0-9]+)?\s*,?\s*)*' # (includes lower-case true, false)
    # - - - - - - - - - -
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE()
        elif not isinstance(value, list) and isValidSFInt32(value):
            value = [value]
        assertValidMFInt32(value)
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            if isValidSFInt32(value):
                if isinstance(value, SFInt32):
                    value = value.value # dereference
                self.__value.append(value)
            elif isValidMFInt32(value):
                for each in value:
                    while isinstance(each, list) and len(each) == 1:
                        each = each[0] # dereference
                    if isinstance(each, SFInt32):
                        each = each.value # dereference
                    self.__value.append(each)
            else:
                assertValidMFInt32(value) # report type failure
    def __bool__(self):
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace('(','').replace(')','').replace(',','').replace('[','').replace(']','')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return '[' + str(self.__value).lower().replace('(','').replace(')','').replace(',','').replace('[','').replace(']','') + ']'

class SFMatrix3d(_X3DField):
    """
    Field type SFMatrix3d specifies a 3x3 matrix of double-precision floating point numbers, organized in row-major fashion. Warning: comma characters within singleton values do not pass strict XML validation.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'SFMatrix3d'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFMatrix3d'
    def DEFAULT_VALUE(self):
        return (1, 0, 0, 0, 1, 0, 0, 0, 1)
    def ARRAY_TYPE(self):
        return False
    def TUPLE_SIZE(self):
        return 9
    def REGEX_XML(self):
        return r'\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){8}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*' # (includes lower-case true, false)
    # - - - - - - - - - -
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE()
        elif isinstance(value, MFMatrix3d) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        assertValidSFMatrix3d(value)
        self.__value = value
    def __bool__(self):
        return len(self.__value) > 0
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace(',','').replace('(','').replace(')','').replace('[','').replace(']','')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return str(self.__value).lower().replace(',','').replace('(','').replace(')','').replace('[','').replace(']','')

class MFMatrix3d(_X3DField):
    """
    Field type MFMatrix3d specifies zero or more 3x3 matrices of double-precision floating point numbers, organized in row-major fashion. Warning: comma characters can only appear between singleton 9-tuple values.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'MFMatrix3d'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFMatrix3d'
    def DEFAULT_VALUE(self):
        return [] # use empty list object, don't keep resetting a mutable python DEFAULT_VALUE
    def ARRAY_TYPE(self):
        return True
    def TUPLE_SIZE(self):
        return 9
    def REGEX_XML(self):
        return r'\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){8}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*' # (includes lower-case true, false)
    # - - - - - - - - - -
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE()
        elif not isinstance(value, list) and isValidSFMatrix3d(value):
            value = [value]
        assertValidMFMatrix3d(value)
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            if isValidSFMatrix3d(value):
                if isinstance(value, SFMatrix3d):
                    value = value.value # dereference
                self.__value.append(value)
            elif isValidMFMatrix3d(value):
                for each in value:
                    while isinstance(each, list) and len(each) == 1:
                        each = each[0] # dereference
                    if isinstance(each, SFMatrix3d):
                        each = each.value # dereference
                    self.__value.append(each)
            else:
                assertValidMFMatrix3d(value) # report type failure
    def __bool__(self):
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace('(','').replace(')','').replace(',','').replace('[','').replace(']','')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return '[' + str(self.__value).lower().replace('(','').replace(')','').replace(',','').replace('[','').replace(']','') + ']'

class SFMatrix3f(_X3DField):
    """
    Field type SFMatrix3f specifies a 3x3 matrix of single-precision floating point numbers, organized in row-major fashion. Warning: comma characters within singleton values do not pass strict XML validation.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'SFMatrix3f'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFMatrix3f'
    def DEFAULT_VALUE(self):
        return (1, 0, 0, 0, 1, 0, 0, 0, 1)
    def ARRAY_TYPE(self):
        return False
    def TUPLE_SIZE(self):
        return 9
    def REGEX_XML(self):
        return r'\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){8}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*' # (includes lower-case true, false)
    # - - - - - - - - - -
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE()
        elif isinstance(value, MFMatrix3f) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        assertValidSFMatrix3f(value)
        self.__value = value
    def __bool__(self):
        return len(self.__value) > 0
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace(',','').replace('(','').replace(')','').replace('[','').replace(']','')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return str(self.__value).lower().replace(',','').replace('(','').replace(')','').replace('[','').replace(']','')

class MFMatrix3f(_X3DField):
    """
    Field type MFMatrix3f specifies zero or more 3x3 matrices of single-precision floating point numbers, organized in row-major fashion. Warning: comma characters can only appear between singleton 9-tuple values.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'MFMatrix3f'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFMatrix3f'
    def DEFAULT_VALUE(self):
        return [] # use empty list object, don't keep resetting a mutable python DEFAULT_VALUE
    def ARRAY_TYPE(self):
        return True
    def TUPLE_SIZE(self):
        return 9
    def REGEX_XML(self):
        return r'\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){8}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*' # (includes lower-case true, false)
    # - - - - - - - - - -
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE()
        elif not isinstance(value, list) and isValidSFMatrix3f(value):
            value = [value]
        assertValidMFMatrix3f(value)
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            if isValidSFMatrix3f(value):
                if isinstance(value, SFMatrix3f):
                    value = value.value # dereference
                self.__value.append(value)
            elif isValidMFMatrix3f(value):
                for each in value:
                    while isinstance(each, list) and len(each) == 1:
                        each = each[0] # dereference
                    if isinstance(each, SFMatrix3f):
                        each = each.value # dereference
                    self.__value.append(each)
            else:
                assertValidMFMatrix3f(value) # report type failure
    def __bool__(self):
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace('(','').replace(')','').replace(',','').replace('[','').replace(']','')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return '[' + str(self.__value).lower().replace('(','').replace(')','').replace(',','').replace('[','').replace(']','') + ']'

class SFMatrix4d(_X3DField):
    """
    Field type SFMatrix4d specifies a 4x4 matrix of double-precision floating point numbers, organized in row-major fashion. Warning: comma characters within singleton values do not pass strict XML validation.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'SFMatrix4d'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFMatrix4d'
    def DEFAULT_VALUE(self):
        return (1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1)
    def ARRAY_TYPE(self):
        return False
    def TUPLE_SIZE(self):
        return 16
    def REGEX_XML(self):
        return r'\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){15}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*' # (includes lower-case true, false)
    # - - - - - - - - - -
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE()
        elif isinstance(value, MFMatrix4d) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        assertValidSFMatrix4d(value)
        self.__value = value
    def __bool__(self):
        return len(self.__value) > 0
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace(',','').replace('(','').replace(')','').replace('[','').replace(']','')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return str(self.__value).lower().replace(',','').replace('(','').replace(')','').replace('[','').replace(']','')

class MFMatrix4d(_X3DField):
    """
    Field type MFMatrix4d specifies zero or more 4x4 matrices of double-precision floating point numbers, organized in row-major fashion. Warning: comma characters can only appear between singleton 16-tuple values.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'MFMatrix4d'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFMatrix4d'
    def DEFAULT_VALUE(self):
        return [] # use empty list object, don't keep resetting a mutable python DEFAULT_VALUE
    def ARRAY_TYPE(self):
        return True
    def TUPLE_SIZE(self):
        return 16
    def REGEX_XML(self):
        return r'\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){15}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*' # (includes lower-case true, false)
    # - - - - - - - - - -
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE()
        elif not isinstance(value, list) and isValidSFMatrix4d(value):
            value = [value]
        assertValidMFMatrix4d(value)
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            if isValidSFMatrix4d(value):
                if isinstance(value, SFMatrix4d):
                    value = value.value # dereference
                self.__value.append(value)
            elif isValidMFMatrix4d(value):
                for each in value:
                    while isinstance(each, list) and len(each) == 1:
                        each = each[0] # dereference
                    if isinstance(each, SFMatrix4d):
                        each = each.value # dereference
                    self.__value.append(each)
            else:
                assertValidMFMatrix4d(value) # report type failure
    def __bool__(self):
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace('(','').replace(')','').replace(',','').replace('[','').replace(']','')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return '[' + str(self.__value).lower().replace('(','').replace(')','').replace(',','').replace('[','').replace(']','') + ']'

class SFMatrix4f(_X3DField):
    """
    Field type SFMatrix4f specifies a 4x4 matrix of single-precision floating point numbers, organized in row-major fashion. Warning: comma characters within singleton values do not pass strict XML validation.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'SFMatrix4f'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFMatrix4f'
    def DEFAULT_VALUE(self):
        return (1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1)
    def ARRAY_TYPE(self):
        return False
    def TUPLE_SIZE(self):
        return 16
    def REGEX_XML(self):
        return r'\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){15}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*' # (includes lower-case true, false)
    # - - - - - - - - - -
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE()
        elif isinstance(value, MFMatrix4f) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        assertValidSFMatrix4f(value)
        self.__value = value
    def __bool__(self):
        return len(self.__value) > 0
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace(',','').replace('(','').replace(')','').replace('[','').replace(']','')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return str(self.__value).lower().replace(',','').replace('(','').replace(')','').replace('[','').replace(']','')

class MFMatrix4f(_X3DField):
    """
    Field type MFMatrix4f specifies zero or more 4x4 matrices of single-precision floating point numbers, organized in row-major fashion. Warning: comma characters can only appear between singleton 16-tuple values.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'MFMatrix4f'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFMatrix4f'
    def DEFAULT_VALUE(self):
        return [] # use empty list object, don't keep resetting a mutable python DEFAULT_VALUE
    def ARRAY_TYPE(self):
        return True
    def TUPLE_SIZE(self):
        return 16
    def REGEX_XML(self):
        return r'\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){15}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*' # (includes lower-case true, false)
    # - - - - - - - - - -
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE()
        elif not isinstance(value, list) and isValidSFMatrix4f(value):
            value = [value]
        assertValidMFMatrix4f(value)
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            if isValidSFMatrix4f(value):
                if isinstance(value, SFMatrix4f):
                    value = value.value # dereference
                self.__value.append(value)
            elif isValidMFMatrix4f(value):
                for each in value:
                    while isinstance(each, list) and len(each) == 1:
                        each = each[0] # dereference
                    if isinstance(each, SFMatrix4f):
                        each = each.value # dereference
                    self.__value.append(each)
            else:
                assertValidMFMatrix4f(value) # report type failure
    def __bool__(self):
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace('(','').replace(')','').replace(',','').replace('[','').replace(']','')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return '[' + str(self.__value).lower().replace('(','').replace(')','').replace(',','').replace('[','').replace(']','') + ']'

class SFNode(_X3DField):
    """
    Field type SFNode specifies an X3D node; the default empty value of an uninitialized SFNode field is sometimes described as NULL.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'SFNode'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFNode'
    def DEFAULT_VALUE(self):
        return None
    def FIELD_DECLARATIONS(self):
        return [('value', 'None', FieldType.SFNode, AccessType.inputOutput, 'SFNode')]
    def ARRAY_TYPE(self):
        return False
    def TUPLE_SIZE(self):
        return 1
    def REGEX_XML(self):
        return r'' # (includes lower-case true, false)
    # - - - - - - - - - -
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE()
        elif isinstance(value, MFNode) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        assertValidSFNode(value)
        self.__value = value
    def __bool__(self):
        return len(self.__value) > 0
    def XML(self):
        """ Provide XML value for this field type. """
        if (not self.__value is None):
            return self.__value.XML()
    def VRML(self):
        """ Provide VRML value for this field type. """
        if (not self.__value is None):
            return self.__value.VRML()

class MFNode(_X3DField):
    """
    Field type MFNode specifies zero or more nodes; the default value of an MFNode field is the empty list.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'MFNode'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFNode'
    def DEFAULT_VALUE(self):
        return [] # use empty list object, don't keep resetting a mutable python DEFAULT_VALUE
    def FIELD_DECLARATIONS(self):
        return [('value', None, FieldType.MFNode, AccessType.inputOutput, 'MFNode')]
    def ARRAY_TYPE(self):
        return True
    def TUPLE_SIZE(self):
        return 1
    def REGEX_XML(self):
        return r'' # (includes lower-case true, false)
    # - - - - - - - - - -
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE()
        elif not isinstance(value, list) and isValidSFNode(value):
            value = [value]
        assertValidMFNode(value)
        self.__value = value
    def __repl__(self):
        result = '['
        for each in self.__value:
            result += str(each) + ', '
        return result.rstrip(', ') + ']'
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            if isValidSFNode(value):
                if isinstance(value, SFNode):
                    value = value.value # dereference
                self.__value.append(value)
            elif isValidMFNode(value):
                for each in value:
                    while isinstance(each, list) and len(each) == 1:
                        each = each[0] # dereference
                    if isinstance(each, SFNode):
                        each = each.value # dereference
                    self.__value.append(each)
            else:
                assertValidMFNode(value) # report type failure
    def __bool__(self):
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)
    def XML(self):
        """ Provide XML value for this field type. """
        result = ''
        for each in self.__value:
            if (not self.__value is None):
                result += each.XML() # has line break '\n' at end, which is OK
        result = result.rstrip('\n')
        return result
    def VRML(self):
        """ Provide VRML value for this field type. """
        result = ''
        for each in self.__value:
            if (not self.__value is None):
                result += each.VRML() # has line break '\n' at end, which is OK
        result = result.rstrip('\n')
        return result

class SFRotation(_X3DField):
    """
    Field type SFRotation is an axis-angle 4-tuple, indicating X-Y-Z direction axis plus angle orientation about that axis. The first three values specify a normalized axis vector about which the rotation takes place, so the first three values shall be within the range [-1..+1] in order to represent a normalized unit vector. The fourth value specifies the amount of right-handed rotation about that axis in radians. Warning: comma characters within singleton values do not pass strict XML validation.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'SFRotation'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFRotation'
    def DEFAULT_VALUE(self):
        return (0, 0, 1, 0)
    def ARRAY_TYPE(self):
        return False
    def TUPLE_SIZE(self):
        return 4
    def REGEX_XML(self):
        return r'\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){3}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*' # (includes lower-case true, false)
    # - - - - - - - - - -
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE()
        elif isinstance(value, list):
            for each in value: # check that elements are not tuples
                if isinstance(each, tuple):
                    break
            else: # no tuples found, create 4-tuples
                value = [(x, y, z, w) for x, y, z, w in value]
        elif isinstance(value, MFRotation) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        assertValidSFRotation(value)
        self.__value = value
    def __bool__(self):
        return len(self.__value) > 0
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace(',','').replace('(','').replace(')','').replace('[','').replace(']','')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return str(self.__value).lower().replace(',','').replace('(','').replace(')','').replace('[','').replace(']','')

class MFRotation(_X3DField):
    """
    Field type MFRotation is an array of SFRotation values. Individual singleton SFRotation array values are optionally separated by commas in XML syntax.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'MFRotation'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFRotation'
    def DEFAULT_VALUE(self):
        return [] # use empty list object, don't keep resetting a mutable python DEFAULT_VALUE
    def ARRAY_TYPE(self):
        return True
    def TUPLE_SIZE(self):
        return 4
    def REGEX_XML(self):
        return r'\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){3}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*' # (includes lower-case true, false)
    # - - - - - - - - - -
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE()
        elif isinstance(value, list):
            for each in value: # check that elements are not tuples
                if isinstance(each, tuple):
                    break
            else: # no tuples found, create 4-tuples
                value = [(x, y, z, w) for x, y, z, w in value]
        elif not isinstance(value, list) and isValidSFRotation(value):
            value = [value]
        assertValidMFRotation(value)
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            if isValidSFRotation(value):
                if isinstance(value, SFRotation):
                    value = value.value # dereference
                self.__value.append(value)
            elif isValidMFRotation(value):
                for each in value:
                    while isinstance(each, list) and len(each) == 1:
                        each = each[0] # dereference
                    if isinstance(each, SFRotation):
                        each = each.value # dereference
                    self.__value.append(each)
            else:
                assertValidMFRotation(value) # report type failure
    def __bool__(self):
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace('(','').replace(')','').replace(',','').replace('[','').replace(']','')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return '[' + str(self.__value).lower().replace('(','').replace(')','').replace(',','').replace('[','').replace(']','') + ']'

class SFString(_X3DField):
    """
    Field type SFString defines a single string encoded with the UTF-8 universal character set.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'SFString'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFString'
    def DEFAULT_VALUE(self):
        return ''
    def ARRAY_TYPE(self):
        return False
    def TUPLE_SIZE(self):
        return 1
    def REGEX_XML(self):
        return r'(\s|\S)*' # (includes lower-case true, false)
    # - - - - - - - - - -
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE()
        elif isinstance(value, MFString) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        assertValidSFString(value)
        self.__value = value
    def __bool__(self):
        return len(self.__value) > 0
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value)
    def VRML(self):
        """ Provide VRML value for this field type. """
        return '"' + str(self.__value) + '"'

class MFString(_X3DField):
    """
    Field type MFString is an array of SFString values, each "quoted" and separated by whitespace. Individual SFString array values are optionally separated by commas in XML syntax.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'MFString'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFString'
    def DEFAULT_VALUE(self):
        return [] # use empty list object, don't keep resetting a mutable python DEFAULT_VALUE
    def ARRAY_TYPE(self):
        return True
    def TUPLE_SIZE(self):
        return 1
    def REGEX_XML(self):
        return r'(\s|\S)*' # (includes lower-case true, false)
    # - - - - - - - - - -
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE()
        elif not isinstance(value, list) and isValidSFString(value):
            value = [value]
        assertValidMFString(value)
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            if isValidSFString(value):
                if isinstance(value, SFString):
                    value = value.value # dereference
                self.__value.append(value)
            elif isValidMFString(value):
                for each in value:
                    while isinstance(each, list) and len(each) == 1:
                        each = each[0] # dereference
                    if isinstance(each, SFString):
                        each = each.value # dereference
                    self.__value.append(each)
            else:
                assertValidMFString(value) # report type failure
    def __bool__(self):
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)
    def XML(self):
        """ Provide XML value for this field type. """
        result = ''
        for each in self.__value:
            result += '"' + str(each) + '"' + ' '
        result = result.rstrip(' ')
        return result
    def VRML(self):
        """ Provide VRML value for this field type. """
        result = ''
        for each in self.__value:
            result += '"' + str(each) + '"' + ' '
        result = '[' + result.rstrip(' ') + ']'
        return result

class SFTime(_X3DField):
    """
    Field type SFTime specifies a single time value, expressed as a double-precision floating point number. Typically, SFTime fields represent the number of seconds since Jan 1, 1970, 00:00:00 GMT.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'SFTime'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFTime'
    def DEFAULT_VALUE(self):
        return -1.0
    def ARRAY_TYPE(self):
        return False
    def TUPLE_SIZE(self):
        return 1
    def REGEX_XML(self):
        return r'(\s)*([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)(\s)*' # (includes lower-case true, false)
    # - - - - - - - - - -
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE()
        elif isinstance(value, MFTime) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        assertValidSFTime(value)
        self.__value = value
    def __bool__(self):
        return len(self.__value) > 0
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace(',','').replace('(','').replace(')','').replace('[','').replace(']','')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return str(self.__value).lower().replace(',','').replace('(','').replace(')','').replace('[','').replace(']','')

class MFTime(_X3DField):
    """
    Field type MFTime is an array of SFTime values. Array values are optionally separated by commas in XML syntax.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'MFTime'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFTime'
    def DEFAULT_VALUE(self):
        return [] # use empty list object, don't keep resetting a mutable python DEFAULT_VALUE
    def ARRAY_TYPE(self):
        return True
    def TUPLE_SIZE(self):
        return 1
    def REGEX_XML(self):
        return r'(\s)*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*' # (includes lower-case true, false)
    # - - - - - - - - - -
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE()
        elif not isinstance(value, list) and isValidSFTime(value):
            value = [value]
        assertValidMFTime(value)
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            if isValidSFTime(value):
                if isinstance(value, SFTime):
                    value = value.value # dereference
                self.__value.append(value)
            elif isValidMFTime(value):
                for each in value:
                    while isinstance(each, list) and len(each) == 1:
                        each = each[0] # dereference
                    if isinstance(each, SFTime):
                        each = each.value # dereference
                    self.__value.append(each)
            else:
                assertValidMFTime(value) # report type failure
    def __bool__(self):
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace('(','').replace(')','').replace(',','').replace('[','').replace(']','')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return '[' + str(self.__value).lower().replace('(','').replace(')','').replace(',','').replace('[','').replace(']','') + ']'

class SFVec2d(_X3DField):
    """
    Field type SFVec2d is a 2-tuple pair of SFDouble values. Hint: SFVec2d can be used to specify a 2D double-precision coordinate. Warning: comma characters within singleton values do not pass strict XML validation.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'SFVec2d'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFVec2d'
    def DEFAULT_VALUE(self):
        return (0.0, 0.0)
    def ARRAY_TYPE(self):
        return False
    def TUPLE_SIZE(self):
        return 2
    def REGEX_XML(self):
        return r'\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){1}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*' # (includes lower-case true, false)
    # - - - - - - - - - -
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE()
        elif isinstance(value, MFVec2d) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        assertValidSFVec2d(value)
        self.__value = value
    def __bool__(self):
        return len(self.__value) > 0
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace(',','').replace('(','').replace(')','').replace('[','').replace(']','')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return str(self.__value).lower().replace(',','').replace('(','').replace(')','').replace('[','').replace(']','')

class MFVec2d(_X3DField):
    """
    Field type MFVec2d is an array of SFVec2d values. Individual singleton SFVec2d array values are optionally separated by commas in XML syntax.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'MFVec2d'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFVec2d'
    def DEFAULT_VALUE(self):
        return [] # use empty list object, don't keep resetting a mutable python DEFAULT_VALUE
    def ARRAY_TYPE(self):
        return True
    def TUPLE_SIZE(self):
        return 2
    def REGEX_XML(self):
        return r'\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){1}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*' # (includes lower-case true, false)
    # - - - - - - - - - -
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE()
        elif not isinstance(value, list) and isValidSFVec2d(value):
            value = [value]
        assertValidMFVec2d(value)
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            if isValidSFVec2d(value):
                if isinstance(value, SFVec2d):
                    value = value.value # dereference
                self.__value.append(value)
            elif isValidMFVec2d(value):
                for each in value:
                    while isinstance(each, list) and len(each) == 1:
                        each = each[0] # dereference
                    if isinstance(each, SFVec2d):
                        each = each.value # dereference
                    self.__value.append(each)
            else:
                assertValidMFVec2d(value) # report type failure
    def __bool__(self):
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace('(','').replace(')','').replace(',','').replace('[','').replace(']','')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return '[' + str(self.__value).lower().replace('(','').replace(')','').replace(',','').replace('[','').replace(']','') + ']'

class SFVec2f(_X3DField):
    """
    Field type SFVec2f is a 2-tuple pair of SFFloat values. Hint: SFVec2f can be used to specify a 2D single-precision coordinate. Warning: comma characters within singleton values do not pass strict XML validation.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'SFVec2f'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFVec2f'
    def DEFAULT_VALUE(self):
        return (0.0, 0.0)
    def ARRAY_TYPE(self):
        return False
    def TUPLE_SIZE(self):
        return 2
    def REGEX_XML(self):
        return r'\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){1}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*' # (includes lower-case true, false)
    # - - - - - - - - - -
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE()
        elif isinstance(value, list):
            for each in value: # check that elements are not tuples
                if isinstance(each, tuple):
                    break
            else: # no tuples found, create 2-tuples
                value = [(x, y) for x, y, in value]
        elif isinstance(value, MFVec2f) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        assertValidSFVec2f(value)
        self.__value = value
    def __bool__(self):
        return len(self.__value) > 0
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace(',','').replace('(','').replace(')','').replace('[','').replace(']','')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return str(self.__value).lower().replace(',','').replace('(','').replace(')','').replace('[','').replace(']','')

class MFVec2f(_X3DField):
    """
    Field type MFVec2f is an array of SFVec2f values. Individual singleton SFVec2f array values are optionally separated by commas in XML syntax.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'MFVec2f'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFVec2f'
    def DEFAULT_VALUE(self):
        return [] # use empty list object, don't keep resetting a mutable python DEFAULT_VALUE
    def ARRAY_TYPE(self):
        return True
    def TUPLE_SIZE(self):
        return 2
    def REGEX_XML(self):
        return r'\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){1}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*' # (includes lower-case true, false)
    # - - - - - - - - - -
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE()
        elif isinstance(value, list):
            for each in value: # check that elements are not tuples
                if isinstance(each, tuple):
                    break
            else: # no tuples found, create 2-tuples
                value = [(x, y) for x, y, in value]
        elif not isinstance(value, list) and isValidSFVec2f(value):
            value = [value]
        assertValidMFVec2f(value)
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            if isValidSFVec2f(value):
                if isinstance(value, SFVec2f):
                    value = value.value # dereference
                self.__value.append(value)
            elif isValidMFVec2f(value):
                for each in value:
                    while isinstance(each, list) and len(each) == 1:
                        each = each[0] # dereference
                    if isinstance(each, SFVec2f):
                        each = each.value # dereference
                    self.__value.append(each)
            else:
                assertValidMFVec2f(value) # report type failure
    def __bool__(self):
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace('(','').replace(')','').replace(',','').replace('[','').replace(']','')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return '[' + str(self.__value).lower().replace('(','').replace(')','').replace(',','').replace('[','').replace(']','') + ']'

class SFVec3d(_X3DField):
    """
    Field type SFVec3d is a 3-tuple triplet of SFDouble values. See GeoVRML 1.0 Recommended Practice, Section 2.3, Limitations of Single Precision. Hint: SFVec3d can be used to specify a georeferenced 3D coordinate. Warning: comma characters within singleton values do not pass strict XML validation.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'SFVec3d'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFVec3d'
    def DEFAULT_VALUE(self):
        return (0.0, 0.0, 0.0)
    def ARRAY_TYPE(self):
        return False
    def TUPLE_SIZE(self):
        return 3
    def REGEX_XML(self):
        return r'\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){2}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*' # (includes lower-case true, false)
    # - - - - - - - - - -
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE()
        elif isinstance(value, MFVec3d) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        assertValidSFVec3d(value)
        self.__value = value
    def __bool__(self):
        return len(self.__value) > 0
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace(',','').replace('(','').replace(')','').replace('[','').replace(']','')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return str(self.__value).lower().replace(',','').replace('(','').replace(')','').replace('[','').replace(']','')

class MFVec3d(_X3DField):
    """
    Field type MFVec3d is an array of SFVec3d values. Individual singleton SFVec3d array values are optionally separated by commas in XML syntax. Original rationale for inclusion: GeoVRML 1.0 Recommended Practice, Section 2.3, Limitations of Single Precision. Hint: MFVec3d can be used to specify a list of georeferenced 3D coordinates.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'MFVec3d'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFVec3d'
    def DEFAULT_VALUE(self):
        return [] # use empty list object, don't keep resetting a mutable python DEFAULT_VALUE
    def ARRAY_TYPE(self):
        return True
    def TUPLE_SIZE(self):
        return 3
    def REGEX_XML(self):
        return r'\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){2}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*' # (includes lower-case true, false)
    # - - - - - - - - - -
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE()
        elif not isinstance(value, list) and isValidSFVec3d(value):
            value = [value]
        assertValidMFVec3d(value)
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            if isValidSFVec3d(value):
                if isinstance(value, SFVec3d):
                    value = value.value # dereference
                self.__value.append(value)
            elif isValidMFVec3d(value):
                for each in value:
                    while isinstance(each, list) and len(each) == 1:
                        each = each[0] # dereference
                    if isinstance(each, SFVec3d):
                        each = each.value # dereference
                    self.__value.append(each)
            else:
                assertValidMFVec3d(value) # report type failure
    def __bool__(self):
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace('(','').replace(')','').replace(',','').replace('[','').replace(']','')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return '[' + str(self.__value).lower().replace('(','').replace(')','').replace(',','').replace('[','').replace(']','') + ']'

class SFVec3f(_X3DField):
    """
    Field type SFVec3f is a 3-tuple triplet of SFFloat values. Hint: SFVec3f can be used to specify a 3D coordinate or a 3D scale value. Warning: comma characters within singleton values do not pass strict XML validation.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'SFVec3f'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFVec3f'
    def DEFAULT_VALUE(self):
        return (0.0, 0.0, 0.0)
    def ARRAY_TYPE(self):
        return False
    def TUPLE_SIZE(self):
        return 3
    def REGEX_XML(self):
        return r'\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){2}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*' # (includes lower-case true, false)
    # - - - - - - - - - -
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE()
        elif isinstance(value, list):
            for each in value: # check that elements are not tuples
                if isinstance(each, tuple):
                    break
            else: # no tuples found, create 3-tuples
                value = [(x, y, z) for x, y, z in value]
        elif isinstance(value, MFVec3f) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        assertValidSFVec3f(value)
        self.__value = value
    def __bool__(self):
        return len(self.__value) > 0
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace(',','').replace('(','').replace(')','').replace('[','').replace(']','')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return str(self.__value).lower().replace(',','').replace('(','').replace(')','').replace('[','').replace(']','')

class MFVec3f(_X3DField):
    """
    Field type MFVec3f is an array of SFVec3f values. Individual singleton SFVec3f array values are optionally separated by commas in XML syntax.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'MFVec3f'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFVec3f'
    def DEFAULT_VALUE(self):
        return [] # use empty list object, don't keep resetting a mutable python DEFAULT_VALUE
    def ARRAY_TYPE(self):
        return True
    def TUPLE_SIZE(self):
        return 3
    def REGEX_XML(self):
        return r'\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){2}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*' # (includes lower-case true, false)
    # - - - - - - - - - -
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE()
        elif isinstance(value, list):
            for each in value: # check that elements are not tuples
                if isinstance(each, tuple):
                    break
            else: # no tuples found, create 3-tuples
                value = [(x, y, z) for x, y, z in value]
        elif not isinstance(value, list) and isValidSFVec3f(value):
            value = [value]
        assertValidMFVec3f(value)
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            if isValidSFVec3f(value):
                if isinstance(value, SFVec3f):
                    value = value.value # dereference
                self.__value.append(value)
            elif isValidMFVec3f(value):
                for each in value:
                    while isinstance(each, list) and len(each) == 1:
                        each = each[0] # dereference
                    if isinstance(each, SFVec3f):
                        each = each.value # dereference
                    self.__value.append(each)
            else:
                assertValidMFVec3f(value) # report type failure
    def __bool__(self):
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace('(','').replace(')','').replace(',','').replace('[','').replace(']','')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return '[' + str(self.__value).lower().replace('(','').replace(')','').replace(',','').replace('[','').replace(']','') + ']'

class SFVec4d(_X3DField):
    """
    Field type SFVec4d is a 4-tuple set of double-precision floating-point values, specifying a 3D homogeneous vector. Warning: comma characters within singleton values do not pass strict XML validation.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'SFVec4d'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFVec4d'
    def DEFAULT_VALUE(self):
        return (0.0, 0.0, 0.0, 0.0)
    def ARRAY_TYPE(self):
        return False
    def TUPLE_SIZE(self):
        return 4
    def REGEX_XML(self):
        return r'\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){3}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*' # (includes lower-case true, false)
    # - - - - - - - - - -
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE()
        elif isinstance(value, MFVec4d) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        assertValidSFVec4d(value)
        self.__value = value
    def __bool__(self):
        return len(self.__value) > 0
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace(',','').replace('(','').replace(')','').replace('[','').replace(']','')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return str(self.__value).lower().replace(',','').replace('(','').replace(')','').replace('[','').replace(']','')

class MFVec4d(_X3DField):
    """
    Field type MFVec4d is zero or more SFVec4d values. Individual singleton SFVec4d array values are optionally separated by commas in XML syntax.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'MFVec4d'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFVec4d'
    def DEFAULT_VALUE(self):
        return [] # use empty list object, don't keep resetting a mutable python DEFAULT_VALUE
    def ARRAY_TYPE(self):
        return True
    def TUPLE_SIZE(self):
        return 4
    def REGEX_XML(self):
        return r'\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){3}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*' # (includes lower-case true, false)
    # - - - - - - - - - -
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE()
        elif not isinstance(value, list) and isValidSFVec4d(value):
            value = [value]
        assertValidMFVec4d(value)
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            if isValidSFVec4d(value):
                if isinstance(value, SFVec4d):
                    value = value.value # dereference
                self.__value.append(value)
            elif isValidMFVec4d(value):
                for each in value:
                    while isinstance(each, list) and len(each) == 1:
                        each = each[0] # dereference
                    if isinstance(each, SFVec4d):
                        each = each.value # dereference
                    self.__value.append(each)
            else:
                assertValidMFVec4d(value) # report type failure
    def __bool__(self):
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace('(','').replace(')','').replace(',','').replace('[','').replace(']','')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return '[' + str(self.__value).lower().replace('(','').replace(')','').replace(',','').replace('[','').replace(']','') + ']'

class SFVec4f(_X3DField):
    """
    Field type SFVec4f is a 4-tuple set of single-precision floating-point values, specifying a 3D homogeneous vector. Warning: comma characters within singleton values do not pass strict XML validation.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'SFVec4f'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#SFVec4f'
    def DEFAULT_VALUE(self):
        return (0.0, 0.0, 0.0, 0.0)
    def ARRAY_TYPE(self):
        return False
    def TUPLE_SIZE(self):
        return 4
    def REGEX_XML(self):
        return r'\s*(([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){3}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*' # (includes lower-case true, false)
    # - - - - - - - - - -
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE()
        elif isinstance(value, list):
            for each in value: # check that elements are not tuples
                if isinstance(each, tuple):
                    break
            else: # no tuples found, create 4-tuples
                value = [(x, y, z, w) for x, y, z, w in value]
        elif isinstance(value, MFVec4f) and isinstance(value.value, list) and len(value.value) == 1:
            print("downcasting by dereferencing simple-list value=" + str(value)[:100] + ", type=" + str(type(value)) + " as " + str(value.value[0]))
            value = value.value[0] # dereference
        elif isinstance(value, list) and len(value) == 1:
            value = value[0] # dereference
        assertValidSFVec4f(value)
        self.__value = value
    def __bool__(self):
        return len(self.__value) > 0
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace(',','').replace('(','').replace(')','').replace('[','').replace(']','')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return str(self.__value).lower().replace(',','').replace('(','').replace(')','').replace('[','').replace(']','')

class MFVec4f(_X3DField):
    """
    Field type MFVec4f is zero or more SFVec4f values. Individual singleton SFVec4f array values are optionally separated by commas in XML syntax.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'MFVec4f'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#MFVec4f'
    def DEFAULT_VALUE(self):
        return [] # use empty list object, don't keep resetting a mutable python DEFAULT_VALUE
    def ARRAY_TYPE(self):
        return True
    def TUPLE_SIZE(self):
        return 4
    def REGEX_XML(self):
        return r'\s*((([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s+){3}([+-]?((0|[1-9][0-9]*)(\.[0-9]*)?|\.[0-9]+)([Ee][+-]?[0-9]+)?)\s*,?\s*)*' # (includes lower-case true, false)
    # - - - - - - - - - -
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide typed value of this field instance. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE()
        elif isinstance(value, list):
            for each in value: # check that elements are not tuples
                if isinstance(each, tuple):
                    break
            else: # no tuples found, create 4-tuples
                value = [(x, y, z, w) for x, y, z, w in value]
        elif not isinstance(value, list) and isValidSFVec4f(value):
            value = [value]
        assertValidMFVec4f(value)
        self.__value = value
    def append(self, value=None):
        """ Add to existing value list, first ensuring that a correctly typed value is applied. """
        if  not value is None:
            if isValidSFVec4f(value):
                if isinstance(value, SFVec4f):
                    value = value.value # dereference
                self.__value.append(value)
            elif isValidMFVec4f(value):
                for each in value:
                    while isinstance(each, list) and len(each) == 1:
                        each = each[0] # dereference
                    if isinstance(each, SFVec4f):
                        each = each.value # dereference
                    self.__value.append(each)
            else:
                assertValidMFVec4f(value) # report type failure
    def __bool__(self):
        return len(self.__value) > 0
    def __len__(self):
        return len(self.__value)
    def XML(self):
        """ Provide XML value for this field type. """
        return str(self.__value).lower().replace('(','').replace(')','').replace(',','').replace('[','').replace(']','')
    def VRML(self):
        """ Provide VRML value for this field type. """
        return '[' + str(self.__value).lower().replace('(','').replace(')','').replace(',','').replace('[','').replace(']','') + ']'

###############################################

# Abstract Node Types

# Note that these package-internal class names are preceded by an underscore _ character for hidden scope, since X3D authors are not expected to use them.

class _X3DNode(object):
    """
    All instantiable nodes implement X3DNode, which corresponds to SFNode in the X3D specification.
    """
    def NAME(self):
        return '_X3DNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF="", USE="", class_="", metadata=None, IS=None):
        self.DEF = DEF
        self.USE = USE
        self.class_ = class_
        self.IS = IS
        self.metadata = metadata
        # if _DEBUG: print('... in X3DNode __init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
    @property # getter - - - - - - - - - -
    def DEF(self):
        """ Unique ID name for this node, referenceable by other nodes. """
        return self.__DEF
    @DEF.setter
    def DEF(self, DEF):
        if  DEF is None:
            DEF = SFString.DEFAULT_VALUE(self)
        assertValidSFString(DEF)
        self.__DEF = str(DEF)
        if self.__DEF:
            self.__USE = None # DEF and USE are mutually exclusive
    @property # getter - - - - - - - - - -
    def USE(self):
        """ Reuse an already DEF-ed node ID, excluding all child nodes and all other attributes. """
        return self.__USE
    @USE.setter
    def USE(self, USE):
        if  USE is None:
            USE = SFString.DEFAULT_VALUE(self)
        assertValidSFString(USE)
        self.__USE = str(USE)
        if self.__USE:
            self.__DEF = None # DEF and USE are mutually exclusive
    @property # getter - - - - - - - - - -
    def class_(self):
        """ Space-separated list of classes, reserved for use by CSS cascading stylesheets. """
        return self.__class_
    @class_.setter
    def class_(self, class_):
        if  class_ is None:
            class_ = SFString.DEFAULT_VALUE(self)
        assertValidSFString(class_)
        self.__class_ = class_
    @property # getter - - - - - - - - - -
    def IS(self):
        """ The IS statement connects node fields defined inside a ProtoBody declaration back to corresponding ProtoInterface fields. """
        return self.__IS
    @IS.setter
    def IS(self, IS):
        if  IS is None:
            IS = SFNode.DEFAULT_VALUE(self)
        assertValidSFNode(IS)
        if not isinstance(IS, object):
            # print(flush=True)
            raise X3DTypeError(str(IS) + ' does not have a valid node type object')
        self.__IS = IS
    @property # getter - - - - - - - - - -
    def metadata(self):
        """ The metadata field can contain a single MetadataBoolean, MetadataInteger, MetadataFloat, MetadataDouble, MetadataString or MetadataSet node. """
        return self.__metadata
    @metadata.setter
    def metadata(self, metadata):
        if  metadata is None:
            metadata = SFNode.DEFAULT_VALUE(self)
        assertValidSFNode(metadata)
        if not isinstance(metadata, object):
            # print(flush=True)
            raise X3DTypeError(str(metadata) + ' does not have a valid node type object')
        self.__metadata = metadata
    def __repl__(self):
        result = self.NAME() + '('
        # TODO put DEF first, matching canonical form
        if self.FIELD_DECLARATIONS():
            for each in self.FIELD_DECLARATIONS():
                # if _DEBUG: print(self.NAME() + ' for each in self.FIELD_DECLARATIONS(): each=' + str(each))
                name = each[0]
                default = each[1]
                type_ = each[2]
                accessType = each[3]
                value = getattr(self, name)
                # if _DEBUG: print('gettattr(self, ' + str(name) + ') value="' + str(value)[:100] + '" for FIELD_DECLARATIONS ' + str(each) + ')', flush=True)
                if value != default:
                    # consider whether indentation is useful; probably not
                    # print("\n\t")
                    if  isinstance(value, list): # avoid X3DTypeError if value is not iterable
                        result += str(name) + '=['
                        for each in value:
                            # if _DEBUG: print('* X3DNode debug: str(each)=' + str(each), flush=True)
                            result += str(each) + ', '
                        result = result.rstrip(', ')
                        result += '],'
                    elif isinstance(value, str) and "'" in value:
                        result += str(name) + '=' + '"' + str(value)[:100] + '"' + ','
                    elif isinstance(value, str) and value != default:
                        result += str(name) + '=' + "'" + str(value)[:100] + "'" + ','
                    elif value != default:
                        result += str(name) + '='       + str(value)[:100]       + ','
                    # elif _DEBUG:
                    #   result += str(name) + '=' + "'" + str(value)[:100] + "'" + ','
        return result.strip().rstrip(',').rstrip(', ') + ')'
    def __str__(self):
        return self.__repl__().strip() # X3DNode


def isX3DNode(value):
    """
    Whether or not value is a concrete node (Shape WorldInfo etc.) meaning any _X3DNode object.
    """
    return isinstance(value, _X3DNode)

class _X3DChildNode(_X3DNode):
    """
    A node that implements X3DChildNode is one of the legal children for a X3DGroupingNode parent.
    """
    def NAME(self):
        return '_X3DChildNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DChildNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DTimeDependentNode(_X3DChildNode):
    """
    Base type from which all time-dependent nodes are derived.
    """
    def NAME(self):
        return '_X3DTimeDependentNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DTimeDependentNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DGeometryNode(_X3DNode):
    """
    Geometry nodes produce renderable geometry and are contained by a Shape node.
    """
    def NAME(self):
        return '_X3DGeometryNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DGeometryNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DParametricGeometryNode(_X3DGeometryNode):
    """
    Base type for all geometry node types that are created parametrically and use control points to describe the final shape of the surface.
    """
    def NAME(self):
        return '_X3DParametricGeometryNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DParametricGeometryNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DAppearanceChildNode(_X3DNode):
    """
    Nodes of this type can be used as child nodes for Appearance.
    """
    def NAME(self):
        return '_X3DAppearanceChildNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DAppearanceChildNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DTextureNode(_X3DAppearanceChildNode):
    """
    Base type for all nodes which specify sources for texture images.
    """
    def NAME(self):
        return '_X3DTextureNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DTextureNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DSensorNode(_X3DChildNode):
    """
    Base type for all sensors.
    """
    def NAME(self):
        return '_X3DSensorNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DSensorNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DPointingDeviceSensorNode(_X3DSensorNode):
    """
    Base type for all pointing device sensors.
    """
    def NAME(self):
        return '_X3DPointingDeviceSensorNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DPointingDeviceSensorNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DVolumeRenderStyleNode(_X3DNode):
    """
    The X3DVolumeRenderStyleNode abstract node type is the base type for all node types that specify a specific visual rendering style to be used when rendering volume data.
    """
    def NAME(self):
        return '_X3DVolumeRenderStyleNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DVolumeRenderStyleNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DGeometricPropertyNode(_X3DNode):
    """
    Base type for all geometric property node types.
    """
    def NAME(self):
        return '_X3DGeometricPropertyNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DGeometricPropertyNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DFollowerNode(_X3DChildNode):
    """
    X3DFollowerNode is the abstract base class for all nodes in the Followers component.
    """
    def NAME(self):
        return '_X3DFollowerNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DFollowerNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DBindableNode(_X3DChildNode):
    """
    Bindable nodes implement the binding stack, so that only one of each node type is active at a given time.
    """
    def NAME(self):
        return '_X3DBindableNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DBindableNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DAppearanceNode(_X3DNode):
    """
    Base type for all Appearance nodes.
    """
    def NAME(self):
        return '_X3DAppearanceNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DAppearanceNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DBackgroundNode(_X3DBindableNode):
    """
    Abstract type from which all backgrounds inherit, also defining a background binding stack.
    """
    def NAME(self):
        return '_X3DBackgroundNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DBackgroundNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DChaserNode(_X3DFollowerNode):
    """
    The X3DChaserNode abstract node type calculates the output on value_changed as a finite impulse response (FIR) based on the events received on set_destination field.
    """
    def NAME(self):
        return '_X3DChaserNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DChaserNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DColorNode(_X3DGeometricPropertyNode):
    """
    Base type for color specifications in X3D.
    """
    def NAME(self):
        return '_X3DColorNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DColorNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DComposableVolumeRenderStyleNode(_X3DVolumeRenderStyleNode):
    """
    The X3DComposableVolumeRenderStyleNode abstract node type is the base type for all node types that allow rendering styles to be sequentially composed together to form a single renderable output.
    """
    def NAME(self):
        return '_X3DComposableVolumeRenderStyleNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DComposableVolumeRenderStyleNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DComposedGeometryNode(_X3DGeometryNode):
    """
    Composed geometry nodes produce renderable geometry, can contain Color Coordinate Normal TextureCoordinate, and are contained by a Shape node.
    """
    def NAME(self):
        return '_X3DComposedGeometryNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DComposedGeometryNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DCoordinateNode(_X3DGeometricPropertyNode):
    """
    Base type for all coordinate node types in X3D.
    """
    def NAME(self):
        return '_X3DCoordinateNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DCoordinateNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DDamperNode(_X3DFollowerNode):
    """
    The X3DDamperNode abstract node type creates an IIR response that approaches the destination value according to the shape of the e-function only asymptotically but very quickly.
    """
    def NAME(self):
        return '_X3DDamperNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DDamperNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DDragSensorNode(_X3DPointingDeviceSensorNode):
    """
    Base type for all drag-style pointing device sensors.
    """
    def NAME(self):
        return '_X3DDragSensorNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DDragSensorNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DEnvironmentalSensorNode(_X3DSensorNode):
    """
    Base type for the environmental sensor nodes ProximitySensor, TransformSensor and VisibilitySensor.
    """
    def NAME(self):
        return '_X3DEnvironmentalSensorNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DEnvironmentalSensorNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DEnvironmentTextureNode(_X3DTextureNode):
    """
    Base type for all nodes that specify cubic environment map sources for texture images.
    """
    def NAME(self):
        return '_X3DEnvironmentTextureNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DEnvironmentTextureNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DFontStyleNode(_X3DNode):
    """
    Base type for all font style nodes.
    """
    def NAME(self):
        return '_X3DFontStyleNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DFontStyleNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DGroupingNode(_X3DChildNode):
    """
    Grouping nodes can contain other nodes as children, thus making up the backbone of a scene graph.
    """
    def NAME(self):
        return '_X3DGroupingNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DGroupingNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DInfoNode(_X3DChildNode):
    """
    Base type for all nodes that contain only information without visual semantics.
    """
    def NAME(self):
        return '_X3DInfoNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DInfoNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DInterpolatorNode(_X3DChildNode):
    """
    Interpolator nodes are designed for linear keyframed animation. Interpolators are driven by an input key ranging [0..1] and produce corresponding piecewise-linear output functions.
    """
    def NAME(self):
        return '_X3DInterpolatorNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DInterpolatorNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DKeyDeviceSensorNode(_X3DSensorNode):
    """
    Base type for all sensor node types that operate using key devices.
    """
    def NAME(self):
        return '_X3DKeyDeviceSensorNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DKeyDeviceSensorNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DLayerNode(_X3DNode):
    """
    The X3DLayerNode abstract node type is the base node type for layer nodes.
    """
    def NAME(self):
        return '_X3DLayerNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DLayerNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DLayoutNode(_X3DChildNode):
    """
    X3DLayoutNode is the base node type for layout nodes.
    """
    def NAME(self):
        return '_X3DLayoutNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DLayoutNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DLightNode(_X3DChildNode):
    """
    Light nodes provide illumination for rendering geometry in the scene. Implementing nodes must include a global field with type SFBool and accessType inputOutput.
    """
    def NAME(self):
        return '_X3DLightNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DLightNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DMaterialNode(_X3DAppearanceChildNode):
    """
    Base type for all Material nodes.
    """
    def NAME(self):
        return '_X3DMaterialNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DMaterialNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DNBodyCollidableNode(_X3DChildNode):
    """
    The X3DNBodyCollidableNode abstract node type represents objects that act as the interface between the rigid body physics, collision geometry proxy, and renderable objects in the scene graph hierarchy.
    """
    def NAME(self):
        return '_X3DNBodyCollidableNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DNBodyCollidableNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DNBodyCollisionSpaceNode(_X3DNode):
    """
    The X3DNBodyCollisionSpaceNode abstract node type represents objects that act as a self-contained spatial collection of objects that can interact through collision detection routines.
    """
    def NAME(self):
        return '_X3DNBodyCollisionSpaceNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DNBodyCollisionSpaceNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DNetworkSensorNode(_X3DSensorNode):
    """
    Base typefor all sensors that generate events based on network activity.
    """
    def NAME(self):
        return '_X3DNetworkSensorNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DNetworkSensorNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DNormalNode(_X3DGeometricPropertyNode):
    """
    Base type for all normal node types in X3D.
    """
    def NAME(self):
        return '_X3DNormalNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DNormalNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DNurbsControlCurveNode(_X3DNode):
    """
    Base type for all nodes that provide control curve information in 2D space.
    """
    def NAME(self):
        return '_X3DNurbsControlCurveNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DNurbsControlCurveNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DNurbsSurfaceGeometryNode(_X3DParametricGeometryNode):
    """
    Abstract geometry type for all types of NURBS surfaces.
    """
    def NAME(self):
        return '_X3DNurbsSurfaceGeometryNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DNurbsSurfaceGeometryNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DParticleEmitterNode(_X3DNode):
    """
    The X3DParticleEmitterNode abstract type represents any node that is an emitter of particles.
    """
    def NAME(self):
        return '_X3DParticleEmitterNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DParticleEmitterNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DParticlePhysicsModelNode(_X3DNode):
    """
    The X3DParticlePhysicsModelNode abstract type represents any node that applies a form of constraints on the particles after they have been generated.
    """
    def NAME(self):
        return '_X3DParticlePhysicsModelNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DParticlePhysicsModelNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DPickSensorNode(_X3DSensorNode):
    """
    The X3DPickSensorNode abstract node type is the base node type that represents the lowest common denominator of picking capabilities.
    """
    def NAME(self):
        return '_X3DPickSensorNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DPickSensorNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DProductStructureChildNode(_X3DChildNode):
    """
    Base type marking nodes that are valid product structure children for the CADGeometry component.
    """
    def NAME(self):
        return '_X3DProductStructureChildNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DProductStructureChildNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DPrototypeInstance(_X3DNode):
    """
    Base type for all prototype instances. Note that direct children nodes are disallowed, instead let fieldValue with type SFNode/MFNode contain them. Current practice is that, if desired, prototype authors must explicitly add the metadata SFNode field in the ProtoInterface.
    """
    def NAME(self):
        return '_X3DPrototypeInstance'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DPrototypeInstance __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DRigidJointNode(_X3DNode):
    """
    The X3DRigidJointNode abstract node type is the base type for all joint types.
    """
    def NAME(self):
        return '_X3DRigidJointNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DRigidJointNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DScriptNode(_X3DChildNode):
    """
    Base type for scripting nodes (but not shader nodes).
    """
    def NAME(self):
        return '_X3DScriptNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DScriptNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DSequencerNode(_X3DChildNode):
    """
    Base type from which all Sequencers are derived.
    """
    def NAME(self):
        return '_X3DSequencerNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DSequencerNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DShaderNode(_X3DAppearanceChildNode):
    """
    Base type for all nodes that specify a programmable shader.
    """
    def NAME(self):
        return '_X3DShaderNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DShaderNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DShapeNode(_X3DChildNode):
    """
    Base type for all Shape nodes.
    """
    def NAME(self):
        return '_X3DShapeNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DShapeNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DSoundNode(_X3DChildNode):
    """
    Base type for all sound nodes.
    """
    def NAME(self):
        return '_X3DSoundNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DSoundNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DSoundSourceNode(_X3DTimeDependentNode):
    """
    Nodes implementing X3DSoundSourceNode are allowed as children of Sound node.
    """
    def NAME(self):
        return '_X3DSoundSourceNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DSoundSourceNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DTexture2DNode(_X3DTextureNode):
    """
    Base type for all nodes which specify 2D sources for texture images.
    """
    def NAME(self):
        return '_X3DTexture2DNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DTexture2DNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DTexture3DNode(_X3DTextureNode):
    """
    Base type for all nodes that specify 3D sources for texture images.
    """
    def NAME(self):
        return '_X3DTexture3DNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DTexture3DNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DTextureCoordinateNode(_X3DGeometricPropertyNode):
    """
    Base type for all nodes which specify texture coordinates.
    """
    def NAME(self):
        return '_X3DTextureCoordinateNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DTextureCoordinateNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DTextureTransformNode(_X3DAppearanceChildNode):
    """
    Base type for all nodes which specify a transformation of texture coordinates.
    """
    def NAME(self):
        return '_X3DTextureTransformNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DTextureTransformNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DTouchSensorNode(_X3DPointingDeviceSensorNode):
    """
    Base type for all touch-style pointing device sensors.
    """
    def NAME(self):
        return '_X3DTouchSensorNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DTouchSensorNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DTriggerNode(_X3DChildNode):
    """
    Base type from which all trigger nodes are derived.
    """
    def NAME(self):
        return '_X3DTriggerNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DTriggerNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DVertexAttributeNode(_X3DGeometricPropertyNode):
    """
    Base type for all nodes that specify per-vertex attribute information to the shader.
    """
    def NAME(self):
        return '_X3DVertexAttributeNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DVertexAttributeNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DViewpointNode(_X3DBindableNode):
    """
    Node type X3DViewpointNode defines a specific location in the local coordinate system from which the user may view the scene, and also defines a viewpoint binding stack.
    """
    def NAME(self):
        return '_X3DViewpointNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DViewpointNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DViewportNode(_X3DGroupingNode):
    """
    The X3DViewportNode abstract node type is the base node type for viewport nodes.
    """
    def NAME(self):
        return '_X3DViewportNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DViewportNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DVolumeDataNode(_X3DChildNode):
    """
    The X3DVolumeDataNode abstract node type is the base type for all node types that describe volumetric data to be rendered.
    """
    def NAME(self):
        return '_X3DVolumeDataNode'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractNodeType X3DVolumeDataNode __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

###############################################

# Abstract Object Types

# Note that these package-internal class names are preceded by an underscore _ character since X3D authors are not expected to use them

class _X3DBoundedObject(_X3DNode):
    """
    X3DBoundedObject indicates that bounding box values can be provided (or computed) to encompass this node and any children.
    """
    def NAME(self):
        return '_X3DBoundedObject'
    def SPECIFICATION_URL(self):
        return ''

class _X3DFogObject(_X3DNode):
    """
    Abstract type describing a node that influences the lighting equation through the use of fog semantics.
    """
    def NAME(self):
        return '_X3DFogObject'
    def SPECIFICATION_URL(self):
        return ''

class _X3DMetadataObject(_X3DNode):
    """
    Each node inheriting the X3DMetadataObject interface contains a single array of strictly typed values: MFBool, MFInt32, MFFloat, MFDouble, MFString, or MFNode, the latter having children that are all Metadata nodes.
    """
    def NAME(self):
        return '_X3DMetadataObject'
    def SPECIFICATION_URL(self):
        return ''
    def __init__(self, DEF, USE, class_, IS, metadata):
        # if _DEBUG: print('... in AbstractObjectType X3DMetadataObject __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only

class _X3DPickableObject(_X3DNode):
    """
    The X3DPickableObject abstract interface marks a node as being capable of having customized picking performed on its contents or children.
    """
    def NAME(self):
        return '_X3DPickableObject'
    def SPECIFICATION_URL(self):
        return ''

class _X3DProgrammableShaderObject(_X3DNode):
    """
    Base type for all nodes that specify arbitrary fields for interfacing with per-object attribute values.
    """
    def NAME(self):
        return '_X3DProgrammableShaderObject'
    def SPECIFICATION_URL(self):
        return ''

class _X3DUrlObject(_X3DNode):
    """
    X3DUrlObject indicates that a node has content loaded from a Uniform Resource Locator (URL) and can be tracked via a LoadSensor. Such child nodes have containerField='watchList' to indicate their relationship to the parent LoadSensor node.
    """
    def NAME(self):
        return '_X3DUrlObject'
    def SPECIFICATION_URL(self):
        return ''

###############################################

# Statements

class _X3DStatement(object):
    """
    All X3D statements implement _X3DStatement abstract type.
    """
    def NAME(self):
        return '_X3DStatement'
    def SPECIFICATION_URL(self):
        return 'https://www.web3d.org/documents/specifications/19775-1/V3.3/Part01/components/core.html#AbstractX3DStructure'
    def __repl__(self):
        result = self.NAME() + '('
        # if _DEBUG: print(self.NAME() + ' self.FIELD_DECLARATIONS() : ' + str(self.FIELD_DECLARATIONS() ))
        if self.FIELD_DECLARATIONS():
            for each in self.FIELD_DECLARATIONS():
                # if _DEBUG: print(self.NAME() + ' for each in self.FIELD_DECLARATIONS(): each=' + str(each))
                name = each[0]
                default = each[1]
                type_ = each[2]
                accessType = each[3]
                value = getattr(self, name)
                # if _DEBUG: print('gettattr(self, ' + str(name) + ') value="' + str(value)[:100] + '" for FIELD_DECLARATIONS ' + str(each) + ')', flush=True)
                if value != default:
                    if  isinstance(value, list): # avoid X3DTypeError if value is not iterable
                        result += str(name) + '=['
                        for each in value:
                            result += str(each) + ', '
                            # if _DEBUG: print('* _X3DStatement debug: str(each)=' + str(each), flush=True)
                        result = result.rstrip(', ')
                        result += '],'
                    elif isinstance(value, str) and "'" in value:
                        result += str(name) + '=' + '"' + str(value)[:100] + '"' + ','
                    elif isinstance(value, str) and value != default:
                        result += str(name) + '=' + "'" + str(value)[:100] + "'" + ','
                    elif value != default:
                        result += str(name) + '='       + str(value)[:100]       + ','
                    # elif _DEBUG:
        #   result += str(name) + '=' + "'" + str(value)[:100] + "'" + ','
        return result.strip().rstrip(',').rstrip(', ') + ')'
    def __str__(self):
        return self.__repl__().strip() # _X3DStatement

def isX3DStatement(value):
    """
    Whether or not value is an _X3DStatement object.
    """
    return isinstance(value, _X3DStatement)

class Comment(_X3DStatement):
    """
    X3D statement containing zero or more comment strings.
    """
    # immutable constant functions - - - - - - - - - -
    def NAME(self):
        return 'Comment'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html'
    def FIELD_DECLARATIONS(self):
        return []
    def REGEX_XML(self):
        return r'(\s|\S)*' # (includes lower-case true, false)
    def __init__(self, value=None):
        self.value = value
    @property # getter - - - - - - - - - -
    def value(self):
        """ Provide list of comment strings. """
        return self.__value
    @value.setter
    def value(self, value):
        """ The value setter only allows correctly typed values. """
        if  value is None:
            value = self.DEFAULT_VALUE()
        self.__value = str(value)
    # immutable constant functions - - - - - - - - - -
    def DEFAULT_VALUE(self):
        return ''
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        result = ''
        indent = '  ' * indentLevel
        if self.value:
            result = indent + '<!-- ' + self.value + ' -->' + '\n'
        return result
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        result = ''
        indent = '  ' * indentLevel
        if self.value:
            result = '\n' + indent + '# ' + self.value
        return result

def isComment(value):
    """
    Whether or not value is an _X3DComment object.
    """
    return isinstance(value, _X3DComment)

class component(_X3DStatement):
    """
    Functional summary: each added component statement indicates needed scene functionality support above the given X3D profile.
    """
    def NAME(self):
        return 'component'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#component'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('level', 1, FieldType.SFInt32, AccessType.inputOutput, 'component'),
        ('name', '', FieldType.SFString, AccessType.inputOutput, 'component')]
    def __init__(self,
                 level=1,
                 name=''):
        self.level = level
        self.name = name
    @property # getter - - - - - - - - - -
    def level(self):
        """Necessary level of support for this scene, as defined in corresponding Support table for a given node's component."""
        return self.__level
    @level.setter
    def level(self, level):
        if  level is None:
            level = 1 # default
        assertValidSFInt32(level)
        assertGreaterThanEquals('level', level, 1)
        assertLessThanEquals('level', level, 5)
        self.__level = level
    @property # getter - - - - - - - - - -
    def name(self):
        """Provides name of this component, as defined in corresponding X3D Specification component Introduction."""
        return self.__name
    @name.setter
    def name(self, name):
        if  name is None:
            name = SFString.DEFAULT_VALUE(self)
        assertValidSFString(name)
        assertValidComponentName('name', name)
        self.__name = name
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return False
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function component.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<component'
        if self.level != 1:
            result += " level='" + SFInt32(self.level).XML() + "'"
        if self.name:
            result += " name='" + self.name + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></component>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            result += indent + '</component>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        result += 'COMPONENT ' + self.name + ':' + str(self.level) + '\n'
#       print('VRML serialization complete.', flush=True)
        return result

class connect(_X3DStatement):
    """
    Functional summary: connect statements define event-routing connections between node fields defined inside a ProtoBody declaration back to corresponding ProtoInterface fields.
    """
    def NAME(self):
        return 'connect'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#connect'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('nodeField', '', FieldType.SFString, AccessType.inputOutput, 'connect'),
        ('protoField', '', FieldType.SFString, AccessType.inputOutput, 'connect')]
    def __init__(self,
                 nodeField='',
                 protoField=''):
        self.nodeField = nodeField
        self.protoField = protoField
    @property # getter - - - - - - - - - -
    def nodeField(self):
        """Name of field within this node which IS CONNECTed to the ancestor ProtoDeclare field definition."""
        return self.__nodeField
    @nodeField.setter
    def nodeField(self, nodeField):
        if  nodeField is None:
            nodeField = SFString.DEFAULT_VALUE(self)
        assertValidSFString(nodeField)
        self.__nodeField = nodeField
    @property # getter - - - - - - - - - -
    def protoField(self):
        """Name of parent ProtoDeclare field definition connecting to field in this node."""
        return self.__protoField
    @protoField.setter
    def protoField(self, protoField):
        if  protoField is None:
            protoField = SFString.DEFAULT_VALUE(self)
        assertValidSFString(protoField)
        self.__protoField = protoField
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return False
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function connect.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<connect'
        if self.nodeField:
            result += " nodeField='" + self.nodeField + "'"
        if self.protoField:
            result += " protoField='" + self.protoField + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></connect>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            result += indent + '</connect>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function connect.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'connect' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'connect' + ' {'
        if self.nodeField:
            result += " nodeField " +  '"' + self.nodeField + '"' + ""
        if self.protoField:
            result += " protoField " +  '"' + self.protoField + '"' + ""
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class EXPORT(_X3DStatement):
    """
    Functional summary: EXPORT exposes a local node for ROUTE passing of event values when the current Scene is included via Inline by a parent external world. These connections allow event values to be exchanged via ROUTE statements between a parent model and a child Inline model.
    """
    def NAME(self):
        return 'EXPORT'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#EXPORT'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('AS', '', FieldType.SFString, AccessType.inputOutput, 'EXPORT'),
        ('localDEF', '', FieldType.SFString, AccessType.inputOutput, 'EXPORT')]
    def __init__(self,
                 AS='',
                 localDEF=''):
        self.AS = AS
        self.localDEF = localDEF
    @property # getter - - - - - - - - - -
    def AS(self):
        """rename localDEF node AS a different name when exporting."""
        return self.__AS
    @AS.setter
    def AS(self, AS):
        if  AS is None:
            AS = SFString.DEFAULT_VALUE(self)
        assertValidSFString(AS)
        self.__AS = AS
    @property # getter - - - - - - - - - -
    def localDEF(self):
        """localDEF is the DEF name of the local node to be EXPORTed."""
        return self.__localDEF
    @localDEF.setter
    def localDEF(self, localDEF):
        if  localDEF is None:
            localDEF = SFString.DEFAULT_VALUE(self)
        assertValidSFString(localDEF)
        self.__localDEF = localDEF
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return False
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function EXPORT.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<EXPORT'
        if self.AS:
            result += " AS='" + self.AS + "'"
        if self.localDEF:
            result += " localDEF='" + self.localDEF + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></EXPORT>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            result += indent + '</EXPORT>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function EXPORT.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'EXPORT' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'EXPORT' + ' {'
        if self.AS:
            result += " AS " +  '"' + self.AS + '"' + ""
        if self.localDEF:
            result += " localDEF " +  '"' + self.localDEF + '"' + ""
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class ExternProtoDeclare(_X3DStatement):
    """
    ExternProtoDeclare refers to a ProtoDeclare node declaration provided in another file. ExternProtoDeclare interfaces are defined by field statements (and without IS/connect statements).
    """
    def NAME(self):
        return 'ExternProtoDeclare'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ExternProtoDeclare'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('appinfo', '', FieldType.SFString, AccessType.inputOutput, 'ExternProtoDeclare'),
        ('documentation', '', FieldType.SFString, AccessType.inputOutput, 'ExternProtoDeclare'),
        ('name', '', FieldType.SFString, AccessType.inputOutput, 'ExternProtoDeclare'),
        ('url', list(), FieldType.MFString, AccessType.inputOutput, 'ExternProtoDeclare'),
        ('field', list(), FieldType.MFNode, AccessType.inputOutput, 'ExternProtoDeclare')]
    def __init__(self,
                 appinfo='',
                 documentation='',
                 name='',
                 url=list(),
                 field=None):
        self.appinfo = appinfo
        self.documentation = documentation
        self.name = name
        self.url = url
        self.field = field
    @property # getter - - - - - - - - - -
    def appinfo(self):
        """Application information to provide simple description usable as a tooltip, similar to XML Schema appinfo tag."""
        return self.__appinfo
    @appinfo.setter
    def appinfo(self, appinfo):
        if  appinfo is None:
            appinfo = SFString.DEFAULT_VALUE(self)
        assertValidSFString(appinfo)
        self.__appinfo = appinfo
    @property # getter - - - - - - - - - -
    def documentation(self):
        """Documentation url for further information, similar to XML Schema documentation tag."""
        return self.__documentation
    @documentation.setter
    def documentation(self, documentation):
        if  documentation is None:
            documentation = SFString.DEFAULT_VALUE(self)
        assertValidSFString(documentation)
        self.__documentation = documentation
    @property # getter - - - - - - - - - -
    def name(self):
        """name of the ExternProtoDeclare (External Prototype Declaration) being referenced."""
        return self.__name
    @name.setter
    def name(self, name):
        if  name is None:
            name = SFString.DEFAULT_VALUE(self)
        assertValidSFString(name)
        self.__name = name
    @property # getter - - - - - - - - - -
    def url(self):
        """Location and filename of ProtoDeclare source declaration of interest."""
        return self.__url
    @url.setter
    def url(self, url):
        if  url is None:
            url = MFString.DEFAULT_VALUE(self)
        assertValidMFString(url)
        self.__url = url
    @property # getter - - - - - - - - - -
    def field(self):
        """Include a field statement for each field declaration in the corresponding original ProtoDeclare."""
        return self.__field
    @field.setter
    def field(self, field):
        if  field is None:
            field = MFNode.DEFAULT_VALUE(self)
        # TODO type-aware checks for field
        if field: # walk each child in MFNode list, if any
            for each in __field:
                assertValidFieldInitializationValue(each.name, type(each.value), each.value, parent='ExternProtoDeclare/field')
        self.__field = field
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.field)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ExternProtoDeclare.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ExternProtoDeclare'
        if self.appinfo:
            result += " appinfo='" + self.appinfo + "'"
        if self.documentation:
            result += " documentation='" + self.documentation + "'"
        if self.name:
            result += " name='" + self.name + "'"
        if self.url != list():
            result += " url='" + MFString(self.url).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></ExternProtoDeclare>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            ### if self.field: # walk each child in MFNode list, if any
            ### print('* ExternProtoDeclare found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(field)=' + str(len(self.field)) + ', now invoking XML(' + str(indentLevel+1) + ')', flush=True)
            for each in self.field:
               result += each.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</ExternProtoDeclare>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function ExternProtoDeclare.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'ExternProtoDeclare' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'ExternProtoDeclare' + ' {'
        if self.appinfo:
            result += " appinfo " +  '"' + self.appinfo + '"' + ""
        if self.documentation:
            result += " documentation " +  '"' + self.documentation + '"' + ""
        if self.name:
            result += " name " +  '"' + self.name + '"' + ""
        if self.url != list():
            result += " url " + MFString(self.url).VRML() + ""
        if self.field: # walk each child in MFNode list, if any
            for each in self.field:
                result += each.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class field(_X3DStatement):
    """
    Functional summary: a field statement defines an interface attribute or node. Each field statement can contain either attribute-value or node content.
    """
    def NAME(self):
        return 'field'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#field'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('accessType', '', FieldType.SFString, AccessType.inputOutput, 'field'),
        ('appinfo', '', FieldType.SFString, AccessType.inputOutput, 'field'),
        ('documentation', '', FieldType.SFString, AccessType.inputOutput, 'field'),
        ('name', '', FieldType.SFString, AccessType.inputOutput, 'field'),
        ('type', '', FieldType.SFString, AccessType.inputOutput, 'field'),
        ('value', '', FieldType.SFString, AccessType.inputOutput, 'field'),
        ('children', list(), FieldType.MFNode, AccessType.inputOutput, 'field')]
    def __init__(self,
                 accessType='',
                 appinfo='',
                 documentation='',
                 name='',
                 type='',
                 value='',
                 children=None):
        self.accessType = accessType
        self.appinfo = appinfo
        self.documentation = documentation
        self.name = name
        self.type = type
        self.value = value
        self.children = children
    @property # getter - - - - - - - - - -
    def accessType(self):
        """Event-model semantics for field set/get capabilities."""
        return self.__accessType
    @accessType.setter
    def accessType(self, accessType):
        if  accessType is None:
            accessType = SFString.DEFAULT_VALUE(self)
        assertValidSFString(accessType)
        assertValidAccessType('accessType', accessType)
        self.__accessType = accessType
    @property # getter - - - - - - - - - -
    def appinfo(self):
        """Application information to provide simple description usable as a tooltip, similar to XML Schema appinfo tag."""
        return self.__appinfo
    @appinfo.setter
    def appinfo(self, appinfo):
        if  appinfo is None:
            appinfo = SFString.DEFAULT_VALUE(self)
        assertValidSFString(appinfo)
        self.__appinfo = appinfo
    @property # getter - - - - - - - - - -
    def documentation(self):
        """Documentation url for further information, similar to XML Schema documentation tag."""
        return self.__documentation
    @documentation.setter
    def documentation(self, documentation):
        if  documentation is None:
            documentation = SFString.DEFAULT_VALUE(self)
        assertValidSFString(documentation)
        self.__documentation = documentation
    @property # getter - - - - - - - - - -
    def name(self):
        """Name of this field declaration."""
        return self.__name
    @name.setter
    def name(self, name):
        if  name is None:
            name = SFString.DEFAULT_VALUE(self)
        assertValidSFString(name)
        self.__name = name
    @property # getter - - - - - - - - - -
    def type(self):
        """Base type of this field variable."""
        return self.__type
    @type.setter
    def type(self, type):
        if  type is None:
            type = SFString.DEFAULT_VALUE(self)
        assertValidSFString(type)
        assertValidFieldType('type', type)
        self.__type = type
    @property # getter - - - - - - - - - -
    def value(self):
        """Provide default initialization value for this field variable (which may be re-initialized later by instantiation value of a named ProtoInstance fieldValue)."""
        return self.__value
    @value.setter
    def value(self, value):
        if  value is None:
            value = SFString.DEFAULT_VALUE(self)
        assertValidFieldInitializationValue(self.name, self.type, value, parent='field/@value')
        self.__value = value
    @property # getter - - - - - - - - - -
    def children(self):
        return self.__children
    @children.setter
    def children(self, children):
        if  children is None:
            children = MFNode.DEFAULT_VALUE(self)
        assertValidMFNode(children)
        self.__children = children
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.children)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function field.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<field'
        if self.accessType:
            result += " accessType='" + self.accessType + "'"
        if self.appinfo:
            result += " appinfo='" + self.appinfo + "'"
        if self.documentation:
            result += " documentation='" + self.documentation + "'"
        if self.name:
            result += " name='" + self.name + "'"
        if self.type:
            result += " type='" + self.type + "'"
        if self.value:
            result += " value='" + SFString(self.value).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></field>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            ### if self.children: # walk each child in MFNode list, if any
            ### print('* field found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(children)=' + str(len(self.children)) + ', now invoking XML(' + str(indentLevel+1) + ')', flush=True)
            for each in self.children:
               result += each.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</field>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function field.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'field' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'field' + ' {'
        if self.accessType:
            result += " accessType " +  '"' + self.accessType + '"' + ""
        if self.appinfo:
            result += " appinfo " +  '"' + self.appinfo + '"' + ""
        if self.documentation:
            result += " documentation " +  '"' + self.documentation + '"' + ""
        if self.name:
            result += " name " +  '"' + self.name + '"' + ""
        if self.type:
            result += " type " +  '"' + self.type + '"' + ""
        if self.value:
            result += " value " +  '"' + self.value + '"' + ""
        if self.children: # walk each child in MFNode list, if any
            for each in self.children:
                result += each.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class fieldValue(_X3DStatement):
    """
    Functional summary: a fieldValue statement re-initializes the default value of a field in a ProtoInstance. Each fieldValue statement can contain either attribute-value or node content.
    """
    def NAME(self):
        return 'fieldValue'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#fieldValue'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('name', '', FieldType.SFString, AccessType.inputOutput, 'fieldValue'),
        ('value', '', FieldType.SFString, AccessType.inputOutput, 'fieldValue'),
        ('children', list(), FieldType.MFNode, AccessType.inputOutput, 'fieldValue')]
    def __init__(self,
                 name='',
                 value='',
                 children=None):
        self.name = name
        self.value = value
        self.children = children
    @property # getter - - - - - - - - - -
    def name(self):
        """Name of the ProtoInstance field being re-initialized (corresponding to field name already defined in ProtoDeclare or ExternProtoDeclare)."""
        return self.__name
    @name.setter
    def name(self, name):
        if  name is None:
            name = SFString.DEFAULT_VALUE(self)
        assertValidSFString(name)
        self.__name = name
    @property # getter - - - - - - - - - -
    def value(self):
        """Initial value for this field, which overrides default initialization value defined in original ProtoDeclare field."""
        return self.__value
    @value.setter
    def value(self, value):
        if  value is None:
            value = SFString.DEFAULT_VALUE(self)
        assertValidFieldInitializationValue(self.name, type(value), value, parent='fieldValue')
        self.__value = value
    @property # getter - - - - - - - - - -
    def children(self):
        return self.__children
    @children.setter
    def children(self, children):
        if  children is None:
            children = MFNode.DEFAULT_VALUE(self)
        assertValidMFNode(children)
        self.__children = children
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.children)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function fieldValue.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<fieldValue'
        if self.name:
            result += " name='" + self.name + "'"
        if self.value:
            result += " value='" + SFString(self.value).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></fieldValue>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            ### if self.children: # walk each child in MFNode list, if any
            ### print('* fieldValue found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(children)=' + str(len(self.children)) + ', now invoking XML(' + str(indentLevel+1) + ')', flush=True)
            for each in self.children:
               result += each.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</fieldValue>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function fieldValue.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'fieldValue' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'fieldValue' + ' {'
        if self.name:
            result += " name " +  '"' + self.name + '"' + ""
        if self.value:
            result += " value " +  '"' + self.value + '"' + ""
        if self.children: # walk each child in MFNode list, if any
            for each in self.children:
                result += each.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class head(_X3DStatement):
    """
    Functional summary: each X3D scene includes a head statement that can contain component, unit and meta statements.
    """
    def NAME(self):
        return 'head'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#head'
    def FIELD_DECLARATIONS(self):
        return [('children', None, FieldType.MFNode, AccessType.inputOutput, 'head')]
    def __init__(self, children=None):
        self.children = children
    @property # getter - - - - - - - - - -
    def children(self):
        """ The head statement has children consisting of component, unit and meta statements. """
        return self.__children
    @children.setter
    def children(self, children):
        if  children is None:
            children = MFNode.DEFAULT_VALUE(self)
        assertValidMFNode(children)
        self.__children = children
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return bool(self.children)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function head.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<head'
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></head>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.__children: # walk each child in MFNode list, if any
                ## print('* head found self.children, now invoking XML(' + str(indentLevel+1) + ')', flush=True)
                # order is significant for component, unit, meta statements
                for each in self.children:
                    if isinstance(each, component):
                        result += each.XML(indentLevel=indentLevel+1,syntax=syntax)
                for each in self.children:
                    if isinstance(each, unit):
                        result += each.XML(indentLevel=indentLevel+1,syntax=syntax)
                for each in self.children:
                    if isinstance(each, meta):
                        result += each.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</head>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        if self.children: # walk each child in MFNode list, if any
            ## print('* head found self.children, now invoking VRML(' + str(indentLevel+1) + ', ' + VRML97 + ')', flush=True)
            # order is significant for component, unit, meta statements
            for each in self.children:
                if isinstance(each, component):
                    result += each.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
            for each in self.children:
                if isinstance(each, unit):
                    result += each.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
            for each in self.children:
                if isinstance(each, meta):
                    result += each.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
#       print('VRML serialization complete.', flush=True)
        return result

class IMPORT(_X3DStatement):
    """
    Functional summary: IMPORT provides ROUTE access to a node that has a corresponding EXPORT statement within an Inline scene. These connections allow event values to be exchanged via ROUTE statements between a parent model and a child Inline model.
    """
    def NAME(self):
        return 'IMPORT'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#IMPORT'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('AS', '', FieldType.SFString, AccessType.inputOutput, 'IMPORT'),
        ('importedDEF', '', FieldType.SFString, AccessType.inputOutput, 'IMPORT'),
        ('inlineDEF', '', FieldType.SFString, AccessType.inputOutput, 'IMPORT')]
    def __init__(self,
                 AS='',
                 importedDEF='',
                 inlineDEF=''):
        self.AS = AS
        self.importedDEF = importedDEF
        self.inlineDEF = inlineDEF
    @property # getter - - - - - - - - - -
    def AS(self):
        """map importedDEF name AS a new name in current scene."""
        return self.__AS
    @AS.setter
    def AS(self, AS):
        if  AS is None:
            AS = SFString.DEFAULT_VALUE(self)
        assertValidSFString(AS)
        self.__AS = AS
    @property # getter - - - - - - - - - -
    def importedDEF(self):
        """importedDEF is DEF name of the node of interest that is contained in the remote inlineDEF scene."""
        return self.__importedDEF
    @importedDEF.setter
    def importedDEF(self, importedDEF):
        if  importedDEF is None:
            importedDEF = SFString.DEFAULT_VALUE(self)
        assertValidSFString(importedDEF)
        self.__importedDEF = importedDEF
    @property # getter - - - - - - - - - -
    def inlineDEF(self):
        """inlineDEF is the DEF name of Inline node in the same scene as this IMPORT statement."""
        return self.__inlineDEF
    @inlineDEF.setter
    def inlineDEF(self, inlineDEF):
        if  inlineDEF is None:
            inlineDEF = SFString.DEFAULT_VALUE(self)
        assertValidSFString(inlineDEF)
        self.__inlineDEF = inlineDEF
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return False
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function IMPORT.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<IMPORT'
        if self.AS:
            result += " AS='" + self.AS + "'"
        if self.importedDEF:
            result += " importedDEF='" + self.importedDEF + "'"
        if self.inlineDEF:
            result += " inlineDEF='" + self.inlineDEF + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></IMPORT>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            result += indent + '</IMPORT>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function IMPORT.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'IMPORT' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'IMPORT' + ' {'
        if self.AS:
            result += " AS " +  '"' + self.AS + '"' + ""
        if self.importedDEF:
            result += " importedDEF " +  '"' + self.importedDEF + '"' + ""
        if self.inlineDEF:
            result += " inlineDEF " +  '"' + self.inlineDEF + '"' + ""
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class IS(_X3DStatement):
    """
    Functional summary: the IS statement connects node fields defined inside a ProtoBody declaration back to corresponding ProtoInterface fields. IS/connect statements can be added if the parent node is within a ProtoBody and connect statements define correspondences between prototype fields and built-in node fields.
    """
    def NAME(self):
        return 'IS'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#IS'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('connect', list(), FieldType.MFNode, AccessType.inputOutput, 'IS')]
    def __init__(self,
                 connect=None):
        self.connect = connect
    @property # getter - - - - - - - - - -
    def connect(self):
        """When inside a ProtoBody declaration and an IS statement, add a connect statement to define event-routing connections between a parent node's field to a corresponding ProtoInterface field."""
        return self.__connect
    @connect.setter
    def connect(self, connect):
        if  connect is None:
            connect = MFNode.DEFAULT_VALUE(self)
        assertValidMFNode(connect)
        self.__connect = connect
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.connect)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function IS.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<IS'
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></IS>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            ### if self.connect: # walk each child in MFNode list, if any
            ### print('* IS found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(connect)=' + str(len(self.connect)) + ', now invoking XML(' + str(indentLevel+1) + ')', flush=True)
            for each in self.connect:
               result += each.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</IS>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function IS.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'IS' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'IS' + ' {'
        if self.connect: # walk each child in MFNode list, if any
            for each in self.connect:
                result += each.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class meta(_X3DStatement):
    """
    Functional summary: the meta statement provides metadata information about a scene, where name and content attributes provide attribute=value metadata pairs.
    """
    def NAME(self):
        return 'meta'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#meta'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('content', '', FieldType.SFString, AccessType.inputOutput, 'meta'),
        ('dir', '', FieldType.SFString, AccessType.inputOutput, 'meta'),
        ('httpequiv', '', FieldType.SFString, AccessType.inputOutput, 'meta'),
        ('lang', '', FieldType.SFString, AccessType.inputOutput, 'meta'),
        ('name', '', FieldType.SFString, AccessType.inputOutput, 'meta'),
        ('scheme', '', FieldType.SFString, AccessType.inputOutput, 'meta')]
    def __init__(self,
                 content='',
                 dir='',
                 httpequiv='',
                 lang='',
                 name='',
                 scheme=''):
        self.content = content
        self.dir = dir
        self.httpequiv = httpequiv
        self.lang = lang
        self.name = name
        self.scheme = scheme
    @property # getter - - - - - - - - - -
    def content(self):
        """The content attribute provides metadata information relevant to the name attribute provided."""
        return self.__content
    @content.setter
    def content(self, content):
        if  content is None:
            content = SFString.DEFAULT_VALUE(self)
        assertValidSFString(content)
        self.__content = content
    @property # getter - - - - - - - - - -
    def dir(self):
        """Direction for weak/neutral text (ltr=left-to-right, rtl=right-to-left)."""
        return self.__dir
    @dir.setter
    def dir(self, dir):
        if  dir is None:
            dir = SFString.DEFAULT_VALUE(self)
        assertValidSFString(dir)
        assertValidMetaDirection('dir', dir)
        self.__dir = dir
    @property # getter - - - - - - - - - -
    def httpequiv(self):
        return self.__httpequiv
    @httpequiv.setter
    def httpequiv(self, httpequiv):
        if  httpequiv is None:
            httpequiv = SFString.DEFAULT_VALUE(self)
        assertValidSFString(httpequiv)
        self.__httpequiv = httpequiv
    @property # getter - - - - - - - - - -
    def lang(self):
        """Language code, as per [IETF BCP47/RFC5646] http://www."""
        return self.__lang
    @lang.setter
    def lang(self, lang):
        if  lang is None:
            lang = SFString.DEFAULT_VALUE(self)
        assertValidSFString(lang)
        self.__lang = lang
    @property # getter - - - - - - - - - -
    def name(self):
        """Keyword name of the meta attribute, following the same naming conventions as HTML's meta tag."""
        return self.__name
    @name.setter
    def name(self, name):
        if  name is None:
            name = SFString.DEFAULT_VALUE(self)
        assertValidSFString(name)
        self.__name = name
    @property # getter - - - - - - - - - -
    def scheme(self):
        """The scheme attribute allows authors to provide user agents more context for the correct interpretation of meta information."""
        return self.__scheme
    @scheme.setter
    def scheme(self, scheme):
        if  scheme is None:
            scheme = SFString.DEFAULT_VALUE(self)
        assertValidSFString(scheme)
        self.__scheme = scheme
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return False
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function meta.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<meta'
        if self.content:
            result += " content='" + self.content + "'"
        if self.dir:
            result += " dir='" + self.dir + "'"
        if self.httpequiv:
            result += " httpequiv='" + self.httpequiv + "'"
        if self.lang:
            result += " lang='" + self.lang + "'"
        if self.name:
            result += " name='" + self.name + "'"
        if self.scheme:
            result += " scheme='" + self.scheme + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></meta>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            result += indent + '</meta>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        result += 'META "' + self.name + '" "' + self.content + '"' + '\n'
#       print('VRML serialization complete.', flush=True)
        return result

class ProtoBody(_X3DStatement):
    """
    ProtoBody contains the definition nodes for new Prototype nodes.
    """
    def NAME(self):
        return 'ProtoBody'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ProtoBody'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('children', list(), FieldType.MFNode, AccessType.inputOutput, 'ProtoBody')]
    def __init__(self,
                 children=None):
        self.children = children
    @property # getter - - - - - - - - - -
    def children(self):
        return self.__children
    @children.setter
    def children(self, children):
        if  children is None:
            children = MFNode.DEFAULT_VALUE(self)
        assertValidMFNode(children)
        self.__children = children
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.children)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ProtoBody.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ProtoBody'
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></ProtoBody>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            ### if self.children: # walk each child in MFNode list, if any
            ### print('* ProtoBody found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(children)=' + str(len(self.children)) + ', now invoking XML(' + str(indentLevel+1) + ')', flush=True)
            for each in self.children:
               result += each.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</ProtoBody>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function ProtoBody.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'ProtoBody' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'ProtoBody' + ' {'
        if self.children: # walk each child in MFNode list, if any
            for each in self.children:
                result += each.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class ProtoDeclare(_X3DStatement):
    """
    ProtoDeclare defines new Prototype nodes. Nested ProtoDeclares and ProtoInstances are allowed by the specification.
    """
    def NAME(self):
        return 'ProtoDeclare'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ProtoDeclare'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('appinfo', '', FieldType.SFString, AccessType.inputOutput, 'ProtoDeclare'),
        ('documentation', '', FieldType.SFString, AccessType.inputOutput, 'ProtoDeclare'),
        ('name', '', FieldType.SFString, AccessType.inputOutput, 'ProtoDeclare'),
        ('ProtoBody', None, FieldType.SFNode, AccessType.inputOutput, 'ProtoDeclare'),
        ('ProtoInterface', None, FieldType.SFNode, AccessType.inputOutput, 'ProtoDeclare')]
    def __init__(self,
                 appinfo='',
                 documentation='',
                 name='',
                 ProtoBody=None,
                 ProtoInterface=None):
        self.appinfo = appinfo
        self.documentation = documentation
        self.name = name
        self.ProtoBody = ProtoBody
        self.ProtoInterface = ProtoInterface
    @property # getter - - - - - - - - - -
    def appinfo(self):
        """Application information to provide simple description usable as a tooltip, similar to XML Schema appinfo tag."""
        return self.__appinfo
    @appinfo.setter
    def appinfo(self, appinfo):
        if  appinfo is None:
            appinfo = SFString.DEFAULT_VALUE(self)
        assertValidSFString(appinfo)
        self.__appinfo = appinfo
    @property # getter - - - - - - - - - -
    def documentation(self):
        """Documentation url for further information, similar to XML Schema documentation tag."""
        return self.__documentation
    @documentation.setter
    def documentation(self, documentation):
        if  documentation is None:
            documentation = SFString.DEFAULT_VALUE(self)
        assertValidSFString(documentation)
        self.__documentation = documentation
    @property # getter - - - - - - - - - -
    def name(self):
        """name of this prototype being declared."""
        return self.__name
    @name.setter
    def name(self, name):
        if  name is None:
            name = SFString.DEFAULT_VALUE(self)
        assertValidSFString(name)
        self.__name = name
    @property # getter - - - - - - - - - -
    def ProtoBody(self):
        """Include one ProtoBody statements after the ProtoInterface statement."""
        return self.__ProtoBody
    @ProtoBody.setter
    def ProtoBody(self, ProtoBody):
        if  ProtoBody is None:
            ProtoBody = None # default
        assertValidSFNode(ProtoBody)
        if not isinstance(ProtoBody, object):
            # print(flush=True)
            raise X3DTypeError(str(ProtoBody) + ' does not have a valid node type object')
        self.__ProtoBody = ProtoBody
    @property # getter - - - - - - - - - -
    def ProtoInterface(self):
        """Include an optional ProtoInterface statement if this ProtoDeclare has field declarations."""
        return self.__ProtoInterface
    @ProtoInterface.setter
    def ProtoInterface(self, ProtoInterface):
        if  ProtoInterface is None:
            ProtoInterface = None # default
        assertValidSFNode(ProtoInterface)
        if not isinstance(ProtoInterface, object):
            # print(flush=True)
            raise X3DTypeError(str(ProtoInterface) + ' does not have a valid node type object')
        self.__ProtoInterface = ProtoInterface
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.ProtoBody) or (self.ProtoInterface)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ProtoDeclare.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ProtoDeclare'
        if self.appinfo:
            result += " appinfo='" + self.appinfo + "'"
        if self.documentation:
            result += " documentation='" + self.documentation + "'"
        if self.name:
            result += " name='" + self.name + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></ProtoDeclare>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.ProtoBody: # output this SFNode
                result += self.ProtoBody.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.ProtoInterface: # output this SFNode
                result += self.ProtoInterface.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</ProtoDeclare>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function ProtoDeclare.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'ProtoDeclare' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'ProtoDeclare' + ' {'
        if self.appinfo:
            result += " appinfo " +  '"' + self.appinfo + '"' + ""
        if self.documentation:
            result += " documentation " +  '"' + self.documentation + '"' + ""
        if self.name:
            result += " name " +  '"' + self.name + '"' + ""
        if self.ProtoBody: # output this SFNode
            result += '\n' + '  ' + indent + 'ProtoBody ' + self.ProtoBody.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.ProtoInterface: # output this SFNode
            result += '\n' + '  ' + indent + 'ProtoInterface ' + self.ProtoInterface.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class ProtoInterface(_X3DStatement):
    """
    ProtoInterface defines fields for new Prototype nodes.
    """
    def NAME(self):
        return 'ProtoInterface'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ProtoInterface'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('field', list(), FieldType.MFNode, AccessType.inputOutput, 'ProtoInterface')]
    def __init__(self,
                 field=None):
        self.field = field
    @property # getter - - - - - - - - - -
    def field(self):
        """Include a field statement for each field declaration in this ProtoDeclare's ProtoInterface."""
        return self.__field
    @field.setter
    def field(self, field):
        if  field is None:
            field = MFNode.DEFAULT_VALUE(self)
        # TODO type-aware checks for field
        if field: # walk each child in MFNode list, if any
            for each in __field:
                assertValidFieldInitializationValue(each.name, each.type, each.value, parent='ProtoInterface')
        self.__field = field
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.field)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ProtoInterface.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ProtoInterface'
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></ProtoInterface>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            ### if self.field: # walk each child in MFNode list, if any
            ### print('* ProtoInterface found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(field)=' + str(len(self.field)) + ', now invoking XML(' + str(indentLevel+1) + ')', flush=True)
            for each in self.field:
               result += each.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</ProtoInterface>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function ProtoInterface.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'ProtoInterface' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'ProtoInterface' + ' {'
        if self.field: # walk each child in MFNode list, if any
            for each in self.field:
                result += each.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class ROUTE(_X3DStatement):
    """
    ROUTE connects output fields of event-producing nodes to input fields of event-consuming nodes.
    """
    def NAME(self):
        return 'ROUTE'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ROUTE'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('fromField', '', FieldType.SFString, AccessType.inputOutput, 'ROUTE'),
        ('fromNode', '', FieldType.SFString, AccessType.inputOutput, 'ROUTE'),
        ('toField', '', FieldType.SFString, AccessType.inputOutput, 'ROUTE'),
        ('toNode', '', FieldType.SFString, AccessType.inputOutput, 'ROUTE')]
    def __init__(self,
                 fromField='',
                 fromNode='',
                 toField='',
                 toNode=''):
        self.fromField = fromField
        self.fromNode = fromNode
        self.toField = toField
        self.toNode = toNode
    @property # getter - - - - - - - - - -
    def fromField(self):
        """fromField is the field name in the source node which is originating an event."""
        return self.__fromField
    @fromField.setter
    def fromField(self, fromField):
        if  fromField is None:
            fromField = SFString.DEFAULT_VALUE(self)
        assertValidSFString(fromField)
        self.__fromField = fromField
    @property # getter - - - - - - - - - -
    def fromNode(self):
        """fromNode is the DEF name of the node originating an event."""
        return self.__fromNode
    @fromNode.setter
    def fromNode(self, fromNode):
        if  fromNode is None:
            fromNode = SFString.DEFAULT_VALUE(self)
        assertValidSFString(fromNode)
        self.__fromNode = fromNode
    @property # getter - - - - - - - - - -
    def toField(self):
        """toField is the field name in the destination node which is receiving an event."""
        return self.__toField
    @toField.setter
    def toField(self, toField):
        if  toField is None:
            toField = SFString.DEFAULT_VALUE(self)
        assertValidSFString(toField)
        self.__toField = toField
    @property # getter - - - - - - - - - -
    def toNode(self):
        """toNode is the DEF name of the destination node receiving an event."""
        return self.__toNode
    @toNode.setter
    def toNode(self, toNode):
        if  toNode is None:
            toNode = SFString.DEFAULT_VALUE(self)
        assertValidSFString(toNode)
        self.__toNode = toNode
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return False
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ROUTE.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ROUTE'
        if self.fromField:
            result += " fromField='" + self.fromField + "'"
        if self.fromNode:
            result += " fromNode='" + self.fromNode + "'"
        if self.toField:
            result += " toField='" + self.toField + "'"
        if self.toNode:
            result += " toNode='" + self.toNode + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></ROUTE>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            result += indent + '</ROUTE>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function ROUTE.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'ROUTE' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'ROUTE' + ' {'
        if self.fromField:
            result += " fromField " +  '"' + self.fromField + '"' + ""
        if self.fromNode:
            result += " fromNode " +  '"' + self.fromNode + '"' + ""
        if self.toField:
            result += " toField " +  '"' + self.toField + '"' + ""
        if self.toNode:
            result += " toNode " +  '"' + self.toNode + '"' + ""
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class Scene(_X3DStatement):
    """
    Scene is the implicit root node of the X3D scene graph.
    """
    def NAME(self):
        return 'Scene'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#Scene'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('children', list(), FieldType.MFNode, AccessType.inputOutput, 'Scene')]
    def __init__(self,
                 children=None):
        self.children = children
    @property # getter - - - - - - - - - -
    def children(self):
        return self.__children
    @children.setter
    def children(self, children):
        if  children is None:
            children = MFNode.DEFAULT_VALUE(self)
        assertValidMFNode(children)
        self.__children = children
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.children)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Scene.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<Scene'
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></Scene>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            ### if self.children: # walk each child in MFNode list, if any
            ### print('* Scene found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(children)=' + str(len(self.children)) + ', now invoking XML(' + str(indentLevel+1) + ')', flush=True)
            for each in self.children:
               result += each.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</Scene>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        if self.children: # walk each child in MFNode list, if any
            for each in self.children:
                result += each.VRML(indentLevel=0,VRML97=VRML97)
#       print('VRML serialization complete.', flush=True)
        return result

class unit(_X3DStatement):
    """
    Functional summary: unit statement defines data-conversion factors for typed values defined in a scene.
    """
    def NAME(self):
        return 'unit'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#unit'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('category', '', FieldType.SFString, AccessType.initializeOnly, 'unit'),
        ('conversionFactor', 1.0, FieldType.SFDouble, AccessType.inputOutput, 'unit'),
        ('name', '', FieldType.SFString, AccessType.inputOutput, 'unit')]
    def __init__(self,
                 category='',
                 conversionFactor=1.0,
                 name=''):
        self.category = category
        self.conversionFactor = conversionFactor
        self.name = name
    @property # getter - - - - - - - - - -
    def category(self):
        """Base-unit category as defined in X3D Specification."""
        return self.__category
    @category.setter
    def category(self, category):
        if  category is None:
            category = SFString.DEFAULT_VALUE(self)
        assertValidSFString(category)
        assertValidUnitCategory('category', category)
        self.__category = category
    @property # getter - - - - - - - - - -
    def conversionFactor(self):
        """[0,+infinity) Positive double-precision factor that converts new base unit to default base unit."""
        return self.__conversionFactor
    @conversionFactor.setter
    def conversionFactor(self, conversionFactor):
        if  conversionFactor is None:
            conversionFactor = 1.0 # default
        assertValidSFDouble(conversionFactor)
        assertPositive('conversionFactor', conversionFactor)
        self.__conversionFactor = conversionFactor
    @property # getter - - - - - - - - - -
    def name(self):
        """Author-defined name for this unit conversionFactor value (for example, FeetToMeters)."""
        return self.__name
    @name.setter
    def name(self, name):
        if  name is None:
            name = SFString.DEFAULT_VALUE(self)
        assertValidSFString(name)
        self.__name = name
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return False
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function unit.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<unit'
        if self.category:
            result += " category='" + self.category + "'"
        if self.conversionFactor != 1.0:
            result += " conversionFactor='" + SFDouble(self.conversionFactor).XML() + "'"
        if self.name:
            result += " name='" + self.name + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></unit>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            result += indent + '</unit>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        result += 'UNIT ' + self.category + ' ' + self.name + ' ' + str(self.conversionFactor) + '\n'
#       print('VRML serialization complete.', flush=True)
        return result

class X3D(_X3DNode):
    """
    X3D is the root node for an Extensible 3D (X3D) Graphics model.
    """
    def NAME(self):
        return 'X3D'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#X3D'
    XML_HEADER = '<?xml version="1.0" encoding="UTF-8"?>'
    XML_DOCTYPE_X3D_3_0 = '<!DOCTYPE X3D PUBLIC "ISO//Web3D//DTD X3D 3.0//EN" "http://www.web3d.org/specifications/x3d-3.0.dtd">'
    XML_DOCTYPE_X3D_3_1 = '<!DOCTYPE X3D PUBLIC "ISO//Web3D//DTD X3D 3.1//EN" "http://www.web3d.org/specifications/x3d-3.1.dtd">'
    XML_DOCTYPE_X3D_3_2 = '<!DOCTYPE X3D PUBLIC "ISO//Web3D//DTD X3D 3.2//EN" "http://www.web3d.org/specifications/x3d-3.2.dtd">'
    XML_DOCTYPE_X3D_3_3 = '<!DOCTYPE X3D PUBLIC "ISO//Web3D//DTD X3D 3.3//EN" "http://www.web3d.org/specifications/x3d-3.3.dtd">'
    XML_DOCTYPE_X3D_4_0 = '<!DOCTYPE X3D PUBLIC "ISO//Web3D//DTD X3D 4.0//EN" "http://www.web3d.org/specifications/x3d-4.0.dtd">'
    X3D_XML_SCHEMA_ATTRIBUTES_3_0 = "xmlns:xsd='http://www.w3.org/2001/XMLSchema-instance' xsd:noNamespaceSchemaLocation='http://www.web3d.org/specifications/x3d-3.0.xsd'"
    X3D_XML_SCHEMA_ATTRIBUTES_3_1 = "xmlns:xsd='http://www.w3.org/2001/XMLSchema-instance' xsd:noNamespaceSchemaLocation='http://www.web3d.org/specifications/x3d-3.1.xsd'"
    X3D_XML_SCHEMA_ATTRIBUTES_3_2 = "xmlns:xsd='http://www.w3.org/2001/XMLSchema-instance' xsd:noNamespaceSchemaLocation='http://www.web3d.org/specifications/x3d-3.2.xsd'"
    X3D_XML_SCHEMA_ATTRIBUTES_3_3 = "xmlns:xsd='http://www.w3.org/2001/XMLSchema-instance' xsd:noNamespaceSchemaLocation='http://www.web3d.org/specifications/x3d-3.3.xsd'"
    X3D_XML_SCHEMA_ATTRIBUTES_4_0 = "xmlns:xsd='http://www.w3.org/2001/XMLSchema-instance' xsd:noNamespaceSchemaLocation='http://www.web3d.org/specifications/x3d-4.0.xsd'"
    VRML97_HEADER = '#VRML V2.0 utf8'
    CLASSIC_VRML_HEADER_PREFIX = '#VRML V' # followed by X3D version number
    CLASSIC_VRML_HEADER_SUFFIX = ' utf8'
    X3DOM_HEADER = """<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Transitional//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd">

<!-- =================================================================== -->
<!-- embedded X3D scene appears after html/head/script and style entries -->
<!-- =================================================================== -->
<html>
   <head>
      <title></title>
      <meta http-equiv="X-UA-Compatible" content="chrome=1,IE=edge"/>
      <meta http-equiv="Content-Type" content="text/html;charset=utf-8"/>
<script type="text/javascript" src="https://ajax.googleapis.com/ajax/libs/jquery/3.2.1/jquery.min.js"></script>
<!-- Numbered X3DOM release versions: https://doc.x3dom.org/download -->
<!-- Developer X3DOM release version: https://doc.x3dom.org/download/dev -->
<link rel="stylesheet"
            type="text/css"
            href="https://x3dom.org/release/x3dom.css"/>
      <script type="text/javascript" src="https://x3dom.org/release/x3dom-full.js"/>
      <meta name="warning"
            content="Webfonts must be loaded prior to using Text node in X3D scene... see https://x3dom.org/x3dom/example/x3dom_text.html"/>
      <!-- X3DOM needs Web Fonts when an X3D Text node is included -->
      <!-- adapted from https://x3dom.org/x3dom/example/x3dom_text.html and http://web.mit.edu/jmorzins/www/fonts.html -->
      <style type="text/css">
/* ============================================================================= */
@font-face {
  font-family: 'SERIF'; /* default original */
  font-style: normal;
  font-weight: 700;
  src: local('Roman'), url('Roman.ttf') format('truetype');
}
@font-face {
  font-family: 'SERIF'; /* default alternate */
  font-style: normal;
  font-weight: 700;
  src: local('Times New Roman'), local('TimesNewRoman'), url('Times New Roman.ttf') format('truetype');
}
/* ============================================================================= */
@font-face {
  font-family: 'SANS'; /* default original */
  font-style: normal;
  font-weight: 400;
  src: local('Arial'), url('Arial.ttf') format('truetype');
}
@font-face {
  font-family: 'SANS'; /* default alternate */
  font-style: normal;
  font-weight: 400;
  src: local('Helvetica'), url('Helvetica.ttf') format('truetype');
}
/* ============================================================================= */
@font-face {
  font-family: 'TYPEWRITER'; /* default original */
  font-style: normal;
  font-weight: 900;
  src: local('Courier'), url('Courier.ttf') format('truetype');
}
@font-face {
  font-family: 'TYPEWRITER'; /* default alternate */
  font-style: normal;
  font-weight: 900;
  src: local('Courier New'), url('Courier New.ttf') format('truetype');
}
/* ============================================================================= */
</style>
   </head>
   <body>"""
    X3DOM_FOOTER = """      
   </body>
</html>
"""
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('profile', 'Immersive', FieldType.SFString, AccessType.inputOutput, 'X3D'),
        ('version', '4.0', FieldType.SFString, AccessType.inputOutput, 'X3D'),
        ('head', None, FieldType.SFNode, AccessType.inputOutput, 'X3D'),
        ('Scene', None, FieldType.SFNode, AccessType.inputOutput, 'X3D')]
    def __init__(self,
                 profile='Immersive',
                 version='4.0',
                 head=None,
                 Scene=None):
        self.profile = profile
        self.version = version
        self.head = head
        self.Scene = Scene
    @property # getter - - - - - - - - - -
    def profile(self):
        """profile attribute is required and defines the player or tool support needed for this model."""
        return self.__profile
    @profile.setter
    def profile(self, profile):
        if  profile is None:
            profile = 'Immersive' # default
        assertValidSFString(profile)
        assertValidProfileName('profile', profile)
        self.__profile = profile
    @property # getter - - - - - - - - - -
    def version(self):
        """Default is highest value matching schema and DOCTYPE in the scene."""
        return self.__version
    @version.setter
    def version(self, version):
        if  version is None:
            version = '4.0' # default
        assertValidSFString(version)
        assertValidX3dVersion('version', version)
        self.__version = version
    @property # getter - - - - - - - - - -
    def head(self):
        """Include a head element to contain component, unit or meta statements for this X3D model."""
        return self.__head
    @head.setter
    def head(self, head):
        if  head is None:
            head = None # default
        assertValidSFNode(head)
        if not isinstance(head, object):
            # print(flush=True)
            raise X3DTypeError(str(head) + ' does not have a valid node type object')
        self.__head = head
    @property # getter - - - - - - - - - -
    def Scene(self):
        """Include a Scene element to contain scene-graph nodes for this X3D model."""
        return self.__Scene
    @Scene.setter
    def Scene(self, Scene):
        if  Scene is None:
            Scene = None # default
        assertValidSFNode(Scene)
        if not isinstance(Scene, object):
            # print(flush=True)
            raise X3DTypeError(str(Scene) + ' does not have a valid node type object')
        self.__Scene = Scene
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.head) or (self.Scene)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function X3D.XML(indentLevel=' + str(indentLevel) + '), indent="' + indent + '"' + '\n'
        result += indent + self.XML_HEADER + '\n'
        if self.version == '3.0':
            result += indent + self.XML_DOCTYPE_X3D_3_0 + '\n'
        elif self.version == '3.1':
            result += indent + self.XML_DOCTYPE_X3D_3_1 + '\n'
        elif self.version == '3.2':
            result += indent + self.XML_DOCTYPE_X3D_3_2 + '\n'
        elif self.version == '3.3':
            result += indent + self.XML_DOCTYPE_X3D_3_3 + '\n'
        elif self.version == '4.0':
            result += indent + self.XML_DOCTYPE_X3D_4_0 + '\n'
        elif self.version == '4.1':
            result += indent + self.XML_DOCTYPE_X3D_4_1 + '\n'
        result += indent + "<X3D profile='" + self.profile +"' version='" + self.version +"' "
        if self.version == '3.0':
            result += self.X3D_XML_SCHEMA_ATTRIBUTES_3_0
        elif self.version == '3.1':
            result += self.X3D_XML_SCHEMA_ATTRIBUTES_3_1
        elif self.version == '3.2':
            result += self.X3D_XML_SCHEMA_ATTRIBUTES_3_2
        elif self.version == '3.3':
            result += self.X3D_XML_SCHEMA_ATTRIBUTES_3_3
        elif self.version == '4.0':
            result += self.X3D_XML_SCHEMA_ATTRIBUTES_4_0
        elif self.version == '4.1':
            result += self.X3D_XML_SCHEMA_ATTRIBUTES_4_1
        result += '>' + '\n' # finish open tag
        if self.head and self.head.hasChild():
            result += str(self.head.XML(indentLevel=indentLevel+1,syntax=syntax))
        if self.Scene and self.Scene.hasChild():
            result += str(self.Scene.XML(indentLevel=indentLevel+1,syntax=syntax))
        result += '</X3D>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def VRML97(self, indentLevel=0):
        """ Provide VRML97 output serialization suitable for .wrl file. """
        return VRML(self, indentLevel=0, VRML97=True)
    # output function - - - - - - - - - -
    def ClassicVRML(self, indentLevel=0):
        """ Provide ClassicVRML output serialization suitable for .x3dv file. """
        return VRML(self, indentLevel=0, VRML97=False)
    # output function - - - - - - - - - -
#    def X_ITE(self): #TODO
#        """ Provide X_ITE output serialization suitable for .html file. """
#        return X3D.X_ITE_HEADER + result + self.XML(indentLevel=0, syntax="XML") + X3D.X_ITE_FOOTER:
    # output function - - - - - - - - - -
    def X3DOM(self, indentLevel=0):
        """ Provide X3DOM output serialization suitable for .html file. """
        return X3D.X3DOM_HEADER + self.XML(indentLevel=0, syntax="HTML5") + X3D.X3DOM_FOOTER
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        if VRML97:
            result += self.VRML97_HEADER + '\n'
            result += '# X3D-to-VRML97 serialization autogenerated by X3DPSAIL x3d.py' + '\n'
        else:
            result += self.CLASSIC_VRML_HEADER_PREFIX + str(self.version) + self.CLASSIC_VRML_HEADER_SUFFIX + '\n'
        result += '# X3D-to-ClassicVRML serialization autogenerated by X3DPSAIL x3d.py' + '\n'
        result += '\n'
        if not VRML97:
            result += 'PROFILE '
            if not self.profile or VRML97:
                result += 'IMMERSIVE' + '\n'
            else:
                result += self.profile + '\n'
                    
        if self.head and self.head.hasChild():
            result += str(self.head.VRML(indentLevel=indentLevel+1,VRML97=VRML97))
        if self.Scene and self.Scene.hasChild():
            result += str(self.Scene.VRML(indentLevel=indentLevel+1,VRML97=VRML97))
        result += '\n'
#       print('VRML serialization complete.', flush=True)
        return result

###############################################

# Concrete Nodes

def isX3DNode(value):
    return isinstance(value, _X3DNode)

class Anchor(_X3DGroupingNode, _X3DUrlObject):
    """
    Anchor is a Grouping node that can contain most nodes.
    """
    def NAME(self):
        return 'Anchor'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#Anchor'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('bboxCenter', (0, 0, 0), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DGroupingNode'),
        ('bboxSize', (-1, -1, -1), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DGroupingNode'),
        ('description', '', FieldType.SFString, AccessType.inputOutput, 'Anchor'),
        ('displayBBox', False, FieldType.SFBool, AccessType.inputOutput, 'X3DGroupingNode'),
        ('parameter', list(), FieldType.MFString, AccessType.inputOutput, 'Anchor'),
        ('url', list(), FieldType.MFString, AccessType.inputOutput, 'X3DUrlObject'),
        ('visible', True, FieldType.SFBool, AccessType.inputOutput, 'X3DGroupingNode'),
        ('children', list(), FieldType.MFNode, AccessType.inputOutput, 'X3DGroupingNode'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 bboxCenter=(0, 0, 0),
                 bboxSize=(-1, -1, -1),
                 description='',
                 displayBBox=False,
                 parameter=list(),
                 url=list(),
                 visible=True,
                 children=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode Anchor __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.bboxCenter = bboxCenter
        self.bboxSize = bboxSize
        self.description = description
        self.displayBBox = displayBBox
        self.parameter = parameter
        self.url = url
        self.visible = visible
        self.children = children
    @property # getter - - - - - - - - - -
    def bboxCenter(self):
        """Bounding box center accompanies bboxSize and provides an optional hint for bounding box position offset from origin of local coordinate system."""
        return self.__bboxCenter
    @bboxCenter.setter
    def bboxCenter(self, bboxCenter):
        if  bboxCenter is None:
            bboxCenter = (0, 0, 0) # default
        assertValidSFVec3f(bboxCenter)
        self.__bboxCenter = bboxCenter
    @property # getter - - - - - - - - - -
    def bboxSize(self):
        """Bounding box size is usually omitted, and can easily be calculated automatically by an X3D player at scene-loading time with minimal computational cost."""
        return self.__bboxSize
    @bboxSize.setter
    def bboxSize(self, bboxSize):
        if  bboxSize is None:
            bboxSize = (-1, -1, -1) # default
        assertValidSFVec3f(bboxSize)
        assertBoundingBox('bboxSize', bboxSize)
        self.__bboxSize = bboxSize
    @property # getter - - - - - - - - - -
    def description(self):
        """Author-provided text tooltip that tells users the expected action of this node."""
        return self.__description
    @description.setter
    def description(self, description):
        if  description is None:
            description = SFString.DEFAULT_VALUE(self)
        assertValidSFString(description)
        self.__description = description
    @property # getter - - - - - - - - - -
    def displayBBox(self):
        """Whether to display bounding box for associated geometry, aligned with world coordinates."""
        return self.__displayBBox
    @displayBBox.setter
    def displayBBox(self, displayBBox):
        if  displayBBox is None:
            displayBBox = False # default
        assertValidSFBool(displayBBox)
        self.__displayBBox = displayBBox
    @property # getter - - - - - - - - - -
    def parameter(self):
        """If provided, parameter tells the X3D player where to to redirect the loaded url."""
        return self.__parameter
    @parameter.setter
    def parameter(self, parameter):
        if  parameter is None:
            parameter = MFString.DEFAULT_VALUE(self)
        assertValidMFString(parameter)
        self.__parameter = parameter
    @property # getter - - - - - - - - - -
    def url(self):
        """Address of replacement world, or #ViewpointDEFName within the current scene, or alternate Web resource, activated by the user selecting Shape geometry within the Anchor children nodes."""
        return self.__url
    @url.setter
    def url(self, url):
        if  url is None:
            url = MFString.DEFAULT_VALUE(self)
        assertValidMFString(url)
        self.__url = url
    @property # getter - - - - - - - - - -
    def visible(self):
        """Whether or not renderable content within this node is visually displayed."""
        return self.__visible
    @visible.setter
    def visible(self, visible):
        if  visible is None:
            visible = True # default
        assertValidSFBool(visible)
        self.__visible = visible
    @property # getter - - - - - - - - - -
    def children(self):
        """[X3DChildNode] Grouping nodes contain an ordered list of children nodes."""
        return self.__children
    @children.setter
    def children(self, children):
        if  children is None:
            children = MFNode.DEFAULT_VALUE(self)
        assertValidMFNode(children)
        self.__children = children
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.children) or (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Anchor.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<Anchor'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.bboxCenter != (0, 0, 0):
            result += " bboxCenter='" + SFVec3f(self.bboxCenter).XML() + "'"
        if self.bboxSize != (-1, -1, -1):
            result += " bboxSize='" + SFVec3f(self.bboxSize).XML() + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.description:
            result += " description='" + self.description + "'"
        if self.displayBBox != False:
            result += " displayBBox='" + SFBool(self.displayBBox).XML() + "'"
        if self.parameter != list():
            result += " parameter='" + MFString(self.parameter).XML() + "'"
        if self.url != list():
            result += " url='" + MFString(self.url).XML() + "'"
        if self.visible != True:
            result += " visible='" + SFBool(self.visible).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></Anchor>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            ### if self.children: # walk each child in MFNode list, if any
            ### print('* Anchor found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(children)=' + str(len(self.children)) + ', now invoking XML(' + str(indentLevel+1) + ')', flush=True)
            for each in self.children:
               result += each.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</Anchor>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function Anchor.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'Anchor' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'Anchor' + ' {'
        if self.bboxCenter != (0, 0, 0):
            result += " bboxCenter " + SFVec3f(self.bboxCenter).VRML() + ""
        if self.bboxSize != (-1, -1, -1):
            result += " bboxSize " + SFVec3f(self.bboxSize).VRML() + ""
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.description:
            result += " description " +  '"' + self.description + '"' + ""
        if self.displayBBox != False:
            result += " displayBBox " + SFBool(self.displayBBox).VRML() + ""
        if self.parameter != list():
            result += " parameter " + MFString(self.parameter).VRML() + ""
        if self.url != list():
            result += " url " + MFString(self.url).VRML() + ""
        if self.visible != True:
            result += " visible " + SFBool(self.visible).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.children: # walk each child in MFNode list, if any
            for each in self.children:
                result += each.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class Appearance(_X3DAppearanceNode):
    """
    Appearance specifies the visual properties of geometry by containing the Material, ImageTexture/MovieTexture/PixelTexture, FillProperties, LineProperties, programmable shader nodes (ComposedShader, PackagedShader, ProgramShader) and TextureTransform nodes.
    """
    def NAME(self):
        return 'Appearance'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#Appearance'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('fillProperties', None, FieldType.SFNode, AccessType.inputOutput, 'Appearance'),
        ('lineProperties', None, FieldType.SFNode, AccessType.inputOutput, 'Appearance'),
        ('material', None, FieldType.SFNode, AccessType.inputOutput, 'Appearance'),
        ('pointProperties', None, FieldType.SFNode, AccessType.inputOutput, 'Appearance'),
        ('texture', None, FieldType.SFNode, AccessType.inputOutput, 'Appearance'),
        ('textureTransform', None, FieldType.SFNode, AccessType.inputOutput, 'Appearance'),
        ('shaders', list(), FieldType.MFNode, AccessType.inputOutput, 'Appearance'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 fillProperties=None,
                 lineProperties=None,
                 material=None,
                 pointProperties=None,
                 texture=None,
                 textureTransform=None,
                 shaders=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode Appearance __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.fillProperties = fillProperties
        self.lineProperties = lineProperties
        self.material = material
        self.pointProperties = pointProperties
        self.texture = texture
        self.textureTransform = textureTransform
        self.shaders = shaders
    @property # getter - - - - - - - - - -
    def fillProperties(self):
        """[FillProperties] Single contained FillProperties node that specifies additional visual attributes applied to polygonal areas of corresponding geometry, on top of whatever other appearance is already defined."""
        return self.__fillProperties
    @fillProperties.setter
    def fillProperties(self, fillProperties):
        if  fillProperties is None:
            fillProperties = None # default
        assertValidSFNode(fillProperties)
        if not isinstance(fillProperties, object):
            # print(flush=True)
            raise X3DTypeError(str(fillProperties) + ' does not have a valid node type object')
        self.__fillProperties = fillProperties
    @property # getter - - - - - - - - - -
    def lineProperties(self):
        """[LineProperties] Single contained LineProperties node that specifies additional visual attributes applied to corresponding line geometry."""
        return self.__lineProperties
    @lineProperties.setter
    def lineProperties(self, lineProperties):
        if  lineProperties is None:
            lineProperties = None # default
        assertValidSFNode(lineProperties)
        if not isinstance(lineProperties, object):
            # print(flush=True)
            raise X3DTypeError(str(lineProperties) + ' does not have a valid node type object')
        self.__lineProperties = lineProperties
    @property # getter - - - - - - - - - -
    def material(self):
        """[X3DMaterialNode] Single contained Material node that specifies visual attributes for lighting response (color types, transparency, etc."""
        return self.__material
    @material.setter
    def material(self, material):
        if  material is None:
            material = None # default
        assertValidSFNode(material)
        if not isinstance(material, object):
            # print(flush=True)
            raise X3DTypeError(str(material) + ' does not have a valid node type object')
        self.__material = material
    @property # getter - - - - - - - - - -
    def pointProperties(self):
        return self.__pointProperties
    @pointProperties.setter
    def pointProperties(self, pointProperties):
        if  pointProperties is None:
            pointProperties = None # default
        assertValidSFNode(pointProperties)
        if not isinstance(pointProperties, object):
            # print(flush=True)
            raise X3DTypeError(str(pointProperties) + ' does not have a valid node type object')
        self.__pointProperties = pointProperties
    @property # getter - - - - - - - - - -
    def texture(self):
        """[X3DTextureNode] Single contained texture node (ImageTexture, MovieTexture, PixelTexture, MultiTexture) that maps image(s) to surface geometry."""
        return self.__texture
    @texture.setter
    def texture(self, texture):
        if  texture is None:
            texture = None # default
        assertValidSFNode(texture)
        if not isinstance(texture, object):
            # print(flush=True)
            raise X3DTypeError(str(texture) + ' does not have a valid node type object')
        self.__texture = texture
    @property # getter - - - - - - - - - -
    def textureTransform(self):
        """[X3DTextureTransformNode] Single contained TextureTransform node that defines 2D transformation applied to texture coordinates."""
        return self.__textureTransform
    @textureTransform.setter
    def textureTransform(self, textureTransform):
        if  textureTransform is None:
            textureTransform = None # default
        assertValidSFNode(textureTransform)
        if not isinstance(textureTransform, object):
            # print(flush=True)
            raise X3DTypeError(str(textureTransform) + ' does not have a valid node type object')
        self.__textureTransform = textureTransform
    @property # getter - - - - - - - - - -
    def shaders(self):
        """[X3DShaderNode] Zero or more contained programmable shader nodes (ComposedShader, PackagedShader, ProgramShader) that specify, in order of preference, author-programmed rendering characteristics."""
        return self.__shaders
    @shaders.setter
    def shaders(self, shaders):
        if  shaders is None:
            shaders = MFNode.DEFAULT_VALUE(self)
        assertValidMFNode(shaders)
        self.__shaders = shaders
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.shaders) or (self.fillProperties) or (self.IS) or (self.lineProperties) or (self.material) or (self.metadata) or (self.pointProperties) or (self.texture) or (self.textureTransform)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Appearance.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<Appearance'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></Appearance>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.fillProperties: # output this SFNode
                result += self.fillProperties.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.lineProperties: # output this SFNode
                result += self.lineProperties.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.material: # output this SFNode
                result += self.material.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.pointProperties: # output this SFNode
                result += self.pointProperties.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.texture: # output this SFNode
                result += self.texture.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.textureTransform: # output this SFNode
                result += self.textureTransform.XML(indentLevel=indentLevel+1,syntax=syntax)
            ### if self.shaders: # walk each child in MFNode list, if any
            ### print('* Appearance found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(shaders)=' + str(len(self.shaders)) + ', now invoking XML(' + str(indentLevel+1) + ')', flush=True)
            for each in self.shaders:
               result += each.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</Appearance>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function Appearance.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'Appearance' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'Appearance' + ' {'
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.fillProperties: # output this SFNode
            result += '\n' + '  ' + indent + 'fillProperties ' + self.fillProperties.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.lineProperties: # output this SFNode
            result += '\n' + '  ' + indent + 'lineProperties ' + self.lineProperties.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.material: # output this SFNode
            result += '\n' + '  ' + indent + 'material ' + self.material.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.pointProperties: # output this SFNode
            result += '\n' + '  ' + indent + 'pointProperties ' + self.pointProperties.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.texture: # output this SFNode
            result += '\n' + '  ' + indent + 'texture ' + self.texture.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.textureTransform: # output this SFNode
            result += '\n' + '  ' + indent + 'textureTransform ' + self.textureTransform.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.shaders: # walk each child in MFNode list, if any
            for each in self.shaders:
                result += each.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class Arc2D(_X3DGeometryNode):
    """
    Arc2D is a line-based geometry node that defines a linear circular arc with center (0,0) in X-Y plane, with angles measured starting at positive x-axis and sweeping towards positive y-axis.
    """
    def NAME(self):
        return 'Arc2D'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#Arc2D'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('endAngle', 1.570796, FieldType.SFFloat, AccessType.initializeOnly, 'Arc2D'),
        ('radius', 1, FieldType.SFFloat, AccessType.initializeOnly, 'Arc2D'),
        ('startAngle', 0, FieldType.SFFloat, AccessType.initializeOnly, 'Arc2D'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 endAngle=1.570796,
                 radius=1,
                 startAngle=0,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode Arc2D __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.endAngle = endAngle
        self.radius = radius
        self.startAngle = startAngle
    @property # getter - - - - - - - - - -
    def endAngle(self):
        """[0,2pi] Arc extends from startAngle counterclockwise to endAngle, in radians."""
        return self.__endAngle
    @endAngle.setter
    def endAngle(self, endAngle):
        if  endAngle is None:
            endAngle = 1.570796 # default
        assertValidSFFloat(endAngle)
        assertGreaterThan('endAngle', endAngle, -6.2832)
        assertLessThan('endAngle', endAngle, 6.2832)
        self.__endAngle = endAngle
    @property # getter - - - - - - - - - -
    def radius(self):
        """(0,+infinity) circle radius, of which the arc is a portion."""
        return self.__radius
    @radius.setter
    def radius(self, radius):
        if  radius is None:
            radius = 1 # default
        assertValidSFFloat(radius)
        assertPositive('radius', radius)
        self.__radius = radius
    @property # getter - - - - - - - - - -
    def startAngle(self):
        """[0,2pi] Arc extends from startAngle counterclockwise to endAngle, in radians."""
        return self.__startAngle
    @startAngle.setter
    def startAngle(self, startAngle):
        if  startAngle is None:
            startAngle = 0 # default
        assertValidSFFloat(startAngle)
        assertGreaterThan('startAngle', startAngle, -6.2832)
        assertLessThan('startAngle', startAngle, 6.2832)
        self.__startAngle = startAngle
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Arc2D.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<Arc2D'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.endAngle != 1.570796:
            result += " endAngle='" + SFFloat(self.endAngle).XML() + "'"
        if self.radius != 1:
            result += " radius='" + SFFloat(self.radius).XML() + "'"
        if self.startAngle != 0:
            result += " startAngle='" + SFFloat(self.startAngle).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></Arc2D>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</Arc2D>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function Arc2D.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'Arc2D' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'Arc2D' + ' {'
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.endAngle != 1.570796:
            result += " endAngle " + SFFloat(self.endAngle).VRML() + ""
        if self.radius != 1:
            result += " radius " + SFFloat(self.radius).VRML() + ""
        if self.startAngle != 0:
            result += " startAngle " + SFFloat(self.startAngle).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class ArcClose2D(_X3DGeometryNode):
    """
    ArcClose2D is a polygonal geometry node that defines a linear circular arc, closed by PIE or CHORD line segments, with center (0,0) in X-Y plane, with angles measured starting at positive x-axis and sweeping towards positive y-axis.
    """
    def NAME(self):
        return 'ArcClose2D'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ArcClose2D'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('closureType', 'PIE', FieldType.SFString, AccessType.initializeOnly, 'ArcClose2D'),
        ('endAngle', 1.570796, FieldType.SFFloat, AccessType.initializeOnly, 'ArcClose2D'),
        ('radius', 1, FieldType.SFFloat, AccessType.initializeOnly, 'ArcClose2D'),
        ('solid', False, FieldType.SFBool, AccessType.initializeOnly, 'ArcClose2D'),
        ('startAngle', 0, FieldType.SFFloat, AccessType.initializeOnly, 'ArcClose2D'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 closureType='PIE',
                 endAngle=1.570796,
                 radius=1,
                 solid=False,
                 startAngle=0,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode ArcClose2D __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.closureType = closureType
        self.endAngle = endAngle
        self.radius = radius
        self.solid = solid
        self.startAngle = startAngle
    @property # getter - - - - - - - - - -
    def closureType(self):
        """Defines whether pair of line segments connect to center (PIE), or single line-segment chord connects arc endpoints (CHORD)."""
        return self.__closureType
    @closureType.setter
    def closureType(self, closureType):
        if  closureType is None:
            closureType = 'PIE' # default
        assertValidSFString(closureType)
        assertValidClosureType('closureType', closureType)
        self.__closureType = closureType
    @property # getter - - - - - - - - - -
    def endAngle(self):
        """[0,2pi] Arc extends from startAngle counterclockwise to endAngle, in radians."""
        return self.__endAngle
    @endAngle.setter
    def endAngle(self, endAngle):
        if  endAngle is None:
            endAngle = 1.570796 # default
        assertValidSFFloat(endAngle)
        assertGreaterThan('endAngle', endAngle, -6.2832)
        assertLessThan('endAngle', endAngle, 6.2832)
        self.__endAngle = endAngle
    @property # getter - - - - - - - - - -
    def radius(self):
        """(0,+infinity) circle radius, of which the arc is a portion."""
        return self.__radius
    @radius.setter
    def radius(self, radius):
        if  radius is None:
            radius = 1 # default
        assertValidSFFloat(radius)
        assertPositive('radius', radius)
        self.__radius = radius
    @property # getter - - - - - - - - - -
    def solid(self):
        """Setting solid true means draw only one side of polygons (backface culling on), setting solid false means draw both sides of polygons (backface culling off)."""
        return self.__solid
    @solid.setter
    def solid(self, solid):
        if  solid is None:
            solid = False # default
        assertValidSFBool(solid)
        self.__solid = solid
    @property # getter - - - - - - - - - -
    def startAngle(self):
        """[0,2pi] Arc extends from startAngle counterclockwise to endAngle, in radians."""
        return self.__startAngle
    @startAngle.setter
    def startAngle(self, startAngle):
        if  startAngle is None:
            startAngle = 0 # default
        assertValidSFFloat(startAngle)
        assertGreaterThan('startAngle', startAngle, -6.2832)
        assertLessThan('startAngle', startAngle, 6.2832)
        self.__startAngle = startAngle
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ArcClose2D.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ArcClose2D'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.closureType != 'PIE':
            result += " closureType='" + self.closureType + "'"
        if self.endAngle != 1.570796:
            result += " endAngle='" + SFFloat(self.endAngle).XML() + "'"
        if self.radius != 1:
            result += " radius='" + SFFloat(self.radius).XML() + "'"
        if self.solid != False:
            result += " solid='" + SFBool(self.solid).XML() + "'"
        if self.startAngle != 0:
            result += " startAngle='" + SFFloat(self.startAngle).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></ArcClose2D>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</ArcClose2D>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function ArcClose2D.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'ArcClose2D' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'ArcClose2D' + ' {'
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.closureType != 'PIE':
            result += " closureType " +  '"' + self.closureType + '"' + ""
        if self.endAngle != 1.570796:
            result += " endAngle " + SFFloat(self.endAngle).VRML() + ""
        if self.radius != 1:
            result += " radius " + SFFloat(self.radius).VRML() + ""
        if self.solid != False:
            result += " solid " + SFBool(self.solid).VRML() + ""
        if self.startAngle != 0:
            result += " startAngle " + SFFloat(self.startAngle).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class AudioClip(_X3DSoundSourceNode, _X3DUrlObject):
    """
    AudioClip provides audio data used by parent Sound nodes.
    """
    def NAME(self):
        return 'AudioClip'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#AudioClip'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('description', '', FieldType.SFString, AccessType.inputOutput, 'X3DSoundSourceNode'),
        ('loop', False, FieldType.SFBool, AccessType.inputOutput, 'X3DTimeDependentNode'),
        ('pauseTime', 0, FieldType.SFTime, AccessType.inputOutput, 'X3DTimeDependentNode'),
        ('pitch', 1.0, FieldType.SFFloat, AccessType.inputOutput, 'X3DSoundSourceNode'),
        ('resumeTime', 0, FieldType.SFTime, AccessType.inputOutput, 'X3DTimeDependentNode'),
        ('startTime', 0, FieldType.SFTime, AccessType.inputOutput, 'X3DTimeDependentNode'),
        ('stopTime', 0, FieldType.SFTime, AccessType.inputOutput, 'X3DTimeDependentNode'),
        ('url', list(), FieldType.MFString, AccessType.inputOutput, 'X3DUrlObject'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 description='',
                 loop=False,
                 pauseTime=0,
                 pitch=1.0,
                 resumeTime=0,
                 startTime=0,
                 stopTime=0,
                 url=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode AudioClip __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.description = description
        self.loop = loop
        self.pauseTime = pauseTime
        self.pitch = pitch
        self.resumeTime = resumeTime
        self.startTime = startTime
        self.stopTime = stopTime
        self.url = url
    @property # getter - - - - - - - - - -
    def description(self):
        """Author-provided text tooltip that tells users the expected action of this node."""
        return self.__description
    @description.setter
    def description(self, description):
        if  description is None:
            description = SFString.DEFAULT_VALUE(self)
        assertValidSFString(description)
        self.__description = description
    @property # getter - - - - - - - - - -
    def loop(self):
        """Repeat indefinitely when loop=true, repeat only once when loop=false."""
        return self.__loop
    @loop.setter
    def loop(self, loop):
        if  loop is None:
            loop = False # default
        assertValidSFBool(loop)
        self.__loop = loop
    @property # getter - - - - - - - - - -
    def pauseTime(self):
        """When time now >= pauseTime, isPaused becomes true and AudioClip becomes paused."""
        return self.__pauseTime
    @pauseTime.setter
    def pauseTime(self, pauseTime):
        if  pauseTime is None:
            pauseTime = 0 # default
        assertValidSFTime(pauseTime)
        self.__pauseTime = pauseTime
    @property # getter - - - - - - - - - -
    def pitch(self):
        """(0,+infinity) Multiplier for the rate at which sampled sound is played."""
        return self.__pitch
    @pitch.setter
    def pitch(self, pitch):
        if  pitch is None:
            pitch = 1.0 # default
        assertValidSFFloat(pitch)
        assertPositive('pitch', pitch)
        self.__pitch = pitch
    @property # getter - - - - - - - - - -
    def resumeTime(self):
        """When resumeTime becomes <= time now, isPaused becomes false and AudioClip becomes active."""
        return self.__resumeTime
    @resumeTime.setter
    def resumeTime(self, resumeTime):
        if  resumeTime is None:
            resumeTime = 0 # default
        assertValidSFTime(resumeTime)
        self.__resumeTime = resumeTime
    @property # getter - - - - - - - - - -
    def startTime(self):
        """Absolute time: number of seconds since January 1, 1970, 00:00:00 GMT."""
        return self.__startTime
    @startTime.setter
    def startTime(self, startTime):
        if  startTime is None:
            startTime = 0 # default
        assertValidSFTime(startTime)
        self.__startTime = startTime
    @property # getter - - - - - - - - - -
    def stopTime(self):
        """Absolute time: number of seconds since January 1, 1970, 00:00:00 GMT."""
        return self.__stopTime
    @stopTime.setter
    def stopTime(self, stopTime):
        if  stopTime is None:
            stopTime = 0 # default
        assertValidSFTime(stopTime)
        self.__stopTime = stopTime
    @property # getter - - - - - - - - - -
    def url(self):
        """Location and filename of sound file or stream."""
        return self.__url
    @url.setter
    def url(self, url):
        if  url is None:
            url = MFString.DEFAULT_VALUE(self)
        assertValidMFString(url)
        self.__url = url
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function AudioClip.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<AudioClip'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.description:
            result += " description='" + self.description + "'"
        if self.loop != False:
            result += " loop='" + SFBool(self.loop).XML() + "'"
        if self.pauseTime != 0:
            result += " pauseTime='" + SFTime(self.pauseTime).XML() + "'"
        if self.pitch != 1.0:
            result += " pitch='" + SFFloat(self.pitch).XML() + "'"
        if self.resumeTime != 0:
            result += " resumeTime='" + SFTime(self.resumeTime).XML() + "'"
        if self.startTime != 0:
            result += " startTime='" + SFTime(self.startTime).XML() + "'"
        if self.stopTime != 0:
            result += " stopTime='" + SFTime(self.stopTime).XML() + "'"
        if self.url != list():
            result += " url='" + MFString(self.url).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></AudioClip>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</AudioClip>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function AudioClip.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'AudioClip' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'AudioClip' + ' {'
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.description:
            result += " description " +  '"' + self.description + '"' + ""
        if self.loop != False:
            result += " loop " + SFBool(self.loop).VRML() + ""
        if self.pauseTime != 0:
            result += " pauseTime " + SFTime(self.pauseTime).VRML() + ""
        if self.pitch != 1.0:
            result += " pitch " + SFFloat(self.pitch).VRML() + ""
        if self.resumeTime != 0:
            result += " resumeTime " + SFTime(self.resumeTime).VRML() + ""
        if self.startTime != 0:
            result += " startTime " + SFTime(self.startTime).VRML() + ""
        if self.stopTime != 0:
            result += " stopTime " + SFTime(self.stopTime).VRML() + ""
        if self.url != list():
            result += " url " + MFString(self.url).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class Background(_X3DBackgroundNode):
    """
    Background simulates ground and sky, using vertical arrays of wraparound color values.
    """
    def NAME(self):
        return 'Background'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#Background'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('backUrl', list(), FieldType.MFString, AccessType.inputOutput, 'Background'),
        ('bottomUrl', list(), FieldType.MFString, AccessType.inputOutput, 'Background'),
        ('frontUrl', list(), FieldType.MFString, AccessType.inputOutput, 'Background'),
        ('groundAngle', list(), FieldType.MFFloat, AccessType.inputOutput, 'X3DBackgroundNode'),
        ('groundColor', list(), FieldType.MFColor, AccessType.inputOutput, 'X3DBackgroundNode'),
        ('leftUrl', list(), FieldType.MFString, AccessType.inputOutput, 'Background'),
        ('rightUrl', list(), FieldType.MFString, AccessType.inputOutput, 'Background'),
        ('skyAngle', list(), FieldType.MFFloat, AccessType.inputOutput, 'X3DBackgroundNode'),
        ('skyColor', [(0, 0, 0)], FieldType.MFColor, AccessType.inputOutput, 'X3DBackgroundNode'),
        ('topUrl', list(), FieldType.MFString, AccessType.inputOutput, 'Background'),
        ('transparency', 0, FieldType.SFFloat, AccessType.inputOutput, 'X3DBackgroundNode'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 backUrl=list(),
                 bottomUrl=list(),
                 frontUrl=list(),
                 groundAngle=list(),
                 groundColor=list(),
                 leftUrl=list(),
                 rightUrl=list(),
                 skyAngle=list(),
                 skyColor=[(0, 0, 0)],
                 topUrl=list(),
                 transparency=0,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode Background __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.backUrl = backUrl
        self.bottomUrl = bottomUrl
        self.frontUrl = frontUrl
        self.groundAngle = groundAngle
        self.groundColor = groundColor
        self.leftUrl = leftUrl
        self.rightUrl = rightUrl
        self.skyAngle = skyAngle
        self.skyColor = skyColor
        self.topUrl = topUrl
        self.transparency = transparency
    @property # getter - - - - - - - - - -
    def backUrl(self):
        """Image background panorama between ground/sky backdrop and scene's geometry."""
        return self.__backUrl
    @backUrl.setter
    def backUrl(self, backUrl):
        if  backUrl is None:
            backUrl = MFString.DEFAULT_VALUE(self)
        assertValidMFString(backUrl)
        self.__backUrl = backUrl
    @property # getter - - - - - - - - - -
    def bottomUrl(self):
        """Image background panorama between ground/sky backdrop and scene's geometry."""
        return self.__bottomUrl
    @bottomUrl.setter
    def bottomUrl(self, bottomUrl):
        if  bottomUrl is None:
            bottomUrl = MFString.DEFAULT_VALUE(self)
        assertValidMFString(bottomUrl)
        self.__bottomUrl = bottomUrl
    @property # getter - - - - - - - - - -
    def frontUrl(self):
        """Image background panorama between ground/sky backdrop and scene's geometry."""
        return self.__frontUrl
    @frontUrl.setter
    def frontUrl(self, frontUrl):
        if  frontUrl is None:
            frontUrl = MFString.DEFAULT_VALUE(self)
        assertValidMFString(frontUrl)
        self.__frontUrl = frontUrl
    @property # getter - - - - - - - - - -
    def groundAngle(self):
        """[0,pi/2] The angle array values increase from 0."""
        return self.__groundAngle
    @groundAngle.setter
    def groundAngle(self, groundAngle):
        if  groundAngle is None:
            groundAngle = MFFloat.DEFAULT_VALUE(self)
        assertValidMFFloat(groundAngle)
        assertGreaterThanEquals('groundAngle', groundAngle, 0)
        assertLessThanEquals('groundAngle', groundAngle, 1.5708)
        self.__groundAngle = groundAngle
    @property # getter - - - - - - - - - -
    def groundColor(self):
        """Color of the ground at the various angles on the ground partial sphere."""
        return self.__groundColor
    @groundColor.setter
    def groundColor(self, groundColor):
        if  groundColor is None:
            groundColor = MFColor.DEFAULT_VALUE(self)
        assertValidMFColor(groundColor)
        assertZeroToOne('groundColor', groundColor)
        self.__groundColor = groundColor
    @property # getter - - - - - - - - - -
    def leftUrl(self):
        """Image background panorama between ground/sky backdrop and scene's geometry."""
        return self.__leftUrl
    @leftUrl.setter
    def leftUrl(self, leftUrl):
        if  leftUrl is None:
            leftUrl = MFString.DEFAULT_VALUE(self)
        assertValidMFString(leftUrl)
        self.__leftUrl = leftUrl
    @property # getter - - - - - - - - - -
    def rightUrl(self):
        """Image background panorama between ground/sky backdrop and scene's geometry."""
        return self.__rightUrl
    @rightUrl.setter
    def rightUrl(self, rightUrl):
        if  rightUrl is None:
            rightUrl = MFString.DEFAULT_VALUE(self)
        assertValidMFString(rightUrl)
        self.__rightUrl = rightUrl
    @property # getter - - - - - - - - - -
    def skyAngle(self):
        """[0,pi] The angle array values increase from 0."""
        return self.__skyAngle
    @skyAngle.setter
    def skyAngle(self, skyAngle):
        if  skyAngle is None:
            skyAngle = MFFloat.DEFAULT_VALUE(self)
        assertValidMFFloat(skyAngle)
        assertGreaterThanEquals('skyAngle', skyAngle, 0)
        assertLessThanEquals('skyAngle', skyAngle, 3.1416)
        self.__skyAngle = skyAngle
    @property # getter - - - - - - - - - -
    def skyColor(self):
        """Color of the sky at various angles on the sky sphere."""
        return self.__skyColor
    @skyColor.setter
    def skyColor(self, skyColor):
        if  skyColor is None:
            skyColor = [(0, 0, 0)] # default
        assertValidMFColor(skyColor)
        assertZeroToOne('skyColor', skyColor)
        self.__skyColor = skyColor
    @property # getter - - - - - - - - - -
    def topUrl(self):
        """Image background panorama between ground/sky backdrop and scene's geometry."""
        return self.__topUrl
    @topUrl.setter
    def topUrl(self, topUrl):
        if  topUrl is None:
            topUrl = MFString.DEFAULT_VALUE(self)
        assertValidMFString(topUrl)
        self.__topUrl = topUrl
    @property # getter - - - - - - - - - -
    def transparency(self):
        """[0,1] how "clear" the background is, allows underlying page to show through: 1."""
        return self.__transparency
    @transparency.setter
    def transparency(self, transparency):
        if  transparency is None:
            transparency = 0 # default
        assertValidSFFloat(transparency)
        assertZeroToOne('transparency', transparency)
        self.__transparency = transparency
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Background.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<Background'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.backUrl != list():
            result += " backUrl='" + MFString(self.backUrl).XML() + "'"
        if self.bottomUrl != list():
            result += " bottomUrl='" + MFString(self.bottomUrl).XML() + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.frontUrl != list():
            result += " frontUrl='" + MFString(self.frontUrl).XML() + "'"
        if self.groundAngle != list():
            result += " groundAngle='" + MFFloat(self.groundAngle).XML() + "'"
        if self.groundColor != list():
            result += " groundColor='" + MFColor(self.groundColor).XML() + "'"
        if self.leftUrl != list():
            result += " leftUrl='" + MFString(self.leftUrl).XML() + "'"
        if self.rightUrl != list():
            result += " rightUrl='" + MFString(self.rightUrl).XML() + "'"
        if self.skyAngle != list():
            result += " skyAngle='" + MFFloat(self.skyAngle).XML() + "'"
        if self.skyColor != [(0, 0, 0)]:
            result += " skyColor='" + MFColor(self.skyColor).XML() + "'"
        if self.topUrl != list():
            result += " topUrl='" + MFString(self.topUrl).XML() + "'"
        if self.transparency != 0:
            result += " transparency='" + SFFloat(self.transparency).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></Background>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</Background>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function Background.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'Background' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'Background' + ' {'
        if self.backUrl != list():
            result += " backUrl " + MFString(self.backUrl).VRML() + ""
        if self.bottomUrl != list():
            result += " bottomUrl " + MFString(self.bottomUrl).VRML() + ""
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.frontUrl != list():
            result += " frontUrl " + MFString(self.frontUrl).VRML() + ""
        if self.groundAngle != list():
            result += " groundAngle " + MFFloat(self.groundAngle).VRML() + ""
        if self.groundColor != list():
            result += " groundColor " + MFColor(self.groundColor).VRML() + ""
        if self.leftUrl != list():
            result += " leftUrl " + MFString(self.leftUrl).VRML() + ""
        if self.rightUrl != list():
            result += " rightUrl " + MFString(self.rightUrl).VRML() + ""
        if self.skyAngle != list():
            result += " skyAngle " + MFFloat(self.skyAngle).VRML() + ""
        if self.skyColor != [(0, 0, 0)]:
            result += " skyColor " + MFColor(self.skyColor).VRML() + ""
        if self.topUrl != list():
            result += " topUrl " + MFString(self.topUrl).VRML() + ""
        if self.transparency != 0:
            result += " transparency " + SFFloat(self.transparency).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class BallJoint(_X3DRigidJointNode):
    """
    BallJoint represents an unconstrained joint between two bodies that pivot about a common anchor point.
    """
    def NAME(self):
        return 'BallJoint'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#BallJoint'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('anchorPoint', (0, 0, 0), FieldType.SFVec3f, AccessType.inputOutput, 'BallJoint'),
        ('forceOutput', ["NONE"], FieldType.MFString, AccessType.inputOutput, 'X3DRigidJointNode'),
        ('body1', None, FieldType.SFNode, AccessType.inputOutput, 'X3DRigidJointNode'),
        ('body2', None, FieldType.SFNode, AccessType.inputOutput, 'X3DRigidJointNode'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 anchorPoint=(0, 0, 0),
                 forceOutput=["NONE"],
                 body1=None,
                 body2=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode BallJoint __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.anchorPoint = anchorPoint
        self.forceOutput = forceOutput
        self.body1 = body1
        self.body2 = body2
    @property # getter - - - - - - - - - -
    def anchorPoint(self):
        """anchorPoint is joint center, specified in world coordinates."""
        return self.__anchorPoint
    @anchorPoint.setter
    def anchorPoint(self, anchorPoint):
        if  anchorPoint is None:
            anchorPoint = (0, 0, 0) # default
        assertValidSFVec3f(anchorPoint)
        self.__anchorPoint = anchorPoint
    @property # getter - - - - - - - - - -
    def forceOutput(self):
        """forceOutput controls which output fields are generated for the next frame."""
        return self.__forceOutput
    @forceOutput.setter
    def forceOutput(self, forceOutput):
        if  forceOutput is None:
            forceOutput = ["NONE"] # default
        assertValidMFString(forceOutput)
        self.__forceOutput = forceOutput
    @property # getter - - - - - - - - - -
    def body1(self):
        return self.__body1
    @body1.setter
    def body1(self, body1):
        if  body1 is None:
            body1 = None # default
        assertValidSFNode(body1)
        if not isinstance(body1, object):
            # print(flush=True)
            raise X3DTypeError(str(body1) + ' does not have a valid node type object')
        self.__body1 = body1
    @property # getter - - - - - - - - - -
    def body2(self):
        return self.__body2
    @body2.setter
    def body2(self, body2):
        if  body2 is None:
            body2 = None # default
        assertValidSFNode(body2)
        if not isinstance(body2, object):
            # print(flush=True)
            raise X3DTypeError(str(body2) + ' does not have a valid node type object')
        self.__body2 = body2
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.body1) or (self.body2) or (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function BallJoint.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<BallJoint'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.anchorPoint != (0, 0, 0):
            result += " anchorPoint='" + SFVec3f(self.anchorPoint).XML() + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.forceOutput != ["NONE"]:
            result += " forceOutput='" + MFString(self.forceOutput).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></BallJoint>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.body1: # output this SFNode
                result += self.body1.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.body2: # output this SFNode
                result += self.body2.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</BallJoint>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function BallJoint.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'BallJoint' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'BallJoint' + ' {'
        if self.anchorPoint != (0, 0, 0):
            result += " anchorPoint " + SFVec3f(self.anchorPoint).VRML() + ""
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.forceOutput != ["NONE"]:
            result += " forceOutput " + MFString(self.forceOutput).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.body1: # output this SFNode
            result += '\n' + '  ' + indent + 'body1 ' + self.body1.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.body2: # output this SFNode
            result += '\n' + '  ' + indent + 'body2 ' + self.body2.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class Billboard(_X3DGroupingNode):
    """
    Billboard is a Grouping node that can contain most nodes.
    """
    def NAME(self):
        return 'Billboard'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#Billboard'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('axisOfRotation', (0, 1, 0), FieldType.SFVec3f, AccessType.inputOutput, 'Billboard'),
        ('bboxCenter', (0, 0, 0), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DGroupingNode'),
        ('bboxSize', (-1, -1, -1), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DGroupingNode'),
        ('displayBBox', False, FieldType.SFBool, AccessType.inputOutput, 'X3DGroupingNode'),
        ('visible', True, FieldType.SFBool, AccessType.inputOutput, 'X3DGroupingNode'),
        ('children', list(), FieldType.MFNode, AccessType.inputOutput, 'X3DGroupingNode'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 axisOfRotation=(0, 1, 0),
                 bboxCenter=(0, 0, 0),
                 bboxSize=(-1, -1, -1),
                 displayBBox=False,
                 visible=True,
                 children=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode Billboard __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.axisOfRotation = axisOfRotation
        self.bboxCenter = bboxCenter
        self.bboxSize = bboxSize
        self.displayBBox = displayBBox
        self.visible = visible
        self.children = children
    @property # getter - - - - - - - - - -
    def axisOfRotation(self):
        """axisOfRotation direction is relative to local coordinate system."""
        return self.__axisOfRotation
    @axisOfRotation.setter
    def axisOfRotation(self, axisOfRotation):
        if  axisOfRotation is None:
            axisOfRotation = (0, 1, 0) # default
        assertValidSFVec3f(axisOfRotation)
        self.__axisOfRotation = axisOfRotation
    @property # getter - - - - - - - - - -
    def bboxCenter(self):
        """Bounding box center accompanies bboxSize and provides an optional hint for bounding box position offset from origin of local coordinate system."""
        return self.__bboxCenter
    @bboxCenter.setter
    def bboxCenter(self, bboxCenter):
        if  bboxCenter is None:
            bboxCenter = (0, 0, 0) # default
        assertValidSFVec3f(bboxCenter)
        self.__bboxCenter = bboxCenter
    @property # getter - - - - - - - - - -
    def bboxSize(self):
        """Bounding box size is usually omitted, and can easily be calculated automatically by an X3D player at scene-loading time with minimal computational cost."""
        return self.__bboxSize
    @bboxSize.setter
    def bboxSize(self, bboxSize):
        if  bboxSize is None:
            bboxSize = (-1, -1, -1) # default
        assertValidSFVec3f(bboxSize)
        assertBoundingBox('bboxSize', bboxSize)
        self.__bboxSize = bboxSize
    @property # getter - - - - - - - - - -
    def displayBBox(self):
        """Whether to display bounding box for associated geometry, aligned with world coordinates."""
        return self.__displayBBox
    @displayBBox.setter
    def displayBBox(self, displayBBox):
        if  displayBBox is None:
            displayBBox = False # default
        assertValidSFBool(displayBBox)
        self.__displayBBox = displayBBox
    @property # getter - - - - - - - - - -
    def visible(self):
        """Whether or not renderable content within this node is visually displayed."""
        return self.__visible
    @visible.setter
    def visible(self, visible):
        if  visible is None:
            visible = True # default
        assertValidSFBool(visible)
        self.__visible = visible
    @property # getter - - - - - - - - - -
    def children(self):
        """[X3DChildNode] Grouping nodes contain an ordered list of children nodes."""
        return self.__children
    @children.setter
    def children(self, children):
        if  children is None:
            children = MFNode.DEFAULT_VALUE(self)
        assertValidMFNode(children)
        self.__children = children
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.children) or (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Billboard.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<Billboard'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.axisOfRotation != (0, 1, 0):
            result += " axisOfRotation='" + SFVec3f(self.axisOfRotation).XML() + "'"
        if self.bboxCenter != (0, 0, 0):
            result += " bboxCenter='" + SFVec3f(self.bboxCenter).XML() + "'"
        if self.bboxSize != (-1, -1, -1):
            result += " bboxSize='" + SFVec3f(self.bboxSize).XML() + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.displayBBox != False:
            result += " displayBBox='" + SFBool(self.displayBBox).XML() + "'"
        if self.visible != True:
            result += " visible='" + SFBool(self.visible).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></Billboard>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            ### if self.children: # walk each child in MFNode list, if any
            ### print('* Billboard found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(children)=' + str(len(self.children)) + ', now invoking XML(' + str(indentLevel+1) + ')', flush=True)
            for each in self.children:
               result += each.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</Billboard>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function Billboard.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'Billboard' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'Billboard' + ' {'
        if self.axisOfRotation != (0, 1, 0):
            result += " axisOfRotation " + SFVec3f(self.axisOfRotation).VRML() + ""
        if self.bboxCenter != (0, 0, 0):
            result += " bboxCenter " + SFVec3f(self.bboxCenter).VRML() + ""
        if self.bboxSize != (-1, -1, -1):
            result += " bboxSize " + SFVec3f(self.bboxSize).VRML() + ""
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.displayBBox != False:
            result += " displayBBox " + SFBool(self.displayBBox).VRML() + ""
        if self.visible != True:
            result += " visible " + SFBool(self.visible).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.children: # walk each child in MFNode list, if any
            for each in self.children:
                result += each.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class BlendedVolumeStyle(_X3DComposableVolumeRenderStyleNode):
    """
    BlendedVolumeStyle combines rendering of two voxel data sets into one by blending voxel values.
    """
    def NAME(self):
        return 'BlendedVolumeStyle'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#BlendedVolumeStyle'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('enabled', True, FieldType.SFBool, AccessType.inputOutput, 'X3DVolumeRenderStyleNode'),
        ('weightConstant1', 0.5, FieldType.SFFloat, AccessType.inputOutput, 'BlendedVolumeStyle'),
        ('weightConstant2', 0.5, FieldType.SFFloat, AccessType.inputOutput, 'BlendedVolumeStyle'),
        ('weightFunction1', 'CONSTANT', FieldType.SFString, AccessType.inputOutput, 'BlendedVolumeStyle'),
        ('weightFunction2', 'CONSTANT', FieldType.SFString, AccessType.inputOutput, 'BlendedVolumeStyle'),
        ('renderStyle', None, FieldType.SFNode, AccessType.inputOutput, 'BlendedVolumeStyle'),
        ('voxels', None, FieldType.SFNode, AccessType.inputOutput, 'BlendedVolumeStyle'),
        ('weightTransferFunction1', None, FieldType.SFNode, AccessType.inputOutput, 'BlendedVolumeStyle'),
        ('weightTransferFunction2', None, FieldType.SFNode, AccessType.inputOutput, 'BlendedVolumeStyle'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 enabled=True,
                 weightConstant1=0.5,
                 weightConstant2=0.5,
                 weightFunction1='CONSTANT',
                 weightFunction2='CONSTANT',
                 renderStyle=None,
                 voxels=None,
                 weightTransferFunction1=None,
                 weightTransferFunction2=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode BlendedVolumeStyle __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.enabled = enabled
        self.weightConstant1 = weightConstant1
        self.weightConstant2 = weightConstant2
        self.weightFunction1 = weightFunction1
        self.weightFunction2 = weightFunction2
        self.renderStyle = renderStyle
        self.voxels = voxels
        self.weightTransferFunction1 = weightTransferFunction1
        self.weightTransferFunction2 = weightTransferFunction2
    @property # getter - - - - - - - - - -
    def enabled(self):
        """Enables/disables node operation."""
        return self.__enabled
    @enabled.setter
    def enabled(self, enabled):
        if  enabled is None:
            enabled = True # default
        assertValidSFBool(enabled)
        self.__enabled = enabled
    @property # getter - - - - - - - - - -
    def weightConstant1(self):
        """[0,1] weightConstant1 is used when weightFunction1=CONSTANT."""
        return self.__weightConstant1
    @weightConstant1.setter
    def weightConstant1(self, weightConstant1):
        if  weightConstant1 is None:
            weightConstant1 = 0.5 # default
        assertValidSFFloat(weightConstant1)
        assertZeroToOne('weightConstant1', weightConstant1)
        self.__weightConstant1 = weightConstant1
    @property # getter - - - - - - - - - -
    def weightConstant2(self):
        """[0,1] weightConstant2 is used when weightFunction2=CONSTANT."""
        return self.__weightConstant2
    @weightConstant2.setter
    def weightConstant2(self, weightConstant2):
        if  weightConstant2 is None:
            weightConstant2 = 0.5 # default
        assertValidSFFloat(weightConstant2)
        assertZeroToOne('weightConstant2', weightConstant2)
        self.__weightConstant2 = weightConstant2
    @property # getter - - - - - - - - - -
    def weightFunction1(self):
        """specifies 2D textures used to determine weight values when weight function is set to TABLE."""
        return self.__weightFunction1
    @weightFunction1.setter
    def weightFunction1(self, weightFunction1):
        if  weightFunction1 is None:
            weightFunction1 = 'CONSTANT' # default
        assertValidSFString(weightFunction1)
        assertValidVolumeRenderingWeightFunction('weightFunction1', weightFunction1)
        self.__weightFunction1 = weightFunction1
    @property # getter - - - - - - - - - -
    def weightFunction2(self):
        """specifies 2D textures used to determine weight values when weight function is set to TABLE."""
        return self.__weightFunction2
    @weightFunction2.setter
    def weightFunction2(self, weightFunction2):
        if  weightFunction2 is None:
            weightFunction2 = 'CONSTANT' # default
        assertValidSFString(weightFunction2)
        assertValidVolumeRenderingWeightFunction('weightFunction2', weightFunction2)
        self.__weightFunction2 = weightFunction2
    @property # getter - - - - - - - - - -
    def renderStyle(self):
        """[X3DComposableVolumeRenderStyleNode] Single contained X3DComposableVolumeRenderStyleNode node that defines specific rendering technique for data in the voxels field, and the result is blended with parent VolumeData or SegmentedVoliumeData node."""
        return self.__renderStyle
    @renderStyle.setter
    def renderStyle(self, renderStyle):
        if  renderStyle is None:
            renderStyle = None # default
        assertValidSFNode(renderStyle)
        if not isinstance(renderStyle, object):
            # print(flush=True)
            raise X3DTypeError(str(renderStyle) + ' does not have a valid node type object')
        self.__renderStyle = renderStyle
    @property # getter - - - - - - - - - -
    def voxels(self):
        """[X3DTexture3DNode] Single contained X3DTexture3DNode (ComposedTexture3D, ImageTexture3D, PixelTexture3D) that provides second set of raw voxel information utilized by corresponding rendering styles."""
        return self.__voxels
    @voxels.setter
    def voxels(self, voxels):
        if  voxels is None:
            voxels = None # default
        assertValidSFNode(voxels)
        if not isinstance(voxels, object):
            # print(flush=True)
            raise X3DTypeError(str(voxels) + ' does not have a valid node type object')
        self.__voxels = voxels
    @property # getter - - - - - - - - - -
    def weightTransferFunction1(self):
        return self.__weightTransferFunction1
    @weightTransferFunction1.setter
    def weightTransferFunction1(self, weightTransferFunction1):
        if  weightTransferFunction1 is None:
            weightTransferFunction1 = None # default
        assertValidSFNode(weightTransferFunction1)
        if not isinstance(weightTransferFunction1, object):
            # print(flush=True)
            raise X3DTypeError(str(weightTransferFunction1) + ' does not have a valid node type object')
        self.__weightTransferFunction1 = weightTransferFunction1
    @property # getter - - - - - - - - - -
    def weightTransferFunction2(self):
        return self.__weightTransferFunction2
    @weightTransferFunction2.setter
    def weightTransferFunction2(self, weightTransferFunction2):
        if  weightTransferFunction2 is None:
            weightTransferFunction2 = None # default
        assertValidSFNode(weightTransferFunction2)
        if not isinstance(weightTransferFunction2, object):
            # print(flush=True)
            raise X3DTypeError(str(weightTransferFunction2) + ' does not have a valid node type object')
        self.__weightTransferFunction2 = weightTransferFunction2
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.IS) or (self.metadata) or (self.renderStyle) or (self.voxels) or (self.weightTransferFunction1) or (self.weightTransferFunction2)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function BlendedVolumeStyle.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<BlendedVolumeStyle'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.enabled != True:
            result += " enabled='" + SFBool(self.enabled).XML() + "'"
        if self.weightConstant1 != 0.5:
            result += " weightConstant1='" + SFFloat(self.weightConstant1).XML() + "'"
        if self.weightConstant2 != 0.5:
            result += " weightConstant2='" + SFFloat(self.weightConstant2).XML() + "'"
        if self.weightFunction1 != 'CONSTANT':
            result += " weightFunction1='" + self.weightFunction1 + "'"
        if self.weightFunction2 != 'CONSTANT':
            result += " weightFunction2='" + self.weightFunction2 + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></BlendedVolumeStyle>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.renderStyle: # output this SFNode
                result += self.renderStyle.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.voxels: # output this SFNode
                result += self.voxels.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.weightTransferFunction1: # output this SFNode
                result += self.weightTransferFunction1.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.weightTransferFunction2: # output this SFNode
                result += self.weightTransferFunction2.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</BlendedVolumeStyle>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function BlendedVolumeStyle.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'BlendedVolumeStyle' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'BlendedVolumeStyle' + ' {'
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.enabled != True:
            result += " enabled " + SFBool(self.enabled).VRML() + ""
        if self.weightConstant1 != 0.5:
            result += " weightConstant1 " + SFFloat(self.weightConstant1).VRML() + ""
        if self.weightConstant2 != 0.5:
            result += " weightConstant2 " + SFFloat(self.weightConstant2).VRML() + ""
        if self.weightFunction1 != 'CONSTANT':
            result += " weightFunction1 " +  '"' + self.weightFunction1 + '"' + ""
        if self.weightFunction2 != 'CONSTANT':
            result += " weightFunction2 " +  '"' + self.weightFunction2 + '"' + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.renderStyle: # output this SFNode
            result += '\n' + '  ' + indent + 'renderStyle ' + self.renderStyle.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.voxels: # output this SFNode
            result += '\n' + '  ' + indent + 'voxels ' + self.voxels.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.weightTransferFunction1: # output this SFNode
            result += '\n' + '  ' + indent + 'weightTransferFunction1 ' + self.weightTransferFunction1.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.weightTransferFunction2: # output this SFNode
            result += '\n' + '  ' + indent + 'weightTransferFunction2 ' + self.weightTransferFunction2.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class BooleanFilter(_X3DChildNode):
    """
    BooleanFilter selectively passes true, false or negated events.
    """
    def NAME(self):
        return 'BooleanFilter'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#BooleanFilter'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode BooleanFilter __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function BooleanFilter.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<BooleanFilter'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></BooleanFilter>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</BooleanFilter>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function BooleanFilter.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'BooleanFilter' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'BooleanFilter' + ' {'
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class BooleanSequencer(_X3DSequencerNode):
    """
    BooleanSequencer generates periodic discrete Boolean values.
    """
    def NAME(self):
        return 'BooleanSequencer'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#BooleanSequencer'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('key', list(), FieldType.MFFloat, AccessType.inputOutput, 'X3DSequencerNode'),
        ('keyValue', list(), FieldType.MFBool, AccessType.inputOutput, 'BooleanSequencer'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 key=list(),
                 keyValue=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode BooleanSequencer __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.key = key
        self.keyValue = keyValue
    @property # getter - - - - - - - - - -
    def key(self):
        """Definition values for linear-interpolation function input intervals, listed in non-decreasing order and corresponding to a value in the keyValue array."""
        return self.__key
    @key.setter
    def key(self, key):
        if  key is None:
            key = MFFloat.DEFAULT_VALUE(self)
        assertValidMFFloat(key)
        self.__key = key
    @property # getter - - - - - - - - - -
    def keyValue(self):
        """Output values for linear sequencing, each corresponding to an input-fraction value in the key array."""
        return self.__keyValue
    @keyValue.setter
    def keyValue(self, keyValue):
        if  keyValue is None:
            keyValue = MFBool.DEFAULT_VALUE(self)
        assertValidMFBool(keyValue)
        self.__keyValue = keyValue
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function BooleanSequencer.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<BooleanSequencer'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.key != list():
            result += " key='" + MFFloat(self.key).XML() + "'"
        if self.keyValue != list():
            result += " keyValue='" + MFBool(self.keyValue).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></BooleanSequencer>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</BooleanSequencer>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function BooleanSequencer.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'BooleanSequencer' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'BooleanSequencer' + ' {'
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.key != list():
            result += " key " + MFFloat(self.key).VRML() + ""
        if self.keyValue != list():
            result += " keyValue " + MFBool(self.keyValue).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class BooleanToggle(_X3DChildNode):
    """
    BooleanToggle maintains state and negates output when a true input is provided.
    """
    def NAME(self):
        return 'BooleanToggle'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#BooleanToggle'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('toggle', False, FieldType.SFBool, AccessType.inputOutput, 'BooleanToggle'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 toggle=False,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode BooleanToggle __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.toggle = toggle
    @property # getter - - - - - - - - - -
    def toggle(self):
        """Persistent state value that gets toggled or reset."""
        return self.__toggle
    @toggle.setter
    def toggle(self, toggle):
        if  toggle is None:
            toggle = False # default
        assertValidSFBool(toggle)
        self.__toggle = toggle
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function BooleanToggle.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<BooleanToggle'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.toggle != False:
            result += " toggle='" + SFBool(self.toggle).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></BooleanToggle>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</BooleanToggle>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function BooleanToggle.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'BooleanToggle' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'BooleanToggle' + ' {'
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.toggle != False:
            result += " toggle " + SFBool(self.toggle).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class BooleanTrigger(_X3DTriggerNode):
    """
    BooleanTrigger converts time events to boolean true events.
    """
    def NAME(self):
        return 'BooleanTrigger'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#BooleanTrigger'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode BooleanTrigger __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function BooleanTrigger.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<BooleanTrigger'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></BooleanTrigger>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</BooleanTrigger>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function BooleanTrigger.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'BooleanTrigger' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'BooleanTrigger' + ' {'
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class BoundaryEnhancementVolumeStyle(_X3DComposableVolumeRenderStyleNode):
    """
    BoundaryEnhancementVolumeStyle provides boundary enhancement for the volume rendering style.
    """
    def NAME(self):
        return 'BoundaryEnhancementVolumeStyle'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#BoundaryEnhancementVolumeStyle'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('boundaryOpacity', 0.9, FieldType.SFFloat, AccessType.inputOutput, 'BoundaryEnhancementVolumeStyle'),
        ('enabled', True, FieldType.SFBool, AccessType.inputOutput, 'X3DVolumeRenderStyleNode'),
        ('opacityFactor', 2, FieldType.SFFloat, AccessType.inputOutput, 'BoundaryEnhancementVolumeStyle'),
        ('retainedOpacity', 0.2, FieldType.SFFloat, AccessType.inputOutput, 'BoundaryEnhancementVolumeStyle'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 boundaryOpacity=0.9,
                 enabled=True,
                 opacityFactor=2,
                 retainedOpacity=0.2,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode BoundaryEnhancementVolumeStyle __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.boundaryOpacity = boundaryOpacity
        self.enabled = enabled
        self.opacityFactor = opacityFactor
        self.retainedOpacity = retainedOpacity
    @property # getter - - - - - - - - - -
    def boundaryOpacity(self):
        """[0,+infinity) boundaryOpacity k_gs is the factored amount of the gradient enhancement to use."""
        return self.__boundaryOpacity
    @boundaryOpacity.setter
    def boundaryOpacity(self, boundaryOpacity):
        if  boundaryOpacity is None:
            boundaryOpacity = 0.9 # default
        assertValidSFFloat(boundaryOpacity)
        assertZeroToOne('boundaryOpacity', boundaryOpacity)
        self.__boundaryOpacity = boundaryOpacity
    @property # getter - - - - - - - - - -
    def enabled(self):
        """Enables/disables node operation."""
        return self.__enabled
    @enabled.setter
    def enabled(self, enabled):
        if  enabled is None:
            enabled = True # default
        assertValidSFBool(enabled)
        self.__enabled = enabled
    @property # getter - - - - - - - - - -
    def opacityFactor(self):
        """[0,+infinity) opacityFactor k_ge is the power function to control the slope of the opacity curve to highlight the set of data."""
        return self.__opacityFactor
    @opacityFactor.setter
    def opacityFactor(self, opacityFactor):
        if  opacityFactor is None:
            opacityFactor = 2 # default
        assertValidSFFloat(opacityFactor)
        assertNonNegative('opacityFactor', opacityFactor)
        self.__opacityFactor = opacityFactor
    @property # getter - - - - - - - - - -
    def retainedOpacity(self):
        """[0,1] retainedOpacity k_gc is the amount of initial opacity to mix into the output."""
        return self.__retainedOpacity
    @retainedOpacity.setter
    def retainedOpacity(self, retainedOpacity):
        if  retainedOpacity is None:
            retainedOpacity = 0.2 # default
        assertValidSFFloat(retainedOpacity)
        assertZeroToOne('retainedOpacity', retainedOpacity)
        self.__retainedOpacity = retainedOpacity
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function BoundaryEnhancementVolumeStyle.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<BoundaryEnhancementVolumeStyle'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.boundaryOpacity != 0.9:
            result += " boundaryOpacity='" + SFFloat(self.boundaryOpacity).XML() + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.enabled != True:
            result += " enabled='" + SFBool(self.enabled).XML() + "'"
        if self.opacityFactor != 2:
            result += " opacityFactor='" + SFFloat(self.opacityFactor).XML() + "'"
        if self.retainedOpacity != 0.2:
            result += " retainedOpacity='" + SFFloat(self.retainedOpacity).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></BoundaryEnhancementVolumeStyle>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</BoundaryEnhancementVolumeStyle>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function BoundaryEnhancementVolumeStyle.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'BoundaryEnhancementVolumeStyle' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'BoundaryEnhancementVolumeStyle' + ' {'
        if self.boundaryOpacity != 0.9:
            result += " boundaryOpacity " + SFFloat(self.boundaryOpacity).VRML() + ""
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.enabled != True:
            result += " enabled " + SFBool(self.enabled).VRML() + ""
        if self.opacityFactor != 2:
            result += " opacityFactor " + SFFloat(self.opacityFactor).VRML() + ""
        if self.retainedOpacity != 0.2:
            result += " retainedOpacity " + SFFloat(self.retainedOpacity).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class BoundedPhysicsModel(_X3DParticlePhysicsModelNode):
    """
    BoundedPhysicsModel provides user-defined geometrical boundaries for particle motion.
    """
    def NAME(self):
        return 'BoundedPhysicsModel'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#BoundedPhysicsModel'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('enabled', True, FieldType.SFBool, AccessType.inputOutput, 'X3DParticlePhysicsModelNode'),
        ('geometry', None, FieldType.SFNode, AccessType.inputOutput, 'BoundedPhysicsModel'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 enabled=True,
                 geometry=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode BoundedPhysicsModel __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.enabled = enabled
        self.geometry = geometry
    @property # getter - - - - - - - - - -
    def enabled(self):
        """Enables/disables node operation."""
        return self.__enabled
    @enabled.setter
    def enabled(self, enabled):
        if  enabled is None:
            enabled = True # default
        assertValidSFBool(enabled)
        self.__enabled = enabled
    @property # getter - - - - - - - - - -
    def geometry(self):
        """[X3DGeometryNode] Single contained geometry node provides the geometry used for each particle when the parent ParticleSystem node has geometryType=GEOMETRY."""
        return self.__geometry
    @geometry.setter
    def geometry(self, geometry):
        if  geometry is None:
            geometry = None # default
        assertValidSFNode(geometry)
        if not isinstance(geometry, object):
            # print(flush=True)
            raise X3DTypeError(str(geometry) + ' does not have a valid node type object')
        self.__geometry = geometry
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.geometry) or (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function BoundedPhysicsModel.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<BoundedPhysicsModel'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.enabled != True:
            result += " enabled='" + SFBool(self.enabled).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></BoundedPhysicsModel>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.geometry: # output this SFNode
                result += self.geometry.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</BoundedPhysicsModel>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function BoundedPhysicsModel.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'BoundedPhysicsModel' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'BoundedPhysicsModel' + ' {'
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.enabled != True:
            result += " enabled " + SFBool(self.enabled).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.geometry: # output this SFNode
            result += '\n' + '  ' + indent + 'geometry ' + self.geometry.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class Box(_X3DGeometryNode):
    """
    Box is a geometry node specifying a rectangular cuboid.
    """
    def NAME(self):
        return 'Box'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#Box'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('size', (2, 2, 2), FieldType.SFVec3f, AccessType.initializeOnly, 'Box'),
        ('solid', True, FieldType.SFBool, AccessType.initializeOnly, 'Box'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 size=(2, 2, 2),
                 solid=True,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode Box __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.size = size
        self.solid = solid
    @property # getter - - - - - - - - - -
    def size(self):
        """(0,+infinity) size x y z in meters."""
        return self.__size
    @size.setter
    def size(self, size):
        if  size is None:
            size = (2, 2, 2) # default
        assertValidSFVec3f(size)
        self.__size = size
    @property # getter - - - - - - - - - -
    def solid(self):
        """Setting solid true means draw only one side of polygons (backface culling on), setting solid false means draw both sides of polygons (backface culling off)."""
        return self.__solid
    @solid.setter
    def solid(self, solid):
        if  solid is None:
            solid = True # default
        assertValidSFBool(solid)
        self.__solid = solid
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Box.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<Box'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.size != (2, 2, 2):
            result += " size='" + SFVec3f(self.size).XML() + "'"
        if self.solid != True:
            result += " solid='" + SFBool(self.solid).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></Box>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</Box>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function Box.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'Box' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'Box' + ' {'
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.size != (2, 2, 2):
            result += " size " + SFVec3f(self.size).VRML() + ""
        if self.solid != True:
            result += " solid " + SFBool(self.solid).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class CADAssembly(_X3DGroupingNode, _X3DProductStructureChildNode):
    """
    CADAssembly holds a set of Computer-Aided Design (CAD) assemblies or parts grouped together.
    """
    def NAME(self):
        return 'CADAssembly'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#CADAssembly'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('bboxCenter', (0, 0, 0), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DGroupingNode'),
        ('bboxSize', (-1, -1, -1), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DGroupingNode'),
        ('displayBBox', False, FieldType.SFBool, AccessType.inputOutput, 'X3DGroupingNode'),
        ('name', '', FieldType.SFString, AccessType.inputOutput, 'X3DProductStructureChildNode'),
        ('visible', True, FieldType.SFBool, AccessType.inputOutput, 'X3DGroupingNode'),
        ('children', list(), FieldType.MFNode, AccessType.inputOutput, 'X3DGroupingNode'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 bboxCenter=(0, 0, 0),
                 bboxSize=(-1, -1, -1),
                 displayBBox=False,
                 name='',
                 visible=True,
                 children=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode CADAssembly __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.bboxCenter = bboxCenter
        self.bboxSize = bboxSize
        self.displayBBox = displayBBox
        self.name = name
        self.visible = visible
        self.children = children
    @property # getter - - - - - - - - - -
    def bboxCenter(self):
        """Bounding box center accompanies bboxSize and provides an optional hint for bounding box position offset from origin of local coordinate system."""
        return self.__bboxCenter
    @bboxCenter.setter
    def bboxCenter(self, bboxCenter):
        if  bboxCenter is None:
            bboxCenter = (0, 0, 0) # default
        assertValidSFVec3f(bboxCenter)
        self.__bboxCenter = bboxCenter
    @property # getter - - - - - - - - - -
    def bboxSize(self):
        """Bounding box size is usually omitted, and can easily be calculated automatically by an X3D player at scene-loading time with minimal computational cost."""
        return self.__bboxSize
    @bboxSize.setter
    def bboxSize(self, bboxSize):
        if  bboxSize is None:
            bboxSize = (-1, -1, -1) # default
        assertValidSFVec3f(bboxSize)
        assertBoundingBox('bboxSize', bboxSize)
        self.__bboxSize = bboxSize
    @property # getter - - - - - - - - - -
    def displayBBox(self):
        """Whether to display bounding box for associated geometry, aligned with world coordinates."""
        return self.__displayBBox
    @displayBBox.setter
    def displayBBox(self, displayBBox):
        if  displayBBox is None:
            displayBBox = False # default
        assertValidSFBool(displayBBox)
        self.__displayBBox = displayBBox
    @property # getter - - - - - - - - - -
    def name(self):
        """Optional name for this particular CAD node."""
        return self.__name
    @name.setter
    def name(self, name):
        if  name is None:
            name = SFString.DEFAULT_VALUE(self)
        assertValidSFString(name)
        self.__name = name
    @property # getter - - - - - - - - - -
    def visible(self):
        """Whether or not renderable content within this node is visually displayed."""
        return self.__visible
    @visible.setter
    def visible(self, visible):
        if  visible is None:
            visible = True # default
        assertValidSFBool(visible)
        self.__visible = visible
    @property # getter - - - - - - - - - -
    def children(self):
        """[X3DChildNode] Grouping nodes contain an ordered list of children nodes."""
        return self.__children
    @children.setter
    def children(self, children):
        if  children is None:
            children = MFNode.DEFAULT_VALUE(self)
        assertValidMFNode(children)
        self.__children = children
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.children) or (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function CADAssembly.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<CADAssembly'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.bboxCenter != (0, 0, 0):
            result += " bboxCenter='" + SFVec3f(self.bboxCenter).XML() + "'"
        if self.bboxSize != (-1, -1, -1):
            result += " bboxSize='" + SFVec3f(self.bboxSize).XML() + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.displayBBox != False:
            result += " displayBBox='" + SFBool(self.displayBBox).XML() + "'"
        if self.name:
            result += " name='" + self.name + "'"
        if self.visible != True:
            result += " visible='" + SFBool(self.visible).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></CADAssembly>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            ### if self.children: # walk each child in MFNode list, if any
            ### print('* CADAssembly found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(children)=' + str(len(self.children)) + ', now invoking XML(' + str(indentLevel+1) + ')', flush=True)
            for each in self.children:
               result += each.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</CADAssembly>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function CADAssembly.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'CADAssembly' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'CADAssembly' + ' {'
        if self.bboxCenter != (0, 0, 0):
            result += " bboxCenter " + SFVec3f(self.bboxCenter).VRML() + ""
        if self.bboxSize != (-1, -1, -1):
            result += " bboxSize " + SFVec3f(self.bboxSize).VRML() + ""
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.displayBBox != False:
            result += " displayBBox " + SFBool(self.displayBBox).VRML() + ""
        if self.name:
            result += " name " +  '"' + self.name + '"' + ""
        if self.visible != True:
            result += " visible " + SFBool(self.visible).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.children: # walk each child in MFNode list, if any
            for each in self.children:
                result += each.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class CADFace(_X3DProductStructureChildNode, _X3DBoundedObject):
    """
    CADFace holds geometry representing one face in a Computer-Aided Design (CAD) CADPart.
    """
    def NAME(self):
        return 'CADFace'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#CADFace'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('bboxCenter', (0, 0, 0), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DBoundedObject'),
        ('bboxSize', (-1, -1, -1), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DBoundedObject'),
        ('displayBBox', False, FieldType.SFBool, AccessType.inputOutput, 'X3DBoundedObject'),
        ('name', '', FieldType.SFString, AccessType.inputOutput, 'X3DProductStructureChildNode'),
        ('visible', True, FieldType.SFBool, AccessType.inputOutput, 'X3DBoundedObject'),
        ('shape', None, FieldType.SFNode, AccessType.inputOutput, 'CADFace'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 bboxCenter=(0, 0, 0),
                 bboxSize=(-1, -1, -1),
                 displayBBox=False,
                 name='',
                 visible=True,
                 shape=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode CADFace __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.bboxCenter = bboxCenter
        self.bboxSize = bboxSize
        self.displayBBox = displayBBox
        self.name = name
        self.visible = visible
        self.shape = shape
    @property # getter - - - - - - - - - -
    def bboxCenter(self):
        """Bounding box center accompanies bboxSize and provides an optional hint for bounding box position offset from origin of local coordinate system."""
        return self.__bboxCenter
    @bboxCenter.setter
    def bboxCenter(self, bboxCenter):
        if  bboxCenter is None:
            bboxCenter = (0, 0, 0) # default
        assertValidSFVec3f(bboxCenter)
        self.__bboxCenter = bboxCenter
    @property # getter - - - - - - - - - -
    def bboxSize(self):
        """Bounding box size is usually omitted, and can easily be calculated automatically by an X3D player at scene-loading time with minimal computational cost."""
        return self.__bboxSize
    @bboxSize.setter
    def bboxSize(self, bboxSize):
        if  bboxSize is None:
            bboxSize = (-1, -1, -1) # default
        assertValidSFVec3f(bboxSize)
        assertBoundingBox('bboxSize', bboxSize)
        self.__bboxSize = bboxSize
    @property # getter - - - - - - - - - -
    def displayBBox(self):
        """Whether to display bounding box for associated geometry, aligned with world coordinates."""
        return self.__displayBBox
    @displayBBox.setter
    def displayBBox(self, displayBBox):
        if  displayBBox is None:
            displayBBox = False # default
        assertValidSFBool(displayBBox)
        self.__displayBBox = displayBBox
    @property # getter - - - - - - - - - -
    def name(self):
        """Optional name for this particular CAD node."""
        return self.__name
    @name.setter
    def name(self, name):
        if  name is None:
            name = SFString.DEFAULT_VALUE(self)
        assertValidSFString(name)
        self.__name = name
    @property # getter - - - - - - - - - -
    def visible(self):
        """Whether or not renderable content within this node is visually displayed."""
        return self.__visible
    @visible.setter
    def visible(self, visible):
        if  visible is None:
            visible = True # default
        assertValidSFBool(visible)
        self.__visible = visible
    @property # getter - - - - - - - - - -
    def shape(self):
        """[X3DShapeNode|LOD|Transform] Contained Shape for this CADPart."""
        return self.__shape
    @shape.setter
    def shape(self, shape):
        if  shape is None:
            shape = None # default
        assertValidSFNode(shape)
        if not isinstance(shape, object):
            # print(flush=True)
            raise X3DTypeError(str(shape) + ' does not have a valid node type object')
        self.__shape = shape
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.IS) or (self.metadata) or (self.shape)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function CADFace.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<CADFace'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.bboxCenter != (0, 0, 0):
            result += " bboxCenter='" + SFVec3f(self.bboxCenter).XML() + "'"
        if self.bboxSize != (-1, -1, -1):
            result += " bboxSize='" + SFVec3f(self.bboxSize).XML() + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.displayBBox != False:
            result += " displayBBox='" + SFBool(self.displayBBox).XML() + "'"
        if self.name:
            result += " name='" + self.name + "'"
        if self.visible != True:
            result += " visible='" + SFBool(self.visible).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></CADFace>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.shape: # output this SFNode
                result += self.shape.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</CADFace>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function CADFace.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'CADFace' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'CADFace' + ' {'
        if self.bboxCenter != (0, 0, 0):
            result += " bboxCenter " + SFVec3f(self.bboxCenter).VRML() + ""
        if self.bboxSize != (-1, -1, -1):
            result += " bboxSize " + SFVec3f(self.bboxSize).VRML() + ""
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.displayBBox != False:
            result += " displayBBox " + SFBool(self.displayBBox).VRML() + ""
        if self.name:
            result += " name " +  '"' + self.name + '"' + ""
        if self.visible != True:
            result += " visible " + SFBool(self.visible).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.shape: # output this SFNode
            result += '\n' + '  ' + indent + 'shape ' + self.shape.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class CADLayer(_X3DGroupingNode):
    """
    CADLayer nodes define a hierarchy that shows layer structure for a Computer-Aided Design (CAD) model.
    """
    def NAME(self):
        return 'CADLayer'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#CADLayer'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('bboxCenter', (0, 0, 0), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DGroupingNode'),
        ('bboxSize', (-1, -1, -1), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DGroupingNode'),
        ('displayBBox', False, FieldType.SFBool, AccessType.inputOutput, 'X3DGroupingNode'),
        ('name', '', FieldType.SFString, AccessType.inputOutput, 'CADLayer'),
        ('visible', True, FieldType.SFBool, AccessType.inputOutput, 'X3DGroupingNode'),
        ('children', list(), FieldType.MFNode, AccessType.inputOutput, 'X3DGroupingNode'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 bboxCenter=(0, 0, 0),
                 bboxSize=(-1, -1, -1),
                 displayBBox=False,
                 name='',
                 visible=True,
                 children=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode CADLayer __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.bboxCenter = bboxCenter
        self.bboxSize = bboxSize
        self.displayBBox = displayBBox
        self.name = name
        self.visible = visible
        self.children = children
    @property # getter - - - - - - - - - -
    def bboxCenter(self):
        """Bounding box center accompanies bboxSize and provides an optional hint for bounding box position offset from origin of local coordinate system."""
        return self.__bboxCenter
    @bboxCenter.setter
    def bboxCenter(self, bboxCenter):
        if  bboxCenter is None:
            bboxCenter = (0, 0, 0) # default
        assertValidSFVec3f(bboxCenter)
        self.__bboxCenter = bboxCenter
    @property # getter - - - - - - - - - -
    def bboxSize(self):
        """Bounding box size is usually omitted, and can easily be calculated automatically by an X3D player at scene-loading time with minimal computational cost."""
        return self.__bboxSize
    @bboxSize.setter
    def bboxSize(self, bboxSize):
        if  bboxSize is None:
            bboxSize = (-1, -1, -1) # default
        assertValidSFVec3f(bboxSize)
        assertBoundingBox('bboxSize', bboxSize)
        self.__bboxSize = bboxSize
    @property # getter - - - - - - - - - -
    def displayBBox(self):
        return self.__displayBBox
    @displayBBox.setter
    def displayBBox(self, displayBBox):
        if  displayBBox is None:
            displayBBox = False # default
        assertValidSFBool(displayBBox)
        self.__displayBBox = displayBBox
    @property # getter - - - - - - - - - -
    def name(self):
        """Optional name for this particular CAD node."""
        return self.__name
    @name.setter
    def name(self, name):
        if  name is None:
            name = SFString.DEFAULT_VALUE(self)
        assertValidSFString(name)
        self.__name = name
    @property # getter - - - - - - - - - -
    def visible(self):
        """Whether or not renderable content within this node is visually displayed."""
        return self.__visible
    @visible.setter
    def visible(self, visible):
        if  visible is None:
            visible = True # default
        assertValidSFBool(visible)
        self.__visible = visible
    @property # getter - - - - - - - - - -
    def children(self):
        """[X3DChildNode] Grouping nodes contain an ordered list of children nodes."""
        return self.__children
    @children.setter
    def children(self, children):
        if  children is None:
            children = MFNode.DEFAULT_VALUE(self)
        assertValidMFNode(children)
        self.__children = children
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.children) or (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function CADLayer.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<CADLayer'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.bboxCenter != (0, 0, 0):
            result += " bboxCenter='" + SFVec3f(self.bboxCenter).XML() + "'"
        if self.bboxSize != (-1, -1, -1):
            result += " bboxSize='" + SFVec3f(self.bboxSize).XML() + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.displayBBox != False:
            result += " displayBBox='" + SFBool(self.displayBBox).XML() + "'"
        if self.name:
            result += " name='" + self.name + "'"
        if self.visible != True:
            result += " visible='" + SFBool(self.visible).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></CADLayer>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            ### if self.children: # walk each child in MFNode list, if any
            ### print('* CADLayer found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(children)=' + str(len(self.children)) + ', now invoking XML(' + str(indentLevel+1) + ')', flush=True)
            for each in self.children:
               result += each.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</CADLayer>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function CADLayer.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'CADLayer' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'CADLayer' + ' {'
        if self.bboxCenter != (0, 0, 0):
            result += " bboxCenter " + SFVec3f(self.bboxCenter).VRML() + ""
        if self.bboxSize != (-1, -1, -1):
            result += " bboxSize " + SFVec3f(self.bboxSize).VRML() + ""
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.displayBBox != False:
            result += " displayBBox " + SFBool(self.displayBBox).VRML() + ""
        if self.name:
            result += " name " +  '"' + self.name + '"' + ""
        if self.visible != True:
            result += " visible " + SFBool(self.visible).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.children: # walk each child in MFNode list, if any
            for each in self.children:
                result += each.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class CADPart(_X3DProductStructureChildNode, _X3DGroupingNode):
    """
    CADPart is an atomic part that defines both coordinate-system location and the faces that constitute a part in a Computer-Aided Design (CAD) model.
    """
    def NAME(self):
        return 'CADPart'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#CADPart'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('bboxCenter', (0, 0, 0), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DGroupingNode'),
        ('bboxSize', (-1, -1, -1), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DGroupingNode'),
        ('center', (0, 0, 0), FieldType.SFVec3f, AccessType.inputOutput, 'CADPart'),
        ('displayBBox', False, FieldType.SFBool, AccessType.inputOutput, 'X3DGroupingNode'),
        ('name', '', FieldType.SFString, AccessType.inputOutput, 'X3DProductStructureChildNode'),
        ('rotation', (0, 0, 1, 0), FieldType.SFRotation, AccessType.inputOutput, 'CADPart'),
        ('scale', (1, 1, 1), FieldType.SFVec3f, AccessType.inputOutput, 'CADPart'),
        ('scaleOrientation', (0, 0, 1, 0), FieldType.SFRotation, AccessType.inputOutput, 'CADPart'),
        ('translation', (0, 0, 0), FieldType.SFVec3f, AccessType.inputOutput, 'CADPart'),
        ('visible', True, FieldType.SFBool, AccessType.inputOutput, 'X3DGroupingNode'),
        ('children', list(), FieldType.MFNode, AccessType.inputOutput, 'CADPart'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 bboxCenter=(0, 0, 0),
                 bboxSize=(-1, -1, -1),
                 center=(0, 0, 0),
                 displayBBox=False,
                 name='',
                 rotation=(0, 0, 1, 0),
                 scale=(1, 1, 1),
                 scaleOrientation=(0, 0, 1, 0),
                 translation=(0, 0, 0),
                 visible=True,
                 children=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode CADPart __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.bboxCenter = bboxCenter
        self.bboxSize = bboxSize
        self.center = center
        self.displayBBox = displayBBox
        self.name = name
        self.rotation = rotation
        self.scale = scale
        self.scaleOrientation = scaleOrientation
        self.translation = translation
        self.visible = visible
        self.children = children
    @property # getter - - - - - - - - - -
    def bboxCenter(self):
        """Bounding box center accompanies bboxSize and provides an optional hint for bounding box position offset from origin of local coordinate system."""
        return self.__bboxCenter
    @bboxCenter.setter
    def bboxCenter(self, bboxCenter):
        if  bboxCenter is None:
            bboxCenter = (0, 0, 0) # default
        assertValidSFVec3f(bboxCenter)
        self.__bboxCenter = bboxCenter
    @property # getter - - - - - - - - - -
    def bboxSize(self):
        """Bounding box size is usually omitted, and can easily be calculated automatically by an X3D player at scene-loading time with minimal computational cost."""
        return self.__bboxSize
    @bboxSize.setter
    def bboxSize(self, bboxSize):
        if  bboxSize is None:
            bboxSize = (-1, -1, -1) # default
        assertValidSFVec3f(bboxSize)
        assertBoundingBox('bboxSize', bboxSize)
        self.__bboxSize = bboxSize
    @property # getter - - - - - - - - - -
    def center(self):
        """Translation offset from origin of local coordinate system, applied prior to rotation or scaling."""
        return self.__center
    @center.setter
    def center(self, center):
        if  center is None:
            center = (0, 0, 0) # default
        assertValidSFVec3f(center)
        self.__center = center
    @property # getter - - - - - - - - - -
    def displayBBox(self):
        """Whether to display bounding box for associated geometry, aligned with world coordinates."""
        return self.__displayBBox
    @displayBBox.setter
    def displayBBox(self, displayBBox):
        if  displayBBox is None:
            displayBBox = False # default
        assertValidSFBool(displayBBox)
        self.__displayBBox = displayBBox
    @property # getter - - - - - - - - - -
    def name(self):
        """Optional name for this particular CAD node."""
        return self.__name
    @name.setter
    def name(self, name):
        if  name is None:
            name = SFString.DEFAULT_VALUE(self)
        assertValidSFString(name)
        self.__name = name
    @property # getter - - - - - - - - - -
    def rotation(self):
        """Orientation (axis, angle in radians) of children relative to local coordinate system."""
        return self.__rotation
    @rotation.setter
    def rotation(self, rotation):
        if  rotation is None:
            rotation = (0, 0, 1, 0) # default
        assertValidSFRotation(rotation)
        self.__rotation = rotation
    @property # getter - - - - - - - - - -
    def scale(self):
        """Non-uniform x-y-z scale of child coordinate system, adjusted by center and scaleOrientation."""
        return self.__scale
    @scale.setter
    def scale(self, scale):
        if  scale is None:
            scale = (1, 1, 1) # default
        assertValidSFVec3f(scale)
        self.__scale = scale
    @property # getter - - - - - - - - - -
    def scaleOrientation(self):
        """Preliminary rotation of coordinate system before scaling (to allow scaling around arbitrary orientations)."""
        return self.__scaleOrientation
    @scaleOrientation.setter
    def scaleOrientation(self, scaleOrientation):
        if  scaleOrientation is None:
            scaleOrientation = (0, 0, 1, 0) # default
        assertValidSFRotation(scaleOrientation)
        self.__scaleOrientation = scaleOrientation
    @property # getter - - - - - - - - - -
    def translation(self):
        """Position (x, y, z in meters) of children relative to local coordinate system."""
        return self.__translation
    @translation.setter
    def translation(self, translation):
        if  translation is None:
            translation = (0, 0, 0) # default
        assertValidSFVec3f(translation)
        self.__translation = translation
    @property # getter - - - - - - - - - -
    def visible(self):
        """Whether or not renderable content within this node is visually displayed."""
        return self.__visible
    @visible.setter
    def visible(self, visible):
        if  visible is None:
            visible = True # default
        assertValidSFBool(visible)
        self.__visible = visible
    @property # getter - - - - - - - - - -
    def children(self):
        """[X3DChildNode] Grouping nodes contain an ordered list of children nodes."""
        return self.__children
    @children.setter
    def children(self, children):
        if  children is None:
            children = MFNode.DEFAULT_VALUE(self)
        assertValidMFNode(children)
        self.__children = children
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.children) or (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function CADPart.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<CADPart'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.bboxCenter != (0, 0, 0):
            result += " bboxCenter='" + SFVec3f(self.bboxCenter).XML() + "'"
        if self.bboxSize != (-1, -1, -1):
            result += " bboxSize='" + SFVec3f(self.bboxSize).XML() + "'"
        if self.center != (0, 0, 0):
            result += " center='" + SFVec3f(self.center).XML() + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.displayBBox != False:
            result += " displayBBox='" + SFBool(self.displayBBox).XML() + "'"
        if self.name:
            result += " name='" + self.name + "'"
        if self.rotation != (0, 0, 1, 0):
            result += " rotation='" + SFRotation(self.rotation).XML() + "'"
        if self.scale != (1, 1, 1):
            result += " scale='" + SFVec3f(self.scale).XML() + "'"
        if self.scaleOrientation != (0, 0, 1, 0):
            result += " scaleOrientation='" + SFRotation(self.scaleOrientation).XML() + "'"
        if self.translation != (0, 0, 0):
            result += " translation='" + SFVec3f(self.translation).XML() + "'"
        if self.visible != True:
            result += " visible='" + SFBool(self.visible).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></CADPart>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            ### if self.children: # walk each child in MFNode list, if any
            ### print('* CADPart found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(children)=' + str(len(self.children)) + ', now invoking XML(' + str(indentLevel+1) + ')', flush=True)
            for each in self.children:
               result += each.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</CADPart>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function CADPart.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'CADPart' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'CADPart' + ' {'
        if self.bboxCenter != (0, 0, 0):
            result += " bboxCenter " + SFVec3f(self.bboxCenter).VRML() + ""
        if self.bboxSize != (-1, -1, -1):
            result += " bboxSize " + SFVec3f(self.bboxSize).VRML() + ""
        if self.center != (0, 0, 0):
            result += " center " + SFVec3f(self.center).VRML() + ""
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.displayBBox != False:
            result += " displayBBox " + SFBool(self.displayBBox).VRML() + ""
        if self.name:
            result += " name " +  '"' + self.name + '"' + ""
        if self.rotation != (0, 0, 1, 0):
            result += " rotation " + SFRotation(self.rotation).VRML() + ""
        if self.scale != (1, 1, 1):
            result += " scale " + SFVec3f(self.scale).VRML() + ""
        if self.scaleOrientation != (0, 0, 1, 0):
            result += " scaleOrientation " + SFRotation(self.scaleOrientation).VRML() + ""
        if self.translation != (0, 0, 0):
            result += " translation " + SFVec3f(self.translation).VRML() + ""
        if self.visible != True:
            result += " visible " + SFBool(self.visible).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.children: # walk each child in MFNode list, if any
            for each in self.children:
                result += each.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class CartoonVolumeStyle(_X3DComposableVolumeRenderStyleNode):
    """
    CartoonVolumeStyle generates cartoon-style non-photorealistic rendering of associated volumetric data.
    """
    def NAME(self):
        return 'CartoonVolumeStyle'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#CartoonVolumeStyle'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('colorSteps', 4, FieldType.SFInt32, AccessType.inputOutput, 'CartoonVolumeStyle'),
        ('enabled', True, FieldType.SFBool, AccessType.inputOutput, 'X3DVolumeRenderStyleNode'),
        ('orthogonalColor', (1, 1, 1, 1), FieldType.SFColorRGBA, AccessType.inputOutput, 'CartoonVolumeStyle'),
        ('parallelColor', (0, 0, 0, 1), FieldType.SFColorRGBA, AccessType.inputOutput, 'CartoonVolumeStyle'),
        ('surfaceNormals', None, FieldType.SFNode, AccessType.inputOutput, 'CartoonVolumeStyle'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 colorSteps=4,
                 enabled=True,
                 orthogonalColor=(1, 1, 1, 1),
                 parallelColor=(0, 0, 0, 1),
                 surfaceNormals=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode CartoonVolumeStyle __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.colorSteps = colorSteps
        self.enabled = enabled
        self.orthogonalColor = orthogonalColor
        self.parallelColor = parallelColor
        self.surfaceNormals = surfaceNormals
    @property # getter - - - - - - - - - -
    def colorSteps(self):
        """[1,64] Number of distinct colors taken from interpolated colors and used to render the object."""
        return self.__colorSteps
    @colorSteps.setter
    def colorSteps(self, colorSteps):
        if  colorSteps is None:
            colorSteps = 4 # default
        assertValidSFInt32(colorSteps)
        assertGreaterThanEquals('colorSteps', colorSteps, 1)
        assertLessThanEquals('colorSteps', colorSteps, 64)
        self.__colorSteps = colorSteps
    @property # getter - - - - - - - - - -
    def enabled(self):
        """Enables/disables node operation."""
        return self.__enabled
    @enabled.setter
    def enabled(self, enabled):
        if  enabled is None:
            enabled = True # default
        assertValidSFBool(enabled)
        self.__enabled = enabled
    @property # getter - - - - - - - - - -
    def orthogonalColor(self):
        """[0,1] orthogonalColor is used for surface normals that are orthogonal (perpendicular) to viewer's current location."""
        return self.__orthogonalColor
    @orthogonalColor.setter
    def orthogonalColor(self, orthogonalColor):
        if  orthogonalColor is None:
            orthogonalColor = (1, 1, 1, 1) # default
        assertValidSFColorRGBA(orthogonalColor)
        assertZeroToOne('orthogonalColor', orthogonalColor)
        self.__orthogonalColor = orthogonalColor
    @property # getter - - - - - - - - - -
    def parallelColor(self):
        """[0,1] parallelColor is used for surface normals that are orthogonal to viewer's current location."""
        return self.__parallelColor
    @parallelColor.setter
    def parallelColor(self, parallelColor):
        if  parallelColor is None:
            parallelColor = (0, 0, 0, 1) # default
        assertValidSFColorRGBA(parallelColor)
        assertZeroToOne('parallelColor', parallelColor)
        self.__parallelColor = parallelColor
    @property # getter - - - - - - - - - -
    def surfaceNormals(self):
        return self.__surfaceNormals
    @surfaceNormals.setter
    def surfaceNormals(self, surfaceNormals):
        if  surfaceNormals is None:
            surfaceNormals = None # default
        assertValidSFNode(surfaceNormals)
        if not isinstance(surfaceNormals, object):
            # print(flush=True)
            raise X3DTypeError(str(surfaceNormals) + ' does not have a valid node type object')
        self.__surfaceNormals = surfaceNormals
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.IS) or (self.metadata) or (self.surfaceNormals)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function CartoonVolumeStyle.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<CartoonVolumeStyle'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.colorSteps != 4:
            result += " colorSteps='" + SFInt32(self.colorSteps).XML() + "'"
        if self.enabled != True:
            result += " enabled='" + SFBool(self.enabled).XML() + "'"
        if self.orthogonalColor != (1, 1, 1, 1):
            result += " orthogonalColor='" + SFColorRGBA(self.orthogonalColor).XML() + "'"
        if self.parallelColor != (0, 0, 0, 1):
            result += " parallelColor='" + SFColorRGBA(self.parallelColor).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></CartoonVolumeStyle>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.surfaceNormals: # output this SFNode
                result += self.surfaceNormals.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</CartoonVolumeStyle>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function CartoonVolumeStyle.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'CartoonVolumeStyle' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'CartoonVolumeStyle' + ' {'
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.colorSteps != 4:
            result += " colorSteps " + SFInt32(self.colorSteps).VRML() + ""
        if self.enabled != True:
            result += " enabled " + SFBool(self.enabled).VRML() + ""
        if self.orthogonalColor != (1, 1, 1, 1):
            result += " orthogonalColor " + SFColorRGBA(self.orthogonalColor).VRML() + ""
        if self.parallelColor != (0, 0, 0, 1):
            result += " parallelColor " + SFColorRGBA(self.parallelColor).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.surfaceNormals: # output this SFNode
            result += '\n' + '  ' + indent + 'surfaceNormals ' + self.surfaceNormals.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class Circle2D(_X3DGeometryNode):
    """
    Circle2D is a geometry node that defines a linear X-Y circle with center (0,0) in X-Y plane.
    """
    def NAME(self):
        return 'Circle2D'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#Circle2D'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('radius', 1, FieldType.SFFloat, AccessType.initializeOnly, 'Circle2D'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 radius=1,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode Circle2D __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.radius = radius
    @property # getter - - - - - - - - - -
    def radius(self):
        """(0,+infinity) circle radius."""
        return self.__radius
    @radius.setter
    def radius(self, radius):
        if  radius is None:
            radius = 1 # default
        assertValidSFFloat(radius)
        assertPositive('radius', radius)
        self.__radius = radius
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Circle2D.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<Circle2D'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.radius != 1:
            result += " radius='" + SFFloat(self.radius).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></Circle2D>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</Circle2D>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function Circle2D.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'Circle2D' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'Circle2D' + ' {'
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.radius != 1:
            result += " radius " + SFFloat(self.radius).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class ClipPlane(_X3DChildNode):
    """
    ClipPlane specifies a single plane equation used to clip (i.
    """
    def NAME(self):
        return 'ClipPlane'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ClipPlane'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('enabled', True, FieldType.SFBool, AccessType.inputOutput, 'ClipPlane'),
        ('plane', (0, 1, 0, 0), FieldType.SFVec4f, AccessType.inputOutput, 'ClipPlane'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 enabled=True,
                 plane=(0, 1, 0, 0),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode ClipPlane __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.enabled = enabled
        self.plane = plane
    @property # getter - - - - - - - - - -
    def enabled(self):
        """Enables/disables node operation."""
        return self.__enabled
    @enabled.setter
    def enabled(self, enabled):
        if  enabled is None:
            enabled = True # default
        assertValidSFBool(enabled)
        self.__enabled = enabled
    @property # getter - - - - - - - - - -
    def plane(self):
        """[0,1] If (a,b,c,d) is the plane, with the first three components being a normalized vector describing the plane's normal direction (and thus the fourth component d being distance from the origin), a point (x,y,z) is visible to the user, with regards to the clipping plane, if a*x+b*y+c*z+d is greater than 0."""
        return self.__plane
    @plane.setter
    def plane(self, plane):
        if  plane is None:
            plane = (0, 1, 0, 0) # default
        assertValidSFVec4f(plane)
        assertGreaterThanEquals('plane', plane, -1)
        assertLessThanEquals('plane', plane, 1)
        self.__plane = plane
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ClipPlane.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ClipPlane'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.enabled != True:
            result += " enabled='" + SFBool(self.enabled).XML() + "'"
        if self.plane != (0, 1, 0, 0):
            result += " plane='" + SFVec4f(self.plane).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></ClipPlane>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</ClipPlane>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function ClipPlane.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'ClipPlane' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'ClipPlane' + ' {'
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.enabled != True:
            result += " enabled " + SFBool(self.enabled).VRML() + ""
        if self.plane != (0, 1, 0, 0):
            result += " plane " + SFVec4f(self.plane).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class CollidableOffset(_X3DNBodyCollidableNode):
    """
    CollidableOffset repositions geometry relative to center of owning body.
    """
    def NAME(self):
        return 'CollidableOffset'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#CollidableOffset'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('bboxCenter', (0, 0, 0), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DNBodyCollidableNode'),
        ('bboxSize', (-1, -1, -1), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DNBodyCollidableNode'),
        ('displayBBox', False, FieldType.SFBool, AccessType.inputOutput, 'X3DNBodyCollidableNode'),
        ('enabled', True, FieldType.SFBool, AccessType.inputOutput, 'X3DNBodyCollidableNode'),
        ('rotation', (0, 0, 1, 0), FieldType.SFRotation, AccessType.inputOutput, 'X3DNBodyCollidableNode'),
        ('translation', (0, 0, 0), FieldType.SFVec3f, AccessType.inputOutput, 'X3DNBodyCollidableNode'),
        ('visible', True, FieldType.SFBool, AccessType.inputOutput, 'X3DNBodyCollidableNode'),
        ('collidable', None, FieldType.SFNode, AccessType.initializeOnly, 'CollidableOffset'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 bboxCenter=(0, 0, 0),
                 bboxSize=(-1, -1, -1),
                 displayBBox=False,
                 enabled=True,
                 rotation=(0, 0, 1, 0),
                 translation=(0, 0, 0),
                 visible=True,
                 collidable=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode CollidableOffset __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.bboxCenter = bboxCenter
        self.bboxSize = bboxSize
        self.displayBBox = displayBBox
        self.enabled = enabled
        self.rotation = rotation
        self.translation = translation
        self.visible = visible
        self.collidable = collidable
    @property # getter - - - - - - - - - -
    def bboxCenter(self):
        """Bounding box center accompanies bboxSize and provides an optional hint for bounding box position offset from origin of local coordinate system."""
        return self.__bboxCenter
    @bboxCenter.setter
    def bboxCenter(self, bboxCenter):
        if  bboxCenter is None:
            bboxCenter = (0, 0, 0) # default
        assertValidSFVec3f(bboxCenter)
        self.__bboxCenter = bboxCenter
    @property # getter - - - - - - - - - -
    def bboxSize(self):
        """Bounding box size is usually omitted, and can easily be calculated automatically by an X3D player at scene-loading time with minimal computational cost."""
        return self.__bboxSize
    @bboxSize.setter
    def bboxSize(self, bboxSize):
        if  bboxSize is None:
            bboxSize = (-1, -1, -1) # default
        assertValidSFVec3f(bboxSize)
        assertBoundingBox('bboxSize', bboxSize)
        self.__bboxSize = bboxSize
    @property # getter - - - - - - - - - -
    def displayBBox(self):
        """Whether to display bounding box for associated geometry, aligned with world coordinates."""
        return self.__displayBBox
    @displayBBox.setter
    def displayBBox(self, displayBBox):
        if  displayBBox is None:
            displayBBox = False # default
        assertValidSFBool(displayBBox)
        self.__displayBBox = displayBBox
    @property # getter - - - - - - - - - -
    def enabled(self):
        """Enables/disables node operation."""
        return self.__enabled
    @enabled.setter
    def enabled(self, enabled):
        if  enabled is None:
            enabled = True # default
        assertValidSFBool(enabled)
        self.__enabled = enabled
    @property # getter - - - - - - - - - -
    def rotation(self):
        """Orientation (axis, angle in radians) of children relative to local coordinate system."""
        return self.__rotation
    @rotation.setter
    def rotation(self, rotation):
        if  rotation is None:
            rotation = (0, 0, 1, 0) # default
        assertValidSFRotation(rotation)
        self.__rotation = rotation
    @property # getter - - - - - - - - - -
    def translation(self):
        """Position (x, y, z in meters) of children relative to local coordinate system."""
        return self.__translation
    @translation.setter
    def translation(self, translation):
        if  translation is None:
            translation = (0, 0, 0) # default
        assertValidSFVec3f(translation)
        self.__translation = translation
    @property # getter - - - - - - - - - -
    def visible(self):
        """Whether or not renderable content within this node is visually displayed."""
        return self.__visible
    @visible.setter
    def visible(self, visible):
        if  visible is None:
            visible = True # default
        assertValidSFBool(visible)
        self.__visible = visible
    @property # getter - - - - - - - - - -
    def collidable(self):
        return self.__collidable
    @collidable.setter
    def collidable(self, collidable):
        if  collidable is None:
            collidable = None # default
        assertValidSFNode(collidable)
        if not isinstance(collidable, object):
            # print(flush=True)
            raise X3DTypeError(str(collidable) + ' does not have a valid node type object')
        self.__collidable = collidable
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.collidable) or (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function CollidableOffset.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<CollidableOffset'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.bboxCenter != (0, 0, 0):
            result += " bboxCenter='" + SFVec3f(self.bboxCenter).XML() + "'"
        if self.bboxSize != (-1, -1, -1):
            result += " bboxSize='" + SFVec3f(self.bboxSize).XML() + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.displayBBox != False:
            result += " displayBBox='" + SFBool(self.displayBBox).XML() + "'"
        if self.enabled != True:
            result += " enabled='" + SFBool(self.enabled).XML() + "'"
        if self.rotation != (0, 0, 1, 0):
            result += " rotation='" + SFRotation(self.rotation).XML() + "'"
        if self.translation != (0, 0, 0):
            result += " translation='" + SFVec3f(self.translation).XML() + "'"
        if self.visible != True:
            result += " visible='" + SFBool(self.visible).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></CollidableOffset>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.collidable: # output this SFNode
                result += self.collidable.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</CollidableOffset>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function CollidableOffset.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'CollidableOffset' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'CollidableOffset' + ' {'
        if self.bboxCenter != (0, 0, 0):
            result += " bboxCenter " + SFVec3f(self.bboxCenter).VRML() + ""
        if self.bboxSize != (-1, -1, -1):
            result += " bboxSize " + SFVec3f(self.bboxSize).VRML() + ""
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.displayBBox != False:
            result += " displayBBox " + SFBool(self.displayBBox).VRML() + ""
        if self.enabled != True:
            result += " enabled " + SFBool(self.enabled).VRML() + ""
        if self.rotation != (0, 0, 1, 0):
            result += " rotation " + SFRotation(self.rotation).VRML() + ""
        if self.translation != (0, 0, 0):
            result += " translation " + SFVec3f(self.translation).VRML() + ""
        if self.visible != True:
            result += " visible " + SFBool(self.visible).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.collidable: # output this SFNode
            result += '\n' + '  ' + indent + 'collidable ' + self.collidable.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class CollidableShape(_X3DNBodyCollidableNode):
    """
    CollidableShape connects the collision detection system, the rigid body model, and the renderable scene graph.
    """
    def NAME(self):
        return 'CollidableShape'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#CollidableShape'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('bboxCenter', (0, 0, 0), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DNBodyCollidableNode'),
        ('bboxSize', (-1, -1, -1), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DNBodyCollidableNode'),
        ('displayBBox', False, FieldType.SFBool, AccessType.inputOutput, 'X3DNBodyCollidableNode'),
        ('enabled', True, FieldType.SFBool, AccessType.inputOutput, 'X3DNBodyCollidableNode'),
        ('rotation', (0, 0, 1, 0), FieldType.SFRotation, AccessType.inputOutput, 'X3DNBodyCollidableNode'),
        ('translation', (0, 0, 0), FieldType.SFVec3f, AccessType.inputOutput, 'X3DNBodyCollidableNode'),
        ('visible', True, FieldType.SFBool, AccessType.inputOutput, 'X3DNBodyCollidableNode'),
        ('shape', None, FieldType.SFNode, AccessType.initializeOnly, 'CollidableShape'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 bboxCenter=(0, 0, 0),
                 bboxSize=(-1, -1, -1),
                 displayBBox=False,
                 enabled=True,
                 rotation=(0, 0, 1, 0),
                 translation=(0, 0, 0),
                 visible=True,
                 shape=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode CollidableShape __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.bboxCenter = bboxCenter
        self.bboxSize = bboxSize
        self.displayBBox = displayBBox
        self.enabled = enabled
        self.rotation = rotation
        self.translation = translation
        self.visible = visible
        self.shape = shape
    @property # getter - - - - - - - - - -
    def bboxCenter(self):
        """Bounding box center accompanies bboxSize and provides an optional hint for bounding box position offset from origin of local coordinate system."""
        return self.__bboxCenter
    @bboxCenter.setter
    def bboxCenter(self, bboxCenter):
        if  bboxCenter is None:
            bboxCenter = (0, 0, 0) # default
        assertValidSFVec3f(bboxCenter)
        self.__bboxCenter = bboxCenter
    @property # getter - - - - - - - - - -
    def bboxSize(self):
        """Bounding box size is usually omitted, and can easily be calculated automatically by an X3D player at scene-loading time with minimal computational cost."""
        return self.__bboxSize
    @bboxSize.setter
    def bboxSize(self, bboxSize):
        if  bboxSize is None:
            bboxSize = (-1, -1, -1) # default
        assertValidSFVec3f(bboxSize)
        assertBoundingBox('bboxSize', bboxSize)
        self.__bboxSize = bboxSize
    @property # getter - - - - - - - - - -
    def displayBBox(self):
        """Whether to display bounding box for associated geometry, aligned with world coordinates."""
        return self.__displayBBox
    @displayBBox.setter
    def displayBBox(self, displayBBox):
        if  displayBBox is None:
            displayBBox = False # default
        assertValidSFBool(displayBBox)
        self.__displayBBox = displayBBox
    @property # getter - - - - - - - - - -
    def enabled(self):
        """Enables/disables node operation."""
        return self.__enabled
    @enabled.setter
    def enabled(self, enabled):
        if  enabled is None:
            enabled = True # default
        assertValidSFBool(enabled)
        self.__enabled = enabled
    @property # getter - - - - - - - - - -
    def rotation(self):
        """Orientation (axis, angle in radians) of children relative to local coordinate system."""
        return self.__rotation
    @rotation.setter
    def rotation(self, rotation):
        if  rotation is None:
            rotation = (0, 0, 1, 0) # default
        assertValidSFRotation(rotation)
        self.__rotation = rotation
    @property # getter - - - - - - - - - -
    def translation(self):
        """Position (x, y, z in meters) of children relative to local coordinate system."""
        return self.__translation
    @translation.setter
    def translation(self, translation):
        if  translation is None:
            translation = (0, 0, 0) # default
        assertValidSFVec3f(translation)
        self.__translation = translation
    @property # getter - - - - - - - - - -
    def visible(self):
        """Whether or not renderable content within this node is visually displayed."""
        return self.__visible
    @visible.setter
    def visible(self, visible):
        if  visible is None:
            visible = True # default
        assertValidSFBool(visible)
        self.__visible = visible
    @property # getter - - - - - - - - - -
    def shape(self):
        return self.__shape
    @shape.setter
    def shape(self, shape):
        if  shape is None:
            shape = None # default
        assertValidSFNode(shape)
        if not isinstance(shape, object):
            # print(flush=True)
            raise X3DTypeError(str(shape) + ' does not have a valid node type object')
        self.__shape = shape
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.IS) or (self.metadata) or (self.shape)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function CollidableShape.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<CollidableShape'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.bboxCenter != (0, 0, 0):
            result += " bboxCenter='" + SFVec3f(self.bboxCenter).XML() + "'"
        if self.bboxSize != (-1, -1, -1):
            result += " bboxSize='" + SFVec3f(self.bboxSize).XML() + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.displayBBox != False:
            result += " displayBBox='" + SFBool(self.displayBBox).XML() + "'"
        if self.enabled != True:
            result += " enabled='" + SFBool(self.enabled).XML() + "'"
        if self.rotation != (0, 0, 1, 0):
            result += " rotation='" + SFRotation(self.rotation).XML() + "'"
        if self.translation != (0, 0, 0):
            result += " translation='" + SFVec3f(self.translation).XML() + "'"
        if self.visible != True:
            result += " visible='" + SFBool(self.visible).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></CollidableShape>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.shape: # output this SFNode
                result += self.shape.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</CollidableShape>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function CollidableShape.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'CollidableShape' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'CollidableShape' + ' {'
        if self.bboxCenter != (0, 0, 0):
            result += " bboxCenter " + SFVec3f(self.bboxCenter).VRML() + ""
        if self.bboxSize != (-1, -1, -1):
            result += " bboxSize " + SFVec3f(self.bboxSize).VRML() + ""
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.displayBBox != False:
            result += " displayBBox " + SFBool(self.displayBBox).VRML() + ""
        if self.enabled != True:
            result += " enabled " + SFBool(self.enabled).VRML() + ""
        if self.rotation != (0, 0, 1, 0):
            result += " rotation " + SFRotation(self.rotation).VRML() + ""
        if self.translation != (0, 0, 0):
            result += " translation " + SFVec3f(self.translation).VRML() + ""
        if self.visible != True:
            result += " visible " + SFBool(self.visible).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.shape: # output this SFNode
            result += '\n' + '  ' + indent + 'shape ' + self.shape.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class Collision(_X3DGroupingNode, _X3DSensorNode):
    """
    Collision detects camera-to-object contact using current view and NavigationInfo avatarSize.
    """
    def NAME(self):
        return 'Collision'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#Collision'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('bboxCenter', (0, 0, 0), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DGroupingNode'),
        ('bboxSize', (-1, -1, -1), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DGroupingNode'),
        ('displayBBox', False, FieldType.SFBool, AccessType.inputOutput, 'X3DGroupingNode'),
        ('enabled', True, FieldType.SFBool, AccessType.inputOutput, 'X3DSensorNode'),
        ('visible', True, FieldType.SFBool, AccessType.inputOutput, 'X3DGroupingNode'),
        ('proxy', None, FieldType.SFNode, AccessType.initializeOnly, 'Collision'),
        ('children', list(), FieldType.MFNode, AccessType.inputOutput, 'X3DGroupingNode'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 bboxCenter=(0, 0, 0),
                 bboxSize=(-1, -1, -1),
                 displayBBox=False,
                 enabled=True,
                 visible=True,
                 proxy=None,
                 children=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode Collision __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.bboxCenter = bboxCenter
        self.bboxSize = bboxSize
        self.displayBBox = displayBBox
        self.enabled = enabled
        self.visible = visible
        self.proxy = proxy
        self.children = children
    @property # getter - - - - - - - - - -
    def bboxCenter(self):
        """Bounding box center accompanies bboxSize and provides an optional hint for bounding box position offset from origin of local coordinate system."""
        return self.__bboxCenter
    @bboxCenter.setter
    def bboxCenter(self, bboxCenter):
        if  bboxCenter is None:
            bboxCenter = (0, 0, 0) # default
        assertValidSFVec3f(bboxCenter)
        self.__bboxCenter = bboxCenter
    @property # getter - - - - - - - - - -
    def bboxSize(self):
        """Bounding box size is usually omitted, and can easily be calculated automatically by an X3D player at scene-loading time with minimal computational cost."""
        return self.__bboxSize
    @bboxSize.setter
    def bboxSize(self, bboxSize):
        if  bboxSize is None:
            bboxSize = (-1, -1, -1) # default
        assertValidSFVec3f(bboxSize)
        assertBoundingBox('bboxSize', bboxSize)
        self.__bboxSize = bboxSize
    @property # getter - - - - - - - - - -
    def displayBBox(self):
        """Whether to display bounding box for associated geometry, aligned with world coordinates."""
        return self.__displayBBox
    @displayBBox.setter
    def displayBBox(self, displayBBox):
        if  displayBBox is None:
            displayBBox = False # default
        assertValidSFBool(displayBBox)
        self.__displayBBox = displayBBox
    @property # getter - - - - - - - - - -
    def enabled(self):
        """Enables/disables collision detection for children and all descendants."""
        return self.__enabled
    @enabled.setter
    def enabled(self, enabled):
        if  enabled is None:
            enabled = True # default
        assertValidSFBool(enabled)
        self.__enabled = enabled
    @property # getter - - - - - - - - - -
    def visible(self):
        """Whether or not renderable content within this node is visually displayed."""
        return self.__visible
    @visible.setter
    def visible(self, visible):
        if  visible is None:
            visible = True # default
        assertValidSFBool(visible)
        self.__visible = visible
    @property # getter - - - - - - - - - -
    def proxy(self):
        """[X3DChildNode] The proxy node is used as a substitute for Collision children during collision detection, simplifying collision-intersection computations."""
        return self.__proxy
    @proxy.setter
    def proxy(self, proxy):
        if  proxy is None:
            proxy = None # default
        assertValidSFNode(proxy)
        if not isinstance(proxy, object):
            # print(flush=True)
            raise X3DTypeError(str(proxy) + ' does not have a valid node type object')
        self.__proxy = proxy
    @property # getter - - - - - - - - - -
    def children(self):
        """[X3DChildNode] Grouping nodes contain an ordered list of children nodes."""
        return self.__children
    @children.setter
    def children(self, children):
        if  children is None:
            children = MFNode.DEFAULT_VALUE(self)
        assertValidMFNode(children)
        self.__children = children
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.children) or (self.IS) or (self.metadata) or (self.proxy)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Collision.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<Collision'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.bboxCenter != (0, 0, 0):
            result += " bboxCenter='" + SFVec3f(self.bboxCenter).XML() + "'"
        if self.bboxSize != (-1, -1, -1):
            result += " bboxSize='" + SFVec3f(self.bboxSize).XML() + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.displayBBox != False:
            result += " displayBBox='" + SFBool(self.displayBBox).XML() + "'"
        if self.enabled != True:
            result += " enabled='" + SFBool(self.enabled).XML() + "'"
        if self.visible != True:
            result += " visible='" + SFBool(self.visible).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></Collision>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.proxy: # output this SFNode
                result += self.proxy.XML(indentLevel=indentLevel+1,syntax=syntax)
            ### if self.children: # walk each child in MFNode list, if any
            ### print('* Collision found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(children)=' + str(len(self.children)) + ', now invoking XML(' + str(indentLevel+1) + ')', flush=True)
            for each in self.children:
               result += each.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</Collision>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function Collision.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'Collision' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'Collision' + ' {'
        if self.bboxCenter != (0, 0, 0):
            result += " bboxCenter " + SFVec3f(self.bboxCenter).VRML() + ""
        if self.bboxSize != (-1, -1, -1):
            result += " bboxSize " + SFVec3f(self.bboxSize).VRML() + ""
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.displayBBox != False:
            result += " displayBBox " + SFBool(self.displayBBox).VRML() + ""
        if self.enabled != True:
            result += " enabled " + SFBool(self.enabled).VRML() + ""
        if self.visible != True:
            result += " visible " + SFBool(self.visible).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.proxy: # output this SFNode
            result += '\n' + '  ' + indent + 'proxy ' + self.proxy.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.children: # walk each child in MFNode list, if any
            for each in self.children:
                result += each.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class CollisionCollection(_X3DChildNode, _X3DBoundedObject):
    """
    CollisionCollection holds a collection of objects that can be managed as a single entity for resolution of inter-object collisions.
    """
    def NAME(self):
        return 'CollisionCollection'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#CollisionCollection'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('appliedParameters', ["BOUNCE"], FieldType.MFString, AccessType.inputOutput, 'CollisionCollection'),
        ('bboxCenter', (0, 0, 0), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DBoundedObject'),
        ('bboxSize', (-1, -1, -1), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DBoundedObject'),
        ('bounce', 0, FieldType.SFFloat, AccessType.inputOutput, 'CollisionCollection'),
        ('displayBBox', False, FieldType.SFBool, AccessType.inputOutput, 'X3DBoundedObject'),
        ('enabled', True, FieldType.SFBool, AccessType.inputOutput, 'CollisionCollection'),
        ('frictionCoefficients', (0, 0), FieldType.SFVec2f, AccessType.inputOutput, 'CollisionCollection'),
        ('minBounceSpeed', 0.1, FieldType.SFFloat, AccessType.inputOutput, 'CollisionCollection'),
        ('slipFactors', (0, 0), FieldType.SFVec2f, AccessType.inputOutput, 'CollisionCollection'),
        ('softnessConstantForceMix', 0.0001, FieldType.SFFloat, AccessType.inputOutput, 'CollisionCollection'),
        ('softnessErrorCorrection', 0.8, FieldType.SFFloat, AccessType.inputOutput, 'CollisionCollection'),
        ('surfaceSpeed', (0, 0), FieldType.SFVec2f, AccessType.inputOutput, 'CollisionCollection'),
        ('visible', True, FieldType.SFBool, AccessType.inputOutput, 'X3DBoundedObject'),
        ('collidables', list(), FieldType.MFNode, AccessType.inputOutput, 'CollisionCollection'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 appliedParameters=["BOUNCE"],
                 bboxCenter=(0, 0, 0),
                 bboxSize=(-1, -1, -1),
                 bounce=0,
                 displayBBox=False,
                 enabled=True,
                 frictionCoefficients=(0, 0),
                 minBounceSpeed=0.1,
                 slipFactors=(0, 0),
                 softnessConstantForceMix=0.0001,
                 softnessErrorCorrection=0.8,
                 surfaceSpeed=(0, 0),
                 visible=True,
                 collidables=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode CollisionCollection __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.appliedParameters = appliedParameters
        self.bboxCenter = bboxCenter
        self.bboxSize = bboxSize
        self.bounce = bounce
        self.displayBBox = displayBBox
        self.enabled = enabled
        self.frictionCoefficients = frictionCoefficients
        self.minBounceSpeed = minBounceSpeed
        self.slipFactors = slipFactors
        self.softnessConstantForceMix = softnessConstantForceMix
        self.softnessErrorCorrection = softnessErrorCorrection
        self.surfaceSpeed = surfaceSpeed
        self.visible = visible
        self.collidables = collidables
    @property # getter - - - - - - - - - -
    def appliedParameters(self):
        """Default global parameters for collision outputs of rigid body physics system."""
        return self.__appliedParameters
    @appliedParameters.setter
    def appliedParameters(self, appliedParameters):
        if  appliedParameters is None:
            appliedParameters = ["BOUNCE"] # default
        assertValidMFString(appliedParameters)
        assertValidAppliedParameters('appliedParameters', appliedParameters)
        self.__appliedParameters = appliedParameters
    @property # getter - - - - - - - - - -
    def bboxCenter(self):
        """Bounding box center accompanies bboxSize and provides an optional hint for bounding box position offset from origin of local coordinate system."""
        return self.__bboxCenter
    @bboxCenter.setter
    def bboxCenter(self, bboxCenter):
        if  bboxCenter is None:
            bboxCenter = (0, 0, 0) # default
        assertValidSFVec3f(bboxCenter)
        self.__bboxCenter = bboxCenter
    @property # getter - - - - - - - - - -
    def bboxSize(self):
        """Bounding box size is usually omitted, and can easily be calculated automatically by an X3D player at scene-loading time with minimal computational cost."""
        return self.__bboxSize
    @bboxSize.setter
    def bboxSize(self, bboxSize):
        if  bboxSize is None:
            bboxSize = (-1, -1, -1) # default
        assertValidSFVec3f(bboxSize)
        assertBoundingBox('bboxSize', bboxSize)
        self.__bboxSize = bboxSize
    @property # getter - - - - - - - - - -
    def bounce(self):
        """[0,1] bounce indicates bounciness (0 = no bounce at all, 1 = maximum bounce)."""
        return self.__bounce
    @bounce.setter
    def bounce(self, bounce):
        if  bounce is None:
            bounce = 0 # default
        assertValidSFFloat(bounce)
        assertZeroToOne('bounce', bounce)
        self.__bounce = bounce
    @property # getter - - - - - - - - - -
    def displayBBox(self):
        """Whether to display bounding box for associated geometry, aligned with world coordinates."""
        return self.__displayBBox
    @displayBBox.setter
    def displayBBox(self, displayBBox):
        if  displayBBox is None:
            displayBBox = False # default
        assertValidSFBool(displayBBox)
        self.__displayBBox = displayBBox
    @property # getter - - - - - - - - - -
    def enabled(self):
        """Enables/disables node operation."""
        return self.__enabled
    @enabled.setter
    def enabled(self, enabled):
        if  enabled is None:
            enabled = True # default
        assertValidSFBool(enabled)
        self.__enabled = enabled
    @property # getter - - - - - - - - - -
    def frictionCoefficients(self):
        """frictionCoefficients used for computing surface drag."""
        return self.__frictionCoefficients
    @frictionCoefficients.setter
    def frictionCoefficients(self, frictionCoefficients):
        if  frictionCoefficients is None:
            frictionCoefficients = (0, 0) # default
        assertValidSFVec2f(frictionCoefficients)
        assertNonNegative('frictionCoefficients', frictionCoefficients)
        self.__frictionCoefficients = frictionCoefficients
    @property # getter - - - - - - - - - -
    def minBounceSpeed(self):
        """[0,+infinity) minBounceSpeed m/s needed to bounce."""
        return self.__minBounceSpeed
    @minBounceSpeed.setter
    def minBounceSpeed(self, minBounceSpeed):
        if  minBounceSpeed is None:
            minBounceSpeed = 0.1 # default
        assertValidSFFloat(minBounceSpeed)
        assertNonNegative('minBounceSpeed', minBounceSpeed)
        self.__minBounceSpeed = minBounceSpeed
    @property # getter - - - - - - - - - -
    def slipFactors(self):
        """slipFactors used for computing surface drag."""
        return self.__slipFactors
    @slipFactors.setter
    def slipFactors(self, slipFactors):
        if  slipFactors is None:
            slipFactors = (0, 0) # default
        assertValidSFVec2f(slipFactors)
        self.__slipFactors = slipFactors
    @property # getter - - - - - - - - - -
    def softnessConstantForceMix(self):
        """[0,1] softnessConstantForceMix value applies a constant force value to make colliding surfaces appear to be somewhat soft."""
        return self.__softnessConstantForceMix
    @softnessConstantForceMix.setter
    def softnessConstantForceMix(self, softnessConstantForceMix):
        if  softnessConstantForceMix is None:
            softnessConstantForceMix = 0.0001 # default
        assertValidSFFloat(softnessConstantForceMix)
        assertZeroToOne('softnessConstantForceMix', softnessConstantForceMix)
        self.__softnessConstantForceMix = softnessConstantForceMix
    @property # getter - - - - - - - - - -
    def softnessErrorCorrection(self):
        """[0,1] softnessErrorCorrection indicates fraction of collision error fixed in a set of evaluations (0 = no error correction, 1 = all errors corrected in single step)."""
        return self.__softnessErrorCorrection
    @softnessErrorCorrection.setter
    def softnessErrorCorrection(self, softnessErrorCorrection):
        if  softnessErrorCorrection is None:
            softnessErrorCorrection = 0.8 # default
        assertValidSFFloat(softnessErrorCorrection)
        assertZeroToOne('softnessErrorCorrection', softnessErrorCorrection)
        self.__softnessErrorCorrection = softnessErrorCorrection
    @property # getter - - - - - - - - - -
    def surfaceSpeed(self):
        """surfaceSpeed defines speed vectors for computing surface drag, if contact surfaces move independently of bodies."""
        return self.__surfaceSpeed
    @surfaceSpeed.setter
    def surfaceSpeed(self, surfaceSpeed):
        if  surfaceSpeed is None:
            surfaceSpeed = (0, 0) # default
        assertValidSFVec2f(surfaceSpeed)
        self.__surfaceSpeed = surfaceSpeed
    @property # getter - - - - - - - - - -
    def visible(self):
        """Whether or not renderable content within this node is visually displayed."""
        return self.__visible
    @visible.setter
    def visible(self, visible):
        if  visible is None:
            visible = True # default
        assertValidSFBool(visible)
        self.__visible = visible
    @property # getter - - - - - - - - - -
    def collidables(self):
        """[CollisionSpace|CollidableShape|CollidableOffset] CollisionCollection node holds a collection of objects in the collidables field that can be managed as a single entity for resolution of inter-object collisions with other groups of collidable objects."""
        return self.__collidables
    @collidables.setter
    def collidables(self, collidables):
        if  collidables is None:
            collidables = MFNode.DEFAULT_VALUE(self)
        assertValidMFNode(collidables)
        self.__collidables = collidables
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.collidables) or (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function CollisionCollection.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<CollisionCollection'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.appliedParameters != ["BOUNCE"]:
            result += " appliedParameters='" + MFString(self.appliedParameters).XML() + "'"
        if self.bboxCenter != (0, 0, 0):
            result += " bboxCenter='" + SFVec3f(self.bboxCenter).XML() + "'"
        if self.bboxSize != (-1, -1, -1):
            result += " bboxSize='" + SFVec3f(self.bboxSize).XML() + "'"
        if self.bounce != 0:
            result += " bounce='" + SFFloat(self.bounce).XML() + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.displayBBox != False:
            result += " displayBBox='" + SFBool(self.displayBBox).XML() + "'"
        if self.enabled != True:
            result += " enabled='" + SFBool(self.enabled).XML() + "'"
        if self.frictionCoefficients != (0, 0):
            result += " frictionCoefficients='" + SFVec2f(self.frictionCoefficients).XML() + "'"
        if self.minBounceSpeed != 0.1:
            result += " minBounceSpeed='" + SFFloat(self.minBounceSpeed).XML() + "'"
        if self.slipFactors != (0, 0):
            result += " slipFactors='" + SFVec2f(self.slipFactors).XML() + "'"
        if self.softnessConstantForceMix != 0.0001:
            result += " softnessConstantForceMix='" + SFFloat(self.softnessConstantForceMix).XML() + "'"
        if self.softnessErrorCorrection != 0.8:
            result += " softnessErrorCorrection='" + SFFloat(self.softnessErrorCorrection).XML() + "'"
        if self.surfaceSpeed != (0, 0):
            result += " surfaceSpeed='" + SFVec2f(self.surfaceSpeed).XML() + "'"
        if self.visible != True:
            result += " visible='" + SFBool(self.visible).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></CollisionCollection>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            ### if self.collidables: # walk each child in MFNode list, if any
            ### print('* CollisionCollection found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(collidables)=' + str(len(self.collidables)) + ', now invoking XML(' + str(indentLevel+1) + ')', flush=True)
            for each in self.collidables:
               result += each.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</CollisionCollection>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function CollisionCollection.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'CollisionCollection' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'CollisionCollection' + ' {'
        if self.appliedParameters != ["BOUNCE"]:
            result += " appliedParameters " + MFString(self.appliedParameters).VRML() + ""
        if self.bboxCenter != (0, 0, 0):
            result += " bboxCenter " + SFVec3f(self.bboxCenter).VRML() + ""
        if self.bboxSize != (-1, -1, -1):
            result += " bboxSize " + SFVec3f(self.bboxSize).VRML() + ""
        if self.bounce != 0:
            result += " bounce " + SFFloat(self.bounce).VRML() + ""
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.displayBBox != False:
            result += " displayBBox " + SFBool(self.displayBBox).VRML() + ""
        if self.enabled != True:
            result += " enabled " + SFBool(self.enabled).VRML() + ""
        if self.frictionCoefficients != (0, 0):
            result += " frictionCoefficients " + SFVec2f(self.frictionCoefficients).VRML() + ""
        if self.minBounceSpeed != 0.1:
            result += " minBounceSpeed " + SFFloat(self.minBounceSpeed).VRML() + ""
        if self.slipFactors != (0, 0):
            result += " slipFactors " + SFVec2f(self.slipFactors).VRML() + ""
        if self.softnessConstantForceMix != 0.0001:
            result += " softnessConstantForceMix " + SFFloat(self.softnessConstantForceMix).VRML() + ""
        if self.softnessErrorCorrection != 0.8:
            result += " softnessErrorCorrection " + SFFloat(self.softnessErrorCorrection).VRML() + ""
        if self.surfaceSpeed != (0, 0):
            result += " surfaceSpeed " + SFVec2f(self.surfaceSpeed).VRML() + ""
        if self.visible != True:
            result += " visible " + SFBool(self.visible).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.collidables: # walk each child in MFNode list, if any
            for each in self.collidables:
                result += each.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class CollisionSensor(_X3DSensorNode):
    """
    CollisionSensor generates collision-detection events.
    """
    def NAME(self):
        return 'CollisionSensor'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#CollisionSensor'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('enabled', True, FieldType.SFBool, AccessType.inputOutput, 'X3DSensorNode'),
        ('collider', None, FieldType.SFNode, AccessType.inputOutput, 'CollisionSensor'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 enabled=True,
                 collider=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode CollisionSensor __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.enabled = enabled
        self.collider = collider
    @property # getter - - - - - - - - - -
    def enabled(self):
        """Enables/disables node operation."""
        return self.__enabled
    @enabled.setter
    def enabled(self, enabled):
        if  enabled is None:
            enabled = True # default
        assertValidSFBool(enabled)
        self.__enabled = enabled
    @property # getter - - - - - - - - - -
    def collider(self):
        return self.__collider
    @collider.setter
    def collider(self, collider):
        if  collider is None:
            collider = None # default
        assertValidSFNode(collider)
        if not isinstance(collider, object):
            # print(flush=True)
            raise X3DTypeError(str(collider) + ' does not have a valid node type object')
        self.__collider = collider
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.collider) or (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function CollisionSensor.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<CollisionSensor'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.enabled != True:
            result += " enabled='" + SFBool(self.enabled).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></CollisionSensor>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.collider: # output this SFNode
                result += self.collider.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</CollisionSensor>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function CollisionSensor.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'CollisionSensor' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'CollisionSensor' + ' {'
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.enabled != True:
            result += " enabled " + SFBool(self.enabled).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.collider: # output this SFNode
            result += '\n' + '  ' + indent + 'collider ' + self.collider.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class CollisionSpace(_X3DNBodyCollisionSpaceNode):
    """
    CollisionSpace holds collection of objects considered together for resolution of inter-object collisions.
    """
    def NAME(self):
        return 'CollisionSpace'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#CollisionSpace'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('bboxCenter', (0, 0, 0), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DNBodyCollisionSpaceNode'),
        ('bboxSize', (-1, -1, -1), FieldType.SFVec3f, AccessType.initializeOnly, 'X3DNBodyCollisionSpaceNode'),
        ('displayBBox', False, FieldType.SFBool, AccessType.inputOutput, 'X3DNBodyCollisionSpaceNode'),
        ('enabled', True, FieldType.SFBool, AccessType.inputOutput, 'X3DNBodyCollisionSpaceNode'),
        ('useGeometry', False, FieldType.SFBool, AccessType.inputOutput, 'CollisionSpace'),
        ('visible', True, FieldType.SFBool, AccessType.inputOutput, 'X3DNBodyCollisionSpaceNode'),
        ('collidables', list(), FieldType.MFNode, AccessType.inputOutput, 'CollisionSpace'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 bboxCenter=(0, 0, 0),
                 bboxSize=(-1, -1, -1),
                 displayBBox=False,
                 enabled=True,
                 useGeometry=False,
                 visible=True,
                 collidables=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode CollisionSpace __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.bboxCenter = bboxCenter
        self.bboxSize = bboxSize
        self.displayBBox = displayBBox
        self.enabled = enabled
        self.useGeometry = useGeometry
        self.visible = visible
        self.collidables = collidables
    @property # getter - - - - - - - - - -
    def bboxCenter(self):
        """Bounding box center accompanies bboxSize and provides an optional hint for bounding box position offset from origin of local coordinate system."""
        return self.__bboxCenter
    @bboxCenter.setter
    def bboxCenter(self, bboxCenter):
        if  bboxCenter is None:
            bboxCenter = (0, 0, 0) # default
        assertValidSFVec3f(bboxCenter)
        self.__bboxCenter = bboxCenter
    @property # getter - - - - - - - - - -
    def bboxSize(self):
        """Bounding box size is usually omitted, and can easily be calculated automatically by an X3D player at scene-loading time with minimal computational cost."""
        return self.__bboxSize
    @bboxSize.setter
    def bboxSize(self, bboxSize):
        if  bboxSize is None:
            bboxSize = (-1, -1, -1) # default
        assertValidSFVec3f(bboxSize)
        assertBoundingBox('bboxSize', bboxSize)
        self.__bboxSize = bboxSize
    @property # getter - - - - - - - - - -
    def displayBBox(self):
        """Whether to display bounding box for associated geometry, aligned with world coordinates."""
        return self.__displayBBox
    @displayBBox.setter
    def displayBBox(self, displayBBox):
        if  displayBBox is None:
            displayBBox = False # default
        assertValidSFBool(displayBBox)
        self.__displayBBox = displayBBox
    @property # getter - - - - - - - - - -
    def enabled(self):
        """Enables/disables node operation."""
        return self.__enabled
    @enabled.setter
    def enabled(self, enabled):
        if  enabled is None:
            enabled = True # default
        assertValidSFBool(enabled)
        self.__enabled = enabled
    @property # getter - - - - - - - - - -
    def useGeometry(self):
        """useGeometry indicates whether collision-detection code checks down to level of geometry, or only make approximations using geometry bounds."""
        return self.__useGeometry
    @useGeometry.setter
    def useGeometry(self, useGeometry):
        if  useGeometry is None:
            useGeometry = False # default
        assertValidSFBool(useGeometry)
        self.__useGeometry = useGeometry
    @property # getter - - - - - - - - - -
    def visible(self):
        """Whether or not renderable content within this node is visually displayed."""
        return self.__visible
    @visible.setter
    def visible(self, visible):
        if  visible is None:
            visible = True # default
        assertValidSFBool(visible)
        self.__visible = visible
    @property # getter - - - - - - - - - -
    def collidables(self):
        return self.__collidables
    @collidables.setter
    def collidables(self, collidables):
        if  collidables is None:
            collidables = MFNode.DEFAULT_VALUE(self)
        assertValidMFNode(collidables)
        self.__collidables = collidables
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.collidables) or (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function CollisionSpace.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<CollisionSpace'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.bboxCenter != (0, 0, 0):
            result += " bboxCenter='" + SFVec3f(self.bboxCenter).XML() + "'"
        if self.bboxSize != (-1, -1, -1):
            result += " bboxSize='" + SFVec3f(self.bboxSize).XML() + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.displayBBox != False:
            result += " displayBBox='" + SFBool(self.displayBBox).XML() + "'"
        if self.enabled != True:
            result += " enabled='" + SFBool(self.enabled).XML() + "'"
        if self.useGeometry != False:
            result += " useGeometry='" + SFBool(self.useGeometry).XML() + "'"
        if self.visible != True:
            result += " visible='" + SFBool(self.visible).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></CollisionSpace>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            ### if self.collidables: # walk each child in MFNode list, if any
            ### print('* CollisionSpace found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(collidables)=' + str(len(self.collidables)) + ', now invoking XML(' + str(indentLevel+1) + ')', flush=True)
            for each in self.collidables:
               result += each.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</CollisionSpace>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function CollisionSpace.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'CollisionSpace' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'CollisionSpace' + ' {'
        if self.bboxCenter != (0, 0, 0):
            result += " bboxCenter " + SFVec3f(self.bboxCenter).VRML() + ""
        if self.bboxSize != (-1, -1, -1):
            result += " bboxSize " + SFVec3f(self.bboxSize).VRML() + ""
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.displayBBox != False:
            result += " displayBBox " + SFBool(self.displayBBox).VRML() + ""
        if self.enabled != True:
            result += " enabled " + SFBool(self.enabled).VRML() + ""
        if self.useGeometry != False:
            result += " useGeometry " + SFBool(self.useGeometry).VRML() + ""
        if self.visible != True:
            result += " visible " + SFBool(self.visible).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.collidables: # walk each child in MFNode list, if any
            for each in self.collidables:
                result += each.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class Color(_X3DColorNode):
    """
    Color node defines a set of RGB color values that apply either to a sibling Coordinate|CoordinateDouble node, or else to a parent ElevationGrid node.
    """
    def NAME(self):
        return 'Color'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#Color'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('color', list(), FieldType.MFColor, AccessType.inputOutput, 'Color'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 color=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode Color __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.color = color
    @property # getter - - - - - - - - - -
    def color(self):
        """The color field defines an array of 3-tuple RGB colors."""
        return self.__color
    @color.setter
    def color(self, color):
        if  color is None:
            color = MFColor.DEFAULT_VALUE(self)
        assertValidMFColor(color)
        assertZeroToOne('color', color)
        self.__color = color
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Color.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<Color'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.color != list():
            result += " color='" + MFColor(self.color).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></Color>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</Color>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function Color.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'Color' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'Color' + ' {'
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.color != list():
            result += " color " + MFColor(self.color).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class ColorChaser(_X3DChaserNode):
    """
    ColorChaser generates a series of SFColor values that progressively change from initial value to destination value.
    """
    def NAME(self):
        return 'ColorChaser'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ColorChaser'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('duration', 1, FieldType.SFTime, AccessType.initializeOnly, 'X3DChaserNode'),
        ('initialDestination', (0.8, 0.8, 0.8), FieldType.SFColor, AccessType.initializeOnly, 'ColorChaser'),
        ('initialValue', (0.8, 0.8, 0.8), FieldType.SFColor, AccessType.initializeOnly, 'ColorChaser'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 duration=1,
                 initialDestination=(0.8, 0.8, 0.8),
                 initialValue=(0.8, 0.8, 0.8),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode ColorChaser __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.duration = duration
        self.initialDestination = initialDestination
        self.initialValue = initialValue
    @property # getter - - - - - - - - - -
    def duration(self):
        """[0,+infinity) duration is the time interval for filter response in seconds."""
        return self.__duration
    @duration.setter
    def duration(self, duration):
        if  duration is None:
            duration = 1 # default
        assertValidSFTime(duration)
        assertNonNegative('duration', duration)
        self.__duration = duration
    @property # getter - - - - - - - - - -
    def initialDestination(self):
        """Initial destination value for this node."""
        return self.__initialDestination
    @initialDestination.setter
    def initialDestination(self, initialDestination):
        if  initialDestination is None:
            initialDestination = (0.8, 0.8, 0.8) # default
        assertValidSFColor(initialDestination)
        assertZeroToOne('initialDestination', initialDestination)
        self.__initialDestination = initialDestination
    @property # getter - - - - - - - - - -
    def initialValue(self):
        """Initial starting value for this node."""
        return self.__initialValue
    @initialValue.setter
    def initialValue(self, initialValue):
        if  initialValue is None:
            initialValue = (0.8, 0.8, 0.8) # default
        assertValidSFColor(initialValue)
        assertZeroToOne('initialValue', initialValue)
        self.__initialValue = initialValue
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ColorChaser.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ColorChaser'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.duration != 1:
            result += " duration='" + SFTime(self.duration).XML() + "'"
        if self.initialDestination != (0.8, 0.8, 0.8):
            result += " initialDestination='" + SFColor(self.initialDestination).XML() + "'"
        if self.initialValue != (0.8, 0.8, 0.8):
            result += " initialValue='" + SFColor(self.initialValue).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></ColorChaser>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</ColorChaser>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function ColorChaser.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'ColorChaser' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'ColorChaser' + ' {'
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.duration != 1:
            result += " duration " + SFTime(self.duration).VRML() + ""
        if self.initialDestination != (0.8, 0.8, 0.8):
            result += " initialDestination " + SFColor(self.initialDestination).VRML() + ""
        if self.initialValue != (0.8, 0.8, 0.8):
            result += " initialValue " + SFColor(self.initialValue).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class ColorDamper(_X3DDamperNode):
    """
    ColorDamper generates a series of RGB color values that progressively change from initial value to destination value.
    """
    def NAME(self):
        return 'ColorDamper'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ColorDamper'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('initialDestination', (0.8, 0.8, 0.8), FieldType.SFColor, AccessType.initializeOnly, 'ColorDamper'),
        ('initialValue', (0.8, 0.8, 0.8), FieldType.SFColor, AccessType.initializeOnly, 'ColorDamper'),
        ('order', 3, FieldType.SFInt32, AccessType.initializeOnly, 'X3DDamperNode'),
        ('tau', 0.3, FieldType.SFTime, AccessType.inputOutput, 'X3DDamperNode'),
        ('tolerance', -1, FieldType.SFFloat, AccessType.inputOutput, 'X3DDamperNode'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 initialDestination=(0.8, 0.8, 0.8),
                 initialValue=(0.8, 0.8, 0.8),
                 order=3,
                 tau=0.3,
                 tolerance=-1,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode ColorDamper __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.initialDestination = initialDestination
        self.initialValue = initialValue
        self.order = order
        self.tau = tau
        self.tolerance = tolerance
    @property # getter - - - - - - - - - -
    def initialDestination(self):
        """Initial destination value for this node."""
        return self.__initialDestination
    @initialDestination.setter
    def initialDestination(self, initialDestination):
        if  initialDestination is None:
            initialDestination = (0.8, 0.8, 0.8) # default
        assertValidSFColor(initialDestination)
        assertZeroToOne('initialDestination', initialDestination)
        self.__initialDestination = initialDestination
    @property # getter - - - - - - - - - -
    def initialValue(self):
        """Initial starting value for this node."""
        return self.__initialValue
    @initialValue.setter
    def initialValue(self, initialValue):
        if  initialValue is None:
            initialValue = (0.8, 0.8, 0.8) # default
        assertValidSFColor(initialValue)
        assertZeroToOne('initialValue', initialValue)
        self.__initialValue = initialValue
    @property # getter - - - - - - - - - -
    def order(self):
        """[0,5] order defines the number of internal filters (larger means smoother response, longer delay)."""
        return self.__order
    @order.setter
    def order(self, order):
        if  order is None:
            order = 3 # default
        assertValidSFInt32(order)
        assertGreaterThanEquals('order', order, 0)
        assertLessThanEquals('order', order, 5)
        self.__order = order
    @property # getter - - - - - - - - - -
    def tau(self):
        """[0,+infinity) tau is the exponential-decay time constant for filter response in seconds."""
        return self.__tau
    @tau.setter
    def tau(self, tau):
        if  tau is None:
            tau = 0.3 # default
        assertValidSFTime(tau)
        assertNonNegative('tau', tau)
        self.__tau = tau
    @property # getter - - - - - - - - - -
    def tolerance(self):
        """[0,+infinity) or -1."""
        return self.__tolerance
    @tolerance.setter
    def tolerance(self, tolerance):
        if  tolerance is None:
            tolerance = -1 # default
        assertValidSFFloat(tolerance)
        self.__tolerance = tolerance
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ColorDamper.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ColorDamper'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.initialDestination != (0.8, 0.8, 0.8):
            result += " initialDestination='" + SFColor(self.initialDestination).XML() + "'"
        if self.initialValue != (0.8, 0.8, 0.8):
            result += " initialValue='" + SFColor(self.initialValue).XML() + "'"
        if self.order != 3:
            result += " order='" + SFInt32(self.order).XML() + "'"
        if self.tau != 0.3:
            result += " tau='" + SFTime(self.tau).XML() + "'"
        if self.tolerance != -1:
            result += " tolerance='" + SFFloat(self.tolerance).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></ColorDamper>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</ColorDamper>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function ColorDamper.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'ColorDamper' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'ColorDamper' + ' {'
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.initialDestination != (0.8, 0.8, 0.8):
            result += " initialDestination " + SFColor(self.initialDestination).VRML() + ""
        if self.initialValue != (0.8, 0.8, 0.8):
            result += " initialValue " + SFColor(self.initialValue).VRML() + ""
        if self.order != 3:
            result += " order " + SFInt32(self.order).VRML() + ""
        if self.tau != 0.3:
            result += " tau " + SFTime(self.tau).VRML() + ""
        if self.tolerance != -1:
            result += " tolerance " + SFFloat(self.tolerance).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class ColorInterpolator(_X3DInterpolatorNode):
    """
    ColorInterpolator generates a range of color values.
    """
    def NAME(self):
        return 'ColorInterpolator'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ColorInterpolator'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('key', list(), FieldType.MFFloat, AccessType.inputOutput, 'X3DInterpolatorNode'),
        ('keyValue', list(), FieldType.MFColor, AccessType.inputOutput, 'ColorInterpolator'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 key=list(),
                 keyValue=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode ColorInterpolator __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.key = key
        self.keyValue = keyValue
    @property # getter - - - - - - - - - -
    def key(self):
        """Definition values for linear-interpolation function input intervals, listed in non-decreasing order and corresponding to a value in the keyValue array."""
        return self.__key
    @key.setter
    def key(self, key):
        if  key is None:
            key = MFFloat.DEFAULT_VALUE(self)
        assertValidMFFloat(key)
        self.__key = key
    @property # getter - - - - - - - - - -
    def keyValue(self):
        """Output values for linear interpolation, each corresponding to an input-fraction value in the key array."""
        return self.__keyValue
    @keyValue.setter
    def keyValue(self, keyValue):
        if  keyValue is None:
            keyValue = MFColor.DEFAULT_VALUE(self)
        assertValidMFColor(keyValue)
        assertZeroToOne('keyValue', keyValue)
        self.__keyValue = keyValue
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ColorInterpolator.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ColorInterpolator'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.key != list():
            result += " key='" + MFFloat(self.key).XML() + "'"
        if self.keyValue != list():
            result += " keyValue='" + MFColor(self.keyValue).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></ColorInterpolator>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</ColorInterpolator>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function ColorInterpolator.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'ColorInterpolator' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'ColorInterpolator' + ' {'
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.key != list():
            result += " key " + MFFloat(self.key).VRML() + ""
        if self.keyValue != list():
            result += " keyValue " + MFColor(self.keyValue).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class ColorRGBA(_X3DColorNode):
    """
    ColorRGBA node defines a set of RGBA color values that apply either to a sibling Coordinate|CoordinateDouble node, or else to a parent ElevationGrid node.
    """
    def NAME(self):
        return 'ColorRGBA'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ColorRGBA'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('color', list(), FieldType.MFColorRGBA, AccessType.inputOutput, 'ColorRGBA'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 color=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode ColorRGBA __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.color = color
    @property # getter - - - - - - - - - -
    def color(self):
        """[0,1] The color field defines an array of 4-tuple RGBA colors."""
        return self.__color
    @color.setter
    def color(self, color):
        if  color is None:
            color = MFColorRGBA.DEFAULT_VALUE(self)
        assertValidMFColorRGBA(color)
        assertZeroToOne('color', color)
        self.__color = color
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ColorRGBA.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ColorRGBA'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.color != list():
            result += " color='" + MFColorRGBA(self.color).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></ColorRGBA>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</ColorRGBA>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function ColorRGBA.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'ColorRGBA' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'ColorRGBA' + ' {'
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.color != list():
            result += " color " + MFColorRGBA(self.color).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class ComposedCubeMapTexture(_X3DEnvironmentTextureNode):
    """
    ComposedCubeMapTexture is a texture node that defines a cubic environment map source as an explicit set of images drawn from individual 2D texture nodes.
    """
    def NAME(self):
        return 'ComposedCubeMapTexture'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ComposedCubeMapTexture'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('back', None, FieldType.SFNode, AccessType.inputOutput, 'ComposedCubeMapTexture'),
        ('bottom', None, FieldType.SFNode, AccessType.inputOutput, 'ComposedCubeMapTexture'),
        ('front', None, FieldType.SFNode, AccessType.inputOutput, 'ComposedCubeMapTexture'),
        ('left', None, FieldType.SFNode, AccessType.inputOutput, 'ComposedCubeMapTexture'),
        ('right', None, FieldType.SFNode, AccessType.inputOutput, 'ComposedCubeMapTexture'),
        ('top', None, FieldType.SFNode, AccessType.inputOutput, 'ComposedCubeMapTexture'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 back=None,
                 bottom=None,
                 front=None,
                 left=None,
                 right=None,
                 top=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode ComposedCubeMapTexture __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.back = back
        self.bottom = bottom
        self.front = front
        self.left = left
        self.right = right
        self.top = top
    @property # getter - - - - - - - - - -
    def back(self):
        return self.__back
    @back.setter
    def back(self, back):
        if  back is None:
            back = None # default
        assertValidSFNode(back)
        if not isinstance(back, object):
            # print(flush=True)
            raise X3DTypeError(str(back) + ' does not have a valid node type object')
        self.__back = back
    @property # getter - - - - - - - - - -
    def bottom(self):
        return self.__bottom
    @bottom.setter
    def bottom(self, bottom):
        if  bottom is None:
            bottom = None # default
        assertValidSFNode(bottom)
        if not isinstance(bottom, object):
            # print(flush=True)
            raise X3DTypeError(str(bottom) + ' does not have a valid node type object')
        self.__bottom = bottom
    @property # getter - - - - - - - - - -
    def front(self):
        return self.__front
    @front.setter
    def front(self, front):
        if  front is None:
            front = None # default
        assertValidSFNode(front)
        if not isinstance(front, object):
            # print(flush=True)
            raise X3DTypeError(str(front) + ' does not have a valid node type object')
        self.__front = front
    @property # getter - - - - - - - - - -
    def left(self):
        return self.__left
    @left.setter
    def left(self, left):
        if  left is None:
            left = None # default
        assertValidSFNode(left)
        if not isinstance(left, object):
            # print(flush=True)
            raise X3DTypeError(str(left) + ' does not have a valid node type object')
        self.__left = left
    @property # getter - - - - - - - - - -
    def right(self):
        return self.__right
    @right.setter
    def right(self, right):
        if  right is None:
            right = None # default
        assertValidSFNode(right)
        if not isinstance(right, object):
            # print(flush=True)
            raise X3DTypeError(str(right) + ' does not have a valid node type object')
        self.__right = right
    @property # getter - - - - - - - - - -
    def top(self):
        return self.__top
    @top.setter
    def top(self, top):
        if  top is None:
            top = None # default
        assertValidSFNode(top)
        if not isinstance(top, object):
            # print(flush=True)
            raise X3DTypeError(str(top) + ' does not have a valid node type object')
        self.__top = top
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.back) or (self.bottom) or (self.front) or (self.IS) or (self.left) or (self.metadata) or (self.right) or (self.top)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ComposedCubeMapTexture.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ComposedCubeMapTexture'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></ComposedCubeMapTexture>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.back: # output this SFNode
                result += self.back.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.bottom: # output this SFNode
                result += self.bottom.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.front: # output this SFNode
                result += self.front.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.left: # output this SFNode
                result += self.left.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.right: # output this SFNode
                result += self.right.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.top: # output this SFNode
                result += self.top.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</ComposedCubeMapTexture>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function ComposedCubeMapTexture.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'ComposedCubeMapTexture' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'ComposedCubeMapTexture' + ' {'
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.back: # output this SFNode
            result += '\n' + '  ' + indent + 'back ' + self.back.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.bottom: # output this SFNode
            result += '\n' + '  ' + indent + 'bottom ' + self.bottom.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.front: # output this SFNode
            result += '\n' + '  ' + indent + 'front ' + self.front.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.left: # output this SFNode
            result += '\n' + '  ' + indent + 'left ' + self.left.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.right: # output this SFNode
            result += '\n' + '  ' + indent + 'right ' + self.right.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.top: # output this SFNode
            result += '\n' + '  ' + indent + 'top ' + self.top.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class ComposedShader(_X3DShaderNode): # , _X3DProgrammableShaderObject # TODO fix additional inheritance method resolution order (MRO)
    """
    ComposedShader can contain field declarations, but no CDATA section of plain-text source code, since programs are composed from child ShaderPart nodes.
    """
    def NAME(self):
        return 'ComposedShader'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ComposedShader'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('language', '', FieldType.SFString, AccessType.initializeOnly, 'X3DShaderNode'),
        ('field', list(), FieldType.MFNode, AccessType.inputOutput, 'ComposedShader'),
        ('parts', list(), FieldType.MFNode, AccessType.inputOutput, 'ComposedShader'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'ComposedShader'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'ComposedShader')]
    def __init__(self,
                 language='',
                 field=None,
                 parts=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode ComposedShader __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.language = language
        self.field = field
        self.parts = parts
    @property # getter - - - - - - - - - -
    def language(self):
        """The language field indicates to the X3D player which shading language is used."""
        return self.__language
    @language.setter
    def language(self, language):
        if  language is None:
            language = SFString.DEFAULT_VALUE(self)
        assertValidSFString(language)
        self.__language = language
    @property # getter - - - - - - - - - -
    def field(self):
        """Include a field statement for each field declaration in the ComposedShader node."""
        return self.__field
    @field.setter
    def field(self, field):
        if  field is None:
            field = MFNode.DEFAULT_VALUE(self)
        # TODO type-aware checks for field
        if field: # walk each child in MFNode list, if any
            for each in __field:
                assertValidFieldInitializationValue(each.name, each.type, each.value, parent='ComposedShader')
        self.__field = field
    @property # getter - - - - - - - - - -
    def parts(self):
        """[ShaderPart] ComposedShader can contain multiple ShaderPart nodes in the parts field."""
        return self.__parts
    @parts.setter
    def parts(self, parts):
        if  parts is None:
            parts = MFNode.DEFAULT_VALUE(self)
        assertValidMFNode(parts)
        self.__parts = parts
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.field) or (self.parts) or (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ComposedShader.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ComposedShader'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.language:
            result += " language='" + self.language + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></ComposedShader>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            ### if self.field: # walk each child in MFNode list, if any
            ### print('* ComposedShader found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(field)=' + str(len(self.field)) + ', now invoking XML(' + str(indentLevel+1) + ')', flush=True)
            for each in self.field:
               result += each.XML(indentLevel=indentLevel+1,syntax=syntax)
            ### if self.parts: # walk each child in MFNode list, if any
            ### print('* ComposedShader found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(parts)=' + str(len(self.parts)) + ', now invoking XML(' + str(indentLevel+1) + ')', flush=True)
            for each in self.parts:
               result += each.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</ComposedShader>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function ComposedShader.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'ComposedShader' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'ComposedShader' + ' {'
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.language:
            result += " language " +  '"' + self.language + '"' + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.field: # walk each child in MFNode list, if any
            for each in self.field:
                result += each.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.parts: # walk each child in MFNode list, if any
            for each in self.parts:
                result += each.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class ComposedTexture3D(_X3DTexture3DNode):
    """
    ComposedTexture3D defines a 3D image-based texture map as a collection of 2D texture sources at various depths.
    """
    def NAME(self):
        return 'ComposedTexture3D'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ComposedTexture3D'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('repeatR', False, FieldType.SFBool, AccessType.initializeOnly, 'X3DTexture3DNode'),
        ('repeatS', False, FieldType.SFBool, AccessType.initializeOnly, 'X3DTexture3DNode'),
        ('repeatT', False, FieldType.SFBool, AccessType.initializeOnly, 'X3DTexture3DNode'),
        ('textureProperties', None, FieldType.SFNode, AccessType.initializeOnly, 'X3DTexture3DNode'),
        ('texture', list(), FieldType.MFNode, AccessType.inputOutput, 'ComposedTexture3D'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 repeatR=False,
                 repeatS=False,
                 repeatT=False,
                 textureProperties=None,
                 texture=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode ComposedTexture3D __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.repeatR = repeatR
        self.repeatS = repeatS
        self.repeatT = repeatT
        self.textureProperties = textureProperties
        self.texture = texture
    @property # getter - - - - - - - - - -
    def repeatR(self):
        """Whether to vertically repeat texture along R axis."""
        return self.__repeatR
    @repeatR.setter
    def repeatR(self, repeatR):
        if  repeatR is None:
            repeatR = False # default
        assertValidSFBool(repeatR)
        self.__repeatR = repeatR
    @property # getter - - - - - - - - - -
    def repeatS(self):
        """Whether to horizontally repeat texture along S axis."""
        return self.__repeatS
    @repeatS.setter
    def repeatS(self, repeatS):
        if  repeatS is None:
            repeatS = False # default
        assertValidSFBool(repeatS)
        self.__repeatS = repeatS
    @property # getter - - - - - - - - - -
    def repeatT(self):
        """Whether to vertically repeat texture along T axis."""
        return self.__repeatT
    @repeatT.setter
    def repeatT(self, repeatT):
        if  repeatT is None:
            repeatT = False # default
        assertValidSFBool(repeatT)
        self.__repeatT = repeatT
    @property # getter - - - - - - - - - -
    def textureProperties(self):
        return self.__textureProperties
    @textureProperties.setter
    def textureProperties(self, textureProperties):
        if  textureProperties is None:
            textureProperties = None # default
        assertValidSFNode(textureProperties)
        if not isinstance(textureProperties, object):
            # print(flush=True)
            raise X3DTypeError(str(textureProperties) + ' does not have a valid node type object')
        self.__textureProperties = textureProperties
    @property # getter - - - - - - - - - -
    def texture(self):
        return self.__texture
    @texture.setter
    def texture(self, texture):
        if  texture is None:
            texture = MFNode.DEFAULT_VALUE(self)
        assertValidMFNode(texture)
        self.__texture = texture
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.texture) or (self.IS) or (self.metadata) or (self.textureProperties)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ComposedTexture3D.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ComposedTexture3D'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.repeatR != False:
            result += " repeatR='" + SFBool(self.repeatR).XML() + "'"
        if self.repeatS != False:
            result += " repeatS='" + SFBool(self.repeatS).XML() + "'"
        if self.repeatT != False:
            result += " repeatT='" + SFBool(self.repeatT).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></ComposedTexture3D>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.textureProperties: # output this SFNode
                result += self.textureProperties.XML(indentLevel=indentLevel+1,syntax=syntax)
            ### if self.texture: # walk each child in MFNode list, if any
            ### print('* ComposedTexture3D found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(texture)=' + str(len(self.texture)) + ', now invoking XML(' + str(indentLevel+1) + ')', flush=True)
            for each in self.texture:
               result += each.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</ComposedTexture3D>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function ComposedTexture3D.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'ComposedTexture3D' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'ComposedTexture3D' + ' {'
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.repeatR != False:
            result += " repeatR " + SFBool(self.repeatR).VRML() + ""
        if self.repeatS != False:
            result += " repeatS " + SFBool(self.repeatS).VRML() + ""
        if self.repeatT != False:
            result += " repeatT " + SFBool(self.repeatT).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.textureProperties: # output this SFNode
            result += '\n' + '  ' + indent + 'textureProperties ' + self.textureProperties.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.texture: # walk each child in MFNode list, if any
            for each in self.texture:
                result += each.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class ComposedVolumeStyle(_X3DComposableVolumeRenderStyleNode):
    """
    ComposedVolumeStyle allows compositing multiple rendering styles into single rendering pass.
    """
    def NAME(self):
        return 'ComposedVolumeStyle'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ComposedVolumeStyle'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('enabled', True, FieldType.SFBool, AccessType.inputOutput, 'X3DVolumeRenderStyleNode'),
        ('renderStyle', list(), FieldType.MFNode, AccessType.inputOutput, 'ComposedVolumeStyle'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 enabled=True,
                 renderStyle=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode ComposedVolumeStyle __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.enabled = enabled
        self.renderStyle = renderStyle
    @property # getter - - - - - - - - - -
    def enabled(self):
        """Enables/disables node operation."""
        return self.__enabled
    @enabled.setter
    def enabled(self, enabled):
        if  enabled is None:
            enabled = True # default
        assertValidSFBool(enabled)
        self.__enabled = enabled
    @property # getter - - - - - - - - - -
    def renderStyle(self):
        """[X3DComposableVolumeRenderStyleNode] List of contributing rendering style nodes or node references that can be applied to the object."""
        return self.__renderStyle
    @renderStyle.setter
    def renderStyle(self, renderStyle):
        if  renderStyle is None:
            renderStyle = MFNode.DEFAULT_VALUE(self)
        assertValidMFNode(renderStyle)
        self.__renderStyle = renderStyle
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.renderStyle) or (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ComposedVolumeStyle.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ComposedVolumeStyle'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.enabled != True:
            result += " enabled='" + SFBool(self.enabled).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></ComposedVolumeStyle>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            ### if self.renderStyle: # walk each child in MFNode list, if any
            ### print('* ComposedVolumeStyle found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(renderStyle)=' + str(len(self.renderStyle)) + ', now invoking XML(' + str(indentLevel+1) + ')', flush=True)
            for each in self.renderStyle:
               result += each.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</ComposedVolumeStyle>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function ComposedVolumeStyle.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'ComposedVolumeStyle' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'ComposedVolumeStyle' + ' {'
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.enabled != True:
            result += " enabled " + SFBool(self.enabled).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.renderStyle: # walk each child in MFNode list, if any
            for each in self.renderStyle:
                result += each.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class Cone(_X3DGeometryNode):
    """
    Cone is a geometry node.
    """
    def NAME(self):
        return 'Cone'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#Cone'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('bottom', True, FieldType.SFBool, AccessType.initializeOnly, 'Cone'),
        ('bottomRadius', 1, FieldType.SFFloat, AccessType.initializeOnly, 'Cone'),
        ('height', 2, FieldType.SFFloat, AccessType.initializeOnly, 'Cone'),
        ('side', True, FieldType.SFBool, AccessType.initializeOnly, 'Cone'),
        ('solid', True, FieldType.SFBool, AccessType.initializeOnly, 'Cone'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 bottom=True,
                 bottomRadius=1,
                 height=2,
                 side=True,
                 solid=True,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode Cone __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.bottom = bottom
        self.bottomRadius = bottomRadius
        self.height = height
        self.side = side
        self.solid = solid
    @property # getter - - - - - - - - - -
    def bottom(self):
        """Whether to draw bottom (other inside faces are not drawn)."""
        return self.__bottom
    @bottom.setter
    def bottom(self, bottom):
        if  bottom is None:
            bottom = True # default
        assertValidSFBool(bottom)
        self.__bottom = bottom
    @property # getter - - - - - - - - - -
    def bottomRadius(self):
        """(0,+infinity) Size in meters."""
        return self.__bottomRadius
    @bottomRadius.setter
    def bottomRadius(self, bottomRadius):
        if  bottomRadius is None:
            bottomRadius = 1 # default
        assertValidSFFloat(bottomRadius)
        assertPositive('bottomRadius', bottomRadius)
        self.__bottomRadius = bottomRadius
    @property # getter - - - - - - - - - -
    def height(self):
        """(0,+infinity) Size in meters."""
        return self.__height
    @height.setter
    def height(self, height):
        if  height is None:
            height = 2 # default
        assertValidSFFloat(height)
        assertPositive('height', height)
        self.__height = height
    @property # getter - - - - - - - - - -
    def side(self):
        """Whether to draw sides (other inside faces are not drawn)."""
        return self.__side
    @side.setter
    def side(self, side):
        if  side is None:
            side = True # default
        assertValidSFBool(side)
        self.__side = side
    @property # getter - - - - - - - - - -
    def solid(self):
        """Setting solid true means draw only one side of polygons (backface culling on), setting solid false means draw both sides of polygons (backface culling off)."""
        return self.__solid
    @solid.setter
    def solid(self, solid):
        if  solid is None:
            solid = True # default
        assertValidSFBool(solid)
        self.__solid = solid
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Cone.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<Cone'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.bottom != True:
            result += " bottom='" + SFBool(self.bottom).XML() + "'"
        if self.bottomRadius != 1:
            result += " bottomRadius='" + SFFloat(self.bottomRadius).XML() + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.height != 2:
            result += " height='" + SFFloat(self.height).XML() + "'"
        if self.side != True:
            result += " side='" + SFBool(self.side).XML() + "'"
        if self.solid != True:
            result += " solid='" + SFBool(self.solid).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></Cone>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</Cone>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function Cone.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'Cone' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'Cone' + ' {'
        if self.bottom != True:
            result += " bottom " + SFBool(self.bottom).VRML() + ""
        if self.bottomRadius != 1:
            result += " bottomRadius " + SFFloat(self.bottomRadius).VRML() + ""
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.height != 2:
            result += " height " + SFFloat(self.height).VRML() + ""
        if self.side != True:
            result += " side " + SFBool(self.side).VRML() + ""
        if self.solid != True:
            result += " solid " + SFBool(self.solid).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class ConeEmitter(_X3DParticleEmitterNode):
    """
    ConeEmitter generates all available particles from a specific point in space.
    """
    def NAME(self):
        return 'ConeEmitter'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ConeEmitter'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('angle', 0.7854, FieldType.SFFloat, AccessType.inputOutput, 'ConeEmitter'),
        ('direction', (0, 1, 0), FieldType.SFVec3f, AccessType.inputOutput, 'ConeEmitter'),
        ('mass', 0, FieldType.SFFloat, AccessType.inputOutput, 'X3DParticleEmitterNode'),
        ('position', (0, 0, 0), FieldType.SFVec3f, AccessType.inputOutput, 'ConeEmitter'),
        ('speed', 0, FieldType.SFFloat, AccessType.inputOutput, 'X3DParticleEmitterNode'),
        ('surfaceArea', 0, FieldType.SFFloat, AccessType.initializeOnly, 'X3DParticleEmitterNode'),
        ('variation', 0.25, FieldType.SFFloat, AccessType.inputOutput, 'X3DParticleEmitterNode'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 angle=0.7854,
                 direction=(0, 1, 0),
                 mass=0,
                 position=(0, 0, 0),
                 speed=0,
                 surfaceArea=0,
                 variation=0.25,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode ConeEmitter __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.angle = angle
        self.direction = direction
        self.mass = mass
        self.position = position
        self.speed = speed
        self.surfaceArea = surfaceArea
        self.variation = variation
    @property # getter - - - - - - - - - -
    def angle(self):
        """[0,+infinity) Cone boundary for random distribution of particles about initial direction."""
        return self.__angle
    @angle.setter
    def angle(self, angle):
        if  angle is None:
            angle = 0.7854 # default
        assertValidSFFloat(angle)
        assertGreaterThanEquals('angle', angle, 0)
        assertLessThanEquals('angle', angle, 3.1416)
        self.__angle = angle
    @property # getter - - - - - - - - - -
    def direction(self):
        """Initial direction from which particles emanate."""
        return self.__direction
    @direction.setter
    def direction(self, direction):
        if  direction is None:
            direction = (0, 1, 0) # default
        assertValidSFVec3f(direction)
        assertGreaterThanEquals('direction', direction, -1)
        assertLessThanEquals('direction', direction, 1)
        self.__direction = direction
    @property # getter - - - - - - - - - -
    def mass(self):
        return self.__mass
    @mass.setter
    def mass(self, mass):
        if  mass is None:
            mass = 0 # default
        assertValidSFFloat(mass)
        assertNonNegative('mass', mass)
        self.__mass = mass
    @property # getter - - - - - - - - - -
    def position(self):
        """Point from which particles emanate."""
        return self.__position
    @position.setter
    def position(self, position):
        if  position is None:
            position = (0, 0, 0) # default
        assertValidSFVec3f(position)
        self.__position = position
    @property # getter - - - - - - - - - -
    def speed(self):
        """[0,+infinity) Initial linear speed (default is m/s) imparted to all particles along their direction of movement."""
        return self.__speed
    @speed.setter
    def speed(self, speed):
        if  speed is None:
            speed = 0 # default
        assertValidSFFloat(speed)
        assertNonNegative('speed', speed)
        self.__speed = speed
    @property # getter - - - - - - - - - -
    def surfaceArea(self):
        """[0,+infinity) Particle surface area in area base units (default is meters squared)."""
        return self.__surfaceArea
    @surfaceArea.setter
    def surfaceArea(self, surfaceArea):
        if  surfaceArea is None:
            surfaceArea = 0 # default
        assertValidSFFloat(surfaceArea)
        assertNonNegative('surfaceArea', surfaceArea)
        self.__surfaceArea = surfaceArea
    @property # getter - - - - - - - - - -
    def variation(self):
        """[0,+infinity) Multiplier for the randomness used to control the range of possible output values."""
        return self.__variation
    @variation.setter
    def variation(self, variation):
        if  variation is None:
            variation = 0.25 # default
        assertValidSFFloat(variation)
        assertNonNegative('variation', variation)
        self.__variation = variation
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ConeEmitter.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ConeEmitter'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.angle != 0.7854:
            result += " angle='" + SFFloat(self.angle).XML() + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.direction != (0, 1, 0):
            result += " direction='" + SFVec3f(self.direction).XML() + "'"
        if self.mass != 0:
            result += " mass='" + SFFloat(self.mass).XML() + "'"
        if self.position != (0, 0, 0):
            result += " position='" + SFVec3f(self.position).XML() + "'"
        if self.speed != 0:
            result += " speed='" + SFFloat(self.speed).XML() + "'"
        if self.surfaceArea != 0:
            result += " surfaceArea='" + SFFloat(self.surfaceArea).XML() + "'"
        if self.variation != 0.25:
            result += " variation='" + SFFloat(self.variation).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></ConeEmitter>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</ConeEmitter>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function ConeEmitter.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'ConeEmitter' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'ConeEmitter' + ' {'
        if self.angle != 0.7854:
            result += " angle " + SFFloat(self.angle).VRML() + ""
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.direction != (0, 1, 0):
            result += " direction " + SFVec3f(self.direction).VRML() + ""
        if self.mass != 0:
            result += " mass " + SFFloat(self.mass).VRML() + ""
        if self.position != (0, 0, 0):
            result += " position " + SFVec3f(self.position).VRML() + ""
        if self.speed != 0:
            result += " speed " + SFFloat(self.speed).VRML() + ""
        if self.surfaceArea != 0:
            result += " surfaceArea " + SFFloat(self.surfaceArea).VRML() + ""
        if self.variation != 0.25:
            result += " variation " + SFFloat(self.variation).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class Contact(_X3DNode):
    """
    Contact nodes are produced as output events when two collidable objects or spaces make contact.
    """
    def NAME(self):
        return 'Contact'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#Contact'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('appliedParameters', ["BOUNCE"], FieldType.MFString, AccessType.inputOutput, 'Contact'),
        ('bounce', 0, FieldType.SFFloat, AccessType.inputOutput, 'Contact'),
        ('contactNormal', (0, 1, 0), FieldType.SFVec3f, AccessType.inputOutput, 'Contact'),
        ('depth', 0, FieldType.SFFloat, AccessType.inputOutput, 'Contact'),
        ('frictionCoefficients', (0, 0), FieldType.SFVec2f, AccessType.inputOutput, 'Contact'),
        ('frictionDirection', (0, 1, 0), FieldType.SFVec3f, AccessType.inputOutput, 'Contact'),
        ('minBounceSpeed', 0, FieldType.SFFloat, AccessType.inputOutput, 'Contact'),
        ('position', (0, 0, 0), FieldType.SFVec3f, AccessType.inputOutput, 'Contact'),
        ('slipCoefficients', (0, 0), FieldType.SFVec2f, AccessType.inputOutput, 'Contact'),
        ('softnessConstantForceMix', 0.0001, FieldType.SFFloat, AccessType.inputOutput, 'Contact'),
        ('softnessErrorCorrection', 0.8, FieldType.SFFloat, AccessType.inputOutput, 'Contact'),
        ('surfaceSpeed', (0, 0), FieldType.SFVec2f, AccessType.inputOutput, 'Contact'),
        ('body1', None, FieldType.SFNode, AccessType.inputOutput, 'Contact'),
        ('body2', None, FieldType.SFNode, AccessType.inputOutput, 'Contact'),
        ('geometry1', None, FieldType.SFNode, AccessType.inputOutput, 'Contact'),
        ('geometry2', None, FieldType.SFNode, AccessType.inputOutput, 'Contact'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 appliedParameters=["BOUNCE"],
                 bounce=0,
                 contactNormal=(0, 1, 0),
                 depth=0,
                 frictionCoefficients=(0, 0),
                 frictionDirection=(0, 1, 0),
                 minBounceSpeed=0,
                 position=(0, 0, 0),
                 slipCoefficients=(0, 0),
                 softnessConstantForceMix=0.0001,
                 softnessErrorCorrection=0.8,
                 surfaceSpeed=(0, 0),
                 body1=None,
                 body2=None,
                 geometry1=None,
                 geometry2=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode Contact __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.appliedParameters = appliedParameters
        self.bounce = bounce
        self.contactNormal = contactNormal
        self.depth = depth
        self.frictionCoefficients = frictionCoefficients
        self.frictionDirection = frictionDirection
        self.minBounceSpeed = minBounceSpeed
        self.position = position
        self.slipCoefficients = slipCoefficients
        self.softnessConstantForceMix = softnessConstantForceMix
        self.softnessErrorCorrection = softnessErrorCorrection
        self.surfaceSpeed = surfaceSpeed
        self.body1 = body1
        self.body2 = body2
        self.geometry1 = geometry1
        self.geometry2 = geometry2
    @property # getter - - - - - - - - - -
    def appliedParameters(self):
        """Default global parameters for collision outputs of rigid body physics system."""
        return self.__appliedParameters
    @appliedParameters.setter
    def appliedParameters(self, appliedParameters):
        if  appliedParameters is None:
            appliedParameters = ["BOUNCE"] # default
        assertValidMFString(appliedParameters)
        assertValidAppliedParameters('appliedParameters', appliedParameters)
        self.__appliedParameters = appliedParameters
    @property # getter - - - - - - - - - -
    def bounce(self):
        """[0,1] bounce indicates bounciness (0 = no bounce at all, 1 = maximum bounce)."""
        return self.__bounce
    @bounce.setter
    def bounce(self, bounce):
        if  bounce is None:
            bounce = 0 # default
        assertValidSFFloat(bounce)
        assertZeroToOne('bounce', bounce)
        self.__bounce = bounce
    @property # getter - - - - - - - - - -
    def contactNormal(self):
        """contactNormal is unit vector describing normal between two colliding bodies."""
        return self.__contactNormal
    @contactNormal.setter
    def contactNormal(self, contactNormal):
        if  contactNormal is None:
            contactNormal = (0, 1, 0) # default
        assertValidSFVec3f(contactNormal)
        self.__contactNormal = contactNormal
    @property # getter - - - - - - - - - -
    def depth(self):
        """[0,1] depth indicates how deep the current intersection is along normal vector."""
        return self.__depth
    @depth.setter
    def depth(self, depth):
        if  depth is None:
            depth = 0 # default
        assertValidSFFloat(depth)
        self.__depth = depth
    @property # getter - - - - - - - - - -
    def frictionCoefficients(self):
        """frictionCoefficients used for computing surface drag."""
        return self.__frictionCoefficients
    @frictionCoefficients.setter
    def frictionCoefficients(self, frictionCoefficients):
        if  frictionCoefficients is None:
            frictionCoefficients = (0, 0) # default
        assertValidSFVec2f(frictionCoefficients)
        assertNonNegative('frictionCoefficients', frictionCoefficients)
        self.__frictionCoefficients = frictionCoefficients
    @property # getter - - - - - - - - - -
    def frictionDirection(self):
        """frictionDirection controls friction vector."""
        return self.__frictionDirection
    @frictionDirection.setter
    def frictionDirection(self, frictionDirection):
        if  frictionDirection is None:
            frictionDirection = (0, 1, 0) # default
        assertValidSFVec3f(frictionDirection)
        self.__frictionDirection = frictionDirection
    @property # getter - - - - - - - - - -
    def minBounceSpeed(self):
        """[0,+infinity) minBounceSpeed m/s needed to bounce."""
        return self.__minBounceSpeed
    @minBounceSpeed.setter
    def minBounceSpeed(self, minBounceSpeed):
        if  minBounceSpeed is None:
            minBounceSpeed = 0 # default
        assertValidSFFloat(minBounceSpeed)
        assertNonNegative('minBounceSpeed', minBounceSpeed)
        self.__minBounceSpeed = minBounceSpeed
    @property # getter - - - - - - - - - -
    def position(self):
        """position (x, y, z in meters) of exact location of collision."""
        return self.__position
    @position.setter
    def position(self, position):
        if  position is None:
            position = (0, 0, 0) # default
        assertValidSFVec3f(position)
        self.__position = position
    @property # getter - - - - - - - - - -
    def slipCoefficients(self):
        """slipCoefficients used for computing surface drag."""
        return self.__slipCoefficients
    @slipCoefficients.setter
    def slipCoefficients(self, slipCoefficients):
        if  slipCoefficients is None:
            slipCoefficients = (0, 0) # default
        assertValidSFVec2f(slipCoefficients)
        self.__slipCoefficients = slipCoefficients
    @property # getter - - - - - - - - - -
    def softnessConstantForceMix(self):
        """[0,1] softnessConstantForceMix value applies a constant force value to make colliding surfaces appear to be somewhat soft."""
        return self.__softnessConstantForceMix
    @softnessConstantForceMix.setter
    def softnessConstantForceMix(self, softnessConstantForceMix):
        if  softnessConstantForceMix is None:
            softnessConstantForceMix = 0.0001 # default
        assertValidSFFloat(softnessConstantForceMix)
        assertZeroToOne('softnessConstantForceMix', softnessConstantForceMix)
        self.__softnessConstantForceMix = softnessConstantForceMix
    @property # getter - - - - - - - - - -
    def softnessErrorCorrection(self):
        """[0,1] softnessErrorCorrection indicates fraction of collision error fixed in a set of evaluations (0 = no error correction, 1 = all errors corrected in single step)."""
        return self.__softnessErrorCorrection
    @softnessErrorCorrection.setter
    def softnessErrorCorrection(self, softnessErrorCorrection):
        if  softnessErrorCorrection is None:
            softnessErrorCorrection = 0.8 # default
        assertValidSFFloat(softnessErrorCorrection)
        assertZeroToOne('softnessErrorCorrection', softnessErrorCorrection)
        self.__softnessErrorCorrection = softnessErrorCorrection
    @property # getter - - - - - - - - - -
    def surfaceSpeed(self):
        """surfaceSpeed defines speed vectors for computing surface drag, if contact surfaces move independently of bodies."""
        return self.__surfaceSpeed
    @surfaceSpeed.setter
    def surfaceSpeed(self, surfaceSpeed):
        if  surfaceSpeed is None:
            surfaceSpeed = (0, 0) # default
        assertValidSFVec2f(surfaceSpeed)
        self.__surfaceSpeed = surfaceSpeed
    @property # getter - - - - - - - - - -
    def body1(self):
        return self.__body1
    @body1.setter
    def body1(self, body1):
        if  body1 is None:
            body1 = None # default
        assertValidSFNode(body1)
        if not isinstance(body1, object):
            # print(flush=True)
            raise X3DTypeError(str(body1) + ' does not have a valid node type object')
        self.__body1 = body1
    @property # getter - - - - - - - - - -
    def body2(self):
        return self.__body2
    @body2.setter
    def body2(self, body2):
        if  body2 is None:
            body2 = None # default
        assertValidSFNode(body2)
        if not isinstance(body2, object):
            # print(flush=True)
            raise X3DTypeError(str(body2) + ' does not have a valid node type object')
        self.__body2 = body2
    @property # getter - - - - - - - - - -
    def geometry1(self):
        return self.__geometry1
    @geometry1.setter
    def geometry1(self, geometry1):
        if  geometry1 is None:
            geometry1 = None # default
        assertValidSFNode(geometry1)
        if not isinstance(geometry1, object):
            # print(flush=True)
            raise X3DTypeError(str(geometry1) + ' does not have a valid node type object')
        self.__geometry1 = geometry1
    @property # getter - - - - - - - - - -
    def geometry2(self):
        return self.__geometry2
    @geometry2.setter
    def geometry2(self, geometry2):
        if  geometry2 is None:
            geometry2 = None # default
        assertValidSFNode(geometry2)
        if not isinstance(geometry2, object):
            # print(flush=True)
            raise X3DTypeError(str(geometry2) + ' does not have a valid node type object')
        self.__geometry2 = geometry2
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.body1) or (self.body2) or (self.geometry1) or (self.geometry2) or (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Contact.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<Contact'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.appliedParameters != ["BOUNCE"]:
            result += " appliedParameters='" + MFString(self.appliedParameters).XML() + "'"
        if self.bounce != 0:
            result += " bounce='" + SFFloat(self.bounce).XML() + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.contactNormal != (0, 1, 0):
            result += " contactNormal='" + SFVec3f(self.contactNormal).XML() + "'"
        if self.depth != 0:
            result += " depth='" + SFFloat(self.depth).XML() + "'"
        if self.frictionCoefficients != (0, 0):
            result += " frictionCoefficients='" + SFVec2f(self.frictionCoefficients).XML() + "'"
        if self.frictionDirection != (0, 1, 0):
            result += " frictionDirection='" + SFVec3f(self.frictionDirection).XML() + "'"
        if self.minBounceSpeed != 0:
            result += " minBounceSpeed='" + SFFloat(self.minBounceSpeed).XML() + "'"
        if self.position != (0, 0, 0):
            result += " position='" + SFVec3f(self.position).XML() + "'"
        if self.slipCoefficients != (0, 0):
            result += " slipCoefficients='" + SFVec2f(self.slipCoefficients).XML() + "'"
        if self.softnessConstantForceMix != 0.0001:
            result += " softnessConstantForceMix='" + SFFloat(self.softnessConstantForceMix).XML() + "'"
        if self.softnessErrorCorrection != 0.8:
            result += " softnessErrorCorrection='" + SFFloat(self.softnessErrorCorrection).XML() + "'"
        if self.surfaceSpeed != (0, 0):
            result += " surfaceSpeed='" + SFVec2f(self.surfaceSpeed).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></Contact>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.body1: # output this SFNode
                result += self.body1.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.body2: # output this SFNode
                result += self.body2.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.geometry1: # output this SFNode
                result += self.geometry1.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.geometry2: # output this SFNode
                result += self.geometry2.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</Contact>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function Contact.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'Contact' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'Contact' + ' {'
        if self.appliedParameters != ["BOUNCE"]:
            result += " appliedParameters " + MFString(self.appliedParameters).VRML() + ""
        if self.bounce != 0:
            result += " bounce " + SFFloat(self.bounce).VRML() + ""
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.contactNormal != (0, 1, 0):
            result += " contactNormal " + SFVec3f(self.contactNormal).VRML() + ""
        if self.depth != 0:
            result += " depth " + SFFloat(self.depth).VRML() + ""
        if self.frictionCoefficients != (0, 0):
            result += " frictionCoefficients " + SFVec2f(self.frictionCoefficients).VRML() + ""
        if self.frictionDirection != (0, 1, 0):
            result += " frictionDirection " + SFVec3f(self.frictionDirection).VRML() + ""
        if self.minBounceSpeed != 0:
            result += " minBounceSpeed " + SFFloat(self.minBounceSpeed).VRML() + ""
        if self.position != (0, 0, 0):
            result += " position " + SFVec3f(self.position).VRML() + ""
        if self.slipCoefficients != (0, 0):
            result += " slipCoefficients " + SFVec2f(self.slipCoefficients).VRML() + ""
        if self.softnessConstantForceMix != 0.0001:
            result += " softnessConstantForceMix " + SFFloat(self.softnessConstantForceMix).VRML() + ""
        if self.softnessErrorCorrection != 0.8:
            result += " softnessErrorCorrection " + SFFloat(self.softnessErrorCorrection).VRML() + ""
        if self.surfaceSpeed != (0, 0):
            result += " surfaceSpeed " + SFVec2f(self.surfaceSpeed).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.body1: # output this SFNode
            result += '\n' + '  ' + indent + 'body1 ' + self.body1.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.body2: # output this SFNode
            result += '\n' + '  ' + indent + 'body2 ' + self.body2.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.geometry1: # output this SFNode
            result += '\n' + '  ' + indent + 'geometry1 ' + self.geometry1.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.geometry2: # output this SFNode
            result += '\n' + '  ' + indent + 'geometry2 ' + self.geometry2.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class Contour2D(_X3DNode):
    """
    Contour2D groups a set of curve segments into a composite contour.
    """
    def NAME(self):
        return 'Contour2D'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#Contour2D'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('children', list(), FieldType.MFNode, AccessType.inputOutput, 'Contour2D'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 children=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode Contour2D __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.children = children
    @property # getter - - - - - - - - - -
    def children(self):
        """[NurbsCurve2D|ContourPolyline2D] The children form a closed loop with first point of first child repeated as last point of last child, and the last point of a segment repeated as first point of the consecutive one."""
        return self.__children
    @children.setter
    def children(self, children):
        if  children is None:
            children = MFNode.DEFAULT_VALUE(self)
        assertValidMFNode(children)
        self.__children = children
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.children) or (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Contour2D.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<Contour2D'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></Contour2D>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            ### if self.children: # walk each child in MFNode list, if any
            ### print('* Contour2D found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(children)=' + str(len(self.children)) + ', now invoking XML(' + str(indentLevel+1) + ')', flush=True)
            for each in self.children:
               result += each.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</Contour2D>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function Contour2D.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'Contour2D' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'Contour2D' + ' {'
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.children: # walk each child in MFNode list, if any
            for each in self.children:
                result += each.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class ContourPolyline2D(_X3DNurbsControlCurveNode):
    """
    ContourPolyline2D defines a linear curve segment as part of a trimming contour in the u-v domain of a NURBS surface.
    """
    def NAME(self):
        return 'ContourPolyline2D'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ContourPolyline2D'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('controlPoint', list(), FieldType.MFVec2d, AccessType.inputOutput, 'X3DNurbsControlCurveNode'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 controlPoint=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode ContourPolyline2D __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.controlPoint = controlPoint
    @property # getter - - - - - - - - - -
    def controlPoint(self):
        """controlPoint specifies the end points of each segment of the piecewise linear curve."""
        return self.__controlPoint
    @controlPoint.setter
    def controlPoint(self, controlPoint):
        if  controlPoint is None:
            controlPoint = MFVec2d.DEFAULT_VALUE(self)
        assertValidMFVec2d(controlPoint)
        self.__controlPoint = controlPoint
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function ContourPolyline2D.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<ContourPolyline2D'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.controlPoint != list():
            result += " controlPoint='" + MFVec2d(self.controlPoint).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></ContourPolyline2D>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</ContourPolyline2D>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function ContourPolyline2D.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'ContourPolyline2D' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'ContourPolyline2D' + ' {'
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.controlPoint != list():
            result += " controlPoint " + MFVec2d(self.controlPoint).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class Coordinate(_X3DCoordinateNode):
    """
    Coordinate builds geometry by defining a set of 3D coordinate (triplet) point values.
    """
    def NAME(self):
        return 'Coordinate'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#Coordinate'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('point', list(), FieldType.MFVec3f, AccessType.inputOutput, 'Coordinate'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 point=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode Coordinate __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.point = point
    @property # getter - - - - - - - - - -
    def point(self):
        """point contains a set of 3D coordinate (triplet) point values."""
        return self.__point
    @point.setter
    def point(self, point):
        if  point is None:
            point = MFVec3f.DEFAULT_VALUE(self)
        assertValidMFVec3f(point)
        self.__point = point
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Coordinate.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<Coordinate'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.point != list():
            result += " point='" + MFVec3f(self.point).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></Coordinate>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</Coordinate>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function Coordinate.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'Coordinate' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'Coordinate' + ' {'
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.point != list():
            result += " point " + MFVec3f(self.point).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class CoordinateChaser(_X3DChaserNode):
    """
    CoordinateChaser generates a series of coordinate arrays that progressively change from initial value to destination value.
    """
    def NAME(self):
        return 'CoordinateChaser'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#CoordinateChaser'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('duration', 1, FieldType.SFTime, AccessType.initializeOnly, 'X3DChaserNode'),
        ('initialDestination', [(0, 0, 0)], FieldType.MFVec3f, AccessType.initializeOnly, 'CoordinateChaser'),
        ('initialValue', [(0, 0, 0)], FieldType.MFVec3f, AccessType.initializeOnly, 'CoordinateChaser'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 duration=1,
                 initialDestination=[(0, 0, 0)],
                 initialValue=[(0, 0, 0)],
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode CoordinateChaser __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.duration = duration
        self.initialDestination = initialDestination
        self.initialValue = initialValue
    @property # getter - - - - - - - - - -
    def duration(self):
        """[0,+infinity) duration is the time interval for filter response in seconds."""
        return self.__duration
    @duration.setter
    def duration(self, duration):
        if  duration is None:
            duration = 1 # default
        assertValidSFTime(duration)
        assertNonNegative('duration', duration)
        self.__duration = duration
    @property # getter - - - - - - - - - -
    def initialDestination(self):
        """Initial destination value for this node."""
        return self.__initialDestination
    @initialDestination.setter
    def initialDestination(self, initialDestination):
        if  initialDestination is None:
            initialDestination = [(0, 0, 0)] # default
        assertValidMFVec3f(initialDestination)
        self.__initialDestination = initialDestination
    @property # getter - - - - - - - - - -
    def initialValue(self):
        """Initial starting value for this node."""
        return self.__initialValue
    @initialValue.setter
    def initialValue(self, initialValue):
        if  initialValue is None:
            initialValue = [(0, 0, 0)] # default
        assertValidMFVec3f(initialValue)
        self.__initialValue = initialValue
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function CoordinateChaser.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<CoordinateChaser'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.duration != 1:
            result += " duration='" + SFTime(self.duration).XML() + "'"
        if self.initialDestination != [(0, 0, 0)]:
            result += " initialDestination='" + MFVec3f(self.initialDestination).XML() + "'"
        if self.initialValue != [(0, 0, 0)]:
            result += " initialValue='" + MFVec3f(self.initialValue).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></CoordinateChaser>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</CoordinateChaser>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function CoordinateChaser.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'CoordinateChaser' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'CoordinateChaser' + ' {'
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.duration != 1:
            result += " duration " + SFTime(self.duration).VRML() + ""
        if self.initialDestination != [(0, 0, 0)]:
            result += " initialDestination " + MFVec3f(self.initialDestination).VRML() + ""
        if self.initialValue != [(0, 0, 0)]:
            result += " initialValue " + MFVec3f(self.initialValue).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class CoordinateDamper(_X3DDamperNode):
    """
    CoordinateDamper generates a series of coordinate arrays that progressively change from initial value to destination value.
    """
    def NAME(self):
        return 'CoordinateDamper'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#CoordinateDamper'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('initialDestination', [(0, 0, 0)], FieldType.MFVec3f, AccessType.initializeOnly, 'CoordinateDamper'),
        ('initialValue', [(0, 0, 0)], FieldType.MFVec3f, AccessType.initializeOnly, 'CoordinateDamper'),
        ('order', 3, FieldType.SFInt32, AccessType.initializeOnly, 'X3DDamperNode'),
        ('tau', 0.3, FieldType.SFTime, AccessType.inputOutput, 'X3DDamperNode'),
        ('tolerance', -1, FieldType.SFFloat, AccessType.inputOutput, 'X3DDamperNode'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 initialDestination=[(0, 0, 0)],
                 initialValue=[(0, 0, 0)],
                 order=3,
                 tau=0.3,
                 tolerance=-1,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode CoordinateDamper __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.initialDestination = initialDestination
        self.initialValue = initialValue
        self.order = order
        self.tau = tau
        self.tolerance = tolerance
    @property # getter - - - - - - - - - -
    def initialDestination(self):
        """Initial destination value for this node."""
        return self.__initialDestination
    @initialDestination.setter
    def initialDestination(self, initialDestination):
        if  initialDestination is None:
            initialDestination = [(0, 0, 0)] # default
        assertValidMFVec3f(initialDestination)
        self.__initialDestination = initialDestination
    @property # getter - - - - - - - - - -
    def initialValue(self):
        """Initial starting value for this node."""
        return self.__initialValue
    @initialValue.setter
    def initialValue(self, initialValue):
        if  initialValue is None:
            initialValue = [(0, 0, 0)] # default
        assertValidMFVec3f(initialValue)
        self.__initialValue = initialValue
    @property # getter - - - - - - - - - -
    def order(self):
        """[0,5] order defines the number of internal filters (larger means smoother response, longer delay)."""
        return self.__order
    @order.setter
    def order(self, order):
        if  order is None:
            order = 3 # default
        assertValidSFInt32(order)
        assertGreaterThanEquals('order', order, 0)
        assertLessThanEquals('order', order, 5)
        self.__order = order
    @property # getter - - - - - - - - - -
    def tau(self):
        """[0,+infinity) tau is the exponential-decay time constant for filter response in seconds."""
        return self.__tau
    @tau.setter
    def tau(self, tau):
        if  tau is None:
            tau = 0.3 # default
        assertValidSFTime(tau)
        assertNonNegative('tau', tau)
        self.__tau = tau
    @property # getter - - - - - - - - - -
    def tolerance(self):
        """[0,+infinity) or -1."""
        return self.__tolerance
    @tolerance.setter
    def tolerance(self, tolerance):
        if  tolerance is None:
            tolerance = -1 # default
        assertValidSFFloat(tolerance)
        self.__tolerance = tolerance
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function CoordinateDamper.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<CoordinateDamper'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.initialDestination != [(0, 0, 0)]:
            result += " initialDestination='" + MFVec3f(self.initialDestination).XML() + "'"
        if self.initialValue != [(0, 0, 0)]:
            result += " initialValue='" + MFVec3f(self.initialValue).XML() + "'"
        if self.order != 3:
            result += " order='" + SFInt32(self.order).XML() + "'"
        if self.tau != 0.3:
            result += " tau='" + SFTime(self.tau).XML() + "'"
        if self.tolerance != -1:
            result += " tolerance='" + SFFloat(self.tolerance).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></CoordinateDamper>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</CoordinateDamper>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function CoordinateDamper.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'CoordinateDamper' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'CoordinateDamper' + ' {'
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.initialDestination != [(0, 0, 0)]:
            result += " initialDestination " + MFVec3f(self.initialDestination).VRML() + ""
        if self.initialValue != [(0, 0, 0)]:
            result += " initialValue " + MFVec3f(self.initialValue).VRML() + ""
        if self.order != 3:
            result += " order " + SFInt32(self.order).VRML() + ""
        if self.tau != 0.3:
            result += " tau " + SFTime(self.tau).VRML() + ""
        if self.tolerance != -1:
            result += " tolerance " + SFFloat(self.tolerance).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class CoordinateDouble(_X3DCoordinateNode):
    """
    CoordinateDouble builds geometry by defining a set of 3D coordinate (triplet) point values.
    """
    def NAME(self):
        return 'CoordinateDouble'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#CoordinateDouble'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('point', list(), FieldType.MFVec3d, AccessType.inputOutput, 'CoordinateDouble'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 point=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode CoordinateDouble __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.point = point
    @property # getter - - - - - - - - - -
    def point(self):
        """point contains a set of 3D coordinate (triplet) point values."""
        return self.__point
    @point.setter
    def point(self, point):
        if  point is None:
            point = MFVec3d.DEFAULT_VALUE(self)
        assertValidMFVec3d(point)
        self.__point = point
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function CoordinateDouble.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<CoordinateDouble'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.point != list():
            result += " point='" + MFVec3d(self.point).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></CoordinateDouble>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</CoordinateDouble>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function CoordinateDouble.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'CoordinateDouble' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'CoordinateDouble' + ' {'
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.point != list():
            result += " point " + MFVec3d(self.point).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class CoordinateInterpolator(_X3DInterpolatorNode):
    """
    CoordinateInterpolator linearly interpolates among a list of 3-tuple MFVec3f arrays, producing a single MFVec3f array that is fractional average between two nearest arrays in the list.
    """
    def NAME(self):
        return 'CoordinateInterpolator'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#CoordinateInterpolator'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('key', list(), FieldType.MFFloat, AccessType.inputOutput, 'X3DInterpolatorNode'),
        ('keyValue', list(), FieldType.MFVec3f, AccessType.inputOutput, 'CoordinateInterpolator'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 key=list(),
                 keyValue=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode CoordinateInterpolator __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.key = key
        self.keyValue = keyValue
    @property # getter - - - - - - - - - -
    def key(self):
        """Definition values for linear-interpolation function input intervals, listed in non-decreasing order and corresponding to a value in the keyValue array."""
        return self.__key
    @key.setter
    def key(self, key):
        if  key is None:
            key = MFFloat.DEFAULT_VALUE(self)
        assertValidMFFloat(key)
        self.__key = key
    @property # getter - - - - - - - - - -
    def keyValue(self):
        """Output values for linear interpolation, each corresponding to an input-fraction value in the key array."""
        return self.__keyValue
    @keyValue.setter
    def keyValue(self, keyValue):
        if  keyValue is None:
            keyValue = MFVec3f.DEFAULT_VALUE(self)
        assertValidMFVec3f(keyValue)
        self.__keyValue = keyValue
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function CoordinateInterpolator.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<CoordinateInterpolator'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.key != list():
            result += " key='" + MFFloat(self.key).XML() + "'"
        if self.keyValue != list():
            result += " keyValue='" + MFVec3f(self.keyValue).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></CoordinateInterpolator>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</CoordinateInterpolator>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function CoordinateInterpolator.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'CoordinateInterpolator' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'CoordinateInterpolator' + ' {'
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.key != list():
            result += " key " + MFFloat(self.key).VRML() + ""
        if self.keyValue != list():
            result += " keyValue " + MFVec3f(self.keyValue).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class CoordinateInterpolator2D(_X3DInterpolatorNode):
    """
    CoordinateInterpolator2D generates a series of SFVec2f or MFVec2f 2-tuple float values.
    """
    def NAME(self):
        return 'CoordinateInterpolator2D'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#CoordinateInterpolator2D'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('key', list(), FieldType.MFFloat, AccessType.inputOutput, 'X3DInterpolatorNode'),
        ('keyValue', list(), FieldType.MFVec2f, AccessType.inputOutput, 'CoordinateInterpolator2D'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 key=list(),
                 keyValue=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode CoordinateInterpolator2D __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.key = key
        self.keyValue = keyValue
    @property # getter - - - - - - - - - -
    def key(self):
        """Definition values for linear-interpolation function input intervals, listed in non-decreasing order and corresponding to a value in the keyValue array."""
        return self.__key
    @key.setter
    def key(self, key):
        if  key is None:
            key = MFFloat.DEFAULT_VALUE(self)
        assertValidMFFloat(key)
        self.__key = key
    @property # getter - - - - - - - - - -
    def keyValue(self):
        """Output values for linear interpolation, each corresponding to an input-fraction value in the key array."""
        return self.__keyValue
    @keyValue.setter
    def keyValue(self, keyValue):
        if  keyValue is None:
            keyValue = MFVec2f.DEFAULT_VALUE(self)
        assertValidMFVec2f(keyValue)
        self.__keyValue = keyValue
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function CoordinateInterpolator2D.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<CoordinateInterpolator2D'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.key != list():
            result += " key='" + MFFloat(self.key).XML() + "'"
        if self.keyValue != list():
            result += " keyValue='" + MFVec2f(self.keyValue).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></CoordinateInterpolator2D>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</CoordinateInterpolator2D>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function CoordinateInterpolator2D.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'CoordinateInterpolator2D' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'CoordinateInterpolator2D' + ' {'
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.key != list():
            result += " key " + MFFloat(self.key).VRML() + ""
        if self.keyValue != list():
            result += " keyValue " + MFVec2f(self.keyValue).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class Cylinder(_X3DGeometryNode):
    """
    Cylinder is a geometry node.
    """
    def NAME(self):
        return 'Cylinder'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#Cylinder'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('bottom', True, FieldType.SFBool, AccessType.initializeOnly, 'Cylinder'),
        ('height', 2, FieldType.SFFloat, AccessType.initializeOnly, 'Cylinder'),
        ('radius', 1, FieldType.SFFloat, AccessType.initializeOnly, 'Cylinder'),
        ('side', True, FieldType.SFBool, AccessType.initializeOnly, 'Cylinder'),
        ('solid', True, FieldType.SFBool, AccessType.initializeOnly, 'Cylinder'),
        ('top', True, FieldType.SFBool, AccessType.inputOutput, 'Cylinder'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 bottom=True,
                 height=2,
                 radius=1,
                 side=True,
                 solid=True,
                 top=True,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode Cylinder __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.bottom = bottom
        self.height = height
        self.radius = radius
        self.side = side
        self.solid = solid
        self.top = top
    @property # getter - - - - - - - - - -
    def bottom(self):
        """Whether to draw bottom (inside faces are never drawn)."""
        return self.__bottom
    @bottom.setter
    def bottom(self, bottom):
        if  bottom is None:
            bottom = True # default
        assertValidSFBool(bottom)
        self.__bottom = bottom
    @property # getter - - - - - - - - - -
    def height(self):
        """(0,+infinity) Size in meters."""
        return self.__height
    @height.setter
    def height(self, height):
        if  height is None:
            height = 2 # default
        assertValidSFFloat(height)
        assertPositive('height', height)
        self.__height = height
    @property # getter - - - - - - - - - -
    def radius(self):
        """(0,+infinity) Size in meters."""
        return self.__radius
    @radius.setter
    def radius(self, radius):
        if  radius is None:
            radius = 1 # default
        assertValidSFFloat(radius)
        assertPositive('radius', radius)
        self.__radius = radius
    @property # getter - - - - - - - - - -
    def side(self):
        """Whether to draw sides (inside faces are never drawn)."""
        return self.__side
    @side.setter
    def side(self, side):
        if  side is None:
            side = True # default
        assertValidSFBool(side)
        self.__side = side
    @property # getter - - - - - - - - - -
    def solid(self):
        """Setting solid true means draw only one side of polygons (backface culling on), setting solid false means draw both sides of polygons (backface culling off)."""
        return self.__solid
    @solid.setter
    def solid(self, solid):
        if  solid is None:
            solid = True # default
        assertValidSFBool(solid)
        self.__solid = solid
    @property # getter - - - - - - - - - -
    def top(self):
        """Whether to draw top (inside faces are never drawn)."""
        return self.__top
    @top.setter
    def top(self, top):
        if  top is None:
            top = True # default
        assertValidSFBool(top)
        self.__top = top
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Cylinder.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<Cylinder'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.bottom != True:
            result += " bottom='" + SFBool(self.bottom).XML() + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.height != 2:
            result += " height='" + SFFloat(self.height).XML() + "'"
        if self.radius != 1:
            result += " radius='" + SFFloat(self.radius).XML() + "'"
        if self.side != True:
            result += " side='" + SFBool(self.side).XML() + "'"
        if self.solid != True:
            result += " solid='" + SFBool(self.solid).XML() + "'"
        if self.top != True:
            result += " top='" + SFBool(self.top).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></Cylinder>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</Cylinder>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function Cylinder.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'Cylinder' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'Cylinder' + ' {'
        if self.bottom != True:
            result += " bottom " + SFBool(self.bottom).VRML() + ""
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.height != 2:
            result += " height " + SFFloat(self.height).VRML() + ""
        if self.radius != 1:
            result += " radius " + SFFloat(self.radius).VRML() + ""
        if self.side != True:
            result += " side " + SFBool(self.side).VRML() + ""
        if self.solid != True:
            result += " solid " + SFBool(self.solid).VRML() + ""
        if self.top != True:
            result += " top " + SFBool(self.top).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class CylinderSensor(_X3DDragSensorNode):
    """
    CylinderSensor converts pointer motion (for example, a mouse or wand) into rotation values using an invisible cylinder aligned with local Y-axis.
    """
    def NAME(self):
        return 'CylinderSensor'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#CylinderSensor'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('autoOffset', True, FieldType.SFBool, AccessType.inputOutput, 'X3DDragSensorNode'),
        ('axisRotation', (0, 1, 0, 0), FieldType.SFRotation, AccessType.inputOutput, 'CylinderSensor'),
        ('description', '', FieldType.SFString, AccessType.inputOutput, 'X3DPointingDeviceSensorNode'),
        ('diskAngle', 0.26179167, FieldType.SFFloat, AccessType.inputOutput, 'CylinderSensor'),
        ('enabled', True, FieldType.SFBool, AccessType.inputOutput, 'X3DSensorNode'),
        ('maxAngle', -1, FieldType.SFFloat, AccessType.inputOutput, 'CylinderSensor'),
        ('minAngle', 0, FieldType.SFFloat, AccessType.inputOutput, 'CylinderSensor'),
        ('offset', 0, FieldType.SFFloat, AccessType.inputOutput, 'CylinderSensor'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 autoOffset=True,
                 axisRotation=(0, 1, 0, 0),
                 description='',
                 diskAngle=0.26179167,
                 enabled=True,
                 maxAngle=-1,
                 minAngle=0,
                 offset=0,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode CylinderSensor __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.autoOffset = autoOffset
        self.axisRotation = axisRotation
        self.description = description
        self.diskAngle = diskAngle
        self.enabled = enabled
        self.maxAngle = maxAngle
        self.minAngle = minAngle
        self.offset = offset
    @property # getter - - - - - - - - - -
    def autoOffset(self):
        """determines whether previous offset values are remembered/accumulated."""
        return self.__autoOffset
    @autoOffset.setter
    def autoOffset(self, autoOffset):
        if  autoOffset is None:
            autoOffset = True # default
        assertValidSFBool(autoOffset)
        self.__autoOffset = autoOffset
    @property # getter - - - - - - - - - -
    def axisRotation(self):
        """axisRotation determines local sensor coordinate system by rotating the local coordinate system."""
        return self.__axisRotation
    @axisRotation.setter
    def axisRotation(self, axisRotation):
        if  axisRotation is None:
            axisRotation = (0, 1, 0, 0) # default
        assertValidSFRotation(axisRotation)
        self.__axisRotation = axisRotation
    @property # getter - - - - - - - - - -
    def description(self):
        """Author-provided text tooltip that tells users the expected action of this node."""
        return self.__description
    @description.setter
    def description(self, description):
        if  description is None:
            description = SFString.DEFAULT_VALUE(self)
        assertValidSFString(description)
        self.__description = description
    @property # getter - - - - - - - - - -
    def diskAngle(self):
        """Help decide rotation behavior from initial relative bearing of pointer drag: acute angle whether cylinder sides or end-cap disks of virtual-geometry sensor are used for manipulation."""
        return self.__diskAngle
    @diskAngle.setter
    def diskAngle(self, diskAngle):
        if  diskAngle is None:
            diskAngle = 0.26179167 # default
        assertValidSFFloat(diskAngle)
        assertGreaterThanEquals('diskAngle', diskAngle, 0)
        assertLessThanEquals('diskAngle', diskAngle, 1.5708)
        self.__diskAngle = diskAngle
    @property # getter - - - - - - - - - -
    def enabled(self):
        """Enables/disables node operation."""
        return self.__enabled
    @enabled.setter
    def enabled(self, enabled):
        if  enabled is None:
            enabled = True # default
        assertValidSFBool(enabled)
        self.__enabled = enabled
    @property # getter - - - - - - - - - -
    def maxAngle(self):
        """clamps rotation_changed events within range of min/max values Hint: if minAngle > maxAngle, rotation is not clamped."""
        return self.__maxAngle
    @maxAngle.setter
    def maxAngle(self, maxAngle):
        if  maxAngle is None:
            maxAngle = -1 # default
        assertValidSFFloat(maxAngle)
        assertGreaterThan('maxAngle', maxAngle, -6.2832)
        assertLessThan('maxAngle', maxAngle, 6.2832)
        self.__maxAngle = maxAngle
    @property # getter - - - - - - - - - -
    def minAngle(self):
        """clamps rotation_changed events within range of min/max values Hint: if minAngle > maxAngle, rotation is not clamped."""
        return self.__minAngle
    @minAngle.setter
    def minAngle(self, minAngle):
        if  minAngle is None:
            minAngle = 0 # default
        assertValidSFFloat(minAngle)
        assertGreaterThan('minAngle', minAngle, -6.2832)
        assertLessThan('minAngle', minAngle, 6.2832)
        self.__minAngle = minAngle
    @property # getter - - - - - - - - - -
    def offset(self):
        """Sends event and remembers last value sensed."""
        return self.__offset
    @offset.setter
    def offset(self, offset):
        if  offset is None:
            offset = 0 # default
        assertValidSFFloat(offset)
        self.__offset = offset
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function CylinderSensor.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<CylinderSensor'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.autoOffset != True:
            result += " autoOffset='" + SFBool(self.autoOffset).XML() + "'"
        if self.axisRotation != (0, 1, 0, 0):
            result += " axisRotation='" + SFRotation(self.axisRotation).XML() + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.description:
            result += " description='" + self.description + "'"
        if self.diskAngle != 0.26179167:
            result += " diskAngle='" + SFFloat(self.diskAngle).XML() + "'"
        if self.enabled != True:
            result += " enabled='" + SFBool(self.enabled).XML() + "'"
        if self.maxAngle != -1:
            result += " maxAngle='" + SFFloat(self.maxAngle).XML() + "'"
        if self.minAngle != 0:
            result += " minAngle='" + SFFloat(self.minAngle).XML() + "'"
        if self.offset != 0:
            result += " offset='" + SFFloat(self.offset).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></CylinderSensor>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</CylinderSensor>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function CylinderSensor.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'CylinderSensor' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'CylinderSensor' + ' {'
        if self.autoOffset != True:
            result += " autoOffset " + SFBool(self.autoOffset).VRML() + ""
        if self.axisRotation != (0, 1, 0, 0):
            result += " axisRotation " + SFRotation(self.axisRotation).VRML() + ""
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.description:
            result += " description " +  '"' + self.description + '"' + ""
        if self.diskAngle != 0.26179167:
            result += " diskAngle " + SFFloat(self.diskAngle).VRML() + ""
        if self.enabled != True:
            result += " enabled " + SFBool(self.enabled).VRML() + ""
        if self.maxAngle != -1:
            result += " maxAngle " + SFFloat(self.maxAngle).VRML() + ""
        if self.minAngle != 0:
            result += " minAngle " + SFFloat(self.minAngle).VRML() + ""
        if self.offset != 0:
            result += " offset " + SFFloat(self.offset).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class DirectionalLight(_X3DLightNode):
    """
    DirectionalLight might not be scoped by parent Group or Transform at levels 1 or 2.
    """
    def NAME(self):
        return 'DirectionalLight'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#DirectionalLight'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('ambientIntensity', 0, FieldType.SFFloat, AccessType.inputOutput, 'X3DLightNode'),
        ('color', (1, 1, 1), FieldType.SFColor, AccessType.inputOutput, 'X3DLightNode'),
        ('direction', (0, 0, -1), FieldType.SFVec3f, AccessType.inputOutput, 'DirectionalLight'),
        ('global_', False, FieldType.SFBool, AccessType.inputOutput, 'DirectionalLight'),
        ('intensity', 1, FieldType.SFFloat, AccessType.inputOutput, 'X3DLightNode'),
        ('on', True, FieldType.SFBool, AccessType.inputOutput, 'X3DLightNode'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 ambientIntensity=0,
                 color=(1, 1, 1),
                 direction=(0, 0, -1),
                 global_=False,
                 intensity=1,
                 on=True,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode DirectionalLight __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.ambientIntensity = ambientIntensity
        self.color = color
        self.direction = direction
        self.global_ = global_
        self.intensity = intensity
        self.on = on
    @property # getter - - - - - - - - - -
    def ambientIntensity(self):
        """[0,1] Brightness of ambient (nondirectional background) emission from the light."""
        return self.__ambientIntensity
    @ambientIntensity.setter
    def ambientIntensity(self, ambientIntensity):
        if  ambientIntensity is None:
            ambientIntensity = 0 # default
        assertValidSFFloat(ambientIntensity)
        assertZeroToOne('ambientIntensity', ambientIntensity)
        self.__ambientIntensity = ambientIntensity
    @property # getter - - - - - - - - - -
    def color(self):
        """[0,1] color of light, applied to colors of objects."""
        return self.__color
    @color.setter
    def color(self, color):
        if  color is None:
            color = (1, 1, 1) # default
        assertValidSFColor(color)
        assertZeroToOne('color', color)
        self.__color = color
    @property # getter - - - - - - - - - -
    def direction(self):
        """Orientation vector of light relative to local coordinate system."""
        return self.__direction
    @direction.setter
    def direction(self, direction):
        if  direction is None:
            direction = (0, 0, -1) # default
        assertValidSFVec3f(direction)
        self.__direction = direction
    @property # getter - - - - - - - - - -
    def global_(self):
        return self.__global_
    @global_.setter
    def global_(self, global_):
        if  global_ is None:
            global_ = False # default
        assertValidSFBool(global_)
        self.__global_ = global_
    @property # getter - - - - - - - - - -
    def intensity(self):
        """[0,1] Brightness of direct emission from the light."""
        return self.__intensity
    @intensity.setter
    def intensity(self, intensity):
        if  intensity is None:
            intensity = 1 # default
        assertValidSFFloat(intensity)
        assertZeroToOne('intensity', intensity)
        self.__intensity = intensity
    @property # getter - - - - - - - - - -
    def on(self):
        """Enables/disables this light source."""
        return self.__on
    @on.setter
    def on(self, on):
        if  on is None:
            on = True # default
        assertValidSFBool(on)
        self.__on = on
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function DirectionalLight.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<DirectionalLight'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.ambientIntensity != 0:
            result += " ambientIntensity='" + SFFloat(self.ambientIntensity).XML() + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.color != (1, 1, 1):
            result += " color='" + SFColor(self.color).XML() + "'"
        if self.direction != (0, 0, -1):
            result += " direction='" + SFVec3f(self.direction).XML() + "'"
        if self.global_ != False:
            result += " global_='" + SFBool(self.global_).XML() + "'"
        if self.intensity != 1:
            result += " intensity='" + SFFloat(self.intensity).XML() + "'"
        if self.on != True:
            result += " on='" + SFBool(self.on).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></DirectionalLight>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</DirectionalLight>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function DirectionalLight.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'DirectionalLight' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'DirectionalLight' + ' {'
        if self.ambientIntensity != 0:
            result += " ambientIntensity " + SFFloat(self.ambientIntensity).VRML() + ""
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.color != (1, 1, 1):
            result += " color " + SFColor(self.color).VRML() + ""
        if self.direction != (0, 0, -1):
            result += " direction " + SFVec3f(self.direction).VRML() + ""
        if self.global_ != False:
            result += " global_ " + SFBool(self.global_).VRML() + ""
        if self.intensity != 1:
            result += " intensity " + SFFloat(self.intensity).VRML() + ""
        if self.on != True:
            result += " on " + SFBool(self.on).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class DISEntityManager(_X3DChildNode):
    """
    DISEntityManager notifies a scene when new DIS ESPDU entities arrive or current entities leave.
    """
    def NAME(self):
        return 'DISEntityManager'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#DISEntityManager'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('address', 'localhost', FieldType.SFString, AccessType.inputOutput, 'DISEntityManager'),
        ('applicationID', 0, FieldType.SFInt32, AccessType.inputOutput, 'DISEntityManager'),
        ('port', 0, FieldType.SFInt32, AccessType.inputOutput, 'DISEntityManager'),
        ('siteID', 0, FieldType.SFInt32, AccessType.inputOutput, 'DISEntityManager'),
        ('mapping', list(), FieldType.MFNode, AccessType.inputOutput, 'DISEntityManager'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 address='localhost',
                 applicationID=0,
                 port=0,
                 siteID=0,
                 mapping=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode DISEntityManager __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.address = address
        self.applicationID = applicationID
        self.port = port
        self.siteID = siteID
        self.mapping = mapping
    @property # getter - - - - - - - - - -
    def address(self):
        """Multicast network address, or else "localhost" example: 224."""
        return self.__address
    @address.setter
    def address(self, address):
        if  address is None:
            address = 'localhost' # default
        assertValidSFString(address)
        self.__address = address
    @property # getter - - - - - - - - - -
    def applicationID(self):
        """Each simulation application that can respond to simulation management PDUs needs to have a unique applicationID."""
        return self.__applicationID
    @applicationID.setter
    def applicationID(self, applicationID):
        if  applicationID is None:
            applicationID = 0 # default
        assertValidSFInt32(applicationID)
        self.__applicationID = applicationID
    @property # getter - - - - - - - - - -
    def port(self):
        """Multicast network port, for example: 3000."""
        return self.__port
    @port.setter
    def port(self, port):
        if  port is None:
            port = 0 # default
        assertValidSFInt32(port)
        self.__port = port
    @property # getter - - - - - - - - - -
    def siteID(self):
        """Simulation/exercise siteID of the participating LAN or organization."""
        return self.__siteID
    @siteID.setter
    def siteID(self, siteID):
        if  siteID is None:
            siteID = 0 # default
        assertValidSFInt32(siteID)
        self.__siteID = siteID
    @property # getter - - - - - - - - - -
    def mapping(self):
        """[DISEntityTypeMapping] mapping field provides a mechanism for automatically creating an X3D model when a new entity arrives over the network."""
        return self.__mapping
    @mapping.setter
    def mapping(self, mapping):
        if  mapping is None:
            mapping = MFNode.DEFAULT_VALUE(self)
        assertValidMFNode(mapping)
        self.__mapping = mapping
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.mapping) or (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function DISEntityManager.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<DISEntityManager'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.address != 'localhost':
            result += " address='" + self.address + "'"
        if self.applicationID != 0:
            result += " applicationID='" + SFInt32(self.applicationID).XML() + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.port != 0:
            result += " port='" + SFInt32(self.port).XML() + "'"
        if self.siteID != 0:
            result += " siteID='" + SFInt32(self.siteID).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></DISEntityManager>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            ### if self.mapping: # walk each child in MFNode list, if any
            ### print('* DISEntityManager found self.children with self.hasChild()=' + str(self.hasChild()) + ' and len(mapping)=' + str(len(self.mapping)) + ', now invoking XML(' + str(indentLevel+1) + ')', flush=True)
            for each in self.mapping:
               result += each.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</DISEntityManager>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function DISEntityManager.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'DISEntityManager' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'DISEntityManager' + ' {'
        if self.address != 'localhost':
            result += " address " +  '"' + self.address + '"' + ""
        if self.applicationID != 0:
            result += " applicationID " + SFInt32(self.applicationID).VRML() + ""
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.port != 0:
            result += " port " + SFInt32(self.port).VRML() + ""
        if self.siteID != 0:
            result += " siteID " + SFInt32(self.siteID).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.mapping: # walk each child in MFNode list, if any
            for each in self.mapping:
                result += each.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class DISEntityTypeMapping(_X3DInfoNode, _X3DUrlObject):
    """
    DISEntityTypeMapping provides a best-match mapping from DIS ESPDU entity type information to a specific X3D model, thus providing a visual and behavioral representation that best matches the entity type.
    """
    def NAME(self):
        return 'DISEntityTypeMapping'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#DISEntityTypeMapping'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('category', 0, FieldType.SFInt32, AccessType.initializeOnly, 'DISEntityTypeMapping'),
        ('country', 0, FieldType.SFInt32, AccessType.initializeOnly, 'DISEntityTypeMapping'),
        ('domain', 0, FieldType.SFInt32, AccessType.initializeOnly, 'DISEntityTypeMapping'),
        ('extra', 0, FieldType.SFInt32, AccessType.initializeOnly, 'DISEntityTypeMapping'),
        ('kind', 0, FieldType.SFInt32, AccessType.initializeOnly, 'DISEntityTypeMapping'),
        ('specific', 0, FieldType.SFInt32, AccessType.initializeOnly, 'DISEntityTypeMapping'),
        ('subcategory', 0, FieldType.SFInt32, AccessType.initializeOnly, 'DISEntityTypeMapping'),
        ('url', list(), FieldType.MFString, AccessType.inputOutput, 'X3DUrlObject'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 category=0,
                 country=0,
                 domain=0,
                 extra=0,
                 kind=0,
                 specific=0,
                 subcategory=0,
                 url=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode DISEntityTypeMapping __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.category = category
        self.country = country
        self.domain = domain
        self.extra = extra
        self.kind = kind
        self.specific = specific
        self.subcategory = subcategory
        self.url = url
    @property # getter - - - - - - - - - -
    def category(self):
        """Integer enumerations value for main category that describes the entity, semantics of each code varies according to domain."""
        return self.__category
    @category.setter
    def category(self, category):
        if  category is None:
            category = 0 # default
        assertValidSFInt32(category)
        assertGreaterThanEquals('category', category, 0)
        assertLessThanEquals('category', category, 255)
        self.__category = category
    @property # getter - - - - - - - - - -
    def country(self):
        """Integer enumerations value for country to which the design of the entity or its design specification is attributed."""
        return self.__country
    @country.setter
    def country(self, country):
        if  country is None:
            country = 0 # default
        assertValidSFInt32(country)
        assertGreaterThanEquals('country', country, 0)
        assertLessThanEquals('country', country, 65535)
        self.__country = country
    @property # getter - - - - - - - - - -
    def domain(self):
        """Integer enumerations value for domain in which the entity operates: LAND, AIR, SURFACE, SUBSURFACE, SPACE or OTHER."""
        return self.__domain
    @domain.setter
    def domain(self, domain):
        if  domain is None:
            domain = 0 # default
        assertValidSFInt32(domain)
        assertGreaterThanEquals('domain', domain, 0)
        assertLessThanEquals('domain', domain, 255)
        self.__domain = domain
    @property # getter - - - - - - - - - -
    def extra(self):
        """Any extra information required to describe a particular entity."""
        return self.__extra
    @extra.setter
    def extra(self, extra):
        if  extra is None:
            extra = 0 # default
        assertValidSFInt32(extra)
        assertGreaterThanEquals('extra', extra, 0)
        assertLessThanEquals('extra', extra, 255)
        self.__extra = extra
    @property # getter - - - - - - - - - -
    def kind(self):
        """Integer enumerations value for whether entity is a PLATFORM, MUNITION, LIFE_FORM, ENVIRONMENTAL, CULTURAL_FEATURE, SUPPLY, RADIO, EXPENDABLE, SENSOR_EMITTER or OTHER."""
        return self.__kind
    @kind.setter
    def kind(self, kind):
        if  kind is None:
            kind = 0 # default
        assertValidSFInt32(kind)
        assertGreaterThanEquals('kind', kind, 0)
        assertLessThanEquals('kind', kind, 255)
        self.__kind = kind
    @property # getter - - - - - - - - - -
    def specific(self):
        """Specific information about an entity based on the Subcategory field."""
        return self.__specific
    @specific.setter
    def specific(self, specific):
        if  specific is None:
            specific = 0 # default
        assertValidSFInt32(specific)
        assertGreaterThanEquals('specific', specific, 0)
        assertLessThanEquals('specific', specific, 255)
        self.__specific = specific
    @property # getter - - - - - - - - - -
    def subcategory(self):
        return self.__subcategory
    @subcategory.setter
    def subcategory(self, subcategory):
        if  subcategory is None:
            subcategory = 0 # default
        assertValidSFInt32(subcategory)
        assertGreaterThanEquals('subcategory', subcategory, 0)
        assertLessThanEquals('subcategory', subcategory, 255)
        self.__subcategory = subcategory
    @property # getter - - - - - - - - - -
    def url(self):
        """Local and/or online addresses of X3D model of interest, for example: "ExtrusionExampleShip."""
        return self.__url
    @url.setter
    def url(self, url):
        if  url is None:
            url = MFString.DEFAULT_VALUE(self)
        assertValidMFString(url)
        self.__url = url
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function DISEntityTypeMapping.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<DISEntityTypeMapping'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.category != 0:
            result += " category='" + SFInt32(self.category).XML() + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.country != 0:
            result += " country='" + SFInt32(self.country).XML() + "'"
        if self.domain != 0:
            result += " domain='" + SFInt32(self.domain).XML() + "'"
        if self.extra != 0:
            result += " extra='" + SFInt32(self.extra).XML() + "'"
        if self.kind != 0:
            result += " kind='" + SFInt32(self.kind).XML() + "'"
        if self.specific != 0:
            result += " specific='" + SFInt32(self.specific).XML() + "'"
        if self.subcategory != 0:
            result += " subcategory='" + SFInt32(self.subcategory).XML() + "'"
        if self.url != list():
            result += " url='" + MFString(self.url).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></DISEntityTypeMapping>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</DISEntityTypeMapping>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function DISEntityTypeMapping.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'DISEntityTypeMapping' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'DISEntityTypeMapping' + ' {'
        if self.category != 0:
            result += " category " + SFInt32(self.category).VRML() + ""
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.country != 0:
            result += " country " + SFInt32(self.country).VRML() + ""
        if self.domain != 0:
            result += " domain " + SFInt32(self.domain).VRML() + ""
        if self.extra != 0:
            result += " extra " + SFInt32(self.extra).VRML() + ""
        if self.kind != 0:
            result += " kind " + SFInt32(self.kind).VRML() + ""
        if self.specific != 0:
            result += " specific " + SFInt32(self.specific).VRML() + ""
        if self.subcategory != 0:
            result += " subcategory " + SFInt32(self.subcategory).VRML() + ""
        if self.url != list():
            result += " url " + MFString(self.url).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class Disk2D(_X3DGeometryNode):
    """
    Disk2D is a geometry node that defines a filled (or partially filled) planar circle with center (0,0).
    """
    def NAME(self):
        return 'Disk2D'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#Disk2D'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('innerRadius', 0, FieldType.SFFloat, AccessType.initializeOnly, 'Disk2D'),
        ('outerRadius', 1, FieldType.SFFloat, AccessType.initializeOnly, 'Disk2D'),
        ('solid', False, FieldType.SFBool, AccessType.initializeOnly, 'Disk2D'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 innerRadius=0,
                 outerRadius=1,
                 solid=False,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode Disk2D __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.innerRadius = innerRadius
        self.outerRadius = outerRadius
        self.solid = solid
    @property # getter - - - - - - - - - -
    def innerRadius(self):
        """[0,+infinity) Inner circle radius, greater than or equal to 0."""
        return self.__innerRadius
    @innerRadius.setter
    def innerRadius(self, innerRadius):
        if  innerRadius is None:
            innerRadius = 0 # default
        assertValidSFFloat(innerRadius)
        assertNonNegative('innerRadius', innerRadius)
        self.__innerRadius = innerRadius
    @property # getter - - - - - - - - - -
    def outerRadius(self):
        """(0,+infinity) Outer radius of circle, greater than or equal to inner radius."""
        return self.__outerRadius
    @outerRadius.setter
    def outerRadius(self, outerRadius):
        if  outerRadius is None:
            outerRadius = 1 # default
        assertValidSFFloat(outerRadius)
        assertPositive('outerRadius', outerRadius)
        self.__outerRadius = outerRadius
    @property # getter - - - - - - - - - -
    def solid(self):
        """Setting solid true means draw only one side of polygons (backface culling on), setting solid false means draw both sides of polygons (backface culling off)."""
        return self.__solid
    @solid.setter
    def solid(self, solid):
        if  solid is None:
            solid = False # default
        assertValidSFBool(solid)
        self.__solid = solid
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function Disk2D.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<Disk2D'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.innerRadius != 0:
            result += " innerRadius='" + SFFloat(self.innerRadius).XML() + "'"
        if self.outerRadius != 1:
            result += " outerRadius='" + SFFloat(self.outerRadius).XML() + "'"
        if self.solid != False:
            result += " solid='" + SFBool(self.solid).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></Disk2D>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</Disk2D>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function Disk2D.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'Disk2D' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'Disk2D' + ' {'
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.innerRadius != 0:
            result += " innerRadius " + SFFloat(self.innerRadius).VRML() + ""
        if self.outerRadius != 1:
            result += " outerRadius " + SFFloat(self.outerRadius).VRML() + ""
        if self.solid != False:
            result += " solid " + SFBool(self.solid).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class DoubleAxisHingeJoint(_X3DRigidJointNode):
    """
    DoubleAxisHingeJoint has two independent axes located around a common anchor point.
    """
    def NAME(self):
        return 'DoubleAxisHingeJoint'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#DoubleAxisHingeJoint'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('anchorPoint', (0, 0, 0), FieldType.SFVec3f, AccessType.inputOutput, 'DoubleAxisHingeJoint'),
        ('axis1', (0, 0, 0), FieldType.SFVec3f, AccessType.inputOutput, 'DoubleAxisHingeJoint'),
        ('axis2', (0, 0, 0), FieldType.SFVec3f, AccessType.inputOutput, 'DoubleAxisHingeJoint'),
        ('desiredAngularVelocity1', 0, FieldType.SFFloat, AccessType.inputOutput, 'DoubleAxisHingeJoint'),
        ('desiredAngularVelocity2', 0, FieldType.SFFloat, AccessType.inputOutput, 'DoubleAxisHingeJoint'),
        ('forceOutput', ["NONE"], FieldType.MFString, AccessType.inputOutput, 'X3DRigidJointNode'),
        ('maxAngle1', 3.141592653, FieldType.SFFloat, AccessType.inputOutput, 'DoubleAxisHingeJoint'),
        ('maxTorque1', 0, FieldType.SFFloat, AccessType.inputOutput, 'DoubleAxisHingeJoint'),
        ('maxTorque2', 0, FieldType.SFFloat, AccessType.inputOutput, 'DoubleAxisHingeJoint'),
        ('minAngle1', -3.141592653, FieldType.SFFloat, AccessType.inputOutput, 'DoubleAxisHingeJoint'),
        ('stop1Bounce', 0, FieldType.SFFloat, AccessType.inputOutput, 'DoubleAxisHingeJoint'),
        ('stop1ConstantForceMix', 0.001, FieldType.SFFloat, AccessType.inputOutput, 'DoubleAxisHingeJoint'),
        ('stop1ErrorCorrection', 0.8, FieldType.SFFloat, AccessType.inputOutput, 'DoubleAxisHingeJoint'),
        ('suspensionErrorCorrection', 0.8, FieldType.SFFloat, AccessType.inputOutput, 'DoubleAxisHingeJoint'),
        ('suspensionForce', 0, FieldType.SFFloat, AccessType.inputOutput, 'DoubleAxisHingeJoint'),
        ('body1', None, FieldType.SFNode, AccessType.inputOutput, 'X3DRigidJointNode'),
        ('body2', None, FieldType.SFNode, AccessType.inputOutput, 'X3DRigidJointNode'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 anchorPoint=(0, 0, 0),
                 axis1=(0, 0, 0),
                 axis2=(0, 0, 0),
                 desiredAngularVelocity1=0,
                 desiredAngularVelocity2=0,
                 forceOutput=["NONE"],
                 maxAngle1=3.141592653,
                 maxTorque1=0,
                 maxTorque2=0,
                 minAngle1=-3.141592653,
                 stop1Bounce=0,
                 stop1ConstantForceMix=0.001,
                 stop1ErrorCorrection=0.8,
                 suspensionErrorCorrection=0.8,
                 suspensionForce=0,
                 body1=None,
                 body2=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode DoubleAxisHingeJoint __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.anchorPoint = anchorPoint
        self.axis1 = axis1
        self.axis2 = axis2
        self.desiredAngularVelocity1 = desiredAngularVelocity1
        self.desiredAngularVelocity2 = desiredAngularVelocity2
        self.forceOutput = forceOutput
        self.maxAngle1 = maxAngle1
        self.maxTorque1 = maxTorque1
        self.maxTorque2 = maxTorque2
        self.minAngle1 = minAngle1
        self.stop1Bounce = stop1Bounce
        self.stop1ConstantForceMix = stop1ConstantForceMix
        self.stop1ErrorCorrection = stop1ErrorCorrection
        self.suspensionErrorCorrection = suspensionErrorCorrection
        self.suspensionForce = suspensionForce
        self.body1 = body1
        self.body2 = body2
    @property # getter - - - - - - - - - -
    def anchorPoint(self):
        """anchorPoint is joint center, specified in world coordinates."""
        return self.__anchorPoint
    @anchorPoint.setter
    def anchorPoint(self, anchorPoint):
        if  anchorPoint is None:
            anchorPoint = (0, 0, 0) # default
        assertValidSFVec3f(anchorPoint)
        self.__anchorPoint = anchorPoint
    @property # getter - - - - - - - - - -
    def axis1(self):
        """axis1 defines axis vector of joint connection to body1."""
        return self.__axis1
    @axis1.setter
    def axis1(self, axis1):
        if  axis1 is None:
            axis1 = (0, 0, 0) # default
        assertValidSFVec3f(axis1)
        self.__axis1 = axis1
    @property # getter - - - - - - - - - -
    def axis2(self):
        """axis2 defines axis vector of joint connection to body2."""
        return self.__axis2
    @axis2.setter
    def axis2(self, axis2):
        if  axis2 is None:
            axis2 = (0, 0, 0) # default
        assertValidSFVec3f(axis2)
        self.__axis2 = axis2
    @property # getter - - - - - - - - - -
    def desiredAngularVelocity1(self):
        """desiredAngularVelocity1 is goal rotation rate for hinge connection to body1."""
        return self.__desiredAngularVelocity1
    @desiredAngularVelocity1.setter
    def desiredAngularVelocity1(self, desiredAngularVelocity1):
        if  desiredAngularVelocity1 is None:
            desiredAngularVelocity1 = 0 # default
        assertValidSFFloat(desiredAngularVelocity1)
        self.__desiredAngularVelocity1 = desiredAngularVelocity1
    @property # getter - - - - - - - - - -
    def desiredAngularVelocity2(self):
        """desiredAngularVelocity2 is goal rotation rate for hinge connection to body2."""
        return self.__desiredAngularVelocity2
    @desiredAngularVelocity2.setter
    def desiredAngularVelocity2(self, desiredAngularVelocity2):
        if  desiredAngularVelocity2 is None:
            desiredAngularVelocity2 = 0 # default
        assertValidSFFloat(desiredAngularVelocity2)
        self.__desiredAngularVelocity2 = desiredAngularVelocity2
    @property # getter - - - - - - - - - -
    def forceOutput(self):
        """forceOutput controls which output fields are generated for the next frame."""
        return self.__forceOutput
    @forceOutput.setter
    def forceOutput(self, forceOutput):
        if  forceOutput is None:
            forceOutput = ["NONE"] # default
        assertValidMFString(forceOutput)
        self.__forceOutput = forceOutput
    @property # getter - - - - - - - - - -
    def maxAngle1(self):
        """[-pi,pi] maxAngle1 is maximum rotation angle for hinge."""
        return self.__maxAngle1
    @maxAngle1.setter
    def maxAngle1(self, maxAngle1):
        if  maxAngle1 is None:
            maxAngle1 = 3.141592653 # default
        assertValidSFFloat(maxAngle1)
        self.__maxAngle1 = maxAngle1
    @property # getter - - - - - - - - - -
    def maxTorque1(self):
        """maxTorque1 is maximum rotational torque applied by corresponding motor axis to achieve desiredAngularVelocity1."""
        return self.__maxTorque1
    @maxTorque1.setter
    def maxTorque1(self, maxTorque1):
        if  maxTorque1 is None:
            maxTorque1 = 0 # default
        assertValidSFFloat(maxTorque1)
        self.__maxTorque1 = maxTorque1
    @property # getter - - - - - - - - - -
    def maxTorque2(self):
        """maxTorque2 is maximum rotational torque applied by corresponding motor axis to achieve desiredAngularVelocity2."""
        return self.__maxTorque2
    @maxTorque2.setter
    def maxTorque2(self, maxTorque2):
        if  maxTorque2 is None:
            maxTorque2 = 0 # default
        assertValidSFFloat(maxTorque2)
        self.__maxTorque2 = maxTorque2
    @property # getter - - - - - - - - - -
    def minAngle1(self):
        """[-pi,pi] minAngle1 is minimum rotation angle for hinge."""
        return self.__minAngle1
    @minAngle1.setter
    def minAngle1(self, minAngle1):
        if  minAngle1 is None:
            minAngle1 = -3.141592653 # default
        assertValidSFFloat(minAngle1)
        self.__minAngle1 = minAngle1
    @property # getter - - - - - - - - - -
    def stop1Bounce(self):
        """[0,1] stop1Bounce is velocity factor for bounce back once stop point is reached."""
        return self.__stop1Bounce
    @stop1Bounce.setter
    def stop1Bounce(self, stop1Bounce):
        if  stop1Bounce is None:
            stop1Bounce = 0 # default
        assertValidSFFloat(stop1Bounce)
        self.__stop1Bounce = stop1Bounce
    @property # getter - - - - - - - - - -
    def stop1ConstantForceMix(self):
        """[0,1] stop1ConstantForceMix value applies a constant force value to make colliding surfaces appear to be somewhat soft."""
        return self.__stop1ConstantForceMix
    @stop1ConstantForceMix.setter
    def stop1ConstantForceMix(self, stop1ConstantForceMix):
        if  stop1ConstantForceMix is None:
            stop1ConstantForceMix = 0.001 # default
        assertValidSFFloat(stop1ConstantForceMix)
        self.__stop1ConstantForceMix = stop1ConstantForceMix
    @property # getter - - - - - - - - - -
    def stop1ErrorCorrection(self):
        """[0,1] stop1ErrorCorrection is fraction of error correction performed during time step once stop point is reached."""
        return self.__stop1ErrorCorrection
    @stop1ErrorCorrection.setter
    def stop1ErrorCorrection(self, stop1ErrorCorrection):
        if  stop1ErrorCorrection is None:
            stop1ErrorCorrection = 0.8 # default
        assertValidSFFloat(stop1ErrorCorrection)
        self.__stop1ErrorCorrection = stop1ErrorCorrection
    @property # getter - - - - - - - - - -
    def suspensionErrorCorrection(self):
        """[0,1] suspensionErrorCorrection describes how quickly the system resolves intersection errors due to floating-point inaccuracies."""
        return self.__suspensionErrorCorrection
    @suspensionErrorCorrection.setter
    def suspensionErrorCorrection(self, suspensionErrorCorrection):
        if  suspensionErrorCorrection is None:
            suspensionErrorCorrection = 0.8 # default
        assertValidSFFloat(suspensionErrorCorrection)
        self.__suspensionErrorCorrection = suspensionErrorCorrection
    @property # getter - - - - - - - - - -
    def suspensionForce(self):
        """[0,1] suspensionForce describes how quickly the system resolves intersection errors due to floating-point inaccuracies."""
        return self.__suspensionForce
    @suspensionForce.setter
    def suspensionForce(self, suspensionForce):
        if  suspensionForce is None:
            suspensionForce = 0 # default
        assertValidSFFloat(suspensionForce)
        self.__suspensionForce = suspensionForce
    @property # getter - - - - - - - - - -
    def body1(self):
        return self.__body1
    @body1.setter
    def body1(self, body1):
        if  body1 is None:
            body1 = None # default
        assertValidSFNode(body1)
        if not isinstance(body1, object):
            # print(flush=True)
            raise X3DTypeError(str(body1) + ' does not have a valid node type object')
        self.__body1 = body1
    @property # getter - - - - - - - - - -
    def body2(self):
        return self.__body2
    @body2.setter
    def body2(self, body2):
        if  body2 is None:
            body2 = None # default
        assertValidSFNode(body2)
        if not isinstance(body2, object):
            # print(flush=True)
            raise X3DTypeError(str(body2) + ' does not have a valid node type object')
        self.__body2 = body2
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.body1) or (self.body2) or (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function DoubleAxisHingeJoint.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<DoubleAxisHingeJoint'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.anchorPoint != (0, 0, 0):
            result += " anchorPoint='" + SFVec3f(self.anchorPoint).XML() + "'"
        if self.axis1 != (0, 0, 0):
            result += " axis1='" + SFVec3f(self.axis1).XML() + "'"
        if self.axis2 != (0, 0, 0):
            result += " axis2='" + SFVec3f(self.axis2).XML() + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.desiredAngularVelocity1 != 0:
            result += " desiredAngularVelocity1='" + SFFloat(self.desiredAngularVelocity1).XML() + "'"
        if self.desiredAngularVelocity2 != 0:
            result += " desiredAngularVelocity2='" + SFFloat(self.desiredAngularVelocity2).XML() + "'"
        if self.forceOutput != ["NONE"]:
            result += " forceOutput='" + MFString(self.forceOutput).XML() + "'"
        if self.maxAngle1 != 3.141592653:
            result += " maxAngle1='" + SFFloat(self.maxAngle1).XML() + "'"
        if self.maxTorque1 != 0:
            result += " maxTorque1='" + SFFloat(self.maxTorque1).XML() + "'"
        if self.maxTorque2 != 0:
            result += " maxTorque2='" + SFFloat(self.maxTorque2).XML() + "'"
        if self.minAngle1 != -3.141592653:
            result += " minAngle1='" + SFFloat(self.minAngle1).XML() + "'"
        if self.stop1Bounce != 0:
            result += " stop1Bounce='" + SFFloat(self.stop1Bounce).XML() + "'"
        if self.stop1ConstantForceMix != 0.001:
            result += " stop1ConstantForceMix='" + SFFloat(self.stop1ConstantForceMix).XML() + "'"
        if self.stop1ErrorCorrection != 0.8:
            result += " stop1ErrorCorrection='" + SFFloat(self.stop1ErrorCorrection).XML() + "'"
        if self.suspensionErrorCorrection != 0.8:
            result += " suspensionErrorCorrection='" + SFFloat(self.suspensionErrorCorrection).XML() + "'"
        if self.suspensionForce != 0:
            result += " suspensionForce='" + SFFloat(self.suspensionForce).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></DoubleAxisHingeJoint>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.body1: # output this SFNode
                result += self.body1.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.body2: # output this SFNode
                result += self.body2.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</DoubleAxisHingeJoint>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function DoubleAxisHingeJoint.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'DoubleAxisHingeJoint' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'DoubleAxisHingeJoint' + ' {'
        if self.anchorPoint != (0, 0, 0):
            result += " anchorPoint " + SFVec3f(self.anchorPoint).VRML() + ""
        if self.axis1 != (0, 0, 0):
            result += " axis1 " + SFVec3f(self.axis1).VRML() + ""
        if self.axis2 != (0, 0, 0):
            result += " axis2 " + SFVec3f(self.axis2).VRML() + ""
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.desiredAngularVelocity1 != 0:
            result += " desiredAngularVelocity1 " + SFFloat(self.desiredAngularVelocity1).VRML() + ""
        if self.desiredAngularVelocity2 != 0:
            result += " desiredAngularVelocity2 " + SFFloat(self.desiredAngularVelocity2).VRML() + ""
        if self.forceOutput != ["NONE"]:
            result += " forceOutput " + MFString(self.forceOutput).VRML() + ""
        if self.maxAngle1 != 3.141592653:
            result += " maxAngle1 " + SFFloat(self.maxAngle1).VRML() + ""
        if self.maxTorque1 != 0:
            result += " maxTorque1 " + SFFloat(self.maxTorque1).VRML() + ""
        if self.maxTorque2 != 0:
            result += " maxTorque2 " + SFFloat(self.maxTorque2).VRML() + ""
        if self.minAngle1 != -3.141592653:
            result += " minAngle1 " + SFFloat(self.minAngle1).VRML() + ""
        if self.stop1Bounce != 0:
            result += " stop1Bounce " + SFFloat(self.stop1Bounce).VRML() + ""
        if self.stop1ConstantForceMix != 0.001:
            result += " stop1ConstantForceMix " + SFFloat(self.stop1ConstantForceMix).VRML() + ""
        if self.stop1ErrorCorrection != 0.8:
            result += " stop1ErrorCorrection " + SFFloat(self.stop1ErrorCorrection).VRML() + ""
        if self.suspensionErrorCorrection != 0.8:
            result += " suspensionErrorCorrection " + SFFloat(self.suspensionErrorCorrection).VRML() + ""
        if self.suspensionForce != 0:
            result += " suspensionForce " + SFFloat(self.suspensionForce).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.body1: # output this SFNode
            result += '\n' + '  ' + indent + 'body1 ' + self.body1.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.body2: # output this SFNode
            result += '\n' + '  ' + indent + 'body2 ' + self.body2.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class EaseInEaseOut(_X3DChildNode):
    """
    EaseInEaseOut enables gradual animation transitions by modifying TimeSensor fraction outputs.
    """
    def NAME(self):
        return 'EaseInEaseOut'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#EaseInEaseOut'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('easeInEaseOut', list(), FieldType.MFVec2f, AccessType.inputOutput, 'EaseInEaseOut'),
        ('key', list(), FieldType.MFFloat, AccessType.inputOutput, 'EaseInEaseOut'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 easeInEaseOut=list(),
                 key=list(),
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode EaseInEaseOut __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.easeInEaseOut = easeInEaseOut
        self.key = key
    @property # getter - - - - - - - - - -
    def easeInEaseOut(self):
        """Array of paired values for easeOut fraction and easeIn fraction within each key interval."""
        return self.__easeInEaseOut
    @easeInEaseOut.setter
    def easeInEaseOut(self, easeInEaseOut):
        if  easeInEaseOut is None:
            easeInEaseOut = MFVec2f.DEFAULT_VALUE(self)
        assertValidMFVec2f(easeInEaseOut)
        self.__easeInEaseOut = easeInEaseOut
    @property # getter - - - - - - - - - -
    def key(self):
        """Definition values for linear-interpolation function input intervals, listed in non-decreasing order and corresponding to easeInEaseOut array."""
        return self.__key
    @key.setter
    def key(self, key):
        if  key is None:
            key = MFFloat.DEFAULT_VALUE(self)
        assertValidMFFloat(key)
        self.__key = key
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.IS) or (self.metadata)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function EaseInEaseOut.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<EaseInEaseOut'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.easeInEaseOut != list():
            result += " easeInEaseOut='" + MFVec2f(self.easeInEaseOut).XML() + "'"
        if self.key != list():
            result += " key='" + MFFloat(self.key).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></EaseInEaseOut>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</EaseInEaseOut>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function EaseInEaseOut.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'EaseInEaseOut' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'EaseInEaseOut' + ' {'
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.easeInEaseOut != list():
            result += " easeInEaseOut " + MFVec2f(self.easeInEaseOut).VRML() + ""
        if self.key != list():
            result += " key " + MFFloat(self.key).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class EdgeEnhancementVolumeStyle(_X3DComposableVolumeRenderStyleNode):
    """
    EdgeEnhancementVolumeStyle specifies edge enhancement for the volume rendering style.
    """
    def NAME(self):
        return 'EdgeEnhancementVolumeStyle'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#EdgeEnhancementVolumeStyle'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('edgeColor', (0, 0, 0, 1), FieldType.SFColorRGBA, AccessType.inputOutput, 'EdgeEnhancementVolumeStyle'),
        ('enabled', True, FieldType.SFBool, AccessType.inputOutput, 'X3DVolumeRenderStyleNode'),
        ('gradientThreshold', 0.4, FieldType.SFFloat, AccessType.inputOutput, 'EdgeEnhancementVolumeStyle'),
        ('surfaceNormals', None, FieldType.SFNode, AccessType.inputOutput, 'EdgeEnhancementVolumeStyle'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 edgeColor=(0, 0, 0, 1),
                 enabled=True,
                 gradientThreshold=0.4,
                 surfaceNormals=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode EdgeEnhancementVolumeStyle __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.edgeColor = edgeColor
        self.enabled = enabled
        self.gradientThreshold = gradientThreshold
        self.surfaceNormals = surfaceNormals
    @property # getter - - - - - - - - - -
    def edgeColor(self):
        """[0,1] color used to highlight edges."""
        return self.__edgeColor
    @edgeColor.setter
    def edgeColor(self, edgeColor):
        if  edgeColor is None:
            edgeColor = (0, 0, 0, 1) # default
        assertValidSFColorRGBA(edgeColor)
        assertZeroToOne('edgeColor', edgeColor)
        self.__edgeColor = edgeColor
    @property # getter - - - - - - - - - -
    def enabled(self):
        """Enables/disables node operation."""
        return self.__enabled
    @enabled.setter
    def enabled(self, enabled):
        if  enabled is None:
            enabled = True # default
        assertValidSFBool(enabled)
        self.__enabled = enabled
    @property # getter - - - - - - - - - -
    def gradientThreshold(self):
        """[0,1] minimum angle (in radians) away from view-direction vector for surface normal before applying enhancement."""
        return self.__gradientThreshold
    @gradientThreshold.setter
    def gradientThreshold(self, gradientThreshold):
        if  gradientThreshold is None:
            gradientThreshold = 0.4 # default
        assertValidSFFloat(gradientThreshold)
        assertGreaterThanEquals('gradientThreshold', gradientThreshold, 0)
        assertLessThanEquals('gradientThreshold', gradientThreshold, 3.1416)
        self.__gradientThreshold = gradientThreshold
    @property # getter - - - - - - - - - -
    def surfaceNormals(self):
        return self.__surfaceNormals
    @surfaceNormals.setter
    def surfaceNormals(self, surfaceNormals):
        if  surfaceNormals is None:
            surfaceNormals = None # default
        assertValidSFNode(surfaceNormals)
        if not isinstance(surfaceNormals, object):
            # print(flush=True)
            raise X3DTypeError(str(surfaceNormals) + ' does not have a valid node type object')
        self.__surfaceNormals = surfaceNormals
    # hasChild() function - - - - - - - - - -
    def hasChild(self):
        ''' Whether or not this node has any child node, statement or comment '''
        return (self.IS) or (self.metadata) or (self.surfaceNormals)
    # output function - - - - - - - - - -
    def XML(self, indentLevel=0, syntax="XML"):
        """ Provide Canonical X3D output serialization using XML encoding. """
        result = ''
        indent = '  ' * indentLevel
        result = indent ### confirm
        # if _DEBUG: result += indent + '# invoked class function EdgeEnhancementVolumeStyle.XML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        result += '<EdgeEnhancementVolumeStyle'
        if self.DEF:
            result += " DEF='" + self.DEF + "'"
        if self.USE:
            result += " USE='" + self.USE + "'"
        if self.class_:
            result += " class_='" + self.class_ + "'"
        if self.edgeColor != (0, 0, 0, 1):
            result += " edgeColor='" + SFColorRGBA(self.edgeColor).XML() + "'"
        if self.enabled != True:
            result += " enabled='" + SFBool(self.enabled).XML() + "'"
        if self.gradientThreshold != 0.4:
            result += " gradientThreshold='" + SFFloat(self.gradientThreshold).XML() + "'"
        if not self.hasChild():
            if syntax.upper() == "HTML5":
                result += '></EdgeEnhancementVolumeStyle>' + '\n' # no self-closing tags allowed by HTML5
            elif syntax.upper() == "XML":
                result += '/>' + '\n' # singleton element
            else:
                raise X3DValueError('.toXML(syntax=' + syntax + ') is incorrect, allowed values are "HTML5" and "XML"')
        else:
            result += '>' + '\n'
            if self.IS: # output this SFNode
                result += self.IS.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.metadata: # output this SFNode
                result += self.metadata.XML(indentLevel=indentLevel+1,syntax=syntax)
            if self.surfaceNormals: # output this SFNode
                result += self.surfaceNormals.XML(indentLevel=indentLevel+1,syntax=syntax)
            result += indent + '</EdgeEnhancementVolumeStyle>' + '\n'
#       print('XML serialization complete.', flush=True)
        return result
    # output function - - - - - - - - - -
    def HTML5(self, indentLevel=0):
        """ Provide HTML5 output serialization using XML encoding with no singleton self-closing elements. """
        return self.XML(indentLevel, syntax="HTML5")
    # output function - - - - - - - - - -
    def VRML(self, indentLevel=0, VRML97=False):
        """ Provide X3D output serialization using VRML encoding. """
        result = ''
        indent = '  ' * indentLevel
        # if _DEBUG: result += indent + '# invoked class function EdgeEnhancementVolumeStyle.VRML(self=' + str(self) + ', indentLevel=' + str(indentLevel) + '), indent="' + indent + '"'
        # print(result)
        if indentLevel == 0:
            result += '\n'
        if self.DEF:
            result += 'DEF ' + self.DEF + ' ' + 'EdgeEnhancementVolumeStyle' + ' {'
        elif self.USE:
            result += 'USE ' + self.USE # no node name, nothing follows
        else:
            result += 'EdgeEnhancementVolumeStyle' + ' {'
        if self.class_:
            result += " class_ " +  '"' + self.class_ + '"' + ""
        if self.edgeColor != (0, 0, 0, 1):
            result += " edgeColor " + SFColorRGBA(self.edgeColor).VRML() + ""
        if self.enabled != True:
            result += " enabled " + SFBool(self.enabled).VRML() + ""
        if self.gradientThreshold != 0.4:
            result += " gradientThreshold " + SFFloat(self.gradientThreshold).VRML() + ""
        if self.IS: # output this SFNode
            result += '\n' + '  ' + indent + 'IS ' + self.IS.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.metadata: # output this SFNode
            result += '\n' + '  ' + indent + 'metadata ' + self.metadata.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if self.surfaceNormals: # output this SFNode
            result += '\n' + '  ' + indent + 'surfaceNormals ' + self.surfaceNormals.VRML(indentLevel=indentLevel+1,VRML97=VRML97)
        if not self.USE:
            result += ' }'
#       print('VRML serialization complete.', flush=True)
        return result

class ElevationGrid(_X3DGeometryNode):
    """
    ElevationGrid is a geometry node defining a rectangular height field, with default values for a 1m by 1m square at height 0.
    """
    def NAME(self):
        return 'ElevationGrid'
    def SPECIFICATION_URL(self):
        return ''
    def TOOLTIP_URL(self):
        return 'https://www.web3d.org/x3d/tooltips/X3dTooltips.html#ElevationGrid'
    def FIELD_DECLARATIONS(self):
        return [ # name, defaultValue, type, accessType, inheritedFrom
        ('ccw', True, FieldType.SFBool, AccessType.initializeOnly, 'ElevationGrid'),
        ('colorPerVertex', True, FieldType.SFBool, AccessType.initializeOnly, 'ElevationGrid'),
        ('creaseAngle', 0, FieldType.SFFloat, AccessType.initializeOnly, 'ElevationGrid'),
        ('height', [0, 0, 0, 0], FieldType.MFFloat, AccessType.initializeOnly, 'ElevationGrid'),
        ('normalPerVertex', True, FieldType.SFBool, AccessType.initializeOnly, 'ElevationGrid'),
        ('solid', True, FieldType.SFBool, AccessType.initializeOnly, 'ElevationGrid'),
        ('xDimension', 2, FieldType.SFInt32, AccessType.initializeOnly, 'ElevationGrid'),
        ('xSpacing', 1.0, FieldType.SFFloat, AccessType.initializeOnly, 'ElevationGrid'),
        ('zDimension', 2, FieldType.SFInt32, AccessType.initializeOnly, 'ElevationGrid'),
        ('zSpacing', 1.0, FieldType.SFFloat, AccessType.initializeOnly, 'ElevationGrid'),
        ('color', None, FieldType.SFNode, AccessType.inputOutput, 'ElevationGrid'),
        ('fogCoord', None, FieldType.SFNode, AccessType.inputOutput, 'ElevationGrid'),
        ('normal', None, FieldType.SFNode, AccessType.inputOutput, 'ElevationGrid'),
        ('texCoord', None, FieldType.SFNode, AccessType.inputOutput, 'ElevationGrid'),
        ('attrib', list(), FieldType.MFNode, AccessType.inputOutput, 'ElevationGrid'),
        ('DEF', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('USE', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('class_', '', FieldType.SFString, AccessType.inputOutput, 'X3DNode'),
        ('IS', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode'),
        ('metadata', None, FieldType.SFNode, AccessType.inputOutput, 'X3DNode')]
    def __init__(self,
                 ccw=True,
                 colorPerVertex=True,
                 creaseAngle=0,
                 height=[0, 0, 0, 0],
                 normalPerVertex=True,
                 solid=True,
                 xDimension=2,
                 xSpacing=1.0,
                 zDimension=2,
                 zSpacing=1.0,
                 color=None,
                 fogCoord=None,
                 normal=None,
                 texCoord=None,
                 attrib=None,
                 DEF='',
                 USE='',
                 class_='',
                 IS=None,
                 metadata=None):
        # if _DEBUG: print('... in ConcreteNode ElevationGrid __init__ calling super.__init__(' + str(DEF) + ',' + str(USE) + ',' + str(class_) + ',' + str(metadata) + ',' + str(IS) + ')', flush=True)
        super().__init__(DEF, USE, class_, IS, metadata) # fields for _X3DNode only
        self.ccw = ccw
        self.colorPerVertex = colorPerVertex
        self.creaseAngle = creaseAngle
        self.height = height
        self.normalPerVertex = normalPerVertex
        self.solid = solid
        self.xDimension = xDimension
        self.xSpacing = xSpacing
        self.zDimension = zDimension
        self.zSpacing = zSpacing
        self.color = color
        self.fogCoord = fogCoord
        self.normal = normal
        self.texCoord = texCoord
        self.attrib = attrib
    @property # getter - - - - - - - - - -
    def ccw(self):
        """ccw defines clockwise/counterclockwise ordering of vertex coordinates, which in turn defines front/back orientation of polygon normals according to Right-Hand Rule (RHR)."""
        return self.__ccw
    @ccw.setter
    def ccw(self, ccw):
        if  ccw is None:
            ccw = True # default
        assertValidSFBool(ccw)
        self.__ccw = ccw
    @property # getter - - - - - - - - - -
    def colorPerVertex(self):
        """Whether Color node color values are applied to each point vertex (true) or per quadrilateral (false)."""
        return self.__colorPerVertex
    @colorPerVertex.setter
    def colorPerVertex(self, colorPerVertex):
        if  colorPerVertex is None:
            colorPerVertex = True # default
        assertValidSFBool(colorPerVertex)
        self.__colorPerVertex = colorPerVertex
    @property # getter - - - - - - - - - -
    def creaseAngle(self):
        """[0,+infinity) creaseAngle defines angle (in radians) for determining whether adjacent polygons are drawn with sharp edges or smooth shading."""
        return self.__creaseAngle
    @creaseAngle.setter
    def creaseAngle(self, creaseAngle):
        if  creaseAngle is None:
            creaseAngle = 0 # default
        assertValidSFFloat(creaseAngle)
        assertNonNegative('creaseAngle', creaseAngle)
        self.__creaseAngle = creaseAngle
    @property # getter - - - - - - - - - -
    def height(s