# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['connector_definition_runner']

package_data = \
{'': ['*']}

install_requires = \
['PyYAML>=6.0,<7.0', 'chevron>=0.14.0,<1.0.0', 'requests==2.28.0']

setup_kwargs = {
    'name': 'connector-def-runner',
    'version': '2.2.0',
    'description': 'Run connectors by schema definitions',
    'long_description': '# Connector Definition Runner\n\nThis package provides the framework for connectors to be created with only schemas.  It can make HTTP \ncalls based on what is defined in the schema.\n## Asset Schema Structure\n\nThe asset schema defines the authentication that is used for the connector. \n\nWe currently support the following authentication methods:\n* API Key (in the header, query parameters, or cookie)\n* OAuth2\n  * Client Credential\n  * Password Grant\n* HTTP Basic\n* HTTP Bearer Token\n\nEach authentication type has an asset file name and inputs that it requires. There is also a \n`meta.security` key in the asset schema that contains the configuration for the authentication.\nThe asset file name must match the `name` field, unless it\'s a custom asset (See the examples below).\n\nExamples:\n\n### http_basic\n\n```yaml\nschema: asset/1\nname: http_basic\ntitle: HTTP Basic Authentication\ndescription: \'Authenticates using username and password.\'\ninputs:\n  type: object\n  properties:\n    url:\n      title: URL\n      description: A URL to the target host.\n      type: string\n      default: https://www.example.com # change if it has a default cloud URL or remove if always custom.\n    username:\n      title: Username\n      description: Username\n      type: string\n    password:\n      title: Password\n      description: Password\n      type: string\n      format: password\n    verify_ssl:\n      title: Verify SSL Certificates\n      description: Verify SSL certificate\n      type: boolean\n    http_proxy:\n      title: HTTP(s) Proxy\n      description: A proxy to route requests through.\n      type: string\n  required:\n    - url\n    - username\n    - password\nmeta:\n  security:\n    type: http\n    scheme: basic\n```\n\n### http_bearer\n\n```yaml\nschema: asset/1\nname: http_bearer\ntitle: HTTP Bearer Authentication\ndescription: \'Authenticates using bearer token such as a JWT, etc.\'\ninputs:\n  type: object\n  properties:\n    url:\n      title: URL\n      description: A URL to the target host.\n      type: string\n      default: https://www.example.com # change if it has a default cloud URL or remove if always custom.\n    token: # NEVER CHANGE THIS\n      title: Token # name this properly\n      description: The API key, token, etc. # name this properly\n      type: string\n      format: password\n    verify_ssl:\n      title: Verify SSL Certificates\n      description: Verify SSL certificate\n      type: boolean\n    http_proxy:\n      title: HTTP(s) Proxy\n      description: A proxy to route requests through.\n      type: string\n  required:\n    - url\n    - username\n    - password\nmeta:\n  security:\n    type: http\n    scheme: bearer\n```\n### apikey\n```yaml\nschema: asset/1\nname: apikey\ntitle: API Key Authentication\ndescription: \'Authenticates using an API Key\'\ninputs:\n  type: object\n  properties:\n    url:\n      title: URL\n      description: A URL to the target host.\n      type: string\n      default: https://www.example.com # change if it has a default cloud URL or remove if always custom.\n    x-apikey: # example, replace with correct key name for product.\n      title: API Key\n      description: API key\n      type: string\n      format: password\n    verify_ssl:\n      title: Verify SSL Certificates\n      description: Verify SSL certificate\n      type: boolean\n    http_proxy:\n      title: HTTP(s) Proxy\n      description: A proxy to route requests through.\n      type: string\n  required:\n    - url\n    - x-apikey # example, replace with correct key name for product.\nmeta:\n  security:\n    type: apiKey\n    name: x-apikey # example, replace with correct key name for product.\n    in: header, cookie or query # please select the one applicable to the API and remove the others.\n\n```\n\nThe `security` field inside `meta` could be an object or an array of objects. \n\n**Examples:**\n\nIf you need to make a request with the header field `x-apikey` and add its value from the inputs, you can use the following schema. \nIn this case the `x-apikey` input field is mandatory.\n\n```yaml\ninputs:\n  type: object\n  properties:\n    x-apikey: # example, replace with correct key name for product.\n      title: API Key\n      type: string\n      format: password\nmeta:\n  security:\n    name: x-apikey\n    in: header\n```\n\nYou can also define a template using mustache syntax with the input values. For example, if you want \nto add to your header the field `Authorization: ApiToken my_token`, you can use the following schema\nand use `my_token` as an input value. \n\n\n```yaml\ninputs:\n  type: object\n  properties:\n    token: # example, replace with correct key name for product.\n      title: Token\n      type: string\n      format: password\nmeta:\n  security:\n    name: Authorization\n    in: header\n    format: ApiToken {{token}}\n```\n\n### oauth2_client_credentials\n\nThe optional field `meta.security.token_endpoint` on this example can be used to set a token endpoint which will be \nconcatenated to the `url` input to create the token url.  If you provide a `token_endpoint`, the `token_url` input\nshould not be required.\n\n```yaml\nschema: asset/1\nname: oauth2_client_credentials\ntitle: Oauth 2.0 Client Credentials\ndescription: \'Authenticates using oauth 2.0 client credentials\'\ninputs:\n  type: object\n  properties:\n    url:\n      title: URL\n      description: A URL to the target host.\n      type: string\n      default: https://www.example.com # change if it has a default cloud URL or remove if always custom.\n    token_url:\n      title: Token URL\n      type: string\n      default: https://www.example.com/oauth/token # remove if this is static. Graph API requires tenant ID and would need the user input.\n    client_id:\n      title: Client ID\n      description: The client ID\n      type: string\n    client_secret:\n      title: Client Secret\n      description: The client secret.\n      type: string\n      format: password\n    scope:\n      title: Scope\n      description: Permission scopes for this action.\n      type: array\n      items:\n        type: string\n      default: [] # Add array of scopes we think are needed for the action.\n    verify_ssl:\n      title: Verify SSL Certificates\n      description: Verify SSL certificate\n      type: boolean\n    http_proxy:\n      title: HTTP(s) Proxy\n      description: A proxy to route requests through.\n      type: string\n  required:\n    - url\n    - client_id\n    - client_secret\n    - token_url\nmeta:\n  security:\n    token_endpoint: "api/oauth2/token"\n    type: oauth2\n    flow: client_credentials\n```\n\n### oauth2_password\n\nThe optional field `meta.security.token_endpoint` on this example can be used to set a token endpoint which will be \nconcatenated to the `url` input to create the token url.  If you provide a `token_endpoint`, the `token_url` input\nshould not be required.\n\n```yaml\nschema: asset/1\nname: oauth2_password\ntitle: Oauth 2.0 Password Grant\ndescription: \'Authenticates using oauth 2.0 client credentials\'\ninputs:\n  type: object\n  properties:\n    url:\n      title: URL\n      description: A URL to the target host.\n      type: string\n      default: https://www.example.com # change if it has a default cloud URL or remove if always custom.\n    token_url:\n      title: Token URL\n      type: string\n      default: https://www.example.com/oauth/token # remove if this is static. Graph API requires tenant ID and would need the user input.\n    oauth2_username:\n      title: OAuth2 Username\n      description: The username for authentication\n      type: string\n    oauth2_password:\n      title: OAuth2 Password\n      description: The password for authentication\n      type: string\n      format: password\n    client_id:\n      title: Client ID\n      description: The client ID\n      type: string\n    client_secret:\n      title: Client Secret\n      description: The client secret.\n      type: string\n      format: password\n    scope:\n      title: Scope\n      description: Permission scopes for this action.\n      type: array\n      items:\n        type: string\n      default: [] # Add array of scopes we think are needed for the action.\n    verify_ssl:\n      title: Verify SSL Certificates\n      description: Verify SSL certificate\n      type: boolean\n    http_proxy:\n      title: HTTP(s) Proxy\n      description: A proxy to route requests through.\n      type: string\n  required:\n    - url\n    - oauth2_username\n    - oauth2_password\n    - token_url\nmeta:\n  security:\n    token_endpoint: "api/oauth2/token"\n    type: oauth2\n    flow: password\n```\n\n## Action Schema Structure\n\nThe action schema will include the following fields:\n\n* schema: The schema type, this differentiates the assets from actions.\n* title: The action title\n* name: The action internal name\n* description: The action description\n* inputs: object of input fields\n* output: object with the output fields\n* meta:\n  * endpoint: The HTTP endpoint\n  * method: The HTTP method\n\nExample schema:\n\n```yaml\nschema: action/1\ntitle: Delete Indicator\nname: delete_indicator\ndescription: Deletes an indicator\ninputs:\n  type: object\n  properties:\n    json_body:\n      title: JSON Body\n      type: object\n      properties:\n        id:\n          title: ID\n          type: string\n      required:\n        - id\n      additionalProperties: false\n  required: \n    - json_body\n  additionalProperties: falase\noutput:\n  type: object\n  properties:\n    status_code:\n      title: Status Code\n      type: number\nmeta:\n  endpoint: iocs/entities/indicators/v1\n  method: DELETE\n```\n### Inputs\n\nThe `inputs` field must be an object type with the following properties:\n\n* headers: Headers to send with the request.\n* parameters: Parameters to send in the query string for the request.\n* data_body: Raw data send in the body of the request.\n* json_body: JSON data to send in the body of the request.\n* files: Object or array of objects with the `contentDisposition: attachment` property. \n* path_parameters: Parameters to be replaced in the URL.\n\n\n**Path Parameters**\n\nYou can use mustaches to build the URL path based in the `path_parameters` values. For example, if you have the following URL:\n\n```\nhttps://api.crowdstrikefalcon/{{session_id}}/download/{{filename}}\n```\n\nand the following `path_parameters` inputs field:\n\n\n```yaml\ninputs:\n  type: object\n  properties:\n    path_parameters:\n      title: Path Parameters\n      type: object\n      properties:\n        session_id:\n          title: Session ID\n          type: string\n        filename:\n          title: File Name\n          type: string\n      required:\n        - session_id\n        - filename\n```\n\nThen the endpoint will be formatted using the input data in the `path_parameters` object.\n\n\n**Files**\n\n`files` inputs could also include binary inputs and additional properties for the file. For example:\n\n```yaml\ninputs:\n  type: object\n  properties:\n    attachments:\n      title: Attachments\n      type: array\n      items:\n        contentDisposition: attachment\n        type: object\n        additionalProperties: false\n        properties:\n          file:\n            type: string\n            format: binary\n          file_name:\n            type: string\n      examples: []\n  required:\n    - attachments\n  additionalProperties: true\n```\n\nYou can also add additional properties into the attachment properties. For example, if you need to replicate the following\ncode:\n\n```python\nimport requests\n\nheaders = {\n    \'accept\': \'application/json\'\n}\n\nfiles = [\n    (\'file\', open(\'decode.py;type=text/x-python-script\', \'rb\')),\n    (\'permission_type\', (None, \'public\')),\n    (\'platform\', (None, \'linux\'))\n]\n\nresponse = requests.post(\'https://api.crowdstrike.com/real-time-response/entities/scripts/v1\', headers=headers, files=files)\n```\n\nYou can use the following input:\n\n```yaml\ninputs:\n  type: object\n  properties:\n    attachments:\n      title: Attachments\n      type: array\n      items:\n        contentDisposition: attachment\n        type: object\n        additionalProperties: false\n        properties:\n          file:\n            type: string\n            format: binary\n          file_name:\n            type: string\n          permission_type:\n            type: string\n          platform:\n            type: string\n  required:\n    - attachments\n  additionalProperties: true\n```\n\n\n### Outputs\n\nThe `output` field works similar to `inputs`, with the difference that it could be an array instead of an object.\n\nIt will contain the following properties:\n\n* `status_code`: The status code of the response.\n* `response_headers`: The headers of the response.\n* `reason`: A text corresponding to the status code. For example, OK for 200, Not Found for 404.\n* `json_body`: A JSON object of the response\n* `response_text`: If the response doesn\'t contain a JSON body and there is no `file` property defined in the manifest, the response body will be returned in text format. \n* `file`: Object or array of objects with the `contentDisposition: attachment` property.\n\nIn order to get files as output, you must manually add a file property to the output section. See the following example:\n\n```yaml\noutput:\n  type: object\n  properties:\n    attachments:\n      title: Attachments\n      type: array\n      items:\n        contentDisposition: attachment\n        type: object\n        additionalProperties: false\n        properties:\n          file:\n            type: string\n            format: binary\n          filename:\n            type: string\n  additionalProperties: true\n```\n\n## Custom Action\n\nIn order to add custom actions, you must create a `.py` file and its file name must match with the corresponding manifest.\nThe source code must have a `RunnerOverride` class with the following interface:\n\n```python\nclass RunnerOverride:\n\n  def __init__(self, asset=asset, asset_schema=asset_schema, http_proxy=http_proxy):\n    pass\n\n  def run(self, inputs=inputs, action_schema=action_schema):\n    pass\n```\n\nFor example, suppose you want to create an action that makes an add operation using LDAP protocol. \nYou can create the following schema in `connector/config/actions/add.yaml`:\n\n```yaml\nschema: action/1\ntitle: Add\nname: add\ndescription: >-\n  The Add operation allows a client to request the addition of an entry into the\n  LDAP directory.\ninputs:\n  type: object\n  properties:\n    dn:\n      title: Dn\n      examples:\n        - CN=Charles,OU=friends,DC=testdomain,DC=local\n      type: string\n    object_class:\n      title: Object Class\n      examples:\n        - - person\n      type: array\n      items:\n        type: string\n    attributes:\n      title: Attributes\n      examples:\n        - name: Charles Darwin\n      type: object\n      properties:\n        name:\n          title: Name\n          examples:\n            - Charles Darwin\n          type: string\n      required: []\n      additionalProperties: true\n  required:\n    - dn\n    - object_class\n  additionalProperties: true\noutput:\n  type: object\n  properties:\n    result:\n      title: Result\n      type: number\n    description:\n      title: Description\n      type: string\n    dn:\n      title: Dn\n      type: string\n    message:\n      title: Message\n      type: string\n    referrals:\n      title: Referrals\n      type: object\n      properties: {}\n      required: []\n      additionalProperties: true\n    type:\n      title: Type\n      type: string\n  required: []\n  additionalProperties: true\nmeta: {}\n```\n\nAnd then create the file `connector/src/add.py` with the source code:\n\n```python\n\nimport json\nimport os\nfrom ldap3 import Server, Connection\n\n\nclass RunnerOverride(LdapActionBasic):\n\n    def __init__(self, asset=asset, asset_schema=asset_schema, http_proxy=None):\n        self.server = Server(asset["ip"],\n                            use_ssl=asset.get("verify_ssl", True),\n                            connect_timeout=asset.get("connect_timeout", 10))\n        self.conn = Connection(self.server,\n                              asset["username"],\n                              asset["password"],\n                              auto_bind=True)\n\n    def run(self, inputs=inputs, action_schema=None):\n        self.conn.add(inputs)\n        return self.conn.result\n```\n\n## Custom Authentication\n\nIn order to add custom authentication, you must create a `runner_override.py` file.\nThe file must have a `RunnerOverride` class with the following interface:\n\n```python\nclass RunnerOverride:\n\n  def __init__(self, asset=asset, asset_schema=asset_schema, http_proxy=http_proxy):\n    pass\n\n  def run(self, inputs=inputs, action_schema=action_schema):\n    pass\n```\nWhen this file is present, all actions will use it.\n\n\n## Custom headers\n\nYou can build your own header field templates using mustache syntax. For example\nif you need to define a custom header field like: `auth: my_username:my_password` for all your actions, you can define the following asset schema:\n\n```yaml\nschema: asset/1\ntitle: HTTP Custom Authentication\ndescription: \'Authenticates using username and password.\'\ninputs:\n  type: object\n  properties:\n    url:\n      title: URL\n      description: A URL to the target host.\n      type: string\n      default: https://www.example.com # change if it has a default cloud URL or remove if always custom.\n    username:\n      title: Username\n      description: Username\n      type: string\n    password:\n      title: Password\n      description: Password\n      type: string\n      format: password\n  required:\n    - url\n    - username\n    - password\nmeta:\n  headers:\n    auth: {{username}}:{{password}}\n```\n\nThis way the username and password will be replaced in the `meta.headers.auth` field. You can also define the same way custom headers for action schemas.',
    'author': 'Swimlane',
    'author_email': 'integrations@swimlane.com',
    'maintainer': 'Swimlane',
    'maintainer_email': 'None',
    'url': 'https://github.com/swimlane/connector-definition-runner',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
