from .....Internal.Core import Core
from .....Internal.CommandsGroup import CommandsGroup
from .....Internal import Conversions


# noinspection PyPep8Naming,PyAttributeOutsideInit,SpellCheckingInspection
class FrequencyCls:
	"""Frequency commands group definition. 2 total commands, 1 Subgroups, 1 group commands"""

	def __init__(self, core: Core, parent):
		self._core = core
		self._cmd_group = CommandsGroup("frequency", core, parent)

	@property
	def step(self):
		"""step commands group. 0 Sub-classes, 1 commands."""
		if not hasattr(self, '_step'):
			from .Step import StepCls
			self._step = StepCls(self._core, self._cmd_group)
		return self._step

	def set(self, frequency: float) -> None:
		"""SCPI: SOURce:GENerator:FREQuency \n
		Snippet: driver.source.generator.frequency.set(frequency = 1.0) \n
		This command defines the frequency of the signal that is generated by the signal source. \n
			:param frequency: numeric value Unit: Hz
		"""
		param = Conversions.decimal_value_to_str(frequency)
		self._core.io.write(f'SOURce:GENerator:FREQuency {param}')

	def get(self) -> float:
		"""SCPI: SOURce:GENerator:FREQuency \n
		Snippet: value: float = driver.source.generator.frequency.get() \n
		This command defines the frequency of the signal that is generated by the signal source. \n
			:return: frequency: numeric value Unit: Hz"""
		response = self._core.io.query_str(f'SOURce:GENerator:FREQuency?')
		return Conversions.str_to_float(response)

	def clone(self) -> 'FrequencyCls':
		"""Clones the group by creating new object from it and its whole existing subgroups
		Also copies all the existing default Repeated Capabilities setting,
		which you can change independently without affecting the original group"""
		new_group = FrequencyCls(self._core, self._cmd_group.parent)
		self._cmd_group.synchronize_repcaps(new_group)
		return new_group
