"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserPool = exports.AccountRecovery = exports.Mfa = exports.VerificationEmailStyle = exports.UserPoolOperation = void 0;
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const cognito_generated_1 = require("./cognito.generated");
const attr_names_1 = require("./private/attr-names");
const user_pool_client_1 = require("./user-pool-client");
const user_pool_domain_1 = require("./user-pool-domain");
/**
 * (experimental) User pool operations to which lambda triggers can be attached.
 *
 * @experimental
 */
class UserPoolOperation {
    constructor(operationName) {
        this.operationName = operationName;
    }
    /**
     * (experimental) A custom user pool operation.
     *
     * @experimental
     */
    static of(name) {
        const lowerCamelCase = name.charAt(0).toLowerCase() + name.slice(1);
        return new UserPoolOperation(lowerCamelCase);
    }
}
exports.UserPoolOperation = UserPoolOperation;
/**
 * (experimental) Creates a challenge in a custom auth flow.
 *
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-create-auth-challenge.html
 * @experimental
 */
UserPoolOperation.CREATE_AUTH_CHALLENGE = new UserPoolOperation('createAuthChallenge');
/**
 * (experimental) Advanced customization and localization of messages.
 *
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-custom-message.html
 * @experimental
 */
UserPoolOperation.CUSTOM_MESSAGE = new UserPoolOperation('customMessage');
/**
 * (experimental) Determines the next challenge in a custom auth flow.
 *
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-define-auth-challenge.html
 * @experimental
 */
UserPoolOperation.DEFINE_AUTH_CHALLENGE = new UserPoolOperation('defineAuthChallenge');
/**
 * (experimental) Event logging for custom analytics.
 *
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-post-authentication.html
 * @experimental
 */
UserPoolOperation.POST_AUTHENTICATION = new UserPoolOperation('postAuthentication');
/**
 * (experimental) Custom welcome messages or event logging for custom analytics.
 *
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-post-confirmation.html
 * @experimental
 */
UserPoolOperation.POST_CONFIRMATION = new UserPoolOperation('postConfirmation');
/**
 * (experimental) Custom validation to accept or deny the sign-in request.
 *
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-authentication.html
 * @experimental
 */
UserPoolOperation.PRE_AUTHENTICATION = new UserPoolOperation('preAuthentication');
/**
 * (experimental) Custom validation to accept or deny the sign-up request.
 *
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-sign-up.html
 * @experimental
 */
UserPoolOperation.PRE_SIGN_UP = new UserPoolOperation('preSignUp');
/**
 * (experimental) Add or remove attributes in Id tokens.
 *
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-token-generation.html
 * @experimental
 */
UserPoolOperation.PRE_TOKEN_GENERATION = new UserPoolOperation('preTokenGeneration');
/**
 * (experimental) Migrate a user from an existing user directory to user pools.
 *
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-migrate-user.html
 * @experimental
 */
UserPoolOperation.USER_MIGRATION = new UserPoolOperation('userMigration');
/**
 * (experimental) Determines if a response is correct in a custom auth flow.
 *
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-verify-auth-challenge-response.html
 * @experimental
 */
UserPoolOperation.VERIFY_AUTH_CHALLENGE_RESPONSE = new UserPoolOperation('verifyAuthChallengeResponse');
/**
 * (experimental) The email verification style.
 *
 * @experimental
 */
var VerificationEmailStyle;
(function (VerificationEmailStyle) {
    VerificationEmailStyle["CODE"] = "CONFIRM_WITH_CODE";
    VerificationEmailStyle["LINK"] = "CONFIRM_WITH_LINK";
})(VerificationEmailStyle = exports.VerificationEmailStyle || (exports.VerificationEmailStyle = {}));
/**
 * (experimental) The different ways in which a user pool's MFA enforcement can be configured.
 *
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-mfa.html
 * @experimental
 */
var Mfa;
(function (Mfa) {
    Mfa["OFF"] = "OFF";
    Mfa["OPTIONAL"] = "OPTIONAL";
    Mfa["REQUIRED"] = "ON";
})(Mfa = exports.Mfa || (exports.Mfa = {}));
/**
 * (experimental) How will a user be able to recover their account?
 *
 * When a user forgets their password, they can have a code sent to their verified email or verified phone to recover their account.
 * You can choose the preferred way to send codes below.
 * We recommend not allowing phone to be used for both password resets and multi-factor authentication (MFA).
 *
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/how-to-recover-a-user-account.html
 * @experimental
 */
var AccountRecovery;
(function (AccountRecovery) {
    AccountRecovery[AccountRecovery["EMAIL_AND_PHONE_WITHOUT_MFA"] = 0] = "EMAIL_AND_PHONE_WITHOUT_MFA";
    AccountRecovery[AccountRecovery["PHONE_WITHOUT_MFA_AND_EMAIL"] = 1] = "PHONE_WITHOUT_MFA_AND_EMAIL";
    AccountRecovery[AccountRecovery["EMAIL_ONLY"] = 2] = "EMAIL_ONLY";
    AccountRecovery[AccountRecovery["PHONE_ONLY_WITHOUT_MFA"] = 3] = "PHONE_ONLY_WITHOUT_MFA";
    AccountRecovery[AccountRecovery["PHONE_AND_EMAIL"] = 4] = "PHONE_AND_EMAIL";
    AccountRecovery[AccountRecovery["NONE"] = 5] = "NONE";
})(AccountRecovery = exports.AccountRecovery || (exports.AccountRecovery = {}));
class UserPoolBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        /**
         * (experimental) Get all identity providers registered with this user pool.
         *
         * @experimental
         */
        this.identityProviders = [];
    }
    /**
     * (experimental) Add a new app client to this user pool.
     *
     * @experimental
     */
    addClient(id, options) {
        return new user_pool_client_1.UserPoolClient(this, id, {
            userPool: this,
            ...options,
        });
    }
    /**
     * (experimental) Associate a domain to this user pool.
     *
     * @experimental
     */
    addDomain(id, options) {
        return new user_pool_domain_1.UserPoolDomain(this, id, {
            userPool: this,
            ...options,
        });
    }
    /**
     * (experimental) Register an identity provider with this user pool.
     *
     * @experimental
     */
    registerIdentityProvider(provider) {
        this.identityProviders.push(provider);
    }
}
/**
 * (experimental) Define a Cognito User Pool.
 *
 * @experimental
 */
class UserPool extends UserPoolBase {
    /**
     * @experimental
     */
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d, _e;
        super(scope, id);
        this.triggers = {};
        const signIn = this.signInConfiguration(props);
        if (props.lambdaTriggers) {
            for (const t of Object.keys(props.lambdaTriggers)) {
                const trigger = props.lambdaTriggers[t];
                if (trigger !== undefined) {
                    this.addLambdaPermission(trigger, t);
                    this.triggers[t] = trigger.functionArn;
                }
            }
        }
        const verificationMessageTemplate = this.verificationMessageConfiguration(props);
        let emailVerificationMessage;
        let emailVerificationSubject;
        if (verificationMessageTemplate.defaultEmailOption === VerificationEmailStyle.CODE) {
            emailVerificationMessage = verificationMessageTemplate.emailMessage;
            emailVerificationSubject = verificationMessageTemplate.emailSubject;
        }
        const smsVerificationMessage = verificationMessageTemplate.smsMessage;
        const inviteMessageTemplate = {
            emailMessage: (_a = props.userInvitation) === null || _a === void 0 ? void 0 : _a.emailBody,
            emailSubject: (_b = props.userInvitation) === null || _b === void 0 ? void 0 : _b.emailSubject,
            smsMessage: (_c = props.userInvitation) === null || _c === void 0 ? void 0 : _c.smsMessage,
        };
        const selfSignUpEnabled = props.selfSignUpEnabled !== undefined ? props.selfSignUpEnabled : false;
        const adminCreateUserConfig = {
            allowAdminCreateUserOnly: !selfSignUpEnabled,
            inviteMessageTemplate: props.userInvitation !== undefined ? inviteMessageTemplate : undefined,
        };
        const passwordPolicy = this.configurePasswordPolicy(props);
        const userPool = new cognito_generated_1.CfnUserPool(this, 'Resource', {
            userPoolName: props.userPoolName,
            usernameAttributes: signIn.usernameAttrs,
            aliasAttributes: signIn.aliasAttrs,
            autoVerifiedAttributes: signIn.autoVerifyAttrs,
            lambdaConfig: core_1.Lazy.anyValue({ produce: () => undefinedIfNoKeys(this.triggers) }),
            smsConfiguration: this.smsConfiguration(props),
            adminCreateUserConfig,
            emailVerificationMessage,
            emailVerificationSubject,
            smsVerificationMessage,
            verificationMessageTemplate,
            schema: this.schemaConfiguration(props),
            mfaConfiguration: props.mfa,
            enabledMfas: this.mfaConfiguration(props),
            policies: passwordPolicy !== undefined ? { passwordPolicy } : undefined,
            emailConfiguration: undefinedIfNoKeys({
                from: (_d = props.emailSettings) === null || _d === void 0 ? void 0 : _d.from,
                replyToEmailAddress: (_e = props.emailSettings) === null || _e === void 0 ? void 0 : _e.replyTo,
            }),
            usernameConfiguration: undefinedIfNoKeys({
                caseSensitive: props.signInCaseSensitive,
            }),
            accountRecoverySetting: this.accountRecovery(props),
        });
        this.userPoolId = userPool.ref;
        this.userPoolArn = userPool.attrArn;
        this.userPoolProviderName = userPool.attrProviderName;
        this.userPoolProviderUrl = userPool.attrProviderUrl;
    }
    /**
     * (experimental) Import an existing user pool based on its id.
     *
     * @experimental
     */
    static fromUserPoolId(scope, id, userPoolId) {
        class Import extends UserPoolBase {
            constructor() {
                super(...arguments);
                this.userPoolId = userPoolId;
                this.userPoolArn = core_1.Stack.of(this).formatArn({
                    service: 'cognito-idp',
                    resource: 'userpool',
                    resourceName: userPoolId,
                });
            }
        }
        return new Import(scope, id);
    }
    /**
     * (experimental) Import an existing user pool based on its ARN.
     *
     * @experimental
     */
    static fromUserPoolArn(scope, id, userPoolArn) {
        return UserPool.fromUserPoolId(scope, id, core_1.Stack.of(scope).parseArn(userPoolArn).resourceName);
    }
    /**
     * (experimental) Add a lambda trigger to a user pool operation.
     *
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html
     * @experimental
     */
    addTrigger(operation, fn) {
        if (operation.operationName in this.triggers) {
            throw new Error(`A trigger for the operation ${operation} already exists.`);
        }
        this.addLambdaPermission(fn, operation.operationName);
        this.triggers[operation.operationName] = fn.functionArn;
    }
    addLambdaPermission(fn, name) {
        const capitalize = name.charAt(0).toUpperCase() + name.slice(1);
        fn.addPermission(`${capitalize}Cognito`, {
            principal: new aws_iam_1.ServicePrincipal('cognito-idp.amazonaws.com'),
            sourceArn: this.userPoolArn,
        });
    }
    verificationMessageConfiguration(props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k;
        const CODE_TEMPLATE = '{####}';
        const VERIFY_EMAIL_TEMPLATE = '{##Verify Email##}';
        const emailStyle = (_b = (_a = props.userVerification) === null || _a === void 0 ? void 0 : _a.emailStyle) !== null && _b !== void 0 ? _b : VerificationEmailStyle.CODE;
        const emailSubject = (_d = (_c = props.userVerification) === null || _c === void 0 ? void 0 : _c.emailSubject) !== null && _d !== void 0 ? _d : 'Verify your new account';
        const smsMessage = (_f = (_e = props.userVerification) === null || _e === void 0 ? void 0 : _e.smsMessage) !== null && _f !== void 0 ? _f : `The verification code to your new account is ${CODE_TEMPLATE}`;
        if (emailStyle === VerificationEmailStyle.CODE) {
            const emailMessage = (_h = (_g = props.userVerification) === null || _g === void 0 ? void 0 : _g.emailBody) !== null && _h !== void 0 ? _h : `The verification code to your new account is ${CODE_TEMPLATE}`;
            if (!core_1.Token.isUnresolved(emailMessage) && emailMessage.indexOf(CODE_TEMPLATE) < 0) {
                throw new Error(`Verification email body must contain the template string '${CODE_TEMPLATE}'`);
            }
            if (!core_1.Token.isUnresolved(smsMessage) && smsMessage.indexOf(CODE_TEMPLATE) < 0) {
                throw new Error(`SMS message must contain the template string '${CODE_TEMPLATE}'`);
            }
            return {
                defaultEmailOption: VerificationEmailStyle.CODE,
                emailMessage,
                emailSubject,
                smsMessage,
            };
        }
        else {
            const emailMessage = (_k = (_j = props.userVerification) === null || _j === void 0 ? void 0 : _j.emailBody) !== null && _k !== void 0 ? _k : `Verify your account by clicking on ${VERIFY_EMAIL_TEMPLATE}`;
            if (!core_1.Token.isUnresolved(emailMessage) && emailMessage.indexOf(VERIFY_EMAIL_TEMPLATE) < 0) {
                throw new Error(`Verification email body must contain the template string '${VERIFY_EMAIL_TEMPLATE}'`);
            }
            return {
                defaultEmailOption: VerificationEmailStyle.LINK,
                emailMessageByLink: emailMessage,
                emailSubjectByLink: emailSubject,
                smsMessage,
            };
        }
    }
    signInConfiguration(props) {
        var _a;
        let aliasAttrs;
        let usernameAttrs;
        let autoVerifyAttrs;
        const signIn = (_a = props.signInAliases) !== null && _a !== void 0 ? _a : { username: true };
        if (signIn.preferredUsername && !signIn.username) {
            throw new Error('username signIn must be enabled if preferredUsername is enabled');
        }
        if (signIn.username) {
            aliasAttrs = [];
            if (signIn.email) {
                aliasAttrs.push(attr_names_1.StandardAttributeNames.email);
            }
            if (signIn.phone) {
                aliasAttrs.push(attr_names_1.StandardAttributeNames.phoneNumber);
            }
            if (signIn.preferredUsername) {
                aliasAttrs.push(attr_names_1.StandardAttributeNames.preferredUsername);
            }
            if (aliasAttrs.length === 0) {
                aliasAttrs = undefined;
            }
        }
        else {
            usernameAttrs = [];
            if (signIn.email) {
                usernameAttrs.push(attr_names_1.StandardAttributeNames.email);
            }
            if (signIn.phone) {
                usernameAttrs.push(attr_names_1.StandardAttributeNames.phoneNumber);
            }
        }
        if (props.autoVerify) {
            autoVerifyAttrs = [];
            if (props.autoVerify.email) {
                autoVerifyAttrs.push(attr_names_1.StandardAttributeNames.email);
            }
            if (props.autoVerify.phone) {
                autoVerifyAttrs.push(attr_names_1.StandardAttributeNames.phoneNumber);
            }
        }
        else if (signIn.email || signIn.phone) {
            autoVerifyAttrs = [];
            if (signIn.email) {
                autoVerifyAttrs.push(attr_names_1.StandardAttributeNames.email);
            }
            if (signIn.phone) {
                autoVerifyAttrs.push(attr_names_1.StandardAttributeNames.phoneNumber);
            }
        }
        return { usernameAttrs, aliasAttrs, autoVerifyAttrs };
    }
    smsConfiguration(props) {
        var _a, _b, _c;
        if (props.enableSmsRole === false && props.smsRole) {
            throw new Error('enableSmsRole cannot be disabled when smsRole is specified');
        }
        if (props.smsRole) {
            return {
                snsCallerArn: props.smsRole.roleArn,
                externalId: props.smsRoleExternalId,
            };
        }
        if (props.enableSmsRole === false) {
            return undefined;
        }
        const mfaConfiguration = this.mfaConfiguration(props);
        const phoneVerification = ((_a = props.signInAliases) === null || _a === void 0 ? void 0 : _a.phone) === true || ((_b = props.autoVerify) === null || _b === void 0 ? void 0 : _b.phone) === true;
        const roleRequired = (mfaConfiguration === null || mfaConfiguration === void 0 ? void 0 : mfaConfiguration.includes('SMS_MFA')) || phoneVerification;
        if (!roleRequired && props.enableSmsRole === undefined) {
            return undefined;
        }
        const smsRoleExternalId = this.node.uniqueId.substr(0, 1223); // sts:ExternalId max length of 1224
        const smsRole = (_c = props.smsRole) !== null && _c !== void 0 ? _c : new aws_iam_1.Role(this, 'smsRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('cognito-idp.amazonaws.com', {
                conditions: {
                    StringEquals: { 'sts:ExternalId': smsRoleExternalId },
                },
            }),
            inlinePolicies: {
                /*
                  * The UserPool is very particular that it must contain an 'sns:Publish' action as an inline policy.
                  * Ideally, a conditional that restricts this action to 'sms' protocol needs to be attached, but the UserPool deployment fails validation.
                  * Seems like a case of being excessively strict.
                  */
                'sns-publish': new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            actions: ['sns:Publish'],
                            resources: ['*'],
                        }),
                    ],
                }),
            },
        });
        return {
            externalId: smsRoleExternalId,
            snsCallerArn: smsRole.roleArn,
        };
    }
    mfaConfiguration(props) {
        if (props.mfa === undefined || props.mfa === Mfa.OFF) {
            // since default is OFF, treat undefined and OFF the same way
            return undefined;
        }
        else if (props.mfaSecondFactor === undefined &&
            (props.mfa === Mfa.OPTIONAL || props.mfa === Mfa.REQUIRED)) {
            return ['SMS_MFA'];
        }
        else {
            const enabledMfas = [];
            if (props.mfaSecondFactor.sms) {
                enabledMfas.push('SMS_MFA');
            }
            if (props.mfaSecondFactor.otp) {
                enabledMfas.push('SOFTWARE_TOKEN_MFA');
            }
            return enabledMfas;
        }
    }
    configurePasswordPolicy(props) {
        var _a, _b, _c, _d, _e, _f;
        const tempPasswordValidity = (_a = props.passwordPolicy) === null || _a === void 0 ? void 0 : _a.tempPasswordValidity;
        if (tempPasswordValidity !== undefined && tempPasswordValidity.toDays() > core_1.Duration.days(365).toDays()) {
            throw new Error(`tempPasswordValidity cannot be greater than 365 days (received: ${tempPasswordValidity.toDays()})`);
        }
        const minLength = props.passwordPolicy ? (_b = props.passwordPolicy.minLength) !== null && _b !== void 0 ? _b : 8 : undefined;
        if (minLength !== undefined && (minLength < 6 || minLength > 99)) {
            throw new Error(`minLength for password must be between 6 and 99 (received: ${minLength})`);
        }
        return undefinedIfNoKeys({
            temporaryPasswordValidityDays: tempPasswordValidity === null || tempPasswordValidity === void 0 ? void 0 : tempPasswordValidity.toDays({ integral: true }),
            minimumLength: minLength,
            requireLowercase: (_c = props.passwordPolicy) === null || _c === void 0 ? void 0 : _c.requireLowercase,
            requireUppercase: (_d = props.passwordPolicy) === null || _d === void 0 ? void 0 : _d.requireUppercase,
            requireNumbers: (_e = props.passwordPolicy) === null || _e === void 0 ? void 0 : _e.requireDigits,
            requireSymbols: (_f = props.passwordPolicy) === null || _f === void 0 ? void 0 : _f.requireSymbols,
        });
    }
    schemaConfiguration(props) {
        const schema = [];
        if (props.standardAttributes) {
            const stdAttributes = Object.entries(props.standardAttributes)
                .filter(([, attr]) => !!attr)
                .map(([attrName, attr]) => {
                var _a, _b;
                return ({
                    name: attr_names_1.StandardAttributeNames[attrName],
                    mutable: (_a = attr.mutable) !== null && _a !== void 0 ? _a : true,
                    required: (_b = attr.required) !== null && _b !== void 0 ? _b : false,
                });
            });
            schema.push(...stdAttributes);
        }
        if (props.customAttributes) {
            const customAttrs = Object.keys(props.customAttributes).map((attrName) => {
                var _a, _b, _c, _d, _e, _f, _g, _h;
                const attrConfig = props.customAttributes[attrName].bind();
                const numberConstraints = {
                    minValue: (_b = (_a = attrConfig.numberConstraints) === null || _a === void 0 ? void 0 : _a.min) === null || _b === void 0 ? void 0 : _b.toString(),
                    maxValue: (_d = (_c = attrConfig.numberConstraints) === null || _c === void 0 ? void 0 : _c.max) === null || _d === void 0 ? void 0 : _d.toString(),
                };
                const stringConstraints = {
                    minLength: (_f = (_e = attrConfig.stringConstraints) === null || _e === void 0 ? void 0 : _e.minLen) === null || _f === void 0 ? void 0 : _f.toString(),
                    maxLength: (_h = (_g = attrConfig.stringConstraints) === null || _g === void 0 ? void 0 : _g.maxLen) === null || _h === void 0 ? void 0 : _h.toString(),
                };
                return {
                    name: attrName,
                    attributeDataType: attrConfig.dataType,
                    numberAttributeConstraints: attrConfig.numberConstraints
                        ? numberConstraints
                        : undefined,
                    stringAttributeConstraints: attrConfig.stringConstraints
                        ? stringConstraints
                        : undefined,
                    mutable: attrConfig.mutable,
                };
            });
            schema.push(...customAttrs);
        }
        if (schema.length === 0) {
            return undefined;
        }
        return schema;
    }
    accountRecovery(props) {
        var _a;
        const accountRecovery = (_a = props.accountRecovery) !== null && _a !== void 0 ? _a : AccountRecovery.PHONE_WITHOUT_MFA_AND_EMAIL;
        switch (accountRecovery) {
            case AccountRecovery.EMAIL_AND_PHONE_WITHOUT_MFA:
                return {
                    recoveryMechanisms: [
                        { name: 'verified_email', priority: 1 },
                        { name: 'verified_phone_number', priority: 2 },
                    ],
                };
            case AccountRecovery.PHONE_WITHOUT_MFA_AND_EMAIL:
                return {
                    recoveryMechanisms: [
                        { name: 'verified_phone_number', priority: 1 },
                        { name: 'verified_email', priority: 2 },
                    ],
                };
            case AccountRecovery.EMAIL_ONLY:
                return {
                    recoveryMechanisms: [{ name: 'verified_email', priority: 1 }],
                };
            case AccountRecovery.PHONE_ONLY_WITHOUT_MFA:
                return {
                    recoveryMechanisms: [{ name: 'verified_phone_number', priority: 1 }],
                };
            case AccountRecovery.NONE:
                return {
                    recoveryMechanisms: [{ name: 'admin_only', priority: 1 }],
                };
            case AccountRecovery.PHONE_AND_EMAIL:
                return undefined;
            default:
                throw new Error(`Unsupported AccountRecovery type - ${accountRecovery}`);
        }
    }
}
exports.UserPool = UserPool;
function undefinedIfNoKeys(struct) {
    const allUndefined = Object.values(struct).reduce((acc, v) => acc && (v === undefined), true);
    return allUndefined ? undefined : struct;
}
//# sourceMappingURL=data:application/json;base64,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