from __future__ import annotations

import enum
from typing import NamedTuple

import numpy


def _central_angle(x: Location, y: Location) -> float:
    """Computes the central angle between two locations.

    Uses the Haversine Formula (see https://en.wikipedia.org/wiki/Haversine_formula) to compute the central angle between
    two locations.

    Args:
        x: the first location
        y: the second location

    Returns:
        the central angle between two points on a sphere in radians
    """
    x_latitude = numpy.radians(x.latitude)
    x_longitude = numpy.radians(x.longitude)
    y_latitude = numpy.radians(y.latitude)
    y_longitude = numpy.radians(y.longitude)

    central_angle = 2 * numpy.arcsin(
        numpy.sqrt(
            numpy.sin((y_latitude - x_latitude) / 2) ** 2
            + numpy.cos(x_latitude) * numpy.cos(y_latitude) * numpy.sin((y_longitude - x_longitude) / 2) ** 2
        )
    )

    return central_angle


class Location(NamedTuple):
    latitude: float
    longitude: float


@enum.unique
class MetarStation(Location, enum.Enum):
    """Enum of METAR stations globally with their locations.

    For more information on the available METAR stations see: https://www.aviationweather.gov/metar.
    """

    DIAD = Location(latitude=5.283, longitude=-3.283)
    DIAP = Location(latitude=5.25, longitude=-3.917)
    DIBK = Location(latitude=7.717, longitude=-5.067)
    DIBU = Location(latitude=8.05, longitude=-2.767)
    DIDK = Location(latitude=6.65, longitude=-4.7)
    DIDL = Location(latitude=6.867, longitude=-6.467)
    DIGA = Location(latitude=6.117, longitude=-5.95)
    DIKO = Location(latitude=9.417, longitude=-5.617)
    DISP = Location(latitude=4.75, longitude=-6.65)
    DISS = Location(latitude=4.95, longitude=-6.067)
    DITB = Location(latitude=4.417, longitude=-7.367)
    DIYO = Location(latitude=6.9, longitude=-5.333)
    DGAA = Location(latitude=5.583, longitude=-0.167)
    HAAB = Location(latitude=8.967, longitude=38.783)
    DAAD = Location(latitude=35.317, longitude=4.2)
    DAAE = Location(latitude=36.717, longitude=5.067)
    DAAG = Location(latitude=36.717, longitude=3.217)
    DAAJ = Location(latitude=24.283, longitude=9.467)
    DAAP = Location(latitude=26.717, longitude=8.617)
    DAAS = Location(latitude=36.183, longitude=5.333)
    DAAT = Location(latitude=22.817, longitude=5.467)
    DAAV = Location(latitude=36.8, longitude=5.883)
    DAAY = Location(latitude=33.533, longitude=-0.25)
    DABB = Location(latitude=36.817, longitude=7.817)
    DABC = Location(latitude=36.267, longitude=6.617)
    DABS = Location(latitude=35.433, longitude=8.117)
    DABT = Location(latitude=35.75, longitude=6.317)
    DAFH = Location(latitude=32.933, longitude=3.317)
    DAOB = Location(latitude=35.333, longitude=1.467)
    DAOF = Location(latitude=27.7, longitude=-8.167)
    DAOI = Location(latitude=36.217, longitude=1.317)
    DAOL = Location(latitude=35.533, longitude=-0.517)
    DAON = Location(latitude=35.017, longitude=-1.467)
    DAOO = Location(latitude=35.617, longitude=-0.6)
    DAOR = Location(latitude=31.65, longitude=-2.267)
    DAOV = Location(latitude=35.2, longitude=0.15)
    DATM = Location(latitude=21.383, longitude=0.917)
    DAUA = Location(latitude=27.833, longitude=-0.183)
    DAUB = Location(latitude=34.783, longitude=5.717)
    DAUE = Location(latitude=30.567, longitude=2.867)
    DAUG = Location(latitude=32.367, longitude=3.817)
    DAUH = Location(latitude=31.667, longitude=6.15)
    DAUI = Location(latitude=27.25, longitude=2.517)
    DAUK = Location(latitude=33.083, longitude=6.117)
    DAUL = Location(latitude=33.767, longitude=2.917)
    DAUO = Location(latitude=33.5, longitude=6.783)
    DAUT = Location(latitude=29.25, longitude=0.267)
    DAUU = Location(latitude=31.917, longitude=5.4)
    DAUZ = Location(latitude=28.05, longitude=9.617)
    DTTR = Location(latitude=31.7, longitude=9.267)
    LEEC = Location(latitude=37.317, longitude=-6)
    GABS = Location(latitude=12.517, longitude=-7.95)
    GAMB = Location(latitude=14.517, longitude=-4.083)
    GANR = Location(latitude=15.233, longitude=-9.583)
    FEFF = Location(latitude=4.4, longitude=18.517)
    GBYD = Location(latitude=13.333, longitude=-16.65)
    GGOV = Location(latitude=11.867, longitude=-15.65)
    FWCL = Location(latitude=-15.667, longitude=34.967)
    FWKI = Location(latitude=-13.767, longitude=33.767)
    FCBB = Location(latitude=-4.25, longitude=15.25)
    FCPP = Location(latitude=-4.817, longitude=11.9)
    HBBA = Location(latitude=-3.317, longitude=29.317)
    HEAR = Location(latitude=31.067, longitude=33.817)
    HEAT = Location(latitude=27.05, longitude=31.017)
    HEAX = Location(latitude=31.183, longitude=29.933)
    HEBA = Location(latitude=30.917, longitude=29.7)
    HEBL = Location(latitude=22.383, longitude=31.617)
    HECA = Location(latitude=30.117, longitude=31.383)
    HEGN = Location(latitude=27.183, longitude=33.8)
    HELX = Location(latitude=25.667, longitude=32.7)
    HEMA = Location(latitude=25.55, longitude=34.583)
    HEMM = Location(latitude=31.317, longitude=27.217)
    HEPS = Location(latitude=31.283, longitude=32.25)
    HESC = Location(latitude=28.683, longitude=34.067)
    HESH = Location(latitude=27.967, longitude=34.383)
    HESN = Location(latitude=23.967, longitude=32.817)
    HETB = Location(latitude=29.583, longitude=34.783)
    GEML = Location(latitude=35.267, longitude=-2.95)
    GMAD = Location(latitude=30.317, longitude=-9.417)
    GMAT = Location(latitude=28.433, longitude=-11.15)
    GMFF = Location(latitude=33.917, longitude=-4.967)
    GMFK = Location(latitude=31.95, longitude=-4.4)
    GMFM = Location(latitude=33.867, longitude=-5.517)
    GMFO = Location(latitude=34.767, longitude=-1.917)
    GMMC = Location(latitude=33.567, longitude=-7.667)
    GMME = Location(latitude=34.033, longitude=-6.767)
    GMMI = Location(latitude=31.4, longitude=-9.683)
    GMMN = Location(latitude=33.367, longitude=-7.567)
    GMMW = Location(latitude=34.983, longitude=-3.017)
    GMMX = Location(latitude=31.617, longitude=-8.017)
    GMMZ = Location(latitude=30.917, longitude=-6.9)
    GMTA = Location(latitude=35.167, longitude=-3.85)
    GMTN = Location(latitude=35.567, longitude=-5.317)
    GMTT = Location(latitude=35.717, longitude=-5.9)
    GUCY = Location(latitude=9.567, longitude=-13.617)
    GOGG = Location(latitude=12.55, longitude=-16.267)
    GOGK = Location(latitude=12.9, longitude=-14.967)
    GOGS = Location(latitude=12.4, longitude=-16.75)
    GOOD = Location(latitude=14.65, longitude=-16.217)
    GOOG = Location(latitude=15.4, longitude=-15.083)
    GOOK = Location(latitude=14.15, longitude=-16.067)
    GOOY = Location(latitude=14.717, longitude=-17.5)
    GOSP = Location(latitude=16.683, longitude=-14.967)
    GOSS = Location(latitude=16.05, longitude=-16.433)
    GOTK = Location(latitude=12.567, longitude=-12.217)
    GOTT = Location(latitude=13.733, longitude=-13.65)
    HTAR = Location(latitude=-3.367, longitude=36.633)
    HTBU = Location(latitude=-1.317, longitude=31.817)
    HTDA = Location(latitude=-6.883, longitude=39.2)
    HTDO = Location(latitude=-6.167, longitude=35.767)
    HTIR = Location(latitude=-7.667, longitude=35.75)
    HTKA = Location(latitude=-4.867, longitude=29.667)
    HTKJ = Location(latitude=-3.417, longitude=37.067)
    HTMB = Location(latitude=-8.917, longitude=33.467)
    HTMG = Location(latitude=-6.817, longitude=37.633)
    HTMT = Location(latitude=-10.267, longitude=40.167)
    HTMU = Location(latitude=-1.5, longitude=33.783)
    HTMW = Location(latitude=-2.467, longitude=32.917)
    HTSO = Location(latitude=-10.667, longitude=35.567)
    HTTB = Location(latitude=-5.067, longitude=32.817)
    HTZA = Location(latitude=-6.217, longitude=39.217)
    HDAM = Location(latitude=11.55, longitude=43.167)
    KQRH = Location(latitude=11.55, longitude=43.15)
    FKKD = Location(latitude=4, longitude=9.717)
    FKKL = Location(latitude=10.45, longitude=14.25)
    FKKN = Location(latitude=7.333, longitude=13.567)
    FKKR = Location(latitude=9.317, longitude=13.367)
    FKYS = Location(latitude=3.817, longitude=11.517)
    GMMH = Location(latitude=23.717, longitude=-15.933)
    GMML = Location(latitude=27.133, longitude=-13.233)
    GFLL = Location(latitude=8.617, longitude=-13.183)
    FBFT = Location(latitude=-21.167, longitude=27.483)
    FBKE = Location(latitude=-17.817, longitude=25.167)
    FBMN = Location(latitude=-19.967, longitude=23.417)
    FBSK = Location(latitude=-24.55, longitude=25.917)
    FBSP = Location(latitude=-22.05, longitude=27.817)
    FVBU = Location(latitude=-20.017, longitude=28.617)
    FVCZ = Location(latitude=-21.017, longitude=31.567)
    FVFA = Location(latitude=-18.1, longitude=25.85)
    FVHA = Location(latitude=-17.917, longitude=31.117)
    FVKB = Location(latitude=-16.517, longitude=28.867)
    FVMV = Location(latitude=-20.067, longitude=30.867)
    FVWN = Location(latitude=-18.617, longitude=27)
    FABL = Location(latitude=-29.1, longitude=26.3)
    FACT = Location(latitude=-33.967, longitude=18.6)
    FACV = Location(latitude=-31.5, longitude=19.717)
    FADY = Location(latitude=-30.667, longitude=23.983)
    FAEL = Location(latitude=-33.017, longitude=27.817)
    FAEO = Location(latitude=-26.5, longitude=29.967)
    FAGC = Location(latitude=-25.983, longitude=28.133)
    FAGG = Location(latitude=-34, longitude=22.367)
    FAIR = Location(latitude=-25.917, longitude=28.217)
    FAJB = Location(latitude=-26.15, longitude=28)
    FAKM = Location(latitude=-28.8, longitude=24.767)
    FAKN = Location(latitude=-25.383, longitude=31.1)
    FALA = Location(latitude=-25.917, longitude=27.917)
    FALE = Location(latitude=-29.617, longitude=31.117)
    FALM = Location(latitude=-23.167, longitude=29.7)
    FALW = Location(latitude=-32.967, longitude=18.167)
    FAMM = Location(latitude=-25.8, longitude=25.517)
    FAOB = Location(latitude=-34.6, longitude=20.267)
    FAPE = Location(latitude=-33.967, longitude=25.6)
    FAPN = Location(latitude=-25.333, longitude=27.167)
    FAPP = Location(latitude=-23.833, longitude=29.45)
    FAPR = Location(latitude=-25.717, longitude=28.167)
    FASB = Location(latitude=-29.683, longitude=17.933)
    FASI = Location(latitude=-26.25, longitude=28.4)
    FAUP = Location(latitude=-28.383, longitude=21.267)
    FAUT = Location(latitude=-31.517, longitude=28.667)
    FAVV = Location(latitude=-26.567, longitude=27.967)
    FAWB = Location(latitude=-25.633, longitude=28.217)
    FAWK = Location(latitude=-25.817, longitude=28.217)
    HUEN = Location(latitude=0.033, longitude=32.45)
    HSSS = Location(latitude=15.583, longitude=32.533)
    HRYR = Location(latitude=-1.967, longitude=30.117)
    FZAA = Location(latitude=-4.383, longitude=15.45)
    DNAA = Location(latitude=9, longitude=7.267)
    DNEN = Location(latitude=6.467, longitude=7.533)
    DNIB = Location(latitude=7.367, longitude=3.983)
    DNKN = Location(latitude=12.05, longitude=8.517)
    DNMM = Location(latitude=6.567, longitude=3.317)
    DNSO = Location(latitude=12.917, longitude=5.2)
    FOOG = Location(latitude=-0.7, longitude=8.75)
    FOOL = Location(latitude=0.45, longitude=9.417)
    FOON = Location(latitude=-1.633, longitude=13.417)
    DXAK = Location(latitude=7.533, longitude=1.117)
    DXNG = Location(latitude=9.767, longitude=1.1)
    DXSK = Location(latitude=9, longitude=1.15)
    DXXX = Location(latitude=6.167, longitude=1.25)
    FNBG = Location(latitude=-12.617, longitude=13.4)
    FNCA = Location(latitude=-5.583, longitude=12.183)
    FNHU = Location(latitude=-12.817, longitude=15.767)
    FNKU = Location(latitude=-12.367, longitude=16.933)
    FNLU = Location(latitude=-8.833, longitude=13.217)
    FNMA = Location(latitude=-9.55, longitude=16.317)
    FNSA = Location(latitude=-9.683, longitude=20.433)
    FNUB = Location(latitude=-14.933, longitude=13.567)
    FLLI = Location(latitude=-17.817, longitude=25.817)
    FLMF = Location(latitude=-13.267, longitude=31.933)
    FGBT = Location(latitude=1.883, longitude=9.8)
    FGSL = Location(latitude=3.75, longitude=8.717)
    FQBR = Location(latitude=-19.8, longitude=34.883)
    FQCH = Location(latitude=-19.15, longitude=33.433)
    FQIN = Location(latitude=-23.867, longitude=35.367)
    FQLC = Location(latitude=-13.267, longitude=35.25)
    FQMA = Location(latitude=-25.917, longitude=32.567)
    FQNP = Location(latitude=-15.083, longitude=39.267)
    FQPB = Location(latitude=-12.967, longitude=40.5)
    FQQL = Location(latitude=-17.867, longitude=36.867)
    FQTT = Location(latitude=-16.1, longitude=33.617)
    FQVL = Location(latitude=-22, longitude=35.317)
    FXMM = Location(latitude=-29.467, longitude=27.55)
    FDJR = Location(latitude=-27.167, longitude=31.433)
    FDMS = Location(latitude=-26.517, longitude=31.3)
    FDPP = Location(latitude=-25.967, longitude=31.25)
    FDST = Location(latitude=-26.433, longitude=31.95)
    GLRB = Location(latitude=6.25, longitude=-10.333)
    HKEL = Location(latitude=0.4, longitude=35.233)
    HKJK = Location(latitude=-1.317, longitude=36.917)
    HKKI = Location(latitude=-0.1, longitude=34.75)
    HKKS = Location(latitude=-0.667, longitude=34.767)
    HKKT = Location(latitude=0.967, longitude=34.967)
    HKLU = Location(latitude=-2.25, longitude=40.917)
    HKML = Location(latitude=-3.217, longitude=40.1)
    HKMO = Location(latitude=-4.017, longitude=39.617)
    HKNC = Location(latitude=-1.3, longitude=36.75)
    HKNK = Location(latitude=-0.3, longitude=36.167)
    HKNW = Location(latitude=-1.317, longitude=36.817)
    HKVO = Location(latitude=-3.383, longitude=38.533)
    HKWJ = Location(latitude=1.75, longitude=40.067)
    KQVF = Location(latitude=-2.15, longitude=40.9)
    FTTA = Location(latitude=9.15, longitude=18.367)
    FTTD = Location(latitude=8.617, longitude=16.067)
    FTTJ = Location(latitude=12.117, longitude=15.017)
    FTTY = Location(latitude=18, longitude=19.167)
    DRRM = Location(latitude=13.5, longitude=7.117)
    DRRN = Location(latitude=13.467, longitude=2.167)
    DRRT = Location(latitude=14.9, longitude=5.25)
    DRZA = Location(latitude=16.967, longitude=8)
    DRZR = Location(latitude=13.767, longitude=8.967)
    GOSM = Location(latitude=15.6, longitude=-13.333)
    GQNN = Location(latitude=18.1, longitude=-15.933)
    GQPA = Location(latitude=20.517, longitude=-13.067)
    GQPP = Location(latitude=20.917, longitude=-17.017)
    DFFD = Location(latitude=12.333, longitude=-1.517)
    DFOO = Location(latitude=11.167, longitude=-4.317)
    DBBB = Location(latitude=6.333, longitude=2.367)
    FPST = Location(latitude=0.367, longitude=6.717)
    HLLB = Location(latitude=32.067, longitude=20.267)
    HLLM = Location(latitude=32.9, longitude=13.283)
    HLLT = Location(latitude=32.667, longitude=13.15)
    DTKA = Location(latitude=36.983, longitude=8.883)
    DTMB = Location(latitude=35.767, longitude=10.75)
    DTTA = Location(latitude=36.85, longitude=10.217)
    DTTB = Location(latitude=37.25, longitude=9.8)
    DTTD = Location(latitude=32.317, longitude=10.4)
    DTTF = Location(latitude=34.417, longitude=8.817)
    DTTG = Location(latitude=33.733, longitude=9.917)
    DTTJ = Location(latitude=33.867, longitude=10.767)
    DTTK = Location(latitude=35.667, longitude=10.083)
    DTTL = Location(latitude=36.85, longitude=11.067)
    DTTN = Location(latitude=36.467, longitude=8.8)
    DTTX = Location(latitude=34.717, longitude=10.667)
    DTTZ = Location(latitude=33.917, longitude=8.083)
    FYGB = Location(latitude=-22.5, longitude=18.967)
    FYGF = Location(latitude=-19.6, longitude=18.117)
    FYKM = Location(latitude=-17.633, longitude=24.183)
    FYKT = Location(latitude=-26.517, longitude=18.117)
    FYLZ = Location(latitude=-26.667, longitude=15.25)
    FYML = Location(latitude=-24.6, longitude=17.933)
    FYOA = Location(latitude=-17.883, longitude=15.967)
    FYOO = Location(latitude=-19.15, longitude=15.917)
    FYOW = Location(latitude=-20.433, longitude=16.667)
    FYWB = Location(latitude=-22.983, longitude=14.65)
    FYWE = Location(latitude=-22.617, longitude=17.083)
    FYWH = Location(latitude=-22.467, longitude=17.467)
    FYWW = Location(latitude=-22.567, longitude=17.1)
    PADK = Location(latitude=51.883, longitude=-176.65)
    PASY = Location(latitude=52.717, longitude=174.117)
    PAAD = Location(latitude=70.133, longitude=-146.283)
    PAAK = Location(latitude=52.217, longitude=-174.2)
    PAAQ = Location(latitude=61.6, longitude=-149.083)
    PABA = Location(latitude=70.133, longitude=-143.583)
    PABE = Location(latitude=60.783, longitude=-161.833)
    PABI = Location(latitude=64, longitude=-145.717)
    PABL = Location(latitude=65.983, longitude=-161.15)
    PABN = Location(latitude=62.4, longitude=-143)
    PABR = Location(latitude=71.283, longitude=-156.767)
    PABT = Location(latitude=66.917, longitude=-151.517)
    PABV = Location(latitude=61.417, longitude=-149.517)
    PACV = Location(latitude=60.5, longitude=-145.467)
    PADL = Location(latitude=59.05, longitude=-158.5)
    PADQ = Location(latitude=57.75, longitude=-152.5)
    PAEC = Location(latitude=62.883, longitude=-149.833)
    PAED = Location(latitude=61.25, longitude=-149.8)
    PAEG = Location(latitude=64.783, longitude=-141.15)
    PAEI = Location(latitude=64.667, longitude=-147.1)
    PAEN = Location(latitude=60.567, longitude=-151.25)
    PAER = Location(latitude=61.267, longitude=-153.8)
    PAFA = Location(latitude=64.8, longitude=-147.85)
    PAFB = Location(latitude=64.833, longitude=-147.617)
    PAFK = Location(latitude=62.533, longitude=-153.6)
    PAFM = Location(latitude=67.1, longitude=-157.85)
    PAFR = Location(latitude=61.267, longitude=-149.65)
    PAFS = Location(latitude=63.017, longitude=-154.35)
    PAGA = Location(latitude=64.733, longitude=-156.933)
    PAGH = Location(latitude=66.883, longitude=-157.167)
    PAGK = Location(latitude=62.15, longitude=-145.45)
    PAHC = Location(latitude=62.183, longitude=-159.783)
    PAHL = Location(latitude=65.7, longitude=-156.35)
    PAHO = Location(latitude=59.633, longitude=-151.467)
    PAHX = Location(latitude=62.7, longitude=-159.567)
    PAIG = Location(latitude=59.317, longitude=-155.9)
    PAII = Location(latitude=58.183, longitude=-157.383)
    PAIK = Location(latitude=66.983, longitude=-160.433)
    PAIL = Location(latitude=59.75, longitude=-154.917)
    PAIM = Location(latitude=66, longitude=-153.7)
    PAIN = Location(latitude=63.733, longitude=-148.917)
    PAJC = Location(latitude=56.317, longitude=-158.367)
    PAJZ = Location(latitude=59.733, longitude=-157.267)
    PAKH = Location(latitude=56.933, longitude=-154.183)
    PAKK = Location(latitude=64.933, longitude=-161.15)
    PAKN = Location(latitude=58.683, longitude=-156.65)
    PAKP = Location(latitude=68.133, longitude=-151.733)
    PAKU = Location(latitude=70.317, longitude=-149.583)
    PAKV = Location(latitude=64.333, longitude=-158.75)
    PALG = Location(latitude=61.533, longitude=-160.333)
    PALH = Location(latitude=61.183, longitude=-149.967)
    PALJ = Location(latitude=60.2, longitude=-154.3)
    PALP = Location(latitude=70.333, longitude=-150.933)
    PAMB = Location(latitude=58.983, longitude=-159.05)
    PAMC = Location(latitude=62.967, longitude=-155.617)
    PAMD = Location(latitude=59.433, longitude=-146.3)
    PAMH = Location(latitude=63.9, longitude=-152.317)
    PAMR = Location(latitude=61.217, longitude=-149.85)
    PAMX = Location(latitude=61.433, longitude=-142.917)
    PANA = Location(latitude=60.683, longitude=-161.983)
    PANC = Location(latitude=61.167, longitude=-150)
    PANI = Location(latitude=61.583, longitude=-159.533)
    PANN = Location(latitude=64.55, longitude=-149.083)
    PANV = Location(latitude=62.65, longitude=-160.183)
    PANW = Location(latitude=59.45, longitude=-157.367)
    PAOR = Location(latitude=62.967, longitude=-141.933)
    PAOU = Location(latitude=56, longitude=-161.167)
    PAPH = Location(latitude=56.95, longitude=-158.617)
    PAPM = Location(latitude=59.017, longitude=-161.817)
    PAPN = Location(latitude=57.583, longitude=-157.567)
    PAPR = Location(latitude=66.817, longitude=-150.65)
    PAQH = Location(latitude=59.75, longitude=-161.85)
    PAQT = Location(latitude=70.217, longitude=-151)
    PARC = Location(latitude=68.117, longitude=-145.583)
    PARS = Location(latitude=61.767, longitude=-161.317)
    PARY = Location(latitude=64.733, longitude=-155.467)
    PASC = Location(latitude=70.2, longitude=-148.467)
    PASD = Location(latitude=55.333, longitude=-160.5)
    PASK = Location(latitude=66.6, longitude=-160)
    PASL = Location(latitude=61.717, longitude=-157.15)
    PASO = Location(latitude=59.45, longitude=-151.7)
    PASV = Location(latitude=61.1, longitude=-155.567)
    PASX = Location(latitude=60.483, longitude=-151.033)
    PATA = Location(latitude=65.167, longitude=-152.1)
    PATG = Location(latitude=59.05, longitude=-160.383)
    PATK = Location(latitude=62.3, longitude=-150.1)
    PATL = Location(latitude=62.9, longitude=-155.967)
    PATO = Location(latitude=60.783, longitude=-148.85)
    PATQ = Location(latitude=70.467, longitude=-157.433)
    PAUN = Location(latitude=63.883, longitude=-160.8)
    PAVD = Location(latitude=61.133, longitude=-146.25)
    PAWD = Location(latitude=60.117, longitude=-149.417)
    PAWI = Location(latitude=70.617, longitude=-159.85)
    PAWS = Location(latitude=61.567, longitude=-149.533)
    PAZK = Location(latitude=61.933, longitude=-147.167)
    PFCL = Location(latitude=58.833, longitude=-158.533)
    PFKW = Location(latitude=60.8, longitude=-161.433)
    PFNO = Location(latitude=66.817, longitude=-161.017)
    PFSH = Location(latitude=64.367, longitude=-161.217)
    PFWS = Location(latitude=58.7, longitude=-157)
    PFYU = Location(latitude=66.567, longitude=-145.267)
    TQPF = Location(latitude=18.183, longitude=-63.033)
    TAPA = Location(latitude=17.117, longitude=-61.783)
    TKPN = Location(latitude=17.183, longitude=-62.567)
    TNCE = Location(latitude=17.467, longitude=-62.967)
    SBAA = Location(latitude=-8.35, longitude=-49.3)
    SBPJ = Location(latitude=-10.283, longitude=-48.367)
    SBPN = Location(latitude=-10.717, longitude=-48.383)
    SABE = Location(latitude=-34.567, longitude=-58.417)
    SADF = Location(latitude=-34.45, longitude=-58.583)
    SAEZ = Location(latitude=-34.817, longitude=-58.517)
    SAZM = Location(latitude=-37.917, longitude=-57.567)
    SAVC = Location(latitude=-45.767, longitude=-67.467)
    SAVE = Location(latitude=-42.9, longitude=-71.15)
    SAAR = Location(latitude=-32.917, longitude=-60.767)
    SACO = Location(latitude=-31.317, longitude=-64.217)
    SAOC = Location(latitude=-33.117, longitude=-64.267)
    SARC = Location(latitude=-27.433, longitude=-58.767)
    SARE = Location(latitude=-27.433, longitude=-59.033)
    SARF = Location(latitude=-26.217, longitude=-58.217)
    SARI = Location(latitude=-25.717, longitude=-54.467)
    SARL = Location(latitude=-29.667, longitude=-57.133)
    SARP = Location(latitude=-27.383, longitude=-55.967)
    SASJ = Location(latitude=-24.367, longitude=-65.1)
    SAME = Location(latitude=-32.817, longitude=-68.767)
    SAWG = Location(latitude=-51.617, longitude=-69.317)
    SAWP = Location(latitude=-46.517, longitude=-71.017)
    SASA = Location(latitude=-24.85, longitude=-65.467)
    SAZN = Location(latitude=-38.95, longitude=-68.15)
    SAZS = Location(latitude=-41.133, longitude=-71.167)
    SANT = Location(latitude=-26.85, longitude=-65.083)
    SAWE = Location(latitude=-53.783, longitude=-67.75)
    SAWH = Location(latitude=-54.85, longitude=-68.3)
    TNCA = Location(latitude=12.5, longitude=-70)
    SGAS = Location(latitude=-25.25, longitude=-57.517)
    SGEN = Location(latitude=-27.317, longitude=-55.817)
    SGES = Location(latitude=-25.45, longitude=-54.85)
    SBIL = Location(latitude=-14.8, longitude=-39.017)
    SBLE = Location(latitude=-12.483, longitude=-41.283)
    SBLP = Location(latitude=-13.267, longitude=-43.417)
    SBPS = Location(latitude=-16.433, longitude=-39.067)
    SBQV = Location(latitude=-14.867, longitude=-40.867)
    SBSV = Location(latitude=-12.9, longitude=-38.317)
    SBTC = Location(latitude=-15.35, longitude=-38.983)
    SBUF = Location(latitude=-9.4, longitude=-38.25)
    TBPB = Location(latitude=13.067, longitude=-59.483)
    SBBE = Location(latitude=-1.367, longitude=-48.467)
    SBCJ = Location(latitude=-6.117, longitude=-50)
    SBMA = Location(latitude=-5.367, longitude=-49.133)
    SBMQ = Location(latitude=0.05, longitude=-51.067)
    SBOI = Location(latitude=3.85, longitude=-51.817)
    SBTU = Location(latitude=-3.783, longitude=-49.717)
    MZBZ = Location(latitude=17.517, longitude=-88.3)
    CWDM = Location(latitude=50.467, longitude=-59.617)
    CYBX = Location(latitude=51.45, longitude=-57.167)
    SBBV = Location(latitude=2.85, longitude=-60.7)
    SBYA = Location(latitude=0.617, longitude=-69.2)
    SKAR = Location(latitude=4.45, longitude=-75.767)
    SKBG = Location(latitude=7.133, longitude=-73.183)
    SKBO = Location(latitude=4.7, longitude=-74.117)
    SKBQ = Location(latitude=10.9, longitude=-74.767)
    SKCC = Location(latitude=7.917, longitude=-72.517)
    SKCG = Location(latitude=10.433, longitude=-75.517)
    SKCL = Location(latitude=3.533, longitude=-76.367)
    SKEJ = Location(latitude=7.017, longitude=-73.783)
    SKIB = Location(latitude=4.417, longitude=-75.15)
    SKIP = Location(latitude=0.883, longitude=-77.667)
    SKLC = Location(latitude=7.817, longitude=-76.7)
    SKLT = Location(latitude=-4.2, longitude=-69.933)
    SKMD = Location(latitude=6.217, longitude=-75.583)
    SKMR = Location(latitude=8.817, longitude=-75.833)
    SKMZ = Location(latitude=5.033, longitude=-75.467)
    SKNV = Location(latitude=2.967, longitude=-75.283)
    SKPC = Location(latitude=6.167, longitude=-67.5)
    SKPE = Location(latitude=4.817, longitude=-75.733)
    SKPS = Location(latitude=1.4, longitude=-77.283)
    SKPV = Location(latitude=13.367, longitude=-81.333)
    SKRG = Location(latitude=6.167, longitude=-75.417)
    SKRH = Location(latitude=11.517, longitude=-72.917)
    SKSM = Location(latitude=11.117, longitude=-74.217)
    SKSP = Location(latitude=12.567, longitude=-81.717)
    SKUC = Location(latitude=7.067, longitude=-70.717)
    SKUI = Location(latitude=5.717, longitude=-76.617)
    SKVP = Location(latitude=10.417, longitude=-73.25)
    SKVV = Location(latitude=4.167, longitude=-73.617)
    KBOI = Location(latitude=43.567, longitude=-116.233)
    KBYI = Location(latitude=42.55, longitude=-113.767)
    KDIJ = Location(latitude=43.75, longitude=-111.1)
    KEUL = Location(latitude=43.617, longitude=-116.617)
    KIDA = Location(latitude=43.517, longitude=-112.067)
    KJER = Location(latitude=42.717, longitude=-114.45)
    KLLJ = Location(latitude=44.517, longitude=-114.217)
    KMAN = Location(latitude=43.583, longitude=-116.517)
    KMUO = Location(latitude=43.033, longitude=-115.867)
    KMYL = Location(latitude=44.9, longitude=-116.1)
    KONO = Location(latitude=44.017, longitude=-117.017)
    KPIH = Location(latitude=42.917, longitude=-112.567)
    KREO = Location(latitude=42.583, longitude=-117.867)
    KRXE = Location(latitude=43.833, longitude=-111.8)
    KSMN = Location(latitude=45.117, longitude=-113.867)
    KSNT = Location(latitude=44.167, longitude=-114.933)
    KSUN = Location(latitude=43.5, longitude=-114.283)
    KTWF = Location(latitude=42.483, longitude=-114.483)
    CWIJ = Location(latitude=65.767, longitude=-111.233)
    CWIL = Location(latitude=68.317, longitude=-100.067)
    CWPX = Location(latitude=69.017, longitude=-107.817)
    CWRF = Location(latitude=69.417, longitude=-89.717)
    CWXR = Location(latitude=69.267, longitude=-119.217)
    CXSE = Location(latitude=73.767, longitude=-105.283)
    CYBB = Location(latitude=68.517, longitude=-89.817)
    CYCB = Location(latitude=69.083, longitude=-105.117)
    CYCO = Location(latitude=67.817, longitude=-115.15)
    CYHK = Location(latitude=68.617, longitude=-95.833)
    CYOA = Location(latitude=64.7, longitude=-110.617)
    CYUS = Location(latitude=68.817, longitude=-93.417)
    CYYH = Location(latitude=69.533, longitude=-93.567)
    CZRP = Location(latitude=75.367, longitude=-105.717)
    SBCG = Location(latitude=-20.467, longitude=-54.667)
    SBCR = Location(latitude=-19, longitude=-57.667)
    SBPP = Location(latitude=-22.55, longitude=-55.7)
    MMCM = Location(latitude=18.5, longitude=-88.333)
    MMCZ = Location(latitude=20.517, longitude=-86.917)
    MMUN = Location(latitude=21.017, longitude=-86.85)
    SVAC = Location(latitude=9.55, longitude=-69.217)
    SVBC = Location(latitude=10.117, longitude=-64.667)
    SVBI = Location(latitude=8.617, longitude=-70.217)
    SVBM = Location(latitude=10.05, longitude=-69.367)
    SVBS = Location(latitude=10.25, longitude=-67.65)
    SVCB = Location(latitude=8.15, longitude=-63.533)
    SVCL = Location(latitude=8.917, longitude=-67.417)
    SVCN = Location(latitude=6.233, longitude=-62.85)
    SVCR = Location(latitude=11.417, longitude=-69.667)
    SVCU = Location(latitude=10.45, longitude=-64.133)
    SVFM = Location(latitude=10.483, longitude=-66.833)
    SVGD = Location(latitude=7.217, longitude=-70.75)
    SVGI = Location(latitude=10.567, longitude=-62.317)
    SVGU = Location(latitude=9.033, longitude=-69.75)
    SVJC = Location(latitude=11.767, longitude=-70.15)
    SVJM = Location(latitude=9.917, longitude=-67.383)
    SVLF = Location(latitude=8.25, longitude=-72.267)
    SVMC = Location(latitude=10.567, longitude=-71.717)
    SVMD = Location(latitude=8.583, longitude=-71.167)
    SVMG = Location(latitude=10.917, longitude=-63.967)
    SVMI = Location(latitude=10.583, longitude=-67)
    SVMN = Location(latitude=9.817, longitude=-70.917)
    SVMT = Location(latitude=9.75, longitude=-63.167)
    SVPA = Location(latitude=5.617, longitude=-67.6)
    SVPC = Location(latitude=10.483, longitude=-68.067)
    SVPM = Location(latitude=7.783, longitude=-72.2)
    SVPR = Location(latitude=8.3, longitude=-62.767)
    SVSA = Location(latitude=7.833, longitude=-72.45)
    SVSE = Location(latitude=4.583, longitude=-61.117)
    SVSO = Location(latitude=7.567, longitude=-72.033)
    SVSP = Location(latitude=10.267, longitude=-68.75)
    SVSR = Location(latitude=7.9, longitude=-67.417)
    SVST = Location(latitude=8.933, longitude=-64.15)
    SVSZ = Location(latitude=8.967, longitude=-71.95)
    SVVA = Location(latitude=10.15, longitude=-67.917)
    SVVG = Location(latitude=8.617, longitude=-71.667)
    SVVL = Location(latitude=9.333, longitude=-70.583)
    TNCC = Location(latitude=12.2, longitude=-68.967)
    SOCA = Location(latitude=4.817, longitude=-52.367)
    MWCB = Location(latitude=19.683, longitude=-79.883)
    MWCR = Location(latitude=19.3, longitude=-81.367)
    K04W = Location(latitude=46.017, longitude=-92.9)
    K06D = Location(latitude=48.883, longitude=-99.617)
    K08D = Location(latitude=48.3, longitude=-102.4)
    K0F2 = Location(latitude=33.6, longitude=-97.783)
    K0J4 = Location(latitude=31.05, longitude=-86.317)
    K11R = Location(latitude=30.217, longitude=-96.367)
    K14Y = Location(latitude=45.9, longitude=-94.867)
    K1A9 = Location(latitude=32.433, longitude=-86.517)
    K1B7 = Location(latitude=29.35, longitude=-89.433)
    K1F0 = Location(latitude=34.15, longitude=-97.117)
    K1H2 = Location(latitude=39.067, longitude=-88.533)
    K1J0 = Location(latitude=30.85, longitude=-85.6)
    K1K1 = Location(latitude=37.767, longitude=-97.117)
    K1L0 = Location(latitude=30.083, longitude=-90.583)
    K1M4 = Location(latitude=34.283, longitude=-87.6)
    K1R7 = Location(latitude=31.6, longitude=-90.417)
    K21D = Location(latitude=45, longitude=-92.85)
    K2C8 = Location(latitude=48.783, longitude=-97.633)
    K2D5 = Location(latitude=46.167, longitude=-98.083)
    K2I0 = Location(latitude=37.35, longitude=-87.4)
    K3AU = Location(latitude=37.667, longitude=-97.083)
    K3LF = Location(latitude=39.167, longitude=-89.667)
    K3N8 = Location(latitude=47.25, longitude=-95.883)
    K3R7 = Location(latitude=30.25, longitude=-92.667)
    K3T5 = Location(latitude=29.9, longitude=-96.95)
    K46D = Location(latitude=47.45, longitude=-99.15)
    K4A6 = Location(latitude=34.683, longitude=-86)
    K4A9 = Location(latitude=34.467, longitude=-85.717)
    K4M9 = Location(latitude=36.4, longitude=-90.65)
    K4O4 = Location(latitude=33.917, longitude=-94.867)
    K54J = Location(latitude=30.733, longitude=-86.15)
    K5A6 = Location(latitude=33.467, longitude=-89.733)
    K5C1 = Location(latitude=29.717, longitude=-98.7)
    K5H4 = Location(latitude=47.783, longitude=-99.933)
    K5M9 = Location(latitude=37.333, longitude=-88.117)
    K5R8 = Location(latitude=30.433, longitude=-93.467)
    K66R = Location(latitude=29.65, longitude=-96.517)
    K6R3 = Location(latitude=30.35, longitude=-95)
    K6R6 = Location(latitude=30.05, longitude=-102.217)
    K79J = Location(latitude=31.317, longitude=-86.4)
    K7L2 = Location(latitude=46.217, longitude=-100.25)
    K7N0 = Location(latitude=29.95, longitude=-90.067)
    K7R3 = Location(latitude=29.683, longitude=-91.083)
    K82C = Location(latitude=43.833, longitude=-90.133)
    K8A0 = Location(latitude=34.233, longitude=-86.25)
    K8D3 = Location(latitude=45.667, longitude=-96.983)
    K96D = Location(latitude=48.933, longitude=-97.9)
    K9D7 = Location(latitude=48.483, longitude=-99.233)
    K9MN = Location(latitude=44.017, longitude=-92.483)
    K9V9 = Location(latitude=43.767, longitude=-99.317)
    KA08 = Location(latitude=32.517, longitude=-87.383)
    KAAA = Location(latitude=40.167, longitude=-89.333)
    KAAO = Location(latitude=37.75, longitude=-97.217)
    KABI = Location(latitude=32.417, longitude=-99.683)
    KABR = Location(latitude=45.45, longitude=-98.417)
    KACP = Location(latitude=30.75, longitude=-92.683)
    KACQ = Location(latitude=44.067, longitude=-93.55)
    KACT = Location(latitude=31.617, longitude=-97.233)
    KADC = Location(latitude=46.45, longitude=-95.217)
    KADF = Location(latitude=34.1, longitude=-93.067)
    KADH = Location(latitude=34.8, longitude=-96.667)
    KADM = Location(latitude=34.283, longitude=-97.017)
    KADS = Location(latitude=32.967, longitude=-96.817)
    KADU = Location(latitude=41.7, longitude=-94.917)
    KAEL = Location(latitude=43.683, longitude=-93.367)
    KAEX = Location(latitude=31.333, longitude=-92.567)
    KAFK = Location(latitude=40.6, longitude=-95.85)
    KAFW = Location(latitude=32.967, longitude=-97.317)
    KAHQ = Location(latitude=41.233, longitude=-96.6)
    KAIG = Location(latitude=45.15, longitude=-89.117)
    KAIO = Location(latitude=41.383, longitude=-95.033)
    KAIT = Location(latitude=46.533, longitude=-93.667)
    KAIZ = Location(latitude=38.1, longitude=-92.533)
    KAJG = Location(latitude=38.617, longitude=-87.733)
    KALI = Location(latitude=27.733, longitude=-98.017)
    KALN = Location(latitude=38.883, longitude=-90.033)
    KALO = Location(latitude=42.55, longitude=-92.4)
    KALX = Location(latitude=32.917, longitude=-85.967)
    KAMA = Location(latitude=35.217, longitude=-101.717)
    KAMW = Location(latitude=41.983, longitude=-93.617)
    KANB = Location(latitude=33.583, longitude=-85.85)
    KANE = Location(latitude=45.15, longitude=-93.217)
    KANW = Location(latitude=42.567, longitude=-100)
    KAPY = Location(latitude=26.967, longitude=-99.25)
    KAQO = Location(latitude=30.783, longitude=-98.667)
    KAQP = Location(latitude=45.217, longitude=-96)
    KAQR = Location(latitude=34.4, longitude=-96.15)
    KAQV = Location(latitude=31.4, longitude=-93.3)
    KARA = Location(latitude=30.033, longitude=-91.883)
    KARG = Location(latitude=36.117, longitude=-90.917)
    KARM = Location(latitude=29.25, longitude=-96.15)
    KARR = Location(latitude=41.767, longitude=-88.483)
    KARV = Location(latitude=45.917, longitude=-89.717)
    KASD = Location(latitude=30.35, longitude=-89.817)
    KASG = Location(latitude=36.167, longitude=-94.117)
    KASL = Location(latitude=32.517, longitude=-94.3)
    KASN = Location(latitude=33.567, longitude=-86.05)
    KASX = Location(latitude=46.55, longitude=-90.917)
    KATT = Location(latitude=30.317, longitude=-97.767)
    KATW = Location(latitude=44.267, longitude=-88.517)
    KATY = Location(latitude=44.9, longitude=-97.15)
    KAUH = Location(latitude=40.883, longitude=-98)
    KAUM = Location(latitude=43.667, longitude=-92.917)
    KAUO = Location(latitude=32.617, longitude=-85.417)
    KAUS = Location(latitude=30.183, longitude=-97.683)
    KAUW = Location(latitude=44.933, longitude=-89.617)
    KAVK = Location(latitude=36.767, longitude=-98.667)
    KAWG = Location(latitude=41.267, longitude=-91.667)
    KAWM = Location(latitude=35.133, longitude=-90.233)
    KAXA = Location(latitude=43.067, longitude=-94.267)
    KAXH = Location(latitude=29.517, longitude=-95.483)
    KAXN = Location(latitude=45.867, longitude=-95.4)
    KAXO = Location(latitude=29.25, longitude=-89.967)
    KAXS = Location(latitude=34.7, longitude=-99.333)
    KBAC = Location(latitude=46.933, longitude=-98.017)
    KBAD = Location(latitude=32.5, longitude=-93.667)
    KBAZ = Location(latitude=29.717, longitude=-98.05)
    KBBB = Location(latitude=45.317, longitude=-95.65)
    KBBD = Location(latitude=31.183, longitude=-99.317)
    KBBF = Location(latitude=27.833, longitude=-96.017)
    KBBG = Location(latitude=36.533, longitude=-93.2)
    KBBW = Location(latitude=41.433, longitude=-99.633)
    KBCK = Location(latitude=44.25, longitude=-90.85)
    KBDE = Location(latitude=48.733, longitude=-94.617)
    KBDH = Location(latitude=45.117, longitude=-95.133)
    KBEA = Location(latitude=28.367, longitude=-97.783)
    KBFM = Location(latitude=30.617, longitude=-88.067)
    KBFW = Location(latitude=47.25, longitude=-91.4)
    KBGD = Location(latitude=35.7, longitude=-101.4)
    KBGF = Location(latitude=35.183, longitude=-86.067)
    KBHM = Location(latitude=33.567, longitude=-86.75)
    KBIE = Location(latitude=40.283, longitude=-96.75)
    KBIS = Location(latitude=46.767, longitude=-100.75)
    KBIX = Location(latitude=30.417, longitude=-88.917)
    KBJI = Location(latitude=47.5, longitude=-94.917)
    KBKB = Location(latitude=31, longitude=-92.917)
    KBKD = Location(latitude=32.717, longitude=-98.883)
    KBKN = Location(latitude=36.75, longitude=-97.35)
    KBKS = Location(latitude=27.2, longitude=-98.117)
    KBKX = Location(latitude=44.283, longitude=-96.817)
    KBLV = Location(latitude=38.533, longitude=-89.833)
    KBMI = Location(latitude=40.467, longitude=-88.917)
    KBMQ = Location(latitude=30.733, longitude=-98.233)
    KBMT = Location(latitude=30.067, longitude=-94.217)
    KBNA = Location(latitude=36.117, longitude=-86.683)
    KBNW = Location(latitude=42.033, longitude=-93.833)
    KBPC = Location(latitude=35.883, longitude=-101.033)
    KBPG = Location(latitude=32.217, longitude=-101.517)
    KBPK = Location(latitude=36.367, longitude=-92.467)
    KBPT = Location(latitude=29.95, longitude=-94.033)
    KBQP = Location(latitude=32.75, longitude=-91.883)
    KBQX = Location(latitude=28.5, longitude=-95.717)
    KBRD = Location(latitude=46.4, longitude=-94.133)
    KBRL = Location(latitude=40.767, longitude=-91.133)
    KBRO = Location(latitude=25.917, longitude=-97.417)
    KBTA = Location(latitude=41.417, longitude=-96.117)
    KBTR = Location(latitude=30.533, longitude=-91.15)
    KBUU = Location(latitude=42.683, longitude=-88.3)
    KBVE = Location(latitude=29.317, longitude=-89.4)
    KBVN = Location(latitude=41.733, longitude=-98.05)
    KBVO = Location(latitude=36.767, longitude=-96.017)
    KBVX = Location(latitude=35.733, longitude=-91.65)
    KBWD = Location(latitude=31.8, longitude=-98.95)
    KBWG = Location(latitude=36.967, longitude=-86.417)
    KBWP = Location(latitude=46.25, longitude=-96.617)
    KBXA = Location(latitude=30.817, longitude=-89.867)
    KBYH = Location(latitude=35.967, longitude=-89.95)
    KBYY = Location(latitude=28.967, longitude=-95.867)
    KC09 = Location(latitude=41.433, longitude=-88.417)
    KC29 = Location(latitude=43.117, longitude=-89.533)
    KC35 = Location(latitude=43.533, longitude=-89.983)
    KC75 = Location(latitude=41.017, longitude=-89.383)
    KCAV = Location(latitude=42.75, longitude=-93.767)
    KCBF = Location(latitude=41.267, longitude=-95.767)
    KCBG = Location(latitude=45.567, longitude=-93.267)
    KCBK = Location(latitude=39.433, longitude=-101.05)
    KCBM = Location(latitude=33.633, longitude=-88.45)
    KCCA = Location(latitude=35.6, longitude=-92.45)
    KCCY = Location(latitude=43.067, longitude=-92.617)
    KCDD = Location(latitude=48.267, longitude=-92.483)
    KCDH = Location(latitude=33.617, longitude=-92.767)
    KCDJ = Location(latitude=39.817, longitude=-93.583)
    KCDS = Location(latitude=34.433, longitude=-100.283)
    KCEW = Location(latitude=30.783, longitude=-86.517)
    KCEY = Location(latitude=36.667, longitude=-88.367)
    KCFD = Location(latitude=30.717, longitude=-96.333)
    KCFE = Location(latitude=45.167, longitude=-93.85)
    KCFV = Location(latitude=37.083, longitude=-95.567)
    KCGI = Location(latitude=37.233, longitude=-89.583)
    KCHK = Location(latitude=35.1, longitude=-97.967)
    KCID = Location(latitude=41.883, longitude=-91.717)
    KCIN = Location(latitude=42.033, longitude=-94.767)
    KCIR = Location(latitude=37.067, longitude=-89.217)
    KCKC = Location(latitude=47.833, longitude=-90.367)
    KCKM = Location(latitude=34.3, longitude=-90.517)
    KCKN = Location(latitude=47.85, longitude=-96.617)
    KCKP = Location(latitude=42.733, longitude=-95.55)
    KCKV = Location(latitude=36.617, longitude=-87.417)
    KCLI = Location(latitude=44.617, longitude=-88.733)
    KCLK = Location(latitude=35.533, longitude=-98.933)
    KCLL = Location(latitude=30.583, longitude=-96.367)
    KCMD = Location(latitude=34.267, longitude=-86.85)
    KCMI = Location(latitude=40.033, longitude=-88.267)
    KCMY = Location(latitude=43.967, longitude=-90.733)
    KCNB = Location(latitude=44.733, longitude=-96.267)
    KCNC = Location(latitude=41.017, longitude=-93.367)
    KCNK = Location(latitude=39.55, longitude=-97.65)
    KCNU = Location(latitude=37.667, longitude=-95.483)
    KCNW = Location(latitude=31.633, longitude=-97.067)
    KCOM = Location(latitude=31.833, longitude=-99.4)
    KCOQ = Location(latitude=46.7, longitude=-92.5)
    KCOT = Location(latitude=28.45, longitude=-99.217)
    KCOU = Location(latitude=38.817, longitude=-92.217)
    KCPS = Location(latitude=38.567, longitude=-90.15)
    KCPT = Location(latitude=32.35, longitude=-97.433)
    KCQB = Location(latitude=35.717, longitude=-96.817)
    KCQF = Location(latitude=30.467, longitude=-87.883)
    KCQM = Location(latitude=47.817, longitude=-92.683)
    KCRH = Location(latitude=28.917, longitude=-93.3)
    KCRP = Location(latitude=27.767, longitude=-97.5)
    KCRS = Location(latitude=32.033, longitude=-96.4)
    KCRX = Location(latitude=34.917, longitude=-88.6)
    KCSM = Location(latitude=35.35, longitude=-99.2)
    KCSQ = Location(latitude=41.017, longitude=-94.367)
    KCSV = Location(latitude=35.95, longitude=-85.083)
    KCUH = Location(latitude=35.95, longitude=-96.767)
    KCUL = Location(latitude=38.083, longitude=-88.117)
    KCVB = Location(latitude=29.35, longitude=-98.85)
    KCVW = Location(latitude=29.783, longitude=-93.3)
    KCWA = Location(latitude=44.783, longitude=-89.667)
    KCWC = Location(latitude=33.85, longitude=-98.483)
    KCWF = Location(latitude=30.217, longitude=-93.15)
    KCWI = Location(latitude=41.817, longitude=-90.317)
    KCXO = Location(latitude=30.35, longitude=-95.417)
    KCZT = Location(latitude=28.517, longitude=-99.817)
    KD25 = Location(latitude=46.117, longitude=-89.883)
    KD39 = Location(latitude=45.7, longitude=-94.933)
    KD50 = Location(latitude=48.933, longitude=-103.3)
    KD55 = Location(latitude=48.75, longitude=-98.4)
    KD60 = Location(latitude=48.383, longitude=-102.9)
    KDAL = Location(latitude=32.85, longitude=-96.85)
    KDBQ = Location(latitude=42.4, longitude=-90.7)
    KDCU = Location(latitude=34.65, longitude=-86.95)
    KDDC = Location(latitude=37.767, longitude=-99.967)
    KDEC = Location(latitude=39.833, longitude=-88.85)
    KDEH = Location(latitude=43.267, longitude=-91.717)
    KDEQ = Location(latitude=34.05, longitude=-94.4)
    KDFW = Location(latitude=32.9, longitude=-97.017)
    KDHN = Location(latitude=31.317, longitude=-85.45)
    KDHT = Location(latitude=36.017, longitude=-102.55)
    KDKB = Location(latitude=41.933, longitude=-88.7)
    KDKR = Location(latitude=31.3, longitude=-95.4)
    KDLF = Location(latitude=29.367, longitude=-100.767)
    KDLH = Location(latitude=46.85, longitude=-92.2)
    KDLL = Location(latitude=43.517, longitude=-89.767)
    KDLP = Location(latitude=29.117, longitude=-89.55)
    KDMO = Location(latitude=38.717, longitude=-93.167)
    KDNS = Location(latitude=41.967, longitude=-95.367)
    KDNV = Location(latitude=40.2, longitude=-87.6)
    KDPA = Location(latitude=41.917, longitude=-88.25)
    KDRI = Location(latitude=30.833, longitude=-93.333)
    KDRT = Location(latitude=29.367, longitude=-100.917)
    KDSM = Location(latitude=41.533, longitude=-93.667)
    KDTL = Location(latitude=46.817, longitude=-95.867)
    KDTN = Location(latitude=32.55, longitude=-93.75)
    KDTO = Location(latitude=33.2, longitude=-97.2)
    KDTS = Location(latitude=30.4, longitude=-86.467)
    KDUA = Location(latitude=33.95, longitude=-96.4)
    KDUC = Location(latitude=34.467, longitude=-97.967)
    KDUX = Location(latitude=35.85, longitude=-102.017)
    KDVL = Location(latitude=48.117, longitude=-98.917)
    KDVN = Location(latitude=41.617, longitude=-90.583)
    KDVP = Location(latitude=43.983, longitude=-95.783)
    KDWH = Location(latitude=30.067, longitude=-95.55)
    KDXX = Location(latitude=44.983, longitude=-96.183)
    KDYR = Location(latitude=36, longitude=-89.4)
    KDYS = Location(latitude=32.417, longitude=-99.833)
    KDYT = Location(latitude=46.717, longitude=-92.033)
    KDZB = Location(latitude=30.533, longitude=-98.367)
    KE11 = Location(latitude=32.333, longitude=-102.533)
    KE38 = Location(latitude=30.383, longitude=-103.683)
    KEAR = Location(latitude=40.717, longitude=-99)
    KEAU = Location(latitude=44.867, longitude=-91.467)
    KEBG = Location(latitude=26.45, longitude=-98.133)
    KEBS = Location(latitude=42.417, longitude=-93.867)
    KECP = Location(latitude=30.35, longitude=-85.8)
    KECU = Location(latitude=29.95, longitude=-100.167)
    KEDC = Location(latitude=30.4, longitude=-97.567)
    KEDN = Location(latitude=31.3, longitude=-85.9)
    KEET = Location(latitude=33.183, longitude=-86.783)
    KEFD = Location(latitude=29.6, longitude=-95.167)
    KEFT = Location(latitude=42.617, longitude=-89.6)
    KEGI = Location(latitude=30.633, longitude=-86.517)
    KEGV = Location(latitude=45.933, longitude=-89.267)
    KEHA = Location(latitude=37, longitude=-101.9)
    KEHC = Location(latitude=28.433, longitude=-92.883)
    KEHR = Location(latitude=37.817, longitude=-87.667)
    KEIR = Location(latitude=28.633, longitude=-91.483)
    KELA = Location(latitude=29.6, longitude=-96.317)
    KELD = Location(latitude=33.217, longitude=-92.817)
    KELK = Location(latitude=35.433, longitude=-99.4)
    KELO = Location(latitude=47.817, longitude=-91.817)
    KEMK = Location(latitude=27.817, longitude=-94.317)
    KEMP = Location(latitude=38.333, longitude=-96.2)
    KEND = Location(latitude=36.317, longitude=-97.917)
    KENL = Location(latitude=38.517, longitude=-89.1)
    KENW = Location(latitude=42.6, longitude=-87.933)
    KEOK = Location(latitude=40.467, longitude=-91.417)
    KEQA = Location(latitude=37.783, longitude=-96.817)
    KERV = Location(latitude=29.983, longitude=-99.083)
    KESF = Location(latitude=31.4, longitude=-92.283)
    KEST = Location(latitude=43.4, longitude=-94.75)
    KETB = Location(latitude=43.417, longitude=-88.133)
    KETH = Location(latitude=45.783, longitude=-96.55)
    KEUF = Location(latitude=31.95, longitude=-85.133)
    KEVM = Location(latitude=47.417, longitude=-92.5)
    KEVV = Location(latitude=38.05, longitude=-87.517)
    KEWK = Location(latitude=38.033, longitude=-97.267)
    KEZS = Location(latitude=44.783, longitude=-88.567)
    KF05 = Location(latitude=34.233, longitude=-99.283)
    KF17 = Location(latitude=31.833, longitude=-94.15)
    KF44 = Location(latitude=32.167, longitude=-95.833)
    KF46 = Location(latitude=32.933, longitude=-96.433)
    KFAM = Location(latitude=37.767, longitude=-90.417)
    KFAR = Location(latitude=46.933, longitude=-96.817)
    KFBL = Location(latitude=44.333, longitude=-93.317)
    KFCM = Location(latitude=44.833, longitude=-93.467)
    KFDR = Location(latitude=34.35, longitude=-98.983)
    KFEP = Location(latitude=42.25, longitude=-89.583)
    KFET = Location(latitude=41.45, longitude=-96.517)
    KFFL = Location(latitude=41.033, longitude=-91.967)
    KFFM = Location(latitude=46.283, longitude=-96.15)
    KFGN = Location(latitude=49.317, longitude=-94.9)
    KFKA = Location(latitude=43.683, longitude=-92.167)
    KFLD = Location(latitude=43.767, longitude=-88.483)
    KFLP = Location(latitude=36.283, longitude=-92.583)
    KFNB = Location(latitude=40.067, longitude=-95.583)
    KFOA = Location(latitude=38.667, longitude=-88.45)
    KFOD = Location(latitude=42.533, longitude=-94.167)
    KFOE = Location(latitude=38.933, longitude=-95.65)
    KFOZ = Location(latitude=47.767, longitude=-93.65)
    KFRI = Location(latitude=39.033, longitude=-96.767)
    KFRM = Location(latitude=43.633, longitude=-94.417)
    KFSD = Location(latitude=43.583, longitude=-96.75)
    KFSE = Location(latitude=47.583, longitude=-95.767)
    KFSI = Location(latitude=34.65, longitude=-98.4)
    KFSM = Location(latitude=35.333, longitude=-94.367)
    KFST = Location(latitude=30.917, longitude=-102.917)
    KFSW = Location(latitude=40.667, longitude=-91.317)
    KFTN = Location(latitude=28.217, longitude=-100.017)
    KFTW = Location(latitude=32.833, longitude=-97.367)
    KFWB = Location(latitude=36.7, longitude=-93.4)
    KFWC = Location(latitude=38.383, longitude=-88.417)
    KFWS = Location(latitude=32.567, longitude=-97.3)
    KFXY = Location(latitude=43.233, longitude=-93.617)
    KFYG = Location(latitude=38.583, longitude=-91)
    KFYM = Location(latitude=35.067, longitude=-86.567)
    KFYV = Location(latitude=36.017, longitude=-94.167)
    KGAD = Location(latitude=33.967, longitude=-86.067)
    KGAF = Location(latitude=48.4, longitude=-97.367)
    KGAG = Location(latitude=36.3, longitude=-99.767)
    KGAO = Location(latitude=29.45, longitude=-90.267)
    KGBD = Location(latitude=38.35, longitude=-98.867)
    KGBG = Location(latitude=40.933, longitude=-90.433)
    KGCK = Location(latitude=37.917, longitude=-100.717)
    KGCM = Location(latitude=36.3, longitude=-95.483)
    KGDB = Location(latitude=44.75, longitude=-95.55)
    KGDJ = Location(latitude=32.45, longitude=-97.817)
    KGDP = Location(latitude=31.833, longitude=-104.817)
    KGFK = Location(latitude=47.95, longitude=-97.183)
    KGGG = Location(latitude=32.383, longitude=-94.717)
    KGGI = Location(latitude=41.717, longitude=-92.733)
    KGHB = Location(latitude=27.833, longitude=-91.983)
    KGHW = Location(latitude=45.633, longitude=-95.317)
    KGKY = Location(latitude=32.667, longitude=-97.1)
    KGLE = Location(latitude=33.65, longitude=-97.2)
    KGLH = Location(latitude=33.5, longitude=-90.983)
    KGLS = Location(latitude=29.267, longitude=-94.867)
    KGLW = Location(latitude=37.033, longitude=-85.95)
    KGLY = Location(latitude=38.35, longitude=-93.683)
    KGMJ = Location(latitude=36.6, longitude=-94.733)
    KGNA = Location(latitude=47.75, longitude=-90.35)
    KGNC = Location(latitude=32.683, longitude=-102.65)
    KGOK = Location(latitude=35.85, longitude=-97.4)
    KGOP = Location(latitude=31.417, longitude=-97.8)
    KGPH = Location(latitude=39.333, longitude=-94.317)
    KGPM = Location(latitude=32.7, longitude=-97.05)
    KGPT = Location(latitude=30.417, longitude=-89.083)
    KGPZ = Location(latitude=47.217, longitude=-93.517)
    KGRB = Location(latitude=44.483, longitude=-88.133)
    KGRI = Location(latitude=40.967, longitude=-98.317)
    KGRK = Location(latitude=31.067, longitude=-97.833)
    KGRY = Location(latitude=27.633, longitude=-90.45)
    KGTR = Location(latitude=33.45, longitude=-88.567)
    KGTU = Location(latitude=30.683, longitude=-97.683)
    KGUY = Location(latitude=36.683, longitude=-101.5)
    KGVT = Location(latitude=33.067, longitude=-96.067)
    KGVX = Location(latitude=28.583, longitude=-94.983)
    KGWO = Location(latitude=33.5, longitude=-90.083)
    KGWR = Location(latitude=46.217, longitude=-97.65)
    KGYB = Location(latitude=30.167, longitude=-96.983)
    KGYI = Location(latitude=33.717, longitude=-96.667)
    KGYL = Location(latitude=44.767, longitude=-94.083)
    KGYY = Location(latitude=41.617, longitude=-87.417)
    KGZH = Location(latitude=31.417, longitude=-87.05)
    KGZL = Location(latitude=35.283, longitude=-95.1)
    KH21 = Location(latitude=37.967, longitude=-92.683)
    KH78 = Location(latitude=28.183, longitude=-88.483)
    KHAE = Location(latitude=39.733, longitude=-91.45)
    KHBG = Location(latitude=31.267, longitude=-89.25)
    KHBR = Location(latitude=34.983, longitude=-99.05)
    KHBV = Location(latitude=27.35, longitude=-98.733)
    KHCD = Location(latitude=44.867, longitude=-94.383)
    KHCO = Location(latitude=48.75, longitude=-96.933)
    KHDC = Location(latitude=30.517, longitude=-90.417)
    KHDE = Location(latitude=40.45, longitude=-99.317)
    KHDO = Location(latitude=29.367, longitude=-99.167)
    KHEY = Location(latitude=31.35, longitude=-85.65)
    KHEZ = Location(latitude=31.617, longitude=-91.3)
    KHFJ = Location(latitude=36.9, longitude=-94.017)
    KHHF = Location(latitude=35.9, longitude=-100.4)
    KHHW = Location(latitude=34.033, longitude=-95.55)
    KHIB = Location(latitude=47.383, longitude=-92.833)
    KHJH = Location(latitude=40.15, longitude=-97.583)
    KHKA = Location(latitude=35.933, longitude=-89.833)
    KHKS = Location(latitude=32.333, longitude=-90.217)
    KHLC = Location(latitude=39.367, longitude=-99.833)
    KHLR = Location(latitude=31.133, longitude=-97.717)
    KHNR = Location(latitude=41.583, longitude=-95.333)
    KHON = Location(latitude=44.383, longitude=-98.233)
    KHOP = Location(latitude=36.667, longitude=-87.5)
    KHOT = Location(latitude=34.483, longitude=-93.1)
    KHOU = Location(latitude=29.633, longitude=-95.283)
    KHQI = Location(latitude=27.933, longitude=-93.667)
    KHQZ = Location(latitude=32.75, longitude=-96.533)
    KHRL = Location(latitude=26.233, longitude=-97.65)
    KHRO = Location(latitude=36.267, longitude=-93.15)
    KHRT = Location(latitude=30.417, longitude=-86.667)
    KHRX = Location(latitude=34.867, longitude=-102.333)
    KHSA = Location(latitude=30.367, longitude=-89.45)
    KHSB = Location(latitude=37.817, longitude=-88.55)
    KHSI = Location(latitude=40.6, longitude=-98.433)
    KHSV = Location(latitude=34.65, longitude=-86.783)
    KHUA = Location(latitude=34.683, longitude=-86.683)
    KHUM = Location(latitude=29.567, longitude=-90.667)
    KHUT = Location(latitude=38.067, longitude=-97.867)
    KHYI = Location(latitude=29.883, longitude=-97.867)
    KHYR = Location(latitude=46.017, longitude=-91.45)
    KHYS = Location(latitude=38.85, longitude=-99.267)
    KHZR = Location(latitude=30.717, longitude=-91.483)
    KHZX = Location(latitude=46.617, longitude=-93.317)
    KI63 = Location(latitude=39.983, longitude=-90.8)
    KI75 = Location(latitude=41.05, longitude=-93.683)
    KIAB = Location(latitude=37.617, longitude=-97.267)
    KIAH = Location(latitude=30, longitude=-95.367)
    KICL = Location(latitude=40.717, longitude=-95.017)
    KICR = Location(latitude=43.383, longitude=-99.85)
    KICT = Location(latitude=37.65, longitude=-97.433)
    KIDP = Location(latitude=37.167, longitude=-95.783)
    KIER = Location(latitude=31.733, longitude=-93.1)
    KIFA = Location(latitude=42.467, longitude=-93.267)
    KIGQ = Location(latitude=41.533, longitude=-87.533)
    KIIB = Location(latitude=42.45, longitude=-91.95)
    KIJX = Location(latitude=39.767, longitude=-90.25)
    KIKK = Location(latitude=41.067, longitude=-87.85)
    KIKT = Location(latitude=28.517, longitude=-88.283)
    KIKV = Location(latitude=41.683, longitude=-93.567)
    KILE = Location(latitude=31.067, longitude=-97.667)
    KINJ = Location(latitude=32.083, longitude=-97.1)
    KINK = Location(latitude=31.783, longitude=-103.2)
    KINL = Location(latitude=48.567, longitude=-93.4)
    KIOW = Location(latitude=41.633, longitude=-91.55)
    KIPN = Location(latitude=28.083, longitude=-87.983)
    KIRK = Location(latitude=40.1, longitude=-92.55)
    KISN = Location(latitude=48.167, longitude=-103.633)
    KISW = Location(latitude=44.367, longitude=-89.833)
    KIXD = Location(latitude=38.817, longitude=-94.883)
    KIYA = Location(latitude=29.983, longitude=-92.083)
    KJAN = Location(latitude=32.317, longitude=-90.083)
    KJAS = Location(latitude=30.883, longitude=-94.033)
    KJBR = Location(latitude=35.833, longitude=-90.65)
    KJCT = Location(latitude=30.5, longitude=-99.767)
    KJDD = Location(latitude=32.75, longitude=-95.5)
    KJEF = Location(latitude=38.6, longitude=-92.15)
    KJFX = Location(latitude=33.9, longitude=-87.317)
    KJHN = Location(latitude=37.583, longitude=-101.733)
    KJKA = Location(latitude=30.283, longitude=-87.667)
    KJKJ = Location(latitude=46.833, longitude=-96.667)
    KJLN = Location(latitude=37.15, longitude=-94.5)
    KJMR = Location(latitude=45.883, longitude=-93.267)
    KJMS = Location(latitude=46.933, longitude=-98.667)
    KJOT = Location(latitude=41.517, longitude=-88.183)
    KJSO = Location(latitude=31.867, longitude=-95.217)
    KJSV = Location(latitude=35.433, longitude=-94.8)
    KJVL = Location(latitude=42.617, longitude=-89.05)
    KJVW = Location(latitude=32.3, longitude=-90.417)
    KJWG = Location(latitude=35.867, longitude=-98.417)
    KJWY = Location(latitude=32.45, longitude=-96.917)
    KJXI = Location(latitude=32.7, longitude=-94.95)
    KJYG = Location(latitude=43.983, longitude=-94.55)
    KJYR = Location(latitude=40.9, longitude=-97.617)
    KL38 = Location(latitude=30.167, longitude=-90.933)
    KLAW = Location(latitude=34.55, longitude=-98.417)
    KLBB = Location(latitude=33.667, longitude=-101.817)
    KLBF = Location(latitude=41.117, longitude=-100.667)
    KLBL = Location(latitude=37.033, longitude=-100.967)
    KLBR = Location(latitude=33.6, longitude=-95.067)
    KLBX = Location(latitude=29.117, longitude=-95.467)
    KLCG = Location(latitude=42.25, longitude=-96.983)
    KLCH = Location(latitude=30.133, longitude=-93.217)
    KLFK = Location(latitude=31.233, longitude=-94.75)
    KLFT = Location(latitude=30.2, longitude=-92)
    KLHB = Location(latitude=30.867, longitude=-96.617)
    KLIT = Location(latitude=34.733, longitude=-92.233)
    KLJF = Location(latitude=45.1, longitude=-94.517)
    KLLQ = Location(latitude=33.633, longitude=-91.75)
    KLNC = Location(latitude=32.583, longitude=-96.717)
    KLNK = Location(latitude=40.85, longitude=-96.767)
    KLNL = Location(latitude=46.15, longitude=-89.217)
    KLNR = Location(latitude=43.217, longitude=-90.183)
    KLOR = Location(latitude=31.35, longitude=-85.75)
    KLOT = Location(latitude=41.6, longitude=-88.1)
    KLRD = Location(latitude=27.55, longitude=-99.467)
    KLRF = Location(latitude=34.917, longitude=-92.15)
    KLRJ = Location(latitude=42.767, longitude=-96.2)
    KLSE = Location(latitude=43.883, longitude=-91.25)
    KLTS = Location(latitude=34.667, longitude=-99.267)
    KLUD = Location(latitude=33.25, longitude=-97.583)
    KLUL = Location(latitude=31.667, longitude=-89.167)
    KLUM = Location(latitude=44.9, longitude=-91.867)
    KLVJ = Location(latitude=29.517, longitude=-95.25)
    KLVN = Location(latitude=44.633, longitude=-93.233)
    KLWC = Location(latitude=39.017, longitude=-95.217)
    KLWD = Location(latitude=40.633, longitude=-93.9)
    KLWV = Location(latitude=38.767, longitude=-87.6)
    KLXL = Location(latitude=45.95, longitude=-94.333)
    KLXN = Location(latitude=40.783, longitude=-99.767)
    KLXT = Location(latitude=38.967, longitude=-94.367)
    KLXY = Location(latitude=31.633, longitude=-96.517)
    KLYV = Location(latitude=43.617, longitude=-96.217)
    KLZZ = Location(latitude=31.1, longitude=-98.2)
    KM19 = Location(latitude=35.65, longitude=-91.167)
    KM25 = Location(latitude=36.767, longitude=-88.583)
    KM30 = Location(latitude=37.183, longitude=-88.75)
    KM40 = Location(latitude=33.867, longitude=-88.483)
    KM91 = Location(latitude=36.533, longitude=-86.917)
    KMAF = Location(latitude=31.95, longitude=-102.2)
    KMAI = Location(latitude=30.833, longitude=-85.183)
    KMBG = Location(latitude=45.55, longitude=-100.417)
    KMBO = Location(latitude=32.433, longitude=-90.1)
    KMCB = Location(latitude=31.183, longitude=-90.467)
    KMCI = Location(latitude=39.3, longitude=-94.733)
    KMCJ = Location(latitude=29.717, longitude=-95.4)
    KMCK = Location(latitude=40.2, longitude=-100.583)
    KMCW = Location(latitude=43.15, longitude=-93.333)
    KMDD = Location(latitude=32.033, longitude=-102.1)
    KMDH = Location(latitude=37.783, longitude=-89.25)
    KMDJ = Location(latitude=28.65, longitude=-89.8)
    KMDQ = Location(latitude=34.867, longitude=-86.55)
    KMDS = Location(latitude=44.017, longitude=-97.083)
    KMDW = Location(latitude=41.783, longitude=-87.75)
    KMDZ = Location(latitude=45.1, longitude=-90.3)
    KMEI = Location(latitude=32.333, longitude=-88.75)
    KMEM = Location(latitude=35.067, longitude=-89.983)
    KMEZ = Location(latitude=34.55, longitude=-94.2)
    KMFE = Location(latitude=26.183, longitude=-98.25)
    KMFI = Location(latitude=44.633, longitude=-90.183)
    KMGC = Location(latitude=41.7, longitude=-86.817)
    KMGG = Location(latitude=45.233, longitude=-93.983)
    KMGM = Location(latitude=32.3, longitude=-86.4)
    KMHE = Location(latitude=43.767, longitude=-98.017)
    KMHK = Location(latitude=39.133, longitude=-96.683)
    KMIB = Location(latitude=48.417, longitude=-101.333)
    KMIC = Location(latitude=45.067, longitude=-93.35)
    KMIS = Location(latitude=29.3, longitude=-88.85)
    KMIW = Location(latitude=42.117, longitude=-92.917)
    KMJQ = Location(latitude=43.633, longitude=-94.967)
    KMKC = Location(latitude=39.117, longitude=-94.6)
    KMKE = Location(latitude=42.95, longitude=-87.9)
    KMKL = Location(latitude=35.6, longitude=-88.917)
    KMKN = Location(latitude=31.917, longitude=-98.6)
    KMKO = Location(latitude=35.65, longitude=-95.367)
    KMKT = Location(latitude=44.217, longitude=-93.917)
    KMLC = Location(latitude=34.883, longitude=-95.783)
    KMLE = Location(latitude=41.2, longitude=-96.117)
    KMLI = Location(latitude=41.45, longitude=-90.517)
    KMLU = Location(latitude=32.517, longitude=-92.033)
    KMML = Location(latitude=44.45, longitude=-95.817)
    KMNE = Location(latitude=32.65, longitude=-93.3)
    KMNZ = Location(latitude=31.667, longitude=-98.15)
    KMOB = Location(latitude=30.683, longitude=-88.25)
    KMOT = Location(latitude=48.25, longitude=-101.267)
    KMOX = Location(latitude=45.567, longitude=-95.967)
    KMPZ = Location(latitude=40.95, longitude=-91.517)
    KMQB = Location(latitude=40.517, longitude=-90.65)
    KMQY = Location(latitude=36.017, longitude=-86.517)
    KMRC = Location(latitude=35.55, longitude=-87.183)
    KMRF = Location(latitude=30.367, longitude=-104.017)
    KMRJ = Location(latitude=42.883, longitude=-90.233)
    KMSL = Location(latitude=34.75, longitude=-87.6)
    KMSN = Location(latitude=43.133, longitude=-89.35)
    KMSP = Location(latitude=44.883, longitude=-93.233)
    KMSY = Location(latitude=30, longitude=-90.25)
    KMTO = Location(latitude=39.467, longitude=-88.267)
    KMTW = Location(latitude=44.117, longitude=-87.667)
    KMUT = Location(latitude=41.367, longitude=-91.15)
    KMVE = Location(latitude=44.967, longitude=-95.717)
    KMVN = Location(latitude=38.317, longitude=-88.867)
    KMWA = Location(latitude=37.767, longitude=-89.017)
    KMWC = Location(latitude=43.117, longitude=-88.033)
    KMWL = Location(latitude=32.783, longitude=-98.067)
    KMWM = Location(latitude=43.9, longitude=-95.1)
    KMWT = Location(latitude=34.55, longitude=-93.583)
    KMXF = Location(latitude=32.367, longitude=-86.367)
    KMXO = Location(latitude=42.217, longitude=-91.167)
    KMYZ = Location(latitude=39.85, longitude=-96.633)
    KMZG = Location(latitude=27.733, longitude=-96.183)
    KMZH = Location(latitude=46.417, longitude=-92.8)
    KN60 = Location(latitude=47.65, longitude=-101.433)
    KNBG = Location(latitude=29.817, longitude=-90.017)
    KNDZ = Location(latitude=30.7, longitude=-87.017)
    KNEW = Location(latitude=30.05, longitude=-90.033)
    KNFW = Location(latitude=32.783, longitude=-97.433)
    KNGP = Location(latitude=27.683, longitude=-97.267)
    KNMM = Location(latitude=32.533, longitude=-88.567)
    KNOG = Location(latitude=27.9, longitude=-98.05)
    KNPA = Location(latitude=30.35, longitude=-87.317)
    KNQA = Location(latitude=35.35, longitude=-89.867)
    KNQI = Location(latitude=27.5, longitude=-97.817)
    KNSE = Location(latitude=30.733, longitude=-87.017)
    KOCH = Location(latitude=31.567, longitude=-94.717)
    KOCQ = Location(latitude=44.867, longitude=-87.917)
    KODO = Location(latitude=31.917, longitude=-102.4)
    KODX = Location(latitude=41.617, longitude=-98.95)
    KOEO = Location(latitude=45.317, longitude=-92.7)
    KOFF = Location(latitude=41.117, longitude=-95.917)
    KOFK = Location(latitude=41.983, longitude=-97.433)
    KOJA = Location(latitude=35.55, longitude=-98.667)
    KOJC = Location(latitude=38.85, longitude=-94.733)
    KOKC = Location(latitude=35.383, longitude=-97.6)
    KOKM = Location(latitude=35.667, longitude=-95.95)
    KOLG = Location(latitude=46.3, longitude=-91.817)
    KOLU = Location(latitude=41.45, longitude=-97.317)
    KOLV = Location(latitude=34.983, longitude=-89.783)
    KOLY = Location(latitude=38.717, longitude=-88.183)
    KOLZ = Location(latitude=42.667, longitude=-91.967)
    KOMA = Location(latitude=41.317, longitude=-95.9)
    KONA = Location(latitude=44.067, longitude=-91.7)
    KONL = Location(latitude=42.467, longitude=-98.667)
    KOOA = Location(latitude=41.233, longitude=-92.5)
    KOPL = Location(latitude=30.55, longitude=-92.1)
    KORB = Location(latitude=48.017, longitude=-92.867)
    KORC = Location(latitude=42.967, longitude=-96.067)
    KORD = Location(latitude=41.983, longitude=-87.933)
    KORG = Location(latitude=30.067, longitude=-93.8)
    KOSA = Location(latitude=33.1, longitude=-94.967)
    KOSH = Location(latitude=43.967, longitude=-88.55)
    KOTG = Location(latitude=43.633, longitude=-95.567)
    KOTM = Location(latitude=41.1, longitude=-92.45)
    KOUN = Location(latitude=35.25, longitude=-97.467)
    KOVL = Location(latitude=44.783, longitude=-95.033)
    KOVS = Location(latitude=43.15, longitude=-90.683)
    KOWA = Location(latitude=44.117, longitude=-93.25)
    KOWB = Location(latitude=37.717, longitude=-87.167)
    KOWP = Location(latitude=36.183, longitude=-96.15)
    KOXV = Location(latitude=41.283, longitude=-93.117)
    KOZA = Location(latitude=30.733, longitude=-101.2)
    KOZR = Location(latitude=31.283, longitude=-85.717)
    KP28 = Location(latitude=37.283, longitude=-98.55)
    KP92 = Location(latitude=29.567, longitude=-91.533)
    KPAH = Location(latitude=37.067, longitude=-88.767)
    KPAM = Location(latitude=30.067, longitude=-85.567)
    KPBF = Location(latitude=34.183, longitude=-91.933)
    KPBH = Location(latitude=45.7, longitude=-90.4)
    KPCZ = Location(latitude=44.333, longitude=-89.017)
    KPDC = Location(latitude=43.017, longitude=-91.117)
    KPEA = Location(latitude=41.4, longitude=-92.95)
    KPEQ = Location(latitude=31.383, longitude=-103.517)
    KPEX = Location(latitude=45.367, longitude=-94.733)
    KPEZ = Location(latitude=28.95, longitude=-98.517)
    KPHT = Location(latitude=36.333, longitude=-88.383)
    KPIA = Location(latitude=40.667, longitude=-89.683)
    KPIB = Location(latitude=31.467, longitude=-89.317)
    KPIL = Location(latitude=26.167, longitude=-97.333)
    KPIR = Location(latitude=44.383, longitude=-100.283)
    KPKD = Location(latitude=46.9, longitude=-95.067)
    KPKV = Location(latitude=28.65, longitude=-96.683)
    KPMV = Location(latitude=40.95, longitude=-95.917)
    KPNC = Location(latitude=36.733, longitude=-97.1)
    KPNM = Location(latitude=45.567, longitude=-93.6)
    KPNS = Location(latitude=30.483, longitude=-87.183)
    KPNT = Location(latitude=40.917, longitude=-88.617)
    KPOE = Location(latitude=31.05, longitude=-93.2)
    KPOF = Location(latitude=36.767, longitude=-90.317)
    KPPA = Location(latitude=35.617, longitude=-101)
    KPPF = Location(latitude=37.333, longitude=-95.5)
    KPPO = Location(latitude=41.567, longitude=-86.733)
    KPPQ = Location(latitude=39.633, longitude=-90.783)
    KPQL = Location(latitude=30.467, longitude=-88.533)
    KPQN = Location(latitude=43.967, longitude=-96.317)
    KPRG = Location(latitude=39.7, longitude=-87.667)
    KPRN = Location(latitude=31.85, longitude=-86.617)
    KPRO = Location(latitude=41.833, longitude=-94.167)
    KPRS = Location(latitude=29.633, longitude=-104.35)
    KPRX = Location(latitude=33.617, longitude=-95.45)
    KPSN = Location(latitude=31.783, longitude=-95.7)
    KPSX = Location(latitude=28.717, longitude=-96.25)
    KPTN = Location(latitude=29.717, longitude=-91.317)
    KPTS = Location(latitude=37.45, longitude=-94.733)
    KPTT = Location(latitude=37.7, longitude=-98.75)
    KPVB = Location(latitude=42.683, longitude=-90.433)
    KPVE = Location(latitude=35.65, longitude=-88.2)
    KPVJ = Location(latitude=34.717, longitude=-97.217)
    KPVW = Location(latitude=34.167, longitude=-101.717)
    KPWA = Location(latitude=35.533, longitude=-97.65)
    KPWC = Location(latitude=46.717, longitude=-94.367)
    KPWG = Location(latitude=31.467, longitude=-97.317)
    KPWK = Location(latitude=42.117, longitude=-87.9)
    KPYX = Location(latitude=36.417, longitude=-100.75)
    KQHY = Location(latitude=30.4, longitude=-86.65)
    KRAC = Location(latitude=42.767, longitude=-87.817)
    KRAS = Location(latitude=27.817, longitude=-97.083)
    KRBD = Location(latitude=32.683, longitude=-96.867)
    KRBO = Location(latitude=27.783, longitude=-97.683)
    KRCE = Location(latitude=35.483, longitude=-97.817)
    KRCM = Location(latitude=38.783, longitude=-93.8)
    KRCX = Location(latitude=45.5, longitude=-91)
    KRDK = Location(latitude=41.017, longitude=-95.267)
    KRDR = Location(latitude=47.967, longitude=-97.4)
    KRFD = Location(latitude=42.2, longitude=-89.1)
    KRFI = Location(latitude=32.15, longitude=-94.85)
    KRGK = Location(latitude=44.583, longitude=-92.483)
    KRHI = Location(latitude=45.633, longitude=-89.483)
    KRKP = Location(latitude=28.083, longitude=-97.05)
    KRKR = Location(latitude=35.017, longitude=-94.617)
    KRNC = Location(latitude=35.7, longitude=-85.85)
    KRND = Location(latitude=29.517, longitude=-98.267)
    KRNH = Location(latitude=45.15, longitude=-92.533)
    KRNV = Location(latitude=33.767, longitude=-90.75)
    KROG = Location(latitude=36.367, longitude=-94.083)
    KROS = Location(latitude=45.7, longitude=-92.95)
    KROX = Location(latitude=48.85, longitude=-95.7)
    KRPD = Location(latitude=45.417, longitude=-91.767)
    KRPH = Location(latitude=33.117, longitude=-98.55)
    KRPJ = Location(latitude=41.883, longitude=-89.083)
    KRQO = Location(latitude=35.467, longitude=-98)
    KRRL = Location(latitude=45.2, longitude=-89.717)
    KRRT = Location(latitude=48.933, longitude=-95.333)
    KRSL = Location(latitude=38.867, longitude=-98.817)
    KRSN = Location(latitude=32.517, longitude=-92.583)
    KRST = Location(latitude=43.9, longitude=-92.5)
    KRSV = Location(latitude=39.017, longitude=-87.65)
    KRUE = Location(latitude=35.25, longitude=-93.1)
    KRUG = Location(latitude=48.383, longitude=-100.017)
    KRVS = Location(latitude=36.05, longitude=-95.983)
    KRWF = Location(latitude=44.55, longitude=-95.083)
    KRWV = Location(latitude=30.517, longitude=-96.7)
    KRYV = Location(latitude=43.167, longitude=-88.717)
    KRYW = Location(latitude=30.5, longitude=-97.967)
    KRZL = Location(latitude=40.95, longitude=-87.183)
    KRZN = Location(latitude=45.817, longitude=-92.367)
    KS25 = Location(latitude=47.8, longitude=-103.25)
    KS32 = Location(latitude=47.417, longitude=-98.1)
    KSAR = Location(latitude=38.15, longitude=-89.7)
    KSAT = Location(latitude=29.533, longitude=-98.467)
    KSAZ = Location(latitude=46.367, longitude=-94.8)
    KSBM = Location(latitude=43.783, longitude=-87.85)
    KSCD = Location(latitude=33.167, longitude=-86.3)
    KSCF = Location(latitude=29.117, longitude=-91.867)
    KSDA = Location(latitude=40.75, longitude=-95.417)
    KSEM = Location(latitude=32.35, longitude=-86.983)
    KSEP = Location(latitude=32.217, longitude=-98.167)
    KSET = Location(latitude=38.933, longitude=-90.433)
    KSFY = Location(latitude=42.05, longitude=-90.117)
    KSGF = Location(latitude=37.233, longitude=-93.383)
    KSGR = Location(latitude=29.617, longitude=-95.65)
    KSGS = Location(latitude=44.867, longitude=-93.033)
    KSGT = Location(latitude=34.6, longitude=-91.567)
    KSHL = Location(latitude=43.217, longitude=-95.817)
    KSHV = Location(latitude=32.45, longitude=-93.833)
    KSJT = Location(latitude=31.367, longitude=-100.5)
    KSKF = Location(latitude=29.367, longitude=-98.567)
    KSLB = Location(latitude=42.6, longitude=-95.217)
    KSLG = Location(latitude=36.2, longitude=-94.467)
    KSLN = Location(latitude=38.783, longitude=-97.65)
    KSLO = Location(latitude=38.65, longitude=-88.967)
    KSLR = Location(latitude=33.167, longitude=-95.617)
    KSNH = Location(latitude=35.167, longitude=-88.217)
    KSNK = Location(latitude=32.7, longitude=-100.95)
    KSNL = Location(latitude=35.35, longitude=-96.95)
    KSOA = Location(latitude=30.583, longitude=-100.65)
    KSPI = Location(latitude=39.85, longitude=-89.683)
    KSPL = Location(latitude=26.067, longitude=-97.15)
    KSPR = Location(latitude=28.6, longitude=-91.2)
    KSPS = Location(latitude=33.983, longitude=-98.5)
    KSPW = Location(latitude=43.167, longitude=-95.217)
    KSQE = Location(latitude=28.067, longitude=-90.817)
    KSQI = Location(latitude=41.75, longitude=-89.667)
    KSRB = Location(latitude=36.05, longitude=-85.533)
    KSRC = Location(latitude=35.217, longitude=-91.733)
    KSSF = Location(latitude=29.333, longitude=-98.467)
    KSTC = Location(latitude=45.55, longitude=-94.05)
    KSTE = Location(latitude=44.55, longitude=-89.533)
    KSTF = Location(latitude=33.433, longitude=-88.85)
    KSTJ = Location(latitude=39.767, longitude=-94.917)
    KSTL = Location(latitude=38.75, longitude=-90.367)
    KSTP = Location(latitude=44.933, longitude=-93.05)
    KSUE = Location(latitude=44.85, longitude=-87.417)
    KSUS = Location(latitude=38.65, longitude=-90.65)
    KSUW = Location(latitude=46.683, longitude=-92.1)
    KSUX = Location(latitude=42.383, longitude=-96.383)
    KSWO = Location(latitude=36.167, longitude=-97.083)
    KSWW = Location(latitude=32.467, longitude=-100.467)
    KSXS = Location(latitude=31.367, longitude=-85.85)
    KSYF = Location(latitude=39.767, longitude=-101.8)
    KSYI = Location(latitude=35.567, longitude=-86.45)
    KSYN = Location(latitude=44.467, longitude=-93.017)
    KSZL = Location(latitude=38.717, longitude=-93.533)
    KT20 = Location(latitude=29.517, longitude=-97.467)
    KT35 = Location(latitude=30.883, longitude=-96.967)
    KT65 = Location(latitude=26.183, longitude=-97.967)
    KT82 = Location(latitude=30.25, longitude=-98.917)
    KTAZ = Location(latitude=39.533, longitude=-89.333)
    KTBN = Location(latitude=37.717, longitude=-92.117)
    KTCL = Location(latitude=33.217, longitude=-87.617)
    KTDR = Location(latitude=30.05, longitude=-85.517)
    KTFP = Location(latitude=27.917, longitude=-97.217)
    KTHA = Location(latitude=35.383, longitude=-86.25)
    KTIF = Location(latitude=41.967, longitude=-100.567)
    KTIK = Location(latitude=35.417, longitude=-97.367)
    KTIP = Location(latitude=40.283, longitude=-88.15)
    KTKC = Location(latitude=44.25, longitude=-95.617)
    KTKI = Location(latitude=33.183, longitude=-96.583)
    KTKV = Location(latitude=45.467, longitude=-89.817)
    KTME = Location(latitude=29.8, longitude=-95.9)
    KTNU = Location(latitude=41.667, longitude=-93.017)
    KTOB = Location(latitude=44.017, longitude=-92.817)
    KTOI = Location(latitude=31.85, longitude=-86.017)
    KTOP = Location(latitude=39.067, longitude=-95.633)
    KTPL = Location(latitude=31.133, longitude=-97.4)
    KTQE = Location(latitude=41.767, longitude=-96.183)
    KTQH = Location(latitude=35.933, longitude=-95)
    KTRL = Location(latitude=32.717, longitude=-96.267)
    KTUL = Location(latitude=36.2, longitude=-95.883)
    KTUP = Location(latitude=34.267, longitude=-88.767)
    KTVF = Location(latitude=48.067, longitude=-96.167)
    KTVK = Location(latitude=40.683, longitude=-92.9)
    KTVR = Location(latitude=32.35, longitude=-91.033)
    KTWM = Location(latitude=47.033, longitude=-91.75)
    KTXK = Location(latitude=33.45, longitude=-93.983)
    KTYR = Location(latitude=32.367, longitude=-95.4)
    KUBE = Location(latitude=45.5, longitude=-91.983)
    KUCY = Location(latitude=36.383, longitude=-88.983)
    KUES = Location(latitude=43.05, longitude=-88.25)
    KUGN = Location(latitude=42.417, longitude=-87.867)
    KUIN = Location(latitude=39.95, longitude=-91.2)
    KULM = Location(latitude=44.317, longitude=-94.5)
    KUNO = Location(latitude=36.883, longitude=-91.9)
    KUNU = Location(latitude=43.417, longitude=-88.7)
    KUOX = Location(latitude=34.383, longitude=-89.533)
    KUTA = Location(latitude=34.683, longitude=-90.35)
    KUTS = Location(latitude=30.75, longitude=-95.583)
    KUVA = Location(latitude=29.217, longitude=-99.75)
    KUXL = Location(latitude=30.133, longitude=-93.383)
    KVAF = Location(latitude=27.35, longitude=-94.633)
    KVBS = Location(latitude=29.483, longitude=-93.633)
    KVBT = Location(latitude=36.35, longitude=-94.217)
    KVCT = Location(latitude=28.867, longitude=-96.933)
    KVER = Location(latitude=38.95, longitude=-92.683)
    KVIH = Location(latitude=38.133, longitude=-91.767)
    KVKS = Location(latitude=32.233, longitude=-90.933)
    KVKY = Location(latitude=29.25, longitude=-88.433)
    KVOA = Location(latitude=29.233, longitude=-87.783)
    KVOK = Location(latitude=43.917, longitude=-90.267)
    KVPS = Location(latitude=30.467, longitude=-86.517)
    KVPZ = Location(latitude=41.45, longitude=-87)
    KVQT = Location(latitude=28.267, longitude=-92.267)
    KVTI = Location(latitude=42.217, longitude=-92.017)
    KVTN = Location(latitude=42.867, longitude=-100.55)
    KVVV = Location(latitude=45.3, longitude=-96.417)
    KVWU = Location(latitude=48.15, longitude=-94.517)
    KVYS = Location(latitude=41.35, longitude=-89.15)
    KWDG = Location(latitude=36.367, longitude=-97.767)
    KWLD = Location(latitude=37.167, longitude=-97.033)
    KWWR = Location(latitude=36.45, longitude=-99.517)
    KXBP = Location(latitude=33.183, longitude=-97.833)
    KXIH = Location(latitude=29.183, longitude=-94.517)
    KXNA = Location(latitude=36.283, longitude=-94.3)
    KXPY = Location(latitude=29.117, longitude=-90.2)
    KXVG = Location(latitude=46.983, longitude=-94.2)
    KY23 = Location(latitude=45.3, longitude=-91.633)
    KY49 = Location(latitude=47.133, longitude=-94.65)
    KY50 = Location(latitude=44.033, longitude=-89.3)
    KY51 = Location(latitude=43.583, longitude=-90.9)
    KY63 = Location(latitude=45.983, longitude=-96)
    KYKN = Location(latitude=42.917, longitude=-97.367)
    MMCU = Location(latitude=28.7, longitude=-105.967)
    CYZS = Location(latitude=64.2, longitude=-83.367)
    MRLB = Location(latitude=10.6, longitude=-85.55)
    MRLM = Location(latitude=9.967, longitude=-83.033)
    MROC = Location(latitude=10, longitude=-84.217)
    MRPV = Location(latitude=9.95, longitude=-84.15)
    CWJR = Location(latitude=49.067, longitude=-116.5)
    SBAT = Location(latitude=-9.867, longitude=-56.1)
    SBBW = Location(latitude=-15.867, longitude=-52.367)
    SBCY = Location(latitude=-15.65, longitude=-56.117)
    CYCQ = Location(latitude=55.7, longitude=-121.617)
    CYDQ = Location(latitude=55.75, longitude=-120.167)
    CYXJ = Location(latitude=56.25, longitude=-120.733)
    K04V = Location(latitude=38.1, longitude=-106.167)
    K0CO = Location(latitude=39.8, longitude=-105.767)
    K0E0 = Location(latitude=34.983, longitude=-106)
    K1V6 = Location(latitude=38.433, longitude=-105.1)
    K20U = Location(latitude=46.933, longitude=-103.983)
    K20V = Location(latitude=40.05, longitude=-106.367)
    K2V5 = Location(latitude=40.1, longitude=-102.25)
    K2WX = Location(latitude=45.6, longitude=-103.55)
    K33V = Location(latitude=40.75, longitude=-106.283)
    K36U = Location(latitude=40.483, longitude=-111.433)
    K3K3 = Location(latitude=38, longitude=-101.75)
    K4BM = Location(latitude=39.05, longitude=-105.517)
    K4MR = Location(latitude=34.3, longitude=-103.8)
    K5SM = Location(latitude=39.433, longitude=-107.383)
    K5T6 = Location(latitude=31.883, longitude=-106.7)
    K7BM = Location(latitude=38.783, longitude=-106.217)
    K82V = Location(latitude=41.15, longitude=-104.133)
    K8S0 = Location(latitude=48.6, longitude=-113.117)
    KABH = Location(latitude=38.75, longitude=-104.3)
    KABQ = Location(latitude=35.05, longitude=-106.617)
    KAEG = Location(latitude=35.133, longitude=-106.8)
    KAEJ = Location(latitude=38.817, longitude=-106.117)
    KAFF = Location(latitude=38.967, longitude=-104.817)
    KAFO = Location(latitude=42.717, longitude=-110.95)
    KAIA = Location(latitude=42.05, longitude=-102.8)
    KAIB = Location(latitude=38.233, longitude=-108.567)
    KAJZ = Location(latitude=38.783, longitude=-108.067)
    KAKO = Location(latitude=40.167, longitude=-103.217)
    KALM = Location(latitude=32.817, longitude=-105.967)
    KALS = Location(latitude=37.433, longitude=-105.867)
    KANK = Location(latitude=38.533, longitude=-106.05)
    KAPA = Location(latitude=39.567, longitude=-104.85)
    KASE = Location(latitude=39.233, longitude=-106.867)
    KATS = Location(latitude=32.85, longitude=-104.467)
    KAXX = Location(latitude=36.417, longitude=-105.283)
    KBCE = Location(latitude=37.7, longitude=-112.15)
    KBDG = Location(latitude=37.583, longitude=-109.467)
    KBDU = Location(latitude=40.033, longitude=-105.233)
    KBFF = Location(latitude=41.867, longitude=-103.583)
    KBHK = Location(latitude=46.35, longitude=-104.25)
    KBIF = Location(latitude=31.85, longitude=-106.383)
    KBIL = Location(latitude=45.8, longitude=-108.55)
    KBJC = Location(latitude=39.9, longitude=-105.1)
    KBKF = Location(latitude=39.717, longitude=-104.75)
    KBMC = Location(latitude=41.55, longitude=-112.067)
    KBPI = Location(latitude=42.583, longitude=-110.1)
    KBTM = Location(latitude=45.967, longitude=-112.5)
    KBYG = Location(latitude=44.383, longitude=-106.717)
    KBZN = Location(latitude=45.783, longitude=-111.167)
    KCAG = Location(latitude=40.5, longitude=-107.517)
    KCAO = Location(latitude=36.45, longitude=-103.15)
    KCCU = Location(latitude=39.483, longitude=-106.15)
    KCDC = Location(latitude=37.7, longitude=-113.1)
    KCDR = Location(latitude=42.833, longitude=-103.1)
    KCEZ = Location(latitude=37.3, longitude=-108.633)
    KCNM = Location(latitude=32.333, longitude=-104.25)
    KCNY = Location(latitude=38.767, longitude=-109.75)
    KCOD = Location(latitude=44.517, longitude=-109.017)
    KCOS = Location(latitude=38.817, longitude=-104.683)
    KCPR = Location(latitude=42.9, longitude=-106.467)
    KCPW = Location(latitude=37.45, longitude=-106.8)
    KCQC = Location(latitude=35, longitude=-105.667)
    KCTB = Location(latitude=48.6, longitude=-112.367)
    KCUT = Location(latitude=43.733, longitude=-103.617)
    KCVN = Location(latitude=34.417, longitude=-103.067)
    KCVS = Location(latitude=34.367, longitude=-103.317)
    KCWN = Location(latitude=38.75, longitude=-104.85)
    KCYS = Location(latitude=41.15, longitude=-104.8)
    KD07 = Location(latitude=45.033, longitude=-102.017)
    KDEN = Location(latitude=39.85, longitude=-104.65)
    KDGW = Location(latitude=42.8, longitude=-105.383)
    KDIK = Location(latitude=46.8, longitude=-102.8)
    KDLN = Location(latitude=45.25, longitude=-112.55)
    KDMN = Location(latitude=32.267, longitude=-107.717)
    KDPG = Location(latitude=40.2, longitude=-112.917)
    KDRO = Location(latitude=37.15, longitude=-107.767)
    KDTA = Location(latitude=39.383, longitude=-112.5)
    KDUB = Location(latitude=43.55, longitude=-109.683)
    KDWX = Location(latitude=41.033, longitude=-107.5)
    KE80 = Location(latitude=34.65, longitude=-106.833)
    KECS = Location(latitude=43.883, longitude=-104.317)
    KEEO = Location(latitude=40.05, longitude=-107.883)
    KEGE = Location(latitude=39.633, longitude=-106.917)
    KEHY = Location(latitude=41.733, longitude=-106.467)
    KEIK = Location(latitude=40.017, longitude=-105.05)
    KEKS = Location(latitude=45.267, longitude=-111.65)
    KELP = Location(latitude=31.817, longitude=-106.383)
    KEMM = Location(latitude=41.817, longitude=-110.55)
    KENV = Location(latitude=40.717, longitude=-114.017)
    KEVW = Location(latitude=41.267, longitude=-111.033)
    KFBR = Location(latitude=41.4, longitude=-110.4)
    KFCS = Location(latitude=38.667, longitude=-104.767)
    KFEW = Location(latitude=41.133, longitude=-104.867)
    KFLY = Location(latitude=38.95, longitude=-104.567)
    KFMM = Location(latitude=40.333, longitude=-103.8)
    KFMN = Location(latitude=36.75, longitude=-108.233)
    KFNL = Location(latitude=40.45, longitude=-105.017)
    KFTG = Location(latitude=39.783, longitude=-104.55)
    KFWZ = Location(latitude=42.517, longitude=-108.783)
    KGCC = Location(latitude=44.333, longitude=-105.55)
    KGDV = Location(latitude=47.117, longitude=-104.783)
    KGEY = Location(latitude=44.517, longitude=-108.083)
    KGFA = Location(latitude=47.5, longitude=-111.167)
    KGGW = Location(latitude=48.217, longitude=-106.617)
    KGJT = Location(latitude=39.117, longitude=-108.517)
    KGLD = Location(latitude=39.367, longitude=-101.7)
    KGNT = Location(latitude=35.167, longitude=-107.9)
    KGPI = Location(latitude=48.317, longitude=-114.25)
    KGTF = Location(latitude=47.467, longitude=-111.383)
    KGUC = Location(latitude=38.517, longitude=-106.917)
    KGUP = Location(latitude=35.517, longitude=-108.8)
    KGUR = Location(latitude=42.25, longitude=-104.717)
    KGXY = Location(latitude=40.417, longitude=-104.617)
    KHDN = Location(latitude=40.467, longitude=-107.217)
    KHEI = Location(latitude=46.017, longitude=-102.65)
    KHEQ = Location(latitude=40.567, longitude=-102.283)
    KHIF = Location(latitude=41.117, longitude=-111.967)
    KHLN = Location(latitude=46.6, longitude=-111.983)
    KHMN = Location(latitude=32.85, longitude=-106.083)
    KHOB = Location(latitude=32.667, longitude=-103.217)
    KHVE = Location(latitude=38.417, longitude=-110.7)
    KHVR = Location(latitude=48.55, longitude=-109.767)
    KIBM = Location(latitude=41.183, longitude=-103.683)
    KIEN = Location(latitude=43.033, longitude=-102.517)
    KIML = Location(latitude=40.517, longitude=-101.617)
    KITR = Location(latitude=39.233, longitude=-102.283)
    KJAC = Location(latitude=43.617, longitude=-110.733)
    KJDN = Location(latitude=47.333, longitude=-106.95)
    KLAA = Location(latitude=38.067, longitude=-102.683)
    KLAM = Location(latitude=35.867, longitude=-106.267)
    KLAR = Location(latitude=41.317, longitude=-105.667)
    KLGU = Location(latitude=41.783, longitude=-111.85)
    KLHX = Location(latitude=38.05, longitude=-103.517)
    KLIC = Location(latitude=39.267, longitude=-103.667)
    KLMO = Location(latitude=40.167, longitude=-105.167)
    KLND = Location(latitude=42.817, longitude=-108.733)
    KLRU = Location(latitude=32.267, longitude=-106.917)
    KLSB = Location(latitude=32.333, longitude=-108.7)
    KLVM = Location(latitude=45.7, longitude=-110.433)
    KLVS = Location(latitude=35.65, longitude=-105.133)
    KLWT = Location(latitude=47.033, longitude=-109.467)
    KLXV = Location(latitude=39.233, longitude=-106.317)
    KM63 = Location(latitude=32.05, longitude=-106.15)
    KMLF = Location(latitude=38.417, longitude=-113.017)
    KMLP = Location(latitude=47.45, longitude=-115.667)
    KMLS = Location(latitude=46.433, longitude=-105.883)
    KMNH = Location(latitude=39.217, longitude=-104.65)
    KMSO = Location(latitude=46.917, longitude=-114.1)
    KMTJ = Location(latitude=38.5, longitude=-107.9)
    KMVH = Location(latitude=46.833, longitude=-108.933)
    KMYP = Location(latitude=38.483, longitude=-106.317)
    KOGA = Location(latitude=41.117, longitude=-101.767)
    KOGD = Location(latitude=41.2, longitude=-112.017)
    KOLF = Location(latitude=48.1, longitude=-105.583)
    KONM = Location(latitude=34.017, longitude=-106.9)
    KP60 = Location(latitude=44.55, longitude=-110.417)
    KPGA = Location(latitude=36.917, longitude=-111.45)
    KPHP = Location(latitude=44.05, longitude=-101.6)
    KPNA = Location(latitude=42.8, longitude=-109.8)
    KPOY = Location(latitude=44.867, longitude=-108.8)
    KPSO = Location(latitude=37.283, longitude=-107.05)
    KPUB = Location(latitude=38.283, longitude=-104.5)
    KPUC = Location(latitude=39.617, longitude=-110.75)
    KPVU = Location(latitude=40.217, longitude=-111.717)
    KRAP = Location(latitude=44.05, longitude=-103.05)
    KRCA = Location(latitude=44.133, longitude=-103.083)
    KRCV = Location(latitude=37.717, longitude=-106.35)
    KRIL = Location(latitude=39.533, longitude=-107.733)
    KRIW = Location(latitude=43.067, longitude=-108.467)
    KRKS = Location(latitude=41.6, longitude=-109.067)
    KROW = Location(latitude=33.3, longitude=-104.5)
    KRPX = Location(latitude=46.467, longitude=-108.55)
    KRQE = Location(latitude=35.65, longitude=-109.067)
    KRTN = Location(latitude=36.733, longitude=-104.5)
    KRWL = Location(latitude=41.783, longitude=-107.2)
    KSAA = Location(latitude=41.45, longitude=-106.833)
    KSAF = Location(latitude=35.617, longitude=-106.1)
    KSBS = Location(latitude=40.517, longitude=-106.867)
    KSDY = Location(latitude=47.7, longitude=-104.2)
    KSGU = Location(latitude=37.067, longitude=-113.517)
    KSHM = Location(latitude=38.8, longitude=-104.517)
    KSHR = Location(latitude=44.767, longitude=-106.967)
    KSKX = Location(latitude=36.45, longitude=-105.667)
    KSLC = Location(latitude=40.767, longitude=-111.967)
    KSNY = Location(latitude=41.1, longitude=-102.983)
    KSPD = Location(latitude=37.283, longitude=-102.617)
    KSPF = Location(latitude=44.467, longitude=-103.767)
    KSRR = Location(latitude=33.467, longitude=-105.517)
    KSTK = Location(latitude=40.617, longitude=-103.267)
    KSVC = Location(latitude=32.633, longitude=-108.15)
    KTAD = Location(latitude=37.267, longitude=-104.333)
    KTBX = Location(latitude=43.467, longitude=-108.233)
    KTCC = Location(latitude=35.183, longitude=-103.6)
    KTCS = Location(latitude=33.233, longitude=-107.267)
    KTEX = Location(latitude=37.95, longitude=-107.9)
    KTOR = Location(latitude=42.067, longitude=-104.15)
    KTVY = Location(latitude=40.617, longitude=-112.35)
    KU16 = Location(latitude=41.05, longitude=-113.067)
    KU42 = Location(latitude=40.617, longitude=-111.983)
    KU68 = Location(latitude=44.917, longitude=-108.45)
    KVEL = Location(latitude=40.45, longitude=-109.517)
    KVTP = Location(latitude=37.5, longitude=-105.167)
    KW43 = Location(latitude=44.667, longitude=-104.567)
    KWRL = Location(latitude=43.967, longitude=-107.95)
    KWYS = Location(latitude=44.667, longitude=-111.117)
    KACB = Location(latitude=44.983, longitude=-85.2)
    KADG = Location(latitude=41.867, longitude=-84.083)
    KAMN = Location(latitude=43.317, longitude=-84.683)
    KANJ = Location(latitude=46.467, longitude=-84.367)
    KAPN = Location(latitude=45.067, longitude=-83.567)
    KARB = Location(latitude=42.217, longitude=-83.733)
    KAZO = Location(latitude=42.233, longitude=-85.55)
    KBAX = Location(latitude=43.783, longitude=-82.983)
    KBEH = Location(latitude=42.133, longitude=-86.417)
    KBIV = Location(latitude=42.75, longitude=-86.1)
    KBTL = Location(latitude=42.317, longitude=-85.233)
    KCAD = Location(latitude=44.267, longitude=-85.417)
    KCFS = Location(latitude=43.467, longitude=-83.45)
    KCIU = Location(latitude=46.25, longitude=-84.467)
    KCMX = Location(latitude=47.167, longitude=-88.483)
    KCVX = Location(latitude=45.3, longitude=-85.267)
    KD95 = Location(latitude=43.067, longitude=-83.267)
    KDET = Location(latitude=42.4, longitude=-83.017)
    KDRM = Location(latitude=46, longitude=-83.75)
    KDTW = Location(latitude=42.233, longitude=-83.333)
    KDUH = Location(latitude=41.733, longitude=-83.65)
    KERY = Location(latitude=46.3, longitude=-85.45)
    KESC = Location(latitude=45.75, longitude=-87.083)
    KFFX = Location(latitude=43.433, longitude=-86)
    KFKS = Location(latitude=44.633, longitude=-86.2)
    KFNT = Location(latitude=42.967, longitude=-83.75)
    KFPK = Location(latitude=42.567, longitude=-84.817)
    KGLR = Location(latitude=45.017, longitude=-84.683)
    KGOV = Location(latitude=44.683, longitude=-84.733)
    KGRR = Location(latitude=42.883, longitude=-85.517)
    KHAI = Location(latitude=41.967, longitude=-85.6)
    KHTL = Location(latitude=44.35, longitude=-84.667)
    KHYX = Location(latitude=43.433, longitude=-83.867)
    KIKW = Location(latitude=43.667, longitude=-84.267)
    KIRS = Location(latitude=41.8, longitude=-85.433)
    KISQ = Location(latitude=45.967, longitude=-86.167)
    KJXN = Location(latitude=42.267, longitude=-84.467)
    KJYM = Location(latitude=41.917, longitude=-84.583)
    KLAN = Location(latitude=42.783, longitude=-84.583)
    KLDM = Location(latitude=43.967, longitude=-86.4)
    KLWA = Location(latitude=42.35, longitude=-86.25)
    KMBL = Location(latitude=44.267, longitude=-86.25)
    KMBS = Location(latitude=43.533, longitude=-84.083)
    KMCD = Location(latitude=45.85, longitude=-84.633)
    KMGN = Location(latitude=45.433, longitude=-84.917)
    KMKG = Location(latitude=43.167, longitude=-86.233)
    KMOP = Location(latitude=43.617, longitude=-84.733)
    KMTC = Location(latitude=42.617, longitude=-82.817)
    KOEB = Location(latitude=41.917, longitude=-85.033)
    KONZ = Location(latitude=42.1, longitude=-83.15)
    KOSC = Location(latitude=44.45, longitude=-83.4)
    KOZW = Location(latitude=42.617, longitude=-83.967)
    KP53 = Location(latitude=46.417, longitude=-86.65)
    KP58 = Location(latitude=44.017, longitude=-82.8)
    KP59 = Location(latitude=47.467, longitude=-87.883)
    KPHN = Location(latitude=42.917, longitude=-82.517)
    KPLN = Location(latitude=45.567, longitude=-84.8)
    KPTK = Location(latitude=42.667, longitude=-83.417)
    KPZQ = Location(latitude=45.4, longitude=-83.817)
    KRMY = Location(latitude=42.233, longitude=-84.95)
    KRNP = Location(latitude=43, longitude=-84.133)
    KRQB = Location(latitude=43.717, longitude=-85.5)
    KSAW = Location(latitude=46.35, longitude=-87.4)
    KSJX = Location(latitude=45.7, longitude=-85.567)
    KSLH = Location(latitude=45.65, longitude=-84.517)
    KTEW = Location(latitude=42.567, longitude=-84.417)
    KTTF = Location(latitude=41.933, longitude=-83.417)
    KTVC = Location(latitude=44.733, longitude=-85.567)
    KVLL = Location(latitude=42.55, longitude=-83.183)
    KY70 = Location(latitude=42.933, longitude=-85.067)
    KYIP = Location(latitude=42.233, longitude=-83.533)
    TDCF = Location(latitude=15.333, longitude=-61.4)
    TDPD = Location(latitude=15.55, longitude=-61.3)
    CPBT = Location(latitude=49.567, longitude=-113.05)
    CPEH = Location(latitude=50.183, longitude=-112.433)
    CPFI = Location(latitude=49.8, longitude=-112.033)
    CPIR = Location(latitude=49.9, longitude=-112.733)
    CPRO = Location(latitude=50.833, longitude=-112.05)
    CPRY = Location(latitude=49.483, longitude=-112.683)
    CPST = Location(latitude=51.033, longitude=-113.283)
    CPSV = Location(latitude=49.917, longitude=-110.917)
    CPXL = Location(latitude=49.867, longitude=-111.383)
    CWAV = Location(latitude=51.767, longitude=-114.667)
    CWBO = Location(latitude=50.533, longitude=-111.833)
    CWCT = Location(latitude=52.067, longitude=-111.467)
    CWDK = Location(latitude=50.017, longitude=-113.617)
    CWDZ = Location(latitude=51.417, longitude=-112.667)
    CWFJ = Location(latitude=49.2, longitude=-113.267)
    CWGM = Location(latitude=49.133, longitude=-113.783)
    CWGW = Location(latitude=49.75, longitude=-114.9)
    CWGY = Location(latitude=51.667, longitude=-110.2)
    CWHI = Location(latitude=51.7, longitude=-113.217)
    CWJW = Location(latitude=52.933, longitude=-118.033)
    CWLB = Location(latitude=54.767, longitude=-112.017)
    CWOE = Location(latitude=49.117, longitude=-110.467)
    CWRM = Location(latitude=52.417, longitude=-114.917)
    CWRT = Location(latitude=49.617, longitude=-114.467)
    CWRY = Location(latitude=49.117, longitude=-112.033)
    CWSW = Location(latitude=49.75, longitude=-114.867)
    CWVI = Location(latitude=53.517, longitude=-112.1)
    CWXA = Location(latitude=51.067, longitude=-115.067)
    CWXL = Location(latitude=49.733, longitude=-111.45)
    CWYL = Location(latitude=51.45, longitude=-116.317)
    CWZG = Location(latitude=51.2, longitude=-115.533)
    CXAF = Location(latitude=54.283, longitude=-112.967)
    CXAG = Location(latitude=53.917, longitude=-112.283)
    CXAJ = Location(latitude=54.767, longitude=-112.817)
    CXAK = Location(latitude=52.35, longitude=-112.6)
    CXBA = Location(latitude=52.117, longitude=-110.1)
    CXBR = Location(latitude=49.617, longitude=-113.817)
    CXBW = Location(latitude=49.8, longitude=-112.3)
    CXCD = Location(latitude=56.317, longitude=-119.75)
    CXCP = Location(latitude=50.283, longitude=-113.35)
    CXCS = Location(latitude=51.933, longitude=-110.717)
    CXDB = Location(latitude=49.05, longitude=-112.817)
    CXDP = Location(latitude=54.317, longitude=-113.95)
    CXEC = Location(latitude=53.567, longitude=-113.517)
    CXEG = Location(latitude=53.3, longitude=-113.6)
    CXFM = Location(latitude=49.483, longitude=-111.483)
    CXFR = Location(latitude=56.083, longitude=-118.433)
    CXHD = Location(latitude=53.183, longitude=-112.25)
    CXHP = Location(latitude=55.4, longitude=-116.483)
    CXHR = Location(latitude=51.183, longitude=-112.5)
    CXKM = Location(latitude=52.817, longitude=-111.85)
    CXMG = Location(latitude=56.967, longitude=-117.45)
    CXMM = Location(latitude=56.65, longitude=-111.217)
    CXMO = Location(latitude=51.667, longitude=-112.683)
    CXOL = Location(latitude=51.767, longitude=-114.083)
    CXOY = Location(latitude=51.383, longitude=-110.35)
    CXPA = Location(latitude=55.617, longitude=-118.3)
    CXPL = Location(latitude=51.133, longitude=-111.7)
    CXRL = Location(latitude=54.5, longitude=-111.7)
    CXSC = Location(latitude=50.3, longitude=-110.083)
    CXSL = Location(latitude=54.283, longitude=-112.5)
    CXSP = Location(latitude=54.017, longitude=-111.267)
    CXSR = Location(latitude=55.7, longitude=-119.233)
    CXTH = Location(latitude=53.633, longitude=-111.683)
    CXVM = Location(latitude=53.35, longitude=-110.883)
    CXVW = Location(latitude=55.1, longitude=-117.2)
    CXWM = Location(latitude=49.5, longitude=-112.117)
    CXZU = Location(latitude=54.15, longitude=-115.783)
    CYBW = Location(latitude=51.1, longitude=-114.367)
    CYED = Location(latitude=53.667, longitude=-113.467)
    CYEG = Location(latitude=53.283, longitude=-113.567)
    CYET = Location(latitude=53.567, longitude=-116.467)
    CYGE = Location(latitude=51.3, longitude=-116.983)
    CYLL = Location(latitude=53.317, longitude=-110.067)
    CYOD = Location(latitude=54.4, longitude=-110.283)
    CYOJ = Location(latitude=58.617, longitude=-117.167)
    CYPE = Location(latitude=56.217, longitude=-117.45)
    CYPY = Location(latitude=58.767, longitude=-111.117)
    CYQF = Location(latitude=52.167, longitude=-113.9)
    CYQL = Location(latitude=49.633, longitude=-112.8)
    CYQU = Location(latitude=55.183, longitude=-118.883)
    CYSD = Location(latitude=50.267, longitude=-111.167)
    CYXC = Location(latitude=49.6, longitude=-115.767)
    CYXH = Location(latitude=50.017, longitude=-110.717)
    CYYC = Location(latitude=51.117, longitude=-114.017)
    CYZH = Location(latitude=55.283, longitude=-114.767)
    CZMU = Location(latitude=53.567, longitude=-112.3)
    CZOL = Location(latitude=53.65, longitude=-113.35)
    CZPC = Location(latitude=49.517, longitude=-114)
    CZPS = Location(latitude=52.433, longitude=-113.6)
    CZVL = Location(latitude=53.667, longitude=-113.85)
    CZZJ = Location(latitude=53.583, longitude=-116.467)
    MSAC = Location(latitude=13.567, longitude=-89.833)
    MSLP = Location(latitude=13.433, longitude=-89.05)
    MSSA = Location(latitude=13.967, longitude=-89.567)
    MSSM = Location(latitude=13.433, longitude=-88.117)
    MSSS = Location(latitude=13.7, longitude=-89.117)
    SBCI = Location(latitude=-7.317, longitude=-47.467)
    SBFZ = Location(latitude=-3.767, longitude=-38.517)
    SBIZ = Location(latitude=-5.517, longitude=-47.45)
    SBJP = Location(latitude=-7.15, longitude=-34.95)
    SBJU = Location(latitude=-7.217, longitude=-39.267)
    SBKG = Location(latitude=-7.267, longitude=-35.867)
    SBMS = Location(latitude=-5.2, longitude=-37.367)
    SBNT = Location(latitude=-5.917, longitude=-35.25)
    SBPB = Location(latitude=-2.917, longitude=-41.75)
    SBSL = Location(latitude=-2.6, longitude=-44.217)
    SBTE = Location(latitude=-5.033, longitude=-42.817)
    CYQY = Location(latitude=46.167, longitude=-60.033)
    BGAA = Location(latitude=68.7, longitude=-52.8)
    BGBW = Location(latitude=61.167, longitude=-45.417)
    BGGH = Location(latitude=64.2, longitude=-51.683)
    BGJN = Location(latitude=69.25, longitude=-51.067)
    BGKK = Location(latitude=65.567, longitude=-37.133)
    BGMQ = Location(latitude=65.417, longitude=-52.933)
    BGPT = Location(latitude=62.017, longitude=-49.667)
    BGSF = Location(latitude=67, longitude=-50.717)
    BGSS = Location(latitude=66.95, longitude=-53.733)
    BGUK = Location(latitude=72.783, longitude=-56.133)
    BGUQ = Location(latitude=70.733, longitude=-52.7)
    CWCA = Location(latitude=53.7, longitude=-57.017)
    CWHO = Location(latitude=55.45, longitude=-60.217)
    CWKW = Location(latitude=59.967, longitude=-64.167)
    CWTU = Location(latitude=54.717, longitude=-58.35)
    CWYK = Location(latitude=57.117, longitude=-61.467)
    CWZZ = Location(latitude=58.317, longitude=-62.567)
    CYDP = Location(latitude=56.533, longitude=-61.667)
    CYFT = Location(latitude=55.067, longitude=-59.167)
    CYWK = Location(latitude=52.917, longitude=-66.867)
    CYYR = Location(latitude=53.317, longitude=-60.417)
    CZUM = Location(latitude=53.533, longitude=-64.083)
    MBGT = Location(latitude=21.433, longitude=-71.15)
    MBPV = Location(latitude=21.767, longitude=-72.267)
    TGPY = Location(latitude=12, longitude=-61.783)
    TFFR = Location(latitude=16.267, longitude=-61.533)
    MGCB = Location(latitude=15.483, longitude=-90.417)
    MGES = Location(latitude=14.567, longitude=-89.317)
    MGGT = Location(latitude=14.567, longitude=-90.533)
    MGHT = Location(latitude=15.317, longitude=-91.5)
    MGMM = Location(latitude=16.967, longitude=-89.867)
    MGPB = Location(latitude=15.717, longitude=-88.583)
    MGQZ = Location(latitude=14.867, longitude=-91.5)
    MGRT = Location(latitude=14.533, longitude=-91.7)
    MGSJ = Location(latitude=13.917, longitude=-90.833)
    SEAM = Location(latitude=-1.183, longitude=-78.567)
    SECU = Location(latitude=-2.867, longitude=-78.967)
    SEGU = Location(latitude=-2.133, longitude=-79.867)
    SELT = Location(latitude=-0.917, longitude=-78.617)
    SEMT = Location(latitude=-0.95, longitude=-80.667)
    SESA = Location(latitude=-2.2, longitude=-80.967)
    SETU = Location(latitude=0.817, longitude=-77.7)
    SYCJ = Location(latitude=6.5, longitude=-58.25)
    SYGO = Location(latitude=6.8, longitude=-58.1)
    CAHR = Location(latitude=46.35, longitude=-63.167)
    CWBK = Location(latitude=45.767, longitude=-62.667)
    CWBV = Location(latitude=44.817, longitude=-62.317)
    CWEF = Location(latitude=47.217, longitude=-60.117)
    CWEP = Location(latitude=46.45, longitude=-61.967)
    CWGR = Location(latitude=47.417, longitude=-61.8)
    CWKG = Location(latitude=44.433, longitude=-65.2)
    CWNE = Location(latitude=47.067, longitude=-64)
    CWRN = Location(latitude=45.35, longitude=-60.983)
    CWSA = Location(latitude=43.933, longitude=-60)
    CWSD = Location(latitude=46.417, longitude=-63.85)
    CWVU = Location(latitude=44.283, longitude=-66.333)
    CWWE = Location(latitude=43.967, longitude=-64.667)
    CWZQ = Location(latitude=46.533, longitude=-61.05)
    CXCH = Location(latitude=46.633, longitude=-60.95)
    CXIB = Location(latitude=46.667, longitude=-60.383)
    CXKT = Location(latitude=45.067, longitude=-64.483)
    CXLB = Location(latitude=44.367, longitude=-64.3)
    CXMI = Location(latitude=44.6, longitude=-63.533)
    CXMY = Location(latitude=44.983, longitude=-62.483)
    CXNM = Location(latitude=46.817, longitude=-60.667)
    CXNP = Location(latitude=45.75, longitude=-64.233)
    CXTD = Location(latitude=45.617, longitude=-61.683)
    CYAW = Location(latitude=44.633, longitude=-63.5)
    CYGR = Location(latitude=47.417, longitude=-61.767)
    CYHZ = Location(latitude=44.867, longitude=-63.5)
    CYPD = Location(latitude=45.667, longitude=-61.367)
    CYQI = Location(latitude=43.833, longitude=-66.083)
    CYYG = Location(latitude=46.267, longitude=-63.117)
    CYZX = Location(latitude=44.983, longitude=-64.917)
    CZDB = Location(latitude=45.417, longitude=-63.467)
    CZSP = Location(latitude=46.45, longitude=-62.583)
    MUCC = Location(latitude=22.467, longitude=-78.333)
    MUCF = Location(latitude=22.133, longitude=-80.4)
    MUCL = Location(latitude=21.633, longitude=-81.55)
    MUCM = Location(latitude=21.417, longitude=-77.85)
    MUCU = Location(latitude=19.967, longitude=-75.833)
    MUHA = Location(latitude=22.967, longitude=-82.417)
    MUHG = Location(latitude=20.767, longitude=-76.317)
    MUMZ = Location(latitude=20.283, longitude=-77.1)
    MUSC = Location(latitude=22.5, longitude=-79.95)
    MUVR = Location(latitude=23.033, longitude=-81.433)
    MUVT = Location(latitude=20.95, longitude=-76.95)
    MMCN = Location(latitude=27.4, longitude=-109.833)
    MMGM = Location(latitude=27.95, longitude=-110.917)
    MMHO = Location(latitude=29.083, longitude=-111.05)
    K1II = Location(latitude=39.333, longitude=-86.033)
    K4I7 = Location(latitude=39.633, longitude=-86.817)
    KAID = Location(latitude=40.117, longitude=-85.617)
    KANQ = Location(latitude=41.633, longitude=-85.083)
    KASW = Location(latitude=41.267, longitude=-85.833)
    KBAK = Location(latitude=39.267, longitude=-85.9)
    KBMG = Location(latitude=39.15, longitude=-86.617)
    KCFJ = Location(latitude=39.983, longitude=-86.917)
    KEKM = Location(latitude=41.717, longitude=-86)
    KEYE = Location(latitude=39.833, longitude=-86.3)
    KFKR = Location(latitude=40.267, longitude=-86.567)
    KFWA = Location(latitude=40.983, longitude=-85.183)
    KGGP = Location(latitude=40.717, longitude=-86.367)
    KGSH = Location(latitude=41.533, longitude=-85.783)
    KGUS = Location(latitude=40.633, longitude=-86.15)
    KGWB = Location(latitude=41.3, longitude=-85.067)
    KHHG = Location(latitude=40.85, longitude=-85.45)
    KHUF = Location(latitude=39.45, longitude=-87.3)
    KIMS = Location(latitude=38.75, longitude=-85.467)
    KIND = Location(latitude=39.717, longitude=-86.3)
    KLAF = Location(latitude=40.417, longitude=-86.933)
    KMCX = Location(latitude=40.717, longitude=-86.767)
    KMIE = Location(latitude=40.233, longitude=-85.4)
    KMQJ = Location(latitude=39.85, longitude=-85.9)
    KMZZ = Location(latitude=40.483, longitude=-85.683)
    KOKK = Location(latitude=40.533, longitude=-86.067)
    KRCR = Location(latitude=41.067, longitude=-86.183)
    KSBN = Location(latitude=41.7, longitude=-86.317)
    KTYQ = Location(latitude=40.033, longitude=-86.25)
    KOXI = Location(latitude=41.333, longitude=-86.667)
    KDCY = Location(latitude=38.7, longitude=-87.133)
    KHNB = Location(latitude=38.25, longitude=-86.95)
    CYEV = Location(latitude=68.317, longitude=-133.5)
    CZEV = Location(latitude=68.317, longitude=-133.517)
    CWGZ = Location(latitude=76.417, longitude=-82.9)
    CWLX = Location(latitude=68.9, longitude=-75.15)
    CWRX = Location(latitude=69.067, longitude=-79.017)
    CWUW = Location(latitude=68.65, longitude=-71.167)
    CYCY = Location(latitude=70.467, longitude=-68.517)
    CYFB = Location(latitude=63.75, longitude=-68.55)
    CYGT = Location(latitude=69.367, longitude=-81.817)
    CYIO = Location(latitude=72.667, longitude=-77.967)
    CYLC = Location(latitude=62.85, longitude=-69.867)
    CYSK = Location(latitude=56.533, longitude=-79.25)
    CYTE = Location(latitude=64.217, longitude=-76.517)
    CYUX = Location(latitude=68.767, longitude=-81.25)
    MKJP = Location(latitude=17.917, longitude=-76.783)
    MKJS = Location(latitude=18.5, longitude=-77.917)
    PAGN = Location(latitude=57.5, longitude=-134.583)
    PAGS = Location(latitude=58.417, longitude=-135.733)
    PAGY = Location(latitude=59.467, longitude=-135.3)
    PAHN = Location(latitude=59.25, longitude=-135.517)
    PAJN = Location(latitude=58.35, longitude=-134.583)
    PAOH = Location(latitude=58.1, longitude=-135.45)
    KEKQ = Location(latitude=36.85, longitude=-84.85)
    TNCB = Location(latitude=12.133, longitude=-68.283)
    SLCA = Location(latitude=-20, longitude=-63.533)
    SLCB = Location(latitude=-17.417, longitude=-66.183)
    SLCO = Location(latitude=-11.033, longitude=-68.783)
    SLCP = Location(latitude=-16.15, longitude=-62.017)
    SLET = Location(latitude=-17.8, longitude=-63.167)
    SLJE = Location(latitude=-17.817, longitude=-60.75)
    SLJO = Location(latitude=-13.067, longitude=-64.667)
    SLLP = Location(latitude=-16.517, longitude=-68.167)
    SLMG = Location(latitude=-13.267, longitude=-64.067)
    SLOR = Location(latitude=-17.967, longitude=-67.083)
    SLPO = Location(latitude=-19.517, longitude=-65.717)
    SLPS = Location(latitude=-18.983, longitude=-57.817)
    SLRB = Location(latitude=-18.317, longitude=-59.75)
    SLRI = Location(latitude=-11.017, longitude=-66.083)
    SLRQ = Location(latitude=-14.417, longitude=-67.5)
    SLRY = Location(latitude=-14.3, longitude=-67.367)
    SLSA = Location(latitude=-13.767, longitude=-65.433)
    SLSI = Location(latitude=-16.367, longitude=-60.95)
    SLSU = Location(latitude=-19.017, longitude=-65.267)
    SLTI = Location(latitude=-16.33, longitude=-58.38)
    SLTJ = Location(latitude=-21.55, longitude=-64.717)
    SLTR = Location(latitude=-14.833, longitude=-64.917)
    SLVM = Location(latitude=-21.25, longitude=-63.4)
    SLVR = Location(latitude=-17.633, longitude=-63.117)
    SLYA = Location(latitude=-21.967, longitude=-63.65)
    SPCL = Location(latitude=-8.383, longitude=-74.583)
    SPGM = Location(latitude=-9.283, longitude=-76)
    SPHI = Location(latitude=-6.767, longitude=-79.817)
    SPHO = Location(latitude=-13.167, longitude=-74.217)
    SPHY = Location(latitude=-13.7, longitude=-73.35)
    SPIM = Location(latitude=-12.017, longitude=-77.117)
    SPJI = Location(latitude=-7.167, longitude=-76.717)
    SPJL = Location(latitude=-15.467, longitude=-70.15)
    SPJR = Location(latitude=-7.117, longitude=-78.467)
    SPLO = Location(latitude=-17.7, longitude=-71.35)
    SPME = Location(latitude=-3.533, longitude=-80.4)
    SPMS = Location(latitude=-5.9, longitude=-76.117)
    SPQT = Location(latitude=-3.783, longitude=-73.317)
    SPQU = Location(latitude=-16.333, longitude=-71.567)
    SPRU = Location(latitude=-8.083, longitude=-79.117)
    SPSO = Location(latitude=-13.733, longitude=-76.217)
    SPST = Location(latitude=-6.517, longitude=-76.367)
    SPTN = Location(latitude=-18.067, longitude=-70.283)
    SPTU = Location(latitude=-12.617, longitude=-69.2)
    SPYL = Location(latitude=-4.567, longitude=-81.25)
    SPZA = Location(latitude=-14.867, longitude=-74.95)
    SPZO = Location(latitude=-13.55, longitude=-71.967)
    K05U = Location(latitude=39.6, longitude=-116)
    K0S9 = Location(latitude=48.05, longitude=-122.817)
    K1QW = Location(latitude=46.8, longitude=-120.167)
    K1S5 = Location(latitude=46.333, longitude=-119.967)
    K65S = Location(latitude=48.733, longitude=-116.3)
    K9L2 = Location(latitude=34.967, longitude=-117.867)
    KAAT = Location(latitude=41.483, longitude=-120.567)
    KACV = Location(latitude=40.983, longitude=-124.1)
    KAJO = Location(latitude=33.9, longitude=-117.6)
    KALW = Location(latitude=46.1, longitude=-118.283)
    KAPC = Location(latitude=38.2, longitude=-122.283)
    KAPV = Location(latitude=34.567, longitude=-117.183)
    KAST = Location(latitude=46.15, longitude=-123.883)
    KAUN = Location(latitude=38.95, longitude=-121.083)
    KAVX = Location(latitude=33.4, longitude=-118.417)
    KAWO = Location(latitude=48.167, longitude=-122.167)
    KBAB = Location(latitude=39.117, longitude=-121.417)
    KBAN = Location(latitude=38.35, longitude=-119.517)
    KBDN = Location(latitude=44.1, longitude=-121.2)
    KBFI = Location(latitude=47.55, longitude=-122.317)
    KBFL = Location(latitude=35.433, longitude=-119.05)
    KBIH = Location(latitude=37.367, longitude=-118.35)
    KBJN = Location(latitude=37.617, longitude=-116.267)
    KBKE = Location(latitude=44.85, longitude=-117.817)
    KBLH = Location(latitude=33.617, longitude=-114.717)
    KBLI = Location(latitude=48.8, longitude=-122.533)
    KBLU = Location(latitude=39.283, longitude=-120.7)
    KBNO = Location(latitude=43.6, longitude=-118.95)
    KBOK = Location(latitude=42.067, longitude=-124.283)
    KBUR = Location(latitude=34.2, longitude=-118.367)
    KBVS = Location(latitude=48.467, longitude=-122.417)
    KBVU = Location(latitude=35.95, longitude=-114.867)
    KBYS = Location(latitude=35.283, longitude=-116.617)
    KCCR = Location(latitude=38, longitude=-122.05)
    KCEC = Location(latitude=41.767, longitude=-124.217)
    KCIC = Location(latitude=39.783, longitude=-121.833)
    KCLM = Location(latitude=48.117, longitude=-123.5)
    KCLS = Location(latitude=46.683, longitude=-122.983)
    KCMA = Location(latitude=34.217, longitude=-119.1)
    KCNO = Location(latitude=33.983, longitude=-117.617)
    KCOE = Location(latitude=47.767, longitude=-116.817)
    KCQT = Location(latitude=34.017, longitude=-118.283)
    KCRQ = Location(latitude=33.133, longitude=-117.283)
    KCVH = Location(latitude=36.9, longitude=-121.417)
    KCVO = Location(latitude=44.5, longitude=-123.267)
    KCXP = Location(latitude=39.183, longitude=-119.733)
    KCZZ = Location(latitude=32.633, longitude=-116.467)
    KDAG = Location(latitude=34.85, longitude=-116.783)
    KDEW = Location(latitude=47.967, longitude=-117.433)
    KDLS = Location(latitude=45.617, longitude=-121.167)
    KDRA = Location(latitude=36.617, longitude=-116.033)
    KDVO = Location(latitude=38.15, longitude=-122.55)
    KE16 = Location(latitude=37.083, longitude=-121.6)
    KEAT = Location(latitude=47.4, longitude=-120.2)
    KEDU = Location(latitude=38.533, longitude=-121.783)
    KEDW = Location(latitude=34.883, longitude=-117.867)
    KEED = Location(latitude=34.767, longitude=-114.617)
    KEKO = Location(latitude=40.817, longitude=-115.783)
    KELN = Location(latitude=47.033, longitude=-120.533)
    KELY = Location(latitude=39.3, longitude=-114.85)
    KEMT = Location(latitude=34.067, longitude=-118.017)
    KEPH = Location(latitude=47.3, longitude=-119.517)
    KEUG = Location(latitude=44.133, longitude=-123.217)
    KF70 = Location(latitude=33.567, longitude=-117.133)
    KFAT = Location(latitude=36.783, longitude=-119.717)
    KFCH = Location(latitude=36.717, longitude=-119.817)
    KFHR = Location(latitude=48.517, longitude=-123.033)
    KFOT = Location(latitude=40.55, longitude=-124.133)
    KFUL = Location(latitude=33.867, longitude=-117.983)
    KGCD = Location(latitude=44.383, longitude=-118.967)
    KGEG = Location(latitude=47.617, longitude=-117.533)
    KGIC = Location(latitude=45.95, longitude=-116.117)
    KGOO = Location(latitude=39.217, longitude=-121)
    KGRF = Location(latitude=47.067, longitude=-122.567)
    KGXA = Location(latitude=34.567, longitude=-117.667)
    KHAF = Location(latitude=37.517, longitude=-122.5)
    KHHR = Location(latitude=33.917, longitude=-118.333)
    KHII = Location(latitude=34.567, longitude=-114.367)
    KHIO = Location(latitude=45.55, longitude=-122.95)
    KHJO = Location(latitude=36.317, longitude=-119.617)
    KHND = Location(latitude=35.983, longitude=-115.133)
    KHQM = Location(latitude=46.967, longitude=-123.933)
    KHRI = Location(latitude=45.833, longitude=-119.267)
    KHWD = Location(latitude=37.667, longitude=-122.117)
    KINS = Location(latitude=36.583, longitude=-115.667)
    KIPL = Location(latitude=32.833, longitude=-115.583)
    KIYK = Location(latitude=35.667, longitude=-117.817)
    KIZA = Location(latitude=34.6, longitude=-120.067)
    KKLS = Location(latitude=46.117, longitude=-122.9)
    KL18 = Location(latitude=33.35, longitude=-117.25)
    KL35 = Location(latitude=34.267, longitude=-116.85)
    KLAS = Location(latitude=36.067, longitude=-115.167)
    KLAX = Location(latitude=33.933, longitude=-118.383)
    KLGB = Location(latitude=33.817, longitude=-118.15)
    KLGD = Location(latitude=45.267, longitude=-118)
    KLHM = Location(latitude=38.917, longitude=-121.35)
    KLKV = Location(latitude=42.167, longitude=-120.4)
    KLMT = Location(latitude=42.15, longitude=-121.717)
    KLOL = Location(latitude=40.067, longitude=-118.567)
    KLPC = Location(latitude=34.667, longitude=-120.467)
    KLSV = Location(latitude=36.217, longitude=-115.017)
    KLVK = Location(latitude=37.7, longitude=-121.817)
    KLWS = Location(latitude=46.367, longitude=-117.017)
    KMAE = Location(latitude=36.983, longitude=-120.117)
    KMCC = Location(latitude=38.667, longitude=-121.4)
    KMCE = Location(latitude=37.283, longitude=-120.5)
    KMEH = Location(latitude=45.517, longitude=-118.417)
    KMER = Location(latitude=37.383, longitude=-120.567)
    KMEV = Location(latitude=38.983, longitude=-119.75)
    KMFR = Location(latitude=42.383, longitude=-122.867)
    KMHR = Location(latitude=38.533, longitude=-121.283)
    KMHS = Location(latitude=41.317, longitude=-122.317)
    KMHV = Location(latitude=35.067, longitude=-118.15)
    KMMH = Location(latitude=37.617, longitude=-118.817)
    KMMV = Location(latitude=45.2, longitude=-123.133)
    KMOD = Location(latitude=37.617, longitude=-120.95)
    KMRY = Location(latitude=36.583, longitude=-121.85)
    KMWH = Location(latitude=47.2, longitude=-119.317)
    KMWS = Location(latitude=34.233, longitude=-118.05)
    KMYF = Location(latitude=32.817, longitude=-117.133)
    KMYV = Location(latitude=39.1, longitude=-121.567)
    KNFG = Location(latitude=33.283, longitude=-117.333)
    KNFL = Location(latitude=39.417, longitude=-118.7)
    KNID = Location(latitude=35.667, longitude=-117.667)
    KNJK = Location(latitude=32.817, longitude=-115.667)
    KNKX = Location(latitude=32.867, longitude=-117.133)
    KNLC = Location(latitude=36.317, longitude=-119.95)
    KNOW = Location(latitude=48.133, longitude=-123.4)
    KNRS = Location(latitude=32.567, longitude=-117.117)
    KNSI = Location(latitude=33.217, longitude=-119.45)
    KNTD = Location(latitude=34.117, longitude=-119.117)
    KNUC = Location(latitude=33.017, longitude=-118.567)
    KNUQ = Location(latitude=37.4, longitude=-122.05)
    KNUW = Location(latitude=48.35, longitude=-122.65)
    KNXF = Location(latitude=33.283, longitude=-117.45)
    KNXP = Location(latitude=34.283, longitude=-116.167)
    KNZY = Location(latitude=32.7, longitude=-117.217)
    KO22 = Location(latitude=38.033, longitude=-120.417)
    KO54 = Location(latitude=40.75, longitude=-122.917)
    KO69 = Location(latitude=38.25, longitude=-122.6)
    KO86 = Location(latitude=40.983, longitude=-122.7)
    KOAK = Location(latitude=37.717, longitude=-122.233)
    KOKB = Location(latitude=33.217, longitude=-117.35)
    KOLM = Location(latitude=46.967, longitude=-122.9)
    KOMK = Location(latitude=48.467, longitude=-119.517)
    KONP = Location(latitude=44.567, longitude=-124.033)
    KONT = Location(latitude=34.05, longitude=-117.583)
    KORS = Location(latitude=48.7, longitude=-122.917)
    KOTH = Location(latitude=43.417, longitude=-124.25)
    KOVE = Location(latitude=39.5, longitude=-121.617)
    KOXR = Location(latitude=34.2, longitude=-119.2)
    KP69 = Location(latitude=46.15, longitude=-115.6)
    KPAE = Location(latitude=47.917, longitude=-122.283)
    KPAO = Location(latitude=37.467, longitude=-122.117)
    KPDT = Location(latitude=45.7, longitude=-118.833)
    KPDX = Location(latitude=45.6, longitude=-122.617)
    KPLU = Location(latitude=47.1, longitude=-122.283)
    KPMD = Location(latitude=34.633, longitude=-118.083)
    KPOC = Location(latitude=34.1, longitude=-117.767)
    KPRB = Location(latitude=35.667, longitude=-120.633)
    KPSC = Location(latitude=46.267, longitude=-119.117)
    KPSP = Location(latitude=33.817, longitude=-116.5)
    KPTV = Location(latitude=36.017, longitude=-119.067)
    KPUW = Location(latitude=46.75, longitude=-117.117)
    KPVF = Location(latitude=38.717, longitude=-120.75)
    KPWT = Location(latitude=47.5, longitude=-122.75)
    KRAL = Location(latitude=33.95, longitude=-117.45)
    KRBG = Location(latitude=43.233, longitude=-123.35)
    KRBL = Location(latitude=40.15, longitude=-122.25)
    KRDD = Location(latitude=40.517, longitude=-122.3)
    KRDM = Location(latitude=44.25, longitude=-121.15)
    KRHV = Location(latitude=37.317, longitude=-121.817)
    KRIV = Location(latitude=33.867, longitude=-117.267)
    KRNM = Location(latitude=33.033, longitude=-116.917)
    KRNO = Location(latitude=39.5, longitude=-119.767)
    KRNT = Location(latitude=47.5, longitude=-122.217)
    KRTS = Location(latitude=39.667, longitude=-119.883)
    KS39 = Location(latitude=44.283, longitude=-120.9)
    KSAC = Location(latitude=38.5, longitude=-121.5)
    KSAN = Location(latitude=32.733, longitude=-117.183)
    KSBA = Location(latitude=34.433, longitude=-119.85)
    KSBD = Location(latitude=34.083, longitude=-117.25)
    KSBP = Location(latitude=35.233, longitude=-120.633)
    KSCK = Location(latitude=37.883, longitude=-121.217)
    KSDB = Location(latitude=34.75, longitude=-118.717)
    KSDM = Location(latitude=32.583, longitude=-117)
    KSEA = Location(latitude=47.45, longitude=-122.317)
    KSEE = Location(latitude=32.817, longitude=-116.967)
    KSFF = Location(latitude=47.683, longitude=-117.317)
    KSFO = Location(latitude=37.617, longitude=-122.367)
    KSHN = Location(latitude=47.233, longitude=-123.133)
    KSIY = Location(latitude=41.767, longitude=-122.467)
    KSJC = Location(latitude=37.367, longitude=-121.917)
    KSKA = Location(latitude=47.617, longitude=-117.65)
    KSLE = Location(latitude=44.9, longitude=-123)
    KSLI = Location(latitude=33.767, longitude=-118.033)
    KSMF = Location(latitude=38.7, longitude=-121.6)
    KSMO = Location(latitude=34.017, longitude=-118.45)
    KSMX = Location(latitude=34.9, longitude=-120.45)
    KSNA = Location(latitude=33.667, longitude=-117.867)
    KSNS = Location(latitude=36.667, longitude=-121.6)
    KSPB = Location(latitude=45.767, longitude=-122.867)
    KSQL = Location(latitude=37.517, longitude=-122.25)
    KSTS = Location(latitude=38.5, longitude=-122.817)
    KSUU = Location(latitude=38.267, longitude=-121.917)
    KSXT = Location(latitude=42.6, longitude=-123.367)
    KSZT = Location(latitude=48.3, longitude=-116.567)
    KTCM = Location(latitude=47.117, longitude=-122.467)
    KTIW = Location(latitude=47.267, longitude=-122.583)
    KTMK = Location(latitude=45.417, longitude=-123.817)
    KTOA = Location(latitude=33.783, longitude=-118.317)
    KTPH = Location(latitude=38.05, longitude=-117.083)
    KTRK = Location(latitude=39.317, longitude=-120.117)
    KTRM = Location(latitude=33.633, longitude=-116.167)
    KTSP = Location(latitude=35.133, longitude=-118.433)
    KTTD = Location(latitude=45.55, longitude=-122.417)
    KTVL = Location(latitude=38.9, longitude=-120)
    KUAO = Location(latitude=45.25, longitude=-122.767)
    KUIL = Location(latitude=47.933, longitude=-124.55)
    KUKI = Location(latitude=39.117, longitude=-123.2)
    KVBG = Location(latitude=34.733, longitude=-120.583)
    KVCB = Location(latitude=38.383, longitude=-121.95)
    KVCV = Location(latitude=34.583, longitude=-117.383)
    KVGT = Location(latitude=36.217, longitude=-115.2)
    KVIS = Location(latitude=36.317, longitude=-119.4)
    KVNY = Location(latitude=34.217, longitude=-118.483)
    KVUO = Location(latitude=45.617, longitude=-122.65)
    KWHP = Location(latitude=34.25, longitude=-118.417)
    KWJF = Location(latitude=34.717, longitude=-118.217)
    KWMC = Location(latitude=40.9, longitude=-117.8)
    KWVI = Location(latitude=36.933, longitude=-121.783)
    KYKM = Location(latitude=46.567, longitude=-120.533)
    MMTJ = Location(latitude=32.533, longitude=-116.967)
    TNCM = Location(latitude=18.05, longitude=-63.117)
    SBAR = Location(latitude=-10.967, longitude=-37.067)
    SBMO = Location(latitude=-9.517, longitude=-35.767)
    MNBL = Location(latitude=12, longitude=-83.767)
    MNCH = Location(latitude=12.617, longitude=-87.117)
    MNJG = Location(latitude=13.083, longitude=-86.017)
    MNJU = Location(latitude=12.083, longitude=-85.367)
    MNMG = Location(latitude=12.15, longitude=-86.167)
    MNPC = Location(latitude=14.05, longitude=-83.383)
    MNRS = Location(latitude=11.417, longitude=-85.817)
    SBEG = Location(latitude=-3.017, longitude=-60.033)
    SBIC = Location(latitude=-3.117, longitude=-58.467)
    SBMN = Location(latitude=-3.133, longitude=-59.967)
    SBMY = Location(latitude=-5.817, longitude=-61.283)
    SBTF = Location(latitude=-3.367, longitude=-64.717)
    SBTT = Location(latitude=-4.25, longitude=-69.933)
    SBUA = Location(latitude=-0.15, longitude=-66.983)
    TFFF = Location(latitude=14.6, longitude=-61)
    MMMA = Location(latitude=25.767, longitude=-97.517)
    MMNL = Location(latitude=27.417, longitude=-99.567)
    MMPG = Location(latitude=28.65, longitude=-100.517)
    MMRX = Location(latitude=26.017, longitude=-98.217)
    MMCL = Location(latitude=24.767, longitude=-107.467)
    MMEP = Location(latitude=21.417, longitude=-104.85)
    MMLM = Location(latitude=25.667, longitude=-109.067)
    MMLP = Location(latitude=24.067, longitude=-110.367)
    MMLT = Location(latitude=26.017, longitude=-111.333)
    MMMZ = Location(latitude=23.167, longitude=-106.267)
    MMSD = Location(latitude=23.133, longitude=-109.7)
    MMSL = Location(latitude=22.95, longitude=-109.933)
    KIMT = Location(latitude=45.817, longitude=-88.117)
    KIWD = Location(latitude=46.517, longitude=-90.117)
    KMNM = Location(latitude=45.117, longitude=-87.617)
    MMCE = Location(latitude=18.65, longitude=-91.8)
    MMCP = Location(latitude=19.817, longitude=-90.5)
    MMCT = Location(latitude=20.633, longitude=-88.45)
    MMMD = Location(latitude=20.95, longitude=-89.65)
    PAMM = Location(latitude=55.133, longitude=-131.583)
    PANT = Location(latitude=55.033, longitude=-131.567)
    MMAA = Location(latitude=16.75, longitude=-99.75)
    MMAS = Location(latitude=21.7, longitude=-102.317)
    MMBT = Location(latitude=15.767, longitude=-96.267)
    MMCB = Location(latitude=18.833, longitude=-99.267)
    MMGL = Location(latitude=20.517, longitude=-103.317)
    MMIA = Location(latitude=19.267, longitude=-103.567)
    MMIT = Location(latitude=16.45, longitude=-95.083)
    MMLO = Location(latitude=21, longitude=-101.467)
    MMMM = Location(latitude=19.85, longitude=-101.017)
    MMMT = Location(latitude=18.1, longitude=-94.567)
    MMMX = Location(latitude=19.417, longitude=-99.083)
    MMOX = Location(latitude=17, longitude=-96.717)
    MMPA = Location(latitude=20.5, longitude=-97.467)
    MMPB = Location(latitude=19.167, longitude=-98.367)
    MMPN = Location(latitude=19.383, longitude=-102.017)
    MMPR = Location(latitude=20.667, longitude=-105.25)
    MMPS = Location(latitude=15.867, longitude=-97.067)
    MMQT = Location(latitude=20.617, longitude=-100.183)
    MMSP = Location(latitude=22.25, longitude=-100.933)
    MMTB = Location(latitude=16.733, longitude=-93.167)
    MMTG = Location(latitude=16.567, longitude=-93.033)
    MMTO = Location(latitude=19.35, longitude=-99.567)
    MMTP = Location(latitude=14.767, longitude=-92.367)
    MMVA = Location(latitude=18, longitude=-92.817)
    MMVR = Location(latitude=19.15, longitude=-96.183)
    MMZC = Location(latitude=22.9, longitude=-102.683)
    MMZH = Location(latitude=17.6, longitude=-101.467)
    MMZO = Location(latitude=19.133, longitude=-104.567)
    MMZP = Location(latitude=20.75, longitude=-103.467)
    LFVP = Location(latitude=46.767, longitude=-56.183)
    CERM = Location(latitude=47.417, longitude=-68.317)
    CWIY = Location(latitude=47.133, longitude=-67.817)
    CWOC = Location(latitude=48.017, longitude=-65.317)
    CWPE = Location(latitude=45.067, longitude=-66.467)
    CWSS = Location(latitude=45.2, longitude=-67.25)
    CYCX = Location(latitude=45.817, longitude=-66.417)
    CYFC = Location(latitude=45.867, longitude=-66.517)
    CYQM = Location(latitude=46.117, longitude=-64.667)
    CYSJ = Location(latitude=45.317, longitude=-65.867)
    CZBF = Location(latitude=47.617, longitude=-65.75)
    CZCR = Location(latitude=47.967, longitude=-66.317)
    MMAN = Location(latitude=25.867, longitude=-100.233)
    MMCV = Location(latitude=23.717, longitude=-98.967)
    MMDO = Location(latitude=24.117, longitude=-104.517)
    MMIO = Location(latitude=25.55, longitude=-100.933)
    MMMV = Location(latitude=26.95, longitude=-101.467)
    MMMY = Location(latitude=25.767, longitude=-100.083)
    MMTC = Location(latitude=25.567, longitude=-103.4)
    MMTM = Location(latitude=22.267, longitude=-97.867)
    SUAA = Location(latitude=-34.767, longitude=-56.25)
    SUCA = Location(latitude=-34.45, longitude=-57.767)
    SUDU = Location(latitude=-33.35, longitude=-56.5)
    SULS = Location(latitude=-34.867, longitude=-55.1)
    SUMU = Location(latitude=-34.817, longitude=-56)
    SUSO = Location(latitude=-31.433, longitude=-57.983)
    CMFM = Location(latitude=47.317, longitude=-71.15)
    CMGB = Location(latitude=45.367, longitude=-72.767)
    CWAF = Location(latitude=48.067, longitude=-69.533)
    CWBA = Location(latitude=46.717, longitude=-79.1)
    CWBS = Location(latitude=46.183, longitude=-72.917)
    CWBT = Location(latitude=50.267, longitude=-64.233)
    CWBY = Location(latitude=49.817, longitude=-64.283)
    CWBZ = Location(latitude=45.117, longitude=-74.267)
    CWDQ = Location(latitude=47.417, longitude=-72.783)
    CWDT = Location(latitude=49.883, longitude=-71.25)
    CWEE = Location(latitude=49.267, longitude=-73.333)
    CWER = Location(latitude=47, longitude=-70.817)
    CWEW = Location(latitude=45.817, longitude=-73.433)
    CWFQ = Location(latitude=45.033, longitude=-72.817)
    CWHM = Location(latitude=45.717, longitude=-73.367)
    CWHP = Location(latitude=49.067, longitude=-61.7)
    CWHQ = Location(latitude=46.683, longitude=-71.967)
    CWHV = Location(latitude=46.2, longitude=-70.767)
    CWIG = Location(latitude=47.067, longitude=-70.533)
    CWIP = Location(latitude=50.167, longitude=-66.433)
    CWIS = Location(latitude=47.283, longitude=-70.633)
    CWIT = Location(latitude=45.167, longitude=-73.667)
    CWIX = Location(latitude=48.6, longitude=-71.717)
    CWIZ = Location(latitude=45.283, longitude=-73.333)
    CWJB = Location(latitude=46.783, longitude=-71.267)
    CWJO = Location(latitude=48.433, longitude=-71.15)
    CWJT = Location(latitude=46.067, longitude=-74.567)
    CWKD = Location(latitude=50.733, longitude=-71.017)
    CWMJ = Location(latitude=46.267, longitude=-76)
    CWNH = Location(latitude=47.783, longitude=-69.533)
    CWNQ = Location(latitude=46.217, longitude=-72.65)
    CWOD = Location(latitude=48.85, longitude=-72.55)
    CWPD = Location(latitude=47.567, longitude=-71.217)
    CWPK = Location(latitude=47.917, longitude=-74.617)
    CWQH = Location(latitude=45.367, longitude=-71.817)
    CWQO = Location(latitude=48.417, longitude=-68.9)
    CWQR = Location(latitude=50.217, longitude=-64.217)
    CWQV = Location(latitude=48.267, longitude=-70.117)
    CWRZ = Location(latitude=48.417, longitude=-64.317)
    CWSF = Location(latitude=49.25, longitude=-65.317)
    CWSG = Location(latitude=49.117, longitude=-66.65)
    CWST = Location(latitude=47.35, longitude=-70.017)
    CWTA = Location(latitude=45.5, longitude=-73.583)
    CWTG = Location(latitude=49.317, longitude=-67.367)
    CWTT = Location(latitude=45.267, longitude=-72.167)
    CWTY = Location(latitude=46.35, longitude=-72.517)
    CWUX = Location(latitude=48.3, longitude=-70.933)
    CWVQ = Location(latitude=45.417, longitude=-73.917)
    CWXC = Location(latitude=48.883, longitude=-71.033)
    CWZS = Location(latitude=48.467, longitude=-67.433)
    CXBO = Location(latitude=46.833, longitude=-71.2)
    CXHF = Location(latitude=45.833, longitude=-75.65)
    CXSH = Location(latitude=46.583, longitude=-72.733)
    CXZV = Location(latitude=50.217, longitude=-66.25)
    CYAD = Location(latitude=53.567, longitude=-76.2)
    CYAH = Location(latitude=53.75, longitude=-73.667)
    CYAS = Location(latitude=60.017, longitude=-70)
    CYBC = Location(latitude=49.117, longitude=-68.2)
    CYBG = Location(latitude=48.333, longitude=-70.983)
    CYGL = Location(latitude=53.617, longitude=-77.7)
    CYGP = Location(latitude=48.767, longitude=-64.467)
    CYGV = Location(latitude=50.267, longitude=-63.617)
    CYGW = Location(latitude=55.267, longitude=-77.767)
    CYHA = Location(latitude=61.033, longitude=-69.617)
    CYHH = Location(latitude=51.7, longitude=-76.117)
    CYHU = Location(latitude=45.517, longitude=-73.417)
    CYIK = Location(latitude=62.417, longitude=-77.933)
    CYKG = Location(latitude=61.567, longitude=-71.917)
    CYKL = Location(latitude=54.783, longitude=-66.817)
    CYKO = Location(latitude=60.817, longitude=-78.15)
    CYKQ = Location(latitude=51.467, longitude=-78.75)
    CYLA = Location(latitude=59.283, longitude=-69.583)
    CYLU = Location(latitude=58.717, longitude=-65.967)
    CYMT = Location(latitude=49.767, longitude=-74.517)
    CYMU = Location(latitude=56.517, longitude=-76.517)
    CYMX = Location(latitude=45.667, longitude=-74.017)
    CYNA = Location(latitude=50.167, longitude=-61.817)
    CYNC = Location(latitude=53.017, longitude=-78.817)
    CYND = Location(latitude=45.517, longitude=-75.567)
    CYNM = Location(latitude=49.767, longitude=-77.817)
    CYOY = Location(latitude=46.883, longitude=-71.5)
    CYPH = Location(latitude=58.45, longitude=-78.083)
    CYPX = Location(latitude=60.033, longitude=-77.267)
    CYQB = Location(latitude=46.783, longitude=-71.367)
    CYRJ = Location(latitude=48.517, longitude=-72.267)
    CYRQ = Location(latitude=46.35, longitude=-72.683)
    CYSC = Location(latitude=45.433, longitude=-71.683)
    CYTQ = Location(latitude=58.667, longitude=-69.95)
    CYUL = Location(latitude=45.467, longitude=-73.733)
    CYUY = Location(latitude=48.2, longitude=-78.817)
    CYVO = Location(latitude=48.067, longitude=-77.767)
    CYVP = Location(latitude=58.117, longitude=-68.417)
    CYYY = Location(latitude=48.6, longitude=-68.217)
    CYZG = Location(latitude=62.167, longitude=-75.667)
    CYZV = Location(latitude=50.217, longitude=-66.267)
    CZEM = Location(latitude=52.217, longitude=-78.517)
    TRPG = Location(latitude=16.783, longitude=-62.2)
    MYGF = Location(latitude=26.55, longitude=-78.7)
    MYNN = Location(latitude=25.033, longitude=-77.467)
    CWTN = Location(latitude=47.067, longitude=-70.783)
    K0A9 = Location(latitude=36.367, longitude=-82.167)
    K0V4 = Location(latitude=37.15, longitude=-79.017)
    K12N = Location(latitude=41.017, longitude=-74.733)
    K18A = Location(latitude=34.35, longitude=-83.133)
    K19A = Location(latitude=34.183, longitude=-83.567)
    K1A5 = Location(latitude=35.217, longitude=-83.417)
    K1A6 = Location(latitude=36.617, longitude=-83.733)
    K1P1 = Location(latitude=43.783, longitude=-71.75)
    K1V4 = Location(latitude=44.417, longitude=-72.017)
    K24J = Location(latitude=30.3, longitude=-83.033)
    K28J = Location(latitude=29.65, longitude=-81.583)
    K2DP = Location(latitude=35.683, longitude=-75.9)
    K2G4 = Location(latitude=39.583, longitude=-79.333)
    K2J9 = Location(latitude=30.6, longitude=-84.55)
    K2W6 = Location(latitude=38.317, longitude=-76.55)
    K3I2 = Location(latitude=38.917, longitude=-82.1)
    K3J7 = Location(latitude=33.6, longitude=-83.133)
    K40B = Location(latitude=46.617, longitude=-69.517)
    K40J = Location(latitude=30.067, longitude=-83.567)
    K42J = Location(latitude=29.85, longitude=-82.05)
    K49A = Location(latitude=34.633, longitude=-84.533)
    K4A7 = Location(latitude=33.383, longitude=-84.333)
    K4I3 = Location(latitude=40.333, longitude=-82.533)
    K5W8 = Location(latitude=35.7, longitude=-79.5)
    K6A1 = Location(latitude=32.567, longitude=-84.25)
    K6A2 = Location(latitude=33.233, longitude=-84.267)
    K6B0 = Location(latitude=43.983, longitude=-73.1)
    K6I2 = Location(latitude=37.633, longitude=-85.25)
    K6L4 = Location(latitude=37.85, longitude=-81.917)
    K7W4 = Location(latitude=37.967, longitude=-77.75)
    K8W2 = Location(latitude=38.65, longitude=-78.7)
    K9A1 = Location(latitude=33.633, longitude=-83.85)
    K9A5 = Location(latitude=34.683, longitude=-85.283)
    KAAF = Location(latitude=29.733, longitude=-85.033)
    KABE = Location(latitude=40.65, longitude=-75.45)
    KABY = Location(latitude=31.533, longitude=-84.2)
    KACJ = Location(latitude=32.117, longitude=-84.183)
    KACK = Location(latitude=41.25, longitude=-70.067)
    KACY = Location(latitude=39.45, longitude=-74.567)
    KADW = Location(latitude=38.817, longitude=-76.867)
    KAFJ = Location(latitude=40.117, longitude=-80.267)
    KAFN = Location(latitude=42.8, longitude=-72)
    KAFP = Location(latitude=35.017, longitude=-80.083)
    KAGC = Location(latitude=40.35, longitude=-79.917)
    KAGS = Location(latitude=33.367, longitude=-81.967)
    KAHN = Location(latitude=33.95, longitude=-83.333)
    KAIK = Location(latitude=33.65, longitude=-81.683)
    KAJR = Location(latitude=34.5, longitude=-83.55)
    KAKH = Location(latitude=35.2, longitude=-81.15)
    KAKQ = Location(latitude=36.983, longitude=-77)
    KAKR = Location(latitude=41.033, longitude=-81.467)
    KALB = Location(latitude=42.75, longitude=-73.8)
    KAMG = Location(latitude=31.533, longitude=-82.5)
    KAND = Location(latitude=34.5, longitude=-82.717)
    KAOH = Location(latitude=40.7, longitude=-84.017)
    KAOO = Location(latitude=40.3, longitude=-78.317)
    KAPF = Location(latitude=26.15, longitude=-81.767)
    KAPG = Location(latitude=39.467, longitude=-76.167)
    KAQW = Location(latitude=42.7, longitude=-73.167)
    KAQX = Location(latitude=32.983, longitude=-81.267)
    KART = Location(latitude=43.983, longitude=-76.033)
    KARW = Location(latitude=32.417, longitude=-80.633)
    KASH = Location(latitude=42.783, longitude=-71.517)
    KASJ = Location(latitude=36.283, longitude=-77.167)
    KATL = Location(latitude=33.633, longitude=-84.45)
    KAUG = Location(latitude=44.317, longitude=-69.8)
    KAVC = Location(latitude=36.683, longitude=-78.033)
    KAVL = Location(latitude=35.433, longitude=-82.533)
    KAVP = Location(latitude=41.333, longitude=-75.717)
    KAXV = Location(latitude=40.5, longitude=-84.3)
    KAYS = Location(latitude=31.25, longitude=-82.4)
    KAZE = Location(latitude=31.883, longitude=-82.65)
    KBAF = Location(latitude=42.167, longitude=-72.717)
    KBBP = Location(latitude=34.617, longitude=-79.733)
    KBCB = Location(latitude=37.217, longitude=-80.417)
    KBCT = Location(latitude=26.383, longitude=-80.1)
    KBDL = Location(latitude=41.933, longitude=-72.683)
    KBDR = Location(latitude=41.167, longitude=-73.133)
    KBED = Location(latitude=42.467, longitude=-71.3)
    KBFD = Location(latitude=41.8, longitude=-78.633)
    KBGE = Location(latitude=30.967, longitude=-84.633)
    KBGM = Location(latitude=42.217, longitude=-75.983)
    KBGR = Location(latitude=44.8, longitude=-68.817)
    KBHB = Location(latitude=44.45, longitude=-68.367)
    KBHC = Location(latitude=31.717, longitude=-82.4)
    KBID = Location(latitude=41.167, longitude=-71.583)
    KBIJ = Location(latitude=31.4, longitude=-84.9)
    KBJJ = Location(latitude=40.867, longitude=-81.883)
    KBKL = Location(latitude=41.533, longitude=-81.667)
    KBKT = Location(latitude=37.067, longitude=-77.95)
    KBKV = Location(latitude=28.467, longitude=-82.45)
    KBKW = Location(latitude=37.8, longitude=-81.117)
    KBLF = Location(latitude=37.3, longitude=-81.2)
    KBLM = Location(latitude=40.167, longitude=-74.117)
    KBML = Location(latitude=44.583, longitude=-71.183)
    KBNL = Location(latitude=33.25, longitude=-81.383)
    KBOS = Location(latitude=42.367, longitude=-71.017)
    KBOW = Location(latitude=27.933, longitude=-81.767)
    KBQK = Location(latitude=31.25, longitude=-81.467)
    KBTP = Location(latitude=40.767, longitude=-79.95)
    KBTV = Location(latitude=44.467, longitude=-73.15)
    KBUF = Location(latitude=42.933, longitude=-78.733)
    KBUY = Location(latitude=36.05, longitude=-79.467)
    KBVI = Location(latitude=40.767, longitude=-80.4)
    KBVY = Location(latitude=42.583, longitude=-70.917)
    KBWI = Location(latitude=39.167, longitude=-76.683)
    KCAE = Location(latitude=33.933, longitude=-81.117)
    KCAK = Location(latitude=40.917, longitude=-81.45)
    KCAR = Location(latitude=46.867, longitude=-68.017)
    KCBE = Location(latitude=39.617, longitude=-78.767)
    KCCO = Location(latitude=33.317, longitude=-84.767)
    KCDA = Location(latitude=44.567, longitude=-72.017)
    KCDN = Location(latitude=34.283, longitude=-80.567)
    KCDW = Location(latitude=40.883, longitude=-74.283)
    KCEF = Location(latitude=42.2, longitude=-72.517)
    KCEU = Location(latitude=34.667, longitude=-82.883)
    KCGC = Location(latitude=28.867, longitude=-82.567)
    KCGE = Location(latitude=38.533, longitude=-76.033)
    KCGF = Location(latitude=41.567, longitude=-81.467)
    KCGS = Location(latitude=38.983, longitude=-76.917)
    KCHA = Location(latitude=35.033, longitude=-85.2)
    KCHO = Location(latitude=38.133, longitude=-78.45)
    KCHS = Location(latitude=32.9, longitude=-80.033)
    KCJR = Location(latitude=38.533, longitude=-77.867)
    KCKB = Location(latitude=39.3, longitude=-80.217)
    KCKF = Location(latitude=31.983, longitude=-83.767)
    KCKI = Location(latitude=33.717, longitude=-79.85)
    KCLE = Location(latitude=41.417, longitude=-81.85)
    KCLT = Location(latitude=35.217, longitude=-80.95)
    KCLW = Location(latitude=27.983, longitude=-82.75)
    KCMH = Location(latitude=40, longitude=-82.883)
    KCNI = Location(latitude=34.317, longitude=-84.417)
    KCOF = Location(latitude=28.233, longitude=-80.6)
    KCON = Location(latitude=43.2, longitude=-71.5)
    KCPC = Location(latitude=34.283, longitude=-78.717)
    KCPK = Location(latitude=36.667, longitude=-76.317)
    KCQW = Location(latitude=34.717, longitude=-79.95)
    KCQX = Location(latitude=41.683, longitude=-70)
    KCRE = Location(latitude=33.817, longitude=-78.717)
    KCRG = Location(latitude=30.333, longitude=-81.517)
    KCRW = Location(latitude=38.383, longitude=-81.583)
    KCSG = Location(latitude=32.517, longitude=-84.95)
    KCTJ = Location(latitude=33.633, longitude=-85.15)
    KCTY = Location(latitude=29.633, longitude=-83.1)
    KCTZ = Location(latitude=34.983, longitude=-78.367)
    KCUB = Location(latitude=33.967, longitude=-81)
    KCVG = Location(latitude=39.05, longitude=-84.667)
    KCWV = Location(latitude=32.2, longitude=-81.867)
    KCXE = Location(latitude=36.783, longitude=-78.5)
    KCXU = Location(latitude=31.217, longitude=-84.233)
    KCXY = Location(latitude=40.217, longitude=-76.85)
    KCZL = Location(latitude=34.467, longitude=-84.933)
    KDAA = Location(latitude=38.717, longitude=-77.167)
    KDAB = Location(latitude=29.183, longitude=-81.067)
    KDAN = Location(latitude=36.567, longitude=-79.333)
    KDAW = Location(latitude=43.283, longitude=-70.917)
    KDAY = Location(latitude=39.9, longitude=-84.217)
    KDBN = Location(latitude=32.567, longitude=-82.983)
    KDCA = Location(latitude=38.85, longitude=-77.033)
    KDCM = Location(latitude=34.783, longitude=-81.2)
    KDDH = Location(latitude=42.9, longitude=-73.25)
    KDED = Location(latitude=29.067, longitude=-81.283)
    KDFI = Location(latitude=41.333, longitude=-84.433)
    KDKK = Location(latitude=42.5, longitude=-79.283)
    KDLZ = Location(latitude=40.283, longitude=-83.117)
    KDMH = Location(latitude=39.283, longitude=-76.617)
    KDMW = Location(latitude=39.6, longitude=-77)
    KDNL = Location(latitude=33.467, longitude=-82.033)
    KDNN = Location(latitude=34.717, longitude=-84.867)
    KDOV = Location(latitude=39.117, longitude=-75.467)
    KDPL = Location(latitude=35, longitude=-77.967)
    KDQH = Location(latitude=31.483, longitude=-82.867)
    KDSV = Location(latitude=42.567, longitude=-77.717)
    KDUJ = Location(latitude=41.167, longitude=-78.9)
    KDVK = Location(latitude=37.583, longitude=-84.767)
    KDWU = Location(latitude=38.55, longitude=-82.733)
    KDXR = Location(latitude=41.367, longitude=-73.483)
    KDYB = Location(latitude=33.067, longitude=-80.283)
    KDYL = Location(latitude=40.333, longitude=-75.117)
    KDZJ = Location(latitude=34.85, longitude=-84)
    KEBA = Location(latitude=34.1, longitude=-82.817)
    KECG = Location(latitude=36.267, longitude=-76.183)
    KEDE = Location(latitude=36.017, longitude=-76.567)
    KEDJ = Location(latitude=40.367, longitude=-83.817)
    KEEN = Location(latitude=42.883, longitude=-72.267)
    KEFK = Location(latitude=44.883, longitude=-72.233)
    KEHO = Location(latitude=35.25, longitude=-81.6)
    KEKN = Location(latitude=38.883, longitude=-79.85)
    KELM = Location(latitude=42.15, longitude=-76.9)
    KELZ = Location(latitude=42.1, longitude=-77.983)
    KEMV = Location(latitude=36.683, longitude=-77.483)
    KEOE = Location(latitude=34.317, longitude=-81.633)
    KEQY = Location(latitude=35.017, longitude=-80.617)
    KERI = Location(latitude=42.083, longitude=-80.183)
    KESN = Location(latitude=38.8, longitude=-76.067)
    KETC = Location(latitude=35.933, longitude=-77.55)
    KEVB = Location(latitude=29.05, longitude=-80.95)
    KEWB = Location(latitude=41.683, longitude=-70.95)
    KEWN = Location(latitude=35.067, longitude=-77.05)
    KEWR = Location(latitude=40.683, longitude=-74.167)
    KEXX = Location(latitude=35.783, longitude=-80.3)
    KEYF = Location(latitude=34.6, longitude=-78.583)
    KEYW = Location(latitude=24.55, longitude=-81.75)
    KEZF = Location(latitude=38.267, longitude=-77.45)
    KEZM = Location(latitude=32.217, longitude=-83.133)
    KFAF = Location(latitude=37.117, longitude=-76.617)
    KFAY = Location(latitude=34.983, longitude=-78.883)
    KFBG = Location(latitude=35.117, longitude=-78.917)
    KFCI = Location(latitude=37.4, longitude=-77.517)
    KFDK = Location(latitude=39.417, longitude=-77.367)
    KFDW = Location(latitude=34.317, longitude=-81.117)
    KFDY = Location(latitude=41.017, longitude=-83.667)
    KFFA = Location(latitude=36.017, longitude=-75.667)
    KFFC = Location(latitude=33.35, longitude=-84.567)
    KFFO = Location(latitude=39.817, longitude=-84.033)
    KFFT = Location(latitude=38.183, longitude=-84.9)
    KFGX = Location(latitude=38.55, longitude=-83.75)
    KFHB = Location(latitude=30.617, longitude=-81.467)
    KFIG = Location(latitude=41.05, longitude=-78.417)
    KFIN = Location(latitude=29.467, longitude=-81.2)
    KFIT = Location(latitude=42.55, longitude=-71.75)
    KFKL = Location(latitude=41.367, longitude=-79.867)
    KFKN = Location(latitude=36.7, longitude=-76.9)
    KFLL = Location(latitude=26.067, longitude=-80.15)
    KFLO = Location(latitude=34.183, longitude=-79.733)
    KFME = Location(latitude=39.083, longitude=-76.767)
    KFMH = Location(latitude=41.633, longitude=-70.517)
    KFMY = Location(latitude=26.583, longitude=-81.867)
    KFOK = Location(latitude=40.85, longitude=-72.617)
    KFPR = Location(latitude=27.5, longitude=-80.383)
    KFQD = Location(latitude=35.417, longitude=-81.933)
    KFRG = Location(latitude=40.733, longitude=-73.417)
    KFRR = Location(latitude=38.917, longitude=-78.25)
    KFSO = Location(latitude=44.933, longitude=-73.1)
    KFTK = Location(latitude=37.883, longitude=-85.967)
    KFTY = Location(latitude=33.783, longitude=-84.517)
    KFVE = Location(latitude=47.283, longitude=-68.3)
    KFVX = Location(latitude=37.35, longitude=-78.417)
    KFWN = Location(latitude=41.2, longitude=-74.633)
    KFXE = Location(latitude=26.2, longitude=-80.183)
    KFYJ = Location(latitude=37.517, longitude=-76.767)
    KFZG = Location(latitude=31.683, longitude=-83.267)
    KFZY = Location(latitude=43.35, longitude=-76.383)
    KGAI = Location(latitude=39.167, longitude=-77.167)
    KGED = Location(latitude=38.683, longitude=-75.367)
    KGEV = Location(latitude=36.433, longitude=-81.417)
    KGEZ = Location(latitude=39.583, longitude=-85.8)
    KGFL = Location(latitude=43.333, longitude=-73.617)
    KGGE = Location(latitude=33.317, longitude=-79.317)
    KGHG = Location(latitude=42.1, longitude=-70.667)
    KGIF = Location(latitude=28.067, longitude=-81.75)
    KGKJ = Location(latitude=41.633, longitude=-80.217)
    KGKT = Location(latitude=35.85, longitude=-83.533)
    KGMU = Location(latitude=34.85, longitude=-82.35)
    KGNR = Location(latitude=45.467, longitude=-69.55)
    KGNV = Location(latitude=29.7, longitude=-82.283)
    KGON = Location(latitude=41.333, longitude=-72.05)
    KGRD = Location(latitude=34.25, longitude=-82.15)
    KGSB = Location(latitude=35.317, longitude=-77.967)
    KGSO = Location(latitude=36.1, longitude=-79.95)
    KGSP = Location(latitude=34.9, longitude=-82.217)
    KGTB = Location(latitude=44.067, longitude=-75.717)
    KGVE = Location(latitude=38.15, longitude=-78.167)
    KGVL = Location(latitude=34.267, longitude=-83.833)
    KGWW = Location(latitude=35.467, longitude=-77.967)
    KGYH = Location(latitude=34.75, longitude=-82.383)
    KHAO = Location(latitude=39.367, longitude=-84.517)
    KHBI = Location(latitude=35.65, longitude=-79.9)
    KHEF = Location(latitude=38.717, longitude=-77.517)
    KHFD = Location(latitude=41.733, longitude=-72.65)
    KHFF = Location(latitude=35.017, longitude=-79.5)
    KHGR = Location(latitude=39.7, longitude=-77.733)
    KHIE = Location(latitude=44.367, longitude=-71.55)
    KHKY = Location(latitude=35.75, longitude=-81.383)
    KHLG = Location(latitude=40.167, longitude=-80.65)
    KHLX = Location(latitude=36.767, longitude=-80.817)
    KHMZ = Location(latitude=40.083, longitude=-78.517)
    KHNZ = Location(latitude=36.367, longitude=-78.533)
    KHOE = Location(latitude=31.05, longitude=-82.783)
    KHPN = Location(latitude=41.067, longitude=-73.7)
    KHQU = Location(latitude=33.533, longitude=-82.517)
    KHRJ = Location(latitude=35.383, longitude=-78.733)
    KHSE = Location(latitude=35.233, longitude=-75.617)
    KHSP = Location(latitude=37.95, longitude=-79.817)
    KHST = Location(latitude=25.467, longitude=-80.367)
    KHTO = Location(latitude=40.95, longitude=-72.25)
    KHTS = Location(latitude=38.367, longitude=-82.55)
    KHUL = Location(latitude=46.117, longitude=-67.8)
    KHVN = Location(latitude=41.267, longitude=-72.867)
    KHVS = Location(latitude=34.4, longitude=-80.117)
    KHWO = Location(latitude=26, longitude=-80.233)
    KHWV = Location(latitude=40.817, longitude=-72.867)
    KHWY = Location(latitude=38.583, longitude=-77.717)
    KHXD = Location(latitude=32.217, longitude=-80.7)
    KHYA = Location(latitude=41.667, longitude=-70.267)
    KHYW = Location(latitude=33.833, longitude=-79.117)
    KHZL = Location(latitude=40.983, longitude=-76)
    KHZY = Location(latitude=41.783, longitude=-80.7)
    KI16 = Location(latitude=37.6, longitude=-81.567)
    KI35 = Location(latitude=36.867, longitude=-83.367)
    KI39 = Location(latitude=37.633, longitude=-84.333)
    KI67 = Location(latitude=39.25, longitude=-84.783)
    KI69 = Location(latitude=39.083, longitude=-84.217)
    KIAD = Location(latitude=38.933, longitude=-77.45)
    KIAG = Location(latitude=43.117, longitude=-78.933)
    KIDI = Location(latitude=40.617, longitude=-79.083)
    KIGX = Location(latitude=35.933, longitude=-79.067)
    KIIY = Location(latitude=33.783, longitude=-82.817)
    KIJD = Location(latitude=41.75, longitude=-72.183)
    KILG = Location(latitude=39.667, longitude=-75.6)
    KILM = Location(latitude=34.267, longitude=-77.9)
    KILN = Location(latitude=39.433, longitude=-83.8)
    KINF = Location(latitude=28.8, longitude=-82.317)
    KINT = Location(latitude=36.133, longitude=-80.217)
    KIOB = Location(latitude=38.05, longitude=-83.983)
    KIPJ = Location(latitude=35.483, longitude=-81.167)
    KIPT = Location(latitude=41.25, longitude=-76.917)
    KISM = Location(latitude=28.283, longitude=-81.433)
    KISO = Location(latitude=35.317, longitude=-77.617)
    KISP = Location(latitude=40.8, longitude=-73.1)
    KITH = Location(latitude=42.467, longitude=-76.45)
    KIWI = Location(latitude=43.967, longitude=-69.717)
    KIXA = Location(latitude=36.333, longitude=-77.633)
    KIZG = Location(latitude=43.983, longitude=-70.95)
    KJAU = Location(latitude=36.333, longitude=-84.167)
    KJAX = Location(latitude=30.5, longitude=-81.683)
    KJES = Location(latitude=31.55, longitude=-81.883)
    KJFK = Location(latitude=40.633, longitude=-73.767)
    KJFZ = Location(latitude=37.067, longitude=-81.8)
    KJGG = Location(latitude=37.233, longitude=-76.717)
    KJHW = Location(latitude=42.133, longitude=-79.267)
    KJKL = Location(latitude=37.6, longitude=-83.317)
    KJNX = Location(latitude=35.533, longitude=-78.383)
    KJQF = Location(latitude=35.383, longitude=-80.717)
    KJST = Location(latitude=40.317, longitude=-78.833)
    KJYL = Location(latitude=32.65, longitude=-81.6)
    KJYO = Location(latitude=39.067, longitude=-77.567)
    KJZI = Location(latitude=32.7, longitude=-80)
    KJZP = Location(latitude=34.45, longitude=-84.467)
    KLAL = Location(latitude=27.967, longitude=-82.017)
    KLBE = Location(latitude=40.267, longitude=-79.4)
    KLBT = Location(latitude=34.6, longitude=-79.067)
    KLCI = Location(latitude=43.567, longitude=-71.417)
    KLCK = Location(latitude=39.817, longitude=-82.917)
    KLCQ = Location(latitude=30.183, longitude=-82.583)
    KLDJ = Location(latitude=40.617, longitude=-74.25)
    KLEB = Location(latitude=43.633, longitude=-72.3)
    KLEE = Location(latitude=28.817, longitude=-81.817)
    KLEW = Location(latitude=44.033, longitude=-70.267)
    KLEX = Location(latitude=38.033, longitude=-84.6)
    KLFI = Location(latitude=37.067, longitude=-76.367)
    KLGA = Location(latitude=40.783, longitude=-73.883)
    KLGC = Location(latitude=33, longitude=-85.067)
    KLHQ = Location(latitude=39.75, longitude=-82.667)
    KLHW = Location(latitude=31.867, longitude=-81.567)
    KLHZ = Location(latitude=36.017, longitude=-78.333)
    KLKR = Location(latitude=34.717, longitude=-80.85)
    KLKU = Location(latitude=38.017, longitude=-77.967)
    KLNN = Location(latitude=41.667, longitude=-81.367)
    KLNP = Location(latitude=36.967, longitude=-82.517)
    KLNS = Location(latitude=40.117, longitude=-76.3)
    KLOM = Location(latitude=40.133, longitude=-75.267)
    KLOU = Location(latitude=38.217, longitude=-85.667)
    KLOZ = Location(latitude=37.083, longitude=-84.067)
    KLPR = Location(latitude=41.35, longitude=-82.183)
    KLQK = Location(latitude=34.817, longitude=-82.7)
    KLRO = Location(latitude=32.9, longitude=-79.783)
    KLSF = Location(latitude=32.317, longitude=-84.967)
    KLUA = Location(latitude=38.667, longitude=-78.5)
    KLUK = Location(latitude=39.1, longitude=-84.417)
    KLUX = Location(latitude=34.5, longitude=-81.95)
    KLVL = Location(latitude=36.783, longitude=-77.8)
    KLWB = Location(latitude=37.867, longitude=-80.4)
    KLWM = Location(latitude=42.717, longitude=-71.133)
    KLYH = Location(latitude=37.317, longitude=-79.2)
    KLZU = Location(latitude=33.983, longitude=-83.967)
    KMAO = Location(latitude=34.183, longitude=-79.333)
    KMCF = Location(latitude=27.85, longitude=-82.517)
    KMCN = Location(latitude=32.683, longitude=-83.65)
    KMCO = Location(latitude=28.417, longitude=-81.333)
    KMDT = Location(latitude=40.2, longitude=-76.767)
    KMEB = Location(latitude=34.783, longitude=-79.367)
    KMFD = Location(latitude=40.817, longitude=-82.517)
    KMFV = Location(latitude=37.633, longitude=-75.767)
    KMGE = Location(latitude=33.917, longitude=-84.517)
    KMGJ = Location(latitude=41.517, longitude=-74.267)
    KMGR = Location(latitude=31.083, longitude=-83.8)
    KMGW = Location(latitude=39.65, longitude=-79.917)
    KMGY = Location(latitude=39.6, longitude=-84.233)
    KMHT = Location(latitude=42.933, longitude=-71.433)
    KMIA = Location(latitude=25.8, longitude=-80.283)
    KMIV = Location(latitude=39.367, longitude=-75.083)
    KMJX = Location(latitude=39.933, longitude=-74.3)
    KMKJ = Location(latitude=36.883, longitude=-81.333)
    KMKS = Location(latitude=33.183, longitude=-80.033)
    KMKY = Location(latitude=26, longitude=-81.667)
    KMLB = Location(latitude=28.1, longitude=-80.65)
    KMLJ = Location(latitude=33.15, longitude=-83.233)
    KMLT = Location(latitude=45.65, longitude=-68.7)
    KMMI = Location(latitude=35.4, longitude=-84.567)
    KMMK = Location(latitude=41.517, longitude=-72.833)
    KMMT = Location(latitude=33.917, longitude=-80.783)
    KMMU = Location(latitude=40.783, longitude=-74.417)
    KMNI = Location(latitude=33.583, longitude=-80.217)
    KMNN = Location(latitude=40.617, longitude=-83.067)
    KMOR = Location(latitude=36.183, longitude=-83.367)
    KMPO = Location(latitude=41.133, longitude=-75.383)
    KMPV = Location(latitude=44.2, longitude=-72.567)
    KMQE = Location(latitude=42.217, longitude=-71.117)
    KMQI = Location(latitude=35.917, longitude=-75.7)
    KMQS = Location(latitude=39.983, longitude=-75.867)
    KMRB = Location(latitude=39.4, longitude=-77.983)
    KMRH = Location(latitude=34.733, longitude=-76.65)
    KMRN = Location(latitude=35.817, longitude=-81.617)
    KMRT = Location(latitude=40.217, longitude=-83.35)
    KMSS = Location(latitude=44.933, longitude=-74.85)
    KMSV = Location(latitude=41.7, longitude=-74.783)
    KMTH = Location(latitude=24.733, longitude=-81.05)
    KMTN = Location(latitude=39.317, longitude=-76.417)
    KMTP = Location(latitude=41.067, longitude=-71.917)
    KMTV = Location(latitude=36.633, longitude=-80.017)
    KMUI = Location(latitude=40.417, longitude=-76.567)
    KMVL = Location(latitude=44.533, longitude=-72.617)
    KMVY = Location(latitude=41.4, longitude=-70.617)
    KMWK = Location(latitude=36.467, longitude=-80.55)
    KMWN = Location(latitude=44.267, longitude=-71.3)
    KMWO = Location(latitude=39.533, longitude=-84.4)
    KMYR = Location(latitude=33.667, longitude=-78.917)
    KN03 = Location(latitude=42.6, longitude=-76.217)
    KN38 = Location(latitude=41.733, longitude=-77.4)
    KNAK = Location(latitude=38.983, longitude=-76.483)
    KNBC = Location(latitude=32.467, longitude=-80.717)
    KNBT = Location(latitude=35.017, longitude=-76.467)
    KNCA = Location(latitude=34.717, longitude=-77.45)
    KNEL = Location(latitude=40.017, longitude=-74.333)
    KNFE = Location(latitude=36.7, longitude=-76.133)
    KNGU = Location(latitude=36.917, longitude=-76.267)
    KNHK = Location(latitude=38.267, longitude=-76.4)
    KNIP = Location(latitude=30.217, longitude=-81.667)
    KNJM = Location(latitude=34.667, longitude=-77.017)
    KNKT = Location(latitude=34.9, longitude=-76.883)
    KNQX = Location(latitude=24.567, longitude=-81.667)
    KNRB = Location(latitude=30.383, longitude=-81.417)
    KNTU = Location(latitude=36.817, longitude=-76.017)
    KNUI = Location(latitude=38.15, longitude=-76.433)
    KNY0 = Location(latitude=42.983, longitude=-74.333)
    KNYC = Location(latitude=40.783, longitude=-73.967)
    KNYG = Location(latitude=38.5, longitude=-77.283)
    KOAJ = Location(latitude=34.817, longitude=-77.617)
    KOBE = Location(latitude=27.267, longitude=-80.85)
    KOCF = Location(latitude=29.167, longitude=-82.217)
    KOCW = Location(latitude=35.567, longitude=-77.05)
    KOFP = Location(latitude=37.7, longitude=-77.433)
    KOGB = Location(latitude=33.467, longitude=-80.85)
    KOKV = Location(latitude=39.133, longitude=-78.15)
    KOKZ = Location(latitude=32.967, longitude=-82.833)
    KOLE = Location(latitude=42.233, longitude=-78.367)
    KOMH = Location(latitude=38.25, longitude=-78.05)
    KOMN = Location(latitude=29.3, longitude=-81.117)
    KONX = Location(latitude=36.4, longitude=-76.017)
    KOPF = Location(latitude=25.917, longitude=-80.283)
    KOPN = Location(latitude=32.95, longitude=-84.267)
    KOQT = Location(latitude=36.017, longitude=-84.233)
    KOQU = Location(latitude=41.6, longitude=-71.417)
    KORE = Location(latitude=42.567, longitude=-72.283)
    KORF = Location(latitude=36.9, longitude=-76.2)
    KORH = Location(latitude=42.267, longitude=-71.867)
    KORL = Location(latitude=28.55, longitude=-81.333)
    KOSU = Location(latitude=40.083, longitude=-83.083)
    KOWD = Location(latitude=42.183, longitude=-71.167)
    KOXB = Location(latitude=38.317, longitude=-75.117)
    KOXC = Location(latitude=41.467, longitude=-73.133)
    KPBG = Location(latitude=44.65, longitude=-73.467)
    KPBI = Location(latitude=26.683, longitude=-80.1)
    KPBX = Location(latitude=37.567, longitude=-82.567)
    KPCM = Location(latitude=28, longitude=-82.167)
    KPCW = Location(latitude=41.517, longitude=-82.867)
    KPDK = Location(latitude=33.883, longitude=-84.3)
    KPEO = Location(latitude=42.65, longitude=-77.05)
    KPGD = Location(latitude=26.917, longitude=-82)
    KPGV = Location(latitude=35.617, longitude=-77.4)
    KPHD = Location(latitude=40.467, longitude=-81.417)
    KPHF = Location(latitude=37.133, longitude=-76.5)
    KPHL = Location(latitude=39.867, longitude=-75.233)
    KPIE = Location(latitude=27.917, longitude=-82.683)
    KPIT = Location(latitude=40.5, longitude=-80.267)
    KPKB = Location(latitude=39.35, longitude=-81.417)
    KPMP = Location(latitude=26.25, longitude=-80.117)
    KPNE = Location(latitude=40.083, longitude=-75.017)
    KPOB = Location(latitude=35.167, longitude=-79.017)
    KPOU = Location(latitude=41.633, longitude=-73.883)
    KPOV = Location(latitude=41.217, longitude=-81.25)
    KPQI = Location(latitude=46.667, longitude=-68.033)
    KPSF = Location(latitude=42.433, longitude=-73.283)
    KPSK = Location(latitude=37.117, longitude=-80.667)
    KPSM = Location(latitude=43.067, longitude=-70.817)
    KPTB = Location(latitude=37.167, longitude=-77.517)
    KPTW = Location(latitude=40.233, longitude=-75.55)
    KPUJ = Location(latitude=33.917, longitude=-84.933)
    KPVC = Location(latitude=42.067, longitude=-70.217)
    KPVD = Location(latitude=41.717, longitude=-71.433)
    KPVG = Location(latitude=36.783, longitude=-76.45)
    KPWM = Location(latitude=43.65, longitude=-70.3)
    KPXE = Location(latitude=32.517, longitude=-83.767)
    KPYM = Location(latitude=41.917, longitude=-70.733)
    KRBW = Location(latitude=32.917, longitude=-80.633)
    KRCZ = Location(latitude=34.883, longitude=-79.767)
    KRDG = Location(latitude=40.367, longitude=-75.967)
    KRDU = Location(latitude=35.9, longitude=-78.783)
    KRHP = Location(latitude=35.183, longitude=-83.85)
    KRIC = Location(latitude=37.517, longitude=-77.317)
    KRKD = Location(latitude=44.067, longitude=-69.083)
    KRME = Location(latitude=43.217, longitude=-75.4)
    KRMG = Location(latitude=34.35, longitude=-85.167)
    KRMN = Location(latitude=38.4, longitude=-77.45)
    KROA = Location(latitude=37.317, longitude=-79.967)
    KROC = Location(latitude=43.117, longitude=-77.683)
    KRSP = Location(latitude=39.65, longitude=-77.467)
    KRSW = Location(latitude=26.533, longitude=-81.75)
    KRUQ = Location(latitude=35.65, longitude=-80.517)
    KRUT = Location(latitude=43.517, longitude=-72.95)
    KRWI = Location(latitude=35.85, longitude=-77.9)
    KRYT = Location(latitude=39.733, longitude=-77.433)
    KRYY = Location(latitude=34.017, longitude=-84.6)
    KRZR = Location(latitude=35.217, longitude=-84.8)
    KRZT = Location(latitude=39.433, longitude=-83.017)
    KSAV = Location(latitude=32.117, longitude=-81.2)
    KSBO = Location(latitude=32.617, longitude=-82.367)
    KSBY = Location(latitude=38.333, longitude=-75.5)
    KSCH = Location(latitude=42.85, longitude=-73.917)
    KSDF = Location(latitude=38.183, longitude=-85.733)
    KSEG = Location(latitude=40.817, longitude=-76.867)
    KSFB = Location(latitude=28.767, longitude=-81.217)
    KSFM = Location(latitude=43.383, longitude=-70.717)
    KSFQ = Location(latitude=36.683, longitude=-76.6)
    KSFZ = Location(latitude=41.917, longitude=-71.5)
    KSGH = Location(latitude=39.833, longitude=-83.833)
    KSGJ = Location(latitude=29.967, longitude=-81.317)
    KSHD = Location(latitude=38.267, longitude=-78.9)
    KSIF = Location(latitude=36.433, longitude=-79.85)
    KSJS = Location(latitude=37.75, longitude=-82.633)
    KSLK = Location(latitude=44.4, longitude=-74.2)
    KSME = Location(latitude=37.05, longitude=-84.617)
    KSMQ = Location(latitude=40.617, longitude=-74.667)
    KSMS = Location(latitude=34, longitude=-80.367)
    KSNC = Location(latitude=41.383, longitude=-72.5)
    KSOP = Location(latitude=35.217, longitude=-79.4)
    KSPA = Location(latitude=34.917, longitude=-81.933)
    KSPG = Location(latitude=27.767, longitude=-82.633)
    KSRQ = Location(latitude=27.4, longitude=-82.567)
    KSSC = Location(latitude=33.967, longitude=-80.467)
    KSSI = Location(latitude=31.15, longitude=-81.383)
    KSUA = Location(latitude=27.183, longitude=-80.217)
    KSUT = Location(latitude=33.933, longitude=-78.083)
    KSVH = Location(latitude=35.75, longitude=-80.95)
    KSVN = Location(latitude=32.017, longitude=-81.15)
    KSWF = Location(latitude=41.5, longitude=-74.083)
    KSYM = Location(latitude=38.217, longitude=-83.583)
    KSYR = Location(latitude=43.117, longitude=-76.1)
    KTAN = Location(latitude=41.883, longitude=-71.017)
    KTBR = Location(latitude=32.483, longitude=-81.733)
    KTDF = Location(latitude=36.283, longitude=-78.983)
    KTDZ = Location(latitude=41.567, longitude=-83.483)
    KTEB = Location(latitude=40.867, longitude=-74.05)
    KTGI = Location(latitude=37.817, longitude=-75.983)
    KTHV = Location(latitude=39.917, longitude=-76.883)
    KTIX = Location(latitude=28.5, longitude=-80.8)
    KTLH = Location(latitude=30.4, longitude=-84.35)
    KTMA = Location(latitude=31.433, longitude=-83.483)
    KTMB = Location(latitude=25.65, longitude=-80.433)
    KTNB = Location(latitude=36.2, longitude=-81.65)
    KTOC = Location(latitude=34.6, longitude=-83.3)
    KTOL = Location(latitude=41.583, longitude=-83.8)
    KTPA = Location(latitude=27.967, longitude=-82.533)
    KTPF = Location(latitude=27.917, longitude=-82.45)
    KTRI = Location(latitude=36.483, longitude=-82.4)
    KTTA = Location(latitude=35.583, longitude=-79.1)
    KTTN = Location(latitude=40.283, longitude=-74.817)
    KTTS = Location(latitude=28.617, longitude=-80.717)
    KTVI = Location(latitude=30.9, longitude=-83.883)
    KTYS = Location(latitude=35.817, longitude=-83.983)
    KTZR = Location(latitude=39.9, longitude=-83.133)
    KUCP = Location(latitude=41.033, longitude=-80.417)
    KUDG = Location(latitude=34.45, longitude=-79.883)
    KUKF = Location(latitude=36.217, longitude=-81.1)
    KUKT = Location(latitude=40.433, longitude=-75.383)
    KUNI = Location(latitude=39.217, longitude=-82.233)
    KUNV = Location(latitude=40.85, longitude=-77.833)
    KUSE = Location(latitude=41.617, longitude=-84.133)
    KUUU = Location(latitude=41.533, longitude=-71.283)
    KUYF = Location(latitude=39.933, longitude=-83.467)
    KUZA = Location(latitude=34.983, longitude=-81.05)
    KVAD = Location(latitude=30.967, longitude=-83.2)
    KVAY = Location(latitude=39.933, longitude=-74.833)
    KVBW = Location(latitude=38.367, longitude=-78.967)
    KVDF = Location(latitude=28.017, longitude=-82.333)
    KVDI = Location(latitude=32.2, longitude=-82.367)
    KVES = Location(latitude=40.2, longitude=-84.533)
    KVJI = Location(latitude=36.667, longitude=-82.017)
    KVLD = Location(latitude=30.783, longitude=-83.267)
    KVNC = Location(latitude=27.067, longitude=-82.433)
    KVPC = Location(latitude=34.133, longitude=-84.85)
    KVQQ = Location(latitude=30.217, longitude=-81.883)
    KVRB = Location(latitude=27.65, longitude=-80.417)
    KVSF = Location(latitude=43.35, longitude=-72.517)
    KVTA = Location(latitude=40.017, longitude=-82.467)
    KVUJ = Location(latitude=35.417, longitude=-80.15)
    KVVG = Location(latitude=28.95, longitude=-81.967)
    KW13 = Location(latitude=38.083, longitude=-78.95)
    KW22 = Location(latitude=39, longitude=-80.267)
    KW29 = Location(latitude=38.983, longitude=-76.333)
    KW31 = Location(latitude=36.95, longitude=-78.183)
    KW63 = Location(latitude=36.6, longitude=-78.55)
    KW75 = Location(latitude=37.6, longitude=-76.45)
    KW78 = Location(latitude=36.683, longitude=-78.85)
    KW81 = Location(latitude=37.183, longitude=-78.1)
    KW96 = Location(latitude=37.5, longitude=-77.117)
    KW99 = Location(latitude=39, longitude=-79.15)
    KWAL = Location(latitude=37.933, longitude=-75.467)
    KWDR = Location(latitude=33.983, longitude=-83.667)
    KWRB = Location(latitude=32.617, longitude=-83.583)
    KWRI = Location(latitude=40.017, longitude=-74.583)
    KWST = Location(latitude=41.35, longitude=-71.8)
    KWVL = Location(latitude=44.517, longitude=-69.667)
    KWWD = Location(latitude=39.017, longitude=-74.917)
    KXLL = Location(latitude=40.567, longitude=-75.483)
    KXMR = Location(latitude=28.467, longitude=-80.567)
    KXSA = Location(latitude=37.867, longitude=-76.9)
    KYNG = Location(latitude=41.25, longitude=-80.667)
    KZPH = Location(latitude=28.233, longitude=-82.15)
    KZZV = Location(latitude=39.95, longitude=-81.9)
    MUGM = Location(latitude=19.9, longitude=-75.2)
    PACD = Location(latitude=55.2, longitude=-162.717)
    PACM = Location(latitude=61.85, longitude=-165.567)
    PACZ = Location(latitude=61.783, longitude=-166.033)
    PADE = Location(latitude=66.067, longitude=-162.767)
    PADG = Location(latitude=68.033, longitude=-162.9)
    PADM = Location(latitude=61.867, longitude=-162.033)
    PADU = Location(latitude=53.9, longitude=-166.533)
    PAEH = Location(latitude=58.65, longitude=-162.067)
    PAEM = Location(latitude=62.783, longitude=-164.483)
    PAGL = Location(latitude=64.55, longitude=-163.033)
    PAGM = Location(latitude=63.767, longitude=-171.733)
    PAHP = Location(latitude=61.517, longitude=-166.15)
    PAIW = Location(latitude=65.617, longitude=-168.1)
    PAKF = Location(latitude=54.85, longitude=-163.417)
    PAKI = Location(latitude=59.933, longitude=-164.033)
    PALU = Location(latitude=68.883, longitude=-166.1)
    PAMK = Location(latitude=63.483, longitude=-162.117)
    PAMO = Location(latitude=62.1, longitude=-163.683)
    PAMY = Location(latitude=60.367, longitude=-166.267)
    PAOM = Location(latitude=64.5, longitude=-165.433)
    PAOO = Location(latitude=60.533, longitude=-165.117)
    PAOT = Location(latitude=66.867, longitude=-162.583)
    PAPB = Location(latitude=56.583, longitude=-169.667)
    PAPO = Location(latitude=68.35, longitude=-166.8)
    PASA = Location(latitude=63.683, longitude=-170.5)
    PASH = Location(latitude=66.267, longitude=-166.083)
    PASM = Location(latitude=62.067, longitude=-163.3)
    PASN = Location(latitude=57.15, longitude=-170.217)
    PATC = Location(latitude=65.567, longitude=-167.917)
    PATE = Location(latitude=65.233, longitude=-166.333)
    PAUT = Location(latitude=54.15, longitude=-165.6)
    PAVA = Location(latitude=61.533, longitude=-165.6)
    PAVC = Location(latitude=55.117, longitude=-162.267)
    PAVL = Location(latitude=67.733, longitude=-164.55)
    PAWM = Location(latitude=64.683, longitude=-163.417)
    PAWN = Location(latitude=67.567, longitude=-162.983)
    PFEL = Location(latitude=64.617, longitude=-162.267)
    PFKT = Location(latitude=65.333, longitude=-166.467)
    PPIT = Location(latitude=60.9, longitude=-162.433)
    PPIZ = Location(latitude=69.733, longitude=-163)
    SBFN = Location(latitude=-3.85, longitude=-32.417)
    KHZE = Location(latitude=47.283, longitude=-101.583)
    KD57 = Location(latitude=46.817, longitude=-101.867)
    KY19 = Location(latitude=46.767, longitude=-100.9)
    MMCS = Location(latitude=31.617, longitude=-106.417)
    MPBO = Location(latitude=9.35, longitude=-82.25)
    MPDA = Location(latitude=8.4, longitude=-82.433)
    MPMG = Location(latitude=8.983, longitude=-79.567)
    MPPA = Location(latitude=8.917, longitude=-79.6)
    MPTO = Location(latitude=9.05, longitude=-79.367)
    CWFD = Location(latitude=66.567, longitude=-61.583)
    CWOB = Location(latitude=63.317, longitude=-64.15)
    CWRH = Location(latitude=61.567, longitude=-64.65)
    CWUP = Location(latitude=68.467, longitude=-66.817)
    CWVD = Location(latitude=67.517, longitude=-63.767)
    CWYM = Location(latitude=64.95, longitude=-63.567)
    CYLT = Location(latitude=82.517, longitude=-62.267)
    CYVM = Location(latitude=67.533, longitude=-64.017)
    CYXP = Location(latitude=66.15, longitude=-65.717)
    SMJP = Location(latitude=5.45, longitude=-55.167)
    SMZO = Location(latitude=5.817, longitude=-55.183)
    KALK = Location(latitude=31.6, longitude=-110.433)
    KAVQ = Location(latitude=32.4, longitude=-111.217)
    KAZC = Location(latitude=36.967, longitude=-113.017)
    KBXK = Location(latitude=33.417, longitude=-112.683)
    KCGZ = Location(latitude=32.95, longitude=-111.767)
    KCHD = Location(latitude=33.267, longitude=-111.817)
    KCMR = Location(latitude=35.3, longitude=-112.2)
    KDMA = Location(latitude=32.167, longitude=-110.867)
    KDUG = Location(latitude=31.467, longitude=-109.583)
    KDVT = Location(latitude=33.683, longitude=-112.067)
    KFFZ = Location(latitude=33.467, longitude=-111.717)
    KFHU = Location(latitude=31.583, longitude=-110.35)
    KFLG = Location(latitude=35.133, longitude=-111.667)
    KGCN = Location(latitude=35.95, longitude=-112.15)
    KGEU = Location(latitude=33.533, longitude=-112.3)
    KGXF = Location(latitude=32.867, longitude=-112.717)
    KGYR = Location(latitude=33.417, longitude=-112.367)
    KIFP = Location(latitude=35.15, longitude=-114.567)
    KIGM = Location(latitude=35.25, longitude=-113.933)
    KINW = Location(latitude=35.033, longitude=-110.717)
    KIWA = Location(latitude=33.3, longitude=-111.65)
    KJTC = Location(latitude=34.133, longitude=-109.317)
    KLUF = Location(latitude=33.517, longitude=-112.367)
    KNYL = Location(latitude=32.633, longitude=-114.617)
    KOLS = Location(latitude=31.417, longitude=-110.85)
    KP08 = Location(latitude=32.933, longitude=-111.433)
    KPAN = Location(latitude=34.25, longitude=-111.333)
    KPHX = Location(latitude=33.433, longitude=-112.017)
    KPRC = Location(latitude=34.65, longitude=-112.417)
    KRYN = Location(latitude=32.15, longitude=-111.167)
    KSAD = Location(latitude=32.85, longitude=-109.633)
    KSDL = Location(latitude=33.617, longitude=-111.917)
    KSEZ = Location(latitude=34.85, longitude=-111.783)
    KSJN = Location(latitude=34.517, longitude=-109.383)
    KSOW = Location(latitude=34.267, longitude=-110)
    KTUS = Location(latitude=32.133, longitude=-110.95)
    TTCP = Location(latitude=11.15, longitude=-60.85)
    TTPP = Location(latitude=10.583, longitude=-61.35)
    MTPP = Location(latitude=18.567, longitude=-72.283)
    SBGM = Location(latitude=-10.783, longitude=-65.283)
    SBPV = Location(latitude=-8.717, longitude=-63.9)
    SBVH = Location(latitude=-12.717, longitude=-60.117)
    TJBQ = Location(latitude=18.5, longitude=-67.133)
    TJIG = Location(latitude=18.45, longitude=-66.1)
    TJMZ = Location(latitude=18.267, longitude=-67.15)
    TJNR = Location(latitude=18.25, longitude=-65.633)
    TJPS = Location(latitude=18.017, longitude=-66.567)
    TJSJ = Location(latitude=18.433, longitude=-66.017)
    CWEU = Location(latitude=80, longitude=-85.917)
    CWJC = Location(latitude=61.133, longitude=-100.9)
    CXAT = Location(latitude=73, longitude=-85.033)
    CXWB = Location(latitude=65.883, longitude=-89.45)
    CYAB = Location(latitude=73, longitude=-85.05)
    CYBK = Location(latitude=64.283, longitude=-96.067)
    CYCS = Location(latitude=63.35, longitude=-90.717)
    CYEK = Location(latitude=61.1, longitude=-94.067)
    CYRT = Location(latitude=62.817, longitude=-92.117)
    CYUT = Location(latitude=66.517, longitude=-86.25)
    CYXN = Location(latitude=62.217, longitude=-92.583)
    SBPL = Location(latitude=-9.367, longitude=-40.567)
    SBRF = Location(latitude=-8.133, longitude=-34.917)
    CWAQ = Location(latitude=49.05, longitude=-105.483)
    CWBU = Location(latitude=53.333, longitude=-104.017)
    CWDC = Location(latitude=59.567, longitude=-108.467)
    CWDJ = Location(latitude=50.4, longitude=-104.583)
    CWEH = Location(latitude=49.417, longitude=-108.967)
    CWFF = Location(latitude=52.817, longitude=-104.583)
    CWHN = Location(latitude=54.883, longitude=-109.967)
    CWIK = Location(latitude=50.367, longitude=-102.583)
    CWIW = Location(latitude=51.667, longitude=-105.4)
    CWJH = Location(latitude=56.317, longitude=-103.267)
    CWJI = Location(latitude=49.717, longitude=-105.917)
    CWJX = Location(latitude=50.883, longitude=-109.5)
    CWKO = Location(latitude=49.167, longitude=-105.967)
    CWLE = Location(latitude=50.95, longitude=-107.15)
    CWMQ = Location(latitude=49.9, longitude=-109.467)
    CWOY = Location(latitude=51.767, longitude=-104.2)
    CWRJ = Location(latitude=51.567, longitude=-107.917)
    CWSR = Location(latitude=53.367, longitude=-107.533)
    CWVN = Location(latitude=49.067, longitude=-107.567)
    CWVP = Location(latitude=49.633, longitude=-109.517)
    CWVT = Location(latitude=55.817, longitude=-108.417)
    CWWF = Location(latitude=49.7, longitude=-103.8)
    CXBK = Location(latitude=50.2, longitude=-104.7)
    CXOX = Location(latitude=55.15, longitude=-105.267)
    CYBU = Location(latitude=53.317, longitude=-104)
    CYEN = Location(latitude=49.217, longitude=-102.967)
    CYKJ = Location(latitude=57.25, longitude=-105.617)
    CYKY = Location(latitude=51.517, longitude=-109.167)
    CYLJ = Location(latitude=54.117, longitude=-108.517)
    CYMJ = Location(latitude=50.317, longitude=-105.533)
    CYPA = Location(latitude=53.217, longitude=-105.667)
    CYQR = Location(latitude=50.433, longitude=-104.667)
    CYQV = Location(latitude=51.267, longitude=-102.467)
    CYQW = Location(latitude=52.767, longitude=-108.25)
    CYSF = Location(latitude=59.25, longitude=-105.817)
    CYXE = Location(latitude=52.167, longitude=-106.7)
    CYYN = Location(latitude=50.267, longitude=-107.667)
    CZMJ = Location(latitude=50.333, longitude=-105.567)
    CXRB = Location(latitude=74.717, longitude=-94.983)
    CYRB = Location(latitude=74.717, longitude=-94.967)
    SBCZ = Location(latitude=-7.617, longitude=-72.767)
    SBRB = Location(latitude=-10, longitude=-67.783)
    SBTK = Location(latitude=-8.167, longitude=-70.767)
    SBCC = Location(latitude=-9.317, longitude=-54.95)
    SBEK = Location(latitude=-6.233, longitude=-57.783)
    SBHT = Location(latitude=-3.25, longitude=-52.267)
    SBIH = Location(latitude=-4.25, longitude=-56)
    SBMD = Location(latitude=-0.867, longitude=-52.6)
    SBSN = Location(latitude=-2.417, longitude=-54.783)
    SCAP = Location(latitude=-43.617, longitude=-71.783)
    SCAR = Location(latitude=-18.35, longitude=-70.317)
    SCAS = Location(latitude=-45.4, longitude=-72.667)
    SCAT = Location(latitude=-27.267, longitude=-70.767)
    SCBA = Location(latitude=-45.917, longitude=-71.667)
    SCCC = Location(latitude=-46.583, longitude=-71.7)
    SCCF = Location(latitude=-22.5, longitude=-68.9)
    SCCH = Location(latitude=-36.567, longitude=-72.017)
    SCCI = Location(latitude=-53, longitude=-70.833)
    SCCY = Location(latitude=-45.567, longitude=-72.1)
    SCDA = Location(latitude=-20.517, longitude=-70.167)
    SCEL = Location(latitude=-33.367, longitude=-70.767)
    SCFA = Location(latitude=-23.417, longitude=-70.417)
    SCFM = Location(latitude=-53.25, longitude=-70.317)
    SCFT = Location(latitude=-43.167, longitude=-71.85)
    SCGE = Location(latitude=-37.383, longitude=-72.417)
    SCGZ = Location(latitude=-54.933, longitude=-67.633)
    SCHR = Location(latitude=-47.217, longitude=-72.533)
    SCIC = Location(latitude=-34.967, longitude=-71.217)
    SCIE = Location(latitude=-36.767, longitude=-73.067)
    SCJO = Location(latitude=-40.6, longitude=-73.033)
    SCMK = Location(latitude=-43.9, longitude=-73.733)
    SCNT = Location(latitude=-51.667, longitude=-72.533)
    SCON = Location(latitude=-43.117, longitude=-73.617)
    SCRD = Location(latitude=-33.067, longitude=-71.55)
    SCRG = Location(latitude=-34.167, longitude=-70.767)
    SCSE = Location(latitude=-29.917, longitude=-71.2)
    SCSN = Location(latitude=-33.633, longitude=-71.617)
    SCTB = Location(latitude=-33.45, longitude=-70.533)
    SCTC = Location(latitude=-38.75, longitude=-72.617)
    SCTE = Location(latitude=-41.417, longitude=-73.067)
    SCVD = Location(latitude=-39.65, longitude=-73.067)
    SCVM = Location(latitude=-32.95, longitude=-71.483)
    MDAB = Location(latitude=19.2, longitude=-69.433)
    MDBH = Location(latitude=18.25, longitude=-71.117)
    MDCY = Location(latitude=19.267, longitude=-69.733)
    MDJB = Location(latitude=18.567, longitude=-69.983)
    MDLR = Location(latitude=18.45, longitude=-68.917)
    MDPC = Location(latitude=18.567, longitude=-68.367)
    MDPP = Location(latitude=19.75, longitude=-70.567)
    MDSD = Location(latitude=18.417, longitude=-69.667)
    MDST = Location(latitude=19.4, longitude=-70.6)
    SBAF = Location(latitude=-22.867, longitude=-43.367)
    SBAN = Location(latitude=-16.217, longitude=-48.967)
    SBAU = Location(latitude=-21.133, longitude=-50.433)
    SBAX = Location(latitude=-19.567, longitude=-46.967)
    SBBG = Location(latitude=-31.383, longitude=-54.117)
    SBBH = Location(latitude=-19.85, longitude=-43.95)
    SBBI = Location(latitude=-25.4, longitude=-49.233)
    SBBQ = Location(latitude=-21.25, longitude=-43.767)
    SBBR = Location(latitude=-15.867, longitude=-47.917)
    SBBU = Location(latitude=-22.317, longitude=-49.067)
    SBCA = Location(latitude=-25, longitude=-53.5)
    SBCB = Location(latitude=-22.917, longitude=-42.083)
    SBCF = Location(latitude=-19.633, longitude=-43.967)
    SBCH = Location(latitude=-27.133, longitude=-52.667)
    SBCM = Location(latitude=-28.717, longitude=-49.417)
    SBCO = Location(latitude=-29.95, longitude=-51.15)
    SBCP = Location(latitude=-21.7, longitude=-41.3)
    SBCT = Location(latitude=-25.517, longitude=-49.167)
    SBCX = Location(latitude=-29.2, longitude=-51.183)
    SBDN = Location(latitude=-22.183, longitude=-51.433)
    SBES = Location(latitude=-22.817, longitude=-42.1)
    SBFI = Location(latitude=-25.6, longitude=-54.483)
    SBFL = Location(latitude=-27.667, longitude=-48.533)
    SBFS = Location(latitude=-22.017, longitude=-41.067)
    SBGL = Location(latitude=-22.817, longitude=-43.25)
    SBGO = Location(latitude=-16.617, longitude=-49.217)
    SBGR = Location(latitude=-23.417, longitude=-46.467)
    SBGW = Location(latitude=-22.767, longitude=-45.2)
    SBIP = Location(latitude=-19.467, longitude=-42.483)
    SBJF = Location(latitude=-21.767, longitude=-43.383)
    SBJR = Location(latitude=-22.967, longitude=-43.367)
    SBJV = Location(latitude=-26.217, longitude=-48.8)
    SBKP = Location(latitude=-23, longitude=-47.117)
    SBLB = Location(latitude=-22.1, longitude=-39.917)
    SBLO = Location(latitude=-23.317, longitude=-51.117)
    SBLS = Location(latitude=-19.667, longitude=-43.883)
    SBME = Location(latitude=-22.35, longitude=-41.767)
    SBMG = Location(latitude=-23.483, longitude=-52.017)
    SBMK = Location(latitude=-16.7, longitude=-43.817)
    SBML = Location(latitude=-22.183, longitude=-49.917)
    SBMT = Location(latitude=-23.517, longitude=-46.617)
    SBNF = Location(latitude=-26.883, longitude=-48.65)
    SBPA = Location(latitude=-30, longitude=-51.167)
    SBPC = Location(latitude=-21.85, longitude=-46.567)
    SBPF = Location(latitude=-28.25, longitude=-52.333)
    SBPK = Location(latitude=-31.717, longitude=-52.333)
    SBPR = Location(latitude=-19.9, longitude=-43.983)
    SBRJ = Location(latitude=-22.883, longitude=-43.167)
    SBRP = Location(latitude=-21.117, longitude=-47.767)
    SBSC = Location(latitude=-22.917, longitude=-43.717)
    SBSJ = Location(latitude=-23.217, longitude=-45.867)
    SBSM = Location(latitude=-29.717, longitude=-53.7)
    SBSP = Location(latitude=-23.617, longitude=-46.633)
    SBSR = Location(latitude=-20.8, longitude=-49.4)
    SBST = Location(latitude=-23.917, longitude=-46.283)
    SBTA = Location(latitude=-23.033, longitude=-45.5)
    SBUG = Location(latitude=-29.767, longitude=-57.017)
    SBUL = Location(latitude=-18.867, longitude=-48.217)
    SBUR = Location(latitude=-19.767, longitude=-47.967)
    SBVT = Location(latitude=-20.267, longitude=-40.267)
    SBYS = Location(latitude=-21.967, longitude=-47.317)
    BGCO = Location(latitude=70.733, longitude=-22.65)
    PAAP = Location(latitude=56.233, longitude=-134.65)
    PAFE = Location(latitude=56.967, longitude=-133.917)
    PAHY = Location(latitude=55.2, longitude=-132.833)
    PAKT = Location(latitude=55.35, longitude=-131.7)
    PAKW = Location(latitude=55.583, longitude=-133.067)
    PAPG = Location(latitude=56.817, longitude=-132.967)
    PASI = Location(latitude=57.067, longitude=-135.35)
    PAWG = Location(latitude=56.483, longitude=-132.367)
    CWAR = Location(latitude=47.283, longitude=-54)
    CWDA = Location(latitude=50.717, longitude=-56.117)
    CWDO = Location(latitude=49.667, longitude=-54.783)
    CWRA = Location(latitude=46.633, longitude=-53.067)
    CWWU = Location(latitude=47.517, longitude=-52.983)
    CWXI = Location(latitude=51.017, longitude=-57.1)
    CWZN = Location(latitude=47.367, longitude=-55.8)
    CXRH = Location(latitude=49.567, longitude=-57.883)
    CXTP = Location(latitude=48.55, longitude=-53.967)
    CYAY = Location(latitude=51.383, longitude=-56.067)
    CYDF = Location(latitude=49.217, longitude=-57.4)
    CYJT = Location(latitude=48.55, longitude=-58.567)
    CYMH = Location(latitude=52.283, longitude=-55.85)
    CYQX = Location(latitude=48.95, longitude=-54.583)
    CYYT = Location(latitude=47.617, longitude=-52.717)
    TKPK = Location(latitude=17.3, longitude=-62.717)
    TLPC = Location(latitude=14.017, longitude=-61)
    TLPL = Location(latitude=13.75, longitude=-60.95)
    TIST = Location(latitude=18.333, longitude=-64.983)
    TISX = Location(latitude=17.7, longitude=-64.8)
    TVSV = Location(latitude=13.133, longitude=-61.2)
    MGZA = Location(latitude=14.967, longitude=-89.517)
    MHAM = Location(latitude=13.267, longitude=-87.65)
    MHCA = Location(latitude=14.9, longitude=-85.783)
    MHCH = Location(latitude=13.317, longitude=-87.15)
    MHLC = Location(latitude=15.75, longitude=-86.85)
    MHLE = Location(latitude=14.317, longitude=-88.15)
    MHLM = Location(latitude=15.45, longitude=-87.933)
    MHNO = Location(latitude=14.433, longitude=-89.183)
    MHPL = Location(latitude=15.267, longitude=-83.783)
    MHRO = Location(latitude=16.317, longitude=-86.533)
    MHSC = Location(latitude=14.367, longitude=-87.617)
    MHSR = Location(latitude=14.783, longitude=-88.783)
    MHTE = Location(latitude=15.783, longitude=-87.483)
    MHTG = Location(latitude=14.067, longitude=-87.217)
    MHTR = Location(latitude=15.933, longitude=-85.933)
    MHYR = Location(latitude=15.133, longitude=-87.117)
    BGQQ = Location(latitude=77.483, longitude=-69.383)
    BGTL = Location(latitude=76.533, longitude=-68.7)
    CYQT = Location(latitude=48.367, longitude=-89.317)
    MMES = Location(latitude=31.8, longitude=-116.6)
    MMML = Location(latitude=32.617, longitude=-115.233)
    CTNK = Location(latitude=45.533, longitude=-78.267)
    CWAJ = Location(latitude=42.25, longitude=-81.9)
    CWBE = Location(latitude=45.983, longitude=-81.483)
    CWCI = Location(latitude=47.317, longitude=-85.817)
    CWCJ = Location(latitude=48.6, longitude=-86.283)
    CWCO = Location(latitude=44.45, longitude=-80.167)
    CWDV = Location(latitude=49.017, longitude=-90.467)
    CWGD = Location(latitude=43.767, longitude=-81.717)
    CWGL = Location(latitude=44.517, longitude=-79.217)
    CWKK = Location(latitude=49.717, longitude=-88.333)
    CWLS = Location(latitude=43.967, longitude=-80.75)
    CWMZ = Location(latitude=45.017, longitude=-80.367)
    CWNB = Location(latitude=41.817, longitude=-82.467)
    CWNC = Location(latitude=43.95, longitude=-78.167)
    CWNL = Location(latitude=45.617, longitude=-82.967)
    CWNZ = Location(latitude=49.75, longitude=-84.167)
    CWPC = Location(latitude=42.867, longitude=-79.25)
    CWPS = Location(latitude=42.567, longitude=-80.033)
    CWQP = Location(latitude=43.833, longitude=-77.15)
    CWRK = Location(latitude=45.067, longitude=-77.867)
    CWWB = Location(latitude=43.3, longitude=-79.783)
    CWWX = Location(latitude=45.317, longitude=-81.733)
    CWWZ = Location(latitude=43.25, longitude=-79.217)
    CXBI = Location(latitude=44.483, longitude=-79.55)
    CXCA = Location(latitude=49.15, longitude=-88.35)
    CXDI = Location(latitude=42.867, longitude=-80.55)
    CXET = Location(latitude=44.233, longitude=-79.783)
    CXHA = Location(latitude=42.033, longitude=-82.9)
    CXHM = Location(latitude=43.283, longitude=-79.9)
    CXKA = Location(latitude=49.4, longitude=-82.45)
    CXKE = Location(latitude=45, longitude=-75.633)
    CXKI = Location(latitude=48.217, longitude=-80)
    CXPC = Location(latitude=45.35, longitude=-80.05)
    CXTO = Location(latitude=43.667, longitude=-79.4)
    CXVN = Location(latitude=43.183, longitude=-79.4)
    CXZC = Location(latitude=51.3, longitude=-80.617)
    CYAM = Location(latitude=46.467, longitude=-84.517)
    CYBN = Location(latitude=44.267, longitude=-79.917)
    CYGK = Location(latitude=44.217, longitude=-76.583)
    CYGQ = Location(latitude=49.767, longitude=-86.917)
    CYHM = Location(latitude=43.167, longitude=-79.933)
    CYKF = Location(latitude=43.467, longitude=-80.383)
    CYKZ = Location(latitude=43.867, longitude=-79.367)
    CYLD = Location(latitude=47.817, longitude=-83.333)
    CYMO = Location(latitude=51.283, longitude=-80.617)
    CYOO = Location(latitude=43.917, longitude=-78.9)
    CYOW = Location(latitude=45.317, longitude=-75.667)
    CYPL = Location(latitude=51.467, longitude=-90.2)
    CYPQ = Location(latitude=44.217, longitude=-78.367)
    CYQA = Location(latitude=44.967, longitude=-79.283)
    CYQG = Location(latitude=42.267, longitude=-82.967)
    CYSB = Location(latitude=46.633, longitude=-80.8)
    CYSN = Location(latitude=43.2, longitude=-79.167)
    CYSP = Location(latitude=48.75, longitude=-86.35)
    CYTR = Location(latitude=44.117, longitude=-77.517)
    CYTS = Location(latitude=48.567, longitude=-81.367)
    CYTZ = Location(latitude=43.633, longitude=-79.4)
    CYVV = Location(latitude=44.75, longitude=-81.083)
    CYWA = Location(latitude=45.95, longitude=-77.317)
    CYXR = Location(latitude=47.7, longitude=-79.833)
    CYXU = Location(latitude=43.017, longitude=-81.15)
    CYXZ = Location(latitude=47.967, longitude=-84.767)
    CYYB = Location(latitude=46.35, longitude=-79.417)
    CYYU = Location(latitude=49.417, longitude=-82.467)
    CYYW = Location(latitude=50.3, longitude=-88.9)
    CYYZ = Location(latitude=43.667, longitude=-79.617)
    CYZE = Location(latitude=45.867, longitude=-82.567)
    CYZR = Location(latitude=43, longitude=-82.3)
    CZEL = Location(latitude=43.65, longitude=-80.417)
    TUPJ = Location(latitude=18.433, longitude=-64.517)
    CBBC = Location(latitude=52.183, longitude=-128.15)
    CWAE = Location(latitude=50.133, longitude=-122.95)
    CWCL = Location(latitude=51.133, longitude=-121.5)
    CWEB = Location(latitude=49.383, longitude=-126.55)
    CWEK = Location(latitude=54.583, longitude=-130.7)
    CWEL = Location(latitude=49.217, longitude=-123.783)
    CWEZ = Location(latitude=48.783, longitude=-123.05)
    CWGB = Location(latitude=49.35, longitude=-124.167)
    CWGT = Location(latitude=49.483, longitude=-124.433)
    CWHL = Location(latitude=54.167, longitude=-130.35)
    CWJU = Location(latitude=54.25, longitude=-133.067)
    CWJV = Location(latitude=50.217, longitude=-119.183)
    CWKH = Location(latitude=48.567, longitude=-123.55)
    CWKX = Location(latitude=58.433, longitude=-130.017)
    CWLC = Location(latitude=54.3, longitude=-130.617)
    CWLM = Location(latitude=48.417, longitude=-123.317)
    CWLP = Location(latitude=50.933, longitude=-127.633)
    CWLY = Location(latitude=50.217, longitude=-121.583)
    CWME = Location(latitude=52.183, longitude=-127.467)
    CWMM = Location(latitude=49.217, longitude=-122.683)
    CWNM = Location(latitude=49.5, longitude=-117.283)
    CWNP = Location(latitude=50.267, longitude=-117.817)
    CWPF = Location(latitude=48.433, longitude=-123.433)
    CWPR = Location(latitude=49.467, longitude=-120.517)
    CWPZ = Location(latitude=54.383, longitude=-125.967)
    CWQK = Location(latitude=48.3, longitude=-123.533)
    CWQS = Location(latitude=53.317, longitude=-132.767)
    CWRO = Location(latitude=54.167, longitude=-131.667)
    CWRU = Location(latitude=50.117, longitude=-127.933)
    CWSK = Location(latitude=49.783, longitude=-123.167)
    CWSL = Location(latitude=50.7, longitude=-119.267)
    CWSP = Location(latitude=48.383, longitude=-123.917)
    CWUS = Location(latitude=49.567, longitude=-119.65)
    CWVF = Location(latitude=49.1, longitude=-123.3)
    CWWA = Location(latitude=49.35, longitude=-123.167)
    CWWK = Location(latitude=49.017, longitude=-122.783)
    CWWL = Location(latitude=53.5, longitude=-130.633)
    CWYJ = Location(latitude=48.45, longitude=-123.3)
    CWYY = Location(latitude=49.033, longitude=-119.433)
    CWZA = Location(latitude=49.25, longitude=-121.767)
    CWZL = Location(latitude=53.033, longitude=-131.6)
    CWZO = Location(latitude=48.55, longitude=-123.233)
    CWZV = Location(latitude=51.933, longitude=-131.017)
    CYAZ = Location(latitude=49.083, longitude=-125.767)
    CYBD = Location(latitude=52.367, longitude=-126.567)
    CYBL = Location(latitude=49.95, longitude=-125.267)
    CYCD = Location(latitude=49.05, longitude=-123.867)
    CYCG = Location(latitude=49.283, longitude=-117.617)
    CYCP = Location(latitude=52.133, longitude=-119.283)
    CYDL = Location(latitude=58.417, longitude=-130.017)
    CYHE = Location(latitude=49.367, longitude=-121.5)
    CYKA = Location(latitude=50.7, longitude=-120.45)
    CYLW = Location(latitude=49.95, longitude=-119.367)
    CYPR = Location(latitude=54.283, longitude=-130.417)
    CYPW = Location(latitude=49.817, longitude=-124.5)
    CYQQ = Location(latitude=49.717, longitude=-124.9)
    CYQZ = Location(latitude=53.017, longitude=-122.517)
    CYRV = Location(latitude=50.967, longitude=-118.167)
    CYVR = Location(latitude=49.167, longitude=-123.167)
    CYWH = Location(latitude=48.417, longitude=-123.383)
    CYWL = Location(latitude=52.183, longitude=-122.067)
    CYXS = Location(latitude=53.9, longitude=-122.683)
    CYXT = Location(latitude=54.467, longitude=-128.567)
    CYXX = Location(latitude=49.033, longitude=-122.383)
    CYYD = Location(latitude=54.817, longitude=-127.183)
    CYYE = Location(latitude=58.817, longitude=-122.567)
    CYYF = Location(latitude=49.467, longitude=-119.583)
    CYYJ = Location(latitude=48.633, longitude=-123.417)
    CYZP = Location(latitude=53.25, longitude=-131.817)
    CYZT = Location(latitude=50.667, longitude=-127.367)
    CYZY = Location(latitude=55.283, longitude=-123.117)
    CZMT = Location(latitude=54.033, longitude=-132.133)
    CZST = Location(latitude=55.933, longitude=-129.983)
    CWHT = Location(latitude=60.767, longitude=-137.583)
    CWKM = Location(latitude=69.6, longitude=-140.167)
    CWZW = Location(latitude=60.167, longitude=-132.733)
    CXCK = Location(latitude=62.117, longitude=-136.2)
    CYDA = Location(latitude=64.033, longitude=-139.117)
    CYDB = Location(latitude=61.367, longitude=-139.05)
    CYMA = Location(latitude=63.617, longitude=-135.867)
    CYOC = Location(latitude=67.567, longitude=-139.817)
    CYQH = Location(latitude=60.117, longitude=-128.817)
    CYUA = Location(latitude=68.95, longitude=-137.283)
    CYXQ = Location(latitude=62.417, longitude=-140.867)
    CYXY = Location(latitude=60.717, longitude=-135.067)
    CYZW = Location(latitude=60.167, longitude=-132.75)
    CZFA = Location(latitude=62.2, longitude=-133.367)
    CTRA = Location(latitude=49.667, longitude=-93.733)
    CWCF = Location(latitude=52.35, longitude=-97.033)
    CWEQ = Location(latitude=52.117, longitude=-101.217)
    CWGX = Location(latitude=56.35, longitude=-94.7)
    CWII = Location(latitude=50.7, longitude=-96.567)
    CWJD = Location(latitude=53.167, longitude=-99.267)
    CWNK = Location(latitude=49.5, longitude=-98.033)
    CWPO = Location(latitude=49.2, longitude=-98.9)
    CWWS = Location(latitude=52.817, longitude=-97.617)
    CXDW = Location(latitude=49.4, longitude=-98.317)
    CXEA = Location(latitude=50.633, longitude=-93.217)
    CXGH = Location(latitude=50.633, longitude=-97)
    CXMD = Location(latitude=49.217, longitude=-98.083)
    CXWN = Location(latitude=49.883, longitude=-97.133)
    CYBQ = Location(latitude=58.7, longitude=-98.5)
    CYBR = Location(latitude=49.917, longitude=-99.95)
    CYDN = Location(latitude=51.1, longitude=-100.033)
    CYFO = Location(latitude=54.667, longitude=-101.667)
    CYHD = Location(latitude=49.817, longitude=-92.75)
    CYIV = Location(latitude=53.85, longitude=-94.65)
    CYNE = Location(latitude=53.967, longitude=-97.817)
    CYPG = Location(latitude=49.883, longitude=-98.267)
    CYQD = Location(latitude=53.967, longitude=-101.083)
    CYQK = Location(latitude=49.767, longitude=-94.367)
    CYRL = Location(latitude=51.067, longitude=-93.783)
    CYTH = Location(latitude=55.783, longitude=-97.867)
    CYTL = Location(latitude=53.817, longitude=-89.883)
    CYWG = Location(latitude=49.883, longitude=-97.217)
    CYXL = Location(latitude=50.117, longitude=-91.9)
    CYYL = Location(latitude=56.867, longitude=-101.067)
    CYYQ = Location(latitude=58.75, longitude=-94.067)
    CZKD = Location(latitude=56.033, longitude=-96.517)
    CZMD = Location(latitude=53.433, longitude=-91.767)
    CZSJ = Location(latitude=53.067, longitude=-93.35)
    PAYA = Location(latitude=59.517, longitude=-139.667)
    CWID = Location(latitude=61.317, longitude=-117.6)
    CWKP = Location(latitude=69.667, longitude=-121.667)
    CWLI = Location(latitude=69.583, longitude=-130.9)
    CWMT = Location(latitude=63.133, longitude=-117.25)
    CWND = Location(latitude=69.633, longitude=-135.433)
    CWVH = Location(latitude=68.9, longitude=-133.933)
    CXDE = Location(latitude=65.217, longitude=-123.433)
    CXDK = Location(latitude=61.267, longitude=-124.467)
    CXLC = Location(latitude=63.6, longitude=-113.85)
    CXLL = Location(latitude=61.117, longitude=-122.85)
    CXTV = Location(latitude=68.75, longitude=-133.5)
    CXYH = Location(latitude=61.25, longitude=-123.75)
    CYFR = Location(latitude=61.167, longitude=-113.667)
    CYFS = Location(latitude=61.75, longitude=-121.217)
    CYGH = Location(latitude=66.267, longitude=-128.617)
    CYHI = Location(latitude=70.767, longitude=-117.783)
    CYHY = Location(latitude=60.817, longitude=-115.767)
    CYJF = Location(latitude=60.217, longitude=-123.467)
    CYKD = Location(latitude=68.217, longitude=-135)
    CYLK = Location(latitude=62.417, longitude=-110.667)
    CYPC = Location(latitude=69.367, longitude=-124.067)
    CYRA = Location(latitude=64.117, longitude=-117.317)
    CYSM = Location(latitude=60.033, longitude=-111.967)
    CYSY = Location(latitude=72, longitude=-125.267)
    CYUB = Location(latitude=69.45, longitude=-133.017)
    CYVQ = Location(latitude=65.267, longitude=-126.783)
    CYWY = Location(latitude=63.217, longitude=-123.417)
    CYZF = Location(latitude=62.467, longitude=-114.45)
    CZCP = Location(latitude=70.167, longitude=-124.717)
    CZFM = Location(latitude=67.4, longitude=-134.867)
    CZFN = Location(latitude=64.917, longitude=-125.567)
    CZFS = Location(latitude=61.767, longitude=-121.233)
    CZHY = Location(latitude=60.833, longitude=-115.783)
    CZSM = Location(latitude=60.033, longitude=-111.933)
    ENSB = Location(latitude=78.25, longitude=15.467)
    OYAA = Location(latitude=12.833, longitude=45.017)
    OYAR = Location(latitude=14.583, longitude=49.25)
    OYAT = Location(latitude=14.55, longitude=46.833)
    OYHD = Location(latitude=14.75, longitude=42.967)
    OYIB = Location(latitude=13.967, longitude=44.167)
    OYMB = Location(latitude=15.467, longitude=45.317)
    OYMK = Location(latitude=13.317, longitude=43.267)
    OYRN = Location(latitude=14.667, longitude=49.383)
    OYSH = Location(latitude=16.967, longitude=43.717)
    OYSN = Location(latitude=15.483, longitude=44.217)
    OYSQ = Location(latitude=12.617, longitude=53.883)
    OYSY = Location(latitude=15.967, longitude=48.783)
    OYTZ = Location(latitude=13.667, longitude=44.117)
    UAAA = Location(latitude=43.35, longitude=77.033)
    UACC = Location(latitude=51.017, longitude=71.467)
    UACP = Location(latitude=54.783, longitude=69.183)
    UADD = Location(latitude=42.85, longitude=71.3)
    UAII = Location(latitude=42.367, longitude=69.483)
    UAKD = Location(latitude=47.7, longitude=67.733)
    UAKK = Location(latitude=49.667, longitude=73.333)
    UASK = Location(latitude=50.033, longitude=82.5)
    UASP = Location(latitude=52.2, longitude=77.067)
    OJAI = Location(latitude=31.717, longitude=36)
    OJAM = Location(latitude=31.967, longitude=35.967)
    UHMA = Location(latitude=64.733, longitude=177.733)
    UATE = Location(latitude=43.867, longitude=51.1)
    UATG = Location(latitude=47.117, longitude=51.833)
    UATT = Location(latitude=50.25, longitude=57.2)
    UTAA = Location(latitude=37.983, longitude=58.367)
    UTAK = Location(latitude=40.05, longitude=53)
    UTAM = Location(latitude=37.617, longitude=61.9)
    UTAT = Location(latitude=41.767, longitude=59.833)
    UTAV = Location(latitude=39.083, longitude=63.617)
    KQAJ = Location(latitude=33.78, longitude=42.43)
    KQBS = Location(latitude=30.55, longitude=47.75)
    KQMH = Location(latitude=33.05, longitude=40.32)
    KQTI = Location(latitude=36.28, longitude=42.4)
    KQTZ = Location(latitude=33.25, longitude=44.23)
    ORBI = Location(latitude=33.267, longitude=44.233)
    ORER = Location(latitude=36.233, longitude=43.967)
    ORMM = Location(latitude=30.567, longitude=47.667)
    ORSU = Location(latitude=35.567, longitude=45.317)
    OBBI = Location(latitude=26.267, longitude=50.633)
    UBBB = Location(latitude=40.45, longitude=50.067)
    UBBG = Location(latitude=40.733, longitude=46.317)
    UBBL = Location(latitude=38.75, longitude=48.817)
    UBBN = Location(latitude=39.183, longitude=45.45)
    UBBY = Location(latitude=41.567, longitude=46.667)
    VLVT = Location(latitude=17.983, longitude=102.567)
    VTBD = Location(latitude=13.917, longitude=100.6)
    VTBO = Location(latitude=12.283, longitude=102.317)
    VTBS = Location(latitude=13.683, longitude=100.75)
    VTBU = Location(latitude=12.683, longitude=101)
    VTCC = Location(latitude=18.767, longitude=98.967)
    VTCL = Location(latitude=18.267, longitude=99.517)
    VTCN = Location(latitude=18.817, longitude=100.767)
    VTCP = Location(latitude=18.133, longitude=100.167)
    VTCT = Location(latitude=19.95, longitude=99.883)
    VTPH = Location(latitude=12.633, longitude=99.95)
    VTPO = Location(latitude=17.233, longitude=99.817)
    VTPP = Location(latitude=16.783, longitude=100.283)
    VTSB = Location(latitude=9.117, longitude=99.133)
    VTSF = Location(latitude=8.533, longitude=99.95)
    VTSG = Location(latitude=8.1, longitude=98.983)
    VTSH = Location(latitude=7.2, longitude=100.583)
    VTSM = Location(latitude=9.55, longitude=100.067)
    VTSP = Location(latitude=8.117, longitude=98.317)
    VTSS = Location(latitude=6.917, longitude=100.383)
    VTST = Location(latitude=7.517, longitude=99.617)
    VTUD = Location(latitude=17.367, longitude=102.783)
    VTUI = Location(latitude=17.2, longitude=104.117)
    VTUK = Location(latitude=16.467, longitude=102.783)
    VTUL = Location(latitude=17.433, longitude=101.717)
    VTUU = Location(latitude=15.25, longitude=104.867)
    VTUV = Location(latitude=16.117, longitude=103.783)
    VTUW = Location(latitude=17.383, longitude=104.633)
    ZJHK = Location(latitude=19.933, longitude=110.467)
    OLBA = Location(latitude=33.817, longitude=35.467)
    KQHT = Location(latitude=43.067, longitude=74.467)
    WBSB = Location(latitude=4.917, longitude=114.917)
    ZBHH = Location(latitude=40.85, longitude=111.817)
    ZGKL = Location(latitude=25.217, longitude=110.033)
    ZGNN = Location(latitude=22.617, longitude=108.167)
    ZJSY = Location(latitude=18.3, longitude=109.417)
    ZLLL = Location(latitude=36.517, longitude=103.617)
    ZLXY = Location(latitude=34.433, longitude=108.75)
    ZPPP = Location(latitude=25, longitude=102.75)
    ZUCK = Location(latitude=29.717, longitude=106.633)
    ZUGY = Location(latitude=26.533, longitude=106.8)
    ZUUU = Location(latitude=30.583, longitude=103.95)
    VCBI = Location(latitude=7.167, longitude=79.867)
    OSDI = Location(latitude=33.417, longitude=36.517)
    OSKL = Location(latitude=37.033, longitude=41.2)
    OSLK = Location(latitude=35.4, longitude=35.95)
    VGEG = Location(latitude=22.267, longitude=91.817)
    VGHS = Location(latitude=23.85, longitude=90.4)
    KQGX = Location(latitude=24.25, longitude=54.55)
    OIBA = Location(latitude=25.883, longitude=55.033)
    OMAA = Location(latitude=24.417, longitude=54.633)
    OMAD = Location(latitude=24.417, longitude=54.467)
    OMAL = Location(latitude=24.25, longitude=55.6)
    OMDB = Location(latitude=25.25, longitude=55.367)
    OMDW = Location(latitude=24.9, longitude=55.167)
    OMFJ = Location(latitude=25.1, longitude=56.317)
    OMRK = Location(latitude=25.617, longitude=55.917)
    OMSJ = Location(latitude=25.317, longitude=55.517)
    UTDD = Location(latitude=38.533, longitude=68.817)
    UTDL = Location(latitude=40.217, longitude=69.7)
    ZYCC = Location(latitude=43.983, longitude=125.683)
    ZYHB = Location(latitude=45.617, longitude=126.25)
    VVDN = Location(latitude=16.017, longitude=108.2)
    VVNB = Location(latitude=21.217, longitude=105.8)
    VVPB = Location(latitude=16.383, longitude=107.7)
    VVTS = Location(latitude=10.817, longitude=106.667)
    VHHH = Location(latitude=22.317, longitude=113.917)
    UIBB = Location(latitude=56.367, longitude=101.7)
    UIII = Location(latitude=52.267, longitude=104.4)
    UIUU = Location(latitude=51.8, longitude=107.45)
    WARR = Location(latitude=-7.367, longitude=112.783)
    WIBB = Location(latitude=0.467, longitude=101.45)
    WIDD = Location(latitude=1.117, longitude=104.117)
    WIHH = Location(latitude=-6.25, longitude=106.883)
    WIII = Location(latitude=-6.133, longitude=106.667)
    WIMM = Location(latitude=3.567, longitude=98.667)
    WIPP = Location(latitude=-2.883, longitude=104.7)
    WIPT = Location(latitude=-0.783, longitude=100.283)
    WAJJ = Location(latitude=-2.583, longitude=140.517)
    LLBG = Location(latitude=32, longitude=34.883)
    LLET = Location(latitude=29.55, longitude=34.95)
    LLHA = Location(latitude=32.783, longitude=35.017)
    LLIB = Location(latitude=32.983, longitude=35.567)
    LLOV = Location(latitude=29.933, longitude=34.933)
    LLSD = Location(latitude=32.117, longitude=34.783)
    OJAQ = Location(latitude=29.617, longitude=35.017)
    KQA4 = Location(latitude=33.4, longitude=62.25)
    KQA7 = Location(latitude=32.6, longitude=65.85)
    KQAR = Location(latitude=32.35, longitude=62.167)
    KQB6 = Location(latitude=33.35, longitude=69.4)
    KQDP = Location(latitude=34.85, longitude=69.633)
    KQHN = Location(latitude=31.62, longitude=65.7)
    KQL5 = Location(latitude=34.4, longitude=70.5)
    KQLT = Location(latitude=32.1, longitude=66.9)
    KQM1 = Location(latitude=34.65, longitude=70.2)
    KQOX = Location(latitude=33.6, longitude=69.217)
    KQPD = Location(latitude=31.85, longitude=65.833)
    KQQS = Location(latitude=34.367, longitude=68.85)
    KQRY = Location(latitude=33, longitude=68)
    KQSA = Location(latitude=34.95, longitude=69.78)
    KQSB = Location(latitude=31, longitude=66.4)
    KQWL = Location(latitude=31.583, longitude=65.417)
    KQXT = Location(latitude=31.1, longitude=64.07)
    KQYU = Location(latitude=34.1, longitude=62.217)
    OAHR = Location(latitude=34.217, longitude=62.217)
    OAKB = Location(latitude=34.55, longitude=69.217)
    OAKN = Location(latitude=31.5, longitude=65.833)
    OAMS = Location(latitude=36.7, longitude=67.2)
    OAUZ = Location(latitude=36.667, longitude=68.917)
    UHPP = Location(latitude=53.167, longitude=158.45)
    OPKC = Location(latitude=24.883, longitude=67.167)
    OPLA = Location(latitude=31.517, longitude=74.4)
    OPMT = Location(latitude=30.183, longitude=71.417)
    OPNH = Location(latitude=26.217, longitude=68.367)
    OPPS = Location(latitude=34.017, longitude=71.517)
    OPRN = Location(latitude=33.617, longitude=73.083)
    OPSK = Location(latitude=27.717, longitude=68.8)
    OPST = Location(latitude=32.533, longitude=74.367)
    ZWSH = Location(latitude=39.55, longitude=76.017)
    VNKT = Location(latitude=27.683, longitude=85.367)
    VAAH = Location(latitude=23.067, longitude=72.617)
    VABB = Location(latitude=19.1, longitude=72.867)
    VANP = Location(latitude=21.083, longitude=79.05)
    VECC = Location(latitude=22.65, longitude=88.433)
    VEPT = Location(latitude=25.6, longitude=85.083)
    VIAR = Location(latitude=31.7, longitude=74.8)
    VIBN = Location(latitude=25.433, longitude=82.867)
    VIDP = Location(latitude=28.567, longitude=77.117)
    VIJP = Location(latitude=26.817, longitude=75.783)
    VILK = Location(latitude=26.75, longitude=80.867)
    VOBL = Location(latitude=13.2, longitude=77.7)
    VOCB = Location(latitude=11.017, longitude=77.033)
    VOCI = Location(latitude=10.15, longitude=76.4)
    VOCL = Location(latitude=11.133, longitude=75.95)
    VOHS = Location(latitude=17.233, longitude=78.433)
    VOHY = Location(latitude=17.45, longitude=78.45)
    VOML = Location(latitude=12.967, longitude=74.883)
    VOMM = Location(latitude=13, longitude=80.167)
    VOTR = Location(latitude=10.767, longitude=78.717)
    VOTV = Location(latitude=8.467, longitude=76.95)
    UNAA = Location(latitude=53.75, longitude=91.4)
    UNKL = Location(latitude=56.167, longitude=92.483)
    UOOO = Location(latitude=69.317, longitude=87.333)
    WMAU = Location(latitude=2.383, longitude=103.883)
    WMBA = Location(latitude=4.217, longitude=100.7)
    WMKA = Location(latitude=6.2, longitude=100.4)
    WMKB = Location(latitude=5.467, longitude=100.367)
    WMKC = Location(latitude=6.167, longitude=102.267)
    WMKD = Location(latitude=3.767, longitude=103.217)
    WMKI = Location(latitude=4.567, longitude=101.083)
    WMKJ = Location(latitude=1.617, longitude=103.667)
    WMKK = Location(latitude=2.75, longitude=101.717)
    WMKL = Location(latitude=6.317, longitude=99.717)
    WMKM = Location(latitude=2.267, longitude=102.25)
    WMKN = Location(latitude=5.367, longitude=103.083)
    WMKP = Location(latitude=5.283, longitude=100.267)
    WMSA = Location(latitude=3.133, longitude=101.55)
    WBGB = Location(latitude=3.117, longitude=113.017)
    WBGG = Location(latitude=1.467, longitude=110.35)
    WBGR = Location(latitude=4.317, longitude=113.967)
    WBGS = Location(latitude=2.267, longitude=111.983)
    WBGY = Location(latitude=1.217, longitude=111.433)
    WBKK = Location(latitude=5.917, longitude=116.033)
    WBKL = Location(latitude=5.283, longitude=115.25)
    WBKS = Location(latitude=5.9, longitude=118.067)
    WBKT = Location(latitude=6.917, longitude=116.817)
    WBKW = Location(latitude=4.317, longitude=118.117)
    KQAY = Location(latitude=28.9, longitude=48.2)
    KQGV = Location(latitude=29.33, longitude=47.52)
    KQWM = Location(latitude=29.67, longitude=47.42)
    OKBK = Location(latitude=29.217, longitude=47.967)
    VMMC = Location(latitude=22.15, longitude=113.583)
    UHMM = Location(latitude=59.917, longitude=150.717)
    WAAA = Location(latitude=-5.067, longitude=119.533)
    WADD = Location(latitude=-8.733, longitude=115.167)
    WALL = Location(latitude=-1.267, longitude=116.9)
    WAMM = Location(latitude=1.55, longitude=124.917)
    WATT = Location(latitude=-10.167, longitude=123.667)
    RPLB = Location(latitude=14.8, longitude=120.267)
    RPLC = Location(latitude=15.167, longitude=120.583)
    RPLI = Location(latitude=18.167, longitude=120.517)
    RPLL = Location(latitude=14.517, longitude=121)
    RPMD = Location(latitude=7.117, longitude=125.65)
    RPMZ = Location(latitude=6.917, longitude=122.067)
    RPVM = Location(latitude=10.317, longitude=123.983)
    OOMS = Location(latitude=23.567, longitude=58.267)
    OOSA = Location(latitude=17.017, longitude=54.067)
    LCEN = Location(latitude=35.133, longitude=33.5)
    LCLK = Location(latitude=34.867, longitude=33.617)
    LCPH = Location(latitude=34.717, longitude=32.467)
    LCRA = Location(latitude=34.567, longitude=32.967)
    UNEE = Location(latitude=55.267, longitude=86.083)
    UNNT = Location(latitude=55.017, longitude=82.633)
    UNBB = Location(latitude=53.367, longitude=83.533)
    UNOO = Location(latitude=54.967, longitude=73.317)
    UARR = Location(latitude=51.15, longitude=51.533)
    VDPP = Location(latitude=11.55, longitude=104.833)
    VDSR = Location(latitude=13.417, longitude=103.833)
    WIOO = Location(latitude=-0.15, longitude=109.4)
    KQIR = Location(latitude=25.2, longitude=51.32)
    OTBD = Location(latitude=25.25, longitude=51.567)
    UAOO = Location(latitude=44.7, longitude=65.6)
    UAUU = Location(latitude=53.2, longitude=63.55)
    VTCH = Location(latitude=19.3, longitude=97.983)
    VYMD = Location(latitude=21.7, longitude=95.983)
    VYYY = Location(latitude=16.9, longitude=96.133)
    OEAB = Location(latitude=18.217, longitude=42.633)
    OEAH = Location(latitude=25.267, longitude=49.467)
    OEAO = Location(latitude=26.483, longitude=38.117)
    OEBA = Location(latitude=20.3, longitude=41.617)
    OEBH = Location(latitude=19.967, longitude=42.617)
    OEDF = Location(latitude=26.467, longitude=49.8)
    OEDM = Location(latitude=24.45, longitude=44.117)
    OEDR = Location(latitude=26.267, longitude=50.167)
    OEGN = Location(latitude=16.883, longitude=42.567)
    OEGS = Location(latitude=26.3, longitude=43.767)
    OEGT = Location(latitude=31.417, longitude=37.267)
    OEHL = Location(latitude=27.417, longitude=41.667)
    OEJN = Location(latitude=21.683, longitude=39.15)
    OEKJ = Location(latitude=24.067, longitude=47.583)
    OEKK = Location(latitude=27.883, longitude=45.517)
    OEKM = Location(latitude=18.3, longitude=42.783)
    OEMA = Location(latitude=24.55, longitude=39.7)
    OEMK = Location(latitude=21.433, longitude=40)
    OENG = Location(latitude=17.617, longitude=44.417)
    OEPA = Location(latitude=28.317, longitude=46.117)
    OERF = Location(latitude=29.617, longitude=43.467)
    OERK = Location(latitude=24.95, longitude=46.717)
    OERR = Location(latitude=30.883, longitude=41.117)
    OERY = Location(latitude=24.717, longitude=46.717)
    OESH = Location(latitude=17.467, longitude=47.117)
    OESK = Location(latitude=29.767, longitude=40.1)
    OETB = Location(latitude=28.367, longitude=36.617)
    OETF = Location(latitude=21.467, longitude=40.533)
    OETR = Location(latitude=31.7, longitude=38.733)
    OEWD = Location(latitude=20.5, longitude=45.2)
    OEWJ = Location(latitude=26.183, longitude=36.467)
    OEYN = Location(latitude=24.133, longitude=38.067)
    UHSS = Location(latitude=46.883, longitude=142.717)
    UTNN = Location(latitude=42.483, longitude=59.633)
    UTNU = Location(latitude=41.583, longitude=60.633)
    UTSA = Location(latitude=40.117, longitude=65.167)
    UTSB = Location(latitude=39.767, longitude=64.483)
    UTSK = Location(latitude=38.833, longitude=65.767)
    UTSS = Location(latitude=39.7, longitude=66.983)
    UTST = Location(latitude=37.283, longitude=67.317)
    KQEI = Location(latitude=37.95, longitude=128.07)
    KQEJ = Location(latitude=38.2, longitude=127.95)
    KQEP = Location(latitude=37.4, longitude=127.27)
    KQEQ = Location(latitude=37.85, longitude=127.55)
    KQES = Location(latitude=36.97, longitude=126.45)
    KQEW = Location(latitude=36.4, longitude=128.15)
    KQEX = Location(latitude=36.1, longitude=128.3)
    KQFB = Location(latitude=35.65, longitude=128.72)
    KQFT = Location(latitude=37.6, longitude=127.35)
    KQFU = Location(latitude=37.8, longitude=126.97)
    KQFV = Location(latitude=38.05, longitude=127.5)
    KQFW = Location(latitude=37.667, longitude=127.883)
    RKJB = Location(latitude=34.983, longitude=126.383)
    RKJK = Location(latitude=35.9, longitude=126.617)
    RKNY = Location(latitude=38.067, longitude=128.667)
    RKPC = Location(latitude=33.5, longitude=126.483)
    RKPK = Location(latitude=35.167, longitude=128.917)
    RKSG = Location(latitude=36.967, longitude=127.017)
    RKSI = Location(latitude=37.467, longitude=126.45)
    RKSM = Location(latitude=37.417, longitude=127.117)
    RKSO = Location(latitude=37.067, longitude=127.017)
    RKSS = Location(latitude=37.533, longitude=126.783)
    RKTN = Location(latitude=35.883, longitude=128.65)
    RKTP = Location(latitude=36.7, longitude=126.483)
    RKTU = Location(latitude=36.7, longitude=127.5)
    ZBAA = Location(latitude=40.067, longitude=116.583)
    ZBSJ = Location(latitude=38.283, longitude=114.7)
    ZBTJ = Location(latitude=39.117, longitude=117.35)
    ZBYN = Location(latitude=37.767, longitude=112.633)
    ZGGG = Location(latitude=23.4, longitude=113.3)
    ZGHA = Location(latitude=28.183, longitude=113.217)
    ZGOW = Location(latitude=23.55, longitude=116.5)
    ZGSZ = Location(latitude=22.633, longitude=113.8)
    ZHCC = Location(latitude=34.517, longitude=113.833)
    ZHHH = Location(latitude=30.783, longitude=114.217)
    ZSAM = Location(latitude=24.533, longitude=118.133)
    ZSFZ = Location(latitude=25.933, longitude=119.667)
    ZSHC = Location(latitude=30.217, longitude=120.433)
    ZSNB = Location(latitude=29.833, longitude=121.467)
    ZSNJ = Location(latitude=31.75, longitude=118.867)
    ZSOF = Location(latitude=31.983, longitude=116.967)
    ZSPD = Location(latitude=31.15, longitude=121.8)
    ZSQD = Location(latitude=36.267, longitude=120.383)
    ZSSS = Location(latitude=31.2, longitude=121.333)
    ZYTL = Location(latitude=38.967, longitude=121.533)
    WSAP = Location(latitude=1.367, longitude=103.917)
    WSSL = Location(latitude=1.417, longitude=103.867)
    WSSS = Location(latitude=1.367, longitude=103.967)
    RCKH = Location(latitude=22.567, longitude=120.333)
    RCMQ = Location(latitude=24.267, longitude=120.617)
    RCSS = Location(latitude=25.067, longitude=121.533)
    RCTP = Location(latitude=25.067, longitude=121.217)
    UTFN = Location(latitude=40.983, longitude=71.567)
    UTTT = Location(latitude=41.267, longitude=69.267)
    UGKO = Location(latitude=42.183, longitude=42.483)
    UGSB = Location(latitude=41.617, longitude=41.6)
    UGTB = Location(latitude=41.667, longitude=44.95)
    OIAA = Location(latitude=30.367, longitude=48.25)
    OIAD = Location(latitude=32.433, longitude=48.4)
    OIAG = Location(latitude=30.767, longitude=49.667)
    OIAH = Location(latitude=30.333, longitude=50.833)
    OIAI = Location(latitude=32, longitude=49.267)
    OIAM = Location(latitude=30.55, longitude=49.15)
    OIAW = Location(latitude=31.317, longitude=48.767)
    OIBB = Location(latitude=28.967, longitude=50.817)
    OIBJ = Location(latitude=27.817, longitude=52.35)
    OIBK = Location(latitude=26.533, longitude=53.983)
    OIBL = Location(latitude=26.533, longitude=54.817)
    OIBP = Location(latitude=27.383, longitude=52.733)
    OIBQ = Location(latitude=29.267, longitude=50.317)
    OIBS = Location(latitude=25.9, longitude=54.55)
    OIBV = Location(latitude=26.817, longitude=53.35)
    OICC = Location(latitude=34.35, longitude=47.167)
    OICI = Location(latitude=33.583, longitude=46.4)
    OICK = Location(latitude=33.433, longitude=48.283)
    OICS = Location(latitude=35.25, longitude=47.017)
    OIFK = Location(latitude=33.9, longitude=51.583)
    OIFM = Location(latitude=32.75, longitude=51.867)
    OIFS = Location(latitude=32.317, longitude=50.85)
    OIGG = Location(latitude=37.333, longitude=49.6)
    OIHH = Location(latitude=34.867, longitude=48.55)
    OIIE = Location(latitude=35.417, longitude=51.15)
    OIII = Location(latitude=35.683, longitude=51.3)
    OIIK = Location(latitude=36.25, longitude=50.05)
    OIIP = Location(latitude=35.783, longitude=50.833)
    OIIS = Location(latitude=35.383, longitude=53.667)
    OIKB = Location(latitude=27.217, longitude=56.367)
    OIKK = Location(latitude=30.25, longitude=56.967)
    OIKQ = Location(latitude=26.75, longitude=55.9)
    OIMB = Location(latitude=32.9, longitude=59.283)
    OIMC = Location(latitude=36.5, longitude=61.067)
    OIMD = Location(latitude=34.383, longitude=58.633)
    OIMM = Location(latitude=36.233, longitude=59.617)
    OIMN = Location(latitude=37.483, longitude=57.3)
    OIMS = Location(latitude=36.167, longitude=57.6)
    OING = Location(latitude=36.9, longitude=54.4)
    OINN = Location(latitude=36.667, longitude=51.467)
    OINR = Location(latitude=36.883, longitude=50.667)
    OINZ = Location(latitude=36.633, longitude=53.2)
    OISA = Location(latitude=31.033, longitude=52.783)
    OISF = Location(latitude=28.9, longitude=53.717)
    OISL = Location(latitude=27.667, longitude=54.367)
    OISS = Location(latitude=29.517, longitude=52.567)
    OISY = Location(latitude=30.7, longitude=51.55)
    OITK = Location(latitude=38.417, longitude=44.983)
    OITL = Location(latitude=38.333, longitude=48.417)
    OITM = Location(latitude=37.35, longitude=46.15)
    OITR = Location(latitude=37.667, longitude=45.067)
    OITT = Location(latitude=38.133, longitude=46.233)
    OITZ = Location(latitude=36.767, longitude=48.367)
    OIYY = Location(latitude=31.9, longitude=54.283)
    OIZB = Location(latitude=31.1, longitude=61.55)
    OIZC = Location(latitude=25.45, longitude=60.383)
    OIZH = Location(latitude=29.467, longitude=60.917)
    OIZI = Location(latitude=27.233, longitude=60.717)
    OIZJ = Location(latitude=25.65, longitude=57.8)
    KQTA = Location(latitude=35.45, longitude=139.45)
    RJAA = Location(latitude=35.767, longitude=140.367)
    RJAF = Location(latitude=36.167, longitude=137.917)
    RJAH = Location(latitude=36.167, longitude=140.417)
    RJAK = Location(latitude=36.017, longitude=140.183)
    RJAW = Location(latitude=24.767, longitude=141.317)
    RJBB = Location(latitude=34.417, longitude=135.25)
    RJBD = Location(latitude=33.667, longitude=135.367)
    RJBE = Location(latitude=34.633, longitude=135.233)
    RJBT = Location(latitude=35.5, longitude=134.8)
    RJCB = Location(latitude=42.717, longitude=143.217)
    RJCC = Location(latitude=42.767, longitude=141.667)
    RJCH = Location(latitude=41.767, longitude=140.817)
    RJCK = Location(latitude=43.017, longitude=144.183)
    RJCM = Location(latitude=43.883, longitude=144.167)
    RJCN = Location(latitude=43.567, longitude=144.967)
    RJCW = Location(latitude=45.383, longitude=141.8)
    RJDC = Location(latitude=33.917, longitude=131.267)
    RJDT = Location(latitude=34.267, longitude=129.317)
    RJEB = Location(latitude=44.3, longitude=143.4)
    RJEC = Location(latitude=43.667, longitude=142.433)
    RJFE = Location(latitude=32.667, longitude=128.817)
    RJFF = Location(latitude=33.567, longitude=130.433)
    RJFG = Location(latitude=30.617, longitude=130.983)
    RJFK = Location(latitude=31.8, longitude=130.717)
    RJFM = Location(latitude=31.867, longitude=131.433)
    RJFO = Location(latitude=33.467, longitude=131.717)
    RJFR = Location(latitude=33.817, longitude=131.033)
    RJFS = Location(latitude=33.15, longitude=130.3)
    RJFT = Location(latitude=32.817, longitude=130.833)
    RJFU = Location(latitude=32.917, longitude=129.917)
    RJFY = Location(latitude=31.367, longitude=130.817)
    RJGG = Location(latitude=34.867, longitude=136.8)
    RJKA = Location(latitude=28.417, longitude=129.717)
    RJNG = Location(latitude=35.367, longitude=136.867)
    RJNK = Location(latitude=36.367, longitude=136.417)
    RJNT = Location(latitude=36.633, longitude=137.167)
    RJNW = Location(latitude=37.3, longitude=136.967)
    RJNY = Location(latitude=34.817, longitude=138.3)
    RJOA = Location(latitude=34.417, longitude=132.917)
    RJOB = Location(latitude=34.75, longitude=133.833)
    RJOC = Location(latitude=35.417, longitude=132.867)
    RJOE = Location(latitude=34.517, longitude=136.667)
    RJOF = Location(latitude=34.017, longitude=131.55)
    RJOH = Location(latitude=35.467, longitude=133.25)
    RJOI = Location(latitude=34.133, longitude=132.217)
    RJOK = Location(latitude=33.517, longitude=133.667)
    RJOM = Location(latitude=33.817, longitude=132.683)
    RJOO = Location(latitude=34.767, longitude=135.433)
    RJOP = Location(latitude=34, longitude=134.617)
    RJOR = Location(latitude=35.517, longitude=134.167)
    RJOS = Location(latitude=34.117, longitude=134.617)
    RJOT = Location(latitude=34.217, longitude=134.017)
    RJOW = Location(latitude=34.667, longitude=131.8)
    RJOY = Location(latitude=34.6, longitude=135.583)
    RJSA = Location(latitude=40.717, longitude=140.683)
    RJSC = Location(latitude=38.417, longitude=140.367)
    RJSF = Location(latitude=37.217, longitude=140.417)
    RJSH = Location(latitude=40.533, longitude=141.467)
    RJSI = Location(latitude=39.417, longitude=141.117)
    RJSK = Location(latitude=39.617, longitude=140.217)
    RJSN = Location(latitude=37.95, longitude=139.117)
    RJSO = Location(latitude=41.217, longitude=141.117)
    RJSR = Location(latitude=40.183, longitude=140.367)
    RJSS = Location(latitude=38.117, longitude=140.917)
    RJSU = Location(latitude=38.217, longitude=140.917)
    RJSY = Location(latitude=38.817, longitude=139.767)
    RJTE = Location(latitude=34.967, longitude=139.817)
    RJTF = Location(latitude=35.667, longitude=139.517)
    RJTI = Location(latitude=35.633, longitude=139.833)
    RJTL = Location(latitude=35.783, longitude=140.017)
    RJTO = Location(latitude=34.767, longitude=139.367)
    RJTT = Location(latitude=35.533, longitude=139.767)
    RJTU = Location(latitude=36.517, longitude=139.867)
    RJTY = Location(latitude=35.75, longitude=139.333)
    ROAH = Location(latitude=26.167, longitude=127.65)
    RODN = Location(latitude=26.35, longitude=127.767)
    ROIG = Location(latitude=24.317, longitude=124.167)
    ROMY = Location(latitude=24.767, longitude=125.267)
    RORS = Location(latitude=24.817, longitude=125.15)
    ROTM = Location(latitude=26.267, longitude=127.75)
    ZMUB = Location(latitude=47.833, longitude=106.783)
    ZWWW = Location(latitude=43.883, longitude=87.467)
    UHHH = Location(latitude=48.517, longitude=135.183)
    UHWW = Location(latitude=43.4, longitude=132.15)
    UEEE = Location(latitude=62.083, longitude=129.767)
    UERP = Location(latitude=66.4, longitude=112.033)
    UERR = Location(latitude=62.533, longitude=114.05)
    UHBB = Location(latitude=50.433, longitude=127.417)
    UIAA = Location(latitude=52.017, longitude=113.317)
    USCC = Location(latitude=55.3, longitude=61.5)
    USCM = Location(latitude=53.4, longitude=58.75)
    USDD = Location(latitude=66.583, longitude=66.617)
    USHH = Location(latitude=61.033, longitude=69.083)
    USNN = Location(latitude=60.95, longitude=76.483)
    USPP = Location(latitude=57.917, longitude=56.017)
    USRR = Location(latitude=61.333, longitude=73.417)
    USSS = Location(latitude=56.75, longitude=60.817)
    USTR = Location(latitude=57.167, longitude=65.317)
    UWOO = Location(latitude=51.8, longitude=55.467)
    UWOR = Location(latitude=51.067, longitude=58.6)
    UWUU = Location(latitude=54.567, longitude=55.867)
    UDSG = Location(latitude=40.75, longitude=43.867)
    UDYZ = Location(latitude=40.15, longitude=44.4)
    LPAZ = Location(latitude=36.967, longitude=-25.167)
    LPFL = Location(latitude=39.45, longitude=-31.117)
    LPHR = Location(latitude=38.517, longitude=-28.717)
    LPLA = Location(latitude=38.767, longitude=-27.1)
    LPPD = Location(latitude=37.717, longitude=-25.683)
    TXKF = Location(latitude=32.367, longitude=-64.683)
    GCFV = Location(latitude=28.433, longitude=-13.867)
    GCGM = Location(latitude=28.033, longitude=-17.217)
    GCHI = Location(latitude=27.817, longitude=-17.867)
    GCLA = Location(latitude=28.617, longitude=-17.75)
    GCLP = Location(latitude=27.917, longitude=-15.367)
    GCRR = Location(latitude=28.933, longitude=-13.583)
    GCTS = Location(latitude=28.05, longitude=-16.567)
    GCXO = Location(latitude=28.483, longitude=-16.35)
    GVAC = Location(latitude=16.717, longitude=-22.933)
    GVBA = Location(latitude=16.133, longitude=-22.883)
    GVNP = Location(latitude=14.95, longitude=-23.483)
    GVSV = Location(latitude=16.833, longitude=-25.067)
    EKVG = Location(latitude=62.067, longitude=-7.3)
    LPMA = Location(latitude=32.7, longitude=-16.767)
    LPPS = Location(latitude=33.067, longitude=-16.35)
    BIAR = Location(latitude=65.667, longitude=-18.067)
    BIEG = Location(latitude=65.267, longitude=-14.4)
    BIHN = Location(latitude=64.3, longitude=-15.233)
    BIIS = Location(latitude=66.05, longitude=-23.133)
    BIKF = Location(latitude=63.983, longitude=-22.6)
    BIRK = Location(latitude=64.133, longitude=-21.933)
    BIVM = Location(latitude=63.417, longitude=-20.267)
    FHAW = Location(latitude=-7.967, longitude=-14.4)
    EGYP = Location(latitude=-51.817, longitude=-58.45)
    YPAD = Location(latitude=-34.95, longitude=138.517)
    YPWR = Location(latitude=-31.117, longitude=136.817)
    YWHA = Location(latitude=-33.067, longitude=137.517)
    YAMB = Location(latitude=-27.633, longitude=152.717)
    YBBN = Location(latitude=-27.383, longitude=153.133)
    YBCS = Location(latitude=-16.867, longitude=145.75)
    YBCV = Location(latitude=-26.383, longitude=146.267)
    YBMA = Location(latitude=-20.667, longitude=139.467)
    YBMK = Location(latitude=-21.167, longitude=149.167)
    YBOK = Location(latitude=-27.417, longitude=151.717)
    YBPN = Location(latitude=-20.5, longitude=148.55)
    YBRK = Location(latitude=-23.367, longitude=150.467)
    YBTL = Location(latitude=-19.25, longitude=146.75)
    YBWP = Location(latitude=-12.683, longitude=141.917)
    YHID = Location(latitude=-10.583, longitude=142.3)
    YAYE = Location(latitude=-25.183, longitude=130.983)
    YBAS = Location(latitude=-23.8, longitude=133.9)
    YPDN = Location(latitude=-12.417, longitude=130.9)
    YPGV = Location(latitude=-12.267, longitude=136.817)
    YPTN = Location(latitude=-14.517, longitude=132.383)
    YTNK = Location(latitude=-19.633, longitude=134.183)
    YMHB = Location(latitude=-42.817, longitude=147.517)
    YMLT = Location(latitude=-41.517, longitude=147.183)
    YBHM = Location(latitude=-20.35, longitude=148.95)
    YMAV = Location(latitude=-38.033, longitude=144.483)
    YMEN = Location(latitude=-37.717, longitude=144.9)
    YMML = Location(latitude=-37.667, longitude=144.817)
    YWGT = Location(latitude=-36.417, longitude=146.3)
    YBRM = Location(latitude=-17.933, longitude=122.217)
    YCIN = Location(latitude=-17.583, longitude=123.833)
    YFRT = Location(latitude=-30.833, longitude=128.117)
    YGEL = Location(latitude=-28.8, longitude=114.7)
    YPEA = Location(latitude=-31.667, longitude=116.017)
    YPJT = Location(latitude=-32.1, longitude=115.883)
    YPKG = Location(latitude=-30.767, longitude=121.45)
    YPKU = Location(latitude=-15.767, longitude=128.683)
    YPLM = Location(latitude=-22.217, longitude=114.067)
    YPPD = Location(latitude=-20.367, longitude=118.617)
    YPPH = Location(latitude=-31.933, longitude=115.983)
    YTEF = Location(latitude=-21.717, longitude=122.233)
    YBCG = Location(latitude=-28.167, longitude=153.5)
    YCFS = Location(latitude=-30.317, longitude=153.117)
    YMAY = Location(latitude=-36.067, longitude=146.933)
    YSCB = Location(latitude=-35.283, longitude=149.167)
    YSDU = Location(latitude=-32.217, longitude=148.567)
    YSNW = Location(latitude=-34.95, longitude=150.517)
    YSRI = Location(latitude=-33.6, longitude=150.767)
    YSSY = Location(latitude=-33.95, longitude=151.167)
    YSTW = Location(latitude=-31.067, longitude=150.85)
    YSWG = Location(latitude=-35.133, longitude=147.467)
    YTRE = Location(latitude=-31.883, longitude=152.517)
    YWLM = Location(latitude=-32.8, longitude=151.833)
    EHAK = Location(latitude=55.417, longitude=3.817)
    EKGF = Location(latitude=55.717, longitude=4.8)
    ENEK = Location(latitude=56.533, longitude=3.25)
    ENHE = Location(latitude=65.333, longitude=2.333)
    NZSP = Location(latitude=-89.983, longitude=179.983)
    PKWA = Location(latitude=8.717, longitude=167.717)
    SFAL = Location(latitude=-51.667, longitude=-57.767)
    OINK = Location(latitude=37.25, longitude=55.1)
    SCRM = Location(latitude=-62.183, longitude=-58.983)
    KATP = Location(latitude=27.2, longitude=-90.033)
    KGBK = Location(latitude=27.2, longitude=-92.2)
    KGUL = Location(latitude=27.3, longitude=-93.533)
    KHHV = Location(latitude=26.933, longitude=-94.683)
    EHAM = Location(latitude=52.317, longitude=4.783)
    EHBK = Location(latitude=50.917, longitude=5.767)
    EHDL = Location(latitude=52.067, longitude=5.867)
    EHDV = Location(latitude=54.317, longitude=2.933)
    EHEH = Location(latitude=51.45, longitude=5.383)
    EHFD = Location(latitude=54.85, longitude=4.7)
    EHFS = Location(latitude=51.45, longitude=3.6)
    EHFZ = Location(latitude=54.117, longitude=4.017)
    EHGG = Location(latitude=53.117, longitude=6.567)
    EHGR = Location(latitude=51.567, longitude=4.917)
    EHJA = Location(latitude=53.817, longitude=2.95)
    EHJR = Location(latitude=53.217, longitude=3.217)
    EHKD = Location(latitude=52.917, longitude=4.767)
    EHKV = Location(latitude=53.267, longitude=3.633)
    EHLW = Location(latitude=53.217, longitude=5.75)
    EHMA = Location(latitude=53.5, longitude=5.95)
    EHMG = Location(latitude=53.617, longitude=4.967)
    EHPG = Location(latitude=52.367, longitude=3.35)
    EHQE = Location(latitude=52.917, longitude=4.15)
    EHRD = Location(latitude=51.95, longitude=4.45)
    EHSA = Location(latitude=52, longitude=3.283)
    EHSC = Location(latitude=51.933, longitude=3.667)
    EHVK = Location(latitude=51.633, longitude=5.7)
    EHVL = Location(latitude=53.233, longitude=4.917)
    EHWO = Location(latitude=51.45, longitude=4.317)
    LGAD = Location(latitude=37.917, longitude=21.267)
    LGAL = Location(latitude=40.85, longitude=25.95)
    LGAV = Location(latitude=37.933, longitude=23.95)
    LGBL = Location(latitude=39.217, longitude=22.8)
    LGEL = Location(latitude=38.067, longitude=23.55)
    LGHI = Location(latitude=38.317, longitude=26.117)
    LGIO = Location(latitude=39.7, longitude=20.817)
    LGIR = Location(latitude=35.317, longitude=25.167)
    LGKC = Location(latitude=36.283, longitude=23.017)
    LGKF = Location(latitude=38.117, longitude=20.5)
    LGKL = Location(latitude=37.067, longitude=22.017)
    LGKO = Location(latitude=36.8, longitude=27.083)
    LGKP = Location(latitude=35.417, longitude=27.15)
    LGKR = Location(latitude=39.617, longitude=19.917)
    LGKV = Location(latitude=40.917, longitude=24.617)
    LGKZ = Location(latitude=40.283, longitude=21.833)
    LGLM = Location(latitude=39.917, longitude=25.233)
    LGLR = Location(latitude=39.65, longitude=22.467)
    LGMK = Location(latitude=37.433, longitude=25.35)
    LGMT = Location(latitude=39.05, longitude=26.6)
    LGNX = Location(latitude=37.083, longitude=25.367)
    LGPA = Location(latitude=37, longitude=25.133)
    LGPZ = Location(latitude=38.917, longitude=20.767)
    LGRP = Location(latitude=36.4, longitude=28.083)
    LGRX = Location(latitude=38.167, longitude=21.417)
    LGSA = Location(latitude=35.533, longitude=24.133)
    LGSK = Location(latitude=39.183, longitude=23.5)
    LGSM = Location(latitude=37.683, longitude=26.917)
    LGSO = Location(latitude=37.417, longitude=24.95)
    LGSR = Location(latitude=36.4, longitude=25.483)
    LGSY = Location(latitude=38.967, longitude=24.483)
    LGTG = Location(latitude=38.333, longitude=23.567)
    LGTS = Location(latitude=40.517, longitude=22.967)
    LGZA = Location(latitude=37.75, longitude=20.867)
    BKPR = Location(latitude=42.583, longitude=21.033)
    LYBE = Location(latitude=44.817, longitude=20.283)
    LYBT = Location(latitude=44.933, longitude=20.25)
    LYKV = Location(latitude=43.817, longitude=20.583)
    LYNI = Location(latitude=43.317, longitude=21.883)
    LYUZ = Location(latitude=43.9, longitude=19.7)
    LYVR = Location(latitude=45.133, longitude=21.317)
    EDAC = Location(latitude=50.983, longitude=12.5)
    EDAH = Location(latitude=53.867, longitude=14.15)
    EDBC = Location(latitude=51.85, longitude=11.417)
    EDDB = Location(latitude=52.383, longitude=13.533)
    EDDC = Location(latitude=51.133, longitude=13.75)
    EDDE = Location(latitude=50.983, longitude=10.967)
    EDDF = Location(latitude=50.05, longitude=8.6)
    EDDG = Location(latitude=52.133, longitude=7.7)
    EDDH = Location(latitude=53.633, longitude=10)
    EDDK = Location(latitude=50.867, longitude=7.15)
    EDDL = Location(latitude=51.3, longitude=6.767)
    EDDM = Location(latitude=48.35, longitude=11.817)
    EDDN = Location(latitude=49.5, longitude=11.083)
    EDDP = Location(latitude=51.433, longitude=12.233)
    EDDR = Location(latitude=49.217, longitude=7.1)
    EDDS = Location(latitude=48.667, longitude=9.217)
    EDDT = Location(latitude=52.567, longitude=13.317)
    EDDV = Location(latitude=52.467, longitude=9.683)
    EDDW = Location(latitude=53.033, longitude=8.8)
    EDFE = Location(latitude=49.967, longitude=8.633)
    EDFH = Location(latitude=49.95, longitude=7.267)
    EDFM = Location(latitude=49.467, longitude=8.517)
    EDGS = Location(latitude=50.7, longitude=8.083)
    EDHI = Location(latitude=53.533, longitude=9.833)
    EDHK = Location(latitude=54.383, longitude=10.15)
    EDHL = Location(latitude=53.8, longitude=10.717)
    EDJA = Location(latitude=47.983, longitude=10.233)
    EDLN = Location(latitude=51.233, longitude=6.5)
    EDLP = Location(latitude=51.617, longitude=8.617)
    EDLV = Location(latitude=51.6, longitude=6.15)
    EDLW = Location(latitude=51.517, longitude=7.617)
    EDMA = Location(latitude=48.417, longitude=10.917)
    EDMO = Location(latitude=48.067, longitude=11.267)
    EDNY = Location(latitude=47.667, longitude=9.517)
    EDOP = Location(latitude=53.433, longitude=11.783)
    EDQM = Location(latitude=50.3, longitude=11.85)
    EDRZ = Location(latitude=49.217, longitude=7.4)
    EDSB = Location(latitude=48.783, longitude=8.083)
    EDTD = Location(latitude=47.967, longitude=8.517)
    EDTL = Location(latitude=48.367, longitude=7.833)
    EDVE = Location(latitude=52.317, longitude=10.55)
    EDVK = Location(latitude=51.383, longitude=9.367)
    EDXW = Location(latitude=54.917, longitude=8.333)
    ETAD = Location(latitude=49.983, longitude=6.7)
    ETAR = Location(latitude=49.417, longitude=7.583)
    ETEB = Location(latitude=49.317, longitude=10.617)
    ETGG = Location(latitude=54.833, longitude=9.5)
    ETGQ = Location(latitude=53.133, longitude=9.35)
    ETGY = Location(latitude=51.75, longitude=6.267)
    ETHA = Location(latitude=47.817, longitude=10.867)
    ETHB = Location(latitude=52.267, longitude=9.067)
    ETHC = Location(latitude=52.6, longitude=10.017)
    ETHE = Location(latitude=52.283, longitude=7.367)
    ETHF = Location(latitude=51.117, longitude=9.267)
    ETHL = Location(latitude=48.217, longitude=9.917)
    ETHN = Location(latitude=49.383, longitude=9.967)
    ETHR = Location(latitude=49.217, longitude=11.083)
    ETHS = Location(latitude=52.917, longitude=10.183)
    ETIC = Location(latitude=49.7, longitude=11.933)
    ETIH = Location(latitude=49.217, longitude=11.817)
    ETIK = Location(latitude=49.467, longitude=10.367)
    ETMN = Location(latitude=53.767, longitude=8.667)
    ETND = Location(latitude=52.583, longitude=8.35)
    ETNG = Location(latitude=50.967, longitude=6.033)
    ETNH = Location(latitude=54.317, longitude=9.55)
    ETNL = Location(latitude=53.917, longitude=12.267)
    ETNN = Location(latitude=50.817, longitude=6.667)
    ETNS = Location(latitude=54.467, longitude=9.517)
    ETNT = Location(latitude=53.533, longitude=7.667)
    ETNW = Location(latitude=52.45, longitude=9.417)
    ETOU = Location(latitude=50.033, longitude=8.317)
    ETSA = Location(latitude=48.067, longitude=10.9)
    ETSB = Location(latitude=50.167, longitude=7.067)
    ETSE = Location(latitude=48.317, longitude=11.933)
    ETSH = Location(latitude=51.767, longitude=13.167)
    ETSI = Location(latitude=48.717, longitude=11.517)
    ETSL = Location(latitude=48.167, longitude=10.867)
    ETSN = Location(latitude=48.717, longitude=11.217)
    ETUO = Location(latitude=51.917, longitude=8.3)
    ETWM = Location(latitude=52.717, longitude=7.333)
    LZIB = Location(latitude=48.167, longitude=17.217)
    LZKZ = Location(latitude=48.667, longitude=21.233)
    LZPP = Location(latitude=48.617, longitude=17.833)
    LZSL = Location(latitude=48.633, longitude=19.133)
    LZTT = Location(latitude=49.067, longitude=20.25)
    LZZI = Location(latitude=49.233, longitude=18.617)
    EBAW = Location(latitude=51.2, longitude=4.467)
    EBBE = Location(latitude=50.75, longitude=4.767)
    EBBL = Location(latitude=51.167, longitude=5.467)
    EBBR = Location(latitude=50.9, longitude=4.5)
    EBCI = Location(latitude=50.467, longitude=4.45)
    EBCV = Location(latitude=50.567, longitude=3.817)
    EBDT = Location(latitude=51.017, longitude=5.067)
    EBFN = Location(latitude=51.067, longitude=2.633)
    EBFS = Location(latitude=50.217, longitude=4.65)
    EBLB = Location(latitude=50.467, longitude=6.167)
    EBLG = Location(latitude=50.633, longitude=5.45)
    EBOS = Location(latitude=51.2, longitude=2.867)
    LRAR = Location(latitude=46.183, longitude=21.267)
    LRBC = Location(latitude=46.517, longitude=26.917)
    LRBM = Location(latitude=47.667, longitude=23.467)
    LRBS = Location(latitude=44.5, longitude=26.083)
    LRCK = Location(latitude=44.367, longitude=28.483)
    LRCL = Location(latitude=46.783, longitude=23.683)
    LRCV = Location(latitude=44.317, longitude=23.883)
    LRIA = Location(latitude=47.167, longitude=27.617)
    LROD = Location(latitude=47.033, longitude=21.917)
    LROP = Location(latitude=44.567, longitude=26.1)
    LRSB = Location(latitude=45.783, longitude=24.083)
    LRSM = Location(latitude=47.7, longitude=22.883)
    LRSV = Location(latitude=47.683, longitude=26.35)
    LRTC = Location(latitude=45.067, longitude=28.717)
    LRTM = Location(latitude=46.467, longitude=24.417)
    LRTR = Location(latitude=45.817, longitude=21.333)
    LHBC = Location(latitude=46.683, longitude=21.167)
    LHBP = Location(latitude=47.417, longitude=19.267)
    LHDC = Location(latitude=47.467, longitude=21.617)
    LHKE = Location(latitude=46.917, longitude=19.75)
    LHPA = Location(latitude=47.35, longitude=17.5)
    LHPP = Location(latitude=45.983, longitude=18.233)
    LHPR = Location(latitude=47.633, longitude=17.817)
    LHSM = Location(latitude=46.683, longitude=17.167)
    LHSN = Location(latitude=47.117, longitude=20.233)
    LHUD = Location(latitude=46.25, longitude=20.1)
    LUBL = Location(latitude=47.833, longitude=27.783)
    LUBM = Location(latitude=47.867, longitude=28.217)
    LUKK = Location(latitude=46.933, longitude=28.933)
    EKAH = Location(latitude=56.283, longitude=10.617)
    EKBI = Location(latitude=55.717, longitude=9.15)
    EKCH = Location(latitude=55.617, longitude=12.667)
    EKEB = Location(latitude=55.517, longitude=8.55)
    EKHN = Location(latitude=55.6, longitude=7.617)
    EKHR = Location(latitude=55.517, longitude=7.883)
    EKKA = Location(latitude=56.3, longitude=9.117)
    EKOD = Location(latitude=55.467, longitude=10.317)
    EKRK = Location(latitude=55.567, longitude=12.117)
    EKRN = Location(latitude=55.067, longitude=14.75)
    EKSB = Location(latitude=54.967, longitude=9.8)
    EKSN = Location(latitude=57.5, longitude=10.217)
    EKSP = Location(latitude=55.217, longitude=9.267)
    EKVD = Location(latitude=55.417, longitude=9.317)
    EKVJ = Location(latitude=55.967, longitude=8.333)
    EKYT = Location(latitude=57.1, longitude=9.867)
    EICK = Location(latitude=51.85, longitude=-8.467)
    EIDW = Location(latitude=53.417, longitude=-6.25)
    EIKN = Location(latitude=53.917, longitude=-8.817)
    EIME = Location(latitude=53.3, longitude=-6.45)
    EINN = Location(latitude=52.7, longitude=-8.917)
    LXGB = Location(latitude=36.133, longitude=-5.333)
    EGJA = Location(latitude=49.717, longitude=-2.2)
    EGJB = Location(latitude=49.417, longitude=-2.6)
    EFET = Location(latitude=68.367, longitude=23.433)
    EFHA = Location(latitude=61.85, longitude=24.8)
    EFHF = Location(latitude=60.25, longitude=25.05)
    EFHK = Location(latitude=60.317, longitude=24.967)
    EFIV = Location(latitude=68.617, longitude=27.417)
    EFJO = Location(latitude=62.667, longitude=29.617)
    EFJY = Location(latitude=62.4, longitude=25.683)
    EFKA = Location(latitude=63.1, longitude=23.05)
    EFKE = Location(latitude=65.767, longitude=24.567)
    EFKI = Location(latitude=64.267, longitude=27.667)
    EFKK = Location(latitude=63.717, longitude=23.133)
    EFKS = Location(latitude=66, longitude=29.233)
    EFKT = Location(latitude=67.7, longitude=24.85)
    EFKU = Location(latitude=63.017, longitude=27.8)
    EFLP = Location(latitude=61.067, longitude=28.133)
    EFMI = Location(latitude=61.683, longitude=27.2)
    EFOU = Location(latitude=64.917, longitude=25.367)
    EFPO = Location(latitude=61.467, longitude=21.8)
    EFRO = Location(latitude=66.567, longitude=25.817)
    EFSA = Location(latitude=61.95, longitude=28.933)
    EFSI = Location(latitude=62.7, longitude=22.833)
    EFTP = Location(latitude=61.417, longitude=23.617)
    EFTU = Location(latitude=60.517, longitude=22.267)
    EFUT = Location(latitude=60.883, longitude=26.917)
    EFVA = Location(latitude=63.033, longitude=21.767)
    EFVR = Location(latitude=62.167, longitude=27.867)
    EGNS = Location(latitude=54.083, longitude=-4.633)
    KQDG = Location(latitude=37, longitude=35.43)
    LTAC = Location(latitude=40.117, longitude=32.967)
    LTAF = Location(latitude=36.967, longitude=35.283)
    LTAG = Location(latitude=37, longitude=35.417)
    LTAI = Location(latitude=36.9, longitude=30.8)
    LTAJ = Location(latitude=36.95, longitude=37.483)
    LTAN = Location(latitude=37.967, longitude=32.533)
    LTAP = Location(latitude=40.833, longitude=35.517)
    LTAR = Location(latitude=39.8, longitude=36.9)
    LTAS = Location(latitude=41.517, longitude=32.083)
    LTAT = Location(latitude=38.417, longitude=38.067)
    LTAU = Location(latitude=38.767, longitude=35.467)
    LTAW = Location(latitude=40.283, longitude=36.367)
    LTAY = Location(latitude=37.767, longitude=29.683)
    LTAZ = Location(latitude=38.767, longitude=34.55)
    LTBA = Location(latitude=40.967, longitude=28.817)
    LTBD = Location(latitude=37.817, longitude=27.883)
    LTBF = Location(latitude=39.617, longitude=27.917)
    LTBH = Location(latitude=40.133, longitude=26.417)
    LTBJ = Location(latitude=38.283, longitude=27.15)
    LTBO = Location(latitude=38.667, longitude=29.467)
    LTBQ = Location(latitude=40.717, longitude=30.067)
    LTBR = Location(latitude=40.25, longitude=29.567)
    LTBS = Location(latitude=36.7, longitude=28.767)
    LTBU = Location(latitude=41.117, longitude=27.917)
    LTBY = Location(latitude=39.817, longitude=30.517)
    LTCA = Location(latitude=38.6, longitude=39.267)
    LTCC = Location(latitude=37.867, longitude=40.2)
    LTCD = Location(latitude=39.717, longitude=39.5)
    LTCE = Location(latitude=39.95, longitude=41.167)
    LTCF = Location(latitude=40.567, longitude=43.117)
    LTCG = Location(latitude=41, longitude=39.8)
    LTCI = Location(latitude=38.45, longitude=43.317)
    LTCJ = Location(latitude=37.933, longitude=41.117)
    LTCK = Location(latitude=38.75, longitude=41.667)
    LTCL = Location(latitude=37.983, longitude=41.833)
    LTCM = Location(latitude=42.017, longitude=35.067)
    LTCN = Location(latitude=37.533, longitude=36.95)
    LTCO = Location(latitude=39.65, longitude=43.033)
    LTCP = Location(latitude=37.733, longitude=38.467)
    LTCR = Location(latitude=37.217, longitude=40.633)
    LTCS = Location(latitude=37.45, longitude=38.9)
    LTDA = Location(latitude=36.367, longitude=36.283)
    LTFC = Location(latitude=37.867, longitude=30.383)
    LTFD = Location(latitude=39.55, longitude=27.017)
    LTFE = Location(latitude=37.25, longitude=27.667)
    LTFG = Location(latitude=36.3, longitude=32.3)
    LTFH = Location(latitude=41.267, longitude=36.55)
    LTFJ = Location(latitude=40.9, longitude=29.3)
    EGJJ = Location(latitude=49.217, longitude=-2.2)
    UMKK = Location(latitude=54.883, longitude=20.6)
    UKBB = Location(latitude=50.333, longitude=30.9)
    UKCC = Location(latitude=48.083, longitude=37.75)
    UKCM = Location(latitude=47.083, longitude=37.45)
    UKCW = Location(latitude=48.417, longitude=39.383)
    UKDD = Location(latitude=48.367, longitude=35.1)
    UKDR = Location(latitude=48.05, longitude=33.217)
    UKHH = Location(latitude=49.917, longitude=36.283)
    UKKE = Location(latitude=49.417, longitude=32)
    UKKK = Location(latitude=50.4, longitude=30.45)
    UKKM = Location(latitude=50.6, longitude=30.2)
    UKLI = Location(latitude=48.883, longitude=24.683)
    UKLL = Location(latitude=49.817, longitude=23.933)
    UKLN = Location(latitude=48.267, longitude=25.983)
    UKLR = Location(latitude=50.6, longitude=26.133)
    UKON = Location(latitude=47.05, longitude=31.917)
    UKOO = Location(latitude=46.417, longitude=30.683)
    LPAR = Location(latitude=38.883, longitude=-9.033)
    LPBJ = Location(latitude=38.083, longitude=-7.933)
    LPFR = Location(latitude=37.017, longitude=-7.967)
    LPMR = Location(latitude=39.817, longitude=-8.867)
    LPMT = Location(latitude=38.7, longitude=-9.05)
    LPOV = Location(latitude=40.917, longitude=-8.65)
    LPPR = Location(latitude=41.217, longitude=-8.667)
    LPPT = Location(latitude=38.767, longitude=-9.117)
    LPST = Location(latitude=38.817, longitude=-9.317)
    LDPL = Location(latitude=44.883, longitude=13.917)
    LJCE = Location(latitude=45.9, longitude=15.517)
    LJLJ = Location(latitude=46.217, longitude=14.467)
    LJMB = Location(latitude=46.467, longitude=15.667)
    LJPZ = Location(latitude=45.467, longitude=13.617)
    EGAA = Location(latitude=54.633, longitude=-6.217)
    EGAC = Location(latitude=54.617, longitude=-5.867)
    EGAE = Location(latitude=55.033, longitude=-7.15)
    EGBB = Location(latitude=52.467, longitude=-1.75)
    EGBE = Location(latitude=52.367, longitude=-1.467)
    EGBJ = Location(latitude=51.9, longitude=-2.167)
    EGCC = Location(latitude=53.35, longitude=-2.267)
    EGCN = Location(latitude=53.483, longitude=-1)
    EGDM = Location(latitude=51.167, longitude=-1.75)
    EGDR = Location(latitude=50.083, longitude=-5.25)
    EGDX = Location(latitude=51.4, longitude=-3.433)
    EGDY = Location(latitude=51, longitude=-2.617)
    EGEC = Location(latitude=55.433, longitude=-5.683)
    EGEO = Location(latitude=56.467, longitude=-5.4)
    EGFF = Location(latitude=51.383, longitude=-3.35)
    EGGD = Location(latitude=51.367, longitude=-2.717)
    EGGP = Location(latitude=53.317, longitude=-2.85)
    EGGW = Location(latitude=51.867, longitude=-0.367)
    EGHE = Location(latitude=49.917, longitude=-6.283)
    EGHH = Location(latitude=50.783, longitude=-1.833)
    EGHI = Location(latitude=50.95, longitude=-1.35)
    EGHQ = Location(latitude=50.433, longitude=-5)
    EGKA = Location(latitude=50.833, longitude=-0.3)
    EGKB = Location(latitude=51.317, longitude=0.017)
    EGKK = Location(latitude=51.133, longitude=-0.167)
    EGLC = Location(latitude=51.5, longitude=0.05)
    EGLF = Location(latitude=51.283, longitude=-0.767)
    EGLL = Location(latitude=51.483, longitude=-0.45)
    EGMC = Location(latitude=51.567, longitude=0.7)
    EGMD = Location(latitude=50.95, longitude=0.917)
    EGMH = Location(latitude=51.35, longitude=1.333)
    EGNC = Location(latitude=54.933, longitude=-2.817)
    EGNH = Location(latitude=53.767, longitude=-3.017)
    EGNJ = Location(latitude=53.567, longitude=-0.35)
    EGNM = Location(latitude=53.867, longitude=-1.633)
    EGNR = Location(latitude=53.167, longitude=-2.967)
    EGNT = Location(latitude=55.017, longitude=-1.7)
    EGNV = Location(latitude=54.517, longitude=-1.417)
    EGNX = Location(latitude=52.817, longitude=-1.317)
    EGOM = Location(latitude=55.05, longitude=-2.55)
    EGOP = Location(latitude=51.717, longitude=-4.317)
    EGOS = Location(latitude=52.8, longitude=-2.667)
    EGOV = Location(latitude=53.25, longitude=-4.533)
    EGOW = Location(latitude=53.583, longitude=-3.05)
    EGPA = Location(latitude=58.95, longitude=-2.883)
    EGPB = Location(latitude=59.867, longitude=-1.3)
    EGPC = Location(latitude=58.45, longitude=-3.067)
    EGPD = Location(latitude=57.2, longitude=-2.217)
    EGPE = Location(latitude=57.517, longitude=-4.033)
    EGPF = Location(latitude=55.867, longitude=-4.417)
    EGPH = Location(latitude=55.95, longitude=-3.35)
    EGPI = Location(latitude=55.683, longitude=-6.25)
    EGPK = Location(latitude=55.5, longitude=-4.567)
    EGPL = Location(latitude=57.467, longitude=-7.367)
    EGPM = Location(latitude=60.417, longitude=-1.3)
    EGPN = Location(latitude=56.45, longitude=-3.017)
    EGPO = Location(latitude=58.217, longitude=-6.317)
    EGPU = Location(latitude=56.5, longitude=-6.867)
    EGQA = Location(latitude=57.833, longitude=-3.95)
    EGQK = Location(latitude=57.65, longitude=-3.567)
    EGQL = Location(latitude=56.383, longitude=-2.867)
    EGQM = Location(latitude=55.417, longitude=-1.6)
    EGQS = Location(latitude=57.717, longitude=-3.317)
    EGSC = Location(latitude=52.2, longitude=0.167)
    EGSH = Location(latitude=52.683, longitude=1.283)
    EGSS = Location(latitude=51.883, longitude=0.217)
    EGTC = Location(latitude=52.067, longitude=-0.617)
    EGTE = Location(latitude=50.717, longitude=-3.417)
    EGUB = Location(latitude=51.617, longitude=-1.067)
    EGUL = Location(latitude=52.417, longitude=0.567)
    EGUN = Location(latitude=52.367, longitude=0.467)
    EGUO = Location(latitude=51.433, longitude=-2.283)
    EGUW = Location(latitude=52.117, longitude=0.967)
    EGUY = Location(latitude=52.35, longitude=-0.117)
    EGVA = Location(latitude=51.667, longitude=-1.767)
    EGVN = Location(latitude=51.767, longitude=-1.583)
    EGVO = Location(latitude=51.233, longitude=-0.95)
    EGVP = Location(latitude=51.133, longitude=-1.567)
    EGWC = Location(latitude=52.617, longitude=-2.283)
    EGWU = Location(latitude=51.55, longitude=-0.417)
    EGXC = Location(latitude=53.1, longitude=-0.167)
    EGXD = Location(latitude=54.133, longitude=-1.417)
    EGXE = Location(latitude=54.283, longitude=-1.517)
    EGXG = Location(latitude=53.833, longitude=-1.2)
    EGXP = Location(latitude=53.3, longitude=-0.55)
    EGXS = Location(latitude=53.483, longitude=0.15)
    EGXT = Location(latitude=52.617, longitude=-0.467)
    EGXU = Location(latitude=54.05, longitude=-1.25)
    EGXV = Location(latitude=53.867, longitude=-0.45)
    EGXW = Location(latitude=53.183, longitude=-0.517)
    EGXZ = Location(latitude=54.2, longitude=-1.383)
    EGYD = Location(latitude=53.033, longitude=-0.5)
    EGYH = Location(latitude=52.867, longitude=0.15)
    EGYM = Location(latitude=52.633, longitude=0.567)
    ENSF = Location(latitude=61.25, longitude=1.85)
    ELLX = Location(latitude=49.633, longitude=6.2)
    LEAB = Location(latitude=38.95, longitude=-1.85)
    LEAL = Location(latitude=38.267, longitude=-0.55)
    LEAM = Location(latitude=36.85, longitude=-2.367)
    LEAO = Location(latitude=38.95, longitude=-3.733)
    LEAS = Location(latitude=43.533, longitude=-6.017)
    LEBA = Location(latitude=37.85, longitude=-4.817)
    LEBB = Location(latitude=43.283, longitude=-2.917)
    LEBG = Location(latitude=42.367, longitude=-3.617)
    LEBL = Location(latitude=41.267, longitude=2.067)
    LEBT = Location(latitude=39.583, longitude=-0.45)
    LEBZ = Location(latitude=38.867, longitude=-6.817)
    LECH = Location(latitude=40.2, longitude=0.067)
    LECO = Location(latitude=43.283, longitude=-8.367)
    LECV = Location(latitude=40.633, longitude=-3.717)
    LEGA = Location(latitude=37.117, longitude=-3.617)
    LEGE = Location(latitude=41.883, longitude=2.767)
    LEGR = Location(latitude=37.167, longitude=-3.767)
    LEGT = Location(latitude=40.283, longitude=-3.717)
    LEHC = Location(latitude=42.083, longitude=-0.317)
    LEIB = Location(latitude=38.867, longitude=1.367)
    LEJR = Location(latitude=36.75, longitude=-6.067)
    LELC = Location(latitude=37.767, longitude=-0.8)
    LELL = Location(latitude=41.517, longitude=2.1)
    LELN = Location(latitude=42.567, longitude=-5.65)
    LELO = Location(latitude=42.45, longitude=-2.317)
    LEMD = Location(latitude=40.483, longitude=-3.567)
    LEMG = Location(latitude=36.667, longitude=-4.5)
    LEMH = Location(latitude=39.867, longitude=4.217)
    LEMO = Location(latitude=37.167, longitude=-5.617)
    LEPA = Location(latitude=39.533, longitude=2.717)
    LEPP = Location(latitude=42.767, longitude=-1.65)
    LERI = Location(latitude=37.95, longitude=-1.217)
    LERS = Location(latitude=41.133, longitude=1.167)
    LERT = Location(latitude=36.633, longitude=-6.333)
    LESA = Location(latitude=40.95, longitude=-5.5)
    LESO = Location(latitude=43.35, longitude=-1.8)
    LEST = Location(latitude=42.883, longitude=-8.417)
    LETO = Location(latitude=40.467, longitude=-3.467)
    LEVC = Location(latitude=39.5, longitude=-0.467)
    LEVD = Location(latitude=41.717, longitude=-4.833)
    LEVS = Location(latitude=40.367, longitude=-3.767)
    LEVT = Location(latitude=42.867, longitude=-2.717)
    LEVX = Location(latitude=42.217, longitude=-8.617)
    LEXJ = Location(latitude=43.417, longitude=-3.817)
    LEZG = Location(latitude=41.667, longitude=-1.017)
    LEZL = Location(latitude=37.417, longitude=-5.9)
    LMML = Location(latitude=35.85, longitude=14.467)
    EFMA = Location(latitude=60.117, longitude=19.883)
    UMBB = Location(latitude=52.117, longitude=23.9)
    UMGG = Location(latitude=52.533, longitude=31.017)
    UMII = Location(latitude=55.133, longitude=30.35)
    UMMG = Location(latitude=53.6, longitude=24.05)
    UMMM = Location(latitude=53.867, longitude=27.517)
    UMMS = Location(latitude=53.883, longitude=28.033)
    UMOO = Location(latitude=53.95, longitude=30.1)
    ULAA = Location(latitude=64.6, longitude=40.717)
    ULLI = Location(latitude=59.8, longitude=30.267)
    ULMM = Location(latitude=68.783, longitude=32.75)
    ULOO = Location(latitude=57.783, longitude=28.4)
    URKA = Location(latitude=45, longitude=37.35)
    URKK = Location(latitude=45.033, longitude=39.15)
    URML = Location(latitude=42.817, longitude=47.65)
    URMM = Location(latitude=44.217, longitude=43.1)
    URMN = Location(latitude=43.517, longitude=43.633)
    URMT = Location(latitude=45.117, longitude=42.117)
    URRR = Location(latitude=47.25, longitude=39.817)
    URSS = Location(latitude=43.45, longitude=39.95)
    URWI = Location(latitude=46.367, longitude=44.333)
    UUBP = Location(latitude=53.217, longitude=34.183)
    UUDD = Location(latitude=55.4, longitude=37.9)
    UUEE = Location(latitude=55.967, longitude=37.417)
    UUMO = Location(latitude=55.517, longitude=37.5)
    UUOB = Location(latitude=50.65, longitude=36.583)
    UUOK = Location(latitude=51.75, longitude=36.283)
    UUOO = Location(latitude=51.817, longitude=39.233)
    UUWW = Location(latitude=55.6, longitude=37.267)
    UUYY = Location(latitude=61.65, longitude=50.817)
    UWGG = Location(latitude=56.233, longitude=43.783)
    UWKD = Location(latitude=55.617, longitude=49.283)
    UWKE = Location(latitude=55.567, longitude=52.083)
    UWKS = Location(latitude=56.083, longitude=47.35)
    UWLW = Location(latitude=54.4, longitude=48.8)
    ENAL = Location(latitude=62.567, longitude=6.117)
    ENAN = Location(latitude=69.283, longitude=16.133)
    ENAT = Location(latitude=69.967, longitude=23.367)
    ENBL = Location(latitude=61.4, longitude=5.767)
    ENBN = Location(latitude=65.467, longitude=12.217)
    ENBO = Location(latitude=67.267, longitude=14.367)
    ENBR = Location(latitude=60.283, longitude=5.217)
    ENBS = Location(latitude=70.617, longitude=29.717)
    ENBV = Location(latitude=70.867, longitude=29.017)
    ENCN = Location(latitude=58.2, longitude=8.067)
    ENDR = Location(latitude=64.35, longitude=7.8)
    ENDU = Location(latitude=69.033, longitude=18.55)
    ENEV = Location(latitude=68.517, longitude=16.667)
    ENFB = Location(latitude=61.2, longitude=1.833)
    ENFG = Location(latitude=61, longitude=9.3)
    ENFL = Location(latitude=61.567, longitude=5.017)
    ENGC = Location(latitude=61.017, longitude=2.017)
    ENGM = Location(latitude=60.2, longitude=11.1)
    ENHD = Location(latitude=59.35, longitude=5.217)
    ENHF = Location(latitude=70.667, longitude=23.667)
    ENHK = Location(latitude=70.467, longitude=22.133)
    ENHM = Location(latitude=59.567, longitude=2.233)
    ENHV = Location(latitude=71.017, longitude=25.967)
    ENKB = Location(latitude=63.117, longitude=7.817)
    ENKR = Location(latitude=69.717, longitude=29.883)
    ENLK = Location(latitude=68.15, longitude=13.617)
    ENMH = Location(latitude=71.017, longitude=27.817)
    ENML = Location(latitude=62.75, longitude=7.267)
    ENMS = Location(latitude=65.783, longitude=13.217)
    ENNA = Location(latitude=70.067, longitude=24.967)
    ENNE = Location(latitude=66.033, longitude=8.083)
    ENNK = Location(latitude=68.467, longitude=17.383)
    ENNM = Location(latitude=64.467, longitude=11.567)
    ENNO = Location(latitude=59.567, longitude=9.217)
    ENOA = Location(latitude=60.5, longitude=2.833)
    ENOL = Location(latitude=63.7, longitude=9.583)
    ENOV = Location(latitude=62.167, longitude=6.067)
    ENQA = Location(latitude=60.65, longitude=3.717)
    ENQC = Location(latitude=60.883, longitude=3.617)
    ENQR = Location(latitude=61.35, longitude=2.217)
    ENRA = Location(latitude=66.367, longitude=14.3)
    ENRM = Location(latitude=64.833, longitude=11.133)
    ENRO = Location(latitude=62.567, longitude=11.333)
    ENRS = Location(latitude=67.517, longitude=12.083)
    ENRY = Location(latitude=59.367, longitude=10.767)
    ENSD = Location(latitude=61.817, longitude=6.117)
    ENSE = Location(latitude=61.45, longitude=2.15)
    ENSG = Location(latitude=61.133, longitude=7.117)
    ENSH = Location(latitude=68.25, longitude=14.667)
    ENSK = Location(latitude=68.567, longitude=15.017)
    ENSL = Location(latitude=58.367, longitude=1.9)
    ENSN = Location(latitude=59.167, longitude=9.567)
    ENSO = Location(latitude=59.783, longitude=5.333)
    ENSR = Location(latitude=69.767, longitude=20.967)
    ENSS = Location(latitude=70.333, longitude=31.05)
    ENST = Location(latitude=65.967, longitude=12.467)
    ENTC = Location(latitude=69.683, longitude=18.917)
    ENTO = Location(latitude=59.2, longitude=10.267)
    ENVA = Location(latitude=63.467, longitude=10.917)
    ENVD = Location(latitude=70.067, longitude=29.85)
    ENZV = Location(latitude=58.867, longitude=5.617)
    LFAQ = Location(latitude=49.967, longitude=2.7)
    LFAT = Location(latitude=50.517, longitude=1.617)
    LFBA = Location(latitude=44.167, longitude=0.6)
    LFBC = Location(latitude=44.517, longitude=-1.117)
    LFBD = Location(latitude=44.817, longitude=-0.7)
    LFBE = Location(latitude=44.817, longitude=0.517)
    LFBG = Location(latitude=45.667, longitude=-0.317)
    LFBH = Location(latitude=46.183, longitude=-1.2)
    LFBI = Location(latitude=46.567, longitude=0.317)
    LFBL = Location(latitude=45.867, longitude=1.167)
    LFBM = Location(latitude=43.917, longitude=-0.5)
    LFBO = Location(latitude=43.617, longitude=1.367)
    LFBP = Location(latitude=43.367, longitude=-0.417)
    LFBT = Location(latitude=43.167, longitude=0)
    LFBU = Location(latitude=45.717, longitude=0.217)
    LFBY = Location(latitude=43.667, longitude=-1.067)
    LFBZ = Location(latitude=43.467, longitude=-1.517)
    LFCR = Location(latitude=44.383, longitude=2.467)
    LFGJ = Location(latitude=47.05, longitude=5.433)
    LFJL = Location(latitude=48.967, longitude=6.25)
    LFJR = Location(latitude=47.567, longitude=-0.3)
    LFKB = Location(latitude=42.533, longitude=9.467)
    LFKC = Location(latitude=42.517, longitude=8.8)
    LFKF = Location(latitude=41.5, longitude=9.083)
    LFKJ = Location(latitude=41.917, longitude=8.8)
    LFKS = Location(latitude=41.917, longitude=9.4)
    LFLB = Location(latitude=45.633, longitude=5.867)
    LFLC = Location(latitude=45.767, longitude=3.167)
    LFLL = Location(latitude=45.717, longitude=5.067)
    LFLN = Location(latitude=46.417, longitude=4.017)
    LFLP = Location(latitude=45.917, longitude=6.083)
    LFLS = Location(latitude=45.367, longitude=5.317)
    LFLW = Location(latitude=44.883, longitude=2.417)
    LFLX = Location(latitude=46.85, longitude=1.717)
    LFLY = Location(latitude=45.717, longitude=4.95)
    LFMC = Location(latitude=43.367, longitude=6.367)
    LFMD = Location(latitude=43.533, longitude=6.95)
    LFMH = Location(latitude=45.517, longitude=4.283)
    LFMI = Location(latitude=43.517, longitude=4.917)
    LFMK = Location(latitude=43.217, longitude=2.317)
    LFML = Location(latitude=43.45, longitude=5.217)
    LFMN = Location(latitude=43.633, longitude=7.2)
    LFMO = Location(latitude=44.117, longitude=4.867)
    LFMP = Location(latitude=42.717, longitude=2.867)
    LFMT = Location(latitude=43.567, longitude=3.967)
    LFMU = Location(latitude=43.317, longitude=3.35)
    LFMV = Location(latitude=43.9, longitude=4.9)
    LFMY = Location(latitude=43.6, longitude=5.083)
    LFOA = Location(latitude=47.033, longitude=2.633)
    LFOB = Location(latitude=49.467, longitude=2.117)
    LFOC = Location(latitude=48.033, longitude=1.367)
    LFOE = Location(latitude=49.017, longitude=1.217)
    LFOH = Location(latitude=49.517, longitude=0.067)
    LFOJ = Location(latitude=47.967, longitude=1.75)
    LFOK = Location(latitude=48.783, longitude=4.183)
    LFOP = Location(latitude=49.367, longitude=1.167)
    LFOT = Location(latitude=47.433, longitude=0.717)
    LFPB = Location(latitude=48.967, longitude=2.45)
    LFPC = Location(latitude=49.25, longitude=2.517)
    LFPG = Location(latitude=49.017, longitude=2.517)
    LFPM = Location(latitude=48.617, longitude=2.667)
    LFPN = Location(latitude=48.75, longitude=2.117)
    LFPO = Location(latitude=48.717, longitude=2.383)
    LFPT = Location(latitude=49.1, longitude=2.033)
    LFPV = Location(latitude=48.767, longitude=2.2)
    LFQB = Location(latitude=48.317, longitude=4.017)
    LFQG = Location(latitude=47, longitude=3.1)
    LFQQ = Location(latitude=50.567, longitude=3.1)
    LFRB = Location(latitude=48.45, longitude=-4.417)
    LFRC = Location(latitude=49.633, longitude=-1.467)
    LFRD = Location(latitude=48.583, longitude=-2.067)
    LFRG = Location(latitude=49.367, longitude=0.15)
    LFRH = Location(latitude=47.767, longitude=-3.45)
    LFRI = Location(latitude=46.7, longitude=-1.367)
    LFRJ = Location(latitude=48.517, longitude=-4.15)
    LFRK = Location(latitude=49.167, longitude=-0.45)
    LFRL = Location(latitude=48.267, longitude=-4.417)
    LFRM = Location(latitude=47.95, longitude=0.2)
    LFRN = Location(latitude=48.067, longitude=-1.717)
    LFRO = Location(latitude=48.75, longitude=-3.467)
    LFRQ = Location(latitude=47.967, longitude=-4.167)
    LFRS = Location(latitude=47.167, longitude=-1.6)
    LFRT = Location(latitude=48.517, longitude=-2.85)
    LFRU = Location(latitude=48.6, longitude=-3.817)
    LFRZ = Location(latitude=47.317, longitude=-2.167)
    LFSB = Location(latitude=47.6, longitude=7.517)
    LFSD = Location(latitude=47.267, longitude=5.067)
    LFSI = Location(latitude=48.617, longitude=4.9)
    LFSL = Location(latitude=45.033, longitude=1.483)
    LFSN = Location(latitude=48.667, longitude=6.217)
    LFSO = Location(latitude=48.567, longitude=5.967)
    LFST = Location(latitude=48.533, longitude=7.617)
    LFSX = Location(latitude=47.767, longitude=6.333)
    LFTH = Location(latitude=43.1, longitude=6.15)
    LFTW = Location(latitude=43.75, longitude=4.417)
    LFYR = Location(latitude=47.317, longitude=1.667)
    LYPG = Location(latitude=42.367, longitude=19.25)
    LYTV = Location(latitude=42.383, longitude=18.717)
    LKCV = Location(latitude=49.933, longitude=15.383)
    LKKB = Location(latitude=50.117, longitude=14.533)
    LKKU = Location(latitude=49.05, longitude=17.45)
    LKKV = Location(latitude=50.2, longitude=12.917)
    LKLN = Location(latitude=49.683, longitude=13.267)
    LKMT = Location(latitude=49.667, longitude=18.117)
    LKNA = Location(latitude=49.167, longitude=16.117)
    LKPD = Location(latitude=50.017, longitude=15.733)
    LKPR = Location(latitude=50.1, longitude=14.267)
    LKTB = Location(latitude=49.133, longitude=16.683)
    EVLA = Location(latitude=56.517, longitude=21.1)
    EVRA = Location(latitude=56.917, longitude=23.967)
    EVVA = Location(latitude=57.35, longitude=21.55)
    LIBA = Location(latitude=41.517, longitude=15.717)
    LIBC = Location(latitude=39, longitude=17.067)
    LIBD = Location(latitude=41.117, longitude=16.767)
    LIBF = Location(latitude=41.417, longitude=15.517)
    LIBG = Location(latitude=40.517, longitude=17.383)
    LIBH = Location(latitude=40.417, longitude=16.867)
    LIBN = Location(latitude=40.217, longitude=18.133)
    LIBP = Location(latitude=42.417, longitude=14.183)
    LIBQ = Location(latitude=39.317, longitude=16.383)
    LIBR = Location(latitude=40.65, longitude=17.95)
    LIBV = Location(latitude=40.767, longitude=16.917)
    LIBY = Location(latitude=39.817, longitude=18.35)
    LICA = Location(latitude=38.883, longitude=16.25)
    LICB = Location(latitude=37, longitude=14.617)
    LICC = Location(latitude=37.467, longitude=15.05)
    LICD = Location(latitude=35.5, longitude=12.617)
    LICF = Location(latitude=38.2, longitude=15.55)
    LICG = Location(latitude=36.817, longitude=11.967)
    LICJ = Location(latitude=38.167, longitude=13.083)
    LICL = Location(latitude=37.067, longitude=14.217)
    LICR = Location(latitude=38.067, longitude=15.65)
    LICT = Location(latitude=37.917, longitude=12.5)
    LICZ = Location(latitude=37.383, longitude=14.917)
    LIEA = Location(latitude=40.617, longitude=8.267)
    LIEB = Location(latitude=39.933, longitude=9.717)
    LIEC = Location(latitude=39.1, longitude=9.517)
    LIED = Location(latitude=39.35, longitude=8.967)
    LIEE = Location(latitude=39.25, longitude=9.067)
    LIEO = Location(latitude=40.883, longitude=9.517)
    LIMA = Location(latitude=45.083, longitude=7.6)
    LIMC = Location(latitude=45.617, longitude=8.717)
    LIME = Location(latitude=45.667, longitude=9.683)
    LIMF = Location(latitude=45.217, longitude=7.65)
    LIMG = Location(latitude=44.033, longitude=8.117)
    LIMH = Location(latitude=45.933, longitude=7.7)
    LIMJ = Location(latitude=44.417, longitude=8.833)
    LIMK = Location(latitude=45.017, longitude=7.717)
    LIML = Location(latitude=45.467, longitude=9.267)
    LIMN = Location(latitude=45.517, longitude=8.667)
    LIMP = Location(latitude=44.817, longitude=10.3)
    LIMS = Location(latitude=44.917, longitude=9.717)
    LIMT = Location(latitude=44.467, longitude=9.917)
    LIMU = Location(latitude=43.95, longitude=8.167)
    LIMV = Location(latitude=44.55, longitude=8.917)
    LIMW = Location(latitude=45.733, longitude=7.367)
    LIMY = Location(latitude=44.25, longitude=7.783)
    LIMZ = Location(latitude=44.533, longitude=7.617)
    LIPA = Location(latitude=46.017, longitude=12.583)
    LIPB = Location(latitude=46.467, longitude=11.317)
    LIPC = Location(latitude=44.217, longitude=12.3)
    LIPE = Location(latitude=44.517, longitude=11.3)
    LIPF = Location(latitude=44.817, longitude=11.617)
    LIPH = Location(latitude=45.633, longitude=12.167)
    LIPI = Location(latitude=45.967, longitude=13.05)
    LIPK = Location(latitude=44.2, longitude=12.067)
    LIPL = Location(latitude=45.417, longitude=10.267)
    LIPO = Location(latitude=45.417, longitude=10.333)
    LIPQ = Location(latitude=45.817, longitude=13.467)
    LIPR = Location(latitude=44.017, longitude=12.617)
    LIPS = Location(latitude=45.667, longitude=12.083)
    LIPX = Location(latitude=45.383, longitude=10.867)
    LIPY = Location(latitude=43.617, longitude=13.367)
    LIPZ = Location(latitude=45.5, longitude=12.35)
    LIQC = Location(latitude=40.55, longitude=14.2)
    LIQN = Location(latitude=42.417, longitude=12.833)
    LIQO = Location(latitude=42.367, longitude=11.167)
    LIQW = Location(latitude=44.067, longitude=9.967)
    LIRA = Location(latitude=41.8, longitude=12.583)
    LIRE = Location(latitude=41.633, longitude=12.417)
    LIRF = Location(latitude=41.783, longitude=12.217)
    LIRG = Location(latitude=42, longitude=12.717)
    LIRH = Location(latitude=41.65, longitude=13.3)
    LIRI = Location(latitude=40.617, longitude=14.917)
    LIRK = Location(latitude=42.467, longitude=12.967)
    LIRL = Location(latitude=41.533, longitude=12.9)
    LIRM = Location(latitude=41.033, longitude=14.067)
    LIRN = Location(latitude=40.883, longitude=14.3)
    LIRP = Location(latitude=43.667, longitude=10.367)
    LIRQ = Location(latitude=43.783, longitude=11.183)
    LIRS = Location(latitude=42.75, longitude=11.067)
    LIRT = Location(latitude=41.033, longitude=15.217)
    LIRU = Location(latitude=41.95, longitude=12.5)
    LIRV = Location(latitude=42.417, longitude=12.067)
    LIRX = Location(latitude=42.733, longitude=10.383)
    LIRZ = Location(latitude=43.1, longitude=12.5)
    LIVC = Location(latitude=44.2, longitude=10.683)
    LIVD = Location(latitude=46.717, longitude=12.217)
    LIVE = Location(latitude=46.85, longitude=10.5)
    LIVF = Location(latitude=43.517, longitude=12.717)
    LIVM = Location(latitude=44.45, longitude=12.3)
    LIVO = Location(latitude=46.5, longitude=13.567)
    LIVP = Location(latitude=46.15, longitude=11.033)
    LIVR = Location(latitude=46.283, longitude=11.767)
    LIVT = Location(latitude=45.683, longitude=13.75)
    LIYW = Location(latitude=46.017, longitude=12.617)
    UWWW = Location(latitude=53.5, longitude=50.167)
    LQBK = Location(latitude=44.933, longitude=17.3)
    LQMO = Location(latitude=43.283, longitude=17.833)
    LQSA = Location(latitude=43.817, longitude=18.317)
    LQTZ = Location(latitude=44.467, longitude=18.733)
    UKFF = Location(latitude=45.033, longitude=33.983)
    LWOH = Location(latitude=41.183, longitude=20.75)
    LWSK = Location(latitude=41.967, longitude=21.633)
    LBBG = Location(latitude=42.567, longitude=27.517)
    LBGO = Location(latitude=43.15, longitude=25.717)
    LBPD = Location(latitude=42.133, longitude=24.8)
    LBSF = Location(latitude=42.7, longitude=23.417)
    LBWN = Location(latitude=43.217, longitude=27.817)
    ESCF = Location(latitude=58.383, longitude=15.517)
    ESDF = Location(latitude=56.267, longitude=15.267)
    ESGG = Location(latitude=57.667, longitude=12.3)
    ESGJ = Location(latitude=57.767, longitude=14.067)
    ESGP = Location(latitude=57.767, longitude=11.867)
    ESGR = Location(latitude=58.45, longitude=13.967)
    ESGT = Location(latitude=58.317, longitude=12.333)
    ESIB = Location(latitude=58.417, longitude=12.683)
    ESKN = Location(latitude=58.783, longitude=16.9)
    ESMK = Location(latitude=55.917, longitude=14.067)
    ESMQ = Location(latitude=56.683, longitude=16.3)
    ESMS = Location(latitude=55.533, longitude=13.367)
    ESMT = Location(latitude=56.667, longitude=12.817)
    ESMX = Location(latitude=56.917, longitude=14.717)
    ESNG = Location(latitude=67.133, longitude=20.817)
    ESNK = Location(latitude=63.033, longitude=17.767)
    ESNL = Location(latitude=64.55, longitude=18.717)
    ESNN = Location(latitude=62.517, longitude=17.433)
    ESNO = Location(latitude=63.383, longitude=18.967)
    ESNQ = Location(latitude=67.817, longitude=20.317)
    ESNS = Location(latitude=64.617, longitude=21.067)
    ESNU = Location(latitude=63.783, longitude=20.267)
    ESNX = Location(latitude=65.583, longitude=19.283)
    ESNZ = Location(latitude=63.167, longitude=14.5)
    ESOE = Location(latitude=59.217, longitude=15.05)
    ESOK = Location(latitude=59.45, longitude=13.333)
    ESOW = Location(latitude=59.567, longitude=16.617)
    ESPA = Location(latitude=65.533, longitude=22.117)
    ESPE = Location(latitude=65.883, longitude=20.15)
    ESSA = Location(latitude=59.633, longitude=17.933)
    ESSB = Location(latitude=59.35, longitude=17.933)
    ESSD = Location(latitude=60.417, longitude=15.5)
    ESSL = Location(latitude=58.383, longitude=15.667)
    ESSP = Location(latitude=58.567, longitude=16.25)
    ESSV = Location(latitude=57.667, longitude=18.35)
    ESTA = Location(latitude=56.283, longitude=12.833)
    ESTL = Location(latitude=56.067, longitude=13.2)
    ESUD = Location(latitude=64.967, longitude=17.7)
    ESUT = Location(latitude=65.8, longitude=15.083)
    EEKA = Location(latitude=58.983, longitude=22.833)
    EEKE = Location(latitude=58.233, longitude=22.517)
    EEPU = Location(latitude=58.417, longitude=24.467)
    EETN = Location(latitude=59.4, longitude=24.817)
    EETU = Location(latitude=58.317, longitude=26.683)
    LATI = Location(latitude=41.417, longitude=19.717)
    UKLU = Location(latitude=48.633, longitude=22.267)
    LOAN = Location(latitude=47.85, longitude=16.25)
    LOAV = Location(latitude=47.967, longitude=16.267)
    LOIH = Location(latitude=47.383, longitude=9.7)
    LOWG = Location(latitude=47, longitude=15.417)
    LOWI = Location(latitude=47.267, longitude=11.333)
    LOWK = Location(latitude=46.65, longitude=14.317)
    LOWL = Location(latitude=48.217, longitude=14.183)
    LOWS = Location(latitude=47.783, longitude=13.017)
    LOWW = Location(latitude=48.117, longitude=16.567)
    LOWZ = Location(latitude=47.3, longitude=12.783)
    LOXT = Location(latitude=48.317, longitude=16.117)
    LOXZ = Location(latitude=47.2, longitude=14.75)
    EYKA = Location(latitude=54.967, longitude=24.083)
    EYPA = Location(latitude=55.967, longitude=21.083)
    EYSA = Location(latitude=55.9, longitude=23.4)
    EYVI = Location(latitude=54.633, longitude=25.283)
    URWA = Location(latitude=46.283, longitude=48.017)
    URWW = Location(latitude=48.783, longitude=44.333)
    UWSS = Location(latitude=51.567, longitude=46.067)
    EPBY = Location(latitude=53.1, longitude=18)
    EPGD = Location(latitude=54.367, longitude=18.467)
    EPKK = Location(latitude=50.067, longitude=19.8)
    EPKT = Location(latitude=50.467, longitude=19.083)
    EPLB = Location(latitude=51.233, longitude=22.717)
    EPLL = Location(latitude=51.717, longitude=19.4)
    EPPO = Location(latitude=52.417, longitude=16.817)
    EPRZ = Location(latitude=50.1, longitude=22.05)
    EPSC = Location(latitude=53.583, longitude=14.9)
    EPWA = Location(latitude=52.167, longitude=20.967)
    EPWR = Location(latitude=51.1, longitude=16.867)
    EPZG = Location(latitude=52.133, longitude=15.8)
    LDDU = Location(latitude=42.567, longitude=18.267)
    LDLO = Location(latitude=44.567, longitude=14.4)
    LDOS = Location(latitude=45.467, longitude=18.817)
    LDRI = Location(latitude=45.217, longitude=14.567)
    LDSB = Location(latitude=43.283, longitude=16.683)
    LDSP = Location(latitude=43.517, longitude=16.3)
    LDZA = Location(latitude=45.717, longitude=16.067)
    LDZD = Location(latitude=44.1, longitude=15.367)
    UKDE = Location(latitude=47.867, longitude=35.317)
    LSGC = Location(latitude=47.083, longitude=6.8)
    LSGG = Location(latitude=46.25, longitude=6.117)
    LSGS = Location(latitude=46.217, longitude=7.317)
    LSMA = Location(latitude=46.95, longitude=8.283)
    LSMD = Location(latitude=47.4, longitude=8.65)
    LSME = Location(latitude=47.1, longitude=8.3)
    LSMM = Location(latitude=46.75, longitude=8.117)
    LSMP = Location(latitude=46.817, longitude=6.917)
    LSZA = Location(latitude=46, longitude=8.917)
    LSZB = Location(latitude=46.917, longitude=7.5)
    LSZC = Location(latitude=46.967, longitude=8.383)
    LSZG = Location(latitude=47.167, longitude=7.417)
    LSZH = Location(latitude=47.483, longitude=8.533)
    LSZR = Location(latitude=47.483, longitude=9.567)
    LSZS = Location(latitude=46.533, longitude=9.883)
    FMMI = Location(latitude=-18.8, longitude=47.467)
    FMMS = Location(latitude=-17.067, longitude=49.817)
    FMMT = Location(latitude=-18.117, longitude=49.383)
    FMNA = Location(latitude=-12.333, longitude=49.283)
    FMNM = Location(latitude=-15.667, longitude=46.35)
    FMNN = Location(latitude=-13.317, longitude=48.317)
    FMSD = Location(latitude=-25.017, longitude=46.95)
    FJDG = Location(latitude=-7.317, longitude=72.417)
    YPXM = Location(latitude=-10.45, longitude=105.667)
    YPCC = Location(latitude=-12.167, longitude=96.817)
    FMCH = Location(latitude=-11.517, longitude=43.267)
    FSIA = Location(latitude=-4.667, longitude=55.517)
    VRMM = Location(latitude=4.2, longitude=73.517)
    FIMP = Location(latitude=-20.417, longitude=57.667)
    FIMR = Location(latitude=-19.75, longitude=63.367)
    FMCZ = Location(latitude=-12.8, longitude=45.267)
    FMEE = Location(latitude=-20.867, longitude=55.517)
    FMEP = Location(latitude=-21.317, longitude=55.417)
    NSFA = Location(latitude=-13.833, longitude=-172)
    NZAA = Location(latitude=-37.017, longitude=174.8)
    NZCH = Location(latitude=-43.467, longitude=172.55)
    NZWN = Location(latitude=-41.317, longitude=174.8)
    PTKK = Location(latitude=7.467, longitude=151.85)
    PTYA = Location(latitude=9.483, longitude=138.083)
    SCIP = Location(latitude=-27.167, longitude=-109.433)
    NVSC = Location(latitude=-13.833, longitude=167.55)
    NVSS = Location(latitude=-15.517, longitude=167.217)
    NVVV = Location(latitude=-17.683, longitude=168.3)
    NFFN = Location(latitude=-17.75, longitude=177.45)
    NFNA = Location(latitude=-18.05, longitude=178.567)
    NGFU = Location(latitude=-8.517, longitude=179.217)
    SEST = Location(latitude=-0.9, longitude=-89.583)
    AGGH = Location(latitude=-9.417, longitude=160.05)
    PGUA = Location(latitude=13.567, longitude=144.917)
    PGUM = Location(latitude=13.483, longitude=144.8)
    PHBK = Location(latitude=22.033, longitude=-159.783)
    PHHI = Location(latitude=21.483, longitude=-158.05)
    PHJH = Location(latitude=20.967, longitude=-156.667)
    PHJR = Location(latitude=21.317, longitude=-158.067)
    PHKO = Location(latitude=19.733, longitude=-156.05)
    PHLI = Location(latitude=21.967, longitude=-159.333)
    PHMK = Location(latitude=21.15, longitude=-157.1)
    PHNG = Location(latitude=21.433, longitude=-157.767)
    PHNL = Location(latitude=21.333, longitude=-157.917)
    PHNY = Location(latitude=20.767, longitude=-156.933)
    PHOG = Location(latitude=20.9, longitude=-156.433)
    PHSF = Location(latitude=19.767, longitude=-155.55)
    PHTO = Location(latitude=19.717, longitude=-155.05)
    PLCH = Location(latitude=1.983, longitude=-157.35)
    PTSA = Location(latitude=5.35, longitude=162.95)
    PKMJ = Location(latitude=7.067, longitude=171.267)
    PKMR = Location(latitude=7.083, longitude=171.383)
    PMDY = Location(latitude=28.217, longitude=-177.367)
    NIUE = Location(latitude=-19.067, longitude=-169.917)
    YSNF = Location(latitude=-29.017, longitude=167.917)
    NWWW = Location(latitude=-22.017, longitude=166.217)
    NSTU = Location(latitude=-14.333, longitude=-170.717)
    PTKR = Location(latitude=7.333, longitude=134.483)
    PTRO = Location(latitude=7.367, longitude=134.55)
    PTPN = Location(latitude=6.967, longitude=158.217)
    AYMO = Location(latitude=-2.067, longitude=147.417)
    AYNZ = Location(latitude=-6.567, longitude=146.717)
    AYPY = Location(latitude=-9.417, longitude=147.217)
    AYWK = Location(latitude=-3.583, longitude=143.667)
    NCRG = Location(latitude=-21.2, longitude=-159.817)
    PGRO = Location(latitude=14.183, longitude=145.25)
    PGSN = Location(latitude=15.117, longitude=145.717)
    PGWT = Location(latitude=14.983, longitude=145.617)
    NTAA = Location(latitude=-17.55, longitude=-149.617)
    NGTT = Location(latitude=1.35, longitude=172.917)
    NFTF = Location(latitude=-21.233, longitude=-175.15)
    NFTL = Location(latitude=-19.783, longitude=-174.333)
    NFTV = Location(latitude=-18.583, longitude=-173.967)
    PWAK = Location(latitude=19.283, longitude=166.65)

    @classmethod
    def from_location(cls, latitude: float, longitude: float) -> MetarStation:
        location = Location(latitude=latitude, longitude=longitude)
        return min(iter(MetarStation), key=lambda metar_station: _central_angle(location, metar_station))
