"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.topologicalSort = void 0;
/**
 * Return a topological sort of all elements of xs, according to the given dependency functions
 *
 * Dependencies outside the referenced set are ignored.
 *
 * Not a stable sort, but in order to keep the order as stable as possible, we'll sort by key
 * among elements of equal precedence.
 */
function topologicalSort(xs, keyFn, depFn) {
    const remaining = new Map();
    for (const element of xs) {
        const key = keyFn(element);
        remaining.set(key, { key, element, dependencies: depFn(element) });
    }
    const ret = new Array();
    while (remaining.size > 0) {
        // All elements with no more deps in the set can be ordered
        const selectable = Array.from(remaining.values()).filter((e) => e.dependencies.every((d) => !remaining.has(d)));
        selectable.sort((a, b) => (a.key < b.key ? -1 : b.key < a.key ? 1 : 0));
        for (const selected of selectable) {
            ret.push(selected.element);
            remaining.delete(selected.key);
        }
        // If we didn't make any progress, we got stuck
        if (selectable.length === 0) {
            throw new Error(`Could not determine ordering between: ${Array.from(remaining.keys()).join(", ")}`);
        }
    }
    return ret;
}
exports.topologicalSort = topologicalSort;
//# sourceMappingURL=data:application/json;base64,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