"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudArtifact = void 0;
const cxschema = require("@alicloud/ros-cdk-assembly-schema");
const metadata_1 = require("./metadata");
/**
 * Represents an artifact within a cloud assembly.
 */
class CloudArtifact {
    constructor(assembly, id, manifest) {
        this.assembly = assembly;
        this.id = id;
        this.manifest = manifest;
        this.messages = this.renderMessages();
        this._dependencyIDs = manifest.dependencies || [];
    }
    /**
     * Returns a subclass of `CloudArtifact` based on the artifact type defined in the artifact manifest.
     * @param assembly The cloud assembly from which to load the artifact
     * @param id The artifact ID
     * @param artifact The artifact manifest
     * @returns the `CloudArtifact` that matches the artifact type or `undefined` if it's an artifact type that is unrecognized by this module.
     */
    static fromManifest(assembly, id, artifact) {
        switch (artifact.type) {
            case cxschema.ArtifactType.ALIYUN_ROS_STACK:
                return new ros_stack_artifact_1.RosStackArtifact(assembly, id, artifact);
            case cxschema.ArtifactType.CDK_TREE:
                return new tree_cloud_artifact_1.TreeCloudArtifact(assembly, id, artifact);
            case cxschema.ArtifactType.NESTED_CLOUD_ASSEMBLY:
                return new nested_cloud_assembly_artifact_1.NestedCloudAssemblyArtifact(assembly, id, artifact);
            default:
                return undefined;
        }
    }
    /**
     * Returns all the artifacts that this artifact depends on.
     */
    get dependencies() {
        // if (this._deps) {
        //   return this._deps;
        // }
        // this._deps = this._dependencyIDs.map((id) => {
        //   const dep = this.assembly.tryGetArtifact(id);
        //   if (!dep) {
        //     throw new Error(`Artifact ${this.id} depends on non-existing artifact ${id}`);
        //   }
        //   return dep;
        // });
        return this._deps;
    }
    /**
     * @returns all the metadata entries of a specific type in this artifact.
     * @param type
     */
    findMetadataByType(type) {
        const result = new Array();
        for (const path of Object.keys(this.manifest.metadata || {})) {
            for (const entry of (this.manifest.metadata || {})[path]) {
                if (entry.type === type) {
                    result.push({ path, ...entry });
                }
            }
        }
        return result;
    }
    renderMessages() {
        const messages = new Array();
        for (const [id, metadata] of Object.entries(this.manifest.metadata || {})) {
            for (const entry of metadata) {
                let level;
                switch (entry.type) {
                    case cxschema.ArtifactMetadataEntryType.WARN:
                        level = metadata_1.SynthesisMessageLevel.WARNING;
                        break;
                    case cxschema.ArtifactMetadataEntryType.ERROR:
                        level = metadata_1.SynthesisMessageLevel.ERROR;
                        break;
                    case cxschema.ArtifactMetadataEntryType.INFO:
                        level = metadata_1.SynthesisMessageLevel.INFO;
                        break;
                    default:
                        continue;
                }
                messages.push({ level, entry, id });
            }
        }
        return messages;
    }
}
exports.CloudArtifact = CloudArtifact;
// needs to be defined at the end to avoid a cyclic dependency
const ros_stack_artifact_1 = require("./artifacts/ros-stack-artifact");
const nested_cloud_assembly_artifact_1 = require("./artifacts/nested-cloud-assembly-artifact");
const tree_cloud_artifact_1 = require("./artifacts/tree-cloud-artifact");
//# sourceMappingURL=data:application/json;base64,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