import { MainAreaWidget } from '@jupyterlab/apputils';
import { PathExt } from '@jupyterlab/coreutils';
import { PromiseDelegate } from '@lumino/coreutils';
import { Panel } from '@lumino/widgets';
import { requestAPI } from '../../utils';
import { NotebookDiff } from '../diff/NotebookDiff';
import { PlainTextDiff } from '../diff/plaintext';
/**
 * FileDiffWidget
 */
export class FileDiffWidget extends MainAreaWidget {
    constructor(props) {
        const content = new Panel();
        const isLoaded = new PromiseDelegate();
        super({
            content,
            reveal: isLoaded.promise
        });
        content.addClass('jp-PullRequestTab');
        FileDiffWidget.loadDiff(props.pullRequestId, props.filename)
            .then(([diff, threads]) => {
            isLoaded.resolve();
            this.showDiff(Object.assign(Object.assign({}, props), { diff,
                threads }));
        })
            .catch(reason => {
            let msg = `Load File Error (${reason.message})`;
            if (reason.message.toLowerCase().includes("'utf-8' codec can't decode")) {
                msg = `Diff for ${props.filename} is not supported.`;
            }
            isLoaded.reject(msg);
        });
    }
    /**
     * Load the file diff and the associated discussions
     *
     * @param pullRequestId Pull request Id
     * @param filename Filename
     */
    static async loadDiff(pullRequestId, filename) {
        return Promise.all([
            requestAPI(`pullrequests/files/content?id=${encodeURIComponent(pullRequestId)}&filename=${encodeURIComponent(filename)}`, 'GET'),
            requestAPI(`pullrequests/files/comments?id=${encodeURIComponent(pullRequestId)}&filename=${encodeURIComponent(filename)}`, 'GET')
        ]);
    }
    /**
     * Display the diff widget depending of the file type
     *
     * @param diffProps Diff properties
     */
    showDiff(diffProps) {
        const fileExtension = PathExt.extname(diffProps.filename).toLowerCase();
        if (fileExtension === '.ipynb') {
            this.content.addWidget(new NotebookDiff(diffProps));
        }
        else {
            try {
                this.content.addWidget(new PlainTextDiff(diffProps));
            }
            catch (reason) {
                this.showError(reason.message || reason);
            }
        }
    }
    /**
     * Display an error message
     *
     * @param message Error message
     */
    showError(message) {
        while (this.children().next()) {
            this.children()
                .next()
                .dispose();
        }
        this.node.innerHTML = `<h2 class="jp-PullRequestTabError"><span style="color: 'var(--jp-ui-font-color1)';">Error Loading File:</span> ${message}</h2>`;
    }
}
//# sourceMappingURL=FileDiffWidget.js.map