import { DocumentRegistry } from '@jupyterlab/docregistry';
import { ActionButton } from '@jupyterlab/git/lib/components/ActionButton';
import { caretDownIcon, caretUpIcon, linkIcon } from '@jupyterlab/ui-components';
import React, { useEffect, useState } from 'react';
import { BeatLoader } from 'react-spinners';
import { CommandIDs } from '../../tokens';
import { requestAPI } from '../../utils';
import { FileItem } from './FileItem';
/**
 * Open an URL in a new window
 *
 * @param link URL to open
 */
function openLink(link) {
    window.open(link, '_blank');
}
/**
 * PullRequestItem component
 *
 * @param props Component properties
 */
export function PullRequestItem(props) {
    var _a;
    const { commands, docRegistry, pullRequest } = props;
    /**
     * Pull request modified files
     */
    const [files, setFiles] = useState(null);
    /**
     * Is the file list expanded?
     */
    const [isExpanded, setIsExpanded] = useState(false);
    /**
     * Is the file list being loaded?
     */
    const [isLoading, setIsLoading] = useState(false);
    /**
     * Error message
     */
    const [error, setError] = useState(null);
    useEffect(() => {
        setFiles(null);
        setIsExpanded(false);
        setIsLoading(false);
        setError(null);
    }, [props.pullRequest]);
    const fetchFiles = async () => {
        setIsLoading(true);
        try {
            const results = (await requestAPI('pullrequests/prs/files?id=' + encodeURIComponent(pullRequest.id), 'GET'));
            setFiles(results.map((rawFile) => {
                const path = rawFile.name;
                return Object.assign(Object.assign({}, rawFile), { fileType: docRegistry.getFileTypesForPath(path)[0] ||
                        DocumentRegistry.defaultTextFileType });
            }));
        }
        finally {
            setIsLoading(false);
        }
    };
    // This makes a shallow copy of data[i], the data[i].files are not copied
    // If files need to be mutated, will need to restructure props / deep copy
    const toggleFilesExpanded = () => {
        if (files === null && !isExpanded) {
            setError(null);
            fetchFiles()
                .then(() => {
                setIsExpanded(!isExpanded);
            })
                .catch(reason => {
                setError(`Failed to get pull request files ${reason}`);
            });
        }
        else {
            setIsExpanded(!isExpanded);
        }
    };
    return (React.createElement("li", { key: pullRequest.id, onClick: () => {
            commands.execute(CommandIDs.prOpenDescription, { pullRequest });
        } },
        React.createElement("div", { className: "jp-PullRequestBrowserItemListItem" },
            React.createElement("h2", null, pullRequest.title),
            React.createElement("div", { className: "jp-PullRequestBrowserItemListItemIconWrapper" },
                React.createElement(ActionButton, { icon: linkIcon, onClick: (e) => {
                        e.stopPropagation();
                        openLink(pullRequest.link);
                    }, title: "Open in new tab" }),
                React.createElement(ActionButton, { icon: isExpanded ? caretUpIcon : caretDownIcon, onClick: (e) => {
                        e.stopPropagation();
                        toggleFilesExpanded();
                    }, title: isExpanded ? 'Hide modified files' : 'Show modified files' }))),
        isLoading ? (React.createElement(BeatLoader, { sizeUnit: 'px', size: 5, color: 'var(--jp-ui-font-color1)', loading: isLoading })) : (isExpanded &&
            (error ? (React.createElement("div", null,
                React.createElement("h2", { className: "jp-PullRequestBrowserGroupError" }, "Error Listing Pull Request Files:"),
                error)) : (React.createElement("ul", { className: "jp-PullRequestBrowserItemFileList" }, (_a = files) === null || _a === void 0 ? void 0 : _a.map(file => (React.createElement("li", { key: `${pullRequest.internalId}-${file.name}`, onClick: (e) => {
                    e.stopPropagation();
                    commands.execute(CommandIDs.prOpenDiff, {
                        file,
                        pullRequest
                    });
                } },
                React.createElement(FileItem, { file: file }))))))))));
}
//# sourceMappingURL=PullRequestItem.js.map