# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['strawberry_resources',
 'strawberry_resources.integrations',
 'strawberry_resources.utils']

package_data = \
{'': ['*']}

install_requires = \
['strawberry-graphql>=0.140.3', 'typing-extensions>=4.2.0']

extras_require = \
{'django': ['strawberry-graphql-django>=0.8', 'django>=3.2'],
 'django-plus': ['strawberry-graphql-django>=0.8',
                 'strawberry-django-plus>=2.0',
                 'django>=3.2',
                 'django-choices-field>=2.0']}

setup_kwargs = {
    'name': 'strawberry-resources',
    'version': '0.1.0',
    'description': 'Introspection utilities to extract data from strawberry graphql',
    'long_description': '# strawberry-resources\n\n[![build status](https://img.shields.io/endpoint.svg?url=https%3A%2F%2Factions-badge.atrox.dev%2Fblb-ventures%2Fstrawberry-resources%2Fbadge%3Fref%3Dmain&style=flat)](https://actions-badge.atrox.dev/blb-ventures/strawberry-resources/goto?ref=main)\n[![coverage](https://img.shields.io/codecov/c/github/blb-ventures/strawberry-resources.svg)](https://codecov.io/gh/blb-ventures/strawberry-resources)\n[![downloads](https://pepy.tech/badge/strawberry-resources)](https://pepy.tech/project/strawberry-resources)\n[![PyPI version](https://img.shields.io/pypi/v/strawberry-resources.svg)](https://pypi.org/project/strawberry-resources/)\n![python version](https://img.shields.io/pypi/pyversions/strawberry-resources.svg)\n![django version](https://img.shields.io/pypi/djversions/strawberry-resources.svg)\n\nIntrospection utilities to extract data from the schema to use as helpers in the\nclient, like building an automatic form for input types.\n\n## Installation\n\nJust install the package with pip or your preferred package manager:\n\n```shell\npip install strawberry-resources\n```\n\n## How to use\n\nThis lib provides a `Query` type that has two queries:\n\n- `resources`: Returns a list of all available resources in the schema\n- `resource`: Returns an specific resource given its name\n\nYou can use [merge_type](https://strawberry.rocks/docs/guides/tools#merge_types)\nto merge it with your own `Query` type.\n\nThen, given this example:\n\n```python\n@strawberry.enum\nclass Color(enum.Enum):\n    YELLOW = strawberry.enum_value("yellow", description="Color Yellow")\n    RED = "red"\n    ORANGE = "orange"\n\n@strawberry.type\nclass Fruit:\n    name: str\n    color: Annotated[Color, config(label="Color")]\n    weight: Annotate[float, strawberry_resource.config(label="Weight")]\n\n@strawberry.type\nclass Market:\n    name: Annotate[str, strawberry_resource.config(label="Market Name")]\n    fruits: Annotate[List[Fruit], strawberry_resource.config(label="Fruits")]\n\n@strawberry.type\nclass Query:\n    market: Market\n```\n\nYou can query `resource(name: "Market")` which would return:\n\n```json\n{\n  "resource": {\n    "name": "Market"\n    "fields": [\n      {\n        "__typename": "Field",\n        "choices": null,\n        "defaultValue": null,\n        "filterable": false,\n        "helpText": null,\n        "kind": "STRING",\n        "label": "Market Name",\n        "multiple": false,\n        "name": "name",\n        "orderable": false,\n        "resource": null,\n        "validation": {\n          "__typename": "BaseFieldValidation",\n          "required": true\n        }\n      },\n      {\n        "__typename": "FieldObject",\n        "label": "Fruits",\n        "name": "fruits",\n        "objKind": "OBJECT_LIST"\n        "fields": [\n          {\n            "__typename": "Field",\n            "choices": null,\n            "defaultValue": null,\n            "filterable": false,\n            "helpText": null,\n            "kind": "STRING",\n            "label": "name",\n            "multiple": false,\n            "name": "name",\n            "orderable": false,\n            "resource": null,\n            "validation": {\n              "__typename": "BaseFieldValidation",\n              "required": true\n            }\n          },\n          {\n            "__typename": "Field",\n            "choices": [\n              {\n                "group": null,\n                "label": "Color Yellow",\n                "value": "YELLOW"\n              },\n              {\n                "group": null,\n                "label": "RED",\n                "value": "RED"\n              },\n              {\n                "group": null,\n                "label": "ORANGE",\n                "value": "ORANGE"\n              }\n            ],\n            "defaultValue": null,\n            "filterable": false,\n            "helpText": null,\n            "kind": "STRING",\n            "label": "Color",\n            "multiple": false,\n            "name": "color",\n            "orderable": false,\n            "resource": null,\n            "validation": {\n              "__typename": "BaseFieldValidation",\n              "required": true\n            }\n          },\n          {\n            "__typename": "Field",\n            "choices": null,\n            "defaultValue": null,\n            "filterable": false,\n            "helpText": null,\n            "kind": "FLOAT",\n            "label": "Weight",\n            "multiple": false,\n            "name": "weight",\n            "orderable": false,\n            "resource": null,\n            "validation": {\n              "__typename": "BaseFieldValidation",\n              "required": true\n            }\n          }\n        ],\n      }\n    ],\n  }\n}\n```\n\n## Customizing the resource\n\nStrawberry resource will introspect the schema to automatically fill some information\nregarding the field. However, you can customize them by annotating your fields with\nyour own config.\n\nIn the example above we customized the `label` for most attributes, except for `Fruit.name`.\nAll possible config options are:\n\n- `kind` (`FieldKind`): The kind of the field\n- `multiple` (`bool`): If the field is multivalued (i.e. a List)\n- `orderable` (`bool`): If the field is orderable`\n- `filterable` (`bool`): If the field is filterable`\n- `label` (`str | None`): An optional human friendly label for the field\n- `help_text` (`str | FieldChoice`): An optional list with available choices for the field\n- `default_value` (`JSON | None`): The default value for the field\n- `validation` (`BaseFieldValidation`): Validation options for the field\n\nCheck the [types.py](strawberry_resources/types.py) module for more details.\n\n## Integrations\n\n### Django\n\nIf you are using Django, and by extend\n[strawberry-graphql-django](https://github.com/strawberry-graphql/strawberry-graphql-django) or\n[strawberry-django-plus](https://github.com/blb-ventures/strawberry-django-plus), the integration\nwill be automatically used to configure some options by introspecting your model.\n\nThe following will be retrieved from the fields in it, specially when typing it with\n`strawberry.auto`:\n\n- `kind`: The field kind will be automatically set based on the model field type. e.g. a `CharField`\n  will generate a kind of `STRING`, a `DateTimeField` will generate a kind of `DATETIME` and so on.\n- `orderable`: Will be automatically filled if the django type has an\n  [ordering](https://strawberry-graphql.github.io/strawberry-graphql-django/references/ordering/)\n  set on it, and the field itself is there\n- `filterable`: Will be automatically filled if the django type has\n  [filters](https://strawberry-graphql.github.io/strawberry-graphql-django/references/filters/)\n  set on it, and the field itself is there\n- `label`: Will be automatically filled using the field\'s `verbose_name` value\n- `help_text`: Will be automatically filled using the field\'s `help_text` value\n- `choices`: Will be automatically filled using the field\'s `choices` value\n- `default_value`: Will be automatically filled using the field\'s `default` value\n\n### Creating your own integration\n\nYou can create your own extension by creating an instance of\n`strawberry_resources.integrations.StrawberryResourceIntegration`. It expects 4 attributes:\n\n- `name`: The name of the integration\n- `get_extra_mappings`: A callable that should return a dict mapping a type to a `FieldKind`\n- `get_field_options`: A mapping that receives the type that contains the field, the field itself,\n  the resolved type of the field and if it is a list of not. It is expect to return a dict with\n  the options mentioned in the section above.\n- `order`: An optional order to be used when running the integrations.\n\nThe integrations will run in the `order` they are defined. The official integrations in\nthis repo all have an order of `0`, so you can define yours to run before them by passing\na negative value, or after them by passing something greater than `0`.\n\nNOTE: strawberry-resources is eager to have more integrations, so feel free to open a PR\nfor us sending yours! :)\n\n## How options are resolved\n\nAll options will be merged recursively to generate the final resource options. That means that\noptions defined later will override the ones defined earlier. The order is the following:\n\n- The options will be created with its `kind` retrieved from the kind mapping (considering the\n  ones returned by the integrations as well), and its `label` will be set the same as its name\n  by default.\n- The integrations will run in the order they were defined, and each option returned will\n  me merged recursively with the current options.\n- At last, options will be retrieved by the field\'s annotations and will have the highest\n  priority when merging with the rest.\n\n## Licensing\n\nThe code in this project is licensed under MIT license. See [LICENSE](./LICENSE)\nfor more information.\n',
    'author': 'Thiago Bellini Ribeiro',
    'author_email': 'thiago@bellini.dev',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/blb-ventures/strawberry-resources',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
