import numpy as np
import gqcms
import scipy.linalg as sp


class DMRG:
    def __init__(self, J) -> None:
        """
        Constructor for the DMRG class
        Sets up the required fields
        """

        # parameter for the Heisenberg model
        self.J = J
        # basis of system hamiltonian
        self.size = 0

        # ground state energy
        self.energy = 0
        # ground state
        self.ground_state = 0

        # amount of sites
        self.sites = 0

        # define basic spinops
        self.s_z = np.array([[0.5, 0], [0, -0.5]])
        self.s_x = np.array([[0, 0.5], [0.5, 0]])
        self.s_y = np.array([[0, -0.5j], [0.5j, 0]])

        self.raising = self.s_x + 1j * self.s_y
        self.lowering = self.s_x - 1j * self.s_y

        # store hamiltonians for all system sizes, start with one site values.
        # separate dicts will be neede for finite size algorithm.
        self.hamiltonians_b = {1: np.zeros((2, 2))}
        self.hamiltonians_u = {1: np.zeros((2, 2))}

        # store spinops for all system sizes
        self.sz_s_b = {1: self.s_z}
        self.raisings_b = {1: self.raising}
        self.lowerings_b = {1: self.lowering}
        self.sz_s_u = {1: self.s_z}
        self.raisings_u = {1: self.raising}
        self.lowerings_u = {1: self.lowering}

        self.one_site_number_operator = gqcms.NumberOperator(
            0,
            [
                gqcms.Determinant(nalpha=1, nbeta=0, sites=1),
                gqcms.Determinant(nalpha=0, nbeta=1, sites=1),
            ],
        )
        self.number_b = {1: self.one_site_number_operator}
        self.number_u = {1: self.one_site_number_operator}

    # suite of modules that accomodate easier interaction of class with other codes.
    def setSize(self, new_size):
        """Set a new size for the system"""
        self.size = new_size

    def setEnergy(self, new_energy):
        """Set a new energy for the system"""
        self.energy = new_energy

    def setGroundState(self, new_state):
        """Set a new ground state"""
        self.ground_state = new_state

    def setSites(self, new_sites):
        """Set a new amount of sites"""
        self.sites = new_sites

    def setHamiltonians(self, new_hams):
        """Set the new hamiltonians"""
        self.hamiltonians = new_hams

    def setSpinops(self, new_sz_s, new_raisings, new_lowerings):
        """Set the new spinops"""
        self.sz_s = new_sz_s
        self.raisings = new_raisings
        self.lowerings = new_lowerings

    @staticmethod
    def constructDensityMatrix(state, system_size):
        """
        Will construct the density matrix for the given single state.
        state: np.array in row form
        """

        dim = system_size * 2
        rho = np.zeros((dim, dim))

        for rownr, row in enumerate(rho):
            for columnnr, value in enumerate(row):
                one = state[rownr::dim]
                two = state[columnnr::dim]
                rho[rownr, columnnr] = np.sum(one.conjugate() * two)

        return rho

    @staticmethod
    def constructDensityMatrixM(state, system_size):
        """
        Control routine from Mishmash in attempt to find what is wrong
        Will calculate the density matrix
        state: np.array in row form
        """

        dim = system_size * 2

        psi0 = state.reshape([dim, -1], order="C")
        rho = np.dot(psi0, psi0.conjugate().transpose())

        return rho

    @staticmethod
    def selectDensityVectors(rho, states_kept=2, mute=True):
        """
        Will calculate the eigenvectors of the density matrix and select the most important ones. We will also calculate the
        von Neumann entropy.
        rho: density matrix as generated by the constructDensityMatrix function.
        states_kept: How many states are we supposed to keep? In order to help define this, we will print the error
        mute: if True, print error message
        """
        E, C = sp.eigh(rho)
        states_kept = min(len(E), states_kept)
        truncation_error = 1 - np.sum(E[-states_kept:])
        # sometimes the eigenvalues of the densitymatrix are effectively -0
        # these will be eliminated by setting it equal to something that is
        # effectively +0, but it does not mess with the np.log function
        mask = np.where(E < 0)
        E[mask] = 1e-18
        logs = np.log(E)
        S = -np.sum(E * logs)

        S_lost = -np.sum(E[:-states_kept] * logs[:-states_kept])
        S_kept = -np.sum(E[-states_kept:] * logs[-states_kept:])
        if not mute:
            print(f"the error is {truncation_error}")

        return C[:, -states_kept:], truncation_error, S, [S_lost, S_kept]

    def infinite(self, maxiter, states_kept=2, mute=False):
        """
        Will perform the infinite size DMRG algorithm as described by Hallberg
        Can also be used to build a system for use with the finite size method

        J: parameter for the Heisenberg model
        maxiter: How many sites do you want to add to a system? Iteration 0 Will have four sites, 1 will have 6 and so on...
            Formula: #sites = 4 + maxiter*2

        states_kept: The amount of states kept in every iteration

        mute: if true, mute all print statements
        """

        # Iteration 0
        # Step 1
        system = gqcms.heisenberg(4)
        H_sb = system.calculateHamiltonian(self.J, periodic=False)

        # Step 2
        E, C = np.linalg.eigh(H_sb)
        ground_state = C[:, 0]

        self.setGroundState(ground_state)

        # We are interested in the energy of the ground state
        ground_state_energy = E[0]

        self.setEnergy(ground_state_energy)

        energy_list = [ground_state_energy]

        # Step 3
        rho = DMRG.constructDensityMatrix(ground_state, system_size=2)
        O, error, entropy, entropy_stats = DMRG.selectDensityVectors(
            rho, states_kept=states_kept, mute=True
        )
        entropy_stat_list = [entropy_stats]

        number_operator_b = np.kron(
            self.one_site_number_operator, self.one_site_number_operator
        )
        # We will be counting the electrons in the open subsystem + the left site
        expect = np.einsum("ij, ij", rho, number_operator_b)
        number_list = [expect]

        if not mute:
            print(f"4 sites: energy: {ground_state_energy}")
            print(f"truncation error: {error}")
            print(f"entropy: {entropy}")

        self.setSites(4)
        error_list = [error]
        entropy_list = [entropy]

        # Step 4
        # hamiltonians for b and u
        # in the first iteration, these are both equal
        H_b = gqcms.heisenberg(2).calculateHamiltonian(self.J, periodic=False)

        H_b = O.T @ H_b @ O

        # spinops for b and u
        S_z_b = O.T @ np.kron(np.eye(2), self.s_z) @ O
        S_m_b = O.T @ np.kron(np.eye(2), self.lowering) @ O
        S_p_b = O.T @ np.kron(np.eye(2), self.raising) @ O

        # number operator
        n_b = (
            O.T
            @ np.kron(self.one_site_number_operator, self.one_site_number_operator)
            @ O
        )

        # save the operators for use in the finite size algorithm
        self.hamiltonians_b[2] = H_b
        self.sz_s_b[2] = S_z_b
        self.lowerings_b[2] = S_m_b
        self.raisings_b[2] = S_p_b
        self.number_b[2] = n_b

        self.hamiltonians_u[2] = H_b
        self.sz_s_u[2] = S_z_b
        self.lowerings_u[2] = S_m_b
        self.raisings_u[2] = S_p_b
        self.number_u[2] = n_b

        dim = H_b.shape[0]

        self.setSize(dim)

        for iteration in range(1, maxiter):
            dim = H_b.shape[0]

            H_b = np.kron(H_b, np.eye(2)) + self.J * (
                np.kron(S_z_b, self.s_z)
                + 0.5 * np.kron(S_m_b, self.raising)
                + 0.5 * np.kron(S_p_b, self.lowering)
            )
            S_z_b = np.kron(np.eye(dim), self.s_z)
            S_m_b = np.kron(np.eye(dim), self.lowering)
            S_p_b = np.kron(np.eye(dim), self.raising)

            n_b = np.kron(n_b, self.one_site_number_operator)

            H_sb = (
                np.kron(H_b, np.eye(H_b.shape[0]))
                + np.kron(np.eye(H_b.shape[0]), H_b)
                + self.J
                * (
                    np.kron(S_z_b, S_z_b)
                    + 0.5 * np.kron(S_p_b, S_m_b)
                    + 0.5 * np.kron(S_m_b, S_p_b)
                )
            )

            E, C = np.linalg.eigh(H_sb)
            ground_state = C[:, 0]
            ground_state_energy = E[0]

            self.setGroundState(ground_state)
            self.setEnergy(ground_state_energy)

            energy_list.append(ground_state_energy)
            rho = DMRG.constructDensityMatrix(ground_state, system_size=dim)
            O, error, entropy, entropy_stats = DMRG.selectDensityVectors(
                rho, mute=True, states_kept=states_kept
            )
            entropy_stat_list.append(entropy_stats)

            expect = np.einsum("ij, ij", rho, n_b)
            number_list.append(expect)

            sites = 4 + iteration * 2
            self.setSites(sites)

            if not mute:
                print(f"{sites} sites: energy: {ground_state_energy}")
                print(f"truncation error: {error}")
                print(f"entropy: {entropy}")

            entropy_list.append(entropy)
            error_list.append(error)
            H_b = O.T @ H_b @ O

            # spinops for b and u
            S_z_b = O.T @ S_z_b @ O
            S_m_b = O.T @ S_m_b @ O
            S_p_b = O.T @ S_p_b @ O
            n_b = O.T @ n_b @ O

            # save the operators for use in the finite size algorithm
            self.hamiltonians_b[sites // 2] = H_b
            self.sz_s_b[sites // 2] = S_z_b
            self.lowerings_b[sites // 2] = S_m_b
            self.raisings_b[sites // 2] = S_p_b
            self.hamiltonians_u[sites // 2] = H_b
            self.sz_s_u[sites // 2] = S_z_b
            self.lowerings_u[sites // 2] = S_m_b
            self.raisings_u[sites // 2] = S_p_b

            self.number_b[sites // 2] = n_b
            self.number_u[sites // 2] = n_b

            dim = H_b.shape[0]
            self.setSize(dim)

        return energy_list, entropy_list, error_list, entropy_stat_list, number_list

    def step(self, system_sites, env_sites, states_kept=2, growing="sys"):
        """
        Will grow the system by one site
        system_sites, the current amount of sites in the system
        env_sites, the current amount of sites in the environment
        """
        # as we will start by growing the system unitll it includes all sites
        # we will have to transform the required operators append these to the stored operators
        H_b = np.kron(self.hamiltonians_b[system_sites], np.eye(2)) + self.J * (
            np.kron(self.sz_s_b[system_sites], self.s_z)
            + 0.5 * np.kron(self.lowerings_b[system_sites], self.raising)
            + 0.5 * np.kron(self.raisings_b[system_sites], self.lowering)
        )

        H_u = np.kron(self.hamiltonians_u[env_sites], np.eye(2)) + self.J * (
            np.kron(self.sz_s_u[env_sites], self.s_z)
            + 0.5 * np.kron(self.lowerings_u[env_sites], self.raising)
            + 0.5 * np.kron(self.raisings_u[env_sites], self.lowering)
        )

        dim_b = self.hamiltonians_b[system_sites].shape[0]
        S_z_b = np.kron(np.eye(dim_b), self.s_z)
        S_p_b = np.kron(np.eye(dim_b), self.raising)
        S_m_b = np.kron(np.eye(dim_b), self.lowering)

        dim_u = self.hamiltonians_u[env_sites].shape[0]
        S_z_u = np.kron(np.eye(dim_u), self.s_z)
        S_p_u = np.kron(np.eye(dim_u), self.raising)
        S_m_u = np.kron(np.eye(dim_u), self.lowering)

        H_sb = (
            np.kron(H_b, np.eye(H_u.shape[0]))
            + np.kron(np.eye(H_b.shape[0]), H_u)
            + self.J
            * (
                np.kron(S_z_b, S_z_u)
                + 0.5 * np.kron(S_m_b, S_p_u)
                + 0.5 * np.kron(S_p_b, S_m_u)
            )
        )

        E, C = sp.eigh(H_sb)
        ground_state_energy = E[0]
        ground_state = C[:, 0]

        self.setGroundState(ground_state)
        self.setEnergy(ground_state_energy)

        # Only the growing system changes its matrices, while the other system uses matrices that already existed
        if growing == "sys":
            rho = DMRG.constructDensityMatrixM(ground_state, system_size=dim_b)
            O, error, entropy, entropy_stats = DMRG.selectDensityVectors(
                rho, states_kept=states_kept, mute=True
            )
            H_b = O.T @ H_b @ O
            S_z_b = O.T @ S_z_b @ O
            S_m_b = O.T @ S_m_b @ O
            S_p_b = O.T @ S_p_b @ O
            self.hamiltonians_b[system_sites + 1] = H_b
            self.sz_s_b[system_sites + 1] = S_z_b
            self.lowerings_b[system_sites + 1] = S_m_b
            self.raisings_b[system_sites + 1] = S_p_b

            n_b = self.number_b[system_sites]
            n_b_l = np.kron(n_b, self.one_site_number_operator)
            expect = np.einsum("ij, ij", rho, n_b_l)
            n_b_l = O.T @ n_b_l @ O
            self.number_b[system_sites + 1] = n_b_l

        if growing == "env":
            rho = DMRG.constructDensityMatrix(ground_state, system_size=dim_u)
            O, error, entropy, entropy_stats = DMRG.selectDensityVectors(
                rho, states_kept=states_kept, mute=True
            )
            H_b = O.T @ H_u @ O
            S_z_b = O.T @ S_z_u @ O
            S_m_b = O.T @ S_m_u @ O
            S_p_b = O.T @ S_p_u @ O
            self.hamiltonians_u[env_sites + 1] = H_b
            self.sz_s_u[env_sites + 1] = S_z_b
            self.lowerings_u[env_sites + 1] = S_m_b
            self.raisings_u[env_sites + 1] = S_p_b

            n_u = self.number_u[env_sites]
            n_u_r = np.kron(self.one_site_number_operator, n_u)
            expect = np.einsum("ij, ij", rho, n_u_r)
            n_u_r = O.T @ n_u_r @ O
            self.number_u[env_sites + 1] = n_u_r

        return ground_state_energy, entropy, error, expect

    def finite(self, states_kept=2, max_sweep=1):
        """
        Finite size algorithm.
        max_iter, the maximum amount of iterations allowed
        """
        system_sites = self.sites // 2 - 1
        env_sites = self.sites // 2 - 1
        growing = "sys"

        energy_list = []
        entropy_list = []
        error_list = []
        pop_list = []

        # track the system size for convenience
        system_size = []
        env_size = []

        energy, entropy, error, expect = self.step(
            system_sites, env_sites, states_kept=states_kept, growing=growing
        )
        system_size.append(system_sites)
        env_size.append(env_sites)
        energy_list.append(energy)
        entropy_list.append(entropy)
        error_list.append(error)
        pop_list.append(expect)

        system_sites += 1
        env_sites -= 1
        sweep_counter = 0
        # go on till full sweep was done
        while True:
            energy, entropy, error, expect = self.step(
                system_sites, env_sites, states_kept=states_kept, growing=growing
            )
            system_size.append(system_sites)
            env_size.append(env_sites)
            energy_list.append(energy)
            entropy_list.append(entropy)
            error_list.append(error)
            pop_list.append(expect)

            if growing == "sys":
                if env_sites == 1:
                    growing = "env"
                    system_sites -= 1
                    env_sites += 1
                else:
                    system_sites += 1
                    env_sites -= 1

            else:
                if system_sites == 1:
                    growing = "sys"
                    system_sites += 1
                    env_sites -= 1
                else:
                    system_sites -= 1
                    env_sites += 1

            if growing == "sys" and system_sites == self.sites // 2:
                sweep_counter += 1
                if sweep_counter == max_sweep:
                    break

        return energy_list, entropy_list, error_list, system_size, env_size, pop_list
