import numpy as np
import pandas as pd
from . import Hubbard


def AdjMatPartitioner(adj_mat: np.ndarray, site_1: int, site_2: int) -> np.ndarray:
    """
    Will make the relevant part of the adjacency matrix for the bond order operator between two sites

    :param adj_mat: the adjacency matrix of a gqcms.Hubbard object
    :param site_1: the first site
    :param site_2: the second site
    """
    new_mat = np.zeros_like(adj_mat)

    relevant_part = adj_mat[site_1 : site_2 + 1, site_1 : site_2 + 1]
    new_mat[site_1 : site_2 + 1, site_1 : site_2 + 1] = relevant_part

    return new_mat


def BondOrderOperator(piece_of_adj_mat: np.ndarray, molecule: Hubbard) -> np.ndarray:
    """
    Will generate a bond order operator in ONV basis of the Hubbard Hamiltonian.

    :param piece_of_adj_mat: a piece of the adjacency matrix as generated by the AdjMatPartitioner
    :param molecule: Hubbard object
    """
    # We will gradually fill up this matrix with the hopping parameter t
    dimension = len(molecule.basis)
    fillmatrix = np.zeros((dimension, dimension))

    # we will look at every combination of determinants
    for det1 in molecule.basis:
        hopped_dets = det1.single_excitations(molecule.sites)
        for det2 in hopped_dets[:-1]:
            # There is only overlap between determinants that differ in only 1 orbital
            # this is 2 sites in the bitsrtring, a zero will have become a one and a one will have become a zero
            unique1, unique2, sign = det1.get_different_orbitals(det2)
            
            i = molecule.basis_dict[det1]
            j = molecule.basis_dict[det2]

            # we are only looking at singlet excitations excitations happened either in alpha of beta, never in both
            # the sites will always both contain an even or an odd number where even numbers are alphas and odd numbers are betas
            # so unique[0] // 2 will always represent the site number
            fillmatrix[i, j] = (
                -molecule.t * sign * piece_of_adj_mat[unique1[0] // 2, unique2[0] // 2]
            )

    return fillmatrix


def BondOrderDataFrame(molecule: Hubbard, wf: pd.DataFrame) -> pd.DataFrame:
    """
    Will plot the bond orders for the different bonds in a molecule

    :param molecule: a gqcms.Hubbard object
    :param wf: a diagonalized wavefunction, as generated by the gqcms.FCI method.
    """
    bond_order_dict = {}
    for site in range(molecule.sites - 1):
        adjusted_adj_mat = AdjMatPartitioner(molecule.adj_mat, site, site + 1)
        bond_order_op = BondOrderOperator(adjusted_adj_mat, molecule.basis_dict)
        if isinstance(wf["1PDM"], pd.core.series.Series):
            state = wf["1PDM"].to_numpy()[0]
        else:
            state = wf["1PDM"]
            bond_order_dict["population"] = wf["expectation_value"]
        bond_order = np.tensordot(state, bond_order_op, axes=2)
        bond_order_dict[f"{site}-{site+1}"] = [float(bond_order)]

    df = pd.DataFrame.from_dict(bond_order_dict, orient="columns")
    return df
