"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.VpcProvider = exports.AmiProvider = exports.Cluster = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const autoscaling = require("@aws-cdk/aws-autoscaling");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const DEFAULT_INSTANCE_TYPE = ec2.InstanceType.of(ec2.InstanceClass.M6G, ec2.InstanceSize.MEDIUM);
/**
 * Represents the k3sCluster construct.
 *
 * @stability stable
 */
class Cluster extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _d, _e, _f, _g;
        super(scope, id);
        // VPC configuration
        const vpc = (_d = props.vpc) !== null && _d !== void 0 ? _d : new ec2.Vpc(this, 'Vpc', { maxAzs: 3, natGateways: 1 });
        // S3 bucket to host K3s token + kubeconfig file
        // support s3 bucket autoDeleteObjects native.
        // see - https://docs.aws.amazon.com/cdk/api/latest/docs/aws-s3-readme.html#bucket-deletion
        // PR - https://github.com/aws/aws-cdk/commit/32e9c23be2852cfca79a57c90e52b9301b1c7081
        let k3sBucket;
        if (props.bucketRemovalPolicy === cdk.RemovalPolicy.DESTROY) {
            k3sBucket = new s3.Bucket(this, 'k3sBucket', {
                removalPolicy: props.bucketRemovalPolicy,
                autoDeleteObjects: true,
            });
        }
        else {
            k3sBucket = new s3.Bucket(this, 'k3sBucket', {
                removalPolicy: cdk.RemovalPolicy.RETAIN,
            });
        }
        // control plane node Security Group
        const k3scontrolplanesg = new ec2.SecurityGroup(this, 'k3s-controlplane-SG', { vpc });
        k3scontrolplanesg.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(22), 'SSH');
        k3scontrolplanesg.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(6443), 'K3s port');
        // worker nodes Security Group
        const k3sworkersg = new ec2.SecurityGroup(this, 'k3s-worker-SG', { vpc });
        // for this prototype the workers are being placed in a public subnet
        // ideally they should land on a private subnet
        /// also ingress traffic - ssh (bastion style) or 6443 - should come from the control plane node only
        k3sworkersg.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(22), 'SSH');
        k3sworkersg.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(6443), 'K3s port');
        // check if the user requires a particular instance type for workers and control plane
        // if not, the default instance type is used
        this.controlPlaneInstanceType = (_e = props.controlPlaneInstanceType) !== null && _e !== void 0 ? _e : DEFAULT_INSTANCE_TYPE;
        this.workerInstanceType = (_f = props.workerInstanceType) !== null && _f !== void 0 ? _f : DEFAULT_INSTANCE_TYPE;
        // create control plane node
        const k3scontrolplane = new ec2.Instance(this, 'k3s-controlplane', {
            instanceType: this.controlPlaneInstanceType,
            machineImage: new AmiProvider().amiId,
            vpc,
            vpcSubnets: {
                subnets: vpc.publicSubnets,
            },
            instanceName: 'k3s-controlplane',
            securityGroup: k3scontrolplanesg,
        });
        k3scontrolplane.addUserData(`
       #!/bin/bash
       curl -L -o k3s https://github.com/rancher/k3s/releases/download/v1.16.9%2Bk3s1/k3s-arm64
       chmod +x k3s
       ./k3s server &
       sleep 30
       ENDPOINT=$(curl http://169.254.169.254/latest/meta-data/public-hostname) 
       cp /etc/rancher/k3s/k3s.yaml /etc/rancher/k3s/kubeconfig.yaml
       sed -i s/127.0.0.1/$ENDPOINT/ /etc/rancher/k3s/kubeconfig.yaml
       aws s3 cp /var/lib/rancher/k3s/server/node-token s3://${k3sBucket.bucketName}/node-token
       aws s3 cp /etc/rancher/k3s/kubeconfig.yaml s3://${k3sBucket.bucketName}/kubeconfig.yaml
     `);
        this.endpointUri = k3scontrolplane.instancePublicIp;
        // create launch template for worker ASG
        // prepare the userData
        const userData = ec2.UserData.forLinux();
        userData.addCommands(`
          #!/bin/bash
          LOGFILE='/var/log/k3s.log'
          curl -L -o k3s https://github.com/rancher/k3s/releases/download/v1.16.13%2Bk3s1/k3s-arm64
          chmod +x k3s
          echo the bucket name is ${k3sBucket.bucketName} 
          aws s3 cp s3://${k3sBucket.bucketName}/node-token /node-token 
          (./k3s agent --server https://${k3scontrolplane.instancePrivateIp}:6443 \
          --token $(cat /node-token) 2>&1 | tee -a $LOGFILE || echo "failed" > $LOGFILE &)
    `);
        // create worker ASG
        const workerAsg = new autoscaling.AutoScalingGroup(this, 'WorkerAsg', {
            instanceType: this.workerInstanceType,
            machineImage: new AmiProvider().amiId,
            vpc,
            vpcSubnets: {
                subnetType: ec2.SubnetType.PUBLIC,
            },
            minCapacity: (_g = props.workerMinCapacity) !== null && _g !== void 0 ? _g : 3,
        });
        const cfnInstanceProfile = workerAsg.node.tryFindChild('InstanceProfile');
        const lt = new ec2.CfnLaunchTemplate(this, 'WorkerLaunchTemplate', {
            launchTemplateData: {
                imageId: new AmiProvider().amiId.getImage(this).imageId,
                instanceType: this.workerInstanceType.toString(),
                instanceMarketOptions: {
                    marketType: props.spotWorkerNodes ? 'spot' : undefined,
                    spotOptions: props.spotWorkerNodes ? {
                        spotInstanceType: 'one-time',
                    } : undefined,
                },
                userData: cdk.Fn.base64(userData.render()),
                iamInstanceProfile: {
                    arn: cfnInstanceProfile.attrArn,
                },
            },
        });
        const cfnAsg = workerAsg.node.tryFindChild('ASG');
        cfnAsg.addPropertyDeletionOverride('LaunchConfigurationName');
        cfnAsg.addPropertyOverride('LaunchTemplate', {
            LaunchTemplateId: lt.ref,
            Version: lt.attrLatestVersionNumber,
        });
        workerAsg.addSecurityGroup(k3sworkersg);
        // enable the SSM session manager
        workerAsg.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
        // grant the S3 write permission to the control plane node and read permissions to the worker nodes
        k3sBucket.grantWrite(k3scontrolplane.role);
        k3sBucket.grantRead(workerAsg.role);
        // endpoint info
        new cdk.CfnOutput(this, 'Endpoint', { value: `https://${k3scontrolplane.instancePublicIp}:6443` });
        // kubeconfig.yaml path
        new cdk.CfnOutput(this, 'Kubernetes configuration file', { value: `s3://${k3sBucket.bucketName}/kubeconfig.yaml` });
        workerAsg.node.addDependency(k3scontrolplane);
    }
}
exports.Cluster = Cluster;
_a = JSII_RTTI_SYMBOL_1;
Cluster[_a] = { fqn: "cdk-k3s-cluster.Cluster", version: "0.0.51" };
/**
 * The AMI provider to get the latest Amazon Linux 2 AMI for ARM64.
 *
 * @stability stable
 */
class AmiProvider {
    /**
     * @stability stable
     */
    get amiId() {
        return ec2.MachineImage.latestAmazonLinux({
            cpuType: ec2.AmazonLinuxCpuType.ARM_64,
            generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2,
        });
    }
}
exports.AmiProvider = AmiProvider;
_b = JSII_RTTI_SYMBOL_1;
AmiProvider[_b] = { fqn: "cdk-k3s-cluster.AmiProvider", version: "0.0.51" };
/**
 * The VPC provider to create or import the VPC.
 *
 * @stability stable
 */
class VpcProvider {
    /**
     * @stability stable
     */
    static getOrCreate(scope) {
        const vpc = scope.node.tryGetContext('use_default_vpc') === '1' ?
            ec2.Vpc.fromLookup(scope, 'Vpc', { isDefault: true }) :
            scope.node.tryGetContext('use_vpc_id') ?
                ec2.Vpc.fromLookup(scope, 'Vpc', { vpcId: scope.node.tryGetContext('use_vpc_id') }) :
                new ec2.Vpc(scope, 'Vpc', { maxAzs: 3, natGateways: 1 });
        return vpc;
    }
}
exports.VpcProvider = VpcProvider;
_c = JSII_RTTI_SYMBOL_1;
VpcProvider[_c] = { fqn: "cdk-k3s-cluster.VpcProvider", version: "0.0.51" };
//# sourceMappingURL=data:application/json;base64,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