"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const api = require("@aws-cdk/aws-apigateway");
// --------------------------------------------------------------
// Test minimal deployment
// --------------------------------------------------------------
test('Test minimal deployment', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {});
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
// --------------------------------------------------------------
// Test deployment w/ DLQ
// --------------------------------------------------------------
test('Test deployment w/ DLQ', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        allowCreateOperation: true,
        allowReadOperation: true,
        allowDeleteOperation: true,
        deployDeadLetterQueue: true
    });
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
// --------------------------------------------------------------
// Test deployment w/o DLQ
// --------------------------------------------------------------
test('Test deployment w/o DLQ', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        deployDeadLetterQueue: false
    });
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    // Assertion 2
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        HttpMethod: "GET",
        AuthorizationType: "AWS_IAM"
    });
});
// --------------------------------------------------------------
// Test deployment w/o allowReadOperation
// --------------------------------------------------------------
test('Test deployment w/o allowReadOperation', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        allowCreateOperation: true,
        allowReadOperation: false,
    });
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    // Assertion 2
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        HttpMethod: "POST",
        AuthorizationType: "AWS_IAM"
    });
});
// --------------------------------------------------------------
// Test deployment w/ allowReadOperation
// --------------------------------------------------------------
test('Test deployment w/ allowReadOperation', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        allowReadOperation: true,
    });
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    // Assertion 2
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        HttpMethod: "GET",
        AuthorizationType: "AWS_IAM"
    });
});
// --------------------------------------------------------------
// Test the getter methods
// --------------------------------------------------------------
test('Test properties', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    const pattern = new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        deployDeadLetterQueue: true,
        maxReceiveCount: 3
    });
    // Assertion 1
    expect(pattern.apiGateway !== null);
    // Assertion 2
    expect(pattern.sqsQueue !== null);
    // Assertion 3
    expect(pattern.apiGatewayRole !== null);
    expect(pattern.apiGatewayCloudWatchRole !== null);
    expect(pattern.apiGatewayLogGroup !== null);
    expect(pattern.deadLetterQueue !== null);
});
// -----------------------------------------------------------------
// Test deployment for override ApiGateway AuthorizationType to NONE
// -----------------------------------------------------------------
test('Test deployment ApiGateway AuthorizationType override', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        apiGatewayProps: {
            defaultMethodOptions: {
                authorizationType: api.AuthorizationType.NONE
            }
        },
        queueProps: {},
        createRequestTemplate: "{}",
        allowCreateOperation: true,
        allowReadOperation: false,
        allowDeleteOperation: true,
        deployDeadLetterQueue: false
    });
    // Assertion 1
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        HttpMethod: "POST",
        AuthorizationType: "NONE"
    });
    // Assertion 2
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        HttpMethod: "DELETE",
        AuthorizationType: "NONE"
    });
});
// -----------------------------------------------------------------
// Test deployment for override ApiGateway createRequestTemplate
// -----------------------------------------------------------------
test('Test deployment for override ApiGateway createRequestTemplate', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        createRequestTemplate: "Action=SendMessage&MessageBody=$util.urlEncode(\"HelloWorld\")",
        allowCreateOperation: true
    });
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        HttpMethod: "POST",
        Integration: {
            RequestTemplates: {
                "application/json": "Action=SendMessage&MessageBody=$util.urlEncode(\"HelloWorld\")"
            }
        }
    });
});
// -----------------------------------------------------------------
// Test deployment for override ApiGateway getRequestTemplate
// -----------------------------------------------------------------
test('Test deployment for override ApiGateway getRequestTemplate', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        readRequestTemplate: "Action=HelloWorld",
        allowReadOperation: true
    });
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        HttpMethod: "GET",
        Integration: {
            RequestTemplates: {
                "application/json": "Action=HelloWorld"
            }
        }
    });
});
// -----------------------------------------------------------------
// Test deployment for override ApiGateway deleteRequestTemplate
// -----------------------------------------------------------------
test('Test deployment for override ApiGateway deleteRequestTemplate', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    new lib_1.ApiGatewayToSqs(stack, 'api-gateway-sqs', {
        deleteRequestTemplate: "Action=HelloWorld",
        allowDeleteOperation: true
    });
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        HttpMethod: "DELETE",
        Integration: {
            RequestTemplates: {
                "application/json": "Action=HelloWorld"
            }
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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