"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Table = exports.TableEncryption = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const cr = require("@aws-cdk/custom-resources");
const glue_generated_1 = require("./glue.generated");
/**
 * Encryption options for a Table.
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/encryption.html
 */
var TableEncryption;
(function (TableEncryption) {
    TableEncryption["UNENCRYPTED"] = "Unencrypted";
    /**
     * Server side encryption (SSE) with an Amazon S3-managed key.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingServerSideEncryption.html
     */
    TableEncryption["S3_MANAGED"] = "SSE-S3";
    /**
     * Server-side encryption (SSE) with an AWS KMS key managed by the account owner.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingKMSEncryption.html
     */
    TableEncryption["KMS"] = "SSE-KMS";
    /**
     * Server-side encryption (SSE) with an AWS KMS key managed by the KMS service.
     */
    TableEncryption["KMS_MANAGED"] = "SSE-KMS-MANAGED";
    /**
     * Client-side encryption (CSE) with an AWS KMS key managed by the account owner.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingClientSideEncryption.html
     */
    TableEncryption["CLIENT_SIDE_KMS"] = "CSE-KMS";
})(TableEncryption = exports.TableEncryption || (exports.TableEncryption = {}));
/**
 * A Glue table.
 */
class Table extends core_1.Resource {
    constructor(scope, id, props) {
        var _b, _c, _d, _e;
        super(scope, id, {
            physicalName: props.tableName,
        });
        /**
         * Partition indexes must be created one at a time. To avoid
         * race conditions, we store the resource and add dependencies
         * each time a new partition index is created.
         */
        this.partitionIndexCustomResources = [];
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_TableProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        this.database = props.database;
        this.dataFormat = props.dataFormat;
        this.s3Prefix = (_b = props.s3Prefix) !== null && _b !== void 0 ? _b : '';
        validateSchema(props.columns, props.partitionKeys);
        this.columns = props.columns;
        this.partitionKeys = props.partitionKeys;
        this.compressed = (_c = props.compressed) !== null && _c !== void 0 ? _c : false;
        const { bucket, encryption, encryptionKey } = createBucket(this, props);
        this.bucket = bucket;
        this.encryption = encryption;
        this.encryptionKey = encryptionKey;
        const tableResource = new glue_generated_1.CfnTable(this, 'Table', {
            catalogId: props.database.catalogId,
            databaseName: props.database.databaseName,
            tableInput: {
                name: this.physicalName,
                description: props.description || `${props.tableName} generated by CDK`,
                partitionKeys: renderColumns(props.partitionKeys),
                parameters: {
                    classification: (_d = props.dataFormat.classificationString) === null || _d === void 0 ? void 0 : _d.value,
                    has_encrypted_data: this.encryption !== TableEncryption.UNENCRYPTED,
                },
                storageDescriptor: {
                    location: `s3://${this.bucket.bucketName}/${this.s3Prefix}`,
                    compressed: this.compressed,
                    storedAsSubDirectories: (_e = props.storedAsSubDirectories) !== null && _e !== void 0 ? _e : false,
                    columns: renderColumns(props.columns),
                    inputFormat: props.dataFormat.inputFormat.className,
                    outputFormat: props.dataFormat.outputFormat.className,
                    serdeInfo: {
                        serializationLibrary: props.dataFormat.serializationLibrary.className,
                    },
                },
                tableType: 'EXTERNAL_TABLE',
            },
        });
        this.tableName = this.getResourceNameAttribute(tableResource.ref);
        this.tableArn = this.stack.formatArn({
            service: 'glue',
            resource: 'table',
            resourceName: `${this.database.databaseName}/${this.tableName}`,
        });
        this.node.defaultChild = tableResource;
        // Partition index creation relies on created table.
        if (props.partitionIndexes) {
            this.partitionIndexes = props.partitionIndexes;
            this.partitionIndexes.forEach((index) => this.addPartitionIndex(index));
        }
    }
    static fromTableArn(scope, id, tableArn) {
        const tableName = core_1.Fn.select(1, core_1.Fn.split('/', core_1.Stack.of(scope).splitArn(tableArn, core_1.ArnFormat.SLASH_RESOURCE_NAME).resourceName));
        return Table.fromTableAttributes(scope, id, {
            tableArn,
            tableName,
        });
    }
    /**
     * Creates a Table construct that represents an external table.
     *
     * @param scope The scope creating construct (usually `this`).
     * @param id The construct's id.
     * @param attrs Import attributes
     */
    static fromTableAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_TableAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromTableAttributes);
            }
            throw error;
        }
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.tableArn = attrs.tableArn;
                this.tableName = attrs.tableName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Add a partition index to the table. You can have a maximum of 3 partition
     * indexes to a table. Partition index keys must be a subset of the table's
     * partition keys.
     *
     * @see https://docs.aws.amazon.com/glue/latest/dg/partition-indexes.html
     */
    addPartitionIndex(index) {
        var _b;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_PartitionIndex(index);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addPartitionIndex);
            }
            throw error;
        }
        const numPartitions = this.partitionIndexCustomResources.length;
        if (numPartitions >= 3) {
            throw new Error('Maximum number of partition indexes allowed is 3');
        }
        this.validatePartitionIndex(index);
        const indexName = (_b = index.indexName) !== null && _b !== void 0 ? _b : this.generateIndexName(index.keyNames);
        const partitionIndexCustomResource = new cr.AwsCustomResource(this, `partition-index-${indexName}`, {
            onCreate: {
                service: 'Glue',
                action: 'createPartitionIndex',
                parameters: {
                    DatabaseName: this.database.databaseName,
                    TableName: this.tableName,
                    PartitionIndex: {
                        IndexName: indexName,
                        Keys: index.keyNames,
                    },
                },
                physicalResourceId: cr.PhysicalResourceId.of(indexName),
            },
            policy: cr.AwsCustomResourcePolicy.fromSdkCalls({
                resources: cr.AwsCustomResourcePolicy.ANY_RESOURCE,
            }),
        });
        this.grantToUnderlyingResources(partitionIndexCustomResource, ['glue:UpdateTable']);
        // Depend on previous partition index if possible, to avoid race condition
        if (numPartitions > 0) {
            this.partitionIndexCustomResources[numPartitions - 1].node.addDependency(partitionIndexCustomResource);
        }
        this.partitionIndexCustomResources.push(partitionIndexCustomResource);
    }
    generateIndexName(keys) {
        const prefix = keys.join('-') + '-';
        const uniqueId = core_1.Names.uniqueId(this);
        const maxIndexLength = 80; // arbitrarily specified
        const startIndex = Math.max(0, uniqueId.length - (maxIndexLength - prefix.length));
        return prefix + uniqueId.substring(startIndex);
    }
    validatePartitionIndex(index) {
        if (index.indexName !== undefined && (index.indexName.length < 1 || index.indexName.length > 255)) {
            throw new Error(`Index name must be between 1 and 255 characters, but got ${index.indexName.length}`);
        }
        if (!this.partitionKeys || this.partitionKeys.length === 0) {
            throw new Error('The table must have partition keys to create a partition index');
        }
        const keyNames = this.partitionKeys.map(pk => pk.name);
        if (!index.keyNames.every(k => keyNames.includes(k))) {
            throw new Error(`All index keys must also be partition keys. Got ${index.keyNames} but partition key names are ${keyNames}`);
        }
    }
    /**
     * Grant read permissions to the table and the underlying data stored in S3 to an IAM principal.
     *
     * @param grantee the principal
     */
    grantRead(grantee) {
        const ret = this.grant(grantee, readPermissions);
        if (this.encryptionKey && this.encryption === TableEncryption.CLIENT_SIDE_KMS) {
            this.encryptionKey.grantDecrypt(grantee);
        }
        this.bucket.grantRead(grantee, this.getS3PrefixForGrant());
        return ret;
    }
    /**
     * Grant write permissions to the table and the underlying data stored in S3 to an IAM principal.
     *
     * @param grantee the principal
     */
    grantWrite(grantee) {
        const ret = this.grant(grantee, writePermissions);
        if (this.encryptionKey && this.encryption === TableEncryption.CLIENT_SIDE_KMS) {
            this.encryptionKey.grantEncrypt(grantee);
        }
        this.bucket.grantWrite(grantee, this.getS3PrefixForGrant());
        return ret;
    }
    /**
     * Grant read and write permissions to the table and the underlying data stored in S3 to an IAM principal.
     *
     * @param grantee the principal
     */
    grantReadWrite(grantee) {
        const ret = this.grant(grantee, [...readPermissions, ...writePermissions]);
        if (this.encryptionKey && this.encryption === TableEncryption.CLIENT_SIDE_KMS) {
            this.encryptionKey.grantEncryptDecrypt(grantee);
        }
        this.bucket.grantReadWrite(grantee, this.getS3PrefixForGrant());
        return ret;
    }
    /**
     * Grant the given identity custom permissions.
     */
    grant(grantee, actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            resourceArns: [this.tableArn],
            actions,
        });
    }
    /**
     * Grant the given identity custom permissions to ALL underlying resources of the table.
     * Permissions will be granted to the catalog, the database, and the table.
     */
    grantToUnderlyingResources(grantee, actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            resourceArns: [
                this.tableArn,
                this.database.catalogArn,
                this.database.databaseArn,
            ],
            actions,
        });
    }
    getS3PrefixForGrant() {
        return this.s3Prefix + '*';
    }
}
exports.Table = Table;
_a = JSII_RTTI_SYMBOL_1;
Table[_a] = { fqn: "@aws-cdk/aws-glue.Table", version: "1.156.1" };
function validateSchema(columns, partitionKeys) {
    if (columns.length === 0) {
        throw new Error('you must specify at least one column for the table');
    }
    // Check there is at least one column and no duplicated column names or partition keys.
    const names = new Set();
    (columns.concat(partitionKeys || [])).forEach(column => {
        if (names.has(column.name)) {
            throw new Error(`column names and partition keys must be unique, but \'${column.name}\' is duplicated`);
        }
        names.add(column.name);
    });
}
// map TableEncryption to bucket's SSE configuration (s3.BucketEncryption)
const encryptionMappings = {
    [TableEncryption.S3_MANAGED]: s3.BucketEncryption.S3_MANAGED,
    [TableEncryption.KMS_MANAGED]: s3.BucketEncryption.KMS_MANAGED,
    [TableEncryption.KMS]: s3.BucketEncryption.KMS,
    [TableEncryption.CLIENT_SIDE_KMS]: s3.BucketEncryption.UNENCRYPTED,
    [TableEncryption.UNENCRYPTED]: s3.BucketEncryption.UNENCRYPTED,
};
// create the bucket to store a table's data depending on the `encryption` and `encryptionKey` properties.
function createBucket(table, props) {
    const encryption = props.encryption || TableEncryption.UNENCRYPTED;
    let bucket = props.bucket;
    if (bucket && (encryption !== TableEncryption.UNENCRYPTED && encryption !== TableEncryption.CLIENT_SIDE_KMS)) {
        throw new Error('you can not specify encryption settings if you also provide a bucket');
    }
    let encryptionKey;
    if (encryption === TableEncryption.CLIENT_SIDE_KMS && props.encryptionKey === undefined) {
        // CSE-KMS should behave the same as SSE-KMS - use the provided key or create one automatically
        // Since Bucket only knows about SSE, we repeat the logic for CSE-KMS at the Table level.
        encryptionKey = new kms.Key(table, 'Key');
    }
    else {
        encryptionKey = props.encryptionKey;
    }
    // create the bucket if none was provided
    if (!bucket) {
        if (encryption === TableEncryption.CLIENT_SIDE_KMS) {
            bucket = new s3.Bucket(table, 'Bucket');
        }
        else {
            bucket = new s3.Bucket(table, 'Bucket', {
                encryption: encryptionMappings[encryption],
                encryptionKey,
            });
            encryptionKey = bucket.encryptionKey;
        }
    }
    return {
        bucket,
        encryption,
        encryptionKey,
    };
}
const readPermissions = [
    'glue:BatchGetPartition',
    'glue:GetPartition',
    'glue:GetPartitions',
    'glue:GetTable',
    'glue:GetTables',
    'glue:GetTableVersion',
    'glue:GetTableVersions',
];
const writePermissions = [
    'glue:BatchCreatePartition',
    'glue:BatchDeletePartition',
    'glue:CreatePartition',
    'glue:DeletePartition',
    'glue:UpdatePartition',
];
function renderColumns(columns) {
    if (columns === undefined) {
        return undefined;
    }
    return columns.map(column => {
        return {
            name: column.name,
            type: column.type.inputString,
            comment: column.comment,
        };
    });
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGFibGUuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ0YWJsZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7Ozs7QUFBQSx3Q0FBd0M7QUFDeEMsd0NBQXdDO0FBQ3hDLHNDQUFzQztBQUN0Qyx3Q0FBaUY7QUFDakYsZ0RBQWdEO0FBS2hELHFEQUE0QztBQWlDNUM7Ozs7R0FJRztBQUNILElBQVksZUE0Qlg7QUE1QkQsV0FBWSxlQUFlO0lBQ3pCLDhDQUEyQixDQUFBO0lBRTNCOzs7O09BSUc7SUFDSCx3Q0FBcUIsQ0FBQTtJQUVyQjs7OztPQUlHO0lBQ0gsa0NBQWUsQ0FBQTtJQUVmOztPQUVHO0lBQ0gsa0RBQStCLENBQUE7SUFFL0I7Ozs7T0FJRztJQUNILDhDQUEyQixDQUFBO0FBQzdCLENBQUMsRUE1QlcsZUFBZSxHQUFmLHVCQUFlLEtBQWYsdUJBQWUsUUE0QjFCO0FBcUdEOztHQUVHO0FBQ0gsTUFBYSxLQUFNLFNBQVEsZUFBUTtJQThGakMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFpQjs7UUFDekQsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDZixZQUFZLEVBQUUsS0FBSyxDQUFDLFNBQVM7U0FDOUIsQ0FBQyxDQUFDO1FBVkw7Ozs7V0FJRztRQUNLLGtDQUE2QixHQUF3QixFQUFFLENBQUM7Ozs7Ozs7Ozs7UUFPOUQsSUFBSSxDQUFDLFFBQVEsR0FBRyxLQUFLLENBQUMsUUFBUSxDQUFDO1FBQy9CLElBQUksQ0FBQyxVQUFVLEdBQUcsS0FBSyxDQUFDLFVBQVUsQ0FBQztRQUNuQyxJQUFJLENBQUMsUUFBUSxTQUFHLEtBQUssQ0FBQyxRQUFRLG1DQUFJLEVBQUUsQ0FBQztRQUVyQyxjQUFjLENBQUMsS0FBSyxDQUFDLE9BQU8sRUFBRSxLQUFLLENBQUMsYUFBYSxDQUFDLENBQUM7UUFDbkQsSUFBSSxDQUFDLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDO1FBQzdCLElBQUksQ0FBQyxhQUFhLEdBQUcsS0FBSyxDQUFDLGFBQWEsQ0FBQztRQUV6QyxJQUFJLENBQUMsVUFBVSxTQUFHLEtBQUssQ0FBQyxVQUFVLG1DQUFJLEtBQUssQ0FBQztRQUM1QyxNQUFNLEVBQUUsTUFBTSxFQUFFLFVBQVUsRUFBRSxhQUFhLEVBQUUsR0FBRyxZQUFZLENBQUMsSUFBSSxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBQ3hFLElBQUksQ0FBQyxNQUFNLEdBQUcsTUFBTSxDQUFDO1FBQ3JCLElBQUksQ0FBQyxVQUFVLEdBQUcsVUFBVSxDQUFDO1FBQzdCLElBQUksQ0FBQyxhQUFhLEdBQUcsYUFBYSxDQUFDO1FBRW5DLE1BQU0sYUFBYSxHQUFHLElBQUkseUJBQVEsQ0FBQyxJQUFJLEVBQUUsT0FBTyxFQUFFO1lBQ2hELFNBQVMsRUFBRSxLQUFLLENBQUMsUUFBUSxDQUFDLFNBQVM7WUFFbkMsWUFBWSxFQUFFLEtBQUssQ0FBQyxRQUFRLENBQUMsWUFBWTtZQUV6QyxVQUFVLEVBQUU7Z0JBQ1YsSUFBSSxFQUFFLElBQUksQ0FBQyxZQUFZO2dCQUN2QixXQUFXLEVBQUUsS0FBSyxDQUFDLFdBQVcsSUFBSSxHQUFHLEtBQUssQ0FBQyxTQUFTLG1CQUFtQjtnQkFFdkUsYUFBYSxFQUFFLGFBQWEsQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDO2dCQUVqRCxVQUFVLEVBQUU7b0JBQ1YsY0FBYyxRQUFFLEtBQUssQ0FBQyxVQUFVLENBQUMsb0JBQW9CLDBDQUFFLEtBQUs7b0JBQzVELGtCQUFrQixFQUFFLElBQUksQ0FBQyxVQUFVLEtBQUssZUFBZSxDQUFDLFdBQVc7aUJBQ3BFO2dCQUNELGlCQUFpQixFQUFFO29CQUNqQixRQUFRLEVBQUUsUUFBUSxJQUFJLENBQUMsTUFBTSxDQUFDLFVBQVUsSUFBSSxJQUFJLENBQUMsUUFBUSxFQUFFO29CQUMzRCxVQUFVLEVBQUUsSUFBSSxDQUFDLFVBQVU7b0JBQzNCLHNCQUFzQixRQUFFLEtBQUssQ0FBQyxzQkFBc0IsbUNBQUksS0FBSztvQkFDN0QsT0FBTyxFQUFFLGFBQWEsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDO29CQUNyQyxXQUFXLEVBQUUsS0FBSyxDQUFDLFVBQVUsQ0FBQyxXQUFXLENBQUMsU0FBUztvQkFDbkQsWUFBWSxFQUFFLEtBQUssQ0FBQyxVQUFVLENBQUMsWUFBWSxDQUFDLFNBQVM7b0JBQ3JELFNBQVMsRUFBRTt3QkFDVCxvQkFBb0IsRUFBRSxLQUFLLENBQUMsVUFBVSxDQUFDLG9CQUFvQixDQUFDLFNBQVM7cUJBQ3RFO2lCQUNGO2dCQUVELFNBQVMsRUFBRSxnQkFBZ0I7YUFDNUI7U0FDRixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsU0FBUyxHQUFHLElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxhQUFhLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDbEUsSUFBSSxDQUFDLFFBQVEsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQztZQUNuQyxPQUFPLEVBQUUsTUFBTTtZQUNmLFFBQVEsRUFBRSxPQUFPO1lBQ2pCLFlBQVksRUFBRSxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsWUFBWSxJQUFJLElBQUksQ0FBQyxTQUFTLEVBQUU7U0FDaEUsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLElBQUksQ0FBQyxZQUFZLEdBQUcsYUFBYSxDQUFDO1FBRXZDLG9EQUFvRDtRQUNwRCxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsRUFBRTtZQUMxQixJQUFJLENBQUMsZ0JBQWdCLEdBQUcsS0FBSyxDQUFDLGdCQUFnQixDQUFDO1lBQy9DLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxPQUFPLENBQUMsQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO1NBQ3pFO0tBQ0Y7SUEzSk0sTUFBTSxDQUFDLFlBQVksQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxRQUFnQjtRQUN2RSxNQUFNLFNBQVMsR0FBRyxTQUFFLENBQUMsTUFBTSxDQUFDLENBQUMsRUFBRSxTQUFFLENBQUMsS0FBSyxDQUFDLEdBQUcsRUFBRSxZQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDLFFBQVEsQ0FBQyxRQUFRLEVBQUUsZ0JBQVMsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDLFlBQWEsQ0FBQyxDQUFDLENBQUM7UUFFL0gsT0FBTyxLQUFLLENBQUMsbUJBQW1CLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRTtZQUMxQyxRQUFRO1lBQ1IsU0FBUztTQUNWLENBQUMsQ0FBQztLQUNKO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksTUFBTSxDQUFDLG1CQUFtQixDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXNCOzs7Ozs7Ozs7O1FBQ3BGLE1BQU0sTUFBTyxTQUFRLGVBQVE7WUFBN0I7O2dCQUNrQixhQUFRLEdBQUcsS0FBSyxDQUFDLFFBQVEsQ0FBQztnQkFDMUIsY0FBUyxHQUFHLEtBQUssQ0FBQyxTQUFTLENBQUM7WUFDOUMsQ0FBQztTQUFBO1FBRUQsT0FBTyxJQUFJLE1BQU0sQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7S0FDOUI7SUFzSUQ7Ozs7OztPQU1HO0lBQ0ksaUJBQWlCLENBQUMsS0FBcUI7Ozs7Ozs7Ozs7O1FBQzVDLE1BQU0sYUFBYSxHQUFHLElBQUksQ0FBQyw2QkFBNkIsQ0FBQyxNQUFNLENBQUM7UUFDaEUsSUFBSSxhQUFhLElBQUksQ0FBQyxFQUFFO1lBQ3RCLE1BQU0sSUFBSSxLQUFLLENBQUMsa0RBQWtELENBQUMsQ0FBQztTQUNyRTtRQUNELElBQUksQ0FBQyxzQkFBc0IsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUVuQyxNQUFNLFNBQVMsU0FBRyxLQUFLLENBQUMsU0FBUyxtQ0FBSSxJQUFJLENBQUMsaUJBQWlCLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQzVFLE1BQU0sNEJBQTRCLEdBQUcsSUFBSSxFQUFFLENBQUMsaUJBQWlCLENBQUMsSUFBSSxFQUFFLG1CQUFtQixTQUFTLEVBQUUsRUFBRTtZQUNsRyxRQUFRLEVBQUU7Z0JBQ1IsT0FBTyxFQUFFLE1BQU07Z0JBQ2YsTUFBTSxFQUFFLHNCQUFzQjtnQkFDOUIsVUFBVSxFQUFFO29CQUNWLFlBQVksRUFBRSxJQUFJLENBQUMsUUFBUSxDQUFDLFlBQVk7b0JBQ3hDLFNBQVMsRUFBRSxJQUFJLENBQUMsU0FBUztvQkFDekIsY0FBYyxFQUFFO3dCQUNkLFNBQVMsRUFBRSxTQUFTO3dCQUNwQixJQUFJLEVBQUUsS0FBSyxDQUFDLFFBQVE7cUJBQ3JCO2lCQUNGO2dCQUNELGtCQUFrQixFQUFFLEVBQUUsQ0FBQyxrQkFBa0IsQ0FBQyxFQUFFLENBQzFDLFNBQVMsQ0FDVjthQUNGO1lBQ0QsTUFBTSxFQUFFLEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQyxZQUFZLENBQUM7Z0JBQzlDLFNBQVMsRUFBRSxFQUFFLENBQUMsdUJBQXVCLENBQUMsWUFBWTthQUNuRCxDQUFDO1NBQ0gsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLDBCQUEwQixDQUFDLDRCQUE0QixFQUFFLENBQUMsa0JBQWtCLENBQUMsQ0FBQyxDQUFDO1FBRXBGLDBFQUEwRTtRQUMxRSxJQUFJLGFBQWEsR0FBRyxDQUFDLEVBQUU7WUFDckIsSUFBSSxDQUFDLDZCQUE2QixDQUFDLGFBQWEsR0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLDRCQUE0QixDQUFDLENBQUM7U0FDdEc7UUFDRCxJQUFJLENBQUMsNkJBQTZCLENBQUMsSUFBSSxDQUFDLDRCQUE0QixDQUFDLENBQUM7S0FDdkU7SUFFTyxpQkFBaUIsQ0FBQyxJQUFjO1FBQ3RDLE1BQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLEdBQUcsR0FBRyxDQUFDO1FBQ3BDLE1BQU0sUUFBUSxHQUFHLFlBQUssQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDdEMsTUFBTSxjQUFjLEdBQUcsRUFBRSxDQUFDLENBQUMsd0JBQXdCO1FBQ25ELE1BQU0sVUFBVSxHQUFHLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQyxFQUFFLFFBQVEsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxjQUFjLEdBQUcsTUFBTSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUM7UUFDbkYsT0FBTyxNQUFNLEdBQUcsUUFBUSxDQUFDLFNBQVMsQ0FBQyxVQUFVLENBQUMsQ0FBQztLQUNoRDtJQUVPLHNCQUFzQixDQUFDLEtBQXFCO1FBQ2xELElBQUksS0FBSyxDQUFDLFNBQVMsS0FBSyxTQUFTLElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUyxDQUFDLE1BQU0sR0FBRyxDQUFDLElBQUksS0FBSyxDQUFDLFNBQVMsQ0FBQyxNQUFNLEdBQUcsR0FBRyxDQUFDLEVBQUU7WUFDakcsTUFBTSxJQUFJLEtBQUssQ0FBQyw0REFBNEQsS0FBSyxDQUFDLFNBQVMsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDO1NBQ3ZHO1FBQ0QsSUFBSSxDQUFDLElBQUksQ0FBQyxhQUFhLElBQUksSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQzFELE1BQU0sSUFBSSxLQUFLLENBQUMsZ0VBQWdFLENBQUMsQ0FBQztTQUNuRjtRQUNELE1BQU0sUUFBUSxHQUFHLElBQUksQ0FBQyxhQUFhLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ3ZELElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRTtZQUNwRCxNQUFNLElBQUksS0FBSyxDQUFDLG1EQUFtRCxLQUFLLENBQUMsUUFBUSxnQ0FBZ0MsUUFBUSxFQUFFLENBQUMsQ0FBQztTQUM5SDtLQUNGO0lBRUQ7Ozs7T0FJRztJQUNJLFNBQVMsQ0FBQyxPQUF1QjtRQUN0QyxNQUFNLEdBQUcsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sRUFBRSxlQUFlLENBQUMsQ0FBQztRQUNqRCxJQUFJLElBQUksQ0FBQyxhQUFhLElBQUksSUFBSSxDQUFDLFVBQVUsS0FBSyxlQUFlLENBQUMsZUFBZSxFQUFFO1lBQUUsSUFBSSxDQUFDLGFBQWEsQ0FBQyxZQUFZLENBQUMsT0FBTyxDQUFDLENBQUM7U0FBRTtRQUM1SCxJQUFJLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxPQUFPLEVBQUUsSUFBSSxDQUFDLG1CQUFtQixFQUFFLENBQUMsQ0FBQztRQUMzRCxPQUFPLEdBQUcsQ0FBQztLQUNaO0lBRUQ7Ozs7T0FJRztJQUNJLFVBQVUsQ0FBQyxPQUF1QjtRQUN2QyxNQUFNLEdBQUcsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sRUFBRSxnQkFBZ0IsQ0FBQyxDQUFDO1FBQ2xELElBQUksSUFBSSxDQUFDLGFBQWEsSUFBSSxJQUFJLENBQUMsVUFBVSxLQUFLLGVBQWUsQ0FBQyxlQUFlLEVBQUU7WUFBRSxJQUFJLENBQUMsYUFBYSxDQUFDLFlBQVksQ0FBQyxPQUFPLENBQUMsQ0FBQztTQUFFO1FBQzVILElBQUksQ0FBQyxNQUFNLENBQUMsVUFBVSxDQUFDLE9BQU8sRUFBRSxJQUFJLENBQUMsbUJBQW1CLEVBQUUsQ0FBQyxDQUFDO1FBQzVELE9BQU8sR0FBRyxDQUFDO0tBQ1o7SUFFRDs7OztPQUlHO0lBQ0ksY0FBYyxDQUFDLE9BQXVCO1FBQzNDLE1BQU0sR0FBRyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxFQUFFLENBQUMsR0FBRyxlQUFlLEVBQUUsR0FBRyxnQkFBZ0IsQ0FBQyxDQUFDLENBQUM7UUFDM0UsSUFBSSxJQUFJLENBQUMsYUFBYSxJQUFJLElBQUksQ0FBQyxVQUFVLEtBQUssZUFBZSxDQUFDLGVBQWUsRUFBRTtZQUFFLElBQUksQ0FBQyxhQUFhLENBQUMsbUJBQW1CLENBQUMsT0FBTyxDQUFDLENBQUM7U0FBRTtRQUNuSSxJQUFJLENBQUMsTUFBTSxDQUFDLGNBQWMsQ0FBQyxPQUFPLEVBQUUsSUFBSSxDQUFDLG1CQUFtQixFQUFFLENBQUMsQ0FBQztRQUNoRSxPQUFPLEdBQUcsQ0FBQztLQUNaO0lBRUQ7O09BRUc7SUFDSSxLQUFLLENBQUMsT0FBdUIsRUFBRSxPQUFpQjtRQUNyRCxPQUFPLEdBQUcsQ0FBQyxLQUFLLENBQUMsY0FBYyxDQUFDO1lBQzlCLE9BQU87WUFDUCxZQUFZLEVBQUUsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDO1lBQzdCLE9BQU87U0FDUixDQUFDLENBQUM7S0FDSjtJQUVEOzs7T0FHRztJQUNJLDBCQUEwQixDQUFDLE9BQXVCLEVBQUUsT0FBaUI7UUFDMUUsT0FBTyxHQUFHLENBQUMsS0FBSyxDQUFDLGNBQWMsQ0FBQztZQUM5QixPQUFPO1lBQ1AsWUFBWSxFQUFFO2dCQUNaLElBQUksQ0FBQyxRQUFRO2dCQUNiLElBQUksQ0FBQyxRQUFRLENBQUMsVUFBVTtnQkFDeEIsSUFBSSxDQUFDLFFBQVEsQ0FBQyxXQUFXO2FBQzFCO1lBQ0QsT0FBTztTQUNSLENBQUMsQ0FBQztLQUNKO0lBRU8sbUJBQW1CO1FBQ3pCLE9BQU8sSUFBSSxDQUFDLFFBQVEsR0FBRyxHQUFHLENBQUM7S0FDNUI7O0FBalNILHNCQWtTQzs7O0FBRUQsU0FBUyxjQUFjLENBQUMsT0FBaUIsRUFBRSxhQUF3QjtJQUNqRSxJQUFJLE9BQU8sQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1FBQ3hCLE1BQU0sSUFBSSxLQUFLLENBQUMsb0RBQW9ELENBQUMsQ0FBQztLQUN2RTtJQUNELHVGQUF1RjtJQUN2RixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsRUFBVSxDQUFDO0lBQ2hDLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxhQUFhLElBQUksRUFBRSxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLEVBQUU7UUFDckQsSUFBSSxLQUFLLENBQUMsR0FBRyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsRUFBRTtZQUMxQixNQUFNLElBQUksS0FBSyxDQUFDLHlEQUF5RCxNQUFNLENBQUMsSUFBSSxrQkFBa0IsQ0FBQyxDQUFDO1NBQ3pHO1FBQ0QsS0FBSyxDQUFDLEdBQUcsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDekIsQ0FBQyxDQUFDLENBQUM7QUFDTCxDQUFDO0FBRUQsMEVBQTBFO0FBQzFFLE1BQU0sa0JBQWtCLEdBQUc7SUFDekIsQ0FBQyxlQUFlLENBQUMsVUFBVSxDQUFDLEVBQUUsRUFBRSxDQUFDLGdCQUFnQixDQUFDLFVBQVU7SUFDNUQsQ0FBQyxlQUFlLENBQUMsV0FBVyxDQUFDLEVBQUUsRUFBRSxDQUFDLGdCQUFnQixDQUFDLFdBQVc7SUFDOUQsQ0FBQyxlQUFlLENBQUMsR0FBRyxDQUFDLEVBQUUsRUFBRSxDQUFDLGdCQUFnQixDQUFDLEdBQUc7SUFDOUMsQ0FBQyxlQUFlLENBQUMsZUFBZSxDQUFDLEVBQUUsRUFBRSxDQUFDLGdCQUFnQixDQUFDLFdBQVc7SUFDbEUsQ0FBQyxlQUFlLENBQUMsV0FBVyxDQUFDLEVBQUUsRUFBRSxDQUFDLGdCQUFnQixDQUFDLFdBQVc7Q0FDL0QsQ0FBQztBQUVGLDBHQUEwRztBQUMxRyxTQUFTLFlBQVksQ0FBQyxLQUFZLEVBQUUsS0FBaUI7SUFDbkQsTUFBTSxVQUFVLEdBQUcsS0FBSyxDQUFDLFVBQVUsSUFBSSxlQUFlLENBQUMsV0FBVyxDQUFDO0lBQ25FLElBQUksTUFBTSxHQUFHLEtBQUssQ0FBQyxNQUFNLENBQUM7SUFFMUIsSUFBSSxNQUFNLElBQUksQ0FBQyxVQUFVLEtBQUssZUFBZSxDQUFDLFdBQVcsSUFBSSxVQUFVLEtBQUssZUFBZSxDQUFDLGVBQWUsQ0FBQyxFQUFFO1FBQzVHLE1BQU0sSUFBSSxLQUFLLENBQUMsc0VBQXNFLENBQUMsQ0FBQztLQUN6RjtJQUVELElBQUksYUFBbUMsQ0FBQztJQUN4QyxJQUFJLFVBQVUsS0FBSyxlQUFlLENBQUMsZUFBZSxJQUFJLEtBQUssQ0FBQyxhQUFhLEtBQUssU0FBUyxFQUFFO1FBQ3ZGLCtGQUErRjtRQUMvRix5RkFBeUY7UUFDekYsYUFBYSxHQUFHLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxLQUFLLEVBQUUsS0FBSyxDQUFDLENBQUM7S0FDM0M7U0FBTTtRQUNMLGFBQWEsR0FBRyxLQUFLLENBQUMsYUFBYSxDQUFDO0tBQ3JDO0lBRUQseUNBQXlDO0lBQ3pDLElBQUksQ0FBQyxNQUFNLEVBQUU7UUFDWCxJQUFJLFVBQVUsS0FBSyxlQUFlLENBQUMsZUFBZSxFQUFFO1lBQ2xELE1BQU0sR0FBRyxJQUFJLEVBQUUsQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1NBQ3pDO2FBQU07WUFDTCxNQUFNLEdBQUcsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUU7Z0JBQ3RDLFVBQVUsRUFBRSxrQkFBa0IsQ0FBQyxVQUFVLENBQUM7Z0JBQzFDLGFBQWE7YUFDZCxDQUFDLENBQUM7WUFDSCxhQUFhLEdBQUcsTUFBTSxDQUFDLGFBQWEsQ0FBQztTQUN0QztLQUNGO0lBRUQsT0FBTztRQUNMLE1BQU07UUFDTixVQUFVO1FBQ1YsYUFBYTtLQUNkLENBQUM7QUFDSixDQUFDO0FBRUQsTUFBTSxlQUFlLEdBQUc7SUFDdEIsd0JBQXdCO0lBQ3hCLG1CQUFtQjtJQUNuQixvQkFBb0I7SUFDcEIsZUFBZTtJQUNmLGdCQUFnQjtJQUNoQixzQkFBc0I7SUFDdEIsdUJBQXVCO0NBQ3hCLENBQUM7QUFFRixNQUFNLGdCQUFnQixHQUFHO0lBQ3ZCLDJCQUEyQjtJQUMzQiwyQkFBMkI7SUFDM0Isc0JBQXNCO0lBQ3RCLHNCQUFzQjtJQUN0QixzQkFBc0I7Q0FDdkIsQ0FBQztBQUVGLFNBQVMsYUFBYSxDQUFDLE9BQWdDO0lBQ3JELElBQUksT0FBTyxLQUFLLFNBQVMsRUFBRTtRQUN6QixPQUFPLFNBQVMsQ0FBQztLQUNsQjtJQUNELE9BQU8sT0FBTyxDQUFDLEdBQUcsQ0FBQyxNQUFNLENBQUMsRUFBRTtRQUMxQixPQUFPO1lBQ0wsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJO1lBQ2pCLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFdBQVc7WUFDN0IsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPO1NBQ3hCLENBQUM7SUFDSixDQUFDLENBQUMsQ0FBQztBQUNMLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBpYW0gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSc7XG5pbXBvcnQgKiBhcyBrbXMgZnJvbSAnQGF3cy1jZGsvYXdzLWttcyc7XG5pbXBvcnQgKiBhcyBzMyBmcm9tICdAYXdzLWNkay9hd3MtczMnO1xuaW1wb3J0IHsgQXJuRm9ybWF0LCBGbiwgSVJlc291cmNlLCBOYW1lcywgUmVzb3VyY2UsIFN0YWNrIH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgKiBhcyBjciBmcm9tICdAYXdzLWNkay9jdXN0b20tcmVzb3VyY2VzJztcbmltcG9ydCB7IEF3c0N1c3RvbVJlc291cmNlIH0gZnJvbSAnQGF3cy1jZGsvY3VzdG9tLXJlc291cmNlcyc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IERhdGFGb3JtYXQgfSBmcm9tICcuL2RhdGEtZm9ybWF0JztcbmltcG9ydCB7IElEYXRhYmFzZSB9IGZyb20gJy4vZGF0YWJhc2UnO1xuaW1wb3J0IHsgQ2ZuVGFibGUgfSBmcm9tICcuL2dsdWUuZ2VuZXJhdGVkJztcbmltcG9ydCB7IENvbHVtbiB9IGZyb20gJy4vc2NoZW1hJztcblxuLyoqXG4gKiBQcm9wZXJ0aWVzIG9mIGEgUGFydGl0aW9uIEluZGV4LlxuICovXG5leHBvcnQgaW50ZXJmYWNlIFBhcnRpdGlvbkluZGV4IHtcbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBwYXJ0aXRpb24gaW5kZXguXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gYSBuYW1lIHdpbGwgYmUgZ2VuZXJhdGVkIGZvciB5b3UuXG4gICAqL1xuICByZWFkb25seSBpbmRleE5hbWU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBwYXJ0aXRpb24ga2V5IG5hbWVzIHRoYXQgY29tcHJpc2UgdGhlIHBhcnRpdGlvblxuICAgKiBpbmRleC4gVGhlIG5hbWVzIG11c3QgY29ycmVzcG9uZCB0byBhIG5hbWUgaW4gdGhlXG4gICAqIHRhYmxlJ3MgcGFydGl0aW9uIGtleXMuXG4gICAqL1xuICByZWFkb25seSBrZXlOYW1lczogc3RyaW5nW107XG59XG5leHBvcnQgaW50ZXJmYWNlIElUYWJsZSBleHRlbmRzIElSZXNvdXJjZSB7XG4gIC8qKlxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICByZWFkb25seSB0YWJsZUFybjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICByZWFkb25seSB0YWJsZU5hbWU6IHN0cmluZztcbn1cblxuLyoqXG4gKiBFbmNyeXB0aW9uIG9wdGlvbnMgZm9yIGEgVGFibGUuXG4gKlxuICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXRoZW5hL2xhdGVzdC91Zy9lbmNyeXB0aW9uLmh0bWxcbiAqL1xuZXhwb3J0IGVudW0gVGFibGVFbmNyeXB0aW9uIHtcbiAgVU5FTkNSWVBURUQgPSAnVW5lbmNyeXB0ZWQnLFxuXG4gIC8qKlxuICAgKiBTZXJ2ZXIgc2lkZSBlbmNyeXB0aW9uIChTU0UpIHdpdGggYW4gQW1hem9uIFMzLW1hbmFnZWQga2V5LlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvZGV2L1VzaW5nU2VydmVyU2lkZUVuY3J5cHRpb24uaHRtbFxuICAgKi9cbiAgUzNfTUFOQUdFRCA9ICdTU0UtUzMnLFxuXG4gIC8qKlxuICAgKiBTZXJ2ZXItc2lkZSBlbmNyeXB0aW9uIChTU0UpIHdpdGggYW4gQVdTIEtNUyBrZXkgbWFuYWdlZCBieSB0aGUgYWNjb3VudCBvd25lci5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L2Rldi9Vc2luZ0tNU0VuY3J5cHRpb24uaHRtbFxuICAgKi9cbiAgS01TID0gJ1NTRS1LTVMnLFxuXG4gIC8qKlxuICAgKiBTZXJ2ZXItc2lkZSBlbmNyeXB0aW9uIChTU0UpIHdpdGggYW4gQVdTIEtNUyBrZXkgbWFuYWdlZCBieSB0aGUgS01TIHNlcnZpY2UuXG4gICAqL1xuICBLTVNfTUFOQUdFRCA9ICdTU0UtS01TLU1BTkFHRUQnLFxuXG4gIC8qKlxuICAgKiBDbGllbnQtc2lkZSBlbmNyeXB0aW9uIChDU0UpIHdpdGggYW4gQVdTIEtNUyBrZXkgbWFuYWdlZCBieSB0aGUgYWNjb3VudCBvd25lci5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L2Rldi9Vc2luZ0NsaWVudFNpZGVFbmNyeXB0aW9uLmh0bWxcbiAgICovXG4gIENMSUVOVF9TSURFX0tNUyA9ICdDU0UtS01TJ1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIFRhYmxlQXR0cmlidXRlcyB7XG4gIHJlYWRvbmx5IHRhYmxlQXJuOiBzdHJpbmc7XG4gIHJlYWRvbmx5IHRhYmxlTmFtZTogc3RyaW5nO1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIFRhYmxlUHJvcHMge1xuICAvKipcbiAgICogTmFtZSBvZiB0aGUgdGFibGUuXG4gICAqL1xuICByZWFkb25seSB0YWJsZU5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogRGVzY3JpcHRpb24gb2YgdGhlIHRhYmxlLlxuICAgKlxuICAgKiBAZGVmYXVsdCBnZW5lcmF0ZWRcbiAgICovXG4gIHJlYWRvbmx5IGRlc2NyaXB0aW9uPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBEYXRhYmFzZSBpbiB3aGljaCB0byBzdG9yZSB0aGUgdGFibGUuXG4gICAqL1xuICByZWFkb25seSBkYXRhYmFzZTogSURhdGFiYXNlO1xuXG4gIC8qKlxuICAgKiBTMyBidWNrZXQgaW4gd2hpY2ggdG8gc3RvcmUgZGF0YS5cbiAgICpcbiAgICogQGRlZmF1bHQgb25lIGlzIGNyZWF0ZWQgZm9yIHlvdVxuICAgKi9cbiAgcmVhZG9ubHkgYnVja2V0PzogczMuSUJ1Y2tldDtcblxuICAvKipcbiAgICogUzMgcHJlZml4IHVuZGVyIHdoaWNoIHRhYmxlIG9iamVjdHMgYXJlIHN0b3JlZC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBwcmVmaXguIFRoZSBkYXRhIHdpbGwgYmUgc3RvcmVkIHVuZGVyIHRoZSByb290IG9mIHRoZSBidWNrZXQuXG4gICAqL1xuICByZWFkb25seSBzM1ByZWZpeD86IHN0cmluZztcblxuICAvKipcbiAgICogQ29sdW1ucyBvZiB0aGUgdGFibGUuXG4gICAqL1xuICByZWFkb25seSBjb2x1bW5zOiBDb2x1bW5bXTtcblxuICAvKipcbiAgICogUGFydGl0aW9uIGNvbHVtbnMgb2YgdGhlIHRhYmxlLlxuICAgKlxuICAgKiBAZGVmYXVsdCB0YWJsZSBpcyBub3QgcGFydGl0aW9uZWRcbiAgICovXG4gIHJlYWRvbmx5IHBhcnRpdGlvbktleXM/OiBDb2x1bW5bXTtcblxuICAvKipcbiAgICogUGFydGl0aW9uIGluZGV4ZXMgb24gdGhlIHRhYmxlLiBBIG1heGltdW0gb2YgMyBpbmRleGVzXG4gICAqIGFyZSBhbGxvd2VkIG9uIGEgdGFibGUuIEtleXMgaW4gdGhlIGluZGV4IG11c3QgYmUgcGFydFxuICAgKiBvZiB0aGUgdGFibGUncyBwYXJ0aXRpb24ga2V5cy5cbiAgICpcbiAgICogQGRlZmF1bHQgdGFibGUgaGFzIG5vIHBhcnRpdGlvbiBpbmRleGVzXG4gICAqL1xuICByZWFkb25seSBwYXJ0aXRpb25JbmRleGVzPzogUGFydGl0aW9uSW5kZXhbXTtcblxuICAvKipcbiAgICogU3RvcmFnZSB0eXBlIG9mIHRoZSB0YWJsZSdzIGRhdGEuXG4gICAqL1xuICByZWFkb25seSBkYXRhRm9ybWF0OiBEYXRhRm9ybWF0O1xuXG4gIC8qKlxuICAgKiBJbmRpY2F0ZXMgd2hldGhlciB0aGUgdGFibGUncyBkYXRhIGlzIGNvbXByZXNzZWQgb3Igbm90LlxuICAgKlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgY29tcHJlc3NlZD86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFRoZSBraW5kIG9mIGVuY3J5cHRpb24gdG8gc2VjdXJlIHRoZSBkYXRhIHdpdGguXG4gICAqXG4gICAqIFlvdSBjYW4gb25seSBwcm92aWRlIHRoaXMgb3B0aW9uIGlmIHlvdSBhcmUgbm90IGV4cGxpY2l0bHkgcGFzc2luZyBpbiBhIGJ1Y2tldC5cbiAgICpcbiAgICogSWYgeW91IGNob29zZSBgU1NFLUtNU2AsIHlvdSAqY2FuKiBwcm92aWRlIGFuIHVuLW1hbmFnZWQgS01TIGtleSB3aXRoIGBlbmNyeXB0aW9uS2V5YC5cbiAgICogSWYgeW91IGNob29zZSBgQ1NFLUtNU2AsIHlvdSAqbXVzdCogcHJvdmlkZSBhbiB1bi1tYW5hZ2VkIEtNUyBrZXkgd2l0aCBgZW5jcnlwdGlvbktleWAuXG4gICAqXG4gICAqIEBkZWZhdWx0IFVuZW5jcnlwdGVkXG4gICAqL1xuICByZWFkb25seSBlbmNyeXB0aW9uPzogVGFibGVFbmNyeXB0aW9uO1xuXG4gIC8qKlxuICAgKiBFeHRlcm5hbCBLTVMga2V5IHRvIHVzZSBmb3IgYnVja2V0IGVuY3J5cHRpb24uXG4gICAqXG4gICAqIFRoZSBgZW5jcnlwdGlvbmAgcHJvcGVydHkgbXVzdCBiZSBgU1NFLUtNU2Agb3IgYENTRS1LTVNgLlxuICAgKlxuICAgKiBAZGVmYXVsdCBrZXkgaXMgbWFuYWdlZCBieSBLTVMuXG4gICAqL1xuICByZWFkb25seSBlbmNyeXB0aW9uS2V5Pzoga21zLklLZXk7XG5cbiAgLyoqXG4gICAqIEluZGljYXRlcyB3aGV0aGVyIHRoZSB0YWJsZSBkYXRhIGlzIHN0b3JlZCBpbiBzdWJkaXJlY3Rvcmllcy5cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IHN0b3JlZEFzU3ViRGlyZWN0b3JpZXM/OiBib29sZWFuO1xufVxuXG4vKipcbiAqIEEgR2x1ZSB0YWJsZS5cbiAqL1xuZXhwb3J0IGNsYXNzIFRhYmxlIGV4dGVuZHMgUmVzb3VyY2UgaW1wbGVtZW50cyBJVGFibGUge1xuXG4gIHB1YmxpYyBzdGF0aWMgZnJvbVRhYmxlQXJuKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHRhYmxlQXJuOiBzdHJpbmcpOiBJVGFibGUge1xuICAgIGNvbnN0IHRhYmxlTmFtZSA9IEZuLnNlbGVjdCgxLCBGbi5zcGxpdCgnLycsIFN0YWNrLm9mKHNjb3BlKS5zcGxpdEFybih0YWJsZUFybiwgQXJuRm9ybWF0LlNMQVNIX1JFU09VUkNFX05BTUUpLnJlc291cmNlTmFtZSEpKTtcblxuICAgIHJldHVybiBUYWJsZS5mcm9tVGFibGVBdHRyaWJ1dGVzKHNjb3BlLCBpZCwge1xuICAgICAgdGFibGVBcm4sXG4gICAgICB0YWJsZU5hbWUsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBhIFRhYmxlIGNvbnN0cnVjdCB0aGF0IHJlcHJlc2VudHMgYW4gZXh0ZXJuYWwgdGFibGUuXG4gICAqXG4gICAqIEBwYXJhbSBzY29wZSBUaGUgc2NvcGUgY3JlYXRpbmcgY29uc3RydWN0ICh1c3VhbGx5IGB0aGlzYCkuXG4gICAqIEBwYXJhbSBpZCBUaGUgY29uc3RydWN0J3MgaWQuXG4gICAqIEBwYXJhbSBhdHRycyBJbXBvcnQgYXR0cmlidXRlc1xuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tVGFibGVBdHRyaWJ1dGVzKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIGF0dHJzOiBUYWJsZUF0dHJpYnV0ZXMpOiBJVGFibGUge1xuICAgIGNsYXNzIEltcG9ydCBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSVRhYmxlIHtcbiAgICAgIHB1YmxpYyByZWFkb25seSB0YWJsZUFybiA9IGF0dHJzLnRhYmxlQXJuO1xuICAgICAgcHVibGljIHJlYWRvbmx5IHRhYmxlTmFtZSA9IGF0dHJzLnRhYmxlTmFtZTtcbiAgICB9XG5cbiAgICByZXR1cm4gbmV3IEltcG9ydChzY29wZSwgaWQpO1xuICB9XG5cbiAgLyoqXG4gICAqIERhdGFiYXNlIHRoaXMgdGFibGUgYmVsb25ncyB0by5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBkYXRhYmFzZTogSURhdGFiYXNlO1xuXG4gIC8qKlxuICAgKiBJbmRpY2F0ZXMgd2hldGhlciB0aGUgdGFibGUncyBkYXRhIGlzIGNvbXByZXNzZWQgb3Igbm90LlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGNvbXByZXNzZWQ6IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFRoZSB0eXBlIG9mIGVuY3J5cHRpb24gZW5hYmxlZCBmb3IgdGhlIHRhYmxlLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGVuY3J5cHRpb246IFRhYmxlRW5jcnlwdGlvbjtcblxuICAvKipcbiAgICogVGhlIEtNUyBrZXkgdXNlZCB0byBzZWN1cmUgdGhlIGRhdGEgaWYgYGVuY3J5cHRpb25gIGlzIHNldCB0byBgQ1NFLUtNU2Agb3IgYFNTRS1LTVNgLiBPdGhlcndpc2UsIGB1bmRlZmluZWRgLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGVuY3J5cHRpb25LZXk/OiBrbXMuSUtleTtcblxuICAvKipcbiAgICogUzMgYnVja2V0IGluIHdoaWNoIHRoZSB0YWJsZSdzIGRhdGEgcmVzaWRlcy5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBidWNrZXQ6IHMzLklCdWNrZXQ7XG5cbiAgLyoqXG4gICAqIFMzIEtleSBQcmVmaXggdW5kZXIgd2hpY2ggdGhpcyB0YWJsZSdzIGZpbGVzIGFyZSBzdG9yZWQgaW4gUzMuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgczNQcmVmaXg6IHN0cmluZztcblxuICAvKipcbiAgICogTmFtZSBvZiB0aGlzIHRhYmxlLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHRhYmxlTmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBBUk4gb2YgdGhpcyB0YWJsZS5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSB0YWJsZUFybjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBGb3JtYXQgb2YgdGhpcyB0YWJsZSdzIGRhdGEgZmlsZXMuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZGF0YUZvcm1hdDogRGF0YUZvcm1hdDtcblxuICAvKipcbiAgICogVGhpcyB0YWJsZSdzIGNvbHVtbnMuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgY29sdW1uczogQ29sdW1uW107XG5cbiAgLyoqXG4gICAqIFRoaXMgdGFibGUncyBwYXJ0aXRpb24ga2V5cyBpZiB0aGUgdGFibGUgaXMgcGFydGl0aW9uZWQuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgcGFydGl0aW9uS2V5cz86IENvbHVtbltdO1xuXG4gIC8qKlxuICAgKiBUaGlzIHRhYmxlJ3MgcGFydGl0aW9uIGluZGV4ZXMuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgcGFydGl0aW9uSW5kZXhlcz86IFBhcnRpdGlvbkluZGV4W107XG5cbiAgLyoqXG4gICAqIFBhcnRpdGlvbiBpbmRleGVzIG11c3QgYmUgY3JlYXRlZCBvbmUgYXQgYSB0aW1lLiBUbyBhdm9pZFxuICAgKiByYWNlIGNvbmRpdGlvbnMsIHdlIHN0b3JlIHRoZSByZXNvdXJjZSBhbmQgYWRkIGRlcGVuZGVuY2llc1xuICAgKiBlYWNoIHRpbWUgYSBuZXcgcGFydGl0aW9uIGluZGV4IGlzIGNyZWF0ZWQuXG4gICAqL1xuICBwcml2YXRlIHBhcnRpdGlvbkluZGV4Q3VzdG9tUmVzb3VyY2VzOiBBd3NDdXN0b21SZXNvdXJjZVtdID0gW107XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IFRhYmxlUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHtcbiAgICAgIHBoeXNpY2FsTmFtZTogcHJvcHMudGFibGVOYW1lLFxuICAgIH0pO1xuXG4gICAgdGhpcy5kYXRhYmFzZSA9IHByb3BzLmRhdGFiYXNlO1xuICAgIHRoaXMuZGF0YUZvcm1hdCA9IHByb3BzLmRhdGFGb3JtYXQ7XG4gICAgdGhpcy5zM1ByZWZpeCA9IHByb3BzLnMzUHJlZml4ID8/ICcnO1xuXG4gICAgdmFsaWRhdGVTY2hlbWEocHJvcHMuY29sdW1ucywgcHJvcHMucGFydGl0aW9uS2V5cyk7XG4gICAgdGhpcy5jb2x1bW5zID0gcHJvcHMuY29sdW1ucztcbiAgICB0aGlzLnBhcnRpdGlvbktleXMgPSBwcm9wcy5wYXJ0aXRpb25LZXlzO1xuXG4gICAgdGhpcy5jb21wcmVzc2VkID0gcHJvcHMuY29tcHJlc3NlZCA/PyBmYWxzZTtcbiAgICBjb25zdCB7IGJ1Y2tldCwgZW5jcnlwdGlvbiwgZW5jcnlwdGlvbktleSB9ID0gY3JlYXRlQnVja2V0KHRoaXMsIHByb3BzKTtcbiAgICB0aGlzLmJ1Y2tldCA9IGJ1Y2tldDtcbiAgICB0aGlzLmVuY3J5cHRpb24gPSBlbmNyeXB0aW9uO1xuICAgIHRoaXMuZW5jcnlwdGlvbktleSA9IGVuY3J5cHRpb25LZXk7XG5cbiAgICBjb25zdCB0YWJsZVJlc291cmNlID0gbmV3IENmblRhYmxlKHRoaXMsICdUYWJsZScsIHtcbiAgICAgIGNhdGFsb2dJZDogcHJvcHMuZGF0YWJhc2UuY2F0YWxvZ0lkLFxuXG4gICAgICBkYXRhYmFzZU5hbWU6IHByb3BzLmRhdGFiYXNlLmRhdGFiYXNlTmFtZSxcblxuICAgICAgdGFibGVJbnB1dDoge1xuICAgICAgICBuYW1lOiB0aGlzLnBoeXNpY2FsTmFtZSxcbiAgICAgICAgZGVzY3JpcHRpb246IHByb3BzLmRlc2NyaXB0aW9uIHx8IGAke3Byb3BzLnRhYmxlTmFtZX0gZ2VuZXJhdGVkIGJ5IENES2AsXG5cbiAgICAgICAgcGFydGl0aW9uS2V5czogcmVuZGVyQ29sdW1ucyhwcm9wcy5wYXJ0aXRpb25LZXlzKSxcblxuICAgICAgICBwYXJhbWV0ZXJzOiB7XG4gICAgICAgICAgY2xhc3NpZmljYXRpb246IHByb3BzLmRhdGFGb3JtYXQuY2xhc3NpZmljYXRpb25TdHJpbmc/LnZhbHVlLFxuICAgICAgICAgIGhhc19lbmNyeXB0ZWRfZGF0YTogdGhpcy5lbmNyeXB0aW9uICE9PSBUYWJsZUVuY3J5cHRpb24uVU5FTkNSWVBURUQsXG4gICAgICAgIH0sXG4gICAgICAgIHN0b3JhZ2VEZXNjcmlwdG9yOiB7XG4gICAgICAgICAgbG9jYXRpb246IGBzMzovLyR7dGhpcy5idWNrZXQuYnVja2V0TmFtZX0vJHt0aGlzLnMzUHJlZml4fWAsXG4gICAgICAgICAgY29tcHJlc3NlZDogdGhpcy5jb21wcmVzc2VkLFxuICAgICAgICAgIHN0b3JlZEFzU3ViRGlyZWN0b3JpZXM6IHByb3BzLnN0b3JlZEFzU3ViRGlyZWN0b3JpZXMgPz8gZmFsc2UsXG4gICAgICAgICAgY29sdW1uczogcmVuZGVyQ29sdW1ucyhwcm9wcy5jb2x1bW5zKSxcbiAgICAgICAgICBpbnB1dEZvcm1hdDogcHJvcHMuZGF0YUZvcm1hdC5pbnB1dEZvcm1hdC5jbGFzc05hbWUsXG4gICAgICAgICAgb3V0cHV0Rm9ybWF0OiBwcm9wcy5kYXRhRm9ybWF0Lm91dHB1dEZvcm1hdC5jbGFzc05hbWUsXG4gICAgICAgICAgc2VyZGVJbmZvOiB7XG4gICAgICAgICAgICBzZXJpYWxpemF0aW9uTGlicmFyeTogcHJvcHMuZGF0YUZvcm1hdC5zZXJpYWxpemF0aW9uTGlicmFyeS5jbGFzc05hbWUsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcblxuICAgICAgICB0YWJsZVR5cGU6ICdFWFRFUk5BTF9UQUJMRScsXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgdGhpcy50YWJsZU5hbWUgPSB0aGlzLmdldFJlc291cmNlTmFtZUF0dHJpYnV0ZSh0YWJsZVJlc291cmNlLnJlZik7XG4gICAgdGhpcy50YWJsZUFybiA9IHRoaXMuc3RhY2suZm9ybWF0QXJuKHtcbiAgICAgIHNlcnZpY2U6ICdnbHVlJyxcbiAgICAgIHJlc291cmNlOiAndGFibGUnLFxuICAgICAgcmVzb3VyY2VOYW1lOiBgJHt0aGlzLmRhdGFiYXNlLmRhdGFiYXNlTmFtZX0vJHt0aGlzLnRhYmxlTmFtZX1gLFxuICAgIH0pO1xuICAgIHRoaXMubm9kZS5kZWZhdWx0Q2hpbGQgPSB0YWJsZVJlc291cmNlO1xuXG4gICAgLy8gUGFydGl0aW9uIGluZGV4IGNyZWF0aW9uIHJlbGllcyBvbiBjcmVhdGVkIHRhYmxlLlxuICAgIGlmIChwcm9wcy5wYXJ0aXRpb25JbmRleGVzKSB7XG4gICAgICB0aGlzLnBhcnRpdGlvbkluZGV4ZXMgPSBwcm9wcy5wYXJ0aXRpb25JbmRleGVzO1xuICAgICAgdGhpcy5wYXJ0aXRpb25JbmRleGVzLmZvckVhY2goKGluZGV4KSA9PiB0aGlzLmFkZFBhcnRpdGlvbkluZGV4KGluZGV4KSk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIEFkZCBhIHBhcnRpdGlvbiBpbmRleCB0byB0aGUgdGFibGUuIFlvdSBjYW4gaGF2ZSBhIG1heGltdW0gb2YgMyBwYXJ0aXRpb25cbiAgICogaW5kZXhlcyB0byBhIHRhYmxlLiBQYXJ0aXRpb24gaW5kZXgga2V5cyBtdXN0IGJlIGEgc3Vic2V0IG9mIHRoZSB0YWJsZSdzXG4gICAqIHBhcnRpdGlvbiBrZXlzLlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9nbHVlL2xhdGVzdC9kZy9wYXJ0aXRpb24taW5kZXhlcy5odG1sXG4gICAqL1xuICBwdWJsaWMgYWRkUGFydGl0aW9uSW5kZXgoaW5kZXg6IFBhcnRpdGlvbkluZGV4KSB7XG4gICAgY29uc3QgbnVtUGFydGl0aW9ucyA9IHRoaXMucGFydGl0aW9uSW5kZXhDdXN0b21SZXNvdXJjZXMubGVuZ3RoO1xuICAgIGlmIChudW1QYXJ0aXRpb25zID49IDMpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignTWF4aW11bSBudW1iZXIgb2YgcGFydGl0aW9uIGluZGV4ZXMgYWxsb3dlZCBpcyAzJyk7XG4gICAgfVxuICAgIHRoaXMudmFsaWRhdGVQYXJ0aXRpb25JbmRleChpbmRleCk7XG5cbiAgICBjb25zdCBpbmRleE5hbWUgPSBpbmRleC5pbmRleE5hbWUgPz8gdGhpcy5nZW5lcmF0ZUluZGV4TmFtZShpbmRleC5rZXlOYW1lcyk7XG4gICAgY29uc3QgcGFydGl0aW9uSW5kZXhDdXN0b21SZXNvdXJjZSA9IG5ldyBjci5Bd3NDdXN0b21SZXNvdXJjZSh0aGlzLCBgcGFydGl0aW9uLWluZGV4LSR7aW5kZXhOYW1lfWAsIHtcbiAgICAgIG9uQ3JlYXRlOiB7XG4gICAgICAgIHNlcnZpY2U6ICdHbHVlJyxcbiAgICAgICAgYWN0aW9uOiAnY3JlYXRlUGFydGl0aW9uSW5kZXgnLFxuICAgICAgICBwYXJhbWV0ZXJzOiB7XG4gICAgICAgICAgRGF0YWJhc2VOYW1lOiB0aGlzLmRhdGFiYXNlLmRhdGFiYXNlTmFtZSxcbiAgICAgICAgICBUYWJsZU5hbWU6IHRoaXMudGFibGVOYW1lLFxuICAgICAgICAgIFBhcnRpdGlvbkluZGV4OiB7XG4gICAgICAgICAgICBJbmRleE5hbWU6IGluZGV4TmFtZSxcbiAgICAgICAgICAgIEtleXM6IGluZGV4LmtleU5hbWVzLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICAgIHBoeXNpY2FsUmVzb3VyY2VJZDogY3IuUGh5c2ljYWxSZXNvdXJjZUlkLm9mKFxuICAgICAgICAgIGluZGV4TmFtZSxcbiAgICAgICAgKSxcbiAgICAgIH0sXG4gICAgICBwb2xpY3k6IGNyLkF3c0N1c3RvbVJlc291cmNlUG9saWN5LmZyb21TZGtDYWxscyh7XG4gICAgICAgIHJlc291cmNlczogY3IuQXdzQ3VzdG9tUmVzb3VyY2VQb2xpY3kuQU5ZX1JFU09VUkNFLFxuICAgICAgfSksXG4gICAgfSk7XG4gICAgdGhpcy5ncmFudFRvVW5kZXJseWluZ1Jlc291cmNlcyhwYXJ0aXRpb25JbmRleEN1c3RvbVJlc291cmNlLCBbJ2dsdWU6VXBkYXRlVGFibGUnXSk7XG5cbiAgICAvLyBEZXBlbmQgb24gcHJldmlvdXMgcGFydGl0aW9uIGluZGV4IGlmIHBvc3NpYmxlLCB0byBhdm9pZCByYWNlIGNvbmRpdGlvblxuICAgIGlmIChudW1QYXJ0aXRpb25zID4gMCkge1xuICAgICAgdGhpcy5wYXJ0aXRpb25JbmRleEN1c3RvbVJlc291cmNlc1tudW1QYXJ0aXRpb25zLTFdLm5vZGUuYWRkRGVwZW5kZW5jeShwYXJ0aXRpb25JbmRleEN1c3RvbVJlc291cmNlKTtcbiAgICB9XG4gICAgdGhpcy5wYXJ0aXRpb25JbmRleEN1c3RvbVJlc291cmNlcy5wdXNoKHBhcnRpdGlvbkluZGV4Q3VzdG9tUmVzb3VyY2UpO1xuICB9XG5cbiAgcHJpdmF0ZSBnZW5lcmF0ZUluZGV4TmFtZShrZXlzOiBzdHJpbmdbXSk6IHN0cmluZyB7XG4gICAgY29uc3QgcHJlZml4ID0ga2V5cy5qb2luKCctJykgKyAnLSc7XG4gICAgY29uc3QgdW5pcXVlSWQgPSBOYW1lcy51bmlxdWVJZCh0aGlzKTtcbiAgICBjb25zdCBtYXhJbmRleExlbmd0aCA9IDgwOyAvLyBhcmJpdHJhcmlseSBzcGVjaWZpZWRcbiAgICBjb25zdCBzdGFydEluZGV4ID0gTWF0aC5tYXgoMCwgdW5pcXVlSWQubGVuZ3RoIC0gKG1heEluZGV4TGVuZ3RoIC0gcHJlZml4Lmxlbmd0aCkpO1xuICAgIHJldHVybiBwcmVmaXggKyB1bmlxdWVJZC5zdWJzdHJpbmcoc3RhcnRJbmRleCk7XG4gIH1cblxuICBwcml2YXRlIHZhbGlkYXRlUGFydGl0aW9uSW5kZXgoaW5kZXg6IFBhcnRpdGlvbkluZGV4KSB7XG4gICAgaWYgKGluZGV4LmluZGV4TmFtZSAhPT0gdW5kZWZpbmVkICYmIChpbmRleC5pbmRleE5hbWUubGVuZ3RoIDwgMSB8fCBpbmRleC5pbmRleE5hbWUubGVuZ3RoID4gMjU1KSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBJbmRleCBuYW1lIG11c3QgYmUgYmV0d2VlbiAxIGFuZCAyNTUgY2hhcmFjdGVycywgYnV0IGdvdCAke2luZGV4LmluZGV4TmFtZS5sZW5ndGh9YCk7XG4gICAgfVxuICAgIGlmICghdGhpcy5wYXJ0aXRpb25LZXlzIHx8IHRoaXMucGFydGl0aW9uS2V5cy5sZW5ndGggPT09IDApIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignVGhlIHRhYmxlIG11c3QgaGF2ZSBwYXJ0aXRpb24ga2V5cyB0byBjcmVhdGUgYSBwYXJ0aXRpb24gaW5kZXgnKTtcbiAgICB9XG4gICAgY29uc3Qga2V5TmFtZXMgPSB0aGlzLnBhcnRpdGlvbktleXMubWFwKHBrID0+IHBrLm5hbWUpO1xuICAgIGlmICghaW5kZXgua2V5TmFtZXMuZXZlcnkoayA9PiBrZXlOYW1lcy5pbmNsdWRlcyhrKSkpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgQWxsIGluZGV4IGtleXMgbXVzdCBhbHNvIGJlIHBhcnRpdGlvbiBrZXlzLiBHb3QgJHtpbmRleC5rZXlOYW1lc30gYnV0IHBhcnRpdGlvbiBrZXkgbmFtZXMgYXJlICR7a2V5TmFtZXN9YCk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50IHJlYWQgcGVybWlzc2lvbnMgdG8gdGhlIHRhYmxlIGFuZCB0aGUgdW5kZXJseWluZyBkYXRhIHN0b3JlZCBpbiBTMyB0byBhbiBJQU0gcHJpbmNpcGFsLlxuICAgKlxuICAgKiBAcGFyYW0gZ3JhbnRlZSB0aGUgcHJpbmNpcGFsXG4gICAqL1xuICBwdWJsaWMgZ3JhbnRSZWFkKGdyYW50ZWU6IGlhbS5JR3JhbnRhYmxlKTogaWFtLkdyYW50IHtcbiAgICBjb25zdCByZXQgPSB0aGlzLmdyYW50KGdyYW50ZWUsIHJlYWRQZXJtaXNzaW9ucyk7XG4gICAgaWYgKHRoaXMuZW5jcnlwdGlvbktleSAmJiB0aGlzLmVuY3J5cHRpb24gPT09IFRhYmxlRW5jcnlwdGlvbi5DTElFTlRfU0lERV9LTVMpIHsgdGhpcy5lbmNyeXB0aW9uS2V5LmdyYW50RGVjcnlwdChncmFudGVlKTsgfVxuICAgIHRoaXMuYnVja2V0LmdyYW50UmVhZChncmFudGVlLCB0aGlzLmdldFMzUHJlZml4Rm9yR3JhbnQoKSk7XG4gICAgcmV0dXJuIHJldDtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudCB3cml0ZSBwZXJtaXNzaW9ucyB0byB0aGUgdGFibGUgYW5kIHRoZSB1bmRlcmx5aW5nIGRhdGEgc3RvcmVkIGluIFMzIHRvIGFuIElBTSBwcmluY2lwYWwuXG4gICAqXG4gICAqIEBwYXJhbSBncmFudGVlIHRoZSBwcmluY2lwYWxcbiAgICovXG4gIHB1YmxpYyBncmFudFdyaXRlKGdyYW50ZWU6IGlhbS5JR3JhbnRhYmxlKTogaWFtLkdyYW50IHtcbiAgICBjb25zdCByZXQgPSB0aGlzLmdyYW50KGdyYW50ZWUsIHdyaXRlUGVybWlzc2lvbnMpO1xuICAgIGlmICh0aGlzLmVuY3J5cHRpb25LZXkgJiYgdGhpcy5lbmNyeXB0aW9uID09PSBUYWJsZUVuY3J5cHRpb24uQ0xJRU5UX1NJREVfS01TKSB7IHRoaXMuZW5jcnlwdGlvbktleS5ncmFudEVuY3J5cHQoZ3JhbnRlZSk7IH1cbiAgICB0aGlzLmJ1Y2tldC5ncmFudFdyaXRlKGdyYW50ZWUsIHRoaXMuZ2V0UzNQcmVmaXhGb3JHcmFudCgpKTtcbiAgICByZXR1cm4gcmV0O1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50IHJlYWQgYW5kIHdyaXRlIHBlcm1pc3Npb25zIHRvIHRoZSB0YWJsZSBhbmQgdGhlIHVuZGVybHlpbmcgZGF0YSBzdG9yZWQgaW4gUzMgdG8gYW4gSUFNIHByaW5jaXBhbC5cbiAgICpcbiAgICogQHBhcmFtIGdyYW50ZWUgdGhlIHByaW5jaXBhbFxuICAgKi9cbiAgcHVibGljIGdyYW50UmVhZFdyaXRlKGdyYW50ZWU6IGlhbS5JR3JhbnRhYmxlKTogaWFtLkdyYW50IHtcbiAgICBjb25zdCByZXQgPSB0aGlzLmdyYW50KGdyYW50ZWUsIFsuLi5yZWFkUGVybWlzc2lvbnMsIC4uLndyaXRlUGVybWlzc2lvbnNdKTtcbiAgICBpZiAodGhpcy5lbmNyeXB0aW9uS2V5ICYmIHRoaXMuZW5jcnlwdGlvbiA9PT0gVGFibGVFbmNyeXB0aW9uLkNMSUVOVF9TSURFX0tNUykgeyB0aGlzLmVuY3J5cHRpb25LZXkuZ3JhbnRFbmNyeXB0RGVjcnlwdChncmFudGVlKTsgfVxuICAgIHRoaXMuYnVja2V0LmdyYW50UmVhZFdyaXRlKGdyYW50ZWUsIHRoaXMuZ2V0UzNQcmVmaXhGb3JHcmFudCgpKTtcbiAgICByZXR1cm4gcmV0O1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50IHRoZSBnaXZlbiBpZGVudGl0eSBjdXN0b20gcGVybWlzc2lvbnMuXG4gICAqL1xuICBwdWJsaWMgZ3JhbnQoZ3JhbnRlZTogaWFtLklHcmFudGFibGUsIGFjdGlvbnM6IHN0cmluZ1tdKSB7XG4gICAgcmV0dXJuIGlhbS5HcmFudC5hZGRUb1ByaW5jaXBhbCh7XG4gICAgICBncmFudGVlLFxuICAgICAgcmVzb3VyY2VBcm5zOiBbdGhpcy50YWJsZUFybl0sXG4gICAgICBhY3Rpb25zLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50IHRoZSBnaXZlbiBpZGVudGl0eSBjdXN0b20gcGVybWlzc2lvbnMgdG8gQUxMIHVuZGVybHlpbmcgcmVzb3VyY2VzIG9mIHRoZSB0YWJsZS5cbiAgICogUGVybWlzc2lvbnMgd2lsbCBiZSBncmFudGVkIHRvIHRoZSBjYXRhbG9nLCB0aGUgZGF0YWJhc2UsIGFuZCB0aGUgdGFibGUuXG4gICAqL1xuICBwdWJsaWMgZ3JhbnRUb1VuZGVybHlpbmdSZXNvdXJjZXMoZ3JhbnRlZTogaWFtLklHcmFudGFibGUsIGFjdGlvbnM6IHN0cmluZ1tdKSB7XG4gICAgcmV0dXJuIGlhbS5HcmFudC5hZGRUb1ByaW5jaXBhbCh7XG4gICAgICBncmFudGVlLFxuICAgICAgcmVzb3VyY2VBcm5zOiBbXG4gICAgICAgIHRoaXMudGFibGVBcm4sXG4gICAgICAgIHRoaXMuZGF0YWJhc2UuY2F0YWxvZ0FybixcbiAgICAgICAgdGhpcy5kYXRhYmFzZS5kYXRhYmFzZUFybixcbiAgICAgIF0sXG4gICAgICBhY3Rpb25zLFxuICAgIH0pO1xuICB9XG5cbiAgcHJpdmF0ZSBnZXRTM1ByZWZpeEZvckdyYW50KCkge1xuICAgIHJldHVybiB0aGlzLnMzUHJlZml4ICsgJyonO1xuICB9XG59XG5cbmZ1bmN0aW9uIHZhbGlkYXRlU2NoZW1hKGNvbHVtbnM6IENvbHVtbltdLCBwYXJ0aXRpb25LZXlzPzogQ29sdW1uW10pOiB2b2lkIHtcbiAgaWYgKGNvbHVtbnMubGVuZ3RoID09PSAwKSB7XG4gICAgdGhyb3cgbmV3IEVycm9yKCd5b3UgbXVzdCBzcGVjaWZ5IGF0IGxlYXN0IG9uZSBjb2x1bW4gZm9yIHRoZSB0YWJsZScpO1xuICB9XG4gIC8vIENoZWNrIHRoZXJlIGlzIGF0IGxlYXN0IG9uZSBjb2x1bW4gYW5kIG5vIGR1cGxpY2F0ZWQgY29sdW1uIG5hbWVzIG9yIHBhcnRpdGlvbiBrZXlzLlxuICBjb25zdCBuYW1lcyA9IG5ldyBTZXQ8c3RyaW5nPigpO1xuICAoY29sdW1ucy5jb25jYXQocGFydGl0aW9uS2V5cyB8fCBbXSkpLmZvckVhY2goY29sdW1uID0+IHtcbiAgICBpZiAobmFtZXMuaGFzKGNvbHVtbi5uYW1lKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBjb2x1bW4gbmFtZXMgYW5kIHBhcnRpdGlvbiBrZXlzIG11c3QgYmUgdW5pcXVlLCBidXQgXFwnJHtjb2x1bW4ubmFtZX1cXCcgaXMgZHVwbGljYXRlZGApO1xuICAgIH1cbiAgICBuYW1lcy5hZGQoY29sdW1uLm5hbWUpO1xuICB9KTtcbn1cblxuLy8gbWFwIFRhYmxlRW5jcnlwdGlvbiB0byBidWNrZXQncyBTU0UgY29uZmlndXJhdGlvbiAoczMuQnVja2V0RW5jcnlwdGlvbilcbmNvbnN0IGVuY3J5cHRpb25NYXBwaW5ncyA9IHtcbiAgW1RhYmxlRW5jcnlwdGlvbi5TM19NQU5BR0VEXTogczMuQnVja2V0RW5jcnlwdGlvbi5TM19NQU5BR0VELFxuICBbVGFibGVFbmNyeXB0aW9uLktNU19NQU5BR0VEXTogczMuQnVja2V0RW5jcnlwdGlvbi5LTVNfTUFOQUdFRCxcbiAgW1RhYmxlRW5jcnlwdGlvbi5LTVNdOiBzMy5CdWNrZXRFbmNyeXB0aW9uLktNUyxcbiAgW1RhYmxlRW5jcnlwdGlvbi5DTElFTlRfU0lERV9LTVNdOiBzMy5CdWNrZXRFbmNyeXB0aW9uLlVORU5DUllQVEVELFxuICBbVGFibGVFbmNyeXB0aW9uLlVORU5DUllQVEVEXTogczMuQnVja2V0RW5jcnlwdGlvbi5VTkVOQ1JZUFRFRCxcbn07XG5cbi8vIGNyZWF0ZSB0aGUgYnVja2V0IHRvIHN0b3JlIGEgdGFibGUncyBkYXRhIGRlcGVuZGluZyBvbiB0aGUgYGVuY3J5cHRpb25gIGFuZCBgZW5jcnlwdGlvbktleWAgcHJvcGVydGllcy5cbmZ1bmN0aW9uIGNyZWF0ZUJ1Y2tldCh0YWJsZTogVGFibGUsIHByb3BzOiBUYWJsZVByb3BzKSB7XG4gIGNvbnN0IGVuY3J5cHRpb24gPSBwcm9wcy5lbmNyeXB0aW9uIHx8IFRhYmxlRW5jcnlwdGlvbi5VTkVOQ1JZUFRFRDtcbiAgbGV0IGJ1Y2tldCA9IHByb3BzLmJ1Y2tldDtcblxuICBpZiAoYnVja2V0ICYmIChlbmNyeXB0aW9uICE9PSBUYWJsZUVuY3J5cHRpb24uVU5FTkNSWVBURUQgJiYgZW5jcnlwdGlvbiAhPT0gVGFibGVFbmNyeXB0aW9uLkNMSUVOVF9TSURFX0tNUykpIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoJ3lvdSBjYW4gbm90IHNwZWNpZnkgZW5jcnlwdGlvbiBzZXR0aW5ncyBpZiB5b3UgYWxzbyBwcm92aWRlIGEgYnVja2V0Jyk7XG4gIH1cblxuICBsZXQgZW5jcnlwdGlvbktleToga21zLklLZXkgfCB1bmRlZmluZWQ7XG4gIGlmIChlbmNyeXB0aW9uID09PSBUYWJsZUVuY3J5cHRpb24uQ0xJRU5UX1NJREVfS01TICYmIHByb3BzLmVuY3J5cHRpb25LZXkgPT09IHVuZGVmaW5lZCkge1xuICAgIC8vIENTRS1LTVMgc2hvdWxkIGJlaGF2ZSB0aGUgc2FtZSBhcyBTU0UtS01TIC0gdXNlIHRoZSBwcm92aWRlZCBrZXkgb3IgY3JlYXRlIG9uZSBhdXRvbWF0aWNhbGx5XG4gICAgLy8gU2luY2UgQnVja2V0IG9ubHkga25vd3MgYWJvdXQgU1NFLCB3ZSByZXBlYXQgdGhlIGxvZ2ljIGZvciBDU0UtS01TIGF0IHRoZSBUYWJsZSBsZXZlbC5cbiAgICBlbmNyeXB0aW9uS2V5ID0gbmV3IGttcy5LZXkodGFibGUsICdLZXknKTtcbiAgfSBlbHNlIHtcbiAgICBlbmNyeXB0aW9uS2V5ID0gcHJvcHMuZW5jcnlwdGlvbktleTtcbiAgfVxuXG4gIC8vIGNyZWF0ZSB0aGUgYnVja2V0IGlmIG5vbmUgd2FzIHByb3ZpZGVkXG4gIGlmICghYnVja2V0KSB7XG4gICAgaWYgKGVuY3J5cHRpb24gPT09IFRhYmxlRW5jcnlwdGlvbi5DTElFTlRfU0lERV9LTVMpIHtcbiAgICAgIGJ1Y2tldCA9IG5ldyBzMy5CdWNrZXQodGFibGUsICdCdWNrZXQnKTtcbiAgICB9IGVsc2Uge1xuICAgICAgYnVja2V0ID0gbmV3IHMzLkJ1Y2tldCh0YWJsZSwgJ0J1Y2tldCcsIHtcbiAgICAgICAgZW5jcnlwdGlvbjogZW5jcnlwdGlvbk1hcHBpbmdzW2VuY3J5cHRpb25dLFxuICAgICAgICBlbmNyeXB0aW9uS2V5LFxuICAgICAgfSk7XG4gICAgICBlbmNyeXB0aW9uS2V5ID0gYnVja2V0LmVuY3J5cHRpb25LZXk7XG4gICAgfVxuICB9XG5cbiAgcmV0dXJuIHtcbiAgICBidWNrZXQsXG4gICAgZW5jcnlwdGlvbixcbiAgICBlbmNyeXB0aW9uS2V5LFxuICB9O1xufVxuXG5jb25zdCByZWFkUGVybWlzc2lvbnMgPSBbXG4gICdnbHVlOkJhdGNoR2V0UGFydGl0aW9uJyxcbiAgJ2dsdWU6R2V0UGFydGl0aW9uJyxcbiAgJ2dsdWU6R2V0UGFydGl0aW9ucycsXG4gICdnbHVlOkdldFRhYmxlJyxcbiAgJ2dsdWU6R2V0VGFibGVzJyxcbiAgJ2dsdWU6R2V0VGFibGVWZXJzaW9uJyxcbiAgJ2dsdWU6R2V0VGFibGVWZXJzaW9ucycsXG5dO1xuXG5jb25zdCB3cml0ZVBlcm1pc3Npb25zID0gW1xuICAnZ2x1ZTpCYXRjaENyZWF0ZVBhcnRpdGlvbicsXG4gICdnbHVlOkJhdGNoRGVsZXRlUGFydGl0aW9uJyxcbiAgJ2dsdWU6Q3JlYXRlUGFydGl0aW9uJyxcbiAgJ2dsdWU6RGVsZXRlUGFydGl0aW9uJyxcbiAgJ2dsdWU6VXBkYXRlUGFydGl0aW9uJyxcbl07XG5cbmZ1bmN0aW9uIHJlbmRlckNvbHVtbnMoY29sdW1ucz86IEFycmF5PENvbHVtbiB8IENvbHVtbj4pIHtcbiAgaWYgKGNvbHVtbnMgPT09IHVuZGVmaW5lZCkge1xuICAgIHJldHVybiB1bmRlZmluZWQ7XG4gIH1cbiAgcmV0dXJuIGNvbHVtbnMubWFwKGNvbHVtbiA9PiB7XG4gICAgcmV0dXJuIHtcbiAgICAgIG5hbWU6IGNvbHVtbi5uYW1lLFxuICAgICAgdHlwZTogY29sdW1uLnR5cGUuaW5wdXRTdHJpbmcsXG4gICAgICBjb21tZW50OiBjb2x1bW4uY29tbWVudCxcbiAgICB9O1xuICB9KTtcbn1cbiJdfQ==