"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ListenerPort = exports.LoadBalancer = exports.LoadBalancingProtocol = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const elasticloadbalancing_generated_1 = require("./elasticloadbalancing.generated");
var LoadBalancingProtocol;
(function (LoadBalancingProtocol) {
    LoadBalancingProtocol["TCP"] = "tcp";
    LoadBalancingProtocol["SSL"] = "ssl";
    LoadBalancingProtocol["HTTP"] = "http";
    LoadBalancingProtocol["HTTPS"] = "https";
})(LoadBalancingProtocol = exports.LoadBalancingProtocol || (exports.LoadBalancingProtocol = {}));
/**
 * A load balancer with a single listener
 *
 * Routes to a fleet of of instances in a VPC.
 */
class LoadBalancer extends core_1.Resource {
    constructor(scope, id, props) {
        var _c;
        super(scope, id);
        /**
         * An object controlling specifically the connections for each listener added to this load balancer
         */
        this.listenerPorts = [];
        this.listeners = [];
        this.instancePorts = [];
        this.targets = [];
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancing_LoadBalancerProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        this.securityGroup = new aws_ec2_1.SecurityGroup(this, 'SecurityGroup', { vpc: props.vpc, allowAllOutbound: false });
        this.connections = new aws_ec2_1.Connections({ securityGroups: [this.securityGroup] });
        // Depending on whether the ELB has public or internal IPs, pick the right backend subnets
        const selectedSubnets = loadBalancerSubnets(props);
        this.elb = new elasticloadbalancing_generated_1.CfnLoadBalancer(this, 'Resource', {
            securityGroups: [this.securityGroup.securityGroupId],
            subnets: selectedSubnets.subnetIds,
            listeners: core_1.Lazy.any({ produce: () => this.listeners }),
            scheme: props.internetFacing ? 'internet-facing' : 'internal',
            healthCheck: props.healthCheck && healthCheckToJSON(props.healthCheck),
            crossZone: (_c = props.crossZone) !== null && _c !== void 0 ? _c : true,
        });
        if (props.internetFacing) {
            this.elb.node.addDependency(selectedSubnets.internetConnectivityEstablished);
        }
        if (props.accessLoggingPolicy !== undefined) {
            this.elb.accessLoggingPolicy = props.accessLoggingPolicy;
        }
        ifUndefined(props.listeners, []).forEach(b => this.addListener(b));
        ifUndefined(props.targets, []).forEach(t => this.addTarget(t));
    }
    /**
     * Add a backend to the load balancer
     *
     * @returns A ListenerPort object that controls connections to the listener port
     */
    addListener(listener) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancing_LoadBalancerListener(listener);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addListener);
            }
            throw error;
        }
        if (listener.sslCertificateArn && listener.sslCertificateId) {
            throw new Error('"sslCertificateId" is deprecated, please use "sslCertificateArn" only.');
        }
        const protocol = ifUndefinedLazy(listener.externalProtocol, () => wellKnownProtocol(listener.externalPort));
        const instancePort = listener.internalPort || listener.externalPort;
        const sslCertificateArn = listener.sslCertificateArn || listener.sslCertificateId;
        const instanceProtocol = ifUndefined(listener.internalProtocol, ifUndefined(tryWellKnownProtocol(instancePort), isHttpProtocol(protocol) ? LoadBalancingProtocol.HTTP : LoadBalancingProtocol.TCP));
        this.listeners.push({
            loadBalancerPort: listener.externalPort.toString(),
            protocol,
            instancePort: instancePort.toString(),
            instanceProtocol,
            sslCertificateId: sslCertificateArn,
            policyNames: listener.policyNames,
        });
        const port = new ListenerPort(this.securityGroup, aws_ec2_1.Port.tcp(listener.externalPort));
        // Allow connections on the public port for all supplied peers (default: everyone)
        ifUndefined(listener.allowConnectionsFrom, [aws_ec2_1.Peer.anyIpv4()]).forEach(peer => {
            port.connections.allowDefaultPortFrom(peer, `Default rule allow on ${listener.externalPort}`);
        });
        this.newInstancePort(instancePort);
        // Keep track using array so user can get to them even if they were all supplied in the constructor
        this.listenerPorts.push(port);
        return port;
    }
    addTarget(target) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancing_ILoadBalancerTarget(target);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addTarget);
            }
            throw error;
        }
        target.attachToClassicLB(this);
        this.newTarget(target);
    }
    /**
     * @attribute
     */
    get loadBalancerName() {
        return this.elb.ref;
    }
    /**
     * @attribute
     */
    get loadBalancerCanonicalHostedZoneNameId() {
        return this.elb.attrCanonicalHostedZoneNameId;
    }
    /**
     * @attribute
     */
    get loadBalancerCanonicalHostedZoneName() {
        return this.elb.attrCanonicalHostedZoneName;
    }
    /**
     * @attribute
     */
    get loadBalancerDnsName() {
        return this.elb.attrDnsName;
    }
    /**
     * @attribute
     */
    get loadBalancerSourceSecurityGroupGroupName() {
        return this.elb.attrSourceSecurityGroupGroupName;
    }
    /**
     * @attribute
     */
    get loadBalancerSourceSecurityGroupOwnerAlias() {
        return this.elb.attrSourceSecurityGroupOwnerAlias;
    }
    /**
     * Allow connections to all existing targets on new instance port
     */
    newInstancePort(instancePort) {
        this.targets.forEach(t => this.allowTargetConnection(instancePort, t));
        // Keep track of port for future targets
        this.instancePorts.push(instancePort);
    }
    /**
     * Allow connections to target on all existing instance ports
     */
    newTarget(target) {
        this.instancePorts.forEach(p => this.allowTargetConnection(p, target));
        // Keep track of target for future listeners.
        this.targets.push(target);
    }
    /**
     * Allow connections for a single (port, target) pair
     */
    allowTargetConnection(instancePort, target) {
        this.connections.allowTo(target, aws_ec2_1.Port.tcp(instancePort), `Port ${instancePort} LB to fleet`);
    }
}
exports.LoadBalancer = LoadBalancer;
_a = JSII_RTTI_SYMBOL_1;
LoadBalancer[_a] = { fqn: "@aws-cdk/aws-elasticloadbalancing.LoadBalancer", version: "1.158.0" };
/**
 * Reference to a listener's port just created.
 *
 * This implements IConnectable with a default port (the port that an ELB
 * listener was just created on) for a given security group so that it can be
 * conveniently used just like any Connectable. E.g:
 *
 *    const listener = elb.addListener(...);
 *
 *    listener.connections.allowDefaultPortFromAnyIPv4();
 *    // or
 *    instance.connections.allowToDefaultPort(listener);
 */
class ListenerPort {
    constructor(securityGroup, defaultPort) {
        this.connections = new aws_ec2_1.Connections({ securityGroups: [securityGroup], defaultPort });
    }
}
exports.ListenerPort = ListenerPort;
_b = JSII_RTTI_SYMBOL_1;
ListenerPort[_b] = { fqn: "@aws-cdk/aws-elasticloadbalancing.ListenerPort", version: "1.158.0" };
function wellKnownProtocol(port) {
    const proto = tryWellKnownProtocol(port);
    if (!proto) {
        throw new Error(`Please supply protocol to go with port ${port}`);
    }
    return proto;
}
function tryWellKnownProtocol(port) {
    if (port === 80) {
        return LoadBalancingProtocol.HTTP;
    }
    if (port === 443) {
        return LoadBalancingProtocol.HTTPS;
    }
    return undefined;
}
function isHttpProtocol(proto) {
    return proto === LoadBalancingProtocol.HTTPS || proto === LoadBalancingProtocol.HTTP;
}
function ifUndefined(x, def) {
    return x != null ? x : def;
}
function ifUndefinedLazy(x, def) {
    return x != null ? x : def();
}
/**
 * Turn health check parameters into a parameter blob for the LB
 */
function healthCheckToJSON(healthCheck) {
    const protocol = ifUndefined(healthCheck.protocol, ifUndefined(tryWellKnownProtocol(healthCheck.port), LoadBalancingProtocol.TCP));
    const path = protocol === LoadBalancingProtocol.HTTP || protocol === LoadBalancingProtocol.HTTPS ? ifUndefined(healthCheck.path, '/') : '';
    const target = `${protocol.toUpperCase()}:${healthCheck.port}${path}`;
    return {
        healthyThreshold: ifUndefined(healthCheck.healthyThreshold, 2).toString(),
        interval: (healthCheck.interval || core_1.Duration.seconds(30)).toSeconds().toString(),
        target,
        timeout: (healthCheck.timeout || core_1.Duration.seconds(5)).toSeconds().toString(),
        unhealthyThreshold: ifUndefined(healthCheck.unhealthyThreshold, 5).toString(),
    };
}
function loadBalancerSubnets(props) {
    if (props.subnetSelection !== undefined) {
        return props.vpc.selectSubnets(props.subnetSelection);
    }
    else if (props.internetFacing) {
        return props.vpc.selectSubnets({
            subnetType: aws_ec2_1.SubnetType.PUBLIC,
        });
    }
    else {
        return props.vpc.selectSubnets({
            subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_NAT,
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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