import numpy as np
import sympy as sp
from .utils import IndexPermutation, Permutation, differentialLimit
from scipy.special import comb
from numpy.typing import NDArray
from joblib import delayed, Parallel, cpu_count

#Parallel is broken in debug mode
import sys
gettrace = getattr(sys, 'gettrace', None)
_inDebugMode = gettrace and gettrace() is not None


class SymKrylovSpaceCache:
  def __init__(self):
    self.value: dict[tuple[int, ...], SymKrylovSpace] = {}
  def __getitem__(self, mults: tuple[int, ...]):
    mults = list(mults) if hasattr(mults, '__iter__') else [mults]
    if len(mults) == 0: return SymKrylovSpace([])

    multsNormalOrdered = tuple(sorted(mults, key=lambda m: -m))
    if multsNormalOrdered not in self.value:
      self.value[multsNormalOrdered] = SymKrylovSpace(multsNormalOrdered)
    sk = self.value[multsNormalOrdered]
    sk.multiplicity = np.array(multsNormalOrdered)
    ret = SymKrylovSpaceCache.permuteEigvals(sk, mults)
    return ret

  @staticmethod
  def permuteEigvals(sk: 'SymKrylovSpace', newOrder: list[int]):
    evL: list[list[int]] = []
    i0 = 0
    for m in sk.multiplicity:
      evL.append(list(range(i0, i0+m)))
      i0 += m

    permMults = Permutation.findIndexPermutation(newOrder, sk.multiplicity)
    permEvs = IndexPermutation(*[ev for evs in (evL @ permMults) for ev in evs])

    ret = SymKrylovSpace([])
    tmpVars: sp.Array = sp.Array(sp.var(f'\\lambda1:{len(permMults)+1}^{{(tmp)}}'))
    subL1 = list(zip(sk.eigvals.tolist(), tmpVars.tolist()))
    subL2 = list(zip(tmpVars.tolist(), sk.eigvals.tolist() @ permMults.inverse))
    ret.multiplicity = np.array(newOrder)
    ret.eigvals = sk.eigvals.copy()
    ret.mVar = sk.mVar
    for attrName in 'lambdaVec betaVec betaM lambdaCoeffs funcCoeffs'.split():
      attr: sp.Array = getattr(sk, attrName)
      attr = attr.subs(subL1).subs(subL2)
      setattr(ret, attrName, attr)
    for attrName in 'lambdaVec betaVec'.split():
      attr: sp.Array = getattr(ret, attrName)
      attr = sp.Array(np.array(attr)[list(permEvs)])
      setattr(ret, attrName, attr)
    for attrName in 'betaM funcCoeffs'.split():
      attr: sp.Array = getattr(ret, attrName)
      attr = sp.Array(np.array(attr)[..., list(permMults), :])
      setattr(ret, attrName, attr)
    return ret

  def __str__(self) -> str:
    return f'SymKrylovSpaceCache{list(self.value)}'
  def __repr__(self) -> str:
    return str(self)


class SymKrylovSpace:
  '''Calculates various objects related to the krylov space of some square matrix but independent from the representation.

  The krylov space is a vector space generated by a square matrix and the cayley-hamilton theorem
  which states that every square matrix is a root of its own characteristic polynomial. From this
  it follows that A^n is a linear combination of 1=A^0, A, A^2, ..., A^(n-1) and therefore every
  polynomial in A is such a linear combination.

  Attributes
  ----------
  eigvals : Eigvals
    The eigenvalues of M
  beta : array-like
    A helper vector for solving the recurrence equation. See the documentation for details
  lambdaCoeffs : array-like
    Coefficients of M^n as a vector in the krylov space.
  funcCoeffs : array-like
    A tensor for calculating any matrix function efficiently
  '''

  def __init__(self, multiplicity: NDArray[np.integer]| int, *additionalMults: int):
    if additionalMults:
      if hasattr(multiplicity, '__iter__'):
        multiplicity = list(multiplicity) + list(additionalMults)
      else:
        multiplicity = list((multiplicity,) + additionalMults)
    elif not hasattr(multiplicity, '__iter__'):
      multiplicity = [multiplicity]

    self.multiplicity = np.array(multiplicity)
    if len(self.multiplicity) > 0:
      if len(self.multiplicity) == 1:
        self.eigvals: sp.Array = sp.Array([sp.var('\\lambda')])
      else:
        self.eigvals: sp.Array = sp.Array(sp.var(f'\\lambda_1:{len(self.multiplicity) + 1}'))
      self.lambdaVec, self.mVar = SymKrylovSpace.LambdaVec(self.eigvals, self.multiplicity)
      self.betaVec = SymKrylovSpace.BetaVec(self.lambdaVec, eigvals=self.eigvals)
      self.betaM = SymKrylovSpace.BetaM(self.multiplicity, self.betaVec, self.eigvals)
      self.lambdaCoeffs = SymKrylovSpace.LambdaCoeffs(self.eigvals, self.multiplicity)
      self.funcCoeffs = SymKrylovSpace.FunctionCoeffs(self.eigvals, self.multiplicity, self.betaM, self.lambdaCoeffs)
    else:
      self.mVar = sp.var('m')
      for attr in 'eigvals lambdaVec betaVec betaM lambdaCoeffs funcCoeffs'.split():
        setattr(self, attr, sp.Array([]))

  @staticmethod
  def LambdaVec(eigvals: sp.Array, multiplicity: NDArray[np.integer], m: sp.Symbol = sp.var('m')) -> tuple[sp.Array, sp.Symbol]:
    assert len(eigvals) == len(multiplicity)
    ret: list[sp.Symbol] = []
    for i in range(len(eigvals)):
      ev = eigvals[i]
      for j in range(multiplicity[i]):
        ret.append((ev**m).diff(ev, j).simplify())
    return sp.Array(ret), m

  @staticmethod
  def BetaVec0(lambdaVec: sp.Array|None = None, eigvals: sp.Array|None = None, multiplicity: NDArray[np.integer]|None=None, m: sp.Symbol = sp.var('m')):
    if lambdaVec is None:
      lambdaVec,_ = SymKrylovSpace.LambdaVec(eigvals, multiplicity, m)
    n = len(lambdaVec)
    if multiplicity is not None:
      assert n == np.sum(multiplicity)
    matrix0 = sp.Matrix([lambdaVec.subs(m, i) for i in range(n-1)])
    eyeN = sp.eye(n)
    if _inDebugMode:
      beta0: sp.Array = sp.Array([
        matrix0.col_join(eyeN[i,:]).det().simplify().factor() for i in range(n)
        ])
    else:
      beta0: sp.Array = sp.Array(Parallel(n_jobs=cpu_count())(delayed(
          lambda i: matrix0.col_join(eyeN[i,:]).det().simplify().factor()
        )(i) for i in range(n)))
    return beta0

  @staticmethod
  def BetaVec(lambdaVec: sp.Array|None=None, betaVec0: sp.Array|None=None, eigvals: sp.Array|None=None, multiplicity: NDArray[np.integer]|None = None, m: sp.Symbol = sp.var('m')) -> sp.Array:
    if lambdaVec is None:
      lambdaVec,_ = SymKrylovSpace.LambdaVec(eigvals, multiplicity, m)
    if betaVec0 is None:
      betaVec0 = SymKrylovSpace.BetaVec0(lambdaVec, eigvals, multiplicity, m)
    n = len(lambdaVec)
    if multiplicity is not None:
      assert n == np.sum(multiplicity)
    #beta: sp.Array = betaVec0 / sp.tensorcontraction(sp.tensorproduct(betaVec0, lambdaVec.subs(m, n-1)), (0,1)).simplify()
    beta: sp.Array = betaVec0 / (np.array(betaVec0.tolist()) @ lambdaVec.subs(m, n-1).tolist()).factor()
    return beta.simplify()

  @staticmethod
  def BetaM(multiplicity: NDArray[np.integer]|None=None, betaVec: sp.Array|None=None, eigvals: sp.Array|None=None, betaVec0: sp.Array|None=None, lambdaVec: sp.Array|None=None, m: sp.Symbol = sp.var('m')) -> sp.Array:
    if betaVec is None:
      betaVec = SymKrylovSpace.BetaVec(lambdaVec, betaVec0, eigvals, multiplicity, m)
    multMax = np.max(multiplicity)
    betaVecL = betaVec.tolist()
    beta = []
    m0 = 0
    for mi in multiplicity:
      row = betaVecL[m0:m0+mi]
      if mi < multMax:
        row += [0]*(multMax-mi)
      beta.append(row)
      m0+=mi
    return sp.Array(beta)


  @staticmethod
  def LambdaCoeffs(eigvals: sp.Array, multiplicity: NDArray[np.integer]) -> sp.Array:
    '''The coefficients of M^n as a vector in the krylov space. There are also equal to the non-trivial negated coeffecients of the characteristic polynomial of `M`
    
    Parameters
    ----------
    eigvals : array-like | None = None
      The eigenvalues of `M`
    M : array-like | None = None
      A square matrix
    inds : list[int] | int | None
      The indices for which the coefficients should be calculated. If `None` all coefficients are calculated.
    Either `eigvals` or `M` should be specified. Optionally specify `inds`.
    '''
    t = sp.var('t')
    charPolyNeg: sp.Poly = -sp.Poly(sp.prod([(t-eigvals[i])**multiplicity[i] for i in range(len(eigvals))]), t)
    return sp.Array(charPolyNeg.all_coeffs()[1:][::-1])
    
    
  @staticmethod
  def FunctionCoeffs(eigvals: sp.Array, multiplicity: NDArray[np.integer], betaM: sp.Array|None=None, lambdaCoeffs: sp.Array|None=None, **kwargs):
    '''A rank 3 tensor which collects all contraction to efficiently calculate the matrix via tensor contraction.
    In total `f(M)_ij = FunctionCoeffs(..)_ijk f(eigvals_k)`

    Parameters
    ----------
    basis : array-like | None = None
      The list A^0, A^1, ..., A^(n-1)
    beta : array-like | None = None
      Beta
    lambdaCoeffs : array-like | None = None
      Lambda
    eigvals : array-like | None = None
      The eigenvalues of `M`
    Either `eigvals` or `M` should be specified. Optionally specify `basis`, `beta` and `lambdaCoeffs`.
    '''
    n = np.sum(multiplicity)
    maxMult = np.max(multiplicity)
    if lambdaCoeffs is None:
      lambdaCoeffs = SymKrylovSpace.LambdaCoeffs(eigvals, multiplicity)
    if betaM is None:
      betaM = SymKrylovSpace.BetaM(multiplicity, eigvals=eigvals)

    def getElem(k,l,q):
      js = np.arange(k+1)
      ps = np.arange(q, multiplicity[l])
      t1 = np.array(lambdaCoeffs[k::-1].tolist()) #(kj)
      t2 = np.array(betaM[l, ps[0]:ps[-1]+1].tolist()) #(lp)
      t3 = np.outer(np.ones_like(js), np.array([comb(p, q, exact=True) for p in ps])*(-1)**(ps-q)) * np.array([[np.arange(j+1, j+1+p-q).prod() for p in ps] for j in js]) #(jp)
      t4 = np.array([[eigvals[l]**-(j+1+p-q) for p in ps] for j in js]) #(jp)
      res = (t1 @ (t3*t4) @ t2).simplify().factor()
      return res
    ret: sp.Array = sp.Array(
      [
        [
          [
            getElem(k,l,q)
            for q in range(multiplicity[l])]
          + [0]*(maxMult - multiplicity[l])
          for l in range(len(eigvals))]
        for k in range(n)]
      )
    return ret


  def reduceFunctionCoeffs(self, ev1, ev2, dummyFunc='f', dummyVar='\\epsilon'):
    if ev1 == ev2 or ev1 not in self.eigvals or ev2 not in self.eigvals:
      return self.funcCoeffs
    iA, iB = [i for i,ev in enumerate(self.eigvals) if ev in [ev1, ev2]]
    auxvars = sp.var(f'\\lambda_1:{len(self.eigvals) + 1}^{{(tmp)}}')
    evA = self.eigvals[iA]
    evB = self.eigvals[iB]
    subL1 = list(zip(self.eigvals[iB+1:], auxvars[iB+1:]))
    subL2 = list(zip(auxvars[iB+1:], self.eigvals[iB:-1]))
    mult = np.append(self.multiplicity[:iB], self.multiplicity[iB+1:])
    mult[iA] += self.multiplicity[iB]
    multMax = mult.max()

    f = sp.Function(dummyFunc)
    eps = sp.var(dummyVar)
    fVec = sp.Array(self.eigvals).applyfunc(f)
    funcCoeffs = self.funcCoeffs
    fTensor = sp.Array([fVec.applyfunc(lambda q: q.diff(*q.args, i)) for i in range(funcCoeffs.shape[2])])
    tensor = sp.Array(np.tensordot(funcCoeffs, fTensor, ((1,2), (1,0))))
    tensorRed: sp.Array = tensor.applyfunc(lambda x:
      differentialLimit(x, evB, evA, multMax))
    
    if fTensor.shape[0] < multMax:
      fTensor = sp.Array([fVec.applyfunc(lambda q: q.diff(*q.args, i)) for i in range(multMax)])
    fTensor1: sp.Array = sp.Array([
        [
          [
            t.subs(fTensor[q, l], 1).subs([(fev, 0) for fev in fTensor[0]]).doit()
          for q in range(multMax)]
        for l in list(range(iB))+list(range(iB+1, len(self.multiplicity)))]
      for t in tensorRed]).subs(subL1).subs(subL2)
    if len(mult) == 1:
      fTensor1 = fTensor1.subs(sp.var('\\lambda_1'), sp.var('\\lambda'))
    return fTensor1
    

  def __repr__(self) -> str:
    evs = ', '.join(map(str, self.multiplicity))
    return f'SymKrylov[ var: {self.mVar}, multiplicity: ({evs}) ]' if evs else 'SymKrylov[]'
  def _repr_latex_(self) -> str:
    evs = ',\\ '.join([f'{ev}\\; ({i})' for ev, i in zip(self.eigvals, self.multiplicity)])
    return f'SymKrylov[ var: ${self.mVar}$, eigvals: ${evs}$ ]' if evs else 'SymKrylov[]'



