from typing import Generator
import numpy as np
import scipy.linalg
from numpy.typing import NDArray


def dimVSpace(span: NDArray, rcond: float | None = None) -> int:
  '''Dimension of vectorspace generated by `span`. Uses `scipy.linalg.orth` to count the number of linear independent vectors in the vectorspace.
  
  Parameters
  ----------
  span: array-like
    Array that either has the span vectors in the rows or coloumns
  rcond: float | None = None
    Passed to `scipy.linalg.orth`
  '''
  basis = scipy.linalg.orth(span, rcond)
  return basis.shape[1]


class Eigvals:
  '''Class for handling eigenvalues and the multiplicity

  Attributes
  ----------
  eigvals: array-like
    Different eigenvalues
  algMult: array-like
    Algebraic multiplicity of the eigenvalues
  irredMult: array-like
    Multiplicity of the eigenvalue in an irreducable representation.
  '''
  def __init__(self, eigvals: NDArray[np.complexfloating], algMult: NDArray[np.integer], irredMult: NDArray[np.integer] | None = None, geomMult: NDArray[np.integer] | None = None):
    self.eigvals = np.real_if_close(eigvals)
    self.algMult = np.array(algMult)
    if irredMult is not None:
      self.irredMult = np.array(irredMult)
    elif geomMult is not None:
      self.irredMult = self.algMult + 1 - np.array(geomMult)
    else:
      self.irredMult = None
  @property
  def fullEigvals(self) -> NDArray[np.complexfloating]:
    'Array containing all eigenvalues with each eigenvalue repeated `algMult` times.'
    return np.concatenate([[l]*n for l,n in zip(self.eigvals, self.algMult)])
  @property
  def geomMult(self) -> NDArray[np.integer] | None:
    'Geometric multiplicity of the eigenvalues. Equal to `algMult + 1 - irredMult`'
    return self.algMult + 1 - self.irredMult if self.irredMult is not None else None
  
  @staticmethod
  def fromFullEigvals(fullEigvals: NDArray[np.complexfloating], thresh: float = 1e-12):
    '''Constructs Eigvals object from full array of eigenvalues. Geometric multiplicity is ignored.

    Parameters
    ----------
    fullEigvals: array-like
      Repeated eigenvalues
    thresh: float = 1e-12
      Threshold used to decide whether two eigenvalues are equal. Checks abs(ev_i-ev_j) <= thresh*max(abs(evs))
    '''
    if (np.array(fullEigvals) == 0).all():
      return Eigvals(np.zeros(1), [len(fullEigvals)])
    fullEigvals = np.array(fullEigvals)
    fullEigvals0 = fullEigvals / np.max(abs(fullEigvals))
    eigvalsD: dict[int, int] = {}
    for i in range(len(fullEigvals)):
      ev0 = fullEigvals0[i]
      for j in eigvalsD:
        if abs(fullEigvals0[j] - ev0) <= thresh:
          eigvalsD[j] += 1
          ev0 = None
          break
      #Not in eigvalsD
      if ev0 is not None:
        eigvalsD[i] = 1
    eigvals, mult = map(list, zip(*[(fullEigvals[i], n) for i, n in eigvalsD.items()]))
    return Eigvals(eigvals, mult)

  @staticmethod
  def fromMatrix(M: NDArray[np.complexfloating]):
    'Constructs Eigvals object from matrix'
    if scipy.linalg.ishermitian(np.array(M)):
      evs, vecs = scipy.linalg.eigh(M)
    else:
      evs, vecs = scipy.linalg.eig(M)
    i0, i1 = 0, 0
    res = []
    while i0 < len(evs):
      while i1 < len(evs) and evs[i1] == evs[i0]:
        i1 += 1
      ev = evs[i0]
      algMult = i1-i0
      geomMult = dimVSpace(vecs[:,i0:i1])
      res.append((ev, algMult, geomMult))
      i0 = i1
    eigs, alg, geom = map(list, zip(*res))
    return Eigvals(eigs, alg, geomMult=geom)

  def __iter__(self) -> Generator[tuple[np.complexfloating, int, int], None, None] | Generator[tuple[np.complexfloating, int], None, None]:
    return zip(*[x for x in [self.eigvals, self.algMult, self.irredMult] if x is not None])

  def __str__(self) -> str:
    if self.irredMult:
      s0 = ', '.join([f'{x} ({am}, {gm})' for x, am, gm in self])
    else:
      s0 = ', '.join([f'{x} ({am})' for x, am in self])
    return f'Eigvals{{ {s0} }}' if s0 else 'Eigvals{ }'
  def __repr__(self) -> str:
    return str(self)

Eigvals([1.,2.], [2,2])
Eigvals.fromFullEigvals([1.,2.]*3)
Eigvals.fromFullEigvals([1, 1+1e-13])
a = .2
Eigvals.fromMatrix(np.array([[1,2*a,a**2],[0,1,a], [0,0,1]]))
Eigvals.fromMatrix(np.eye(3))


class AbstrKrylovSpace:
  '''Calculates various objects related to the krylov space of some square matrix but independent from the representation.

  The krylov space is a vector space generated by a square matrix and the cayley-hamilton theorem
  which states that every square matrix is a root of its own characteristic polynomial. From this
  it follows that A^n is a linear combination of 1=A^0, A, A^2, ..., A^(n-1) and therefore every
  polynomial in A is such a linear combination.

  Attributes
  ----------
  eigvals : Eigvals
    The eigenvalues of M
  beta : array-like
    A helper vector for solving the recurrence equation. See the documentation for details
  lambdaCoeffs : array-like
    Coefficients of M^n as a vector in the krylov space.
  funcCoeffs : array-like
    A tensor for calculating any matrix function efficiently
  '''

  def __init__(self, eigvals: Eigvals):
    self.eigvals = eigvals
    self.beta = AbstrKrylovSpace.Beta(eigvals=self.eigvals)
    self.lambdaCoeffs = AbstrKrylovSpace.Lambda(eigvals=self.eigvals)
    self.funcCoeffs = AbstrKrylovSpace.FunctionCoeffs(self.basis, self.beta, self.lambdaCoeffs, self.eigvals)

  @staticmethod
  def Beta0(M: np.ndarray | None =None, eigvals: np.ndarray | None =None):
    '''A vector which is perpendicular to e_i*ev_i^m for m=0..n-2 where ev_i are the eigenvalues and n the number of eigenvalues

    Parameters
    ----------
    M : array-like | None = None
      A square matrix
    eigvals : array-like | None = None
      The eigenvalues of `M`
    Either `eigvals` or `M` should be specified.
    '''
    if eigvals is None:
      if M is None:
        raise ValueError('M or eigvals have to be specified')
      else:
        eigvals = np.linalg.eigvals(M)
    else:
      eigvals = np.array(eigvals)
    nEigvals = eigvals.shape[0]
    if nEigvals == 1: return 1.
    if nEigvals == 2: return np.array([1,-1], dtype=eigvals.dtype)
    Q = ([np.ones_like(eigvals), eigvals] + [eigvals**k for k in range(2, nEigvals-1)])[::-1]
    ret = [np.linalg.det([ei]+Q) for ei in np.eye(nEigvals)]
    return np.array(ret)

  @staticmethod
  def Beta(beta0: np.ndarray|None=None, eigvals: np.ndarray|None=None, M:np.ndarray|None=None):
    '''beta0/(beta0 @ e_i*ev_i^(n-1)), see also method `KrylovSpace.Beta0`.

    Parameters
    ----------
    beta0 : array-like | None = None
      Beta0
    eigvals : array-like | None = None
      The eigenvalues of `M`
    M : array-like | None = None
      A square matrix
    Either `eigvals` or `M` should be specified. Optionally specify beta0.
    '''
    if eigvals is None:
      if M is None:
        raise ValueError('M or eigvals have to be specified')
      else:
        eigvals = np.linalg.eigvals(M)
    else:
      eigvals = np.array(eigvals)
    if beta0 is None:
      beta0 = KrylovSpace.Beta0(M, eigvals)
    else:
      beta0 = np.array(beta0)

    nEigvals = eigvals.shape[0]
    lms = eigvals**(nEigvals-1)
    ret: np.ndarray = beta0 / (beta0 @ lms)
    return ret

  @staticmethod
  def Lambda(eigvals: np.ndarray|None=None, M: np.ndarray|None=None, inds: list[int]|int|None=None):
    '''The coefficients of M^n as a vector in the krylov space. There are also equal to the non-trivial negated coeffecients of the characteristic polynomial of `M`
    
    Parameters
    ----------
    eigvals : array-like | None = None
      The eigenvalues of `M`
    M : array-like | None = None
      A square matrix
    inds : list[int] | int | None
      The indices for which the coefficients should be calculated. If `None` all coefficients are calculated.
    Either `eigvals` or `M` should be specified. Optionally specify `inds`.
    '''
    if eigvals is None:
      if M is None:
        raise ValueError('M or eigvals have to be specified')
      else:
        eigvals = np.linalg.eigvals(M)
    else:
      eigvals = np.array(eigvals)
    nEigvals = eigvals.shape[0]
    if inds is None:
      inds = range(nEigvals)
    def getLambdaI(k:int) -> float|complex:
      def multiI():
        js = list(range(nEigvals-k))
        while True:
          yield js
          i = next((i for i in range(1, len(js)+1) if js[-i] < nEigvals-i), None)
          if i is None:
            break
          else:
            js[-i]+=1
            for j in range(-i+1, 0):
              js[j] = js[j-1]+1
          # if js[-1] < nEigvals-1:
          #   js[-1]+=1
          # elif js[-2] < nEigvals-2:
          #   js[-2]+=1
          #   js[-1]=js[-2]+1
          # #...
          # else:
          #   break
      arr = []
      for js in multiI():
        arr.append(-np.prod(-eigvals[js]))
      return np.sum(arr)

    if hasattr(inds, '__iter__'):
      return np.array([getLambdaI(i) for i in inds])
    else:
      return getLambdaI(inds)
    
  @staticmethod
  def FunctionCoeffs(basis: np.ndarray|None=None, beta:np.ndarray|None=None, lambdaCoeffs: np.ndarray|None=None, eigvals: np.ndarray|None=None, **kwargs):
    '''A rank 3 tensor which collects all contraction to efficiently calculate the matrix via tensor contraction.
    In total `f(M)_ij = FunctionCoeffs(..)_ijk f(eigvals_k)`

    Parameters
    ----------
    basis : array-like | None = None
      The list A^0, A^1, ..., A^(n-1)
    beta : array-like | None = None
      Beta
    lambdaCoeffs : array-like | None = None
      Lambda
    eigvals : array-like | None = None
      The eigenvalues of `M`
    Either `eigvals` or `M` should be specified. Optionally specify `basis`, `beta` and `lambdaCoeffs`.
    '''
    if eigvals is None:
      M = kwargs.get('M', None)
      if M is None:
        raise ValueError('M or eigvals have to be specified')
      else:
        eigvals = np.linalg.eigvals(M)
    else:
      eigvals = np.array(eigvals)
    nEigvals = eigvals.shape[0]
    if basis is None:
      basis = KrylovSpace.Basis(eigvals=eigvals, **kwargs)
    if beta is None:
      beta = KrylovSpace.Beta(eigvals=eigvals, **kwargs)
    if lambdaCoeffs is None:
      lambdaCoeffs = KrylovSpace.Lambda(eigvals=eigvals, **kwargs)
    eigvalTensor = np.array([eigvals**q for q in range(-nEigvals, 0)]).T
    #eigvalTensorK = lambda k: np.array([eigvals**q for q in range(-k-1, 0)]).T
    eigvalTensorK = lambda k: eigvalTensor[:, nEigvals-k-1:]
    ret : np.ndarray = np.sum([np.outer(basis[k], beta*(eigvalTensorK(k) @ lambdaCoeffs[:k+1])).reshape((nEigvals,)*3) for k in range(nEigvals)], axis=0)
    #ret = np.sum([np.einsum('ij,k,km,m->ijk', basis[k], beta, eigvalTensorK(k), lambdaCoeffs[:k+1]) for k in range(nEigvals)], axis=0)
    return ret

  def __repr__(self) -> str:
    return 'KrylovSpace'+repr(dict(
      M = self.M,
      Eigvals = self.eigvals,
      Basis = self.basis,
      Beta0 = KrylovSpace.Beta0(self.eigvals),
      Beta = self.beta,
      Lambda = self.lambdaCoeffs, 
      Coeffs = self.funcCoeffs,
    ))

