# -*- coding: utf-8 -*-
from setuptools import setup

modules = \
['eventsourcing_orjsontranscoder']
install_requires = \
['orjson>=3.6.7,<3.7.0']

setup_kwargs = {
    'name': 'eventsourcing-orjsontranscoder',
    'version': '0.1.1',
    'description': 'Python package for OrjsonTranscoder.',
    'long_description': '# Welcome to the OrjsonTranscoder project\n\nThis package provides a `OrjsonTranscoder` class for use with\nthe Python eventsourcing library that uses the [orjson\nlibrary](https://pypi.org/project/orjson).\n\nMost importantly, `OrjsonTranscoder` supports custom transcoding of instances\nof `tuple` and subclasses of `str`, `int`, `dict` and `tuple`. This is an\nimportant improvement on the core library\'s `JSONTranscoder` class which converts\n`tuple` to `list` and loses type information for subclasses of `str`, `int`, `dict`\nand `tuple`.\n\nIt is also faster than `JSONTranscoder`, encoding approximately x3 faster\nand decoding approximately x2 faster. This is less important than the preservation\nof type information (see above) because latency in your application will\nusually be dominated by database interactions. However, it\'s nice that it\nis not slower.\n\n| class            | encode  | decode  |\n|------------------|---------|---------|\n| OrjsonTranscoder | 6.8 μs  | 13.8 μs |\n| JSON Transcoder  | 20.1 μs | 25.7 μs |\n\n\nThis package uses Cython, so relevant build tools may need to be\ninstalled before this package can be installed successfully.\n\n## Installation\n\nUse pip to install the [stable distribution](https://pypi.org/project/eventsourcing-orjsontranscoder/)\nfrom the Python Package Index.\n\n    $ pip install eventsourcing_orjsontranscoder\n\nPlease note, it is recommended to install Python packages into a Python virtual environment.\n\n## Custom Transcodings\n\nDefine custom transcodings for your custom value object types by subclassing\n``CTranscoding``. The prefix ``C`` is used to distinguish these classes from the\n``Transcoding`` classes provided by the core Python eventsourcing library.\n\nFor example, consider the custom value object ``MyInt`` below.\n\n```python\nclass MyInt(int):\n    def __repr__(self):\n        return f"{type(self).__name__}({super().__repr__()})"\n\n    def __eq__(self, other):\n        return type(self) == type(other) and super().__eq__(other)\n```\n\nYou can define a custom transcoding for ``MyInt`` as a normal Python class in a\nnormal Python module (``.py`` file) using the ``CTranscoding`` class.\n\n```python\nclass CMyIntAsInt(CTranscoding):\n    def type(self):\n        return MyInt\n\n    def name(self):\n        return "myint_as_int"\n\n    def encode(self, obj):\n        return int(obj)\n\n    def decode(self, data):\n        return MyInt(data)\n```\n\nAlternatively for greater speed, you can define a custom transcoding for ``MyInt``\nas a Cython extension type class in a Cython module (``.pyx`` file) using the\n``CTranscoding`` extension type. See this project\'s Git repository for examples.\n\n```cython\nfrom _eventsourcing_orjsontranscoder cimport CTranscoding\n\nfrom my_domain_model import MyInt\n\ncdef class CMyIntAsInt(CTranscoding):\n    cpdef object type(self):\n        return MyInt\n\n    cpdef object name(self):\n        return "myint_as_int"\n\n    cpdef object encode(self, object obj):\n        return int(obj)\n\n    cpdef object decode(self, object data):\n        return MyInt(data)\n```\n\nIf you define Cython modules, you will need to build them in-place before you\ncan use them. If you are distributing your code, you will also need to configure\nyour distribution to build the Cython module when your code is installed.\n\n```commandline\n$ cythonize -i my_transcodings.pyx\n```\n\nSee the Cython documentation for more information about Cython.\n\n\n## Using the OrjsonTranscoder\n\nTo use the ``OrjsonTranscoder`` class in a Python eventsourcing application\nobject, override  the `construct_transcoder()` and `register_transcodings()`\nmethods.\n\n```python\n\nfrom eventsourcing.application import Application\nfrom eventsourcing.domain import Aggregate, event\nfrom eventsourcing_orjsontranscoder import (\n    CDatetimeAsISO,\n    CTupleAsList,\n    CUUIDAsHex,\n    OrjsonTranscoder,\n)\n\n\nclass DogSchool(Application):\n    def construct_transcoder(self):\n        transcoder = OrjsonTranscoder()\n        self.register_transcodings(transcoder)\n        return transcoder\n\n    def register_transcodings(self, transcoder):\n        transcoder.register(CDatetimeAsISO())\n        transcoder.register(CTupleAsList())\n        transcoder.register(CUUIDAsHex())\n        transcoder.register(CMyIntAsInt())\n\n    def register_dog(self, name, age):\n        dog = Dog(name, age)\n        self.save(dog)\n        return dog.id\n\n    def add_trick(self, dog_id, trick):\n        dog = self.repository.get(dog_id)\n        dog.add_trick(trick)\n        self.save(dog)\n\n    def update_age(self, dog_id, age):\n        dog = self.repository.get(dog_id)\n        dog.update_age(age)\n        self.save(dog)\n\n    def get_dog(self, dog_id):\n        dog = self.repository.get(dog_id)\n        return {"name": dog.name, "tricks": tuple(dog.tricks), "age": dog.age}\n\n\nclass Dog(Aggregate):\n    @event("Registered")\n    def __init__(self, name, age):\n        self.name = name\n        self.age = age\n        self.tricks = []\n\n    @event("TrickAdded")\n    def add_trick(self, trick):\n        self.tricks.append(trick)\n\n    @event("AgeUpdated")\n    def update_age(self, age):\n        self.age = age\n\n\ndef test_dog_school():\n    # Construct application object.\n    school = DogSchool()\n\n    # Evolve application state.\n    dog_id = school.register_dog("Fido", MyInt(2))\n    school.add_trick(dog_id, "roll over")\n    school.add_trick(dog_id, "play dead")\n    school.update_age(dog_id, MyInt(5))\n\n    # Query application state.\n    dog = school.get_dog(dog_id)\n    assert dog["name"] == "Fido"\n    assert dog["tricks"] == ("roll over", "play dead")\n    assert dog["age"] == MyInt(5)\n\n    # Select notifications.\n    notifications = school.notification_log.select(start=1, limit=10)\n    assert len(notifications) == 4\n```\n\nSee the [library docs](https://eventsourcing.readthedocs.io/en/stable/topics/persistence.html#transcodings)\nfor more information about transcoding, but please note the `CTranscoder` uses a slightly\ndifferent API.\n\n\n## Developers\n\nAfter cloning the repository, you can set up a virtual environment and\ninstall dependencies by running the following command in the root\nfolder.\n\n    $ make install\n\nAfter making changes, please run the tests.\n\n    $ make test\n\nCheck the formatting of the code.\n\n    $ make lint\n\nYou can automatically reformat the code by running the following command.\n\n    $ make fmt\n\nIf the project dependencies change, you can update your packages by running\nthe following command.\n\n    $ make update-packages\n\nPlease submit changes for review by making a pull request.\n',
    'author': 'John Bywater',
    'author_email': 'john.bywater@appropriatesoftware.net',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://eventsourcing.readthedocs.io/',
    'py_modules': modules,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}
from build import *
build(setup_kwargs)

setup(**setup_kwargs)
