# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['vban_cmd']

package_data = \
{'': ['*']}

extras_require = \
{':python_version < "3.11"': ['tomli>=2.0.1,<3.0.0']}

setup_kwargs = {
    'name': 'vban-cmd',
    'version': '1.5.2',
    'description': 'Python interface for the VBAN RT Packet Service (Sendtext)',
    'long_description': '[![PyPI version](https://badge.fury.io/py/vban-cmd.svg)](https://badge.fury.io/py/vban-cmd)\n[![License: MIT](https://img.shields.io/badge/License-MIT-yellow.svg)](https://github.com/onyx-and-iris/vban-cmd-python/blob/dev/LICENSE)\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n[![Imports: isort](https://img.shields.io/badge/%20imports-isort-%231674b1?style=flat&labelColor=ef8336)](https://pycqa.github.io/isort/)\n![Tests Status](./tests/basic.svg?dummy=8484744)\n![Tests Status](./tests/banana.svg?dummy=8484744)\n![Tests Status](./tests/potato.svg?dummy=8484744)\n\n# VBAN CMD\n\nThis python interface allows you to get and set Voicemeeter parameter values over a network.\n\nIt may be used standalone or to extend the [Voicemeeter Remote Python API](https://github.com/onyx-and-iris/voicemeeter-api-python)\n\nThere is no support for audio transfer in this package, only parameters.\n\nFor an outline of past/future changes refer to: [CHANGELOG](CHANGELOG.md)\n\n## Tested against\n\n-   Basic 1.0.8.4\n-   Banana 2.0.6.4\n-   Potato 3.0.2.4\n\n## Requirements\n\n-   [Voicemeeter](https://voicemeeter.com/)\n-   Python 3.10 or greater\n\n## Installation\n\n### `Pip`\n\nInstall vban-cmd package from your console\n\n`pip install vban-cmd`\n\n## `Use`\n\nSimplest use case, use a context manager to request a VbanCmd class of a kind.\n\nLogin and logout are handled for you in this scenario.\n\n#### `__main__.py`\n\n```python\nimport vban_cmd\n\n\nclass ManyThings:\n    def __init__(self, vban):\n        self.vban = vban\n\n    def things(self):\n        self.vban.strip[0].label = "podmic"\n        self.vban.strip[0].mute = True\n        print(\n            f"strip 0 ({self.vban.strip[0].label}) mute has been set to {self.vban.strip[0].mute}"\n        )\n\n    def other_things(self):\n        info = (\n            f"bus 3 gain has been set to {self.vban.bus[3].gain}",\n            f"bus 4 eq has been set to {self.vban.bus[4].eq}",\n        )\n        self.vban.bus[3].gain = -6.3\n        self.vban.bus[4].eq = True\n        print("\\n".join(info))\n\n\ndef main():\n    with vban_cmd.api(kind_id, **opts) as vban:\n        do = ManyThings(vban)\n        do.things()\n        do.other_things()\n\n        # set many parameters at once\n        vban.apply(\n            {\n                "strip-2": {"A1": True, "B1": True, "gain": -6.0},\n                "bus-2": {"mute": True},\n            }\n        )\n\n\nif __name__ == "__main__":\n    kind_id = "banana"\n    opts = {\n        "ip": "<ip address>",\n        "streamname": "Command1",\n        "port": 6980,\n    }\n\n    main()\n```\n\nOtherwise you must remember to call `vban.login()`, `vban.logout()` at the start/end of your code.\n\n## `kind_id`\n\nPass the kind of Voicemeeter as an argument. kind_id may be:\n\n-   `basic`\n-   `banana`\n-   `potato`\n\n## `Available commands`\n\n### Strip\n\nThe following properties are available.\n\n-   `mono`: boolean\n-   `solo`: boolean\n-   `mute`: boolean\n-   `label`: string\n-   `gain`: float, -60 to 12\n-   `A1 - A5`, `B1 - B3`: boolean\n-   `comp`: float, from 0.0 to 10.0\n-   `gate`: float, from 0.0 to 10.0\n-   `limit`: int, from -40 to 12\n\nexample:\n\n```python\nvban.strip[3].gain = 3.7\nprint(strip[0].label)\n```\n\n##### Gainlayers\n\n-   `gain`: float, from -60.0 to 12.0\n\nexample:\n\n```python\nvban.strip[3].gainlayer[3].gain = 3.7\n```\n\nGainlayers are defined for potato version only.\n\n##### Levels\n\nThe following properties are available.\n\n-   `prefader`\n\nexample:\n\n```python\nprint(vban.strip[3].levels.prefader)\n```\n\nLevel properties will return -200.0 if no audio detected.\n\n### Bus\n\nThe following properties are available.\n\n-   `mono`: boolean\n-   `eq`: boolean\n-   `eq_ab`: boolean\n-   `mute`: boolean\n-   `label`: string\n-   `gain`: float, -60 to 12\n\nexample:\n\n```python\nvban.bus[4].eq = true\nprint(vban.bus[0].label)\n```\n\n##### Modes\n\nThe following properties are available.\n\n-   `normal`: boolean\n-   `amix`: boolean\n-   `bmix`: boolean\n-   `composite`: boolean\n-   `tvmix`: boolean\n-   `upmix21`: boolean\n-   `upmix41`: boolean\n-   `upmix61`: boolean\n-   `centeronly`: boolean\n-   `lfeonly`: boolean\n-   `rearonly`: boolean\n\nThe following methods are available.\n\n-   `get()`: Returns the current bus mode\n\nexample:\n\n```python\nvban.bus[4].mode.amix = True\n\nprint(vban.bus[2].mode.get())\n```\n\n##### Levels\n\nThe following properties are available.\n\n-   `all`\n\nexample:\n\n```python\nprint(vban.bus[0].levels.all)\n```\n\n`levels.all` will return -200.0 if no audio detected.\n\n### Command\n\nCertain \'special\' commands are defined by the API as performing actions rather than setting values. The following methods are available:\n\n-   `show()` : Bring Voiceemeter GUI to the front\n-   `shutdown()` : Shuts down the GUI\n-   `restart()` : Restart the audio engine\n-   `reset()`: Applies the `reset` config. (phys strip B1, virt strip A1, gains, comp, gate 0.0, mute, mono, solo, eq false)\n\nThe following properties are write only and accept boolean values.\n\n-   `showvbanchat`: boolean\n-   `lock`: boolean\n\nexample:\n\n```python\nvban.command.restart()\nvban.command.showvbanchat = true\n```\n\n### Multiple parameters\n\n-   `apply`\n    Set many strip/bus parameters at once, for example:\n\n```python\nvban.apply(\n    {\n        "strip-2": {"A1": True, "B1": True, "gain": -6.0},\n        "bus-2": {"mute": True},\n    }\n)\n```\n\nOr for each class you may do:\n\n```python\nvban.strip[0].apply(mute: true, gain: 3.2, A1: true)\nvban.vban.outstream[0].apply(on: true, name: \'streamname\', bit: 24)\n```\n\n## Config Files\n\n`vban.apply_config(<configname>)`\n\nYou may load config files in TOML format.\nThree example configs have been included with the package. Remember to save\ncurrent settings before loading a user config. To set one you may do:\n\n```python\nimport vban_cmd\nwith vban_cmd.api(\'banana\') as vban:\n    vban.apply_config(\'example\')\n```\n\nwill load a config file at configs/banana/example.toml for Voicemeeter Banana.\n\n## Events\n\nLevel updates are considered high volume, by default they are NOT listened for. Use `subs` keyword arg to initialize event updates.\n\nexample:\n\n```python\nimport vban_cmd\n# Listen for level updates\nopts = {\n    "ip": "<ip address>",\n    "streamname": "Command1",\n    "port": 6980,\n    "subs": {"ldirty": True},\n}\nwith vban_cmd.api(\'banana\', **opts) as vban:\n    ...\n```\n\n#### `vban.subject`\n\nUse the Subject class to register an app as event observer.\n\nThe following methods are available:\n\n-   `add`: registers an app as an event observer\n-   `remove`: deregisters an app as an event observer\n\nexample:\n\n```python\n# register the app self as an event observer\nself.vban.subject.add(self)\n```\n\n#### `vban.event`\n\nYou may also add/remove event subscriptions as necessary with the Event class.\n\nexample:\n\n```python\nvban.event.add("ldirty")\n\nvban.event.remove("pdirty")\n\n# get a list of currently subscribed\nprint(vban.event.get())\n```\n\n## VbanCmd class\n\n`vban_cmd.api(kind_id: str, **opts: dict)`\n\nYou may pass the following optional keyword arguments:\n\n-   `ip`: str, ip or hostname of remote machine\n-   `streamname`: str, name of the stream to connect to.\n-   `port`: int=6980, vban udp port of remote machine.\n-   `subs`: dict={"pdirty": True, "ldirty": False}, controls which updates to listen for.\n    -   `pdirty`: parameter updates\n    -   `ldirty`: level updates\n\n#### `vban.pdirty`\n\nTrue iff a parameter has been changed.\n\n#### `vban.ldirty`\n\nTrue iff a level value has been changed.\n\n#### `vban.sendtext(script)`\n\nSends a script block as a string request, for example:\n\n```python\nvban.sendtext("Strip[0].Mute=1;Bus[0].Mono=1")\n```\n\n#### `vban.public_packet`\n\nReturns a Voicemeeter rt data packet object. Designed to be used internally by the interface but available for parsing through this read only property object. States not guaranteed to be current (requires use of dirty parameters to confirm).\n\n### `Errors`\n\n-   `errors.VBANCMDError`: Base VMCMD error class.\n\n### `Tests`\n\nFirst make sure you installed the [development dependencies](https://github.com/onyx-and-iris/vban-cmd-python#installation)\n\nThen from tests directory:\n\n`pytest -v`\n\n## Resources\n\n-   [Voicemeeter VBAN TEXT](https://vb-audio.com/Voicemeeter/VBANProtocol_Specifications.pdf#page=19)\n\n-   [Voicemeeter RT Packet Service](https://vb-audio.com/Voicemeeter/VBANProtocol_Specifications.pdf#page=27)\n',
    'author': 'onyx-and-iris',
    'author_email': 'code@onyxandiris.online',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/onyx-and-iris/vban-cmd-python',
    'packages': packages,
    'package_data': package_data,
    'extras_require': extras_require,
    'python_requires': '>=3.10,<4.0',
}


setup(**setup_kwargs)
