import re
from getpass import getpass
from typing import Optional

from requests import get, post

from domains_api.constants import api_responses
from domains_api.exceptions import UserInstanceNotRecognised, UserNotSetup
from domains_api.file_handlers import FileHandlers
from domains_api.user import User


class IPChanger:
    fh = FileHandlers()

    def __init__(self, cli: bool = False):
        self.user: Optional[User] = None
        self.cli = cli
        if self.cli:
            self.run()

    def run(self):
        """Attempt to load user if no user and run check, updating DDNS if necessary."""
        try:
            if self.user is None:
                self.load_user()
            self.check_ip()
        except UserNotSetup:
            if self.cli:
                self.user_setup_wizard()

    def load_user(self, user_file: Optional[str] = None):
        """Load user from file or raise UserException"""
        try:
            if user_file:
                user = self.fh.load_user(user_file)
            else:
                try:
                    user = self.fh.load_user(self.fh.user_file)
                except FileNotFoundError:
                    return self.user_setup_wizard()
            if not isinstance(user, User):
                raise UserInstanceNotRecognised
            self.user = User.update_user_instance(user)
            self.user.send_notification(clear=True, log_fn=self.fh.log)
            if user_file is not None:
                self.fh.save_user(self.user)
        except UserInstanceNotRecognised:
            self.check_user()

    @staticmethod
    def get_ip() -> str:
        return get("https://api.ipify.org").text

    def check_ip(self):
        """Check last recorded IP address against current external IP address"""
        self.check_user()
        try:
            if (ip := self.get_ip()) != self.user.last_ip:
                self.user.last_ip = ip
                self.fh.save_user(self.user)
                self.parse_api_response(self.call_api(ip))
        except Exception as e:
            self.user.send_notification(self.user.last_ip, "error", e.__str__())

    def call_api(self, ip: str) -> str:
        return post(f"{self.user.req_url}&myip={ip}").text

    def parse_api_response(self, response: str) -> bool:
        """Parse response from Google Domains API call"""
        key = None
        for key in api_responses.keys():
            if key in response:
                break
        help_text = f'API response: {response}{": " + api_responses[key]["help_text"] if key is not None else ""}'
        status = api_responses[key]["status"] if key is not None else False
        if not status:
            help_text += (
                " ...see https://support.google.com/domains/answer/6147083?hl=en-CA "
                "for API documentation"
            )
        log_type = "info" if status else "error"
        self.fh.log(help_text, log_type)
        if "good" in response:
            ip = response.split()[1]
            self.user.send_notification(ip, log_fn=self.fh.log)
        elif "nochg" in response:
            pass
        else:
            self.user.send_notification(
                msg_type="error", error=f"{response}: {help_text}", log_fn=self.fh.log
            )
        return status

    def check_user(self):
        if self.user is None:
            raise UserNotSetup

    def force_change(self, ip: str = None):
        """Forces a change of IP and saves to the user instance"""
        if ip is not None:
            self.fh.log("***Forcing API call***", "info")
            pattern = r"(\d{1,3})\.(\d{1,3})\.(\d{1,3})\.(\d{1,3})"
            if re.search(pattern, ip.strip()):
                self.user.last_ip = ip
                self.fh.log(f"Overriding IP: {ip}", "info")
                self.fh.save_user(self.user)
            else:
                print(f"'{ip}' is not a recognised IPv4 address")
                print("Using last IP")
        self.parse_api_response(self.call_api(self.user.last_ip))

    def user_setup_wizard(self):
        """Set up user profile from command line input"""
        print("Performing user profile setup wizard....")
        self.user = User(
            domain=input(
                "What's the domain you wish to update? (include subdomain if relevant) "
            ),
            api_key=input("What's the autogenerated API key for this domain? "),
            api_sec=getpass("What's the autogenerated API secret for this domain? "),
            email_notifications=input(
                "Would you like to turn on email notifications for this domain? (You can change this later) "
                "[Y: yes to all; e: errors only; n: no emails] "
            ).lower()
            or "y",
        )
        if self.user.email_notifications != "n" and not self.user.gmail_address:
            self.user.email_wizard()
        self.fh.save_user(self.user)
