"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SchemaAsset = void 0;
const fs_1 = require("fs");
const os_1 = require("os");
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_s3_assets_1 = require("aws-cdk-lib/aws-s3-assets");
const change_case_1 = require("change-case");
class SchemaAsset extends aws_s3_assets_1.Asset {
    /**
     * Construct a new Schema Definition Asset that will be uploaded into S3
     * and then referenced in CloudFormation template.
     */
    constructor(scope, id, document, props) {
        const dirPath = SchemaAsset.resolveDir(props);
        SchemaAsset.ensureDir(dirPath);
        const fileName = SchemaAsset.resolveFileName(scope, id);
        const filePath = SchemaAsset.resolveFilePath(dirPath, fileName);
        const data = SchemaAsset.resolveSchemaTokens(scope, document);
        // S3 Asset requires a file path, so let's write the data to filesystem
        SchemaAsset.write(filePath, data);
        super(scope, id, {
            path: filePath,
        });
        this.cloudFormationDocument = data;
    }
    /**
     * Resolve a tokenized value in the context of the current stack.
     *
     * This is required for the S3 uploaded definition to work.
     */
    static resolveSchemaTokens(scope, document) {
        return aws_cdk_lib_1.Stack.of(scope).resolve(document);
    }
    /** Resolve the target directory for the schema asset. */
    static resolveDir(props) {
        return props?.dir || path_1.join(os_1.tmpdir(), '@alma-cdk', 'openapix');
    }
    /** The target directory must exists. */
    static ensureDir(path) {
        fs_1.mkdirSync(path, { recursive: true });
    }
    /** Resolve filename for the generated file. */
    static resolveFileName(scope, id) {
        // TODO ensure does not conflict with anything
        return `${change_case_1.paramCase(scope.node.path)}-${change_case_1.paramCase(id)}.schema.yaml`;
    }
    /** Resolve the full path for the generated file. */
    static resolveFilePath(dirPath, fileName) {
        return path_1.join(dirPath, fileName);
    }
    /** S3 Asset requires the data to be available on filesystem, so let's write it there. */
    static write(filePath, data) {
        // TODO investigate if we can perform this in-memory
        fs_1.writeFileSync(filePath, JSON.stringify(data), 'utf-8');
    }
}
exports.SchemaAsset = SchemaAsset;
//# sourceMappingURL=data:application/json;base64,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