"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiDefinition = void 0;
const apigateway = require("aws-cdk-lib/aws-apigateway");
const add_1 = require("../errors/add");
const schema_1 = require("../schema");
const utils_1 = require("./utils");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const omitDeep = require('omit-deep-lodash');
/** Represents an OpenAPI v3 definition asset. */
class ApiDefinition extends apigateway.ApiDefinition {
    /** Represents an OpenAPI v3 definition asset. */
    constructor(scope, props) {
        super();
        this.scope = scope;
        this.upload = props.upload === true;
        this.schema = this.resolveSource(props.source);
        // Handle injects/rejects
        this.schema.inject(props.injections);
        this.schema.reject(props.rejections);
        this.schema.rejectDeep(props.rejectionsDeep);
        // Configurate integrations
        this.configureValidators(props.validators);
        this.configureAuthorizers(props.authorizers);
        this.configurePaths(props.paths, props.defaultCors, props.defaultIntegration);
        // Finally expose the processed OpenApi v3 document
        this.document = this.schema.toDocument();
        this.validateDocument(this.document);
    }
    /** Configures API Gateway validators (if any). */
    configureValidators(validators = {}) {
        const keys = Object.keys(validators);
        // Do not assign x-amazon-apigateway-request-validators object if none provided
        if (keys.length < 1) {
            return;
        }
        const defaults = keys.filter(k => validators[k].default === true);
        // Ensure only single validator configured as default
        if (defaults.length > 1) {
            add_1.addError(this.scope, 'You may only configure one default validator');
            return;
        }
        // Configure the default validator if provided
        if (defaults.length === 1) {
            const defaultValidator = defaults[0];
            this.schema.set('x-amazon-apigateway-request-validator', defaultValidator);
        }
        // Omit the non-standard "default" field
        const cleaned = omitDeep(validators, 'default');
        this.schema.set('x-amazon-apigateway-request-validators', cleaned);
    }
    /**
     * Configure Authorizers within OpenApi `components.securitySchemes`.
     */
    configureAuthorizers(authorizers = []) {
        authorizers.map(a => {
            const authorizerComponentSecuritySchemePath = `components.securitySchemes.${a.id}`;
            if (!this.schema.has(authorizerComponentSecuritySchemePath)) {
                add_1.addError(this.scope, `Security Scheme ${a.id} not found in OpenAPI Definition`);
                return;
            }
            /**
            * if current authorizer is defined for whole api, add security to all paths
            */
            const schemaSecurity = this.schema.get('security');
            // check if security includes authorizer
            const schemaSecurityIncludesCurrentAuthorizer = Array.isArray(schemaSecurity) && schemaSecurity.some(s => Object.keys(s).includes(a.id));
            if (schemaSecurityIncludesCurrentAuthorizer) {
                // loop schema paths
                const schemaPaths = utils_1.getSchemaPaths(this.schema);
                if (schemaPaths !== undefined) {
                    Object.keys(schemaPaths).forEach(path => {
                        // loop methods
                        const schemaPathMethods = utils_1.getMethodsFromSchemaPath(schemaPaths[path]);
                        Object.keys(schemaPathMethods).forEach(method => {
                            // check if method has security
                            const methodSecurity = this.schema.get(`paths.${path}.${method}.security`);
                            if (methodSecurity && Array.isArray(methodSecurity)) {
                                // check if security includes authorizer
                                const methodSecurityIncludesCurrentAuthorizer = methodSecurity.some(s => Object.keys(s).includes(a.id));
                                if (!methodSecurityIncludesCurrentAuthorizer) {
                                    this.schema.set(`paths.${path}.${method}.security`, [...methodSecurity, { [a.id]: [] }]);
                                }
                            }
                            else {
                                this.schema.set(`paths.${path}.${method}.security`, [{ [a.id]: [] }]);
                            }
                        });
                    });
                }
            }
            const authorizer = this.schema.get(authorizerComponentSecuritySchemePath);
            authorizer['x-amazon-apigateway-authtype'] = a.xAmazonApigatewayAuthtype;
            authorizer['x-amazon-apigateway-authorizer'] = a.xAmazonApigatewayAuthorizer;
            this.schema.set(authorizerComponentSecuritySchemePath, authorizer);
        });
    }
    /**
     * Configure all `x-amazon-apigateway-integration` values within OpenApi `paths`.
     */
    configurePaths(paths = {}, defaultCors, defaultIntegration) {
        const schemaPaths = utils_1.getSchemaPaths(this.schema);
        // Check that schema has paths object
        if (typeof schemaPaths === 'undefined') {
            add_1.addError(this.scope, 'OpenAPI Definition does not have paths object');
            return;
        }
        // Loop through paths to ensure all paths are defined in OpenAPI schema
        Object.keys(paths).forEach(path => {
            if (!schemaPaths[path]) {
                const message = `Path ${path} not found in OpenAPI Definition. Check paths-props in definition.`;
                add_1.addError(this.scope, message);
            }
        });
        // Loop through all schema paths
        Object.keys(schemaPaths).map((path) => {
            if (!defaultIntegration && !paths[path]) {
                const message = `Missing integration for path: ${path}. Check paths-props in definition, or add a default integration.`;
                add_1.addError(this.scope, message);
                return;
            }
            if (typeof defaultCors !== 'undefined') {
                this.configureDefaultCors(path, defaultCors);
            }
            const methods = paths[path];
            this.configurePathMethods(path, schemaPaths[path], methods, defaultIntegration, defaultCors);
        });
    }
    configureDefaultCors(path, defaultCors) {
        this.schema.set(`paths.${path}.options`, {
            'summary': 'CORS support',
            'description': 'Enable CORS by returning correct headers',
            'consumes': [
                'application/json',
            ],
            'produces': [
                'application/json',
            ],
            'tags': [
                'CORS',
            ],
            'responses': {
                204: {
                    description: 'Default response for CORS method',
                    headers: {
                        'Access-Control-Allow-Headers': {
                            type: 'string',
                        },
                        'Access-Control-Allow-Methods': {
                            type: 'string',
                        },
                        'Access-Control-Allow-Origin': {
                            type: 'string',
                        },
                    },
                },
            },
            'x-amazon-apigateway-integration': defaultCors.xAmazonApigatewayIntegration,
        });
    }
    /**
     * Configure `x-amazon-apigateway-integration` for given method.
     */
    configurePathMethods(schemaPathName, schemaPath, methods = {}, defaultIntegration, defaultCors) {
        // Loop through given methods to ensure they are defined
        // and dont have an existing integration
        Object.keys(methods).map((method) => {
            const methodName = method.toLowerCase();
            this.ensureMethodExists(schemaPathName, methodName);
            this.ensureNoIntegrationAlready(schemaPathName, methodName);
        });
        const schemaPathMethods = utils_1.getMethodsFromSchemaPath(schemaPath);
        // Look through all schema path methods
        Object.keys(schemaPathMethods).map((schemaPathMethod) => {
            const method = methods[schemaPathMethod];
            // Do not process options method because it has been modified already
            // and no override method is present
            if (defaultCors && schemaPathMethod === 'options' && !method) {
                return;
            }
            if (!defaultIntegration && !method) {
                const message = `OpenAPI schema has an unhandled path method: ${schemaPathName}/${schemaPathMethod}`;
                add_1.addError(this.scope, message);
                return;
            }
            let integration;
            // Generate mock integration if requested
            if (defaultIntegration && !method) {
                integration = defaultIntegration;
            }
            else {
                integration = method;
            }
            const methodPath = `paths['${schemaPathName}']['${schemaPathMethod}']`;
            const validator = integration.validator;
            if (typeof validator === 'string') {
                this.schema.set(`${methodPath}['x-amazon-apigateway-request-validator']`, validator);
            }
            this.schema.set(`${methodPath}['x-amazon-apigateway-integration']`, integration.xAmazonApigatewayIntegration);
        });
    }
    /**
     * End-User can provide the OpenAPI definition either as a path to a file or
     * as an Asset. This method handles that and always returns Asset Source.
     */
    resolveSource(source) {
        if (typeof source === 'string') {
            return schema_1.Schema.fromAsset(source);
        }
        return source;
    }
    /**
     * Implement the functionality of exposing the API definition, either as `s3Location` or as `inlineDefinition`.
     *
     * Called by `apigateway.SpecRestApi`.
     *
     * @see https://github.com/aws/aws-cdk/blob/87dd2a6eb0b8208e49ff5f0cc8486ad58410d3ef/packages/%40aws-cdk/aws-apigateway/lib/restapi.ts#L636
     * @see https://github.com/aws/aws-cdk/blob/87dd2a6eb0b8208e49ff5f0cc8486ad58410d3ef/packages/%40aws-cdk/aws-apigateway/lib/api-definition.ts#L81-L88
     */
    bind(_) {
        if (this.upload === true) {
            const asset = this.schema.toAsset(this.scope, 'SchemaAsset');
            return {
                s3Location: {
                    bucket: asset.bucket.bucketArn,
                    key: asset.s3ObjectKey,
                },
            };
        }
        return {
            inlineDefinition: this.schema.toDocument(),
        };
    }
    /**
     * Ensures OpenAPI definition contains a given method for the path.
     */
    ensureMethodExists(path, method) {
        const value = this.schema.get(`paths[${path}][${method}]`);
        if (typeof value === 'undefined') {
            const message = `OpenAPI schema is missing method ${method} for path: ${path}`;
            add_1.addError(this.scope, message);
        }
    }
    /**
     * Ensures OpenAPI definition does not already have
     * `x-amazon-apigateway-integration` configuration for given method for the path.
     */
    ensureNoIntegrationAlready(path, method) {
        const value = this.schema.get(`paths[${path}][${method}]['x-amazon-apigateway-integration']`);
        if (typeof value !== 'undefined') {
            const message = `OpenAPI schema already has x-amazon-apigateway-integration configuration for method ${method} in path: ${path}`;
            add_1.addError(this.scope, message);
        }
    }
    /** Validate final OpenApi v3 document. */
    validateDocument(document) {
        if (typeof (document) !== 'object') {
            add_1.addError(this.scope, 'definition should be of type object');
        }
        if (Object.keys(document).length === 0) {
            add_1.addError(this.scope, 'JSON definition cannot be empty');
        }
    }
}
exports.ApiDefinition = ApiDefinition;
//# sourceMappingURL=data:application/json;base64,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