"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.startExecution = exports.cfnResponseFailed = exports.cfnResponseSuccess = exports.CREATE_FAILED_PHYSICAL_ID_MARKER = void 0;
const client_sfn_1 = require("@aws-sdk/client-sfn"); // eslint-disable-line import/no-extraneous-dependencies
const http_1 = require("./http");
exports.CREATE_FAILED_PHYSICAL_ID_MARKER = 'AWSCDK::StateMachineProvider::CREATE_FAILED';
async function cfnResponseSuccess(event) {
    console.log('Event: %j', event);
    await http_1.respond('SUCCESS', {
        ...event.Input,
        PhysicalResourceId: event.Output?.PhysicalResourceId ?? event.Input.PhysicalResourceId ?? event.Input.RequestId,
        Data: event.Output?.Data ?? {},
        NoEcho: event.Output?.NoEcho,
    });
}
exports.cfnResponseSuccess = cfnResponseSuccess;
async function cfnResponseFailed(event) {
    console.log('Event: %j', event);
    const parsedCause = JSON.parse(event.Cause);
    const executionResult = {
        ...parsedCause,
        Input: JSON.parse(parsedCause.Input),
    };
    console.log('Execution result: %j', executionResult);
    let physicalResourceId = executionResult.Output?.PhysicalResourceId ?? executionResult.Input.PhysicalResourceId;
    if (!physicalResourceId) {
        // special case: if CREATE fails, which usually implies, we usually don't
        // have a physical resource id. in this case, the subsequent DELETE
        // operation does not have any meaning, and will likely fail as well. to
        // address this, we use a marker so the provider framework can simply
        // ignore the subsequent DELETE.
        if (executionResult.Input.RequestType === 'Create') {
            console.log('CREATE failed, responding with a marker physical resource id so that the subsequent DELETE will be ignored');
            physicalResourceId = exports.CREATE_FAILED_PHYSICAL_ID_MARKER;
        }
        else {
            console.log(`ERROR: Malformed event. "PhysicalResourceId" is required: ${JSON.stringify(event)}`);
        }
    }
    await http_1.respond('FAILED', {
        ...executionResult.Input,
        Reason: `${event.Error}: ${event.Cause}`,
        PhysicalResourceId: physicalResourceId,
    });
}
exports.cfnResponseFailed = cfnResponseFailed;
const stepFunctionsClient = new client_sfn_1.SFNClient({});
async function startExecution(event) {
    try {
        console.log('Event: %j', event);
        if (!process.env.STATE_MACHINE_ARN) {
            throw new Error('Missing STATE_MACHINE_ARN.');
        }
        // ignore DELETE event when the physical resource ID is the marker that
        // indicates that this DELETE is a subsequent DELETE to a failed CREATE
        // operation.
        if (event.RequestType === 'Delete' && event.PhysicalResourceId === exports.CREATE_FAILED_PHYSICAL_ID_MARKER) {
            console.log('ignoring DELETE event caused by a failed CREATE event');
            await http_1.respond('SUCCESS', event);
            return;
        }
        await stepFunctionsClient.send(new client_sfn_1.StartExecutionCommand({
            stateMachineArn: process.env.STATE_MACHINE_ARN,
            input: JSON.stringify(event),
        }));
    }
    catch (err) {
        console.log(err);
        await http_1.respond('FAILED', {
            ...event,
            Reason: err.message,
        });
    }
}
exports.startExecution = startExecution;
//# sourceMappingURL=data:application/json;base64,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