# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['hermes', 'hermes.typeo']

package_data = \
{'': ['*']}

install_requires = \
['toml>=0.10.2,<0.11.0']

setup_kwargs = {
    'name': 'hermes.typeo',
    'version': '0.1.3',
    'description': 'Utilities for running functions as scripts',
    'long_description': '# Typeo\n## (Pronounced like the abbreviation of "typographical error")\nTurn annotated functions into command line scripts with a single line of code, and keep all your documentation!\n\nUses type annotations on functions to parse command line arguments, and strips help strings from function documentation.\n\n\n## Basic Usage\n\nSay we have a file `say_hello.py` that looks like:\n```python\ndef say_hello(name: str, friendliness: int):\n    """Say hello with various degrees of friendliness\n\n    Args:\n        name:\n            The name of the person to greet\n        friendliness:\n            The level of friendliness to greet them with\n    """\n\n    if friendliness == 0:\n        print("hey.")\n    elif friendliness == 1:\n        print(f"Hi {name}")\n    elif friendliness > 1:\n        print(f"Hello {name}!")\n    else:\n        raise ValueError(\n            "Friendliness level cannot be less than 0"\n        )\n```\n\nThis function can be run as a command line utility by just adding\n\n```python\nfrom hermes.typeo import typeo\n\n\n@typeo\ndef say_hello(name: str, friendliness: int = 1):\n    """Say hello to someone with various degrees of friendliness\n\n    Args:\n        name:\n            The name of the person to greet\n        friendliness:\n            The level of friendliness to greet them with\n    """\n\n    if friendliness == 0:\n        print("hey.")\n    elif friendliness == 1:\n        print(f"Hi {name}")\n    elif friendliness > 1:\n        print(f"Hello {name}!")\n    else:\n        raise ValueError(\n            "Friendliness level cannot be less than 0"\n        )\n\n\nif __name__ == "__main__":\n    say_hello()\n```\n\nNow when we run from the command line:\n```console\n$ python say_hello.py -h\nusage: say_hello [-h] --name NAME [--friendliness FRIENDLINESS]\n\nSay hello to someone with various degrees of friendliness\n\noptional arguments:\n  -h, --help            show this help message and exit\n  --name NAME           The name of the person to greet (default: None)\n  --friendliness FRIENDLINESS\n                        The level of friendliness to greet them with (default: 1)\n\n$ python say_hello.py --name Thom\nHi Thom\n\n$ python say_hello.py --name Thom --friendliness 2\nHello Thom!\n\n$ python say_hello.py --friendliness 0\nusage: say_hello [-h] --name NAME [--friendliness FRIENDLINESS]\nsay_hello: error: the following arguments are required: --name\n```\n\nNote that we can still import `say_hello` in other scripts and call it with regular arguments, and its behavior won\'t be impacted. `typeo` works by only reading from the command line if no arguments are passed in.\n\n```python\nfrom say_hello import say_hello\n\n\n# prints "Hi Thom"\nsay_hello("Thom", 1)\n```\n\nNote also that we can drop the `if __name__ == "__main__"` syntax if we integrate with Poetry package scripts. For example, if my `pyproject.toml` in the directory where I host `say_hello.py` has a section like:\n\n```toml\n[tool.poetry.scripts]\ngreet = "say_hello:say_hello"\n```\n\nThen I can drop the `if __name__ == "__main__"` from `say_hello.py` and run my script like this\n\n```console\n$ poetry run greet --name Thom\nHi Thom\n```\n\n## Subcommands\n\nWe can also add subcommands to our scripts. Let\'s say `greet.py` looks like\n\n```python\nfrom hermes.typeo import typeo\n\n\ndef validate_friendliness(friendliness: int):\n    if friendliness < 0:\n        raise ValueError(\n            "Friendliness level cannot be less than 0"\n        )\n\n\ndef say_goodbye(name: str, friendliness: int = 1):\n    """Say goodbye to someone with various degrees of friendliness\n\n    Args:\n        name:\n            The name of the person to bid farewell\n        friendliness:\n            The level of friendliness to bid them farewell with\n    """\n\n    validate_friendliness(friendliness)\n\n    if friendliness == 0:\n        print("bye.")\n    elif friendliness == 1:\n        print(f"Goodbye {name}")\n    else:\n        print(f"So long {name}!")\n\n\ndef say_hello(name: str, friendliness: int = 1):\n    """Say hello to someone with various degrees of friendliness\n\n    Args:\n        name:\n            The name of the person to greet\n        friendliness:\n            The level of friendliness to greet them with\n    """\n\n    validate_friendliness(friendliness)\n    if friendliness == 0:\n        print("hey.")\n    elif friendliness == 1:\n        print(f"Hi {name}")\n    else:\n        print(f"Hello {name}!")\n\n\n@typeo(hello=say_hello, goodbye=say_goodbye)\ndef greet(greeter: str):\n    print(f"This is a greeting from {greeter}:")\n\n\nif __name__ == "__main__":\n    greet()\n```\n\n```console\n$ python greet.py -h\nusage: greet [-h] --greeter GREETER {hello,goodbye} ...\n\npositional arguments:\n  {hello,goodbye}\n\noptional arguments:\n  -h, --help         show this help message and exit\n  --greeter GREETER\n\n$ python greet.py hello -h\nusage: greet hello [-h] --name NAME [--friendliness FRIENDLINESS]\n\nSay hello to someone with various degrees of friendliness\n\n\n\noptional arguments:\n  -h, --help            show this help message and exit\n  --name NAME           The name of the person to greet (default: None)\n  --friendliness FRIENDLINESS\n                        The level of friendliness to greet them with (default: 1)\n\n$ python greet.py goodbye -h\nusage: greet goodbye [-h] --name NAME [--friendliness FRIENDLINESS]\n\nSay goodbye to someone with various degrees of friendliness\n\n\n\noptional arguments:\n  -h, --help            show this help message and exit\n  --name NAME           The name of the person to bid farewell (default: None)\n  --friendliness FRIENDLINESS\n                        The level of friendliness to bid them farewell with (default: 1)\n\n$ python greet.py --greeter Jonny hello --name Thom\nThis is a greeting from Jonny:\nHi Thom\n\n$ python greet.py --greeter Phil goodbye --name Jonny --friendliness 2\nThis is a greeting from Phil:\nSo long Jonny!\n```\n',
    'author': 'Alec Gunny',
    'author_email': 'alec.gunny@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'None',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
