"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ColumnEncoding = exports.TableSortStyle = exports.TableDistStyle = exports.Table = exports.TableAction = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/* eslint-disable import/no-extraneous-dependencies */
const cdk = require("aws-cdk-lib");
const cx_api_1 = require("aws-cdk-lib/cx-api");
const constructs_1 = require("constructs");
const database_query_1 = require("./private/database-query");
const handler_name_1 = require("./private/database-query-provider/handler-name");
const util_1 = require("./private/database-query-provider/util");
/**
 * An action that a Redshift user can be granted privilege to perform on a table.
 */
var TableAction;
(function (TableAction) {
    /**
     * Grants privilege to select data from a table or view using a SELECT statement.
     */
    TableAction[TableAction["SELECT"] = 0] = "SELECT";
    /**
     * Grants privilege to load data into a table using an INSERT statement or a COPY statement.
     */
    TableAction[TableAction["INSERT"] = 1] = "INSERT";
    /**
     * Grants privilege to update a table column using an UPDATE statement.
     */
    TableAction[TableAction["UPDATE"] = 2] = "UPDATE";
    /**
     * Grants privilege to delete a data row from a table.
     */
    TableAction[TableAction["DELETE"] = 3] = "DELETE";
    /**
     * Grants privilege to drop a table.
     */
    TableAction[TableAction["DROP"] = 4] = "DROP";
    /**
     * Grants privilege to create a foreign key constraint.
     *
     * You need to grant this privilege on both the referenced table and the referencing table; otherwise, the user can't create the constraint.
     */
    TableAction[TableAction["REFERENCES"] = 5] = "REFERENCES";
    /**
     * Grants all available privileges at once to the specified user or user group.
     */
    TableAction[TableAction["ALL"] = 6] = "ALL";
})(TableAction = exports.TableAction || (exports.TableAction = {}));
class TableBase extends constructs_1.Construct {
    grant(user, ...actions) {
        user.addTablePrivileges(this, ...actions);
    }
}
/**
 * A table in a Redshift cluster.
 */
class Table extends TableBase {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_redshift_alpha_TableProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Table);
            }
            throw error;
        }
        this.addColumnIds(props.tableColumns);
        this.validateDistKeyColumns(props.tableColumns);
        if (props.distStyle) {
            this.validateDistStyle(props.distStyle, props.tableColumns);
        }
        if (props.sortStyle) {
            this.validateSortStyle(props.sortStyle, props.tableColumns);
        }
        this.tableColumns = props.tableColumns;
        this.cluster = props.cluster;
        this.databaseName = props.databaseName;
        const useColumnIds = !!cdk.FeatureFlags.of(this).isEnabled(cx_api_1.REDSHIFT_COLUMN_ID);
        this.resource = new database_query_1.DatabaseQuery(this, 'Resource', {
            removalPolicy: cdk.RemovalPolicy.RETAIN,
            ...props,
            handler: handler_name_1.HandlerName.Table,
            properties: {
                tableName: {
                    prefix: props.tableName ?? cdk.Names.uniqueId(this),
                    generateSuffix: !props.tableName ? 'true' : 'false',
                },
                tableColumns: this.tableColumns,
                distStyle: props.distStyle,
                sortStyle: props.sortStyle ?? this.getDefaultSortStyle(props.tableColumns),
                tableComment: props.tableComment,
                useColumnIds,
            },
        });
        this.tableName = this.resource.ref;
    }
    /**
     * Specify a Redshift table using a table name and schema that already exists.
     */
    static fromTableAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_redshift_alpha_TableAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromTableAttributes);
            }
            throw error;
        }
        return new class extends TableBase {
            constructor() {
                super(...arguments);
                this.tableName = attrs.tableName;
                this.tableColumns = attrs.tableColumns;
                this.cluster = attrs.cluster;
                this.databaseName = attrs.databaseName;
            }
        }(scope, id);
    }
    /**
     * Apply the given removal policy to this resource
     *
     * The Removal Policy controls what happens to this resource when it stops
     * being managed by CloudFormation, either because you've removed it from the
     * CDK application or because you've made a change that requires the resource
     * to be replaced.
     *
     * The resource can be destroyed (`RemovalPolicy.DESTROY`), or left in your AWS
     * account for data recovery and cleanup later (`RemovalPolicy.RETAIN`).
     *
     * This resource is retained by default.
     */
    applyRemovalPolicy(policy) {
        this.resource.applyRemovalPolicy(policy);
    }
    validateDistKeyColumns(columns) {
        try {
            util_1.getDistKeyColumn(columns);
        }
        catch (err) {
            throw new Error('Only one column can be configured as distKey.');
        }
    }
    validateDistStyle(distStyle, columns) {
        const distKeyColumn = util_1.getDistKeyColumn(columns);
        if (distKeyColumn && distStyle !== TableDistStyle.KEY) {
            throw new Error(`Only 'TableDistStyle.KEY' can be configured when distKey is also configured. Found ${distStyle}`);
        }
        if (!distKeyColumn && distStyle === TableDistStyle.KEY) {
            throw new Error('distStyle of "TableDistStyle.KEY" can only be configured when distKey is also configured.');
        }
    }
    validateSortStyle(sortStyle, columns) {
        const sortKeyColumns = util_1.getSortKeyColumns(columns);
        if (sortKeyColumns.length === 0 && sortStyle !== TableSortStyle.AUTO) {
            throw new Error(`sortStyle of '${sortStyle}' can only be configured when sortKey is also configured.`);
        }
        if (sortKeyColumns.length > 0 && sortStyle === TableSortStyle.AUTO) {
            throw new Error(`sortStyle of '${TableSortStyle.AUTO}' cannot be configured when sortKey is also configured.`);
        }
    }
    getDefaultSortStyle(columns) {
        const sortKeyColumns = util_1.getSortKeyColumns(columns);
        return (sortKeyColumns.length === 0) ? TableSortStyle.AUTO : TableSortStyle.COMPOUND;
    }
    addColumnIds(columns) {
        const columnIds = new Set();
        for (const column of columns) {
            if (column.id) {
                if (columnIds.has(column.id)) {
                    throw new Error(`Column id '${column.id}' is not unique.`);
                }
                columnIds.add(column.id);
            }
        }
    }
}
exports.Table = Table;
_a = JSII_RTTI_SYMBOL_1;
Table[_a] = { fqn: "@aws-cdk/aws-redshift-alpha.Table", version: "2.68.0-alpha.0" };
/**
 * The data distribution style of a table.
 */
var TableDistStyle;
(function (TableDistStyle) {
    /**
     *  Amazon Redshift assigns an optimal distribution style based on the table data
     */
    TableDistStyle["AUTO"] = "AUTO";
    /**
     * The data in the table is spread evenly across the nodes in a cluster in a round-robin distribution.
     */
    TableDistStyle["EVEN"] = "EVEN";
    /**
     * The data is distributed by the values in the DISTKEY column.
     */
    TableDistStyle["KEY"] = "KEY";
    /**
     * A copy of the entire table is distributed to every node.
     */
    TableDistStyle["ALL"] = "ALL";
})(TableDistStyle = exports.TableDistStyle || (exports.TableDistStyle = {}));
/**
 * The sort style of a table.
 */
var TableSortStyle;
(function (TableSortStyle) {
    /**
     * Amazon Redshift assigns an optimal sort key based on the table data.
     */
    TableSortStyle["AUTO"] = "AUTO";
    /**
     * Specifies that the data is sorted using a compound key made up of all of the listed columns,
     * in the order they are listed.
     */
    TableSortStyle["COMPOUND"] = "COMPOUND";
    /**
     * Specifies that the data is sorted using an interleaved sort key.
     */
    TableSortStyle["INTERLEAVED"] = "INTERLEAVED";
})(TableSortStyle = exports.TableSortStyle || (exports.TableSortStyle = {}));
/**
 * The compression encoding of a column.
 *
 * @see https://docs.aws.amazon.com/redshift/latest/dg/c_Compression_encodings.html
 */
var ColumnEncoding;
(function (ColumnEncoding) {
    /**
     * Amazon Redshift assigns an optimal encoding based on the column data.
     * This is the default.
     */
    ColumnEncoding["AUTO"] = "AUTO";
    /**
     * The column is not compressed.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/c_Raw_encoding.html
     */
    ColumnEncoding["RAW"] = "RAW";
    /**
     * The column is compressed using the AZ64 algorithm.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/az64-encoding.html
     */
    ColumnEncoding["AZ64"] = "AZ64";
    /**
     * The column is compressed using a separate dictionary for each block column value on disk.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/c_Byte_dictionary_encoding.html
     */
    ColumnEncoding["BYTEDICT"] = "BYTEDICT";
    /**
     * The column is compressed based on the difference between values in the column.
     * This records differences as 1-byte values.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/c_Delta_encoding.html
     */
    ColumnEncoding["DELTA"] = "DELTA";
    /**
     * The column is compressed based on the difference between values in the column.
     * This records differences as 2-byte values.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/c_Delta_encoding.html
     */
    ColumnEncoding["DELTA32K"] = "DELTA32K";
    /**
     * The column is compressed using the LZO algorithm.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/lzo-encoding.html
     */
    ColumnEncoding["LZO"] = "LZO";
    /**
     * The column is compressed to a smaller storage size than the original data type.
     * The compressed storage size is 1 byte.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/c_MostlyN_encoding.html
     */
    ColumnEncoding["MOSTLY8"] = "MOSTLY8";
    /**
     * The column is compressed to a smaller storage size than the original data type.
     * The compressed storage size is 2 bytes.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/c_MostlyN_encoding.html
     */
    ColumnEncoding["MOSTLY16"] = "MOSTLY16";
    /**
     * The column is compressed to a smaller storage size than the original data type.
     * The compressed storage size is 4 bytes.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/c_MostlyN_encoding.html
     */
    ColumnEncoding["MOSTLY32"] = "MOSTLY32";
    /**
     * The column is compressed by recording the number of occurrences of each value in the column.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/c_Runlength_encoding.html
     */
    ColumnEncoding["RUNLENGTH"] = "RUNLENGTH";
    /**
     * The column is compressed by recording the first 245 unique words and then using a 1-byte index to represent each word.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/c_Text255_encoding.html
     */
    ColumnEncoding["TEXT255"] = "TEXT255";
    /**
     * The column is compressed by recording the first 32K unique words and then using a 2-byte index to represent each word.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/c_Text255_encoding.html
     */
    ColumnEncoding["TEXT32K"] = "TEXT32K";
    /**
     * The column is compressed using the ZSTD algorithm.
     *
     * @see https://docs.aws.amazon.com/redshift/latest/dg/zstd-encoding.html
     */
    ColumnEncoding["ZSTD"] = "ZSTD";
})(ColumnEncoding = exports.ColumnEncoding || (exports.ColumnEncoding = {}));
//# sourceMappingURL=data:application/json;base64,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