# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['tgintegration', 'tgintegration.containers', 'tgintegration.utils']

package_data = \
{'': ['*']}

install_requires = \
['pyrogram>=1.0.0,<2.0.0', 'typing-extensions>=3.7.4,<4.0.0']

setup_kwargs = {
    'name': 'tgintegration',
    'version': '1.1.0',
    'description': 'Integration test and automation library for Telegram Messenger Bots based on Pyrogram.',
    'long_description': 'TgIntegration\n=============\n\nAn integration test and automation library for [Telegram Bots](https://core.telegram.org/bots)\nbased on [Pyrogram](https://github.com/pyrogram/pyrogram).\n<br />**Test your bot in realtime scenarios!**\n\n[Features](#features) • [Requirements](#requirements) • [Installation](#installation) • [**Quick Start Guide**](#quick-start-guide) • [Test Frameworks](#integrating-with-test-frameworks)\n\n- 📖 [Documentation](https://josxa.github.io/tgintegration/)\n- 👥 [Telegram Chat](https://t.me/TgIntegration)\n- 📄 Free software: [MIT License](https://tldrlegal.com/license/mit-license)\n\nFeatures\n--------\n\n▶️ [**See it in action!** 🎬](https://josxa.github.io/tgintegration/introduction/#see-it-in-action)\n\n- 👤 Log into a Telegram user account and interact with bots or other users\n- ✅ Write **realtime integration tests** to ensure that your bot works as expected! ▶️ [Pytest examples](https://github.com/JosXa/tgintegration/tree/master/examples/pytest)\n- ⚡️ **Automate any interaction** on Telegram! ▶️ [Automation examples](https://github.com/JosXa/tgintegration/tree/master/examples/automation)\n- 🛡 Fully typed for safety and **autocompletion** with your favorite IDE\n- 🐍 Built for modern Python (3.7+) with high test coverage\n\n\nPrerequisites\n-------------\n\n[Same as Pyrogram](https://github.com/pyrogram/pyrogram#requirements):\n\n- Python **3.7** or higher.\n- A [Telegram API key](https://docs.pyrogram.ml/start/ProjectSetup#api-keys).\n- A user session (seeing things happen in your own account is great for getting started)\n\n\nInstallation\n------------\n\nAll hail pip!\n\n$ `pip install tgintegration --upgrade`\n\n<br />\n<details><summary><i>Feeling adventurous?</i></summary><p>\n\nFor bleeding edge, install the master branch:\n\n$ `pip install git+https://github.com/JosXa/tgintegration.git`\n\n</p></details>\n\n\nQuick Start Guide\n-----------------\n\n_You can [follow along by running the example](https://github.com/JosXa/tgintegration/blob/master/examples/readme_example/readmeexample.py) ([README](https://github.com/JosXa/tgintegration/blob/master/examples/README.md))_\n\n#### Setup\n\nSuppose we want to write integration tests for [@BotListBot](https://t.me/BotListBot) by sending it a couple of\nmessages and checking that it responds the way it should.\n\nAfter [configuring a Pyrogram **user client**](https://docs.pyrogram.org/intro/setup),\nlet\'s start by creating a `BotController`:\n\n``` python\nfrom tgintegration import BotController\n\ncontroller = BotController(\n    peer="@BotListBot",      # The bot under test is https://t.me/BotListBot 🤖\n    client=client,           # This assumes you already have a Pyrogram user client available\n    max_wait=8,              # Maximum timeout for responses (optional)\n    wait_consecutive=2,      # Minimum time to wait for more/consecutive messages (optional)\n    raise_no_response=True,  # Raise `InvalidResponseError` when no response is received (defaults to True)\n    global_action_delay=2.5  # Choosing a rather high delay so we can observe what\'s happening (optional)\n)\n\nawait controller.clear_chat()  # Start with a blank screen (⚠️)\n```\n\nNow, let\'s send `/start` to the bot and wait until exactly three messages have been received by using the asynchronous `collect` context manager:\n\n``` python\nasync with controller.collect(count=3) as response:\n    await controller.send_command("start")\n\nassert response.num_messages == 3  # Three messages received, bundled under a `Response` object\nassert response.messages[0].sticker  # The first message is a sticker\n```\n\nThe result should look like this:\n\n![image](https://raw.githubusercontent.com/JosXa/tgintegration/master/docs/assets/start_botlistbot.png)\n\nExamining the buttons in the response...\n\n``` python\n# Get first (and only) inline keyboard from the replies\ninline_keyboard = response.inline_keyboards[0]\n\n# Three buttons in the first row\nassert len(inline_keyboard.rows[0]) == 3\n```\n\nWe can also press the inline keyboard buttons, for example based on a regular expression:\n\n``` python\nexamples = await inline_keyboard.click(pattern=r".*Examples")\n```\n\nAs the bot edits the message, `.click()` automatically listens for "message edited" updates and returns\nthe new state as another `Response`.\n\n![image](https://raw.githubusercontent.com/JosXa/tgintegration/master/docs/assets/examples_botlistbot.png)\n\n``` python\nassert "Examples for contributing to the BotList" in examples.full_text\n```\n\n#### Error handling\n\nSo what happens when we send an invalid query or the peer fails to respond?\n\nThe following instruction will raise an `InvalidResponseError` after `controller.max_wait` seconds.\nThis is because we passed `raise_no_response=True` during controller initialization.\n\n``` python\ntry:\n    async with controller.collect():\n        await controller.send_command("ayylmao")\nexcept InvalidResponseError:\n    pass  # OK\n```\n\nLet\'s explicitly set `raise_` to `False` so that no exception occurs:\n\n``` python\nasync with controller.collect(raise_=False) as response:\n    await client.send_message(controller.peer_id, "Henlo Fren")\n```\n\nIn this case, _tgintegration_ will simply emit a warning, but you can still assert\nthat no response has been received by using the `is_empty` property:\n\n``` python\nassert response.is_empty\n```\n\n\nIntegrating with Test Frameworks\n--------------------------------\n\n### [pytest](https://docs.pytest.org/en/stable/index.html)\n\nPytest is the recommended test framework for use with _tgintegration_. You can\n[browse through several examples](https://github.com/JosXa/tgintegration/tree/master/examples/pytest)\nand _tgintegration_ also uses pytest for its own test suite.\n\n### unittest\n\nI haven\'t tried out the builtin `unittest` library in combination with _tgintegration_ yet,\nbut theoretically I don\'t see any problems with it.\nIf you do decide to try it, it would be awesome if you could tell me about your\nexperience and whether anything could be improved 🙂\nLet us know at 👉 https://t.me/TgIntegration or in an issue.\n',
    'author': 'JosXa',
    'author_email': 'joscha.goetzer@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/JosXa/tgintegration',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0.0',
}


setup(**setup_kwargs)
