"""
The pycity_scheduling framework


Copyright (C) 2022,
Institute for Automation of Complex Power Systems (ACS),
E.ON Energy Research Center (E.ON ERC),
RWTH Aachen University

Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated
documentation files (the "Software"), to deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit
persons to whom the Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all copies or substantial portions of the
Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
"""


import numpy as np
import pyomo.environ as pyomo
import pycity_base.classes.supply.combined_heat_power as chp

from pycity_scheduling.classes.thermal_entity_heating import ThermalEntityHeating
from pycity_scheduling.classes.electrical_entity import ElectricalEntity
from pycity_scheduling.util.generic_constraints import LowerActivationLimit


class CombinedHeatPower(ThermalEntityHeating, ElectricalEntity, chp.CHP):
    """
    Extension of pyCity_base class CHP for scheduling purposes.

    Parameters
    ----------
    environment : Environment
        Common to all other objects. Includes time and weather instances.
    p_th_nom : float
        Nominal thermal power output in [kW].
    p_el_nom : float, optional
        Nominal electrical power output in [kW]. Defaults to `p_th_nom`.
    eta : float, optional
        Total efficiency of the CHP (without unit).
    lower_activation_limit : float, optional (only adhered to in integer mode)
        Must be in [0, 1]. Lower activation limit of the CHP as a
        percentage of the rated power. When the CHP is in operation, its power
        must be zero or between the lower activation limit and its rated
        power.

        - `lower_activation_limit = 0`: Linear behavior
        - `lower_activation_limit = 1`: Two-point controlled

    Notes
    -----
    - CHPs offer sets of constraints for operation. In the `convex` mode the
      following constraints and bounds are generated by the CHP:

    .. math::
        0 &\\geq& p_{th\\_heat} \\geq -p_{th\\_nom} \\\\
        0 &\\geq& p_{el} \\geq -p_{el\\_nom} \\\\
        \\frac{p_{el}}{p_{el\\_nom}} &=& \\frac{p_{th\\_heat}}{p_{th\\_nom}}

    - See also:
        - pycity_scheduling.util.generic_constraints.LowerActivationLimit:
          Generates additional constraints for the `lower_activation_limit` in `integer` mode.
    """

    def __init__(self, environment, p_th_nom, p_el_nom=None, eta=1, lower_activation_limit=0):
        q_nominal = p_th_nom * 1000
        if p_el_nom is None:
            p_nominal = q_nominal
            self.p_el_nom = p_th_nom
        else:
            p_nominal = p_el_nom * 1000
            self.p_el_nom = p_el_nom
        # Flow temperature of 55 C
        super().__init__(environment, p_nominal, q_nominal, eta, 55, lower_activation_limit)
        self._long_id = "CHP_" + self._id_string
        self.p_th_nom = p_th_nom
        self.activation_constr = LowerActivationLimit(self, "p_th_heat", lower_activation_limit, -p_th_nom)

    def populate_model(self, model, mode="convex"):
        """
        Add device block to pyomo ConcreteModel.

        Call both parents' `populate_model` methods and set the upper bounds
        of the thermal variables to `self.p_th_nom`, the lower bounds of the
        electrical variables to `-self.p_el_nom` and the upper bounds to zero.
        Also add constraints to bind electrical demand to thermal output.

        Parameters
        ----------
        model : pyomo.ConcreteModel
        mode : str, optional
            Specifies which set of constraints to use.

            - `convex`  : Use linear constraints
            - `integer`  : Use integer variables representing discrete control decisions
        """
        super().populate_model(model, mode)
        m = self.model

        if mode in ["convex", "integer"]:
            m.p_th_heat_vars.setlb(-self.p_th_nom)
            m.p_th_heat_vars.setub(0.0)
            m.p_el_vars.setlb(-self.p_el_nom)
            m.p_el_vars.setub(0.0)

            def p_coupl_rule(model, t):
                return model.p_th_heat_vars[t] * self.sigma == model.p_el_vars[t]
            m.P_coupl_constr = pyomo.Constraint(m.t, rule=p_coupl_rule)

            self.activation_constr.apply(m, mode)
        else:
            raise ValueError(
                "Mode %s is not implemented by class CombinedHeatPower." % str(mode)
            )
        return
