# Copyright 2019 DeepMind Technologies Limited
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""Default data for dynamic routing game."""

from open_spiel.python.games import dynamic_routing_utils

# The line network is a very simple network (O -> A -> D) with the goal of
# testing the routing game. There is no possible action and all cars will go
# from node O (being at node O means being on the link bef_O->O) to node D
# (being at node D means being on the link D->aft_D).
LINE_NETWORK = dynamic_routing_utils.Network({
    "bef_O": "O",
    "O": ["A"],
    "A": ["D"],
    "D": ["aft_D"],
    "aft_D": []
})

LINE_NETWORK_VEHICLES_DEMAND = [
    dynamic_routing_utils.Vehicle("bef_O->O", "D->aft_D") for _ in range(2)
]

LINE_NETWORK_OD_DEMAND = [
    dynamic_routing_utils.OriginDestinationDemand("bef_O->O", "D->aft_D", 0,
                                                  100)
]

# The Braess network comes from the Braess paradox: Braess, D., 1968. "Uber ein
# Paradoxon aus der Verkehrsplanung". Unternehmensforschung 12, 258-268.
BRAESS_NUM_PLAYER = 5
BRAESS_NETWORK = dynamic_routing_utils.Network(
    {
        "O": "A",
        "A": ["B", "C"],
        "B": ["C", "D"],
        "C": ["D"],
        "D": ["E"],
        "E": []
    },
    node_position={
        "O": (0, 0),
        "A": (1, 0),
        "B": (2, 1),
        "C": (2, -1),
        "D": (3, 0),
        "E": (4, 0)
    },
    bpr_a_coefficient={
        "O->A": 0,
        "A->B": 1.0,
        "A->C": 0,
        "B->C": 0,
        "B->D": 0,
        "C->D": 1.0,
        "D->E": 0
    },
    bpr_b_coefficient={
        "O->A": 1.0,
        "A->B": 1.0,
        "A->C": 1.0,
        "B->C": 1.0,
        "B->D": 1.0,
        "C->D": 1.0,
        "D->E": 1.0
    },
    capacity={
        "O->A": BRAESS_NUM_PLAYER,
        "A->B": BRAESS_NUM_PLAYER,
        "A->C": BRAESS_NUM_PLAYER,
        "B->C": BRAESS_NUM_PLAYER,
        "B->D": BRAESS_NUM_PLAYER,
        "C->D": BRAESS_NUM_PLAYER,
        "D->E": BRAESS_NUM_PLAYER
    },
    free_flow_travel_time={
        "O->A": 0,
        "A->B": 1.0,
        "A->C": 2.0,
        "B->C": 0.25,
        "B->D": 2.0,
        "C->D": 1.0,
        "D->E": 0
    })

BRAESS_NETWORK_VEHICLES_DEMAND = [
    dynamic_routing_utils.Vehicle("O->A", "D->E")
    for _ in range(BRAESS_NUM_PLAYER)
]

BRAESS_NETWORK_OD_DEMAND = [
    dynamic_routing_utils.OriginDestinationDemand("O->A", "D->E", 0,
                                                  BRAESS_NUM_PLAYER)
]

# The Sioux Falls data comes from "An Efficient Approach to Solving the Road
# Network Equilibrium Traffic Assignment Problem" by L. J. LeBlanc and E. K.
# Morlok (http://doi.org/10.1016/0041-1647(75)90030-1). We scale uniformly the
# data to decrease the number of time steps needed to cross the network. The
# demand and congestion functions data has been copied and pasted from the
# paper. The node position has been created from the paper's figure with a
# simple scale.
__SIOUX_FALLS_ADJACENCY = {
    "1": ["2", "3"],
    "2": ["1", "6"],
    "3": ["1", "4", "12"],
    "4": ["3", "5", "11"],
    "5": ["4", "6", "9"],
    "6": ["2", "5", "8"],
    "7": ["8", "18"],
    "8": ["6", "7", "9", "16"],
    "9": ["5", "8", "10"],
    "10": ["9", "11", "15", "16", "17"],
    "11": ["4", "10", "12", "14"],
    "12": ["3", "11", "13"],
    "13": ["12", "24"],
    "14": ["11", "15", "23"],
    "15": ["10", "14", "19", "22"],
    "16": ["8", "10", "17", "18"],
    "17": ["10", "16", "19"],
    "18": ["7", "16", "20"],
    "19": ["15", "17", "20"],
    "20": ["18", "19", "21", "22"],
    "21": ["20", "22", "24"],
    "22": ["15", "20", "21", "23"],
    "23": ["14", "22", "24"],
    "24": ["13", "21", "23"]
}

__SIOUX_FALLS_FREE_FLOW_TRAVEL_TIME = {
    "1->2": 6, "1->3": 4, "2->1": 6, "2->6": 5, "3->1": 4, "3->4": 4,
    "3->12": 4, "4->3": 4, "4->5": 2, "4->11": 6, "5->4": 2, "5->6": 4,
    "5->9": 5, "6->2": 5, "6->5": 4, "6->8": 2, "7->8": 3, "7->18": 2,
    "8->6": 2, "8->7": 3, "8->9": 10, "8->16": 5, "9->5": 5, "9->8": 10,
    "9->10": 3, "10->9": 3, "10->11": 5, "10->15": 6, "10->16": 4, "10->17": 8,
    "11->4": 6, "11->10": 5, "11->12": 6, "11->14": 4, "12->3": 4, "12->11": 6,
    "12->13": 3, "13->12": 3, "13->24": 4, "14->11": 4, "14->15": 5,
    "14->23": 4, "15->10": 6, "15->14": 5, "15->19": 3, "15->22": 3, "16->8": 5,
    "16->10": 4, "16->17": 2, "16->18": 3, "17->10": 8, "17->16": 2,
    "17->19": 2, "18->7": 2, "18->16": 3, "18->20": 4, "19->15": 3, "19->17": 2,
    "19->20": 4, "20->18": 4, "20->19": 4, "20->21": 6, "20->22": 5,
    "21->20": 6, "21->22": 2, "21->24": 3, "22->15": 3, "22->20": 5,
    "22->21": 2, "22->23": 4, "23->14": 4, "23->22": 4, "23->24": 2,
    "24->13": 4, "24->21": 3, "24->23": 2
}

__SIOUX_FALLS_BPR_A_COEFF = {
    "1->2": 2 * 1e-18,
    "1->3": 2 * 1e-18,
    "2->1": 2 * 1e-18,
    "2->6": 1240 * 1e-18,
    "3->1": 2 * 1e-18,
    "3->4": 6 * 1e-18,
    "3->12": 2 * 1e-18,
    "4->3": 6 * 1e-18,
    "4->5": 3 * 1e-18,
    "4->11": 1550 * 1e-18,
    "5->4": 3 * 1e-18,
    "5->6": 1000 * 1e-18,
    "5->9": 75 * 1e-18,
    "6->2": 1240 * 1e-18,
    "6->5": 1000 * 1e-18,
    "6->8": 520 * 1e-18,
    "7->8": 119 * 1e-18,
    "7->18": 1 * 1e-18,
    "8->6": 520 * 1e-18,
    "8->7": 119 * 1e-18,
    "8->9": 2306 * 1e-18,
    "8->16": 1156 * 1e-18,
    "9->5": 75 * 1e-18,
    "9->8": 2306 * 1e-18,
    "9->10": 11 * 1e-18,
    "10->9": 11 * 1e-18,
    "10->11": 75 * 1e-18,
    "10->15": 26 * 1e-18,
    "10->16": 1080 * 1e-18,
    "10->17": 1929 * 1e-18,
    "11->4": 1550 * 1e-18,
    "11->10": 75 * 1e-18,
    "11->12": 1550 * 1e-18,
    "11->14": 1061 * 1e-18,
    "12->3": 2 * 1e-18,
    "12->11": 1550 * 1e-18,
    "12->13": 1 * 1e-18,
    "13->12": 1 * 1e-18,
    "13->24": 893 * 1e-18,
    "14->11": 1061 * 1e-18,
    "14->15": 1085 * 1e-18,
    "14->23": 1020 * 1e-18,
    "15->10": 26 * 1e-18,
    "15->14": 1085 * 1e-18,
    "15->19": 10 * 1e-18,
    "15->22": 53 * 1e-18,
    "16->8": 1156 * 1e-18,
    "16->10": 1080 * 1e-18,
    "16->17": 401 * 1e-18,
    "16->18": 3 * 1e-18,
    "17->10": 1929 * 1e-18,
    "17->16": 401 * 1e-18,
    "17->19": 553 * 1e-18,
    "18->7": 1 * 1e-18,
    "18->16": 3 * 1e-18,
    "18->20": 2 * 1e-18,
    "19->15": 10 * 1e-18,
    "19->17": 553 * 1e-18,
    "19->20": 957 * 1e-18,
    "20->18": 2 * 1e-18,
    "20->19": 957 * 1e-18,
    "20->21": 1373 * 1e-18,
    "20->22": 1130 * 1e-18,
    "21->20": 1373 * 1e-18,
    "21->22": 401 * 1e-18,
    "21->24": 789 * 1e-18,
    "22->15": 53 * 1e-18,
    "22->20": 1130 * 1e-18,
    "22->21": 401 * 1e-18,
    "22->23": 960 * 1e-18,
    "23->14": 1020 * 1e-18,
    "23->22": 960 * 1e-18,
    "23->24": 451 * 1e-18,
    "24->13": 893 * 1e-18,
    "24->21": 789 * 1e-18,
    "24->23": 451 * 1e-18,
}

__SIOUX_FALLS_NODES = {
    "1": (0, 9), "2": (5, 9), "3": (0, 8), "4": (1, 8), "5": (3, 8),
    "6": (5, 8), "7": (7, 6), "8": (5, 6), "9": (3, 6), "10": (3, 5),
    "11": (1, 5), "12": (0, 5), "13": (0, 0), "14": (1, 2), "15": (3, 2),
    "16": (5, 5), "17": (5, 4), "18": (7, 5), "19": (5, 2), "20": (5, 0),
    "21": (3, 0), "22": (3, 1), "23": (1, 1), "24": (1, 0)
}

__SIOUX_FALLS_DEMAND_AUX = [
    ("2", "1", 1), ("3", "1", 1), ("4", "1", 5), ("5", "1", 2),
    ("6", "1", 3), ("7", "1", 5), ("8", "1", 8), ("9", "1", 5),
    ("10", "1", 13), ("11", "1", 5), ("12", "1", 2), ("13", "1", 5),
    ("14", "1", 3), ("15", "1", 5), ("16", "1", 5), ("17", "1", 4),
    ("18", "1", 1), ("19", "1", 3), ("20", "1", 3), ("21", "1", 1),
    ("22", "1", 4), ("23", "1", 3), ("24", "1", 1), ("1", "2", 1),
    ("3", "2", 1), ("4", "2", 2), ("5", "2", 1), ("6", "2", 4),
    ("7", "2", 2), ("8", "2", 4), ("9", "2", 2), ("10", "2", 6),
    ("11", "2", 2), ("12", "2", 1), ("13", "2", 3), ("14", "2", 1),
    ("15", "2", 1), ("16", "2", 4), ("17", "2", 2), ("19", "2", 1),
    ("20", "2", 1), ("22", "2", 1), ("1", "3", 1), ("2", "3", 1),
    ("4", "3", 2), ("5", "3", 1), ("6", "3", 3), ("7", "3", 1),
    ("8", "3", 2), ("9", "3", 1), ("10", "3", 3), ("11", "3", 3),
    ("12", "3", 2), ("13", "3", 1), ("14", "3", 1), ("15", "3", 1),
    ("16", "3", 2), ("17", "3", 1), ("22", "3", 1), ("23", "3", 1),
    ("1", "4", 5), ("2", "4", 2), ("3", "4", 2), ("5", "4", 5),
    ("6", "4", 4), ("7", "4", 4), ("8", "4", 7), ("9", "4", 7),
    ("10", "4", 12), ("11", "4", 14), ("12", "4", 6), ("13", "4", 6),
    ("14", "4", 5), ("15", "4", 5), ("16", "4", 8), ("17", "4", 5),
    ("18", "4", 1), ("19", "4", 2), ("20", "4", 3), ("21", "4", 2),
    ("22", "4", 4), ("23", "4", 5), ("24", "4", 2), ("1", "5", 2),
    ("2", "5", 1), ("3", "5", 1), ("4", "5", 5), ("6", "5", 2),
    ("7", "5", 2), ("8", "5", 5), ("9", "5", 8), ("10", "5", 10),
    ("11", "5", 5), ("12", "5", 2), ("13", "5", 2), ("14", "5", 1),
    ("15", "5", 2), ("16", "5", 5), ("17", "5", 2), ("19", "5", 1),
    ("20", "5", 1), ("21", "5", 1), ("22", "5", 2), ("23", "5", 1),
    ("1", "6", 3), ("2", "6", 4), ("3", "6", 3), ("4", "6", 4),
    ("5", "6", 2), ("7", "6", 4), ("8", "6", 8), ("9", "6", 4),
    ("10", "6", 8), ("11", "6", 4), ("12", "6", 2), ("13", "6", 2),
    ("14", "6", 1), ("15", "6", 2), ("16", "6", 9), ("17", "6", 5),
    ("18", "6", 1), ("19", "6", 2), ("20", "6", 3), ("21", "6", 1),
    ("22", "6", 2), ("23", "6", 1), ("24", "6", 1), ("1", "7", 5),
    ("2", "7", 2), ("3", "7", 1), ("4", "7", 4), ("5", "7", 2),
    ("6", "7", 4), ("8", "7", 10), ("9", "7", 6), ("10", "7", 19),
    ("11", "7", 5), ("12", "7", 7), ("13", "7", 4), ("14", "7", 2),
    ("15", "7", 5), ("16", "7", 14), ("17", "7", 10), ("18", "7", 2),
    ("19", "7", 4), ("20", "7", 5), ("21", "7", 2), ("22", "7", 5),
    ("23", "7", 2), ("24", "7", 1), ("1", "8", 8), ("2", "8", 4),
    ("3", "8", 2), ("4", "8", 7), ("5", "8", 5), ("6", "8", 8),
    ("7", "8", 10), ("9", "8", 8), ("10", "8", 16), ("11", "8", 8),
    ("12", "8", 6), ("13", "8", 6), ("14", "8", 4), ("15", "8", 6),
    ("16", "8", 22), ("17", "8", 14), ("18", "8", 3), ("19", "8", 7),
    ("20", "8", 9), ("21", "8", 4), ("22", "8", 5), ("23", "8", 3),
    ("24", "8", 2), ("1", "9", 5), ("2", "9", 2), ("3", "9", 1),
    ("4", "9", 7), ("5", "9", 8), ("6", "9", 4), ("7", "9", 6),
    ("8", "9", 8), ("10", "9", 28), ("11", "9", 14), ("12", "9", 6),
    ("13", "9", 6), ("14", "9", 6), ("15", "9", 9), ("16", "9", 14),
    ("17", "9", 9), ("18", "9", 2), ("19", "9", 4), ("20", "9", 6),
    ("21", "9", 3), ("22", "9", 7), ("23", "9", 5), ("24", "9", 2),
    ("1", "10", 13), ("2", "10", 6), ("3", "10", 3), ("4", "10", 12),
    ("5", "10", 10), ("6", "10", 8), ("7", "10", 19), ("8", "10", 16),
    ("9", "10", 28), ("11", "10", 40), ("12", "10", 20), ("13", "10", 19),
    ("14", "10", 21), ("15", "10", 40), ("16", "10", 44), ("17", "10", 39),
    ("18", "10", 7), ("19", "10", 18), ("20", "10", 25), ("21", "10", 12),
    ("22", "10", 26), ("23", "10", 18), ("24", "10", 8), ("1", "11", 5),
    ("2", "11", 2), ("3", "11", 3), ("4", "11", 15), ("5", "11", 5),
    ("6", "11", 4), ("7", "11", 5), ("8", "11", 8), ("9", "11", 14),
    ("10", "11", 39), ("12", "11", 14), ("13", "11", 10), ("14", "11", 16),
    ("15", "11", 14), ("16", "11", 14), ("17", "11", 10), ("18", "11", 1),
    ("19", "11", 4), ("20", "11", 6), ("21", "11", 4), ("22", "11", 11),
    ("23", "11", 13), ("24", "11", 6), ("1", "12", 2), ("2", "12", 1),
    ("3", "12", 2), ("4", "12", 6), ("5", "12", 2), ("6", "12", 2),
    ("7", "12", 7), ("8", "12", 6), ("9", "12", 6), ("10", "12", 20),
    ("11", "12", 14), ("13", "12", 13), ("14", "12", 7), ("15", "12", 7),
    ("16", "12", 7), ("17", "12", 6), ("18", "12", 2), ("19", "12", 3),
    ("20", "12", 4), ("21", "12", 3), ("22", "12", 7), ("23", "12", 7),
    ("24", "12", 5), ("1", "13", 5), ("2", "13", 3), ("3", "13", 1),
    ("4", "13", 6), ("5", "13", 2), ("6", "13", 2), ("7", "13", 4),
    ("8", "13", 6), ("9", "13", 6), ("10", "13", 19), ("11", "13", 10),
    ("12", "13", 13), ("14", "13", 6), ("15", "13", 7), ("16", "13", 6),
    ("17", "13", 5), ("18", "13", 1), ("19", "13", 3), ("20", "13", 6),
    ("21", "13", 6), ("22", "13", 13), ("23", "13", 8), ("24", "13", 8),
    ("1", "14", 3), ("2", "14", 1), ("3", "14", 1), ("4", "14", 5),
    ("5", "14", 1), ("6", "14", 1), ("7", "14", 2), ("8", "14", 4),
    ("9", "14", 6), ("10", "14", 21), ("11", "14", 16), ("12", "14", 7),
    ("13", "14", 6), ("15", "14", 13), ("16", "14", 7), ("17", "14", 7),
    ("18", "14", 1), ("19", "14", 3), ("20", "14", 5), ("21", "14", 4),
    ("22", "14", 12), ("23", "14", 11), ("24", "14", 4), ("1", "15", 5),
    ("2", "15", 1), ("3", "15", 1), ("4", "15", 5), ("5", "15", 2),
    ("6", "15", 2), ("7", "15", 5), ("8", "15", 6), ("9", "15", 10),
    ("10", "15", 40), ("11", "15", 14), ("12", "15", 7), ("13", "15", 7),
    ("14", "15", 13), ("16", "15", 12), ("17", "15", 15), ("18", "15", 2),
    ("19", "15", 8), ("20", "15", 11), ("21", "15", 8), ("22", "15", 26),
    ("23", "15", 10), ("24", "15", 4), ("1", "16", 5), ("2", "16", 4),
    ("3", "16", 2), ("4", "16", 8), ("5", "16", 5), ("6", "16", 9),
    ("7", "16", 14), ("8", "16", 22), ("9", "16", 14), ("10", "16", 44),
    ("11", "16", 14), ("12", "16", 7), ("13", "16", 6), ("14", "16", 7),
    ("15", "16", 12), ("17", "16", 28), ("18", "16", 5), ("19", "16", 13),
    ("20", "16", 16), ("21", "16", 6), ("22", "16", 12), ("23", "16", 5),
    ("24", "16", 3), ("1", "17", 4), ("2", "17", 2), ("3", "17", 1),
    ("4", "17", 5), ("5", "17", 2), ("6", "17", 5), ("7", "17", 10),
    ("8", "17", 14), ("9", "17", 9), ("10", "17", 39), ("11", "17", 10),
    ("12", "17", 6), ("13", "17", 5), ("14", "17", 7), ("15", "17", 15),
    ("16", "17", 28), ("18", "17", 6), ("19", "17", 17), ("20", "17", 17),
    ("21", "17", 6), ("22", "17", 17), ("23", "17", 6), ("24", "17", 3),
    ("1", "18", 1), ("4", "18", 1), ("6", "18", 1), ("7", "18", 2),
    ("8", "18", 3), ("9", "18", 2), ("10", "18", 7), ("11", "18", 2),
    ("12", "18", 2), ("13", "18", 1), ("14", "18", 1), ("15", "18", 2),
    ("16", "18", 5), ("17", "18", 6), ("19", "18", 3), ("20", "18", 4),
    ("21", "18", 1), ("22", "18", 3), ("23", "18", 1), ("1", "19", 3),
    ("2", "19", 1), ("4", "19", 2), ("5", "19", 1), ("6", "19", 2),
    ("7", "19", 4), ("8", "19", 7), ("9", "19", 4), ("10", "19", 18),
    ("11", "19", 4), ("12", "19", 3), ("13", "19", 3), ("14", "19", 3),
    ("15", "19", 8), ("16", "19", 13), ("17", "19", 17), ("18", "19", 3),
    ("20", "19", 12), ("21", "19", 4), ("22", "19", 12), ("23", "19", 3),
    ("24", "19", 1), ("1", "20", 3), ("2", "20", 1), ("4", "20", 3),
    ("5", "20", 1), ("6", "20", 3), ("7", "20", 5), ("8", "20", 9),
    ("9", "20", 6), ("10", "20", 25), ("11", "20", 6), ("12", "20", 5),
    ("13", "20", 6), ("14", "20", 5), ("15", "20", 11), ("16", "20", 16),
    ("17", "20", 17), ("18", "20", 4), ("19", "20", 12), ("21", "20", 12),
    ("22", "20", 24), ("23", "20", 7), ("24", "20", 4), ("1", "21", 1),
    ("4", "21", 2), ("5", "21", 1), ("6", "21", 1), ("7", "21", 2),
    ("8", "21", 4), ("9", "21", 3), ("10", "21", 12), ("11", "21", 4),
    ("12", "21", 3), ("13", "21", 6), ("14", "21", 4), ("15", "21", 8),
    ("16", "21", 6), ("17", "21", 6), ("18", "21", 1), ("19", "21", 4),
    ("20", "21", 12), ("22", "21", 18), ("23", "21", 7), ("24", "21", 5),
    ("1", "22", 4), ("2", "22", 1), ("3", "22", 1), ("4", "22", 4),
    ("5", "22", 2), ("6", "22", 2), ("7", "22", 5), ("8", "22", 5),
    ("9", "22", 7), ("10", "22", 26), ("11", "22", 11), ("12", "22", 7),
    ("13", "22", 13), ("14", "22", 12), ("15", "22", 26), ("16", "22", 12),
    ("17", "22", 17), ("18", "22", 3), ("19", "22", 12), ("20", "22", 24),
    ("21", "22", 18), ("23", "22", 21), ("24", "22", 11), ("1", "23", 3),
    ("3", "23", 1), ("4", "23", 5), ("5", "23", 1), ("6", "23", 1),
    ("7", "23", 2), ("8", "23", 3), ("9", "23", 5), ("10", "23", 18),
    ("11", "23", 13), ("12", "23", 7), ("13", "23", 8), ("14", "23", 11),
    ("15", "23", 10), ("16", "23", 5), ("17", "23", 6), ("18", "23", 1),
    ("19", "23", 3), ("20", "23", 7), ("21", "23", 7), ("22", "23", 21),
    ("24", "23", 7), ("1", "24", 1), ("4", "24", 2), ("6", "24", 1),
    ("7", "24", 1), ("8", "24", 2), ("9", "24", 2), ("10", "24", 8),
    ("11", "24", 6), ("12", "24", 5), ("13", "24", 7), ("14", "24", 4),
    ("15", "24", 4), ("16", "24", 3), ("17", "24", 3), ("19", "24", 1),
    ("20", "24", 4), ("21", "24", 5), ("22", "24", 11), ("23", "24", 7)
]


def create_sioux_falls_network():
  """Returns Sioux Falls network object (Network).

  Adds the origin and destination link to the adjacency list
  __SIOUX_FALLS_ADJACENCY, to the BPR coefficients
  __SIOUX_FALLS_FREE_FLOW_TRAVEL_TIME and __SIOUX_FALLS_BPR_A_COEFF and to the
  node positions __SIOUX_FALLS_NODES and returns the network.
  The BPR (Burean of Public Roads) coefficients are the coefficients used to
  compute the travel time as a function of the volume on each link.
  """
  adjacency = {}
  free_flow_travel_time = __SIOUX_FALLS_FREE_FLOW_TRAVEL_TIME.copy()
  bpr_a_coeff = __SIOUX_FALLS_BPR_A_COEFF.copy()
  node_position = {}

  for k, nodes in __SIOUX_FALLS_ADJACENCY.items():
    adjacency[k] = nodes + [f"aft_{k}"]
    adjacency[f"bef_{k}"] = [k]
    adjacency[f"aft_{k}"] = []
    free_flow_travel_time[f"bef_{k}->{k}"] = 0
    free_flow_travel_time[f"{k}->aft_{k}"] = 0
    bpr_a_coeff[f"bef_{k}->{k}"] = 0
    bpr_a_coeff[f"{k}->aft_{k}"] = 0

  for node, coord in __SIOUX_FALLS_NODES.items():
    node_position[node] = coord
    node_position[f"bef_{node}"] = coord
    node_position[f"aft_{node}"] = coord

  return dynamic_routing_utils.Network(
      adjacency,
      node_position=node_position,
      bpr_a_coefficient=bpr_a_coeff,
      bpr_b_coefficient={k: 4 for k in bpr_a_coeff},
      capacity={k: 1 for k in bpr_a_coeff},
      free_flow_travel_time=free_flow_travel_time)


SIOUX_FALLS_NETWORK = create_sioux_falls_network()

SIOUX_FALLS_OD_DEMAND = [
    dynamic_routing_utils.OriginDestinationDemand(
        f"bef_{origin}->{origin}", f"{dest}->aft_{dest}", 0, count * 1e2)
    for (origin, dest, count) in __SIOUX_FALLS_DEMAND_AUX]

SIOUX_FALLS_DUMMY_OD_DEMAND = [
    dynamic_routing_utils.OriginDestinationDemand("bef_19->19", "1->aft_1", 0,
                                                  70 * 1e2),
    dynamic_routing_utils.OriginDestinationDemand("bef_1->1", "19->aft_19", 0,
                                                  70 * 1e2)
]
