#
# PRESTO .inf file parsing module
# (c) Vincent Morello, 2020
#
import os
import numpy
import astropy.units as uu
from astropy.coordinates import SkyCoord


# Template valid for all .inf files except those generated by makedata
# --------------------------------------------------------------------
#  Data file name without suffix          =
#  Telescope used                         =
#  Instrument used                        =
#  Object being observed                  =
#  J2000 Right Ascension (hh:mm:ss.ssss)  =
#  J2000 Declination     (dd:mm:ss.ssss)  =
#  Data observed by                       =
#  Epoch of observation (MJD)             =
#  Barycentered?           (1 yes, 0 no)  =
#  Number of bins in the time series      =
#  Width of each time series bin (sec)    =
#  Any breaks in the data? (1 yes, 0 no)  =
#  [ON/OFF BIN PAIR #1]
#  [...]
#  [ON/OFF BIN PAIR #N]
#  Type of observation (EM band)          =
#  [BLOCK DEPENDENT ON EM BAND]
#  Any additional notes:
#  [ADDITIONAL NOTES]


SEP = '='
SEP_COLUMN = 40
FAKE_TELESCOPE = "None (Artificial Data Set)"


def parse_inf_value(line, vtype):
    # A "standard" .inf line has an '=' character at column 40
    if not (len(line) > SEP_COLUMN and line[SEP_COLUMN] == SEP):
        raise ValueError(f"Expected '=' character at column {SEP_COLUMN}")
    val = line[SEP_COLUMN+1:].strip()
    return vtype(val)


def parse_bool(s):
    """ Convert '1' or '0' to boolean """
    return int(s) != 0


def parse_int_pair(s):
    a, b = s.split(',')
    return int(a), int(b)


def inf2dict(text):
    lines = text.strip('\n').splitlines()

    def p(n, vtype):
        return parse_inf_value(lines[n], vtype)

    # The first two lines are always the file basename and the telescope
    basename = p(0, str)
    telescope = p(1, str)

    if telescope == FAKE_TELESCOPE:
        raise ValueError("Reading data generated with PRESTO's makedata is not supported")

    # Parse block common to all EM bands
    items = {
        'basename': basename,
        'telescope': telescope,
        'instrument': p(2, str),
        'source_name': p(3, str),
        'raj': p(4, str),
        'decj': p(5, str),
        'observer': p(6, str),
        'mjd': p(7, float),
        'barycentered': p(8, parse_bool),
        'nsamp': p(9, int),
        'tsamp': p(10, float),
        'breaks': p(11, parse_bool),
        'onoff_pairs': []
    }

    lines = lines[12:]

    # Parse on/off pairs, if any
    if items['breaks']:
        for line in lines:
            try:
                pair = parse_inf_value(line, parse_int_pair)
                items['onoff_pairs'].append(pair)
            except:
                break
    num_onoff = len(items['onoff_pairs'])

    lines = lines[num_onoff:]
    em_band = p(0, str)
    items['em_band'] = em_band

    if em_band == 'Radio':
        items['fov_arcsec'] = p(1, float)
        items['dm'] = p(2, float)
        items['fbot'] = p(3, float)
        items['bandwidth'] = p(4, float)
        items['nchan'] = p(5, int)
        items['cbw'] = p(6, float)
        items['analyst'] = p(7, str)

    elif em_band in ('X-ray', 'Gamma'):
        items['fov_arcsec'] = p(1, float)
        items['central_energy_kev'] = p(2, float)
        items['energy_bandpass_kev'] = p(3, float)
        items['analyst'] = p(4, str)

    else:
        raise ValueError(f"EM Band {em_band!r} not supported")

    return items


class PrestoInf(dict):
    """ Parse PRESTO's .inf files that contain dedispersed time series metadata. """
    def __init__(self, fname):
        self._fname = os.path.realpath(fname)
        with open(fname, 'r') as fobj:
            items = inf2dict(fobj.read())
        super(PrestoInf, self).__init__(items)

    @property
    def fname(self):
        """ Absolute path to original file. """
        return self._fname

    @property
    def data_fname(self):
        """ Path to the associated .dat file """
        return self.fname.rsplit('.', maxsplit=1)[0] + '.dat'

    @property
    def skycoord(self):
        """ astropy.SkyCoord object with the coordinates of the source. """
        return SkyCoord(self['raj'], self['decj'], unit=(uu.hour, uu.degree))

    def load_data(self):
        """ Returns the associated time series data as a numpy float32 array. """
        return numpy.fromfile(self.data_fname, dtype=numpy.float32)
