# This file is part of aiootp, an asynchronous one-time-pad based crypto
# and anonymity library.
#
# Licensed under the AGPLv3: https://www.gnu.org/licenses/agpl-3.0.html
# Copyright © 2019-2021 Gonzo Investigatory Journalism Agency, LLC
#            <gonzo.development@protonmail.ch>
#           © 2019-2021 Richard Machado <rmlibre@riseup.net>
# All rights reserved.
#


import hashlib


__all__ = [
    "bits",
    "primes",
    "NUM",
    "NONE",
    "WORD_LIST",
    "ASCII_TABLE",
    "ASCII_TABLE_128",
    "BYTES_TABLE",
    "BASE_36_TABLE",
    "BASE_64_TABLE",
    "URL_SAFE_TABLE",
    "ONION_CHAR_TABLE",
    "ASCII_ALPHANUMERIC",
    "PrimeGroups",
]


ASCII_TABLE = """ !"#$%&'()*+,-./0123456789:;<=>?@"""
ASCII_TABLE += "ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^_`"
ASCII_TABLE += "abcdefghijklmnopqrstuvwxyz{|}~"


ASCII_TABLE_128 = """\x00\x01\x02\x03\x04\x05\x06\x07\x08\t\n\x0b"""
ASCII_TABLE_128 += """\x0c\r\x0e\x0f\x10\x11\x12\x13\x14\x15\x16\x17"""
ASCII_TABLE_128 += """\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f !"#$%&\'()*+,"""
ASCII_TABLE_128 += """-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ"""
ASCII_TABLE_128 += """[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~\x7f"""


ASCII_ALPHANUMERIC = "0123456789abcdefghijklmnopqrstuvwxyz"
ASCII_ALPHANUMERIC += "ABCDEFGHIJKLMNOPQRSTUVWXYZ"


BASE_36_TABLE = "0123456789abcdefghijklmnopqrstuvwxyz"


BASE_64_TABLE = "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
BASE_64_TABLE += "abcdefghijklmnopqrstuvwxyz0123456789+/"


ONION_CHAR_TABLE = "234567abcdefghijklmnopqrstuvwxyz"


URL_SAFE_TABLE = "-0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ_abcdefghijklmn"
URL_SAFE_TABLE += "opqrstuvwxyz"


BYTES_TABLE = [
    b"\x00",
    b"\x01",
    b"\x02",
    b"\x03",
    b"\x04",
    b"\x05",
    b"\x06",
    b"\x07",
    b"\x08",
    b"\t",
    b"\n",
    b"\x0b",
    b"\x0c",
    b"\r",
    b"\x0e",
    b"\x0f",
    b"\x10",
    b"\x11",
    b"\x12",
    b"\x13",
    b"\x14",
    b"\x15",
    b"\x16",
    b"\x17",
    b"\x18",
    b"\x19",
    b"\x1a",
    b"\x1b",
    b"\x1c",
    b"\x1d",
    b"\x1e",
    b"\x1f",
    b" ",
    b"!",
    b'"',
    b"#",
    b"$",
    b"%",
    b"&",
    b"'",
    b"(",
    b")",
    b"*",
    b"+",
    b",",
    b"-",
    b".",
    b"/",
    b"0",
    b"1",
    b"2",
    b"3",
    b"4",
    b"5",
    b"6",
    b"7",
    b"8",
    b"9",
    b":",
    b";",
    b"<",
    b"=",
    b">",
    b"?",
    b"@",
    b"A",
    b"B",
    b"C",
    b"D",
    b"E",
    b"F",
    b"G",
    b"H",
    b"I",
    b"J",
    b"K",
    b"L",
    b"M",
    b"N",
    b"O",
    b"P",
    b"Q",
    b"R",
    b"S",
    b"T",
    b"U",
    b"V",
    b"W",
    b"X",
    b"Y",
    b"Z",
    b"[",
    b"\\",
    b"]",
    b"^",
    b"_",
    b"`",
    b"a",
    b"b",
    b"c",
    b"d",
    b"e",
    b"f",
    b"g",
    b"h",
    b"i",
    b"j",
    b"k",
    b"l",
    b"m",
    b"n",
    b"o",
    b"p",
    b"q",
    b"r",
    b"s",
    b"t",
    b"u",
    b"v",
    b"w",
    b"x",
    b"y",
    b"z",
    b"{",
    b"|",
    b"}",
    b"~",
    b"\x7f",
    b"\x80",
    b"\x81",
    b"\x82",
    b"\x83",
    b"\x84",
    b"\x85",
    b"\x86",
    b"\x87",
    b"\x88",
    b"\x89",
    b"\x8a",
    b"\x8b",
    b"\x8c",
    b"\x8d",
    b"\x8e",
    b"\x8f",
    b"\x90",
    b"\x91",
    b"\x92",
    b"\x93",
    b"\x94",
    b"\x95",
    b"\x96",
    b"\x97",
    b"\x98",
    b"\x99",
    b"\x9a",
    b"\x9b",
    b"\x9c",
    b"\x9d",
    b"\x9e",
    b"\x9f",
    b"\xa0",
    b"\xa1",
    b"\xa2",
    b"\xa3",
    b"\xa4",
    b"\xa5",
    b"\xa6",
    b"\xa7",
    b"\xa8",
    b"\xa9",
    b"\xaa",
    b"\xab",
    b"\xac",
    b"\xad",
    b"\xae",
    b"\xaf",
    b"\xb0",
    b"\xb1",
    b"\xb2",
    b"\xb3",
    b"\xb4",
    b"\xb5",
    b"\xb6",
    b"\xb7",
    b"\xb8",
    b"\xb9",
    b"\xba",
    b"\xbb",
    b"\xbc",
    b"\xbd",
    b"\xbe",
    b"\xbf",
    b"\xc0",
    b"\xc1",
    b"\xc2",
    b"\xc3",
    b"\xc4",
    b"\xc5",
    b"\xc6",
    b"\xc7",
    b"\xc8",
    b"\xc9",
    b"\xca",
    b"\xcb",
    b"\xcc",
    b"\xcd",
    b"\xce",
    b"\xcf",
    b"\xd0",
    b"\xd1",
    b"\xd2",
    b"\xd3",
    b"\xd4",
    b"\xd5",
    b"\xd6",
    b"\xd7",
    b"\xd8",
    b"\xd9",
    b"\xda",
    b"\xdb",
    b"\xdc",
    b"\xdd",
    b"\xde",
    b"\xdf",
    b"\xe0",
    b"\xe1",
    b"\xe2",
    b"\xe3",
    b"\xe4",
    b"\xe5",
    b"\xe6",
    b"\xe7",
    b"\xe8",
    b"\xe9",
    b"\xea",
    b"\xeb",
    b"\xec",
    b"\xed",
    b"\xee",
    b"\xef",
    b"\xf0",
    b"\xf1",
    b"\xf2",
    b"\xf3",
    b"\xf4",
    b"\xf5",
    b"\xf6",
    b"\xf7",
    b"\xf8",
    b"\xf9",
    b"\xfa",
    b"\xfb",
    b"\xfc",
    b"\xfd",
    b"\xfe",
    b"\xff",
]


bits = {
    # A static table of powers of 2, for efficient, more readable logic
    # requiring bit and entropy calculations. Each entry is named after
    # how many bits the value is.
    0: 1,
    1: 2,
    2: 4,
    3: 8,
    4: 16,
    5: 32,
    6: 64,
    7: 128,
    8: 256,
    9: 512,
    10: 1024,
    11: 2048,
    12: 4096,
    13: 8192,
    14: 16384,
    15: 32768,
    16: 65536,
    17: 131072,
    18: 262144,
    19: 524288,
    20: 1048576,
    21: 2097152,
    22: 4194304,
    23: 8388608,
    24: 16777216,
    25: 33554432,
    26: 67108864,
    27: 134217728,
    28: 268435456,
    29: 536870912,
    30: 1073741824,
    31: 2147483648,
    32: 4294967296,
    33: 8589934592,
    34: 17179869184,
    35: 34359738368,
    36: 68719476736,
    37: 137438953472,
    38: 274877906944,
    39: 549755813888,
    40: 1099511627776,
    41: 2199023255552,
    42: 4398046511104,
    43: 8796093022208,
    44: 17592186044416,
    45: 35184372088832,
    46: 70368744177664,
    47: 140737488355328,
    48: 281474976710656,
    49: 562949953421312,
    50: 1125899906842624,
    51: 2251799813685248,
    52: 4503599627370496,
    53: 9007199254740992,
    54: 18014398509481984,
    55: 36028797018963968,
    56: 72057594037927936,
    57: 144115188075855872,
    58: 288230376151711744,
    59: 576460752303423488,
    60: 1152921504606846976,
    61: 2305843009213693952,
    62: 4611686018427387904,
    63: 9223372036854775808,
    64: 18446744073709551616,
    65: 36893488147419103232,
    66: 73786976294838206464,
    67: 147573952589676412928,
    68: 295147905179352825856,
    69: 590295810358705651712,
    70: 1180591620717411303424,
    71: 2361183241434822606848,
    72: 4722366482869645213696,
    73: 9444732965739290427392,
    74: 18889465931478580854784,
    75: 37778931862957161709568,
    76: 75557863725914323419136,
    77: 151115727451828646838272,
    78: 302231454903657293676544,
    79: 604462909807314587353088,
    80: 1208925819614629174706176,
    81: 2417851639229258349412352,
    82: 4835703278458516698824704,
    83: 9671406556917033397649408,
    84: 19342813113834066795298816,
    85: 38685626227668133590597632,
    86: 77371252455336267181195264,
    87: 154742504910672534362390528,
    88: 309485009821345068724781056,
    89: 618970019642690137449562112,
    90: 1237940039285380274899124224,
    91: 2475880078570760549798248448,
    92: 4951760157141521099596496896,
    93: 9903520314283042199192993792,
    94: 19807040628566084398385987584,
    95: 39614081257132168796771975168,
    96: 79228162514264337593543950336,
    97: 158456325028528675187087900672,
    98: 316912650057057350374175801344,
    99: 633825300114114700748351602688,
    100: 1267650600228229401496703205376,
    101: 2535301200456458802993406410752,
    102: 5070602400912917605986812821504,
    103: 10141204801825835211973625643008,
    104: 20282409603651670423947251286016,
    105: 40564819207303340847894502572032,
    106: 81129638414606681695789005144064,
    107: 162259276829213363391578010288128,
    108: 324518553658426726783156020576256,
    109: 649037107316853453566312041152512,
    110: 1298074214633706907132624082305024,
    111: 2596148429267413814265248164610048,
    112: 5192296858534827628530496329220096,
    113: 10384593717069655257060992658440192,
    114: 20769187434139310514121985316880384,
    115: 41538374868278621028243970633760768,
    116: 83076749736557242056487941267521536,
    117: 166153499473114484112975882535043072,
    118: 332306998946228968225951765070086144,
    119: 664613997892457936451903530140172288,
    120: 1329227995784915872903807060280344576,
    121: 2658455991569831745807614120560689152,
    122: 5316911983139663491615228241121378304,
    123: 10633823966279326983230456482242756608,
    124: 21267647932558653966460912964485513216,
    125: 42535295865117307932921825928971026432,
    126: 85070591730234615865843651857942052864,
    127: 170141183460469231731687303715884105728,
    128: 340282366920938463463374607431768211456,
    129: 680564733841876926926749214863536422912,
    130: 1361129467683753853853498429727072845824,
    131: 2722258935367507707706996859454145691648,
    132: 5444517870735015415413993718908291383296,
    133: 10889035741470030830827987437816582766592,
    134: 21778071482940061661655974875633165533184,
    135: 43556142965880123323311949751266331066368,
    136: 87112285931760246646623899502532662132736,
    137: 174224571863520493293247799005065324265472,
    138: 348449143727040986586495598010130648530944,
    139: 696898287454081973172991196020261297061888,
    140: 1393796574908163946345982392040522594123776,
    141: 2787593149816327892691964784081045188247552,
    142: 5575186299632655785383929568162090376495104,
    143: 11150372599265311570767859136324180752990208,
    144: 22300745198530623141535718272648361505980416,
    145: 44601490397061246283071436545296723011960832,
    146: 89202980794122492566142873090593446023921664,
    147: 178405961588244985132285746181186892047843328,
    148: 356811923176489970264571492362373784095686656,
    149: 713623846352979940529142984724747568191373312,
    150: 1427247692705959881058285969449495136382746624,
    151: 2854495385411919762116571938898990272765493248,
    152: 5708990770823839524233143877797980545530986496,
    153: 11417981541647679048466287755595961091061972992,
    154: 22835963083295358096932575511191922182123945984,
    155: 45671926166590716193865151022383844364247891968,
    156: 91343852333181432387730302044767688728495783936,
    157: 182687704666362864775460604089535377456991567872,
    158: 365375409332725729550921208179070754913983135744,
    159: 730750818665451459101842416358141509827966271488,
    160: 1461501637330902918203684832716283019655932542976,
    161: 2923003274661805836407369665432566039311865085952,
    162: 5846006549323611672814739330865132078623730171904,
    163: 11692013098647223345629478661730264157247460343808,
    164: 23384026197294446691258957323460528314494920687616,
    165: 46768052394588893382517914646921056628989841375232,
    166: 93536104789177786765035829293842113257979682750464,
    167: 187072209578355573530071658587684226515959365500928,
    168: 374144419156711147060143317175368453031918731001856,
    169: 748288838313422294120286634350736906063837462003712,
    170: 1496577676626844588240573268701473812127674924007424,
    171: 2993155353253689176481146537402947624255349848014848,
    172: 5986310706507378352962293074805895248510699696029696,
    173: 11972621413014756705924586149611790497021399392059392,
    174: 23945242826029513411849172299223580994042798784118784,
    175: 47890485652059026823698344598447161988085597568237568,
    176: 95780971304118053647396689196894323976171195136475136,
    177: 191561942608236107294793378393788647952342390272950272,
    178: 383123885216472214589586756787577295904684780545900544,
    179: 766247770432944429179173513575154591809369561091801088,
    180: 1532495540865888858358347027150309183618739122183602176,
    181: 3064991081731777716716694054300618367237478244367204352,
    182: 6129982163463555433433388108601236734474956488734408704,
    183: 12259964326927110866866776217202473468949912977468817408,
    184: 24519928653854221733733552434404946937899825954937634816,
    185: 49039857307708443467467104868809893875799651909875269632,
    186: 98079714615416886934934209737619787751599303819750539264,
    187: 196159429230833773869868419475239575503198607639501078528,
    188: 392318858461667547739736838950479151006397215279002157056,
    189: 784637716923335095479473677900958302012794430558004314112,
    190: 1569275433846670190958947355801916604025588861116008628224,
    191: 3138550867693340381917894711603833208051177722232017256448,
    192: 6277101735386680763835789423207666416102355444464034512896,
    193: 12554203470773361527671578846415332832204710888928069025792,
    194: 25108406941546723055343157692830665664409421777856138051584,
    195: 50216813883093446110686315385661331328818843555712276103168,
    196: 100433627766186892221372630771322662657637687111424552206336,
    197: 200867255532373784442745261542645325315275374222849104412672,
    198: 401734511064747568885490523085290650630550748445698208825344,
    199: 803469022129495137770981046170581301261101496891396417650688,
    200: 1606938044258990275541962092341162602522202993782792835301376,
    201: 3213876088517980551083924184682325205044405987565585670602752,
    202: 6427752177035961102167848369364650410088811975131171341205504,
    203: 12855504354071922204335696738729300820177623950262342682411008,
    204: 25711008708143844408671393477458601640355247900524685364822016,
    205: 51422017416287688817342786954917203280710495801049370729644032,
    206: 102844034832575377634685573909834406561420991602098741459288064,
    207: 205688069665150755269371147819668813122841983204197482918576128,
    208: 411376139330301510538742295639337626245683966408394965837152256,
    209: 822752278660603021077484591278675252491367932816789931674304512,
    210: 1645504557321206042154969182557350504982735865633579863348609024,
    211: 3291009114642412084309938365114701009965471731267159726697218048,
    212: 6582018229284824168619876730229402019930943462534319453394436096,
    213: 13164036458569648337239753460458804039861886925068638906788872192,
    214: 26328072917139296674479506920917608079723773850137277813577744384,
    215: 52656145834278593348959013841835216159447547700274555627155488768,
    216: 105312291668557186697918027683670432318895095400549111254310977536,
    217: 210624583337114373395836055367340864637790190801098222508621955072,
    218: 421249166674228746791672110734681729275580381602196445017243910144,
    219: 842498333348457493583344221469363458551160763204392890034487820288,
    220: 1684996666696914987166688442938726917102321526408785780068975640576,
    221: 3369993333393829974333376885877453834204643052817571560137951281152,
    222: 6739986666787659948666753771754907668409286105635143120275902562304,
    223: 13479973333575319897333507543509815336818572211270286240551805124608,
    224: 26959946667150639794667015087019630673637144422540572481103610249216,
    225: 53919893334301279589334030174039261347274288845081144962207220498432,
    226: 107839786668602559178668060348078522694548577690162289924414440996864,
    227: 215679573337205118357336120696157045389097155380324579848828881993728,
    228: 431359146674410236714672241392314090778194310760649159697657763987456,
    229: 862718293348820473429344482784628181556388621521298319395315527974912,
    230: 1725436586697640946858688965569256363112777243042596638790631055949824,
    231: 3450873173395281893717377931138512726225554486085193277581262111899648,
    232: 6901746346790563787434755862277025452451108972170386555162524223799296,
    233: 13803492693581127574869511724554050904902217944340773110325048447598592,
    234: 27606985387162255149739023449108101809804435888681546220650096895197184,
    235: 55213970774324510299478046898216203619608871777363092441300193790394368,
    236: 110427941548649020598956093796432407239217743554726184882600387580788736,
    237: 220855883097298041197912187592864814478435487109452369765200775161577472,
    238: 441711766194596082395824375185729628956870974218904739530401550323154944,
    239: 883423532389192164791648750371459257913741948437809479060803100646309888,
    240: 1766847064778384329583297500742918515827483896875618958121606201292619776,
    241: 3533694129556768659166595001485837031654967793751237916243212402585239552,
    242: 7067388259113537318333190002971674063309935587502475832486424805170479104,
    243: 14134776518227074636666380005943348126619871175004951664972849610340958208,
    244: 28269553036454149273332760011886696253239742350009903329945699220681916416,
    245: 56539106072908298546665520023773392506479484700019806659891398441363832832,
    246: 113078212145816597093331040047546785012958969400039613319782796882727665664,
    247: 226156424291633194186662080095093570025917938800079226639565593765455331328,
    248: 452312848583266388373324160190187140051835877600158453279131187530910662656,
    249: 904625697166532776746648320380374280103671755200316906558262375061821325312,
    250: 1809251394333065553493296640760748560207343510400633813116524750123642650624,
    251: 3618502788666131106986593281521497120414687020801267626233049500247285301248,
    252: 7237005577332262213973186563042994240829374041602535252466099000494570602496,
    253: 14474011154664524427946373126085988481658748083205070504932198000989141204992,
    254: 28948022309329048855892746252171976963317496166410141009864396001978282409984,
    255: 57896044618658097711785492504343953926634992332820282019728792003956564819968,
    256: 115792089237316195423570985008687907853269984665640564039457584007913129639936,
    257: 231584178474632390847141970017375815706539969331281128078915168015826259279872,
    383: 19701003098197239606139520050071806902539869635232723333974146702122860885748605305707133127442457820403313995153408,
    384: 39402006196394479212279040100143613805079739270465446667948293404245721771497210611414266254884915640806627990306816,
    385: 78804012392788958424558080200287227610159478540930893335896586808491443542994421222828532509769831281613255980613632,
    511: 6703903964971298549787012499102923063739682910296196688861780721860882015036773488400937149083451713845015929093243025426876941405973284973216824503042048,
    512: 13407807929942597099574024998205846127479365820592393377723561443721764030073546976801874298166903427690031858186486050853753882811946569946433649006084096,
    513: 26815615859885194199148049996411692254958731641184786755447122887443528060147093953603748596333806855380063716372972101707507765623893139892867298012168192,
    1023: 89884656743115795386465259539451236680898848947115328636715040578866337902750481566354238661203768010560056939935696678829394884407208311246423715319737062188883946712432742638151109800623047059726541476042502884419075341171231440736956555270413618581675255342293149119973622969239858152417678164812112068608,
    1024: 179769313486231590772930519078902473361797697894230657273430081157732675805500963132708477322407536021120113879871393357658789768814416622492847430639474124377767893424865485276302219601246094119453082952085005768838150682342462881473913110540827237163350510684586298239947245938479716304835356329624224137216,
    1025: 359538626972463181545861038157804946723595395788461314546860162315465351611001926265416954644815072042240227759742786715317579537628833244985694861278948248755535786849730970552604439202492188238906165904170011537676301364684925762947826221081654474326701021369172596479894491876959432609670712659248448274432,
    1535: 1205156213460516294290058303014157056456046623972844475679837519532628695795901600334542512053673024831724383140444002393931208489397479162484806493945387325727606669690812612385391038958840749838422771568693910028798672928952299554730693561049753982498907820671150338814736677640808714205897081983892935185184484554610795971527116005781379225040289793925450496857446141738323315590757531902436687591130253123496418949352985506262921662200616493428502380169658368,
    1536: 2410312426921032588580116606028314112912093247945688951359675039065257391591803200669085024107346049663448766280888004787862416978794958324969612987890774651455213339381625224770782077917681499676845543137387820057597345857904599109461387122099507964997815641342300677629473355281617428411794163967785870370368969109221591943054232011562758450080579587850900993714892283476646631181515063804873375182260506246992837898705971012525843324401232986857004760339316736,
    1537: 4820624853842065177160233212056628225824186495891377902719350078130514783183606401338170048214692099326897532561776009575724833957589916649939225975781549302910426678763250449541564155835362999353691086274775640115194691715809198218922774244199015929995631282684601355258946710563234856823588327935571740740737938218443183886108464023125516900161159175701801987429784566953293262363030127609746750364521012493985675797411942025051686648802465973714009520678633472,
    2047: 16158503035655503650357438344334975980222051334857742016065172713762327569433945446598600705761456731844358980460949009747059779575245460547544076193224141560315438683650498045875098875194826053398028819192033784138396109321309878080919047169238085235290822926018152521443787945770532904303776199561965192760957166694834171210342487393282284747428088017663161029038902829665513096354230157075129296432088558362971801859230928678799175576150822952201848806616643615613562842355410104862578550863465661734839271290328348967522998634176499319107762583194718667771801067716614802322659239302476074096777926805529798115328,
    2048: 32317006071311007300714876688669951960444102669715484032130345427524655138867890893197201411522913463688717960921898019494119559150490921095088152386448283120630877367300996091750197750389652106796057638384067568276792218642619756161838094338476170470581645852036305042887575891541065808607552399123930385521914333389668342420684974786564569494856176035326322058077805659331026192708460314150258592864177116725943603718461857357598351152301645904403697613233287231227125684710820209725157101726931323469678542580656697935045997268352998638215525166389437335543602135433229604645318478604952148193555853611059596230656,
    2049: 64634012142622014601429753377339903920888205339430968064260690855049310277735781786394402823045826927377435921843796038988239118300981842190176304772896566241261754734601992183500395500779304213592115276768135136553584437285239512323676188676952340941163291704072610085775151783082131617215104798247860771043828666779336684841369949573129138989712352070652644116155611318662052385416920628300517185728354233451887207436923714715196702304603291808807395226466574462454251369421640419450314203453862646939357085161313395870091994536705997276431050332778874671087204270866459209290636957209904296387111707222119192461312,
    2559: 216650105137463389650617861497565264563425962156126783138415683273548827976634139436795999684226879687676229560321054584637120055310421758985051155612229213782023308113580557900831076179697725979446917740452505123351619721850962726871841863469394883970288762277862181198519679293724124370496977854191158541765925566879265766422347152368591546709285595026646498858206907857208870217127157835133667776604803272694950479326770851109383831436474554999730298499562859577884233389221614859959589867945489268513288025053750434670157094443550547631669932950540393572885957861184360422260925520187791145637412287222231209920958093437529187202164501625349287416117927935584199570041141450359215058287808548529345867633458852670693085809226336531763392019626315521018494466274623488,
    2560: 433300210274926779301235722995130529126851924312253566276831366547097655953268278873591999368453759375352459120642109169274240110620843517970102311224458427564046616227161115801662152359395451958893835480905010246703239443701925453743683726938789767940577524555724362397039358587448248740993955708382317083531851133758531532844694304737183093418571190053292997716413815714417740434254315670267335553209606545389900958653541702218767662872949109999460596999125719155768466778443229719919179735890978537026576050107500869340314188887101095263339865901080787145771915722368720844521851040375582291274824574444462419841916186875058374404329003250698574832235855871168399140082282900718430116575617097058691735266917705341386171618452673063526784039252631042036988932549246976,
    2561: 866600420549853558602471445990261058253703848624507132553662733094195311906536557747183998736907518750704918241284218338548480221241687035940204622448916855128093232454322231603324304718790903917787670961810020493406478887403850907487367453877579535881155049111448724794078717174896497481987911416764634167063702267517063065689388609474366186837142380106585995432827631428835480868508631340534671106419213090779801917307083404437535325745898219998921193998251438311536933556886459439838359471781957074053152100215001738680628377774202190526679731802161574291543831444737441689043702080751164582549649148888924839683832373750116748808658006501397149664471711742336798280164565801436860233151234194117383470533835410682772343236905346127053568078505262084073977865098493952,
    3071: 2904802997684979031429751266652287185343487588181447618330743076143601865498555112868668022266559203625663078877490258721995264797270023560831442836093516200516055819853220249422024925494525813600122382903520906197364840270012052413988292184690761146180604389522384946371612875869038489784405654789562755666546621759776892408153190790080930100123746284224075121257652224788593802068214369290495086275786967073127915183202957500434821866026609283416272645553951861415817069299793203345162979862593723584529770402506155104819505875374380008547680367117472878708136497428006654308479264979152338818509590797044264172530642931949135881728647441773319439777155807723223165099627191170008146028545375587766944080959493647795765768349350646133842732758718957895411577422317390130051445859016247698037520949742756905563488653739484537428521855358075060657961012278379620619506576459855478234203189721457470807178553957231283664846848,
    3072: 5809605995369958062859502533304574370686975176362895236661486152287203730997110225737336044533118407251326157754980517443990529594540047121662885672187032401032111639706440498844049850989051627200244765807041812394729680540024104827976584369381522292361208779044769892743225751738076979568811309579125511333093243519553784816306381580161860200247492568448150242515304449577187604136428738580990172551573934146255830366405915000869643732053218566832545291107903722831634138599586406690325959725187447169059540805012310209639011750748760017095360734234945757416272994856013308616958529958304677637019181594088528345061285863898271763457294883546638879554311615446446330199254382340016292057090751175533888161918987295591531536698701292267685465517437915790823154844634780260102891718032495396075041899485513811126977307478969074857043710716150121315922024556759241239013152919710956468406379442914941614357107914462567329693696,
    3073: 11619211990739916125719005066609148741373950352725790473322972304574407461994220451474672089066236814502652315509961034887981059189080094243325771344374064802064223279412880997688099701978103254400489531614083624789459361080048209655953168738763044584722417558089539785486451503476153959137622619158251022666186487039107569632612763160323720400494985136896300485030608899154375208272857477161980345103147868292511660732811830001739287464106437133665090582215807445663268277199172813380651919450374894338119081610024620419278023501497520034190721468469891514832545989712026617233917059916609355274038363188177056690122571727796543526914589767093277759108623230892892660398508764680032584114181502351067776323837974591183063073397402584535370931034875831581646309689269560520205783436064990792150083798971027622253954614957938149714087421432300242631844049113518482478026305839421912936812758885829883228714215828925134659387392,
    3583: 38947040667281689382254353725954299317181937896273053110495640486022545056571878123675414026837261387202082953137189972896247458774155287031623920822040964948589318773064700632107807079976295030781077460207726198293417869742491045883812644379740779924235838722056404143528353491961302515100905739180945234463749117814971371854735149297618231771895205448047581869036739990701238939123227617556414594513884852829118611297418625205298301228523706879215075014925865791460771763221122763958612427913913908310742775528968700838474469171501116935712407241634951353918086503727521537313584321336354300702090755194042828426528834180836453047850210518827101272970172368515183289183178786399796991077780160661742187642695507876168777794589932410371477524160198842491374886601286122761289064656316267072385897867613185015054476279468890949551641068532865528616069237517153274019287734013622385516991644530345778405646488565895551972503778313204509139728577577834190694474871418451492670973482367448545292057903483946665755750810240897272178630444779252431655195945691356582402808512048529408,
    3584: 77894081334563378764508707451908598634363875792546106220991280972045090113143756247350828053674522774404165906274379945792494917548310574063247841644081929897178637546129401264215614159952590061562154920415452396586835739484982091767625288759481559848471677444112808287056706983922605030201811478361890468927498235629942743709470298595236463543790410896095163738073479981402477878246455235112829189027769705658237222594837250410596602457047413758430150029851731582921543526442245527917224855827827816621485551057937401676948938343002233871424814483269902707836173007455043074627168642672708601404181510388085656853057668361672906095700421037654202545940344737030366578366357572799593982155560321323484375285391015752337555589179864820742955048320397684982749773202572245522578129312632534144771795735226370030108952558937781899103282137065731057232138475034306548038575468027244771033983289060691556811292977131791103945007556626409018279457155155668381388949742836902985341946964734897090584115806967893331511501620481794544357260889558504863310391891382713164805617024097058816,
    3585: 155788162669126757529017414903817197268727751585092212441982561944090180226287512494701656107349045548808331812548759891584989835096621148126495683288163859794357275092258802528431228319905180123124309840830904793173671478969964183535250577518963119696943354888225616574113413967845210060403622956723780937854996471259885487418940597190472927087580821792190327476146959962804955756492910470225658378055539411316474445189674500821193204914094827516860300059703463165843087052884491055834449711655655633242971102115874803353897876686004467742849628966539805415672346014910086149254337285345417202808363020776171313706115336723345812191400842075308405091880689474060733156732715145599187964311120642646968750570782031504675111178359729641485910096640795369965499546405144491045156258625265068289543591470452740060217905117875563798206564274131462114464276950068613096077150936054489542067966578121383113622585954263582207890015113252818036558914310311336762777899485673805970683893929469794181168231613935786663023003240963589088714521779117009726620783782765426329611234048194117632,
    4095: 522194440706576253345876355358312191289982124523691890192116741641976953985778728424413405967498779170445053357219631418993786719092896803631618043925682638972978488271854999170180795067191859157214035005927973113188159419698856372836167342172293308748403954352901852035642024370059304557233988891799014503343469488440893892973452815095130470299789726716411734651513348221529512507986199933857107770846917779942645743159118957217248367043905936319748237550094520674504208530837546834166925275516486044134775384991808184705966507606898412918594045916828375610659246423184062775112999150206172392431297837246097308511903252956622805412865917690043804311051417135098849101156584508839003337597742539960818209685142687562392007453579567729991395256699805775897135553415567045292136442139895777424891477161767258532611634530697452993846501061481697843891439474220308003706472837459911525285821188577408160690315522951458068463354171428220365223949985950890732881736611925133626529949897998045399734600887312408859224933727829625089164535236559716582775403784110923285873186648442456409760158728501220463308455437074192539205964902261490928669488824051563042951500651206733594863336608245755565801460390869016718045121902354170201577095168,
    4096: 1044388881413152506691752710716624382579964249047383780384233483283953907971557456848826811934997558340890106714439262837987573438185793607263236087851365277945956976543709998340361590134383718314428070011855946226376318839397712745672334684344586617496807908705803704071284048740118609114467977783598029006686938976881787785946905630190260940599579453432823469303026696443059025015972399867714215541693835559885291486318237914434496734087811872639496475100189041349008417061675093668333850551032972088269550769983616369411933015213796825837188091833656751221318492846368125550225998300412344784862595674492194617023806505913245610825731835380087608622102834270197698202313169017678006675195485079921636419370285375124784014907159135459982790513399611551794271106831134090584272884279791554849782954323534517065223269061394905987693002122963395687782878948440616007412945674919823050571642377154816321380631045902916136926708342856440730447899971901781465763473223850267253059899795996090799469201774624817718449867455659250178329070473119433165550807568221846571746373296884912819520317457002440926616910874148385078411929804522981857338977648103126085903001302413467189726673216491511131602920781738033436090243804708340403154190336,
    4097: 2088777762826305013383505421433248765159928498094767560768466966567907815943114913697653623869995116681780213428878525675975146876371587214526472175702730555891913953087419996680723180268767436628856140023711892452752637678795425491344669368689173234993615817411607408142568097480237218228935955567196058013373877953763575571893811260380521881199158906865646938606053392886118050031944799735428431083387671119770582972636475828868993468175623745278992950200378082698016834123350187336667701102065944176539101539967232738823866030427593651674376183667313502442636985692736251100451996600824689569725191348984389234047613011826491221651463670760175217244205668540395396404626338035356013350390970159843272838740570750249568029814318270919965581026799223103588542213662268181168545768559583109699565908647069034130446538122789811975386004245926791375565757896881232014825891349839646101143284754309632642761262091805832273853416685712881460895799943803562931526946447700534506119799591992181598938403549249635436899734911318500356658140946238866331101615136443693143492746593769825639040634914004881853233821748296770156823859609045963714677955296206252171806002604826934379453346432983022263205841563476066872180487609416680806308380672,
}


primes = {
    # A static table of primes for efficient and/or deterministic
    # functionality requiring prime numbers. The primes are organized
    # so there's two primes in a list in each element of the dictionary.
    # The first & last element of each list contains the first & last
    # prime of bit length ``key``.
    0: [],
    1: [],
    2: [2, 3],
    3: [5, 7],
    4: [11, 13],
    5: [19, 31],
    6: [37, 61],
    7: [67, 127],
    8: [131, 251],
    9: [257, 509],
    10: [521, 1021],
    11: [1031, 2039],
    12: [2053, 4093],
    13: [4099, 8191],
    14: [8209, 16381],
    15: [16411, 32749],
    16: [32771, 65521],
    17: [65537, 131071],
    18: [131101, 262139],
    19: [262147, 524287],
    20: [524309, 1048573],
    21: [1048583, 2097143],
    22: [2097169, 4194301],
    23: [4194319, 8388593],
    24: [8388617, 16777213],
    25: [16777259, 33554393],
    26: [33554467, 67108859],
    27: [67108879, 134217689],
    28: [134217757, 268435399],
    29: [268435459, 536870909],
    30: [536870923, 1073741789],
    31: [1073741827, 2147483647],
    32: [2147483659, 4294967291],
    33: [4294967311, 8589934583],
    34: [8589934609, 17179869143],
    35: [17179869209, 34359738337],
    36: [34359738421, 68719476731],
    37: [68719476767, 137438953447],
    38: [137438953481, 274877906899],
    39: [274877906951, 549755813881],
    40: [549755813911, 1099511627689],
    41: [1099511627791, 2199023255531],
    42: [2199023255579, 4398046511093],
    43: [4398046511119, 8796093022151],
    44: [8796093022237, 17592186044399],
    45: [17592186044423, 35184372088777],
    46: [35184372088891, 70368744177643],
    47: [70368744177679, 140737488355213],
    48: [140737488355333, 281474976710597],
    49: [281474976710677, 562949953421231],
    50: [562949953421381, 1125899906842597],
    51: [1125899906842679, 2251799813685119],
    52: [2251799813685269, 4503599627370449],
    53: [4503599627370517, 9007199254740881],
    54: [9007199254740997, 18014398509481951],
    55: [18014398509482143, 36028797018963913],
    56: [36028797018963971, 72057594037927931],
    57: [72057594037928017, 144115188075855859],
    58: [144115188075855881, 288230376151711717],
    59: [288230376151711813, 576460752303423433],
    60: [576460752303423619, 1152921504606846883],
    61: [1152921504606847009, 2305843009213693951],
    62: [2305843009213693967, 4611686018427387847],
    63: [4611686018427388039, 9223372036854775783],
    64: [9223372036854775837, 18446744073709551557],
    65: [18446744073709551629, 36893488147419103183],
    66: [36893488147419103363, 73786976294838206459],
    67: [73786976294838206473, 147573952589676412909],
    68: [147573952589676412931, 295147905179352825833],
    69: [295147905179352825889, 590295810358705651693],
    70: [590295810358705651741, 1180591620717411303389],
    71: [1180591620717411303449, 2361183241434822606617],
    72: [2361183241434822606859, 4722366482869645213603],
    73: [4722366482869645213711, 9444732965739290427323],
    74: [9444732965739290427421, 18889465931478580854749],
    75: [18889465931478580854821, 37778931862957161709471],
    76: [37778931862957161709601, 75557863725914323419121],
    77: [75557863725914323419151, 151115727451828646838239],
    78: [151115727451828646838283, 302231454903657293676533],
    79: [302231454903657293676551, 604462909807314587353021],
    80: [604462909807314587353111, 1208925819614629174706111],
    81: [1208925819614629174706189, 2417851639229258349412301],
    82: [2417851639229258349412369, 4835703278458516698824647],
    83: [4835703278458516698824713, 9671406556917033397649353],
    84: [9671406556917033397649483, 19342813113834066795298781],
    85: [19342813113834066795298819, 38685626227668133590597613],
    86: [38685626227668133590597803, 77371252455336267181195229],
    87: [77371252455336267181195291, 154742504910672534362390461],
    88: [154742504910672534362390567, 309485009821345068724780757],
    89: [309485009821345068724781063, 618970019642690137449562111],
    90: [618970019642690137449562141, 1237940039285380274899124191],
    91: [1237940039285380274899124357, 2475880078570760549798248403],
    92: [2475880078570760549798248507, 4951760157141521099596496813],
    93: [4951760157141521099596496921, 9903520314283042199192993767],
    94: [9903520314283042199192993897, 19807040628566084398385987581],
    95: [19807040628566084398385987713, 39614081257132168796771975153],
    96: [39614081257132168796771975177, 79228162514264337593543950319],
    97: [79228162514264337593543950397, 158456325028528675187087900531],
    98: [158456325028528675187087900777, 316912650057057350374175801293],
    99: [316912650057057350374175801351, 633825300114114700748351602573],
    100: [633825300114114700748351602943, 1267650600228229401496703205361],
    101: [1267650600228229401496703205653, 2535301200456458802993406410683],
    102: [2535301200456458802993406410833, 5070602400912917605986812821471],
    103: [
        5070602400912917605986812821771,
        10141204801825835211973625642911,
    ],
    104: [
        10141204801825835211973625643089,
        20282409603651670423947251285999,
    ],
    105: [
        20282409603651670423947251286127,
        40564819207303340847894502572019,
    ],
    106: [
        40564819207303340847894502572071,
        81129638414606681695789005143947,
    ],
    107: [
        81129638414606681695789005144163,
        162259276829213363391578010288127,
    ],
    108: [
        162259276829213363391578010288167,
        324518553658426726783156020576197,
    ],
    109: [
        324518553658426726783156020576289,
        649037107316853453566312041152481,
    ],
    110: [
        649037107316853453566312041152659,
        1298074214633706907132624082305003,
    ],
    111: [
        1298074214633706907132624082305051,
        2596148429267413814265248164610011,
    ],
    112: [
        2596148429267413814265248164610099,
        5192296858534827628530496329220021,
    ],
    113: [
        5192296858534827628530496329220121,
        10384593717069655257060992658440059,
    ],
    114: [
        10384593717069655257060992658440473,
        20769187434139310514121985316880373,
    ],
    115: [
        20769187434139310514121985316880427,
        41538374868278621028243970633760701,
    ],
    116: [
        41538374868278621028243970633760839,
        83076749736557242056487941267521533,
    ],
    117: [
        83076749736557242056487941267521569,
        166153499473114484112975882535042793,
    ],
    118: [
        166153499473114484112975882535043101,
        332306998946228968225951765070086139,
    ],
    119: [
        332306998946228968225951765070086169,
        664613997892457936451903530140172219,
    ],
    120: [
        664613997892457936451903530140172297,
        1329227995784915872903807060280344457,
    ],
    121: [
        1329227995784915872903807060280345027,
        2658455991569831745807614120560689079,
    ],
    122: [
        2658455991569831745807614120560689193,
        5316911983139663491615228241121378301,
    ],
    123: [
        5316911983139663491615228241121378581,
        10633823966279326983230456482242756541,
    ],
    124: [
        10633823966279326983230456482242756773,
        21267647932558653966460912964485513157,
    ],
    125: [
        21267647932558653966460912964485513283,
        42535295865117307932921825928971026423,
    ],
    126: [
        42535295865117307932921825928971026459,
        85070591730234615865843651857942052727,
    ],
    127: [
        85070591730234615865843651857942052871,
        170141183460469231731687303715884105727,
    ],
    128: [
        170141183460469231731687303715884105757,
        340282366920938463463374607431768211297,
    ],
    129: [
        340282366920938463463374607431768211507,
        680564733841876926926749214863536422887,
    ],
    130: [
        680564733841876926926749214863536422929,
        1361129467683753853853498429727072845819,
    ],
    131: [
        1361129467683753853853498429727072845993,
        2722258935367507707706996859454145691579,
    ],
    132: [
        2722258935367507707706996859454145691687,
        5444517870735015415413993718908291382949,
    ],
    133: [
        5444517870735015415413993718908291383363,
        10889035741470030830827987437816582766493,
    ],
    134: [
        10889035741470030830827987437816582766619,
        21778071482940061661655974875633165533139,
    ],
    135: [
        21778071482940061661655974875633165533211,
        43556142965880123323311949751266331066323,
    ],
    136: [
        43556142965880123323311949751266331066401,
        87112285931760246646623899502532662132623,
    ],
    137: [
        87112285931760246646623899502532662132821,
        174224571863520493293247799005065324265459,
    ],
    138: [
        174224571863520493293247799005065324265627,
        348449143727040986586495598010130648530839,
    ],
    139: [
        348449143727040986586495598010130648531031,
        696898287454081973172991196020261297061701,
    ],
    140: [
        696898287454081973172991196020261297062043,
        1393796574908163946345982392040522594123749,
    ],
    141: [
        1393796574908163946345982392040522594123813,
        2787593149816327892691964784081045188247543,
    ],
    142: [
        2787593149816327892691964784081045188247557,
        5575186299632655785383929568162090376494993,
    ],
    143: [
        5575186299632655785383929568162090376495321,
        11150372599265311570767859136324180752990139,
    ],
    144: [
        11150372599265311570767859136324180752990213,
        22300745198530623141535718272648361505980333,
    ],
    145: [
        22300745198530623141535718272648361505980591,
        44601490397061246283071436545296723011960681,
    ],
    146: [
        44601490397061246283071436545296723011960859,
        89202980794122492566142873090593446023921511,
    ],
    147: [
        89202980794122492566142873090593446023921749,
        178405961588244985132285746181186892047843183,
    ],
    148: [
        178405961588244985132285746181186892047843379,
        356811923176489970264571492362373784095686489,
    ],
    149: [
        356811923176489970264571492362373784095686747,
        713623846352979940529142984724747568191373281,
    ],
    150: [
        713623846352979940529142984724747568191373381,
        1427247692705959881058285969449495136382746621,
    ],
    151: [
        1427247692705959881058285969449495136382746771,
        2854495385411919762116571938898990272765493053,
    ],
    152: [
        2854495385411919762116571938898990272765493293,
        5708990770823839524233143877797980545530986479,
    ],
    153: [
        5708990770823839524233143877797980545530986749,
        11417981541647679048466287755595961091061972923,
    ],
    154: [
        11417981541647679048466287755595961091061973087,
        22835963083295358096932575511191922182123945741,
    ],
    155: [
        22835963083295358096932575511191922182123946011,
        45671926166590716193865151022383844364247891937,
    ],
    156: [
        45671926166590716193865151022383844364247891983,
        91343852333181432387730302044767688728495783793,
    ],
    157: [
        91343852333181432387730302044767688728495783981,
        182687704666362864775460604089535377456991567853,
    ],
    158: [
        182687704666362864775460604089535377456991567941,
        365375409332725729550921208179070754913983135729,
    ],
    159: [
        365375409332725729550921208179070754913983135841,
        730750818665451459101842416358141509827966271397,
    ],
    160: [
        730750818665451459101842416358141509827966271787,
        1461501637330902918203684832716283019655932542929,
    ],
    161: [
        1461501637330902918203684832716283019655932542983,
        2923003274661805836407369665432566039311865085793,
    ],
    162: [
        2923003274661805836407369665432566039311865086059,
        5846006549323611672814739330865132078623730171803,
    ],
    163: [
        5846006549323611672814739330865132078623730171923,
        11692013098647223345629478661730264157247460343753,
    ],
    164: [
        11692013098647223345629478661730264157247460343829,
        23384026197294446691258957323460528314494920687553,
    ],
    165: [
        23384026197294446691258957323460528314494920687733,
        46768052394588893382517914646921056628989841375207,
    ],
    166: [
        46768052394588893382517914646921056628989841375373,
        93536104789177786765035829293842113257979682750459,
    ],
    167: [
        93536104789177786765035829293842113257979682750549,
        187072209578355573530071658587684226515959365500793,
    ],
    168: [
        187072209578355573530071658587684226515959365501011,
        374144419156711147060143317175368453031918731001599,
    ],
    169: [
        374144419156711147060143317175368453031918731001943,
        748288838313422294120286634350736906063837462003069,
    ],
    170: [
        748288838313422294120286634350736906063837462003859,
        1496577676626844588240573268701473812127674924007281,
    ],
    171: [
        1496577676626844588240573268701473812127674924007473,
        2993155353253689176481146537402947624255349848014829,
    ],
    172: [
        2993155353253689176481146537402947624255349848014977,
        5986310706507378352962293074805895248510699696029601,
    ],
    173: [
        5986310706507378352962293074805895248510699696029801,
        11972621413014756705924586149611790497021399392059337,
    ],
    174: [
        11972621413014756705924586149611790497021399392059469,
        23945242826029513411849172299223580994042798784118781,
    ],
    175: [
        23945242826029513411849172299223580994042798784118791,
        47890485652059026823698344598447161988085597568237339,
    ],
    176: [
        47890485652059026823698344598447161988085597568237577,
        95780971304118053647396689196894323976171195136474903,
    ],
    177: [
        95780971304118053647396689196894323976171195136475563,
        191561942608236107294793378393788647952342390272949933,
    ],
    178: [
        191561942608236107294793378393788647952342390272950347,
        383123885216472214589586756787577295904684780545900503,
    ],
    179: [
        383123885216472214589586756787577295904684780545900631,
        766247770432944429179173513575154591809369561091801039,
    ],
    180: [
        766247770432944429179173513575154591809369561091801397,
        1532495540865888858358347027150309183618739122183602129,
    ],
    181: [
        1532495540865888858358347027150309183618739122183602191,
        3064991081731777716716694054300618367237478244367204187,
    ],
    182: [
        3064991081731777716716694054300618367237478244367204517,
        6129982163463555433433388108601236734474956488734408543,
    ],
    183: [
        6129982163463555433433388108601236734474956488734408753,
        12259964326927110866866776217202473468949912977468817261,
    ],
    184: [
        12259964326927110866866776217202473468949912977468817623,
        24519928653854221733733552434404946937899825954937634783,
    ],
    185: [
        24519928653854221733733552434404946937899825954937634843,
        49039857307708443467467104868809893875799651909875269329,
    ],
    186: [
        49039857307708443467467104868809893875799651909875269791,
        98079714615416886934934209737619787751599303819750538893,
    ],
    187: [
        98079714615416886934934209737619787751599303819750539469,
        196159429230833773869868419475239575503198607639501078443,
    ],
    188: [
        196159429230833773869868419475239575503198607639501078831,
        392318858461667547739736838950479151006397215279002156931,
    ],
    189: [
        392318858461667547739736838950479151006397215279002157113,
        784637716923335095479473677900958302012794430558004314087,
    ],
    190: [
        784637716923335095479473677900958302012794430558004314147,
        1569275433846670190958947355801916604025588861116008628213,
    ],
    191: [
        1569275433846670190958947355801916604025588861116008628353,
        3138550867693340381917894711603833208051177722232017256429,
    ],
    192: [
        3138550867693340381917894711603833208051177722232017256453,
        6277101735386680763835789423207666416102355444464034512659,
    ],
    193: [
        6277101735386680763835789423207666416102355444464034513029,
        12554203470773361527671578846415332832204710888928069025761,
    ],
    194: [
        12554203470773361527671578846415332832204710888928069025857,
        25108406941546723055343157692830665664409421777856138051551,
    ],
    195: [
        25108406941546723055343157692830665664409421777856138051611,
        50216813883093446110686315385661331328818843555712276103033,
    ],
    196: [
        50216813883093446110686315385661331328818843555712276103203,
        100433627766186892221372630771322662657637687111424552206321,
    ],
    197: [
        100433627766186892221372630771322662657637687111424552206357,
        200867255532373784442745261542645325315275374222849104412597,
    ],
    198: [
        200867255532373784442745261542645325315275374222849104412779,
        401734511064747568885490523085290650630550748445698208825327,
    ],
    199: [
        401734511064747568885490523085290650630550748445698208825359,
        803469022129495137770981046170581301261101496891396417650639,
    ],
    200: [
        803469022129495137770981046170581301261101496891396417650789,
        1606938044258990275541962092341162602522202993782792835301301,
    ],
    201: [
        1606938044258990275541962092341162602522202993782792835301611,
        3213876088517980551083924184682325205044405987565585670602697,
    ],
    202: [
        3213876088517980551083924184682325205044405987565585670603103,
        6427752177035961102167848369364650410088811975131171341205321,
    ],
    203: [
        6427752177035961102167848369364650410088811975131171341205571,
        12855504354071922204335696738729300820177623950262342682410849,
    ],
    204: [
        12855504354071922204335696738729300820177623950262342682411023,
        25711008708143844408671393477458601640355247900524685364821849,
    ],
    205: [
        25711008708143844408671393477458601640355247900524685364822023,
        51422017416287688817342786954917203280710495801049370729643951,
    ],
    206: [
        51422017416287688817342786954917203280710495801049370729644613,
        102844034832575377634685573909834406561420991602098741459288059,
    ],
    207: [
        102844034832575377634685573909834406561420991602098741459288097,
        205688069665150755269371147819668813122841983204197482918576037,
    ],
    208: [
        205688069665150755269371147819668813122841983204197482918576331,
        411376139330301510538742295639337626245683966408394965837151957,
    ],
    209: [
        411376139330301510538742295639337626245683966408394965837152631,
        822752278660603021077484591278675252491367932816789931674304479,
    ],
    210: [
        822752278660603021077484591278675252491367932816789931674304559,
        1645504557321206042154969182557350504982735865633579863348608977,
    ],
    211: [
        1645504557321206042154969182557350504982735865633579863348609057,
        3291009114642412084309938365114701009965471731267159726697217873,
    ],
    212: [
        3291009114642412084309938365114701009965471731267159726697218119,
        6582018229284824168619876730229402019930943462534319453394436073,
    ],
    213: [
        6582018229284824168619876730229402019930943462534319453394436153,
        13164036458569648337239753460458804039861886925068638906788872189,
    ],
    214: [
        13164036458569648337239753460458804039861886925068638906788872267,
        26328072917139296674479506920917608079723773850137277813577744199,
    ],
    215: [
        26328072917139296674479506920917608079723773850137277813577744391,
        52656145834278593348959013841835216159447547700274555627155488611,
    ],
    216: [
        52656145834278593348959013841835216159447547700274555627155489019,
        105312291668557186697918027683670432318895095400549111254310977159,
    ],
    217: [
        105312291668557186697918027683670432318895095400549111254310977959,
        210624583337114373395836055367340864637790190801098222508621955011,
    ],
    218: [
        210624583337114373395836055367340864637790190801098222508621955201,
        421249166674228746791672110734681729275580381602196445017243910111,
    ],
    219: [
        421249166674228746791672110734681729275580381602196445017243910307,
        842498333348457493583344221469363458551160763204392890034487820167,
    ],
    220: [
        842498333348457493583344221469363458551160763204392890034487820473,
        1684996666696914987166688442938726917102321526408785780068975640499,
    ],
    221: [
        1684996666696914987166688442938726917102321526408785780068975640793,
        3369993333393829974333376885877453834204643052817571560137951281149,
    ],
    222: [
        3369993333393829974333376885877453834204643052817571560137951281233,
        6739986666787659948666753771754907668409286105635143120275902562187,
    ],
    223: [
        6739986666787659948666753771754907668409286105635143120275902562353,
        13479973333575319897333507543509815336818572211270286240551805124373,
    ],
    224: [
        13479973333575319897333507543509815336818572211270286240551805124797,
        26959946667150639794667015087019630673637144422540572481103610249153,
    ],
    225: [
        26959946667150639794667015087019630673637144422540572481103610249951,
        53919893334301279589334030174039261347274288845081144962207220498383,
    ],
    226: [
        53919893334301279589334030174039261347274288845081144962207220498551,
        107839786668602559178668060348078522694548577690162289924414440996859,
    ],
    227: [
        107839786668602559178668060348078522694548577690162289924414440997599,
        215679573337205118357336120696157045389097155380324579848828881993323,
    ],
    228: [
        215679573337205118357336120696157045389097155380324579848828881994211,
        431359146674410236714672241392314090778194310760649159697657763987363,
    ],
    229: [
        431359146674410236714672241392314090778194310760649159697657763987459,
        862718293348820473429344482784628181556388621521298319395315527974821,
    ],
    230: [
        862718293348820473429344482784628181556388621521298319395315527975161,
        1725436586697640946858688965569256363112777243042596638790631055949797,
    ],
    231: [
        1725436586697640946858688965569256363112777243042596638790631055949891,
        3450873173395281893717377931138512726225554486085193277581262111899483,
    ],
    232: [
        3450873173395281893717377931138512726225554486085193277581262111899753,
        6901746346790563787434755862277025452451108972170386555162524223798729,
    ],
    233: [
        6901746346790563787434755862277025452451108972170386555162524223799653,
        13803492693581127574869511724554050904902217944340773110325048447598589,
    ],
    234: [
        13803492693581127574869511724554050904902217944340773110325048447599023,
        27606985387162255149739023449108101809804435888681546220650096895197101,
    ],
    235: [
        27606985387162255149739023449108101809804435888681546220650096895197227,
        55213970774324510299478046898216203619608871777363092441300193790394353,
    ],
    236: [
        55213970774324510299478046898216203619608871777363092441300193790394449,
        110427941548649020598956093796432407239217743554726184882600387580788527,
    ],
    237: [
        110427941548649020598956093796432407239217743554726184882600387580788761,
        220855883097298041197912187592864814478435487109452369765200775161577291,
    ],
    238: [
        220855883097298041197912187592864814478435487109452369765200775161577721,
        441711766194596082395824375185729628956870974218904739530401550323154783,
    ],
    239: [
        441711766194596082395824375185729628956870974218904739530401550323155011,
        883423532389192164791648750371459257913741948437809479060803100646309801,
    ],
    240: [
        883423532389192164791648750371459257913741948437809479060803100646309917,
        1766847064778384329583297500742918515827483896875618958121606201292619309,
    ],
    241: [
        1766847064778384329583297500742918515827483896875618958121606201292619891,
        3533694129556768659166595001485837031654967793751237916243212402585239513,
    ],
    242: [
        3533694129556768659166595001485837031654967793751237916243212402585239813,
        7067388259113537318333190002971674063309935587502475832486424805170479041,
    ],
    243: [
        7067388259113537318333190002971674063309935587502475832486424805170479173,
        14134776518227074636666380005943348126619871175004951664972849610340958199,
    ],
    244: [
        14134776518227074636666380005943348126619871175004951664972849610340958267,
        28269553036454149273332760011886696253239742350009903329945699220681916227,
    ],
    245: [
        28269553036454149273332760011886696253239742350009903329945699220681916549,
        56539106072908298546665520023773392506479484700019806659891398441363832669,
    ],
    246: [
        56539106072908298546665520023773392506479484700019806659891398441363833147,
        113078212145816597093331040047546785012958969400039613319782796882727665557,
    ],
    247: [
        113078212145816597093331040047546785012958969400039613319782796882727666001,
        226156424291633194186662080095093570025917938800079226639565593765455331247,
    ],
    248: [
        226156424291633194186662080095093570025917938800079226639565593765455331391,
        452312848583266388373324160190187140051835877600158453279131187530910662419,
    ],
    249: [
        452312848583266388373324160190187140051835877600158453279131187530910662737,
        904625697166532776746648320380374280103671755200316906558262375061821325237,
    ],
    250: [
        904625697166532776746648320380374280103671755200316906558262375061821325431,
        1809251394333065553493296640760748560207343510400633813116524750123642650417,
    ],
    251: [
        1809251394333065553493296640760748560207343510400633813116524750123642650649,
        3618502788666131106986593281521497120414687020801267626233049500247285301239,
    ],
    252: [
        3618502788666131106986593281521497120414687020801267626233049500247285301313,
        7237005577332262213973186563042994240829374041602535252466099000494570602367,
    ],
    253: [
        7237005577332262213973186563042994240829374041602535252466099000494570602917,
        14474011154664524427946373126085988481658748083205070504932198000989141204719,
    ],
    254: [
        14474011154664524427946373126085988481658748083205070504932198000989141205031,
        28948022309329048855892746252171976963317496166410141009864396001978282409739,
    ],
    255: [
        28948022309329048855892746252171976963317496166410141009864396001978282410063,
        57896044618658097711785492504343953926634992332820282019728792003956564819949,
    ],
    256: [
        57896044618658097711785492504343953926634992332820282019728792003956564820063,
        115792089237316195423570985008687907853269984665640564039457584007913129639747,
    ],
    257: [
        115792089237316195423570985008687907853269984665640564039457584007913129640233,
        231584178474632390847141970017375815706539969331281128078915168015826259279779,
    ],
    383: [
        9850501549098619803069760025035903451269934817616361666987073351061430442874302652853566563721228910201656997576959,
        19701003098197239606139520050071806902539869635232723333974146702122860885748605305707133127442457820403313995153377,
    ],
    384: [
        19701003098197239606139520050071806902539869635232723333974146702122860885748605305707133127442457820403313995153777,
        39402006196394479212279040100143613805079739270465446667948293404245721771497210611414266254884915640806627990306499,
    ],
    385: [
        39402006196394479212279040100143613805079739270465446667948293404245721771497210611414266254884915640806627990307047,
        78804012392788958424558080200287227610159478540930893335896586808491443542994421222828532509769831281613255980613367,
    ],
    511: [
        3351951982485649274893506249551461531869841455148098344430890360930441007518386744200468574541725856922507964546621512713438470702986642486608412251521039,
        6703903964971298549787012499102923063739682910296196688861780721860882015036773488400937149083451713845015929093243025426876941405973284973216824503041861,
    ],
    512: [
        6703903964971298549787012499102923063739682910296196688861780721860882015036773488400937149083451713845015929093243025426876941405973284973216824503042159,
        13407807929942597099574024998205846127479365820592393377723561443721764030073546976801874298166903427690031858186486050853753882811946569946433649006083527,
    ],
    513: [
        13407807929942597099574024998205846127479365820592393377723561443721764030073546976801874298166903427690031858186486050853753882811946569946433649006084171,
        26815615859885194199148049996411692254958731641184786755447122887443528060147093953603748596333806855380063716372972101707507765623893139892867298012167747,
    ],
    1023: [
        44942328371557897693232629769725618340449424473557664318357520289433168951375240783177119330601884005280028469967848339414697442203604155623211857659868531094441973356216371319075554900311523529863270738021251442209537670585615720368478277635206809290837627671146574559986811484619929076208839082406056035747,
        89884656743115795386465259539451236680898848947115328636715040578866337902750481566354238661203768010560056939935696678829394884407208311246423715319737062188883946712432742638151109800623047059726541476042502884419075341171231440736956555270413618581675255342293149119973622969239858152417678164812112068247,
    ],
    1024: [
        89884656743115795386465259539451236680898848947115328636715040578866337902750481566354238661203768010560056939935696678829394884407208311246423715319737062188883946712432742638151109800623047059726541476042502884419075341171231440736956555270413618581675255342293149119973622969239858152417678164812112069763,
        179769313486231590772930519078902473361797697894230657273430081157732675805500963132708477322407536021120113879871393357658789768814416622492847430639474124377767893424865485276302219601246094119453082952085005768838150682342462881473913110540827237163350510684586298239947245938479716304835356329624224137111,
    ],
    1025: [
        179769313486231590772930519078902473361797697894230657273430081157732675805500963132708477322407536021120113879871393357658789768814416622492847430639474124377767893424865485276302219601246094119453082952085005768838150682342462881473913110540827237163350510684586298239947245938479716304835356329624224137859,
        359538626972463181545861038157804946723595395788461314546860162315465351611001926265416954644815072042240227759742786715317579537628833244985694861278948248755535786849730970552604439202492188238906165904170011537676301364684925762947826221081654474326701021369172596479894491876959432609670712659248448271759,
    ],
    1535: [
        602578106730258147145029151507078528228023311986422237839918759766314347897950800167271256026836512415862191570222001196965604244698739581242403246972693662863803334845406306192695519479420374919211385784346955014399336464476149777365346780524876991249453910335575169407368338820404357102948540991946467592592242277305397985763558002890689612520144896962725248428723070869161657795378765951218343795565126561748209474676492753131460831100308246714251190084829449,
        1205156213460516294290058303014157056456046623972844475679837519532628695795901600334542512053673024831724383140444002393931208489397479162484806493945387325727606669690812612385391038958840749838422771568693910028798672928952299554730693561049753982498907820671150338814736677640808714205897081983892935185184484554610795971527116005781379225040289793925450496857446141738323315590757531902436687591130253123496418949352985506262921662200616493428502380169657051,
    ],
    1536: [
        1205156213460516294290058303014157056456046623972844475679837519532628695795901600334542512053673024831724383140444002393931208489397479162484806493945387325727606669690812612385391038958840749838422771568693910028798672928952299554730693561049753982498907820671150338814736677640808714205897081983892935185184484554610795971527116005781379225040289793925450496857446141738323315590757531902436687591130253123496418949352985506262921662200616493428502380169659067,
        2410312426921032588580116606028314112912093247945688951359675039065257391591803200669085024107346049663448766280888004787862416978794958324969612987890774651455213339381625224770782077917681499676845543137387820057597345857904599109461387122099507964997815641342300677629473355281617428411794163967785870370368969109221591943054232011562758450080579587850900993714892283476646631181515063804873375182260506246992837898705971012525843324401232986857004760339313283,
    ],
    1537: [
        2410312426921032588580116606028314112912093247945688951359675039065257391591803200669085024107346049663448766280888004787862416978794958324969612987890774651455213339381625224770782077917681499676845543137387820057597345857904599109461387122099507964997815641342300677629473355281617428411794163967785870370368969109221591943054232011562758450080579587850900993714892283476646631181515063804873375182260506246992837898705971012525843324401232986857004760339316811,
        4820624853842065177160233212056628225824186495891377902719350078130514783183606401338170048214692099326897532561776009575724833957589916649939225975781549302910426678763250449541564155835362999353691086274775640115194691715809198218922774244199015929995631282684601355258946710563234856823588327935571740740737938218443183886108464023125516900161159175701801987429784566953293262363030127609746750364521012493985675797411942025051686648802465973714009520678633297,
    ],
    2047: [
        8079251517827751825178719172167487990111025667428871008032586356881163784716972723299300352880728365922179490230474504873529889787622730273772038096612070780157719341825249022937549437597413026699014409596016892069198054660654939040459523584619042617645411463009076260721893972885266452151888099780982596380478583347417085605171243696641142373714044008831580514519451414832756548177115078537564648216044279181485900929615464339399587788075411476100924403308321807806781421177705052431289275431732830867419635645164174483761499317088249659553881291597359333885900533858307401161329619651238037048388963402764899061811,
        16158503035655503650357438344334975980222051334857742016065172713762327569433945446598600705761456731844358980460949009747059779575245460547544076193224141560315438683650498045875098875194826053398028819192033784138396109321309878080919047169238085235290822926018152521443787945770532904303776199561965192760957166694834171210342487393282284747428088017663161029038902829665513096354230157075129296432088558362971801859230928678799175576150822952201848806616643615613562842355410104862578550863465661734839271290328348967522998634176499319107762583194718667771801067716614802322659239302476074096777926805529798115243,
    ],
    2048: [
        16158503035655503650357438344334975980222051334857742016065172713762327569433945446598600705761456731844358980460949009747059779575245460547544076193224141560315438683650498045875098875194826053398028819192033784138396109321309878080919047169238085235290822926018152521443787945770532904303776199561965192760957166694834171210342487393282284747428088017663161029038902829665513096354230157075129296432088558362971801859230928678799175576150822952201848806616643615613562842355410104862578550863465661734839271290328348967522998634176499319107762583194718667771801067716614802322659239302476074096777926805529798117247,
        32317006071311007300714876688669951960444102669715484032130345427524655138867890893197201411522913463688717960921898019494119559150490921095088152386448283120630877367300996091750197750389652106796057638384067568276792218642619756161838094338476170470581645852036305042887575891541065808607552399123930385521914333389668342420684974786564569494856176035326322058077805659331026192708460314150258592864177116725943603718461857357598351152301645904403697613233287231227125684710820209725157101726931323469678542580656697935045997268352998638215525166389437335543602135433229604645318478604952148193555853611059596229099,
    ],
    2049: [
        32317006071311007300714876688669951960444102669715484032130345427524655138867890893197201411522913463688717960921898019494119559150490921095088152386448283120630877367300996091750197750389652106796057638384067568276792218642619756161838094338476170470581645852036305042887575891541065808607552399123930385521914333389668342420684974786564569494856176035326322058077805659331026192708460314150258592864177116725943603718461857357598351152301645904403697613233287231227125684710820209725157101726931323469678542580656697935045997268352998638215525166389437335543602135433229604645318478604952148193555853611059596231637,
        64634012142622014601429753377339903920888205339430968064260690855049310277735781786394402823045826927377435921843796038988239118300981842190176304772896566241261754734601992183500395500779304213592115276768135136553584437285239512323676188676952340941163291704072610085775151783082131617215104798247860771043828666779336684841369949573129138989712352070652644116155611318662052385416920628300517185728354233451887207436923714715196702304603291808807395226466574462454251369421640419450314203453862646939357085161313395870091994536705997276431050332778874671087204270866459209290636957209904296387111707222119192459863,
    ],
    2559: [
        108325052568731694825308930748782632281712981078063391569207841636774413988317069718397999842113439843838114780160527292318560027655210879492525577806114606891011654056790278950415538089848862989723458870226252561675809860925481363435920931734697441985144381138931090599259839646862062185248488927095579270882962783439632883211173576184295773354642797513323249429103453928604435108563578917566833888302401636347475239663385425554691915718237277499865149249781429788942116694610807429979794933972744634256644012526875217335078547221775273815834966475270196786442978930592180211130462760093895572818706143611115604960479046718764593601082250812674643708058963967792099785020570725179607529143904274264672933816729426335346542904613168265881696009813157760509247233137312647,
        216650105137463389650617861497565264563425962156126783138415683273548827976634139436795999684226879687676229560321054584637120055310421758985051155612229213782023308113580557900831076179697725979446917740452505123351619721850962726871841863469394883970288762277862181198519679293724124370496977854191158541765925566879265766422347152368591546709285595026646498858206907857208870217127157835133667776604803272694950479326770851109383831436474554999730298499562859577884233389221614859959589867945489268513288025053750434670157094443550547631669932950540393572885957861184360422260925520187791145637412287222231209920958093437529187202164501625349287416117927935584199570041141450359215058287808548529345867633458852670693085809226336531763392019626315521018494466274622779,
    ],
    2560: [
        216650105137463389650617861497565264563425962156126783138415683273548827976634139436795999684226879687676229560321054584637120055310421758985051155612229213782023308113580557900831076179697725979446917740452505123351619721850962726871841863469394883970288762277862181198519679293724124370496977854191158541765925566879265766422347152368591546709285595026646498858206907857208870217127157835133667776604803272694950479326770851109383831436474554999730298499562859577884233389221614859959589867945489268513288025053750434670157094443550547631669932950540393572885957861184360422260925520187791145637412287222231209920958093437529187202164501625349287416117927935584199570041141450359215058287808548529345867633458852670693085809226336531763392019626315521018494466274625771,
        433300210274926779301235722995130529126851924312253566276831366547097655953268278873591999368453759375352459120642109169274240110620843517970102311224458427564046616227161115801662152359395451958893835480905010246703239443701925453743683726938789767940577524555724362397039358587448248740993955708382317083531851133758531532844694304737183093418571190053292997716413815714417740434254315670267335553209606545389900958653541702218767662872949109999460596999125719155768466778443229719919179735890978537026576050107500869340314188887101095263339865901080787145771915722368720844521851040375582291274824574444462419841916186875058374404329003250698574832235855871168399140082282900718430116575617097058691735266917705341386171618452673063526784039252631042036988932549246901,
    ],
    2561: [
        433300210274926779301235722995130529126851924312253566276831366547097655953268278873591999368453759375352459120642109169274240110620843517970102311224458427564046616227161115801662152359395451958893835480905010246703239443701925453743683726938789767940577524555724362397039358587448248740993955708382317083531851133758531532844694304737183093418571190053292997716413815714417740434254315670267335553209606545389900958653541702218767662872949109999460596999125719155768466778443229719919179735890978537026576050107500869340314188887101095263339865901080787145771915722368720844521851040375582291274824574444462419841916186875058374404329003250698574832235855871168399140082282900718430116575617097058691735266917705341386171618452673063526784039252631042036988932549247879,
        866600420549853558602471445990261058253703848624507132553662733094195311906536557747183998736907518750704918241284218338548480221241687035940204622448916855128093232454322231603324304718790903917787670961810020493406478887403850907487367453877579535881155049111448724794078717174896497481987911416764634167063702267517063065689388609474366186837142380106585995432827631428835480868508631340534671106419213090779801917307083404437535325745898219998921193998251438311536933556886459439838359471781957074053152100215001738680628377774202190526679731802161574291543831444737441689043702080751164582549649148888924839683832373750116748808658006501397149664471711742336798280164565801436860233151234194117383470533835410682772343236905346127053568078505262084073977865098493337,
    ],
    3071: [
        1452401498842489515714875633326143592671743794090723809165371538071800932749277556434334011133279601812831539438745129360997632398635011780415721418046758100258027909926610124711012462747262906800061191451760453098682420135006026206994146092345380573090302194761192473185806437934519244892202827394781377833273310879888446204076595395040465050061873142112037560628826112394296901034107184645247543137893483536563957591601478750217410933013304641708136322776975930707908534649896601672581489931296861792264885201253077552409752937687190004273840183558736439354068248714003327154239632489576169409254795398522132086265321465974567940864323720886659719888577903861611582549813595585004073014272687793883472040479746823897882884174675323066921366379359478947705788711158695065025722929508123849018760474871378452781744326869742268714260927679037530328980506139189810309753288229927739117101594860728735403589276978615641832423619,
        2904802997684979031429751266652287185343487588181447618330743076143601865498555112868668022266559203625663078877490258721995264797270023560831442836093516200516055819853220249422024925494525813600122382903520906197364840270012052413988292184690761146180604389522384946371612875869038489784405654789562755666546621759776892408153190790080930100123746284224075121257652224788593802068214369290495086275786967073127915183202957500434821866026609283416272645553951861415817069299793203345162979862593723584529770402506155104819505875374380008547680367117472878708136497428006654308479264979152338818509590797044264172530642931949135881728647441773319439777155807723223165099627191170008146028545375587766944080959493647795765768349350646133842732758718957895411577422317390130051445859016247698037520949742756905563488653739484537428521855358075060657961012278379620619506576459855478234203189721457470807178553957231283664846661,
    ],
    3072: [
        2904802997684979031429751266652287185343487588181447618330743076143601865498555112868668022266559203625663078877490258721995264797270023560831442836093516200516055819853220249422024925494525813600122382903520906197364840270012052413988292184690761146180604389522384946371612875869038489784405654789562755666546621759776892408153190790080930100123746284224075121257652224788593802068214369290495086275786967073127915183202957500434821866026609283416272645553951861415817069299793203345162979862593723584529770402506155104819505875374380008547680367117472878708136497428006654308479264979152338818509590797044264172530642931949135881728647441773319439777155807723223165099627191170008146028545375587766944080959493647795765768349350646133842732758718957895411577422317390130051445859016247698037520949742756905563488653739484537428521855358075060657961012278379620619506576459855478234203189721457470807178553957231283664849139,
        5809605995369958062859502533304574370686975176362895236661486152287203730997110225737336044533118407251326157754980517443990529594540047121662885672187032401032111639706440498844049850989051627200244765807041812394729680540024104827976584369381522292361208779044769892743225751738076979568811309579125511333093243519553784816306381580161860200247492568448150242515304449577187604136428738580990172551573934146255830366405915000869643732053218566832545291107903722831634138599586406690325959725187447169059540805012310209639011750748760017095360734234945757416272994856013308616958529958304677637019181594088528345061285863898271763457294883546638879554311615446446330199254382340016292057090751175533888161918987295591531536698701292267685465517437915790823154844634780260102891718032495396075041899485513811126977307478969074857043710716150121315922024556759241239013152919710956468406379442914941614357107914462567329693649,
    ],
    3073: [
        5809605995369958062859502533304574370686975176362895236661486152287203730997110225737336044533118407251326157754980517443990529594540047121662885672187032401032111639706440498844049850989051627200244765807041812394729680540024104827976584369381522292361208779044769892743225751738076979568811309579125511333093243519553784816306381580161860200247492568448150242515304449577187604136428738580990172551573934146255830366405915000869643732053218566832545291107903722831634138599586406690325959725187447169059540805012310209639011750748760017095360734234945757416272994856013308616958529958304677637019181594088528345061285863898271763457294883546638879554311615446446330199254382340016292057090751175533888161918987295591531536698701292267685465517437915790823154844634780260102891718032495396075041899485513811126977307478969074857043710716150121315922024556759241239013152919710956468406379442914941614357107914462567329694509,
        11619211990739916125719005066609148741373950352725790473322972304574407461994220451474672089066236814502652315509961034887981059189080094243325771344374064802064223279412880997688099701978103254400489531614083624789459361080048209655953168738763044584722417558089539785486451503476153959137622619158251022666186487039107569632612763160323720400494985136896300485030608899154375208272857477161980345103147868292511660732811830001739287464106437133665090582215807445663268277199172813380651919450374894338119081610024620419278023501497520034190721468469891514832545989712026617233917059916609355274038363188177056690122571727796543526914589767093277759108623230892892660398508764680032584114181502351067776323837974591183063073397402584535370931034875831581646309689269560520205783436064990792150083798971027622253954614957938149714087421432300242631844049113518482478026305839421912936812758885829883228714215828925134659386551,
    ],
    3583: [
        19473520333640844691127176862977149658590968948136526555247820243011272528285939061837707013418630693601041476568594986448123729387077643515811960411020482474294659386532350316053903539988147515390538730103863099146708934871245522941906322189870389962117919361028202071764176745980651257550452869590472617231874558907485685927367574648809115885947602724023790934518369995350619469561613808778207297256942426414559305648709312602649150614261853439607537507462932895730385881610561381979306213956956954155371387764484350419237234585750558467856203620817475676959043251863760768656792160668177150351045377597021414213264417090418226523925105259413550636485086184257591644591589393199898495538890080330871093821347753938084388897294966205185738762080099421245687443300643061380644532328158133536192948933806592507527238139734445474775820534266432764308034618758576637009643867006811192758495822265172889202823244282947775986251889156602254569864288788917095347237435709225746335486741183724272646028951741973332877875405120448636089315222389626215827597972845678291201404256024267047,
        38947040667281689382254353725954299317181937896273053110495640486022545056571878123675414026837261387202082953137189972896247458774155287031623920822040964948589318773064700632107807079976295030781077460207726198293417869742491045883812644379740779924235838722056404143528353491961302515100905739180945234463749117814971371854735149297618231771895205448047581869036739990701238939123227617556414594513884852829118611297418625205298301228523706879215075014925865791460771763221122763958612427913913908310742775528968700838474469171501116935712407241634951353918086503727521537313584321336354300702090755194042828426528834180836453047850210518827101272970172368515183289183178786399796991077780160661742187642695507876168777794589932410371477524160198842491374886601286122761289064656316267072385897867613185015054476279468890949551641068532865528616069237517153274019287734013622385516991644530345778405646488565895551972503778313204509139728577577834190694474871418451492670973482367448545292057903483946665755750810240897272178630444779252431655195945691356582402808512048526311,
    ],
    3584: [
        38947040667281689382254353725954299317181937896273053110495640486022545056571878123675414026837261387202082953137189972896247458774155287031623920822040964948589318773064700632107807079976295030781077460207726198293417869742491045883812644379740779924235838722056404143528353491961302515100905739180945234463749117814971371854735149297618231771895205448047581869036739990701238939123227617556414594513884852829118611297418625205298301228523706879215075014925865791460771763221122763958612427913913908310742775528968700838474469171501116935712407241634951353918086503727521537313584321336354300702090755194042828426528834180836453047850210518827101272970172368515183289183178786399796991077780160661742187642695507876168777794589932410371477524160198842491374886601286122761289064656316267072385897867613185015054476279468890949551641068532865528616069237517153274019287734013622385516991644530345778405646488565895551972503778313204509139728577577834190694474871418451492670973482367448545292057903483946665755750810240897272178630444779252431655195945691356582402808512048533667,
        77894081334563378764508707451908598634363875792546106220991280972045090113143756247350828053674522774404165906274379945792494917548310574063247841644081929897178637546129401264215614159952590061562154920415452396586835739484982091767625288759481559848471677444112808287056706983922605030201811478361890468927498235629942743709470298595236463543790410896095163738073479981402477878246455235112829189027769705658237222594837250410596602457047413758430150029851731582921543526442245527917224855827827816621485551057937401676948938343002233871424814483269902707836173007455043074627168642672708601404181510388085656853057668361672906095700421037654202545940344737030366578366357572799593982155560321323484375285391015752337555589179864820742955048320397684982749773202572245522578129312632534144771795735226370030108952558937781899103282137065731057232138475034306548038575468027244771033983289060691556811292977131791103945007556626409018279457155155668381388949742836902985341946964734897090584115806967893331511501620481794544357260889558504863310391891382713164805617024097058387,
    ],
    3585: [
        77894081334563378764508707451908598634363875792546106220991280972045090113143756247350828053674522774404165906274379945792494917548310574063247841644081929897178637546129401264215614159952590061562154920415452396586835739484982091767625288759481559848471677444112808287056706983922605030201811478361890468927498235629942743709470298595236463543790410896095163738073479981402477878246455235112829189027769705658237222594837250410596602457047413758430150029851731582921543526442245527917224855827827816621485551057937401676948938343002233871424814483269902707836173007455043074627168642672708601404181510388085656853057668361672906095700421037654202545940344737030366578366357572799593982155560321323484375285391015752337555589179864820742955048320397684982749773202572245522578129312632534144771795735226370030108952558937781899103282137065731057232138475034306548038575468027244771033983289060691556811292977131791103945007556626409018279457155155668381388949742836902985341946964734897090584115806967893331511501620481794544357260889558504863310391891382713164805617024097058837,
        155788162669126757529017414903817197268727751585092212441982561944090180226287512494701656107349045548808331812548759891584989835096621148126495683288163859794357275092258802528431228319905180123124309840830904793173671478969964183535250577518963119696943354888225616574113413967845210060403622956723780937854996471259885487418940597190472927087580821792190327476146959962804955756492910470225658378055539411316474445189674500821193204914094827516860300059703463165843087052884491055834449711655655633242971102115874803353897876686004467742849628966539805415672346014910086149254337285345417202808363020776171313706115336723345812191400842075308405091880689474060733156732715145599187964311120642646968750570782031504675111178359729641485910096640795369965499546405144491045156258625265068289543591470452740060217905117875563798206564274131462114464276950068613096077150936054489542067966578121383113622585954263582207890015113252818036558914310311336762777899485673805970683893929469794181168231613935786663023003240963589088714521779117009726620783782765426329611234048194117221,
    ],
    4095: [
        261097220353288126672938177679156095644991062261845945096058370820988476992889364212206702983749389585222526678609815709496893359546448401815809021962841319486489244135927499585090397533595929578607017502963986556594079709849428186418083671086146654374201977176450926017821012185029652278616994445899507251671734744220446946486726407547565235149894863358205867325756674110764756253993099966928553885423458889971322871579559478608624183521952968159874118775047260337252104265418773417083462637758243022067387692495904092352983253803449206459297022958414187805329623211592031387556499575103086196215648918623048654255951626478311402706432958845021902155525708567549424550578292254419501668798871269980409104842571343781196003726789783864995697628349902887948567776707783522646068221069947888712445738580883629266305817265348726496923250530740848921945719737110154001853236418729955762642910594288704080345157761475729034231677085714110182611974992975445366440868305962566813264974948999022699867300443656204429612466863914812544582267618279858291387701892055461642936593324221228204880079364250610231654227718537096269602982451130745464334744412025781521475750325603366797431668304122877782900730195434508359022560951177085100788548073,
        522194440706576253345876355358312191289982124523691890192116741641976953985778728424413405967498779170445053357219631418993786719092896803631618043925682638972978488271854999170180795067191859157214035005927973113188159419698856372836167342172293308748403954352901852035642024370059304557233988891799014503343469488440893892973452815095130470299789726716411734651513348221529512507986199933857107770846917779942645743159118957217248367043905936319748237550094520674504208530837546834166925275516486044134775384991808184705966507606898412918594045916828375610659246423184062775112999150206172392431297837246097308511903252956622805412865917690043804311051417135098849101156584508839003337597742539960818209685142687562392007453579567729991395256699805775897135553415567045292136442139895777424891477161767258532611634530697452993846501061481697843891439474220308003706472837459911525285821188577408160690315522951458068463354171428220365223949985950890732881736611925133626529949897998045399734600887312408859224933727829625089164535236559716582775403784110923285873186648442456409760158728501220463308455437074192539205964902261490928669488824051563042951500651206733594863336608245755565801460390869016718045121902354170201577093553,
    ],
    4096: [
        522194440706576253345876355358312191289982124523691890192116741641976953985778728424413405967498779170445053357219631418993786719092896803631618043925682638972978488271854999170180795067191859157214035005927973113188159419698856372836167342172293308748403954352901852035642024370059304557233988891799014503343469488440893892973452815095130470299789726716411734651513348221529512507986199933857107770846917779942645743159118957217248367043905936319748237550094520674504208530837546834166925275516486044134775384991808184705966507606898412918594045916828375610659246423184062775112999150206172392431297837246097308511903252956622805412865917690043804311051417135098849101156584508839003337597742539960818209685142687562392007453579567729991395256699805775897135553415567045292136442139895777424891477161767258532611634530697452993846501061481697843891439474220308003706472837459911525285821188577408160690315522951458068463354171428220365223949985950890732881736611925133626529949897998045399734600887312408859224933727829625089164535236559716582775403784110923285873186648442456409760158728501220463308455437074192539205964902261490928669488824051563042951500651206733594863336608245755565801460390869016718045121902354170201577095747,
        1044388881413152506691752710716624382579964249047383780384233483283953907971557456848826811934997558340890106714439262837987573438185793607263236087851365277945956976543709998340361590134383718314428070011855946226376318839397712745672334684344586617496807908705803704071284048740118609114467977783598029006686938976881787785946905630190260940599579453432823469303026696443059025015972399867714215541693835559885291486318237914434496734087811872639496475100189041349008417061675093668333850551032972088269550769983616369411933015213796825837188091833656751221318492846368125550225998300412344784862595674492194617023806505913245610825731835380087608622102834270197698202313169017678006675195485079921636419370285375124784014907159135459982790513399611551794271106831134090584272884279791554849782954323534517065223269061394905987693002122963395687782878948440616007412945674919823050571642377154816321380631045902916136926708342856440730447899971901781465763473223850267253059899795996090799469201774624817718449867455659250178329070473119433165550807568221846571746373296884912819520317457002440926616910874148385078411929804522981857338977648103126085903001302413467189726673216491511131602920781738033436090243804708340403154187787,
    ],
    4097: [
        1044388881413152506691752710716624382579964249047383780384233483283953907971557456848826811934997558340890106714439262837987573438185793607263236087851365277945956976543709998340361590134383718314428070011855946226376318839397712745672334684344586617496807908705803704071284048740118609114467977783598029006686938976881787785946905630190260940599579453432823469303026696443059025015972399867714215541693835559885291486318237914434496734087811872639496475100189041349008417061675093668333850551032972088269550769983616369411933015213796825837188091833656751221318492846368125550225998300412344784862595674492194617023806505913245610825731835380087608622102834270197698202313169017678006675195485079921636419370285375124784014907159135459982790513399611551794271106831134090584272884279791554849782954323534517065223269061394905987693002122963395687782878948440616007412945674919823050571642377154816321380631045902916136926708342856440730447899971901781465763473223850267253059899795996090799469201774624817718449867455659250178329070473119433165550807568221846571746373296884912819520317457002440926616910874148385078411929804522981857338977648103126085903001302413467189726673216491511131602920781738033436090243804708340403154192097,
        2088777762826305013383505421433248765159928498094767560768466966567907815943114913697653623869995116681780213428878525675975146876371587214526472175702730555891913953087419996680723180268767436628856140023711892452752637678795425491344669368689173234993615817411607408142568097480237218228935955567196058013373877953763575571893811260380521881199158906865646938606053392886118050031944799735428431083387671119770582972636475828868993468175623745278992950200378082698016834123350187336667701102065944176539101539967232738823866030427593651674376183667313502442636985692736251100451996600824689569725191348984389234047613011826491221651463670760175217244205668540395396404626338035356013350390970159843272838740570750249568029814318270919965581026799223103588542213662268181168545768559583109699565908647069034130446538122789811975386004245926791375565757896881232014825891349839646101143284754309632642761262091805832273853416685712881460895799943803562931526946447700534506119799591992181598938403549249635436899734911318500356658140946238866331101615136443693143492746593769825639040634914004881853233821748296770156823859609045963714677955296206252171806002604826934379453346432983022263205841563476066872180487609416680806308379889,
    ],
}


NUM = bits[512]


NONE = hashlib.sha3_512(f"__NONE__{NUM}".encode()).hexdigest()


class PrimeGroups:
    """
    A collection of primes moduli & each of their respective primitive
    roots organized by bit length.
    """

    GENERATOR_32: int = 2
    GENERATOR_40: int = 3
    GENERATOR_48: int = 3
    GENERATOR_56: int = 2
    GENERATOR_64: int = 3
    GENERATOR_128: int = 11
    GENERATOR_192: int = 2
    GENERATOR_256: int = 2
    GENERATOR_384: int = 3
    GENERATOR_512: int = 7
    DH_GENERATOR_4096_GROUP_16: int = 2
    MOD_32: int = 0xfffffffb
    MOD_40: int = 0xfffffffc29
    MOD_48: int = 0xfffffffffab5
    MOD_56: int = 0xfffffffffff833
    MOD_64: int = 0xfffffffffffff9b9
    MOD_128: int = 0xffffffffffffffffffffffffffffe807
    MOD_192: int = 0xfffffffffffffffffffffffffffffffffffffffffffffa85
    MOD_256: int = 0xffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff8dcb
    MOD_384: int = 0xffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff5e31
    MOD_512: int = 0xffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffffff9203
    DH_PRIME_4096_GROUP_16: int = 0xffffffffffffffffc90fdaa22168c234c4c6628b80dc1cd129024e088a67cc74020bbea63b139b22514a08798e3404ddef9519b3cd3a431b302b0a6df25f14374fe1356d6d51c245e485b576625e7ec6f44c42e9a637ed6b0bff5cb6f406b7edee386bfb5a899fa5ae9f24117c4b1fe649286651ece45b3dc2007cb8a163bf0598da48361c55d39a69163fa8fd24cf5f83655d23dca3ad961c62f356208552bb9ed529077096966d670c354e4abc9804f1746c08ca18217c32905e462e36ce3be39e772c180e86039b2783a2ec07a28fb5c55df06f4c52c9de2bcbf6955817183995497cea956ae515d2261898fa051015728e5a8aaac42dad33170d04507a33a85521abdf1cba64ecfb850458dbef0a8aea71575d060c7db3970f85a6e1e4c7abf5ae8cdb0933d71e8c94e04a25619dcee3d2261ad2ee6bf12ffa06d98a0864d87602733ec86a64521f2b18177b200cbbe117577a615d6c770988c0bad946e208e24fa074e5ab3143db5bfce0fd108e4b82d120a92108011a723c12a787e6d788719a10bdba5b2699c327186af4e23c1a946834b6150bda2583e9ca2ad44ce8dbbbc2db04de8ef92e8efc141fbecaa6287c59474e6bc05d99b2964fa090c3a2233ba186515be7ed1f612970cee2d7afb81bdd762170481cd0069127d5b05aa993b4ea988d8fddc186ffb7dc90a6c08f4df435c934063199ffffffffffffffff


WORD_LIST = [
    "sausage",
    "power",
    "twelve",
    "soft",
    "athlete",
    "will",
    "chunk",
    "pear",
    "broccoli",
    "butter",
    "exchange",
    "pass",
    "rubber",
    "wealth",
    "suspect",
    "sustain",
    "dream",
    "circle",
    "gadget",
    "try",
    "unknown",
    "wrong",
    "next",
    "exact",
    "rhythm",
    "rally",
    "own",
    "island",
    "craft",
    "ketchup",
    "smart",
    "brief",
    "creek",
    "illegal",
    "busy",
    "patrol",
    "profit",
    "arrest",
    "carpet",
    "adjust",
    "black",
    "avocado",
    "cloud",
    "effort",
    "jelly",
    "subway",
    "rocket",
    "local",
    "oval",
    "twenty",
    "bridge",
    "hint",
    "super",
    "liquid",
    "casual",
    "velvet",
    "quarter",
    "blush",
    "leaf",
    "frame",
    "element",
    "song",
    "upgrade",
    "cry",
    "pilot",
    "proud",
    "video",
    "approve",
    "odor",
    "bundle",
    "food",
    "news",
    "mansion",
    "load",
    "near",
    "argue",
    "drum",
    "curtain",
    "solution",
    "remove",
    "person",
    "universe",
    "before",
    "raven",
    "capable",
    "humor",
    "grant",
    "live",
    "amount",
    "solve",
    "garment",
    "siege",
    "fitness",
    "post",
    "rotate",
    "flight",
    "mosquito",
    "crawl",
    "hedgehog",
    "cause",
    "decline",
    "cattle",
    "guard",
    "flip",
    "magnet",
    "glove",
    "vessel",
    "diary",
    "require",
    "hollow",
    "saddle",
    "history",
    "rival",
    "fox",
    "around",
    "yellow",
    "hill",
    "assume",
    "celery",
    "reflect",
    "force",
    "merry",
    "crush",
    "hospital",
    "echo",
    "behave",
    "reunion",
    "uphold",
    "pudding",
    "truly",
    "vintage",
    "april",
    "gossip",
    "slice",
    "gift",
    "category",
    "rail",
    "hobby",
    "dinosaur",
    "scene",
    "ribbon",
    "mass",
    "aware",
    "cruel",
    "account",
    "gap",
    "coil",
    "vendor",
    "grunt",
    "farm",
    "room",
    "confirm",
    "crisp",
    "cherry",
    "film",
    "great",
    "milk",
    "name",
    "scheme",
    "sport",
    "short",
    "special",
    "ready",
    "police",
    "ring",
    "appear",
    "produce",
    "prepare",
    "chronic",
    "funny",
    "curve",
    "what",
    "section",
    "glue",
    "rebuild",
    "earth",
    "opera",
    "chimney",
    "improve",
    "winter",
    "bottom",
    "fire",
    "recall",
    "panda",
    "shove",
    "hip",
    "manage",
    "debris",
    "under",
    "blade",
    "bird",
    "meadow",
    "hawk",
    "broom",
    "book",
    "soul",
    "aunt",
    "paddle",
    "catalog",
    "inspire",
    "pair",
    "twice",
    "exhaust",
    "much",
    "inmate",
    "dignity",
    "powder",
    "angry",
    "draw",
    "clap",
    "pool",
    "report",
    "push",
    "choose",
    "mad",
    "hybrid",
    "purchase",
    "icon",
    "certain",
    "balcony",
    "doll",
    "oxygen",
    "style",
    "digital",
    "toddler",
    "tilt",
    "cousin",
    "fault",
    "welcome",
    "imitate",
    "general",
    "wheel",
    "fortune",
    "daring",
    "horse",
    "half",
    "crucial",
    "crouch",
    "roast",
    "nasty",
    "hello",
    "poverty",
    "female",
    "happy",
    "despair",
    "breeze",
    "isolate",
    "december",
    "drive",
    "habit",
    "honey",
    "assault",
    "shock",
    "riot",
    "mail",
    "scout",
    "urge",
    "prefer",
    "monster",
    "donate",
    "shadow",
    "engine",
    "raise",
    "harsh",
    "struggle",
    "portion",
    "satoshi",
    "supreme",
    "miracle",
    "genre",
    "metal",
    "foot",
    "join",
    "decade",
    "place",
    "hazard",
    "loyal",
    "raccoon",
    "lend",
    "diamond",
    "trade",
    "acquire",
    "normal",
    "virus",
    "average",
    "broken",
    "final",
    "mom",
    "hunt",
    "rebel",
    "wear",
    "wrap",
    "leisure",
    "choice",
    "double",
    "stock",
    "tribe",
    "label",
    "avoid",
    "van",
    "tell",
    "announce",
    "waste",
    "setup",
    "end",
    "health",
    "cancel",
    "about",
    "lobster",
    "blur",
    "become",
    "cart",
    "genius",
    "salute",
    "loan",
    "dose",
    "senior",
    "brother",
    "wheat",
    "inside",
    "eye",
    "hover",
    "trophy",
    "often",
    "clinic",
    "enhance",
    "vague",
    "afford",
    "toilet",
    "source",
    "attend",
    "skate",
    "wet",
    "issue",
    "piece",
    "major",
    "since",
    "peanut",
    "right",
    "sibling",
    "nest",
    "robot",
    "ordinary",
    "embark",
    "state",
    "decorate",
    "patient",
    "ripple",
    "truck",
    "penalty",
    "energy",
    "nature",
    "sister",
    "list",
    "photo",
    "such",
    "federal",
    "tide",
    "settle",
    "pretty",
    "top",
    "snake",
    "strike",
    "warfare",
    "expire",
    "transfer",
    "sand",
    "baby",
    "aerobic",
    "cable",
    "obtain",
    "husband",
    "side",
    "exist",
    "coach",
    "cereal",
    "grief",
    "smoke",
    "success",
    "moment",
    "cake",
    "blame",
    "rent",
    "utility",
    "matrix",
    "basket",
    "visit",
    "salad",
    "bounce",
    "fruit",
    "dynamic",
    "device",
    "flag",
    "save",
    "review",
    "provide",
    "ladder",
    "aisle",
    "salt",
    "little",
    "seek",
    "sunset",
    "depend",
    "occur",
    "search",
    "soup",
    "add",
    "figure",
    "cube",
    "team",
    "phrase",
    "gorilla",
    "stool",
    "faint",
    "dad",
    "size",
    "midnight",
    "open",
    "stamp",
    "puzzle",
    "dirt",
    "lawsuit",
    "miss",
    "tragic",
    "one",
    "strong",
    "sadness",
    "man",
    "eight",
    "wink",
    "employ",
    "urban",
    "emerge",
    "machine",
    "scatter",
    "brisk",
    "verify",
    "tuition",
    "fade",
    "true",
    "biology",
    "gauge",
    "key",
    "hub",
    "poet",
    "myself",
    "silly",
    "frown",
    "also",
    "smooth",
    "brush",
    "mimic",
    "share",
    "whale",
    "fantasy",
    "tooth",
    "script",
    "spatial",
    "world",
    "peasant",
    "census",
    "impose",
    "remember",
    "second",
    "arena",
    "index",
    "width",
    "main",
    "lumber",
    "jacket",
    "afraid",
    "fence",
    "attack",
    "destroy",
    "suggest",
    "clean",
    "depart",
    "never",
    "six",
    "game",
    "grit",
    "screen",
    "water",
    "inform",
    "swift",
    "weather",
    "evil",
    "injury",
    "grocery",
    "stadium",
    "essence",
    "ball",
    "invite",
    "mango",
    "autumn",
    "castle",
    "example",
    "forum",
    "lava",
    "safe",
    "sick",
    "toast",
    "cave",
    "hockey",
    "volume",
    "unaware",
    "unusual",
    "spoon",
    "first",
    "physical",
    "foster",
    "have",
    "merge",
    "pill",
    "sleep",
    "absurd",
    "cage",
    "lunar",
    "exile",
    "minimum",
    "calm",
    "deny",
    "manual",
    "where",
    "asthma",
    "topic",
    "wing",
    "kite",
    "update",
    "mask",
    "rude",
    "abuse",
    "shallow",
    "kind",
    "chest",
    "tip",
    "result",
    "shoot",
    "quick",
    "shield",
    "maple",
    "inquiry",
    "scissors",
    "street",
    "high",
    "siren",
    "lazy",
    "cliff",
    "unfold",
    "wool",
    "crazy",
    "prison",
    "build",
    "must",
    "good",
    "oblige",
    "upper",
    "nephew",
    "alcohol",
    "reform",
    "floor",
    "grab",
    "apology",
    "arrive",
    "custom",
    "month",
    "shoulder",
    "shell",
    "office",
    "doctor",
    "speak",
    "valid",
    "battle",
    "mention",
    "witness",
    "border",
    "capital",
    "collect",
    "frozen",
    "party",
    "venue",
    "parrot",
    "potato",
    "net",
    "visual",
    "traffic",
    "invest",
    "pledge",
    "begin",
    "rug",
    "all",
    "trip",
    "other",
    "west",
    "teach",
    "umbrella",
    "swamp",
    "educate",
    "canoe",
    "gate",
    "best",
    "divert",
    "tenant",
    "toe",
    "fetch",
    "claim",
    "crowd",
    "select",
    "bench",
    "demise",
    "absent",
    "goat",
    "that",
    "either",
    "trial",
    "turkey",
    "erase",
    "bulk",
    "burger",
    "wise",
    "offer",
    "together",
    "practice",
    "captain",
    "dolphin",
    "treat",
    "eternal",
    "retreat",
    "margin",
    "area",
    "elite",
    "merit",
    "negative",
    "stick",
    "axis",
    "ivory",
    "include",
    "hole",
    "simple",
    "away",
    "nerve",
    "assist",
    "sorry",
    "lab",
    "tattoo",
    "pole",
    "blanket",
    "pyramid",
    "easily",
    "glass",
    "above",
    "erosion",
    "anchor",
    "vault",
    "gentle",
    "actress",
    "property",
    "receive",
    "spring",
    "shift",
    "walnut",
    "junior",
    "spice",
    "subject",
    "industry",
    "drift",
    "bind",
    "deputy",
    "cheap",
    "allow",
    "mushroom",
    "indoor",
    "furnace",
    "empower",
    "clever",
    "annual",
    "fabric",
    "unveil",
    "school",
    "library",
    "enrich",
    "tiger",
    "elephant",
    "fine",
    "viable",
    "couple",
    "client",
    "pride",
    "divorce",
    "ugly",
    "lesson",
    "excess",
    "achieve",
    "noble",
    "present",
    "wild",
    "joy",
    "trumpet",
    "knee",
    "page",
    "reward",
    "demand",
    "off",
    "license",
    "gather",
    "knock",
    "conduct",
    "equal",
    "make",
    "slogan",
    "convince",
    "please",
    "solid",
    "dragon",
    "ecology",
    "kick",
    "art",
    "boat",
    "enemy",
    "climb",
    "romance",
    "trap",
    "want",
    "seven",
    "cross",
    "exotic",
    "original",
    "bronze",
    "segment",
    "law",
    "slam",
    "suffer",
    "match",
    "sail",
    "mind",
    "recycle",
    "infant",
    "file",
    "theory",
    "club",
    "oil",
    "toy",
    "question",
    "sort",
    "soccer",
    "stone",
    "time",
    "lucky",
    "economy",
    "flock",
    "inner",
    "neglect",
    "uniform",
    "flower",
    "glare",
    "rose",
    "fever",
    "into",
    "wall",
    "violin",
    "recipe",
    "angle",
    "asset",
    "control",
    "zone",
    "void",
    "morning",
    "vacuum",
    "expect",
    "garbage",
    "museum",
    "wish",
    "again",
    "affair",
    "vote",
    "door",
    "express",
    "blue",
    "south",
    "olympic",
    "tornado",
    "memory",
    "sell",
    "return",
    "wasp",
    "sudden",
    "inherit",
    "visa",
    "airport",
    "tray",
    "vicious",
    "resource",
    "virtual",
    "pistol",
    "syrup",
    "faculty",
    "fall",
    "repair",
    "raw",
    "armed",
    "damage",
    "region",
    "object",
    "predict",
    "rescue",
    "festival",
    "wide",
    "picture",
    "early",
    "taxi",
    "today",
    "spray",
    "rate",
    "maximum",
    "day",
    "pluck",
    "theme",
    "trash",
    "banana",
    "river",
    "fun",
    "company",
    "reject",
    "error",
    "usage",
    "track",
    "tent",
    "grape",
    "hammer",
    "dust",
    "brown",
    "security",
    "concert",
    "alone",
    "scan",
    "damp",
    "flee",
    "myth",
    "version",
    "gallery",
    "parade",
    "fame",
    "awkward",
    "quiz",
    "tennis",
    "slush",
    "coral",
    "ask",
    "tongue",
    "measure",
    "cricket",
    "magic",
    "answer",
    "ghost",
    "denial",
    "market",
    "group",
    "any",
    "poem",
    "february",
    "panther",
    "tomorrow",
    "episode",
    "estate",
    "valley",
    "obscure",
    "excite",
    "surge",
    "multiply",
    "zoo",
    "you",
    "bitter",
    "later",
    "swing",
    "blouse",
    "color",
    "blood",
    "enact",
    "call",
    "output",
    "letter",
    "sponsor",
    "when",
    "target",
    "decrease",
    "fancy",
    "father",
    "nothing",
    "now",
    "vacant",
    "walk",
    "october",
    "beef",
    "bright",
    "this",
    "duck",
    "rely",
    "balance",
    "mesh",
    "skull",
    "liar",
    "retire",
    "vital",
    "orient",
    "tiny",
    "inflict",
    "lion",
    "onion",
    "stuff",
    "enforce",
    "need",
    "olive",
    "opinion",
    "point",
    "favorite",
    "risk",
    "churn",
    "spend",
    "mystery",
    "lottery",
    "weasel",
    "dumb",
    "dice",
    "expose",
    "core",
    "nice",
    "snack",
    "prize",
    "wolf",
    "pumpkin",
    "horn",
    "shiver",
    "hold",
    "prosper",
    "skirt",
    "exit",
    "ostrich",
    "parent",
    "refuse",
    "know",
    "tortoise",
    "adult",
    "case",
    "dress",
    "lady",
    "sauce",
    "antique",
    "nominee",
    "outdoor",
    "reason",
    "over",
    "blast",
    "dance",
    "village",
    "across",
    "defy",
    "ocean",
    "dish",
    "phone",
    "girl",
    "obvious",
    "muscle",
    "cannon",
    "diesel",
    "better",
    "primary",
    "small",
    "payment",
    "swap",
    "skin",
    "swallow",
    "cup",
    "left",
    "pen",
    "awful",
    "useless",
    "pony",
    "vocal",
    "race",
    "start",
    "enough",
    "corn",
    "access",
    "network",
    "grow",
    "bring",
    "drop",
    "cool",
    "modify",
    "item",
    "mountain",
    "step",
    "bacon",
    "cream",
    "rapid",
    "correct",
    "change",
    "glimpse",
    "move",
    "abstract",
    "zebra",
    "elder",
    "differ",
    "develop",
    "truth",
    "switch",
    "dog",
    "purse",
    "pull",
    "east",
    "maze",
    "feel",
    "dentist",
    "beach",
    "aim",
    "faith",
    "hamster",
    "wire",
    "donkey",
    "cement",
    "silent",
    "giant",
    "space",
    "edit",
    "love",
    "tape",
    "act",
    "gain",
    "pioneer",
    "chalk",
    "range",
    "tissue",
    "runway",
    "giraffe",
    "eagle",
    "quality",
    "term",
    "arrange",
    "plug",
    "weekend",
    "labor",
    "kid",
    "dutch",
    "duty",
    "camera",
    "chase",
    "mixture",
    "lonely",
    "income",
    "repeat",
    "rough",
    "naive",
    "reduce",
    "summer",
    "fat",
    "citizen",
    "voice",
    "dizzy",
    "coin",
    "wedding",
    "woman",
    "resemble",
    "canvas",
    "kitchen",
    "sample",
    "brand",
    "then",
    "dash",
    "drip",
    "program",
    "salmon",
    "equip",
    "radio",
    "hotel",
    "embody",
    "stomach",
    "bid",
    "chuckle",
    "friend",
    "learn",
    "punch",
    "hire",
    "language",
    "jar",
    "quit",
    "wave",
    "model",
    "vivid",
    "increase",
    "box",
    "ethics",
    "almost",
    "hand",
    "story",
    "pink",
    "gospel",
    "envelope",
    "rigid",
    "stable",
    "orange",
    "egg",
    "lecture",
    "whip",
    "pipe",
    "order",
    "chief",
    "replace",
    "auction",
    "inject",
    "scare",
    "accuse",
    "lyrics",
    "chair",
    "arctic",
    "earn",
    "regret",
    "fringe",
    "crunch",
    "jazz",
    "glance",
    "spider",
    "rain",
    "burden",
    "tone",
    "medal",
    "paper",
    "nation",
    "lizard",
    "govern",
    "joke",
    "there",
    "more",
    "explain",
    "thumb",
    "flat",
    "direct",
    "limb",
    "thought",
    "identify",
    "master",
    "eyebrow",
    "use",
    "light",
    "stand",
    "barrel",
    "million",
    "bike",
    "hero",
    "fury",
    "bullet",
    "banner",
    "stage",
    "cycle",
    "pitch",
    "grain",
    "vapor",
    "volcano",
    "response",
    "fiber",
    "height",
    "just",
    "ice",
    "rack",
    "heart",
    "charge",
    "involve",
    "artwork",
    "fork",
    "submit",
    "cook",
    "clip",
    "already",
    "cost",
    "hair",
    "catch",
    "purpose",
    "junk",
    "fish",
    "whisper",
    "float",
    "face",
    "bus",
    "sentence",
    "toss",
    "country",
    "monitor",
    "deer",
    "empty",
    "planet",
    "tower",
    "drink",
    "cluster",
    "donor",
    "gesture",
    "fiction",
    "globe",
    "initial",
    "notice",
    "host",
    "polar",
    "prevent",
    "shrug",
    "kit",
    "course",
    "note",
    "critic",
    "melt",
    "illness",
    "bone",
    "curious",
    "spy",
    "length",
    "ski",
    "unlock",
    "defense",
    "cheese",
    "dry",
    "display",
    "rather",
    "clay",
    "radar",
    "else",
    "mixed",
    "proof",
    "night",
    "caught",
    "situate",
    "wage",
    "material",
    "knife",
    "bachelor",
    "menu",
    "release",
    "sword",
    "pause",
    "scale",
    "enroll",
    "dismiss",
    "resist",
    "elegant",
    "garlic",
    "useful",
    "supply",
    "alert",
    "rice",
    "kiwi",
    "razor",
    "cushion",
    "salon",
    "mutual",
    "flavor",
    "flash",
    "found",
    "harvest",
    "lawn",
    "mercy",
    "series",
    "accident",
    "extra",
    "trim",
    "giggle",
    "age",
    "rare",
    "card",
    "uncover",
    "combine",
    "agent",
    "lock",
    "they",
    "author",
    "legend",
    "method",
    "latin",
    "sing",
    "leader",
    "sheriff",
    "disorder",
    "debate",
    "animal",
    "slide",
    "skill",
    "omit",
    "park",
    "drastic",
    "gas",
    "sphere",
    "dilemma",
    "taste",
    "window",
    "unhappy",
    "sight",
    "quote",
    "laundry",
    "acoustic",
    "blind",
    "pond",
    "tag",
    "home",
    "ancient",
    "mother",
    "derive",
    "vehicle",
    "hurry",
    "spirit",
    "journey",
    "soda",
    "attitude",
    "worry",
    "immense",
    "twist",
    "blossom",
    "piano",
    "way",
    "notable",
    "meat",
    "hen",
    "uncle",
    "pattern",
    "brass",
    "north",
    "forget",
    "service",
    "congress",
    "steak",
    "ceiling",
    "few",
    "crop",
    "sad",
    "enlist",
    "print",
    "cute",
    "bubble",
    "atom",
    "bar",
    "brain",
    "novel",
    "snap",
    "tuna",
    "travel",
    "legal",
    "gravity",
    "ride",
    "close",
    "crack",
    "total",
    "amused",
    "boss",
    "bonus",
    "warrior",
    "artefact",
    "bleak",
    "swear",
    "maid",
    "satisfy",
    "clog",
    "camp",
    "marble",
    "slot",
    "unable",
    "candy",
    "amazing",
    "oppose",
    "vibrant",
    "crew",
    "toward",
    "movie",
    "arm",
    "follow",
    "document",
    "harbor",
    "season",
    "session",
    "outside",
    "sun",
    "talent",
    "seminar",
    "two",
    "once",
    "three",
    "insect",
    "voyage",
    "because",
    "marine",
    "evolve",
    "enter",
    "crumble",
    "betray",
    "family",
    "lunch",
    "option",
    "hidden",
    "turn",
    "famous",
    "throw",
    "innocent",
    "spoil",
    "quantum",
    "fold",
    "wife",
    "admit",
    "address",
    "foil",
    "path",
    "upset",
    "degree",
    "kitten",
    "lake",
    "oven",
    "win",
    "wash",
    "kiss",
    "bracket",
    "tail",
    "similar",
    "veteran",
    "fresh",
    "feature",
    "sock",
    "insane",
    "work",
    "distance",
    "write",
    "ahead",
    "slow",
    "thrive",
    "ill",
    "image",
    "describe",
    "guilt",
    "iron",
    "stove",
    "year",
    "depth",
    "problem",
    "auto",
    "shrimp",
    "text",
    "rabbit",
    "lounge",
    "tube",
    "tackle",
    "slight",
    "foam",
    "another",
    "tourist",
    "can",
    "jewel",
    "soap",
    "basic",
    "finish",
    "pizza",
    "hurt",
    "glide",
    "thing",
    "green",
    "cash",
    "endorse",
    "monkey",
    "connect",
    "grass",
    "sound",
    "bargain",
    "sniff",
    "absorb",
    "anger",
    "ensure",
    "panel",
    "detect",
    "chicken",
    "noise",
    "coffee",
    "frog",
    "reopen",
    "secret",
    "render",
    "expand",
    "bean",
    "desert",
    "spare",
    "entire",
    "cabbage",
    "yard",
    "squirrel",
    "boy",
    "remind",
    "pelican",
    "actor",
    "evoke",
    "relief",
    "anxiety",
    "hood",
    "mirror",
    "excuse",
    "eager",
    "claw",
    "soon",
    "buddy",
    "popular",
    "very",
    "actual",
    "hat",
    "bamboo",
    "strategy",
    "shaft",
    "perfect",
    "city",
    "cargo",
    "hour",
    "flush",
    "cotton",
    "ankle",
    "plate",
    "enable",
    "clutch",
    "pepper",
    "cupboard",
    "human",
    "bunker",
    "ginger",
    "column",
    "help",
    "impulse",
    "cat",
    "priority",
    "lift",
    "shoe",
    "sting",
    "trend",
    "fit",
    "check",
    "dune",
    "humble",
    "sugar",
    "mouse",
    "fiscal",
    "diet",
    "jaguar",
    "tool",
    "dial",
    "garage",
    "project",
    "armor",
    "okay",
    "budget",
    "cigar",
    "fly",
    "dawn",
    "remain",
    "belt",
    "rural",
    "same",
    "gym",
    "wine",
    "flame",
    "spike",
    "deliver",
    "buyer",
    "antenna",
    "renew",
    "grid",
    "level",
    "devote",
    "agree",
    "buzz",
    "emotion",
    "oak",
    "vast",
    "peace",
    "warm",
    "mean",
    "brick",
    "wonder",
    "life",
    "swarm",
    "action",
    "plunge",
    "business",
    "clock",
    "split",
    "real",
    "tunnel",
    "logic",
    "rib",
    "seat",
    "embrace",
    "member",
    "picnic",
    "promote",
    "kangaroo",
    "cover",
    "ritual",
    "shuffle",
    "old",
    "glory",
    "system",
    "surface",
    "amateur",
    "frequent",
    "tree",
    "extend",
    "scorpion",
    "false",
    "plastic",
    "victory",
    "clarify",
    "palm",
    "spawn",
    "town",
    "genuine",
    "between",
    "coyote",
    "reveal",
    "crane",
    "guitar",
    "trigger",
    "holiday",
    "web",
    "fan",
    "talk",
    "danger",
    "long",
    "feed",
    "child",
    "trust",
    "thunder",
    "deposit",
    "fragile",
    "ramp",
    "word",
    "body",
    "consider",
    "click",
    "vanish",
    "various",
    "turtle",
    "divide",
    "swim",
    "stairs",
    "minor",
    "jump",
    "bag",
    "grace",
    "analyst",
    "leg",
    "discover",
    "used",
    "air",
    "weapon",
    "leopard",
    "view",
    "laptop",
    "shed",
    "birth",
    "design",
    "erupt",
    "symptom",
    "wood",
    "nose",
    "sunny",
    "venture",
    "gown",
    "daughter",
    "price",
    "pet",
    "unit",
    "electric",
    "abandon",
    "sea",
    "gloom",
    "keen",
    "sweet",
    "stumble",
    "puppy",
    "exclude",
    "regular",
    "wagon",
    "crash",
    "public",
    "narrow",
    "rookie",
    "inhale",
    "goose",
    "robust",
    "august",
    "laugh",
    "burst",
    "behind",
    "coast",
    "ignore",
    "upon",
    "caution",
    "fog",
    "like",
    "fatal",
    "always",
    "fee",
    "gold",
    "huge",
    "slim",
    "during",
    "wreck",
    "owner",
    "alpha",
    "worth",
    "rifle",
    "crime",
    "audit",
    "motor",
    "cradle",
    "pulse",
    "rich",
    "cloth",
    "hurdle",
    "nut",
    "install",
    "culture",
    "message",
    "play",
    "badge",
    "galaxy",
    "purity",
    "arch",
    "slab",
    "garden",
    "disease",
    "benefit",
    "hard",
    "clown",
    "dwarf",
    "matter",
    "front",
    "common",
    "gaze",
    "ship",
    "century",
    "luxury",
    "melody",
    "hope",
    "spread",
    "alley",
    "immune",
    "scrap",
    "wrist",
    "dove",
    "squeeze",
    "artist",
    "smile",
    "verb",
    "acid",
    "large",
    "exercise",
    "unfair",
    "shop",
    "thank",
    "edge",
    "army",
    "road",
    "pottery",
    "evidence",
    "orphan",
    "tumble",
    "loud",
    "run",
    "frost",
    "youth",
    "horror",
    "barely",
    "pact",
    "liberty",
    "boil",
    "misery",
    "clump",
    "layer",
    "neither",
    "people",
    "unique",
    "noodle",
    "oyster",
    "comic",
    "panic",
    "math",
    "say",
    "put",
    "beyond",
    "stereo",
    "random",
    "aspect",
    "hungry",
    "surround",
    "symbol",
    "guide",
    "tobacco",
    "survey",
    "stem",
    "drama",
    "judge",
    "domain",
    "sure",
    "spot",
    "napkin",
    "praise",
    "advance",
    "alter",
    "suit",
    "inch",
    "gun",
    "possible",
    "pulp",
    "casino",
    "couch",
    "cactus",
    "stay",
    "record",
    "elbow",
    "define",
    "elevator",
    "disagree",
    "ranch",
    "goddess",
    "adapt",
    "mix",
    "essay",
    "album",
    "nuclear",
    "beauty",
    "scrub",
    "lemon",
    "palace",
    "chapter",
    "orchard",
    "weird",
    "tired",
    "deal",
    "diagram",
    "march",
    "ten",
    "spell",
    "undo",
    "limit",
    "surprise",
    "shine",
    "entry",
    "chat",
    "heavy",
    "trick",
    "brave",
    "kidney",
    "escape",
    "boring",
    "outer",
    "indicate",
    "usual",
    "loop",
    "tank",
    "torch",
    "train",
    "interest",
    "lens",
    "car",
    "spin",
    "lamp",
    "dinner",
    "intact",
    "idle",
    "link",
    "detail",
    "guess",
    "roof",
    "sketch",
    "era",
    "code",
    "enjoy",
    "future",
    "comfort",
    "task",
    "easy",
    "trouble",
    "bread",
    "wisdom",
    "route",
    "believe",
    "young",
    "sense",
    "social",
    "seed",
    "juice",
    "crystal",
    "obey",
    "focus",
    "type",
    "bless",
    "pigeon",
    "drill",
    "carry",
    "zero",
    "canal",
    "find",
    "fatigue",
    "credit",
    "topple",
    "input",
    "ability",
    "ticket",
    "someone",
    "table",
    "fix",
    "moon",
    "below",
    "science",
    "jealous",
    "student",
    "twin",
    "idea",
    "board",
    "steel",
    "pencil",
    "minute",
    "motion",
    "observe",
    "execute",
    "among",
    "title",
    "bicycle",
    "token",
    "alien",
    "base",
    "jungle",
    "attract",
    "media",
    "middle",
    "position",
    "orbit",
    "delay",
    "muffin",
    "forest",
    "music",
    "bomb",
    "mammal",
    "mandate",
    "addict",
    "pig",
    "winner",
    "bulb",
    "field",
    "buffalo",
    "shy",
    "until",
    "carbon",
    "protect",
    "awake",
    "tomato",
    "coconut",
    "neck",
    "head",
    "mule",
    "fluid",
    "chaos",
    "endless",
    "glad",
    "fossil",
    "fuel",
    "process",
    "keep",
    "come",
    "impact",
    "nurse",
    "cruise",
    "able",
    "fashion",
    "copy",
    "forward",
    "jeans",
    "mistake",
    "give",
    "glow",
    "gasp",
    "valve",
    "pupil",
    "apart",
    "mobile",
    "borrow",
    "copper",
    "timber",
    "test",
    "kingdom",
    "engage",
    "neutral",
    "draft",
    "alarm",
    "desk",
    "advice",
    "soldier",
    "crater",
    "ridge",
    "rule",
    "luggage",
    "private",
    "finger",
    "moral",
    "still",
    "wrestle",
    "mechanic",
    "sign",
    "silk",
    "cabin",
    "decide",
    "organ",
    "silver",
    "apple",
    "helmet",
    "pave",
    "snow",
    "number",
    "champion",
    "ozone",
    "job",
    "online",
    "solar",
    "canyon",
    "typical",
    "royal",
    "permit",
    "firm",
    "only",
    "awesome",
    "cram",
    "patch",
    "boost",
    "cinnamon",
    "current",
    "filter",
    "hundred",
    "marriage",
    "staff",
    "erode",
    "round",
    "exhibit",
    "arrow",
    "wait",
    "tonight",
    "speed",
    "slender",
    "civil",
    "leave",
    "square",
    "chef",
    "clerk",
    "relax",
]

