# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['sagemcom_api']

package_data = \
{'': ['*']}

install_requires = \
['aiohttp>=3.7.3,<4.0.0', 'pyhumps>=3.0.2,<4.0.0']

setup_kwargs = {
    'name': 'sagemcom-api',
    'version': '1.0.6',
    'description': "Python client to interact with SagemCom F@st routers via internal API's.",
    'long_description': '# Sagemcom API Client in Python\n\n(Unofficial) async Python client to interact with Sagemcom F@st routers via internal API\'s. This client offers helper functions to retrieve common used functions, but also offers functionality to do custom requests via XPATH notation.\n\nPython 3.9+ required.\n\n## Features\n\n- Retrieve detailed information of your Sagemcom F@st device\n- Retrieve connected devices (wifi and ethernet)\n- Reboot Sagemcom F@st device\n- Retrieve and set all values of your Sagemcom F@st device\n\n## Supported devices\n\nThe Sagemcom F@st series is used by multiple cable companies, where some cable companies did rebrand the router. Examples are the b-box from Proximus, Home Hub from bell and the Smart Hub from BT.\n\n| Router Model          | Provider(s)          | Authentication Method | Comments                      |\n| --------------------- | -------------------- | --------------------- | ----------------------------- |\n| Sagemcom F@st 3864    | Optus                | sha512                | username: guest, password: "" |\n| Sagemcom F@st 3865b   | Proximus (b-box3)    | md5                   |                               |\n| Sagemcom F@st 3890V3  | Delta / Zeelandnet   | md5                   |                               |\n| Sagemcom F@st 4360Air | KPN                  | md5                   |                               |\n| Sagemcom F@st 5250    | Bell (Home Hub 2000) | md5                   | username: guest, password: "" |\n| Sagemcom F@st 5280    |                      | sha512                |                               |\n| Sagemcom F@st 5364    | BT (Smart Hub)       | md5                   | username: guest, password: "" |\n| SagemCom F@st 5366SD  | Eir F3000            | md5                   |                               |\n| Sagemcom F@st 5370e   | Telia                | sha512                |                               |\n| Sagemcom F@st 5566    | Bell (Home Hub 3000) | md5                   | username: guest, password: "" |\n| Sagemcom F@st 5655V2  | MásMóvil             | md5                   |                               |\n| Sagemcom F@st 5657IL  |                      | md5                   |                               |\n| Speedport Pro         | Telekom              | md5                   |                               |\n\n> Contributions welcome. If you router model is supported by this package, but not in the list above, please create [an issue](https://github.com/iMicknl/python-sagemcom-api/issues/new) or pull request.\n\n## Installation\n\n```bash\npip install sagemcom_api\n```\n\n## Getting Started\n\nDepending on the router model, Sagemcom is using different encryption methods for authentication, which can be found in [the table above](#supported-devices). This package supports MD5 and SHA512 encryption. If you receive a `LoginTimeoutException`, you will probably need to use another encryption type.\n\nThe following script can be used as a quickstart.\n\n```python\nimport asyncio\nfrom sagemcom_api.enums import EncryptionMethod\nfrom sagemcom_api.client import SagemcomClient\n\nHOST = ""\nUSERNAME = ""\nPASSWORD = ""\nENCRYPTION_METHOD = EncryptionMethod.MD5 # or EncryptionMethod.SHA512\n\nasync def main() -> None:\n    async with SagemcomClient(HOST, USERNAME, PASSWORD, ENCRYPTION_METHOD) as client:\n        try:\n            await client.login()\n        except Exception as exception:  # pylint: disable=broad-except\n            print(exception)\n            return\n\n        # Print device information of Sagemcom F@st router\n        device_info = await client.get_device_info()\n        print(f"{device_info.id} {device_info.model_name}")\n\n        # Print connected devices\n        devices = await client.get_hosts()\n\n        for device in devices:\n            if device.active:\n                print(f"{device.id} - {device.name}")\n\n        # Retrieve values via XPath notation, output is a dict\n        custom_command_output = await client.get_value_by_xpath("Device/UserInterface/AdvancedMode")\n        print(custom_command_output)\n\n        # Set value via XPath notation\n        custom_command_output = await client.set_value_by_xpath("Device/UserInterface/AdvancedMode", "true")\n        print(custom_command_output)\n\nasyncio.run(main())\n```\n\n## Functions\n\n- `login()`\n- `get_device_info()`\n- `get_hosts()`\n- `get_port_mappings()`\n- `reboot()`\n- `get_value_by_xpath(xpath)`\n- `set_value_by_xpath(xpath, value)`\n\n## Advanced\n\n### Determine the EncryptionMethod\n\n(not supported yet)\n\n### Handle exceptions\n\nSome functions may cause an error when an attempt is made to execute it. These exceptions are thrown by the client and need to be [handled in your Python program](https://docs.python.org/3/tutorial/errors.html#handling-exceptions). Best practice is to catch some specific exceptions and handle them gracefully.\n\n```python\nfrom sagemcom_api.exceptions import *\n\ntry:\n    await client.set_value_by_xpath("Device/UserInterface/AdvancedMode", "true")\nexcept NonWritableParameterException as exception:\n    print("You don\'t have rights to write to this parameter.")\nexcept UnknownPathException as exception:\n    print("The xpath does not exist.")\n```\n\n### Run your custom commands\n\nNot all values can be retrieved by helper functions in this client implementation. By using XPath, you are able to return all values via the API. The result will be a dict response, or [an exception](#handle-exceptions) when the attempt was not successful.\n\n```python\ntry:\n    result = await client.get_value_by_xpath("Device/DeviceSummary")\nexcept Exception as exception:\n    print(exception)\n```\n\n### Use your own aiohttp ClientSession\n\n> ClientSession is the heart and the main entry point for all client API operations. The session contains a cookie storage and connection pool, thus cookies and connections are shared between HTTP requests sent by the same session.\n\nIn order to change settings like the time-out, it is possible to pass your custom [aiohttp ClientSession](https://docs.aiohttp.org/en/stable/client_advanced.html).\n\n```python\nfrom aiohttp import ClientSession, ClientTimeout\n\nsession = ClientSession(timeout=ClientTimeout(100))\nclient = SagemcomClient(session=session)\n```\n\n## Inspired by\n\n- [wuseman/SAGEMCOM-FAST-5370e-TELIA](https://github.com/wuseman/SAGEMCOM-FAST-5370e-TELIA)\n- [insou22/optus-router-tools](https://github.com/insou22/optus-router-tools)\n- [onegambler/bthomehub_client](https://github.com/onegambler/bthomehub_client)\n',
    'author': 'Mick Vleeshouwer',
    'author_email': 'mick@imick.nl',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/iMicknl/python-sagemcom-api',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
