"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const semver = require("semver");
const my_package_json_1 = require("./private/my-package-json");
const schema_helpers_1 = require("./private/schema-helpers");
const PACKAGE_VERSION = my_package_json_1.loadMyPackageJson().version;
/**
 * Static class with loader routines
 *
 * This class mostly exists to put the schema structs into input position
 * (taken into a function), so that the jsii-diff checker will make sure all
 * structs are only allowed to be weakened in future updates. For example,
 * it is now allowed to add new required fields, since old CDK frameworks
 * would not be emitting those fields yet.
 *
 * At the same time, we might as well validate the structure so code doesn't
 * barf on invalid disk input.
 */
class AssetManifestSchema {
    /**
     * Validate the given structured object as a valid ManifestFile schema
     */
    static validate(file) {
        const obj = file;
        if (typeof obj !== 'object' || obj === null) {
            throw new Error(`Expected object, got '${obj}`);
        }
        schema_helpers_1.expectKey(obj, 'version', schema_helpers_1.isString);
        // Current tool must be >= the version used to write the manifest
        // (disregarding MVs) which we can verify by ^-prefixing the file version.
        if (!semver.satisfies(AssetManifestSchema.currentVersion(), `^${obj.version}`)) {
            throw new Error(`Need CDK Tools >= '${obj.version}' to read this file (current is '${AssetManifestSchema.currentVersion()}')`);
        }
        schema_helpers_1.expectKey(obj, 'files', schema_helpers_1.isMapOf(schema_helpers_1.isObjectAnd(isFileAsset)), true);
        schema_helpers_1.expectKey(obj, 'dockerImages', schema_helpers_1.isMapOf(schema_helpers_1.isObjectAnd(isDockerImageAsset)), true);
    }
    /**
     * Take a ManifestFile as input
     *
     * The presence of this method makes sure the struct is only ever weakened
     * in future releases.
     */
    static input(file) {
        this.validate(file);
    }
    /**
     * Return the version of the schema module
     */
    static currentVersion() {
        return PACKAGE_VERSION;
    }
}
exports.AssetManifestSchema = AssetManifestSchema;
function isFileAsset(entry) {
    schema_helpers_1.expectKey(entry, 'source', source => {
        schema_helpers_1.assertIsObject(source);
        schema_helpers_1.expectKey(source, 'path', schema_helpers_1.isString);
        schema_helpers_1.expectKey(source, 'packaging', schema_helpers_1.isString, true);
        return source;
    });
    schema_helpers_1.expectKey(entry, 'destinations', schema_helpers_1.isMapOf(destination => {
        schema_helpers_1.assertIsObject(destination);
        schema_helpers_1.expectKey(destination, 'region', schema_helpers_1.isString, true);
        schema_helpers_1.expectKey(destination, 'assumeRoleArn', schema_helpers_1.isString, true);
        schema_helpers_1.expectKey(destination, 'assumeRoleExternalId', schema_helpers_1.isString, true);
        schema_helpers_1.expectKey(destination, 'bucketName', schema_helpers_1.isString);
        schema_helpers_1.expectKey(destination, 'objectKey', schema_helpers_1.isString);
        return destination;
    }));
    return entry;
}
function isDockerImageAsset(entry) {
    schema_helpers_1.expectKey(entry, 'source', source => {
        schema_helpers_1.assertIsObject(source);
        schema_helpers_1.expectKey(source, 'directory', schema_helpers_1.isString);
        schema_helpers_1.expectKey(source, 'dockerFile', schema_helpers_1.isString, true);
        schema_helpers_1.expectKey(source, 'dockerBuildTarget', schema_helpers_1.isString, true);
        schema_helpers_1.expectKey(source, 'dockerBuildArgs', schema_helpers_1.isMapOf(schema_helpers_1.isString), true);
        return source;
    });
    schema_helpers_1.expectKey(entry, 'destinations', schema_helpers_1.isMapOf(destination => {
        schema_helpers_1.assertIsObject(destination);
        schema_helpers_1.expectKey(destination, 'region', schema_helpers_1.isString, true);
        schema_helpers_1.expectKey(destination, 'assumeRoleArn', schema_helpers_1.isString, true);
        schema_helpers_1.expectKey(destination, 'assumeRoleExternalId', schema_helpers_1.isString, true);
        schema_helpers_1.expectKey(destination, 'repositoryName', schema_helpers_1.isString);
        schema_helpers_1.expectKey(destination, 'imageTag', schema_helpers_1.isString);
        schema_helpers_1.expectKey(destination, 'imageUri', schema_helpers_1.isString);
        return destination;
    }));
    return entry;
}
//# sourceMappingURL=data:application/json;base64,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