import pytest
from bs4 import BeautifulSoup

from critparse import CriterionParser


def test_extract_next_url_collection(soup_collection):
    expected_url = None
    url = CriterionParser.CriterionParser.extract_next_url(soup_collection)
    assert expected_url == url


def test_extract_next_url_unusual(soup_unusual_collection):
    expected_url = None
    url = CriterionParser.CriterionParser.extract_next_url(soup_unusual_collection)
    assert expected_url == url


def test_extract_next_url_none(soup_none):
    expected_url = None
    url = CriterionParser.CriterionParser.extract_next_url(soup_none)
    assert url == expected_url


def test_determine_url_type_collection(soup_collection):
    one = CriterionParser.CriterionParser.determine_url_type(soup_collection)
    assert one == 'collection'


def test_determine_url_type_special_sub(soup_special_subtitle):
    one = CriterionParser.CriterionParser.determine_url_type(soup_special_subtitle)
    assert one == 'movie'


def test_determine_url_type_none(soup_none):
    one = CriterionParser.CriterionParser.determine_url_type(soup_none)
    assert one is None


def test_extract_collection_name_and_description_collection(soup_collection):
    one, two = CriterionParser.CriterionParser.extract_collection_name_and_description(soup_collection)
    assert one == 'The Front Page'
    assert two == 'Starring Adolphe Menjou, Pat O’Brien, Mary Brian'

def test_extract_collection_name_and_description_coll_dir(soup_collection_director):
    one, two = CriterionParser.CriterionParser.extract_collection_name_and_description(soup_collection_director)
    assert one == 'La Jetée'
    assert two == 'Directed by Chris Marker • 1963 • France'


def test_extract_collection_name_and_description_special_sub(soup_special_subtitle):
    one, two = CriterionParser.CriterionParser.extract_collection_name_and_description(soup_special_subtitle)
    assert one == 'NoName'
    assert two == 'NoDescription'


def test_extract_collection_name_and_description_none(soup_none):
    one, two = CriterionParser.CriterionParser.extract_collection_name_and_description(soup_none)
    assert one == 'Three by Jafar Panahi'
    assert two == 'The brilliant Iranian auteur Jafar Panahi was sentenced to six years in prison in July 2022, after refusing to stop making urgent, perceptive films when he was banned from the profession in 2010. With NO BEARS, his latest film, coming out in December, there’s no better time to revisit three of his most beloved masterpieces, all of which view modern Iran through the eyes of young girls.'


def test_extract_collection_name_and_description_unusual(soup_unusual_collection):
    one, two = CriterionParser.CriterionParser.extract_collection_name_and_description(soup_unusual_collection)
    assert one == 'The Organizer'
    assert two == 'Starring Marcello Mastroianni, Renato Salvatori, Annie Girardot'


def test_extract_title_feature_collection(soup_collection):
    expected = [['1:41:27', 'https://www.criterionchannel.com/the-front-page/videos/the-front-page', 'NoTitle']]
    ret = CriterionParser.CriterionParser.extract_title_feature_from_collection(soup_collection)
    assert ret == expected


def test_extract_title_feature_special_sub(soup_special_subtitle):
    expected = [['1:21:40',
  'https://www.criterionchannel.com/foreign-language-oscar-winners/season:1/videos/war-and-peace-part-3',
  'NoTitle']]
    ret = CriterionParser.CriterionParser.extract_title_feature_from_collection(soup_special_subtitle)
    assert ret == expected


def test_extract_title_feature_none(soup_none):
    expected = [['1:24:49', 'https://www.criterionchannel.com/three-by-jafar-panahi/season:1/videos/the-white-balloon', 'NoTitle']]
    ret = CriterionParser.CriterionParser.extract_title_feature_from_collection(soup_none)
    assert ret == expected


def test_extract_title_feature_unusual(soup_unusual_collection):
    expected = [['2:09:54', 'https://www.criterionchannel.com/the-organizer/videos/the-organizer', 'NoTitle']]
    ret = CriterionParser.CriterionParser.extract_title_feature_from_collection(soup_unusual_collection)
    assert ret == expected


def test_extract_episode_time_and_url_collection(soup_collection):
    expected = [['24:04', 'https://www.criterionchannel.com/the-front-page/videos/restoring-the-front-page',
                 'Restoring THE FRONT PAGE'], ['25:47',
                                               'https://www.criterionchannel.com/the-front-page/videos/david-brendel-on-screenwriter-ben-hecht',
                                               'David Brendel on Screenwriter Ben Hecht'],
                ['58:47', 'https://www.criterionchannel.com/the-front-page/videos/the-front-page-radio-adaptation-1937',
                 'THE FRONT PAGE Radio Adaptation: 1937'],
                ['31:44', 'https://www.criterionchannel.com/the-front-page/videos/the-front-page-radio-adaptation-1946',
                 'THE FRONT PAGE Radio Adaptation: 1946']]
    ret = CriterionParser.CriterionParser.extract_episode_time_and_url(soup_collection)
    assert ret == expected


def test_extract_episode_time_and_url_special_sub(soup_special_subtitle):
    expected = []
    ret = CriterionParser.CriterionParser.extract_episode_time_and_url(soup_special_subtitle)
    assert ret == expected


def test_extract_episode_time_and_url_none(soup_none):
    expected = [['1:24:49', 'https://www.criterionchannel.com/three-by-jafar-panahi/season:1/videos/the-white-balloon', 'The White Balloon'], ['1:34:33', 'https://www.criterionchannel.com/three-by-jafar-panahi/season:1/videos/the-mirror-1', 'The Mirror'], ['1:31:46', 'https://www.criterionchannel.com/three-by-jafar-panahi/season:1/videos/offside', 'Offside']]
    ret = CriterionParser.CriterionParser.extract_episode_time_and_url(soup_none)
    assert ret == expected


def test_extract_episode_time_and_url_unusual(soup_unusual_collection):
    expected = [['06:03', 'https://www.criterionchannel.com/the-organizer/videos/spotlight-on-the-organizer-with-imogen-sara-smith', 'Spotlight on THE ORGANIZER with Imogen Sara Smith'], ['10:38', 'https://www.criterionchannel.com/the-organizer/videos/mario-monicelli', 'Mario Monicelli on THE ORGANIZER']]
    ret = CriterionParser.CriterionParser.extract_episode_time_and_url(soup_unusual_collection)
    assert ret == expected


def test_guess_collection_url():
    url = 'https://www.criterionchannel.com/something-different/videos/something-different'
    expected = 'https://www.criterionchannel.com/something-different'
    result = CriterionParser.CriterionParser.guess_collection_url(url)
    assert result == expected


def test_get_collection_info_collection(soup_collection):
    expected_name = 'Collection:The Front Page'
    expected_series = [['1:41:27', 'https://www.criterionchannel.com/the-front-page/videos/the-front-page', 'NoTitle'],
                       ['24:04', 'https://www.criterionchannel.com/the-front-page/videos/restoring-the-front-page',
                        'Restoring THE FRONT PAGE'], ['25:47',
                                                      'https://www.criterionchannel.com/the-front-page/videos/david-brendel-on-screenwriter-ben-hecht',
                                                      'David Brendel on Screenwriter Ben Hecht'], ['58:47',
                                                                                                   'https://www.criterionchannel.com/the-front-page/videos/the-front-page-radio-adaptation-1937',
                                                                                                   'THE FRONT PAGE Radio Adaptation: 1937'],
                       ['31:44',
                        'https://www.criterionchannel.com/the-front-page/videos/the-front-page-radio-adaptation-1946',
                        'THE FRONT PAGE Radio Adaptation: 1946']]

    name, series = CriterionParser.CriterionParser.get_collection_info(soup_collection)
    assert name == expected_name
    assert series == expected_series


def test_get_collection_info_special_sub(soup_special_subtitle):
    expected_name = 'Collection:NoName'
    expected_series = [['1:21:40',
  'https://www.criterionchannel.com/foreign-language-oscar-winners/season:1/videos/war-and-peace-part-3',
  'NoTitle']]

    name, series = CriterionParser.CriterionParser.get_collection_info(soup_special_subtitle)
    assert name == expected_name
    assert series == expected_series


def test_get_collection_info_none(soup_none):
    expected_name = 'Collection:Three by Jafar Panahi'
    expected_series = [
        ['1:24:49', 'https://www.criterionchannel.com/three-by-jafar-panahi/season:1/videos/the-white-balloon',
         'NoTitle'],
        ['1:24:49', 'https://www.criterionchannel.com/three-by-jafar-panahi/season:1/videos/the-white-balloon',
         'The White Balloon'],
        ['1:34:33', 'https://www.criterionchannel.com/three-by-jafar-panahi/season:1/videos/the-mirror-1',
         'The Mirror'],
        ['1:31:46', 'https://www.criterionchannel.com/three-by-jafar-panahi/season:1/videos/offside', 'Offside']]

    name, series = CriterionParser.CriterionParser.get_collection_info(soup_none)
    assert name == expected_name
    assert series == expected_series


def test_get_collection_info_unusual(soup_unusual_collection):
    expected_name = 'Collection:The Organizer'
    expected_series = [['2:09:54', 'https://www.criterionchannel.com/the-organizer/videos/the-organizer', 'NoTitle'], ['06:03', 'https://www.criterionchannel.com/the-organizer/videos/spotlight-on-the-organizer-with-imogen-sara-smith', 'Spotlight on THE ORGANIZER with Imogen Sara Smith'], ['10:38', 'https://www.criterionchannel.com/the-organizer/videos/mario-monicelli', 'Mario Monicelli on THE ORGANIZER']]

    name, series = CriterionParser.CriterionParser.get_collection_info(soup_unusual_collection)
    assert name == expected_name
    assert series == expected_series


def test_get_edition_info_collection(soup_collection):
    expected_name = 'The Front Page'
    expected_series = [['1:41:27', 'https://www.criterionchannel.com/the-front-page/videos/the-front-page', 'NoTitle'],
                       ['24:04', 'https://www.criterionchannel.com/the-front-page/videos/restoring-the-front-page',
                        'Restoring THE FRONT PAGE'], ['25:47',
                                                      'https://www.criterionchannel.com/the-front-page/videos/david-brendel-on-screenwriter-ben-hecht',
                                                      'David Brendel on Screenwriter Ben Hecht'], ['58:47',
                                                                                                   'https://www.criterionchannel.com/the-front-page/videos/the-front-page-radio-adaptation-1937',
                                                                                                   'THE FRONT PAGE Radio Adaptation: 1937'],
                       ['31:44',
                        'https://www.criterionchannel.com/the-front-page/videos/the-front-page-radio-adaptation-1946',
                        'THE FRONT PAGE Radio Adaptation: 1946']]

    name, series = CriterionParser.CriterionParser.get_edition_info(soup_collection)
    assert name == expected_name
    assert series == expected_series


def test_get_edition_info_special_sub(soup_special_subtitle):
    expected_name = 'NoName'
    expected_series = [['1:21:40',
  'https://www.criterionchannel.com/foreign-language-oscar-winners/season:1/videos/war-and-peace-part-3',
  'NoTitle']]

    name, series = CriterionParser.CriterionParser.get_edition_info(soup_special_subtitle)
    assert name == expected_name
    assert series == expected_series


def test_get_edition_info_none(soup_none):
    expected_name = 'Three by Jafar Panahi'
    expected_series = [
        ['1:24:49', 'https://www.criterionchannel.com/three-by-jafar-panahi/season:1/videos/the-white-balloon',
         'NoTitle'],
        ['1:24:49', 'https://www.criterionchannel.com/three-by-jafar-panahi/season:1/videos/the-white-balloon',
         'The White Balloon'],
        ['1:34:33', 'https://www.criterionchannel.com/three-by-jafar-panahi/season:1/videos/the-mirror-1',
         'The Mirror'],
        ['1:31:46', 'https://www.criterionchannel.com/three-by-jafar-panahi/season:1/videos/offside', 'Offside']]

    name, series = CriterionParser.CriterionParser.get_edition_info(soup_none)
    assert name == expected_name
    assert series == expected_series


def test_get_series_info_collection(soup_collection):
    expected_name = 'Criterion:The Front Page'
    expected_descr = 'Starring Adolphe Menjou, Pat O’Brien, Mary Brian'
    expected_series = [['24:04', 'https://www.criterionchannel.com/the-front-page/videos/restoring-the-front-page', 'Restoring THE FRONT PAGE'], ['25:47', 'https://www.criterionchannel.com/the-front-page/videos/david-brendel-on-screenwriter-ben-hecht', 'David Brendel on Screenwriter Ben Hecht'], ['58:47', 'https://www.criterionchannel.com/the-front-page/videos/the-front-page-radio-adaptation-1937', 'THE FRONT PAGE Radio Adaptation: 1937'], ['31:44', 'https://www.criterionchannel.com/the-front-page/videos/the-front-page-radio-adaptation-1946', 'THE FRONT PAGE Radio Adaptation: 1946']]

    name, descr, series = CriterionParser.CriterionParser.get_series_info(soup_collection)
    assert name == expected_name == name
    assert descr == expected_descr
    assert series == expected_series


def test_get_series_info_special_sub(soup_special_subtitle):
    expected_name = 'Criterion:NoName'
    expected_descr = 'NoDescription'
    expected_series = []

    name, descr, series = CriterionParser.CriterionParser.get_series_info(soup_special_subtitle)
    assert name == expected_name == name
    assert descr == expected_descr
    assert series == expected_series


def test_get_series_info_none(soup_none):
    expected_name = 'Criterion:Three by Jafar Panahi'
    expected_descr = 'The brilliant Iranian auteur Jafar Panahi was sentenced to six years in prison in July 2022, after refusing to stop making urgent, perceptive films when he was banned from the profession in 2010. With NO BEARS, his latest film, coming out in December, there’s no better time to revisit three of his most beloved masterpieces, all of which view modern Iran through the eyes of young girls.'
    expected_series = [
        ['1:24:49', 'https://www.criterionchannel.com/three-by-jafar-panahi/season:1/videos/the-white-balloon',
         'The White Balloon'],
        ['1:34:33', 'https://www.criterionchannel.com/three-by-jafar-panahi/season:1/videos/the-mirror-1',
         'The Mirror'],
        ['1:31:46', 'https://www.criterionchannel.com/three-by-jafar-panahi/season:1/videos/offside', 'Offside']]

    name, descr, series = CriterionParser.CriterionParser.get_series_info(soup_none)
    assert name == expected_name == name
    assert descr == expected_descr
    assert series == expected_series

# ========================================================================================
# ========================================================================================
# ========================================================================================
# ========================================================================================


@pytest.fixture
def soup_unusual_collection():
    content = b'<!doctype html>\n<!--[if lt IE 7]> <html class="no-js lt-ie9 lt-ie8 lt-ie7" lang="en" xml:lang="en"> <![endif]-->\n<!--[if IE 7]>    <html class="no-js lt-ie9 lt-ie8" lang="en" xml:lang="en"> <![endif]-->\n<!--[if IE 8]>    <html class="no-js lt-ie9" lang="en" xml:lang="en"> <![endif]-->\n<!--[if gt IE 8]><!--> <html class="no-js" lang="en" xml:lang="en"> <!--<![endif]-->\n<head lang="en" xml:lang="en">\n  <meta charset="utf-8">\n  <meta http-equiv="X-UA-Compatible" content="IE=edge,chrome=1">\n  <meta name="viewport" content="width=device-width, initial-scale=1">\n\n    <link rel="shortcut icon" href="https://dr56wvhu2c8zo.cloudfront.net/criterionchannel/assets/5922c7f6-cdda-4786-8ac9-521602da043a.png" type="image/png">\n\n    <meta name="apple-itunes-app" content="app-id=1454275199, app-argument=https://criterionchannel.vhx.tv/the-organizer">\n\n    <meta name="facebook-domain-verification" content="qk8q6gxhll8cwf99kggygprhe0y9fz">\n\n  <title>The Organizer - The Criterion Channel</title>\n  <meta name="description" content="Directed by Mario Monicelli \xe2\x80\xa2 1963 \xe2\x80\xa2 France, Yugoslavia, Italy\nStarring Marcello Mastroianni, Renato Salvatori, Annie Girardot\n\nIn turn-of-the-twentieth-century Turin, an accident in a textile factory incites workers to stage a walkout. But it\xe2\x80\x99s not until they receive unexpected aid from a traveling professor (Marcello Mastroianni) that they find their voice, unite, and stand up for themselves. This historical drama by Mario Monicelli, brimming with humor and honesty, is a beautiful and moving ode to the power of the people, and features engaging, naturalistic performances; cinematography by the great Giuseppe Rotunno; and a multilayered, Oscar-nominated screenplay by Monicelli, Agenore Incrocci, and Furio Scarpelli.">\n  <link rel="canonical" href="https://www.criterionchannel.com/the-organizer" />\n\n    <meta property="og:url" content="https://www.criterionchannel.com/the-organizer">\n  <meta property="fb:app_id" content="136862186391940">\n  <meta property="og:title" content="The Organizer - The Criterion Channel">\n  <meta property="og:site_name" content="The Criterion Channel">\n  <meta property="og:image" content="https://vhx.imgix.net/criterionchannelchartersu/assets/1dd7e4a4-f23b-40fc-8821-e39055a9315e.jpg?auto=format%2Ccompress&fit=crop&h=720&q=75&w=1280">\n  <meta property="og:image:width" content="1280">\n  <meta property="og:image:height" content="720">\n  <meta property="og:description" content="Directed by Mario Monicelli \xe2\x80\xa2 1963 \xe2\x80\xa2 France, Yugoslavia, Italy\nStarring Marcello Mastroianni, Renato Salvatori, Annie Girardot\n\nIn turn-of-the-twentieth-century Turin, an accident in a textile factory incites workers to stage a walkout. But it\xe2\x80\x99s not until they receive unexpected aid from a travel...">\n  <meta property="og:type" content="website">\n\n  <meta name="twitter:card" content="summary_large_image">\n    <meta name="twitter:site" content="@criterionchannl">\n    <meta name="twitter:creator" content="@criterionchannl">\n  <meta name="twitter:title" content="The Organizer - The Criterion Channel">\n  <meta name="twitter:description" content="Directed by Mario Monicelli \xe2\x80\xa2 1963 \xe2\x80\xa2 France, Yugoslavia, Italy\nStarring Marcello Mastroianni, Renato Salvatori, Annie Girardot\n\nIn turn-of-the-twentieth-century Turin, an accident in a textile factory incites workers to stage a walkout. But it\xe2\x80\x99s not until they receive unexpected aid from a travel...">\n  <meta name="twitter:image" content="https://vhx.imgix.net/criterionchannelchartersu/assets/1dd7e4a4-f23b-40fc-8821-e39055a9315e.jpg?auto=format%2Ccompress&fit=crop&h=720&q=75&w=1280">\n  <meta name="twitter:image:width" content="1280">\n  <meta name="twitter:image:height" content="720">\n\n\n\n<link rel="author" href="/humans.txt">\n<meta name="csrf-param" content="authenticity_token" />\n<meta name="csrf-token" content="wwa50ZnIMFSwtDP7mAFy0f+Bbm8c7W3pHziYNTkKfVlRJNZ16xzLvHqw04+gMl7rRW5jXVqzeedqTE7UnKRIQw==" />\n\n\n    <!-- apple / android  -->\n  <meta name="mobile-web-app-capable" content="no">\n  <meta name="apple-mobile-web-app-capable" content="no">\n  <meta name="apple-mobile-web-app-title" content="The Criterion Channel">\n  <!-- microsoft tile color -->\n  <meta name="msapplication-TileColor" content="#b9bcc7">\n  <!-- android browser bar color -->\n  <meta name="theme-color" content="#b9bcc7">\n\n    <!-- apple and other browsers -->\n    <link rel="apple-touch-icon" sizes="57x57"    href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=57&w=57">\n    <link rel="apple-touch-icon" sizes="114x114"  href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=114&w=114">\n    <link rel="apple-touch-icon" sizes="72x72"    href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=72&w=72">\n    <link rel="apple-touch-icon" sizes="144x144"  href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=144&w=144">\n    <link rel="apple-touch-icon" sizes="60x60"    href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=60&w=60">\n    <link rel="apple-touch-icon" sizes="120x120"  href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=120&w=120">\n    <link rel="apple-touch-icon" sizes="76x76"    href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=76&w=76">\n    <link rel="apple-touch-icon" sizes="152x152"  href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=152&w=152">\n    <link rel="apple-touch-icon" sizes="180x180"  href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=180&w=180">\n    <!-- windows tile -->\n    <meta name="msapplication-TileImage" content="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=144&w=144">\n    <!-- android icons -->\n    <link rel="icon" type="image/png" sizes="32x32" href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=32&w=32">\n    <link rel="icon" type="image/png" sizes="16x16" href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=16&w=16">\n\n\n    <meta class="swiftype" name="type" data-type="enum" content="Channel" />\n\n\n\n  <link rel="stylesheet" media="all" href="https://cdn.vhx.tv/assets/webpack/icons.svg-ottbundle-798d02cf9d8ee37d5dfe.css" />\n  <link rel="stylesheet" media="all" href="https://cdn.vhx.tv/assets/webpack/sites-ottbundle-c9ea5b6487f913b79233.css" />\n\n    <link rel="stylesheet" media="screen" href="https://cdn.vhx.tv/assets/templates/_shared/typography.montserrat-3b605b0b18ab721686d69bf9c5bd93fc8f657562bc6a431ae71c409675dfb333.css" />\n\n<style>\n\n\n  /* main colors */\n  .site-background-color {\n    background-color: #141414 !important;\n  }\n  .site-primary-highlight-color {\n    background: #b9bcc7 !important;\n  }\n  .site-primary-color,\n  strong.site-primary-color,\n  .site-primary-color strong,\n  a.site-primary-color:link,\n  a.site-primary-color:visited {\n    color: #b9bcc7 !important;\n  }\n  a.site-primary-color:hover,\n  a.site-primary-color:active {\n    color: #9da0a9 !important;\n  }\n\n  .site-border-color,\n  .site-border-color.border-light {\n    border-color: #323232 !important;\n  }\n  .site-border-primary-color,\n  .site-border-primary-color.border-light {\n    border-color: #b9bcc7 !important;\n  }\n\n  /* custom font primary family */\n  .site-font-primary-family {\n    font-family: \'montserrat\', sans-serif !important;\n  }\n\n  /* font primary color */\n  .site-font-primary-color,\n  .site-font-primary-color strong {\n    color: #ffffff !important;\n  }\n\n  /* font secondary color */\n  .site-font-secondary-color,\n  .site-font-secondary-color strong {\n    color: #bebebb !important;\n  }\n\n  /* link primary color */\n  a.site-link-primary-color:link,\n  a.site-link-primary-color:visited,\n  a.site-link-primary-color strong {\n    color: #ffffff !important;\n  }\n  a.site-link-primary-color:hover,\n  a.site-link-primary-color:active {\n    color: #d9d9d9 !important;\n    opacity: .7;\n  }\n  .site-link-primary-color-contain a:link,\n  .site-link-primary-color-contain a:visited {\n    color: #b9bcc7 !important;\n  }\n  .site-link-primary-color-contain a:hover,\n  .site-link-primary-color-contain a:active {\n    color: #9da0a9 !important;\n    opacity: .7;\n  }\n\n  /* link secondary color */\n  .pagination a,\n  a.site-link-secondary-color:link,\n  a.site-link-secondary-color:visited,\n  a.site-link-secondary-color strong,\n  .site-link-secondary-color-contain a:link,\n  .site-link-secondary-color-contain a:visited {\n    color: #7c7c7c;\n  }\n  .pagination a:hover,\n  a.site-link-secondary-color:hover,\n  a.site-link-secondary-color:active,\n  .site-link-secondary-color-contain a:hover,\n  .site-link-secondary-color-contain a:active {\n    color: #696969;\n    opacity: .7;\n  }\n\n  /* link underline font primary color */\n  .site-link-contain a:link,\n  .site-link-contain a:visited {\n    color: #ffffff !important;\n    text-decoration: underline;\n  }\n  .site-link-contain a:hover,\n  .site-link-contain a:active {\n    color: #ffffff !important;\n    opacity: .7;\n    text-decoration: underline;\n  }\n\n  /* border color same as primary color */\n  .site-border-primary-color {\n    border-color: #ffffff !important;\n  }\n  .site-border-top-primary-color {\n    border-top-color: #ffffff !important;\n  }\n  .site-border-bottom-primary-color {\n    border-bottom-color: #ffffff !important;\n  }\n  .site-border-left-primary-color {\n    border-left-color: #ffffff !important;\n  }\n  .site-border-right-primary-color {\n    border-right-color: #ffffff !important;\n  }\n\n  /* link header color */\n  a.site-link-header-color:link,\n  a.site-link-header-color:visited,\n  a.site-link-header-color:link strong,\n  a.site-link-header-color:visited strong {\n    color: #ffffff !important;\n  }\n  a.site-link-header-color:hover,\n  a.site-link-header-color:active {\n    color: #d9d9d9;\n    opacity: .7;\n  }\n .avatar-chevron.icon {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2223%22%20height%3D%2223%22%20viewBox%3D%220%200%2023%2023%22%3E%3Cpolygon%20fill%3D%22%23ffffff%22%20points%3D%2222.9%2C7.2%2020%2C4.4%2011.5%2C12.9%203%2C4.4%200.1%2C7.2%208.7%2C15.7%208.7%2C15.7%2011.5%2C18.6%2011.5%2C18.6%2011.5%2C18.6%2014.3%2C15.7%2014.3%2C15.7%20%22%2F%3E%3C%2Fsvg%3E%0A");\n    background-repeat: no-repeat;\n  }\n  .user-dropdown a:hover {\n    background-color: rgba(51,51,51,0.2);\n    color: #fff;\n  }\n  .user-dropdown input {\n    background-color: #ffffff;\n  }\n  .user-dropdown input:hover {\n    background-color: rgba(51,51,51,0.2);\n    color: #fff;\n  }\n\n  /* buttons site primary */\n  .btn-site-primary,\n  .btn-site-primary:link,\n  .btn-site-primary:visited {\n    color: #000;\n    border-color: #9da0a9;\n    background-color: #b9bcc7;\n  }\n  .btn-site-primary:hover,\n  .btn-site-primary.is-hover,\n  .btn-site-primary:focus,\n  .btn-site-primary:focus:hover,\n  .btn-site-primary:active,\n  .btn-site-primary.is-active {\n    border-color: #9da0a9;\n    background-color: #9da0a9;\n    color: #000;\n  }\n\n  .btn-site-primary[disabled],\n  .btn-site-primary.is-disabled,\n  .btn-site-primary[disabled]:visited,\n  .btn-site-primary.is-disabled:visited,\n  .btn-site-primary[disabled]:hover,\n  .btn-site-primary.is-disabled:hover,\n  .btn-site-primary[disabled].is-hover,\n  .btn-site-primary.is-disabled.is-hover,\n  .btn-site-primary[disabled]:active,\n  .btn-site-primary.is-disabled:active,\n  .btn-site-primary[disabled]:focus,\n  .btn-site-primary.is-disabled:focus,\n  .btn-site-primary[disabled]:focus:hover,\n  .btn-site-primary.is-disabled:focus:hover,\n  .btn-site-primary[disabled].is-active,\n  .btn-site-primary.is-disabled.is-active {\n    opacity: .5;\n    background-color: #b9bcc7;\n    color: #000;\n    border-color: #9da0a9;\n  }\n\n  /* buttons site secondary */\n  .btn-site-secondary,\n  .btn-site-secondary:link,\n  .btn-site-secondary:visited {\n    color: #bebebb;\n    border-color: #141414;\n    background-color: #323232;\n  }\n  .btn-site-secondary:hover,\n  .btn-site-secondary.is-hover,\n  .btn-site-secondary:focus,\n  .btn-site-secondary:active,\n  .btn-site-secondary.is-active {\n    color: #5f5f5e;\n    border-color: #0f0f0f;\n    background-color: #323232;\n  }\n\n  .btn-site-primary.is-processing,\n  .btn-site-secondary.is-processing {\n    color: transparent !important;\n    background-position: center center !important;\n    background-repeat: no-repeat;\n    background-size: 23px 23px;\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3Csvg%20version%3D%221.1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%20%20viewBox%3D%220%200%2040%2040%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cpath%20opacity%3D%220.2%22%20fill%3D%22%23000000%22%20d%3D%22M20.201%2C5.169c-8.254%2C0-14.946%2C6.692-14.946%2C14.946c0%2C8.255%2C6.692%2C14.946%2C14.946%2C14.946%0D%0A%20%20s14.946-6.691%2C14.946-14.946C35.146%2C11.861%2C28.455%2C5.169%2C20.201%2C5.169z%20M20.201%2C31.749c-6.425%2C0-11.634-5.208-11.634-11.634%0D%0A%20%20c0-6.425%2C5.209-11.634%2C11.634-11.634c6.425%2C0%2C11.633%2C5.209%2C11.633%2C11.634C31.834%2C26.541%2C26.626%2C31.749%2C20.201%2C31.749z%22%2F%3E%0D%0A%3Cpath%20fill%3D%22%23000000%22%20d%3D%22M26.013%2C10.047l1.654-2.866c-2.198-1.272-4.743-2.012-7.466-2.012h0v3.312h0%0D%0A%20%20C22.32%2C8.481%2C24.301%2C9.057%2C26.013%2C10.047z%22%3E%0D%0A%20%20%3CanimateTransform%20attributeType%3D%22xml%22%0D%0A%20%20%20%20attributeName%3D%22transform%22%0D%0A%20%20%20%20type%3D%22rotate%22%0D%0A%20%20%20%20from%3D%220%2020%2020%22%0D%0A%20%20%20%20to%3D%22360%2020%2020%22%0D%0A%20%20%20%20dur%3D%220.5s%22%0D%0A%20%20%20%20repeatCount%3D%22indefinite%22%2F%3E%0D%0A%20%20%3C%2Fpath%3E%0D%0A%3C%2Fsvg%3E\');\n  }\n\n  /* buttons dropdown */\n  .btn-dropdown-site,\n  .btn-dropdown-site:link,\n  .btn-dropdown-site:visited {\n    color: #000000;\n    border-color: #b9bcc7;\n    background-color: #b9bcc7;\n  }\n  .btn-dropdown-site:hover,\n  .btn-dropdown-site.is-hover,\n  .btn-dropdown-site:focus,\n  .btn-dropdown-site:active,\n  .btn-dropdown-site.is-active {\n    border-color: #b9bcc7;\n    background-color: #9da0a9;\n  }\n\n  .site-btn-block,\n  .site-btn-block:link,\n  .site-btn-block:visited {\n    color: #ffffff;\n  }\n\n  .site-btn-block:hover,\n  .site-btn-block.is-hover,\n  .site-btn-block.is-active,\n  .site-btn-block:active {\n    box-shadow: inset 0px -4px 0px #ffffff;\n  }\n\n  .site-form-color input[type="email"],\n  .site-form-color input[type="text"],\n  .site-form-color input[type="search"],\n  .site-form-color input[type="password"],\n  .site-form-color input[type="tel"],\n  .site-form-color textarea {\n    color: #323232 !important;\n    border-color: #696969 !important;\n    background-color:  #fcfcfc !important;\n  }\n  .site-form-color label,\n  .site-form-color-minimal input[type="email"],\n  .site-form-color-minimal input[type="text"],\n  .site-form-color-minimal input[type="search"],\n  .site-form-color-minimal input[type="password"],\n  .site-form-color-minimal textarea {\n    color: #ffffff;\n    background: transparent !important;\n  }\n  .site-form-color input[type="text"]:focus,\n  .site-form-color input[type="search"]:focus,\n  .site-form-color input[type="password"]:focus,\n  .site-form-color input[type="tel"]:focus,\n  .site-form-color textarea:focus {\n    border-color: #7c7c7c !important;\n  }\n\n  .site-form-color input[type="text"]::-webkit-input-placeholder,\n  .site-form-color-minimal input[type="text"]::-webkit-input-placeholder,\n  .site-form-color input[type="email"]::-webkit-input-placeholder,\n  .site-form-color-minimal input[type="email"]::-webkit-input-placeholder,\n  .site-form-color input[type="search"]::-webkit-input-placeholder,\n  .site-form-color-minimal input[type="search"]::-webkit-input-placeholder,\n  .site-form-color input[type="password"]::-webkit-input-placeholder,\n  .site-form-color-minimal input[type="password"]::-webkit-input-placeholder,\n  .site-form-color input[type="tel"]::-webkit-input-placeholder,\n  .site-form-color-minimal input[type="tel"]::-webkit-input-placeholder,\n  .site-form-color textarea::-webkit-input-placeholder,\n  .site-form-color-minimal textarea::-webkit-input-placeholder {\n    color: #323232 !important;\n    opacity: .5;\n  }\n  .site-form-color input[type="text"]::-moz-input-placeholder,\n  .site-form-color-minimal input[type="text"]::-moz-input-placeholder,\n  .site-form-color input[type="email"]::-moz-input-placeholder,\n  .site-form-color-minimal input[type="email"]::-moz-input-placeholder,\n  .site-form-color input[type="search"]::-moz-input-placeholder,\n  .site-form-color-minimal input[type="search"]::-moz-input-placeholder,\n  .site-form-color input[type="password"]::-moz-input-placeholder,\n  .site-form-color-minimal input[type="password"]::-moz-input-placeholder,\n  .site-form-color input[type="tel"]::-moz-input-placeholder,\n  .site-form-color-minimal input[type="tel"]::-moz-input-placeholder,\n  .site-form-color textarea::-moz-input-placeholder,\n  .site-form-color-minimal textarea::-moz-input-placeholder {\n    color: #323232 !important;\n    opacity: .5;\n  }\n\n  .site-form-color input[type="text"]:-moz-input-placeholder,\n  .site-form-color-minimal input[type="text"]:-moz-input-placeholder,\n  .site-form-color input[type="email"]:-moz-input-placeholder,\n  .site-form-color-minimal input[type="email"]:-moz-input-placeholder,\n  .site-form-color input[type="search"]:-moz-input-placeholder,\n  .site-form-color-minimal input[type="search"]:-moz-input-placeholder,\n  .site-form-color input[type="password"]:-moz-input-placeholder,\n  .site-form-color-minimal input[type="password"]:-moz-input-placeholder,\n  .site-form-color input[type="tel"]:-moz-input-placeholder,\n  .site-form-color-minimal input[type="tel"]:-moz-input-placeholder,\n  .site-form-color textarea:-moz-input-placeholder,\n  .site-form-color-minimal textarea:-moz-input-placeholder {\n    color: #323232 !important;\n    opacity: .5;\n  }\n\n  .site-form-color input[type="text"]:-ms-input-placeholder,\n  .site-form-color-minimal input[type="text"]:-ms-input-placeholder,\n  .site-form-color input[type="email"]:-ms-input-placeholder,\n  .site-form-color-minimal input[type="email"]:-ms-input-placeholder,\n  .site-form-color input[type="search"]:-ms-input-placeholder,\n  .site-form-color-minimal input[type="search"]:-ms-input-placeholder,\n  .site-form-color input[type="password"]:-ms-input-placeholder,\n  .site-form-color-minimal input[type="password"]:-ms-input-placeholder,\n  .site-form-color input[type="tel"]:-ms-input-placeholder,\n  .site-form-color-minimal input[type="tel"]:-ms-input-placeholder,\n  .site-form-color textarea:-ms-input-placeholder,\n  .site-form-color-minimal textarea:-ms-input-placeholder {\n    color: #323232 !important;\n    opacity: .5;\n  }\n\n  .site-form-color .radio label .radio--icon {\n    border-color: #bebebb !important;\n  }\n  .site-form-color .radio.primary .circle-top span {\n    background: #b9bcc7 !important;\n  }\n  .site-form-color .radio.primary .circle-bottom span {\n    background: #94969f !important;\n  }\n\n  /* header */\n  .site-header {\n    border-bottom: 1px solid #2e2e2e;\n    background-color: #141414;\n\n\n  }\n  .site-header-background-color {\n    background-color: #141414;\n  }\n  @media only screen and (max-width: 40em) {\n     .site-header-background-color-small-only {\n      background-color: #141414;\n    }\n  }\n\n  /* modals */\n  .site-modal {\n    box-shadow: inset 0 0 2px #7c7c7c !important;\n  }\n\n  /* tooltips */\n  .tooltip.bottom-right:after,\n  .tooltip.bottom-left:after,\n  .tooltip.bottom:after,\n  .tooltip.bottom-right:before,\n  .tooltip.bottom-left:before,\n  .tooltip.bottom:before {\n    border-top-color: #7c7c7c;\n  }\n  .tooltip.right:after,\n  .tooltip.right:after\n  .tooltip.right-top:after,\n  .tooltip.right-top:before,\n  .tooltip.right-bottom:before,\n  .tooltip.right-bottom:before {\n    border-right-color: #7c7c7c;\n  }\n\n  .tooltip.left-bottom:after,\n  .tooltip.left-top:after,\n  .tooltip.left:after,\n  .tooltip.left-bottom:before,\n  .tooltip.left-top:before,\n  .tooltip.left:before {\n    border-left-color: #7c7c7c;\n  }\n\n  .tooltip.top-right:after,\n  .tooltip.top-left:after,\n  .tooltip.top:after,\n  .tooltip.top-right:before,\n  .tooltip.top-left:before,\n  .tooltip.top:before {\n    border-bottom-color: #7c7c7c;\n  }\n\n  .tooltip {\n    border: 1px solid #7c7c7c;\n    background: #7c7c7c;\n    color: #ffffff;\n  }\n\n  .reveal-modal {\n    border-color: #323232;\n    box-shadow: 0 0 3px #cbcbcb;\n  }\n  .reveal-modal-bg {\n    background-color: rgba(0,0,0,.8);\n  }\n  .play-outer-circle-large:hover {\n    border-color: #b9bcc7;\n  }\n  .forums-container tr.clickable:hover .thread-activity {\n    background-color: #141414;\n  }\n\n    [data-flash].is-inline.is-error {\n      background: rgba(211, 16, 74, 1) !important;\n    }\n    [data-flash].is-inline.is-error .flash-msg,\n    [data-flash].is-inline.is-error .flash-msg strong {\n      color: #fff !important;\n    }\n    [data-flash].is-inline.is-error .flash-close .icon-remove-mono {\n      display: none !important;\n    }\n    [data-flash].is-inline.is-error .flash-close .icon-remove-inverse {\n      display: block !important;\n    }\n\n  /* custom banner */\n\n\n</style>\n<style>\n  /* icons site primary */\n  .site-icon-arrow-right {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2223%22%20height%3D%2223%22%20viewBox%3D%220%200%2023%2023%22%3E%3Ctitle%3Eicons-arrows-set%20copy%3C%2Ftitle%3E%3Cpath%20d%3D%22M5.61%2C19.75l8.26-8.26L5.61%2C3.24c-0.68-.69.61-1.58%2C1-2%2C0.72-.69%2C1.76.78%2C2.2%2C1.22l3.39%2C3.39L17.41%2C11c0.71%2C0.69-.8%2C1.77-1.22%2C2.19L12.8%2C16.59%2C7.63%2C21.77c-0.7.67-1.58-.61-2-1A0.68%2C0.68%2C0%2C0%2C1%2C5.61%2C19.75Z%22%20fill%3D%22%23b9bcc7%22%2F%3E%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-menu {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Hamburger%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Crect%20x%3D%222.6%22%20y%3D%223.6%22%20width%3D%2218%22%20height%3D%223%22%20fill%3D%22%23ffffff%22%2F%3E%0D%0A%3Crect%20x%3D%222.6%22%20y%3D%2210%22%20width%3D%2218%22%20height%3D%223%22%20fill%3D%22%23ffffff%22%2F%3E%0D%0A%3Crect%20x%3D%222.6%22%20y%3D%2216.3%22%20width%3D%2218%22%20height%3D%223%22%20fill%3D%22%23ffffff%22%2F%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-chrevron-down {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21--%20Generator%3A%20Adobe%20Illustrator%2018.1.1%2C%20SVG%20Export%20Plug-In%20.%20SVG%20Version%3A%206.00%20Build%200%29%20%20--%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%20%20%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%22-294%20385%2023%2023%22%20enable-background%3D%22new%20-294%20385%2023%2023%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cg%3E%0D%0A%20%20%3Cpolygon%20fill%3D%22%23ffffff%22%20points%3D%22-271.1%2C392.9%20-275.4%2C388.7%20-282.5%2C395.8%20-289.6%2C388.7%20-293.9%2C392.9%20-286.7%2C400%20-286.8%2C400.1%20%0D%0A%20%20%20%20-282.5%2C404.3%20-282.5%2C404.3%20-282.5%2C404.3%20-278.2%2C400.1%20-278.3%2C400%20%20%22%2F%3E%0D%0A%3C%2Fg%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-chat {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21--%20Generator%3A%20Adobe%20Illustrator%2018.1.1%2C%20SVG%20Export%20Plug-In%20.%20SVG%20Version%3A%206.00%20Build%200%29%20%20--%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%22-294%20385%2023%2023%22%20enable-background%3D%22new%20-294%20385%2023%2023%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cg%3E%0D%0A%09%3Cg%3E%0D%0A%09%09%3Cpath%20fill%3D%22%23ffffff%22%20d%3D%22M-282.5%2C388.6c2.6%2C0%2C5.1%2C0.1%2C6.8%2C1.3c1.9%2C1.4%2C1.9%2C4.3%2C1.7%2C6c-0.2%2C2.1-1.2%2C3.5-3%2C4.2%0D%0A%09%09%09c-1.3%2C0.5-3%2C0.7-5.1%2C0.7c-1%2C0-2.1-0.1-3.4-0.2l-1-0.1l0%2C0l-1.2-0.4c-1.1-0.3-2.5-1-3.1-2.6c-0.7-1.8-0.7-4.2%2C0-5.9%0D%0A%09%09%09C-289.6%2C389.1-286.6%2C388.6-282.5%2C388.6%20M-282.5%2C386.6c-3.7%2C0-8.4%2C0.3-9.9%2C4.3c-0.9%2C2.2-0.9%2C5.1%2C0%2C7.4c0.7%2C2%2C2.4%2C3.2%2C4.3%2C3.8v3.2%0D%0A%09%09%09l2.6-2.7c1.1%2C0.1%2C2.4%2C0.2%2C3.6%2C0.2c2%2C0%2C4-0.2%2C5.7-0.9c2.6-1%2C4-3.1%2C4.3-5.8c0.3-2.8-0.1-6.1-2.6-7.9%0D%0A%09%09%09C-276.7%2C386.6-279.8%2C386.6-282.5%2C386.6L-282.5%2C386.6z%22%2F%3E%0D%0A%09%3C%2Fg%3E%0D%0A%3C%2Fg%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n\n  /* icons site social */\n  .site-icon-twitter {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%22-294%20385%2023%2023%22%20enable-background%3D%22new%20-294%20385%2023%2023%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cg%3E%0D%0A%09%3Cpath%20fill%3D%22%23ffffff%22%20d%3D%22M-272.5%2C390.3c-0.7%2C0.3-1.5%2C0.5-2.4%2C0.6c0.8-0.5%2C1.5-1.3%2C1.8-2.3c-0.8%2C0.5-1.7%2C0.8-2.6%2C1%0D%0A%09%09c-2.8-3-7.9-0.2-7%2C3.7c-3.4-0.1-6.4-1.7-8.4-4.2c-1.1%2C1.9-0.5%2C4.3%2C1.3%2C5.5c-0.7%2C0-1.3-0.2-1.9-0.5c0%2C1.9%2C1.4%2C3.7%2C3.3%2C4.1%0D%0A%09%09c-0.6%2C0.2-1.2%2C0.2-1.9%2C0.1c0.5%2C1.6%2C2%2C2.8%2C3.8%2C2.8c-1.7%2C1.3-3.9%2C2-6.1%2C1.7c4%2C2.6%2C9.5%2C2.5%2C13.4-0.5c3-2.3%2C4.8-6.1%2C4.6-9.9%0D%0A%09%09C-273.7%2C391.8-273.1%2C391.1-272.5%2C390.3C-273.2%2C390.6-273.1%2C391.1-272.5%2C390.3z%22%2F%3E%0D%0A%3C%2Fg%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-facebook {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21--%20Generator%3A%20Adobe%20Illustrator%2018.1.1%2C%20SVG%20Export%20Plug-In%20.%20SVG%20Version%3A%206.00%20Build%200%29%20%20--%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%22-294%20385%2023%2023%22%20enable-background%3D%22new%20-294%20385%2023%2023%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cpath%20fill%3D%22%23ffffff%22%20d%3D%22M-280.4%2C406.8v-9.3h3.4v-4h-3.4v-2.1c0-0.5%2C0.3-0.9%2C0.9-0.9h2.6v-4.5h-2.8c-3%2C0-5.3%2C2.4-5.3%2C5.5v2.1h-3.1v4%0D%0A%09h3.1v9.3h4.6L-280.4%2C406.8z%22%2F%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-tumblr {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cg%3E%0D%0A%09%3Cpath%20style%3D%22fill%3A%23ffffff%3B%22%20d%3D%22M17.6%2C20.4c-2%2C1-4.4%2C1.4-6.6%2C0.8c-2.3-0.6-3.4-2.3-3.4-4.5v-7H5.4V6.9c2.5-0.8%2C3.7-2.9%2C4-5.4h2.8%0D%0A%09%09v5.1h4.7v3.1h-4.7v5.1c0%2C1-0.2%2C2.4%2C0.9%2C3c1.4%2C0.8%2C3.2%2C0.2%2C4.5-0.6V20.4z%22%2F%3E%0D%0A%3C%2Fg%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-envelope {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cg%3E%0D%0A%09%3Cpath%20style%3D%22fill%3A%23ffffff%3B%22%20d%3D%22M22.6%2C9c0-0.2-0.1-0.5-0.3-0.6L12%2C0.8c-0.3-0.2-0.6-0.2-0.9%2C0L0.7%2C8.4C0.5%2C8.5%2C0.4%2C8.8%2C0.4%2C9v12.6%0D%0A%09%09c0%2C0.4%2C0.3%2C0.8%2C0.8%2C0.8h20.7c0.4%2C0%2C0.8-0.4%2C0.8-0.8V9z%20M11.5%2C2.4l9.6%2C7v0.9l-6.6%2C4.7l5.2%2C3.3c0.8%2C0.5%2C0%2C1.8-0.8%2C1.3L11.5%2C15%0D%0A%09%09l-7.4%2C4.7c-0.8%2C0.5-1.7-0.8-0.8-1.3l5.2-3.3l-6.6-4.7V9.4L11.5%2C2.4z%22%2F%3E%0D%0A%09%3Cpath%20style%3D%22fill%3A%23ffffff%3B%22%20d%3D%22M19.7%2C18.4%22%2F%3E%0D%0A%09%3Cpolygon%20style%3D%22fill%3A%23ffffff%3B%22%20points%3D%2211.5%2C13.1%2013.4%2C14.3%2019.5%2C9.9%2011.5%2C4%203.5%2C9.9%209.6%2C14.3%20%09%22%2F%3E%0D%0A%3C%2Fg%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-check {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cg%3E%0D%0A%09%0D%0A%09%09%3Crect%20x%3D%2211.9%22%20y%3D%222.1%22%20transform%3D%22matrix%28-0.7072%20-0.7071%200.7071%20-0.7072%2016.2216%2029.719%29%22%20style%3D%22fill%3A%23ffffff%3B%22%20width%3D%224.7%22%20height%3D%2218.8%22%2F%3E%0D%0A%09%0D%0A%09%09%3Crect%20x%3D%220.2%22%20y%3D%2211.3%22%20transform%3D%22matrix%280.7072%200.707%20-0.707%200.7072%2011.5962%20-0.598%29%22%20style%3D%22fill%3A%23ffffff%3B%22%20width%3D%2212.5%22%20height%3D%224.7%22%2F%3E%0D%0A%3C%2Fg%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n\n  /* icons site secondary */\n  .site-icon-clock {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cg%3E%0D%0A%09%3Cpath%20style%3D%22fill%3A%23bebebb%3B%22%20d%3D%22M11.5%2C22c-4.5%2C0-8.6-3-10-7.3c-1.4-4.3%2C0.2-9.2%2C4-11.8c3.7-2.6%2C8.9-2.5%2C12.5%2C0.3%0D%0A%09%09c3.6%2C2.8%2C5%2C7.7%2C3.4%2C12C19.8%2C19.3%2C15.8%2C22%2C11.5%2C22z%20M11.5%2C3.3C7%2C3.3%2C3.3%2C7%2C3.3%2C11.5c0%2C4.5%2C3.7%2C8.2%2C8.2%2C8.2c4.5%2C0%2C8.2-3.7%2C8.2-8.2%0D%0A%09%09C19.7%2C7%2C16%2C3.3%2C11.5%2C3.3z%22%2F%3E%0D%0A%09%3Cpath%20style%3D%22fill%3A%23ffffff%3B%22%20d%3D%22M12.9%2C12.9v-2.9V5.5c0-0.4-0.4-0.8-0.8-0.8h-1.3c-0.4%2C0-0.8%2C0.4-0.8%2C0.8v4.5H6.5%0D%0A%09%09c-0.4%2C0-0.8%2C0.4-0.8%2C0.8v1.3c0%2C0.4%2C0.4%2C0.8%2C0.8%2C0.8h3.5H12.9z%22%2F%3E%0D%0A%3C%2Fg%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-reply {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cg%3E%0D%0A%09%3Cpath%20style%3D%22fill%3A%23bebebb%3B%22%20d%3D%22M20.7%2C20c-0.1%2C0.3-0.3%2C0.7-0.4%2C0.9c-0.1%2C0.1-0.2%2C0.2-0.3%2C0.2c-0.2%2C0-0.4-0.2-0.4-0.4%0D%0A%09%09c0-0.2%2C0-0.4%2C0.1-0.6c0-0.5%2C0.1-1%2C0.1-1.5c0-5.7-3.4-6.7-8.6-6.7H8.4V15c0%2C0.4-0.3%2C0.8-0.8%2C0.8c-0.2%2C0-0.4-0.1-0.5-0.2L1%2C9.4%0D%0A%09%09C0.8%2C9.2%2C0.8%2C9%2C0.8%2C8.8c0-0.2%2C0.1-0.4%2C0.2-0.5l6.1-6.1C7.3%2C2%2C7.5%2C1.9%2C7.7%2C1.9c0.4%2C0%2C0.8%2C0.3%2C0.8%2C0.8v3.1h2.7%0D%0A%09%09c3.9%2C0%2C8.8%2C0.7%2C10.5%2C4.8c0.5%2C1.3%2C0.6%2C2.6%2C0.6%2C4C22.2%2C16.2%2C21.4%2C18.4%2C20.7%2C20z%22%2F%3E%0D%0A%3C%2Fg%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-remove {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cpath%20style%3D%22fill%3A%23bebebb%3B%22%20d%3D%22M19.6%2C17.5l-6-6l6-6c0.3-0.3%2C0.3-0.8%2C0-1.1l-0.9-0.9c-0.3-0.3-0.8-0.3-1.1%2C0l-6%2C6l-6-6%0D%0A%09c-0.3-0.3-0.8-0.3-1.1%2C0L3.4%2C4.4c-0.3%2C0.3-0.3%2C0.8%2C0%2C1.1l6%2C6l-6%2C6c-0.3%2C0.3-0.3%2C0.8%2C0%2C1.1l0.9%2C0.9c0.3%2C0.3%2C0.8%2C0.3%2C1.1%2C0l6-6l6%2C6%0D%0A%09c0.3%2C0.3%2C0.8%2C0.3%2C1.1%2C0l0.9-0.9C19.9%2C18.3%2C19.9%2C17.8%2C19.6%2C17.5z%22%2F%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-back {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cpath%20style%3D%22fill%3A%23bebebb%3B%22%20d%3D%22M11.5%2C20.9l1-1c0.3-0.3%2C0.3-0.7%2C0-0.9L6%2C12.5h14.6c0.4%2C0%2C0.7-0.3%2C0.7-0.7v-1.3c0-0.4-0.3-0.7-0.7-0.7%0D%0A%09H6l6.5-6.5c0.3-0.3%2C0.3-0.7%2C0-0.9l-1-1c-0.3-0.3-0.7-0.3-0.9%2C0l-9.4%2C9.4c-0.3%2C0.3-0.3%2C0.7%2C0%2C0.9l9.4%2C9.4%0D%0A%09C10.9%2C21.2%2C11.3%2C21.2%2C11.5%2C20.9z%22%2F%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-link {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cpath%20style%3D%22fill-rule%3Aevenodd%3Bclip-rule%3Aevenodd%3Bfill%3A%23bebebb%3B%22%20d%3D%22M19.3%2C8.2c-1.8%2C1.8-3.7%2C4.6-6.6%2C3.8c0.8-0.8%2C1.6-1.6%2C2.4-2.4%0D%0A%09c0.9-0.9%2C1.8-1.8%2C2.6-2.7c1.9-1.9-0.8-4.8-2.6-2.9c-0.5%2C0.5-0.9%2C0.9-1.4%2C1.4c-0.2%2C0.2-1.4%2C1.1-1.4%2C1.4l0%2C0c-0.8%2C0.8-1.6%2C1.6-2.4%2C2.4%0D%0A%09c-0.4-1.3%2C0-2.8%2C1-3.8c1.3-1.3%2C2.7-3.3%2C4.5-3.8C19.3%2C0.5%2C22.1%2C5.4%2C19.3%2C8.2%22%2F%3E%0D%0A%3Cpath%20style%3D%22fill-rule%3Aevenodd%3Bclip-rule%3Aevenodd%3Bfill%3A%23bebebb%3B%22%20d%3D%22M7.9%2C15.6C7%2C16.5%2C5.5%2C15%2C6.5%2C14.1c1.4-1.4%2C2.7-2.7%2C4.1-4.1%0D%0A%09c0.8-0.8%2C1.5-1.5%2C2.3-2.3c0.2-0.2%2C0.3-0.3%2C0.5-0.5c1-0.6%2C2%2C0.7%2C1.2%2C1.6C12.4%2C11.2%2C10.1%2C13.3%2C7.9%2C15.6z%22%2F%3E%0D%0A%3Cpath%20style%3D%22fill-rule%3Aevenodd%3Bclip-rule%3Aevenodd%3Bfill%3A%23bebebb%3B%22%20d%3D%22M5.8%2C13.4c-0.9%2C0.9-1.8%2C1.8-2.6%2C2.7c-1.9%2C1.9%2C0.8%2C4.8%2C2.6%2C2.9%0D%0A%09c0.9-0.9%2C2-1.8%2C2.8-2.9c0.8-0.8%2C1.6-1.6%2C2.4-2.4c0.4%2C1.3%2C0%2C2.8-1%2C3.8c-1.3%2C1.3-2.7%2C3.3-4.5%2C3.8c-3.8%2C1.1-6.7-3.8-3.9-6.6%0D%0A%09C3.4%2C13%2C5.4%2C10.2%2C8.2%2C11C7.4%2C11.8%2C6.6%2C12.6%2C5.8%2C13.4%22%2F%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-trash {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%221792px%22%20height%3D%221792px%22%20viewBox%3D%22590%20-500%201792%201792%22%20enable-background%3D%22new%20590%20-500%201792%201792%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cpath%20fill%3D%22%23bebebb%22%20d%3D%22M1294%2C876V172c0-9.3-3-17-9-23s-13.7-9-23-9h-64c-9.3%2C0-17%2C3-23%2C9s-9%2C13.7-9%2C23v704c0%2C9.3%2C3%2C17%2C9%2C23%0D%0A%09s13.7%2C9%2C23%2C9h64c9.3%2C0%2C17-3%2C23-9S1294%2C885.3%2C1294%2C876z%20M1550%2C876V172c0-9.3-3-17-9-23s-13.7-9-23-9h-64c-9.3%2C0-17%2C3-23%2C9%0D%0A%09s-9%2C13.7-9%2C23v704c0%2C9.3%2C3%2C17%2C9%2C23s13.7%2C9%2C23%2C9h64c9.3%2C0%2C17-3%2C23-9S1550%2C885.3%2C1550%2C876z%20M1806%2C876V172c0-9.3-3-17-9-23%0D%0A%09s-13.7-9-23-9h-64c-9.3%2C0-17%2C3-23%2C9s-9%2C13.7-9%2C23v704c0%2C9.3%2C3%2C17%2C9%2C23s13.7%2C9%2C23%2C9h64c9.3%2C0%2C17-3%2C23-9S1806%2C885.3%2C1806%2C876z%0D%0A%09%20M1262-116h448l-48-117c-4.7-6-10.3-9.7-17-11h-317c-6.7%2C1.3-12.3%2C5-17%2C11L1262-116z%20M2190-84v64c0%2C9.3-3%2C17-9%2C23s-13.7%2C9-23%2C9h-96%0D%0A%09v948c0%2C55.3-15.7%2C103.2-47%2C143.5s-69%2C60.5-113%2C60.5h-832c-44%2C0-81.7-19.5-113-58.5s-47-86.2-47-141.5V12h-96c-9.3%2C0-17-3-23-9%0D%0A%09s-9-13.7-9-23v-64c0-9.3%2C3-17%2C9-23s13.7-9%2C23-9h309l70-167c10-24.7%2C28-45.7%2C54-63s52.3-26%2C79-26h320c26.7%2C0%2C53%2C8.7%2C79%2C26%0D%0A%09s44%2C38.3%2C54%2C63l70%2C167h309c9.3%2C0%2C17%2C3%2C23%2C9S2190-93.3%2C2190-84z%22%2F%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-replay {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2223%22%20height%3D%2223%22%20viewBox%3D%220%200%2023%2023%22%3E%3Cpath%20d%3D%22M16.4%2016.9c-2.8%202.4-7.1%202.3-9.8-.3l-2%202c3.8%203.8%209.9%203.9%2013.9.3l-2.1-2z%22%20fill%3D%22%23b9bcc7%22%2F%3E%3Cpath%20d%3D%22M15.3%205.3l.9.6c3%202.6%203.7%207.2.9%2010.3l2%202c2.3-2.5%203.1-6%202.3-9.3-.9-3.6-3.8-6.4-7.4-7.3-3.8-.8-8.1.7-10.4%204l.1-1.2c0-.2%200-.5-.2-.7-.6-.8-1.7-.4-1.8.4l-.5%204.6c-.1.6.4%201.1%201%201.1h4.6c.9%200%201.3-1.1.8-1.7-.3-.2-.5-.3-.8-.3h-1.2c2-3.2%206.3-4.4%209.7-2.5z%22%20fill%3D%22%23b9bcc7%22%2F%3E%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-alt-download {\n    background-image: url(\'data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2223%22%20height%3D%2223%22%20viewBox%3D%220%200%2023%2023%22%3E%3Cg%20fill-rule%3D%22evenodd%22%20clip-rule%3D%22evenodd%22%20fill%3D%22%23ffffff%22%3E%3Cpath%20d%3D%22M17.3%2017.2c-1.1%200-1.4-1.5-.5-2%20.4-.2%201-.1%201.5-.3.6-.2%201-.6%201.4-1%201.3-1.6.5-4.1-1.4-4.8-.3-.1-.7-.2-1.1-.2-.2%200-.3-.8-.4-1-.4-.9-1-1.8-1.7-2.4-1.8-1.6-4.5-1.8-6.6-.5-1%20.5-1.7%201.4-2.2%202.3-.2.3-.3.7-.4%201-.2.7-.2.5-.8.7-1.9.4-3%202.5-2.2%204.3.4.9%201.3%201.6%202.3%201.8.5.1%201%200%201.3.4.4.4.3%201.1-.2%201.4-.7.6-2.2.1-2.9-.3-1-.5-1.8-1.3-2.3-2.3-.8-1.6-.7-3.5.2-4.9.4-.8%201.1-1.4%201.8-1.9.4-.2.9-.3%201.1-.7.2-.4.5-.8.7-1.2%202.1-3.1%206.2-4.4%209.6-2.9%201.6.7%203%201.9%203.8%203.4.2.4.3.9.7%201.1l1.2.6c.8.5%201.4%201.2%201.8%202%201.6%203.4-1%207.4-4.7%207.4%22%20fill%3D%22%23000000%22%3E%3C%2Fpath%3E%3Cpath%20d%3D%22M15.4%2017.2l-3.1%203.4c-.4.4-1.1.5-1.5%200-.5-.6-1.1-1.2-1.6-1.8-.2-.2-1.2-1.6-1.5-1.6-.9-1%20.7-2.4%201.5-1.4l1.3%201.4v-3.9c0-.4-.1-.9%200-1.3%200-1.3%202.1-1.3%202.1%200v5.2c.5-.5%201-1.4%201.7-1.7.8-.4%201.8.9%201.1%201.7z%22%20fill%3D%22%23000000%22%3E%3C%2Fpath%3E%3C%2Fg%3E%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-random {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2223%22%20height%3D%2223%22%20viewBox%3D%220%200%2023%2023%22%3E%3Cpath%20d%3D%22M5.7%2015.2v-6H8L4%204.6-.1%209.2h2.3v7c0%201.3%201%202.3%202.3%202.3H15l-3-3.2H5.7z%22%20fill%3D%22#ffffff%22%2F%3E%3Cpath%20d%3D%22M20.7%2013.8v-7c0-1.3-1-2.3-2.3-2.3H8l3%203.2h6.3v6H15l4.1%204.6%204.1-4.6h-2.5z%22%20fill%3D%22#ffffff%22%2F%3E%3C%2Fsvg%3E");\n    background-repeat: no-repeat;\n  }\n  .site-icon-visa {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2271%22%20height%3D%2243%22%20viewBox%3D%220%200%2071%2043%22%3E%3Cpath%20d%3D%22M51.9%2017.9c-.2.5-.5%201.4-.5%201.4l-1.5%204H53c-.2-.7-.9-4.2-.9-4.2l-.2-1.2zM69%200H2C.9%200%200%20.9%200%202v39c0%201.1.9%202%202%202h67c1.1%200%202-.9%202-2V2c0-1.1-.9-2-2-2zM11.6%2014h6c.8%200%201.5.3%201.7%201.2l1.3%206.6c-1.3-3.5-4.4-6.4-9.1-7.6l.1-.2zm7.1%2014.4l-3-11.5c2.5%201.5%204.2%203.5%205%205.9l.2%201%203.7-9.8h3.9l-5.9%2014.4h-3.9zm9%200L30.1%2014h3.7l-2.3%2014.4h-3.8zm10.2.2c-1.7%200-3.3-.4-4.2-.8l.5-3.2.5.2c1.2.5%202.4.8%203.8.8.9-.1%202.2-.6%201.9-1.7-.3-.9-1.6-1.3-2.3-1.7-1-.5-2-1.2-2.5-2.2-1-2%20.2-4.2%201.9-5.3%202.2-1.3%205.1-1.3%207.4-.4l-.5%203.1-.3-.2c-1-.4-4.9-1.3-4.9.7%200%20.8%201.2%201.3%201.8%201.6%201%20.5%202.1%201%202.8%202%201.1%201.4.8%203.5-.2%204.8-1.4%201.8-3.6%202.3-5.7%202.3zm16.2-.2s-.3-1.7-.4-2.2h-4.8c-.1.4-.8%202.2-.8%202.2h-3.9l5.5-13.2c.4-.9%201.1-1.2%202-1.2h2.9l3%2014.4h-3.5z%22%20fill%3D%22#ffffff%22%3E%3C%2Fpath%3E%3C%2Fsvg%3E");\n    background-repeat: no-repeat;\n  }\n  .site-icon-amex {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2271%22%20height%3D%2243%22%20viewBox%3D%220%200%2071%2043%22%3E%3Cpath%20d%3D%22M35.5%2023.5h-5l-2%202.2-1.9-2.2h-6.3v6.6h6.2l2-2.2%201.9%202.2h3.1v-2.2h2c1.5%200%202.7-.6%202.7-2.2%200-1.7-1.3-2.2-2.7-2.2zm-9.9%205.2h-3.9v-1.3h3.5V26h-3.5v-1.2h4l1.7%201.9-1.8%202zm6.3.8l-2.4-2.7%202.4-2.6v5.3zm3.6-3h-2v-1.7h2.1c1.2%200%201.2%201.7-.1%201.7zm9.4-7.8h1.8l.6-1.5h3.3l.6%201.5h3.2v-5l3%205h2.2v-6.6H58v4.6l-2.8-4.6h-2.4v6.2l-2.7-6.2h-2.4l-2.2%205.2H45c-1.6%200-2.1-2.3-1.1-3.4.3-.3.6-.4%201.2-.4h1.5v-1.4h-1.5c-2.1%200-3.2%201.3-3.2%203.3-.1%202%201%203.3%203%203.3zm4.1-5.5l1.1%202.6h-2.2l1.1-2.6zm-15.2%203.1h1.7c.6%200%20.9.1%201.1.3.3.3.2.9.2%201.3v.8h1.6v-1.3c0-.6%200-.9-.3-1.2-.1-.2-.4-.4-.8-.6.9-.4%201.3-1.4%201-2.3-.3-1.2-1.7-1.2-2.6-1.2h-3.6v6.6h1.6v-2.4zm0-2.8h1.9c.3%200%20.6%200%20.7.1.5.2.4%201%200%201.2-.2.1-.5.1-.8.1h-1.9v-1.4zm-2.7%203.8h-3.7V16H31v-1.3h-3.6v-1.2h3.7v-1.4h-5.3v6.6h5.3zM23%2018.7h1.6v-6.6H22l-1.9%204.5-2.1-4.5h-2.6v6.2l-2.7-6.2h-2.3l-2.8%206.6h1.7l.6-1.5h3.3l.6%201.5H17v-5.2l2.3%205.2h1.4l2.3-5.2v5.2zm-12.5-2.9l1.1-2.6%201.1%202.6h-2.2zm35.6%2014.3h5.3v-1.4h-3.7v-1.3h3.6V26h-3.6v-1.2h3.7v-1.3h-5.3zM61.8%2026H61c-.3%200-.6%200-.8-.2-.2-.2-.1-.6%200-.7l.1-.1c.2-.1.3-.1.6-.1h2.9v-1.4h-3.2c-1%200-1.6.4-1.9.8l-.1.1-.1.2c-.1.1-.1.3-.1.4v.7l.1.6c0%20.1%200%20.1.1.2%200%20.1.1.2.1.2l.1.1.1.1s.1%200%20.1.1c.1.1.2.1.3.1%200%200%20.1%200%20.1.1.7.2%201.4.2%202.2.2%201%200%20.9%201.2-.1%201.2h-3V30h3c1.1%200%202.2-.4%202.4-1.6.4-1.6-.6-2.3-2.1-2.4zm-16.5-.8c0-.8-.3-1.2-.8-1.5-.5-.3-1.1-.3-1.9-.3H39V30h1.6v-2.4h1.7c1.1%200%201.4.5%201.4%201.6v.8h1.6v-1.3c0-.8-.2-1.5-1-1.8.3-.1%201-.6%201-1.7zm-2.1%201c-.2.1-.5.1-.8.1h-1.9v-1.5h1.9c.3%200%20.6%200%20.8.1.5.3.4%201%200%201.3-.2.1.2-.2%200%200zM69%200H2C.9%200%200%20.9%200%202v39c0%201.1.9%202%202%202h67c1.1%200%202-.9%202-2V2c0-1.1-.9-2-2-2zM18.2%2018.9V20h-5.3l-.6-1.5h-1.4l-.6%201.5H5.5l4-9.2h4.1l.5%201.1v-1.1H19l1.1%202.4%201-2.4h15.3c.7%200%201.3.1%201.8.5v-.5h4.2v.5c.7-.4%201.6-.5%202.6-.5h6.1l.6%201.1v-1.1h4.5l.6%201.1v-1.1h4.4V20h-4.4l-.8-1.4V20h-5.5l-.6-1.5h-1.4l-.8%201.5h-2.9c-1.1%200-2-.3-2.5-.6v.6h-6.8v-2.1c0-.3-.1-.3-.2-.3H35V20H21.9v-1.1l-.5%201.1h-2.8l-.4-1.1zm46.9%2011c-.7%201.3-2%201.5-3.3%201.5h-4.4v-.6c-.5.4-1.3.6-2.1.6h-13v-2.1c0-.3%200-.3-.3-.3h-.2v2.4h-4.2v-2.5c-.7.3-1.5.3-2.2.3h-.5v2.2h-5.1L28.5%2030l-1.3%201.4h-8.3v-9.2h8.5l1.2%201.4%201.3-1.4h5.7c.7%200%201.7.1%202.2.5v-.5h5.1c.5%200%201.5.1%202.1.5v-.5h7.7v.5c.4-.4%201.2-.5%201.9-.5h4.3v.5c.4-.3%201-.5%201.8-.5h4.5v3.3h-.9c1.3.9%201.6%203%20.8%204.4-.4.7.3-.5%200%200zM39.4%2012.1H41v6.6h-1.6zM55.6%2026h-.8c-.5%200-1.5-.5-.7-1%20.2-.1.3-.1.6-.1h2.9v-1.4h-3.2c-1.5%200-2.6%201.2-2.1%202.7.4%201.3%202.1%201.2%203.2%201.3%201%200%20.9%201.2-.1%201.2h-3.1v1.4h3.1c1.1%200%202.2-.4%202.4-1.6.4-1.7-.7-2.4-2.2-2.5z%22%20fill%3D%22#ffffff%22%3E%3C%2Fpath%3E%3C%2Fsvg%3E");\n    background-repeat: no-repeat;\n  }\n  .site-icon-mc {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2271%22%20height%3D%2243%22%20viewBox%3D%220%200%2071%2043%22%3E%3Cpath%20d%3D%22M36.2%2022.5c.2-.9.3-2.2-.8-2.7-1.4-.7-3%20.1-3.6%201.5-.3.7-.3%201.5-.1%202.2.6%201.4%202.7%201.8%204%201.1l.2-1.1c-.9.2-3.1.5-2.9-1h3.2zm-1.8-1.9c.6%200%20.7.4.6.9h-1.8c.2-.6.6-.9%201.2-.9zm-3.8%202.3l.4-2.1h.7l.3-1.1h-.8l.3-1.4h-1.4l-1%204.9c-.3%201.3%201%202.1%202.2%201.4l.1-1c-.8%200-.8-.3-.8-.7zm8-2.7l-.1-.5h-1.2l-1%205.1h1.4l.4-2.3c.4-1.5%201.1-1.4%201.4-1.4l.4-1.4c-.7%200-1.1.2-1.3.5zm-10.2%201.9c-.2-.1-2.1-1-1-1.3.3%200%201%200%201.4.1l.3-1c-1.3-.5-3.7-.2-3.6%201.6.1%201.1%201.3.9%201.8%201.6.7.9-1.9.6-2.2.5l-.1%201c.8.2%202%20.3%202.8%200%201-.3%201.7-1.9.6-2.5-.3-.2.3.1%200%200zm14.5-2.4c.5%200%20.7%200%201.1.3l.9-.9c-1.7-1.9-4.5-.3-5.1%201.8-.7%202.7%201.4%204.8%204%203.6l-.4-1.3c-2.8%201-2.8-3.5-.5-3.5zM23%2019.6c-.4%200-.9%200-1.8.2l-.2%201.1c.6-.1%202.9-.7%202.6.6-1.1.1-2.2-.1-2.9%201-1.1%201.6.8%203.2%202.3%202.1l.2.3h1.3L25%2022c.2-.7.4-2.4-2-2.4zm.3%203.4c-.1.5-1.3%201-1.5.3-.3-.9%201.1-1%201.6-.9l-.1.6zm27.9-2.9l-.2-.5h-1.2l-1%205.2h1.4l.4-2.4c.4-1.5%201.1-1.5%201.4-1.5l.4-1.4c-.6.1-1%20.4-1.2.6zM69%200H2C.9%200%200%20.9%200%202v39c0%201.1.9%202%202%202h67c1.1%200%202-.9%202-2V2c0-1.1-.9-2-2-2zM45.1%2035.5c-3.4%200-6.4-1.2-8.8-3.1%201.8-1.8%203.2-4%204-6.5h-1.2c-.7%202.2-2%204.2-3.7%205.8-1.7-1.6-2.9-3.6-3.6-5.8h-1.2c.8%202.5%202.1%204.7%204%206.5-4%203.2-9.5%204.1-14.3%202-4-1.7-7.1-5.4-8.1-9.6C10%2015.4%2018%206.4%2027.5%207.6c2.6.3%205.1%201.4%207.1%203-1.8%201.8-3.2%204-4%206.5h1.2c.7-2.2%202-4.2%203.6-5.8%201.7%201.6%202.9%203.6%203.7%205.8h1.2c-.8-2.5-2.2-4.7-4-6.5%207.6-6.2%2019.4-2.5%2022.2%206.8%202.7%208.9-4.1%2018.1-13.4%2018.1zm7.2-13.7c-.5%201.6.5%203%201.5%203%20.5%200%20.9-.2%201.2-.4l.2.4h1.5l1.2-6.6h-1.5l-.4%202c-.7-.5-1.2-.8-2-.6-1%20.3-1.5%201.3-1.7%202.2-.5%201.6.7-2.4%200%200zm3.1-.8c.2.2.2.8.1%201.3-.2.4-.4.9-.9%201-.6.1-1-.4-1-1%20.1-.6%201.1-2.2%201.8-1.3.2.3-.4-.5%200%200zm-36.9-2.5l-1.7%203.8-.3-3.8h-2.2l-1.2%206.3h1.4l.8-4.5.4%204.5h1.4l1.9-4.4-.8%204.4h1.5l1.1-6.3zm28.4%201.1c-.4%200-.9%200-1.8.2l-.2%201.1c.6-.1%202.9-.7%202.6.6-1.1.1-2.2-.1-3%201-1.1%201.6.8%203.2%202.3%202.1l.2.3h1.3l.6-2.8c.1-.8.4-2.5-2-2.5zm.2%203.4c-.1.5-1.3%201-1.5.3-.3-.9%201.1-1%201.6-.9l-.1.6z%22%20fill%3D%22#ffffff%22%3E%3C%2Fpath%3E%3C%2Fsvg%3E");\n    background-repeat: no-repeat;\n  }\n  .site-icon-discover {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2271%22%20height%3D%2243%22%20viewBox%3D%220%200%2071%2043%22%3E%3Cpath%20d%3D%22M6.5%2012.1v6.5s4.2.5%204.2-3.2v-.1c-.1-3.6-4.2-3.2-4.2-3.2zM69%200H2C.9%200%200%20.9%200%202v39c0%201.1.9%202%202%202h67c1.1%200%202-.9%202-2V2c0-1.1-.9-2-2-2zm-9.8%2010.4h3.2c1.7%200%203.1%201.2%203.1%202.9%200%201.3-.8%202.5-2.1%202.9l3%204.2H64l-2.7-4h-.2v4h-1.9v-10zm-6.8%200H58v1.8h-3.5v2.2h3.4v1.8h-3.4v2.5H58v1.8h-5.5V10.4zm-8.2%200l2.7%206.5%202.6-6.5h2.1l-4.2%2010.2h-1.2L42%2010.4h2.2zm-31.5%205c0%203.2-2.9%204.9-4.8%204.9H4.6v-9.9h3.3c1.9%200%204.8%201.8%204.8%205zm2.8%204.9h-2v-10h2v10zm4.2.3c-2.5%200-3.6-1.9-3.6-1.9l1.2-1.2c.7%201%203.1%202.2%203.5.4.5-1.8-1.8-1.9-2.9-2.4-1.7-.9-1.8-3.4-.2-4.6%201.5-1.1%203.5-.6%204.8.6l-1.1%201.3c-.7-.8-3-1.2-2.9.4%200%201%202%201.3%202.7%201.7%203.2%201.3%201.6%205.7-1.5%205.7zm11.6-.6c-3.4%201.8-7.7-.8-7.7-4.6%200-3.9%204.2-6.4%207.7-4.6v2.4c-2.1-2.1-5.8-.8-5.8%202.3%200%203%203.7%204.4%205.8%202.3V20zm6-9.8c2.8%200%205.2%202.3%205.2%205.2%200%202.8-2.3%205.2-5.2%205.2-2.8%200-5.2-2.3-5.2-5.2%200-2.8%202.4-5.2%205.2-5.2zM68%2040H32.8C45%2036.8%2057.2%2032.4%2068%2025.7V40zm-4.4-26.5v-.1c0-.6-.5-1.4-1.3-1.4h-1.1v2.9h1.1c.8%200%201.3-.7%201.3-1.4z%22%20fill%3D%22#ffffff%22%3E%3C%2Fpath%3E%3C%2Fsvg%3E");\n    background-repeat: no-repeat;\n  }\n  .site-icon-jcb {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2271%22%20height%3D%2243%22%20viewBox%3D%220%200%2071%2043%22%3E%3Cpath%20d%3D%22M69%200H2C-.1%200%200%202.1%200%203.5v37.4C0%2042.6%201.4%2043%202.7%2043h66c1.4%200%202.4-.8%202.4-2.3V3c-.1-1.4-.3-3-2.1-3zM28.3%2030.8c0%202.4-1.9%204.4-4.3%204.6-2.1.3-4.3%200-6.5%200-.9%200-.8.1-.8-.8v-9c2.8.7%209.5%201.7%2010.4-2.2.3-1.2.1-2.6.1-3.8v-2.1c0-.4.1-.5-.3-.5h-3.7v3.4c0%201.5.3%203.5-1.5%204.2-.8.3-1.7.2-2.5%200-.9-.2-1.8-.5-2.6-.9-.1-.1%200-.5%200-.6v-6.4c0-2.1-.4-4.7.5-6.7.5-1%201.3-1.6%202.2-2.1%201.4-.6%208.7-.5%208.7-.3v15.1c.3%202.7.3%205.4.3%208.1zm13%20.1c0%202.4-2%204.4-4.4%204.6-.9.1-2%200-2.9%200h-4c-.4%200-.2-1.3-.2-1.6v-9.1c1.4%201.2%203.4%201.5%205.2%201.5%201.1%200%202.1%200%203.1-.2l1.2-.2c.1%200%20.6-.1.7-.2.2-.3%200-1.5%200-1.9v-.1c-1.4.7-3.4%201.4-5%201-3-.4-3.4-4.8-.8-6.1%201.9-.7%204.1-.1%205.8.7v-1.8c0-.3-.3-.3-.5-.3-.4-.1-.8-.2-1.2-.2-.9-.1-1.9-.2-2.8-.2-1.8%200-3.7.2-5.3%201.2l-.5.3v-3.5c0-1.2-.2-2.6.1-3.8.3-1.3%201.2-2.2%202.2-2.9.7-.4%201.4-.6%202.2-.6%202.3-.1%204.6%200%206.8%200%20.3%200%20.3%2021.2.3%2023.4zm12.9%200c0%202.5-2.1%204.5-4.5%204.6-2.3.1-4.6%200-6.9%200-.2%200-.1-6.1-.1-6.7V26h7c.4%200%20.7%200%201.1-.1%201.3-.3%202.5-1.4%202-2.9-.3-.9-1.3-1.4-2.3-1.6h-.3l1.1-.4c.7-.4%201.1-1.2%201-2-.1-1.2-1.2-1.8-2.3-1.9-1.8-.2-3.8%200-5.6%200h-1.6c-.3%200-.2-.3-.2-.6v-4.1c0-2.5%201.8-4.6%204.3-4.8%202.2-.2%204.4%200%206.6%200%20.6%200%20.7-.1.7.5v22.8zm-6.5-8.8c1.5.2%201.6%202.5%200%202.5h-2.3c-.1%200-.4.1-.4-.1v-2.4c0-.2.4-.1.5-.1.7.1%201.4.1%202.2.1zm-2.8-1.4v-2c0-.6.9-.3%201.3-.3.5%200%201.5-.2%201.9.3.7.5.4%201.7-.4%202-.8.2-1.9%200-2.8%200z%22%20fill%3D%22#ffffff%22%3E%3C%2Fpath%3E%3C%2Fsvg%3E");\n    background-repeat: no-repeat;\n  }\n  .site-icon-calendar {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2223%22%20height%3D%2223%22%20viewBox%3D%220%200%2023%2023%22%3E%3Cpath%20d%3D%22M19.3%202.8h-1.1v2.5h-4V2.8H8.8v2.5h-4V2.8H3.7c-1.4%200-2.5%201.1-2.5%202.5v14.9c0%201.4%201.1%202.5%202.5%202.5h15.6c1.4%200%202.5-1.1%202.5-2.5V5.3c0-1.4-1.1-2.5-2.5-2.5zm0%2017.4H3.7v-9.9h15.6v9.9zM7.7.3H5.9v4.2h1.7V.3zm9.4%200h-1.7v4.2h1.7V.3z%22%20fill%3D%22#ffffff%22%2F%3E%3Cg%20fill%3D%22#ffffff%22%3E%3Cpath%20d%3D%22M5.1%2011.7h1.4v1.4H5.1zM5.1%2014.5h1.4v1.4H5.1zM5.1%2017.4h1.4v1.4H5.1zM8%2011.7h1.4v1.4H8zM8%2014.5h1.4v1.4H8zM8%2017.4h1.4v1.4H8zM10.8%2011.7h1.4v1.4h-1.4zM10.8%2014.5h1.4v1.4h-1.4zM10.8%2017.4h1.4v1.4h-1.4zM13.6%2011.7H15v1.4h-1.4zM13.6%2014.5H15v1.4h-1.4zM13.6%2017.4H15v1.4h-1.4zM16.5%2011.7h1.4v1.4h-1.4zM16.5%2014.5h1.4v1.4h-1.4zM16.5%2017.4h1.4v1.4h-1.4z%22%2F%3E%3C%2Fg%3E%3C%2Fsvg%3E");\n    background-repeat: no-repeat;\n  }\n  .site-icon-credit {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2271%22%20height%3D%2243%22%20viewBox%3D%220%200%2071%2043%22%3E%3Cstyle%20type%3D%22text%2Fcss%22%3Ecircle%2C%20ellipse%2C%20line%2C%20path%2C%20polygon%2C%20polyline%2C%20rect%2C%20text%20%7B%20fill%3A%20#ffffff%20!important%3B%20%7D%3C%2Fstyle%3E%3Cpath%20d%3D%22M71%202c0-1.1-.9-2-2-2h-67c-1.1%200-2%20.9-2%202v8.6h71v-8.6zm-71%2018.4v20.6c0%201.1.9%202%202%202h67c1.1%200%202-.9%202-2v-20.6h-71zm17.7%2014.9h-9.6v-3.6h9.6v3.6zm17.3%200h-13.9v-3.6h13.9v3.6z%22%20fill%3D%22%23fff%22%2F%3E%3C%2Fsvg%3E");\n    background-repeat: no-repeat;\n  }\n  .site-icon-search {\n    background-image: url(\'data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2223%22%20height%3D%2223%22%20viewBox%3D%220%200%2023%2023%22%3E%3Cpath%20d%3D%22M21.8%2019.5L17.4%2015c-.6.9-1.5%201.7-2.4%202.4l4.5%204.4c1.4%201.6%203.8-.8%202.3-2.3z%22%20fill%3D%22%23bebebb%22%3E%3C%2Fpath%3E%3Cpath%20d%3D%22M15%2017.4c4.4-3%205.2-9.4%201.8-13.4C13.8.5%208.6-.3%204.7%202.3.9%204.7-.4%209.9%201.7%2014c2.2%204.4%207.7%206.2%2012.1%204.1.4-.2.8-.5%201.2-.7zm-5.2-1.7C6.7%2015.7%204%2013%204%209.8%204%206.7%206.7%204%209.8%204c3.2%200%205.8%202.7%205.8%205.8.1%203.2-2.6%205.9-5.8%205.9z%22%20fill%3D%22%23bebebb%22%3E%3C%2Fpath%3E%3Cpath%20d%3D%22M9.8%202.2c4.2%200%207.7%203.4%207.7%207.7%200%201.5-.5%203.1-1.4%204.3-.5.7-1%201.3-1.7%201.8l-.3.2c-1.2.9-2.8%201.3-4.3%201.3-4.2%200-7.7-3.4-7.7-7.7.1-4.2%203.5-7.6%207.7-7.6m0-1.5C4.7.7.6%204.8.6%209.9s4.1%209.2%209.2%209.2c1.8%200%203.7-.6%205.2-1.6.9-.6%201.7-1.4%202.3-2.4%201.1-1.5%201.6-3.3%201.6-5.2.1-5.1-4-9.2-9.1-9.2z%22%20fill%3D%22%23bebebb%22%3E%3C%2Fpath%3E%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-is-searching {\n     background-position: center center !important;\n     background-image: url(\'data:image/svg+xml;charset=utf-8,%3Csvg%20version%3D%221.1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%20%20viewBox%3D%220%200%2040%2040%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cpath%20opacity%3D%220.2%22%20fill%3D%22%23323232%22%20d%3D%22M20.201%2C5.169c-8.254%2C0-14.946%2C6.692-14.946%2C14.946c0%2C8.255%2C6.692%2C14.946%2C14.946%2C14.946%0D%0A%20%20s14.946-6.691%2C14.946-14.946C35.146%2C11.861%2C28.455%2C5.169%2C20.201%2C5.169z%20M20.201%2C31.749c-6.425%2C0-11.634-5.208-11.634-11.634%0D%0A%20%20c0-6.425%2C5.209-11.634%2C11.634-11.634c6.425%2C0%2C11.633%2C5.209%2C11.633%2C11.634C31.834%2C26.541%2C26.626%2C31.749%2C20.201%2C31.749z%22%2F%3E%0D%0A%3Cpath%20fill%3D%22%23323232%22%20d%3D%22M26.013%2C10.047l1.654-2.866c-2.198-1.272-4.743-2.012-7.466-2.012h0v3.312h0%0D%0A%20%20C22.32%2C8.481%2C24.301%2C9.057%2C26.013%2C10.047z%22%3E%0D%0A%20%20%3CanimateTransform%20attributeType%3D%22xml%22%0D%0A%20%20%20%20attributeName%3D%22transform%22%0D%0A%20%20%20%20type%3D%22rotate%22%0D%0A%20%20%20%20from%3D%220%2020%2020%22%0D%0A%20%20%20%20to%3D%22360%2020%2020%22%0D%0A%20%20%20%20dur%3D%220.5s%22%0D%0A%20%20%20%20repeatCount%3D%22indefinite%22%2F%3E%0D%0A%20%20%3C%2Fpath%3E%0D%0A%3C%2Fsvg%3E\');\n     background-repeat: no-repeat;\n     background-size: contain;\n     z-index: 999;\n     width: 80px;\n     height: 80px;\n     margin: 0 auto;\n  }\n  .site-icon-report {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3Csvg%20viewBox%3D%220%200%201792%202031%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%3E%3Cpath%20d%3D%22M1122.987%20238.933L1075.2%200H0v2030.933h238.933v-836.266h669.014l47.786%20238.933H1792V238.933z%22%20fill%3D%22%23bebebb%22%20fill-rule%3D%22evenodd%22%2F%3E%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n    background-size: contain;\n  }\n  .site-icon-block {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3Csvg%20viewBox%3D%220%200%201792%201792%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%3E%3Cpath%20d%3D%22M896%201612.8c-165.76%200-318.08-56.448-439.04-151.424L1461.376%20456.96A708.02%20708.02%200%200%201%201612.8%20896c0%20396.032-320.768%20716.8-716.8%20716.8M179.2%20896c0-396.032%20320.768-716.8%20716.8-716.8%20165.76%200%20318.08%2056.448%20439.04%20151.424L330.624%201335.04A708.02%20708.02%200%200%201%20179.2%20896M896%200C401.408%200%200%20401.408%200%20896s401.408%20896%20896%20896%20896-401.408%20896-896S1390.592%200%20896%200%22%20fill%3D%22%23ffffff%22%20fill-rule%3D%22evenodd%22%20fill-opacity%3D%22.6%22%2F%3E%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n    background-size: contain;\n  }\n</style>\n\n\n  <style>\n\n    /*\r\n * This file is for use on the Vimeo VHX platform.\r\n *\r\n * 1. prototype changes in browser dev tools using Vimeo VHX demo site\r\n * 2. edit this file locally\r\n * 3. git commit and push to the remote repo\r\n * 4. share for code review\r\n * 5. paste into "CUSTOM STYLES" field in the Vimeo VHX theme demo admin\r\n * 6. share for approval by Criterion team\r\n * 7. once approved and code reviewed, paste into "CUSTOM STYLES" field in the Vimeo VHX theme production admin\r\n *\r\n * Note for Vimeo engineers: please share any updates to this file to the Criterion.com team.\r\n */\r\n\r\n @import url(\'https://fonts.googleapis.com/css?family=Montserrat:400,500,600,700,900|Spectral:400,500,600,700\');\r\n\r\n p, div, span, a, h1, h2, h3, h4, h5, h6, .site-font-primary-family, .site-font-secondary-family {\r\n     font-family: \'Montserrat\', sans-serif !important;\r\n }\r\n\r\n div {\r\n     font-family: \'Montserrat\', sans-serif !important;\r\n }\r\n\r\n .browse-row h1 a {\r\n     font-family: \'Montserrat\', sans-serif !important;\r\n     font-weight: 700;\r\n     text-transform: uppercase;\r\n }\r\n\r\n .head-1, .head-2 {\r\n     font-weight: 700;\r\n }\r\n\r\n .browse-row h1.horizontal-row-header a {\r\n     font-size: 1rem !important;\r\n }\r\n\r\n .browse-link {\r\n     font-family: \'Montserrat\', sans-serif !important;\r\n }\r\n\r\n /*\r\n * Nav\r\n */\r\n\r\n .navigation-inner a {\r\n    opacity: 1;\r\n    font-family: \'Montserrat\', sans-serif !important;\r\n    -webkit-transition: all 0.3s ease;\r\n    -moz-transition: all 0.3s ease;\r\n    -ms-transition: all 0.3s ease;\r\n    -o-transition: all 0.3s ease;\r\n    transition: all 0.3s ease;\r\n}\r\n\r\n .navigation-inner.is-ready a {\r\n    opacity: 1;\r\n}\r\n\r\n  /*\r\n  * Other\r\n  */\r\n\r\n h1.gigantic.uppercase { text-transform: none !important; }\r\n .percentage-bar-container { display: none !important; }\r\n\r\n .slide-content {\r\n     bottom: 40px;\r\n }\r\n\r\n .slide-title {\r\n     font-size: 28px;\r\n     font-weight: 600;\r\n }\r\n\r\n .browse-item-title .subtext-container > div.site-font-primary-color.truncate {\r\n     white-space: normal;\r\n }\r\n\r\n .collection-description .read-more-wrap {\r\n     max-height: 120px !important;\r\n }\r\n\r\n .site-header nav a {\r\n     font-weight: 600;\r\n     letter-spacing: 0;\r\n     padding-left: 25px;\r\n }\r\n\r\n div.slide-subtitle {\r\n     display: none;\r\n }\r\n\r\n .slide-content {\r\n     bottom: 40px;\r\n }\r\n\r\n .collection-title {\r\n     font-weight: 600;\r\n }\r\n\r\n\r\n #watch-info h5 {\r\n     display: none;\r\n }\r\n\r\n .head.primary strong {\r\n     font-weight: 600;\r\n }\r\n\r\n .item-type-movie .play-outer-rectangle {\r\n     display: none;\r\n }\r\n\r\n /*\r\n  * Borders\r\n  */\r\n\r\n .border-bottom,\r\n .border-bottom-light {\r\n     border-bottom: none !important;\r\n }\r\n\r\n .border-top,\r\n .border-top-light {\r\n     border-top: none !important;\r\n }\r\n\r\n .horizontal-row {\r\n     border-top: none !important;\r\n     border-bottom: none !important;\r\n }\r\n\r\n /*\r\n  * Buttons\r\n  */\r\n\r\n .button {\r\n     border-radius: 0px;\r\n     border: none;\r\n     font-weight: 600;\r\n }\r\n\r\n .slide-button,\r\n .slide-button--alt {\r\n     border-radius: 0px !important;\r\n }\r\n\r\n .btn-site-primary,\r\n .btn-gray:link,\r\n .btn-dropdown-gray:link,\r\n .btn-gray:active,\r\n .btn-dropdown-gray:active,\r\n .btn-gray:visited,\r\n .btn-dropdown-gray:visited {\r\n     border: 1px solid #fff !important;\r\n     background-color: #fff !important;\r\n     color: #252525;\r\n     border-radius: 0px;\r\n     text-transform: uppercase;\r\n     line-height: 46px;\r\n     font-size: 13px;\r\n     transition: all 300ms ease-out;\r\n }\r\n .site-settings-billing-panel .btn-gray:link {\r\n   color: #252525 !important;\r\n   line-height: 33px;\r\n }\r\n .btn-site-primary,\r\n .btn-site-secondary {\r\n     line-height: 33px;\r\n }\r\n\r\n .btn-site-secondary,\r\n .btn-transparent,\r\n .btn-dropdown-transparent {\r\n     border: 1px solid #313131 !important;\r\n     background-color: #313131 !important;\r\n     color: #fff !important;\r\n     transition: all 300ms ease-out;\r\n     font-size: 13px;\r\n }\r\n\r\n .btn-gray:hover,\r\n .btn-dropdown-gray:hover,\r\n .btn-site-primary:hover,\r\n .btn-dropdown-transparent:hover,\r\n .btn-transparent:hover,\r\n .btn-site-secondary:hover {\r\n     border: 1px solid #CDCDCB !important;\r\n     background-color: #CDCDCB !important;\r\n }\r\n\r\n .icon-play-white {\r\n     background-image: url("data:image/svg+xml;charset=utf-8,%3Csvg xmlns=\'http://www.w3.org/2000/svg\' width=\'23\' height=\'23\' viewBox=\'0 0 23 23\'%3E%3Ctitle%3Eicons-ui-set copy%3C/title%3E%3Cpath d=\'M20.49 10.52L4.13.39C2.94-.4 2 .18 2 1.67v19.66c0 1.48 1 2.06 2.17 1.28l16.32-10.13a1.15 1.15 0 0 0 0-1.96z\' fill=\'%23252525\'/%3E%3C/svg%3E");\r\n }\r\n\r\n .btn-black:link,\r\n .btn-dropdown-black:link,\r\n .btn-black:active,\r\n .btn-dropdown-black:active,\r\n .btn-black:visited,\r\n .btn-dropdown-black:visited {\r\n     border: 1px solid #B9BCC6;\r\n     background-color: #B9BCC6;\r\n     color: #000;\r\n     border-radius: 0px;\r\n     text-transform: uppercase;\r\n     line-height: 46px;\r\n     font-size: 13px;\r\n }\r\n\r\n .btn-transparent,\r\n .btn-dropdown-transparent,\r\n .btn-transparent:link,\r\n .btn-dropdown-transparent:link,\r\n .btn-transparent:active,\r\n .btn-dropdown-transparent:active,\r\n .btn-transparent:visited,\r\n .btn-dropdown-transparent:visited {\r\n     border: none;\r\n     border-radius: 0px;\r\n     text-transform: uppercase;\r\n     line-height: 28px;\r\n     font-size: 13px;\r\n     padding: 10px 16px;\r\n     background-position: 16px 18px !important;\r\n }\r\n\r\n select.btn-dropdown-transparent {\r\n     padding: 0 20px 0 40px;\r\n }\r\n\r\n .coin {\r\n     border-radius: 4px;\r\n }\r\n\r\n /*\r\n  * Search link + collection card\r\n  */\r\n\r\n .browse-item-subtext {\r\n     display: none !important;\r\n }\r\n\r\n /*\r\n  * Search results\r\n  */\r\n\r\n .item-type-movie.js-collection-item .duration-container {\r\n     display: none !important;\r\n }\r\n\r\n /*\r\n  * Carousel\r\n  */\r\n\r\n .slide-content {\r\n     transform: translate3d(0,0,0);\r\n     top: auto;\r\n     bottom: 10%;\r\n     padding-left: 57px;\r\n }\r\n\r\n .slide-image-container:after,\r\n .slide-image-container:before {\r\n     display: none !important;\r\n }\r\n\r\n @media screen and (max-width: 39.9375em) {\r\n     .slide-content {\r\n         padding-left: 40px;\r\n         bottom: 13%;\r\n     }\r\n     .slide-bg--mobile .slide-bg-img {\r\n         float: left;\r\n         width: 100%;\r\n         height: auto;\r\n     }\r\n\r\n }\r\n\r\n /*\r\n  * Browse\r\n  */\r\n\r\n .browse-item-card.item-type-movie .duration-container {\r\n   display: none !important;\r\n }\r\n\r\n /*\r\n  * Collection Pages\r\n  */\r\n\r\n .vertical-bottom {\r\n     position: absolute;\r\n     bottom: 0;\r\n }\r\n\r\n .cover-art.columns {\r\n     padding-right: 0px !important;\r\n }\r\n\r\n select.js-sort-items,\r\n h2.collection-stats,\r\n h2.content-label,\r\n .extras-header {\r\n     display: none !important;\r\n }\r\n\r\n .browse-item-title .browse-image-container:before {\r\n     width: 102%;\r\n     height: 102%;\r\n }\r\n\r\n .browse-image-container:before {\r\n     background-image: none !important;\r\n }\r\n\r\n .media-identifier.media-episode {\r\n     display: none;\r\n }\r\n\r\n .read-more-trigger {\r\n     font-weight: bold;\r\n }\r\n\r\nsection.site-settings-purchases-panel .subscription-button {\r\n    background-color: #222 !important\r\n}\r\n\r\n.platforms-modal--container .platform-modal-devices-list-tv li:last-child {\r\n    display: none;\r\n}\r\n\r\n .collection-actions .margin-bottom-small {\r\n    margin-bottom: 0.9375rem !important;\r\n }\r\n .collection-actions .margin-right-small {\r\n    margin-right: 0.9375rem !important;\r\n }\r\n .collection-actions .btn-site-primary {\r\n    line-height: 46px;\r\n }\r\n\r\n/* Fix for button issue on pause / cancellation modal */\r\nbody.settings.is-modal-active .btn-site-secondary,\r\nbody.settings.is-modal-active .btn-site-primary {\r\n    line-height: 38px !important;\r\n    color: #252525 !important;\r\n}\r\n\r\n.site-header .user-dropdown {\r\n    width: 190px;\r\n}\r\n\r\n.tooltip.background-white span.media-count, .tooltip.background-white span.media-identifier, .tooltip.background-white span.media-seperator { display: none !important; }\r\n\r\n.tooltip.background-white .tooltip-item-title { margin-bottom: 15px; }\r\n\r\ndiv[id^="collection-tooltip"] .tooltip-item-title { margin-bottom: 0px !important; }\n\n\n    html.wf-loading { opacity: 0; }\n    html.wf-active, html.is-loaded { opacity: 1 !important; }\n  </style>\n\n  <script>\n  var Tracking = undefined;\n  window.Page = undefined;\n  var Segment_ID = undefined;\n  window._current_user = undefined;\n  window._current_site = undefined;\n</script>\n\n\n  <script>\n    window._vhx = {\n      environment: \'production\',\n      site: { id: \'59054\', domain: \'www.criterionchannel.com\', host: \'criterionchannel.vhx.tv\', previewUrl: \'https://www.criterionchannel.com/login\' },\n      geo:  { country: \'us\' },\n      product: {id: \'\' }\n    };\n\n    window._current_user = {"id":null,"email":null,"name":null,"city":null,"state":null,"country":null,"external_user_id":null};\n\n    window._current_site = {"id":59054,"facebook_pixel_codes":[{"pixel_code":"980569266079632","product_id":null}],"twitter_pixel_codes":[],"adwords_pixel_codes":[],"key":"criterionchannelchartersu","title":"The Criterion Channel","subdomain":"criterionchannel","made_for_kids":false,"made_for_kids_tracking_blocked":false,"algolia_search_enabled":true,"passwords_disabled":false,"segment_heartbeat_frequency":0,"segment_includes_user_id":false,"sso_enabled":false};\n  </script>\n\n<script>\n  window.Tracking = {\n    EVENTS:     {"EVENT_SALE_INITIATED":"sale_initiated","EVENT_SALE_COMPLETE":"sale_complete","EVENT_SALE_COMPLETE_GIFT":"sale_conversion-gift","EVENT_SALE_ERROR":"sale_error","EVENT_SALE_COMPLETE_SAVE_CARD":"sale_complete_save_card","EVENT_SALE_MISSING_EMAIL":"sale_missing_email","EVENT_SALE_UNCONFIRMED_EMAIL":"sale_unconfirmed_email","EVENT_SALE_MISSING_CC":"sale_missing_cc","EVENT_SALE_MISSING_CVC":"sale_missing_cvc","EVENT_SALE_MISSING_EXP":"sale_missing_exp","EVENT_SALE_MISSING_ZIP":"sale_missing_zip","EVENT_SALE_INVALID_CC":"sale_invalid_cc","EVENT_SALE_INVALID_CVC":"sale_invalid_cvc","EVENT_SALE_INVALID_EXP":"sale_invalid_exp","EVENT_SALE_INVALID_ZIP":"sale_invalid_zip","EVENT_SALE_INVALID_COUPON":"sale_invalid_coupon","EVENT_SALE_INACTIVE_COUPON":"sale_inactive_coupon","EVENT_SALE_GEOBLOCKED_COUPON":"sale_geoblocked_coupon","EVENT_SALE_EXPIRED_CC":"sale_expired_cc","EVENT_SALE_PLAY_TRAILER":"sale_play_trailer","EVENT_SALE_PROCESSING_ERROR":"sale_processing_error","EVENT_SALE_UNKNOWN_ERROR":"sale_unknown_error","EVENT_SALE_INCOMPLETE_FORM":"sale_incomplete_form","EVENT_SALE_SHARE_FACEBOOK":"sale_share_facebook","EVENT_SALE_SHARE_TWITTER":"sale_share_twitter","EVENT_SALE_SHARE_EMAIL":"sale_share_email","EVENT_SALE_START_WATCHING":"sale_start_watching","EVENT_SALE_SAVE_CARD":"sale_save_card","EVENT_SALE_SHOW_PAYMENT_TAB":"sale_show_payment_tab","EVENT_SALE_SHOW_DETAILS_TAB":"sale_show_details_tab","EVENT_ADMIN_SIGNUP":"admin_signup","EVENT_ADMIN_ADD_SITE":"admin_add_site","EVENT_ADMIN_PUBLISH_THEME":"admin_publish_theme","EVENT_ADMIN_CREATE_VIDEO":"admin_create_video","EVENT_ADMIN_UPLOAD_VIDEO":"admin_upload_video","EVENT_ADMIN_UPLOAD_VIDEO_NATIVE":"admin_upload_video_native","EVENT_ADMIN_UPLOAD_VIDEO_DROPBOX":"admin_upload_video_dropbox","EVENT_ADMIN_TRANSCODE_VIDEO":"admin_transcode_video","EVENT_ADMIN_CREATE_EXTRA":"admin_create_extra","EVENT_ADMIN_UPLOAD_EXTRA":"admin_upload_extra","EVENT_ADMIN_UPLOAD_EXTRA_NATIVE":"admin_upload_video_native","EVENT_ADMIN_CONNECT_PAYOUT":"admin_connect_payout","EVENT_ADMIN_CREATE_SCREENER":"admin_create_screener","EVENT_ADMIN_CREATE_COUPON":"admin_create_coupon","EVENT_ADMIN_CREATE_UPDATE":"admin_create_update","EVENT_ADMIN_PUBLISH_UPDATE":"admin_publish_update","EVENT_ADMIN_CREATE_PARTNER":"admin_create_partner","EVENT_ADMIN_UPLOADED_FOLLOWERS_CS":"admin_uploaded_followers_csv","EVENT_ADMIN_ADDED_FOLLOWERS":"admin_added_followers","EVENT_ADMIN_CREATE_SUBSCRIPTION":"admin_create_subscription","EVENT_ADMIN_LINK_HELP":"admin_link_help","EVENT_ADMIN_LINK_FAQ":"admin_link_faq","EVENT_ADMIN_LINK_GETTING_STARTED":"admin_link_getting_started","EVENT_ADMIN_LINK_HOWTO":"admin_link_howto","EVENT_ADMIN_LINK_BEST_PRACTICES":"admin_link_best_practices","EVENT_ADMIN_LINK_DEV_DOCS":"admin_link_dev_docs","EVENT_FORUM_CREATE_THREAD":"forum_create_thread","EVENT_FORUM_CREATE_POST":"forum_create_post","EVENT_UPDATE_COMMENT":"comment_update","EVENT_CONCURRENT_LIMIT":"concurrent_limit","EVENT_VIDEO_COMMENT":"video_comment","EVENT_SITE_HERO_WATCH_TRAILER":"site_hero_watch_trailer","EVENT_SITE_HERO_BUY":"site_hero_buy","EVENT_SITE_FOOTER_BUY":"site_footer_buy","EVENT_SITE_FOOTER_IOS":"site_footer_ios","EVENT_SITE_BUY":"site_buy","EVENT_SITE_UNAVAILABLE_REQUEST":"site_unavailable_request","EVENT_SITE_SEARCH":"site_search","EVENT_SITE_SEARCH_EMPTY":"site_search_empty","EVENT_SITE_SEARCH_VIDEO_CLICK":"search_video","EVENT_SITE_SEARCH_COLLECTION_CLICK":"search_collection","EVENT_SITE_SEARCH_PRODUCT_CLICK":"search_product","EVENT_SITE_SEARCH_ITEM_SELECTION":"site_search_item_selected","EVENT_SITE_VIDEO_BUY":"site_video_buy","EVENT_SITE_VIDEO_WATCH_TRAILER":"site_video_watch_trailer","EVENT_SITE_COLLECTION_BUY":"site_collection_buy","EVENT_SITE_PRODUCT_PREORDER":"site_product_preorder","EVENT_SITE_PRODUCT_BUY":"site_product_buy","EVENT_SITE_PRODUCT_RENT":"site_product_rent","EVENT_SITE_CAROUSEL_NEXT":"site_carousel_next","EVENT_SITE_CAROUSEL_PREV":"site_carousel_prev","EVENT_SITE_CAROUSEL_COIN":"site_carousel_coin","EVENT_SITE_CAROUSEL_WATCH_NOW":"site_carousel_watch_now","EVENT_SITE_CAROUSEL_TRAILER":"site_carousel_trailer","EVENT_SITE_HOME":"site_home","EVENT_SITE_BROWSE":"site_browse","EVENT_SITE_SETTINGS":"site_settings","EVENT_SITE_FORUMS":"site_forums","EVENT_SITE_SUPPORT":"site_help","EVENT_SITE_CONTACT":"site_contact","EVENT_SITE_CONTACT_HELP":"site_contact_help","EVENT_SITE_VIEW_PRODUCTS":"site_products","EVENT_SITE_VIEW_PRODUCT":"site_product","EVENT_SITE_COLLECTION_CLICK":"site_collection","EVENT_SITE_VIDEO_CLICK":"site_video","EVENT_SITE_EXTRA_CLICK":"site_extras","EVENT_SITE_CATEGORY_CLICK":"site_category","EVENT_SITE_MOVIE_CLICK":"site_movie","EVENT_SITE_PLAYLIST_CLICK":"site_playlist","EVENT_SITE_SERIES_CLICK":"site_series","EVENT_SITE_SEASON_CLICK":"site_season","EVENT_SITE_TERMS":"site_tos","EVENT_SITE_PRIVACY":"site_privacy","EVENT_SITE_COOKIES":"site_cookies","EVENT_SITE_COPYRIGHT":"site_copyright","EVENT_SITE_DATA_PROCESSING":"site_data_processing","EVENT_SITE_WATCHLIST_REMOVE":"site_removewatchlist","EVENT_SITE_WATCHLIST_ADD":"site_addwatchlist","EVENT_SITE_PAGE_VIEW":"site_page_view","EVENT_SITE_CONTEXT_HOMEPAGE":"site_context_homepage","EVENT_SITE_CONTEXT_COLLECTION":"site_context_collection","EVENT_SITE_CONTEXT_CATEGORY":"site_context_category","EVENT_SITE_CONTEXT_MOVIE":"site_context_movie","EVENT_SITE_CONTEXT_PLAYLIST":"site_context_playlist","EVENT_SITE_CONTEXT_SERIES":"site_context_series","EVENT_SITE_CONTEXT_UPNEXT":"site_context_upnext","EVENT_AUTHENTICATION_LOGIN":"authentication_login","EVENT_AUTHENTICATION_ACTIVATION_STARTED":"activationstarted","EVENT_AUTHENTICATION_ACTIVATION_COMPLETE":"activation","EVENT_AUTHENTICATION_SIGNIN_VIEW":"signin","EVENT_AUTHENTICATION_SIGNIN_STARTED":"signinstarted","EVENT_AUTHENTICATION_SIGNIN_COMPLETE":"authentication","EVENT_AUTHENTICATION_SIGNUP_VIEW":"signup","EVENT_AUTHENTICATION_SIGNUP_STARTED":"signupstarted","EVENT_AUTHENTICATION_PURCHASE_STARTED":"purchasestarted","EVENT_AUTHENTICATION_PURCHASE_COMPLETED":"purchasecompleted","EVENT_AUTHENTICATION_CONVERSION":"conversion","EVENT_AUTHENTICATION_REGISTRATION":"register","EVENT_AUTHENTICATION_IDENTIFY_USER":"identify_user","EVENT_SETTINGS_SAVE_PROFILE":"settings_save_profile","EVENT_SETTINGS_SAVE_NOTIFICATIONS":"settings_save_notifications","EVENT_SETTINGS_SAVE_CARD":"settings_save_card","EVENT_DEVICES_DOWNLOAD_IOS":"devices_download_ios","EVENT_DEVICES_DOWNLOAD_APPLETV":"devices_download_appletv","EVENT_DEVICES_DOWNLOAD_ANDROID":"devices_download_android","EVENT_DEVICES_DOWNLOAD_ANDROIDTV":"devices_download_androidtv","EVENT_DEVICES_DOWNLOAD_AMAZON_FIRE":"devices_download_amazon_fire","EVENT_DEVICES_DOWNLOAD_XBOX":"devices_download_xbox","EVENT_DEVICES_DOWNLOAD_ROKU":"devices_download_roku","EVENT_DEVICES_DOWNLOAD_TIZEN":"devices_download_tizen","EVENT_DEVICES_DOWNLOAD_CHROMECAST":"devices_download_chromecast","EVENT_NEWSLETTER_SIGNUP":"event_newsletter_signup","EVENT_METADATA_ADVISORY":"metadata_click_advisory","EVENT_METADATA_CAST":"metadata_click_cast","EVENT_METADATA_CREW":"metadata_click_crew","EVENT_METADATA_GENRES":"metadata_click_genres","EVENT_METADATA_RATING":"metadata_click_rating","EVENT_METADATA_TAGS":"metadata_click_tags","EVENT_SEGMENT_AUTHENTICATION_CONVERSION":"Order Completed","EVENT_SEGMENT_AUTHENTICATION_PURCHASE_STARTED":"Checkout Started","EVENT_SEGMENT_AUTHENTICATION_REGISTRATION":"Registration Completed","EVENT_SEGMENT_AUTHENTICATION_SIGNIN_COMPLETE":"Sign In Complete","EVENT_SEGMENT_AUTHENTICATION_SIGNIN_VIEW":"Sign In Viewed","EVENT_SEGMENT_AUTHENTICATION_SIGNUP_STARTED":"Signed Up","EVENT_SEGMENT_AUTHENTICATION_SIGNUP_VIEW":"Sign Up Viewed","EVENT_SEGMENT_EXTRA_VIEWED":"Extra Content Viewed","EVENT_SEGMENT_SEARCH_EXECUTED":"Search Executed","EVENT_SEGMENT_SEARCH_ITEM_SELECTED":"Search Result Selected","EVENT_SEGMENT_VIDEO_AD_START":"Video Ad Started","EVENT_SEGMENT_VIDEO_AD_CLICKED":"Video Ad Clicked","EVENT_SEGMENT_VIDEO_AD_FINISH":"Video Ad Completed","EVENT_SEGMENT_VIDEO_AD_FAIL":"Video Ad Failed","EVENT_SEGMENT_VIDEO_AD_SKIP":"Video Ad Skipped","EVENT_SEGMENT_VIDEO_COMMENT_ADDED":"Video Comment Added","EVENT_SEGMENT_VIDEO_COMPLETE_PROGRESS_PERCENTILE":"Video Content Completed","EVENT_SEGMENT_VIDEO_ENDED":"Video Content Ended","EVENT_SEGMENT_VIDEO_END_BUFFER":"Video Playback Buffer Completed","EVENT_SEGMENT_VIDEO_HALF_PROGRESS_PERCENTILE":"50% Content Completed","EVENT_SEGMENT_VIDEO_PAUSE":"Video Playback Paused","EVENT_SEGMENT_VIDEO_PLAY":"Video Playback Started","EVENT_SEGMENT_VIDEO_QUARTER_PROGRESS_PERCENTILE":"25% Content Completed","EVENT_SEGMENT_VIDEO_RESUME":"Video Playback Resumed","EVENT_SEGMENT_VIDEO_SEEKED":"Video Playback Seek Completed","EVENT_SEGMENT_VIDEO_SEEKING":"Video Playback Seek Started","EVENT_SEGMENT_VIDEO_START_BUFFER":"Video Playback Buffer Started","EVENT_SEGMENT_VIDEO_TIMEUPDATE":"Video Content Playing","EVENT_SEGMENT_VIDEO_THIRD_PROGRESS_PERCENTILE":"75% Content Completed","EVENT_SEGMENT_VIDEO_WATCHLIST_ADD":"Added to Watch List","EVENT_SEGMENT_VIDEO_WATCHLIST_REMOVE":"Removed from Watch List","EVENT_SEARCH":"search","EVENT_SEARCH_ITEM_SELECTED":"searchitemselected","EVENT_EXTRA_VIEWED":"extraviewed"},\n    PROPERTIES: {"PROPERTY_COLLECTOR_URL":"https://collector.vhx.tv/pixel.gif","PROPERTY_COLLECTION_ID":"collection_id","PROPERTY_DEVICE":"device","PROPERTY_DEVICE_ID":"device_id","PROPERTY_NAME":"name","PROPERTY_PLATFORM":"platform","PROPERTY_PLATFORM_ID":"platform_id","PROPERTY_PLATFORM_VERSION":"platform_version","PROPERTY_PRODUCT_ID":"product_id","PROPERTY_REFERRER":"referrer","PROPERTY_SESSION_ID":"session_id","PROPERTY_SITE_ID":"site_id","PROPERTY_TIMESTAMP":"timestamp","PROPERTY_TYPE":"type","PROPERTY_URL":"url","PROPERTY_USER_AGENT":"user_agent","PROPERTY_USER_EMAIL":"user_email","PROPERTY_USER_ID":"user_id","PROPERTY_VIDEO_ID":"video_id","PROPERTY_VIEW":"view"},\n    DEPRECATED_EVENTS: ["admin_connect_payout","admin_create_coupon","admin_create_screener","admin_link_dev_docs","admin_link_faq","admin_link_help","admin_publish_theme","metadata_click_cast","metadata_click_crew","metadata_click_genres","metadata_click_rating","metadata_click_tags","metadata_click_advisory"],\n    UPDATED_EVENTS: {"search_video":"searchitemselected","search_collection":"searchitemselected","site_search_item_selected":"searchitemselected","site_extras":"extraviewed"},\n    SITE_GANALYTICS_ENABLED: false,\n    SESSION: \'51b8cf1564f3684336e4100926180e2e\'\n  };\n</script>\n  <script>\n     window.Page = {"PROPERTIES":{"VIEW_TYPE":"collection","PRODUCT_ID":39621,"COLLECTION_ID":95691,"COLLECTION_TITLE":"The Organizer"}}\n  </script>\n\n\n\n  <script>\n  var _gaq = _gaq || [];\n\n  (function(i,s,o,g,r,a,m){i[\'GoogleAnalyticsObject\']=r;i[r]=i[r]||function(){\n  (i[r].q=i[r].q||[]).push(arguments)},i[r].l=1*new Date();a=s.createElement(o),\n  m=s.getElementsByTagName(o)[0];a.async=1;a.src=g;m.parentNode.insertBefore(a,m)\n  })(window,document,\'script\',\'//www.google-analytics.com/analytics.js\',\'ga\');\n\n      ga(\'create\', \'UA-19540423-6\', \'auto\');\n    ga(\'require\', \'displayfeatures\');\n\n      ga(\'set\', \'dimension1\', \'criterionchannelchartersu\');\n\n    ga(\'send\', \'pageview\');\n\n\n      (function() {\n        var vhx = document.createElement(\'script\'); vhx.type = \'text/javascript\'; vhx.async = true;\n        vhx.src = \'https://cdn.vhx.tv/assets/tracker-e98d1630e0a189cf4d65d6c65df70445332c77633d19674d44390116fe52123a.js\';\n        var s = document.getElementsByTagName(\'script\')[0]; s.parentNode.insertBefore(vhx, s);\n      })();\n\n\n    Tracking.SITE_GANALYTICS_ENABLED = true;\n\n      ga(\'create\', \'UA-1852531-11\', \'auto\', { \'name\': \'site\', \'allowLinker\': true});\n\n    ga(\'site.require\', \'displayfeatures\');\n      ga(\'site.require\', \'linker\');\n      ga(\'site.linker:autoLink\', [\'www.criterionchannel.com\',\'criterionchannel.vhx.tv\'] );\n    ga(\'site.set\', \'dimension1\', \'criterionchannelchartersu\');\n    ga(\'site.send\', \'pageview\');\n\n    // <!-- Google Tag Manager -->\n    (function(w,d,s,l,i){w[l]=w[l]||[];w[l].push({\'gtm.start\':\n    new Date().getTime(),event:\'gtm.js\'});var f=d.getElementsByTagName(s)[0],\n    j=d.createElement(s),dl=l!=\'dataLayer\'?\'&l=\'+l:\'\';j.async=true;j.src=\n    \'https://www.googletagmanager.com/gtm.js?id=\'+i+dl;f.parentNode.insertBefore(j,f);\n    })(window,document,\'script\',\'dataLayer\',\'GTM-PKL3NF6\');\n    // <!-- End Google Tag Manager -->\n\n  if (window._current_site && window._current_site.facebook_pixel_codes.length > 0) {\n    // <!-- Facebook Pixel Code -->\n    !function(f,b,e,v,n,t,s)\n    {if(f.fbq)return;n=f.fbq=function(){n.callMethod?\n    n.callMethod.apply(n,arguments):n.queue.push(arguments)};\n    if(!f._fbq)f._fbq=n;n.push=n;n.loaded=!0;n.version=\'2.0\';\n    n.queue=[];t=b.createElement(e);t.async=!0;\n    t.src=v;s=b.getElementsByTagName(e)[0];\n    s.parentNode.insertBefore(t,s)}(window, document,\'script\',\n    \'https://connect.facebook.net/en_US/fbevents.js\');\n    //<!-- End Facebook Pixel Code -->\n  }\n\n  if (window._current_site && window._current_site.twitter_pixel_codes.length > 0) {\n    // <!-- Twitter Pixel Code -->\n    !function(e,t,n,s,u,a){e.twq||(s=e.twq=function(){s.exe?s.exe.apply(s,arguments):s.queue.push(arguments);\n    },s.version=\'1.1\',s.queue=[],u=t.createElement(n),u.async=!0,u.src=\'//static.ads-twitter.com/uwt.js\',\n    a=t.getElementsByTagName(n)[0],a.parentNode.insertBefore(u,a))}(window,document,\'script\');\n    //<!-- End Twitter Pixel Code -->\n  }\n\n      Segment_ID = \'D22V0IgjPYTwUKXVEtSzbmfcnolk4CNB\';\n      !function(){var analytics=window.analytics=window.analytics||[];if(!analytics.initialize)if(analytics.invoked)window.console&&console.error&&console.error("Segment snippet included twice.");else{analytics.invoked=!0;analytics.methods=["trackSubmit","trackClick","trackLink","trackForm","pageview","identify","reset","group","track","ready","alias","debug","page","once","off","on"];analytics.factory=function(t){return function(){var e=Array.prototype.slice.call(arguments);e.unshift(t);analytics.push(e);return analytics}};for(var t=0;t<analytics.methods.length;t++){var e=analytics.methods[t];analytics[e]=analytics.factory(e)}analytics.load=function(t,e){var n=document.createElement("script");n.type="text/javascript";n.async=!0;n.src="https://cdn.segment.com/analytics.js/v1/"+t+"/analytics.min.js";var a=document.getElementsByTagName("script")[0];a.parentNode.insertBefore(n,a);analytics._loadOptions=e};analytics.SNIPPET_VERSION="4.1.0";\n      analytics.load("D22V0IgjPYTwUKXVEtSzbmfcnolk4CNB");\n      }}();\n  </script>\n\n\n\n    \n\n  <script>\n  if (/ticket=/.test(location.search) && window.history.replaceState) {\n    var url = location.href.replace(/\\?([^#]*)/, function(_, search) {\n      search = search.split(\'&\').map(function(v) {\n        return !/^ticket=/.test(v) && !/^support=/.test(v) && !/^sso=/.test(v) && v;\n      }).filter(Boolean).join(\'&\');\n      return search ? \'?\' + search : \'\';\n    });\n\n    if (url != location.href) {\n      window.history.replaceState({}, \'\', url);\n    }\n  }\n\n  if (/return_to=/.test(location.search) && window.history.replaceState) {\n    var url = location.href.replace(/\\?([^#]*)/, function(_, search) {\n      search = search.split(\'&\').map(function(v) {\n        return !/^return_to=/.test(v) && v;\n      }).filter(Boolean).join(\'&\');\n      return search ? \'?\' + search : \'\';\n    });\n\n    if (url != location.href) {\n      window.history.replaceState({}, \'\', url);\n    }\n  }\n\n  if (/logout=/.test(location.search) && window.history.replaceState) {\n    var url = location.href.replace(/\\?([^#]*)/, function(_, search) {\n      search = search.split(\'&\').map(function(v) {\n        return !/^logout=/.test(v) && v;\n      }).filter(Boolean).join(\'&\');\n      return search ? \'?\' + search : \'\';\n    });\n\n    if (url != location.href) {\n      window.history.replaceState({}, \'\', url);\n    }\n  }\n</script>\n\n\n  <script>\n    window.TOKEN = "eyJhbGciOiJSUzI1NiIsImtpZCI6ImQ2YmZlZmMzNGIyNTdhYTE4Y2E2NDUzNDE2ZTlmZmRjNjk4MDAxMDdhZTQ2ZWJhODg0YTU2ZDBjOGQ4NTYzMzgifQ.eyJhcHBfaWQiOjY4NTExLCJleHAiOjE2NzgzOTgwMDcsIm5vbmNlIjoiZTU4M2VjZTVmNDRiNTg0NCIsInNjb3BlcyI6WyJwdWJsaWMiXSwic2Vzc2lvbl9pZCI6ImoyUVNNL3BZYU5BWHpaMFVMM1R3NkE9PSIsInNpdGVfaWQiOjU5MDU0fQ.tMPhDR9upjH9nJpOW6tPltFT1nvvsfBPrtuprqn-l6yI-ruqPwqofUVGSlLYJNmUi2bdiD6Kz3IQ2Oi5Sedznpte6OoDjecbZTaRp7CY59CbykqXrqRhO1UYtvJZYjGzmpWoKHE9dCE68WR6Lpu5L8J_XfG-FlWR9lTwmBy067RZJ57RscTa0xRanVFCHl7yVbFbYizEiexLVn-guHSIlenaKFXTLIdAB1J_DNLbwlwnRN9-aCU9EAKTvPwBxdxpLyP2aGqK8oXovW97EvmrDWIdg_doltCqrhIuljMgeWU2KRbtfoAjd0eMxnSvAZ7UCeTS5HECE0pU310HnmKLI7i0XRI_VaHwKBO_98qzSyPBBkuXNGBkwSxaZemocRCLZC7lzWni3Z59v5h3EgsPGhqR6gjb6K6yC14NxzymRYnRJ8Apc7Fx6_uCV4VDVt_W_XM2zkPBG6PP_ME-EpUDWH2IPcIdDrZ0p2sxQc8jDGC3AkG-6MzhRmgLoveY15W_5suR49HPy1WZcBZ_Mo1eDhQe-FFGWhZD2shRF8iJ8RovDv7U0G9sEvT6tIkMeta-1XWtcJx_ywwn-lsj3CMC07850pGCpvI_6oZCdNjDqmUxKmWojLiRJng98uRtTASX9m2UKveaA0l8g9Z0aySQDdLxfinVdDeqyTE31G2iErE";\n\n    (function() {\n      var config = { kitId: \'kyo4vqy\', scriptTimeout: 3000 };\n      var h=document.getElementsByTagName("html")[0];h.className+=" wf-loading";var t=setTimeout(function(){h.className=h.className.replace(/(\\s|^)wf-loading(\\s|$)/g," ");h.className+=" wf-inactive"},config.scriptTimeout);var tk=document.createElement("script"),d=false;tk.src=\'//use.typekit.net/\'+config.kitId+\'.js\';tk.type="text/javascript";tk.async="true";tk.onload=tk.onreadystatechange=function(){var a=this.readyState;if(d||a&&a!="complete"&&a!="loaded")return;d=true;clearTimeout(t);try{Typekit.load(config)}catch(b){}};var s=document.getElementsByTagName("script")[0];s.parentNode.insertBefore(tk,s);\n    })();\n  </script>\n\n  \n  \n  \n</head>\n\n<body class="site-background-color logged-out collections collections-show group-a site-color-is-light collection-the-organizer sites" data-site-color="#b9bcc7" data-track-event="site_movie">\n    <!-- Google Tag Manager (noscript) -->\n  <noscript><iframe src="https://www.googletagmanager.com/ns.html?id=GTM-PKL3NF6"\n  height="0" width="0" style="display:none;visibility:hidden"></iframe></noscript>\n  <!-- End Google Tag Manager (noscript) -->\n\n  \n\n  <a href="#maincontent" class="skipto">Skip to main content</a>\n    <div style="display: none !important;" rel="description" class="is-hidden">\n      Classics and discoveries from around the world, thematically programmed with special features, on a streaming service brought to you by the Criterion Collection.\n    </div>\n\n  <div data-flash="{&quot;auto_hide&quot;:true,&quot;style&quot;:&quot;overlay&quot;,&quot;container&quot;:null,&quot;type&quot;:&quot;notice&quot;,&quot;message&quot;:{&quot;body&quot;:null},&quot;show_on_load&quot;:false}" class="is-error is-hidden">\n    <div class="flash-icon">\n      <i class="icon icon-check-inverse icon-check-white"></i>\n      <i class="icon icon-alert-inverse icon-alert-white"></i>\n      <i class="icon icon-check-alt icon-check-blue"></i>\n      <i class="icon medium icon-check"></i>\n      <i class="icon medium icon-alert"></i>\n    </div>\n\n    <div class="flash-msg"></div>\n\n    <div class="flash-close">\n      <i class="icon icon-remove-inverse icon--xsmall"></i>\n      <i class="icon icon-remove-mono icon-x-black icon--xsmall"></i>\n    </div>\n</div>\n\n\n\n\n  <main class="main-container">\n      \n\n\n\n\n\n\n\n\n\n<header class="nav-background-color border-bottom site-border-color site-header has-logo ">\n  <div class="row full-width padding-top-medium padding-bottom-medium primary-row-container flex-container">\n    <div class="small-16 medium-11 columns identity-container flex-container has-image">\n      <span class="show-for-small-only flex-item padding-right-small show-menu-container">\n        <label for="show-menu" class="show-menu-trigger">\n          <span class="site-link-header-color inverse"><i class="icon site-icon-menu"></i></span>\n        </label>\n      </span>\n\n      <h1 class="identity site-font-primary-family head primary flex-item padding-horizontal-medium">\n            <a href="/">\n              <img class="logo" src="https://vhx.imgix.net/criterionchannel/assets/1c5933b9-19a1-4ea8-9761-726ca79faf44.png?fit=max&h=84&q=92&w=400" alt="The Criterion Channel" />\n</a>      </h1>\n\n      <nav class="navigation-container inline flex-item" role="navigation" aria-label="Primary">\n        <input type="checkbox" id="show-menu" name="show-menu" role="button">\n\n        <div class="navigation-inner site-header-background-color-small-only" id="primary-navigation">\n\n          <a href="/browse" class="browse-link site-font-primary-family head site-link-header-color">\n            Browse\n          </a>\n\n\n              <a href="#"\n                id="search-nav-toggle"\n                class="search-link site-font-primary-family head site-link-header-color">\n                Search\n              </a>\n\n                <a href="https://films.criterionchannel.com/" rel="noopener" class="custom-nav-all-films site-font-primary-family head site-link-header-color" target="_blank">ALL FILMS</a>\n                <a href="https://www.criterion.com/" rel="noopener" class="custom-nav-criterion-com site-font-primary-family head site-link-header-color" target="_blank">Criterion.com</a>\n\n            <a class="show-for-small-only flex-item site-link-header-color site-font-primary-family head subscribe-link" data-track-event-properties="{&quot;type&quot;:&quot;subscription&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/the-organizer&quot;}" data-track-event="site_buy" href="https://www.criterionchannel.com/checkout/subscribe">\n              <strong style="font-weight: bold;">\n                Start Free Trial\n              </strong>\n</a>\n\n              <a href="/login" class="show-for-small-only flex-item user-action-link head site-link-header-color site-font-primary-family">\n                Sign in\n              </a>\n\n        </div>\n      </nav>\n    </div>\n\n    <div class="hide-for-small-only text-right flex-right small-16 medium-5 columns small-only-text-center user-activity site-border-color">\n      <div class="user-actions has-image padding-right-medium flex-container" role="navigation" aria-label="Secondary">\n            <a class="flex-item site-link-header-color site-font-primary-family head subscribe-link user-action-link" data-track-event-properties="{&quot;type&quot;:&quot;subscription&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/the-organizer&quot;}" data-track-event="site_buy" href="https://www.criterionchannel.com/checkout/subscribe">\n              <strong style="font-weight: bold;">\n                Start Free Trial\n              </strong>\n</a>\n\n              <a href="/login" class="flex-item user-action-link head site-link-header-color  site-font-primary-family">\n                Sign In\n              </a>\n      </div>\n    </div>\n  </div>\n</header>\n\n  <div id="search-react-app"\n    data-props="{&quot;overlay&quot;:true,&quot;site_id&quot;:59054,&quot;api_v1_url&quot;:&quot;https://api.vhx.tv&quot;,&quot;api_v2_url&quot;:&quot;https://api.vhx.com&quot;,&quot;user_subscribed&quot;:false,&quot;purchased_product_ids&quot;:[],&quot;theme&quot;:&quot;dark&quot;,&quot;show_filters&quot;:false}">\n  </div>\n\n\n\n    \n\n\n\n<section class="site-background-color site-font-primary-color margin-bottom-large collection-movie top-container">\n  <div class="blurred-background-image"  style="background-image: url(\'https://vhx.imgix.net/criterionchannelchartersu/assets/1dd7e4a4-f23b-40fc-8821-e39055a9315e.jpg?blur=180\')">\n    <div class="gradient-blend" data-gradient-blend></div>\n  </div>\n  <section class="product-feature padding-horizontal-medium padding-top-large padding-bottom-xlarge border-bottom border-bottom-light">\n\n    <div class="row relative">\n      <div class="small-16 medium-8 large-8 collection-details-container columns end">\n\n        <div class="collection-details grid-padding-left">\n            <div class="show-for-small-only cover-art padding-bottom-medium">\n              <div class="embed">\n                <img src="https://vhx.imgix.net/criterionchannelchartersu/assets/1dd7e4a4-f23b-40fc-8821-e39055a9315e.jpg?auto=format%2Ccompress&fit=crop&h=360&q=70&w=640" alt="The Organizer" data-image-primary>\n              </div>\n            </div>\n\n          <h1 class="head primary site-font-primary-color collection-title">The Organizer</h1>\n\n            <div class="contain padding-top-medium collection-description word-break">\n    <div class="site-font-secondary-color"\n         data-text-show-less="Show less"\n         data-text-show-more="Show more"\n         data-truncate=\'{"max_height": 60, "speed": 300}\'>\n      <p>Directed by Mario Monicelli \xe2\x80\xa2 1963 \xe2\x80\xa2 France, Yugoslavia, Italy\n<br />Starring Marcello Mastroianni, Renato Salvatori, Annie Girardot</p>\n\n<p>In turn-of-the-twentieth-century Turin, an accident in a textile factory incites workers to stage a walkout. But it\xe2\x80\x99s not until they receive unexpected aid from a traveling professor (Marcello Mastroianni) that they find their voice, unite, and stand up for themselves. This historical drama by Mario Monicelli, brimming with humor and honesty, is a beautiful and moving ode to the power of the people, and features engaging, naturalistic performances; cinematography by the great Giuseppe Rotunno; and a multilayered, Oscar-nominated screenplay by Monicelli, Agenore Incrocci, and Furio Scarpelli.</p>\n    </div>\n  </div>\n\n\n          <div class="collection-actions padding-bottom-small padding-top-large">\n                  <a href="https://www.criterionchannel.com/checkout/subscribe?return_to=https%3A%2F%2Fcriterionchannel.vhx.tv%2Fthe-organizer"\n      class="btn btn-site-primary custom-btn-action-transact site-font-primary-family uppercase margin-bottom-small margin-right-small"\n      data-track-event="site_collection_buy">\n      Subscribe\n    </a>\n\n              <a href="https://www.criterionchannel.com/the-organizer/videos/the-organizer-trailer" class="btn btn-black custom-btn-action-trailer margin-bottom-small margin-right-medium">\n                <span class="icon-adjustment relative inverse-icon-bg dark-bg">\n                  <i class="icon small icon-play-inverse"></i>\n                </span>\n                Watch Trailer\n              </a>\n              <a href="#" class="btn btn-black custom-btn-action-share margin-bottom-small margin-right-medium " data-reveal-id="modal-the-organizer">\n                Share\n              </a>\n              \n\n\n\n<div class="reveal-modal site-background-color text-center share-modal-container" id="modal-the-organizer" data-reveal>\n  <a class="close-reveal-modal"><i class="icon icon-remove-inverse"></i><i class="icon icon-remove-gray"></i></a>\n  <div class="share-title-container padding-bottom-medium">\n    <h5 class="head secondary site-font-secondary-color site-font-primary-family">\n      Share with your friends\n    </h5>\n  </div>\n  <div class="share-link-container padding-bottom-medium">\n    <ul class="small-block-grid-3">\n      <li>\n        <a href="https://www.facebook.com/sharer/sharer.php?u=https%3A%2F%2Fwww.criterionchannel.com%2Fthe-organizer"\n           data-social-popup\n           data-share="facebook"\n           class="btn btn-facebook block border site-border-color"\n           title="Facebook">\n          <i class="vertical-align icon site-icon-facebook icon-center"></i>\n        </a>\n      </li>\n      <li>\n        <a href="https://twitter.com/share?url=https%3A%2F%2Fwww.criterionchannel.com%2Fthe-organizer&text=Watch+The+Organizer+from+%40criterionchannl&related=criterionchannl"\n           data-social-popup\n           data-share="twitter"\n           class="btn-twitter btn block border site-border-color"\n           title="Twitter">\n          <i class="vertical-align icon site-icon-twitter icon-center"></i>\n        </a>\n      </li>\n      <li>\n        <a target="_blank" href="/cdn-cgi/l/email-protection#57682422353d3234236a003623343f77033f327718253036393e2d3225773125383a77033f327714253e2332253e383977143f363939323b71363a276c3538332e6a3f232327247264167265117265112020207934253e2332253e3839343f363939323b7934383a726511233f327a38253036393e2d3225" class="btn btn-black block border site-border-color" title="Email">\n          <i class="vertical-align icon site-icon-envelope icon-center"></i>\n        </a>\n      </li>\n    </ul>\n  </div>\n    <div class="share-url-container">\n      <form class="form"><input value="https://www.criterionchannel.com/the-organizer" type="text" class="text text-center border-none site-background-color site-font-secondary-color" /></form>\n    </div>\n</div>\n\n          </div>\n        </div>\n      </div>\n\n        <div class="hide-for-small-only medium-8 large-offset-2 large-6 end cover-art columns">\n          <div class="embed-container grid-padding-right">\n            <div class="embed">\n              <img src="https://vhx.imgix.net/criterionchannelchartersu/assets/1dd7e4a4-f23b-40fc-8821-e39055a9315e.jpg?auto=format%2Ccompress&fit=crop&h=720&q=75&w=1280" alt="The Organizer" data-image-primary>\n            </div>\n          </div>\n        </div>\n\n    </div>\n  </section>\n\n  <section class="movie-container video-container padding-top-large padding-horizontal-medium">\n    <div class="row relative">\n      <div class="small-16 columns item-grid flush-grid flush-row">\n        \n  <div class="row full-width padding-horizontal-medium">\n    <ul class="small-block-grid-1 medium-block-grid-3 large-block-grid-3 ">\n\n\n\n  \n<li\n  class="js-collection-item\n    collection-item-456225 item-type-video\n    \n    "\n  data-item-id="456225"\n  data-item-type="video"\n  \n    data-tooltip="id: collection-tooltip-456225;\n    position: right;\n    offset: 0;\n    animate: true;\n    timer: 650;\n    delay: false;\n    is_centered: true;"\n    \n>\n  <div class="browse-item-card">\n    <div class="grid-item-padding">\n      <a href="https://www.criterionchannel.com/the-organizer/videos/the-organizer" class="browse-item-link" data-track-event="site_video" data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/the-organizer&quot;,&quot;id&quot;:456225,&quot;context&quot;:&quot;site_context_movie&quot;,&quot;label&quot;:&quot;The Organizer&quot;,&quot;canonical_collection&quot;:null}">\n        <div class="browse-image-container">\n            <span class="freemium-lock">\n              <i class="icon icon--xsmall icon-lock-inverse"></i>\n            </span>\n\n\n          <img alt="The Organizer" src="https://vhx.imgix.net/criterionchannelchartersu/assets/1dd7e4a4-f23b-40fc-8821-e39055a9315e.jpg?auto=format%2Ccompress&fit=crop&h=360&q=70&w=640" />\n\n\n          <div class="browse-info-container">\n              <div class="duration-container\n                is-locked\n                "\n              >\n                 2:09:54\n              </div>\n          </div>\n        </div> \n      </a>\n\n      <div class="padding-small">\n        <div class="site-font-primary-color browse-item-title">\n           <a\n            href="https://www.criterionchannel.com/the-organizer/videos/the-organizer"\n            data-track-event="site_video"\n            data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/the-organizer&quot;,&quot;id&quot;:456225,&quot;context&quot;:&quot;site_context_movie&quot;,&quot;label&quot;:&quot;The Organizer&quot;,&quot;canonical_collection&quot;:null}"\n          >\n            <strong title="The Organizer">\n              The Organizer\n            </strong>\n          </a>\n        </div>\n\n      </div>\n    </div>\n\n\n    <div \n         class="tooltip background-white" id="collection-tooltip-456225">\n      <h3 class="tooltip-item-title site-font-primary-family"><strong>The Organizer</strong></h3>\n\n        <div class="transparent padding-top-medium">\n          <p>Directed by Mario Monicelli \xe2\x80\xa2 1963 \xe2\x80\xa2 France, Yugoslavia, Italy\n<br />Starring Marcello Mastroianni, Renato Salvatori, Annie Girardot</p>\n\n<p>In turn-of-the-twentieth-century Turin, an accident in a textile factory incites workers to stage a walkout. But it\xe2\x80\x99s not until they receive unexpected aid from a travel...</p>\n        </div>\n    </div>\n  </div>\n</li>\n\n\n\n\n    </ul>\n  </div>\n\n      </div>\n    </div>\n  </section>\n    <section class="extras-container video-container padding-horizontal-medium padding-bottom-medium">\n      <div class="row relative border-top border-top-light padding-top-large">\n        <div class="small-16 columns item-grid flush-grid flush-row">\n          <h2 class="site-font-primary-color site-font-primary-family collection-title extras-header grid-padding-left padding-bottom-medium">\n            Extras\n          </h2>\n          \n  <div class="row full-width padding-horizontal-medium">\n    <ul class="small-block-grid-1 medium-block-grid-3 large-block-grid-4 js-load-more-items-container">\n\n\n\n  \n<li\n  class="js-collection-item\n    collection-item-2587567 item-type-video\n    \n    "\n  data-item-id="2587567"\n  data-item-type="video"\n  \n    data-tooltip="id: collection-tooltip-2587567;\n    position: right;\n    offset: 0;\n    animate: true;\n    timer: 650;\n    delay: false;\n    is_centered: true;"\n    \n>\n  <div class="browse-item-card">\n    <div class="grid-item-padding">\n      <a href="https://www.criterionchannel.com/the-organizer/videos/spotlight-on-the-organizer-with-imogen-sara-smith" class="browse-item-link" data-track-event="site_video" data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/the-organizer&quot;,&quot;id&quot;:2587567,&quot;context&quot;:&quot;site_context_movie&quot;,&quot;label&quot;:&quot;Spotlight on THE ORGANIZER with Imogen Sara Smith&quot;,&quot;canonical_collection&quot;:null}">\n        <div class="browse-image-container">\n            <span class="freemium-lock">\n              <i class="icon icon--xsmall icon-lock-inverse"></i>\n            </span>\n\n\n          <img alt="Spotlight on THE ORGANIZER with Imogen Sara Smith" src="https://vhx.imgix.net/criterionchannelchartersu/assets/93bd9c80-6537-4c28-a429-10bcd286399c.jpg?auto=format%2Ccompress&fit=crop&h=360&q=70&w=640" />\n\n\n          <div class="browse-info-container">\n              <div class="duration-container\n                is-locked\n                "\n              >\n                06:03\n              </div>\n          </div>\n        </div> \n      </a>\n\n      <div class="padding-small">\n        <div class="site-font-primary-color browse-item-title">\n           <a\n            href="https://www.criterionchannel.com/the-organizer/videos/spotlight-on-the-organizer-with-imogen-sara-smith"\n            data-track-event="site_video"\n            data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/the-organizer&quot;,&quot;id&quot;:2587567,&quot;context&quot;:&quot;site_context_movie&quot;,&quot;label&quot;:&quot;Spotlight on THE ORGANIZER with Imogen Sara Smith&quot;,&quot;canonical_collection&quot;:null}"\n          >\n            <strong title="Spotlight on THE ORGANIZER with Imogen Sara Smith">\n              Spotlight on THE ORGANIZER with Imogen Sara Smith\n            </strong>\n          </a>\n        </div>\n\n      </div>\n    </div>\n\n\n    <div \n         class="tooltip background-white" id="collection-tooltip-2587567">\n      <h3 class="tooltip-item-title site-font-primary-family"><strong>Spotlight on THE ORGANIZER with Imogen Sara Smith</strong></h3>\n\n        <div class="transparent padding-top-medium">\n          <p>In the latest installment of our Spotlight series, critic Imogen Sara Smith examines Mario Monicelli\xe2\x80\x99s politically engaged comedy THE ORGANIZER, a film that asks whether comedy can find room for such serious subject matter as trade unionism and class struggle.</p>\n        </div>\n    </div>\n  </div>\n</li>\n\n\n\n\n  \n<li\n  class="js-collection-item\n    collection-item-458423 item-type-video\n    \n    "\n  data-item-id="458423"\n  data-item-type="video"\n  \n    data-tooltip="id: collection-tooltip-458423;\n    position: right;\n    offset: 0;\n    animate: true;\n    timer: 650;\n    delay: false;\n    is_centered: true;"\n    \n>\n  <div class="browse-item-card">\n    <div class="grid-item-padding">\n      <a href="https://www.criterionchannel.com/the-organizer/videos/mario-monicelli" class="browse-item-link" data-track-event="site_video" data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/the-organizer&quot;,&quot;id&quot;:458423,&quot;context&quot;:&quot;site_context_movie&quot;,&quot;label&quot;:&quot;Mario Monicelli on THE ORGANIZER&quot;,&quot;canonical_collection&quot;:null}">\n        <div class="browse-image-container">\n            <span class="freemium-lock">\n              <i class="icon icon--xsmall icon-lock-inverse"></i>\n            </span>\n\n\n          <img alt="Mario Monicelli on THE ORGANIZER" src="https://vhx.imgix.net/criterionchannelchartersu/assets/154cd637-206a-4535-9729-26feb6edbc8c-ef5faed8.jpg?auto=format%2Ccompress&fit=crop&h=360&q=70&w=640" />\n\n\n          <div class="browse-info-container">\n              <div class="duration-container\n                is-locked\n                "\n              >\n                10:38\n              </div>\n          </div>\n        </div> \n      </a>\n\n      <div class="padding-small">\n        <div class="site-font-primary-color browse-item-title">\n           <a\n            href="https://www.criterionchannel.com/the-organizer/videos/mario-monicelli"\n            data-track-event="site_video"\n            data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/the-organizer&quot;,&quot;id&quot;:458423,&quot;context&quot;:&quot;site_context_movie&quot;,&quot;label&quot;:&quot;Mario Monicelli on THE ORGANIZER&quot;,&quot;canonical_collection&quot;:null}"\n          >\n            <strong title="Mario Monicelli on THE ORGANIZER">\n              Mario Monicelli on THE ORGANIZER\n            </strong>\n          </a>\n        </div>\n\n      </div>\n    </div>\n\n\n    <div \n         class="tooltip background-white" id="collection-tooltip-458423">\n      <h3 class="tooltip-item-title site-font-primary-family"><strong>Mario Monicelli on THE ORGANIZER</strong></h3>\n\n        <div class="transparent padding-top-medium">\n          <p>In the following piece, recorded in 2006, director Mario Monicelli introduces THE ORGANIZER.</p>\n        </div>\n    </div>\n  </div>\n</li>\n\n\n\n\n    </ul>\n  </div>\n\n          \n        </div>\n      </div>\n    </section>\n</section>\n\n  </main>\n\n    <style>\n  a.site-link-footer-color,\n  a.site-link-footer-color:visited {\n    color: #ffffff !important;\n  }\n  .btn-site-footer,\n  .btn-site-footer:link,\n  .btn-site-footer:visited {\n    color: #b9bcc7;\n    background-color: #000;\n  }\n  .btn-site-footer:hover,\n  .btn-site-footer.is-hover,\n  .btn-site-footer:focus,\n  .btn-site-footer:focus:hover,\n  .btn-site-footer:active,\n  .btn-site-footer.is-active {\n    color: #000;\n    border-color: #000;\n    background-color: #b9bcc7;\n  }\n\n  .btn-logout-form {\n    display: inline;\n    line-height: 0;\n  }\n\n  .btn-logout {\n    background-color: transparent;\n    border: none;\n    cursor: pointer;\n    font-size: 12px;\n    padding: 0;\n  }\n\n  .btn-logout:hover {\n    opacity: .5;\n  }\n</style>\n\n\n<footer class="footer--site border-top site-border-color site-background-color padding-top-medium padding-bottom-medium ">\n  <div class="row full-width">\n    <div class="small-16 medium-6 large-10 columns helper-links small-only-text-center">\n      <div class="padding-left-medium display-flex-container flex-start force-min-height">\n              <a href="https://www.criterion.com/" class="site-font-secondary-color" target="_blank" rel="noopener">THE CRITERION COLLECTION</a>\n\n\n        <a href="/help" class="site-font-secondary-color legal-link help-link" rel="noopener">\n          Help\n        </a>\n        <a href="/tos" class="site-font-secondary-color legal-link terms-link" rel="noopener">\n          Terms\n        </a>\n        <a href="/privacy" class="site-font-secondary-color legal-link privacy-link" rel="noopener">\n          Privacy\n        </a>\n        <a href="/cookies" class="site-font-secondary-color legal-link cookies-link" rel="noopener">\n          Cookies\n        </a>\n\n\n            <a href="/login" class="site-font-secondary-color signin-link">\n              Sign in\n            </a>\n      </div>\n    </div> <!-- columns -->\n    <div class="small-16 medium-10 large-6 small-only-text-center text-right columns vhx-link">\n      <div class="padding-right-small display-flex-container flex-end force-min-height">\n          &nbsp;\n        <div id="language-switcher-react-app"\n             class="padding-right-small"\n             data-supported-locales="[]"\n        </div>\n        <script data-cfasync="false" src="/cdn-cgi/scripts/5c5dd728/cloudflare-static/email-decode.min.js"></script><script src="https://cdn.vhx.tv/assets/vite/assets/language-switcher.94cd8c1c.js" crossorigin="anonymous" type="module"></script><link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/jsx-runtime.763a5c32.js" as="script" crossorigin="anonymous">\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/LanguageSwitcher.47e95f91.js" as="script" crossorigin="anonymous">\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/index.71ea8908.js" as="script" crossorigin="anonymous">\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/intl.helpers.f379992a.js" as="script" crossorigin="anonymous">\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/index.61149844.js" as="script" crossorigin="anonymous">\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/deviceSizes.990ebe17.js" as="script" crossorigin="anonymous">\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/message.3bf4adb2.js" as="script" crossorigin="anonymous">\n      </div> <!-- display-flex-container -->\n    </div>\n  </div> <!-- row -->\n</footer>\n\n<div class="modal">\n  <div class="close"><a href="#close" alt="Close Window"><strong class="is-inverse">&times;</strong></a></div>\n  <div class="content"></div>\n</div>\n\n<div id="blanket"></div>\n\n\n  <div class="modals">\n    \n  </div>\n\n  <script src="https://cdn.vhx.tv/assets/webpack/vendor-ottbundle-8f0450ca8ec63c03ff2b.js"></script>\n\n  <script>\n    window.VHX = window.VHX || {};\n    window.VHX.config = {\n      api_url: "https://api.vhx.tv",\n      token: "eyJhbGciOiJSUzI1NiIsImtpZCI6ImQ2YmZlZmMzNGIyNTdhYTE4Y2E2NDUzNDE2ZTlmZmRjNjk4MDAxMDdhZTQ2ZWJhODg0YTU2ZDBjOGQ4NTYzMzgifQ.eyJhcHBfaWQiOjY4NTExLCJleHAiOjE2NzgzOTgwMDcsIm5vbmNlIjoiZTU4M2VjZTVmNDRiNTg0NCIsInNjb3BlcyI6WyJwdWJsaWMiXSwic2Vzc2lvbl9pZCI6ImoyUVNNL3BZYU5BWHpaMFVMM1R3NkE9PSIsInNpdGVfaWQiOjU5MDU0fQ.tMPhDR9upjH9nJpOW6tPltFT1nvvsfBPrtuprqn-l6yI-ruqPwqofUVGSlLYJNmUi2bdiD6Kz3IQ2Oi5Sedznpte6OoDjecbZTaRp7CY59CbykqXrqRhO1UYtvJZYjGzmpWoKHE9dCE68WR6Lpu5L8J_XfG-FlWR9lTwmBy067RZJ57RscTa0xRanVFCHl7yVbFbYizEiexLVn-guHSIlenaKFXTLIdAB1J_DNLbwlwnRN9-aCU9EAKTvPwBxdxpLyP2aGqK8oXovW97EvmrDWIdg_doltCqrhIuljMgeWU2KRbtfoAjd0eMxnSvAZ7UCeTS5HECE0pU310HnmKLI7i0XRI_VaHwKBO_98qzSyPBBkuXNGBkwSxaZemocRCLZC7lzWni3Z59v5h3EgsPGhqR6gjb6K6yC14NxzymRYnRJ8Apc7Fx6_uCV4VDVt_W_XM2zkPBG6PP_ME-EpUDWH2IPcIdDrZ0p2sxQc8jDGC3AkG-6MzhRmgLoveY15W_5suR49HPy1WZcBZ_Mo1eDhQe-FFGWhZD2shRF8iJ8RovDv7U0G9sEvT6tIkMeta-1XWtcJx_ywwn-lsj3CMC07850pGCpvI_6oZCdNjDqmUxKmWojLiRJng98uRtTASX9m2UKveaA0l8g9Z0aySQDdLxfinVdDeqyTE31G2iErE",\n      token_expires_in: "7200000",\n      user_has_subscription: "false"\n    };\n  </script>\n\n  <script src="https://cdn.vhx.tv/assets/webpack/sites-ottbundle-39186b2977137610deba.js"></script>\n  \n\n      <script>\n    window.VHX = window.VHX || {};\n    window.VHX.config = {\n      api_url: "https://api.vhx.tv",\n      token: "eyJhbGciOiJSUzI1NiIsImtpZCI6ImQ2YmZlZmMzNGIyNTdhYTE4Y2E2NDUzNDE2ZTlmZmRjNjk4MDAxMDdhZTQ2ZWJhODg0YTU2ZDBjOGQ4NTYzMzgifQ.eyJhcHBfaWQiOjY4NTExLCJleHAiOjE2NzgzOTgwMDcsIm5vbmNlIjoiZTU4M2VjZTVmNDRiNTg0NCIsInNjb3BlcyI6WyJwdWJsaWMiXSwic2Vzc2lvbl9pZCI6ImoyUVNNL3BZYU5BWHpaMFVMM1R3NkE9PSIsInNpdGVfaWQiOjU5MDU0fQ.tMPhDR9upjH9nJpOW6tPltFT1nvvsfBPrtuprqn-l6yI-ruqPwqofUVGSlLYJNmUi2bdiD6Kz3IQ2Oi5Sedznpte6OoDjecbZTaRp7CY59CbykqXrqRhO1UYtvJZYjGzmpWoKHE9dCE68WR6Lpu5L8J_XfG-FlWR9lTwmBy067RZJ57RscTa0xRanVFCHl7yVbFbYizEiexLVn-guHSIlenaKFXTLIdAB1J_DNLbwlwnRN9-aCU9EAKTvPwBxdxpLyP2aGqK8oXovW97EvmrDWIdg_doltCqrhIuljMgeWU2KRbtfoAjd0eMxnSvAZ7UCeTS5HECE0pU310HnmKLI7i0XRI_VaHwKBO_98qzSyPBBkuXNGBkwSxaZemocRCLZC7lzWni3Z59v5h3EgsPGhqR6gjb6K6yC14NxzymRYnRJ8Apc7Fx6_uCV4VDVt_W_XM2zkPBG6PP_ME-EpUDWH2IPcIdDrZ0p2sxQc8jDGC3AkG-6MzhRmgLoveY15W_5suR49HPy1WZcBZ_Mo1eDhQe-FFGWhZD2shRF8iJ8RovDv7U0G9sEvT6tIkMeta-1XWtcJx_ywwn-lsj3CMC07850pGCpvI_6oZCdNjDqmUxKmWojLiRJng98uRtTASX9m2UKveaA0l8g9Z0aySQDdLxfinVdDeqyTE31G2iErE",\n      token_expires_in: "7200000",\n      user_has_subscription: "false"\n    };\n  </script>\n\n  <script src="https://cdn.vhx.tv/assets/webpack/products-ottbundle-80e221c1ef7842c46bb7.js"></script>\n    <script src="https://cdn.vhx.tv/assets/vite/assets/search.5b19401f.js" crossorigin="anonymous" type="module"></script><link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/jsx-runtime.763a5c32.js" as="script" crossorigin="anonymous">\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/index.esm.f84f874f.js" as="script" crossorigin="anonymous">\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/index.0e8c968f.js" as="script" crossorigin="anonymous">\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/intl.helpers.f379992a.js" as="script" crossorigin="anonymous">\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/api.utils.c8000605.js" as="script" crossorigin="anonymous">\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/Menu.esm.73b9a255.js" as="script" crossorigin="anonymous">\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/Button.esm.01a08ee1.js" as="script" crossorigin="anonymous">\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/Text.esm.bfe09214.js" as="script" crossorigin="anonymous">\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/message.3bf4adb2.js" as="script" crossorigin="anonymous">\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/Checkmark.esm.69ba937a.js" as="script" crossorigin="anonymous">\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/Link.esm.63fcdd9f.js" as="script" crossorigin="anonymous">\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/Search.esm.d6eb5953.js" as="script" crossorigin="anonymous">\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/index.71ea8908.js" as="script" crossorigin="anonymous">\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/text-helpers.aa9b5961.js" as="script" crossorigin="anonymous">\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/time-helpers.96f382b5.js" as="script" crossorigin="anonymous">\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/_includes.3534e71e.js" as="script" crossorigin="anonymous">\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/_curry2.3e499763.js" as="script" crossorigin="anonymous">\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/path.235ca80c.js" as="script" crossorigin="anonymous">\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/Header.esm.945377e9.js" as="script" crossorigin="anonymous">\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/DismissX.esm.cee35906.js" as="script" crossorigin="anonymous">\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/colors.esm.e8046dae.js" as="script" crossorigin="anonymous">\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/inheritsLoose.a8857277.js" as="script" crossorigin="anonymous">\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/index.61149844.js" as="script" crossorigin="anonymous">\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/ChevronDown.esm.55a41fde.js" as="script" crossorigin="anonymous">\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/_isArray.68eb59b2.js" as="script" crossorigin="anonymous">\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/index.096cf963.js" as="script" crossorigin="anonymous">\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/equals.d9a066f9.js" as="script" crossorigin="anonymous">\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/_isString.019dcf3c.js" as="script" crossorigin="anonymous"><link rel="stylesheet" media="screen" href="https://cdn.vhx.tv/assets/vite/assets/search.b3f20860.css" />\n\n\n<script>\n  setTimeout(function() {\n    document.documentElement.className += \' is-loaded\';\n    var selects = document.getElementsByClassName(\'select\'),\n        checkboxes = document.getElementsByClassName(\'checkbox\'),\n        i;\n\n    if (selects.length) {\n      for (i = 0; i < selects.length; i++) {\n        selects[i].className += \' ready\';\n      }\n    }\n\n    if (checkboxes.length) {\n      for (i = 0; i < checkboxes.length; i++) {\n        for (var p=0; p < checkboxes[i].childNodes.length; p++) {\n          if (checkboxes[i].childNodes[p].tagName == \'LABEL\') {\n            checkboxes[i].childNodes[p].className += \' ready\';\n            break;\n          }\n        }\n      }\n    }\n  }, 500);\n</script>\n\n  <script>\n  window.addEventListener(\'load\', function() {\n    jQuery(\'a:contains("Sign In")\').click(function(){\n      ga(\'site.send\',\'event\',\'button\',\'click\',\'sign in\');\n    });\n\n    jQuery(\'a:contains("Sign Up")\').click(function(){\n      ga(\'site.send\',\'event\',\'button\',\'click\',\'sign up\');\n    });\n\n    jQuery(\'a:contains("Watch now")\').click(function(){\n      ga(\'site.send\',\'event\',\'button\',\'click\',\'watch now\');\n    });\n\n    jQuery(\'a:contains("Share")\').click(function(){\n      ga(\'site.send\',\'event\',\'button\',\'click\',\'share\');\n    });\n\n    jQuery(\'[href="https://www.criterionchannel.com/the-deviant-and-the-divine/season:1/videos/freaks"],[href="https://www.criterionchannel.com/the-deviant-and-the-divine/season:1/videos/multiple-maniacs"]\').click(function(){\n      ga(\'site.send\',\'event\',\'button\',\'click\',\'video\');\n    });\n  });\n</script>\n\n\n\n    <script>\n      const javascript = \'\\$(document).ready(function(){\\n\\$(\\\'header .logo\\\').attr(\\\'data-swiftype-index\\\', false);\\n\\n\\$(\\"a[href*=\\\'/checkout\\\']\\").attr(\\"data html(\\"Sign Up\\");\\n  \\$(\\"a[href*=\\\'/checkout/subscribe/gift\\\']\\").html(\\"Buy a gift\\");\\n  \\$(\\"a[href*=\\\'/buy/subscription\\\']\\").attr(\\"data-trigger-disabled\\",true);\\n  \\$(\\"a[href*=\\\'/buy/subscription\\\']\\").html(\\"Sign Up\\");\\n  \\$(\\"a.browse-link\\").html(\\"Now Playing\\");\\n  landingPageChanges();\\n  collectionChanges();\\n  helpChanges();\\n  \\$(\\"body.browse .slide-subtitle\\").hide();\\n  \\$(\\"body.logged-in h1.identity a\\").attr(\\"href\\",\\"/\\");\\n  \\$(\\".navigation-inner\\").addClass(\\"is-ready\\");\\n  \\$(\\"nav .custom-nav-all-films\\").attr(\\"target\\", \\"_self\\");\\n\\n  amazon_fire_tv_supported = \\$(\\\'.amazon-fire-tv\\\').parent().parent().children()[1]\\n  \\$(amazon_fire_tv_supported).text(\\"2nd Gen +\\")\\n\\n  android_tv_supported_text = \\$(\\\'.android-tv\\\').parent().parent().children()[1]\\n  \\$(android_tv_supported_text).text(\\"Selected models\\")\\n  android_tv_4k_text = \\$(\\\'.android-tv\\\').parent().parent().children()[2]\\n  \\$(android_tv_4k_text).hide()\\n\\n  new_xbox = \\"<li>Xbox Series X|S<\\/li>\\";\\n  \\$(new_xbox).insertAfter(\\$(\\\'.platforms-modal-content-tv .platforms-modal-icon-container .platform-modal-devices-list > li:nth-of-type(5) li:nth-of-type(2)\\\'));\\n\\n  \\$(\\\'.platforms-modal-content-tv .platforms-modal-icon-container .platform-modal-devices-list > li:nth-of-type(6) li:nth-of-type(2)\\\').text(\\\'Select 2018, 2019 & 2020 Models\\\');\\n\\n  gift_link = \\\'<a href=\\"https://www.criterionchannel.com/checkout/subscribe/gift?gift=1&duration=6\\" class=\\"is-block text dark block padding-medium\\">Give a Gift<\\/a>\\\';\\n  \\$(gift_link).insertAfter(\\$(\\\'.user-dropdown a[href*=\\"/purchases\\"]\\\'));\\n\\n  if (\\$(\\\'meta[name*=\\"twitter:image\\"]\\\').length > 0) {\\n    twitter_content = \\$(\\\'meta[name*=\\"twitter:image\\"]\\\').attr(\\\'content\\\')\\n    new_meta = \\\'<meta class=\\"swiftype\\" name=\\"image\\" data-type=\\"enum\\" content=\\"\\\' + twitter_content + \\\'\\">\\\';\\n    \\$(\\\'head\\\').append(new_meta);\\n  }\\n});\\n\\nfunction collectionChanges(){\\n  if (\\$(\\"body\\").hasClass(\\"collections\\")){\\n    \\$(\\\'.cover-art\\\').removeClass(\\\'large-offset-2 large-6\\\').addClass(\\\'large-8 large-offset-1\\\');\\n\\n    \\$(\\\'section.product-feature\\\').removeClass(\\\'padding-horizontal-medium padding-top-large padding-bottom-xlarge\\\').addClass(\\\'padding-bottom-large\\\');\\n    \\$(\\\'section.product-feature .row.relative\\\').addClass(\\\'full-width\\\');\\n\\n    \\$(\\\'section.product-feature .collection-details-container\\\').removeClass(\\"large-8 end\\").addClass(\\"large-7\\");\\n      \\$(\\\'section.product-feature .collection-details\\\').addClass(\\"padding-left-large padding-right-large padding-top-xlarge\\");\\n\\n      \\$(\\".embed-container\\").removeClass(\\"grid-padding-right\\");\\n  }\\n  if (\\$(\\\'.collection-playlist\\\').length > 0 || \\$(\\\'.collection-series\\\').length > 0) {\\n    \\$(\\\'.video-controls\\\').removeClass(\\\'padding-top-large\\\').addClass(\\\'padding-top-small\\\');\\n  }\\n\\n  if (\\$(\\\'.collection-movie\\\').length > 0) {\\n    \\$(\\\'.collection-actions a:contains(\\"Trailer\\")\\\').html(\\\'<span class=\\"icon-adjustment relative inverse-icon-bg dark-bg margin-bottom-small\\"><i class=\\"icon small icon-play-inverse\\"><\\/i><\\/span> Trailer\\\');\\n    \\$(\\\'.collection-actions a:contains(\\"Movie\\")\\\').html(\\\'<span class=\\"icon-adjustment relative inverse-icon-bg dark-bg margin-bottom-small\\"><i class=\\"icon small icon-play-inverse\\"><\\/i><\\/span> Watch\\\');\\n  }\\n\\n\\n}\\n\\nfunction landingPageChanges(){\\n  \\$(\\\'a:contains(\\"Start Your Free Month\\")\\\').text(\\"Sign Up Now\\");\\n}\\nfunction helpChanges(){\\n  // Custom Help Center Updates\\n  setTimeout(function() {\\n    \\$(\\\'article li:contains(\\"You can plug your laptop or mobile device into your TV using an HDMI cable\\")\\\').each(function() {\\n      \\$(this).html(\\\'<li>It may be possible to connect your laptop or mobile device to your TV using an HDMI cable if all components in the chain are <a href=\\"https://www.criterionchannel.com/help/articles/drm-and-hdcp#HDCP\\">HDCP/DRM compliant.<\\/a><\\/li>\\\');\\n    });\\n\\n    \\$(\\\'article li:contains(\\" and select the language of choice\\")\\\').each(function() {\\n      \\$(this).text(\\$(this).text().replace(\\" and select the language of choice\\", \\"\\"));\\n    });\\n    \\n    a = \\\'<li>If you have a supported device connected to your TV, Criterion also has an app for Apple TV, Amazon Fire, Roku, and Android TV.<\\/li>\\\'\\n    b = \\$(\\\'li:contains(\\"You can mirror or Airplay from your iOS device\xe2\x80\x99s browser.\\")\\\')\\n    b.after(a);\\n\\n    \\$(\\\'li a[href*=\\"670-how-do-i-login-on-the-xbox-app\\"]\\\').parent().hide();\\n  }, 500);\\n\\n  // hide mentions of the VHX articles\\n  // we have to settimeout because the help pages are ajax\\\'ing in content.\\n  // this is best guess on timing\\n  if (\\$(\\"body\\").hasClass(\\"help-articles\\") || \\$(\\"body\\").hasClass(\\"sites-help\\") || \\$(\\"body\\").hasClass(\\"help-categories\\")){\\n   \\$(\\\'#mithril-help\\\').bind(\\"DOMSubtreeModified\\",function(){\\n    var waittoremove = setTimeout(function(){ removeHelpArticles(); }, 500);\\n  });\\n\\n    var waittoremove = setTimeout(function(){ removeHelpArticles(); }, 700);\\n  }\\n}\\nfunction removeHelpArticles(){\\n  var vhxhelp = Array(\\"589e1e742c7d3a784630ccd5\\",\\"581396999033604deb0ec504\\",\\"57fe5ceb9033600277a6980b\\",\\"54b59261e4b0934e822d2132\\",\\"54dbc064e4b086c0c0966908\\",\\"5453fe87e4b02ce342671537\\",\\"555119ebe4b027e1978dfc04\\",\\"549206b9e4b07d03cb251286\\", \\"5c7eee4f04286350d088ab37\\");\\n  for (i=0;i<=vhxhelp.length-1;i++){\\n    \\$(\\".article-\\"+vhxhelp[i]).remove();                                                        \\n  }\\n}\';\n      try {\n        // Using eval so we can catch syntax errors\n        eval(javascript);\n      } catch (err) {\n        console.error(err);\n      }\n    </script>\n</body>\n</html>\n'
    return BeautifulSoup(content, 'html5lib')


@pytest.fixture
def soup_special_subtitle():
    content = b'<!doctype html>\n<!--[if lt IE 7]> <html class="no-js lt-ie9 lt-ie8 lt-ie7" lang="en" xml:lang="en"> <![endif]-->\n<!--[if IE 7]>    <html class="no-js lt-ie9 lt-ie8" lang="en" xml:lang="en"> <![endif]-->\n<!--[if IE 8]>    <html class="no-js lt-ie9" lang="en" xml:lang="en"> <![endif]-->\n<!--[if gt IE 8]><!--> <html class="no-js" lang="en" xml:lang="en"> <!--<![endif]-->\n<head lang="en" xml:lang="en">\n  <meta charset="utf-8">\n  <meta http-equiv="X-UA-Compatible" content="IE=edge,chrome=1">\n  <meta name="viewport" content="width=device-width, initial-scale=1">\n\n    <link rel="shortcut icon" href="https://dr56wvhu2c8zo.cloudfront.net/criterionchannel/assets/5922c7f6-cdda-4786-8ac9-521602da043a.png" type="image/png">\n\n    <meta name="apple-itunes-app" content="app-id=1454275199, app-argument=https://criterionchannel.vhx.tv/foreign-language-oscar-winners/season:1/videos/war-and-peace-part-2">\n\n    <meta name="facebook-domain-verification" content="qk8q6gxhll8cwf99kggygprhe0y9fz">\n\n  <title>WAR AND PEACE: Part 2 - Foreign-Language Oscar Winners - The Criterion Channel</title>\n  <meta name="description" content="NATASHA ROSTOVA\nDirected by Sergei Bondarchuk \xe2\x80\xa2 1966 \xe2\x80\xa2 Soviet Union\n\nAt the height of the Cold War, the Soviet film industry set out to prove it could outdo Hollywood with a production that would dazzle the world: a titanic, awe-inspiring adaptation of Tolstoy\xe2\x80\x99s classic tome in which the fates of three souls\xe2\x80\x94the blundering, good-hearted Pierre; the heroically tragic Prince Andrei; and the radiant, tempestuous Natasha\xe2\x80\x94collide amid the tumult of the Napoleonic Wars. Employing a cast of thousands and an array of innovative camera techniques, director Sergei Bondarchuk conjures a sweeping vision of grand balls that glitter with rococo beauty and breathtaking battles that overwhelm with their expressionistic power. As a statement of Soviet cinema\xe2\x80\x99s might, WAR AND PEACE succeeded wildly, garnering the Academy Award for best foreign-language film and setting a new standard for epic moviemaking.">\n  <link rel="canonical" href="https://www.criterionchannel.com/videos/war-and-peace-part-2" />\n\n  \n\n    <meta property="og:url" content="https://www.criterionchannel.com/foreign-language-oscar-winners/season:1/videos/war-and-peace-part-2">\n    <meta property="fb:app_id" content="136862186391940">\n    <meta property="og:title" content="WAR AND PEACE: Part 2 - Foreign-Language Oscar Winners - The Criterion Channel">\n    <meta property="og:site_name" content="The Criterion Channel">\n    <meta property="og:image" content="https://vhx.imgix.net/criterionchannelchartersu/assets/e5be2fc9-5a9c-4754-85b5-b887f4f00d1d.jpg?auto=format%2Ccompress&fit=crop&h=720&w=1280">\n    <meta property="og:image:width" content="1280">\n    <meta property="og:image:height" content="720">\n    <meta property="og:description" content="NATASHA ROSTOVA\nDirected by Sergei Bondarchuk \xe2\x80\xa2 1966 \xe2\x80\xa2 Soviet Union\n\nAt the height of the Cold War, the Soviet film industry set out to prove it could outdo Hollywood with a production that would dazzle the world: a titanic, awe-inspiring adaptation of Tolstoy\xe2\x80\x99s classic tome in which the fates of...">\n    <meta property="og:type" content="website">\n\n    <meta name="twitter:card" content="summary_large_image">\n      <meta name="twitter:site" content="@criterionchannl">\n      <meta name="twitter:creator" content="@criterionchannl">\n    <meta name="twitter:title" content="WAR AND PEACE: Part 2 - Foreign-Language Oscar Winners - The Criterion Channel">\n    <meta name="twitter:description" content="NATASHA ROSTOVA\nDirected by Sergei Bondarchuk \xe2\x80\xa2 1966 \xe2\x80\xa2 Soviet Union\n\nAt the height of the Cold War, the Soviet film industry set out to prove it could outdo Hollywood with a production that would dazzle the world: a titanic, awe-inspiring adaptation of Tolstoy\xe2\x80\x99s classic tome in which the fates of...">\n    <meta name="twitter:image" content="https://vhx.imgix.net/criterionchannelchartersu/assets/e5be2fc9-5a9c-4754-85b5-b887f4f00d1d.jpg?auto=format%2Ccompress&fit=crop&h=720&w=1280">\n    <meta name="twitter:image:width" content="1280">\n    <meta name="twitter:image:height" content="720">\n\n\n\n<link rel="author" href="/humans.txt">\n<meta name="csrf-param" content="authenticity_token" />\n<meta name="csrf-token" content="qV/HBikA8a2pwiNrS4JVFU2xC5YePkRNvqoBBETeM0sPBPo0dnArEk9By4pD8/qo2bnn0x8+fuL7qYk9GB3VrA==" />\n\n\n    <!-- apple / android  -->\n  <meta name="mobile-web-app-capable" content="no">\n  <meta name="apple-mobile-web-app-capable" content="no">\n  <meta name="apple-mobile-web-app-title" content="The Criterion Channel">\n  <!-- microsoft tile color -->\n  <meta name="msapplication-TileColor" content="#b9bcc7">\n  <!-- android browser bar color -->\n  <meta name="theme-color" content="#b9bcc7">\n\n    <!-- apple and other browsers -->\n    <link rel="apple-touch-icon" sizes="57x57"    href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=57&w=57">\n    <link rel="apple-touch-icon" sizes="114x114"  href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=114&w=114">\n    <link rel="apple-touch-icon" sizes="72x72"    href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=72&w=72">\n    <link rel="apple-touch-icon" sizes="144x144"  href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=144&w=144">\n    <link rel="apple-touch-icon" sizes="60x60"    href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=60&w=60">\n    <link rel="apple-touch-icon" sizes="120x120"  href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=120&w=120">\n    <link rel="apple-touch-icon" sizes="76x76"    href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=76&w=76">\n    <link rel="apple-touch-icon" sizes="152x152"  href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=152&w=152">\n    <link rel="apple-touch-icon" sizes="180x180"  href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=180&w=180">\n    <!-- windows tile -->\n    <meta name="msapplication-TileImage" content="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=144&w=144">\n    <!-- android icons -->\n    <link rel="icon" type="image/png" sizes="32x32" href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=32&w=32">\n    <link rel="icon" type="image/png" sizes="16x16" href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=16&w=16">\n\n\n    <meta class="swiftype" name="type" data-type="enum" content="Channel" />\n\n\n\n  <link rel="stylesheet" media="all" href="https://cdn.vhx.tv/assets/webpack/icons.svg-ottbundle-798d02cf9d8ee37d5dfe.css" />\n  <link rel="stylesheet" media="all" href="https://cdn.vhx.tv/assets/webpack/sites-ottbundle-45f07117e8bdae2b07b3.css" />\n\n      <link rel="stylesheet" media="all" href="https://cdn.vhx.tv/assets/webpack/videos-ottbundle-38db67622c529d3c8cae.css" />\n  <script>\n  var disableInputOnSubmit = function(el) {\n    if (document.createElement(\'div\').setAttribute && typeof(document.getElementsByClassName) !== \'undefined\')  {\n      el.getElementsByClassName(\'comment-submit\')[0].setAttribute(\'disabled\',\'disabled\');\n    }\n  }\n</script>\n\n\n\n    <link rel="stylesheet" media="screen" href="https://cdn.vhx.tv/assets/templates/_shared/typography.montserrat-3b605b0b18ab721686d69bf9c5bd93fc8f657562bc6a431ae71c409675dfb333.css" />\n\n<style>\n\n\n  /* main colors */\n  .site-background-color {\n    background-color: #141414 !important;\n  }\n  .site-primary-highlight-color {\n    background: #b9bcc7 !important;\n  }\n  .site-primary-color,\n  strong.site-primary-color,\n  .site-primary-color strong,\n  a.site-primary-color:link,\n  a.site-primary-color:visited {\n    color: #b9bcc7 !important;\n  }\n  a.site-primary-color:hover,\n  a.site-primary-color:active {\n    color: #9da0a9 !important;\n  }\n\n  .site-border-color,\n  .site-border-color.border-light {\n    border-color: #323232 !important;\n  }\n  .site-border-primary-color,\n  .site-border-primary-color.border-light {\n    border-color: #b9bcc7 !important;\n  }\n\n  /* custom font primary family */\n  .site-font-primary-family {\n    font-family: \'montserrat\', sans-serif !important;\n  }\n\n  /* font primary color */\n  .site-font-primary-color,\n  .site-font-primary-color strong {\n    color: #ffffff !important;\n  }\n\n  /* font secondary color */\n  .site-font-secondary-color,\n  .site-font-secondary-color strong {\n    color: #bebebb !important;\n  }\n\n  /* link primary color */\n  a.site-link-primary-color:link,\n  a.site-link-primary-color:visited,\n  a.site-link-primary-color strong {\n    color: #ffffff !important;\n  }\n  a.site-link-primary-color:hover,\n  a.site-link-primary-color:active {\n    color: #d9d9d9 !important;\n    opacity: .7;\n  }\n  .site-link-primary-color-contain a:link,\n  .site-link-primary-color-contain a:visited {\n    color: #b9bcc7 !important;\n  }\n  .site-link-primary-color-contain a:hover,\n  .site-link-primary-color-contain a:active {\n    color: #9da0a9 !important;\n    opacity: .7;\n  }\n\n  /* link secondary color */\n  .pagination a,\n  a.site-link-secondary-color:link,\n  a.site-link-secondary-color:visited,\n  a.site-link-secondary-color strong,\n  .site-link-secondary-color-contain a:link,\n  .site-link-secondary-color-contain a:visited {\n    color: #7c7c7c;\n  }\n  .pagination a:hover,\n  a.site-link-secondary-color:hover,\n  a.site-link-secondary-color:active,\n  .site-link-secondary-color-contain a:hover,\n  .site-link-secondary-color-contain a:active {\n    color: #696969;\n    opacity: .7;\n  }\n\n  /* link underline font primary color */\n  .site-link-contain a:link,\n  .site-link-contain a:visited {\n    color: #ffffff !important;\n    text-decoration: underline;\n  }\n  .site-link-contain a:hover,\n  .site-link-contain a:active {\n    color: #ffffff !important;\n    opacity: .7;\n    text-decoration: underline;\n  }\n\n  /* border color same as primary color */\n  .site-border-primary-color {\n    border-color: #ffffff !important;\n  }\n  .site-border-top-primary-color {\n    border-top-color: #ffffff !important;\n  }\n  .site-border-bottom-primary-color {\n    border-bottom-color: #ffffff !important;\n  }\n  .site-border-left-primary-color {\n    border-left-color: #ffffff !important;\n  }\n  .site-border-right-primary-color {\n    border-right-color: #ffffff !important;\n  }\n\n  /* link header color */\n  a.site-link-header-color:link,\n  a.site-link-header-color:visited,\n  a.site-link-header-color:link strong,\n  a.site-link-header-color:visited strong {\n    color: #ffffff !important;\n  }\n  a.site-link-header-color:hover,\n  a.site-link-header-color:active {\n    color: #d9d9d9;\n    opacity: .7;\n  }\n .avatar-chevron.icon {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2223%22%20height%3D%2223%22%20viewBox%3D%220%200%2023%2023%22%3E%3Cpolygon%20fill%3D%22%23ffffff%22%20points%3D%2222.9%2C7.2%2020%2C4.4%2011.5%2C12.9%203%2C4.4%200.1%2C7.2%208.7%2C15.7%208.7%2C15.7%2011.5%2C18.6%2011.5%2C18.6%2011.5%2C18.6%2014.3%2C15.7%2014.3%2C15.7%20%22%2F%3E%3C%2Fsvg%3E%0A");\n    background-repeat: no-repeat;\n  }\n  .user-dropdown a:hover {\n    background-color: rgba(51,51,51,0.2);\n    color: #fff;\n  }\n  .user-dropdown input {\n    background-color: #ffffff;\n  }\n  .user-dropdown input:hover {\n    background-color: rgba(51,51,51,0.2);\n    color: #fff;\n  }\n\n  /* buttons site primary */\n  .btn-site-primary,\n  .btn-site-primary:link,\n  .btn-site-primary:visited {\n    color: #000;\n    border-color: #9da0a9;\n    background-color: #b9bcc7;\n  }\n  .btn-site-primary:hover,\n  .btn-site-primary.is-hover,\n  .btn-site-primary:focus,\n  .btn-site-primary:focus:hover,\n  .btn-site-primary:active,\n  .btn-site-primary.is-active {\n    border-color: #9da0a9;\n    background-color: #9da0a9;\n    color: #000;\n  }\n\n  .btn-site-primary[disabled],\n  .btn-site-primary.is-disabled,\n  .btn-site-primary[disabled]:visited,\n  .btn-site-primary.is-disabled:visited,\n  .btn-site-primary[disabled]:hover,\n  .btn-site-primary.is-disabled:hover,\n  .btn-site-primary[disabled].is-hover,\n  .btn-site-primary.is-disabled.is-hover,\n  .btn-site-primary[disabled]:active,\n  .btn-site-primary.is-disabled:active,\n  .btn-site-primary[disabled]:focus,\n  .btn-site-primary.is-disabled:focus,\n  .btn-site-primary[disabled]:focus:hover,\n  .btn-site-primary.is-disabled:focus:hover,\n  .btn-site-primary[disabled].is-active,\n  .btn-site-primary.is-disabled.is-active {\n    opacity: .5;\n    background-color: #b9bcc7;\n    color: #000;\n    border-color: #9da0a9;\n  }\n\n  /* buttons site secondary */\n  .btn-site-secondary,\n  .btn-site-secondary:link,\n  .btn-site-secondary:visited {\n    color: #bebebb;\n    border-color: #141414;\n    background-color: #323232;\n  }\n  .btn-site-secondary:hover,\n  .btn-site-secondary.is-hover,\n  .btn-site-secondary:focus,\n  .btn-site-secondary:active,\n  .btn-site-secondary.is-active {\n    color: #5f5f5e;\n    border-color: #0f0f0f;\n    background-color: #323232;\n  }\n\n  .btn-site-primary.is-processing,\n  .btn-site-secondary.is-processing {\n    color: transparent !important;\n    background-position: center center !important;\n    background-repeat: no-repeat;\n    background-size: 23px 23px;\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3Csvg%20version%3D%221.1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%20%20viewBox%3D%220%200%2040%2040%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cpath%20opacity%3D%220.2%22%20fill%3D%22%23000000%22%20d%3D%22M20.201%2C5.169c-8.254%2C0-14.946%2C6.692-14.946%2C14.946c0%2C8.255%2C6.692%2C14.946%2C14.946%2C14.946%0D%0A%20%20s14.946-6.691%2C14.946-14.946C35.146%2C11.861%2C28.455%2C5.169%2C20.201%2C5.169z%20M20.201%2C31.749c-6.425%2C0-11.634-5.208-11.634-11.634%0D%0A%20%20c0-6.425%2C5.209-11.634%2C11.634-11.634c6.425%2C0%2C11.633%2C5.209%2C11.633%2C11.634C31.834%2C26.541%2C26.626%2C31.749%2C20.201%2C31.749z%22%2F%3E%0D%0A%3Cpath%20fill%3D%22%23000000%22%20d%3D%22M26.013%2C10.047l1.654-2.866c-2.198-1.272-4.743-2.012-7.466-2.012h0v3.312h0%0D%0A%20%20C22.32%2C8.481%2C24.301%2C9.057%2C26.013%2C10.047z%22%3E%0D%0A%20%20%3CanimateTransform%20attributeType%3D%22xml%22%0D%0A%20%20%20%20attributeName%3D%22transform%22%0D%0A%20%20%20%20type%3D%22rotate%22%0D%0A%20%20%20%20from%3D%220%2020%2020%22%0D%0A%20%20%20%20to%3D%22360%2020%2020%22%0D%0A%20%20%20%20dur%3D%220.5s%22%0D%0A%20%20%20%20repeatCount%3D%22indefinite%22%2F%3E%0D%0A%20%20%3C%2Fpath%3E%0D%0A%3C%2Fsvg%3E\');\n  }\n\n  /* buttons dropdown */\n  .btn-dropdown-site,\n  .btn-dropdown-site:link,\n  .btn-dropdown-site:visited {\n    color: #000000;\n    border-color: #b9bcc7;\n    background-color: #b9bcc7;\n  }\n  .btn-dropdown-site:hover,\n  .btn-dropdown-site.is-hover,\n  .btn-dropdown-site:focus,\n  .btn-dropdown-site:active,\n  .btn-dropdown-site.is-active {\n    border-color: #b9bcc7;\n    background-color: #9da0a9;\n  }\n\n  .site-btn-block,\n  .site-btn-block:link,\n  .site-btn-block:visited {\n    color: #ffffff;\n  }\n\n  .site-btn-block:hover,\n  .site-btn-block.is-hover,\n  .site-btn-block.is-active,\n  .site-btn-block:active {\n    box-shadow: inset 0px -4px 0px #ffffff;\n  }\n\n  .site-form-color input[type="email"],\n  .site-form-color input[type="text"],\n  .site-form-color input[type="search"],\n  .site-form-color input[type="password"],\n  .site-form-color input[type="tel"],\n  .site-form-color textarea {\n    color: #323232 !important;\n    border-color: #696969 !important;\n    background-color:  #fcfcfc !important;\n  }\n  .site-form-color label,\n  .site-form-color-minimal input[type="email"],\n  .site-form-color-minimal input[type="text"],\n  .site-form-color-minimal input[type="search"],\n  .site-form-color-minimal input[type="password"],\n  .site-form-color-minimal textarea {\n    color: #ffffff;\n    background: transparent !important;\n  }\n  .site-form-color input[type="text"]:focus,\n  .site-form-color input[type="search"]:focus,\n  .site-form-color input[type="password"]:focus,\n  .site-form-color input[type="tel"]:focus,\n  .site-form-color textarea:focus {\n    border-color: #7c7c7c !important;\n  }\n\n  .site-form-color input[type="text"]::-webkit-input-placeholder,\n  .site-form-color-minimal input[type="text"]::-webkit-input-placeholder,\n  .site-form-color input[type="email"]::-webkit-input-placeholder,\n  .site-form-color-minimal input[type="email"]::-webkit-input-placeholder,\n  .site-form-color input[type="search"]::-webkit-input-placeholder,\n  .site-form-color-minimal input[type="search"]::-webkit-input-placeholder,\n  .site-form-color input[type="password"]::-webkit-input-placeholder,\n  .site-form-color-minimal input[type="password"]::-webkit-input-placeholder,\n  .site-form-color input[type="tel"]::-webkit-input-placeholder,\n  .site-form-color-minimal input[type="tel"]::-webkit-input-placeholder,\n  .site-form-color textarea::-webkit-input-placeholder,\n  .site-form-color-minimal textarea::-webkit-input-placeholder {\n    color: #323232 !important;\n    opacity: .5;\n  }\n  .site-form-color input[type="text"]::-moz-input-placeholder,\n  .site-form-color-minimal input[type="text"]::-moz-input-placeholder,\n  .site-form-color input[type="email"]::-moz-input-placeholder,\n  .site-form-color-minimal input[type="email"]::-moz-input-placeholder,\n  .site-form-color input[type="search"]::-moz-input-placeholder,\n  .site-form-color-minimal input[type="search"]::-moz-input-placeholder,\n  .site-form-color input[type="password"]::-moz-input-placeholder,\n  .site-form-color-minimal input[type="password"]::-moz-input-placeholder,\n  .site-form-color input[type="tel"]::-moz-input-placeholder,\n  .site-form-color-minimal input[type="tel"]::-moz-input-placeholder,\n  .site-form-color textarea::-moz-input-placeholder,\n  .site-form-color-minimal textarea::-moz-input-placeholder {\n    color: #323232 !important;\n    opacity: .5;\n  }\n\n  .site-form-color input[type="text"]:-moz-input-placeholder,\n  .site-form-color-minimal input[type="text"]:-moz-input-placeholder,\n  .site-form-color input[type="email"]:-moz-input-placeholder,\n  .site-form-color-minimal input[type="email"]:-moz-input-placeholder,\n  .site-form-color input[type="search"]:-moz-input-placeholder,\n  .site-form-color-minimal input[type="search"]:-moz-input-placeholder,\n  .site-form-color input[type="password"]:-moz-input-placeholder,\n  .site-form-color-minimal input[type="password"]:-moz-input-placeholder,\n  .site-form-color input[type="tel"]:-moz-input-placeholder,\n  .site-form-color-minimal input[type="tel"]:-moz-input-placeholder,\n  .site-form-color textarea:-moz-input-placeholder,\n  .site-form-color-minimal textarea:-moz-input-placeholder {\n    color: #323232 !important;\n    opacity: .5;\n  }\n\n  .site-form-color input[type="text"]:-ms-input-placeholder,\n  .site-form-color-minimal input[type="text"]:-ms-input-placeholder,\n  .site-form-color input[type="email"]:-ms-input-placeholder,\n  .site-form-color-minimal input[type="email"]:-ms-input-placeholder,\n  .site-form-color input[type="search"]:-ms-input-placeholder,\n  .site-form-color-minimal input[type="search"]:-ms-input-placeholder,\n  .site-form-color input[type="password"]:-ms-input-placeholder,\n  .site-form-color-minimal input[type="password"]:-ms-input-placeholder,\n  .site-form-color input[type="tel"]:-ms-input-placeholder,\n  .site-form-color-minimal input[type="tel"]:-ms-input-placeholder,\n  .site-form-color textarea:-ms-input-placeholder,\n  .site-form-color-minimal textarea:-ms-input-placeholder {\n    color: #323232 !important;\n    opacity: .5;\n  }\n\n  .site-form-color .radio label .radio--icon {\n    border-color: #bebebb !important;\n  }\n  .site-form-color .radio.primary .circle-top span {\n    background: #b9bcc7 !important;\n  }\n  .site-form-color .radio.primary .circle-bottom span {\n    background: #94969f !important;\n  }\n\n  /* header */\n  .site-header {\n    border-bottom: 1px solid #2e2e2e;\n    background-color: #141414;\n\n\n  }\n  .site-header-background-color {\n    background-color: #141414;\n  }\n  @media only screen and (max-width: 40em) {\n     .site-header-background-color-small-only {\n      background-color: #141414;\n    }\n  }\n\n  /* modals */\n  .site-modal {\n    box-shadow: inset 0 0 2px #7c7c7c !important;\n  }\n\n  /* tooltips */\n  .tooltip.bottom-right:after,\n  .tooltip.bottom-left:after,\n  .tooltip.bottom:after,\n  .tooltip.bottom-right:before,\n  .tooltip.bottom-left:before,\n  .tooltip.bottom:before {\n    border-top-color: #7c7c7c;\n  }\n  .tooltip.right:after,\n  .tooltip.right:after\n  .tooltip.right-top:after,\n  .tooltip.right-top:before,\n  .tooltip.right-bottom:before,\n  .tooltip.right-bottom:before {\n    border-right-color: #7c7c7c;\n  }\n\n  .tooltip.left-bottom:after,\n  .tooltip.left-top:after,\n  .tooltip.left:after,\n  .tooltip.left-bottom:before,\n  .tooltip.left-top:before,\n  .tooltip.left:before {\n    border-left-color: #7c7c7c;\n  }\n\n  .tooltip.top-right:after,\n  .tooltip.top-left:after,\n  .tooltip.top:after,\n  .tooltip.top-right:before,\n  .tooltip.top-left:before,\n  .tooltip.top:before {\n    border-bottom-color: #7c7c7c;\n  }\n\n  .tooltip {\n    border: 1px solid #7c7c7c;\n    background: #7c7c7c;\n    color: #ffffff;\n  }\n\n  .reveal-modal {\n    border-color: #323232;\n    box-shadow: 0 0 3px #cbcbcb;\n  }\n  .reveal-modal-bg {\n    background-color: rgba(0,0,0,.8);\n  }\n  .play-outer-circle-large:hover {\n    border-color: #b9bcc7;\n  }\n  .forums-container tr.clickable:hover .thread-activity {\n    background-color: #141414;\n  }\n\n    [data-flash].is-inline.is-error {\n      background: rgba(211, 16, 74, 1) !important;\n    }\n    [data-flash].is-inline.is-error .flash-msg,\n    [data-flash].is-inline.is-error .flash-msg strong {\n      color: #fff !important;\n    }\n    [data-flash].is-inline.is-error .flash-close .icon-remove-mono {\n      display: none !important;\n    }\n    [data-flash].is-inline.is-error .flash-close .icon-remove-inverse {\n      display: block !important;\n    }\n\n  /* custom banner */\n\n\n</style>\n<style>\n  /* icons site primary */\n  .site-icon-arrow-right {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2223%22%20height%3D%2223%22%20viewBox%3D%220%200%2023%2023%22%3E%3Ctitle%3Eicons-arrows-set%20copy%3C%2Ftitle%3E%3Cpath%20d%3D%22M5.61%2C19.75l8.26-8.26L5.61%2C3.24c-0.68-.69.61-1.58%2C1-2%2C0.72-.69%2C1.76.78%2C2.2%2C1.22l3.39%2C3.39L17.41%2C11c0.71%2C0.69-.8%2C1.77-1.22%2C2.19L12.8%2C16.59%2C7.63%2C21.77c-0.7.67-1.58-.61-2-1A0.68%2C0.68%2C0%2C0%2C1%2C5.61%2C19.75Z%22%20fill%3D%22%23b9bcc7%22%2F%3E%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-menu {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Hamburger%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Crect%20x%3D%222.6%22%20y%3D%223.6%22%20width%3D%2218%22%20height%3D%223%22%20fill%3D%22%23ffffff%22%2F%3E%0D%0A%3Crect%20x%3D%222.6%22%20y%3D%2210%22%20width%3D%2218%22%20height%3D%223%22%20fill%3D%22%23ffffff%22%2F%3E%0D%0A%3Crect%20x%3D%222.6%22%20y%3D%2216.3%22%20width%3D%2218%22%20height%3D%223%22%20fill%3D%22%23ffffff%22%2F%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-chrevron-down {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21--%20Generator%3A%20Adobe%20Illustrator%2018.1.1%2C%20SVG%20Export%20Plug-In%20.%20SVG%20Version%3A%206.00%20Build%200%29%20%20--%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%20%20%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%22-294%20385%2023%2023%22%20enable-background%3D%22new%20-294%20385%2023%2023%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cg%3E%0D%0A%20%20%3Cpolygon%20fill%3D%22%23ffffff%22%20points%3D%22-271.1%2C392.9%20-275.4%2C388.7%20-282.5%2C395.8%20-289.6%2C388.7%20-293.9%2C392.9%20-286.7%2C400%20-286.8%2C400.1%20%0D%0A%20%20%20%20-282.5%2C404.3%20-282.5%2C404.3%20-282.5%2C404.3%20-278.2%2C400.1%20-278.3%2C400%20%20%22%2F%3E%0D%0A%3C%2Fg%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-chat {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21--%20Generator%3A%20Adobe%20Illustrator%2018.1.1%2C%20SVG%20Export%20Plug-In%20.%20SVG%20Version%3A%206.00%20Build%200%29%20%20--%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%22-294%20385%2023%2023%22%20enable-background%3D%22new%20-294%20385%2023%2023%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cg%3E%0D%0A%09%3Cg%3E%0D%0A%09%09%3Cpath%20fill%3D%22%23ffffff%22%20d%3D%22M-282.5%2C388.6c2.6%2C0%2C5.1%2C0.1%2C6.8%2C1.3c1.9%2C1.4%2C1.9%2C4.3%2C1.7%2C6c-0.2%2C2.1-1.2%2C3.5-3%2C4.2%0D%0A%09%09%09c-1.3%2C0.5-3%2C0.7-5.1%2C0.7c-1%2C0-2.1-0.1-3.4-0.2l-1-0.1l0%2C0l-1.2-0.4c-1.1-0.3-2.5-1-3.1-2.6c-0.7-1.8-0.7-4.2%2C0-5.9%0D%0A%09%09%09C-289.6%2C389.1-286.6%2C388.6-282.5%2C388.6%20M-282.5%2C386.6c-3.7%2C0-8.4%2C0.3-9.9%2C4.3c-0.9%2C2.2-0.9%2C5.1%2C0%2C7.4c0.7%2C2%2C2.4%2C3.2%2C4.3%2C3.8v3.2%0D%0A%09%09%09l2.6-2.7c1.1%2C0.1%2C2.4%2C0.2%2C3.6%2C0.2c2%2C0%2C4-0.2%2C5.7-0.9c2.6-1%2C4-3.1%2C4.3-5.8c0.3-2.8-0.1-6.1-2.6-7.9%0D%0A%09%09%09C-276.7%2C386.6-279.8%2C386.6-282.5%2C386.6L-282.5%2C386.6z%22%2F%3E%0D%0A%09%3C%2Fg%3E%0D%0A%3C%2Fg%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n\n  /* icons site social */\n  .site-icon-twitter {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%22-294%20385%2023%2023%22%20enable-background%3D%22new%20-294%20385%2023%2023%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cg%3E%0D%0A%09%3Cpath%20fill%3D%22%23ffffff%22%20d%3D%22M-272.5%2C390.3c-0.7%2C0.3-1.5%2C0.5-2.4%2C0.6c0.8-0.5%2C1.5-1.3%2C1.8-2.3c-0.8%2C0.5-1.7%2C0.8-2.6%2C1%0D%0A%09%09c-2.8-3-7.9-0.2-7%2C3.7c-3.4-0.1-6.4-1.7-8.4-4.2c-1.1%2C1.9-0.5%2C4.3%2C1.3%2C5.5c-0.7%2C0-1.3-0.2-1.9-0.5c0%2C1.9%2C1.4%2C3.7%2C3.3%2C4.1%0D%0A%09%09c-0.6%2C0.2-1.2%2C0.2-1.9%2C0.1c0.5%2C1.6%2C2%2C2.8%2C3.8%2C2.8c-1.7%2C1.3-3.9%2C2-6.1%2C1.7c4%2C2.6%2C9.5%2C2.5%2C13.4-0.5c3-2.3%2C4.8-6.1%2C4.6-9.9%0D%0A%09%09C-273.7%2C391.8-273.1%2C391.1-272.5%2C390.3C-273.2%2C390.6-273.1%2C391.1-272.5%2C390.3z%22%2F%3E%0D%0A%3C%2Fg%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-facebook {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21--%20Generator%3A%20Adobe%20Illustrator%2018.1.1%2C%20SVG%20Export%20Plug-In%20.%20SVG%20Version%3A%206.00%20Build%200%29%20%20--%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%22-294%20385%2023%2023%22%20enable-background%3D%22new%20-294%20385%2023%2023%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cpath%20fill%3D%22%23ffffff%22%20d%3D%22M-280.4%2C406.8v-9.3h3.4v-4h-3.4v-2.1c0-0.5%2C0.3-0.9%2C0.9-0.9h2.6v-4.5h-2.8c-3%2C0-5.3%2C2.4-5.3%2C5.5v2.1h-3.1v4%0D%0A%09h3.1v9.3h4.6L-280.4%2C406.8z%22%2F%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-tumblr {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cg%3E%0D%0A%09%3Cpath%20style%3D%22fill%3A%23ffffff%3B%22%20d%3D%22M17.6%2C20.4c-2%2C1-4.4%2C1.4-6.6%2C0.8c-2.3-0.6-3.4-2.3-3.4-4.5v-7H5.4V6.9c2.5-0.8%2C3.7-2.9%2C4-5.4h2.8%0D%0A%09%09v5.1h4.7v3.1h-4.7v5.1c0%2C1-0.2%2C2.4%2C0.9%2C3c1.4%2C0.8%2C3.2%2C0.2%2C4.5-0.6V20.4z%22%2F%3E%0D%0A%3C%2Fg%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-envelope {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cg%3E%0D%0A%09%3Cpath%20style%3D%22fill%3A%23ffffff%3B%22%20d%3D%22M22.6%2C9c0-0.2-0.1-0.5-0.3-0.6L12%2C0.8c-0.3-0.2-0.6-0.2-0.9%2C0L0.7%2C8.4C0.5%2C8.5%2C0.4%2C8.8%2C0.4%2C9v12.6%0D%0A%09%09c0%2C0.4%2C0.3%2C0.8%2C0.8%2C0.8h20.7c0.4%2C0%2C0.8-0.4%2C0.8-0.8V9z%20M11.5%2C2.4l9.6%2C7v0.9l-6.6%2C4.7l5.2%2C3.3c0.8%2C0.5%2C0%2C1.8-0.8%2C1.3L11.5%2C15%0D%0A%09%09l-7.4%2C4.7c-0.8%2C0.5-1.7-0.8-0.8-1.3l5.2-3.3l-6.6-4.7V9.4L11.5%2C2.4z%22%2F%3E%0D%0A%09%3Cpath%20style%3D%22fill%3A%23ffffff%3B%22%20d%3D%22M19.7%2C18.4%22%2F%3E%0D%0A%09%3Cpolygon%20style%3D%22fill%3A%23ffffff%3B%22%20points%3D%2211.5%2C13.1%2013.4%2C14.3%2019.5%2C9.9%2011.5%2C4%203.5%2C9.9%209.6%2C14.3%20%09%22%2F%3E%0D%0A%3C%2Fg%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-check {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cg%3E%0D%0A%09%0D%0A%09%09%3Crect%20x%3D%2211.9%22%20y%3D%222.1%22%20transform%3D%22matrix%28-0.7072%20-0.7071%200.7071%20-0.7072%2016.2216%2029.719%29%22%20style%3D%22fill%3A%23ffffff%3B%22%20width%3D%224.7%22%20height%3D%2218.8%22%2F%3E%0D%0A%09%0D%0A%09%09%3Crect%20x%3D%220.2%22%20y%3D%2211.3%22%20transform%3D%22matrix%280.7072%200.707%20-0.707%200.7072%2011.5962%20-0.598%29%22%20style%3D%22fill%3A%23ffffff%3B%22%20width%3D%2212.5%22%20height%3D%224.7%22%2F%3E%0D%0A%3C%2Fg%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n\n  /* icons site secondary */\n  .site-icon-clock {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cg%3E%0D%0A%09%3Cpath%20style%3D%22fill%3A%23bebebb%3B%22%20d%3D%22M11.5%2C22c-4.5%2C0-8.6-3-10-7.3c-1.4-4.3%2C0.2-9.2%2C4-11.8c3.7-2.6%2C8.9-2.5%2C12.5%2C0.3%0D%0A%09%09c3.6%2C2.8%2C5%2C7.7%2C3.4%2C12C19.8%2C19.3%2C15.8%2C22%2C11.5%2C22z%20M11.5%2C3.3C7%2C3.3%2C3.3%2C7%2C3.3%2C11.5c0%2C4.5%2C3.7%2C8.2%2C8.2%2C8.2c4.5%2C0%2C8.2-3.7%2C8.2-8.2%0D%0A%09%09C19.7%2C7%2C16%2C3.3%2C11.5%2C3.3z%22%2F%3E%0D%0A%09%3Cpath%20style%3D%22fill%3A%23ffffff%3B%22%20d%3D%22M12.9%2C12.9v-2.9V5.5c0-0.4-0.4-0.8-0.8-0.8h-1.3c-0.4%2C0-0.8%2C0.4-0.8%2C0.8v4.5H6.5%0D%0A%09%09c-0.4%2C0-0.8%2C0.4-0.8%2C0.8v1.3c0%2C0.4%2C0.4%2C0.8%2C0.8%2C0.8h3.5H12.9z%22%2F%3E%0D%0A%3C%2Fg%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-reply {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cg%3E%0D%0A%09%3Cpath%20style%3D%22fill%3A%23bebebb%3B%22%20d%3D%22M20.7%2C20c-0.1%2C0.3-0.3%2C0.7-0.4%2C0.9c-0.1%2C0.1-0.2%2C0.2-0.3%2C0.2c-0.2%2C0-0.4-0.2-0.4-0.4%0D%0A%09%09c0-0.2%2C0-0.4%2C0.1-0.6c0-0.5%2C0.1-1%2C0.1-1.5c0-5.7-3.4-6.7-8.6-6.7H8.4V15c0%2C0.4-0.3%2C0.8-0.8%2C0.8c-0.2%2C0-0.4-0.1-0.5-0.2L1%2C9.4%0D%0A%09%09C0.8%2C9.2%2C0.8%2C9%2C0.8%2C8.8c0-0.2%2C0.1-0.4%2C0.2-0.5l6.1-6.1C7.3%2C2%2C7.5%2C1.9%2C7.7%2C1.9c0.4%2C0%2C0.8%2C0.3%2C0.8%2C0.8v3.1h2.7%0D%0A%09%09c3.9%2C0%2C8.8%2C0.7%2C10.5%2C4.8c0.5%2C1.3%2C0.6%2C2.6%2C0.6%2C4C22.2%2C16.2%2C21.4%2C18.4%2C20.7%2C20z%22%2F%3E%0D%0A%3C%2Fg%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-remove {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cpath%20style%3D%22fill%3A%23bebebb%3B%22%20d%3D%22M19.6%2C17.5l-6-6l6-6c0.3-0.3%2C0.3-0.8%2C0-1.1l-0.9-0.9c-0.3-0.3-0.8-0.3-1.1%2C0l-6%2C6l-6-6%0D%0A%09c-0.3-0.3-0.8-0.3-1.1%2C0L3.4%2C4.4c-0.3%2C0.3-0.3%2C0.8%2C0%2C1.1l6%2C6l-6%2C6c-0.3%2C0.3-0.3%2C0.8%2C0%2C1.1l0.9%2C0.9c0.3%2C0.3%2C0.8%2C0.3%2C1.1%2C0l6-6l6%2C6%0D%0A%09c0.3%2C0.3%2C0.8%2C0.3%2C1.1%2C0l0.9-0.9C19.9%2C18.3%2C19.9%2C17.8%2C19.6%2C17.5z%22%2F%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-back {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cpath%20style%3D%22fill%3A%23bebebb%3B%22%20d%3D%22M11.5%2C20.9l1-1c0.3-0.3%2C0.3-0.7%2C0-0.9L6%2C12.5h14.6c0.4%2C0%2C0.7-0.3%2C0.7-0.7v-1.3c0-0.4-0.3-0.7-0.7-0.7%0D%0A%09H6l6.5-6.5c0.3-0.3%2C0.3-0.7%2C0-0.9l-1-1c-0.3-0.3-0.7-0.3-0.9%2C0l-9.4%2C9.4c-0.3%2C0.3-0.3%2C0.7%2C0%2C0.9l9.4%2C9.4%0D%0A%09C10.9%2C21.2%2C11.3%2C21.2%2C11.5%2C20.9z%22%2F%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-link {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cpath%20style%3D%22fill-rule%3Aevenodd%3Bclip-rule%3Aevenodd%3Bfill%3A%23bebebb%3B%22%20d%3D%22M19.3%2C8.2c-1.8%2C1.8-3.7%2C4.6-6.6%2C3.8c0.8-0.8%2C1.6-1.6%2C2.4-2.4%0D%0A%09c0.9-0.9%2C1.8-1.8%2C2.6-2.7c1.9-1.9-0.8-4.8-2.6-2.9c-0.5%2C0.5-0.9%2C0.9-1.4%2C1.4c-0.2%2C0.2-1.4%2C1.1-1.4%2C1.4l0%2C0c-0.8%2C0.8-1.6%2C1.6-2.4%2C2.4%0D%0A%09c-0.4-1.3%2C0-2.8%2C1-3.8c1.3-1.3%2C2.7-3.3%2C4.5-3.8C19.3%2C0.5%2C22.1%2C5.4%2C19.3%2C8.2%22%2F%3E%0D%0A%3Cpath%20style%3D%22fill-rule%3Aevenodd%3Bclip-rule%3Aevenodd%3Bfill%3A%23bebebb%3B%22%20d%3D%22M7.9%2C15.6C7%2C16.5%2C5.5%2C15%2C6.5%2C14.1c1.4-1.4%2C2.7-2.7%2C4.1-4.1%0D%0A%09c0.8-0.8%2C1.5-1.5%2C2.3-2.3c0.2-0.2%2C0.3-0.3%2C0.5-0.5c1-0.6%2C2%2C0.7%2C1.2%2C1.6C12.4%2C11.2%2C10.1%2C13.3%2C7.9%2C15.6z%22%2F%3E%0D%0A%3Cpath%20style%3D%22fill-rule%3Aevenodd%3Bclip-rule%3Aevenodd%3Bfill%3A%23bebebb%3B%22%20d%3D%22M5.8%2C13.4c-0.9%2C0.9-1.8%2C1.8-2.6%2C2.7c-1.9%2C1.9%2C0.8%2C4.8%2C2.6%2C2.9%0D%0A%09c0.9-0.9%2C2-1.8%2C2.8-2.9c0.8-0.8%2C1.6-1.6%2C2.4-2.4c0.4%2C1.3%2C0%2C2.8-1%2C3.8c-1.3%2C1.3-2.7%2C3.3-4.5%2C3.8c-3.8%2C1.1-6.7-3.8-3.9-6.6%0D%0A%09C3.4%2C13%2C5.4%2C10.2%2C8.2%2C11C7.4%2C11.8%2C6.6%2C12.6%2C5.8%2C13.4%22%2F%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-trash {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%221792px%22%20height%3D%221792px%22%20viewBox%3D%22590%20-500%201792%201792%22%20enable-background%3D%22new%20590%20-500%201792%201792%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cpath%20fill%3D%22%23bebebb%22%20d%3D%22M1294%2C876V172c0-9.3-3-17-9-23s-13.7-9-23-9h-64c-9.3%2C0-17%2C3-23%2C9s-9%2C13.7-9%2C23v704c0%2C9.3%2C3%2C17%2C9%2C23%0D%0A%09s13.7%2C9%2C23%2C9h64c9.3%2C0%2C17-3%2C23-9S1294%2C885.3%2C1294%2C876z%20M1550%2C876V172c0-9.3-3-17-9-23s-13.7-9-23-9h-64c-9.3%2C0-17%2C3-23%2C9%0D%0A%09s-9%2C13.7-9%2C23v704c0%2C9.3%2C3%2C17%2C9%2C23s13.7%2C9%2C23%2C9h64c9.3%2C0%2C17-3%2C23-9S1550%2C885.3%2C1550%2C876z%20M1806%2C876V172c0-9.3-3-17-9-23%0D%0A%09s-13.7-9-23-9h-64c-9.3%2C0-17%2C3-23%2C9s-9%2C13.7-9%2C23v704c0%2C9.3%2C3%2C17%2C9%2C23s13.7%2C9%2C23%2C9h64c9.3%2C0%2C17-3%2C23-9S1806%2C885.3%2C1806%2C876z%0D%0A%09%20M1262-116h448l-48-117c-4.7-6-10.3-9.7-17-11h-317c-6.7%2C1.3-12.3%2C5-17%2C11L1262-116z%20M2190-84v64c0%2C9.3-3%2C17-9%2C23s-13.7%2C9-23%2C9h-96%0D%0A%09v948c0%2C55.3-15.7%2C103.2-47%2C143.5s-69%2C60.5-113%2C60.5h-832c-44%2C0-81.7-19.5-113-58.5s-47-86.2-47-141.5V12h-96c-9.3%2C0-17-3-23-9%0D%0A%09s-9-13.7-9-23v-64c0-9.3%2C3-17%2C9-23s13.7-9%2C23-9h309l70-167c10-24.7%2C28-45.7%2C54-63s52.3-26%2C79-26h320c26.7%2C0%2C53%2C8.7%2C79%2C26%0D%0A%09s44%2C38.3%2C54%2C63l70%2C167h309c9.3%2C0%2C17%2C3%2C23%2C9S2190-93.3%2C2190-84z%22%2F%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-replay {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2223%22%20height%3D%2223%22%20viewBox%3D%220%200%2023%2023%22%3E%3Cpath%20d%3D%22M16.4%2016.9c-2.8%202.4-7.1%202.3-9.8-.3l-2%202c3.8%203.8%209.9%203.9%2013.9.3l-2.1-2z%22%20fill%3D%22%23b9bcc7%22%2F%3E%3Cpath%20d%3D%22M15.3%205.3l.9.6c3%202.6%203.7%207.2.9%2010.3l2%202c2.3-2.5%203.1-6%202.3-9.3-.9-3.6-3.8-6.4-7.4-7.3-3.8-.8-8.1.7-10.4%204l.1-1.2c0-.2%200-.5-.2-.7-.6-.8-1.7-.4-1.8.4l-.5%204.6c-.1.6.4%201.1%201%201.1h4.6c.9%200%201.3-1.1.8-1.7-.3-.2-.5-.3-.8-.3h-1.2c2-3.2%206.3-4.4%209.7-2.5z%22%20fill%3D%22%23b9bcc7%22%2F%3E%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-alt-download {\n    background-image: url(\'data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2223%22%20height%3D%2223%22%20viewBox%3D%220%200%2023%2023%22%3E%3Cg%20fill-rule%3D%22evenodd%22%20clip-rule%3D%22evenodd%22%20fill%3D%22%23ffffff%22%3E%3Cpath%20d%3D%22M17.3%2017.2c-1.1%200-1.4-1.5-.5-2%20.4-.2%201-.1%201.5-.3.6-.2%201-.6%201.4-1%201.3-1.6.5-4.1-1.4-4.8-.3-.1-.7-.2-1.1-.2-.2%200-.3-.8-.4-1-.4-.9-1-1.8-1.7-2.4-1.8-1.6-4.5-1.8-6.6-.5-1%20.5-1.7%201.4-2.2%202.3-.2.3-.3.7-.4%201-.2.7-.2.5-.8.7-1.9.4-3%202.5-2.2%204.3.4.9%201.3%201.6%202.3%201.8.5.1%201%200%201.3.4.4.4.3%201.1-.2%201.4-.7.6-2.2.1-2.9-.3-1-.5-1.8-1.3-2.3-2.3-.8-1.6-.7-3.5.2-4.9.4-.8%201.1-1.4%201.8-1.9.4-.2.9-.3%201.1-.7.2-.4.5-.8.7-1.2%202.1-3.1%206.2-4.4%209.6-2.9%201.6.7%203%201.9%203.8%203.4.2.4.3.9.7%201.1l1.2.6c.8.5%201.4%201.2%201.8%202%201.6%203.4-1%207.4-4.7%207.4%22%20fill%3D%22%23000000%22%3E%3C%2Fpath%3E%3Cpath%20d%3D%22M15.4%2017.2l-3.1%203.4c-.4.4-1.1.5-1.5%200-.5-.6-1.1-1.2-1.6-1.8-.2-.2-1.2-1.6-1.5-1.6-.9-1%20.7-2.4%201.5-1.4l1.3%201.4v-3.9c0-.4-.1-.9%200-1.3%200-1.3%202.1-1.3%202.1%200v5.2c.5-.5%201-1.4%201.7-1.7.8-.4%201.8.9%201.1%201.7z%22%20fill%3D%22%23000000%22%3E%3C%2Fpath%3E%3C%2Fg%3E%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-random {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2223%22%20height%3D%2223%22%20viewBox%3D%220%200%2023%2023%22%3E%3Cpath%20d%3D%22M5.7%2015.2v-6H8L4%204.6-.1%209.2h2.3v7c0%201.3%201%202.3%202.3%202.3H15l-3-3.2H5.7z%22%20fill%3D%22#ffffff%22%2F%3E%3Cpath%20d%3D%22M20.7%2013.8v-7c0-1.3-1-2.3-2.3-2.3H8l3%203.2h6.3v6H15l4.1%204.6%204.1-4.6h-2.5z%22%20fill%3D%22#ffffff%22%2F%3E%3C%2Fsvg%3E");\n    background-repeat: no-repeat;\n  }\n  .site-icon-visa {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2271%22%20height%3D%2243%22%20viewBox%3D%220%200%2071%2043%22%3E%3Cpath%20d%3D%22M51.9%2017.9c-.2.5-.5%201.4-.5%201.4l-1.5%204H53c-.2-.7-.9-4.2-.9-4.2l-.2-1.2zM69%200H2C.9%200%200%20.9%200%202v39c0%201.1.9%202%202%202h67c1.1%200%202-.9%202-2V2c0-1.1-.9-2-2-2zM11.6%2014h6c.8%200%201.5.3%201.7%201.2l1.3%206.6c-1.3-3.5-4.4-6.4-9.1-7.6l.1-.2zm7.1%2014.4l-3-11.5c2.5%201.5%204.2%203.5%205%205.9l.2%201%203.7-9.8h3.9l-5.9%2014.4h-3.9zm9%200L30.1%2014h3.7l-2.3%2014.4h-3.8zm10.2.2c-1.7%200-3.3-.4-4.2-.8l.5-3.2.5.2c1.2.5%202.4.8%203.8.8.9-.1%202.2-.6%201.9-1.7-.3-.9-1.6-1.3-2.3-1.7-1-.5-2-1.2-2.5-2.2-1-2%20.2-4.2%201.9-5.3%202.2-1.3%205.1-1.3%207.4-.4l-.5%203.1-.3-.2c-1-.4-4.9-1.3-4.9.7%200%20.8%201.2%201.3%201.8%201.6%201%20.5%202.1%201%202.8%202%201.1%201.4.8%203.5-.2%204.8-1.4%201.8-3.6%202.3-5.7%202.3zm16.2-.2s-.3-1.7-.4-2.2h-4.8c-.1.4-.8%202.2-.8%202.2h-3.9l5.5-13.2c.4-.9%201.1-1.2%202-1.2h2.9l3%2014.4h-3.5z%22%20fill%3D%22#ffffff%22%3E%3C%2Fpath%3E%3C%2Fsvg%3E");\n    background-repeat: no-repeat;\n  }\n  .site-icon-amex {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2271%22%20height%3D%2243%22%20viewBox%3D%220%200%2071%2043%22%3E%3Cpath%20d%3D%22M35.5%2023.5h-5l-2%202.2-1.9-2.2h-6.3v6.6h6.2l2-2.2%201.9%202.2h3.1v-2.2h2c1.5%200%202.7-.6%202.7-2.2%200-1.7-1.3-2.2-2.7-2.2zm-9.9%205.2h-3.9v-1.3h3.5V26h-3.5v-1.2h4l1.7%201.9-1.8%202zm6.3.8l-2.4-2.7%202.4-2.6v5.3zm3.6-3h-2v-1.7h2.1c1.2%200%201.2%201.7-.1%201.7zm9.4-7.8h1.8l.6-1.5h3.3l.6%201.5h3.2v-5l3%205h2.2v-6.6H58v4.6l-2.8-4.6h-2.4v6.2l-2.7-6.2h-2.4l-2.2%205.2H45c-1.6%200-2.1-2.3-1.1-3.4.3-.3.6-.4%201.2-.4h1.5v-1.4h-1.5c-2.1%200-3.2%201.3-3.2%203.3-.1%202%201%203.3%203%203.3zm4.1-5.5l1.1%202.6h-2.2l1.1-2.6zm-15.2%203.1h1.7c.6%200%20.9.1%201.1.3.3.3.2.9.2%201.3v.8h1.6v-1.3c0-.6%200-.9-.3-1.2-.1-.2-.4-.4-.8-.6.9-.4%201.3-1.4%201-2.3-.3-1.2-1.7-1.2-2.6-1.2h-3.6v6.6h1.6v-2.4zm0-2.8h1.9c.3%200%20.6%200%20.7.1.5.2.4%201%200%201.2-.2.1-.5.1-.8.1h-1.9v-1.4zm-2.7%203.8h-3.7V16H31v-1.3h-3.6v-1.2h3.7v-1.4h-5.3v6.6h5.3zM23%2018.7h1.6v-6.6H22l-1.9%204.5-2.1-4.5h-2.6v6.2l-2.7-6.2h-2.3l-2.8%206.6h1.7l.6-1.5h3.3l.6%201.5H17v-5.2l2.3%205.2h1.4l2.3-5.2v5.2zm-12.5-2.9l1.1-2.6%201.1%202.6h-2.2zm35.6%2014.3h5.3v-1.4h-3.7v-1.3h3.6V26h-3.6v-1.2h3.7v-1.3h-5.3zM61.8%2026H61c-.3%200-.6%200-.8-.2-.2-.2-.1-.6%200-.7l.1-.1c.2-.1.3-.1.6-.1h2.9v-1.4h-3.2c-1%200-1.6.4-1.9.8l-.1.1-.1.2c-.1.1-.1.3-.1.4v.7l.1.6c0%20.1%200%20.1.1.2%200%20.1.1.2.1.2l.1.1.1.1s.1%200%20.1.1c.1.1.2.1.3.1%200%200%20.1%200%20.1.1.7.2%201.4.2%202.2.2%201%200%20.9%201.2-.1%201.2h-3V30h3c1.1%200%202.2-.4%202.4-1.6.4-1.6-.6-2.3-2.1-2.4zm-16.5-.8c0-.8-.3-1.2-.8-1.5-.5-.3-1.1-.3-1.9-.3H39V30h1.6v-2.4h1.7c1.1%200%201.4.5%201.4%201.6v.8h1.6v-1.3c0-.8-.2-1.5-1-1.8.3-.1%201-.6%201-1.7zm-2.1%201c-.2.1-.5.1-.8.1h-1.9v-1.5h1.9c.3%200%20.6%200%20.8.1.5.3.4%201%200%201.3-.2.1.2-.2%200%200zM69%200H2C.9%200%200%20.9%200%202v39c0%201.1.9%202%202%202h67c1.1%200%202-.9%202-2V2c0-1.1-.9-2-2-2zM18.2%2018.9V20h-5.3l-.6-1.5h-1.4l-.6%201.5H5.5l4-9.2h4.1l.5%201.1v-1.1H19l1.1%202.4%201-2.4h15.3c.7%200%201.3.1%201.8.5v-.5h4.2v.5c.7-.4%201.6-.5%202.6-.5h6.1l.6%201.1v-1.1h4.5l.6%201.1v-1.1h4.4V20h-4.4l-.8-1.4V20h-5.5l-.6-1.5h-1.4l-.8%201.5h-2.9c-1.1%200-2-.3-2.5-.6v.6h-6.8v-2.1c0-.3-.1-.3-.2-.3H35V20H21.9v-1.1l-.5%201.1h-2.8l-.4-1.1zm46.9%2011c-.7%201.3-2%201.5-3.3%201.5h-4.4v-.6c-.5.4-1.3.6-2.1.6h-13v-2.1c0-.3%200-.3-.3-.3h-.2v2.4h-4.2v-2.5c-.7.3-1.5.3-2.2.3h-.5v2.2h-5.1L28.5%2030l-1.3%201.4h-8.3v-9.2h8.5l1.2%201.4%201.3-1.4h5.7c.7%200%201.7.1%202.2.5v-.5h5.1c.5%200%201.5.1%202.1.5v-.5h7.7v.5c.4-.4%201.2-.5%201.9-.5h4.3v.5c.4-.3%201-.5%201.8-.5h4.5v3.3h-.9c1.3.9%201.6%203%20.8%204.4-.4.7.3-.5%200%200zM39.4%2012.1H41v6.6h-1.6zM55.6%2026h-.8c-.5%200-1.5-.5-.7-1%20.2-.1.3-.1.6-.1h2.9v-1.4h-3.2c-1.5%200-2.6%201.2-2.1%202.7.4%201.3%202.1%201.2%203.2%201.3%201%200%20.9%201.2-.1%201.2h-3.1v1.4h3.1c1.1%200%202.2-.4%202.4-1.6.4-1.7-.7-2.4-2.2-2.5z%22%20fill%3D%22#ffffff%22%3E%3C%2Fpath%3E%3C%2Fsvg%3E");\n    background-repeat: no-repeat;\n  }\n  .site-icon-mc {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2271%22%20height%3D%2243%22%20viewBox%3D%220%200%2071%2043%22%3E%3Cpath%20d%3D%22M36.2%2022.5c.2-.9.3-2.2-.8-2.7-1.4-.7-3%20.1-3.6%201.5-.3.7-.3%201.5-.1%202.2.6%201.4%202.7%201.8%204%201.1l.2-1.1c-.9.2-3.1.5-2.9-1h3.2zm-1.8-1.9c.6%200%20.7.4.6.9h-1.8c.2-.6.6-.9%201.2-.9zm-3.8%202.3l.4-2.1h.7l.3-1.1h-.8l.3-1.4h-1.4l-1%204.9c-.3%201.3%201%202.1%202.2%201.4l.1-1c-.8%200-.8-.3-.8-.7zm8-2.7l-.1-.5h-1.2l-1%205.1h1.4l.4-2.3c.4-1.5%201.1-1.4%201.4-1.4l.4-1.4c-.7%200-1.1.2-1.3.5zm-10.2%201.9c-.2-.1-2.1-1-1-1.3.3%200%201%200%201.4.1l.3-1c-1.3-.5-3.7-.2-3.6%201.6.1%201.1%201.3.9%201.8%201.6.7.9-1.9.6-2.2.5l-.1%201c.8.2%202%20.3%202.8%200%201-.3%201.7-1.9.6-2.5-.3-.2.3.1%200%200zm14.5-2.4c.5%200%20.7%200%201.1.3l.9-.9c-1.7-1.9-4.5-.3-5.1%201.8-.7%202.7%201.4%204.8%204%203.6l-.4-1.3c-2.8%201-2.8-3.5-.5-3.5zM23%2019.6c-.4%200-.9%200-1.8.2l-.2%201.1c.6-.1%202.9-.7%202.6.6-1.1.1-2.2-.1-2.9%201-1.1%201.6.8%203.2%202.3%202.1l.2.3h1.3L25%2022c.2-.7.4-2.4-2-2.4zm.3%203.4c-.1.5-1.3%201-1.5.3-.3-.9%201.1-1%201.6-.9l-.1.6zm27.9-2.9l-.2-.5h-1.2l-1%205.2h1.4l.4-2.4c.4-1.5%201.1-1.5%201.4-1.5l.4-1.4c-.6.1-1%20.4-1.2.6zM69%200H2C.9%200%200%20.9%200%202v39c0%201.1.9%202%202%202h67c1.1%200%202-.9%202-2V2c0-1.1-.9-2-2-2zM45.1%2035.5c-3.4%200-6.4-1.2-8.8-3.1%201.8-1.8%203.2-4%204-6.5h-1.2c-.7%202.2-2%204.2-3.7%205.8-1.7-1.6-2.9-3.6-3.6-5.8h-1.2c.8%202.5%202.1%204.7%204%206.5-4%203.2-9.5%204.1-14.3%202-4-1.7-7.1-5.4-8.1-9.6C10%2015.4%2018%206.4%2027.5%207.6c2.6.3%205.1%201.4%207.1%203-1.8%201.8-3.2%204-4%206.5h1.2c.7-2.2%202-4.2%203.6-5.8%201.7%201.6%202.9%203.6%203.7%205.8h1.2c-.8-2.5-2.2-4.7-4-6.5%207.6-6.2%2019.4-2.5%2022.2%206.8%202.7%208.9-4.1%2018.1-13.4%2018.1zm7.2-13.7c-.5%201.6.5%203%201.5%203%20.5%200%20.9-.2%201.2-.4l.2.4h1.5l1.2-6.6h-1.5l-.4%202c-.7-.5-1.2-.8-2-.6-1%20.3-1.5%201.3-1.7%202.2-.5%201.6.7-2.4%200%200zm3.1-.8c.2.2.2.8.1%201.3-.2.4-.4.9-.9%201-.6.1-1-.4-1-1%20.1-.6%201.1-2.2%201.8-1.3.2.3-.4-.5%200%200zm-36.9-2.5l-1.7%203.8-.3-3.8h-2.2l-1.2%206.3h1.4l.8-4.5.4%204.5h1.4l1.9-4.4-.8%204.4h1.5l1.1-6.3zm28.4%201.1c-.4%200-.9%200-1.8.2l-.2%201.1c.6-.1%202.9-.7%202.6.6-1.1.1-2.2-.1-3%201-1.1%201.6.8%203.2%202.3%202.1l.2.3h1.3l.6-2.8c.1-.8.4-2.5-2-2.5zm.2%203.4c-.1.5-1.3%201-1.5.3-.3-.9%201.1-1%201.6-.9l-.1.6z%22%20fill%3D%22#ffffff%22%3E%3C%2Fpath%3E%3C%2Fsvg%3E");\n    background-repeat: no-repeat;\n  }\n  .site-icon-discover {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2271%22%20height%3D%2243%22%20viewBox%3D%220%200%2071%2043%22%3E%3Cpath%20d%3D%22M6.5%2012.1v6.5s4.2.5%204.2-3.2v-.1c-.1-3.6-4.2-3.2-4.2-3.2zM69%200H2C.9%200%200%20.9%200%202v39c0%201.1.9%202%202%202h67c1.1%200%202-.9%202-2V2c0-1.1-.9-2-2-2zm-9.8%2010.4h3.2c1.7%200%203.1%201.2%203.1%202.9%200%201.3-.8%202.5-2.1%202.9l3%204.2H64l-2.7-4h-.2v4h-1.9v-10zm-6.8%200H58v1.8h-3.5v2.2h3.4v1.8h-3.4v2.5H58v1.8h-5.5V10.4zm-8.2%200l2.7%206.5%202.6-6.5h2.1l-4.2%2010.2h-1.2L42%2010.4h2.2zm-31.5%205c0%203.2-2.9%204.9-4.8%204.9H4.6v-9.9h3.3c1.9%200%204.8%201.8%204.8%205zm2.8%204.9h-2v-10h2v10zm4.2.3c-2.5%200-3.6-1.9-3.6-1.9l1.2-1.2c.7%201%203.1%202.2%203.5.4.5-1.8-1.8-1.9-2.9-2.4-1.7-.9-1.8-3.4-.2-4.6%201.5-1.1%203.5-.6%204.8.6l-1.1%201.3c-.7-.8-3-1.2-2.9.4%200%201%202%201.3%202.7%201.7%203.2%201.3%201.6%205.7-1.5%205.7zm11.6-.6c-3.4%201.8-7.7-.8-7.7-4.6%200-3.9%204.2-6.4%207.7-4.6v2.4c-2.1-2.1-5.8-.8-5.8%202.3%200%203%203.7%204.4%205.8%202.3V20zm6-9.8c2.8%200%205.2%202.3%205.2%205.2%200%202.8-2.3%205.2-5.2%205.2-2.8%200-5.2-2.3-5.2-5.2%200-2.8%202.4-5.2%205.2-5.2zM68%2040H32.8C45%2036.8%2057.2%2032.4%2068%2025.7V40zm-4.4-26.5v-.1c0-.6-.5-1.4-1.3-1.4h-1.1v2.9h1.1c.8%200%201.3-.7%201.3-1.4z%22%20fill%3D%22#ffffff%22%3E%3C%2Fpath%3E%3C%2Fsvg%3E");\n    background-repeat: no-repeat;\n  }\n  .site-icon-jcb {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2271%22%20height%3D%2243%22%20viewBox%3D%220%200%2071%2043%22%3E%3Cpath%20d%3D%22M69%200H2C-.1%200%200%202.1%200%203.5v37.4C0%2042.6%201.4%2043%202.7%2043h66c1.4%200%202.4-.8%202.4-2.3V3c-.1-1.4-.3-3-2.1-3zM28.3%2030.8c0%202.4-1.9%204.4-4.3%204.6-2.1.3-4.3%200-6.5%200-.9%200-.8.1-.8-.8v-9c2.8.7%209.5%201.7%2010.4-2.2.3-1.2.1-2.6.1-3.8v-2.1c0-.4.1-.5-.3-.5h-3.7v3.4c0%201.5.3%203.5-1.5%204.2-.8.3-1.7.2-2.5%200-.9-.2-1.8-.5-2.6-.9-.1-.1%200-.5%200-.6v-6.4c0-2.1-.4-4.7.5-6.7.5-1%201.3-1.6%202.2-2.1%201.4-.6%208.7-.5%208.7-.3v15.1c.3%202.7.3%205.4.3%208.1zm13%20.1c0%202.4-2%204.4-4.4%204.6-.9.1-2%200-2.9%200h-4c-.4%200-.2-1.3-.2-1.6v-9.1c1.4%201.2%203.4%201.5%205.2%201.5%201.1%200%202.1%200%203.1-.2l1.2-.2c.1%200%20.6-.1.7-.2.2-.3%200-1.5%200-1.9v-.1c-1.4.7-3.4%201.4-5%201-3-.4-3.4-4.8-.8-6.1%201.9-.7%204.1-.1%205.8.7v-1.8c0-.3-.3-.3-.5-.3-.4-.1-.8-.2-1.2-.2-.9-.1-1.9-.2-2.8-.2-1.8%200-3.7.2-5.3%201.2l-.5.3v-3.5c0-1.2-.2-2.6.1-3.8.3-1.3%201.2-2.2%202.2-2.9.7-.4%201.4-.6%202.2-.6%202.3-.1%204.6%200%206.8%200%20.3%200%20.3%2021.2.3%2023.4zm12.9%200c0%202.5-2.1%204.5-4.5%204.6-2.3.1-4.6%200-6.9%200-.2%200-.1-6.1-.1-6.7V26h7c.4%200%20.7%200%201.1-.1%201.3-.3%202.5-1.4%202-2.9-.3-.9-1.3-1.4-2.3-1.6h-.3l1.1-.4c.7-.4%201.1-1.2%201-2-.1-1.2-1.2-1.8-2.3-1.9-1.8-.2-3.8%200-5.6%200h-1.6c-.3%200-.2-.3-.2-.6v-4.1c0-2.5%201.8-4.6%204.3-4.8%202.2-.2%204.4%200%206.6%200%20.6%200%20.7-.1.7.5v22.8zm-6.5-8.8c1.5.2%201.6%202.5%200%202.5h-2.3c-.1%200-.4.1-.4-.1v-2.4c0-.2.4-.1.5-.1.7.1%201.4.1%202.2.1zm-2.8-1.4v-2c0-.6.9-.3%201.3-.3.5%200%201.5-.2%201.9.3.7.5.4%201.7-.4%202-.8.2-1.9%200-2.8%200z%22%20fill%3D%22#ffffff%22%3E%3C%2Fpath%3E%3C%2Fsvg%3E");\n    background-repeat: no-repeat;\n  }\n  .site-icon-calendar {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2223%22%20height%3D%2223%22%20viewBox%3D%220%200%2023%2023%22%3E%3Cpath%20d%3D%22M19.3%202.8h-1.1v2.5h-4V2.8H8.8v2.5h-4V2.8H3.7c-1.4%200-2.5%201.1-2.5%202.5v14.9c0%201.4%201.1%202.5%202.5%202.5h15.6c1.4%200%202.5-1.1%202.5-2.5V5.3c0-1.4-1.1-2.5-2.5-2.5zm0%2017.4H3.7v-9.9h15.6v9.9zM7.7.3H5.9v4.2h1.7V.3zm9.4%200h-1.7v4.2h1.7V.3z%22%20fill%3D%22#ffffff%22%2F%3E%3Cg%20fill%3D%22#ffffff%22%3E%3Cpath%20d%3D%22M5.1%2011.7h1.4v1.4H5.1zM5.1%2014.5h1.4v1.4H5.1zM5.1%2017.4h1.4v1.4H5.1zM8%2011.7h1.4v1.4H8zM8%2014.5h1.4v1.4H8zM8%2017.4h1.4v1.4H8zM10.8%2011.7h1.4v1.4h-1.4zM10.8%2014.5h1.4v1.4h-1.4zM10.8%2017.4h1.4v1.4h-1.4zM13.6%2011.7H15v1.4h-1.4zM13.6%2014.5H15v1.4h-1.4zM13.6%2017.4H15v1.4h-1.4zM16.5%2011.7h1.4v1.4h-1.4zM16.5%2014.5h1.4v1.4h-1.4zM16.5%2017.4h1.4v1.4h-1.4z%22%2F%3E%3C%2Fg%3E%3C%2Fsvg%3E");\n    background-repeat: no-repeat;\n  }\n  .site-icon-credit {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2271%22%20height%3D%2243%22%20viewBox%3D%220%200%2071%2043%22%3E%3Cstyle%20type%3D%22text%2Fcss%22%3Ecircle%2C%20ellipse%2C%20line%2C%20path%2C%20polygon%2C%20polyline%2C%20rect%2C%20text%20%7B%20fill%3A%20#ffffff%20!important%3B%20%7D%3C%2Fstyle%3E%3Cpath%20d%3D%22M71%202c0-1.1-.9-2-2-2h-67c-1.1%200-2%20.9-2%202v8.6h71v-8.6zm-71%2018.4v20.6c0%201.1.9%202%202%202h67c1.1%200%202-.9%202-2v-20.6h-71zm17.7%2014.9h-9.6v-3.6h9.6v3.6zm17.3%200h-13.9v-3.6h13.9v3.6z%22%20fill%3D%22%23fff%22%2F%3E%3C%2Fsvg%3E");\n    background-repeat: no-repeat;\n  }\n  .site-icon-search {\n    background-image: url(\'data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2223%22%20height%3D%2223%22%20viewBox%3D%220%200%2023%2023%22%3E%3Cpath%20d%3D%22M21.8%2019.5L17.4%2015c-.6.9-1.5%201.7-2.4%202.4l4.5%204.4c1.4%201.6%203.8-.8%202.3-2.3z%22%20fill%3D%22%23bebebb%22%3E%3C%2Fpath%3E%3Cpath%20d%3D%22M15%2017.4c4.4-3%205.2-9.4%201.8-13.4C13.8.5%208.6-.3%204.7%202.3.9%204.7-.4%209.9%201.7%2014c2.2%204.4%207.7%206.2%2012.1%204.1.4-.2.8-.5%201.2-.7zm-5.2-1.7C6.7%2015.7%204%2013%204%209.8%204%206.7%206.7%204%209.8%204c3.2%200%205.8%202.7%205.8%205.8.1%203.2-2.6%205.9-5.8%205.9z%22%20fill%3D%22%23bebebb%22%3E%3C%2Fpath%3E%3Cpath%20d%3D%22M9.8%202.2c4.2%200%207.7%203.4%207.7%207.7%200%201.5-.5%203.1-1.4%204.3-.5.7-1%201.3-1.7%201.8l-.3.2c-1.2.9-2.8%201.3-4.3%201.3-4.2%200-7.7-3.4-7.7-7.7.1-4.2%203.5-7.6%207.7-7.6m0-1.5C4.7.7.6%204.8.6%209.9s4.1%209.2%209.2%209.2c1.8%200%203.7-.6%205.2-1.6.9-.6%201.7-1.4%202.3-2.4%201.1-1.5%201.6-3.3%201.6-5.2.1-5.1-4-9.2-9.1-9.2z%22%20fill%3D%22%23bebebb%22%3E%3C%2Fpath%3E%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-is-searching {\n     background-position: center center !important;\n     background-image: url(\'data:image/svg+xml;charset=utf-8,%3Csvg%20version%3D%221.1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%20%20viewBox%3D%220%200%2040%2040%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cpath%20opacity%3D%220.2%22%20fill%3D%22%23323232%22%20d%3D%22M20.201%2C5.169c-8.254%2C0-14.946%2C6.692-14.946%2C14.946c0%2C8.255%2C6.692%2C14.946%2C14.946%2C14.946%0D%0A%20%20s14.946-6.691%2C14.946-14.946C35.146%2C11.861%2C28.455%2C5.169%2C20.201%2C5.169z%20M20.201%2C31.749c-6.425%2C0-11.634-5.208-11.634-11.634%0D%0A%20%20c0-6.425%2C5.209-11.634%2C11.634-11.634c6.425%2C0%2C11.633%2C5.209%2C11.633%2C11.634C31.834%2C26.541%2C26.626%2C31.749%2C20.201%2C31.749z%22%2F%3E%0D%0A%3Cpath%20fill%3D%22%23323232%22%20d%3D%22M26.013%2C10.047l1.654-2.866c-2.198-1.272-4.743-2.012-7.466-2.012h0v3.312h0%0D%0A%20%20C22.32%2C8.481%2C24.301%2C9.057%2C26.013%2C10.047z%22%3E%0D%0A%20%20%3CanimateTransform%20attributeType%3D%22xml%22%0D%0A%20%20%20%20attributeName%3D%22transform%22%0D%0A%20%20%20%20type%3D%22rotate%22%0D%0A%20%20%20%20from%3D%220%2020%2020%22%0D%0A%20%20%20%20to%3D%22360%2020%2020%22%0D%0A%20%20%20%20dur%3D%220.5s%22%0D%0A%20%20%20%20repeatCount%3D%22indefinite%22%2F%3E%0D%0A%20%20%3C%2Fpath%3E%0D%0A%3C%2Fsvg%3E\');\n     background-repeat: no-repeat;\n     background-size: contain;\n     z-index: 999;\n     width: 80px;\n     height: 80px;\n     margin: 0 auto;\n  }\n  .site-icon-report {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3Csvg%20viewBox%3D%220%200%201792%202031%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%3E%3Cpath%20d%3D%22M1122.987%20238.933L1075.2%200H0v2030.933h238.933v-836.266h669.014l47.786%20238.933H1792V238.933z%22%20fill%3D%22%23bebebb%22%20fill-rule%3D%22evenodd%22%2F%3E%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n    background-size: contain;\n  }\n  .site-icon-block {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3Csvg%20viewBox%3D%220%200%201792%201792%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%3E%3Cpath%20d%3D%22M896%201612.8c-165.76%200-318.08-56.448-439.04-151.424L1461.376%20456.96A708.02%20708.02%200%200%201%201612.8%20896c0%20396.032-320.768%20716.8-716.8%20716.8M179.2%20896c0-396.032%20320.768-716.8%20716.8-716.8%20165.76%200%20318.08%2056.448%20439.04%20151.424L330.624%201335.04A708.02%20708.02%200%200%201%20179.2%20896M896%200C401.408%200%200%20401.408%200%20896s401.408%20896%20896%20896%20896-401.408%20896-896S1390.592%200%20896%200%22%20fill%3D%22%23ffffff%22%20fill-rule%3D%22evenodd%22%20fill-opacity%3D%22.6%22%2F%3E%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n    background-size: contain;\n  }\n</style>\n\n\n  <style>\n\n    /*\r\n * This file is for use on the Vimeo VHX platform.\r\n *\r\n * 1. prototype changes in browser dev tools using Vimeo VHX demo site\r\n * 2. edit this file locally\r\n * 3. git commit and push to the remote repo\r\n * 4. share for code review\r\n * 5. paste into "CUSTOM STYLES" field in the Vimeo VHX theme demo admin\r\n * 6. share for approval by Criterion team\r\n * 7. once approved and code reviewed, paste into "CUSTOM STYLES" field in the Vimeo VHX theme production admin\r\n *\r\n * Note for Vimeo engineers: please share any updates to this file to the Criterion.com team.\r\n */\r\n\r\n @import url(\'https://fonts.googleapis.com/css?family=Montserrat:400,500,600,700,900|Spectral:400,500,600,700\');\r\n\r\n p, div, span, a, h1, h2, h3, h4, h5, h6, .site-font-primary-family, .site-font-secondary-family {\r\n     font-family: \'Montserrat\', sans-serif !important;\r\n }\r\n\r\n div {\r\n     font-family: \'Montserrat\', sans-serif !important;\r\n }\r\n\r\n .browse-row h1 a {\r\n     font-family: \'Montserrat\', sans-serif !important;\r\n     font-weight: 700;\r\n     text-transform: uppercase;\r\n }\r\n\r\n .head-1, .head-2 {\r\n     font-weight: 700;\r\n }\r\n\r\n .browse-row h1.horizontal-row-header a {\r\n     font-size: 1rem !important;\r\n }\r\n\r\n .browse-link {\r\n     font-family: \'Montserrat\', sans-serif !important;\r\n }\r\n\r\n /*\r\n * Nav\r\n */\r\n\r\n .navigation-inner a {\r\n    opacity: 1;\r\n    font-family: \'Montserrat\', sans-serif !important;\r\n    -webkit-transition: all 0.3s ease;\r\n    -moz-transition: all 0.3s ease;\r\n    -ms-transition: all 0.3s ease;\r\n    -o-transition: all 0.3s ease;\r\n    transition: all 0.3s ease;\r\n}\r\n\r\n .navigation-inner.is-ready a {\r\n    opacity: 1;\r\n}\r\n\r\n  /*\r\n  * Other\r\n  */\r\n\r\n h1.gigantic.uppercase { text-transform: none !important; }\r\n .percentage-bar-container { display: none !important; }\r\n\r\n .slide-content {\r\n     bottom: 40px;\r\n }\r\n\r\n .slide-title {\r\n     font-size: 28px;\r\n     font-weight: 600;\r\n }\r\n\r\n .browse-item-title .subtext-container > div.site-font-primary-color.truncate {\r\n     white-space: normal;\r\n }\r\n\r\n .collection-description .read-more-wrap {\r\n     max-height: 120px !important;\r\n }\r\n\r\n .site-header nav a {\r\n     font-weight: 600;\r\n     letter-spacing: 0;\r\n     padding-left: 25px;\r\n }\r\n\r\n div.slide-subtitle {\r\n     display: none;\r\n }\r\n\r\n .slide-content {\r\n     bottom: 40px;\r\n }\r\n\r\n .collection-title {\r\n     font-weight: 600;\r\n }\r\n\r\n\r\n #watch-info h5 {\r\n     display: none;\r\n }\r\n\r\n .head.primary strong {\r\n     font-weight: 600;\r\n }\r\n\r\n .item-type-movie .play-outer-rectangle {\r\n     display: none;\r\n }\r\n\r\n /*\r\n  * Borders\r\n  */\r\n\r\n .border-bottom,\r\n .border-bottom-light {\r\n     border-bottom: none !important;\r\n }\r\n\r\n .border-top,\r\n .border-top-light {\r\n     border-top: none !important;\r\n }\r\n\r\n .horizontal-row {\r\n     border-top: none !important;\r\n     border-bottom: none !important;\r\n }\r\n\r\n /*\r\n  * Buttons\r\n  */\r\n\r\n .button {\r\n     border-radius: 0px;\r\n     border: none;\r\n     font-weight: 600;\r\n }\r\n\r\n .slide-button,\r\n .slide-button--alt {\r\n     border-radius: 0px !important;\r\n }\r\n\r\n .btn-site-primary,\r\n .btn-gray:link,\r\n .btn-dropdown-gray:link,\r\n .btn-gray:active,\r\n .btn-dropdown-gray:active,\r\n .btn-gray:visited,\r\n .btn-dropdown-gray:visited {\r\n     border: 1px solid #fff !important;\r\n     background-color: #fff !important;\r\n     color: #252525;\r\n     border-radius: 0px;\r\n     text-transform: uppercase;\r\n     line-height: 46px;\r\n     font-size: 13px;\r\n     transition: all 300ms ease-out;\r\n }\r\n .site-settings-billing-panel .btn-gray:link {\r\n   color: #252525 !important;\r\n   line-height: 33px;\r\n }\r\n .btn-site-primary,\r\n .btn-site-secondary {\r\n     line-height: 33px;\r\n }\r\n\r\n .btn-site-secondary,\r\n .btn-transparent,\r\n .btn-dropdown-transparent {\r\n     border: 1px solid #313131 !important;\r\n     background-color: #313131 !important;\r\n     color: #fff !important;\r\n     transition: all 300ms ease-out;\r\n     font-size: 13px;\r\n }\r\n\r\n .btn-gray:hover,\r\n .btn-dropdown-gray:hover,\r\n .btn-site-primary:hover,\r\n .btn-dropdown-transparent:hover,\r\n .btn-transparent:hover,\r\n .btn-site-secondary:hover {\r\n     border: 1px solid #CDCDCB !important;\r\n     background-color: #CDCDCB !important;\r\n }\r\n\r\n .icon-play-white {\r\n     background-image: url("data:image/svg+xml;charset=utf-8,%3Csvg xmlns=\'http://www.w3.org/2000/svg\' width=\'23\' height=\'23\' viewBox=\'0 0 23 23\'%3E%3Ctitle%3Eicons-ui-set copy%3C/title%3E%3Cpath d=\'M20.49 10.52L4.13.39C2.94-.4 2 .18 2 1.67v19.66c0 1.48 1 2.06 2.17 1.28l16.32-10.13a1.15 1.15 0 0 0 0-1.96z\' fill=\'%23252525\'/%3E%3C/svg%3E");\r\n }\r\n\r\n .btn-black:link,\r\n .btn-dropdown-black:link,\r\n .btn-black:active,\r\n .btn-dropdown-black:active,\r\n .btn-black:visited,\r\n .btn-dropdown-black:visited {\r\n     border: 1px solid #B9BCC6;\r\n     background-color: #B9BCC6;\r\n     color: #000;\r\n     border-radius: 0px;\r\n     text-transform: uppercase;\r\n     line-height: 46px;\r\n     font-size: 13px;\r\n }\r\n\r\n .btn-transparent,\r\n .btn-dropdown-transparent,\r\n .btn-transparent:link,\r\n .btn-dropdown-transparent:link,\r\n .btn-transparent:active,\r\n .btn-dropdown-transparent:active,\r\n .btn-transparent:visited,\r\n .btn-dropdown-transparent:visited {\r\n     border: none;\r\n     border-radius: 0px;\r\n     text-transform: uppercase;\r\n     line-height: 28px;\r\n     font-size: 13px;\r\n     padding: 10px 16px;\r\n     background-position: 16px 18px !important;\r\n }\r\n\r\n select.btn-dropdown-transparent {\r\n     padding: 0 20px 0 40px;\r\n }\r\n\r\n .coin {\r\n     border-radius: 4px;\r\n }\r\n\r\n /*\r\n  * Search link + collection card\r\n  */\r\n\r\n .browse-item-subtext {\r\n     display: none !important;\r\n }\r\n\r\n /*\r\n  * Search results\r\n  */\r\n\r\n .item-type-movie.js-collection-item .duration-container {\r\n     display: none !important;\r\n }\r\n\r\n /*\r\n  * Carousel\r\n  */\r\n\r\n .slide-content {\r\n     transform: translate3d(0,0,0);\r\n     top: auto;\r\n     bottom: 10%;\r\n     padding-left: 57px;\r\n }\r\n\r\n .slide-image-container:after,\r\n .slide-image-container:before {\r\n     display: none !important;\r\n }\r\n\r\n @media screen and (max-width: 39.9375em) {\r\n     .slide-content {\r\n         padding-left: 40px;\r\n         bottom: 13%;\r\n     }\r\n     .slide-bg--mobile .slide-bg-img {\r\n         float: left;\r\n         width: 100%;\r\n         height: auto;\r\n     }\r\n\r\n }\r\n\r\n /*\r\n  * Browse\r\n  */\r\n\r\n .browse-item-card.item-type-movie .duration-container {\r\n   display: none !important;\r\n }\r\n\r\n /*\r\n  * Collection Pages\r\n  */\r\n\r\n .vertical-bottom {\r\n     position: absolute;\r\n     bottom: 0;\r\n }\r\n\r\n .cover-art.columns {\r\n     padding-right: 0px !important;\r\n }\r\n\r\n select.js-sort-items,\r\n h2.collection-stats,\r\n h2.content-label,\r\n .extras-header {\r\n     display: none !important;\r\n }\r\n\r\n .browse-item-title .browse-image-container:before {\r\n     width: 102%;\r\n     height: 102%;\r\n }\r\n\r\n .browse-image-container:before {\r\n     background-image: none !important;\r\n }\r\n\r\n .media-identifier.media-episode {\r\n     display: none;\r\n }\r\n\r\n .read-more-trigger {\r\n     font-weight: bold;\r\n }\r\n\r\nsection.site-settings-purchases-panel .subscription-button {\r\n    background-color: #222 !important\r\n}\r\n\r\n.platforms-modal--container .platform-modal-devices-list-tv li:last-child {\r\n    display: none;\r\n}\r\n\r\n .collection-actions .margin-bottom-small {\r\n    margin-bottom: 0.9375rem !important;\r\n }\r\n .collection-actions .margin-right-small {\r\n    margin-right: 0.9375rem !important;\r\n }\r\n .collection-actions .btn-site-primary {\r\n    line-height: 46px;\r\n }\r\n\r\n/* Fix for button issue on pause / cancellation modal */\r\nbody.settings.is-modal-active .btn-site-secondary,\r\nbody.settings.is-modal-active .btn-site-primary {\r\n    line-height: 38px !important;\r\n    color: #252525 !important;\r\n}\r\n\r\n.site-header .user-dropdown {\r\n    width: 190px;\r\n}\r\n\r\n.tooltip.background-white span.media-count, .tooltip.background-white span.media-identifier, .tooltip.background-white span.media-seperator { display: none !important; }\r\n\r\n.tooltip.background-white .tooltip-item-title { margin-bottom: 15px; }\r\n\r\ndiv[id^="collection-tooltip"] .tooltip-item-title { margin-bottom: 0px !important; }\n\n\n    html.wf-loading { opacity: 0; }\n    html.wf-active, html.is-loaded { opacity: 1 !important; }\n  </style>\n\n  <script>\n  var Tracking = undefined;\n  window.Page = undefined;\n  var Segment_ID = undefined;\n  window._current_user = undefined;\n  window._current_site = undefined;\n</script>\n\n\n  <script>\n    window._vhx = {\n      environment: \'production\',\n      site: { id: \'59054\', domain: \'www.criterionchannel.com\', host: \'criterionchannel.vhx.tv\', previewUrl: \'https://www.criterionchannel.com/login\' },\n      geo:  { country: \'us\' },\n      product: {id: \'\' }\n    };\n\n    window._current_user = {"id":null,"email":null,"name":null,"city":null,"state":null,"country":null,"external_user_id":null};\n\n    window._current_site = {"id":59054,"facebook_pixel_codes":[{"pixel_code":"980569266079632","product_id":null}],"twitter_pixel_codes":[],"adwords_pixel_codes":[],"key":"criterionchannelchartersu","title":"The Criterion Channel","subdomain":"criterionchannel","made_for_kids":false,"made_for_kids_tracking_blocked":false,"algolia_search_enabled":true,"passwords_disabled":false,"segment_heartbeat_frequency":0,"segment_includes_user_id":false,"sso_enabled":false};\n  </script>\n\n<script>\n  window.Tracking = {\n    EVENTS:     {"EVENT_SALE_INITIATED":"sale_initiated","EVENT_SALE_COMPLETE":"sale_complete","EVENT_SALE_COMPLETE_GIFT":"sale_conversion-gift","EVENT_SALE_ERROR":"sale_error","EVENT_SALE_COMPLETE_SAVE_CARD":"sale_complete_save_card","EVENT_SALE_MISSING_EMAIL":"sale_missing_email","EVENT_SALE_UNCONFIRMED_EMAIL":"sale_unconfirmed_email","EVENT_SALE_MISSING_CC":"sale_missing_cc","EVENT_SALE_MISSING_CVC":"sale_missing_cvc","EVENT_SALE_MISSING_EXP":"sale_missing_exp","EVENT_SALE_MISSING_ZIP":"sale_missing_zip","EVENT_SALE_INVALID_CC":"sale_invalid_cc","EVENT_SALE_INVALID_CVC":"sale_invalid_cvc","EVENT_SALE_INVALID_EXP":"sale_invalid_exp","EVENT_SALE_INVALID_ZIP":"sale_invalid_zip","EVENT_SALE_INVALID_COUPON":"sale_invalid_coupon","EVENT_SALE_INACTIVE_COUPON":"sale_inactive_coupon","EVENT_SALE_GEOBLOCKED_COUPON":"sale_geoblocked_coupon","EVENT_SALE_EXPIRED_CC":"sale_expired_cc","EVENT_SALE_PLAY_TRAILER":"sale_play_trailer","EVENT_SALE_PROCESSING_ERROR":"sale_processing_error","EVENT_SALE_UNKNOWN_ERROR":"sale_unknown_error","EVENT_SALE_INCOMPLETE_FORM":"sale_incomplete_form","EVENT_SALE_SHARE_FACEBOOK":"sale_share_facebook","EVENT_SALE_SHARE_TWITTER":"sale_share_twitter","EVENT_SALE_SHARE_EMAIL":"sale_share_email","EVENT_SALE_START_WATCHING":"sale_start_watching","EVENT_SALE_SAVE_CARD":"sale_save_card","EVENT_SALE_SHOW_PAYMENT_TAB":"sale_show_payment_tab","EVENT_SALE_SHOW_DETAILS_TAB":"sale_show_details_tab","EVENT_ADMIN_SIGNUP":"admin_signup","EVENT_ADMIN_ADD_SITE":"admin_add_site","EVENT_ADMIN_PUBLISH_THEME":"admin_publish_theme","EVENT_ADMIN_CREATE_VIDEO":"admin_create_video","EVENT_ADMIN_UPLOAD_VIDEO":"admin_upload_video","EVENT_ADMIN_UPLOAD_VIDEO_NATIVE":"admin_upload_video_native","EVENT_ADMIN_UPLOAD_VIDEO_DROPBOX":"admin_upload_video_dropbox","EVENT_ADMIN_TRANSCODE_VIDEO":"admin_transcode_video","EVENT_ADMIN_CREATE_EXTRA":"admin_create_extra","EVENT_ADMIN_UPLOAD_EXTRA":"admin_upload_extra","EVENT_ADMIN_UPLOAD_EXTRA_NATIVE":"admin_upload_video_native","EVENT_ADMIN_CONNECT_PAYOUT":"admin_connect_payout","EVENT_ADMIN_CREATE_SCREENER":"admin_create_screener","EVENT_ADMIN_CREATE_COUPON":"admin_create_coupon","EVENT_ADMIN_CREATE_UPDATE":"admin_create_update","EVENT_ADMIN_PUBLISH_UPDATE":"admin_publish_update","EVENT_ADMIN_CREATE_PARTNER":"admin_create_partner","EVENT_ADMIN_UPLOADED_FOLLOWERS_CS":"admin_uploaded_followers_csv","EVENT_ADMIN_ADDED_FOLLOWERS":"admin_added_followers","EVENT_ADMIN_CREATE_SUBSCRIPTION":"admin_create_subscription","EVENT_ADMIN_LINK_HELP":"admin_link_help","EVENT_ADMIN_LINK_FAQ":"admin_link_faq","EVENT_ADMIN_LINK_GETTING_STARTED":"admin_link_getting_started","EVENT_ADMIN_LINK_HOWTO":"admin_link_howto","EVENT_ADMIN_LINK_BEST_PRACTICES":"admin_link_best_practices","EVENT_ADMIN_LINK_DEV_DOCS":"admin_link_dev_docs","EVENT_FORUM_CREATE_THREAD":"forum_create_thread","EVENT_FORUM_CREATE_POST":"forum_create_post","EVENT_UPDATE_COMMENT":"comment_update","EVENT_CONCURRENT_LIMIT":"concurrent_limit","EVENT_VIDEO_COMMENT":"video_comment","EVENT_SITE_HERO_WATCH_TRAILER":"site_hero_watch_trailer","EVENT_SITE_HERO_BUY":"site_hero_buy","EVENT_SITE_FOOTER_BUY":"site_footer_buy","EVENT_SITE_FOOTER_IOS":"site_footer_ios","EVENT_SITE_BUY":"site_buy","EVENT_SITE_UNAVAILABLE_REQUEST":"site_unavailable_request","EVENT_SITE_SEARCH":"site_search","EVENT_SITE_SEARCH_EMPTY":"site_search_empty","EVENT_SITE_SEARCH_VIDEO_CLICK":"search_video","EVENT_SITE_SEARCH_COLLECTION_CLICK":"search_collection","EVENT_SITE_SEARCH_PRODUCT_CLICK":"search_product","EVENT_SITE_SEARCH_ITEM_SELECTION":"site_search_item_selected","EVENT_SITE_VIDEO_BUY":"site_video_buy","EVENT_SITE_VIDEO_WATCH_TRAILER":"site_video_watch_trailer","EVENT_SITE_COLLECTION_BUY":"site_collection_buy","EVENT_SITE_PRODUCT_PREORDER":"site_product_preorder","EVENT_SITE_PRODUCT_BUY":"site_product_buy","EVENT_SITE_PRODUCT_RENT":"site_product_rent","EVENT_SITE_CAROUSEL_NEXT":"site_carousel_next","EVENT_SITE_CAROUSEL_PREV":"site_carousel_prev","EVENT_SITE_CAROUSEL_COIN":"site_carousel_coin","EVENT_SITE_CAROUSEL_WATCH_NOW":"site_carousel_watch_now","EVENT_SITE_CAROUSEL_TRAILER":"site_carousel_trailer","EVENT_SITE_HOME":"site_home","EVENT_SITE_BROWSE":"site_browse","EVENT_SITE_SETTINGS":"site_settings","EVENT_SITE_FORUMS":"site_forums","EVENT_SITE_SUPPORT":"site_help","EVENT_SITE_CONTACT":"site_contact","EVENT_SITE_CONTACT_HELP":"site_contact_help","EVENT_SITE_VIEW_PRODUCTS":"site_products","EVENT_SITE_VIEW_PRODUCT":"site_product","EVENT_SITE_COLLECTION_CLICK":"site_collection","EVENT_SITE_VIDEO_CLICK":"site_video","EVENT_SITE_EXTRA_CLICK":"site_extras","EVENT_SITE_CATEGORY_CLICK":"site_category","EVENT_SITE_MOVIE_CLICK":"site_movie","EVENT_SITE_PLAYLIST_CLICK":"site_playlist","EVENT_SITE_SERIES_CLICK":"site_series","EVENT_SITE_SEASON_CLICK":"site_season","EVENT_SITE_TERMS":"site_tos","EVENT_SITE_PRIVACY":"site_privacy","EVENT_SITE_COOKIES":"site_cookies","EVENT_SITE_COPYRIGHT":"site_copyright","EVENT_SITE_DATA_PROCESSING":"site_data_processing","EVENT_SITE_WATCHLIST_REMOVE":"site_removewatchlist","EVENT_SITE_WATCHLIST_ADD":"site_addwatchlist","EVENT_SITE_PAGE_VIEW":"site_page_view","EVENT_SITE_CONTEXT_HOMEPAGE":"site_context_homepage","EVENT_SITE_CONTEXT_COLLECTION":"site_context_collection","EVENT_SITE_CONTEXT_CATEGORY":"site_context_category","EVENT_SITE_CONTEXT_MOVIE":"site_context_movie","EVENT_SITE_CONTEXT_PLAYLIST":"site_context_playlist","EVENT_SITE_CONTEXT_SERIES":"site_context_series","EVENT_SITE_CONTEXT_UPNEXT":"site_context_upnext","EVENT_AUTHENTICATION_LOGIN":"authentication_login","EVENT_AUTHENTICATION_ACTIVATION_STARTED":"activationstarted","EVENT_AUTHENTICATION_ACTIVATION_COMPLETE":"activation","EVENT_AUTHENTICATION_SIGNIN_VIEW":"signin","EVENT_AUTHENTICATION_SIGNIN_STARTED":"signinstarted","EVENT_AUTHENTICATION_SIGNIN_COMPLETE":"authentication","EVENT_AUTHENTICATION_SIGNUP_VIEW":"signup","EVENT_AUTHENTICATION_SIGNUP_STARTED":"signupstarted","EVENT_AUTHENTICATION_PURCHASE_STARTED":"purchasestarted","EVENT_AUTHENTICATION_PURCHASE_COMPLETED":"purchasecompleted","EVENT_AUTHENTICATION_CONVERSION":"conversion","EVENT_AUTHENTICATION_REGISTRATION":"register","EVENT_AUTHENTICATION_IDENTIFY_USER":"identify_user","EVENT_SETTINGS_SAVE_PROFILE":"settings_save_profile","EVENT_SETTINGS_SAVE_NOTIFICATIONS":"settings_save_notifications","EVENT_SETTINGS_SAVE_CARD":"settings_save_card","EVENT_DEVICES_DOWNLOAD_IOS":"devices_download_ios","EVENT_DEVICES_DOWNLOAD_APPLETV":"devices_download_appletv","EVENT_DEVICES_DOWNLOAD_ANDROID":"devices_download_android","EVENT_DEVICES_DOWNLOAD_ANDROIDTV":"devices_download_androidtv","EVENT_DEVICES_DOWNLOAD_AMAZON_FIRE":"devices_download_amazon_fire","EVENT_DEVICES_DOWNLOAD_XBOX":"devices_download_xbox","EVENT_DEVICES_DOWNLOAD_ROKU":"devices_download_roku","EVENT_DEVICES_DOWNLOAD_TIZEN":"devices_download_tizen","EVENT_DEVICES_DOWNLOAD_CHROMECAST":"devices_download_chromecast","EVENT_NEWSLETTER_SIGNUP":"event_newsletter_signup","EVENT_METADATA_ADVISORY":"metadata_click_advisory","EVENT_METADATA_CAST":"metadata_click_cast","EVENT_METADATA_CREW":"metadata_click_crew","EVENT_METADATA_GENRES":"metadata_click_genres","EVENT_METADATA_RATING":"metadata_click_rating","EVENT_METADATA_TAGS":"metadata_click_tags","EVENT_SEGMENT_AUTHENTICATION_CONVERSION":"Order Completed","EVENT_SEGMENT_AUTHENTICATION_PURCHASE_STARTED":"Checkout Started","EVENT_SEGMENT_AUTHENTICATION_REGISTRATION":"Registration Completed","EVENT_SEGMENT_AUTHENTICATION_SIGNIN_COMPLETE":"Sign In Complete","EVENT_SEGMENT_AUTHENTICATION_SIGNIN_VIEW":"Sign In Viewed","EVENT_SEGMENT_AUTHENTICATION_SIGNUP_STARTED":"Signed Up","EVENT_SEGMENT_AUTHENTICATION_SIGNUP_VIEW":"Sign Up Viewed","EVENT_SEGMENT_EXTRA_VIEWED":"Extra Content Viewed","EVENT_SEGMENT_SEARCH_EXECUTED":"Search Executed","EVENT_SEGMENT_SEARCH_ITEM_SELECTED":"Search Result Selected","EVENT_SEGMENT_VIDEO_AD_START":"Video Ad Started","EVENT_SEGMENT_VIDEO_AD_CLICKED":"Video Ad Clicked","EVENT_SEGMENT_VIDEO_AD_FINISH":"Video Ad Completed","EVENT_SEGMENT_VIDEO_AD_FAIL":"Video Ad Failed","EVENT_SEGMENT_VIDEO_AD_SKIP":"Video Ad Skipped","EVENT_SEGMENT_VIDEO_COMMENT_ADDED":"Video Comment Added","EVENT_SEGMENT_VIDEO_COMPLETE_PROGRESS_PERCENTILE":"Video Content Completed","EVENT_SEGMENT_VIDEO_ENDED":"Video Content Ended","EVENT_SEGMENT_VIDEO_END_BUFFER":"Video Playback Buffer Completed","EVENT_SEGMENT_VIDEO_HALF_PROGRESS_PERCENTILE":"50% Content Completed","EVENT_SEGMENT_VIDEO_PAUSE":"Video Playback Paused","EVENT_SEGMENT_VIDEO_PLAY":"Video Playback Started","EVENT_SEGMENT_VIDEO_QUARTER_PROGRESS_PERCENTILE":"25% Content Completed","EVENT_SEGMENT_VIDEO_RESUME":"Video Playback Resumed","EVENT_SEGMENT_VIDEO_SEEKED":"Video Playback Seek Completed","EVENT_SEGMENT_VIDEO_SEEKING":"Video Playback Seek Started","EVENT_SEGMENT_VIDEO_START_BUFFER":"Video Playback Buffer Started","EVENT_SEGMENT_VIDEO_TIMEUPDATE":"Video Content Playing","EVENT_SEGMENT_VIDEO_THIRD_PROGRESS_PERCENTILE":"75% Content Completed","EVENT_SEGMENT_VIDEO_WATCHLIST_ADD":"Added to Watch List","EVENT_SEGMENT_VIDEO_WATCHLIST_REMOVE":"Removed from Watch List","EVENT_SEARCH":"search","EVENT_SEARCH_ITEM_SELECTED":"searchitemselected","EVENT_EXTRA_VIEWED":"extraviewed"},\n    PROPERTIES: {"PROPERTY_COLLECTOR_URL":"https://collector.vhx.tv/pixel.gif","PROPERTY_COLLECTION_ID":"collection_id","PROPERTY_DEVICE":"device","PROPERTY_DEVICE_ID":"device_id","PROPERTY_NAME":"name","PROPERTY_PLATFORM":"platform","PROPERTY_PLATFORM_ID":"platform_id","PROPERTY_PLATFORM_VERSION":"platform_version","PROPERTY_PRODUCT_ID":"product_id","PROPERTY_REFERRER":"referrer","PROPERTY_SESSION_ID":"session_id","PROPERTY_SITE_ID":"site_id","PROPERTY_TIMESTAMP":"timestamp","PROPERTY_TYPE":"type","PROPERTY_URL":"url","PROPERTY_USER_AGENT":"user_agent","PROPERTY_USER_EMAIL":"user_email","PROPERTY_USER_ID":"user_id","PROPERTY_VIDEO_ID":"video_id","PROPERTY_VIEW":"view"},\n    DEPRECATED_EVENTS: ["admin_connect_payout","admin_create_coupon","admin_create_screener","admin_link_dev_docs","admin_link_faq","admin_link_help","admin_publish_theme","metadata_click_cast","metadata_click_crew","metadata_click_genres","metadata_click_rating","metadata_click_tags","metadata_click_advisory"],\n    UPDATED_EVENTS: {"search_video":"searchitemselected","search_collection":"searchitemselected","site_search_item_selected":"searchitemselected","site_extras":"extraviewed"},\n    SITE_GANALYTICS_ENABLED: false,\n    SESSION: \'4a080361f97a5621442775a417f88f7c\'\n  };\n</script>\n  <script>\n    window.Page = {"PROPERTIES":{"VIEW_TYPE":"video","VIDEO_ID":533128,"COLLECTION_ID":143189,"COLLECTION_TITLE":"Foreign-Language Oscar Winners","PRODUCT_ID":39621,"VIDEO_TITLE":"WAR AND PEACE: Part 2","CANONICAL_COLLECTION":null}}\n  </script>\n\n\n\n  <script>\n  var _gaq = _gaq || [];\n\n  (function(i,s,o,g,r,a,m){i[\'GoogleAnalyticsObject\']=r;i[r]=i[r]||function(){\n  (i[r].q=i[r].q||[]).push(arguments)},i[r].l=1*new Date();a=s.createElement(o),\n  m=s.getElementsByTagName(o)[0];a.async=1;a.src=g;m.parentNode.insertBefore(a,m)\n  })(window,document,\'script\',\'//www.google-analytics.com/analytics.js\',\'ga\');\n\n      ga(\'create\', \'UA-19540423-6\', \'auto\');\n    ga(\'require\', \'displayfeatures\');\n\n      ga(\'set\', \'dimension1\', \'criterionchannelchartersu\');\n\n    ga(\'send\', \'pageview\');\n\n\n      (function() {\n        var vhx = document.createElement(\'script\'); vhx.type = \'text/javascript\'; vhx.async = true;\n        vhx.src = \'https://cdn.vhx.tv/assets/tracker-be253e44a9312b41e422e9acec66d157e9c650341ea1701c01c2c8b31b26b9e6.js\';\n        var s = document.getElementsByTagName(\'script\')[0]; s.parentNode.insertBefore(vhx, s);\n      })();\n\n\n    Tracking.SITE_GANALYTICS_ENABLED = true;\n\n      ga(\'create\', \'UA-1852531-11\', \'auto\', { \'name\': \'site\', \'allowLinker\': true});\n\n    ga(\'site.require\', \'displayfeatures\');\n      ga(\'site.require\', \'linker\');\n      ga(\'site.linker:autoLink\', [\'www.criterionchannel.com\',\'criterionchannel.vhx.tv\'] );\n    ga(\'site.set\', \'dimension1\', \'criterionchannelchartersu\');\n    ga(\'site.send\', \'pageview\');\n\n    // <!-- Google Tag Manager -->\n    (function(w,d,s,l,i){w[l]=w[l]||[];w[l].push({\'gtm.start\':\n    new Date().getTime(),event:\'gtm.js\'});var f=d.getElementsByTagName(s)[0],\n    j=d.createElement(s),dl=l!=\'dataLayer\'?\'&l=\'+l:\'\';j.async=true;j.src=\n    \'https://www.googletagmanager.com/gtm.js?id=\'+i+dl;f.parentNode.insertBefore(j,f);\n    })(window,document,\'script\',\'dataLayer\',\'GTM-PKL3NF6\');\n    // <!-- End Google Tag Manager -->\n\n  if (window._current_site && window._current_site.facebook_pixel_codes.length > 0) {\n    // <!-- Facebook Pixel Code -->\n    !function(f,b,e,v,n,t,s)\n    {if(f.fbq)return;n=f.fbq=function(){n.callMethod?\n    n.callMethod.apply(n,arguments):n.queue.push(arguments)};\n    if(!f._fbq)f._fbq=n;n.push=n;n.loaded=!0;n.version=\'2.0\';\n    n.queue=[];t=b.createElement(e);t.async=!0;\n    t.src=v;s=b.getElementsByTagName(e)[0];\n    s.parentNode.insertBefore(t,s)}(window, document,\'script\',\n    \'https://connect.facebook.net/en_US/fbevents.js\');\n    //<!-- End Facebook Pixel Code -->\n  }\n\n  if (window._current_site && window._current_site.twitter_pixel_codes.length > 0) {\n    // <!-- Twitter Pixel Code -->\n    !function(e,t,n,s,u,a){e.twq||(s=e.twq=function(){s.exe?s.exe.apply(s,arguments):s.queue.push(arguments);\n    },s.version=\'1.1\',s.queue=[],u=t.createElement(n),u.async=!0,u.src=\'//static.ads-twitter.com/uwt.js\',\n    a=t.getElementsByTagName(n)[0],a.parentNode.insertBefore(u,a))}(window,document,\'script\');\n    //<!-- End Twitter Pixel Code -->\n  }\n\n      Segment_ID = \'D22V0IgjPYTwUKXVEtSzbmfcnolk4CNB\';\n      !function(){var analytics=window.analytics=window.analytics||[];if(!analytics.initialize)if(analytics.invoked)window.console&&console.error&&console.error("Segment snippet included twice.");else{analytics.invoked=!0;analytics.methods=["trackSubmit","trackClick","trackLink","trackForm","pageview","identify","reset","group","track","ready","alias","debug","page","once","off","on"];analytics.factory=function(t){return function(){var e=Array.prototype.slice.call(arguments);e.unshift(t);analytics.push(e);return analytics}};for(var t=0;t<analytics.methods.length;t++){var e=analytics.methods[t];analytics[e]=analytics.factory(e)}analytics.load=function(t,e){var n=document.createElement("script");n.type="text/javascript";n.async=!0;n.src="https://cdn.segment.com/analytics.js/v1/"+t+"/analytics.min.js";var a=document.getElementsByTagName("script")[0];a.parentNode.insertBefore(n,a);analytics._loadOptions=e};analytics.SNIPPET_VERSION="4.1.0";\n      analytics.load("D22V0IgjPYTwUKXVEtSzbmfcnolk4CNB");\n      }}();\n  </script>\n\n\n\n    \n\n  <script>\n  if (/ticket=/.test(location.search) && window.history.replaceState) {\n    var url = location.href.replace(/\\?([^#]*)/, function(_, search) {\n      search = search.split(\'&\').map(function(v) {\n        return !/^ticket=/.test(v) && !/^support=/.test(v) && !/^sso=/.test(v) && v;\n      }).filter(Boolean).join(\'&\');\n      return search ? \'?\' + search : \'\';\n    });\n\n    if (url != location.href) {\n      window.history.replaceState({}, \'\', url);\n    }\n  }\n\n  if (/return_to=/.test(location.search) && window.history.replaceState) {\n    var url = location.href.replace(/\\?([^#]*)/, function(_, search) {\n      search = search.split(\'&\').map(function(v) {\n        return !/^return_to=/.test(v) && v;\n      }).filter(Boolean).join(\'&\');\n      return search ? \'?\' + search : \'\';\n    });\n\n    if (url != location.href) {\n      window.history.replaceState({}, \'\', url);\n    }\n  }\n\n  if (/logout=/.test(location.search) && window.history.replaceState) {\n    var url = location.href.replace(/\\?([^#]*)/, function(_, search) {\n      search = search.split(\'&\').map(function(v) {\n        return !/^logout=/.test(v) && v;\n      }).filter(Boolean).join(\'&\');\n      return search ? \'?\' + search : \'\';\n    });\n\n    if (url != location.href) {\n      window.history.replaceState({}, \'\', url);\n    }\n  }\n</script>\n\n\n  <script>\n    window.TOKEN = "eyJhbGciOiJSUzI1NiIsImtpZCI6ImQ2YmZlZmMzNGIyNTdhYTE4Y2E2NDUzNDE2ZTlmZmRjNjk4MDAxMDdhZTQ2ZWJhODg0YTU2ZDBjOGQ4NTYzMzgifQ.eyJhcHBfaWQiOjY4NTExLCJleHAiOjE2NzU0NTU0MDMsIm5vbmNlIjoiNGI5MjY4ZTdjMTM0NTdhOSIsInNjb3BlcyI6WyJwdWJsaWMiXSwic2Vzc2lvbl9pZCI6Ik1ieHNtT3RxelNWRytGaXQ5Szd6d2c9PSIsInNpdGVfaWQiOjU5MDU0fQ.C4rkmFBWRVhGngmKbcCVxRQjvxf0Vp5VUXKhFudt_DmKwIjj2PRvyLKvez5hFjOlMUGNVNke7vEEyl106eyRP9LvVi5L1BmHDv_YuGbSl99A6YWBFG-Nlm7oDgQO43eWf2J5SyLrWToAB541T-tWHsQj6xHEbOyptUknIvs4sZGzGpOz8VAGpQme9Evqb_ailtbaKW9pgR6Rm2EBw1k_ZpFBhpRkd26Fa2WiZSInqqG8kbWh_2IWPVxEFF7R6AaST7wF89aYdp6QcCvxiL8dglBQJ7FHulrUwCKGpAY4AvGuBGN0m8yRLkId0eGH9GLOFbbjPjJYUnSfHGV7qxWk0zqj_mRs2E5_UEpWmKaLqdSyANI6gl_vKiV_H_OGIncoqceqo1yUQktGCIEfpAwATFJNKv_gDHCAKPMFQv2wrHRkigb4rO7e8pmJ3H15TDVCfrOF6M7Hu84RWAYbjNBOAhdQ0w2C_qMIGcR-9IAaxnESccKrt-vULczpVOI0-7X1maAUhSuRHcJFVLnh5sGGxXRE7VkYfxE_eBf7C89e5cmoLJ2s0M5afYTEouc0pKtx2p3b1X5YpI_c6-I-_Pps7ThkUT7MsCFj_mmk8AMZ1xYNoeI2FwfotvDr0YJDa2dCxtoUCx1Q6AComQZoI8C-golRuolPvHwpHsTnoSna-AI";\n\n    (function() {\n      var config = { kitId: \'kyo4vqy\', scriptTimeout: 3000 };\n      var h=document.getElementsByTagName("html")[0];h.className+=" wf-loading";var t=setTimeout(function(){h.className=h.className.replace(/(\\s|^)wf-loading(\\s|$)/g," ");h.className+=" wf-inactive"},config.scriptTimeout);var tk=document.createElement("script"),d=false;tk.src=\'//use.typekit.net/\'+config.kitId+\'.js\';tk.type="text/javascript";tk.async="true";tk.onload=tk.onreadystatechange=function(){var a=this.readyState;if(d||a&&a!="complete"&&a!="loaded")return;d=true;clearTimeout(t);try{Typekit.load(config)}catch(b){}};var s=document.getElementsByTagName("script")[0];s.parentNode.insertBefore(tk,s);\n    })();\n  </script>\n\n  \n  \n  \n</head>\n\n<body class="site-background-color logged-out videos videos-show group-a site-color-is-light collection-foreign-language-oscar-winners video-war-and-peace-part-2 sites" data-site-color="#b9bcc7" data-track-event="">\n    <!-- Google Tag Manager (noscript) -->\n  <noscript><iframe src="https://www.googletagmanager.com/ns.html?id=GTM-PKL3NF6"\n  height="0" width="0" style="display:none;visibility:hidden"></iframe></noscript>\n  <!-- End Google Tag Manager (noscript) -->\n\n  \n\n  <a href="#maincontent" class="skipto">Skip to main content</a>\n    <div style="display: none !important;" rel="description" class="is-hidden">\n      Classics and discoveries from around the world, thematically programmed with special features, on a streaming service brought to you by the Criterion Collection.\n    </div>\n\n  <div data-flash="{&quot;auto_hide&quot;:true,&quot;style&quot;:&quot;overlay&quot;,&quot;container&quot;:null,&quot;type&quot;:&quot;notice&quot;,&quot;message&quot;:{&quot;body&quot;:null},&quot;show_on_load&quot;:false}" class="is-error is-hidden">\n    <div class="flash-icon">\n      <i class="icon icon-check-inverse icon-check-white"></i>\n      <i class="icon icon-alert-inverse icon-alert-white"></i>\n      <i class="icon icon-check-alt icon-check-blue"></i>\n      <i class="icon medium icon-check"></i>\n      <i class="icon medium icon-alert"></i>\n    </div>\n\n    <div class="flash-msg"></div>\n\n    <div class="flash-close">\n      <i class="icon icon-remove-inverse icon--xsmall"></i>\n      <i class="icon icon-remove-mono icon-x-black icon--xsmall"></i>\n    </div>\n</div>\n\n\n\n\n    \n\n\n\n\n\n\n\n\n\n<header class="nav-background-color border-bottom site-border-color site-header has-logo ">\n  <div class="row full-width padding-top-medium padding-bottom-medium primary-row-container flex-container">\n    <div class="small-16 medium-11 columns identity-container flex-container has-image">\n      <span class="show-for-small-only flex-item padding-right-small show-menu-container">\n        <label for="show-menu" class="show-menu-trigger">\n          <span class="site-link-header-color inverse"><i class="icon site-icon-menu"></i></span>\n        </label>\n      </span>\n\n      <h1 class="identity site-font-primary-family head primary flex-item padding-horizontal-medium">\n            <a href="/">\n              <img class="logo" src="https://vhx.imgix.net/criterionchannel/assets/1c5933b9-19a1-4ea8-9761-726ca79faf44.png?fit=max&h=84&q=92&w=400" alt="The Criterion Channel" />\n</a>      </h1>\n\n      <nav class="navigation-container inline flex-item" role="navigation" aria-label="Primary">\n        <input type="checkbox" id="show-menu" name="show-menu" role="button">\n\n        <div class="navigation-inner site-header-background-color-small-only" id="primary-navigation">\n\n          <a href="/browse" class="browse-link site-font-primary-family head site-link-header-color">\n            Browse\n          </a>\n\n\n              <a href="#"\n                id="search-nav-toggle"\n                class="search-link site-font-primary-family head site-link-header-color">\n                Search\n              </a>\n\n                <a href="https://films.criterionchannel.com/" rel="noopener" class="custom-nav-all-films site-font-primary-family head site-link-header-color" target="_blank">ALL FILMS</a>\n                <a href="https://www.criterion.com/" rel="noopener" class="custom-nav-criterion-com site-font-primary-family head site-link-header-color" target="_blank">Criterion.com</a>\n\n            <a class="show-for-small-only flex-item site-link-header-color site-font-primary-family head subscribe-link" data-track-event-properties="{&quot;type&quot;:&quot;subscription&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/foreign-language-oscar-winners/season:1/videos/war-and-peace-part-2&quot;}" data-track-event="site_buy" href="https://www.criterionchannel.com/checkout/subscribe">\n              <strong style="font-weight: bold;">\n                Start Free Trial\n              </strong>\n</a>\n\n              <a href="/login" class="show-for-small-only flex-item user-action-link head site-link-header-color site-font-primary-family">\n                Sign in\n              </a>\n\n        </div>\n      </nav>\n    </div>\n\n    <div class="hide-for-small-only text-right flex-right small-16 medium-5 columns small-only-text-center user-activity site-border-color">\n      <div class="user-actions has-image padding-right-medium flex-container" role="navigation" aria-label="Secondary">\n            <a class="flex-item site-link-header-color site-font-primary-family head subscribe-link user-action-link" data-track-event-properties="{&quot;type&quot;:&quot;subscription&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/foreign-language-oscar-winners/season:1/videos/war-and-peace-part-2&quot;}" data-track-event="site_buy" href="https://www.criterionchannel.com/checkout/subscribe">\n              <strong style="font-weight: bold;">\n                Start Free Trial\n              </strong>\n</a>\n\n              <a href="/login" class="flex-item user-action-link head site-link-header-color  site-font-primary-family">\n                Sign In\n              </a>\n      </div>\n    </div>\n  </div>\n</header>\n\n  <div id="search-react-app"\n    data-props="{&quot;overlay&quot;:true,&quot;site_id&quot;:59054,&quot;api_v1_url&quot;:&quot;https://api.vhx.tv&quot;,&quot;api_v2_url&quot;:&quot;https://api.vhx.com&quot;,&quot;user_subscribed&quot;:false,&quot;purchased_product_ids&quot;:[],&quot;theme&quot;:&quot;dark&quot;,&quot;show_filters&quot;:false}">\n  </div>\n\n\n\n  \n\n\n<div id="preview-notification" class="preview-notification">Live stream preview</div>\n<section id="watch-player" class="full-width-container background-black contain top-container ">\n  <div class="embed " data-sticky-player>\n        <div id="watch-unauthorized" class="embed-content border-reset background-image-cover flex-vertical-middle" style="background-image: url(https://vhx.imgix.net/criterionchannelchartersu/assets/e5be2fc9-5a9c-4754-85b5-b887f4f00d1d.jpg?auto=format%2Ccompress&amp;fit=crop&amp;h=360&amp;w=640);">\n  <div class="row padding-large embed-content-container">\n    <div class="column small-centered small-14 text-center padding-bottom-medium hide-for-small-only">\n      <h2 class="head primary inverse site-font-primary-family">Watch this video and more on The Criterion Channel</h2>\n    </div>\n    <div class="column small-centered small-14 text-center padding-bottom-small show-for-small-only">\n      <p class="text text-medium inverse site-font-primary-family">Watch this video and more on The Criterion Channel</p>\n    </div>\n\n      <div class="column small-centered small-14 large-8 text-center">\n        <a data-track-event="site_video_buy" href="https://www.criterionchannel.com/checkout/subscribe" class="btn btn--brandon small-only-btn--fill btn--half btn-site-primary margin-bottom-medium">\n          Start your free trial\n        </a>\n\n\n          <p class="text text-medium inverse">\n            Already subscribed?\n            <a href="/login?return_to=https%3A%2F%2Fcriterionchannel.vhx.tv%2Fforeign-language-oscar-winners%2Fseason%3A1%2Fvideos%2Fwar-and-peace-part-2" class="site-primary-color">Sign in</a>\n          </p>\n      </div>\n  </div>\n</div>\n\n\n  </div>\n</section>\n\n\n<section id="watch-info" class="full-width-container">\n  <div class="standard-width-container row">\n    <div class="column small-centered margin-bottom-xlarge small-14">\n      <div class="row margin-vertical-medium">\n  <div class="column small-16 medium-8 large-10">\n    <div class="contain margin-top-large column small-16">\n        <h3 class="secondary padding-bottom-small series-title site-font-primary-color">\n          <a href="/foreign-language-oscar-winners" class="context-link">\n            Foreign-Language Oscar Winners\n          </a>\n        </h3>\n      <h1 class="head primary site-font-primary-color site-font-primary-family margin-bottom-small collection-title video-title">\n        <strong>WAR AND PEACE: Part 2</strong>\n      </h1>\n      <h5 class="text site-font-secondary-color">\n          1h 38m\n      </h5>\n    </div>\n\n\n    <section class="contain column small-16 margin-top-large">\n      <div class="text site-font-secondary-color site-link-primary-color-contain"\n           data-text-show-less="Show less"\n           data-text-show-more="Show more"\n           data-truncate>\n        <p>NATASHA ROSTOVA\n<br />Directed by Sergei Bondarchuk \xe2\x80\xa2 1966 \xe2\x80\xa2 Soviet Union</p>\n\n<p>At the height of the Cold War, the Soviet film industry set out to prove it could outdo Hollywood with a production that would dazzle the world: a titanic, awe-inspiring adaptation of Tolstoy\xe2\x80\x99s classic tome in which the fates of three souls\xe2\x80\x94the blundering, good-hearted Pierre; the heroically tragic Prince Andrei; and the radiant, tempestuous Natasha\xe2\x80\x94collide amid the tumult of the Napoleonic Wars. Employing a cast of thousands and an array of innovative camera techniques, director Sergei Bondarchuk conjures a sweeping vision of grand balls that glitter with rococo beauty and breathtaking battles that overwhelm with their expressionistic power. As a statement of Soviet cinema\xe2\x80\x99s might, WAR AND PEACE succeeded wildly, garnering the Academy Award for best foreign-language film and setting a new standard for epic moviemaking.</p>\n      </div>\n    </section>\n  </div>\n\n\n  <div class="column small-16 medium-8 large-6">\n    <div class="contain padding-vertical-medium padding-horizontal-large share-tools">\n        <h5 class="margin-top-large padding-bottom-medium text-center border-bottom site-border-color site-font-secondary-color">\n  Share with friends\n</h5>\n<div class="text-center margin-top-medium margin-bottom-large video-share-container">\n  <a\n    href="https://www.facebook.com/sharer/sharer.php?u=https%3A%2F%2Fwww.criterionchannel.com%2Fforeign-language-oscar-winners%2Fseason%3A1%2Fvideos%2Fwar-and-peace-part-2"\n    data-social-popup data-share="facebook"\n    data-tooltip=\'{"id": "tooltip-facebook", "width": "auto", "position": "top"}\'\n    class="btn btn-site-secondary icon--center site-icon-facebook margin-horizontal-small">\n    Facebook\n  </a>\n  <a\n    href="https://twitter.com/share?url=https%3A%2F%2Fwww.criterionchannel.com%2Fforeign-language-oscar-winners%2Fseason%3A1%2Fvideos%2Fwar-and-peace-part-2&text=Watch+WAR+AND+PEACE%3A+Part+2+from+%40criterionchannl&related=criterionchannl"\n    data-social-popup data-share="twitter"\n    data-tooltip=\'{"id": "tooltip-twitter", "width": "auto", "position": "top"}\'\n    class="btn btn-site-secondary icon--center site-icon-twitter margin-horizontal-small">\n    Twitter\n  </a>\n  <a target="_blank" href="/cdn-cgi/l/email-protection#6b54181e09010e081f563c0a1f08034b3c2a394b2a252f4b3b2e2a282e514b3b0a191f4b594b0d1904064b3f030e4b2819021f0e190204054b28030a05050e074d0a061b5009040f1256031f1f1b184e582a4e592d4e592d1c1c1c450819021f0e1902040508030a05050e07450804064e592d0d04190e020c0546070a050c1e0a0c0e460418080a19461c0205050e19184e592d180e0a1804054e582a5a4e592d1d020f0e04184e592d1c0a19460a050f461b0e0a080e461b0a191f4659" data-tooltip=\'{"id": "tooltip-email", "width": "auto", "position": "top"}\' class="btn btn-site-secondary icon--center site-icon-envelope margin-horizontal-small">\n    Email\n  </a>\n\n  <div id="tooltip-facebook" class="tooltip medium">\n    Share on Facebook\n  </div>\n  <div id="tooltip-twitter" class="tooltip medium">\n    Share on Twitter\n  </div>\n  <div id="tooltip-email" class="tooltip medium">\n    Share via Email\n  </div>\n</div>\n\n          <div class="video-devices-container">\n    <h5 class="padding-vertical-medium text-center border-bottom site-border-color site-font-secondary-color">\n      Watch anywhere, anytime\n    </h5>\n    <p class="text-center padding-top-medium">\n\n      <a class="site-link-secondary-color padding-horizontal-small" target="_blank" href="https://itunes.apple.com/us/app/id1454275199" data-track-event="devices_download_ios">\n        <strong>iPhone</strong>\n      </a>\n\n      <a class="site-link-secondary-color padding-horizontal-small" target="_blank" href="http://www.criterionchannel.com/help/using-the-apple-tv-app/apple-tv-apps" data-track-event="devices_download_appletv">\n        <strong>Apple TV</strong>\n      </a>\n\n      <a class="site-link-secondary-color padding-horizontal-small" target="_blank" href="https://play.google.com/store/apps/details?id=com.criterionchannel" data-track-event="devices_download_android">\n        <strong>Android</strong>\n      </a>\n\n      <a class="site-link-secondary-color padding-horizontal-small" target="_blank" href="https://play.google.com/store/apps/details?id=com.criterionchannel" data-track-event="devices_download_androidtv">\n        <strong>Android TV</strong>\n      </a>\n\n      <a class="site-link-secondary-color padding-horizontal-small" target="_blank" href="https://www.amazon.com/gp/product/B07PPB3PTV/" data-track-event="devices_download_amazon_fire">\n        <strong>Fire TV</strong>\n      </a>\n\n      <a class="site-link-secondary-color padding-horizontal-small" target="_blank" href="https://www.microsoft.com/store/apps/9PM5D0MQRT2T" data-track-event="devices_download_xbox">\n        <strong>XBox One</strong>\n      </a>\n\n      <a class="site-link-secondary-color padding-horizontal-small" target="_blank" href="https://channelstore.roku.com/details/276871" data-track-event="devices_download_roku">\n        <strong>\n          Roku <span class=\'tiny superscript\'>&reg;</span>\n        </strong>\n      </a>\n\n      <a class="site-link-secondary-color padding-horizontal-small" target="_blank" href="http://www.criterionchannel.com/help/articles/how-do-i-log-in-on-the-tizen-app" data-track-event="devices_download_tizen">\n        <strong>Samsung Smart TV</strong>\n      </a>\n    </p>\n  </div>\n\n    </div>\n  </div>\n</div>\n\n          <div class="row margin-top-large">\n    <h3 class="head secondary site-font-primary-color text-center margin-bottom-medium">\n      Up Next in <a href=\'https://www.criterionchannel.com/foreign-language-oscar-winners/season:1\'               class=\'site-font-primary-color\'>Foreign-Language Oscar Winners</a>\n    </h3>\n    <ul class="small-block-grid-1 medium-block-grid-2 large-block-grid-3">\n\n        <li\n  class="js-collection-item\n    collection-item-533879 item-type-video\n    \n    "\n  data-item-id="533879"\n  data-item-type="video"\n  \n    data-tooltip="id: collection-tooltip-533879;\n    position: right;\n    offset: 0;\n    animate: true;\n    timer: 650;\n    delay: false;\n    is_centered: true;"\n    \n>\n  <div class="browse-item-card">\n    <div class="grid-item-padding">\n      <a href="https://www.criterionchannel.com/foreign-language-oscar-winners/season:1/videos/war-and-peace-part-3" class="browse-item-link" data-track-event="site_video" data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/foreign-language-oscar-winners/season:1/videos/war-and-peace-part-2&quot;,&quot;id&quot;:533879,&quot;context&quot;:&quot;site_context_upnext&quot;,&quot;label&quot;:&quot;WAR AND PEACE: Part 3&quot;,&quot;canonical_collection&quot;:null}">\n        <div class="browse-image-container">\n            <span class="freemium-lock">\n              <i class="icon icon--xsmall icon-lock-inverse"></i>\n            </span>\n\n\n          <img alt="WAR AND PEACE: Part 3" src="https://vhx.imgix.net/criterionchannelchartersu/assets/ca8701d0-afe1-4820-a8c3-91bd6f2721c6.jpg?auto=format%2Ccompress&fit=crop&h=360&w=640" />\n\n\n          <div class="browse-info-container">\n              <div class="duration-container\n                is-locked\n                "\n              >\n                 1:21:40\n              </div>\n          </div>\n        </div> \n      </a>\n\n      <div class="padding-small">\n        <div class="site-font-primary-color browse-item-title">\n           <a\n            href="https://www.criterionchannel.com/foreign-language-oscar-winners/season:1/videos/war-and-peace-part-3"\n            data-track-event="site_video"\n            data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/foreign-language-oscar-winners/season:1/videos/war-and-peace-part-2&quot;,&quot;id&quot;:533879,&quot;context&quot;:&quot;site_context_upnext&quot;,&quot;label&quot;:&quot;WAR AND PEACE: Part 3&quot;,&quot;canonical_collection&quot;:null}"\n          >\n            <strong title="WAR AND PEACE: Part 3">\n              WAR AND PEACE: Part 3\n            </strong>\n          </a>\n        </div>\n\n      </div>\n    </div>\n\n\n    <div \n         class="tooltip background-white" id="collection-tooltip-533879">\n      <h3 class="tooltip-item-title site-font-primary-family"><strong>WAR AND PEACE: Part 3</strong></h3>\n\n        <div class="transparent padding-top-medium">\n          <p>THE YEAR 1812\n<br />Directed by Sergei Bondarchuk \xe2\x80\xa2 1967 \xe2\x80\xa2 Soviet Union</p>\n\n<p>At the height of the Cold War, the Soviet film industry set out to prove it could outdo Hollywood with a production that would dazzle the world: a titanic, awe-inspiring adaptation of Tolstoy\xe2\x80\x99s classic tome in which the fates of t...</p>\n        </div>\n    </div>\n  </div>\n</li>\n\n\n        <li\n  class="js-collection-item\n    collection-item-533880 item-type-video\n    \n    "\n  data-item-id="533880"\n  data-item-type="video"\n  \n    data-tooltip="id: collection-tooltip-533880;\n    position: right;\n    offset: 0;\n    animate: true;\n    timer: 650;\n    delay: false;\n    is_centered: true;"\n    \n>\n  <div class="browse-item-card">\n    <div class="grid-item-padding">\n      <a href="https://www.criterionchannel.com/foreign-language-oscar-winners/season:1/videos/war-and-peace-part-4" class="browse-item-link" data-track-event="site_video" data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/foreign-language-oscar-winners/season:1/videos/war-and-peace-part-2&quot;,&quot;id&quot;:533880,&quot;context&quot;:&quot;site_context_upnext&quot;,&quot;label&quot;:&quot;WAR AND PEACE: Part 4&quot;,&quot;canonical_collection&quot;:null}">\n        <div class="browse-image-container">\n            <span class="freemium-lock">\n              <i class="icon icon--xsmall icon-lock-inverse"></i>\n            </span>\n\n\n          <img alt="WAR AND PEACE: Part 4" src="https://vhx.imgix.net/criterionchannelchartersu/assets/26a12fc6-bd9f-4708-9723-0cbe89ba17ba.jpg?auto=format%2Ccompress&fit=crop&h=360&w=640" />\n\n\n          <div class="browse-info-container">\n              <div class="duration-container\n                is-locked\n                "\n              >\n                 1:36:37\n              </div>\n          </div>\n        </div> \n      </a>\n\n      <div class="padding-small">\n        <div class="site-font-primary-color browse-item-title">\n           <a\n            href="https://www.criterionchannel.com/foreign-language-oscar-winners/season:1/videos/war-and-peace-part-4"\n            data-track-event="site_video"\n            data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/foreign-language-oscar-winners/season:1/videos/war-and-peace-part-2&quot;,&quot;id&quot;:533880,&quot;context&quot;:&quot;site_context_upnext&quot;,&quot;label&quot;:&quot;WAR AND PEACE: Part 4&quot;,&quot;canonical_collection&quot;:null}"\n          >\n            <strong title="WAR AND PEACE: Part 4">\n              WAR AND PEACE: Part 4\n            </strong>\n          </a>\n        </div>\n\n      </div>\n    </div>\n\n\n    <div \n         class="tooltip background-white" id="collection-tooltip-533880">\n      <h3 class="tooltip-item-title site-font-primary-family"><strong>WAR AND PEACE: Part 4</strong></h3>\n\n        <div class="transparent padding-top-medium">\n          <p>PIERRE BEZUKHOV\n<br />Directed by Sergei Bondarchuk \xe2\x80\xa2 1967 \xe2\x80\xa2 Soviet Union</p>\n\n<p>At the height of the Cold War, the Soviet film industry set out to prove it could outdo Hollywood with a production that would dazzle the world: a titanic, awe-inspiring adaptation of Tolstoy\xe2\x80\x99s classic tome in which the fates of...</p>\n        </div>\n    </div>\n  </div>\n</li>\n\n\n        <li\n  class="js-collection-item\n    collection-item-455781 item-type-video\n    \n    "\n  data-item-id="455781"\n  data-item-type="video"\n  \n    data-tooltip="id: collection-tooltip-455781;\n    position: right;\n    offset: 0;\n    animate: true;\n    timer: 650;\n    delay: false;\n    is_centered: true;"\n    \n>\n  <div class="browse-item-card">\n    <div class="grid-item-padding">\n      <a href="https://www.criterionchannel.com/foreign-language-oscar-winners/season:1/videos/z" class="browse-item-link" data-track-event="site_video" data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/foreign-language-oscar-winners/season:1/videos/war-and-peace-part-2&quot;,&quot;id&quot;:455781,&quot;context&quot;:&quot;site_context_upnext&quot;,&quot;label&quot;:&quot;Z&quot;,&quot;canonical_collection&quot;:null}">\n        <div class="browse-image-container">\n            <span class="freemium-lock">\n              <i class="icon icon--xsmall icon-lock-inverse"></i>\n            </span>\n\n\n          <img alt="Z" src="https://vhx.imgix.net/criterionchannelchartersu/assets/a549d9a6-bf8c-4d58-8b7f-ef9c9fb11ff4.jpg?auto=format%2Ccompress&fit=crop&h=360&w=640" />\n\n\n          <div class="browse-info-container">\n              <div class="duration-container\n                is-locked\n                "\n              >\n                 2:07:12\n              </div>\n          </div>\n        </div> \n      </a>\n\n      <div class="padding-small">\n        <div class="site-font-primary-color browse-item-title">\n           <a\n            href="https://www.criterionchannel.com/foreign-language-oscar-winners/season:1/videos/z"\n            data-track-event="site_video"\n            data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/foreign-language-oscar-winners/season:1/videos/war-and-peace-part-2&quot;,&quot;id&quot;:455781,&quot;context&quot;:&quot;site_context_upnext&quot;,&quot;label&quot;:&quot;Z&quot;,&quot;canonical_collection&quot;:null}"\n          >\n            <strong title="Z">\n              Z\n            </strong>\n          </a>\n        </div>\n\n      </div>\n    </div>\n\n\n    <div \n         class="tooltip background-white" id="collection-tooltip-455781">\n      <h3 class="tooltip-item-title site-font-primary-family"><strong>Z</strong></h3>\n\n        <div class="transparent padding-top-medium">\n          <p>Directed by Costa-Gavras \xe2\x80\xa2 1969 \xe2\x80\xa2 Algeria, France\n<br />Starring Yves Montand, Jean-Louis Trintignant, Irene Papas</p>\n\n<p>A pulse-pounding political thriller, Greek expatriate director Costa-Gavras\xe2\x80\x99s Z was one of the cinematic sensations of the late sixties, and remains among the most vital dispatches from t...</p>\n        </div>\n    </div>\n  </div>\n</li>\n\n    </ul>\n  </div>\n\n    </div>\n  </div>\n</section>\n\n\n\n\n    <style>\n  a.site-link-footer-color,\n  a.site-link-footer-color:visited {\n    color: #ffffff !important;\n  }\n  .btn-site-footer,\n  .btn-site-footer:link,\n  .btn-site-footer:visited {\n    color: #b9bcc7;\n    background-color: #000;\n  }\n  .btn-site-footer:hover,\n  .btn-site-footer.is-hover,\n  .btn-site-footer:focus,\n  .btn-site-footer:focus:hover,\n  .btn-site-footer:active,\n  .btn-site-footer.is-active {\n    color: #000;\n    border-color: #000;\n    background-color: #b9bcc7;\n  }\n\n  .btn-logout-form {\n    display: inline;\n    line-height: 0;\n  }\n\n  .btn-logout {\n    background-color: transparent;\n    border: none;\n    cursor: pointer;\n    font-size: 12px;\n    padding: 0;\n  }\n\n  .btn-logout:hover {\n    opacity: .5;\n  }\n</style>\n\n\n<footer class="footer--site border-top site-border-color site-background-color padding-top-medium padding-bottom-medium ">\n  <div class="row full-width">\n    <div class="small-16 medium-6 large-10 columns helper-links small-only-text-center">\n      <div class="padding-left-medium display-flex-container flex-start force-min-height">\n              <a href="https://www.criterion.com/" class="site-font-secondary-color" target="_blank" rel="noopener">THE CRITERION COLLECTION</a>\n\n\n        <a href="/help" class="site-font-secondary-color legal-link help-link" rel="noopener">\n          Help\n        </a>\n        <a href="/tos" class="site-font-secondary-color legal-link terms-link" rel="noopener">\n          Terms\n        </a>\n        <a href="/privacy" class="site-font-secondary-color legal-link privacy-link" rel="noopener">\n          Privacy\n        </a>\n        <a href="/cookies" class="site-font-secondary-color legal-link cookies-link" rel="noopener">\n          Cookies\n        </a>\n\n\n            <a href="/login" class="site-font-secondary-color signin-link">\n              Sign in\n            </a>\n      </div>\n    </div> <!-- columns -->\n    <div class="small-16 medium-10 large-6 small-only-text-center text-right columns vhx-link">\n      <div class="padding-right-small display-flex-container flex-end force-min-height">\n          &nbsp;\n        <div id="language-switcher-react-app"\n             class="padding-right-small"\n             data-supported-locales="[]"\n        </div>\n        <script data-cfasync="false" src="/cdn-cgi/scripts/5c5dd728/cloudflare-static/email-decode.min.js"></script><script src="https://cdn.vhx.tv/assets/vite/assets/language-switcher.3abc5616.js" crossorigin="anonymous" type="module"></script><link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/jsx-runtime.75dbc984.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/LanguageSwitcher.864c5f4d.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/index.71ea8908.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/intl.helpers.a161606f.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/index.61149844.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/deviceSizes.9973a5fd.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/message.5870a508.js" as="script" crossorigin="anonymous" />\n      </div> <!-- display-flex-container -->\n    </div>\n  </div> <!-- row -->\n</footer>\n\n<div class="modal">\n  <div class="close"><a href="#close" alt="Close Window"><strong class="is-inverse">&times;</strong></a></div>\n  <div class="content"></div>\n</div>\n\n<div id="blanket"></div>\n\n\n  <div class="modals">\n    \n  </div>\n\n  <script src="https://cdn.vhx.tv/assets/webpack/vendor-ottbundle-8f0450ca8ec63c03ff2b.js"></script>\n\n  <script>\n    window.VHX = window.VHX || {};\n    window.VHX.config = {\n      api_url: "https://api.vhx.tv",\n      token: "eyJhbGciOiJSUzI1NiIsImtpZCI6ImQ2YmZlZmMzNGIyNTdhYTE4Y2E2NDUzNDE2ZTlmZmRjNjk4MDAxMDdhZTQ2ZWJhODg0YTU2ZDBjOGQ4NTYzMzgifQ.eyJhcHBfaWQiOjY4NTExLCJleHAiOjE2NzU0NTU0MDMsIm5vbmNlIjoiNGI5MjY4ZTdjMTM0NTdhOSIsInNjb3BlcyI6WyJwdWJsaWMiXSwic2Vzc2lvbl9pZCI6Ik1ieHNtT3RxelNWRytGaXQ5Szd6d2c9PSIsInNpdGVfaWQiOjU5MDU0fQ.C4rkmFBWRVhGngmKbcCVxRQjvxf0Vp5VUXKhFudt_DmKwIjj2PRvyLKvez5hFjOlMUGNVNke7vEEyl106eyRP9LvVi5L1BmHDv_YuGbSl99A6YWBFG-Nlm7oDgQO43eWf2J5SyLrWToAB541T-tWHsQj6xHEbOyptUknIvs4sZGzGpOz8VAGpQme9Evqb_ailtbaKW9pgR6Rm2EBw1k_ZpFBhpRkd26Fa2WiZSInqqG8kbWh_2IWPVxEFF7R6AaST7wF89aYdp6QcCvxiL8dglBQJ7FHulrUwCKGpAY4AvGuBGN0m8yRLkId0eGH9GLOFbbjPjJYUnSfHGV7qxWk0zqj_mRs2E5_UEpWmKaLqdSyANI6gl_vKiV_H_OGIncoqceqo1yUQktGCIEfpAwATFJNKv_gDHCAKPMFQv2wrHRkigb4rO7e8pmJ3H15TDVCfrOF6M7Hu84RWAYbjNBOAhdQ0w2C_qMIGcR-9IAaxnESccKrt-vULczpVOI0-7X1maAUhSuRHcJFVLnh5sGGxXRE7VkYfxE_eBf7C89e5cmoLJ2s0M5afYTEouc0pKtx2p3b1X5YpI_c6-I-_Pps7ThkUT7MsCFj_mmk8AMZ1xYNoeI2FwfotvDr0YJDa2dCxtoUCx1Q6AComQZoI8C-golRuolPvHwpHsTnoSna-AI",\n      token_expires_in: "7200000",\n      user_has_subscription: "false"\n    };\n  </script>\n\n  <script src="https://cdn.vhx.tv/assets/webpack/sites-ottbundle-5d471cab41d88dfe692d.js"></script>\n  \n\n      <script>\n    window.COMMENTABLE_ID = 533128;\n    window.VHX = window.VHX || {};\n    window.VHX.config = {\n      api_url: "https://api.vhx.tv",\n      embed_url: "https://embed.criterionchannel.com/videos/533128?api=1&amp;autoplay=1&amp;back=Foreign-Language%20Oscar%20Winners%20-%20Foreign-Language%20Oscar%20Winners&amp;collection_id=143189&amp;color=b9bcc7&amp;context=https%3A%2F%2Fwww.criterionchannel.com%2Fforeign-language-oscar-winners%2Fseason%3A1&amp;is_trailer=false&amp;live=0&amp;locale=en&amp;playsinline=1&amp;product_id=39621&amp;referrer=&amp;sharing=1&amp;title=0&amp;vimeo=1",\n      token: "eyJhbGciOiJSUzI1NiIsImtpZCI6ImQ2YmZlZmMzNGIyNTdhYTE4Y2E2NDUzNDE2ZTlmZmRjNjk4MDAxMDdhZTQ2ZWJhODg0YTU2ZDBjOGQ4NTYzMzgifQ.eyJhcHBfaWQiOjY4NTExLCJleHAiOjE2NzU0NTU0MDMsIm5vbmNlIjoiNGI5MjY4ZTdjMTM0NTdhOSIsInNjb3BlcyI6WyJwdWJsaWMiXSwic2Vzc2lvbl9pZCI6Ik1ieHNtT3RxelNWRytGaXQ5Szd6d2c9PSIsInNpdGVfaWQiOjU5MDU0fQ.C4rkmFBWRVhGngmKbcCVxRQjvxf0Vp5VUXKhFudt_DmKwIjj2PRvyLKvez5hFjOlMUGNVNke7vEEyl106eyRP9LvVi5L1BmHDv_YuGbSl99A6YWBFG-Nlm7oDgQO43eWf2J5SyLrWToAB541T-tWHsQj6xHEbOyptUknIvs4sZGzGpOz8VAGpQme9Evqb_ailtbaKW9pgR6Rm2EBw1k_ZpFBhpRkd26Fa2WiZSInqqG8kbWh_2IWPVxEFF7R6AaST7wF89aYdp6QcCvxiL8dglBQJ7FHulrUwCKGpAY4AvGuBGN0m8yRLkId0eGH9GLOFbbjPjJYUnSfHGV7qxWk0zqj_mRs2E5_UEpWmKaLqdSyANI6gl_vKiV_H_OGIncoqceqo1yUQktGCIEfpAwATFJNKv_gDHCAKPMFQv2wrHRkigb4rO7e8pmJ3H15TDVCfrOF6M7Hu84RWAYbjNBOAhdQ0w2C_qMIGcR-9IAaxnESccKrt-vULczpVOI0-7X1maAUhSuRHcJFVLnh5sGGxXRE7VkYfxE_eBf7C89e5cmoLJ2s0M5afYTEouc0pKtx2p3b1X5YpI_c6-I-_Pps7ThkUT7MsCFj_mmk8AMZ1xYNoeI2FwfotvDr0YJDa2dCxtoUCx1Q6AComQZoI8C-golRuolPvHwpHsTnoSna-AI",\n      token_expires_in: "7200000",\n      user_has_subscription: "false",\n      is_avod: false,\n      user_has_plan: false,\n      is_admin: false\n    };\n    window.VHX.video = {\n      id: "533128",\n      status: "pending",\n      isLive: "false",\n      scheduled_at: "",\n      isDRM: "true",\n      isTrailer: "false",\n      \n    };\n  </script>\n\n  <script src="https://cdn.vhx.tv/assets/webpack/videos-ottbundle-6717392db1468efccdad.js"></script>\n  <script src="https://cdn.vhx.tv/assets/player.js"></script>\n\n    <script src="https://cdn.vhx.tv/assets/vite/assets/search.e53736d1.js" crossorigin="anonymous" type="module"></script><link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/jsx-runtime.75dbc984.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/index.esm.f84f874f.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/index.0e8c968f.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/intl.helpers.a161606f.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/api.utils.c8000605.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/Menu.esm.ddfda87f.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/Button.esm.61e4a4cb.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/Text.esm.c0f99a84.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/message.5870a508.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/Checkmark.esm.fb551793.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/Link.esm.a6ddba18.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/Search.esm.a44cb748.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/index.71ea8908.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/text-helpers.5cfd261d.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/time-helpers.a73ca797.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/_includes.3534e71e.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/_curry2.3e499763.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/path.235ca80c.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/Header.esm.2a01f61a.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/DismissX.esm.54e5d2b0.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/colors.esm.e8046dae.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/inheritsLoose.a8857277.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/index.61149844.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/ChevronDown.esm.405e6674.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/_isArray.68eb59b2.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/index.096cf963.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/equals.d9a066f9.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/_isString.019dcf3c.js" as="script" crossorigin="anonymous" /><link rel="stylesheet" media="screen" href="https://cdn.vhx.tv/assets/vite/assets/search.c6f13361.css" crossorigin="anonymous" />\n\n\n<script>\n  setTimeout(function() {\n    document.documentElement.className += \' is-loaded\';\n    var selects = document.getElementsByClassName(\'select\'),\n        checkboxes = document.getElementsByClassName(\'checkbox\'),\n        i;\n\n    if (selects.length) {\n      for (i = 0; i < selects.length; i++) {\n        selects[i].className += \' ready\';\n      }\n    }\n\n    if (checkboxes.length) {\n      for (i = 0; i < checkboxes.length; i++) {\n        for (var p=0; p < checkboxes[i].childNodes.length; p++) {\n          if (checkboxes[i].childNodes[p].tagName == \'LABEL\') {\n            checkboxes[i].childNodes[p].className += \' ready\';\n            break;\n          }\n        }\n      }\n    }\n  }, 500);\n</script>\n\n  <script>\n  window.addEventListener(\'load\', function() {\n    jQuery(\'a:contains("Sign In")\').click(function(){\n      ga(\'site.send\',\'event\',\'button\',\'click\',\'sign in\');\n    });\n\n    jQuery(\'a:contains("Sign Up")\').click(function(){\n      ga(\'site.send\',\'event\',\'button\',\'click\',\'sign up\');\n    });\n\n    jQuery(\'a:contains("Watch now")\').click(function(){\n      ga(\'site.send\',\'event\',\'button\',\'click\',\'watch now\');\n    });\n\n    jQuery(\'a:contains("Share")\').click(function(){\n      ga(\'site.send\',\'event\',\'button\',\'click\',\'share\');\n    });\n\n    jQuery(\'[href="https://www.criterionchannel.com/the-deviant-and-the-divine/season:1/videos/freaks"],[href="https://www.criterionchannel.com/the-deviant-and-the-divine/season:1/videos/multiple-maniacs"]\').click(function(){\n      ga(\'site.send\',\'event\',\'button\',\'click\',\'video\');\n    });\n  });\n</script>\n\n\n\n    <script>\n      const javascript = \'\\$(document).ready(function(){\\n\\$(\\\'header .logo\\\').attr(\\\'data-swiftype-index\\\', false);\\n\\n\\$(\\"a[href*=\\\'/checkout\\\']\\").attr(\\"data html(\\"Sign Up\\");\\n  \\$(\\"a[href*=\\\'/checkout/subscribe/gift\\\']\\").html(\\"Buy a gift\\");\\n  \\$(\\"a[href*=\\\'/buy/subscription\\\']\\").attr(\\"data-trigger-disabled\\",true);\\n  \\$(\\"a[href*=\\\'/buy/subscription\\\']\\").html(\\"Sign Up\\");\\n  \\$(\\"a.browse-link\\").html(\\"Now Playing\\");\\n  landingPageChanges();\\n  collectionChanges();\\n  helpChanges();\\n  \\$(\\"body.browse .slide-subtitle\\").hide();\\n  \\$(\\"body.logged-in h1.identity a\\").attr(\\"href\\",\\"/\\");\\n  \\$(\\".navigation-inner\\").addClass(\\"is-ready\\");\\n  \\$(\\"nav .custom-nav-all-films\\").attr(\\"target\\", \\"_self\\");\\n\\n  amazon_fire_tv_supported = \\$(\\\'.amazon-fire-tv\\\').parent().parent().children()[1]\\n  \\$(amazon_fire_tv_supported).text(\\"2nd Gen +\\")\\n\\n  android_tv_supported_text = \\$(\\\'.android-tv\\\').parent().parent().children()[1]\\n  \\$(android_tv_supported_text).text(\\"Selected models\\")\\n  android_tv_4k_text = \\$(\\\'.android-tv\\\').parent().parent().children()[2]\\n  \\$(android_tv_4k_text).hide()\\n\\n  new_xbox = \\"<li>Xbox Series X|S<\\/li>\\";\\n  \\$(new_xbox).insertAfter(\\$(\\\'.platforms-modal-content-tv .platforms-modal-icon-container .platform-modal-devices-list > li:nth-of-type(5) li:nth-of-type(2)\\\'));\\n\\n  \\$(\\\'.platforms-modal-content-tv .platforms-modal-icon-container .platform-modal-devices-list > li:nth-of-type(6) li:nth-of-type(2)\\\').text(\\\'Select 2018, 2019 & 2020 Models\\\');\\n\\n  gift_link = \\\'<a href=\\"https://www.criterionchannel.com/checkout/subscribe/gift?gift=1&duration=6\\" class=\\"is-block text dark block padding-medium\\">Give a Gift<\\/a>\\\';\\n  \\$(gift_link).insertAfter(\\$(\\\'.user-dropdown a[href*=\\"/purchases\\"]\\\'));\\n\\n  if (\\$(\\\'meta[name*=\\"twitter:image\\"]\\\').length > 0) {\\n    twitter_content = \\$(\\\'meta[name*=\\"twitter:image\\"]\\\').attr(\\\'content\\\')\\n    new_meta = \\\'<meta class=\\"swiftype\\" name=\\"image\\" data-type=\\"enum\\" content=\\"\\\' + twitter_content + \\\'\\">\\\';\\n    \\$(\\\'head\\\').append(new_meta);\\n  }\\n});\\n\\nfunction collectionChanges(){\\n  if (\\$(\\"body\\").hasClass(\\"collections\\")){\\n    \\$(\\\'.cover-art\\\').removeClass(\\\'large-offset-2 large-6\\\').addClass(\\\'large-8 large-offset-1\\\');\\n\\n    \\$(\\\'section.product-feature\\\').removeClass(\\\'padding-horizontal-medium padding-top-large padding-bottom-xlarge\\\').addClass(\\\'padding-bottom-large\\\');\\n    \\$(\\\'section.product-feature .row.relative\\\').addClass(\\\'full-width\\\');\\n\\n    \\$(\\\'section.product-feature .collection-details-container\\\').removeClass(\\"large-8 end\\").addClass(\\"large-7\\");\\n      \\$(\\\'section.product-feature .collection-details\\\').addClass(\\"padding-left-large padding-right-large padding-top-xlarge\\");\\n\\n      \\$(\\".embed-container\\").removeClass(\\"grid-padding-right\\");\\n  }\\n  if (\\$(\\\'.collection-playlist\\\').length > 0 || \\$(\\\'.collection-series\\\').length > 0) {\\n    \\$(\\\'.video-controls\\\').removeClass(\\\'padding-top-large\\\').addClass(\\\'padding-top-small\\\');\\n  }\\n\\n  if (\\$(\\\'.collection-movie\\\').length > 0) {\\n    \\$(\\\'.collection-actions a:contains(\\"Trailer\\")\\\').html(\\\'<span class=\\"icon-adjustment relative inverse-icon-bg dark-bg margin-bottom-small\\"><i class=\\"icon small icon-play-inverse\\"><\\/i><\\/span> Trailer\\\');\\n    \\$(\\\'.collection-actions a:contains(\\"Movie\\")\\\').html(\\\'<span class=\\"icon-adjustment relative inverse-icon-bg dark-bg margin-bottom-small\\"><i class=\\"icon small icon-play-inverse\\"><\\/i><\\/span> Watch\\\');\\n  }\\n\\n\\n}\\n\\nfunction landingPageChanges(){\\n  \\$(\\\'a:contains(\\"Start Your Free Month\\")\\\').text(\\"Sign Up Now\\");\\n}\\nfunction helpChanges(){\\n  // Custom Help Center Updates\\n  setTimeout(function() {\\n    \\$(\\\'article li:contains(\\"You can plug your laptop or mobile device into your TV using an HDMI cable\\")\\\').each(function() {\\n      \\$(this).html(\\\'<li>It may be possible to connect your laptop or mobile device to your TV using an HDMI cable if all components in the chain are <a href=\\"https://www.criterionchannel.com/help/articles/drm-and-hdcp#HDCP\\">HDCP/DRM compliant.<\\/a><\\/li>\\\');\\n    });\\n\\n    \\$(\\\'article li:contains(\\" and select the language of choice\\")\\\').each(function() {\\n      \\$(this).text(\\$(this).text().replace(\\" and select the language of choice\\", \\"\\"));\\n    });\\n    \\n    a = \\\'<li>If you have a supported device connected to your TV, Criterion also has an app for Apple TV, Amazon Fire, Roku, and Android TV.<\\/li>\\\'\\n    b = \\$(\\\'li:contains(\\"You can mirror or Airplay from your iOS device\xe2\x80\x99s browser.\\")\\\')\\n    b.after(a);\\n\\n    \\$(\\\'li a[href*=\\"670-how-do-i-login-on-the-xbox-app\\"]\\\').parent().hide();\\n  }, 500);\\n\\n  // hide mentions of the VHX articles\\n  // we have to settimeout because the help pages are ajax\\\'ing in content.\\n  // this is best guess on timing\\n  if (\\$(\\"body\\").hasClass(\\"help-articles\\") || \\$(\\"body\\").hasClass(\\"sites-help\\") || \\$(\\"body\\").hasClass(\\"help-categories\\")){\\n   \\$(\\\'#mithril-help\\\').bind(\\"DOMSubtreeModified\\",function(){\\n    var waittoremove = setTimeout(function(){ removeHelpArticles(); }, 500);\\n  });\\n\\n    var waittoremove = setTimeout(function(){ removeHelpArticles(); }, 700);\\n  }\\n}\\nfunction removeHelpArticles(){\\n  var vhxhelp = Array(\\"589e1e742c7d3a784630ccd5\\",\\"581396999033604deb0ec504\\",\\"57fe5ceb9033600277a6980b\\",\\"54b59261e4b0934e822d2132\\",\\"54dbc064e4b086c0c0966908\\",\\"5453fe87e4b02ce342671537\\",\\"555119ebe4b027e1978dfc04\\",\\"549206b9e4b07d03cb251286\\", \\"5c7eee4f04286350d088ab37\\");\\n  for (i=0;i<=vhxhelp.length-1;i++){\\n    \\$(\\".article-\\"+vhxhelp[i]).remove();                                                        \\n  }\\n}\';\n      try {\n        // Using eval so we can catch syntax errors\n        eval(javascript);\n      } catch (err) {\n        console.error(err);\n      }\n    </script>\n</body>\n</html>\n'
    return BeautifulSoup(content, 'html5lib')


@pytest.fixture
def soup_collection():
    content = b'<!doctype html>\n<!--[if lt IE 7]> <html class="no-js lt-ie9 lt-ie8 lt-ie7" lang="en" xml:lang="en"> <![endif]-->\n<!--[if IE 7]>    <html class="no-js lt-ie9 lt-ie8" lang="en" xml:lang="en"> <![endif]-->\n<!--[if IE 8]>    <html class="no-js lt-ie9" lang="en" xml:lang="en"> <![endif]-->\n<!--[if gt IE 8]><!--> <html class="no-js" lang="en" xml:lang="en"> <!--<![endif]-->\n<head lang="en" xml:lang="en">\n  <meta charset="utf-8">\n  <meta http-equiv="X-UA-Compatible" content="IE=edge,chrome=1">\n  <meta name="viewport" content="width=device-width, initial-scale=1">\n\n    <link rel="shortcut icon" href="https://dr56wvhu2c8zo.cloudfront.net/criterionchannel/assets/5922c7f6-cdda-4786-8ac9-521602da043a.png" type="image/png">\n\n    <meta name="apple-itunes-app" content="app-id=1454275199, app-argument=https://criterionchannel.vhx.tv/the-front-page">\n\n    <meta name="facebook-domain-verification" content="qk8q6gxhll8cwf99kggygprhe0y9fz">\n\n  <title>The Front Page - The Criterion Channel</title>\n  <meta name="description" content="Directed by Lewis Milestone \xe2\x80\xa2 1931 \xe2\x80\xa2 United States\nStarring Adolphe Menjou, Pat O\xe2\x80\x99Brien, Mary Brian\n\nDirected with pre-Code verve by Lewis Milestone, the first screen adaptation of the oft-filmed Broadway hit by Ben Hecht and Charles MacArthur (later the basis for Howard Hawks\xe2\x80\x99s screwball classic HIS GIRL FRIDAY) stars Adolphe Menjou and Pat O\xe2\x80\x99Brien as, respectively, a newspaper editor and his ace reporter, who must pull out all the stops in order to get the scoop of a lifetime: the story of an escaped death-row inmate whose whereabouts are known only to them. The crackling dialogue, cynical wit, and snappy pace set the standard for a generation of newsroom pictures to follow.\n\nRestored by the Academy Film Archive and The Film Foundation. Restoration funding provided by the George Lucas Family Foundation. Elements for this restoration provided by the Howard Hughes Corporation and by the University of Nevada, Las Vegas, College of Fine Arts\xe2\x80\x99 Department of Film and its Howard Hughes Collection at the Academy Film Archive.\nRestored by the Academy Film Archive and The Film Foundation. Restoration funding provided by the George Lucas Family Foundation. Elements for this restoration provided by the Howard Hughes Corporation and by the University of Nevada, Las Vegas, College of Fine Arts\xe2\x80\x99 Department of Film and its Howard Hughes Collection at the Academy Film Archive.">\n  <link rel="canonical" href="https://www.criterionchannel.com/the-front-page" />\n\n    <meta property="og:url" content="https://www.criterionchannel.com/the-front-page">\n  <meta property="fb:app_id" content="136862186391940">\n  <meta property="og:title" content="The Front Page - The Criterion Channel">\n  <meta property="og:site_name" content="The Criterion Channel">\n  <meta property="og:image" content="https://vhx.imgix.net/criterionchannelchartersu/assets/8066de39-d6e3-4196-9097-60053d6c940f.jpg?auto=format%2Ccompress&fit=crop&h=720&q=75&w=1280">\n  <meta property="og:image:width" content="1280">\n  <meta property="og:image:height" content="720">\n  <meta property="og:description" content="Directed by Lewis Milestone \xe2\x80\xa2 1931 \xe2\x80\xa2 United States\nStarring Adolphe Menjou, Pat O\xe2\x80\x99Brien, Mary Brian\n\nDirected with pre-Code verve by Lewis Milestone, the first screen adaptation of the oft-filmed Broadway hit by Ben Hecht and Charles MacArthur (later the basis for Howard Hawks\xe2\x80\x99s screwball classic...">\n  <meta property="og:type" content="website">\n\n  <meta name="twitter:card" content="summary_large_image">\n    <meta name="twitter:site" content="@criterionchannl">\n    <meta name="twitter:creator" content="@criterionchannl">\n  <meta name="twitter:title" content="The Front Page - The Criterion Channel">\n  <meta name="twitter:description" content="Directed by Lewis Milestone \xe2\x80\xa2 1931 \xe2\x80\xa2 United States\nStarring Adolphe Menjou, Pat O\xe2\x80\x99Brien, Mary Brian\n\nDirected with pre-Code verve by Lewis Milestone, the first screen adaptation of the oft-filmed Broadway hit by Ben Hecht and Charles MacArthur (later the basis for Howard Hawks\xe2\x80\x99s screwball classic...">\n  <meta name="twitter:image" content="https://vhx.imgix.net/criterionchannelchartersu/assets/8066de39-d6e3-4196-9097-60053d6c940f.jpg?auto=format%2Ccompress&fit=crop&h=720&q=75&w=1280">\n  <meta name="twitter:image:width" content="1280">\n  <meta name="twitter:image:height" content="720">\n\n\n\n<link rel="author" href="/humans.txt">\n<meta name="csrf-param" content="authenticity_token" />\n<meta name="csrf-token" content="yPhDLT7fReRTFm177WZsWJfyEBnn5SyFuq0KCRAEQ2BZf4CpUO9+3R5/JgJfHEKaKlHyIIzMjy815nOSFxfecw==" />\n\n\n    <!-- apple / android  -->\n  <meta name="mobile-web-app-capable" content="no">\n  <meta name="apple-mobile-web-app-capable" content="no">\n  <meta name="apple-mobile-web-app-title" content="The Criterion Channel">\n  <!-- microsoft tile color -->\n  <meta name="msapplication-TileColor" content="#b9bcc7">\n  <!-- android browser bar color -->\n  <meta name="theme-color" content="#b9bcc7">\n\n    <!-- apple and other browsers -->\n    <link rel="apple-touch-icon" sizes="57x57"    href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=57&w=57">\n    <link rel="apple-touch-icon" sizes="114x114"  href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=114&w=114">\n    <link rel="apple-touch-icon" sizes="72x72"    href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=72&w=72">\n    <link rel="apple-touch-icon" sizes="144x144"  href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=144&w=144">\n    <link rel="apple-touch-icon" sizes="60x60"    href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=60&w=60">\n    <link rel="apple-touch-icon" sizes="120x120"  href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=120&w=120">\n    <link rel="apple-touch-icon" sizes="76x76"    href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=76&w=76">\n    <link rel="apple-touch-icon" sizes="152x152"  href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=152&w=152">\n    <link rel="apple-touch-icon" sizes="180x180"  href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=180&w=180">\n    <!-- windows tile -->\n    <meta name="msapplication-TileImage" content="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=144&w=144">\n    <!-- android icons -->\n    <link rel="icon" type="image/png" sizes="32x32" href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=32&w=32">\n    <link rel="icon" type="image/png" sizes="16x16" href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=16&w=16">\n\n\n    <meta class="swiftype" name="type" data-type="enum" content="Channel" />\n\n\n\n  <link rel="stylesheet" media="all" href="https://cdn.vhx.tv/assets/webpack/icons.svg-ottbundle-798d02cf9d8ee37d5dfe.css" />\n  <link rel="stylesheet" media="all" href="https://cdn.vhx.tv/assets/webpack/sites-ottbundle-edf8dd6d4f778f83ed2d.css" />\n\n    <link rel="stylesheet" media="screen" href="https://cdn.vhx.tv/assets/templates/_shared/typography.montserrat-3b605b0b18ab721686d69bf9c5bd93fc8f657562bc6a431ae71c409675dfb333.css" />\n\n<style>\n\n\n  /* main colors */\n  .site-background-color {\n    background-color: #141414 !important;\n  }\n  .site-primary-highlight-color {\n    background: #b9bcc7 !important;\n  }\n  .site-primary-color,\n  strong.site-primary-color,\n  .site-primary-color strong,\n  a.site-primary-color:link,\n  a.site-primary-color:visited {\n    color: #b9bcc7 !important;\n  }\n  a.site-primary-color:hover,\n  a.site-primary-color:active {\n    color: #9da0a9 !important;\n  }\n\n  .site-border-color,\n  .site-border-color.border-light {\n    border-color: #323232 !important;\n  }\n  .site-border-primary-color,\n  .site-border-primary-color.border-light {\n    border-color: #b9bcc7 !important;\n  }\n\n  /* custom font primary family */\n  .site-font-primary-family {\n    font-family: \'montserrat\', sans-serif !important;\n  }\n\n  /* font primary color */\n  .site-font-primary-color,\n  .site-font-primary-color strong {\n    color: #ffffff !important;\n  }\n\n  /* font secondary color */\n  .site-font-secondary-color,\n  .site-font-secondary-color strong {\n    color: #bebebb !important;\n  }\n\n  /* link primary color */\n  a.site-link-primary-color:link,\n  a.site-link-primary-color:visited,\n  a.site-link-primary-color strong {\n    color: #ffffff !important;\n  }\n  a.site-link-primary-color:hover,\n  a.site-link-primary-color:active {\n    color: #d9d9d9 !important;\n    opacity: .7;\n  }\n  .site-link-primary-color-contain a:link,\n  .site-link-primary-color-contain a:visited {\n    color: #b9bcc7 !important;\n  }\n  .site-link-primary-color-contain a:hover,\n  .site-link-primary-color-contain a:active {\n    color: #9da0a9 !important;\n    opacity: .7;\n  }\n\n  /* link secondary color */\n  .pagination a,\n  a.site-link-secondary-color:link,\n  a.site-link-secondary-color:visited,\n  a.site-link-secondary-color strong,\n  .site-link-secondary-color-contain a:link,\n  .site-link-secondary-color-contain a:visited {\n    color: #7c7c7c;\n  }\n  .pagination a:hover,\n  a.site-link-secondary-color:hover,\n  a.site-link-secondary-color:active,\n  .site-link-secondary-color-contain a:hover,\n  .site-link-secondary-color-contain a:active {\n    color: #696969;\n    opacity: .7;\n  }\n\n  /* link underline font primary color */\n  .site-link-contain a:link,\n  .site-link-contain a:visited {\n    color: #ffffff !important;\n    text-decoration: underline;\n  }\n  .site-link-contain a:hover,\n  .site-link-contain a:active {\n    color: #ffffff !important;\n    opacity: .7;\n    text-decoration: underline;\n  }\n\n  /* border color same as primary color */\n  .site-border-primary-color {\n    border-color: #ffffff !important;\n  }\n  .site-border-top-primary-color {\n    border-top-color: #ffffff !important;\n  }\n  .site-border-bottom-primary-color {\n    border-bottom-color: #ffffff !important;\n  }\n  .site-border-left-primary-color {\n    border-left-color: #ffffff !important;\n  }\n  .site-border-right-primary-color {\n    border-right-color: #ffffff !important;\n  }\n\n  /* link header color */\n  a.site-link-header-color:link,\n  a.site-link-header-color:visited,\n  a.site-link-header-color:link strong,\n  a.site-link-header-color:visited strong {\n    color: #ffffff !important;\n  }\n  a.site-link-header-color:hover,\n  a.site-link-header-color:active {\n    color: #d9d9d9;\n    opacity: .7;\n  }\n .avatar-chevron.icon {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2223%22%20height%3D%2223%22%20viewBox%3D%220%200%2023%2023%22%3E%3Cpolygon%20fill%3D%22%23ffffff%22%20points%3D%2222.9%2C7.2%2020%2C4.4%2011.5%2C12.9%203%2C4.4%200.1%2C7.2%208.7%2C15.7%208.7%2C15.7%2011.5%2C18.6%2011.5%2C18.6%2011.5%2C18.6%2014.3%2C15.7%2014.3%2C15.7%20%22%2F%3E%3C%2Fsvg%3E%0A");\n    background-repeat: no-repeat;\n  }\n  .user-dropdown a:hover {\n    background-color: rgba(51,51,51,0.2);\n    color: #fff;\n  }\n  .user-dropdown input {\n    background-color: #ffffff;\n  }\n  .user-dropdown input:hover {\n    background-color: rgba(51,51,51,0.2);\n    color: #fff;\n  }\n\n  /* buttons site primary */\n  .btn-site-primary,\n  .btn-site-primary:link,\n  .btn-site-primary:visited {\n    color: #000;\n    border-color: #9da0a9;\n    background-color: #b9bcc7;\n  }\n  .btn-site-primary:hover,\n  .btn-site-primary.is-hover,\n  .btn-site-primary:focus,\n  .btn-site-primary:focus:hover,\n  .btn-site-primary:active,\n  .btn-site-primary.is-active {\n    border-color: #9da0a9;\n    background-color: #9da0a9;\n    color: #000;\n  }\n\n  .btn-site-primary[disabled],\n  .btn-site-primary.is-disabled,\n  .btn-site-primary[disabled]:visited,\n  .btn-site-primary.is-disabled:visited,\n  .btn-site-primary[disabled]:hover,\n  .btn-site-primary.is-disabled:hover,\n  .btn-site-primary[disabled].is-hover,\n  .btn-site-primary.is-disabled.is-hover,\n  .btn-site-primary[disabled]:active,\n  .btn-site-primary.is-disabled:active,\n  .btn-site-primary[disabled]:focus,\n  .btn-site-primary.is-disabled:focus,\n  .btn-site-primary[disabled]:focus:hover,\n  .btn-site-primary.is-disabled:focus:hover,\n  .btn-site-primary[disabled].is-active,\n  .btn-site-primary.is-disabled.is-active {\n    opacity: .5;\n    background-color: #b9bcc7;\n    color: #000;\n    border-color: #9da0a9;\n  }\n\n  /* buttons site secondary */\n  .btn-site-secondary,\n  .btn-site-secondary:link,\n  .btn-site-secondary:visited {\n    color: #bebebb;\n    border-color: #141414;\n    background-color: #323232;\n  }\n  .btn-site-secondary:hover,\n  .btn-site-secondary.is-hover,\n  .btn-site-secondary:focus,\n  .btn-site-secondary:active,\n  .btn-site-secondary.is-active {\n    color: #5f5f5e;\n    border-color: #0f0f0f;\n    background-color: #323232;\n  }\n\n  .btn-site-primary.is-processing,\n  .btn-site-secondary.is-processing {\n    color: transparent !important;\n    background-position: center center !important;\n    background-repeat: no-repeat;\n    background-size: 23px 23px;\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3Csvg%20version%3D%221.1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%20%20viewBox%3D%220%200%2040%2040%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cpath%20opacity%3D%220.2%22%20fill%3D%22%23000000%22%20d%3D%22M20.201%2C5.169c-8.254%2C0-14.946%2C6.692-14.946%2C14.946c0%2C8.255%2C6.692%2C14.946%2C14.946%2C14.946%0D%0A%20%20s14.946-6.691%2C14.946-14.946C35.146%2C11.861%2C28.455%2C5.169%2C20.201%2C5.169z%20M20.201%2C31.749c-6.425%2C0-11.634-5.208-11.634-11.634%0D%0A%20%20c0-6.425%2C5.209-11.634%2C11.634-11.634c6.425%2C0%2C11.633%2C5.209%2C11.633%2C11.634C31.834%2C26.541%2C26.626%2C31.749%2C20.201%2C31.749z%22%2F%3E%0D%0A%3Cpath%20fill%3D%22%23000000%22%20d%3D%22M26.013%2C10.047l1.654-2.866c-2.198-1.272-4.743-2.012-7.466-2.012h0v3.312h0%0D%0A%20%20C22.32%2C8.481%2C24.301%2C9.057%2C26.013%2C10.047z%22%3E%0D%0A%20%20%3CanimateTransform%20attributeType%3D%22xml%22%0D%0A%20%20%20%20attributeName%3D%22transform%22%0D%0A%20%20%20%20type%3D%22rotate%22%0D%0A%20%20%20%20from%3D%220%2020%2020%22%0D%0A%20%20%20%20to%3D%22360%2020%2020%22%0D%0A%20%20%20%20dur%3D%220.5s%22%0D%0A%20%20%20%20repeatCount%3D%22indefinite%22%2F%3E%0D%0A%20%20%3C%2Fpath%3E%0D%0A%3C%2Fsvg%3E\');\n  }\n\n  /* buttons dropdown */\n  .btn-dropdown-site,\n  .btn-dropdown-site:link,\n  .btn-dropdown-site:visited {\n    color: #000000;\n    border-color: #b9bcc7;\n    background-color: #b9bcc7;\n  }\n  .btn-dropdown-site:hover,\n  .btn-dropdown-site.is-hover,\n  .btn-dropdown-site:focus,\n  .btn-dropdown-site:active,\n  .btn-dropdown-site.is-active {\n    border-color: #b9bcc7;\n    background-color: #9da0a9;\n  }\n\n  .site-btn-block,\n  .site-btn-block:link,\n  .site-btn-block:visited {\n    color: #ffffff;\n  }\n\n  .site-btn-block:hover,\n  .site-btn-block.is-hover,\n  .site-btn-block.is-active,\n  .site-btn-block:active {\n    box-shadow: inset 0px -4px 0px #ffffff;\n  }\n\n  .site-form-color input[type="email"],\n  .site-form-color input[type="text"],\n  .site-form-color input[type="search"],\n  .site-form-color input[type="password"],\n  .site-form-color input[type="tel"],\n  .site-form-color textarea {\n    color: #323232 !important;\n    border-color: #696969 !important;\n    background-color:  #fcfcfc !important;\n  }\n  .site-form-color label,\n  .site-form-color-minimal input[type="email"],\n  .site-form-color-minimal input[type="text"],\n  .site-form-color-minimal input[type="search"],\n  .site-form-color-minimal input[type="password"],\n  .site-form-color-minimal textarea {\n    color: #ffffff;\n    background: transparent !important;\n  }\n  .site-form-color input[type="text"]:focus,\n  .site-form-color input[type="search"]:focus,\n  .site-form-color input[type="password"]:focus,\n  .site-form-color input[type="tel"]:focus,\n  .site-form-color textarea:focus {\n    border-color: #7c7c7c !important;\n  }\n\n  .site-form-color input[type="text"]::-webkit-input-placeholder,\n  .site-form-color-minimal input[type="text"]::-webkit-input-placeholder,\n  .site-form-color input[type="email"]::-webkit-input-placeholder,\n  .site-form-color-minimal input[type="email"]::-webkit-input-placeholder,\n  .site-form-color input[type="search"]::-webkit-input-placeholder,\n  .site-form-color-minimal input[type="search"]::-webkit-input-placeholder,\n  .site-form-color input[type="password"]::-webkit-input-placeholder,\n  .site-form-color-minimal input[type="password"]::-webkit-input-placeholder,\n  .site-form-color input[type="tel"]::-webkit-input-placeholder,\n  .site-form-color-minimal input[type="tel"]::-webkit-input-placeholder,\n  .site-form-color textarea::-webkit-input-placeholder,\n  .site-form-color-minimal textarea::-webkit-input-placeholder {\n    color: #323232 !important;\n    opacity: .5;\n  }\n  .site-form-color input[type="text"]::-moz-input-placeholder,\n  .site-form-color-minimal input[type="text"]::-moz-input-placeholder,\n  .site-form-color input[type="email"]::-moz-input-placeholder,\n  .site-form-color-minimal input[type="email"]::-moz-input-placeholder,\n  .site-form-color input[type="search"]::-moz-input-placeholder,\n  .site-form-color-minimal input[type="search"]::-moz-input-placeholder,\n  .site-form-color input[type="password"]::-moz-input-placeholder,\n  .site-form-color-minimal input[type="password"]::-moz-input-placeholder,\n  .site-form-color input[type="tel"]::-moz-input-placeholder,\n  .site-form-color-minimal input[type="tel"]::-moz-input-placeholder,\n  .site-form-color textarea::-moz-input-placeholder,\n  .site-form-color-minimal textarea::-moz-input-placeholder {\n    color: #323232 !important;\n    opacity: .5;\n  }\n\n  .site-form-color input[type="text"]:-moz-input-placeholder,\n  .site-form-color-minimal input[type="text"]:-moz-input-placeholder,\n  .site-form-color input[type="email"]:-moz-input-placeholder,\n  .site-form-color-minimal input[type="email"]:-moz-input-placeholder,\n  .site-form-color input[type="search"]:-moz-input-placeholder,\n  .site-form-color-minimal input[type="search"]:-moz-input-placeholder,\n  .site-form-color input[type="password"]:-moz-input-placeholder,\n  .site-form-color-minimal input[type="password"]:-moz-input-placeholder,\n  .site-form-color input[type="tel"]:-moz-input-placeholder,\n  .site-form-color-minimal input[type="tel"]:-moz-input-placeholder,\n  .site-form-color textarea:-moz-input-placeholder,\n  .site-form-color-minimal textarea:-moz-input-placeholder {\n    color: #323232 !important;\n    opacity: .5;\n  }\n\n  .site-form-color input[type="text"]:-ms-input-placeholder,\n  .site-form-color-minimal input[type="text"]:-ms-input-placeholder,\n  .site-form-color input[type="email"]:-ms-input-placeholder,\n  .site-form-color-minimal input[type="email"]:-ms-input-placeholder,\n  .site-form-color input[type="search"]:-ms-input-placeholder,\n  .site-form-color-minimal input[type="search"]:-ms-input-placeholder,\n  .site-form-color input[type="password"]:-ms-input-placeholder,\n  .site-form-color-minimal input[type="password"]:-ms-input-placeholder,\n  .site-form-color input[type="tel"]:-ms-input-placeholder,\n  .site-form-color-minimal input[type="tel"]:-ms-input-placeholder,\n  .site-form-color textarea:-ms-input-placeholder,\n  .site-form-color-minimal textarea:-ms-input-placeholder {\n    color: #323232 !important;\n    opacity: .5;\n  }\n\n  .site-form-color .radio label .radio--icon {\n    border-color: #bebebb !important;\n  }\n  .site-form-color .radio.primary .circle-top span {\n    background: #b9bcc7 !important;\n  }\n  .site-form-color .radio.primary .circle-bottom span {\n    background: #94969f !important;\n  }\n\n  /* header */\n  .site-header {\n    border-bottom: 1px solid #2e2e2e;\n    background-color: #141414;\n\n\n  }\n  .site-header-background-color {\n    background-color: #141414;\n  }\n  @media only screen and (max-width: 40em) {\n     .site-header-background-color-small-only {\n      background-color: #141414;\n    }\n  }\n\n  /* modals */\n  .site-modal {\n    box-shadow: inset 0 0 2px #7c7c7c !important;\n  }\n\n  /* tooltips */\n  .tooltip.bottom-right:after,\n  .tooltip.bottom-left:after,\n  .tooltip.bottom:after,\n  .tooltip.bottom-right:before,\n  .tooltip.bottom-left:before,\n  .tooltip.bottom:before {\n    border-top-color: #7c7c7c;\n  }\n  .tooltip.right:after,\n  .tooltip.right:after\n  .tooltip.right-top:after,\n  .tooltip.right-top:before,\n  .tooltip.right-bottom:before,\n  .tooltip.right-bottom:before {\n    border-right-color: #7c7c7c;\n  }\n\n  .tooltip.left-bottom:after,\n  .tooltip.left-top:after,\n  .tooltip.left:after,\n  .tooltip.left-bottom:before,\n  .tooltip.left-top:before,\n  .tooltip.left:before {\n    border-left-color: #7c7c7c;\n  }\n\n  .tooltip.top-right:after,\n  .tooltip.top-left:after,\n  .tooltip.top:after,\n  .tooltip.top-right:before,\n  .tooltip.top-left:before,\n  .tooltip.top:before {\n    border-bottom-color: #7c7c7c;\n  }\n\n  .tooltip {\n    border: 1px solid #7c7c7c;\n    background: #7c7c7c;\n    color: #ffffff;\n  }\n\n  .reveal-modal {\n    border-color: #323232;\n    box-shadow: 0 0 3px #cbcbcb;\n  }\n  .reveal-modal-bg {\n    background-color: rgba(0,0,0,.8);\n  }\n  .play-outer-circle-large:hover {\n    border-color: #b9bcc7;\n  }\n  .forums-container tr.clickable:hover .thread-activity {\n    background-color: #141414;\n  }\n\n    [data-flash].is-inline.is-error {\n      background: rgba(211, 16, 74, 1) !important;\n    }\n    [data-flash].is-inline.is-error .flash-msg,\n    [data-flash].is-inline.is-error .flash-msg strong {\n      color: #fff !important;\n    }\n    [data-flash].is-inline.is-error .flash-close .icon-remove-mono {\n      display: none !important;\n    }\n    [data-flash].is-inline.is-error .flash-close .icon-remove-inverse {\n      display: block !important;\n    }\n\n  /* custom banner */\n\n\n</style>\n<style>\n  /* icons site primary */\n  .site-icon-arrow-right {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2223%22%20height%3D%2223%22%20viewBox%3D%220%200%2023%2023%22%3E%3Ctitle%3Eicons-arrows-set%20copy%3C%2Ftitle%3E%3Cpath%20d%3D%22M5.61%2C19.75l8.26-8.26L5.61%2C3.24c-0.68-.69.61-1.58%2C1-2%2C0.72-.69%2C1.76.78%2C2.2%2C1.22l3.39%2C3.39L17.41%2C11c0.71%2C0.69-.8%2C1.77-1.22%2C2.19L12.8%2C16.59%2C7.63%2C21.77c-0.7.67-1.58-.61-2-1A0.68%2C0.68%2C0%2C0%2C1%2C5.61%2C19.75Z%22%20fill%3D%22%23b9bcc7%22%2F%3E%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-menu {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Hamburger%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Crect%20x%3D%222.6%22%20y%3D%223.6%22%20width%3D%2218%22%20height%3D%223%22%20fill%3D%22%23ffffff%22%2F%3E%0D%0A%3Crect%20x%3D%222.6%22%20y%3D%2210%22%20width%3D%2218%22%20height%3D%223%22%20fill%3D%22%23ffffff%22%2F%3E%0D%0A%3Crect%20x%3D%222.6%22%20y%3D%2216.3%22%20width%3D%2218%22%20height%3D%223%22%20fill%3D%22%23ffffff%22%2F%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-chrevron-down {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21--%20Generator%3A%20Adobe%20Illustrator%2018.1.1%2C%20SVG%20Export%20Plug-In%20.%20SVG%20Version%3A%206.00%20Build%200%29%20%20--%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%20%20%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%22-294%20385%2023%2023%22%20enable-background%3D%22new%20-294%20385%2023%2023%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cg%3E%0D%0A%20%20%3Cpolygon%20fill%3D%22%23ffffff%22%20points%3D%22-271.1%2C392.9%20-275.4%2C388.7%20-282.5%2C395.8%20-289.6%2C388.7%20-293.9%2C392.9%20-286.7%2C400%20-286.8%2C400.1%20%0D%0A%20%20%20%20-282.5%2C404.3%20-282.5%2C404.3%20-282.5%2C404.3%20-278.2%2C400.1%20-278.3%2C400%20%20%22%2F%3E%0D%0A%3C%2Fg%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-chat {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21--%20Generator%3A%20Adobe%20Illustrator%2018.1.1%2C%20SVG%20Export%20Plug-In%20.%20SVG%20Version%3A%206.00%20Build%200%29%20%20--%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%22-294%20385%2023%2023%22%20enable-background%3D%22new%20-294%20385%2023%2023%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cg%3E%0D%0A%09%3Cg%3E%0D%0A%09%09%3Cpath%20fill%3D%22%23ffffff%22%20d%3D%22M-282.5%2C388.6c2.6%2C0%2C5.1%2C0.1%2C6.8%2C1.3c1.9%2C1.4%2C1.9%2C4.3%2C1.7%2C6c-0.2%2C2.1-1.2%2C3.5-3%2C4.2%0D%0A%09%09%09c-1.3%2C0.5-3%2C0.7-5.1%2C0.7c-1%2C0-2.1-0.1-3.4-0.2l-1-0.1l0%2C0l-1.2-0.4c-1.1-0.3-2.5-1-3.1-2.6c-0.7-1.8-0.7-4.2%2C0-5.9%0D%0A%09%09%09C-289.6%2C389.1-286.6%2C388.6-282.5%2C388.6%20M-282.5%2C386.6c-3.7%2C0-8.4%2C0.3-9.9%2C4.3c-0.9%2C2.2-0.9%2C5.1%2C0%2C7.4c0.7%2C2%2C2.4%2C3.2%2C4.3%2C3.8v3.2%0D%0A%09%09%09l2.6-2.7c1.1%2C0.1%2C2.4%2C0.2%2C3.6%2C0.2c2%2C0%2C4-0.2%2C5.7-0.9c2.6-1%2C4-3.1%2C4.3-5.8c0.3-2.8-0.1-6.1-2.6-7.9%0D%0A%09%09%09C-276.7%2C386.6-279.8%2C386.6-282.5%2C386.6L-282.5%2C386.6z%22%2F%3E%0D%0A%09%3C%2Fg%3E%0D%0A%3C%2Fg%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n\n  /* icons site social */\n  .site-icon-twitter {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%22-294%20385%2023%2023%22%20enable-background%3D%22new%20-294%20385%2023%2023%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cg%3E%0D%0A%09%3Cpath%20fill%3D%22%23ffffff%22%20d%3D%22M-272.5%2C390.3c-0.7%2C0.3-1.5%2C0.5-2.4%2C0.6c0.8-0.5%2C1.5-1.3%2C1.8-2.3c-0.8%2C0.5-1.7%2C0.8-2.6%2C1%0D%0A%09%09c-2.8-3-7.9-0.2-7%2C3.7c-3.4-0.1-6.4-1.7-8.4-4.2c-1.1%2C1.9-0.5%2C4.3%2C1.3%2C5.5c-0.7%2C0-1.3-0.2-1.9-0.5c0%2C1.9%2C1.4%2C3.7%2C3.3%2C4.1%0D%0A%09%09c-0.6%2C0.2-1.2%2C0.2-1.9%2C0.1c0.5%2C1.6%2C2%2C2.8%2C3.8%2C2.8c-1.7%2C1.3-3.9%2C2-6.1%2C1.7c4%2C2.6%2C9.5%2C2.5%2C13.4-0.5c3-2.3%2C4.8-6.1%2C4.6-9.9%0D%0A%09%09C-273.7%2C391.8-273.1%2C391.1-272.5%2C390.3C-273.2%2C390.6-273.1%2C391.1-272.5%2C390.3z%22%2F%3E%0D%0A%3C%2Fg%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-facebook {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21--%20Generator%3A%20Adobe%20Illustrator%2018.1.1%2C%20SVG%20Export%20Plug-In%20.%20SVG%20Version%3A%206.00%20Build%200%29%20%20--%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%22-294%20385%2023%2023%22%20enable-background%3D%22new%20-294%20385%2023%2023%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cpath%20fill%3D%22%23ffffff%22%20d%3D%22M-280.4%2C406.8v-9.3h3.4v-4h-3.4v-2.1c0-0.5%2C0.3-0.9%2C0.9-0.9h2.6v-4.5h-2.8c-3%2C0-5.3%2C2.4-5.3%2C5.5v2.1h-3.1v4%0D%0A%09h3.1v9.3h4.6L-280.4%2C406.8z%22%2F%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-tumblr {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cg%3E%0D%0A%09%3Cpath%20style%3D%22fill%3A%23ffffff%3B%22%20d%3D%22M17.6%2C20.4c-2%2C1-4.4%2C1.4-6.6%2C0.8c-2.3-0.6-3.4-2.3-3.4-4.5v-7H5.4V6.9c2.5-0.8%2C3.7-2.9%2C4-5.4h2.8%0D%0A%09%09v5.1h4.7v3.1h-4.7v5.1c0%2C1-0.2%2C2.4%2C0.9%2C3c1.4%2C0.8%2C3.2%2C0.2%2C4.5-0.6V20.4z%22%2F%3E%0D%0A%3C%2Fg%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-envelope {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cg%3E%0D%0A%09%3Cpath%20style%3D%22fill%3A%23ffffff%3B%22%20d%3D%22M22.6%2C9c0-0.2-0.1-0.5-0.3-0.6L12%2C0.8c-0.3-0.2-0.6-0.2-0.9%2C0L0.7%2C8.4C0.5%2C8.5%2C0.4%2C8.8%2C0.4%2C9v12.6%0D%0A%09%09c0%2C0.4%2C0.3%2C0.8%2C0.8%2C0.8h20.7c0.4%2C0%2C0.8-0.4%2C0.8-0.8V9z%20M11.5%2C2.4l9.6%2C7v0.9l-6.6%2C4.7l5.2%2C3.3c0.8%2C0.5%2C0%2C1.8-0.8%2C1.3L11.5%2C15%0D%0A%09%09l-7.4%2C4.7c-0.8%2C0.5-1.7-0.8-0.8-1.3l5.2-3.3l-6.6-4.7V9.4L11.5%2C2.4z%22%2F%3E%0D%0A%09%3Cpath%20style%3D%22fill%3A%23ffffff%3B%22%20d%3D%22M19.7%2C18.4%22%2F%3E%0D%0A%09%3Cpolygon%20style%3D%22fill%3A%23ffffff%3B%22%20points%3D%2211.5%2C13.1%2013.4%2C14.3%2019.5%2C9.9%2011.5%2C4%203.5%2C9.9%209.6%2C14.3%20%09%22%2F%3E%0D%0A%3C%2Fg%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-check {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cg%3E%0D%0A%09%0D%0A%09%09%3Crect%20x%3D%2211.9%22%20y%3D%222.1%22%20transform%3D%22matrix%28-0.7072%20-0.7071%200.7071%20-0.7072%2016.2216%2029.719%29%22%20style%3D%22fill%3A%23ffffff%3B%22%20width%3D%224.7%22%20height%3D%2218.8%22%2F%3E%0D%0A%09%0D%0A%09%09%3Crect%20x%3D%220.2%22%20y%3D%2211.3%22%20transform%3D%22matrix%280.7072%200.707%20-0.707%200.7072%2011.5962%20-0.598%29%22%20style%3D%22fill%3A%23ffffff%3B%22%20width%3D%2212.5%22%20height%3D%224.7%22%2F%3E%0D%0A%3C%2Fg%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n\n  /* icons site secondary */\n  .site-icon-clock {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cg%3E%0D%0A%09%3Cpath%20style%3D%22fill%3A%23bebebb%3B%22%20d%3D%22M11.5%2C22c-4.5%2C0-8.6-3-10-7.3c-1.4-4.3%2C0.2-9.2%2C4-11.8c3.7-2.6%2C8.9-2.5%2C12.5%2C0.3%0D%0A%09%09c3.6%2C2.8%2C5%2C7.7%2C3.4%2C12C19.8%2C19.3%2C15.8%2C22%2C11.5%2C22z%20M11.5%2C3.3C7%2C3.3%2C3.3%2C7%2C3.3%2C11.5c0%2C4.5%2C3.7%2C8.2%2C8.2%2C8.2c4.5%2C0%2C8.2-3.7%2C8.2-8.2%0D%0A%09%09C19.7%2C7%2C16%2C3.3%2C11.5%2C3.3z%22%2F%3E%0D%0A%09%3Cpath%20style%3D%22fill%3A%23ffffff%3B%22%20d%3D%22M12.9%2C12.9v-2.9V5.5c0-0.4-0.4-0.8-0.8-0.8h-1.3c-0.4%2C0-0.8%2C0.4-0.8%2C0.8v4.5H6.5%0D%0A%09%09c-0.4%2C0-0.8%2C0.4-0.8%2C0.8v1.3c0%2C0.4%2C0.4%2C0.8%2C0.8%2C0.8h3.5H12.9z%22%2F%3E%0D%0A%3C%2Fg%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-reply {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cg%3E%0D%0A%09%3Cpath%20style%3D%22fill%3A%23bebebb%3B%22%20d%3D%22M20.7%2C20c-0.1%2C0.3-0.3%2C0.7-0.4%2C0.9c-0.1%2C0.1-0.2%2C0.2-0.3%2C0.2c-0.2%2C0-0.4-0.2-0.4-0.4%0D%0A%09%09c0-0.2%2C0-0.4%2C0.1-0.6c0-0.5%2C0.1-1%2C0.1-1.5c0-5.7-3.4-6.7-8.6-6.7H8.4V15c0%2C0.4-0.3%2C0.8-0.8%2C0.8c-0.2%2C0-0.4-0.1-0.5-0.2L1%2C9.4%0D%0A%09%09C0.8%2C9.2%2C0.8%2C9%2C0.8%2C8.8c0-0.2%2C0.1-0.4%2C0.2-0.5l6.1-6.1C7.3%2C2%2C7.5%2C1.9%2C7.7%2C1.9c0.4%2C0%2C0.8%2C0.3%2C0.8%2C0.8v3.1h2.7%0D%0A%09%09c3.9%2C0%2C8.8%2C0.7%2C10.5%2C4.8c0.5%2C1.3%2C0.6%2C2.6%2C0.6%2C4C22.2%2C16.2%2C21.4%2C18.4%2C20.7%2C20z%22%2F%3E%0D%0A%3C%2Fg%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-remove {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cpath%20style%3D%22fill%3A%23bebebb%3B%22%20d%3D%22M19.6%2C17.5l-6-6l6-6c0.3-0.3%2C0.3-0.8%2C0-1.1l-0.9-0.9c-0.3-0.3-0.8-0.3-1.1%2C0l-6%2C6l-6-6%0D%0A%09c-0.3-0.3-0.8-0.3-1.1%2C0L3.4%2C4.4c-0.3%2C0.3-0.3%2C0.8%2C0%2C1.1l6%2C6l-6%2C6c-0.3%2C0.3-0.3%2C0.8%2C0%2C1.1l0.9%2C0.9c0.3%2C0.3%2C0.8%2C0.3%2C1.1%2C0l6-6l6%2C6%0D%0A%09c0.3%2C0.3%2C0.8%2C0.3%2C1.1%2C0l0.9-0.9C19.9%2C18.3%2C19.9%2C17.8%2C19.6%2C17.5z%22%2F%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-back {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cpath%20style%3D%22fill%3A%23bebebb%3B%22%20d%3D%22M11.5%2C20.9l1-1c0.3-0.3%2C0.3-0.7%2C0-0.9L6%2C12.5h14.6c0.4%2C0%2C0.7-0.3%2C0.7-0.7v-1.3c0-0.4-0.3-0.7-0.7-0.7%0D%0A%09H6l6.5-6.5c0.3-0.3%2C0.3-0.7%2C0-0.9l-1-1c-0.3-0.3-0.7-0.3-0.9%2C0l-9.4%2C9.4c-0.3%2C0.3-0.3%2C0.7%2C0%2C0.9l9.4%2C9.4%0D%0A%09C10.9%2C21.2%2C11.3%2C21.2%2C11.5%2C20.9z%22%2F%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-link {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cpath%20style%3D%22fill-rule%3Aevenodd%3Bclip-rule%3Aevenodd%3Bfill%3A%23bebebb%3B%22%20d%3D%22M19.3%2C8.2c-1.8%2C1.8-3.7%2C4.6-6.6%2C3.8c0.8-0.8%2C1.6-1.6%2C2.4-2.4%0D%0A%09c0.9-0.9%2C1.8-1.8%2C2.6-2.7c1.9-1.9-0.8-4.8-2.6-2.9c-0.5%2C0.5-0.9%2C0.9-1.4%2C1.4c-0.2%2C0.2-1.4%2C1.1-1.4%2C1.4l0%2C0c-0.8%2C0.8-1.6%2C1.6-2.4%2C2.4%0D%0A%09c-0.4-1.3%2C0-2.8%2C1-3.8c1.3-1.3%2C2.7-3.3%2C4.5-3.8C19.3%2C0.5%2C22.1%2C5.4%2C19.3%2C8.2%22%2F%3E%0D%0A%3Cpath%20style%3D%22fill-rule%3Aevenodd%3Bclip-rule%3Aevenodd%3Bfill%3A%23bebebb%3B%22%20d%3D%22M7.9%2C15.6C7%2C16.5%2C5.5%2C15%2C6.5%2C14.1c1.4-1.4%2C2.7-2.7%2C4.1-4.1%0D%0A%09c0.8-0.8%2C1.5-1.5%2C2.3-2.3c0.2-0.2%2C0.3-0.3%2C0.5-0.5c1-0.6%2C2%2C0.7%2C1.2%2C1.6C12.4%2C11.2%2C10.1%2C13.3%2C7.9%2C15.6z%22%2F%3E%0D%0A%3Cpath%20style%3D%22fill-rule%3Aevenodd%3Bclip-rule%3Aevenodd%3Bfill%3A%23bebebb%3B%22%20d%3D%22M5.8%2C13.4c-0.9%2C0.9-1.8%2C1.8-2.6%2C2.7c-1.9%2C1.9%2C0.8%2C4.8%2C2.6%2C2.9%0D%0A%09c0.9-0.9%2C2-1.8%2C2.8-2.9c0.8-0.8%2C1.6-1.6%2C2.4-2.4c0.4%2C1.3%2C0%2C2.8-1%2C3.8c-1.3%2C1.3-2.7%2C3.3-4.5%2C3.8c-3.8%2C1.1-6.7-3.8-3.9-6.6%0D%0A%09C3.4%2C13%2C5.4%2C10.2%2C8.2%2C11C7.4%2C11.8%2C6.6%2C12.6%2C5.8%2C13.4%22%2F%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-trash {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%221792px%22%20height%3D%221792px%22%20viewBox%3D%22590%20-500%201792%201792%22%20enable-background%3D%22new%20590%20-500%201792%201792%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cpath%20fill%3D%22%23bebebb%22%20d%3D%22M1294%2C876V172c0-9.3-3-17-9-23s-13.7-9-23-9h-64c-9.3%2C0-17%2C3-23%2C9s-9%2C13.7-9%2C23v704c0%2C9.3%2C3%2C17%2C9%2C23%0D%0A%09s13.7%2C9%2C23%2C9h64c9.3%2C0%2C17-3%2C23-9S1294%2C885.3%2C1294%2C876z%20M1550%2C876V172c0-9.3-3-17-9-23s-13.7-9-23-9h-64c-9.3%2C0-17%2C3-23%2C9%0D%0A%09s-9%2C13.7-9%2C23v704c0%2C9.3%2C3%2C17%2C9%2C23s13.7%2C9%2C23%2C9h64c9.3%2C0%2C17-3%2C23-9S1550%2C885.3%2C1550%2C876z%20M1806%2C876V172c0-9.3-3-17-9-23%0D%0A%09s-13.7-9-23-9h-64c-9.3%2C0-17%2C3-23%2C9s-9%2C13.7-9%2C23v704c0%2C9.3%2C3%2C17%2C9%2C23s13.7%2C9%2C23%2C9h64c9.3%2C0%2C17-3%2C23-9S1806%2C885.3%2C1806%2C876z%0D%0A%09%20M1262-116h448l-48-117c-4.7-6-10.3-9.7-17-11h-317c-6.7%2C1.3-12.3%2C5-17%2C11L1262-116z%20M2190-84v64c0%2C9.3-3%2C17-9%2C23s-13.7%2C9-23%2C9h-96%0D%0A%09v948c0%2C55.3-15.7%2C103.2-47%2C143.5s-69%2C60.5-113%2C60.5h-832c-44%2C0-81.7-19.5-113-58.5s-47-86.2-47-141.5V12h-96c-9.3%2C0-17-3-23-9%0D%0A%09s-9-13.7-9-23v-64c0-9.3%2C3-17%2C9-23s13.7-9%2C23-9h309l70-167c10-24.7%2C28-45.7%2C54-63s52.3-26%2C79-26h320c26.7%2C0%2C53%2C8.7%2C79%2C26%0D%0A%09s44%2C38.3%2C54%2C63l70%2C167h309c9.3%2C0%2C17%2C3%2C23%2C9S2190-93.3%2C2190-84z%22%2F%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-replay {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2223%22%20height%3D%2223%22%20viewBox%3D%220%200%2023%2023%22%3E%3Cpath%20d%3D%22M16.4%2016.9c-2.8%202.4-7.1%202.3-9.8-.3l-2%202c3.8%203.8%209.9%203.9%2013.9.3l-2.1-2z%22%20fill%3D%22%23b9bcc7%22%2F%3E%3Cpath%20d%3D%22M15.3%205.3l.9.6c3%202.6%203.7%207.2.9%2010.3l2%202c2.3-2.5%203.1-6%202.3-9.3-.9-3.6-3.8-6.4-7.4-7.3-3.8-.8-8.1.7-10.4%204l.1-1.2c0-.2%200-.5-.2-.7-.6-.8-1.7-.4-1.8.4l-.5%204.6c-.1.6.4%201.1%201%201.1h4.6c.9%200%201.3-1.1.8-1.7-.3-.2-.5-.3-.8-.3h-1.2c2-3.2%206.3-4.4%209.7-2.5z%22%20fill%3D%22%23b9bcc7%22%2F%3E%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-alt-download {\n    background-image: url(\'data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2223%22%20height%3D%2223%22%20viewBox%3D%220%200%2023%2023%22%3E%3Cg%20fill-rule%3D%22evenodd%22%20clip-rule%3D%22evenodd%22%20fill%3D%22%23ffffff%22%3E%3Cpath%20d%3D%22M17.3%2017.2c-1.1%200-1.4-1.5-.5-2%20.4-.2%201-.1%201.5-.3.6-.2%201-.6%201.4-1%201.3-1.6.5-4.1-1.4-4.8-.3-.1-.7-.2-1.1-.2-.2%200-.3-.8-.4-1-.4-.9-1-1.8-1.7-2.4-1.8-1.6-4.5-1.8-6.6-.5-1%20.5-1.7%201.4-2.2%202.3-.2.3-.3.7-.4%201-.2.7-.2.5-.8.7-1.9.4-3%202.5-2.2%204.3.4.9%201.3%201.6%202.3%201.8.5.1%201%200%201.3.4.4.4.3%201.1-.2%201.4-.7.6-2.2.1-2.9-.3-1-.5-1.8-1.3-2.3-2.3-.8-1.6-.7-3.5.2-4.9.4-.8%201.1-1.4%201.8-1.9.4-.2.9-.3%201.1-.7.2-.4.5-.8.7-1.2%202.1-3.1%206.2-4.4%209.6-2.9%201.6.7%203%201.9%203.8%203.4.2.4.3.9.7%201.1l1.2.6c.8.5%201.4%201.2%201.8%202%201.6%203.4-1%207.4-4.7%207.4%22%20fill%3D%22%23000000%22%3E%3C%2Fpath%3E%3Cpath%20d%3D%22M15.4%2017.2l-3.1%203.4c-.4.4-1.1.5-1.5%200-.5-.6-1.1-1.2-1.6-1.8-.2-.2-1.2-1.6-1.5-1.6-.9-1%20.7-2.4%201.5-1.4l1.3%201.4v-3.9c0-.4-.1-.9%200-1.3%200-1.3%202.1-1.3%202.1%200v5.2c.5-.5%201-1.4%201.7-1.7.8-.4%201.8.9%201.1%201.7z%22%20fill%3D%22%23000000%22%3E%3C%2Fpath%3E%3C%2Fg%3E%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-random {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2223%22%20height%3D%2223%22%20viewBox%3D%220%200%2023%2023%22%3E%3Cpath%20d%3D%22M5.7%2015.2v-6H8L4%204.6-.1%209.2h2.3v7c0%201.3%201%202.3%202.3%202.3H15l-3-3.2H5.7z%22%20fill%3D%22#ffffff%22%2F%3E%3Cpath%20d%3D%22M20.7%2013.8v-7c0-1.3-1-2.3-2.3-2.3H8l3%203.2h6.3v6H15l4.1%204.6%204.1-4.6h-2.5z%22%20fill%3D%22#ffffff%22%2F%3E%3C%2Fsvg%3E");\n    background-repeat: no-repeat;\n  }\n  .site-icon-visa {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2271%22%20height%3D%2243%22%20viewBox%3D%220%200%2071%2043%22%3E%3Cpath%20d%3D%22M51.9%2017.9c-.2.5-.5%201.4-.5%201.4l-1.5%204H53c-.2-.7-.9-4.2-.9-4.2l-.2-1.2zM69%200H2C.9%200%200%20.9%200%202v39c0%201.1.9%202%202%202h67c1.1%200%202-.9%202-2V2c0-1.1-.9-2-2-2zM11.6%2014h6c.8%200%201.5.3%201.7%201.2l1.3%206.6c-1.3-3.5-4.4-6.4-9.1-7.6l.1-.2zm7.1%2014.4l-3-11.5c2.5%201.5%204.2%203.5%205%205.9l.2%201%203.7-9.8h3.9l-5.9%2014.4h-3.9zm9%200L30.1%2014h3.7l-2.3%2014.4h-3.8zm10.2.2c-1.7%200-3.3-.4-4.2-.8l.5-3.2.5.2c1.2.5%202.4.8%203.8.8.9-.1%202.2-.6%201.9-1.7-.3-.9-1.6-1.3-2.3-1.7-1-.5-2-1.2-2.5-2.2-1-2%20.2-4.2%201.9-5.3%202.2-1.3%205.1-1.3%207.4-.4l-.5%203.1-.3-.2c-1-.4-4.9-1.3-4.9.7%200%20.8%201.2%201.3%201.8%201.6%201%20.5%202.1%201%202.8%202%201.1%201.4.8%203.5-.2%204.8-1.4%201.8-3.6%202.3-5.7%202.3zm16.2-.2s-.3-1.7-.4-2.2h-4.8c-.1.4-.8%202.2-.8%202.2h-3.9l5.5-13.2c.4-.9%201.1-1.2%202-1.2h2.9l3%2014.4h-3.5z%22%20fill%3D%22#ffffff%22%3E%3C%2Fpath%3E%3C%2Fsvg%3E");\n    background-repeat: no-repeat;\n  }\n  .site-icon-amex {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2271%22%20height%3D%2243%22%20viewBox%3D%220%200%2071%2043%22%3E%3Cpath%20d%3D%22M35.5%2023.5h-5l-2%202.2-1.9-2.2h-6.3v6.6h6.2l2-2.2%201.9%202.2h3.1v-2.2h2c1.5%200%202.7-.6%202.7-2.2%200-1.7-1.3-2.2-2.7-2.2zm-9.9%205.2h-3.9v-1.3h3.5V26h-3.5v-1.2h4l1.7%201.9-1.8%202zm6.3.8l-2.4-2.7%202.4-2.6v5.3zm3.6-3h-2v-1.7h2.1c1.2%200%201.2%201.7-.1%201.7zm9.4-7.8h1.8l.6-1.5h3.3l.6%201.5h3.2v-5l3%205h2.2v-6.6H58v4.6l-2.8-4.6h-2.4v6.2l-2.7-6.2h-2.4l-2.2%205.2H45c-1.6%200-2.1-2.3-1.1-3.4.3-.3.6-.4%201.2-.4h1.5v-1.4h-1.5c-2.1%200-3.2%201.3-3.2%203.3-.1%202%201%203.3%203%203.3zm4.1-5.5l1.1%202.6h-2.2l1.1-2.6zm-15.2%203.1h1.7c.6%200%20.9.1%201.1.3.3.3.2.9.2%201.3v.8h1.6v-1.3c0-.6%200-.9-.3-1.2-.1-.2-.4-.4-.8-.6.9-.4%201.3-1.4%201-2.3-.3-1.2-1.7-1.2-2.6-1.2h-3.6v6.6h1.6v-2.4zm0-2.8h1.9c.3%200%20.6%200%20.7.1.5.2.4%201%200%201.2-.2.1-.5.1-.8.1h-1.9v-1.4zm-2.7%203.8h-3.7V16H31v-1.3h-3.6v-1.2h3.7v-1.4h-5.3v6.6h5.3zM23%2018.7h1.6v-6.6H22l-1.9%204.5-2.1-4.5h-2.6v6.2l-2.7-6.2h-2.3l-2.8%206.6h1.7l.6-1.5h3.3l.6%201.5H17v-5.2l2.3%205.2h1.4l2.3-5.2v5.2zm-12.5-2.9l1.1-2.6%201.1%202.6h-2.2zm35.6%2014.3h5.3v-1.4h-3.7v-1.3h3.6V26h-3.6v-1.2h3.7v-1.3h-5.3zM61.8%2026H61c-.3%200-.6%200-.8-.2-.2-.2-.1-.6%200-.7l.1-.1c.2-.1.3-.1.6-.1h2.9v-1.4h-3.2c-1%200-1.6.4-1.9.8l-.1.1-.1.2c-.1.1-.1.3-.1.4v.7l.1.6c0%20.1%200%20.1.1.2%200%20.1.1.2.1.2l.1.1.1.1s.1%200%20.1.1c.1.1.2.1.3.1%200%200%20.1%200%20.1.1.7.2%201.4.2%202.2.2%201%200%20.9%201.2-.1%201.2h-3V30h3c1.1%200%202.2-.4%202.4-1.6.4-1.6-.6-2.3-2.1-2.4zm-16.5-.8c0-.8-.3-1.2-.8-1.5-.5-.3-1.1-.3-1.9-.3H39V30h1.6v-2.4h1.7c1.1%200%201.4.5%201.4%201.6v.8h1.6v-1.3c0-.8-.2-1.5-1-1.8.3-.1%201-.6%201-1.7zm-2.1%201c-.2.1-.5.1-.8.1h-1.9v-1.5h1.9c.3%200%20.6%200%20.8.1.5.3.4%201%200%201.3-.2.1.2-.2%200%200zM69%200H2C.9%200%200%20.9%200%202v39c0%201.1.9%202%202%202h67c1.1%200%202-.9%202-2V2c0-1.1-.9-2-2-2zM18.2%2018.9V20h-5.3l-.6-1.5h-1.4l-.6%201.5H5.5l4-9.2h4.1l.5%201.1v-1.1H19l1.1%202.4%201-2.4h15.3c.7%200%201.3.1%201.8.5v-.5h4.2v.5c.7-.4%201.6-.5%202.6-.5h6.1l.6%201.1v-1.1h4.5l.6%201.1v-1.1h4.4V20h-4.4l-.8-1.4V20h-5.5l-.6-1.5h-1.4l-.8%201.5h-2.9c-1.1%200-2-.3-2.5-.6v.6h-6.8v-2.1c0-.3-.1-.3-.2-.3H35V20H21.9v-1.1l-.5%201.1h-2.8l-.4-1.1zm46.9%2011c-.7%201.3-2%201.5-3.3%201.5h-4.4v-.6c-.5.4-1.3.6-2.1.6h-13v-2.1c0-.3%200-.3-.3-.3h-.2v2.4h-4.2v-2.5c-.7.3-1.5.3-2.2.3h-.5v2.2h-5.1L28.5%2030l-1.3%201.4h-8.3v-9.2h8.5l1.2%201.4%201.3-1.4h5.7c.7%200%201.7.1%202.2.5v-.5h5.1c.5%200%201.5.1%202.1.5v-.5h7.7v.5c.4-.4%201.2-.5%201.9-.5h4.3v.5c.4-.3%201-.5%201.8-.5h4.5v3.3h-.9c1.3.9%201.6%203%20.8%204.4-.4.7.3-.5%200%200zM39.4%2012.1H41v6.6h-1.6zM55.6%2026h-.8c-.5%200-1.5-.5-.7-1%20.2-.1.3-.1.6-.1h2.9v-1.4h-3.2c-1.5%200-2.6%201.2-2.1%202.7.4%201.3%202.1%201.2%203.2%201.3%201%200%20.9%201.2-.1%201.2h-3.1v1.4h3.1c1.1%200%202.2-.4%202.4-1.6.4-1.7-.7-2.4-2.2-2.5z%22%20fill%3D%22#ffffff%22%3E%3C%2Fpath%3E%3C%2Fsvg%3E");\n    background-repeat: no-repeat;\n  }\n  .site-icon-mc {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2271%22%20height%3D%2243%22%20viewBox%3D%220%200%2071%2043%22%3E%3Cpath%20d%3D%22M36.2%2022.5c.2-.9.3-2.2-.8-2.7-1.4-.7-3%20.1-3.6%201.5-.3.7-.3%201.5-.1%202.2.6%201.4%202.7%201.8%204%201.1l.2-1.1c-.9.2-3.1.5-2.9-1h3.2zm-1.8-1.9c.6%200%20.7.4.6.9h-1.8c.2-.6.6-.9%201.2-.9zm-3.8%202.3l.4-2.1h.7l.3-1.1h-.8l.3-1.4h-1.4l-1%204.9c-.3%201.3%201%202.1%202.2%201.4l.1-1c-.8%200-.8-.3-.8-.7zm8-2.7l-.1-.5h-1.2l-1%205.1h1.4l.4-2.3c.4-1.5%201.1-1.4%201.4-1.4l.4-1.4c-.7%200-1.1.2-1.3.5zm-10.2%201.9c-.2-.1-2.1-1-1-1.3.3%200%201%200%201.4.1l.3-1c-1.3-.5-3.7-.2-3.6%201.6.1%201.1%201.3.9%201.8%201.6.7.9-1.9.6-2.2.5l-.1%201c.8.2%202%20.3%202.8%200%201-.3%201.7-1.9.6-2.5-.3-.2.3.1%200%200zm14.5-2.4c.5%200%20.7%200%201.1.3l.9-.9c-1.7-1.9-4.5-.3-5.1%201.8-.7%202.7%201.4%204.8%204%203.6l-.4-1.3c-2.8%201-2.8-3.5-.5-3.5zM23%2019.6c-.4%200-.9%200-1.8.2l-.2%201.1c.6-.1%202.9-.7%202.6.6-1.1.1-2.2-.1-2.9%201-1.1%201.6.8%203.2%202.3%202.1l.2.3h1.3L25%2022c.2-.7.4-2.4-2-2.4zm.3%203.4c-.1.5-1.3%201-1.5.3-.3-.9%201.1-1%201.6-.9l-.1.6zm27.9-2.9l-.2-.5h-1.2l-1%205.2h1.4l.4-2.4c.4-1.5%201.1-1.5%201.4-1.5l.4-1.4c-.6.1-1%20.4-1.2.6zM69%200H2C.9%200%200%20.9%200%202v39c0%201.1.9%202%202%202h67c1.1%200%202-.9%202-2V2c0-1.1-.9-2-2-2zM45.1%2035.5c-3.4%200-6.4-1.2-8.8-3.1%201.8-1.8%203.2-4%204-6.5h-1.2c-.7%202.2-2%204.2-3.7%205.8-1.7-1.6-2.9-3.6-3.6-5.8h-1.2c.8%202.5%202.1%204.7%204%206.5-4%203.2-9.5%204.1-14.3%202-4-1.7-7.1-5.4-8.1-9.6C10%2015.4%2018%206.4%2027.5%207.6c2.6.3%205.1%201.4%207.1%203-1.8%201.8-3.2%204-4%206.5h1.2c.7-2.2%202-4.2%203.6-5.8%201.7%201.6%202.9%203.6%203.7%205.8h1.2c-.8-2.5-2.2-4.7-4-6.5%207.6-6.2%2019.4-2.5%2022.2%206.8%202.7%208.9-4.1%2018.1-13.4%2018.1zm7.2-13.7c-.5%201.6.5%203%201.5%203%20.5%200%20.9-.2%201.2-.4l.2.4h1.5l1.2-6.6h-1.5l-.4%202c-.7-.5-1.2-.8-2-.6-1%20.3-1.5%201.3-1.7%202.2-.5%201.6.7-2.4%200%200zm3.1-.8c.2.2.2.8.1%201.3-.2.4-.4.9-.9%201-.6.1-1-.4-1-1%20.1-.6%201.1-2.2%201.8-1.3.2.3-.4-.5%200%200zm-36.9-2.5l-1.7%203.8-.3-3.8h-2.2l-1.2%206.3h1.4l.8-4.5.4%204.5h1.4l1.9-4.4-.8%204.4h1.5l1.1-6.3zm28.4%201.1c-.4%200-.9%200-1.8.2l-.2%201.1c.6-.1%202.9-.7%202.6.6-1.1.1-2.2-.1-3%201-1.1%201.6.8%203.2%202.3%202.1l.2.3h1.3l.6-2.8c.1-.8.4-2.5-2-2.5zm.2%203.4c-.1.5-1.3%201-1.5.3-.3-.9%201.1-1%201.6-.9l-.1.6z%22%20fill%3D%22#ffffff%22%3E%3C%2Fpath%3E%3C%2Fsvg%3E");\n    background-repeat: no-repeat;\n  }\n  .site-icon-discover {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2271%22%20height%3D%2243%22%20viewBox%3D%220%200%2071%2043%22%3E%3Cpath%20d%3D%22M6.5%2012.1v6.5s4.2.5%204.2-3.2v-.1c-.1-3.6-4.2-3.2-4.2-3.2zM69%200H2C.9%200%200%20.9%200%202v39c0%201.1.9%202%202%202h67c1.1%200%202-.9%202-2V2c0-1.1-.9-2-2-2zm-9.8%2010.4h3.2c1.7%200%203.1%201.2%203.1%202.9%200%201.3-.8%202.5-2.1%202.9l3%204.2H64l-2.7-4h-.2v4h-1.9v-10zm-6.8%200H58v1.8h-3.5v2.2h3.4v1.8h-3.4v2.5H58v1.8h-5.5V10.4zm-8.2%200l2.7%206.5%202.6-6.5h2.1l-4.2%2010.2h-1.2L42%2010.4h2.2zm-31.5%205c0%203.2-2.9%204.9-4.8%204.9H4.6v-9.9h3.3c1.9%200%204.8%201.8%204.8%205zm2.8%204.9h-2v-10h2v10zm4.2.3c-2.5%200-3.6-1.9-3.6-1.9l1.2-1.2c.7%201%203.1%202.2%203.5.4.5-1.8-1.8-1.9-2.9-2.4-1.7-.9-1.8-3.4-.2-4.6%201.5-1.1%203.5-.6%204.8.6l-1.1%201.3c-.7-.8-3-1.2-2.9.4%200%201%202%201.3%202.7%201.7%203.2%201.3%201.6%205.7-1.5%205.7zm11.6-.6c-3.4%201.8-7.7-.8-7.7-4.6%200-3.9%204.2-6.4%207.7-4.6v2.4c-2.1-2.1-5.8-.8-5.8%202.3%200%203%203.7%204.4%205.8%202.3V20zm6-9.8c2.8%200%205.2%202.3%205.2%205.2%200%202.8-2.3%205.2-5.2%205.2-2.8%200-5.2-2.3-5.2-5.2%200-2.8%202.4-5.2%205.2-5.2zM68%2040H32.8C45%2036.8%2057.2%2032.4%2068%2025.7V40zm-4.4-26.5v-.1c0-.6-.5-1.4-1.3-1.4h-1.1v2.9h1.1c.8%200%201.3-.7%201.3-1.4z%22%20fill%3D%22#ffffff%22%3E%3C%2Fpath%3E%3C%2Fsvg%3E");\n    background-repeat: no-repeat;\n  }\n  .site-icon-jcb {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2271%22%20height%3D%2243%22%20viewBox%3D%220%200%2071%2043%22%3E%3Cpath%20d%3D%22M69%200H2C-.1%200%200%202.1%200%203.5v37.4C0%2042.6%201.4%2043%202.7%2043h66c1.4%200%202.4-.8%202.4-2.3V3c-.1-1.4-.3-3-2.1-3zM28.3%2030.8c0%202.4-1.9%204.4-4.3%204.6-2.1.3-4.3%200-6.5%200-.9%200-.8.1-.8-.8v-9c2.8.7%209.5%201.7%2010.4-2.2.3-1.2.1-2.6.1-3.8v-2.1c0-.4.1-.5-.3-.5h-3.7v3.4c0%201.5.3%203.5-1.5%204.2-.8.3-1.7.2-2.5%200-.9-.2-1.8-.5-2.6-.9-.1-.1%200-.5%200-.6v-6.4c0-2.1-.4-4.7.5-6.7.5-1%201.3-1.6%202.2-2.1%201.4-.6%208.7-.5%208.7-.3v15.1c.3%202.7.3%205.4.3%208.1zm13%20.1c0%202.4-2%204.4-4.4%204.6-.9.1-2%200-2.9%200h-4c-.4%200-.2-1.3-.2-1.6v-9.1c1.4%201.2%203.4%201.5%205.2%201.5%201.1%200%202.1%200%203.1-.2l1.2-.2c.1%200%20.6-.1.7-.2.2-.3%200-1.5%200-1.9v-.1c-1.4.7-3.4%201.4-5%201-3-.4-3.4-4.8-.8-6.1%201.9-.7%204.1-.1%205.8.7v-1.8c0-.3-.3-.3-.5-.3-.4-.1-.8-.2-1.2-.2-.9-.1-1.9-.2-2.8-.2-1.8%200-3.7.2-5.3%201.2l-.5.3v-3.5c0-1.2-.2-2.6.1-3.8.3-1.3%201.2-2.2%202.2-2.9.7-.4%201.4-.6%202.2-.6%202.3-.1%204.6%200%206.8%200%20.3%200%20.3%2021.2.3%2023.4zm12.9%200c0%202.5-2.1%204.5-4.5%204.6-2.3.1-4.6%200-6.9%200-.2%200-.1-6.1-.1-6.7V26h7c.4%200%20.7%200%201.1-.1%201.3-.3%202.5-1.4%202-2.9-.3-.9-1.3-1.4-2.3-1.6h-.3l1.1-.4c.7-.4%201.1-1.2%201-2-.1-1.2-1.2-1.8-2.3-1.9-1.8-.2-3.8%200-5.6%200h-1.6c-.3%200-.2-.3-.2-.6v-4.1c0-2.5%201.8-4.6%204.3-4.8%202.2-.2%204.4%200%206.6%200%20.6%200%20.7-.1.7.5v22.8zm-6.5-8.8c1.5.2%201.6%202.5%200%202.5h-2.3c-.1%200-.4.1-.4-.1v-2.4c0-.2.4-.1.5-.1.7.1%201.4.1%202.2.1zm-2.8-1.4v-2c0-.6.9-.3%201.3-.3.5%200%201.5-.2%201.9.3.7.5.4%201.7-.4%202-.8.2-1.9%200-2.8%200z%22%20fill%3D%22#ffffff%22%3E%3C%2Fpath%3E%3C%2Fsvg%3E");\n    background-repeat: no-repeat;\n  }\n  .site-icon-calendar {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2223%22%20height%3D%2223%22%20viewBox%3D%220%200%2023%2023%22%3E%3Cpath%20d%3D%22M19.3%202.8h-1.1v2.5h-4V2.8H8.8v2.5h-4V2.8H3.7c-1.4%200-2.5%201.1-2.5%202.5v14.9c0%201.4%201.1%202.5%202.5%202.5h15.6c1.4%200%202.5-1.1%202.5-2.5V5.3c0-1.4-1.1-2.5-2.5-2.5zm0%2017.4H3.7v-9.9h15.6v9.9zM7.7.3H5.9v4.2h1.7V.3zm9.4%200h-1.7v4.2h1.7V.3z%22%20fill%3D%22#ffffff%22%2F%3E%3Cg%20fill%3D%22#ffffff%22%3E%3Cpath%20d%3D%22M5.1%2011.7h1.4v1.4H5.1zM5.1%2014.5h1.4v1.4H5.1zM5.1%2017.4h1.4v1.4H5.1zM8%2011.7h1.4v1.4H8zM8%2014.5h1.4v1.4H8zM8%2017.4h1.4v1.4H8zM10.8%2011.7h1.4v1.4h-1.4zM10.8%2014.5h1.4v1.4h-1.4zM10.8%2017.4h1.4v1.4h-1.4zM13.6%2011.7H15v1.4h-1.4zM13.6%2014.5H15v1.4h-1.4zM13.6%2017.4H15v1.4h-1.4zM16.5%2011.7h1.4v1.4h-1.4zM16.5%2014.5h1.4v1.4h-1.4zM16.5%2017.4h1.4v1.4h-1.4z%22%2F%3E%3C%2Fg%3E%3C%2Fsvg%3E");\n    background-repeat: no-repeat;\n  }\n  .site-icon-credit {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2271%22%20height%3D%2243%22%20viewBox%3D%220%200%2071%2043%22%3E%3Cstyle%20type%3D%22text%2Fcss%22%3Ecircle%2C%20ellipse%2C%20line%2C%20path%2C%20polygon%2C%20polyline%2C%20rect%2C%20text%20%7B%20fill%3A%20#ffffff%20!important%3B%20%7D%3C%2Fstyle%3E%3Cpath%20d%3D%22M71%202c0-1.1-.9-2-2-2h-67c-1.1%200-2%20.9-2%202v8.6h71v-8.6zm-71%2018.4v20.6c0%201.1.9%202%202%202h67c1.1%200%202-.9%202-2v-20.6h-71zm17.7%2014.9h-9.6v-3.6h9.6v3.6zm17.3%200h-13.9v-3.6h13.9v3.6z%22%20fill%3D%22%23fff%22%2F%3E%3C%2Fsvg%3E");\n    background-repeat: no-repeat;\n  }\n  .site-icon-search {\n    background-image: url(\'data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2223%22%20height%3D%2223%22%20viewBox%3D%220%200%2023%2023%22%3E%3Cpath%20d%3D%22M21.8%2019.5L17.4%2015c-.6.9-1.5%201.7-2.4%202.4l4.5%204.4c1.4%201.6%203.8-.8%202.3-2.3z%22%20fill%3D%22%23bebebb%22%3E%3C%2Fpath%3E%3Cpath%20d%3D%22M15%2017.4c4.4-3%205.2-9.4%201.8-13.4C13.8.5%208.6-.3%204.7%202.3.9%204.7-.4%209.9%201.7%2014c2.2%204.4%207.7%206.2%2012.1%204.1.4-.2.8-.5%201.2-.7zm-5.2-1.7C6.7%2015.7%204%2013%204%209.8%204%206.7%206.7%204%209.8%204c3.2%200%205.8%202.7%205.8%205.8.1%203.2-2.6%205.9-5.8%205.9z%22%20fill%3D%22%23bebebb%22%3E%3C%2Fpath%3E%3Cpath%20d%3D%22M9.8%202.2c4.2%200%207.7%203.4%207.7%207.7%200%201.5-.5%203.1-1.4%204.3-.5.7-1%201.3-1.7%201.8l-.3.2c-1.2.9-2.8%201.3-4.3%201.3-4.2%200-7.7-3.4-7.7-7.7.1-4.2%203.5-7.6%207.7-7.6m0-1.5C4.7.7.6%204.8.6%209.9s4.1%209.2%209.2%209.2c1.8%200%203.7-.6%205.2-1.6.9-.6%201.7-1.4%202.3-2.4%201.1-1.5%201.6-3.3%201.6-5.2.1-5.1-4-9.2-9.1-9.2z%22%20fill%3D%22%23bebebb%22%3E%3C%2Fpath%3E%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-is-searching {\n     background-position: center center !important;\n     background-image: url(\'data:image/svg+xml;charset=utf-8,%3Csvg%20version%3D%221.1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%20%20viewBox%3D%220%200%2040%2040%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cpath%20opacity%3D%220.2%22%20fill%3D%22%23323232%22%20d%3D%22M20.201%2C5.169c-8.254%2C0-14.946%2C6.692-14.946%2C14.946c0%2C8.255%2C6.692%2C14.946%2C14.946%2C14.946%0D%0A%20%20s14.946-6.691%2C14.946-14.946C35.146%2C11.861%2C28.455%2C5.169%2C20.201%2C5.169z%20M20.201%2C31.749c-6.425%2C0-11.634-5.208-11.634-11.634%0D%0A%20%20c0-6.425%2C5.209-11.634%2C11.634-11.634c6.425%2C0%2C11.633%2C5.209%2C11.633%2C11.634C31.834%2C26.541%2C26.626%2C31.749%2C20.201%2C31.749z%22%2F%3E%0D%0A%3Cpath%20fill%3D%22%23323232%22%20d%3D%22M26.013%2C10.047l1.654-2.866c-2.198-1.272-4.743-2.012-7.466-2.012h0v3.312h0%0D%0A%20%20C22.32%2C8.481%2C24.301%2C9.057%2C26.013%2C10.047z%22%3E%0D%0A%20%20%3CanimateTransform%20attributeType%3D%22xml%22%0D%0A%20%20%20%20attributeName%3D%22transform%22%0D%0A%20%20%20%20type%3D%22rotate%22%0D%0A%20%20%20%20from%3D%220%2020%2020%22%0D%0A%20%20%20%20to%3D%22360%2020%2020%22%0D%0A%20%20%20%20dur%3D%220.5s%22%0D%0A%20%20%20%20repeatCount%3D%22indefinite%22%2F%3E%0D%0A%20%20%3C%2Fpath%3E%0D%0A%3C%2Fsvg%3E\');\n     background-repeat: no-repeat;\n     background-size: contain;\n     z-index: 999;\n     width: 80px;\n     height: 80px;\n     margin: 0 auto;\n  }\n  .site-icon-report {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3Csvg%20viewBox%3D%220%200%201792%202031%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%3E%3Cpath%20d%3D%22M1122.987%20238.933L1075.2%200H0v2030.933h238.933v-836.266h669.014l47.786%20238.933H1792V238.933z%22%20fill%3D%22%23bebebb%22%20fill-rule%3D%22evenodd%22%2F%3E%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n    background-size: contain;\n  }\n  .site-icon-block {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3Csvg%20viewBox%3D%220%200%201792%201792%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%3E%3Cpath%20d%3D%22M896%201612.8c-165.76%200-318.08-56.448-439.04-151.424L1461.376%20456.96A708.02%20708.02%200%200%201%201612.8%20896c0%20396.032-320.768%20716.8-716.8%20716.8M179.2%20896c0-396.032%20320.768-716.8%20716.8-716.8%20165.76%200%20318.08%2056.448%20439.04%20151.424L330.624%201335.04A708.02%20708.02%200%200%201%20179.2%20896M896%200C401.408%200%200%20401.408%200%20896s401.408%20896%20896%20896%20896-401.408%20896-896S1390.592%200%20896%200%22%20fill%3D%22%23ffffff%22%20fill-rule%3D%22evenodd%22%20fill-opacity%3D%22.6%22%2F%3E%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n    background-size: contain;\n  }\n</style>\n\n\n  <style>\n\n    /*\r\n * This file is for use on the Vimeo VHX platform.\r\n *\r\n * 1. prototype changes in browser dev tools using Vimeo VHX demo site\r\n * 2. edit this file locally\r\n * 3. git commit and push to the remote repo\r\n * 4. share for code review\r\n * 5. paste into "CUSTOM STYLES" field in the Vimeo VHX theme demo admin\r\n * 6. share for approval by Criterion team\r\n * 7. once approved and code reviewed, paste into "CUSTOM STYLES" field in the Vimeo VHX theme production admin\r\n *\r\n * Note for Vimeo engineers: please share any updates to this file to the Criterion.com team.\r\n */\r\n\r\n @import url(\'https://fonts.googleapis.com/css?family=Montserrat:400,500,600,700,900|Spectral:400,500,600,700\');\r\n\r\n p, div, span, a, h1, h2, h3, h4, h5, h6, .site-font-primary-family, .site-font-secondary-family {\r\n     font-family: \'Montserrat\', sans-serif !important;\r\n }\r\n\r\n div {\r\n     font-family: \'Montserrat\', sans-serif !important;\r\n }\r\n\r\n .browse-row h1 a {\r\n     font-family: \'Montserrat\', sans-serif !important;\r\n     font-weight: 700;\r\n     text-transform: uppercase;\r\n }\r\n\r\n .head-1, .head-2 {\r\n     font-weight: 700;\r\n }\r\n\r\n .browse-row h1.horizontal-row-header a {\r\n     font-size: 1rem !important;\r\n }\r\n\r\n .browse-link {\r\n     font-family: \'Montserrat\', sans-serif !important;\r\n }\r\n\r\n /*\r\n * Nav\r\n */\r\n\r\n .navigation-inner a {\r\n    opacity: 1;\r\n    font-family: \'Montserrat\', sans-serif !important;\r\n    -webkit-transition: all 0.3s ease;\r\n    -moz-transition: all 0.3s ease;\r\n    -ms-transition: all 0.3s ease;\r\n    -o-transition: all 0.3s ease;\r\n    transition: all 0.3s ease;\r\n}\r\n\r\n .navigation-inner.is-ready a {\r\n    opacity: 1;\r\n}\r\n\r\n  /*\r\n  * Other\r\n  */\r\n\r\n h1.gigantic.uppercase { text-transform: none !important; }\r\n .percentage-bar-container { display: none !important; }\r\n\r\n .slide-content {\r\n     bottom: 40px;\r\n }\r\n\r\n .slide-title {\r\n     font-size: 28px;\r\n     font-weight: 600;\r\n }\r\n\r\n .browse-item-title .subtext-container > div.site-font-primary-color.truncate {\r\n     white-space: normal;\r\n }\r\n\r\n .collection-description .read-more-wrap {\r\n     max-height: 120px !important;\r\n }\r\n\r\n .site-header nav a {\r\n     font-weight: 600;\r\n     letter-spacing: 0;\r\n     padding-left: 25px;\r\n }\r\n\r\n div.slide-subtitle {\r\n     display: none;\r\n }\r\n\r\n .slide-content {\r\n     bottom: 40px;\r\n }\r\n\r\n .collection-title {\r\n     font-weight: 600;\r\n }\r\n\r\n\r\n #watch-info h5 {\r\n     display: none;\r\n }\r\n\r\n .head.primary strong {\r\n     font-weight: 600;\r\n }\r\n\r\n .item-type-movie .play-outer-rectangle {\r\n     display: none;\r\n }\r\n\r\n /*\r\n  * Borders\r\n  */\r\n\r\n .border-bottom,\r\n .border-bottom-light {\r\n     border-bottom: none !important;\r\n }\r\n\r\n .border-top,\r\n .border-top-light {\r\n     border-top: none !important;\r\n }\r\n\r\n .horizontal-row {\r\n     border-top: none !important;\r\n     border-bottom: none !important;\r\n }\r\n\r\n /*\r\n  * Buttons\r\n  */\r\n\r\n .button {\r\n     border-radius: 0px;\r\n     border: none;\r\n     font-weight: 600;\r\n }\r\n\r\n .slide-button,\r\n .slide-button--alt {\r\n     border-radius: 0px !important;\r\n }\r\n\r\n .btn-site-primary,\r\n .btn-gray:link,\r\n .btn-dropdown-gray:link,\r\n .btn-gray:active,\r\n .btn-dropdown-gray:active,\r\n .btn-gray:visited,\r\n .btn-dropdown-gray:visited {\r\n     border: 1px solid #fff !important;\r\n     background-color: #fff !important;\r\n     color: #252525;\r\n     border-radius: 0px;\r\n     text-transform: uppercase;\r\n     line-height: 46px;\r\n     font-size: 13px;\r\n     transition: all 300ms ease-out;\r\n }\r\n .site-settings-billing-panel .btn-gray:link {\r\n   color: #252525 !important;\r\n   line-height: 33px;\r\n }\r\n .btn-site-primary,\r\n .btn-site-secondary {\r\n     line-height: 33px;\r\n }\r\n\r\n .btn-site-secondary,\r\n .btn-transparent,\r\n .btn-dropdown-transparent {\r\n     border: 1px solid #313131 !important;\r\n     background-color: #313131 !important;\r\n     color: #fff !important;\r\n     transition: all 300ms ease-out;\r\n     font-size: 13px;\r\n }\r\n\r\n .btn-gray:hover,\r\n .btn-dropdown-gray:hover,\r\n .btn-site-primary:hover,\r\n .btn-dropdown-transparent:hover,\r\n .btn-transparent:hover,\r\n .btn-site-secondary:hover {\r\n     border: 1px solid #CDCDCB !important;\r\n     background-color: #CDCDCB !important;\r\n }\r\n\r\n .icon-play-white {\r\n     background-image: url("data:image/svg+xml;charset=utf-8,%3Csvg xmlns=\'http://www.w3.org/2000/svg\' width=\'23\' height=\'23\' viewBox=\'0 0 23 23\'%3E%3Ctitle%3Eicons-ui-set copy%3C/title%3E%3Cpath d=\'M20.49 10.52L4.13.39C2.94-.4 2 .18 2 1.67v19.66c0 1.48 1 2.06 2.17 1.28l16.32-10.13a1.15 1.15 0 0 0 0-1.96z\' fill=\'%23252525\'/%3E%3C/svg%3E");\r\n }\r\n\r\n .btn-black:link,\r\n .btn-dropdown-black:link,\r\n .btn-black:active,\r\n .btn-dropdown-black:active,\r\n .btn-black:visited,\r\n .btn-dropdown-black:visited {\r\n     border: 1px solid #B9BCC6;\r\n     background-color: #B9BCC6;\r\n     color: #000;\r\n     border-radius: 0px;\r\n     text-transform: uppercase;\r\n     line-height: 46px;\r\n     font-size: 13px;\r\n }\r\n\r\n .btn-transparent,\r\n .btn-dropdown-transparent,\r\n .btn-transparent:link,\r\n .btn-dropdown-transparent:link,\r\n .btn-transparent:active,\r\n .btn-dropdown-transparent:active,\r\n .btn-transparent:visited,\r\n .btn-dropdown-transparent:visited {\r\n     border: none;\r\n     border-radius: 0px;\r\n     text-transform: uppercase;\r\n     line-height: 28px;\r\n     font-size: 13px;\r\n     padding: 10px 16px;\r\n     background-position: 16px 18px !important;\r\n }\r\n\r\n select.btn-dropdown-transparent {\r\n     padding: 0 20px 0 40px;\r\n }\r\n\r\n .coin {\r\n     border-radius: 4px;\r\n }\r\n\r\n /*\r\n  * Search link + collection card\r\n  */\r\n\r\n .browse-item-subtext {\r\n     display: none !important;\r\n }\r\n\r\n /*\r\n  * Search results\r\n  */\r\n\r\n .item-type-movie.js-collection-item .duration-container {\r\n     display: none !important;\r\n }\r\n\r\n /*\r\n  * Carousel\r\n  */\r\n\r\n .slide-content {\r\n     transform: translate3d(0,0,0);\r\n     top: auto;\r\n     bottom: 10%;\r\n     padding-left: 57px;\r\n }\r\n\r\n .slide-image-container:after,\r\n .slide-image-container:before {\r\n     display: none !important;\r\n }\r\n\r\n @media screen and (max-width: 39.9375em) {\r\n     .slide-content {\r\n         padding-left: 40px;\r\n         bottom: 13%;\r\n     }\r\n     .slide-bg--mobile .slide-bg-img {\r\n         float: left;\r\n         width: 100%;\r\n         height: auto;\r\n     }\r\n\r\n }\r\n\r\n /*\r\n  * Browse\r\n  */\r\n\r\n .browse-item-card.item-type-movie .duration-container {\r\n   display: none !important;\r\n }\r\n\r\n /*\r\n  * Collection Pages\r\n  */\r\n\r\n .vertical-bottom {\r\n     position: absolute;\r\n     bottom: 0;\r\n }\r\n\r\n .cover-art.columns {\r\n     padding-right: 0px !important;\r\n }\r\n\r\n select.js-sort-items,\r\n h2.collection-stats,\r\n h2.content-label,\r\n .extras-header {\r\n     display: none !important;\r\n }\r\n\r\n .browse-item-title .browse-image-container:before {\r\n     width: 102%;\r\n     height: 102%;\r\n }\r\n\r\n .browse-image-container:before {\r\n     background-image: none !important;\r\n }\r\n\r\n .media-identifier.media-episode {\r\n     display: none;\r\n }\r\n\r\n .read-more-trigger {\r\n     font-weight: bold;\r\n }\r\n\r\nsection.site-settings-purchases-panel .subscription-button {\r\n    background-color: #222 !important\r\n}\r\n\r\n.platforms-modal--container .platform-modal-devices-list-tv li:last-child {\r\n    display: none;\r\n}\r\n\r\n .collection-actions .margin-bottom-small {\r\n    margin-bottom: 0.9375rem !important;\r\n }\r\n .collection-actions .margin-right-small {\r\n    margin-right: 0.9375rem !important;\r\n }\r\n .collection-actions .btn-site-primary {\r\n    line-height: 46px;\r\n }\r\n\r\n/* Fix for button issue on pause / cancellation modal */\r\nbody.settings.is-modal-active .btn-site-secondary,\r\nbody.settings.is-modal-active .btn-site-primary {\r\n    line-height: 38px !important;\r\n    color: #252525 !important;\r\n}\r\n\r\n.site-header .user-dropdown {\r\n    width: 190px;\r\n}\r\n\r\n.tooltip.background-white span.media-count, .tooltip.background-white span.media-identifier, .tooltip.background-white span.media-seperator { display: none !important; }\r\n\r\n.tooltip.background-white .tooltip-item-title { margin-bottom: 15px; }\r\n\r\ndiv[id^="collection-tooltip"] .tooltip-item-title { margin-bottom: 0px !important; }\n\n\n    html.wf-loading { opacity: 0; }\n    html.wf-active, html.is-loaded { opacity: 1 !important; }\n  </style>\n\n  <script>\n  var Tracking = undefined;\n  window.Page = undefined;\n  var Segment_ID = undefined;\n  window._current_user = undefined;\n  window._current_site = undefined;\n</script>\n\n\n  <script>\n    window._vhx = {\n      environment: \'production\',\n      site: { id: \'59054\', domain: \'www.criterionchannel.com\', host: \'criterionchannel.vhx.tv\', previewUrl: \'https://www.criterionchannel.com/login\' },\n      geo:  { country: \'us\' },\n      product: {id: \'\' }\n    };\n\n    window._current_user = {"id":null,"email":null,"name":null,"city":null,"state":null,"country":null,"external_user_id":null};\n\n    window._current_site = {"id":59054,"facebook_pixel_codes":[{"pixel_code":"980569266079632","product_id":null}],"twitter_pixel_codes":[],"adwords_pixel_codes":[],"key":"criterionchannelchartersu","title":"The Criterion Channel","subdomain":"criterionchannel","made_for_kids":false,"made_for_kids_tracking_blocked":false,"algolia_search_enabled":true,"passwords_disabled":false,"segment_heartbeat_frequency":0,"segment_includes_user_id":false,"sso_enabled":false};\n  </script>\n\n<script>\n  window.Tracking = {\n    EVENTS:     {"EVENT_SALE_INITIATED":"sale_initiated","EVENT_SALE_COMPLETE":"sale_complete","EVENT_SALE_COMPLETE_GIFT":"sale_conversion-gift","EVENT_SALE_ERROR":"sale_error","EVENT_SALE_COMPLETE_SAVE_CARD":"sale_complete_save_card","EVENT_SALE_MISSING_EMAIL":"sale_missing_email","EVENT_SALE_UNCONFIRMED_EMAIL":"sale_unconfirmed_email","EVENT_SALE_MISSING_CC":"sale_missing_cc","EVENT_SALE_MISSING_CVC":"sale_missing_cvc","EVENT_SALE_MISSING_EXP":"sale_missing_exp","EVENT_SALE_MISSING_ZIP":"sale_missing_zip","EVENT_SALE_INVALID_CC":"sale_invalid_cc","EVENT_SALE_INVALID_CVC":"sale_invalid_cvc","EVENT_SALE_INVALID_EXP":"sale_invalid_exp","EVENT_SALE_INVALID_ZIP":"sale_invalid_zip","EVENT_SALE_INVALID_COUPON":"sale_invalid_coupon","EVENT_SALE_INACTIVE_COUPON":"sale_inactive_coupon","EVENT_SALE_GEOBLOCKED_COUPON":"sale_geoblocked_coupon","EVENT_SALE_EXPIRED_CC":"sale_expired_cc","EVENT_SALE_PLAY_TRAILER":"sale_play_trailer","EVENT_SALE_PROCESSING_ERROR":"sale_processing_error","EVENT_SALE_UNKNOWN_ERROR":"sale_unknown_error","EVENT_SALE_INCOMPLETE_FORM":"sale_incomplete_form","EVENT_SALE_SHARE_FACEBOOK":"sale_share_facebook","EVENT_SALE_SHARE_TWITTER":"sale_share_twitter","EVENT_SALE_SHARE_EMAIL":"sale_share_email","EVENT_SALE_START_WATCHING":"sale_start_watching","EVENT_SALE_SAVE_CARD":"sale_save_card","EVENT_SALE_SHOW_PAYMENT_TAB":"sale_show_payment_tab","EVENT_SALE_SHOW_DETAILS_TAB":"sale_show_details_tab","EVENT_ADMIN_SIGNUP":"admin_signup","EVENT_ADMIN_ADD_SITE":"admin_add_site","EVENT_ADMIN_PUBLISH_THEME":"admin_publish_theme","EVENT_ADMIN_CREATE_VIDEO":"admin_create_video","EVENT_ADMIN_UPLOAD_VIDEO":"admin_upload_video","EVENT_ADMIN_UPLOAD_VIDEO_NATIVE":"admin_upload_video_native","EVENT_ADMIN_UPLOAD_VIDEO_DROPBOX":"admin_upload_video_dropbox","EVENT_ADMIN_TRANSCODE_VIDEO":"admin_transcode_video","EVENT_ADMIN_CREATE_EXTRA":"admin_create_extra","EVENT_ADMIN_UPLOAD_EXTRA":"admin_upload_extra","EVENT_ADMIN_UPLOAD_EXTRA_NATIVE":"admin_upload_video_native","EVENT_ADMIN_CONNECT_PAYOUT":"admin_connect_payout","EVENT_ADMIN_CREATE_SCREENER":"admin_create_screener","EVENT_ADMIN_CREATE_COUPON":"admin_create_coupon","EVENT_ADMIN_CREATE_UPDATE":"admin_create_update","EVENT_ADMIN_PUBLISH_UPDATE":"admin_publish_update","EVENT_ADMIN_CREATE_PARTNER":"admin_create_partner","EVENT_ADMIN_UPLOADED_FOLLOWERS_CS":"admin_uploaded_followers_csv","EVENT_ADMIN_ADDED_FOLLOWERS":"admin_added_followers","EVENT_ADMIN_CREATE_SUBSCRIPTION":"admin_create_subscription","EVENT_ADMIN_LINK_HELP":"admin_link_help","EVENT_ADMIN_LINK_FAQ":"admin_link_faq","EVENT_ADMIN_LINK_GETTING_STARTED":"admin_link_getting_started","EVENT_ADMIN_LINK_HOWTO":"admin_link_howto","EVENT_ADMIN_LINK_BEST_PRACTICES":"admin_link_best_practices","EVENT_ADMIN_LINK_DEV_DOCS":"admin_link_dev_docs","EVENT_FORUM_CREATE_THREAD":"forum_create_thread","EVENT_FORUM_CREATE_POST":"forum_create_post","EVENT_UPDATE_COMMENT":"comment_update","EVENT_CONCURRENT_LIMIT":"concurrent_limit","EVENT_VIDEO_COMMENT":"video_comment","EVENT_SITE_HERO_WATCH_TRAILER":"site_hero_watch_trailer","EVENT_SITE_HERO_BUY":"site_hero_buy","EVENT_SITE_FOOTER_BUY":"site_footer_buy","EVENT_SITE_FOOTER_IOS":"site_footer_ios","EVENT_SITE_BUY":"site_buy","EVENT_SITE_UNAVAILABLE_REQUEST":"site_unavailable_request","EVENT_SITE_SEARCH":"site_search","EVENT_SITE_SEARCH_EMPTY":"site_search_empty","EVENT_SITE_SEARCH_VIDEO_CLICK":"search_video","EVENT_SITE_SEARCH_COLLECTION_CLICK":"search_collection","EVENT_SITE_SEARCH_PRODUCT_CLICK":"search_product","EVENT_SITE_SEARCH_ITEM_SELECTION":"site_search_item_selected","EVENT_SITE_VIDEO_BUY":"site_video_buy","EVENT_SITE_VIDEO_WATCH_TRAILER":"site_video_watch_trailer","EVENT_SITE_COLLECTION_BUY":"site_collection_buy","EVENT_SITE_PRODUCT_PREORDER":"site_product_preorder","EVENT_SITE_PRODUCT_BUY":"site_product_buy","EVENT_SITE_PRODUCT_RENT":"site_product_rent","EVENT_SITE_CAROUSEL_NEXT":"site_carousel_next","EVENT_SITE_CAROUSEL_PREV":"site_carousel_prev","EVENT_SITE_CAROUSEL_COIN":"site_carousel_coin","EVENT_SITE_CAROUSEL_WATCH_NOW":"site_carousel_watch_now","EVENT_SITE_CAROUSEL_TRAILER":"site_carousel_trailer","EVENT_SITE_HOME":"site_home","EVENT_SITE_BROWSE":"site_browse","EVENT_SITE_SETTINGS":"site_settings","EVENT_SITE_FORUMS":"site_forums","EVENT_SITE_SUPPORT":"site_help","EVENT_SITE_CONTACT":"site_contact","EVENT_SITE_CONTACT_HELP":"site_contact_help","EVENT_SITE_VIEW_PRODUCTS":"site_products","EVENT_SITE_VIEW_PRODUCT":"site_product","EVENT_SITE_COLLECTION_CLICK":"site_collection","EVENT_SITE_VIDEO_CLICK":"site_video","EVENT_SITE_EXTRA_CLICK":"site_extras","EVENT_SITE_CATEGORY_CLICK":"site_category","EVENT_SITE_MOVIE_CLICK":"site_movie","EVENT_SITE_PLAYLIST_CLICK":"site_playlist","EVENT_SITE_SERIES_CLICK":"site_series","EVENT_SITE_SEASON_CLICK":"site_season","EVENT_SITE_TERMS":"site_tos","EVENT_SITE_PRIVACY":"site_privacy","EVENT_SITE_COOKIES":"site_cookies","EVENT_SITE_COPYRIGHT":"site_copyright","EVENT_SITE_DATA_PROCESSING":"site_data_processing","EVENT_SITE_WATCHLIST_REMOVE":"site_removewatchlist","EVENT_SITE_WATCHLIST_ADD":"site_addwatchlist","EVENT_SITE_PAGE_VIEW":"site_page_view","EVENT_SITE_CONTEXT_HOMEPAGE":"site_context_homepage","EVENT_SITE_CONTEXT_COLLECTION":"site_context_collection","EVENT_SITE_CONTEXT_CATEGORY":"site_context_category","EVENT_SITE_CONTEXT_MOVIE":"site_context_movie","EVENT_SITE_CONTEXT_PLAYLIST":"site_context_playlist","EVENT_SITE_CONTEXT_SERIES":"site_context_series","EVENT_SITE_CONTEXT_UPNEXT":"site_context_upnext","EVENT_AUTHENTICATION_LOGIN":"authentication_login","EVENT_AUTHENTICATION_ACTIVATION_STARTED":"activationstarted","EVENT_AUTHENTICATION_ACTIVATION_COMPLETE":"activation","EVENT_AUTHENTICATION_SIGNIN_VIEW":"signin","EVENT_AUTHENTICATION_SIGNIN_STARTED":"signinstarted","EVENT_AUTHENTICATION_SIGNIN_COMPLETE":"authentication","EVENT_AUTHENTICATION_SIGNUP_VIEW":"signup","EVENT_AUTHENTICATION_SIGNUP_STARTED":"signupstarted","EVENT_AUTHENTICATION_PURCHASE_STARTED":"purchasestarted","EVENT_AUTHENTICATION_PURCHASE_COMPLETED":"purchasecompleted","EVENT_AUTHENTICATION_CONVERSION":"conversion","EVENT_AUTHENTICATION_REGISTRATION":"register","EVENT_AUTHENTICATION_IDENTIFY_USER":"identify_user","EVENT_SETTINGS_SAVE_PROFILE":"settings_save_profile","EVENT_SETTINGS_SAVE_NOTIFICATIONS":"settings_save_notifications","EVENT_SETTINGS_SAVE_CARD":"settings_save_card","EVENT_DEVICES_DOWNLOAD_IOS":"devices_download_ios","EVENT_DEVICES_DOWNLOAD_APPLETV":"devices_download_appletv","EVENT_DEVICES_DOWNLOAD_ANDROID":"devices_download_android","EVENT_DEVICES_DOWNLOAD_ANDROIDTV":"devices_download_androidtv","EVENT_DEVICES_DOWNLOAD_AMAZON_FIRE":"devices_download_amazon_fire","EVENT_DEVICES_DOWNLOAD_XBOX":"devices_download_xbox","EVENT_DEVICES_DOWNLOAD_ROKU":"devices_download_roku","EVENT_DEVICES_DOWNLOAD_TIZEN":"devices_download_tizen","EVENT_DEVICES_DOWNLOAD_CHROMECAST":"devices_download_chromecast","EVENT_NEWSLETTER_SIGNUP":"event_newsletter_signup","EVENT_METADATA_ADVISORY":"metadata_click_advisory","EVENT_METADATA_CAST":"metadata_click_cast","EVENT_METADATA_CREW":"metadata_click_crew","EVENT_METADATA_GENRES":"metadata_click_genres","EVENT_METADATA_RATING":"metadata_click_rating","EVENT_METADATA_TAGS":"metadata_click_tags","EVENT_SEGMENT_AUTHENTICATION_CONVERSION":"Order Completed","EVENT_SEGMENT_AUTHENTICATION_PURCHASE_STARTED":"Checkout Started","EVENT_SEGMENT_AUTHENTICATION_REGISTRATION":"Registration Completed","EVENT_SEGMENT_AUTHENTICATION_SIGNIN_COMPLETE":"Sign In Complete","EVENT_SEGMENT_AUTHENTICATION_SIGNIN_VIEW":"Sign In Viewed","EVENT_SEGMENT_AUTHENTICATION_SIGNUP_STARTED":"Signed Up","EVENT_SEGMENT_AUTHENTICATION_SIGNUP_VIEW":"Sign Up Viewed","EVENT_SEGMENT_EXTRA_VIEWED":"Extra Content Viewed","EVENT_SEGMENT_SEARCH_EXECUTED":"Search Executed","EVENT_SEGMENT_SEARCH_ITEM_SELECTED":"Search Result Selected","EVENT_SEGMENT_VIDEO_AD_START":"Video Ad Started","EVENT_SEGMENT_VIDEO_AD_CLICKED":"Video Ad Clicked","EVENT_SEGMENT_VIDEO_AD_FINISH":"Video Ad Completed","EVENT_SEGMENT_VIDEO_AD_FAIL":"Video Ad Failed","EVENT_SEGMENT_VIDEO_AD_SKIP":"Video Ad Skipped","EVENT_SEGMENT_VIDEO_COMMENT_ADDED":"Video Comment Added","EVENT_SEGMENT_VIDEO_COMPLETE_PROGRESS_PERCENTILE":"Video Content Completed","EVENT_SEGMENT_VIDEO_ENDED":"Video Content Ended","EVENT_SEGMENT_VIDEO_END_BUFFER":"Video Playback Buffer Completed","EVENT_SEGMENT_VIDEO_HALF_PROGRESS_PERCENTILE":"50% Content Completed","EVENT_SEGMENT_VIDEO_PAUSE":"Video Playback Paused","EVENT_SEGMENT_VIDEO_PLAY":"Video Playback Started","EVENT_SEGMENT_VIDEO_QUARTER_PROGRESS_PERCENTILE":"25% Content Completed","EVENT_SEGMENT_VIDEO_RESUME":"Video Playback Resumed","EVENT_SEGMENT_VIDEO_SEEKED":"Video Playback Seek Completed","EVENT_SEGMENT_VIDEO_SEEKING":"Video Playback Seek Started","EVENT_SEGMENT_VIDEO_START_BUFFER":"Video Playback Buffer Started","EVENT_SEGMENT_VIDEO_TIMEUPDATE":"Video Content Playing","EVENT_SEGMENT_VIDEO_THIRD_PROGRESS_PERCENTILE":"75% Content Completed","EVENT_SEGMENT_VIDEO_WATCHLIST_ADD":"Added to Watch List","EVENT_SEGMENT_VIDEO_WATCHLIST_REMOVE":"Removed from Watch List","EVENT_SEARCH":"search","EVENT_SEARCH_ITEM_SELECTED":"searchitemselected","EVENT_EXTRA_VIEWED":"extraviewed"},\n    PROPERTIES: {"PROPERTY_COLLECTOR_URL":"https://collector.vhx.tv/pixel.gif","PROPERTY_COLLECTION_ID":"collection_id","PROPERTY_DEVICE":"device","PROPERTY_DEVICE_ID":"device_id","PROPERTY_NAME":"name","PROPERTY_PLATFORM":"platform","PROPERTY_PLATFORM_ID":"platform_id","PROPERTY_PLATFORM_VERSION":"platform_version","PROPERTY_PRODUCT_ID":"product_id","PROPERTY_REFERRER":"referrer","PROPERTY_SESSION_ID":"session_id","PROPERTY_SITE_ID":"site_id","PROPERTY_TIMESTAMP":"timestamp","PROPERTY_TYPE":"type","PROPERTY_URL":"url","PROPERTY_USER_AGENT":"user_agent","PROPERTY_USER_EMAIL":"user_email","PROPERTY_USER_ID":"user_id","PROPERTY_VIDEO_ID":"video_id","PROPERTY_VIEW":"view"},\n    DEPRECATED_EVENTS: ["admin_connect_payout","admin_create_coupon","admin_create_screener","admin_link_dev_docs","admin_link_faq","admin_link_help","admin_publish_theme","metadata_click_cast","metadata_click_crew","metadata_click_genres","metadata_click_rating","metadata_click_tags","metadata_click_advisory"],\n    UPDATED_EVENTS: {"search_video":"searchitemselected","search_collection":"searchitemselected","site_search_item_selected":"searchitemselected","site_extras":"extraviewed"},\n    SITE_GANALYTICS_ENABLED: false,\n    SESSION: \'8d093f147120eb715dd8def89f5b6d00\'\n  };\n</script>\n  <script>\n     window.Page = {"PROPERTIES":{"VIEW_TYPE":"collection","PRODUCT_ID":39621,"COLLECTION_ID":96158,"COLLECTION_TITLE":"The Front Page"}}\n  </script>\n\n\n\n  <script>\n  var _gaq = _gaq || [];\n\n  (function(i,s,o,g,r,a,m){i[\'GoogleAnalyticsObject\']=r;i[r]=i[r]||function(){\n  (i[r].q=i[r].q||[]).push(arguments)},i[r].l=1*new Date();a=s.createElement(o),\n  m=s.getElementsByTagName(o)[0];a.async=1;a.src=g;m.parentNode.insertBefore(a,m)\n  })(window,document,\'script\',\'//www.google-analytics.com/analytics.js\',\'ga\');\n\n      ga(\'create\', \'UA-19540423-6\', \'auto\');\n    ga(\'require\', \'displayfeatures\');\n\n      ga(\'set\', \'dimension1\', \'criterionchannelchartersu\');\n\n    ga(\'send\', \'pageview\');\n\n\n      (function() {\n        var vhx = document.createElement(\'script\'); vhx.type = \'text/javascript\'; vhx.async = true;\n        vhx.src = \'https://cdn.vhx.tv/assets/tracker-be253e44a9312b41e422e9acec66d157e9c650341ea1701c01c2c8b31b26b9e6.js\';\n        var s = document.getElementsByTagName(\'script\')[0]; s.parentNode.insertBefore(vhx, s);\n      })();\n\n\n    Tracking.SITE_GANALYTICS_ENABLED = true;\n\n      ga(\'create\', \'UA-1852531-11\', \'auto\', { \'name\': \'site\', \'allowLinker\': true});\n\n    ga(\'site.require\', \'displayfeatures\');\n      ga(\'site.require\', \'linker\');\n      ga(\'site.linker:autoLink\', [\'www.criterionchannel.com\',\'criterionchannel.vhx.tv\'] );\n    ga(\'site.set\', \'dimension1\', \'criterionchannelchartersu\');\n    ga(\'site.send\', \'pageview\');\n\n    // <!-- Google Tag Manager -->\n    (function(w,d,s,l,i){w[l]=w[l]||[];w[l].push({\'gtm.start\':\n    new Date().getTime(),event:\'gtm.js\'});var f=d.getElementsByTagName(s)[0],\n    j=d.createElement(s),dl=l!=\'dataLayer\'?\'&l=\'+l:\'\';j.async=true;j.src=\n    \'https://www.googletagmanager.com/gtm.js?id=\'+i+dl;f.parentNode.insertBefore(j,f);\n    })(window,document,\'script\',\'dataLayer\',\'GTM-PKL3NF6\');\n    // <!-- End Google Tag Manager -->\n\n  if (window._current_site && window._current_site.facebook_pixel_codes.length > 0) {\n    // <!-- Facebook Pixel Code -->\n    !function(f,b,e,v,n,t,s)\n    {if(f.fbq)return;n=f.fbq=function(){n.callMethod?\n    n.callMethod.apply(n,arguments):n.queue.push(arguments)};\n    if(!f._fbq)f._fbq=n;n.push=n;n.loaded=!0;n.version=\'2.0\';\n    n.queue=[];t=b.createElement(e);t.async=!0;\n    t.src=v;s=b.getElementsByTagName(e)[0];\n    s.parentNode.insertBefore(t,s)}(window, document,\'script\',\n    \'https://connect.facebook.net/en_US/fbevents.js\');\n    //<!-- End Facebook Pixel Code -->\n  }\n\n  if (window._current_site && window._current_site.twitter_pixel_codes.length > 0) {\n    // <!-- Twitter Pixel Code -->\n    !function(e,t,n,s,u,a){e.twq||(s=e.twq=function(){s.exe?s.exe.apply(s,arguments):s.queue.push(arguments);\n    },s.version=\'1.1\',s.queue=[],u=t.createElement(n),u.async=!0,u.src=\'//static.ads-twitter.com/uwt.js\',\n    a=t.getElementsByTagName(n)[0],a.parentNode.insertBefore(u,a))}(window,document,\'script\');\n    //<!-- End Twitter Pixel Code -->\n  }\n\n      Segment_ID = \'D22V0IgjPYTwUKXVEtSzbmfcnolk4CNB\';\n      !function(){var analytics=window.analytics=window.analytics||[];if(!analytics.initialize)if(analytics.invoked)window.console&&console.error&&console.error("Segment snippet included twice.");else{analytics.invoked=!0;analytics.methods=["trackSubmit","trackClick","trackLink","trackForm","pageview","identify","reset","group","track","ready","alias","debug","page","once","off","on"];analytics.factory=function(t){return function(){var e=Array.prototype.slice.call(arguments);e.unshift(t);analytics.push(e);return analytics}};for(var t=0;t<analytics.methods.length;t++){var e=analytics.methods[t];analytics[e]=analytics.factory(e)}analytics.load=function(t,e){var n=document.createElement("script");n.type="text/javascript";n.async=!0;n.src="https://cdn.segment.com/analytics.js/v1/"+t+"/analytics.min.js";var a=document.getElementsByTagName("script")[0];a.parentNode.insertBefore(n,a);analytics._loadOptions=e};analytics.SNIPPET_VERSION="4.1.0";\n      analytics.load("D22V0IgjPYTwUKXVEtSzbmfcnolk4CNB");\n      }}();\n  </script>\n\n\n\n    \n\n  <script>\n  if (/ticket=/.test(location.search) && window.history.replaceState) {\n    var url = location.href.replace(/\\?([^#]*)/, function(_, search) {\n      search = search.split(\'&\').map(function(v) {\n        return !/^ticket=/.test(v) && !/^support=/.test(v) && !/^sso=/.test(v) && v;\n      }).filter(Boolean).join(\'&\');\n      return search ? \'?\' + search : \'\';\n    });\n\n    if (url != location.href) {\n      window.history.replaceState({}, \'\', url);\n    }\n  }\n\n  if (/return_to=/.test(location.search) && window.history.replaceState) {\n    var url = location.href.replace(/\\?([^#]*)/, function(_, search) {\n      search = search.split(\'&\').map(function(v) {\n        return !/^return_to=/.test(v) && v;\n      }).filter(Boolean).join(\'&\');\n      return search ? \'?\' + search : \'\';\n    });\n\n    if (url != location.href) {\n      window.history.replaceState({}, \'\', url);\n    }\n  }\n\n  if (/logout=/.test(location.search) && window.history.replaceState) {\n    var url = location.href.replace(/\\?([^#]*)/, function(_, search) {\n      search = search.split(\'&\').map(function(v) {\n        return !/^logout=/.test(v) && v;\n      }).filter(Boolean).join(\'&\');\n      return search ? \'?\' + search : \'\';\n    });\n\n    if (url != location.href) {\n      window.history.replaceState({}, \'\', url);\n    }\n  }\n</script>\n\n\n  <script>\n    window.TOKEN = "eyJhbGciOiJSUzI1NiIsImtpZCI6ImQ2YmZlZmMzNGIyNTdhYTE4Y2E2NDUzNDE2ZTlmZmRjNjk4MDAxMDdhZTQ2ZWJhODg0YTU2ZDBjOGQ4NTYzMzgifQ.eyJhcHBfaWQiOjY4NTExLCJleHAiOjE2NzE4MTg1MTMsIm5vbmNlIjoiMzkzNzczODU3YmQxY2I0ZiIsInNjb3BlcyI6WyJwdWJsaWMiXSwic2Vzc2lvbl9pZCI6InkraVNmKzczWm1Ia3lPOVB0RnQxUVE9PSIsInNpdGVfaWQiOjU5MDU0fQ.SPu8Ratcf5y5kP9a4yp7MUX4LtXz0ub2HHTr4HFY3d-3r82qno0CYYMUrM0fMppOuHMwziK9x0dVFOc-AXzKfDacT10XDYpa8q52X5clzYLDPL8FRYhc3QLx0OQqqcf9hL7E_lpCRZWs0bn8Wbb8uRquFmh4GmNT8wl1wEVMn8toyI49LngW-3sQjEYd9__0gn3U8c5QK94Y87pWTA5XEtfQpKqPfzpyieoN6M--AfSwUZpl7Svct7KWrbuh1-whMkJzEr21HVrs5_7FRYAnMhUBbedLJRwb5CqAr975ViQ7vegL9PNZAue-qxzoz16JKT_Tt0_leMSWYF5JcwkOG1CGb2oxDS1wUxin_oCrd8VDgWbrP2VHtT74pQZcu9M7s-C1JaH70L3cwkpXcc8nai5XdSbdJkVpZCpjUbaeY66gqFnq4N761zOCFo4GVG1ECsi3H5sm-5VVSPHKx9tDuxZimqvPG2BVwPB7KX_8NZHMJ6ZS_yAGSbjrhBfQZNQeL90a8CWDq5OLewSAtYJAeGqfIuu9Ow7Ej-8969nBTqeqSPYUaIcwKikBRnujfEK4TDpvzm55lZiO-CYgzQuJCjvfCanT2iIJRW2iDERfSkgUl9scYi8SpZseD34kZqLM2YIeeY6aQs76vpB0MQ8bMyFEL0y9fFbP8kxim7LK70Q";\n\n    (function() {\n      var config = { kitId: \'kyo4vqy\', scriptTimeout: 3000 };\n      var h=document.getElementsByTagName("html")[0];h.className+=" wf-loading";var t=setTimeout(function(){h.className=h.className.replace(/(\\s|^)wf-loading(\\s|$)/g," ");h.className+=" wf-inactive"},config.scriptTimeout);var tk=document.createElement("script"),d=false;tk.src=\'//use.typekit.net/\'+config.kitId+\'.js\';tk.type="text/javascript";tk.async="true";tk.onload=tk.onreadystatechange=function(){var a=this.readyState;if(d||a&&a!="complete"&&a!="loaded")return;d=true;clearTimeout(t);try{Typekit.load(config)}catch(b){}};var s=document.getElementsByTagName("script")[0];s.parentNode.insertBefore(tk,s);\n    })();\n  </script>\n\n  \n  \n  \n</head>\n\n\n<body class="site-background-color  site-color-is-light logged-out collections group-a collection-the-front-page" data-site-color="#b9bcc7" data-track-event="site_movie">\n\n    <!-- Google Tag Manager (noscript) -->\n  <noscript><iframe src="https://www.googletagmanager.com/ns.html?id=GTM-PKL3NF6"\n  height="0" width="0" style="display:none;visibility:hidden"></iframe></noscript>\n  <!-- End Google Tag Manager (noscript) -->\n\n  \n\n  <a href="#maincontent" class="skipto">Skip to main content</a>\n    <div style="display: none !important;" rel="description" class="is-hidden">\n      Classics and discoveries from around the world, thematically programmed with special features, on a streaming service brought to you by the Criterion Collection.\n    </div>\n\n  <div data-flash="{&quot;auto_hide&quot;:true,&quot;style&quot;:&quot;overlay&quot;,&quot;container&quot;:null,&quot;type&quot;:&quot;notice&quot;,&quot;message&quot;:{&quot;body&quot;:null},&quot;show_on_load&quot;:false}" class="is-error is-hidden">\n    <div class="flash-icon">\n      <i class="icon icon-check-inverse icon-check-white"></i>\n      <i class="icon icon-alert-inverse icon-alert-white"></i>\n      <i class="icon icon-check-alt icon-check-blue"></i>\n      <i class="icon medium icon-check"></i>\n      <i class="icon medium icon-alert"></i>\n    </div>\n\n    <div class="flash-msg"></div>\n\n    <div class="flash-close">\n      <i class="icon icon-remove-inverse icon--xsmall"></i>\n      <i class="icon icon-remove-mono icon-x-black icon--xsmall"></i>\n    </div>\n</div>\n\n\n\n\n    \n\n\n\n\n\n\n\n\n\n<header class="nav-background-color border-bottom site-border-color site-header has-logo ">\n  <div class="row full-width padding-top-medium padding-bottom-medium primary-row-container flex-container">\n    <div class="small-16 medium-11 columns identity-container flex-container has-image">\n      <span class="show-for-small-only flex-item padding-right-small show-menu-container">\n        <label for="show-menu" class="show-menu-trigger">\n          <span class="site-link-header-color inverse"><i class="icon site-icon-menu"></i></span>\n        </label>\n      </span>\n\n      <h1 class="identity site-font-primary-family head primary flex-item padding-horizontal-medium">\n            <a href="/">\n              <img class="logo" src="https://vhx.imgix.net/criterionchannel/assets/1c5933b9-19a1-4ea8-9761-726ca79faf44.png?fit=max&h=84&q=92&w=400" alt="The Criterion Channel" />\n</a>      </h1>\n\n      <nav class="navigation-container inline flex-item" role="navigation" aria-label="Primary">\n        <input type="checkbox" id="show-menu" name="show-menu" role="button">\n\n        <div class="navigation-inner site-header-background-color-small-only" id="primary-navigation">\n\n          <a href="/browse" class="browse-link site-font-primary-family head site-link-header-color">\n            Browse\n          </a>\n\n\n              <a href="#"\n                id="search-nav-toggle"\n                class="search-link site-font-primary-family head site-link-header-color">\n                Search\n              </a>\n\n                <a href="https://films.criterionchannel.com/" rel="noopener" class="custom-nav-all-films site-font-primary-family head site-link-header-color" target="_blank">ALL FILMS</a>\n                <a href="https://www.criterion.com/" rel="noopener" class="custom-nav-criterion-com site-font-primary-family head site-link-header-color" target="_blank">Criterion.com</a>\n\n            <a class="show-for-small-only flex-item site-link-header-color site-font-primary-family head subscribe-link" data-track-event-properties="{&quot;type&quot;:&quot;subscription&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/the-front-page&quot;}" data-track-event="site_buy" href="https://www.criterionchannel.com/checkout/subscribe">\n              <strong style="font-weight: bold;">\n                Start Free Trial\n              </strong>\n</a>\n\n              <a href="/login" class="show-for-small-only flex-item user-action-link head site-link-header-color site-font-primary-family">\n                Sign in\n              </a>\n\n        </div>\n      </nav>\n    </div>\n\n    <div class="hide-for-small-only text-right flex-right small-16 medium-5 columns small-only-text-center user-activity site-border-color">\n      <div class="user-actions has-image padding-right-medium flex-container" role="navigation" aria-label="Secondary">\n            <a class="flex-item site-link-header-color site-font-primary-family head subscribe-link user-action-link" data-track-event-properties="{&quot;type&quot;:&quot;subscription&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/the-front-page&quot;}" data-track-event="site_buy" href="https://www.criterionchannel.com/checkout/subscribe">\n              <strong style="font-weight: bold;">\n                Start Free Trial\n              </strong>\n</a>\n\n              <a href="/login" class="flex-item user-action-link head site-link-header-color  site-font-primary-family">\n                Sign In\n              </a>\n      </div>\n    </div>\n  </div>\n</header>\n\n  <div id="search-react-app"\n    data-props="{&quot;overlay&quot;:true,&quot;site_id&quot;:59054,&quot;api_v1_url&quot;:&quot;https://api.vhx.tv&quot;,&quot;api_v2_url&quot;:&quot;https://api.vhx.com&quot;,&quot;user_subscribed&quot;:false,&quot;purchased_product_ids&quot;:[],&quot;theme&quot;:&quot;dark&quot;,&quot;show_filters&quot;:false}">\n  </div>\n\n\n\n  \n\n\n\n<section class="site-background-color site-font-primary-color margin-bottom-large collection-movie top-container">\n  <div class="blurred-background-image"  style="background-image: url(\'https://vhx.imgix.net/criterionchannelchartersu/assets/8066de39-d6e3-4196-9097-60053d6c940f.jpg?blur=180\')">\n    <div class="gradient-blend" data-gradient-blend></div>\n  </div>\n  <section class="product-feature padding-horizontal-medium padding-top-large padding-bottom-xlarge border-bottom border-bottom-light">\n\n    <div class="row relative">\n      <div class="small-16 medium-8 large-8 collection-details-container columns end">\n\n        <div class="collection-details grid-padding-left">\n            <div class="show-for-small-only cover-art padding-bottom-medium">\n              <div class="embed">\n                <img src="https://vhx.imgix.net/criterionchannelchartersu/assets/8066de39-d6e3-4196-9097-60053d6c940f.jpg?auto=format%2Ccompress&fit=crop&h=360&q=70&w=640" alt="The Front Page" data-image-primary>\n              </div>\n            </div>\n\n          <h1 class="head primary site-font-primary-color collection-title">The Front Page</h1>\n\n            <div class="contain padding-top-medium collection-description word-break">\n    <div class="site-font-secondary-color"\n         data-text-show-less="Show less"\n         data-text-show-more="Show more"\n         data-truncate=\'{"max_height": 60, "speed": 300}\'>\n      <p>Directed by Lewis Milestone \xe2\x80\xa2 1931 \xe2\x80\xa2 United States\n<br />Starring Adolphe Menjou, Pat O\xe2\x80\x99Brien, Mary Brian</p>\n\n<p>Directed with pre-Code verve by Lewis Milestone, the first screen adaptation of the oft-filmed Broadway hit by Ben Hecht and Charles MacArthur (later the basis for Howard Hawks\xe2\x80\x99s screwball classic HIS GIRL FRIDAY) stars Adolphe Menjou and Pat O\xe2\x80\x99Brien as, respectively, a newspaper editor and his ace reporter, who must pull out all the stops in order to get the scoop of a lifetime: the story of an escaped death-row inmate whose whereabouts are known only to them. The crackling dialogue, cynical wit, and snappy pace set the standard for a generation of newsroom pictures to follow.</p>\n\n<p>Restored by the Academy Film Archive and The Film Foundation. Restoration funding provided by the George Lucas Family Foundation. Elements for this restoration provided by the Howard Hughes Corporation and by the University of Nevada, Las Vegas, College of Fine Arts\xe2\x80\x99 Department of Film and its Howard Hughes Collection at the Academy Film Archive.\n<br />Restored by the Academy Film Archive and The Film Foundation. Restoration funding provided by the George Lucas Family Foundation. Elements for this restoration provided by the Howard Hughes Corporation and by the University of Nevada, Las Vegas, College of Fine Arts\xe2\x80\x99 Department of Film and its Howard Hughes Collection at the Academy Film Archive. </p>\n    </div>\n  </div>\n\n\n          <div class="collection-actions padding-bottom-small padding-top-large">\n                  <a href="https://www.criterionchannel.com/checkout/subscribe?return_to=https%3A%2F%2Fcriterionchannel.vhx.tv%2Fthe-front-page"\n      class="btn btn-site-primary custom-btn-action-transact site-font-primary-family uppercase margin-bottom-small margin-right-small"\n      data-track-event="site_collection_buy">\n      Subscribe\n    </a>\n\n              <a href="#" class="btn btn-black custom-btn-action-share margin-bottom-small margin-right-medium " data-reveal-id="modal-the-front-page">\n                Share\n              </a>\n              \n\n\n\n<div class="reveal-modal site-background-color text-center share-modal-container" id="modal-the-front-page" data-reveal>\n  <a class="close-reveal-modal"><i class="icon icon-remove-inverse"></i><i class="icon icon-remove-gray"></i></a>\n  <div class="share-title-container padding-bottom-medium">\n    <h5 class="head secondary site-font-secondary-color site-font-primary-family">\n      Share with your friends\n    </h5>\n  </div>\n  <div class="share-link-container padding-bottom-medium">\n    <ul class="small-block-grid-3">\n      <li>\n        <a href="https://www.facebook.com/sharer/sharer.php?u=https%3A%2F%2Fwww.criterionchannel.com%2Fthe-front-page"\n           data-social-popup\n           data-share="facebook"\n           class="btn btn-facebook block border site-border-color"\n           title="Facebook">\n          <i class="vertical-align icon site-icon-facebook icon-center"></i>\n        </a>\n      </li>\n      <li>\n        <a href="https://twitter.com/share?url=https%3A%2F%2Fwww.criterionchannel.com%2Fthe-front-page&text=Watch+The+Front+Page+from+%40criterionchannl&related=criterionchannl"\n           data-social-popup\n           data-share="twitter"\n           class="btn-twitter btn block border site-border-color"\n           title="Twitter">\n          <i class="vertical-align icon site-icon-twitter icon-center"></i>\n        </a>\n      </li>\n      <li>\n        <a target="_blank" href="/cdn-cgi/l/email-protection#3a05494f58505f594e076d5b4e59521a6e525f1a7c4855544e1a6a5b5d5f1a5c4855571a6e525f1a7948534e5f485355541a79525b54545f561c5b574a0158555e4307524e4e4a491f097b1f087c1f087c4d4d4d145948534e5f4853555459525b54545f56145955571f087c4e525f175c4855544e174a5b5d5f" class="btn btn-black block border site-border-color" title="Email">\n          <i class="vertical-align icon site-icon-envelope icon-center"></i>\n        </a>\n      </li>\n    </ul>\n  </div>\n    <div class="share-url-container">\n      <form class="form"><input value="https://www.criterionchannel.com/the-front-page" type="text" class="text text-center border-none site-background-color site-font-secondary-color" /></form>\n    </div>\n</div>\n\n          </div>\n        </div>\n      </div>\n\n        <div class="hide-for-small-only medium-8 large-offset-2 large-6 end cover-art columns">\n          <div class="embed-container grid-padding-right">\n            <div class="embed">\n              <img src="https://vhx.imgix.net/criterionchannelchartersu/assets/8066de39-d6e3-4196-9097-60053d6c940f.jpg?auto=format%2Ccompress&fit=crop&h=720&q=75&w=1280" alt="The Front Page" data-image-primary>\n            </div>\n          </div>\n        </div>\n\n    </div>\n  </section>\n\n  <section class="movie-container video-container padding-top-large padding-horizontal-medium">\n    <div class="row relative">\n      <div class="small-16 columns item-grid flush-grid flush-row">\n        \n  <div class="row full-width padding-horizontal-medium">\n    <ul class="small-block-grid-1 medium-block-grid-3 large-block-grid-3 ">\n\n\n\n  \n<li\n  class="js-collection-item\n    collection-item-459099 item-type-video\n    \n    "\n  data-item-id="459099"\n  data-item-type="video"\n  \n    data-tooltip="id: collection-tooltip-459099;\n    position: right;\n    offset: 0;\n    animate: true;\n    timer: 650;\n    delay: false;\n    is_centered: true;"\n    \n>\n  <div class="browse-item-card">\n    <div class="grid-item-padding">\n      <a href="https://www.criterionchannel.com/the-front-page/videos/the-front-page" class="browse-item-link" data-track-event="site_video" data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/the-front-page&quot;,&quot;id&quot;:459099,&quot;context&quot;:&quot;site_context_movie&quot;,&quot;label&quot;:&quot;The Front Page&quot;,&quot;canonical_collection&quot;:null}">\n        <div class="browse-image-container">\n            <span class="freemium-lock">\n              <i class="icon icon--xsmall icon-lock-inverse"></i>\n            </span>\n\n\n          <img alt="The Front Page" src="https://vhx.imgix.net/criterionchannelchartersu/assets/8066de39-d6e3-4196-9097-60053d6c940f.jpg?auto=format%2Ccompress&fit=crop&h=360&q=70&w=640" />\n\n\n          <div class="browse-info-container">\n              <div class="duration-container\n                is-locked\n                "\n              >\n                 1:41:27\n              </div>\n          </div>\n        </div> \n      </a>\n\n      <div class="padding-small">\n        <div class="site-font-primary-color browse-item-title">\n           <a\n            href="https://www.criterionchannel.com/the-front-page/videos/the-front-page"\n            data-track-event="site_video"\n            data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/the-front-page&quot;,&quot;id&quot;:459099,&quot;context&quot;:&quot;site_context_movie&quot;,&quot;label&quot;:&quot;The Front Page&quot;,&quot;canonical_collection&quot;:null}"\n          >\n            <strong title="The Front Page">\n              The Front Page\n            </strong>\n          </a>\n        </div>\n\n      </div>\n    </div>\n\n\n    <div \n         class="tooltip background-white" id="collection-tooltip-459099">\n      <h3 class="tooltip-item-title site-font-primary-family"><strong>The Front Page</strong></h3>\n\n        <div class="transparent padding-top-medium">\n          <p>Directed by Lewis Milestone \xe2\x80\xa2 1931 \xe2\x80\xa2 United States\n<br />Starring Adolphe Menjou, Pat O\xe2\x80\x99Brien, Mary Brian</p>\n\n<p>Directed with pre-Code verve by Lewis Milestone, the first screen adaptation of the oft-filmed Broadway hit by Ben Hecht and Charles MacArthur (later the basis for Howard Hawks\xe2\x80\x99s screwball classic...</p>\n        </div>\n    </div>\n  </div>\n</li>\n\n\n\n\n    </ul>\n  </div>\n\n      </div>\n    </div>\n  </section>\n    <section class="extras-container video-container padding-horizontal-medium padding-bottom-medium">\n      <div class="row relative border-top border-top-light padding-top-large">\n        <div class="small-16 columns item-grid flush-grid flush-row">\n          <h2 class="site-font-primary-color site-font-primary-family collection-title extras-header grid-padding-left padding-bottom-medium">\n            Extras\n          </h2>\n          \n  <div class="row full-width padding-horizontal-medium">\n    <ul class="small-block-grid-1 medium-block-grid-3 large-block-grid-4 js-load-more-items-container">\n\n\n\n  \n<li\n  class="js-collection-item\n    collection-item-460792 item-type-video\n    \n    "\n  data-item-id="460792"\n  data-item-type="video"\n  \n    data-tooltip="id: collection-tooltip-460792;\n    position: right;\n    offset: 0;\n    animate: true;\n    timer: 650;\n    delay: false;\n    is_centered: true;"\n    \n>\n  <div class="browse-item-card">\n    <div class="grid-item-padding">\n      <a href="https://www.criterionchannel.com/the-front-page/videos/restoring-the-front-page" class="browse-item-link" data-track-event="site_video" data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/the-front-page&quot;,&quot;id&quot;:460792,&quot;context&quot;:&quot;site_context_movie&quot;,&quot;label&quot;:&quot;Restoring THE FRONT PAGE&quot;,&quot;canonical_collection&quot;:null}">\n        <div class="browse-image-container">\n            <span class="freemium-lock">\n              <i class="icon icon--xsmall icon-lock-inverse"></i>\n            </span>\n\n\n          <img alt="Restoring THE FRONT PAGE" src="https://vhx.imgix.net/criterionchannelchartersu/assets/b2540c07-8100-43cb-835e-ee1905c5c9f9-2429f2d3.jpg?auto=format%2Ccompress&fit=crop&h=360&q=70&w=640" />\n\n\n          <div class="browse-info-container">\n              <div class="duration-container\n                is-locked\n                "\n              >\n                24:04\n              </div>\n          </div>\n        </div> \n      </a>\n\n      <div class="padding-small">\n        <div class="site-font-primary-color browse-item-title">\n           <a\n            href="https://www.criterionchannel.com/the-front-page/videos/restoring-the-front-page"\n            data-track-event="site_video"\n            data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/the-front-page&quot;,&quot;id&quot;:460792,&quot;context&quot;:&quot;site_context_movie&quot;,&quot;label&quot;:&quot;Restoring THE FRONT PAGE&quot;,&quot;canonical_collection&quot;:null}"\n          >\n            <strong title="Restoring THE FRONT PAGE">\n              Restoring THE FRONT PAGE\n            </strong>\n          </a>\n        </div>\n\n      </div>\n    </div>\n\n\n    <div \n         class="tooltip background-white" id="collection-tooltip-460792">\n      <h3 class="tooltip-item-title site-font-primary-family"><strong>Restoring THE FRONT PAGE</strong></h3>\n\n        <div class="transparent padding-top-medium">\n          <p>This 2016 piece details the work of the Academy Film Archive and The Film Foundation in bringing this version of THE FRONT PAGE to life. It highlights the differences between this restoration, in which certain scenes reflect director Lewis Milestone\xe2\x80\x99s preferred presentation for domestic distribut...</p>\n        </div>\n    </div>\n  </div>\n</li>\n\n\n\n\n  \n<li\n  class="js-collection-item\n    collection-item-460793 item-type-video\n    \n    "\n  data-item-id="460793"\n  data-item-type="video"\n  \n    data-tooltip="id: collection-tooltip-460793;\n    position: right;\n    offset: 0;\n    animate: true;\n    timer: 650;\n    delay: false;\n    is_centered: true;"\n    \n>\n  <div class="browse-item-card">\n    <div class="grid-item-padding">\n      <a href="https://www.criterionchannel.com/the-front-page/videos/david-brendel-on-screenwriter-ben-hecht" class="browse-item-link" data-track-event="site_video" data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/the-front-page&quot;,&quot;id&quot;:460793,&quot;context&quot;:&quot;site_context_movie&quot;,&quot;label&quot;:&quot;David Brendel on Screenwriter Ben Hecht&quot;,&quot;canonical_collection&quot;:null}">\n        <div class="browse-image-container">\n            <span class="freemium-lock">\n              <i class="icon icon--xsmall icon-lock-inverse"></i>\n            </span>\n\n\n          <img alt="David Brendel on Screenwriter Ben Hecht" src="https://vhx.imgix.net/criterionchannelchartersu/assets/e95c2638-22b7-4365-9071-3bce071f9d67-3dac8bf1.jpg?auto=format%2Ccompress&fit=crop&h=360&q=70&w=640" />\n\n\n          <div class="browse-info-container">\n              <div class="duration-container\n                is-locked\n                "\n              >\n                25:47\n              </div>\n          </div>\n        </div> \n      </a>\n\n      <div class="padding-small">\n        <div class="site-font-primary-color browse-item-title">\n           <a\n            href="https://www.criterionchannel.com/the-front-page/videos/david-brendel-on-screenwriter-ben-hecht"\n            data-track-event="site_video"\n            data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/the-front-page&quot;,&quot;id&quot;:460793,&quot;context&quot;:&quot;site_context_movie&quot;,&quot;label&quot;:&quot;David Brendel on Screenwriter Ben Hecht&quot;,&quot;canonical_collection&quot;:null}"\n          >\n            <strong title="David Brendel on Screenwriter Ben Hecht">\n              David Brendel on Screenwriter Ben Hecht\n            </strong>\n          </a>\n        </div>\n\n      </div>\n    </div>\n\n\n    <div \n         class="tooltip background-white" id="collection-tooltip-460793">\n      <h3 class="tooltip-item-title site-font-primary-family"><strong>David Brendel on Screenwriter Ben Hecht</strong></h3>\n\n        <div class="transparent padding-top-medium">\n          <p>Writer Ben Hecht\xe2\x80\x99s art made extensive use of his life, and the work he did in Hollywood drew on his experiences growing up in Al Capone\xe2\x80\x93era Chicago and working in newsrooms to capture the worlds of gangsters, reporters, and spies. In this 2016 program, Hecht expert David Brendel explores the life...</p>\n        </div>\n    </div>\n  </div>\n</li>\n\n\n\n\n  \n<li\n  class="js-collection-item\n    collection-item-460794 item-type-video\n    \n    "\n  data-item-id="460794"\n  data-item-type="video"\n  \n    data-tooltip="id: collection-tooltip-460794;\n    position: right;\n    offset: 0;\n    animate: true;\n    timer: 650;\n    delay: false;\n    is_centered: true;"\n    \n>\n  <div class="browse-item-card">\n    <div class="grid-item-padding">\n      <a href="https://www.criterionchannel.com/the-front-page/videos/the-front-page-radio-adaptation-1937" class="browse-item-link" data-track-event="site_video" data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/the-front-page&quot;,&quot;id&quot;:460794,&quot;context&quot;:&quot;site_context_movie&quot;,&quot;label&quot;:&quot;THE FRONT PAGE Radio Adaptation: 1937&quot;,&quot;canonical_collection&quot;:null}">\n        <div class="browse-image-container">\n            <span class="freemium-lock">\n              <i class="icon icon--xsmall icon-lock-inverse"></i>\n            </span>\n\n\n          <img alt="THE FRONT PAGE Radio Adaptation: 1937" src="https://vhx.imgix.net/criterionchannelchartersu/assets/bb0de182-619e-4997-bcd6-373ea06a24c7-310bccb5.jpg?auto=format%2Ccompress&fit=crop&h=360&q=70&w=640" />\n\n\n          <div class="browse-info-container">\n              <div class="duration-container\n                is-locked\n                "\n              >\n                58:47\n              </div>\n          </div>\n        </div> \n      </a>\n\n      <div class="padding-small">\n        <div class="site-font-primary-color browse-item-title">\n           <a\n            href="https://www.criterionchannel.com/the-front-page/videos/the-front-page-radio-adaptation-1937"\n            data-track-event="site_video"\n            data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/the-front-page&quot;,&quot;id&quot;:460794,&quot;context&quot;:&quot;site_context_movie&quot;,&quot;label&quot;:&quot;THE FRONT PAGE Radio Adaptation: 1937&quot;,&quot;canonical_collection&quot;:null}"\n          >\n            <strong title="THE FRONT PAGE Radio Adaptation: 1937">\n              THE FRONT PAGE Radio Adaptation: 1937\n            </strong>\n          </a>\n        </div>\n\n      </div>\n    </div>\n\n\n    <div \n         class="tooltip background-white" id="collection-tooltip-460794">\n      <h3 class="tooltip-item-title site-font-primary-family"><strong>THE FRONT PAGE Radio Adaptation: 1937</strong></h3>\n\n        <div class="transparent padding-top-medium">\n          <p>This radio adaptation of THE FRONT PAGE aired in 1937. It features Walter Winchell as Hildy Johnson and James Gleason as Walter Burns.</p>\n        </div>\n    </div>\n  </div>\n</li>\n\n\n\n\n  \n<li\n  class="js-collection-item\n    collection-item-460795 item-type-video\n    \n    "\n  data-item-id="460795"\n  data-item-type="video"\n  \n    data-tooltip="id: collection-tooltip-460795;\n    position: right;\n    offset: 0;\n    animate: true;\n    timer: 650;\n    delay: false;\n    is_centered: true;"\n    \n>\n  <div class="browse-item-card">\n    <div class="grid-item-padding">\n      <a href="https://www.criterionchannel.com/the-front-page/videos/the-front-page-radio-adaptation-1946" class="browse-item-link" data-track-event="site_video" data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/the-front-page&quot;,&quot;id&quot;:460795,&quot;context&quot;:&quot;site_context_movie&quot;,&quot;label&quot;:&quot;THE FRONT PAGE Radio Adaptation: 1946&quot;,&quot;canonical_collection&quot;:null}">\n        <div class="browse-image-container">\n            <span class="freemium-lock">\n              <i class="icon icon--xsmall icon-lock-inverse"></i>\n            </span>\n\n\n          <img alt="THE FRONT PAGE Radio Adaptation: 1946" src="https://vhx.imgix.net/criterionchannelchartersu/assets/96aff9cd-fb53-4891-b6b4-7604e56d777a-c896f362.jpg?auto=format%2Ccompress&fit=crop&h=360&q=70&w=640" />\n\n\n          <div class="browse-info-container">\n              <div class="duration-container\n                is-locked\n                "\n              >\n                31:44\n              </div>\n          </div>\n        </div> \n      </a>\n\n      <div class="padding-small">\n        <div class="site-font-primary-color browse-item-title">\n           <a\n            href="https://www.criterionchannel.com/the-front-page/videos/the-front-page-radio-adaptation-1946"\n            data-track-event="site_video"\n            data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/the-front-page&quot;,&quot;id&quot;:460795,&quot;context&quot;:&quot;site_context_movie&quot;,&quot;label&quot;:&quot;THE FRONT PAGE Radio Adaptation: 1946&quot;,&quot;canonical_collection&quot;:null}"\n          >\n            <strong title="THE FRONT PAGE Radio Adaptation: 1946">\n              THE FRONT PAGE Radio Adaptation: 1946\n            </strong>\n          </a>\n        </div>\n\n      </div>\n    </div>\n\n\n    <div \n         class="tooltip background-white" id="collection-tooltip-460795">\n      <h3 class="tooltip-item-title site-font-primary-family"><strong>THE FRONT PAGE Radio Adaptation: 1946</strong></h3>\n\n        <div class="transparent padding-top-medium">\n          <p>This radio adaptation of THE FRONT PAGE aired in 1937. It reunites Pat O\xe2\x80\x99Brien and Adolphe Menjou in the roles they played in Lewis Milestone\xe2\x80\x99s 1931 film.</p>\n        </div>\n    </div>\n  </div>\n</li>\n\n\n\n\n    </ul>\n  </div>\n\n          \n        </div>\n      </div>\n    </section>\n</section>\n\n\n    <style>\n  a.site-link-footer-color,\n  a.site-link-footer-color:visited {\n    color: #ffffff !important;\n  }\n  .btn-site-footer,\n  .btn-site-footer:link,\n  .btn-site-footer:visited {\n    color: #b9bcc7;\n    background-color: #000;\n  }\n  .btn-site-footer:hover,\n  .btn-site-footer.is-hover,\n  .btn-site-footer:focus,\n  .btn-site-footer:focus:hover,\n  .btn-site-footer:active,\n  .btn-site-footer.is-active {\n    color: #000;\n    border-color: #000;\n    background-color: #b9bcc7;\n  }\n\n  .btn-logout-form {\n    display: inline;\n    line-height: 0;\n  }\n\n  .btn-logout {\n    background-color: transparent;\n    border: none;\n    cursor: pointer;\n    font-size: 12px;\n    padding: 0;\n  }\n\n  .btn-logout:hover {\n    opacity: .5;\n  }\n</style>\n\n\n<footer class="footer--site border-top site-border-color site-background-color padding-top-medium padding-bottom-medium ">\n  <div class="row full-width">\n    <div class="small-16 medium-6 large-10 columns helper-links small-only-text-center">\n      <div class="padding-left-medium display-flex-container flex-start force-min-height">\n              <a href="https://www.criterion.com/" class="site-font-secondary-color" target="_blank" rel="noopener">THE CRITERION COLLECTION</a>\n\n\n        <a href="/help" class="site-font-secondary-color legal-link help-link" rel="noopener">\n          Help\n        </a>\n        <a href="/tos" class="site-font-secondary-color legal-link terms-link" rel="noopener">\n          Terms\n        </a>\n        <a href="/privacy" class="site-font-secondary-color legal-link privacy-link" rel="noopener">\n          Privacy\n        </a>\n        <a href="/cookies" class="site-font-secondary-color legal-link cookies-link" rel="noopener">\n          Cookies\n        </a>\n\n\n            <a href="/login" class="site-font-secondary-color signin-link">\n              Sign in\n            </a>\n      </div>\n    </div> <!-- columns -->\n    <div class="small-16 medium-10 large-6 small-only-text-center text-right columns vhx-link">\n      <div class="padding-right-small display-flex-container flex-end force-min-height">\n          &nbsp;\n        <div id="language-switcher-react-app"\n             class="padding-right-small"\n             data-supported-locales="[]"\n        </div>\n        <script data-cfasync="false" src="/cdn-cgi/scripts/5c5dd728/cloudflare-static/email-decode.min.js"></script><script src="https://cdn.vhx.tv/assets/vite/assets/language-switcher.c19b2f61.js" crossorigin="anonymous" type="module"></script><link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/jsx-runtime.75dbc984.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/LanguageSwitcher.2b83283a.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/index.71ea8908.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/intl.helpers.2a168b49.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/index.61149844.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/deviceSizes.9973a5fd.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/message.bd28a54d.js" as="script" crossorigin="anonymous" />\n      </div> <!-- display-flex-container -->\n    </div>\n  </div> <!-- row -->\n</footer>\n\n<div class="modal">\n  <div class="close"><a href="#close" alt="Close Window"><strong class="is-inverse">&times;</strong></a></div>\n  <div class="content"></div>\n</div>\n\n<div id="blanket"></div>\n\n\n  <div class="modals">\n    \n  </div>\n\n  <script src="https://cdn.vhx.tv/assets/webpack/vendor-ottbundle-5b2989a58aa1f2ae17a5.js"></script>\n\n  <script>\n    window.VHX = window.VHX || {};\n    window.VHX.config = {\n      api_url: "https://api.vhx.tv",\n      token: "eyJhbGciOiJSUzI1NiIsImtpZCI6ImQ2YmZlZmMzNGIyNTdhYTE4Y2E2NDUzNDE2ZTlmZmRjNjk4MDAxMDdhZTQ2ZWJhODg0YTU2ZDBjOGQ4NTYzMzgifQ.eyJhcHBfaWQiOjY4NTExLCJleHAiOjE2NzE4MTg1MTMsIm5vbmNlIjoiMzkzNzczODU3YmQxY2I0ZiIsInNjb3BlcyI6WyJwdWJsaWMiXSwic2Vzc2lvbl9pZCI6InkraVNmKzczWm1Ia3lPOVB0RnQxUVE9PSIsInNpdGVfaWQiOjU5MDU0fQ.SPu8Ratcf5y5kP9a4yp7MUX4LtXz0ub2HHTr4HFY3d-3r82qno0CYYMUrM0fMppOuHMwziK9x0dVFOc-AXzKfDacT10XDYpa8q52X5clzYLDPL8FRYhc3QLx0OQqqcf9hL7E_lpCRZWs0bn8Wbb8uRquFmh4GmNT8wl1wEVMn8toyI49LngW-3sQjEYd9__0gn3U8c5QK94Y87pWTA5XEtfQpKqPfzpyieoN6M--AfSwUZpl7Svct7KWrbuh1-whMkJzEr21HVrs5_7FRYAnMhUBbedLJRwb5CqAr975ViQ7vegL9PNZAue-qxzoz16JKT_Tt0_leMSWYF5JcwkOG1CGb2oxDS1wUxin_oCrd8VDgWbrP2VHtT74pQZcu9M7s-C1JaH70L3cwkpXcc8nai5XdSbdJkVpZCpjUbaeY66gqFnq4N761zOCFo4GVG1ECsi3H5sm-5VVSPHKx9tDuxZimqvPG2BVwPB7KX_8NZHMJ6ZS_yAGSbjrhBfQZNQeL90a8CWDq5OLewSAtYJAeGqfIuu9Ow7Ej-8969nBTqeqSPYUaIcwKikBRnujfEK4TDpvzm55lZiO-CYgzQuJCjvfCanT2iIJRW2iDERfSkgUl9scYi8SpZseD34kZqLM2YIeeY6aQs76vpB0MQ8bMyFEL0y9fFbP8kxim7LK70Q",\n      token_expires_in: "7200000",\n      user_has_subscription: "false"\n    };\n  </script>\n\n  <script src="https://cdn.vhx.tv/assets/webpack/sites-ottbundle-1e2d12c471814f99fb7e.js"></script>\n  \n\n      <script>\n    window.VHX = window.VHX || {};\n    window.VHX.config = {\n      api_url: "https://api.vhx.tv",\n      token: "eyJhbGciOiJSUzI1NiIsImtpZCI6ImQ2YmZlZmMzNGIyNTdhYTE4Y2E2NDUzNDE2ZTlmZmRjNjk4MDAxMDdhZTQ2ZWJhODg0YTU2ZDBjOGQ4NTYzMzgifQ.eyJhcHBfaWQiOjY4NTExLCJleHAiOjE2NzE4MTg1MTMsIm5vbmNlIjoiMzkzNzczODU3YmQxY2I0ZiIsInNjb3BlcyI6WyJwdWJsaWMiXSwic2Vzc2lvbl9pZCI6InkraVNmKzczWm1Ia3lPOVB0RnQxUVE9PSIsInNpdGVfaWQiOjU5MDU0fQ.SPu8Ratcf5y5kP9a4yp7MUX4LtXz0ub2HHTr4HFY3d-3r82qno0CYYMUrM0fMppOuHMwziK9x0dVFOc-AXzKfDacT10XDYpa8q52X5clzYLDPL8FRYhc3QLx0OQqqcf9hL7E_lpCRZWs0bn8Wbb8uRquFmh4GmNT8wl1wEVMn8toyI49LngW-3sQjEYd9__0gn3U8c5QK94Y87pWTA5XEtfQpKqPfzpyieoN6M--AfSwUZpl7Svct7KWrbuh1-whMkJzEr21HVrs5_7FRYAnMhUBbedLJRwb5CqAr975ViQ7vegL9PNZAue-qxzoz16JKT_Tt0_leMSWYF5JcwkOG1CGb2oxDS1wUxin_oCrd8VDgWbrP2VHtT74pQZcu9M7s-C1JaH70L3cwkpXcc8nai5XdSbdJkVpZCpjUbaeY66gqFnq4N761zOCFo4GVG1ECsi3H5sm-5VVSPHKx9tDuxZimqvPG2BVwPB7KX_8NZHMJ6ZS_yAGSbjrhBfQZNQeL90a8CWDq5OLewSAtYJAeGqfIuu9Ow7Ej-8969nBTqeqSPYUaIcwKikBRnujfEK4TDpvzm55lZiO-CYgzQuJCjvfCanT2iIJRW2iDERfSkgUl9scYi8SpZseD34kZqLM2YIeeY6aQs76vpB0MQ8bMyFEL0y9fFbP8kxim7LK70Q",\n      token_expires_in: "7200000",\n      user_has_subscription: "false"\n    };\n  </script>\n\n  <script src="https://cdn.vhx.tv/assets/webpack/products-ottbundle-12994ed95e958f7c2e65.js"></script>\n    <script src="https://cdn.vhx.tv/assets/vite/assets/search.f6a28d3c.js" crossorigin="anonymous" type="module"></script><link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/jsx-runtime.75dbc984.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/index.esm.f84f874f.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/index.0e8c968f.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/intl.helpers.2a168b49.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/api.utils.c8000605.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/Menu.esm.959a8f77.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/Button.esm.61e4a4cb.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/Text.esm.c0f99a84.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/message.bd28a54d.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/Checkmark.esm.fb551793.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/Link.esm.a6ddba18.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/Search.esm.a44cb748.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/index.71ea8908.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/text-helpers.5cfd261d.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/time-helpers.4df27d81.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/_includes.3534e71e.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/_curry2.3e499763.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/path.235ca80c.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/Header.esm.2a01f61a.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/DismissX.esm.54e5d2b0.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/colors.esm.e8046dae.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/inheritsLoose.a8857277.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/index.61149844.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/ChevronDown.esm.405e6674.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/_isArray.68eb59b2.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/index.096cf963.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/equals.d9a066f9.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/_isString.019dcf3c.js" as="script" crossorigin="anonymous" /><link rel="stylesheet" media="screen" href="https://cdn.vhx.tv/assets/vite/assets/search.c6f13361.css" crossorigin="anonymous" />\n\n\n<script>\n  setTimeout(function() {\n    document.documentElement.className += \' is-loaded\';\n    var selects = document.getElementsByClassName(\'select\'),\n        checkboxes = document.getElementsByClassName(\'checkbox\'),\n        i;\n\n    if (selects.length) {\n      for (i = 0; i < selects.length; i++) {\n        selects[i].className += \' ready\';\n      }\n    }\n\n    if (checkboxes.length) {\n      for (i = 0; i < checkboxes.length; i++) {\n        for (var p=0; p < checkboxes[i].childNodes.length; p++) {\n          if (checkboxes[i].childNodes[p].tagName == \'LABEL\') {\n            checkboxes[i].childNodes[p].className += \' ready\';\n            break;\n          }\n        }\n      }\n    }\n  }, 500);\n</script>\n\n  <script>\n  window.addEventListener(\'load\', function() {\n    jQuery(\'a:contains("Sign In")\').click(function(){\n      ga(\'site.send\',\'event\',\'button\',\'click\',\'sign in\');\n    });\n\n    jQuery(\'a:contains("Sign Up")\').click(function(){\n      ga(\'site.send\',\'event\',\'button\',\'click\',\'sign up\');\n    });\n\n    jQuery(\'a:contains("Watch now")\').click(function(){\n      ga(\'site.send\',\'event\',\'button\',\'click\',\'watch now\');\n    });\n\n    jQuery(\'a:contains("Share")\').click(function(){\n      ga(\'site.send\',\'event\',\'button\',\'click\',\'share\');\n    });\n\n    jQuery(\'[href="https://www.criterionchannel.com/the-deviant-and-the-divine/season:1/videos/freaks"],[href="https://www.criterionchannel.com/the-deviant-and-the-divine/season:1/videos/multiple-maniacs"]\').click(function(){\n      ga(\'site.send\',\'event\',\'button\',\'click\',\'video\');\n    });\n  });\n</script>\n\n\n\n    <script>\n      const javascript = \'\\$(document).ready(function(){\\n\\$(\\\'header .logo\\\').attr(\\\'data-swiftype-index\\\', false);\\n\\n\\$(\\"a[href*=\\\'/checkout\\\']\\").attr(\\"data html(\\"Sign Up\\");\\n  \\$(\\"a[href*=\\\'/checkout/subscribe/gift\\\']\\").html(\\"Buy a gift\\");\\n  \\$(\\"a[href*=\\\'/buy/subscription\\\']\\").attr(\\"data-trigger-disabled\\",true);\\n  \\$(\\"a[href*=\\\'/buy/subscription\\\']\\").html(\\"Sign Up\\");\\n  \\$(\\"a.browse-link\\").html(\\"Now Playing\\");\\n  landingPageChanges();\\n  collectionChanges();\\n  helpChanges();\\n  \\$(\\"body.browse .slide-subtitle\\").hide();\\n  \\$(\\"body.logged-in h1.identity a\\").attr(\\"href\\",\\"/\\");\\n  \\$(\\".navigation-inner\\").addClass(\\"is-ready\\");\\n  \\$(\\"nav .custom-nav-all-films\\").attr(\\"target\\", \\"_self\\");\\n\\n  amazon_fire_tv_supported = \\$(\\\'.amazon-fire-tv\\\').parent().parent().children()[1]\\n  \\$(amazon_fire_tv_supported).text(\\"2nd Gen +\\")\\n\\n  android_tv_supported_text = \\$(\\\'.android-tv\\\').parent().parent().children()[1]\\n  \\$(android_tv_supported_text).text(\\"Selected models\\")\\n  android_tv_4k_text = \\$(\\\'.android-tv\\\').parent().parent().children()[2]\\n  \\$(android_tv_4k_text).hide()\\n\\n  new_xbox = \\"<li>Xbox Series X|S<\\/li>\\";\\n  \\$(new_xbox).insertAfter(\\$(\\\'.platforms-modal-content-tv .platforms-modal-icon-container .platform-modal-devices-list > li:nth-of-type(5) li:nth-of-type(2)\\\'));\\n\\n  \\$(\\\'.platforms-modal-content-tv .platforms-modal-icon-container .platform-modal-devices-list > li:nth-of-type(6) li:nth-of-type(2)\\\').text(\\\'Select 2018, 2019 & 2020 Models\\\');\\n\\n  gift_link = \\\'<a href=\\"https://www.criterionchannel.com/checkout/subscribe/gift?gift=1&duration=6\\" class=\\"is-block text dark block padding-medium\\">Give a Gift<\\/a>\\\';\\n  \\$(gift_link).insertAfter(\\$(\\\'.user-dropdown a[href*=\\"/purchases\\"]\\\'));\\n\\n  if (\\$(\\\'meta[name*=\\"twitter:image\\"]\\\').length > 0) {\\n    twitter_content = \\$(\\\'meta[name*=\\"twitter:image\\"]\\\').attr(\\\'content\\\')\\n    new_meta = \\\'<meta class=\\"swiftype\\" name=\\"image\\" data-type=\\"enum\\" content=\\"\\\' + twitter_content + \\\'\\">\\\';\\n    \\$(\\\'head\\\').append(new_meta);\\n  }\\n});\\n\\nfunction collectionChanges(){\\n  if (\\$(\\"body\\").hasClass(\\"collections\\")){\\n    \\$(\\\'.cover-art\\\').removeClass(\\\'large-offset-2 large-6\\\').addClass(\\\'large-8 large-offset-1\\\');\\n\\n    \\$(\\\'section.product-feature\\\').removeClass(\\\'padding-horizontal-medium padding-top-large padding-bottom-xlarge\\\').addClass(\\\'padding-bottom-large\\\');\\n    \\$(\\\'section.product-feature .row.relative\\\').addClass(\\\'full-width\\\');\\n\\n    \\$(\\\'section.product-feature .collection-details-container\\\').removeClass(\\"large-8 end\\").addClass(\\"large-7\\");\\n      \\$(\\\'section.product-feature .collection-details\\\').addClass(\\"padding-left-large padding-right-large padding-top-xlarge\\");\\n\\n      \\$(\\".embed-container\\").removeClass(\\"grid-padding-right\\");\\n  }\\n  if (\\$(\\\'.collection-playlist\\\').length > 0 || \\$(\\\'.collection-series\\\').length > 0) {\\n    \\$(\\\'.video-controls\\\').removeClass(\\\'padding-top-large\\\').addClass(\\\'padding-top-small\\\');\\n  }\\n\\n  if (\\$(\\\'.collection-movie\\\').length > 0) {\\n    \\$(\\\'.collection-actions a:contains(\\"Trailer\\")\\\').html(\\\'<span class=\\"icon-adjustment relative inverse-icon-bg dark-bg margin-bottom-small\\"><i class=\\"icon small icon-play-inverse\\"><\\/i><\\/span> Trailer\\\');\\n    \\$(\\\'.collection-actions a:contains(\\"Movie\\")\\\').html(\\\'<span class=\\"icon-adjustment relative inverse-icon-bg dark-bg margin-bottom-small\\"><i class=\\"icon small icon-play-inverse\\"><\\/i><\\/span> Watch\\\');\\n  }\\n\\n\\n}\\n\\nfunction landingPageChanges(){\\n  \\$(\\\'a:contains(\\"Start Your Free Month\\")\\\').text(\\"Sign Up Now\\");\\n}\\nfunction helpChanges(){\\n  // Custom Help Center Updates\\n  setTimeout(function() {\\n    \\$(\\\'article li:contains(\\"You can plug your laptop or mobile device into your TV using an HDMI cable\\")\\\').each(function() {\\n      \\$(this).html(\\\'<li>It may be possible to connect your laptop or mobile device to your TV using an HDMI cable if all components in the chain are <a href=\\"https://www.criterionchannel.com/help/articles/drm-and-hdcp#HDCP\\">HDCP/DRM compliant.<\\/a><\\/li>\\\');\\n    });\\n\\n    \\$(\\\'article li:contains(\\" and select the language of choice\\")\\\').each(function() {\\n      \\$(this).text(\\$(this).text().replace(\\" and select the language of choice\\", \\"\\"));\\n    });\\n    \\n    a = \\\'<li>If you have a supported device connected to your TV, Criterion also has an app for Apple TV, Amazon Fire, Roku, and Android TV.<\\/li>\\\'\\n    b = \\$(\\\'li:contains(\\"You can mirror or Airplay from your iOS device\xe2\x80\x99s browser.\\")\\\')\\n    b.after(a);\\n\\n    \\$(\\\'li a[href*=\\"670-how-do-i-login-on-the-xbox-app\\"]\\\').parent().hide();\\n  }, 500);\\n\\n  // hide mentions of the VHX articles\\n  // we have to settimeout because the help pages are ajax\\\'ing in content.\\n  // this is best guess on timing\\n  if (\\$(\\"body\\").hasClass(\\"help-articles\\") || \\$(\\"body\\").hasClass(\\"sites-help\\") || \\$(\\"body\\").hasClass(\\"help-categories\\")){\\n   \\$(\\\'#mithril-help\\\').bind(\\"DOMSubtreeModified\\",function(){\\n    var waittoremove = setTimeout(function(){ removeHelpArticles(); }, 500);\\n  });\\n\\n    var waittoremove = setTimeout(function(){ removeHelpArticles(); }, 700);\\n  }\\n}\\nfunction removeHelpArticles(){\\n  var vhxhelp = Array(\\"589e1e742c7d3a784630ccd5\\",\\"581396999033604deb0ec504\\",\\"57fe5ceb9033600277a6980b\\",\\"54b59261e4b0934e822d2132\\",\\"54dbc064e4b086c0c0966908\\",\\"5453fe87e4b02ce342671537\\",\\"555119ebe4b027e1978dfc04\\",\\"549206b9e4b07d03cb251286\\", \\"5c7eee4f04286350d088ab37\\");\\n  for (i=0;i<=vhxhelp.length-1;i++){\\n    \\$(\\".article-\\"+vhxhelp[i]).remove();                                                        \\n  }\\n}\';\n      try {\n        // Using eval so we can catch syntax errors\n        eval(javascript);\n      } catch (err) {\n        console.error(err);\n      }\n    </script>\n</body>\n</html>\n'
    return BeautifulSoup(content, 'html5lib')


@pytest.fixture
def soup_none():
    content = b'<!doctype html>\n<!--[if lt IE 7]> <html class="no-js lt-ie9 lt-ie8 lt-ie7" lang="en" xml:lang="en"> <![endif]-->\n<!--[if IE 7]>    <html class="no-js lt-ie9 lt-ie8" lang="en" xml:lang="en"> <![endif]-->\n<!--[if IE 8]>    <html class="no-js lt-ie9" lang="en" xml:lang="en"> <![endif]-->\n<!--[if gt IE 8]><!--> <html class="no-js" lang="en" xml:lang="en"> <!--<![endif]-->\n<head lang="en" xml:lang="en">\n  <meta charset="utf-8">\n  <meta http-equiv="X-UA-Compatible" content="IE=edge,chrome=1">\n  <meta name="viewport" content="width=device-width, initial-scale=1">\n\n    <link rel="shortcut icon" href="https://dr56wvhu2c8zo.cloudfront.net/criterionchannel/assets/5922c7f6-cdda-4786-8ac9-521602da043a.png" type="image/png">\n\n    <meta name="apple-itunes-app" content="app-id=1454275199, app-argument=https://criterionchannel.vhx.tv/three-by-jafar-panahi">\n\n    <meta name="facebook-domain-verification" content="qk8q6gxhll8cwf99kggygprhe0y9fz">\n\n  <title>Three by Jafar Panahi - The Criterion Channel</title>\n  <meta name="description" content="The brilliant Iranian auteur Jafar Panahi was sentenced to six years in prison in July 2022, after refusing to stop making urgent, perceptive films when he was banned from the profession in 2010. With NO BEARS, his latest film, coming out in December, there\xe2\x80\x99s no better time to revisit three of his most beloved masterpieces, all of which view modern Iran through the eyes of young girls.">\n  <link rel="canonical" href="https://www.criterionchannel.com/three-by-jafar-panahi" />\n\n    <meta property="og:url" content="https://www.criterionchannel.com/three-by-jafar-panahi">\n  <meta property="fb:app_id" content="136862186391940">\n  <meta property="og:title" content="Three by Jafar Panahi - The Criterion Channel">\n  <meta property="og:site_name" content="The Criterion Channel">\n  <meta property="og:image" content="https://vhx.imgix.net/criterionchannelchartersu/assets/2848bf05-4da0-4898-9e36-7e7f143bb35c.jpg?auto=format%2Ccompress&fit=crop&h=720&q=75&w=1280">\n  <meta property="og:image:width" content="1280">\n  <meta property="og:image:height" content="720">\n  <meta property="og:description" content="The brilliant Iranian auteur Jafar Panahi was sentenced to six years in prison in July 2022, after refusing to stop making urgent, perceptive films when he was banned from the profession in 2010. With NO BEARS, his latest film, coming out in December, there\xe2\x80\x99s no better time to revisit three of hi...">\n  <meta property="og:type" content="website">\n\n  <meta name="twitter:card" content="summary_large_image">\n    <meta name="twitter:site" content="@criterionchannl">\n    <meta name="twitter:creator" content="@criterionchannl">\n  <meta name="twitter:title" content="Three by Jafar Panahi - The Criterion Channel">\n  <meta name="twitter:description" content="The brilliant Iranian auteur Jafar Panahi was sentenced to six years in prison in July 2022, after refusing to stop making urgent, perceptive films when he was banned from the profession in 2010. With NO BEARS, his latest film, coming out in December, there\xe2\x80\x99s no better time to revisit three of hi...">\n  <meta name="twitter:image" content="https://vhx.imgix.net/criterionchannelchartersu/assets/2848bf05-4da0-4898-9e36-7e7f143bb35c.jpg?auto=format%2Ccompress&fit=crop&h=720&q=75&w=1280">\n  <meta name="twitter:image:width" content="1280">\n  <meta name="twitter:image:height" content="720">\n\n\n\n<link rel="author" href="/humans.txt">\n<meta name="csrf-param" content="authenticity_token" />\n<meta name="csrf-token" content="uO5q5vC8c272T8RQPgd/ZZsyLOlgVXipSiQHR06WjCjZWoaHSw7uZKMXlIQ4ZV/MkI+gRqNrvAySd8BiX4S1Eg==" />\n\n\n    <!-- apple / android  -->\n  <meta name="mobile-web-app-capable" content="no">\n  <meta name="apple-mobile-web-app-capable" content="no">\n  <meta name="apple-mobile-web-app-title" content="The Criterion Channel">\n  <!-- microsoft tile color -->\n  <meta name="msapplication-TileColor" content="#b9bcc7">\n  <!-- android browser bar color -->\n  <meta name="theme-color" content="#b9bcc7">\n\n    <!-- apple and other browsers -->\n    <link rel="apple-touch-icon" sizes="57x57"    href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=57&w=57">\n    <link rel="apple-touch-icon" sizes="114x114"  href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=114&w=114">\n    <link rel="apple-touch-icon" sizes="72x72"    href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=72&w=72">\n    <link rel="apple-touch-icon" sizes="144x144"  href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=144&w=144">\n    <link rel="apple-touch-icon" sizes="60x60"    href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=60&w=60">\n    <link rel="apple-touch-icon" sizes="120x120"  href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=120&w=120">\n    <link rel="apple-touch-icon" sizes="76x76"    href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=76&w=76">\n    <link rel="apple-touch-icon" sizes="152x152"  href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=152&w=152">\n    <link rel="apple-touch-icon" sizes="180x180"  href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=180&w=180">\n    <!-- windows tile -->\n    <meta name="msapplication-TileImage" content="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=144&w=144">\n    <!-- android icons -->\n    <link rel="icon" type="image/png" sizes="32x32" href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=32&w=32">\n    <link rel="icon" type="image/png" sizes="16x16" href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=16&w=16">\n\n\n    <meta class="swiftype" name="type" data-type="enum" content="Channel" />\n\n\n\n  <link rel="stylesheet" media="all" href="https://cdn.vhx.tv/assets/webpack/icons.svg-ottbundle-798d02cf9d8ee37d5dfe.css" />\n  <link rel="stylesheet" media="all" href="https://cdn.vhx.tv/assets/webpack/sites-ottbundle-edf8dd6d4f778f83ed2d.css" />\n\n    <link rel="stylesheet" media="screen" href="https://cdn.vhx.tv/assets/templates/_shared/typography.montserrat-3b605b0b18ab721686d69bf9c5bd93fc8f657562bc6a431ae71c409675dfb333.css" />\n\n<style>\n\n\n  /* main colors */\n  .site-background-color {\n    background-color: #141414 !important;\n  }\n  .site-primary-highlight-color {\n    background: #b9bcc7 !important;\n  }\n  .site-primary-color,\n  strong.site-primary-color,\n  .site-primary-color strong,\n  a.site-primary-color:link,\n  a.site-primary-color:visited {\n    color: #b9bcc7 !important;\n  }\n  a.site-primary-color:hover,\n  a.site-primary-color:active {\n    color: #9da0a9 !important;\n  }\n\n  .site-border-color,\n  .site-border-color.border-light {\n    border-color: #323232 !important;\n  }\n  .site-border-primary-color,\n  .site-border-primary-color.border-light {\n    border-color: #b9bcc7 !important;\n  }\n\n  /* custom font primary family */\n  .site-font-primary-family {\n    font-family: \'montserrat\', sans-serif !important;\n  }\n\n  /* font primary color */\n  .site-font-primary-color,\n  .site-font-primary-color strong {\n    color: #ffffff !important;\n  }\n\n  /* font secondary color */\n  .site-font-secondary-color,\n  .site-font-secondary-color strong {\n    color: #bebebb !important;\n  }\n\n  /* link primary color */\n  a.site-link-primary-color:link,\n  a.site-link-primary-color:visited,\n  a.site-link-primary-color strong {\n    color: #ffffff !important;\n  }\n  a.site-link-primary-color:hover,\n  a.site-link-primary-color:active {\n    color: #d9d9d9 !important;\n    opacity: .7;\n  }\n  .site-link-primary-color-contain a:link,\n  .site-link-primary-color-contain a:visited {\n    color: #b9bcc7 !important;\n  }\n  .site-link-primary-color-contain a:hover,\n  .site-link-primary-color-contain a:active {\n    color: #9da0a9 !important;\n    opacity: .7;\n  }\n\n  /* link secondary color */\n  .pagination a,\n  a.site-link-secondary-color:link,\n  a.site-link-secondary-color:visited,\n  a.site-link-secondary-color strong,\n  .site-link-secondary-color-contain a:link,\n  .site-link-secondary-color-contain a:visited {\n    color: #7c7c7c;\n  }\n  .pagination a:hover,\n  a.site-link-secondary-color:hover,\n  a.site-link-secondary-color:active,\n  .site-link-secondary-color-contain a:hover,\n  .site-link-secondary-color-contain a:active {\n    color: #696969;\n    opacity: .7;\n  }\n\n  /* link underline font primary color */\n  .site-link-contain a:link,\n  .site-link-contain a:visited {\n    color: #ffffff !important;\n    text-decoration: underline;\n  }\n  .site-link-contain a:hover,\n  .site-link-contain a:active {\n    color: #ffffff !important;\n    opacity: .7;\n    text-decoration: underline;\n  }\n\n  /* border color same as primary color */\n  .site-border-primary-color {\n    border-color: #ffffff !important;\n  }\n  .site-border-top-primary-color {\n    border-top-color: #ffffff !important;\n  }\n  .site-border-bottom-primary-color {\n    border-bottom-color: #ffffff !important;\n  }\n  .site-border-left-primary-color {\n    border-left-color: #ffffff !important;\n  }\n  .site-border-right-primary-color {\n    border-right-color: #ffffff !important;\n  }\n\n  /* link header color */\n  a.site-link-header-color:link,\n  a.site-link-header-color:visited,\n  a.site-link-header-color:link strong,\n  a.site-link-header-color:visited strong {\n    color: #ffffff !important;\n  }\n  a.site-link-header-color:hover,\n  a.site-link-header-color:active {\n    color: #d9d9d9;\n    opacity: .7;\n  }\n .avatar-chevron.icon {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2223%22%20height%3D%2223%22%20viewBox%3D%220%200%2023%2023%22%3E%3Cpolygon%20fill%3D%22%23ffffff%22%20points%3D%2222.9%2C7.2%2020%2C4.4%2011.5%2C12.9%203%2C4.4%200.1%2C7.2%208.7%2C15.7%208.7%2C15.7%2011.5%2C18.6%2011.5%2C18.6%2011.5%2C18.6%2014.3%2C15.7%2014.3%2C15.7%20%22%2F%3E%3C%2Fsvg%3E%0A");\n    background-repeat: no-repeat;\n  }\n  .user-dropdown a:hover {\n    background-color: rgba(51,51,51,0.2);\n    color: #fff;\n  }\n  .user-dropdown input {\n    background-color: #ffffff;\n  }\n  .user-dropdown input:hover {\n    background-color: rgba(51,51,51,0.2);\n    color: #fff;\n  }\n\n  /* buttons site primary */\n  .btn-site-primary,\n  .btn-site-primary:link,\n  .btn-site-primary:visited {\n    color: #000;\n    border-color: #9da0a9;\n    background-color: #b9bcc7;\n  }\n  .btn-site-primary:hover,\n  .btn-site-primary.is-hover,\n  .btn-site-primary:focus,\n  .btn-site-primary:focus:hover,\n  .btn-site-primary:active,\n  .btn-site-primary.is-active {\n    border-color: #9da0a9;\n    background-color: #9da0a9;\n    color: #000;\n  }\n\n  .btn-site-primary[disabled],\n  .btn-site-primary.is-disabled,\n  .btn-site-primary[disabled]:visited,\n  .btn-site-primary.is-disabled:visited,\n  .btn-site-primary[disabled]:hover,\n  .btn-site-primary.is-disabled:hover,\n  .btn-site-primary[disabled].is-hover,\n  .btn-site-primary.is-disabled.is-hover,\n  .btn-site-primary[disabled]:active,\n  .btn-site-primary.is-disabled:active,\n  .btn-site-primary[disabled]:focus,\n  .btn-site-primary.is-disabled:focus,\n  .btn-site-primary[disabled]:focus:hover,\n  .btn-site-primary.is-disabled:focus:hover,\n  .btn-site-primary[disabled].is-active,\n  .btn-site-primary.is-disabled.is-active {\n    opacity: .5;\n    background-color: #b9bcc7;\n    color: #000;\n    border-color: #9da0a9;\n  }\n\n  /* buttons site secondary */\n  .btn-site-secondary,\n  .btn-site-secondary:link,\n  .btn-site-secondary:visited {\n    color: #bebebb;\n    border-color: #141414;\n    background-color: #323232;\n  }\n  .btn-site-secondary:hover,\n  .btn-site-secondary.is-hover,\n  .btn-site-secondary:focus,\n  .btn-site-secondary:active,\n  .btn-site-secondary.is-active {\n    color: #5f5f5e;\n    border-color: #0f0f0f;\n    background-color: #323232;\n  }\n\n  .btn-site-primary.is-processing,\n  .btn-site-secondary.is-processing {\n    color: transparent !important;\n    background-position: center center !important;\n    background-repeat: no-repeat;\n    background-size: 23px 23px;\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3Csvg%20version%3D%221.1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%20%20viewBox%3D%220%200%2040%2040%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cpath%20opacity%3D%220.2%22%20fill%3D%22%23000000%22%20d%3D%22M20.201%2C5.169c-8.254%2C0-14.946%2C6.692-14.946%2C14.946c0%2C8.255%2C6.692%2C14.946%2C14.946%2C14.946%0D%0A%20%20s14.946-6.691%2C14.946-14.946C35.146%2C11.861%2C28.455%2C5.169%2C20.201%2C5.169z%20M20.201%2C31.749c-6.425%2C0-11.634-5.208-11.634-11.634%0D%0A%20%20c0-6.425%2C5.209-11.634%2C11.634-11.634c6.425%2C0%2C11.633%2C5.209%2C11.633%2C11.634C31.834%2C26.541%2C26.626%2C31.749%2C20.201%2C31.749z%22%2F%3E%0D%0A%3Cpath%20fill%3D%22%23000000%22%20d%3D%22M26.013%2C10.047l1.654-2.866c-2.198-1.272-4.743-2.012-7.466-2.012h0v3.312h0%0D%0A%20%20C22.32%2C8.481%2C24.301%2C9.057%2C26.013%2C10.047z%22%3E%0D%0A%20%20%3CanimateTransform%20attributeType%3D%22xml%22%0D%0A%20%20%20%20attributeName%3D%22transform%22%0D%0A%20%20%20%20type%3D%22rotate%22%0D%0A%20%20%20%20from%3D%220%2020%2020%22%0D%0A%20%20%20%20to%3D%22360%2020%2020%22%0D%0A%20%20%20%20dur%3D%220.5s%22%0D%0A%20%20%20%20repeatCount%3D%22indefinite%22%2F%3E%0D%0A%20%20%3C%2Fpath%3E%0D%0A%3C%2Fsvg%3E\');\n  }\n\n  /* buttons dropdown */\n  .btn-dropdown-site,\n  .btn-dropdown-site:link,\n  .btn-dropdown-site:visited {\n    color: #000000;\n    border-color: #b9bcc7;\n    background-color: #b9bcc7;\n  }\n  .btn-dropdown-site:hover,\n  .btn-dropdown-site.is-hover,\n  .btn-dropdown-site:focus,\n  .btn-dropdown-site:active,\n  .btn-dropdown-site.is-active {\n    border-color: #b9bcc7;\n    background-color: #9da0a9;\n  }\n\n  .site-btn-block,\n  .site-btn-block:link,\n  .site-btn-block:visited {\n    color: #ffffff;\n  }\n\n  .site-btn-block:hover,\n  .site-btn-block.is-hover,\n  .site-btn-block.is-active,\n  .site-btn-block:active {\n    box-shadow: inset 0px -4px 0px #ffffff;\n  }\n\n  .site-form-color input[type="email"],\n  .site-form-color input[type="text"],\n  .site-form-color input[type="search"],\n  .site-form-color input[type="password"],\n  .site-form-color input[type="tel"],\n  .site-form-color textarea {\n    color: #323232 !important;\n    border-color: #696969 !important;\n    background-color:  #fcfcfc !important;\n  }\n  .site-form-color label,\n  .site-form-color-minimal input[type="email"],\n  .site-form-color-minimal input[type="text"],\n  .site-form-color-minimal input[type="search"],\n  .site-form-color-minimal input[type="password"],\n  .site-form-color-minimal textarea {\n    color: #ffffff;\n    background: transparent !important;\n  }\n  .site-form-color input[type="text"]:focus,\n  .site-form-color input[type="search"]:focus,\n  .site-form-color input[type="password"]:focus,\n  .site-form-color input[type="tel"]:focus,\n  .site-form-color textarea:focus {\n    border-color: #7c7c7c !important;\n  }\n\n  .site-form-color input[type="text"]::-webkit-input-placeholder,\n  .site-form-color-minimal input[type="text"]::-webkit-input-placeholder,\n  .site-form-color input[type="email"]::-webkit-input-placeholder,\n  .site-form-color-minimal input[type="email"]::-webkit-input-placeholder,\n  .site-form-color input[type="search"]::-webkit-input-placeholder,\n  .site-form-color-minimal input[type="search"]::-webkit-input-placeholder,\n  .site-form-color input[type="password"]::-webkit-input-placeholder,\n  .site-form-color-minimal input[type="password"]::-webkit-input-placeholder,\n  .site-form-color input[type="tel"]::-webkit-input-placeholder,\n  .site-form-color-minimal input[type="tel"]::-webkit-input-placeholder,\n  .site-form-color textarea::-webkit-input-placeholder,\n  .site-form-color-minimal textarea::-webkit-input-placeholder {\n    color: #323232 !important;\n    opacity: .5;\n  }\n  .site-form-color input[type="text"]::-moz-input-placeholder,\n  .site-form-color-minimal input[type="text"]::-moz-input-placeholder,\n  .site-form-color input[type="email"]::-moz-input-placeholder,\n  .site-form-color-minimal input[type="email"]::-moz-input-placeholder,\n  .site-form-color input[type="search"]::-moz-input-placeholder,\n  .site-form-color-minimal input[type="search"]::-moz-input-placeholder,\n  .site-form-color input[type="password"]::-moz-input-placeholder,\n  .site-form-color-minimal input[type="password"]::-moz-input-placeholder,\n  .site-form-color input[type="tel"]::-moz-input-placeholder,\n  .site-form-color-minimal input[type="tel"]::-moz-input-placeholder,\n  .site-form-color textarea::-moz-input-placeholder,\n  .site-form-color-minimal textarea::-moz-input-placeholder {\n    color: #323232 !important;\n    opacity: .5;\n  }\n\n  .site-form-color input[type="text"]:-moz-input-placeholder,\n  .site-form-color-minimal input[type="text"]:-moz-input-placeholder,\n  .site-form-color input[type="email"]:-moz-input-placeholder,\n  .site-form-color-minimal input[type="email"]:-moz-input-placeholder,\n  .site-form-color input[type="search"]:-moz-input-placeholder,\n  .site-form-color-minimal input[type="search"]:-moz-input-placeholder,\n  .site-form-color input[type="password"]:-moz-input-placeholder,\n  .site-form-color-minimal input[type="password"]:-moz-input-placeholder,\n  .site-form-color input[type="tel"]:-moz-input-placeholder,\n  .site-form-color-minimal input[type="tel"]:-moz-input-placeholder,\n  .site-form-color textarea:-moz-input-placeholder,\n  .site-form-color-minimal textarea:-moz-input-placeholder {\n    color: #323232 !important;\n    opacity: .5;\n  }\n\n  .site-form-color input[type="text"]:-ms-input-placeholder,\n  .site-form-color-minimal input[type="text"]:-ms-input-placeholder,\n  .site-form-color input[type="email"]:-ms-input-placeholder,\n  .site-form-color-minimal input[type="email"]:-ms-input-placeholder,\n  .site-form-color input[type="search"]:-ms-input-placeholder,\n  .site-form-color-minimal input[type="search"]:-ms-input-placeholder,\n  .site-form-color input[type="password"]:-ms-input-placeholder,\n  .site-form-color-minimal input[type="password"]:-ms-input-placeholder,\n  .site-form-color input[type="tel"]:-ms-input-placeholder,\n  .site-form-color-minimal input[type="tel"]:-ms-input-placeholder,\n  .site-form-color textarea:-ms-input-placeholder,\n  .site-form-color-minimal textarea:-ms-input-placeholder {\n    color: #323232 !important;\n    opacity: .5;\n  }\n\n  .site-form-color .radio label .radio--icon {\n    border-color: #bebebb !important;\n  }\n  .site-form-color .radio.primary .circle-top span {\n    background: #b9bcc7 !important;\n  }\n  .site-form-color .radio.primary .circle-bottom span {\n    background: #94969f !important;\n  }\n\n  /* header */\n  .site-header {\n    border-bottom: 1px solid #2e2e2e;\n    background-color: #141414;\n\n\n  }\n  .site-header-background-color {\n    background-color: #141414;\n  }\n  @media only screen and (max-width: 40em) {\n     .site-header-background-color-small-only {\n      background-color: #141414;\n    }\n  }\n\n  /* modals */\n  .site-modal {\n    box-shadow: inset 0 0 2px #7c7c7c !important;\n  }\n\n  /* tooltips */\n  .tooltip.bottom-right:after,\n  .tooltip.bottom-left:after,\n  .tooltip.bottom:after,\n  .tooltip.bottom-right:before,\n  .tooltip.bottom-left:before,\n  .tooltip.bottom:before {\n    border-top-color: #7c7c7c;\n  }\n  .tooltip.right:after,\n  .tooltip.right:after\n  .tooltip.right-top:after,\n  .tooltip.right-top:before,\n  .tooltip.right-bottom:before,\n  .tooltip.right-bottom:before {\n    border-right-color: #7c7c7c;\n  }\n\n  .tooltip.left-bottom:after,\n  .tooltip.left-top:after,\n  .tooltip.left:after,\n  .tooltip.left-bottom:before,\n  .tooltip.left-top:before,\n  .tooltip.left:before {\n    border-left-color: #7c7c7c;\n  }\n\n  .tooltip.top-right:after,\n  .tooltip.top-left:after,\n  .tooltip.top:after,\n  .tooltip.top-right:before,\n  .tooltip.top-left:before,\n  .tooltip.top:before {\n    border-bottom-color: #7c7c7c;\n  }\n\n  .tooltip {\n    border: 1px solid #7c7c7c;\n    background: #7c7c7c;\n    color: #ffffff;\n  }\n\n  .reveal-modal {\n    border-color: #323232;\n    box-shadow: 0 0 3px #cbcbcb;\n  }\n  .reveal-modal-bg {\n    background-color: rgba(0,0,0,.8);\n  }\n  .play-outer-circle-large:hover {\n    border-color: #b9bcc7;\n  }\n  .forums-container tr.clickable:hover .thread-activity {\n    background-color: #141414;\n  }\n\n    [data-flash].is-inline.is-error {\n      background: rgba(211, 16, 74, 1) !important;\n    }\n    [data-flash].is-inline.is-error .flash-msg,\n    [data-flash].is-inline.is-error .flash-msg strong {\n      color: #fff !important;\n    }\n    [data-flash].is-inline.is-error .flash-close .icon-remove-mono {\n      display: none !important;\n    }\n    [data-flash].is-inline.is-error .flash-close .icon-remove-inverse {\n      display: block !important;\n    }\n\n  /* custom banner */\n\n\n</style>\n<style>\n  /* icons site primary */\n  .site-icon-arrow-right {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2223%22%20height%3D%2223%22%20viewBox%3D%220%200%2023%2023%22%3E%3Ctitle%3Eicons-arrows-set%20copy%3C%2Ftitle%3E%3Cpath%20d%3D%22M5.61%2C19.75l8.26-8.26L5.61%2C3.24c-0.68-.69.61-1.58%2C1-2%2C0.72-.69%2C1.76.78%2C2.2%2C1.22l3.39%2C3.39L17.41%2C11c0.71%2C0.69-.8%2C1.77-1.22%2C2.19L12.8%2C16.59%2C7.63%2C21.77c-0.7.67-1.58-.61-2-1A0.68%2C0.68%2C0%2C0%2C1%2C5.61%2C19.75Z%22%20fill%3D%22%23b9bcc7%22%2F%3E%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-menu {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Hamburger%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Crect%20x%3D%222.6%22%20y%3D%223.6%22%20width%3D%2218%22%20height%3D%223%22%20fill%3D%22%23ffffff%22%2F%3E%0D%0A%3Crect%20x%3D%222.6%22%20y%3D%2210%22%20width%3D%2218%22%20height%3D%223%22%20fill%3D%22%23ffffff%22%2F%3E%0D%0A%3Crect%20x%3D%222.6%22%20y%3D%2216.3%22%20width%3D%2218%22%20height%3D%223%22%20fill%3D%22%23ffffff%22%2F%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-chrevron-down {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21--%20Generator%3A%20Adobe%20Illustrator%2018.1.1%2C%20SVG%20Export%20Plug-In%20.%20SVG%20Version%3A%206.00%20Build%200%29%20%20--%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%20%20%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%22-294%20385%2023%2023%22%20enable-background%3D%22new%20-294%20385%2023%2023%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cg%3E%0D%0A%20%20%3Cpolygon%20fill%3D%22%23ffffff%22%20points%3D%22-271.1%2C392.9%20-275.4%2C388.7%20-282.5%2C395.8%20-289.6%2C388.7%20-293.9%2C392.9%20-286.7%2C400%20-286.8%2C400.1%20%0D%0A%20%20%20%20-282.5%2C404.3%20-282.5%2C404.3%20-282.5%2C404.3%20-278.2%2C400.1%20-278.3%2C400%20%20%22%2F%3E%0D%0A%3C%2Fg%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-chat {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21--%20Generator%3A%20Adobe%20Illustrator%2018.1.1%2C%20SVG%20Export%20Plug-In%20.%20SVG%20Version%3A%206.00%20Build%200%29%20%20--%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%22-294%20385%2023%2023%22%20enable-background%3D%22new%20-294%20385%2023%2023%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cg%3E%0D%0A%09%3Cg%3E%0D%0A%09%09%3Cpath%20fill%3D%22%23ffffff%22%20d%3D%22M-282.5%2C388.6c2.6%2C0%2C5.1%2C0.1%2C6.8%2C1.3c1.9%2C1.4%2C1.9%2C4.3%2C1.7%2C6c-0.2%2C2.1-1.2%2C3.5-3%2C4.2%0D%0A%09%09%09c-1.3%2C0.5-3%2C0.7-5.1%2C0.7c-1%2C0-2.1-0.1-3.4-0.2l-1-0.1l0%2C0l-1.2-0.4c-1.1-0.3-2.5-1-3.1-2.6c-0.7-1.8-0.7-4.2%2C0-5.9%0D%0A%09%09%09C-289.6%2C389.1-286.6%2C388.6-282.5%2C388.6%20M-282.5%2C386.6c-3.7%2C0-8.4%2C0.3-9.9%2C4.3c-0.9%2C2.2-0.9%2C5.1%2C0%2C7.4c0.7%2C2%2C2.4%2C3.2%2C4.3%2C3.8v3.2%0D%0A%09%09%09l2.6-2.7c1.1%2C0.1%2C2.4%2C0.2%2C3.6%2C0.2c2%2C0%2C4-0.2%2C5.7-0.9c2.6-1%2C4-3.1%2C4.3-5.8c0.3-2.8-0.1-6.1-2.6-7.9%0D%0A%09%09%09C-276.7%2C386.6-279.8%2C386.6-282.5%2C386.6L-282.5%2C386.6z%22%2F%3E%0D%0A%09%3C%2Fg%3E%0D%0A%3C%2Fg%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n\n  /* icons site social */\n  .site-icon-twitter {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%22-294%20385%2023%2023%22%20enable-background%3D%22new%20-294%20385%2023%2023%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cg%3E%0D%0A%09%3Cpath%20fill%3D%22%23ffffff%22%20d%3D%22M-272.5%2C390.3c-0.7%2C0.3-1.5%2C0.5-2.4%2C0.6c0.8-0.5%2C1.5-1.3%2C1.8-2.3c-0.8%2C0.5-1.7%2C0.8-2.6%2C1%0D%0A%09%09c-2.8-3-7.9-0.2-7%2C3.7c-3.4-0.1-6.4-1.7-8.4-4.2c-1.1%2C1.9-0.5%2C4.3%2C1.3%2C5.5c-0.7%2C0-1.3-0.2-1.9-0.5c0%2C1.9%2C1.4%2C3.7%2C3.3%2C4.1%0D%0A%09%09c-0.6%2C0.2-1.2%2C0.2-1.9%2C0.1c0.5%2C1.6%2C2%2C2.8%2C3.8%2C2.8c-1.7%2C1.3-3.9%2C2-6.1%2C1.7c4%2C2.6%2C9.5%2C2.5%2C13.4-0.5c3-2.3%2C4.8-6.1%2C4.6-9.9%0D%0A%09%09C-273.7%2C391.8-273.1%2C391.1-272.5%2C390.3C-273.2%2C390.6-273.1%2C391.1-272.5%2C390.3z%22%2F%3E%0D%0A%3C%2Fg%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-facebook {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21--%20Generator%3A%20Adobe%20Illustrator%2018.1.1%2C%20SVG%20Export%20Plug-In%20.%20SVG%20Version%3A%206.00%20Build%200%29%20%20--%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%22-294%20385%2023%2023%22%20enable-background%3D%22new%20-294%20385%2023%2023%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cpath%20fill%3D%22%23ffffff%22%20d%3D%22M-280.4%2C406.8v-9.3h3.4v-4h-3.4v-2.1c0-0.5%2C0.3-0.9%2C0.9-0.9h2.6v-4.5h-2.8c-3%2C0-5.3%2C2.4-5.3%2C5.5v2.1h-3.1v4%0D%0A%09h3.1v9.3h4.6L-280.4%2C406.8z%22%2F%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-tumblr {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cg%3E%0D%0A%09%3Cpath%20style%3D%22fill%3A%23ffffff%3B%22%20d%3D%22M17.6%2C20.4c-2%2C1-4.4%2C1.4-6.6%2C0.8c-2.3-0.6-3.4-2.3-3.4-4.5v-7H5.4V6.9c2.5-0.8%2C3.7-2.9%2C4-5.4h2.8%0D%0A%09%09v5.1h4.7v3.1h-4.7v5.1c0%2C1-0.2%2C2.4%2C0.9%2C3c1.4%2C0.8%2C3.2%2C0.2%2C4.5-0.6V20.4z%22%2F%3E%0D%0A%3C%2Fg%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-envelope {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cg%3E%0D%0A%09%3Cpath%20style%3D%22fill%3A%23ffffff%3B%22%20d%3D%22M22.6%2C9c0-0.2-0.1-0.5-0.3-0.6L12%2C0.8c-0.3-0.2-0.6-0.2-0.9%2C0L0.7%2C8.4C0.5%2C8.5%2C0.4%2C8.8%2C0.4%2C9v12.6%0D%0A%09%09c0%2C0.4%2C0.3%2C0.8%2C0.8%2C0.8h20.7c0.4%2C0%2C0.8-0.4%2C0.8-0.8V9z%20M11.5%2C2.4l9.6%2C7v0.9l-6.6%2C4.7l5.2%2C3.3c0.8%2C0.5%2C0%2C1.8-0.8%2C1.3L11.5%2C15%0D%0A%09%09l-7.4%2C4.7c-0.8%2C0.5-1.7-0.8-0.8-1.3l5.2-3.3l-6.6-4.7V9.4L11.5%2C2.4z%22%2F%3E%0D%0A%09%3Cpath%20style%3D%22fill%3A%23ffffff%3B%22%20d%3D%22M19.7%2C18.4%22%2F%3E%0D%0A%09%3Cpolygon%20style%3D%22fill%3A%23ffffff%3B%22%20points%3D%2211.5%2C13.1%2013.4%2C14.3%2019.5%2C9.9%2011.5%2C4%203.5%2C9.9%209.6%2C14.3%20%09%22%2F%3E%0D%0A%3C%2Fg%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-check {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cg%3E%0D%0A%09%0D%0A%09%09%3Crect%20x%3D%2211.9%22%20y%3D%222.1%22%20transform%3D%22matrix%28-0.7072%20-0.7071%200.7071%20-0.7072%2016.2216%2029.719%29%22%20style%3D%22fill%3A%23ffffff%3B%22%20width%3D%224.7%22%20height%3D%2218.8%22%2F%3E%0D%0A%09%0D%0A%09%09%3Crect%20x%3D%220.2%22%20y%3D%2211.3%22%20transform%3D%22matrix%280.7072%200.707%20-0.707%200.7072%2011.5962%20-0.598%29%22%20style%3D%22fill%3A%23ffffff%3B%22%20width%3D%2212.5%22%20height%3D%224.7%22%2F%3E%0D%0A%3C%2Fg%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n\n  /* icons site secondary */\n  .site-icon-clock {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cg%3E%0D%0A%09%3Cpath%20style%3D%22fill%3A%23bebebb%3B%22%20d%3D%22M11.5%2C22c-4.5%2C0-8.6-3-10-7.3c-1.4-4.3%2C0.2-9.2%2C4-11.8c3.7-2.6%2C8.9-2.5%2C12.5%2C0.3%0D%0A%09%09c3.6%2C2.8%2C5%2C7.7%2C3.4%2C12C19.8%2C19.3%2C15.8%2C22%2C11.5%2C22z%20M11.5%2C3.3C7%2C3.3%2C3.3%2C7%2C3.3%2C11.5c0%2C4.5%2C3.7%2C8.2%2C8.2%2C8.2c4.5%2C0%2C8.2-3.7%2C8.2-8.2%0D%0A%09%09C19.7%2C7%2C16%2C3.3%2C11.5%2C3.3z%22%2F%3E%0D%0A%09%3Cpath%20style%3D%22fill%3A%23ffffff%3B%22%20d%3D%22M12.9%2C12.9v-2.9V5.5c0-0.4-0.4-0.8-0.8-0.8h-1.3c-0.4%2C0-0.8%2C0.4-0.8%2C0.8v4.5H6.5%0D%0A%09%09c-0.4%2C0-0.8%2C0.4-0.8%2C0.8v1.3c0%2C0.4%2C0.4%2C0.8%2C0.8%2C0.8h3.5H12.9z%22%2F%3E%0D%0A%3C%2Fg%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-reply {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cg%3E%0D%0A%09%3Cpath%20style%3D%22fill%3A%23bebebb%3B%22%20d%3D%22M20.7%2C20c-0.1%2C0.3-0.3%2C0.7-0.4%2C0.9c-0.1%2C0.1-0.2%2C0.2-0.3%2C0.2c-0.2%2C0-0.4-0.2-0.4-0.4%0D%0A%09%09c0-0.2%2C0-0.4%2C0.1-0.6c0-0.5%2C0.1-1%2C0.1-1.5c0-5.7-3.4-6.7-8.6-6.7H8.4V15c0%2C0.4-0.3%2C0.8-0.8%2C0.8c-0.2%2C0-0.4-0.1-0.5-0.2L1%2C9.4%0D%0A%09%09C0.8%2C9.2%2C0.8%2C9%2C0.8%2C8.8c0-0.2%2C0.1-0.4%2C0.2-0.5l6.1-6.1C7.3%2C2%2C7.5%2C1.9%2C7.7%2C1.9c0.4%2C0%2C0.8%2C0.3%2C0.8%2C0.8v3.1h2.7%0D%0A%09%09c3.9%2C0%2C8.8%2C0.7%2C10.5%2C4.8c0.5%2C1.3%2C0.6%2C2.6%2C0.6%2C4C22.2%2C16.2%2C21.4%2C18.4%2C20.7%2C20z%22%2F%3E%0D%0A%3C%2Fg%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-remove {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cpath%20style%3D%22fill%3A%23bebebb%3B%22%20d%3D%22M19.6%2C17.5l-6-6l6-6c0.3-0.3%2C0.3-0.8%2C0-1.1l-0.9-0.9c-0.3-0.3-0.8-0.3-1.1%2C0l-6%2C6l-6-6%0D%0A%09c-0.3-0.3-0.8-0.3-1.1%2C0L3.4%2C4.4c-0.3%2C0.3-0.3%2C0.8%2C0%2C1.1l6%2C6l-6%2C6c-0.3%2C0.3-0.3%2C0.8%2C0%2C1.1l0.9%2C0.9c0.3%2C0.3%2C0.8%2C0.3%2C1.1%2C0l6-6l6%2C6%0D%0A%09c0.3%2C0.3%2C0.8%2C0.3%2C1.1%2C0l0.9-0.9C19.9%2C18.3%2C19.9%2C17.8%2C19.6%2C17.5z%22%2F%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-back {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cpath%20style%3D%22fill%3A%23bebebb%3B%22%20d%3D%22M11.5%2C20.9l1-1c0.3-0.3%2C0.3-0.7%2C0-0.9L6%2C12.5h14.6c0.4%2C0%2C0.7-0.3%2C0.7-0.7v-1.3c0-0.4-0.3-0.7-0.7-0.7%0D%0A%09H6l6.5-6.5c0.3-0.3%2C0.3-0.7%2C0-0.9l-1-1c-0.3-0.3-0.7-0.3-0.9%2C0l-9.4%2C9.4c-0.3%2C0.3-0.3%2C0.7%2C0%2C0.9l9.4%2C9.4%0D%0A%09C10.9%2C21.2%2C11.3%2C21.2%2C11.5%2C20.9z%22%2F%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-link {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cpath%20style%3D%22fill-rule%3Aevenodd%3Bclip-rule%3Aevenodd%3Bfill%3A%23bebebb%3B%22%20d%3D%22M19.3%2C8.2c-1.8%2C1.8-3.7%2C4.6-6.6%2C3.8c0.8-0.8%2C1.6-1.6%2C2.4-2.4%0D%0A%09c0.9-0.9%2C1.8-1.8%2C2.6-2.7c1.9-1.9-0.8-4.8-2.6-2.9c-0.5%2C0.5-0.9%2C0.9-1.4%2C1.4c-0.2%2C0.2-1.4%2C1.1-1.4%2C1.4l0%2C0c-0.8%2C0.8-1.6%2C1.6-2.4%2C2.4%0D%0A%09c-0.4-1.3%2C0-2.8%2C1-3.8c1.3-1.3%2C2.7-3.3%2C4.5-3.8C19.3%2C0.5%2C22.1%2C5.4%2C19.3%2C8.2%22%2F%3E%0D%0A%3Cpath%20style%3D%22fill-rule%3Aevenodd%3Bclip-rule%3Aevenodd%3Bfill%3A%23bebebb%3B%22%20d%3D%22M7.9%2C15.6C7%2C16.5%2C5.5%2C15%2C6.5%2C14.1c1.4-1.4%2C2.7-2.7%2C4.1-4.1%0D%0A%09c0.8-0.8%2C1.5-1.5%2C2.3-2.3c0.2-0.2%2C0.3-0.3%2C0.5-0.5c1-0.6%2C2%2C0.7%2C1.2%2C1.6C12.4%2C11.2%2C10.1%2C13.3%2C7.9%2C15.6z%22%2F%3E%0D%0A%3Cpath%20style%3D%22fill-rule%3Aevenodd%3Bclip-rule%3Aevenodd%3Bfill%3A%23bebebb%3B%22%20d%3D%22M5.8%2C13.4c-0.9%2C0.9-1.8%2C1.8-2.6%2C2.7c-1.9%2C1.9%2C0.8%2C4.8%2C2.6%2C2.9%0D%0A%09c0.9-0.9%2C2-1.8%2C2.8-2.9c0.8-0.8%2C1.6-1.6%2C2.4-2.4c0.4%2C1.3%2C0%2C2.8-1%2C3.8c-1.3%2C1.3-2.7%2C3.3-4.5%2C3.8c-3.8%2C1.1-6.7-3.8-3.9-6.6%0D%0A%09C3.4%2C13%2C5.4%2C10.2%2C8.2%2C11C7.4%2C11.8%2C6.6%2C12.6%2C5.8%2C13.4%22%2F%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-trash {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%221792px%22%20height%3D%221792px%22%20viewBox%3D%22590%20-500%201792%201792%22%20enable-background%3D%22new%20590%20-500%201792%201792%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cpath%20fill%3D%22%23bebebb%22%20d%3D%22M1294%2C876V172c0-9.3-3-17-9-23s-13.7-9-23-9h-64c-9.3%2C0-17%2C3-23%2C9s-9%2C13.7-9%2C23v704c0%2C9.3%2C3%2C17%2C9%2C23%0D%0A%09s13.7%2C9%2C23%2C9h64c9.3%2C0%2C17-3%2C23-9S1294%2C885.3%2C1294%2C876z%20M1550%2C876V172c0-9.3-3-17-9-23s-13.7-9-23-9h-64c-9.3%2C0-17%2C3-23%2C9%0D%0A%09s-9%2C13.7-9%2C23v704c0%2C9.3%2C3%2C17%2C9%2C23s13.7%2C9%2C23%2C9h64c9.3%2C0%2C17-3%2C23-9S1550%2C885.3%2C1550%2C876z%20M1806%2C876V172c0-9.3-3-17-9-23%0D%0A%09s-13.7-9-23-9h-64c-9.3%2C0-17%2C3-23%2C9s-9%2C13.7-9%2C23v704c0%2C9.3%2C3%2C17%2C9%2C23s13.7%2C9%2C23%2C9h64c9.3%2C0%2C17-3%2C23-9S1806%2C885.3%2C1806%2C876z%0D%0A%09%20M1262-116h448l-48-117c-4.7-6-10.3-9.7-17-11h-317c-6.7%2C1.3-12.3%2C5-17%2C11L1262-116z%20M2190-84v64c0%2C9.3-3%2C17-9%2C23s-13.7%2C9-23%2C9h-96%0D%0A%09v948c0%2C55.3-15.7%2C103.2-47%2C143.5s-69%2C60.5-113%2C60.5h-832c-44%2C0-81.7-19.5-113-58.5s-47-86.2-47-141.5V12h-96c-9.3%2C0-17-3-23-9%0D%0A%09s-9-13.7-9-23v-64c0-9.3%2C3-17%2C9-23s13.7-9%2C23-9h309l70-167c10-24.7%2C28-45.7%2C54-63s52.3-26%2C79-26h320c26.7%2C0%2C53%2C8.7%2C79%2C26%0D%0A%09s44%2C38.3%2C54%2C63l70%2C167h309c9.3%2C0%2C17%2C3%2C23%2C9S2190-93.3%2C2190-84z%22%2F%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-replay {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2223%22%20height%3D%2223%22%20viewBox%3D%220%200%2023%2023%22%3E%3Cpath%20d%3D%22M16.4%2016.9c-2.8%202.4-7.1%202.3-9.8-.3l-2%202c3.8%203.8%209.9%203.9%2013.9.3l-2.1-2z%22%20fill%3D%22%23b9bcc7%22%2F%3E%3Cpath%20d%3D%22M15.3%205.3l.9.6c3%202.6%203.7%207.2.9%2010.3l2%202c2.3-2.5%203.1-6%202.3-9.3-.9-3.6-3.8-6.4-7.4-7.3-3.8-.8-8.1.7-10.4%204l.1-1.2c0-.2%200-.5-.2-.7-.6-.8-1.7-.4-1.8.4l-.5%204.6c-.1.6.4%201.1%201%201.1h4.6c.9%200%201.3-1.1.8-1.7-.3-.2-.5-.3-.8-.3h-1.2c2-3.2%206.3-4.4%209.7-2.5z%22%20fill%3D%22%23b9bcc7%22%2F%3E%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-alt-download {\n    background-image: url(\'data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2223%22%20height%3D%2223%22%20viewBox%3D%220%200%2023%2023%22%3E%3Cg%20fill-rule%3D%22evenodd%22%20clip-rule%3D%22evenodd%22%20fill%3D%22%23ffffff%22%3E%3Cpath%20d%3D%22M17.3%2017.2c-1.1%200-1.4-1.5-.5-2%20.4-.2%201-.1%201.5-.3.6-.2%201-.6%201.4-1%201.3-1.6.5-4.1-1.4-4.8-.3-.1-.7-.2-1.1-.2-.2%200-.3-.8-.4-1-.4-.9-1-1.8-1.7-2.4-1.8-1.6-4.5-1.8-6.6-.5-1%20.5-1.7%201.4-2.2%202.3-.2.3-.3.7-.4%201-.2.7-.2.5-.8.7-1.9.4-3%202.5-2.2%204.3.4.9%201.3%201.6%202.3%201.8.5.1%201%200%201.3.4.4.4.3%201.1-.2%201.4-.7.6-2.2.1-2.9-.3-1-.5-1.8-1.3-2.3-2.3-.8-1.6-.7-3.5.2-4.9.4-.8%201.1-1.4%201.8-1.9.4-.2.9-.3%201.1-.7.2-.4.5-.8.7-1.2%202.1-3.1%206.2-4.4%209.6-2.9%201.6.7%203%201.9%203.8%203.4.2.4.3.9.7%201.1l1.2.6c.8.5%201.4%201.2%201.8%202%201.6%203.4-1%207.4-4.7%207.4%22%20fill%3D%22%23000000%22%3E%3C%2Fpath%3E%3Cpath%20d%3D%22M15.4%2017.2l-3.1%203.4c-.4.4-1.1.5-1.5%200-.5-.6-1.1-1.2-1.6-1.8-.2-.2-1.2-1.6-1.5-1.6-.9-1%20.7-2.4%201.5-1.4l1.3%201.4v-3.9c0-.4-.1-.9%200-1.3%200-1.3%202.1-1.3%202.1%200v5.2c.5-.5%201-1.4%201.7-1.7.8-.4%201.8.9%201.1%201.7z%22%20fill%3D%22%23000000%22%3E%3C%2Fpath%3E%3C%2Fg%3E%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-random {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2223%22%20height%3D%2223%22%20viewBox%3D%220%200%2023%2023%22%3E%3Cpath%20d%3D%22M5.7%2015.2v-6H8L4%204.6-.1%209.2h2.3v7c0%201.3%201%202.3%202.3%202.3H15l-3-3.2H5.7z%22%20fill%3D%22#ffffff%22%2F%3E%3Cpath%20d%3D%22M20.7%2013.8v-7c0-1.3-1-2.3-2.3-2.3H8l3%203.2h6.3v6H15l4.1%204.6%204.1-4.6h-2.5z%22%20fill%3D%22#ffffff%22%2F%3E%3C%2Fsvg%3E");\n    background-repeat: no-repeat;\n  }\n  .site-icon-visa {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2271%22%20height%3D%2243%22%20viewBox%3D%220%200%2071%2043%22%3E%3Cpath%20d%3D%22M51.9%2017.9c-.2.5-.5%201.4-.5%201.4l-1.5%204H53c-.2-.7-.9-4.2-.9-4.2l-.2-1.2zM69%200H2C.9%200%200%20.9%200%202v39c0%201.1.9%202%202%202h67c1.1%200%202-.9%202-2V2c0-1.1-.9-2-2-2zM11.6%2014h6c.8%200%201.5.3%201.7%201.2l1.3%206.6c-1.3-3.5-4.4-6.4-9.1-7.6l.1-.2zm7.1%2014.4l-3-11.5c2.5%201.5%204.2%203.5%205%205.9l.2%201%203.7-9.8h3.9l-5.9%2014.4h-3.9zm9%200L30.1%2014h3.7l-2.3%2014.4h-3.8zm10.2.2c-1.7%200-3.3-.4-4.2-.8l.5-3.2.5.2c1.2.5%202.4.8%203.8.8.9-.1%202.2-.6%201.9-1.7-.3-.9-1.6-1.3-2.3-1.7-1-.5-2-1.2-2.5-2.2-1-2%20.2-4.2%201.9-5.3%202.2-1.3%205.1-1.3%207.4-.4l-.5%203.1-.3-.2c-1-.4-4.9-1.3-4.9.7%200%20.8%201.2%201.3%201.8%201.6%201%20.5%202.1%201%202.8%202%201.1%201.4.8%203.5-.2%204.8-1.4%201.8-3.6%202.3-5.7%202.3zm16.2-.2s-.3-1.7-.4-2.2h-4.8c-.1.4-.8%202.2-.8%202.2h-3.9l5.5-13.2c.4-.9%201.1-1.2%202-1.2h2.9l3%2014.4h-3.5z%22%20fill%3D%22#ffffff%22%3E%3C%2Fpath%3E%3C%2Fsvg%3E");\n    background-repeat: no-repeat;\n  }\n  .site-icon-amex {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2271%22%20height%3D%2243%22%20viewBox%3D%220%200%2071%2043%22%3E%3Cpath%20d%3D%22M35.5%2023.5h-5l-2%202.2-1.9-2.2h-6.3v6.6h6.2l2-2.2%201.9%202.2h3.1v-2.2h2c1.5%200%202.7-.6%202.7-2.2%200-1.7-1.3-2.2-2.7-2.2zm-9.9%205.2h-3.9v-1.3h3.5V26h-3.5v-1.2h4l1.7%201.9-1.8%202zm6.3.8l-2.4-2.7%202.4-2.6v5.3zm3.6-3h-2v-1.7h2.1c1.2%200%201.2%201.7-.1%201.7zm9.4-7.8h1.8l.6-1.5h3.3l.6%201.5h3.2v-5l3%205h2.2v-6.6H58v4.6l-2.8-4.6h-2.4v6.2l-2.7-6.2h-2.4l-2.2%205.2H45c-1.6%200-2.1-2.3-1.1-3.4.3-.3.6-.4%201.2-.4h1.5v-1.4h-1.5c-2.1%200-3.2%201.3-3.2%203.3-.1%202%201%203.3%203%203.3zm4.1-5.5l1.1%202.6h-2.2l1.1-2.6zm-15.2%203.1h1.7c.6%200%20.9.1%201.1.3.3.3.2.9.2%201.3v.8h1.6v-1.3c0-.6%200-.9-.3-1.2-.1-.2-.4-.4-.8-.6.9-.4%201.3-1.4%201-2.3-.3-1.2-1.7-1.2-2.6-1.2h-3.6v6.6h1.6v-2.4zm0-2.8h1.9c.3%200%20.6%200%20.7.1.5.2.4%201%200%201.2-.2.1-.5.1-.8.1h-1.9v-1.4zm-2.7%203.8h-3.7V16H31v-1.3h-3.6v-1.2h3.7v-1.4h-5.3v6.6h5.3zM23%2018.7h1.6v-6.6H22l-1.9%204.5-2.1-4.5h-2.6v6.2l-2.7-6.2h-2.3l-2.8%206.6h1.7l.6-1.5h3.3l.6%201.5H17v-5.2l2.3%205.2h1.4l2.3-5.2v5.2zm-12.5-2.9l1.1-2.6%201.1%202.6h-2.2zm35.6%2014.3h5.3v-1.4h-3.7v-1.3h3.6V26h-3.6v-1.2h3.7v-1.3h-5.3zM61.8%2026H61c-.3%200-.6%200-.8-.2-.2-.2-.1-.6%200-.7l.1-.1c.2-.1.3-.1.6-.1h2.9v-1.4h-3.2c-1%200-1.6.4-1.9.8l-.1.1-.1.2c-.1.1-.1.3-.1.4v.7l.1.6c0%20.1%200%20.1.1.2%200%20.1.1.2.1.2l.1.1.1.1s.1%200%20.1.1c.1.1.2.1.3.1%200%200%20.1%200%20.1.1.7.2%201.4.2%202.2.2%201%200%20.9%201.2-.1%201.2h-3V30h3c1.1%200%202.2-.4%202.4-1.6.4-1.6-.6-2.3-2.1-2.4zm-16.5-.8c0-.8-.3-1.2-.8-1.5-.5-.3-1.1-.3-1.9-.3H39V30h1.6v-2.4h1.7c1.1%200%201.4.5%201.4%201.6v.8h1.6v-1.3c0-.8-.2-1.5-1-1.8.3-.1%201-.6%201-1.7zm-2.1%201c-.2.1-.5.1-.8.1h-1.9v-1.5h1.9c.3%200%20.6%200%20.8.1.5.3.4%201%200%201.3-.2.1.2-.2%200%200zM69%200H2C.9%200%200%20.9%200%202v39c0%201.1.9%202%202%202h67c1.1%200%202-.9%202-2V2c0-1.1-.9-2-2-2zM18.2%2018.9V20h-5.3l-.6-1.5h-1.4l-.6%201.5H5.5l4-9.2h4.1l.5%201.1v-1.1H19l1.1%202.4%201-2.4h15.3c.7%200%201.3.1%201.8.5v-.5h4.2v.5c.7-.4%201.6-.5%202.6-.5h6.1l.6%201.1v-1.1h4.5l.6%201.1v-1.1h4.4V20h-4.4l-.8-1.4V20h-5.5l-.6-1.5h-1.4l-.8%201.5h-2.9c-1.1%200-2-.3-2.5-.6v.6h-6.8v-2.1c0-.3-.1-.3-.2-.3H35V20H21.9v-1.1l-.5%201.1h-2.8l-.4-1.1zm46.9%2011c-.7%201.3-2%201.5-3.3%201.5h-4.4v-.6c-.5.4-1.3.6-2.1.6h-13v-2.1c0-.3%200-.3-.3-.3h-.2v2.4h-4.2v-2.5c-.7.3-1.5.3-2.2.3h-.5v2.2h-5.1L28.5%2030l-1.3%201.4h-8.3v-9.2h8.5l1.2%201.4%201.3-1.4h5.7c.7%200%201.7.1%202.2.5v-.5h5.1c.5%200%201.5.1%202.1.5v-.5h7.7v.5c.4-.4%201.2-.5%201.9-.5h4.3v.5c.4-.3%201-.5%201.8-.5h4.5v3.3h-.9c1.3.9%201.6%203%20.8%204.4-.4.7.3-.5%200%200zM39.4%2012.1H41v6.6h-1.6zM55.6%2026h-.8c-.5%200-1.5-.5-.7-1%20.2-.1.3-.1.6-.1h2.9v-1.4h-3.2c-1.5%200-2.6%201.2-2.1%202.7.4%201.3%202.1%201.2%203.2%201.3%201%200%20.9%201.2-.1%201.2h-3.1v1.4h3.1c1.1%200%202.2-.4%202.4-1.6.4-1.7-.7-2.4-2.2-2.5z%22%20fill%3D%22#ffffff%22%3E%3C%2Fpath%3E%3C%2Fsvg%3E");\n    background-repeat: no-repeat;\n  }\n  .site-icon-mc {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2271%22%20height%3D%2243%22%20viewBox%3D%220%200%2071%2043%22%3E%3Cpath%20d%3D%22M36.2%2022.5c.2-.9.3-2.2-.8-2.7-1.4-.7-3%20.1-3.6%201.5-.3.7-.3%201.5-.1%202.2.6%201.4%202.7%201.8%204%201.1l.2-1.1c-.9.2-3.1.5-2.9-1h3.2zm-1.8-1.9c.6%200%20.7.4.6.9h-1.8c.2-.6.6-.9%201.2-.9zm-3.8%202.3l.4-2.1h.7l.3-1.1h-.8l.3-1.4h-1.4l-1%204.9c-.3%201.3%201%202.1%202.2%201.4l.1-1c-.8%200-.8-.3-.8-.7zm8-2.7l-.1-.5h-1.2l-1%205.1h1.4l.4-2.3c.4-1.5%201.1-1.4%201.4-1.4l.4-1.4c-.7%200-1.1.2-1.3.5zm-10.2%201.9c-.2-.1-2.1-1-1-1.3.3%200%201%200%201.4.1l.3-1c-1.3-.5-3.7-.2-3.6%201.6.1%201.1%201.3.9%201.8%201.6.7.9-1.9.6-2.2.5l-.1%201c.8.2%202%20.3%202.8%200%201-.3%201.7-1.9.6-2.5-.3-.2.3.1%200%200zm14.5-2.4c.5%200%20.7%200%201.1.3l.9-.9c-1.7-1.9-4.5-.3-5.1%201.8-.7%202.7%201.4%204.8%204%203.6l-.4-1.3c-2.8%201-2.8-3.5-.5-3.5zM23%2019.6c-.4%200-.9%200-1.8.2l-.2%201.1c.6-.1%202.9-.7%202.6.6-1.1.1-2.2-.1-2.9%201-1.1%201.6.8%203.2%202.3%202.1l.2.3h1.3L25%2022c.2-.7.4-2.4-2-2.4zm.3%203.4c-.1.5-1.3%201-1.5.3-.3-.9%201.1-1%201.6-.9l-.1.6zm27.9-2.9l-.2-.5h-1.2l-1%205.2h1.4l.4-2.4c.4-1.5%201.1-1.5%201.4-1.5l.4-1.4c-.6.1-1%20.4-1.2.6zM69%200H2C.9%200%200%20.9%200%202v39c0%201.1.9%202%202%202h67c1.1%200%202-.9%202-2V2c0-1.1-.9-2-2-2zM45.1%2035.5c-3.4%200-6.4-1.2-8.8-3.1%201.8-1.8%203.2-4%204-6.5h-1.2c-.7%202.2-2%204.2-3.7%205.8-1.7-1.6-2.9-3.6-3.6-5.8h-1.2c.8%202.5%202.1%204.7%204%206.5-4%203.2-9.5%204.1-14.3%202-4-1.7-7.1-5.4-8.1-9.6C10%2015.4%2018%206.4%2027.5%207.6c2.6.3%205.1%201.4%207.1%203-1.8%201.8-3.2%204-4%206.5h1.2c.7-2.2%202-4.2%203.6-5.8%201.7%201.6%202.9%203.6%203.7%205.8h1.2c-.8-2.5-2.2-4.7-4-6.5%207.6-6.2%2019.4-2.5%2022.2%206.8%202.7%208.9-4.1%2018.1-13.4%2018.1zm7.2-13.7c-.5%201.6.5%203%201.5%203%20.5%200%20.9-.2%201.2-.4l.2.4h1.5l1.2-6.6h-1.5l-.4%202c-.7-.5-1.2-.8-2-.6-1%20.3-1.5%201.3-1.7%202.2-.5%201.6.7-2.4%200%200zm3.1-.8c.2.2.2.8.1%201.3-.2.4-.4.9-.9%201-.6.1-1-.4-1-1%20.1-.6%201.1-2.2%201.8-1.3.2.3-.4-.5%200%200zm-36.9-2.5l-1.7%203.8-.3-3.8h-2.2l-1.2%206.3h1.4l.8-4.5.4%204.5h1.4l1.9-4.4-.8%204.4h1.5l1.1-6.3zm28.4%201.1c-.4%200-.9%200-1.8.2l-.2%201.1c.6-.1%202.9-.7%202.6.6-1.1.1-2.2-.1-3%201-1.1%201.6.8%203.2%202.3%202.1l.2.3h1.3l.6-2.8c.1-.8.4-2.5-2-2.5zm.2%203.4c-.1.5-1.3%201-1.5.3-.3-.9%201.1-1%201.6-.9l-.1.6z%22%20fill%3D%22#ffffff%22%3E%3C%2Fpath%3E%3C%2Fsvg%3E");\n    background-repeat: no-repeat;\n  }\n  .site-icon-discover {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2271%22%20height%3D%2243%22%20viewBox%3D%220%200%2071%2043%22%3E%3Cpath%20d%3D%22M6.5%2012.1v6.5s4.2.5%204.2-3.2v-.1c-.1-3.6-4.2-3.2-4.2-3.2zM69%200H2C.9%200%200%20.9%200%202v39c0%201.1.9%202%202%202h67c1.1%200%202-.9%202-2V2c0-1.1-.9-2-2-2zm-9.8%2010.4h3.2c1.7%200%203.1%201.2%203.1%202.9%200%201.3-.8%202.5-2.1%202.9l3%204.2H64l-2.7-4h-.2v4h-1.9v-10zm-6.8%200H58v1.8h-3.5v2.2h3.4v1.8h-3.4v2.5H58v1.8h-5.5V10.4zm-8.2%200l2.7%206.5%202.6-6.5h2.1l-4.2%2010.2h-1.2L42%2010.4h2.2zm-31.5%205c0%203.2-2.9%204.9-4.8%204.9H4.6v-9.9h3.3c1.9%200%204.8%201.8%204.8%205zm2.8%204.9h-2v-10h2v10zm4.2.3c-2.5%200-3.6-1.9-3.6-1.9l1.2-1.2c.7%201%203.1%202.2%203.5.4.5-1.8-1.8-1.9-2.9-2.4-1.7-.9-1.8-3.4-.2-4.6%201.5-1.1%203.5-.6%204.8.6l-1.1%201.3c-.7-.8-3-1.2-2.9.4%200%201%202%201.3%202.7%201.7%203.2%201.3%201.6%205.7-1.5%205.7zm11.6-.6c-3.4%201.8-7.7-.8-7.7-4.6%200-3.9%204.2-6.4%207.7-4.6v2.4c-2.1-2.1-5.8-.8-5.8%202.3%200%203%203.7%204.4%205.8%202.3V20zm6-9.8c2.8%200%205.2%202.3%205.2%205.2%200%202.8-2.3%205.2-5.2%205.2-2.8%200-5.2-2.3-5.2-5.2%200-2.8%202.4-5.2%205.2-5.2zM68%2040H32.8C45%2036.8%2057.2%2032.4%2068%2025.7V40zm-4.4-26.5v-.1c0-.6-.5-1.4-1.3-1.4h-1.1v2.9h1.1c.8%200%201.3-.7%201.3-1.4z%22%20fill%3D%22#ffffff%22%3E%3C%2Fpath%3E%3C%2Fsvg%3E");\n    background-repeat: no-repeat;\n  }\n  .site-icon-jcb {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2271%22%20height%3D%2243%22%20viewBox%3D%220%200%2071%2043%22%3E%3Cpath%20d%3D%22M69%200H2C-.1%200%200%202.1%200%203.5v37.4C0%2042.6%201.4%2043%202.7%2043h66c1.4%200%202.4-.8%202.4-2.3V3c-.1-1.4-.3-3-2.1-3zM28.3%2030.8c0%202.4-1.9%204.4-4.3%204.6-2.1.3-4.3%200-6.5%200-.9%200-.8.1-.8-.8v-9c2.8.7%209.5%201.7%2010.4-2.2.3-1.2.1-2.6.1-3.8v-2.1c0-.4.1-.5-.3-.5h-3.7v3.4c0%201.5.3%203.5-1.5%204.2-.8.3-1.7.2-2.5%200-.9-.2-1.8-.5-2.6-.9-.1-.1%200-.5%200-.6v-6.4c0-2.1-.4-4.7.5-6.7.5-1%201.3-1.6%202.2-2.1%201.4-.6%208.7-.5%208.7-.3v15.1c.3%202.7.3%205.4.3%208.1zm13%20.1c0%202.4-2%204.4-4.4%204.6-.9.1-2%200-2.9%200h-4c-.4%200-.2-1.3-.2-1.6v-9.1c1.4%201.2%203.4%201.5%205.2%201.5%201.1%200%202.1%200%203.1-.2l1.2-.2c.1%200%20.6-.1.7-.2.2-.3%200-1.5%200-1.9v-.1c-1.4.7-3.4%201.4-5%201-3-.4-3.4-4.8-.8-6.1%201.9-.7%204.1-.1%205.8.7v-1.8c0-.3-.3-.3-.5-.3-.4-.1-.8-.2-1.2-.2-.9-.1-1.9-.2-2.8-.2-1.8%200-3.7.2-5.3%201.2l-.5.3v-3.5c0-1.2-.2-2.6.1-3.8.3-1.3%201.2-2.2%202.2-2.9.7-.4%201.4-.6%202.2-.6%202.3-.1%204.6%200%206.8%200%20.3%200%20.3%2021.2.3%2023.4zm12.9%200c0%202.5-2.1%204.5-4.5%204.6-2.3.1-4.6%200-6.9%200-.2%200-.1-6.1-.1-6.7V26h7c.4%200%20.7%200%201.1-.1%201.3-.3%202.5-1.4%202-2.9-.3-.9-1.3-1.4-2.3-1.6h-.3l1.1-.4c.7-.4%201.1-1.2%201-2-.1-1.2-1.2-1.8-2.3-1.9-1.8-.2-3.8%200-5.6%200h-1.6c-.3%200-.2-.3-.2-.6v-4.1c0-2.5%201.8-4.6%204.3-4.8%202.2-.2%204.4%200%206.6%200%20.6%200%20.7-.1.7.5v22.8zm-6.5-8.8c1.5.2%201.6%202.5%200%202.5h-2.3c-.1%200-.4.1-.4-.1v-2.4c0-.2.4-.1.5-.1.7.1%201.4.1%202.2.1zm-2.8-1.4v-2c0-.6.9-.3%201.3-.3.5%200%201.5-.2%201.9.3.7.5.4%201.7-.4%202-.8.2-1.9%200-2.8%200z%22%20fill%3D%22#ffffff%22%3E%3C%2Fpath%3E%3C%2Fsvg%3E");\n    background-repeat: no-repeat;\n  }\n  .site-icon-calendar {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2223%22%20height%3D%2223%22%20viewBox%3D%220%200%2023%2023%22%3E%3Cpath%20d%3D%22M19.3%202.8h-1.1v2.5h-4V2.8H8.8v2.5h-4V2.8H3.7c-1.4%200-2.5%201.1-2.5%202.5v14.9c0%201.4%201.1%202.5%202.5%202.5h15.6c1.4%200%202.5-1.1%202.5-2.5V5.3c0-1.4-1.1-2.5-2.5-2.5zm0%2017.4H3.7v-9.9h15.6v9.9zM7.7.3H5.9v4.2h1.7V.3zm9.4%200h-1.7v4.2h1.7V.3z%22%20fill%3D%22#ffffff%22%2F%3E%3Cg%20fill%3D%22#ffffff%22%3E%3Cpath%20d%3D%22M5.1%2011.7h1.4v1.4H5.1zM5.1%2014.5h1.4v1.4H5.1zM5.1%2017.4h1.4v1.4H5.1zM8%2011.7h1.4v1.4H8zM8%2014.5h1.4v1.4H8zM8%2017.4h1.4v1.4H8zM10.8%2011.7h1.4v1.4h-1.4zM10.8%2014.5h1.4v1.4h-1.4zM10.8%2017.4h1.4v1.4h-1.4zM13.6%2011.7H15v1.4h-1.4zM13.6%2014.5H15v1.4h-1.4zM13.6%2017.4H15v1.4h-1.4zM16.5%2011.7h1.4v1.4h-1.4zM16.5%2014.5h1.4v1.4h-1.4zM16.5%2017.4h1.4v1.4h-1.4z%22%2F%3E%3C%2Fg%3E%3C%2Fsvg%3E");\n    background-repeat: no-repeat;\n  }\n  .site-icon-credit {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2271%22%20height%3D%2243%22%20viewBox%3D%220%200%2071%2043%22%3E%3Cstyle%20type%3D%22text%2Fcss%22%3Ecircle%2C%20ellipse%2C%20line%2C%20path%2C%20polygon%2C%20polyline%2C%20rect%2C%20text%20%7B%20fill%3A%20#ffffff%20!important%3B%20%7D%3C%2Fstyle%3E%3Cpath%20d%3D%22M71%202c0-1.1-.9-2-2-2h-67c-1.1%200-2%20.9-2%202v8.6h71v-8.6zm-71%2018.4v20.6c0%201.1.9%202%202%202h67c1.1%200%202-.9%202-2v-20.6h-71zm17.7%2014.9h-9.6v-3.6h9.6v3.6zm17.3%200h-13.9v-3.6h13.9v3.6z%22%20fill%3D%22%23fff%22%2F%3E%3C%2Fsvg%3E");\n    background-repeat: no-repeat;\n  }\n  .site-icon-search {\n    background-image: url(\'data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2223%22%20height%3D%2223%22%20viewBox%3D%220%200%2023%2023%22%3E%3Cpath%20d%3D%22M21.8%2019.5L17.4%2015c-.6.9-1.5%201.7-2.4%202.4l4.5%204.4c1.4%201.6%203.8-.8%202.3-2.3z%22%20fill%3D%22%23bebebb%22%3E%3C%2Fpath%3E%3Cpath%20d%3D%22M15%2017.4c4.4-3%205.2-9.4%201.8-13.4C13.8.5%208.6-.3%204.7%202.3.9%204.7-.4%209.9%201.7%2014c2.2%204.4%207.7%206.2%2012.1%204.1.4-.2.8-.5%201.2-.7zm-5.2-1.7C6.7%2015.7%204%2013%204%209.8%204%206.7%206.7%204%209.8%204c3.2%200%205.8%202.7%205.8%205.8.1%203.2-2.6%205.9-5.8%205.9z%22%20fill%3D%22%23bebebb%22%3E%3C%2Fpath%3E%3Cpath%20d%3D%22M9.8%202.2c4.2%200%207.7%203.4%207.7%207.7%200%201.5-.5%203.1-1.4%204.3-.5.7-1%201.3-1.7%201.8l-.3.2c-1.2.9-2.8%201.3-4.3%201.3-4.2%200-7.7-3.4-7.7-7.7.1-4.2%203.5-7.6%207.7-7.6m0-1.5C4.7.7.6%204.8.6%209.9s4.1%209.2%209.2%209.2c1.8%200%203.7-.6%205.2-1.6.9-.6%201.7-1.4%202.3-2.4%201.1-1.5%201.6-3.3%201.6-5.2.1-5.1-4-9.2-9.1-9.2z%22%20fill%3D%22%23bebebb%22%3E%3C%2Fpath%3E%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-is-searching {\n     background-position: center center !important;\n     background-image: url(\'data:image/svg+xml;charset=utf-8,%3Csvg%20version%3D%221.1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%20%20viewBox%3D%220%200%2040%2040%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cpath%20opacity%3D%220.2%22%20fill%3D%22%23323232%22%20d%3D%22M20.201%2C5.169c-8.254%2C0-14.946%2C6.692-14.946%2C14.946c0%2C8.255%2C6.692%2C14.946%2C14.946%2C14.946%0D%0A%20%20s14.946-6.691%2C14.946-14.946C35.146%2C11.861%2C28.455%2C5.169%2C20.201%2C5.169z%20M20.201%2C31.749c-6.425%2C0-11.634-5.208-11.634-11.634%0D%0A%20%20c0-6.425%2C5.209-11.634%2C11.634-11.634c6.425%2C0%2C11.633%2C5.209%2C11.633%2C11.634C31.834%2C26.541%2C26.626%2C31.749%2C20.201%2C31.749z%22%2F%3E%0D%0A%3Cpath%20fill%3D%22%23323232%22%20d%3D%22M26.013%2C10.047l1.654-2.866c-2.198-1.272-4.743-2.012-7.466-2.012h0v3.312h0%0D%0A%20%20C22.32%2C8.481%2C24.301%2C9.057%2C26.013%2C10.047z%22%3E%0D%0A%20%20%3CanimateTransform%20attributeType%3D%22xml%22%0D%0A%20%20%20%20attributeName%3D%22transform%22%0D%0A%20%20%20%20type%3D%22rotate%22%0D%0A%20%20%20%20from%3D%220%2020%2020%22%0D%0A%20%20%20%20to%3D%22360%2020%2020%22%0D%0A%20%20%20%20dur%3D%220.5s%22%0D%0A%20%20%20%20repeatCount%3D%22indefinite%22%2F%3E%0D%0A%20%20%3C%2Fpath%3E%0D%0A%3C%2Fsvg%3E\');\n     background-repeat: no-repeat;\n     background-size: contain;\n     z-index: 999;\n     width: 80px;\n     height: 80px;\n     margin: 0 auto;\n  }\n  .site-icon-report {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3Csvg%20viewBox%3D%220%200%201792%202031%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%3E%3Cpath%20d%3D%22M1122.987%20238.933L1075.2%200H0v2030.933h238.933v-836.266h669.014l47.786%20238.933H1792V238.933z%22%20fill%3D%22%23bebebb%22%20fill-rule%3D%22evenodd%22%2F%3E%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n    background-size: contain;\n  }\n  .site-icon-block {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3Csvg%20viewBox%3D%220%200%201792%201792%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%3E%3Cpath%20d%3D%22M896%201612.8c-165.76%200-318.08-56.448-439.04-151.424L1461.376%20456.96A708.02%20708.02%200%200%201%201612.8%20896c0%20396.032-320.768%20716.8-716.8%20716.8M179.2%20896c0-396.032%20320.768-716.8%20716.8-716.8%20165.76%200%20318.08%2056.448%20439.04%20151.424L330.624%201335.04A708.02%20708.02%200%200%201%20179.2%20896M896%200C401.408%200%200%20401.408%200%20896s401.408%20896%20896%20896%20896-401.408%20896-896S1390.592%200%20896%200%22%20fill%3D%22%23ffffff%22%20fill-rule%3D%22evenodd%22%20fill-opacity%3D%22.6%22%2F%3E%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n    background-size: contain;\n  }\n</style>\n\n\n  <style>\n\n    /*\r\n * This file is for use on the Vimeo VHX platform.\r\n *\r\n * 1. prototype changes in browser dev tools using Vimeo VHX demo site\r\n * 2. edit this file locally\r\n * 3. git commit and push to the remote repo\r\n * 4. share for code review\r\n * 5. paste into "CUSTOM STYLES" field in the Vimeo VHX theme demo admin\r\n * 6. share for approval by Criterion team\r\n * 7. once approved and code reviewed, paste into "CUSTOM STYLES" field in the Vimeo VHX theme production admin\r\n *\r\n * Note for Vimeo engineers: please share any updates to this file to the Criterion.com team.\r\n */\r\n\r\n @import url(\'https://fonts.googleapis.com/css?family=Montserrat:400,500,600,700,900|Spectral:400,500,600,700\');\r\n\r\n p, div, span, a, h1, h2, h3, h4, h5, h6, .site-font-primary-family, .site-font-secondary-family {\r\n     font-family: \'Montserrat\', sans-serif !important;\r\n }\r\n\r\n div {\r\n     font-family: \'Montserrat\', sans-serif !important;\r\n }\r\n\r\n .browse-row h1 a {\r\n     font-family: \'Montserrat\', sans-serif !important;\r\n     font-weight: 700;\r\n     text-transform: uppercase;\r\n }\r\n\r\n .head-1, .head-2 {\r\n     font-weight: 700;\r\n }\r\n\r\n .browse-row h1.horizontal-row-header a {\r\n     font-size: 1rem !important;\r\n }\r\n\r\n .browse-link {\r\n     font-family: \'Montserrat\', sans-serif !important;\r\n }\r\n\r\n /*\r\n * Nav\r\n */\r\n\r\n .navigation-inner a {\r\n    opacity: 1;\r\n    font-family: \'Montserrat\', sans-serif !important;\r\n    -webkit-transition: all 0.3s ease;\r\n    -moz-transition: all 0.3s ease;\r\n    -ms-transition: all 0.3s ease;\r\n    -o-transition: all 0.3s ease;\r\n    transition: all 0.3s ease;\r\n}\r\n\r\n .navigation-inner.is-ready a {\r\n    opacity: 1;\r\n}\r\n\r\n  /*\r\n  * Other\r\n  */\r\n\r\n h1.gigantic.uppercase { text-transform: none !important; }\r\n .percentage-bar-container { display: none !important; }\r\n\r\n .slide-content {\r\n     bottom: 40px;\r\n }\r\n\r\n .slide-title {\r\n     font-size: 28px;\r\n     font-weight: 600;\r\n }\r\n\r\n .browse-item-title .subtext-container > div.site-font-primary-color.truncate {\r\n     white-space: normal;\r\n }\r\n\r\n .collection-description .read-more-wrap {\r\n     max-height: 120px !important;\r\n }\r\n\r\n .site-header nav a {\r\n     font-weight: 600;\r\n     letter-spacing: 0;\r\n     padding-left: 25px;\r\n }\r\n\r\n div.slide-subtitle {\r\n     display: none;\r\n }\r\n\r\n .slide-content {\r\n     bottom: 40px;\r\n }\r\n\r\n .collection-title {\r\n     font-weight: 600;\r\n }\r\n\r\n\r\n #watch-info h5 {\r\n     display: none;\r\n }\r\n\r\n .head.primary strong {\r\n     font-weight: 600;\r\n }\r\n\r\n .item-type-movie .play-outer-rectangle {\r\n     display: none;\r\n }\r\n\r\n /*\r\n  * Borders\r\n  */\r\n\r\n .border-bottom,\r\n .border-bottom-light {\r\n     border-bottom: none !important;\r\n }\r\n\r\n .border-top,\r\n .border-top-light {\r\n     border-top: none !important;\r\n }\r\n\r\n .horizontal-row {\r\n     border-top: none !important;\r\n     border-bottom: none !important;\r\n }\r\n\r\n /*\r\n  * Buttons\r\n  */\r\n\r\n .button {\r\n     border-radius: 0px;\r\n     border: none;\r\n     font-weight: 600;\r\n }\r\n\r\n .slide-button,\r\n .slide-button--alt {\r\n     border-radius: 0px !important;\r\n }\r\n\r\n .btn-site-primary,\r\n .btn-gray:link,\r\n .btn-dropdown-gray:link,\r\n .btn-gray:active,\r\n .btn-dropdown-gray:active,\r\n .btn-gray:visited,\r\n .btn-dropdown-gray:visited {\r\n     border: 1px solid #fff !important;\r\n     background-color: #fff !important;\r\n     color: #252525;\r\n     border-radius: 0px;\r\n     text-transform: uppercase;\r\n     line-height: 46px;\r\n     font-size: 13px;\r\n     transition: all 300ms ease-out;\r\n }\r\n .site-settings-billing-panel .btn-gray:link {\r\n   color: #252525 !important;\r\n   line-height: 33px;\r\n }\r\n .btn-site-primary,\r\n .btn-site-secondary {\r\n     line-height: 33px;\r\n }\r\n\r\n .btn-site-secondary,\r\n .btn-transparent,\r\n .btn-dropdown-transparent {\r\n     border: 1px solid #313131 !important;\r\n     background-color: #313131 !important;\r\n     color: #fff !important;\r\n     transition: all 300ms ease-out;\r\n     font-size: 13px;\r\n }\r\n\r\n .btn-gray:hover,\r\n .btn-dropdown-gray:hover,\r\n .btn-site-primary:hover,\r\n .btn-dropdown-transparent:hover,\r\n .btn-transparent:hover,\r\n .btn-site-secondary:hover {\r\n     border: 1px solid #CDCDCB !important;\r\n     background-color: #CDCDCB !important;\r\n }\r\n\r\n .icon-play-white {\r\n     background-image: url("data:image/svg+xml;charset=utf-8,%3Csvg xmlns=\'http://www.w3.org/2000/svg\' width=\'23\' height=\'23\' viewBox=\'0 0 23 23\'%3E%3Ctitle%3Eicons-ui-set copy%3C/title%3E%3Cpath d=\'M20.49 10.52L4.13.39C2.94-.4 2 .18 2 1.67v19.66c0 1.48 1 2.06 2.17 1.28l16.32-10.13a1.15 1.15 0 0 0 0-1.96z\' fill=\'%23252525\'/%3E%3C/svg%3E");\r\n }\r\n\r\n .btn-black:link,\r\n .btn-dropdown-black:link,\r\n .btn-black:active,\r\n .btn-dropdown-black:active,\r\n .btn-black:visited,\r\n .btn-dropdown-black:visited {\r\n     border: 1px solid #B9BCC6;\r\n     background-color: #B9BCC6;\r\n     color: #000;\r\n     border-radius: 0px;\r\n     text-transform: uppercase;\r\n     line-height: 46px;\r\n     font-size: 13px;\r\n }\r\n\r\n .btn-transparent,\r\n .btn-dropdown-transparent,\r\n .btn-transparent:link,\r\n .btn-dropdown-transparent:link,\r\n .btn-transparent:active,\r\n .btn-dropdown-transparent:active,\r\n .btn-transparent:visited,\r\n .btn-dropdown-transparent:visited {\r\n     border: none;\r\n     border-radius: 0px;\r\n     text-transform: uppercase;\r\n     line-height: 28px;\r\n     font-size: 13px;\r\n     padding: 10px 16px;\r\n     background-position: 16px 18px !important;\r\n }\r\n\r\n select.btn-dropdown-transparent {\r\n     padding: 0 20px 0 40px;\r\n }\r\n\r\n .coin {\r\n     border-radius: 4px;\r\n }\r\n\r\n /*\r\n  * Search link + collection card\r\n  */\r\n\r\n .browse-item-subtext {\r\n     display: none !important;\r\n }\r\n\r\n /*\r\n  * Search results\r\n  */\r\n\r\n .item-type-movie.js-collection-item .duration-container {\r\n     display: none !important;\r\n }\r\n\r\n /*\r\n  * Carousel\r\n  */\r\n\r\n .slide-content {\r\n     transform: translate3d(0,0,0);\r\n     top: auto;\r\n     bottom: 10%;\r\n     padding-left: 57px;\r\n }\r\n\r\n .slide-image-container:after,\r\n .slide-image-container:before {\r\n     display: none !important;\r\n }\r\n\r\n @media screen and (max-width: 39.9375em) {\r\n     .slide-content {\r\n         padding-left: 40px;\r\n         bottom: 13%;\r\n     }\r\n     .slide-bg--mobile .slide-bg-img {\r\n         float: left;\r\n         width: 100%;\r\n         height: auto;\r\n     }\r\n\r\n }\r\n\r\n /*\r\n  * Browse\r\n  */\r\n\r\n .browse-item-card.item-type-movie .duration-container {\r\n   display: none !important;\r\n }\r\n\r\n /*\r\n  * Collection Pages\r\n  */\r\n\r\n .vertical-bottom {\r\n     position: absolute;\r\n     bottom: 0;\r\n }\r\n\r\n .cover-art.columns {\r\n     padding-right: 0px !important;\r\n }\r\n\r\n select.js-sort-items,\r\n h2.collection-stats,\r\n h2.content-label,\r\n .extras-header {\r\n     display: none !important;\r\n }\r\n\r\n .browse-item-title .browse-image-container:before {\r\n     width: 102%;\r\n     height: 102%;\r\n }\r\n\r\n .browse-image-container:before {\r\n     background-image: none !important;\r\n }\r\n\r\n .media-identifier.media-episode {\r\n     display: none;\r\n }\r\n\r\n .read-more-trigger {\r\n     font-weight: bold;\r\n }\r\n\r\nsection.site-settings-purchases-panel .subscription-button {\r\n    background-color: #222 !important\r\n}\r\n\r\n.platforms-modal--container .platform-modal-devices-list-tv li:last-child {\r\n    display: none;\r\n}\r\n\r\n .collection-actions .margin-bottom-small {\r\n    margin-bottom: 0.9375rem !important;\r\n }\r\n .collection-actions .margin-right-small {\r\n    margin-right: 0.9375rem !important;\r\n }\r\n .collection-actions .btn-site-primary {\r\n    line-height: 46px;\r\n }\r\n\r\n/* Fix for button issue on pause / cancellation modal */\r\nbody.settings.is-modal-active .btn-site-secondary,\r\nbody.settings.is-modal-active .btn-site-primary {\r\n    line-height: 38px !important;\r\n    color: #252525 !important;\r\n}\r\n\r\n.site-header .user-dropdown {\r\n    width: 190px;\r\n}\r\n\r\n.tooltip.background-white span.media-count, .tooltip.background-white span.media-identifier, .tooltip.background-white span.media-seperator { display: none !important; }\r\n\r\n.tooltip.background-white .tooltip-item-title { margin-bottom: 15px; }\r\n\r\ndiv[id^="collection-tooltip"] .tooltip-item-title { margin-bottom: 0px !important; }\n\n\n    html.wf-loading { opacity: 0; }\n    html.wf-active, html.is-loaded { opacity: 1 !important; }\n  </style>\n\n  <script>\n  var Tracking = undefined;\n  window.Page = undefined;\n  var Segment_ID = undefined;\n  window._current_user = undefined;\n  window._current_site = undefined;\n</script>\n\n\n  <script>\n    window._vhx = {\n      environment: \'production\',\n      site: { id: \'59054\', domain: \'www.criterionchannel.com\', host: \'criterionchannel.vhx.tv\', previewUrl: \'https://www.criterionchannel.com/login\' },\n      geo:  { country: \'us\' },\n      product: {id: \'\' }\n    };\n\n    window._current_user = {"id":null,"email":null,"name":null,"city":null,"state":null,"country":null,"external_user_id":null};\n\n    window._current_site = {"id":59054,"facebook_pixel_codes":[{"pixel_code":"980569266079632","product_id":null}],"twitter_pixel_codes":[],"adwords_pixel_codes":[],"key":"criterionchannelchartersu","title":"The Criterion Channel","subdomain":"criterionchannel","made_for_kids":false,"made_for_kids_tracking_blocked":false,"algolia_search_enabled":true,"passwords_disabled":false,"segment_heartbeat_frequency":0,"segment_includes_user_id":false,"sso_enabled":false};\n  </script>\n\n<script>\n  window.Tracking = {\n    EVENTS:     {"EVENT_SALE_INITIATED":"sale_initiated","EVENT_SALE_COMPLETE":"sale_complete","EVENT_SALE_COMPLETE_GIFT":"sale_conversion-gift","EVENT_SALE_ERROR":"sale_error","EVENT_SALE_COMPLETE_SAVE_CARD":"sale_complete_save_card","EVENT_SALE_MISSING_EMAIL":"sale_missing_email","EVENT_SALE_UNCONFIRMED_EMAIL":"sale_unconfirmed_email","EVENT_SALE_MISSING_CC":"sale_missing_cc","EVENT_SALE_MISSING_CVC":"sale_missing_cvc","EVENT_SALE_MISSING_EXP":"sale_missing_exp","EVENT_SALE_MISSING_ZIP":"sale_missing_zip","EVENT_SALE_INVALID_CC":"sale_invalid_cc","EVENT_SALE_INVALID_CVC":"sale_invalid_cvc","EVENT_SALE_INVALID_EXP":"sale_invalid_exp","EVENT_SALE_INVALID_ZIP":"sale_invalid_zip","EVENT_SALE_INVALID_COUPON":"sale_invalid_coupon","EVENT_SALE_INACTIVE_COUPON":"sale_inactive_coupon","EVENT_SALE_GEOBLOCKED_COUPON":"sale_geoblocked_coupon","EVENT_SALE_EXPIRED_CC":"sale_expired_cc","EVENT_SALE_PLAY_TRAILER":"sale_play_trailer","EVENT_SALE_PROCESSING_ERROR":"sale_processing_error","EVENT_SALE_UNKNOWN_ERROR":"sale_unknown_error","EVENT_SALE_INCOMPLETE_FORM":"sale_incomplete_form","EVENT_SALE_SHARE_FACEBOOK":"sale_share_facebook","EVENT_SALE_SHARE_TWITTER":"sale_share_twitter","EVENT_SALE_SHARE_EMAIL":"sale_share_email","EVENT_SALE_START_WATCHING":"sale_start_watching","EVENT_SALE_SAVE_CARD":"sale_save_card","EVENT_SALE_SHOW_PAYMENT_TAB":"sale_show_payment_tab","EVENT_SALE_SHOW_DETAILS_TAB":"sale_show_details_tab","EVENT_ADMIN_SIGNUP":"admin_signup","EVENT_ADMIN_ADD_SITE":"admin_add_site","EVENT_ADMIN_PUBLISH_THEME":"admin_publish_theme","EVENT_ADMIN_CREATE_VIDEO":"admin_create_video","EVENT_ADMIN_UPLOAD_VIDEO":"admin_upload_video","EVENT_ADMIN_UPLOAD_VIDEO_NATIVE":"admin_upload_video_native","EVENT_ADMIN_UPLOAD_VIDEO_DROPBOX":"admin_upload_video_dropbox","EVENT_ADMIN_TRANSCODE_VIDEO":"admin_transcode_video","EVENT_ADMIN_CREATE_EXTRA":"admin_create_extra","EVENT_ADMIN_UPLOAD_EXTRA":"admin_upload_extra","EVENT_ADMIN_UPLOAD_EXTRA_NATIVE":"admin_upload_video_native","EVENT_ADMIN_CONNECT_PAYOUT":"admin_connect_payout","EVENT_ADMIN_CREATE_SCREENER":"admin_create_screener","EVENT_ADMIN_CREATE_COUPON":"admin_create_coupon","EVENT_ADMIN_CREATE_UPDATE":"admin_create_update","EVENT_ADMIN_PUBLISH_UPDATE":"admin_publish_update","EVENT_ADMIN_CREATE_PARTNER":"admin_create_partner","EVENT_ADMIN_UPLOADED_FOLLOWERS_CS":"admin_uploaded_followers_csv","EVENT_ADMIN_ADDED_FOLLOWERS":"admin_added_followers","EVENT_ADMIN_CREATE_SUBSCRIPTION":"admin_create_subscription","EVENT_ADMIN_LINK_HELP":"admin_link_help","EVENT_ADMIN_LINK_FAQ":"admin_link_faq","EVENT_ADMIN_LINK_GETTING_STARTED":"admin_link_getting_started","EVENT_ADMIN_LINK_HOWTO":"admin_link_howto","EVENT_ADMIN_LINK_BEST_PRACTICES":"admin_link_best_practices","EVENT_ADMIN_LINK_DEV_DOCS":"admin_link_dev_docs","EVENT_FORUM_CREATE_THREAD":"forum_create_thread","EVENT_FORUM_CREATE_POST":"forum_create_post","EVENT_UPDATE_COMMENT":"comment_update","EVENT_CONCURRENT_LIMIT":"concurrent_limit","EVENT_VIDEO_COMMENT":"video_comment","EVENT_SITE_HERO_WATCH_TRAILER":"site_hero_watch_trailer","EVENT_SITE_HERO_BUY":"site_hero_buy","EVENT_SITE_FOOTER_BUY":"site_footer_buy","EVENT_SITE_FOOTER_IOS":"site_footer_ios","EVENT_SITE_BUY":"site_buy","EVENT_SITE_UNAVAILABLE_REQUEST":"site_unavailable_request","EVENT_SITE_SEARCH":"site_search","EVENT_SITE_SEARCH_EMPTY":"site_search_empty","EVENT_SITE_SEARCH_VIDEO_CLICK":"search_video","EVENT_SITE_SEARCH_COLLECTION_CLICK":"search_collection","EVENT_SITE_SEARCH_PRODUCT_CLICK":"search_product","EVENT_SITE_SEARCH_ITEM_SELECTION":"site_search_item_selected","EVENT_SITE_VIDEO_BUY":"site_video_buy","EVENT_SITE_VIDEO_WATCH_TRAILER":"site_video_watch_trailer","EVENT_SITE_COLLECTION_BUY":"site_collection_buy","EVENT_SITE_PRODUCT_PREORDER":"site_product_preorder","EVENT_SITE_PRODUCT_BUY":"site_product_buy","EVENT_SITE_PRODUCT_RENT":"site_product_rent","EVENT_SITE_CAROUSEL_NEXT":"site_carousel_next","EVENT_SITE_CAROUSEL_PREV":"site_carousel_prev","EVENT_SITE_CAROUSEL_COIN":"site_carousel_coin","EVENT_SITE_CAROUSEL_WATCH_NOW":"site_carousel_watch_now","EVENT_SITE_CAROUSEL_TRAILER":"site_carousel_trailer","EVENT_SITE_HOME":"site_home","EVENT_SITE_BROWSE":"site_browse","EVENT_SITE_SETTINGS":"site_settings","EVENT_SITE_FORUMS":"site_forums","EVENT_SITE_SUPPORT":"site_help","EVENT_SITE_CONTACT":"site_contact","EVENT_SITE_CONTACT_HELP":"site_contact_help","EVENT_SITE_VIEW_PRODUCTS":"site_products","EVENT_SITE_VIEW_PRODUCT":"site_product","EVENT_SITE_COLLECTION_CLICK":"site_collection","EVENT_SITE_VIDEO_CLICK":"site_video","EVENT_SITE_EXTRA_CLICK":"site_extras","EVENT_SITE_CATEGORY_CLICK":"site_category","EVENT_SITE_MOVIE_CLICK":"site_movie","EVENT_SITE_PLAYLIST_CLICK":"site_playlist","EVENT_SITE_SERIES_CLICK":"site_series","EVENT_SITE_SEASON_CLICK":"site_season","EVENT_SITE_TERMS":"site_tos","EVENT_SITE_PRIVACY":"site_privacy","EVENT_SITE_COOKIES":"site_cookies","EVENT_SITE_COPYRIGHT":"site_copyright","EVENT_SITE_DATA_PROCESSING":"site_data_processing","EVENT_SITE_WATCHLIST_REMOVE":"site_removewatchlist","EVENT_SITE_WATCHLIST_ADD":"site_addwatchlist","EVENT_SITE_PAGE_VIEW":"site_page_view","EVENT_SITE_CONTEXT_HOMEPAGE":"site_context_homepage","EVENT_SITE_CONTEXT_COLLECTION":"site_context_collection","EVENT_SITE_CONTEXT_CATEGORY":"site_context_category","EVENT_SITE_CONTEXT_MOVIE":"site_context_movie","EVENT_SITE_CONTEXT_PLAYLIST":"site_context_playlist","EVENT_SITE_CONTEXT_SERIES":"site_context_series","EVENT_SITE_CONTEXT_UPNEXT":"site_context_upnext","EVENT_AUTHENTICATION_LOGIN":"authentication_login","EVENT_AUTHENTICATION_ACTIVATION_STARTED":"activationstarted","EVENT_AUTHENTICATION_ACTIVATION_COMPLETE":"activation","EVENT_AUTHENTICATION_SIGNIN_VIEW":"signin","EVENT_AUTHENTICATION_SIGNIN_STARTED":"signinstarted","EVENT_AUTHENTICATION_SIGNIN_COMPLETE":"authentication","EVENT_AUTHENTICATION_SIGNUP_VIEW":"signup","EVENT_AUTHENTICATION_SIGNUP_STARTED":"signupstarted","EVENT_AUTHENTICATION_PURCHASE_STARTED":"purchasestarted","EVENT_AUTHENTICATION_PURCHASE_COMPLETED":"purchasecompleted","EVENT_AUTHENTICATION_CONVERSION":"conversion","EVENT_AUTHENTICATION_REGISTRATION":"register","EVENT_AUTHENTICATION_IDENTIFY_USER":"identify_user","EVENT_SETTINGS_SAVE_PROFILE":"settings_save_profile","EVENT_SETTINGS_SAVE_NOTIFICATIONS":"settings_save_notifications","EVENT_SETTINGS_SAVE_CARD":"settings_save_card","EVENT_DEVICES_DOWNLOAD_IOS":"devices_download_ios","EVENT_DEVICES_DOWNLOAD_APPLETV":"devices_download_appletv","EVENT_DEVICES_DOWNLOAD_ANDROID":"devices_download_android","EVENT_DEVICES_DOWNLOAD_ANDROIDTV":"devices_download_androidtv","EVENT_DEVICES_DOWNLOAD_AMAZON_FIRE":"devices_download_amazon_fire","EVENT_DEVICES_DOWNLOAD_XBOX":"devices_download_xbox","EVENT_DEVICES_DOWNLOAD_ROKU":"devices_download_roku","EVENT_DEVICES_DOWNLOAD_TIZEN":"devices_download_tizen","EVENT_DEVICES_DOWNLOAD_CHROMECAST":"devices_download_chromecast","EVENT_NEWSLETTER_SIGNUP":"event_newsletter_signup","EVENT_METADATA_ADVISORY":"metadata_click_advisory","EVENT_METADATA_CAST":"metadata_click_cast","EVENT_METADATA_CREW":"metadata_click_crew","EVENT_METADATA_GENRES":"metadata_click_genres","EVENT_METADATA_RATING":"metadata_click_rating","EVENT_METADATA_TAGS":"metadata_click_tags","EVENT_SEGMENT_AUTHENTICATION_CONVERSION":"Order Completed","EVENT_SEGMENT_AUTHENTICATION_PURCHASE_STARTED":"Checkout Started","EVENT_SEGMENT_AUTHENTICATION_REGISTRATION":"Registration Completed","EVENT_SEGMENT_AUTHENTICATION_SIGNIN_COMPLETE":"Sign In Complete","EVENT_SEGMENT_AUTHENTICATION_SIGNIN_VIEW":"Sign In Viewed","EVENT_SEGMENT_AUTHENTICATION_SIGNUP_STARTED":"Signed Up","EVENT_SEGMENT_AUTHENTICATION_SIGNUP_VIEW":"Sign Up Viewed","EVENT_SEGMENT_EXTRA_VIEWED":"Extra Content Viewed","EVENT_SEGMENT_SEARCH_EXECUTED":"Search Executed","EVENT_SEGMENT_SEARCH_ITEM_SELECTED":"Search Result Selected","EVENT_SEGMENT_VIDEO_AD_START":"Video Ad Started","EVENT_SEGMENT_VIDEO_AD_CLICKED":"Video Ad Clicked","EVENT_SEGMENT_VIDEO_AD_FINISH":"Video Ad Completed","EVENT_SEGMENT_VIDEO_AD_FAIL":"Video Ad Failed","EVENT_SEGMENT_VIDEO_AD_SKIP":"Video Ad Skipped","EVENT_SEGMENT_VIDEO_COMMENT_ADDED":"Video Comment Added","EVENT_SEGMENT_VIDEO_COMPLETE_PROGRESS_PERCENTILE":"Video Content Completed","EVENT_SEGMENT_VIDEO_ENDED":"Video Content Ended","EVENT_SEGMENT_VIDEO_END_BUFFER":"Video Playback Buffer Completed","EVENT_SEGMENT_VIDEO_HALF_PROGRESS_PERCENTILE":"50% Content Completed","EVENT_SEGMENT_VIDEO_PAUSE":"Video Playback Paused","EVENT_SEGMENT_VIDEO_PLAY":"Video Playback Started","EVENT_SEGMENT_VIDEO_QUARTER_PROGRESS_PERCENTILE":"25% Content Completed","EVENT_SEGMENT_VIDEO_RESUME":"Video Playback Resumed","EVENT_SEGMENT_VIDEO_SEEKED":"Video Playback Seek Completed","EVENT_SEGMENT_VIDEO_SEEKING":"Video Playback Seek Started","EVENT_SEGMENT_VIDEO_START_BUFFER":"Video Playback Buffer Started","EVENT_SEGMENT_VIDEO_TIMEUPDATE":"Video Content Playing","EVENT_SEGMENT_VIDEO_THIRD_PROGRESS_PERCENTILE":"75% Content Completed","EVENT_SEGMENT_VIDEO_WATCHLIST_ADD":"Added to Watch List","EVENT_SEGMENT_VIDEO_WATCHLIST_REMOVE":"Removed from Watch List","EVENT_SEARCH":"search","EVENT_SEARCH_ITEM_SELECTED":"searchitemselected","EVENT_EXTRA_VIEWED":"extraviewed"},\n    PROPERTIES: {"PROPERTY_COLLECTOR_URL":"https://collector.vhx.tv/pixel.gif","PROPERTY_COLLECTION_ID":"collection_id","PROPERTY_DEVICE":"device","PROPERTY_DEVICE_ID":"device_id","PROPERTY_NAME":"name","PROPERTY_PLATFORM":"platform","PROPERTY_PLATFORM_ID":"platform_id","PROPERTY_PLATFORM_VERSION":"platform_version","PROPERTY_PRODUCT_ID":"product_id","PROPERTY_REFERRER":"referrer","PROPERTY_SESSION_ID":"session_id","PROPERTY_SITE_ID":"site_id","PROPERTY_TIMESTAMP":"timestamp","PROPERTY_TYPE":"type","PROPERTY_URL":"url","PROPERTY_USER_AGENT":"user_agent","PROPERTY_USER_EMAIL":"user_email","PROPERTY_USER_ID":"user_id","PROPERTY_VIDEO_ID":"video_id","PROPERTY_VIEW":"view"},\n    DEPRECATED_EVENTS: ["admin_connect_payout","admin_create_coupon","admin_create_screener","admin_link_dev_docs","admin_link_faq","admin_link_help","admin_publish_theme","metadata_click_cast","metadata_click_crew","metadata_click_genres","metadata_click_rating","metadata_click_tags","metadata_click_advisory"],\n    UPDATED_EVENTS: {"search_video":"searchitemselected","search_collection":"searchitemselected","site_search_item_selected":"searchitemselected","site_extras":"extraviewed"},\n    SITE_GANALYTICS_ENABLED: false,\n    SESSION: \'171038f7d4a5b9727879477ef14e5209\'\n  };\n</script>\n  <script>\n     window.Page = {"PROPERTIES":{"VIEW_TYPE":"collection","PRODUCT_ID":39621,"COLLECTION_ID":795549,"COLLECTION_TITLE":"Three by Jafar Panahi"}}\n  </script>\n\n\n\n  <script>\n  var _gaq = _gaq || [];\n\n  (function(i,s,o,g,r,a,m){i[\'GoogleAnalyticsObject\']=r;i[r]=i[r]||function(){\n  (i[r].q=i[r].q||[]).push(arguments)},i[r].l=1*new Date();a=s.createElement(o),\n  m=s.getElementsByTagName(o)[0];a.async=1;a.src=g;m.parentNode.insertBefore(a,m)\n  })(window,document,\'script\',\'//www.google-analytics.com/analytics.js\',\'ga\');\n\n      ga(\'create\', \'UA-19540423-6\', \'auto\');\n    ga(\'require\', \'displayfeatures\');\n\n      ga(\'set\', \'dimension1\', \'criterionchannelchartersu\');\n\n    ga(\'send\', \'pageview\');\n\n\n      (function() {\n        var vhx = document.createElement(\'script\'); vhx.type = \'text/javascript\'; vhx.async = true;\n        vhx.src = \'https://cdn.vhx.tv/assets/tracker-be253e44a9312b41e422e9acec66d157e9c650341ea1701c01c2c8b31b26b9e6.js\';\n        var s = document.getElementsByTagName(\'script\')[0]; s.parentNode.insertBefore(vhx, s);\n      })();\n\n\n    Tracking.SITE_GANALYTICS_ENABLED = true;\n\n      ga(\'create\', \'UA-1852531-11\', \'auto\', { \'name\': \'site\', \'allowLinker\': true});\n\n    ga(\'site.require\', \'displayfeatures\');\n      ga(\'site.require\', \'linker\');\n      ga(\'site.linker:autoLink\', [\'www.criterionchannel.com\',\'criterionchannel.vhx.tv\'] );\n    ga(\'site.set\', \'dimension1\', \'criterionchannelchartersu\');\n    ga(\'site.send\', \'pageview\');\n\n    // <!-- Google Tag Manager -->\n    (function(w,d,s,l,i){w[l]=w[l]||[];w[l].push({\'gtm.start\':\n    new Date().getTime(),event:\'gtm.js\'});var f=d.getElementsByTagName(s)[0],\n    j=d.createElement(s),dl=l!=\'dataLayer\'?\'&l=\'+l:\'\';j.async=true;j.src=\n    \'https://www.googletagmanager.com/gtm.js?id=\'+i+dl;f.parentNode.insertBefore(j,f);\n    })(window,document,\'script\',\'dataLayer\',\'GTM-PKL3NF6\');\n    // <!-- End Google Tag Manager -->\n\n  if (window._current_site && window._current_site.facebook_pixel_codes.length > 0) {\n    // <!-- Facebook Pixel Code -->\n    !function(f,b,e,v,n,t,s)\n    {if(f.fbq)return;n=f.fbq=function(){n.callMethod?\n    n.callMethod.apply(n,arguments):n.queue.push(arguments)};\n    if(!f._fbq)f._fbq=n;n.push=n;n.loaded=!0;n.version=\'2.0\';\n    n.queue=[];t=b.createElement(e);t.async=!0;\n    t.src=v;s=b.getElementsByTagName(e)[0];\n    s.parentNode.insertBefore(t,s)}(window, document,\'script\',\n    \'https://connect.facebook.net/en_US/fbevents.js\');\n    //<!-- End Facebook Pixel Code -->\n  }\n\n  if (window._current_site && window._current_site.twitter_pixel_codes.length > 0) {\n    // <!-- Twitter Pixel Code -->\n    !function(e,t,n,s,u,a){e.twq||(s=e.twq=function(){s.exe?s.exe.apply(s,arguments):s.queue.push(arguments);\n    },s.version=\'1.1\',s.queue=[],u=t.createElement(n),u.async=!0,u.src=\'//static.ads-twitter.com/uwt.js\',\n    a=t.getElementsByTagName(n)[0],a.parentNode.insertBefore(u,a))}(window,document,\'script\');\n    //<!-- End Twitter Pixel Code -->\n  }\n\n      Segment_ID = \'D22V0IgjPYTwUKXVEtSzbmfcnolk4CNB\';\n      !function(){var analytics=window.analytics=window.analytics||[];if(!analytics.initialize)if(analytics.invoked)window.console&&console.error&&console.error("Segment snippet included twice.");else{analytics.invoked=!0;analytics.methods=["trackSubmit","trackClick","trackLink","trackForm","pageview","identify","reset","group","track","ready","alias","debug","page","once","off","on"];analytics.factory=function(t){return function(){var e=Array.prototype.slice.call(arguments);e.unshift(t);analytics.push(e);return analytics}};for(var t=0;t<analytics.methods.length;t++){var e=analytics.methods[t];analytics[e]=analytics.factory(e)}analytics.load=function(t,e){var n=document.createElement("script");n.type="text/javascript";n.async=!0;n.src="https://cdn.segment.com/analytics.js/v1/"+t+"/analytics.min.js";var a=document.getElementsByTagName("script")[0];a.parentNode.insertBefore(n,a);analytics._loadOptions=e};analytics.SNIPPET_VERSION="4.1.0";\n      analytics.load("D22V0IgjPYTwUKXVEtSzbmfcnolk4CNB");\n      }}();\n  </script>\n\n\n\n    \n\n  <script>\n  if (/ticket=/.test(location.search) && window.history.replaceState) {\n    var url = location.href.replace(/\\?([^#]*)/, function(_, search) {\n      search = search.split(\'&\').map(function(v) {\n        return !/^ticket=/.test(v) && !/^support=/.test(v) && !/^sso=/.test(v) && v;\n      }).filter(Boolean).join(\'&\');\n      return search ? \'?\' + search : \'\';\n    });\n\n    if (url != location.href) {\n      window.history.replaceState({}, \'\', url);\n    }\n  }\n\n  if (/return_to=/.test(location.search) && window.history.replaceState) {\n    var url = location.href.replace(/\\?([^#]*)/, function(_, search) {\n      search = search.split(\'&\').map(function(v) {\n        return !/^return_to=/.test(v) && v;\n      }).filter(Boolean).join(\'&\');\n      return search ? \'?\' + search : \'\';\n    });\n\n    if (url != location.href) {\n      window.history.replaceState({}, \'\', url);\n    }\n  }\n\n  if (/logout=/.test(location.search) && window.history.replaceState) {\n    var url = location.href.replace(/\\?([^#]*)/, function(_, search) {\n      search = search.split(\'&\').map(function(v) {\n        return !/^logout=/.test(v) && v;\n      }).filter(Boolean).join(\'&\');\n      return search ? \'?\' + search : \'\';\n    });\n\n    if (url != location.href) {\n      window.history.replaceState({}, \'\', url);\n    }\n  }\n</script>\n\n\n  <script>\n    window.TOKEN = "eyJhbGciOiJSUzI1NiIsImtpZCI6ImQ2YmZlZmMzNGIyNTdhYTE4Y2E2NDUzNDE2ZTlmZmRjNjk4MDAxMDdhZTQ2ZWJhODg0YTU2ZDBjOGQ4NTYzMzgifQ.eyJhcHBfaWQiOjY4NTExLCJleHAiOjE2NzE4NDM3MjEsIm5vbmNlIjoiMThjYTQ5MzFhODFjZDExNSIsInNjb3BlcyI6WyJwdWJsaWMiXSwic2Vzc2lvbl9pZCI6IkhsZUlhTk5DUWFGcWg3cklqZTJFYnc9PSIsInNpdGVfaWQiOjU5MDU0fQ.hxhFdkBtNgoN-agZaREtB3u8dnp95ZVYtt6TGMNL3-_d2gKARSPfxvKQnUgydfMB8Nby0QDwHoYA4V6wi9fCrAdU1dcbOr7rwt4x0EkNoGT_7ku8dvDssk4HXPPsdqHTB-N-66uV24KhHmbXAMGof1mlEme516VbKAlU11JE7cxH9JOfVcR2uoDsD87ZeKqYoFSz3LHwuzku9orzRXcDyQifAH_HdjdYCBkZxLadbCdy9ABfeoKG4pxfu_fCKsA0O_MeDJ3ohimncJsNvr26C7q0eHWBnAwlpeR9FPQMkWCNkpz-6OHk4PNovZilQBTIk6TFvI5dGk4a0nXjNvOkYaImY_n5yxYvir21EMMT-x1h6i0RkjCNDbeJkbZ--0rrP5aC3Qg0CH8s1EnF9fxe42bMcddpFgZUtNqzIsuYyLb0WEjbYZ5W7E7LmYpasAwVnbTQOCu5XOKclLl-vxQMdy-sxCoV8OgEkJJ38zeBtPJbl9IY5LiIfUvpwiWsvg_hj2rtaAe0bAZmTIGzYU1F1tZaKuYSBhwTTwKgBwtJ0jElNWfdAKAGoH5r_efQUY0EUb03nj4ucC0o4pvz75pxiFyauy5elpMRBnxkE_4Q5iN925ER_-cyJraPWfdNb8RS66hLSA9-24aujhJmXKuk87YbzMx0rxqs7XRYAoX7Zak";\n\n    (function() {\n      var config = { kitId: \'kyo4vqy\', scriptTimeout: 3000 };\n      var h=document.getElementsByTagName("html")[0];h.className+=" wf-loading";var t=setTimeout(function(){h.className=h.className.replace(/(\\s|^)wf-loading(\\s|$)/g," ");h.className+=" wf-inactive"},config.scriptTimeout);var tk=document.createElement("script"),d=false;tk.src=\'//use.typekit.net/\'+config.kitId+\'.js\';tk.type="text/javascript";tk.async="true";tk.onload=tk.onreadystatechange=function(){var a=this.readyState;if(d||a&&a!="complete"&&a!="loaded")return;d=true;clearTimeout(t);try{Typekit.load(config)}catch(b){}};var s=document.getElementsByTagName("script")[0];s.parentNode.insertBefore(tk,s);\n    })();\n  </script>\n\n  \n  \n  \n</head>\n\n\n<body class="site-background-color  site-color-is-light logged-out collections group-a collection-three-by-jafar-panahi" data-site-color="#b9bcc7" data-track-event="site_series">\n\n    <!-- Google Tag Manager (noscript) -->\n  <noscript><iframe src="https://www.googletagmanager.com/ns.html?id=GTM-PKL3NF6"\n  height="0" width="0" style="display:none;visibility:hidden"></iframe></noscript>\n  <!-- End Google Tag Manager (noscript) -->\n\n  \n\n  <a href="#maincontent" class="skipto">Skip to main content</a>\n    <div style="display: none !important;" rel="description" class="is-hidden">\n      Classics and discoveries from around the world, thematically programmed with special features, on a streaming service brought to you by the Criterion Collection.\n    </div>\n\n  <div data-flash="{&quot;auto_hide&quot;:true,&quot;style&quot;:&quot;overlay&quot;,&quot;container&quot;:null,&quot;type&quot;:&quot;notice&quot;,&quot;message&quot;:{&quot;body&quot;:null},&quot;show_on_load&quot;:false}" class="is-error is-hidden">\n    <div class="flash-icon">\n      <i class="icon icon-check-inverse icon-check-white"></i>\n      <i class="icon icon-alert-inverse icon-alert-white"></i>\n      <i class="icon icon-check-alt icon-check-blue"></i>\n      <i class="icon medium icon-check"></i>\n      <i class="icon medium icon-alert"></i>\n    </div>\n\n    <div class="flash-msg"></div>\n\n    <div class="flash-close">\n      <i class="icon icon-remove-inverse icon--xsmall"></i>\n      <i class="icon icon-remove-mono icon-x-black icon--xsmall"></i>\n    </div>\n</div>\n\n\n\n\n    \n\n\n\n\n\n\n\n\n\n<header class="nav-background-color border-bottom site-border-color site-header has-logo ">\n  <div class="row full-width padding-top-medium padding-bottom-medium primary-row-container flex-container">\n    <div class="small-16 medium-11 columns identity-container flex-container has-image">\n      <span class="show-for-small-only flex-item padding-right-small show-menu-container">\n        <label for="show-menu" class="show-menu-trigger">\n          <span class="site-link-header-color inverse"><i class="icon site-icon-menu"></i></span>\n        </label>\n      </span>\n\n      <h1 class="identity site-font-primary-family head primary flex-item padding-horizontal-medium">\n            <a href="/">\n              <img class="logo" src="https://vhx.imgix.net/criterionchannel/assets/1c5933b9-19a1-4ea8-9761-726ca79faf44.png?fit=max&h=84&q=92&w=400" alt="The Criterion Channel" />\n</a>      </h1>\n\n      <nav class="navigation-container inline flex-item" role="navigation" aria-label="Primary">\n        <input type="checkbox" id="show-menu" name="show-menu" role="button">\n\n        <div class="navigation-inner site-header-background-color-small-only" id="primary-navigation">\n\n          <a href="/browse" class="browse-link site-font-primary-family head site-link-header-color">\n            Browse\n          </a>\n\n\n              <a href="#"\n                id="search-nav-toggle"\n                class="search-link site-font-primary-family head site-link-header-color">\n                Search\n              </a>\n\n                <a href="https://films.criterionchannel.com/" rel="noopener" class="custom-nav-all-films site-font-primary-family head site-link-header-color" target="_blank">ALL FILMS</a>\n                <a href="https://www.criterion.com/" rel="noopener" class="custom-nav-criterion-com site-font-primary-family head site-link-header-color" target="_blank">Criterion.com</a>\n\n            <a class="show-for-small-only flex-item site-link-header-color site-font-primary-family head subscribe-link" data-track-event-properties="{&quot;type&quot;:&quot;subscription&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/three-by-jafar-panahi&quot;}" data-track-event="site_buy" href="https://www.criterionchannel.com/checkout/subscribe">\n              <strong style="font-weight: bold;">\n                Start Free Trial\n              </strong>\n</a>\n\n              <a href="/login" class="show-for-small-only flex-item user-action-link head site-link-header-color site-font-primary-family">\n                Sign in\n              </a>\n\n        </div>\n      </nav>\n    </div>\n\n    <div class="hide-for-small-only text-right flex-right small-16 medium-5 columns small-only-text-center user-activity site-border-color">\n      <div class="user-actions has-image padding-right-medium flex-container" role="navigation" aria-label="Secondary">\n            <a class="flex-item site-link-header-color site-font-primary-family head subscribe-link user-action-link" data-track-event-properties="{&quot;type&quot;:&quot;subscription&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/three-by-jafar-panahi&quot;}" data-track-event="site_buy" href="https://www.criterionchannel.com/checkout/subscribe">\n              <strong style="font-weight: bold;">\n                Start Free Trial\n              </strong>\n</a>\n\n              <a href="/login" class="flex-item user-action-link head site-link-header-color  site-font-primary-family">\n                Sign In\n              </a>\n      </div>\n    </div>\n  </div>\n</header>\n\n  <div id="search-react-app"\n    data-props="{&quot;overlay&quot;:true,&quot;site_id&quot;:59054,&quot;api_v1_url&quot;:&quot;https://api.vhx.tv&quot;,&quot;api_v2_url&quot;:&quot;https://api.vhx.com&quot;,&quot;user_subscribed&quot;:false,&quot;purchased_product_ids&quot;:[],&quot;theme&quot;:&quot;dark&quot;,&quot;show_filters&quot;:false}">\n  </div>\n\n\n\n  \n\n\n\n<section class="site-background-color site-font-primary-color margin-bottom-large collection-series top-container">\n  <div class="blurred-background-image"  style="background-image: url(\'https://vhx.imgix.net/criterionchannelchartersu/assets/2848bf05-4da0-4898-9e36-7e7f143bb35c.jpg?blur=180\')">\n    <div class="gradient-blend" data-gradient-blend></div>\n  </div>\n  <section class="product-feature padding-horizontal-medium padding-top-large padding-bottom-xlarge border-bottom border-bottom-light">\n\n    <div class="row relative ">\n      <div class="small-16 medium-8 large-8 collection-details-container columns end">\n\n        <div class="collection-details grid-padding-left">\n            <div class="show-for-small-only cover-art padding-bottom-medium">\n              <div class="embed">\n                <img src="https://vhx.imgix.net/criterionchannelchartersu/assets/2848bf05-4da0-4898-9e36-7e7f143bb35c.jpg?auto=format%2Ccompress&fit=crop&h=360&q=70&w=640" alt="Three by Jafar Panahi" data-image-primary>\n              </div>\n            </div>\n\n          <h1 class="head primary site-font-primary-color collection-title">\n            Three by Jafar Panahi\n          </h1>\n          <h2 class="site-font-secondary-color site-font-primary-family collection-stats">\n              3 Episodes\n          </h2>\n\n            <div class="contain padding-top-medium collection-description word-break">\n    <div class="site-font-secondary-color"\n         data-text-show-less="Show less"\n         data-text-show-more="Show more"\n         data-truncate=\'{"max_height": 60, "speed": 300}\'>\n      <p>The brilliant Iranian auteur Jafar Panahi was sentenced to six years in prison in July 2022, after refusing to stop making urgent, perceptive films when he was banned from the profession in 2010. With NO BEARS, his latest film, coming out in December, there\xe2\x80\x99s no better time to revisit three of his most beloved masterpieces, all of which view modern Iran through the eyes of young girls. </p>\n    </div>\n  </div>\n\n\n          <div class="collection-actions padding-bottom-small padding-top-large">\n                  <a href="https://www.criterionchannel.com/checkout/subscribe?return_to=https%3A%2F%2Fcriterionchannel.vhx.tv%2Fthree-by-jafar-panahi"\n      class="btn btn-site-primary custom-btn-action-transact site-font-primary-family uppercase margin-bottom-small margin-right-small"\n      data-track-event="site_collection_buy">\n      Subscribe\n    </a>\n\n\n\n              <a href="#"\n                 class="btn btn-transparent custom-btn-action-share margin-bottom-small margin-right-small"\n                 data-reveal-id="modal-three-by-jafar-panahi">\n                <span class="icon-adjustment relative inverse-icon-bg dark-bg padding-right-small">\n                  <i class="icon icon-share"></i>\n                </span>\n                Share\n              </a>\n              \n\n\n\n<div class="reveal-modal site-background-color text-center share-modal-container" id="modal-three-by-jafar-panahi" data-reveal>\n  <a class="close-reveal-modal"><i class="icon icon-remove-inverse"></i><i class="icon icon-remove-gray"></i></a>\n  <div class="share-title-container padding-bottom-medium">\n    <h5 class="head secondary site-font-secondary-color site-font-primary-family">\n      Share with your friends\n    </h5>\n  </div>\n  <div class="share-link-container padding-bottom-medium">\n    <ul class="small-block-grid-3">\n      <li>\n        <a href="https://www.facebook.com/sharer/sharer.php?u=https%3A%2F%2Fwww.criterionchannel.com%2Fthree-by-jafar-panahi"\n           data-social-popup\n           data-share="facebook"\n           class="btn btn-facebook block border site-border-color"\n           title="Facebook">\n          <i class="vertical-align icon site-icon-facebook icon-center"></i>\n        </a>\n      </li>\n      <li>\n        <a href="https://twitter.com/share?url=https%3A%2F%2Fwww.criterionchannel.com%2Fthree-by-jafar-panahi&text=Watch+Three+by+Jafar+Panahi+from+%40criterionchannl&related=criterionchannl"\n           data-social-popup\n           data-share="twitter"\n           class="btn-twitter btn block border site-border-color"\n           title="Twitter">\n          <i class="vertical-align icon site-icon-twitter icon-center"></i>\n        </a>\n      </li>\n      <li>\n        <a target="_blank" href="/cdn-cgi/l/email-protection#92ade1e7f0f8f7f1e6afc5f3e6f1fab2c6fae0f7f7b2f0ebb2d8f3f4f3e0b2c2f3fcf3fafbb2f4e0fdffb2c6faf7b2d1e0fbe6f7e0fbfdfcb2d1faf3fcfcf7feb4f3ffe2a9f0fdf6ebaffae6e6e2e1b7a1d3b7a0d4b7a0d4e5e5e5bcf1e0fbe6f7e0fbfdfcf1faf3fcfcf7febcf1fdffb7a0d4e6fae0f7f7bff0ebbff8f3f4f3e0bfe2f3fcf3fafb" class="btn btn-black block border site-border-color" title="Email">\n          <i class="vertical-align icon site-icon-envelope icon-center"></i>\n        </a>\n      </li>\n    </ul>\n  </div>\n    <div class="share-url-container">\n      <form class="form"><input value="https://www.criterionchannel.com/three-by-jafar-panahi" type="text" class="text text-center border-none site-background-color site-font-secondary-color" /></form>\n    </div>\n</div>\n\n          </div>\n\n        </div>\n      </div>\n\n        <div class="hide-for-small-only medium-8 large-offset-2 large-6 end cover-art columns">\n          <div class="embed-container grid-padding-right">\n            <div class="embed">\n              <img src="https://vhx.imgix.net/criterionchannelchartersu/assets/2848bf05-4da0-4898-9e36-7e7f143bb35c.jpg?auto=format%2Ccompress&fit=crop&h=720&q=75&w=1280" alt="Three by Jafar Panahi" data-image-primary>\n            </div>\n          </div>\n        </div>\n\n    </div>\n  </section>\n\n  <section class="episode-container video-container padding-bottom-large  padding-horizontal-medium ">\n    <div class="row relative season-controls padding-bottom-medium padding-top-large">\n      <div class="small-16 medium-11 large-13 columns text-left small-only-text-center">\n        <div class="grid-padding-left">\n            <form class="select-dropdown-wrapper" autocomplete="off">\n              <select class="js-switch-season btn-dropdown-transparent margin-right-small" data-switch-season disabled >\n                  <option value="https://www.criterionchannel.com/three-by-jafar-panahi/season:1" selected>\n                    Three by Jafar Panahi\n                  </option>\n              </select>\n            </form>\n          \n        </div>\n      </div>\n      <div class="hide-for-small-only medium-5 large-3 columns text-right">\n        <h2 class="site-font-secondary-color site-font-primary-family content-label padding-top-medium grid-padding-right">\n          3 Episodes\n        </h2>\n      </div>\n    </div>\n\n    <div class="row relative">\n      <div class="small-16 columns item-grid flush-row flush-grid">\n        \n  <div class="row full-width padding-horizontal-medium">\n    <ul class="small-block-grid-1 medium-block-grid-3 large-block-grid-4 js-load-more-items-container">\n\n\n\n  \n<li\n  class="js-collection-item\n    collection-item-936546 item-type-video\n    \n    "\n  data-item-id="936546"\n  data-item-type="video"\n  \n    data-tooltip="id: collection-tooltip-936546;\n    position: right;\n    offset: 0;\n    animate: true;\n    timer: 650;\n    delay: false;\n    is_centered: true;"\n    \n>\n  <div class="browse-item-card">\n    <div class="grid-item-padding">\n      <a href="https://www.criterionchannel.com/three-by-jafar-panahi/season:1/videos/the-white-balloon" class="browse-item-link" data-track-event="site_video" data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/three-by-jafar-panahi&quot;,&quot;id&quot;:936546,&quot;context&quot;:&quot;site_context_series&quot;,&quot;label&quot;:&quot;The White Balloon&quot;,&quot;canonical_collection&quot;:null}">\n        <div class="browse-image-container">\n            <span class="freemium-lock">\n              <i class="icon icon--xsmall icon-lock-inverse"></i>\n            </span>\n\n\n          <img alt="The White Balloon" src="https://vhx.imgix.net/criterionchannelchartersu/assets/53328e42-e15a-4a94-b2ef-8cb90c823967.jpg?auto=format%2Ccompress&fit=crop&h=360&q=70&w=640" />\n\n\n          <div class="browse-info-container">\n              <div class="duration-container\n                is-locked\n                "\n              >\n                 1:24:49\n              </div>\n          </div>\n        </div> \n      </a>\n\n      <div class="padding-small">\n        <div class="site-font-primary-color browse-item-title">\n           <a\n            href="https://www.criterionchannel.com/three-by-jafar-panahi/season:1/videos/the-white-balloon"\n            data-track-event="site_video"\n            data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/three-by-jafar-panahi&quot;,&quot;id&quot;:936546,&quot;context&quot;:&quot;site_context_series&quot;,&quot;label&quot;:&quot;The White Balloon&quot;,&quot;canonical_collection&quot;:null}"\n          >\n            <strong title="The White Balloon">\n              The White Balloon\n            </strong>\n          </a>\n        </div>\n\n          <span class="site-font-secondary-color browse-item-subtext">\n            <span class=\'media-identifier media-episode\'>Episode 1</span> \n          </span>\n      </div>\n    </div>\n\n\n    <div \n         class="tooltip background-white" id="collection-tooltip-936546">\n      <h3 class="tooltip-item-title site-font-primary-family"><strong>The White Balloon</strong></h3>\n        <h4 class="transparent"><span class=\'media-identifier media-episode\'>Episode 1</span> </h4>\n\n        <div class="transparent padding-top-medium">\n          <p>Directed by Jafar Panahi \xe2\x80\xa2 1995 \xe2\x80\xa2 Iran\n<br />Starring Aida Mohammadkhani, Mohsen Kafili, Fereshteh Sadre Orafaiy</p>\n\n<p>Jafar Panahi\xe2\x80\x99s revelatory debut feature is a child\xe2\x80\x99s-eye adventure in which a young girl\xe2\x80\x99s quest to buy a goldfish leads her on a detour-filled journey through the streets of Tehran on the ...</p>\n        </div>\n    </div>\n  </div>\n</li>\n\n\n\n\n  \n<li\n  class="js-collection-item\n    collection-item-840975 item-type-video\n    \n    "\n  data-item-id="840975"\n  data-item-type="video"\n  \n    data-tooltip="id: collection-tooltip-840975;\n    position: right;\n    offset: 0;\n    animate: true;\n    timer: 650;\n    delay: false;\n    is_centered: true;"\n    \n>\n  <div class="browse-item-card">\n    <div class="grid-item-padding">\n      <a href="https://www.criterionchannel.com/three-by-jafar-panahi/season:1/videos/the-mirror-1" class="browse-item-link" data-track-event="site_video" data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/three-by-jafar-panahi&quot;,&quot;id&quot;:840975,&quot;context&quot;:&quot;site_context_series&quot;,&quot;label&quot;:&quot;The Mirror&quot;,&quot;canonical_collection&quot;:null}">\n        <div class="browse-image-container">\n            <span class="freemium-lock">\n              <i class="icon icon--xsmall icon-lock-inverse"></i>\n            </span>\n\n\n          <img alt="The Mirror" src="https://vhx.imgix.net/criterionchannelchartersu/assets/cfd3258f-5dc4-4f4e-a167-be60825ba1b7-d217556a.jpg?auto=format%2Ccompress&fit=crop&h=360&q=70&w=640" />\n\n\n          <div class="browse-info-container">\n              <div class="duration-container\n                is-locked\n                "\n              >\n                 1:34:33\n              </div>\n          </div>\n        </div> \n      </a>\n\n      <div class="padding-small">\n        <div class="site-font-primary-color browse-item-title">\n           <a\n            href="https://www.criterionchannel.com/three-by-jafar-panahi/season:1/videos/the-mirror-1"\n            data-track-event="site_video"\n            data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/three-by-jafar-panahi&quot;,&quot;id&quot;:840975,&quot;context&quot;:&quot;site_context_series&quot;,&quot;label&quot;:&quot;The Mirror&quot;,&quot;canonical_collection&quot;:null}"\n          >\n            <strong title="The Mirror">\n              The Mirror\n            </strong>\n          </a>\n        </div>\n\n          <span class="site-font-secondary-color browse-item-subtext">\n            <span class=\'media-identifier media-episode\'>Episode 2</span> \n          </span>\n      </div>\n    </div>\n\n\n    <div \n         class="tooltip background-white" id="collection-tooltip-840975">\n      <h3 class="tooltip-item-title site-font-primary-family"><strong>The Mirror</strong></h3>\n        <h4 class="transparent"><span class=\'media-identifier media-episode\'>Episode 2</span> </h4>\n\n        <div class="transparent padding-top-medium">\n          <p>Directed by Jafar Panahi \xe2\x80\xa2 1997 \xe2\x80\xa2 Iran\n<br />Starring Mina Mohammad Khani, Aida Mohammadkhani, Kazem Mojdehi</p>\n\n<p>Iranian master Jafar Panahi explores the interplay of imagination and reality in this slyly inventive meta-film marvel. When her mother is late to pick her up from school, first grader Mina (Mi...</p>\n        </div>\n    </div>\n  </div>\n</li>\n\n\n\n\n  \n<li\n  class="js-collection-item\n    collection-item-2400849 item-type-video\n    \n    "\n  data-item-id="2400849"\n  data-item-type="video"\n  \n    data-tooltip="id: collection-tooltip-2400849;\n    position: right;\n    offset: 0;\n    animate: true;\n    timer: 650;\n    delay: false;\n    is_centered: true;"\n    \n>\n  <div class="browse-item-card">\n    <div class="grid-item-padding">\n      <a href="https://www.criterionchannel.com/three-by-jafar-panahi/season:1/videos/offside" class="browse-item-link" data-track-event="site_video" data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/three-by-jafar-panahi&quot;,&quot;id&quot;:2400849,&quot;context&quot;:&quot;site_context_series&quot;,&quot;label&quot;:&quot;Offside&quot;,&quot;canonical_collection&quot;:null}">\n        <div class="browse-image-container">\n            <span class="freemium-lock">\n              <i class="icon icon--xsmall icon-lock-inverse"></i>\n            </span>\n\n\n          <img alt="Offside" src="https://vhx.imgix.net/criterionchannelchartersu/assets/cbf0a27c-95c3-4c02-aa6a-2ea1bfd273bb.jpg?auto=format%2Ccompress&fit=crop&h=360&q=70&w=640" />\n\n\n          <div class="browse-info-container">\n              <div class="duration-container\n                is-locked\n                "\n              >\n                 1:31:46\n              </div>\n          </div>\n        </div> \n      </a>\n\n      <div class="padding-small">\n        <div class="site-font-primary-color browse-item-title">\n           <a\n            href="https://www.criterionchannel.com/three-by-jafar-panahi/season:1/videos/offside"\n            data-track-event="site_video"\n            data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/three-by-jafar-panahi&quot;,&quot;id&quot;:2400849,&quot;context&quot;:&quot;site_context_series&quot;,&quot;label&quot;:&quot;Offside&quot;,&quot;canonical_collection&quot;:null}"\n          >\n            <strong title="Offside">\n              Offside\n            </strong>\n          </a>\n        </div>\n\n          <span class="site-font-secondary-color browse-item-subtext">\n            <span class=\'media-identifier media-episode\'>Episode 3</span> \n          </span>\n      </div>\n    </div>\n\n\n    <div \n         class="tooltip background-white" id="collection-tooltip-2400849">\n      <h3 class="tooltip-item-title site-font-primary-family"><strong>Offside</strong></h3>\n        <h4 class="transparent"><span class=\'media-identifier media-episode\'>Episode 3</span> </h4>\n\n        <div class="transparent padding-top-medium">\n          <p>Directed by Jafar Panahi \xe2\x80\xa2 2006 \xe2\x80\xa2\xc2\xa0Iran\n<br />Starring Mohammad Asarian, Shayesteh Irani, Sima Mobarak-Shahi</p>\n\n<p>A by turns funny and caustic vision of gender politics in contemporary Iran comes into focus in an exuberant sports film unlike any other. The Tehran soccer stadium roars with 100,000 cheering m...</p>\n        </div>\n    </div>\n  </div>\n</li>\n\n\n\n\n    </ul>\n  </div>\n\n        \n      </div>\n    </div>\n\n  </section>\n</section>\n\n\n    <style>\n  a.site-link-footer-color,\n  a.site-link-footer-color:visited {\n    color: #ffffff !important;\n  }\n  .btn-site-footer,\n  .btn-site-footer:link,\n  .btn-site-footer:visited {\n    color: #b9bcc7;\n    background-color: #000;\n  }\n  .btn-site-footer:hover,\n  .btn-site-footer.is-hover,\n  .btn-site-footer:focus,\n  .btn-site-footer:focus:hover,\n  .btn-site-footer:active,\n  .btn-site-footer.is-active {\n    color: #000;\n    border-color: #000;\n    background-color: #b9bcc7;\n  }\n\n  .btn-logout-form {\n    display: inline;\n    line-height: 0;\n  }\n\n  .btn-logout {\n    background-color: transparent;\n    border: none;\n    cursor: pointer;\n    font-size: 12px;\n    padding: 0;\n  }\n\n  .btn-logout:hover {\n    opacity: .5;\n  }\n</style>\n\n\n<footer class="footer--site border-top site-border-color site-background-color padding-top-medium padding-bottom-medium ">\n  <div class="row full-width">\n    <div class="small-16 medium-6 large-10 columns helper-links small-only-text-center">\n      <div class="padding-left-medium display-flex-container flex-start force-min-height">\n              <a href="https://www.criterion.com/" class="site-font-secondary-color" target="_blank" rel="noopener">THE CRITERION COLLECTION</a>\n\n\n        <a href="/help" class="site-font-secondary-color legal-link help-link" rel="noopener">\n          Help\n        </a>\n        <a href="/tos" class="site-font-secondary-color legal-link terms-link" rel="noopener">\n          Terms\n        </a>\n        <a href="/privacy" class="site-font-secondary-color legal-link privacy-link" rel="noopener">\n          Privacy\n        </a>\n        <a href="/cookies" class="site-font-secondary-color legal-link cookies-link" rel="noopener">\n          Cookies\n        </a>\n\n\n            <a href="/login" class="site-font-secondary-color signin-link">\n              Sign in\n            </a>\n      </div>\n    </div> <!-- columns -->\n    <div class="small-16 medium-10 large-6 small-only-text-center text-right columns vhx-link">\n      <div class="padding-right-small display-flex-container flex-end force-min-height">\n          &nbsp;\n        <div id="language-switcher-react-app"\n             class="padding-right-small"\n             data-supported-locales="[]"\n        </div>\n        <script data-cfasync="false" src="/cdn-cgi/scripts/5c5dd728/cloudflare-static/email-decode.min.js"></script><script src="https://cdn.vhx.tv/assets/vite/assets/language-switcher.c19b2f61.js" crossorigin="anonymous" type="module"></script><link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/jsx-runtime.75dbc984.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/LanguageSwitcher.2b83283a.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/index.71ea8908.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/intl.helpers.2a168b49.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/index.61149844.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/deviceSizes.9973a5fd.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/message.bd28a54d.js" as="script" crossorigin="anonymous" />\n      </div> <!-- display-flex-container -->\n    </div>\n  </div> <!-- row -->\n</footer>\n\n<div class="modal">\n  <div class="close"><a href="#close" alt="Close Window"><strong class="is-inverse">&times;</strong></a></div>\n  <div class="content"></div>\n</div>\n\n<div id="blanket"></div>\n\n\n  <div class="modals">\n    \n  </div>\n\n  <script src="https://cdn.vhx.tv/assets/webpack/vendor-ottbundle-5b2989a58aa1f2ae17a5.js"></script>\n\n  <script>\n    window.VHX = window.VHX || {};\n    window.VHX.config = {\n      api_url: "https://api.vhx.tv",\n      token: "eyJhbGciOiJSUzI1NiIsImtpZCI6ImQ2YmZlZmMzNGIyNTdhYTE4Y2E2NDUzNDE2ZTlmZmRjNjk4MDAxMDdhZTQ2ZWJhODg0YTU2ZDBjOGQ4NTYzMzgifQ.eyJhcHBfaWQiOjY4NTExLCJleHAiOjE2NzE4NDM3MjEsIm5vbmNlIjoiMThjYTQ5MzFhODFjZDExNSIsInNjb3BlcyI6WyJwdWJsaWMiXSwic2Vzc2lvbl9pZCI6IkhsZUlhTk5DUWFGcWg3cklqZTJFYnc9PSIsInNpdGVfaWQiOjU5MDU0fQ.hxhFdkBtNgoN-agZaREtB3u8dnp95ZVYtt6TGMNL3-_d2gKARSPfxvKQnUgydfMB8Nby0QDwHoYA4V6wi9fCrAdU1dcbOr7rwt4x0EkNoGT_7ku8dvDssk4HXPPsdqHTB-N-66uV24KhHmbXAMGof1mlEme516VbKAlU11JE7cxH9JOfVcR2uoDsD87ZeKqYoFSz3LHwuzku9orzRXcDyQifAH_HdjdYCBkZxLadbCdy9ABfeoKG4pxfu_fCKsA0O_MeDJ3ohimncJsNvr26C7q0eHWBnAwlpeR9FPQMkWCNkpz-6OHk4PNovZilQBTIk6TFvI5dGk4a0nXjNvOkYaImY_n5yxYvir21EMMT-x1h6i0RkjCNDbeJkbZ--0rrP5aC3Qg0CH8s1EnF9fxe42bMcddpFgZUtNqzIsuYyLb0WEjbYZ5W7E7LmYpasAwVnbTQOCu5XOKclLl-vxQMdy-sxCoV8OgEkJJ38zeBtPJbl9IY5LiIfUvpwiWsvg_hj2rtaAe0bAZmTIGzYU1F1tZaKuYSBhwTTwKgBwtJ0jElNWfdAKAGoH5r_efQUY0EUb03nj4ucC0o4pvz75pxiFyauy5elpMRBnxkE_4Q5iN925ER_-cyJraPWfdNb8RS66hLSA9-24aujhJmXKuk87YbzMx0rxqs7XRYAoX7Zak",\n      token_expires_in: "7200000",\n      user_has_subscription: "false"\n    };\n  </script>\n\n  <script src="https://cdn.vhx.tv/assets/webpack/sites-ottbundle-1e2d12c471814f99fb7e.js"></script>\n  \n\n      <script>\n    window.VHX = window.VHX || {};\n    window.VHX.config = {\n      api_url: "https://api.vhx.tv",\n      token: "eyJhbGciOiJSUzI1NiIsImtpZCI6ImQ2YmZlZmMzNGIyNTdhYTE4Y2E2NDUzNDE2ZTlmZmRjNjk4MDAxMDdhZTQ2ZWJhODg0YTU2ZDBjOGQ4NTYzMzgifQ.eyJhcHBfaWQiOjY4NTExLCJleHAiOjE2NzE4NDM3MjEsIm5vbmNlIjoiMThjYTQ5MzFhODFjZDExNSIsInNjb3BlcyI6WyJwdWJsaWMiXSwic2Vzc2lvbl9pZCI6IkhsZUlhTk5DUWFGcWg3cklqZTJFYnc9PSIsInNpdGVfaWQiOjU5MDU0fQ.hxhFdkBtNgoN-agZaREtB3u8dnp95ZVYtt6TGMNL3-_d2gKARSPfxvKQnUgydfMB8Nby0QDwHoYA4V6wi9fCrAdU1dcbOr7rwt4x0EkNoGT_7ku8dvDssk4HXPPsdqHTB-N-66uV24KhHmbXAMGof1mlEme516VbKAlU11JE7cxH9JOfVcR2uoDsD87ZeKqYoFSz3LHwuzku9orzRXcDyQifAH_HdjdYCBkZxLadbCdy9ABfeoKG4pxfu_fCKsA0O_MeDJ3ohimncJsNvr26C7q0eHWBnAwlpeR9FPQMkWCNkpz-6OHk4PNovZilQBTIk6TFvI5dGk4a0nXjNvOkYaImY_n5yxYvir21EMMT-x1h6i0RkjCNDbeJkbZ--0rrP5aC3Qg0CH8s1EnF9fxe42bMcddpFgZUtNqzIsuYyLb0WEjbYZ5W7E7LmYpasAwVnbTQOCu5XOKclLl-vxQMdy-sxCoV8OgEkJJ38zeBtPJbl9IY5LiIfUvpwiWsvg_hj2rtaAe0bAZmTIGzYU1F1tZaKuYSBhwTTwKgBwtJ0jElNWfdAKAGoH5r_efQUY0EUb03nj4ucC0o4pvz75pxiFyauy5elpMRBnxkE_4Q5iN925ER_-cyJraPWfdNb8RS66hLSA9-24aujhJmXKuk87YbzMx0rxqs7XRYAoX7Zak",\n      token_expires_in: "7200000",\n      user_has_subscription: "false"\n    };\n  </script>\n\n  <script src="https://cdn.vhx.tv/assets/webpack/products-ottbundle-12994ed95e958f7c2e65.js"></script>\n    <script src="https://cdn.vhx.tv/assets/vite/assets/search.f6a28d3c.js" crossorigin="anonymous" type="module"></script><link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/jsx-runtime.75dbc984.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/index.esm.f84f874f.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/index.0e8c968f.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/intl.helpers.2a168b49.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/api.utils.c8000605.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/Menu.esm.959a8f77.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/Button.esm.61e4a4cb.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/Text.esm.c0f99a84.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/message.bd28a54d.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/Checkmark.esm.fb551793.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/Link.esm.a6ddba18.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/Search.esm.a44cb748.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/index.71ea8908.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/text-helpers.5cfd261d.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/time-helpers.4df27d81.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/_includes.3534e71e.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/_curry2.3e499763.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/path.235ca80c.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/Header.esm.2a01f61a.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/DismissX.esm.54e5d2b0.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/colors.esm.e8046dae.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/inheritsLoose.a8857277.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/index.61149844.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/ChevronDown.esm.405e6674.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/_isArray.68eb59b2.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/index.096cf963.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/equals.d9a066f9.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/_isString.019dcf3c.js" as="script" crossorigin="anonymous" /><link rel="stylesheet" media="screen" href="https://cdn.vhx.tv/assets/vite/assets/search.c6f13361.css" crossorigin="anonymous" />\n\n\n<script>\n  setTimeout(function() {\n    document.documentElement.className += \' is-loaded\';\n    var selects = document.getElementsByClassName(\'select\'),\n        checkboxes = document.getElementsByClassName(\'checkbox\'),\n        i;\n\n    if (selects.length) {\n      for (i = 0; i < selects.length; i++) {\n        selects[i].className += \' ready\';\n      }\n    }\n\n    if (checkboxes.length) {\n      for (i = 0; i < checkboxes.length; i++) {\n        for (var p=0; p < checkboxes[i].childNodes.length; p++) {\n          if (checkboxes[i].childNodes[p].tagName == \'LABEL\') {\n            checkboxes[i].childNodes[p].className += \' ready\';\n            break;\n          }\n        }\n      }\n    }\n  }, 500);\n</script>\n\n  <script>\n  window.addEventListener(\'load\', function() {\n    jQuery(\'a:contains("Sign In")\').click(function(){\n      ga(\'site.send\',\'event\',\'button\',\'click\',\'sign in\');\n    });\n\n    jQuery(\'a:contains("Sign Up")\').click(function(){\n      ga(\'site.send\',\'event\',\'button\',\'click\',\'sign up\');\n    });\n\n    jQuery(\'a:contains("Watch now")\').click(function(){\n      ga(\'site.send\',\'event\',\'button\',\'click\',\'watch now\');\n    });\n\n    jQuery(\'a:contains("Share")\').click(function(){\n      ga(\'site.send\',\'event\',\'button\',\'click\',\'share\');\n    });\n\n    jQuery(\'[href="https://www.criterionchannel.com/the-deviant-and-the-divine/season:1/videos/freaks"],[href="https://www.criterionchannel.com/the-deviant-and-the-divine/season:1/videos/multiple-maniacs"]\').click(function(){\n      ga(\'site.send\',\'event\',\'button\',\'click\',\'video\');\n    });\n  });\n</script>\n\n\n\n    <script>\n      const javascript = \'\\$(document).ready(function(){\\n\\$(\\\'header .logo\\\').attr(\\\'data-swiftype-index\\\', false);\\n\\n\\$(\\"a[href*=\\\'/checkout\\\']\\").attr(\\"data html(\\"Sign Up\\");\\n  \\$(\\"a[href*=\\\'/checkout/subscribe/gift\\\']\\").html(\\"Buy a gift\\");\\n  \\$(\\"a[href*=\\\'/buy/subscription\\\']\\").attr(\\"data-trigger-disabled\\",true);\\n  \\$(\\"a[href*=\\\'/buy/subscription\\\']\\").html(\\"Sign Up\\");\\n  \\$(\\"a.browse-link\\").html(\\"Now Playing\\");\\n  landingPageChanges();\\n  collectionChanges();\\n  helpChanges();\\n  \\$(\\"body.browse .slide-subtitle\\").hide();\\n  \\$(\\"body.logged-in h1.identity a\\").attr(\\"href\\",\\"/\\");\\n  \\$(\\".navigation-inner\\").addClass(\\"is-ready\\");\\n  \\$(\\"nav .custom-nav-all-films\\").attr(\\"target\\", \\"_self\\");\\n\\n  amazon_fire_tv_supported = \\$(\\\'.amazon-fire-tv\\\').parent().parent().children()[1]\\n  \\$(amazon_fire_tv_supported).text(\\"2nd Gen +\\")\\n\\n  android_tv_supported_text = \\$(\\\'.android-tv\\\').parent().parent().children()[1]\\n  \\$(android_tv_supported_text).text(\\"Selected models\\")\\n  android_tv_4k_text = \\$(\\\'.android-tv\\\').parent().parent().children()[2]\\n  \\$(android_tv_4k_text).hide()\\n\\n  new_xbox = \\"<li>Xbox Series X|S<\\/li>\\";\\n  \\$(new_xbox).insertAfter(\\$(\\\'.platforms-modal-content-tv .platforms-modal-icon-container .platform-modal-devices-list > li:nth-of-type(5) li:nth-of-type(2)\\\'));\\n\\n  \\$(\\\'.platforms-modal-content-tv .platforms-modal-icon-container .platform-modal-devices-list > li:nth-of-type(6) li:nth-of-type(2)\\\').text(\\\'Select 2018, 2019 & 2020 Models\\\');\\n\\n  gift_link = \\\'<a href=\\"https://www.criterionchannel.com/checkout/subscribe/gift?gift=1&duration=6\\" class=\\"is-block text dark block padding-medium\\">Give a Gift<\\/a>\\\';\\n  \\$(gift_link).insertAfter(\\$(\\\'.user-dropdown a[href*=\\"/purchases\\"]\\\'));\\n\\n  if (\\$(\\\'meta[name*=\\"twitter:image\\"]\\\').length > 0) {\\n    twitter_content = \\$(\\\'meta[name*=\\"twitter:image\\"]\\\').attr(\\\'content\\\')\\n    new_meta = \\\'<meta class=\\"swiftype\\" name=\\"image\\" data-type=\\"enum\\" content=\\"\\\' + twitter_content + \\\'\\">\\\';\\n    \\$(\\\'head\\\').append(new_meta);\\n  }\\n});\\n\\nfunction collectionChanges(){\\n  if (\\$(\\"body\\").hasClass(\\"collections\\")){\\n    \\$(\\\'.cover-art\\\').removeClass(\\\'large-offset-2 large-6\\\').addClass(\\\'large-8 large-offset-1\\\');\\n\\n    \\$(\\\'section.product-feature\\\').removeClass(\\\'padding-horizontal-medium padding-top-large padding-bottom-xlarge\\\').addClass(\\\'padding-bottom-large\\\');\\n    \\$(\\\'section.product-feature .row.relative\\\').addClass(\\\'full-width\\\');\\n\\n    \\$(\\\'section.product-feature .collection-details-container\\\').removeClass(\\"large-8 end\\").addClass(\\"large-7\\");\\n      \\$(\\\'section.product-feature .collection-details\\\').addClass(\\"padding-left-large padding-right-large padding-top-xlarge\\");\\n\\n      \\$(\\".embed-container\\").removeClass(\\"grid-padding-right\\");\\n  }\\n  if (\\$(\\\'.collection-playlist\\\').length > 0 || \\$(\\\'.collection-series\\\').length > 0) {\\n    \\$(\\\'.video-controls\\\').removeClass(\\\'padding-top-large\\\').addClass(\\\'padding-top-small\\\');\\n  }\\n\\n  if (\\$(\\\'.collection-movie\\\').length > 0) {\\n    \\$(\\\'.collection-actions a:contains(\\"Trailer\\")\\\').html(\\\'<span class=\\"icon-adjustment relative inverse-icon-bg dark-bg margin-bottom-small\\"><i class=\\"icon small icon-play-inverse\\"><\\/i><\\/span> Trailer\\\');\\n    \\$(\\\'.collection-actions a:contains(\\"Movie\\")\\\').html(\\\'<span class=\\"icon-adjustment relative inverse-icon-bg dark-bg margin-bottom-small\\"><i class=\\"icon small icon-play-inverse\\"><\\/i><\\/span> Watch\\\');\\n  }\\n\\n\\n}\\n\\nfunction landingPageChanges(){\\n  \\$(\\\'a:contains(\\"Start Your Free Month\\")\\\').text(\\"Sign Up Now\\");\\n}\\nfunction helpChanges(){\\n  // Custom Help Center Updates\\n  setTimeout(function() {\\n    \\$(\\\'article li:contains(\\"You can plug your laptop or mobile device into your TV using an HDMI cable\\")\\\').each(function() {\\n      \\$(this).html(\\\'<li>It may be possible to connect your laptop or mobile device to your TV using an HDMI cable if all components in the chain are <a href=\\"https://www.criterionchannel.com/help/articles/drm-and-hdcp#HDCP\\">HDCP/DRM compliant.<\\/a><\\/li>\\\');\\n    });\\n\\n    \\$(\\\'article li:contains(\\" and select the language of choice\\")\\\').each(function() {\\n      \\$(this).text(\\$(this).text().replace(\\" and select the language of choice\\", \\"\\"));\\n    });\\n    \\n    a = \\\'<li>If you have a supported device connected to your TV, Criterion also has an app for Apple TV, Amazon Fire, Roku, and Android TV.<\\/li>\\\'\\n    b = \\$(\\\'li:contains(\\"You can mirror or Airplay from your iOS device\xe2\x80\x99s browser.\\")\\\')\\n    b.after(a);\\n\\n    \\$(\\\'li a[href*=\\"670-how-do-i-login-on-the-xbox-app\\"]\\\').parent().hide();\\n  }, 500);\\n\\n  // hide mentions of the VHX articles\\n  // we have to settimeout because the help pages are ajax\\\'ing in content.\\n  // this is best guess on timing\\n  if (\\$(\\"body\\").hasClass(\\"help-articles\\") || \\$(\\"body\\").hasClass(\\"sites-help\\") || \\$(\\"body\\").hasClass(\\"help-categories\\")){\\n   \\$(\\\'#mithril-help\\\').bind(\\"DOMSubtreeModified\\",function(){\\n    var waittoremove = setTimeout(function(){ removeHelpArticles(); }, 500);\\n  });\\n\\n    var waittoremove = setTimeout(function(){ removeHelpArticles(); }, 700);\\n  }\\n}\\nfunction removeHelpArticles(){\\n  var vhxhelp = Array(\\"589e1e742c7d3a784630ccd5\\",\\"581396999033604deb0ec504\\",\\"57fe5ceb9033600277a6980b\\",\\"54b59261e4b0934e822d2132\\",\\"54dbc064e4b086c0c0966908\\",\\"5453fe87e4b02ce342671537\\",\\"555119ebe4b027e1978dfc04\\",\\"549206b9e4b07d03cb251286\\", \\"5c7eee4f04286350d088ab37\\");\\n  for (i=0;i<=vhxhelp.length-1;i++){\\n    \\$(\\".article-\\"+vhxhelp[i]).remove();                                                        \\n  }\\n}\';\n      try {\n        // Using eval so we can catch syntax errors\n        eval(javascript);\n      } catch (err) {\n        console.error(err);\n      }\n    </script>\n</body>\n</html>\n'
    return BeautifulSoup(content, 'html5lib')


@pytest.fixture
def soup_collection_director():
    content = b'<!doctype html>\n<!--[if lt IE 7]> <html class="no-js lt-ie9 lt-ie8 lt-ie7" lang="en" xml:lang="en"> <![endif]-->\n<!--[if IE 7]>    <html class="no-js lt-ie9 lt-ie8" lang="en" xml:lang="en"> <![endif]-->\n<!--[if IE 8]>    <html class="no-js lt-ie9" lang="en" xml:lang="en"> <![endif]-->\n<!--[if gt IE 8]><!--> <html class="no-js" lang="en" xml:lang="en"> <!--<![endif]-->\n<head lang="en" xml:lang="en">\n  <meta charset="utf-8">\n  <meta http-equiv="X-UA-Compatible" content="IE=edge,chrome=1">\n  <meta name="viewport" content="width=device-width, initial-scale=1">\n\n    <link rel="shortcut icon" href="https://dr56wvhu2c8zo.cloudfront.net/criterionchannel/assets/5922c7f6-cdda-4786-8ac9-521602da043a.png" type="image/png">\n\n    <meta name="apple-itunes-app" content="app-id=1454275199, app-argument=https://criterionchannel.vhx.tv/la-jetee">\n\n    <meta name="facebook-domain-verification" content="qk8q6gxhll8cwf99kggygprhe0y9fz">\n\n  <title>La Jet\xc3\xa9e - The Criterion Channel</title>\n  <meta name="description" content="Directed by Chris Marker \xe2\x80\xa2 1963 \xe2\x80\xa2 France\n\nChris Marker, filmmaker, poet, novelist, photographer, editor, and now videographer and digital multimedia artist, has been challenging moviegoers, philosophers, and himself for years with his complex queries about time, memory, and the rapid advancement of life on this planet. Marker\xe2\x80\x99s LA JET\xc3\x89E is one of the most influential, radical science-fiction films ever made, a tale of time travel told in still images.">\n  <link rel="canonical" href="https://www.criterionchannel.com/la-jetee" />\n\n    <meta property="og:url" content="https://www.criterionchannel.com/la-jetee">\n  <meta property="fb:app_id" content="136862186391940">\n  <meta property="og:title" content="La Jet\xc3\xa9e - The Criterion Channel">\n  <meta property="og:site_name" content="The Criterion Channel">\n  <meta property="og:image" content="https://vhx.imgix.net/criterionchannelchartersu/assets/9dfaaba6-1c22-473c-afe2-0ef43c00c663-663d30eb.jpg?auto=format%2Ccompress&fit=crop&h=720&q=75&w=1280">\n  <meta property="og:image:width" content="1280">\n  <meta property="og:image:height" content="720">\n  <meta property="og:description" content="Directed by Chris Marker \xe2\x80\xa2 1963 \xe2\x80\xa2 France\n\nChris Marker, filmmaker, poet, novelist, photographer, editor, and now videographer and digital multimedia artist, has been challenging moviegoers, philosophers, and himself for years with his complex queries about time, memory, and the rapid advancement ...">\n  <meta property="og:type" content="website">\n\n  <meta name="twitter:card" content="summary_large_image">\n    <meta name="twitter:site" content="@criterionchannl">\n    <meta name="twitter:creator" content="@criterionchannl">\n  <meta name="twitter:title" content="La Jet\xc3\xa9e - The Criterion Channel">\n  <meta name="twitter:description" content="Directed by Chris Marker \xe2\x80\xa2 1963 \xe2\x80\xa2 France\n\nChris Marker, filmmaker, poet, novelist, photographer, editor, and now videographer and digital multimedia artist, has been challenging moviegoers, philosophers, and himself for years with his complex queries about time, memory, and the rapid advancement ...">\n  <meta name="twitter:image" content="https://vhx.imgix.net/criterionchannelchartersu/assets/9dfaaba6-1c22-473c-afe2-0ef43c00c663-663d30eb.jpg?auto=format%2Ccompress&fit=crop&h=720&q=75&w=1280">\n  <meta name="twitter:image:width" content="1280">\n  <meta name="twitter:image:height" content="720">\n\n\n\n<link rel="author" href="/humans.txt">\n<meta name="csrf-param" content="authenticity_token" />\n<meta name="csrf-token" content="gpJMEYjMxBamzX0JfoGi16ERF4vpFZM8dDINiQhygbsnTr2vFa+IoZ+BvBS/0689jGPPZNeT3uORUuAvo8+D6w==" />\n\n\n    <!-- apple / android  -->\n  <meta name="mobile-web-app-capable" content="no">\n  <meta name="apple-mobile-web-app-capable" content="no">\n  <meta name="apple-mobile-web-app-title" content="The Criterion Channel">\n  <!-- microsoft tile color -->\n  <meta name="msapplication-TileColor" content="#b9bcc7">\n  <!-- android browser bar color -->\n  <meta name="theme-color" content="#b9bcc7">\n\n    <!-- apple and other browsers -->\n    <link rel="apple-touch-icon" sizes="57x57"    href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=57&w=57">\n    <link rel="apple-touch-icon" sizes="114x114"  href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=114&w=114">\n    <link rel="apple-touch-icon" sizes="72x72"    href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=72&w=72">\n    <link rel="apple-touch-icon" sizes="144x144"  href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=144&w=144">\n    <link rel="apple-touch-icon" sizes="60x60"    href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=60&w=60">\n    <link rel="apple-touch-icon" sizes="120x120"  href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=120&w=120">\n    <link rel="apple-touch-icon" sizes="76x76"    href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=76&w=76">\n    <link rel="apple-touch-icon" sizes="152x152"  href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=152&w=152">\n    <link rel="apple-touch-icon" sizes="180x180"  href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=180&w=180">\n    <!-- windows tile -->\n    <meta name="msapplication-TileImage" content="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=144&w=144">\n    <!-- android icons -->\n    <link rel="icon" type="image/png" sizes="32x32" href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=32&w=32">\n    <link rel="icon" type="image/png" sizes="16x16" href="https://vhx.imgix.net/criterionchannelchartersu/assets/15749e13-43d7-4c08-88bf-f634ca8756b3.png?fm=png&q=16&w=16">\n\n\n    <meta class="swiftype" name="type" data-type="enum" content="Channel" />\n\n\n\n  <link rel="stylesheet" media="all" href="https://cdn.vhx.tv/assets/webpack/icons.svg-ottbundle-798d02cf9d8ee37d5dfe.css" />\n  <link rel="stylesheet" media="all" href="https://cdn.vhx.tv/assets/webpack/sites-ottbundle-edf8dd6d4f778f83ed2d.css" />\n\n    <link rel="stylesheet" media="screen" href="https://cdn.vhx.tv/assets/templates/_shared/typography.montserrat-3b605b0b18ab721686d69bf9c5bd93fc8f657562bc6a431ae71c409675dfb333.css" />\n\n<style>\n\n\n  /* main colors */\n  .site-background-color {\n    background-color: #141414 !important;\n  }\n  .site-primary-highlight-color {\n    background: #b9bcc7 !important;\n  }\n  .site-primary-color,\n  strong.site-primary-color,\n  .site-primary-color strong,\n  a.site-primary-color:link,\n  a.site-primary-color:visited {\n    color: #b9bcc7 !important;\n  }\n  a.site-primary-color:hover,\n  a.site-primary-color:active {\n    color: #9da0a9 !important;\n  }\n\n  .site-border-color,\n  .site-border-color.border-light {\n    border-color: #323232 !important;\n  }\n  .site-border-primary-color,\n  .site-border-primary-color.border-light {\n    border-color: #b9bcc7 !important;\n  }\n\n  /* custom font primary family */\n  .site-font-primary-family {\n    font-family: \'montserrat\', sans-serif !important;\n  }\n\n  /* font primary color */\n  .site-font-primary-color,\n  .site-font-primary-color strong {\n    color: #ffffff !important;\n  }\n\n  /* font secondary color */\n  .site-font-secondary-color,\n  .site-font-secondary-color strong {\n    color: #bebebb !important;\n  }\n\n  /* link primary color */\n  a.site-link-primary-color:link,\n  a.site-link-primary-color:visited,\n  a.site-link-primary-color strong {\n    color: #ffffff !important;\n  }\n  a.site-link-primary-color:hover,\n  a.site-link-primary-color:active {\n    color: #d9d9d9 !important;\n    opacity: .7;\n  }\n  .site-link-primary-color-contain a:link,\n  .site-link-primary-color-contain a:visited {\n    color: #b9bcc7 !important;\n  }\n  .site-link-primary-color-contain a:hover,\n  .site-link-primary-color-contain a:active {\n    color: #9da0a9 !important;\n    opacity: .7;\n  }\n\n  /* link secondary color */\n  .pagination a,\n  a.site-link-secondary-color:link,\n  a.site-link-secondary-color:visited,\n  a.site-link-secondary-color strong,\n  .site-link-secondary-color-contain a:link,\n  .site-link-secondary-color-contain a:visited {\n    color: #7c7c7c;\n  }\n  .pagination a:hover,\n  a.site-link-secondary-color:hover,\n  a.site-link-secondary-color:active,\n  .site-link-secondary-color-contain a:hover,\n  .site-link-secondary-color-contain a:active {\n    color: #696969;\n    opacity: .7;\n  }\n\n  /* link underline font primary color */\n  .site-link-contain a:link,\n  .site-link-contain a:visited {\n    color: #ffffff !important;\n    text-decoration: underline;\n  }\n  .site-link-contain a:hover,\n  .site-link-contain a:active {\n    color: #ffffff !important;\n    opacity: .7;\n    text-decoration: underline;\n  }\n\n  /* border color same as primary color */\n  .site-border-primary-color {\n    border-color: #ffffff !important;\n  }\n  .site-border-top-primary-color {\n    border-top-color: #ffffff !important;\n  }\n  .site-border-bottom-primary-color {\n    border-bottom-color: #ffffff !important;\n  }\n  .site-border-left-primary-color {\n    border-left-color: #ffffff !important;\n  }\n  .site-border-right-primary-color {\n    border-right-color: #ffffff !important;\n  }\n\n  /* link header color */\n  a.site-link-header-color:link,\n  a.site-link-header-color:visited,\n  a.site-link-header-color:link strong,\n  a.site-link-header-color:visited strong {\n    color: #ffffff !important;\n  }\n  a.site-link-header-color:hover,\n  a.site-link-header-color:active {\n    color: #d9d9d9;\n    opacity: .7;\n  }\n .avatar-chevron.icon {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2223%22%20height%3D%2223%22%20viewBox%3D%220%200%2023%2023%22%3E%3Cpolygon%20fill%3D%22%23ffffff%22%20points%3D%2222.9%2C7.2%2020%2C4.4%2011.5%2C12.9%203%2C4.4%200.1%2C7.2%208.7%2C15.7%208.7%2C15.7%2011.5%2C18.6%2011.5%2C18.6%2011.5%2C18.6%2014.3%2C15.7%2014.3%2C15.7%20%22%2F%3E%3C%2Fsvg%3E%0A");\n    background-repeat: no-repeat;\n  }\n  .user-dropdown a:hover {\n    background-color: rgba(51,51,51,0.2);\n    color: #fff;\n  }\n  .user-dropdown input {\n    background-color: #ffffff;\n  }\n  .user-dropdown input:hover {\n    background-color: rgba(51,51,51,0.2);\n    color: #fff;\n  }\n\n  /* buttons site primary */\n  .btn-site-primary,\n  .btn-site-primary:link,\n  .btn-site-primary:visited {\n    color: #000;\n    border-color: #9da0a9;\n    background-color: #b9bcc7;\n  }\n  .btn-site-primary:hover,\n  .btn-site-primary.is-hover,\n  .btn-site-primary:focus,\n  .btn-site-primary:focus:hover,\n  .btn-site-primary:active,\n  .btn-site-primary.is-active {\n    border-color: #9da0a9;\n    background-color: #9da0a9;\n    color: #000;\n  }\n\n  .btn-site-primary[disabled],\n  .btn-site-primary.is-disabled,\n  .btn-site-primary[disabled]:visited,\n  .btn-site-primary.is-disabled:visited,\n  .btn-site-primary[disabled]:hover,\n  .btn-site-primary.is-disabled:hover,\n  .btn-site-primary[disabled].is-hover,\n  .btn-site-primary.is-disabled.is-hover,\n  .btn-site-primary[disabled]:active,\n  .btn-site-primary.is-disabled:active,\n  .btn-site-primary[disabled]:focus,\n  .btn-site-primary.is-disabled:focus,\n  .btn-site-primary[disabled]:focus:hover,\n  .btn-site-primary.is-disabled:focus:hover,\n  .btn-site-primary[disabled].is-active,\n  .btn-site-primary.is-disabled.is-active {\n    opacity: .5;\n    background-color: #b9bcc7;\n    color: #000;\n    border-color: #9da0a9;\n  }\n\n  /* buttons site secondary */\n  .btn-site-secondary,\n  .btn-site-secondary:link,\n  .btn-site-secondary:visited {\n    color: #bebebb;\n    border-color: #141414;\n    background-color: #323232;\n  }\n  .btn-site-secondary:hover,\n  .btn-site-secondary.is-hover,\n  .btn-site-secondary:focus,\n  .btn-site-secondary:active,\n  .btn-site-secondary.is-active {\n    color: #5f5f5e;\n    border-color: #0f0f0f;\n    background-color: #323232;\n  }\n\n  .btn-site-primary.is-processing,\n  .btn-site-secondary.is-processing {\n    color: transparent !important;\n    background-position: center center !important;\n    background-repeat: no-repeat;\n    background-size: 23px 23px;\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3Csvg%20version%3D%221.1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%20%20viewBox%3D%220%200%2040%2040%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cpath%20opacity%3D%220.2%22%20fill%3D%22%23000000%22%20d%3D%22M20.201%2C5.169c-8.254%2C0-14.946%2C6.692-14.946%2C14.946c0%2C8.255%2C6.692%2C14.946%2C14.946%2C14.946%0D%0A%20%20s14.946-6.691%2C14.946-14.946C35.146%2C11.861%2C28.455%2C5.169%2C20.201%2C5.169z%20M20.201%2C31.749c-6.425%2C0-11.634-5.208-11.634-11.634%0D%0A%20%20c0-6.425%2C5.209-11.634%2C11.634-11.634c6.425%2C0%2C11.633%2C5.209%2C11.633%2C11.634C31.834%2C26.541%2C26.626%2C31.749%2C20.201%2C31.749z%22%2F%3E%0D%0A%3Cpath%20fill%3D%22%23000000%22%20d%3D%22M26.013%2C10.047l1.654-2.866c-2.198-1.272-4.743-2.012-7.466-2.012h0v3.312h0%0D%0A%20%20C22.32%2C8.481%2C24.301%2C9.057%2C26.013%2C10.047z%22%3E%0D%0A%20%20%3CanimateTransform%20attributeType%3D%22xml%22%0D%0A%20%20%20%20attributeName%3D%22transform%22%0D%0A%20%20%20%20type%3D%22rotate%22%0D%0A%20%20%20%20from%3D%220%2020%2020%22%0D%0A%20%20%20%20to%3D%22360%2020%2020%22%0D%0A%20%20%20%20dur%3D%220.5s%22%0D%0A%20%20%20%20repeatCount%3D%22indefinite%22%2F%3E%0D%0A%20%20%3C%2Fpath%3E%0D%0A%3C%2Fsvg%3E\');\n  }\n\n  /* buttons dropdown */\n  .btn-dropdown-site,\n  .btn-dropdown-site:link,\n  .btn-dropdown-site:visited {\n    color: #000000;\n    border-color: #b9bcc7;\n    background-color: #b9bcc7;\n  }\n  .btn-dropdown-site:hover,\n  .btn-dropdown-site.is-hover,\n  .btn-dropdown-site:focus,\n  .btn-dropdown-site:active,\n  .btn-dropdown-site.is-active {\n    border-color: #b9bcc7;\n    background-color: #9da0a9;\n  }\n\n  .site-btn-block,\n  .site-btn-block:link,\n  .site-btn-block:visited {\n    color: #ffffff;\n  }\n\n  .site-btn-block:hover,\n  .site-btn-block.is-hover,\n  .site-btn-block.is-active,\n  .site-btn-block:active {\n    box-shadow: inset 0px -4px 0px #ffffff;\n  }\n\n  .site-form-color input[type="email"],\n  .site-form-color input[type="text"],\n  .site-form-color input[type="search"],\n  .site-form-color input[type="password"],\n  .site-form-color input[type="tel"],\n  .site-form-color textarea {\n    color: #323232 !important;\n    border-color: #696969 !important;\n    background-color:  #fcfcfc !important;\n  }\n  .site-form-color label,\n  .site-form-color-minimal input[type="email"],\n  .site-form-color-minimal input[type="text"],\n  .site-form-color-minimal input[type="search"],\n  .site-form-color-minimal input[type="password"],\n  .site-form-color-minimal textarea {\n    color: #ffffff;\n    background: transparent !important;\n  }\n  .site-form-color input[type="text"]:focus,\n  .site-form-color input[type="search"]:focus,\n  .site-form-color input[type="password"]:focus,\n  .site-form-color input[type="tel"]:focus,\n  .site-form-color textarea:focus {\n    border-color: #7c7c7c !important;\n  }\n\n  .site-form-color input[type="text"]::-webkit-input-placeholder,\n  .site-form-color-minimal input[type="text"]::-webkit-input-placeholder,\n  .site-form-color input[type="email"]::-webkit-input-placeholder,\n  .site-form-color-minimal input[type="email"]::-webkit-input-placeholder,\n  .site-form-color input[type="search"]::-webkit-input-placeholder,\n  .site-form-color-minimal input[type="search"]::-webkit-input-placeholder,\n  .site-form-color input[type="password"]::-webkit-input-placeholder,\n  .site-form-color-minimal input[type="password"]::-webkit-input-placeholder,\n  .site-form-color input[type="tel"]::-webkit-input-placeholder,\n  .site-form-color-minimal input[type="tel"]::-webkit-input-placeholder,\n  .site-form-color textarea::-webkit-input-placeholder,\n  .site-form-color-minimal textarea::-webkit-input-placeholder {\n    color: #323232 !important;\n    opacity: .5;\n  }\n  .site-form-color input[type="text"]::-moz-input-placeholder,\n  .site-form-color-minimal input[type="text"]::-moz-input-placeholder,\n  .site-form-color input[type="email"]::-moz-input-placeholder,\n  .site-form-color-minimal input[type="email"]::-moz-input-placeholder,\n  .site-form-color input[type="search"]::-moz-input-placeholder,\n  .site-form-color-minimal input[type="search"]::-moz-input-placeholder,\n  .site-form-color input[type="password"]::-moz-input-placeholder,\n  .site-form-color-minimal input[type="password"]::-moz-input-placeholder,\n  .site-form-color input[type="tel"]::-moz-input-placeholder,\n  .site-form-color-minimal input[type="tel"]::-moz-input-placeholder,\n  .site-form-color textarea::-moz-input-placeholder,\n  .site-form-color-minimal textarea::-moz-input-placeholder {\n    color: #323232 !important;\n    opacity: .5;\n  }\n\n  .site-form-color input[type="text"]:-moz-input-placeholder,\n  .site-form-color-minimal input[type="text"]:-moz-input-placeholder,\n  .site-form-color input[type="email"]:-moz-input-placeholder,\n  .site-form-color-minimal input[type="email"]:-moz-input-placeholder,\n  .site-form-color input[type="search"]:-moz-input-placeholder,\n  .site-form-color-minimal input[type="search"]:-moz-input-placeholder,\n  .site-form-color input[type="password"]:-moz-input-placeholder,\n  .site-form-color-minimal input[type="password"]:-moz-input-placeholder,\n  .site-form-color input[type="tel"]:-moz-input-placeholder,\n  .site-form-color-minimal input[type="tel"]:-moz-input-placeholder,\n  .site-form-color textarea:-moz-input-placeholder,\n  .site-form-color-minimal textarea:-moz-input-placeholder {\n    color: #323232 !important;\n    opacity: .5;\n  }\n\n  .site-form-color input[type="text"]:-ms-input-placeholder,\n  .site-form-color-minimal input[type="text"]:-ms-input-placeholder,\n  .site-form-color input[type="email"]:-ms-input-placeholder,\n  .site-form-color-minimal input[type="email"]:-ms-input-placeholder,\n  .site-form-color input[type="search"]:-ms-input-placeholder,\n  .site-form-color-minimal input[type="search"]:-ms-input-placeholder,\n  .site-form-color input[type="password"]:-ms-input-placeholder,\n  .site-form-color-minimal input[type="password"]:-ms-input-placeholder,\n  .site-form-color input[type="tel"]:-ms-input-placeholder,\n  .site-form-color-minimal input[type="tel"]:-ms-input-placeholder,\n  .site-form-color textarea:-ms-input-placeholder,\n  .site-form-color-minimal textarea:-ms-input-placeholder {\n    color: #323232 !important;\n    opacity: .5;\n  }\n\n  .site-form-color .radio label .radio--icon {\n    border-color: #bebebb !important;\n  }\n  .site-form-color .radio.primary .circle-top span {\n    background: #b9bcc7 !important;\n  }\n  .site-form-color .radio.primary .circle-bottom span {\n    background: #94969f !important;\n  }\n\n  /* header */\n  .site-header {\n    border-bottom: 1px solid #2e2e2e;\n    background-color: #141414;\n\n\n  }\n  .site-header-background-color {\n    background-color: #141414;\n  }\n  @media only screen and (max-width: 40em) {\n     .site-header-background-color-small-only {\n      background-color: #141414;\n    }\n  }\n\n  /* modals */\n  .site-modal {\n    box-shadow: inset 0 0 2px #7c7c7c !important;\n  }\n\n  /* tooltips */\n  .tooltip.bottom-right:after,\n  .tooltip.bottom-left:after,\n  .tooltip.bottom:after,\n  .tooltip.bottom-right:before,\n  .tooltip.bottom-left:before,\n  .tooltip.bottom:before {\n    border-top-color: #7c7c7c;\n  }\n  .tooltip.right:after,\n  .tooltip.right:after\n  .tooltip.right-top:after,\n  .tooltip.right-top:before,\n  .tooltip.right-bottom:before,\n  .tooltip.right-bottom:before {\n    border-right-color: #7c7c7c;\n  }\n\n  .tooltip.left-bottom:after,\n  .tooltip.left-top:after,\n  .tooltip.left:after,\n  .tooltip.left-bottom:before,\n  .tooltip.left-top:before,\n  .tooltip.left:before {\n    border-left-color: #7c7c7c;\n  }\n\n  .tooltip.top-right:after,\n  .tooltip.top-left:after,\n  .tooltip.top:after,\n  .tooltip.top-right:before,\n  .tooltip.top-left:before,\n  .tooltip.top:before {\n    border-bottom-color: #7c7c7c;\n  }\n\n  .tooltip {\n    border: 1px solid #7c7c7c;\n    background: #7c7c7c;\n    color: #ffffff;\n  }\n\n  .reveal-modal {\n    border-color: #323232;\n    box-shadow: 0 0 3px #cbcbcb;\n  }\n  .reveal-modal-bg {\n    background-color: rgba(0,0,0,.8);\n  }\n  .play-outer-circle-large:hover {\n    border-color: #b9bcc7;\n  }\n  .forums-container tr.clickable:hover .thread-activity {\n    background-color: #141414;\n  }\n\n    [data-flash].is-inline.is-error {\n      background: rgba(211, 16, 74, 1) !important;\n    }\n    [data-flash].is-inline.is-error .flash-msg,\n    [data-flash].is-inline.is-error .flash-msg strong {\n      color: #fff !important;\n    }\n    [data-flash].is-inline.is-error .flash-close .icon-remove-mono {\n      display: none !important;\n    }\n    [data-flash].is-inline.is-error .flash-close .icon-remove-inverse {\n      display: block !important;\n    }\n\n  /* custom banner */\n\n\n</style>\n<style>\n  /* icons site primary */\n  .site-icon-arrow-right {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2223%22%20height%3D%2223%22%20viewBox%3D%220%200%2023%2023%22%3E%3Ctitle%3Eicons-arrows-set%20copy%3C%2Ftitle%3E%3Cpath%20d%3D%22M5.61%2C19.75l8.26-8.26L5.61%2C3.24c-0.68-.69.61-1.58%2C1-2%2C0.72-.69%2C1.76.78%2C2.2%2C1.22l3.39%2C3.39L17.41%2C11c0.71%2C0.69-.8%2C1.77-1.22%2C2.19L12.8%2C16.59%2C7.63%2C21.77c-0.7.67-1.58-.61-2-1A0.68%2C0.68%2C0%2C0%2C1%2C5.61%2C19.75Z%22%20fill%3D%22%23b9bcc7%22%2F%3E%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-menu {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Hamburger%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Crect%20x%3D%222.6%22%20y%3D%223.6%22%20width%3D%2218%22%20height%3D%223%22%20fill%3D%22%23ffffff%22%2F%3E%0D%0A%3Crect%20x%3D%222.6%22%20y%3D%2210%22%20width%3D%2218%22%20height%3D%223%22%20fill%3D%22%23ffffff%22%2F%3E%0D%0A%3Crect%20x%3D%222.6%22%20y%3D%2216.3%22%20width%3D%2218%22%20height%3D%223%22%20fill%3D%22%23ffffff%22%2F%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-chrevron-down {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21--%20Generator%3A%20Adobe%20Illustrator%2018.1.1%2C%20SVG%20Export%20Plug-In%20.%20SVG%20Version%3A%206.00%20Build%200%29%20%20--%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%20%20%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%22-294%20385%2023%2023%22%20enable-background%3D%22new%20-294%20385%2023%2023%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cg%3E%0D%0A%20%20%3Cpolygon%20fill%3D%22%23ffffff%22%20points%3D%22-271.1%2C392.9%20-275.4%2C388.7%20-282.5%2C395.8%20-289.6%2C388.7%20-293.9%2C392.9%20-286.7%2C400%20-286.8%2C400.1%20%0D%0A%20%20%20%20-282.5%2C404.3%20-282.5%2C404.3%20-282.5%2C404.3%20-278.2%2C400.1%20-278.3%2C400%20%20%22%2F%3E%0D%0A%3C%2Fg%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-chat {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21--%20Generator%3A%20Adobe%20Illustrator%2018.1.1%2C%20SVG%20Export%20Plug-In%20.%20SVG%20Version%3A%206.00%20Build%200%29%20%20--%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%22-294%20385%2023%2023%22%20enable-background%3D%22new%20-294%20385%2023%2023%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cg%3E%0D%0A%09%3Cg%3E%0D%0A%09%09%3Cpath%20fill%3D%22%23ffffff%22%20d%3D%22M-282.5%2C388.6c2.6%2C0%2C5.1%2C0.1%2C6.8%2C1.3c1.9%2C1.4%2C1.9%2C4.3%2C1.7%2C6c-0.2%2C2.1-1.2%2C3.5-3%2C4.2%0D%0A%09%09%09c-1.3%2C0.5-3%2C0.7-5.1%2C0.7c-1%2C0-2.1-0.1-3.4-0.2l-1-0.1l0%2C0l-1.2-0.4c-1.1-0.3-2.5-1-3.1-2.6c-0.7-1.8-0.7-4.2%2C0-5.9%0D%0A%09%09%09C-289.6%2C389.1-286.6%2C388.6-282.5%2C388.6%20M-282.5%2C386.6c-3.7%2C0-8.4%2C0.3-9.9%2C4.3c-0.9%2C2.2-0.9%2C5.1%2C0%2C7.4c0.7%2C2%2C2.4%2C3.2%2C4.3%2C3.8v3.2%0D%0A%09%09%09l2.6-2.7c1.1%2C0.1%2C2.4%2C0.2%2C3.6%2C0.2c2%2C0%2C4-0.2%2C5.7-0.9c2.6-1%2C4-3.1%2C4.3-5.8c0.3-2.8-0.1-6.1-2.6-7.9%0D%0A%09%09%09C-276.7%2C386.6-279.8%2C386.6-282.5%2C386.6L-282.5%2C386.6z%22%2F%3E%0D%0A%09%3C%2Fg%3E%0D%0A%3C%2Fg%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n\n  /* icons site social */\n  .site-icon-twitter {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%22-294%20385%2023%2023%22%20enable-background%3D%22new%20-294%20385%2023%2023%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cg%3E%0D%0A%09%3Cpath%20fill%3D%22%23ffffff%22%20d%3D%22M-272.5%2C390.3c-0.7%2C0.3-1.5%2C0.5-2.4%2C0.6c0.8-0.5%2C1.5-1.3%2C1.8-2.3c-0.8%2C0.5-1.7%2C0.8-2.6%2C1%0D%0A%09%09c-2.8-3-7.9-0.2-7%2C3.7c-3.4-0.1-6.4-1.7-8.4-4.2c-1.1%2C1.9-0.5%2C4.3%2C1.3%2C5.5c-0.7%2C0-1.3-0.2-1.9-0.5c0%2C1.9%2C1.4%2C3.7%2C3.3%2C4.1%0D%0A%09%09c-0.6%2C0.2-1.2%2C0.2-1.9%2C0.1c0.5%2C1.6%2C2%2C2.8%2C3.8%2C2.8c-1.7%2C1.3-3.9%2C2-6.1%2C1.7c4%2C2.6%2C9.5%2C2.5%2C13.4-0.5c3-2.3%2C4.8-6.1%2C4.6-9.9%0D%0A%09%09C-273.7%2C391.8-273.1%2C391.1-272.5%2C390.3C-273.2%2C390.6-273.1%2C391.1-272.5%2C390.3z%22%2F%3E%0D%0A%3C%2Fg%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-facebook {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21--%20Generator%3A%20Adobe%20Illustrator%2018.1.1%2C%20SVG%20Export%20Plug-In%20.%20SVG%20Version%3A%206.00%20Build%200%29%20%20--%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%22-294%20385%2023%2023%22%20enable-background%3D%22new%20-294%20385%2023%2023%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cpath%20fill%3D%22%23ffffff%22%20d%3D%22M-280.4%2C406.8v-9.3h3.4v-4h-3.4v-2.1c0-0.5%2C0.3-0.9%2C0.9-0.9h2.6v-4.5h-2.8c-3%2C0-5.3%2C2.4-5.3%2C5.5v2.1h-3.1v4%0D%0A%09h3.1v9.3h4.6L-280.4%2C406.8z%22%2F%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-tumblr {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cg%3E%0D%0A%09%3Cpath%20style%3D%22fill%3A%23ffffff%3B%22%20d%3D%22M17.6%2C20.4c-2%2C1-4.4%2C1.4-6.6%2C0.8c-2.3-0.6-3.4-2.3-3.4-4.5v-7H5.4V6.9c2.5-0.8%2C3.7-2.9%2C4-5.4h2.8%0D%0A%09%09v5.1h4.7v3.1h-4.7v5.1c0%2C1-0.2%2C2.4%2C0.9%2C3c1.4%2C0.8%2C3.2%2C0.2%2C4.5-0.6V20.4z%22%2F%3E%0D%0A%3C%2Fg%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-envelope {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cg%3E%0D%0A%09%3Cpath%20style%3D%22fill%3A%23ffffff%3B%22%20d%3D%22M22.6%2C9c0-0.2-0.1-0.5-0.3-0.6L12%2C0.8c-0.3-0.2-0.6-0.2-0.9%2C0L0.7%2C8.4C0.5%2C8.5%2C0.4%2C8.8%2C0.4%2C9v12.6%0D%0A%09%09c0%2C0.4%2C0.3%2C0.8%2C0.8%2C0.8h20.7c0.4%2C0%2C0.8-0.4%2C0.8-0.8V9z%20M11.5%2C2.4l9.6%2C7v0.9l-6.6%2C4.7l5.2%2C3.3c0.8%2C0.5%2C0%2C1.8-0.8%2C1.3L11.5%2C15%0D%0A%09%09l-7.4%2C4.7c-0.8%2C0.5-1.7-0.8-0.8-1.3l5.2-3.3l-6.6-4.7V9.4L11.5%2C2.4z%22%2F%3E%0D%0A%09%3Cpath%20style%3D%22fill%3A%23ffffff%3B%22%20d%3D%22M19.7%2C18.4%22%2F%3E%0D%0A%09%3Cpolygon%20style%3D%22fill%3A%23ffffff%3B%22%20points%3D%2211.5%2C13.1%2013.4%2C14.3%2019.5%2C9.9%2011.5%2C4%203.5%2C9.9%209.6%2C14.3%20%09%22%2F%3E%0D%0A%3C%2Fg%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-check {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cg%3E%0D%0A%09%0D%0A%09%09%3Crect%20x%3D%2211.9%22%20y%3D%222.1%22%20transform%3D%22matrix%28-0.7072%20-0.7071%200.7071%20-0.7072%2016.2216%2029.719%29%22%20style%3D%22fill%3A%23ffffff%3B%22%20width%3D%224.7%22%20height%3D%2218.8%22%2F%3E%0D%0A%09%0D%0A%09%09%3Crect%20x%3D%220.2%22%20y%3D%2211.3%22%20transform%3D%22matrix%280.7072%200.707%20-0.707%200.7072%2011.5962%20-0.598%29%22%20style%3D%22fill%3A%23ffffff%3B%22%20width%3D%2212.5%22%20height%3D%224.7%22%2F%3E%0D%0A%3C%2Fg%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n\n  /* icons site secondary */\n  .site-icon-clock {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cg%3E%0D%0A%09%3Cpath%20style%3D%22fill%3A%23bebebb%3B%22%20d%3D%22M11.5%2C22c-4.5%2C0-8.6-3-10-7.3c-1.4-4.3%2C0.2-9.2%2C4-11.8c3.7-2.6%2C8.9-2.5%2C12.5%2C0.3%0D%0A%09%09c3.6%2C2.8%2C5%2C7.7%2C3.4%2C12C19.8%2C19.3%2C15.8%2C22%2C11.5%2C22z%20M11.5%2C3.3C7%2C3.3%2C3.3%2C7%2C3.3%2C11.5c0%2C4.5%2C3.7%2C8.2%2C8.2%2C8.2c4.5%2C0%2C8.2-3.7%2C8.2-8.2%0D%0A%09%09C19.7%2C7%2C16%2C3.3%2C11.5%2C3.3z%22%2F%3E%0D%0A%09%3Cpath%20style%3D%22fill%3A%23ffffff%3B%22%20d%3D%22M12.9%2C12.9v-2.9V5.5c0-0.4-0.4-0.8-0.8-0.8h-1.3c-0.4%2C0-0.8%2C0.4-0.8%2C0.8v4.5H6.5%0D%0A%09%09c-0.4%2C0-0.8%2C0.4-0.8%2C0.8v1.3c0%2C0.4%2C0.4%2C0.8%2C0.8%2C0.8h3.5H12.9z%22%2F%3E%0D%0A%3C%2Fg%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-reply {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cg%3E%0D%0A%09%3Cpath%20style%3D%22fill%3A%23bebebb%3B%22%20d%3D%22M20.7%2C20c-0.1%2C0.3-0.3%2C0.7-0.4%2C0.9c-0.1%2C0.1-0.2%2C0.2-0.3%2C0.2c-0.2%2C0-0.4-0.2-0.4-0.4%0D%0A%09%09c0-0.2%2C0-0.4%2C0.1-0.6c0-0.5%2C0.1-1%2C0.1-1.5c0-5.7-3.4-6.7-8.6-6.7H8.4V15c0%2C0.4-0.3%2C0.8-0.8%2C0.8c-0.2%2C0-0.4-0.1-0.5-0.2L1%2C9.4%0D%0A%09%09C0.8%2C9.2%2C0.8%2C9%2C0.8%2C8.8c0-0.2%2C0.1-0.4%2C0.2-0.5l6.1-6.1C7.3%2C2%2C7.5%2C1.9%2C7.7%2C1.9c0.4%2C0%2C0.8%2C0.3%2C0.8%2C0.8v3.1h2.7%0D%0A%09%09c3.9%2C0%2C8.8%2C0.7%2C10.5%2C4.8c0.5%2C1.3%2C0.6%2C2.6%2C0.6%2C4C22.2%2C16.2%2C21.4%2C18.4%2C20.7%2C20z%22%2F%3E%0D%0A%3C%2Fg%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-remove {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cpath%20style%3D%22fill%3A%23bebebb%3B%22%20d%3D%22M19.6%2C17.5l-6-6l6-6c0.3-0.3%2C0.3-0.8%2C0-1.1l-0.9-0.9c-0.3-0.3-0.8-0.3-1.1%2C0l-6%2C6l-6-6%0D%0A%09c-0.3-0.3-0.8-0.3-1.1%2C0L3.4%2C4.4c-0.3%2C0.3-0.3%2C0.8%2C0%2C1.1l6%2C6l-6%2C6c-0.3%2C0.3-0.3%2C0.8%2C0%2C1.1l0.9%2C0.9c0.3%2C0.3%2C0.8%2C0.3%2C1.1%2C0l6-6l6%2C6%0D%0A%09c0.3%2C0.3%2C0.8%2C0.3%2C1.1%2C0l0.9-0.9C19.9%2C18.3%2C19.9%2C17.8%2C19.6%2C17.5z%22%2F%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-back {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cpath%20style%3D%22fill%3A%23bebebb%3B%22%20d%3D%22M11.5%2C20.9l1-1c0.3-0.3%2C0.3-0.7%2C0-0.9L6%2C12.5h14.6c0.4%2C0%2C0.7-0.3%2C0.7-0.7v-1.3c0-0.4-0.3-0.7-0.7-0.7%0D%0A%09H6l6.5-6.5c0.3-0.3%2C0.3-0.7%2C0-0.9l-1-1c-0.3-0.3-0.7-0.3-0.9%2C0l-9.4%2C9.4c-0.3%2C0.3-0.3%2C0.7%2C0%2C0.9l9.4%2C9.4%0D%0A%09C10.9%2C21.2%2C11.3%2C21.2%2C11.5%2C20.9z%22%2F%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-link {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%2223px%22%20height%3D%2223px%22%20viewBox%3D%220%200%2023%2023%22%20style%3D%22enable-background%3Anew%200%200%2023%2023%3B%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cpath%20style%3D%22fill-rule%3Aevenodd%3Bclip-rule%3Aevenodd%3Bfill%3A%23bebebb%3B%22%20d%3D%22M19.3%2C8.2c-1.8%2C1.8-3.7%2C4.6-6.6%2C3.8c0.8-0.8%2C1.6-1.6%2C2.4-2.4%0D%0A%09c0.9-0.9%2C1.8-1.8%2C2.6-2.7c1.9-1.9-0.8-4.8-2.6-2.9c-0.5%2C0.5-0.9%2C0.9-1.4%2C1.4c-0.2%2C0.2-1.4%2C1.1-1.4%2C1.4l0%2C0c-0.8%2C0.8-1.6%2C1.6-2.4%2C2.4%0D%0A%09c-0.4-1.3%2C0-2.8%2C1-3.8c1.3-1.3%2C2.7-3.3%2C4.5-3.8C19.3%2C0.5%2C22.1%2C5.4%2C19.3%2C8.2%22%2F%3E%0D%0A%3Cpath%20style%3D%22fill-rule%3Aevenodd%3Bclip-rule%3Aevenodd%3Bfill%3A%23bebebb%3B%22%20d%3D%22M7.9%2C15.6C7%2C16.5%2C5.5%2C15%2C6.5%2C14.1c1.4-1.4%2C2.7-2.7%2C4.1-4.1%0D%0A%09c0.8-0.8%2C1.5-1.5%2C2.3-2.3c0.2-0.2%2C0.3-0.3%2C0.5-0.5c1-0.6%2C2%2C0.7%2C1.2%2C1.6C12.4%2C11.2%2C10.1%2C13.3%2C7.9%2C15.6z%22%2F%3E%0D%0A%3Cpath%20style%3D%22fill-rule%3Aevenodd%3Bclip-rule%3Aevenodd%3Bfill%3A%23bebebb%3B%22%20d%3D%22M5.8%2C13.4c-0.9%2C0.9-1.8%2C1.8-2.6%2C2.7c-1.9%2C1.9%2C0.8%2C4.8%2C2.6%2C2.9%0D%0A%09c0.9-0.9%2C2-1.8%2C2.8-2.9c0.8-0.8%2C1.6-1.6%2C2.4-2.4c0.4%2C1.3%2C0%2C2.8-1%2C3.8c-1.3%2C1.3-2.7%2C3.3-4.5%2C3.8c-3.8%2C1.1-6.7-3.8-3.9-6.6%0D%0A%09C3.4%2C13%2C5.4%2C10.2%2C8.2%2C11C7.4%2C11.8%2C6.6%2C12.6%2C5.8%2C13.4%22%2F%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-trash {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3C%3Fxml%20version%3D%221.0%22%20encoding%3D%22utf-8%22%3F%3E%0D%0A%3C%21DOCTYPE%20svg%20PUBLIC%20%22-%2F%2FW3C%2F%2FDTD%20SVG%201.1%2F%2FEN%22%20%22http%3A%2F%2Fwww.w3.org%2FGraphics%2FSVG%2F1.1%2FDTD%2Fsvg11.dtd%22%3E%0D%0A%3Csvg%20version%3D%221.1%22%20id%3D%22Layer_1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%09%20width%3D%221792px%22%20height%3D%221792px%22%20viewBox%3D%22590%20-500%201792%201792%22%20enable-background%3D%22new%20590%20-500%201792%201792%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cpath%20fill%3D%22%23bebebb%22%20d%3D%22M1294%2C876V172c0-9.3-3-17-9-23s-13.7-9-23-9h-64c-9.3%2C0-17%2C3-23%2C9s-9%2C13.7-9%2C23v704c0%2C9.3%2C3%2C17%2C9%2C23%0D%0A%09s13.7%2C9%2C23%2C9h64c9.3%2C0%2C17-3%2C23-9S1294%2C885.3%2C1294%2C876z%20M1550%2C876V172c0-9.3-3-17-9-23s-13.7-9-23-9h-64c-9.3%2C0-17%2C3-23%2C9%0D%0A%09s-9%2C13.7-9%2C23v704c0%2C9.3%2C3%2C17%2C9%2C23s13.7%2C9%2C23%2C9h64c9.3%2C0%2C17-3%2C23-9S1550%2C885.3%2C1550%2C876z%20M1806%2C876V172c0-9.3-3-17-9-23%0D%0A%09s-13.7-9-23-9h-64c-9.3%2C0-17%2C3-23%2C9s-9%2C13.7-9%2C23v704c0%2C9.3%2C3%2C17%2C9%2C23s13.7%2C9%2C23%2C9h64c9.3%2C0%2C17-3%2C23-9S1806%2C885.3%2C1806%2C876z%0D%0A%09%20M1262-116h448l-48-117c-4.7-6-10.3-9.7-17-11h-317c-6.7%2C1.3-12.3%2C5-17%2C11L1262-116z%20M2190-84v64c0%2C9.3-3%2C17-9%2C23s-13.7%2C9-23%2C9h-96%0D%0A%09v948c0%2C55.3-15.7%2C103.2-47%2C143.5s-69%2C60.5-113%2C60.5h-832c-44%2C0-81.7-19.5-113-58.5s-47-86.2-47-141.5V12h-96c-9.3%2C0-17-3-23-9%0D%0A%09s-9-13.7-9-23v-64c0-9.3%2C3-17%2C9-23s13.7-9%2C23-9h309l70-167c10-24.7%2C28-45.7%2C54-63s52.3-26%2C79-26h320c26.7%2C0%2C53%2C8.7%2C79%2C26%0D%0A%09s44%2C38.3%2C54%2C63l70%2C167h309c9.3%2C0%2C17%2C3%2C23%2C9S2190-93.3%2C2190-84z%22%2F%3E%0D%0A%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-replay {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2223%22%20height%3D%2223%22%20viewBox%3D%220%200%2023%2023%22%3E%3Cpath%20d%3D%22M16.4%2016.9c-2.8%202.4-7.1%202.3-9.8-.3l-2%202c3.8%203.8%209.9%203.9%2013.9.3l-2.1-2z%22%20fill%3D%22%23b9bcc7%22%2F%3E%3Cpath%20d%3D%22M15.3%205.3l.9.6c3%202.6%203.7%207.2.9%2010.3l2%202c2.3-2.5%203.1-6%202.3-9.3-.9-3.6-3.8-6.4-7.4-7.3-3.8-.8-8.1.7-10.4%204l.1-1.2c0-.2%200-.5-.2-.7-.6-.8-1.7-.4-1.8.4l-.5%204.6c-.1.6.4%201.1%201%201.1h4.6c.9%200%201.3-1.1.8-1.7-.3-.2-.5-.3-.8-.3h-1.2c2-3.2%206.3-4.4%209.7-2.5z%22%20fill%3D%22%23b9bcc7%22%2F%3E%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-alt-download {\n    background-image: url(\'data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2223%22%20height%3D%2223%22%20viewBox%3D%220%200%2023%2023%22%3E%3Cg%20fill-rule%3D%22evenodd%22%20clip-rule%3D%22evenodd%22%20fill%3D%22%23ffffff%22%3E%3Cpath%20d%3D%22M17.3%2017.2c-1.1%200-1.4-1.5-.5-2%20.4-.2%201-.1%201.5-.3.6-.2%201-.6%201.4-1%201.3-1.6.5-4.1-1.4-4.8-.3-.1-.7-.2-1.1-.2-.2%200-.3-.8-.4-1-.4-.9-1-1.8-1.7-2.4-1.8-1.6-4.5-1.8-6.6-.5-1%20.5-1.7%201.4-2.2%202.3-.2.3-.3.7-.4%201-.2.7-.2.5-.8.7-1.9.4-3%202.5-2.2%204.3.4.9%201.3%201.6%202.3%201.8.5.1%201%200%201.3.4.4.4.3%201.1-.2%201.4-.7.6-2.2.1-2.9-.3-1-.5-1.8-1.3-2.3-2.3-.8-1.6-.7-3.5.2-4.9.4-.8%201.1-1.4%201.8-1.9.4-.2.9-.3%201.1-.7.2-.4.5-.8.7-1.2%202.1-3.1%206.2-4.4%209.6-2.9%201.6.7%203%201.9%203.8%203.4.2.4.3.9.7%201.1l1.2.6c.8.5%201.4%201.2%201.8%202%201.6%203.4-1%207.4-4.7%207.4%22%20fill%3D%22%23000000%22%3E%3C%2Fpath%3E%3Cpath%20d%3D%22M15.4%2017.2l-3.1%203.4c-.4.4-1.1.5-1.5%200-.5-.6-1.1-1.2-1.6-1.8-.2-.2-1.2-1.6-1.5-1.6-.9-1%20.7-2.4%201.5-1.4l1.3%201.4v-3.9c0-.4-.1-.9%200-1.3%200-1.3%202.1-1.3%202.1%200v5.2c.5-.5%201-1.4%201.7-1.7.8-.4%201.8.9%201.1%201.7z%22%20fill%3D%22%23000000%22%3E%3C%2Fpath%3E%3C%2Fg%3E%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-random {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2223%22%20height%3D%2223%22%20viewBox%3D%220%200%2023%2023%22%3E%3Cpath%20d%3D%22M5.7%2015.2v-6H8L4%204.6-.1%209.2h2.3v7c0%201.3%201%202.3%202.3%202.3H15l-3-3.2H5.7z%22%20fill%3D%22#ffffff%22%2F%3E%3Cpath%20d%3D%22M20.7%2013.8v-7c0-1.3-1-2.3-2.3-2.3H8l3%203.2h6.3v6H15l4.1%204.6%204.1-4.6h-2.5z%22%20fill%3D%22#ffffff%22%2F%3E%3C%2Fsvg%3E");\n    background-repeat: no-repeat;\n  }\n  .site-icon-visa {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2271%22%20height%3D%2243%22%20viewBox%3D%220%200%2071%2043%22%3E%3Cpath%20d%3D%22M51.9%2017.9c-.2.5-.5%201.4-.5%201.4l-1.5%204H53c-.2-.7-.9-4.2-.9-4.2l-.2-1.2zM69%200H2C.9%200%200%20.9%200%202v39c0%201.1.9%202%202%202h67c1.1%200%202-.9%202-2V2c0-1.1-.9-2-2-2zM11.6%2014h6c.8%200%201.5.3%201.7%201.2l1.3%206.6c-1.3-3.5-4.4-6.4-9.1-7.6l.1-.2zm7.1%2014.4l-3-11.5c2.5%201.5%204.2%203.5%205%205.9l.2%201%203.7-9.8h3.9l-5.9%2014.4h-3.9zm9%200L30.1%2014h3.7l-2.3%2014.4h-3.8zm10.2.2c-1.7%200-3.3-.4-4.2-.8l.5-3.2.5.2c1.2.5%202.4.8%203.8.8.9-.1%202.2-.6%201.9-1.7-.3-.9-1.6-1.3-2.3-1.7-1-.5-2-1.2-2.5-2.2-1-2%20.2-4.2%201.9-5.3%202.2-1.3%205.1-1.3%207.4-.4l-.5%203.1-.3-.2c-1-.4-4.9-1.3-4.9.7%200%20.8%201.2%201.3%201.8%201.6%201%20.5%202.1%201%202.8%202%201.1%201.4.8%203.5-.2%204.8-1.4%201.8-3.6%202.3-5.7%202.3zm16.2-.2s-.3-1.7-.4-2.2h-4.8c-.1.4-.8%202.2-.8%202.2h-3.9l5.5-13.2c.4-.9%201.1-1.2%202-1.2h2.9l3%2014.4h-3.5z%22%20fill%3D%22#ffffff%22%3E%3C%2Fpath%3E%3C%2Fsvg%3E");\n    background-repeat: no-repeat;\n  }\n  .site-icon-amex {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2271%22%20height%3D%2243%22%20viewBox%3D%220%200%2071%2043%22%3E%3Cpath%20d%3D%22M35.5%2023.5h-5l-2%202.2-1.9-2.2h-6.3v6.6h6.2l2-2.2%201.9%202.2h3.1v-2.2h2c1.5%200%202.7-.6%202.7-2.2%200-1.7-1.3-2.2-2.7-2.2zm-9.9%205.2h-3.9v-1.3h3.5V26h-3.5v-1.2h4l1.7%201.9-1.8%202zm6.3.8l-2.4-2.7%202.4-2.6v5.3zm3.6-3h-2v-1.7h2.1c1.2%200%201.2%201.7-.1%201.7zm9.4-7.8h1.8l.6-1.5h3.3l.6%201.5h3.2v-5l3%205h2.2v-6.6H58v4.6l-2.8-4.6h-2.4v6.2l-2.7-6.2h-2.4l-2.2%205.2H45c-1.6%200-2.1-2.3-1.1-3.4.3-.3.6-.4%201.2-.4h1.5v-1.4h-1.5c-2.1%200-3.2%201.3-3.2%203.3-.1%202%201%203.3%203%203.3zm4.1-5.5l1.1%202.6h-2.2l1.1-2.6zm-15.2%203.1h1.7c.6%200%20.9.1%201.1.3.3.3.2.9.2%201.3v.8h1.6v-1.3c0-.6%200-.9-.3-1.2-.1-.2-.4-.4-.8-.6.9-.4%201.3-1.4%201-2.3-.3-1.2-1.7-1.2-2.6-1.2h-3.6v6.6h1.6v-2.4zm0-2.8h1.9c.3%200%20.6%200%20.7.1.5.2.4%201%200%201.2-.2.1-.5.1-.8.1h-1.9v-1.4zm-2.7%203.8h-3.7V16H31v-1.3h-3.6v-1.2h3.7v-1.4h-5.3v6.6h5.3zM23%2018.7h1.6v-6.6H22l-1.9%204.5-2.1-4.5h-2.6v6.2l-2.7-6.2h-2.3l-2.8%206.6h1.7l.6-1.5h3.3l.6%201.5H17v-5.2l2.3%205.2h1.4l2.3-5.2v5.2zm-12.5-2.9l1.1-2.6%201.1%202.6h-2.2zm35.6%2014.3h5.3v-1.4h-3.7v-1.3h3.6V26h-3.6v-1.2h3.7v-1.3h-5.3zM61.8%2026H61c-.3%200-.6%200-.8-.2-.2-.2-.1-.6%200-.7l.1-.1c.2-.1.3-.1.6-.1h2.9v-1.4h-3.2c-1%200-1.6.4-1.9.8l-.1.1-.1.2c-.1.1-.1.3-.1.4v.7l.1.6c0%20.1%200%20.1.1.2%200%20.1.1.2.1.2l.1.1.1.1s.1%200%20.1.1c.1.1.2.1.3.1%200%200%20.1%200%20.1.1.7.2%201.4.2%202.2.2%201%200%20.9%201.2-.1%201.2h-3V30h3c1.1%200%202.2-.4%202.4-1.6.4-1.6-.6-2.3-2.1-2.4zm-16.5-.8c0-.8-.3-1.2-.8-1.5-.5-.3-1.1-.3-1.9-.3H39V30h1.6v-2.4h1.7c1.1%200%201.4.5%201.4%201.6v.8h1.6v-1.3c0-.8-.2-1.5-1-1.8.3-.1%201-.6%201-1.7zm-2.1%201c-.2.1-.5.1-.8.1h-1.9v-1.5h1.9c.3%200%20.6%200%20.8.1.5.3.4%201%200%201.3-.2.1.2-.2%200%200zM69%200H2C.9%200%200%20.9%200%202v39c0%201.1.9%202%202%202h67c1.1%200%202-.9%202-2V2c0-1.1-.9-2-2-2zM18.2%2018.9V20h-5.3l-.6-1.5h-1.4l-.6%201.5H5.5l4-9.2h4.1l.5%201.1v-1.1H19l1.1%202.4%201-2.4h15.3c.7%200%201.3.1%201.8.5v-.5h4.2v.5c.7-.4%201.6-.5%202.6-.5h6.1l.6%201.1v-1.1h4.5l.6%201.1v-1.1h4.4V20h-4.4l-.8-1.4V20h-5.5l-.6-1.5h-1.4l-.8%201.5h-2.9c-1.1%200-2-.3-2.5-.6v.6h-6.8v-2.1c0-.3-.1-.3-.2-.3H35V20H21.9v-1.1l-.5%201.1h-2.8l-.4-1.1zm46.9%2011c-.7%201.3-2%201.5-3.3%201.5h-4.4v-.6c-.5.4-1.3.6-2.1.6h-13v-2.1c0-.3%200-.3-.3-.3h-.2v2.4h-4.2v-2.5c-.7.3-1.5.3-2.2.3h-.5v2.2h-5.1L28.5%2030l-1.3%201.4h-8.3v-9.2h8.5l1.2%201.4%201.3-1.4h5.7c.7%200%201.7.1%202.2.5v-.5h5.1c.5%200%201.5.1%202.1.5v-.5h7.7v.5c.4-.4%201.2-.5%201.9-.5h4.3v.5c.4-.3%201-.5%201.8-.5h4.5v3.3h-.9c1.3.9%201.6%203%20.8%204.4-.4.7.3-.5%200%200zM39.4%2012.1H41v6.6h-1.6zM55.6%2026h-.8c-.5%200-1.5-.5-.7-1%20.2-.1.3-.1.6-.1h2.9v-1.4h-3.2c-1.5%200-2.6%201.2-2.1%202.7.4%201.3%202.1%201.2%203.2%201.3%201%200%20.9%201.2-.1%201.2h-3.1v1.4h3.1c1.1%200%202.2-.4%202.4-1.6.4-1.7-.7-2.4-2.2-2.5z%22%20fill%3D%22#ffffff%22%3E%3C%2Fpath%3E%3C%2Fsvg%3E");\n    background-repeat: no-repeat;\n  }\n  .site-icon-mc {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2271%22%20height%3D%2243%22%20viewBox%3D%220%200%2071%2043%22%3E%3Cpath%20d%3D%22M36.2%2022.5c.2-.9.3-2.2-.8-2.7-1.4-.7-3%20.1-3.6%201.5-.3.7-.3%201.5-.1%202.2.6%201.4%202.7%201.8%204%201.1l.2-1.1c-.9.2-3.1.5-2.9-1h3.2zm-1.8-1.9c.6%200%20.7.4.6.9h-1.8c.2-.6.6-.9%201.2-.9zm-3.8%202.3l.4-2.1h.7l.3-1.1h-.8l.3-1.4h-1.4l-1%204.9c-.3%201.3%201%202.1%202.2%201.4l.1-1c-.8%200-.8-.3-.8-.7zm8-2.7l-.1-.5h-1.2l-1%205.1h1.4l.4-2.3c.4-1.5%201.1-1.4%201.4-1.4l.4-1.4c-.7%200-1.1.2-1.3.5zm-10.2%201.9c-.2-.1-2.1-1-1-1.3.3%200%201%200%201.4.1l.3-1c-1.3-.5-3.7-.2-3.6%201.6.1%201.1%201.3.9%201.8%201.6.7.9-1.9.6-2.2.5l-.1%201c.8.2%202%20.3%202.8%200%201-.3%201.7-1.9.6-2.5-.3-.2.3.1%200%200zm14.5-2.4c.5%200%20.7%200%201.1.3l.9-.9c-1.7-1.9-4.5-.3-5.1%201.8-.7%202.7%201.4%204.8%204%203.6l-.4-1.3c-2.8%201-2.8-3.5-.5-3.5zM23%2019.6c-.4%200-.9%200-1.8.2l-.2%201.1c.6-.1%202.9-.7%202.6.6-1.1.1-2.2-.1-2.9%201-1.1%201.6.8%203.2%202.3%202.1l.2.3h1.3L25%2022c.2-.7.4-2.4-2-2.4zm.3%203.4c-.1.5-1.3%201-1.5.3-.3-.9%201.1-1%201.6-.9l-.1.6zm27.9-2.9l-.2-.5h-1.2l-1%205.2h1.4l.4-2.4c.4-1.5%201.1-1.5%201.4-1.5l.4-1.4c-.6.1-1%20.4-1.2.6zM69%200H2C.9%200%200%20.9%200%202v39c0%201.1.9%202%202%202h67c1.1%200%202-.9%202-2V2c0-1.1-.9-2-2-2zM45.1%2035.5c-3.4%200-6.4-1.2-8.8-3.1%201.8-1.8%203.2-4%204-6.5h-1.2c-.7%202.2-2%204.2-3.7%205.8-1.7-1.6-2.9-3.6-3.6-5.8h-1.2c.8%202.5%202.1%204.7%204%206.5-4%203.2-9.5%204.1-14.3%202-4-1.7-7.1-5.4-8.1-9.6C10%2015.4%2018%206.4%2027.5%207.6c2.6.3%205.1%201.4%207.1%203-1.8%201.8-3.2%204-4%206.5h1.2c.7-2.2%202-4.2%203.6-5.8%201.7%201.6%202.9%203.6%203.7%205.8h1.2c-.8-2.5-2.2-4.7-4-6.5%207.6-6.2%2019.4-2.5%2022.2%206.8%202.7%208.9-4.1%2018.1-13.4%2018.1zm7.2-13.7c-.5%201.6.5%203%201.5%203%20.5%200%20.9-.2%201.2-.4l.2.4h1.5l1.2-6.6h-1.5l-.4%202c-.7-.5-1.2-.8-2-.6-1%20.3-1.5%201.3-1.7%202.2-.5%201.6.7-2.4%200%200zm3.1-.8c.2.2.2.8.1%201.3-.2.4-.4.9-.9%201-.6.1-1-.4-1-1%20.1-.6%201.1-2.2%201.8-1.3.2.3-.4-.5%200%200zm-36.9-2.5l-1.7%203.8-.3-3.8h-2.2l-1.2%206.3h1.4l.8-4.5.4%204.5h1.4l1.9-4.4-.8%204.4h1.5l1.1-6.3zm28.4%201.1c-.4%200-.9%200-1.8.2l-.2%201.1c.6-.1%202.9-.7%202.6.6-1.1.1-2.2-.1-3%201-1.1%201.6.8%203.2%202.3%202.1l.2.3h1.3l.6-2.8c.1-.8.4-2.5-2-2.5zm.2%203.4c-.1.5-1.3%201-1.5.3-.3-.9%201.1-1%201.6-.9l-.1.6z%22%20fill%3D%22#ffffff%22%3E%3C%2Fpath%3E%3C%2Fsvg%3E");\n    background-repeat: no-repeat;\n  }\n  .site-icon-discover {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2271%22%20height%3D%2243%22%20viewBox%3D%220%200%2071%2043%22%3E%3Cpath%20d%3D%22M6.5%2012.1v6.5s4.2.5%204.2-3.2v-.1c-.1-3.6-4.2-3.2-4.2-3.2zM69%200H2C.9%200%200%20.9%200%202v39c0%201.1.9%202%202%202h67c1.1%200%202-.9%202-2V2c0-1.1-.9-2-2-2zm-9.8%2010.4h3.2c1.7%200%203.1%201.2%203.1%202.9%200%201.3-.8%202.5-2.1%202.9l3%204.2H64l-2.7-4h-.2v4h-1.9v-10zm-6.8%200H58v1.8h-3.5v2.2h3.4v1.8h-3.4v2.5H58v1.8h-5.5V10.4zm-8.2%200l2.7%206.5%202.6-6.5h2.1l-4.2%2010.2h-1.2L42%2010.4h2.2zm-31.5%205c0%203.2-2.9%204.9-4.8%204.9H4.6v-9.9h3.3c1.9%200%204.8%201.8%204.8%205zm2.8%204.9h-2v-10h2v10zm4.2.3c-2.5%200-3.6-1.9-3.6-1.9l1.2-1.2c.7%201%203.1%202.2%203.5.4.5-1.8-1.8-1.9-2.9-2.4-1.7-.9-1.8-3.4-.2-4.6%201.5-1.1%203.5-.6%204.8.6l-1.1%201.3c-.7-.8-3-1.2-2.9.4%200%201%202%201.3%202.7%201.7%203.2%201.3%201.6%205.7-1.5%205.7zm11.6-.6c-3.4%201.8-7.7-.8-7.7-4.6%200-3.9%204.2-6.4%207.7-4.6v2.4c-2.1-2.1-5.8-.8-5.8%202.3%200%203%203.7%204.4%205.8%202.3V20zm6-9.8c2.8%200%205.2%202.3%205.2%205.2%200%202.8-2.3%205.2-5.2%205.2-2.8%200-5.2-2.3-5.2-5.2%200-2.8%202.4-5.2%205.2-5.2zM68%2040H32.8C45%2036.8%2057.2%2032.4%2068%2025.7V40zm-4.4-26.5v-.1c0-.6-.5-1.4-1.3-1.4h-1.1v2.9h1.1c.8%200%201.3-.7%201.3-1.4z%22%20fill%3D%22#ffffff%22%3E%3C%2Fpath%3E%3C%2Fsvg%3E");\n    background-repeat: no-repeat;\n  }\n  .site-icon-jcb {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2271%22%20height%3D%2243%22%20viewBox%3D%220%200%2071%2043%22%3E%3Cpath%20d%3D%22M69%200H2C-.1%200%200%202.1%200%203.5v37.4C0%2042.6%201.4%2043%202.7%2043h66c1.4%200%202.4-.8%202.4-2.3V3c-.1-1.4-.3-3-2.1-3zM28.3%2030.8c0%202.4-1.9%204.4-4.3%204.6-2.1.3-4.3%200-6.5%200-.9%200-.8.1-.8-.8v-9c2.8.7%209.5%201.7%2010.4-2.2.3-1.2.1-2.6.1-3.8v-2.1c0-.4.1-.5-.3-.5h-3.7v3.4c0%201.5.3%203.5-1.5%204.2-.8.3-1.7.2-2.5%200-.9-.2-1.8-.5-2.6-.9-.1-.1%200-.5%200-.6v-6.4c0-2.1-.4-4.7.5-6.7.5-1%201.3-1.6%202.2-2.1%201.4-.6%208.7-.5%208.7-.3v15.1c.3%202.7.3%205.4.3%208.1zm13%20.1c0%202.4-2%204.4-4.4%204.6-.9.1-2%200-2.9%200h-4c-.4%200-.2-1.3-.2-1.6v-9.1c1.4%201.2%203.4%201.5%205.2%201.5%201.1%200%202.1%200%203.1-.2l1.2-.2c.1%200%20.6-.1.7-.2.2-.3%200-1.5%200-1.9v-.1c-1.4.7-3.4%201.4-5%201-3-.4-3.4-4.8-.8-6.1%201.9-.7%204.1-.1%205.8.7v-1.8c0-.3-.3-.3-.5-.3-.4-.1-.8-.2-1.2-.2-.9-.1-1.9-.2-2.8-.2-1.8%200-3.7.2-5.3%201.2l-.5.3v-3.5c0-1.2-.2-2.6.1-3.8.3-1.3%201.2-2.2%202.2-2.9.7-.4%201.4-.6%202.2-.6%202.3-.1%204.6%200%206.8%200%20.3%200%20.3%2021.2.3%2023.4zm12.9%200c0%202.5-2.1%204.5-4.5%204.6-2.3.1-4.6%200-6.9%200-.2%200-.1-6.1-.1-6.7V26h7c.4%200%20.7%200%201.1-.1%201.3-.3%202.5-1.4%202-2.9-.3-.9-1.3-1.4-2.3-1.6h-.3l1.1-.4c.7-.4%201.1-1.2%201-2-.1-1.2-1.2-1.8-2.3-1.9-1.8-.2-3.8%200-5.6%200h-1.6c-.3%200-.2-.3-.2-.6v-4.1c0-2.5%201.8-4.6%204.3-4.8%202.2-.2%204.4%200%206.6%200%20.6%200%20.7-.1.7.5v22.8zm-6.5-8.8c1.5.2%201.6%202.5%200%202.5h-2.3c-.1%200-.4.1-.4-.1v-2.4c0-.2.4-.1.5-.1.7.1%201.4.1%202.2.1zm-2.8-1.4v-2c0-.6.9-.3%201.3-.3.5%200%201.5-.2%201.9.3.7.5.4%201.7-.4%202-.8.2-1.9%200-2.8%200z%22%20fill%3D%22#ffffff%22%3E%3C%2Fpath%3E%3C%2Fsvg%3E");\n    background-repeat: no-repeat;\n  }\n  .site-icon-calendar {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2223%22%20height%3D%2223%22%20viewBox%3D%220%200%2023%2023%22%3E%3Cpath%20d%3D%22M19.3%202.8h-1.1v2.5h-4V2.8H8.8v2.5h-4V2.8H3.7c-1.4%200-2.5%201.1-2.5%202.5v14.9c0%201.4%201.1%202.5%202.5%202.5h15.6c1.4%200%202.5-1.1%202.5-2.5V5.3c0-1.4-1.1-2.5-2.5-2.5zm0%2017.4H3.7v-9.9h15.6v9.9zM7.7.3H5.9v4.2h1.7V.3zm9.4%200h-1.7v4.2h1.7V.3z%22%20fill%3D%22#ffffff%22%2F%3E%3Cg%20fill%3D%22#ffffff%22%3E%3Cpath%20d%3D%22M5.1%2011.7h1.4v1.4H5.1zM5.1%2014.5h1.4v1.4H5.1zM5.1%2017.4h1.4v1.4H5.1zM8%2011.7h1.4v1.4H8zM8%2014.5h1.4v1.4H8zM8%2017.4h1.4v1.4H8zM10.8%2011.7h1.4v1.4h-1.4zM10.8%2014.5h1.4v1.4h-1.4zM10.8%2017.4h1.4v1.4h-1.4zM13.6%2011.7H15v1.4h-1.4zM13.6%2014.5H15v1.4h-1.4zM13.6%2017.4H15v1.4h-1.4zM16.5%2011.7h1.4v1.4h-1.4zM16.5%2014.5h1.4v1.4h-1.4zM16.5%2017.4h1.4v1.4h-1.4z%22%2F%3E%3C%2Fg%3E%3C%2Fsvg%3E");\n    background-repeat: no-repeat;\n  }\n  .site-icon-credit {\n    background-image: url("data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2271%22%20height%3D%2243%22%20viewBox%3D%220%200%2071%2043%22%3E%3Cstyle%20type%3D%22text%2Fcss%22%3Ecircle%2C%20ellipse%2C%20line%2C%20path%2C%20polygon%2C%20polyline%2C%20rect%2C%20text%20%7B%20fill%3A%20#ffffff%20!important%3B%20%7D%3C%2Fstyle%3E%3Cpath%20d%3D%22M71%202c0-1.1-.9-2-2-2h-67c-1.1%200-2%20.9-2%202v8.6h71v-8.6zm-71%2018.4v20.6c0%201.1.9%202%202%202h67c1.1%200%202-.9%202-2v-20.6h-71zm17.7%2014.9h-9.6v-3.6h9.6v3.6zm17.3%200h-13.9v-3.6h13.9v3.6z%22%20fill%3D%22%23fff%22%2F%3E%3C%2Fsvg%3E");\n    background-repeat: no-repeat;\n  }\n  .site-icon-search {\n    background-image: url(\'data:image/svg+xml;charset=US-ASCII,%3Csvg%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20width%3D%2223%22%20height%3D%2223%22%20viewBox%3D%220%200%2023%2023%22%3E%3Cpath%20d%3D%22M21.8%2019.5L17.4%2015c-.6.9-1.5%201.7-2.4%202.4l4.5%204.4c1.4%201.6%203.8-.8%202.3-2.3z%22%20fill%3D%22%23bebebb%22%3E%3C%2Fpath%3E%3Cpath%20d%3D%22M15%2017.4c4.4-3%205.2-9.4%201.8-13.4C13.8.5%208.6-.3%204.7%202.3.9%204.7-.4%209.9%201.7%2014c2.2%204.4%207.7%206.2%2012.1%204.1.4-.2.8-.5%201.2-.7zm-5.2-1.7C6.7%2015.7%204%2013%204%209.8%204%206.7%206.7%204%209.8%204c3.2%200%205.8%202.7%205.8%205.8.1%203.2-2.6%205.9-5.8%205.9z%22%20fill%3D%22%23bebebb%22%3E%3C%2Fpath%3E%3Cpath%20d%3D%22M9.8%202.2c4.2%200%207.7%203.4%207.7%207.7%200%201.5-.5%203.1-1.4%204.3-.5.7-1%201.3-1.7%201.8l-.3.2c-1.2.9-2.8%201.3-4.3%201.3-4.2%200-7.7-3.4-7.7-7.7.1-4.2%203.5-7.6%207.7-7.6m0-1.5C4.7.7.6%204.8.6%209.9s4.1%209.2%209.2%209.2c1.8%200%203.7-.6%205.2-1.6.9-.6%201.7-1.4%202.3-2.4%201.1-1.5%201.6-3.3%201.6-5.2.1-5.1-4-9.2-9.1-9.2z%22%20fill%3D%22%23bebebb%22%3E%3C%2Fpath%3E%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n  }\n  .site-icon-is-searching {\n     background-position: center center !important;\n     background-image: url(\'data:image/svg+xml;charset=utf-8,%3Csvg%20version%3D%221.1%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%20xmlns%3Axlink%3D%22http%3A%2F%2Fwww.w3.org%2F1999%2Fxlink%22%20x%3D%220px%22%20y%3D%220px%22%0D%0A%20%20viewBox%3D%220%200%2040%2040%22%20xml%3Aspace%3D%22preserve%22%3E%0D%0A%3Cpath%20opacity%3D%220.2%22%20fill%3D%22%23323232%22%20d%3D%22M20.201%2C5.169c-8.254%2C0-14.946%2C6.692-14.946%2C14.946c0%2C8.255%2C6.692%2C14.946%2C14.946%2C14.946%0D%0A%20%20s14.946-6.691%2C14.946-14.946C35.146%2C11.861%2C28.455%2C5.169%2C20.201%2C5.169z%20M20.201%2C31.749c-6.425%2C0-11.634-5.208-11.634-11.634%0D%0A%20%20c0-6.425%2C5.209-11.634%2C11.634-11.634c6.425%2C0%2C11.633%2C5.209%2C11.633%2C11.634C31.834%2C26.541%2C26.626%2C31.749%2C20.201%2C31.749z%22%2F%3E%0D%0A%3Cpath%20fill%3D%22%23323232%22%20d%3D%22M26.013%2C10.047l1.654-2.866c-2.198-1.272-4.743-2.012-7.466-2.012h0v3.312h0%0D%0A%20%20C22.32%2C8.481%2C24.301%2C9.057%2C26.013%2C10.047z%22%3E%0D%0A%20%20%3CanimateTransform%20attributeType%3D%22xml%22%0D%0A%20%20%20%20attributeName%3D%22transform%22%0D%0A%20%20%20%20type%3D%22rotate%22%0D%0A%20%20%20%20from%3D%220%2020%2020%22%0D%0A%20%20%20%20to%3D%22360%2020%2020%22%0D%0A%20%20%20%20dur%3D%220.5s%22%0D%0A%20%20%20%20repeatCount%3D%22indefinite%22%2F%3E%0D%0A%20%20%3C%2Fpath%3E%0D%0A%3C%2Fsvg%3E\');\n     background-repeat: no-repeat;\n     background-size: contain;\n     z-index: 999;\n     width: 80px;\n     height: 80px;\n     margin: 0 auto;\n  }\n  .site-icon-report {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3Csvg%20viewBox%3D%220%200%201792%202031%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%3E%3Cpath%20d%3D%22M1122.987%20238.933L1075.2%200H0v2030.933h238.933v-836.266h669.014l47.786%20238.933H1792V238.933z%22%20fill%3D%22%23bebebb%22%20fill-rule%3D%22evenodd%22%2F%3E%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n    background-size: contain;\n  }\n  .site-icon-block {\n    background-image: url(\'data:image/svg+xml;charset=utf-8,%3Csvg%20viewBox%3D%220%200%201792%201792%22%20xmlns%3D%22http%3A%2F%2Fwww.w3.org%2F2000%2Fsvg%22%3E%3Cpath%20d%3D%22M896%201612.8c-165.76%200-318.08-56.448-439.04-151.424L1461.376%20456.96A708.02%20708.02%200%200%201%201612.8%20896c0%20396.032-320.768%20716.8-716.8%20716.8M179.2%20896c0-396.032%20320.768-716.8%20716.8-716.8%20165.76%200%20318.08%2056.448%20439.04%20151.424L330.624%201335.04A708.02%20708.02%200%200%201%20179.2%20896M896%200C401.408%200%200%20401.408%200%20896s401.408%20896%20896%20896%20896-401.408%20896-896S1390.592%200%20896%200%22%20fill%3D%22%23ffffff%22%20fill-rule%3D%22evenodd%22%20fill-opacity%3D%22.6%22%2F%3E%3C%2Fsvg%3E\');\n    background-repeat: no-repeat;\n    background-size: contain;\n  }\n</style>\n\n\n  <style>\n\n    /*\r\n * This file is for use on the Vimeo VHX platform.\r\n *\r\n * 1. prototype changes in browser dev tools using Vimeo VHX demo site\r\n * 2. edit this file locally\r\n * 3. git commit and push to the remote repo\r\n * 4. share for code review\r\n * 5. paste into "CUSTOM STYLES" field in the Vimeo VHX theme demo admin\r\n * 6. share for approval by Criterion team\r\n * 7. once approved and code reviewed, paste into "CUSTOM STYLES" field in the Vimeo VHX theme production admin\r\n *\r\n * Note for Vimeo engineers: please share any updates to this file to the Criterion.com team.\r\n */\r\n\r\n @import url(\'https://fonts.googleapis.com/css?family=Montserrat:400,500,600,700,900|Spectral:400,500,600,700\');\r\n\r\n p, div, span, a, h1, h2, h3, h4, h5, h6, .site-font-primary-family, .site-font-secondary-family {\r\n     font-family: \'Montserrat\', sans-serif !important;\r\n }\r\n\r\n div {\r\n     font-family: \'Montserrat\', sans-serif !important;\r\n }\r\n\r\n .browse-row h1 a {\r\n     font-family: \'Montserrat\', sans-serif !important;\r\n     font-weight: 700;\r\n     text-transform: uppercase;\r\n }\r\n\r\n .head-1, .head-2 {\r\n     font-weight: 700;\r\n }\r\n\r\n .browse-row h1.horizontal-row-header a {\r\n     font-size: 1rem !important;\r\n }\r\n\r\n .browse-link {\r\n     font-family: \'Montserrat\', sans-serif !important;\r\n }\r\n\r\n /*\r\n * Nav\r\n */\r\n\r\n .navigation-inner a {\r\n    opacity: 1;\r\n    font-family: \'Montserrat\', sans-serif !important;\r\n    -webkit-transition: all 0.3s ease;\r\n    -moz-transition: all 0.3s ease;\r\n    -ms-transition: all 0.3s ease;\r\n    -o-transition: all 0.3s ease;\r\n    transition: all 0.3s ease;\r\n}\r\n\r\n .navigation-inner.is-ready a {\r\n    opacity: 1;\r\n}\r\n\r\n  /*\r\n  * Other\r\n  */\r\n\r\n h1.gigantic.uppercase { text-transform: none !important; }\r\n .percentage-bar-container { display: none !important; }\r\n\r\n .slide-content {\r\n     bottom: 40px;\r\n }\r\n\r\n .slide-title {\r\n     font-size: 28px;\r\n     font-weight: 600;\r\n }\r\n\r\n .browse-item-title .subtext-container > div.site-font-primary-color.truncate {\r\n     white-space: normal;\r\n }\r\n\r\n .collection-description .read-more-wrap {\r\n     max-height: 120px !important;\r\n }\r\n\r\n .site-header nav a {\r\n     font-weight: 600;\r\n     letter-spacing: 0;\r\n     padding-left: 25px;\r\n }\r\n\r\n div.slide-subtitle {\r\n     display: none;\r\n }\r\n\r\n .slide-content {\r\n     bottom: 40px;\r\n }\r\n\r\n .collection-title {\r\n     font-weight: 600;\r\n }\r\n\r\n\r\n #watch-info h5 {\r\n     display: none;\r\n }\r\n\r\n .head.primary strong {\r\n     font-weight: 600;\r\n }\r\n\r\n .item-type-movie .play-outer-rectangle {\r\n     display: none;\r\n }\r\n\r\n /*\r\n  * Borders\r\n  */\r\n\r\n .border-bottom,\r\n .border-bottom-light {\r\n     border-bottom: none !important;\r\n }\r\n\r\n .border-top,\r\n .border-top-light {\r\n     border-top: none !important;\r\n }\r\n\r\n .horizontal-row {\r\n     border-top: none !important;\r\n     border-bottom: none !important;\r\n }\r\n\r\n /*\r\n  * Buttons\r\n  */\r\n\r\n .button {\r\n     border-radius: 0px;\r\n     border: none;\r\n     font-weight: 600;\r\n }\r\n\r\n .slide-button,\r\n .slide-button--alt {\r\n     border-radius: 0px !important;\r\n }\r\n\r\n .btn-site-primary,\r\n .btn-gray:link,\r\n .btn-dropdown-gray:link,\r\n .btn-gray:active,\r\n .btn-dropdown-gray:active,\r\n .btn-gray:visited,\r\n .btn-dropdown-gray:visited {\r\n     border: 1px solid #fff !important;\r\n     background-color: #fff !important;\r\n     color: #252525;\r\n     border-radius: 0px;\r\n     text-transform: uppercase;\r\n     line-height: 46px;\r\n     font-size: 13px;\r\n     transition: all 300ms ease-out;\r\n }\r\n .site-settings-billing-panel .btn-gray:link {\r\n   color: #252525 !important;\r\n   line-height: 33px;\r\n }\r\n .btn-site-primary,\r\n .btn-site-secondary {\r\n     line-height: 33px;\r\n }\r\n\r\n .btn-site-secondary,\r\n .btn-transparent,\r\n .btn-dropdown-transparent {\r\n     border: 1px solid #313131 !important;\r\n     background-color: #313131 !important;\r\n     color: #fff !important;\r\n     transition: all 300ms ease-out;\r\n     font-size: 13px;\r\n }\r\n\r\n .btn-gray:hover,\r\n .btn-dropdown-gray:hover,\r\n .btn-site-primary:hover,\r\n .btn-dropdown-transparent:hover,\r\n .btn-transparent:hover,\r\n .btn-site-secondary:hover {\r\n     border: 1px solid #CDCDCB !important;\r\n     background-color: #CDCDCB !important;\r\n }\r\n\r\n .icon-play-white {\r\n     background-image: url("data:image/svg+xml;charset=utf-8,%3Csvg xmlns=\'http://www.w3.org/2000/svg\' width=\'23\' height=\'23\' viewBox=\'0 0 23 23\'%3E%3Ctitle%3Eicons-ui-set copy%3C/title%3E%3Cpath d=\'M20.49 10.52L4.13.39C2.94-.4 2 .18 2 1.67v19.66c0 1.48 1 2.06 2.17 1.28l16.32-10.13a1.15 1.15 0 0 0 0-1.96z\' fill=\'%23252525\'/%3E%3C/svg%3E");\r\n }\r\n\r\n .btn-black:link,\r\n .btn-dropdown-black:link,\r\n .btn-black:active,\r\n .btn-dropdown-black:active,\r\n .btn-black:visited,\r\n .btn-dropdown-black:visited {\r\n     border: 1px solid #B9BCC6;\r\n     background-color: #B9BCC6;\r\n     color: #000;\r\n     border-radius: 0px;\r\n     text-transform: uppercase;\r\n     line-height: 46px;\r\n     font-size: 13px;\r\n }\r\n\r\n .btn-transparent,\r\n .btn-dropdown-transparent,\r\n .btn-transparent:link,\r\n .btn-dropdown-transparent:link,\r\n .btn-transparent:active,\r\n .btn-dropdown-transparent:active,\r\n .btn-transparent:visited,\r\n .btn-dropdown-transparent:visited {\r\n     border: none;\r\n     border-radius: 0px;\r\n     text-transform: uppercase;\r\n     line-height: 28px;\r\n     font-size: 13px;\r\n     padding: 10px 16px;\r\n     background-position: 16px 18px !important;\r\n }\r\n\r\n select.btn-dropdown-transparent {\r\n     padding: 0 20px 0 40px;\r\n }\r\n\r\n .coin {\r\n     border-radius: 4px;\r\n }\r\n\r\n /*\r\n  * Search link + collection card\r\n  */\r\n\r\n .browse-item-subtext {\r\n     display: none !important;\r\n }\r\n\r\n /*\r\n  * Search results\r\n  */\r\n\r\n .item-type-movie.js-collection-item .duration-container {\r\n     display: none !important;\r\n }\r\n\r\n /*\r\n  * Carousel\r\n  */\r\n\r\n .slide-content {\r\n     transform: translate3d(0,0,0);\r\n     top: auto;\r\n     bottom: 10%;\r\n     padding-left: 57px;\r\n }\r\n\r\n .slide-image-container:after,\r\n .slide-image-container:before {\r\n     display: none !important;\r\n }\r\n\r\n @media screen and (max-width: 39.9375em) {\r\n     .slide-content {\r\n         padding-left: 40px;\r\n         bottom: 13%;\r\n     }\r\n     .slide-bg--mobile .slide-bg-img {\r\n         float: left;\r\n         width: 100%;\r\n         height: auto;\r\n     }\r\n\r\n }\r\n\r\n /*\r\n  * Browse\r\n  */\r\n\r\n .browse-item-card.item-type-movie .duration-container {\r\n   display: none !important;\r\n }\r\n\r\n /*\r\n  * Collection Pages\r\n  */\r\n\r\n .vertical-bottom {\r\n     position: absolute;\r\n     bottom: 0;\r\n }\r\n\r\n .cover-art.columns {\r\n     padding-right: 0px !important;\r\n }\r\n\r\n select.js-sort-items,\r\n h2.collection-stats,\r\n h2.content-label,\r\n .extras-header {\r\n     display: none !important;\r\n }\r\n\r\n .browse-item-title .browse-image-container:before {\r\n     width: 102%;\r\n     height: 102%;\r\n }\r\n\r\n .browse-image-container:before {\r\n     background-image: none !important;\r\n }\r\n\r\n .media-identifier.media-episode {\r\n     display: none;\r\n }\r\n\r\n .read-more-trigger {\r\n     font-weight: bold;\r\n }\r\n\r\nsection.site-settings-purchases-panel .subscription-button {\r\n    background-color: #222 !important\r\n}\r\n\r\n.platforms-modal--container .platform-modal-devices-list-tv li:last-child {\r\n    display: none;\r\n}\r\n\r\n .collection-actions .margin-bottom-small {\r\n    margin-bottom: 0.9375rem !important;\r\n }\r\n .collection-actions .margin-right-small {\r\n    margin-right: 0.9375rem !important;\r\n }\r\n .collection-actions .btn-site-primary {\r\n    line-height: 46px;\r\n }\r\n\r\n/* Fix for button issue on pause / cancellation modal */\r\nbody.settings.is-modal-active .btn-site-secondary,\r\nbody.settings.is-modal-active .btn-site-primary {\r\n    line-height: 38px !important;\r\n    color: #252525 !important;\r\n}\r\n\r\n.site-header .user-dropdown {\r\n    width: 190px;\r\n}\r\n\r\n.tooltip.background-white span.media-count, .tooltip.background-white span.media-identifier, .tooltip.background-white span.media-seperator { display: none !important; }\r\n\r\n.tooltip.background-white .tooltip-item-title { margin-bottom: 15px; }\r\n\r\ndiv[id^="collection-tooltip"] .tooltip-item-title { margin-bottom: 0px !important; }\n\n\n    html.wf-loading { opacity: 0; }\n    html.wf-active, html.is-loaded { opacity: 1 !important; }\n  </style>\n\n  <script>\n  var Tracking = undefined;\n  window.Page = undefined;\n  var Segment_ID = undefined;\n  window._current_user = undefined;\n  window._current_site = undefined;\n</script>\n\n\n  <script>\n    window._vhx = {\n      environment: \'production\',\n      site: { id: \'59054\', domain: \'www.criterionchannel.com\', host: \'criterionchannel.vhx.tv\', previewUrl: \'https://www.criterionchannel.com/login\' },\n      geo:  { country: \'us\' },\n      product: {id: \'\' }\n    };\n\n    window._current_user = {"id":null,"email":null,"name":null,"city":null,"state":null,"country":null,"external_user_id":null};\n\n    window._current_site = {"id":59054,"facebook_pixel_codes":[{"pixel_code":"980569266079632","product_id":null}],"twitter_pixel_codes":[],"adwords_pixel_codes":[],"key":"criterionchannelchartersu","title":"The Criterion Channel","subdomain":"criterionchannel","made_for_kids":false,"made_for_kids_tracking_blocked":false,"algolia_search_enabled":true,"passwords_disabled":false,"segment_heartbeat_frequency":0,"segment_includes_user_id":false,"sso_enabled":false};\n  </script>\n\n<script>\n  window.Tracking = {\n    EVENTS:     {"EVENT_SALE_INITIATED":"sale_initiated","EVENT_SALE_COMPLETE":"sale_complete","EVENT_SALE_COMPLETE_GIFT":"sale_conversion-gift","EVENT_SALE_ERROR":"sale_error","EVENT_SALE_COMPLETE_SAVE_CARD":"sale_complete_save_card","EVENT_SALE_MISSING_EMAIL":"sale_missing_email","EVENT_SALE_UNCONFIRMED_EMAIL":"sale_unconfirmed_email","EVENT_SALE_MISSING_CC":"sale_missing_cc","EVENT_SALE_MISSING_CVC":"sale_missing_cvc","EVENT_SALE_MISSING_EXP":"sale_missing_exp","EVENT_SALE_MISSING_ZIP":"sale_missing_zip","EVENT_SALE_INVALID_CC":"sale_invalid_cc","EVENT_SALE_INVALID_CVC":"sale_invalid_cvc","EVENT_SALE_INVALID_EXP":"sale_invalid_exp","EVENT_SALE_INVALID_ZIP":"sale_invalid_zip","EVENT_SALE_INVALID_COUPON":"sale_invalid_coupon","EVENT_SALE_INACTIVE_COUPON":"sale_inactive_coupon","EVENT_SALE_GEOBLOCKED_COUPON":"sale_geoblocked_coupon","EVENT_SALE_EXPIRED_CC":"sale_expired_cc","EVENT_SALE_PLAY_TRAILER":"sale_play_trailer","EVENT_SALE_PROCESSING_ERROR":"sale_processing_error","EVENT_SALE_UNKNOWN_ERROR":"sale_unknown_error","EVENT_SALE_INCOMPLETE_FORM":"sale_incomplete_form","EVENT_SALE_SHARE_FACEBOOK":"sale_share_facebook","EVENT_SALE_SHARE_TWITTER":"sale_share_twitter","EVENT_SALE_SHARE_EMAIL":"sale_share_email","EVENT_SALE_START_WATCHING":"sale_start_watching","EVENT_SALE_SAVE_CARD":"sale_save_card","EVENT_SALE_SHOW_PAYMENT_TAB":"sale_show_payment_tab","EVENT_SALE_SHOW_DETAILS_TAB":"sale_show_details_tab","EVENT_ADMIN_SIGNUP":"admin_signup","EVENT_ADMIN_ADD_SITE":"admin_add_site","EVENT_ADMIN_PUBLISH_THEME":"admin_publish_theme","EVENT_ADMIN_CREATE_VIDEO":"admin_create_video","EVENT_ADMIN_UPLOAD_VIDEO":"admin_upload_video","EVENT_ADMIN_UPLOAD_VIDEO_NATIVE":"admin_upload_video_native","EVENT_ADMIN_UPLOAD_VIDEO_DROPBOX":"admin_upload_video_dropbox","EVENT_ADMIN_TRANSCODE_VIDEO":"admin_transcode_video","EVENT_ADMIN_CREATE_EXTRA":"admin_create_extra","EVENT_ADMIN_UPLOAD_EXTRA":"admin_upload_extra","EVENT_ADMIN_UPLOAD_EXTRA_NATIVE":"admin_upload_video_native","EVENT_ADMIN_CONNECT_PAYOUT":"admin_connect_payout","EVENT_ADMIN_CREATE_SCREENER":"admin_create_screener","EVENT_ADMIN_CREATE_COUPON":"admin_create_coupon","EVENT_ADMIN_CREATE_UPDATE":"admin_create_update","EVENT_ADMIN_PUBLISH_UPDATE":"admin_publish_update","EVENT_ADMIN_CREATE_PARTNER":"admin_create_partner","EVENT_ADMIN_UPLOADED_FOLLOWERS_CS":"admin_uploaded_followers_csv","EVENT_ADMIN_ADDED_FOLLOWERS":"admin_added_followers","EVENT_ADMIN_CREATE_SUBSCRIPTION":"admin_create_subscription","EVENT_ADMIN_LINK_HELP":"admin_link_help","EVENT_ADMIN_LINK_FAQ":"admin_link_faq","EVENT_ADMIN_LINK_GETTING_STARTED":"admin_link_getting_started","EVENT_ADMIN_LINK_HOWTO":"admin_link_howto","EVENT_ADMIN_LINK_BEST_PRACTICES":"admin_link_best_practices","EVENT_ADMIN_LINK_DEV_DOCS":"admin_link_dev_docs","EVENT_FORUM_CREATE_THREAD":"forum_create_thread","EVENT_FORUM_CREATE_POST":"forum_create_post","EVENT_UPDATE_COMMENT":"comment_update","EVENT_CONCURRENT_LIMIT":"concurrent_limit","EVENT_VIDEO_COMMENT":"video_comment","EVENT_SITE_HERO_WATCH_TRAILER":"site_hero_watch_trailer","EVENT_SITE_HERO_BUY":"site_hero_buy","EVENT_SITE_FOOTER_BUY":"site_footer_buy","EVENT_SITE_FOOTER_IOS":"site_footer_ios","EVENT_SITE_BUY":"site_buy","EVENT_SITE_UNAVAILABLE_REQUEST":"site_unavailable_request","EVENT_SITE_SEARCH":"site_search","EVENT_SITE_SEARCH_EMPTY":"site_search_empty","EVENT_SITE_SEARCH_VIDEO_CLICK":"search_video","EVENT_SITE_SEARCH_COLLECTION_CLICK":"search_collection","EVENT_SITE_SEARCH_PRODUCT_CLICK":"search_product","EVENT_SITE_SEARCH_ITEM_SELECTION":"site_search_item_selected","EVENT_SITE_VIDEO_BUY":"site_video_buy","EVENT_SITE_VIDEO_WATCH_TRAILER":"site_video_watch_trailer","EVENT_SITE_COLLECTION_BUY":"site_collection_buy","EVENT_SITE_PRODUCT_PREORDER":"site_product_preorder","EVENT_SITE_PRODUCT_BUY":"site_product_buy","EVENT_SITE_PRODUCT_RENT":"site_product_rent","EVENT_SITE_CAROUSEL_NEXT":"site_carousel_next","EVENT_SITE_CAROUSEL_PREV":"site_carousel_prev","EVENT_SITE_CAROUSEL_COIN":"site_carousel_coin","EVENT_SITE_CAROUSEL_WATCH_NOW":"site_carousel_watch_now","EVENT_SITE_CAROUSEL_TRAILER":"site_carousel_trailer","EVENT_SITE_HOME":"site_home","EVENT_SITE_BROWSE":"site_browse","EVENT_SITE_SETTINGS":"site_settings","EVENT_SITE_FORUMS":"site_forums","EVENT_SITE_SUPPORT":"site_help","EVENT_SITE_CONTACT":"site_contact","EVENT_SITE_CONTACT_HELP":"site_contact_help","EVENT_SITE_VIEW_PRODUCTS":"site_products","EVENT_SITE_VIEW_PRODUCT":"site_product","EVENT_SITE_COLLECTION_CLICK":"site_collection","EVENT_SITE_VIDEO_CLICK":"site_video","EVENT_SITE_EXTRA_CLICK":"site_extras","EVENT_SITE_CATEGORY_CLICK":"site_category","EVENT_SITE_MOVIE_CLICK":"site_movie","EVENT_SITE_PLAYLIST_CLICK":"site_playlist","EVENT_SITE_SERIES_CLICK":"site_series","EVENT_SITE_SEASON_CLICK":"site_season","EVENT_SITE_TERMS":"site_tos","EVENT_SITE_PRIVACY":"site_privacy","EVENT_SITE_COOKIES":"site_cookies","EVENT_SITE_COPYRIGHT":"site_copyright","EVENT_SITE_DATA_PROCESSING":"site_data_processing","EVENT_SITE_WATCHLIST_REMOVE":"site_removewatchlist","EVENT_SITE_WATCHLIST_ADD":"site_addwatchlist","EVENT_SITE_PAGE_VIEW":"site_page_view","EVENT_SITE_CONTEXT_HOMEPAGE":"site_context_homepage","EVENT_SITE_CONTEXT_COLLECTION":"site_context_collection","EVENT_SITE_CONTEXT_CATEGORY":"site_context_category","EVENT_SITE_CONTEXT_MOVIE":"site_context_movie","EVENT_SITE_CONTEXT_PLAYLIST":"site_context_playlist","EVENT_SITE_CONTEXT_SERIES":"site_context_series","EVENT_SITE_CONTEXT_UPNEXT":"site_context_upnext","EVENT_AUTHENTICATION_LOGIN":"authentication_login","EVENT_AUTHENTICATION_ACTIVATION_STARTED":"activationstarted","EVENT_AUTHENTICATION_ACTIVATION_COMPLETE":"activation","EVENT_AUTHENTICATION_SIGNIN_VIEW":"signin","EVENT_AUTHENTICATION_SIGNIN_STARTED":"signinstarted","EVENT_AUTHENTICATION_SIGNIN_COMPLETE":"authentication","EVENT_AUTHENTICATION_SIGNUP_VIEW":"signup","EVENT_AUTHENTICATION_SIGNUP_STARTED":"signupstarted","EVENT_AUTHENTICATION_PURCHASE_STARTED":"purchasestarted","EVENT_AUTHENTICATION_PURCHASE_COMPLETED":"purchasecompleted","EVENT_AUTHENTICATION_CONVERSION":"conversion","EVENT_AUTHENTICATION_REGISTRATION":"register","EVENT_AUTHENTICATION_IDENTIFY_USER":"identify_user","EVENT_SETTINGS_SAVE_PROFILE":"settings_save_profile","EVENT_SETTINGS_SAVE_NOTIFICATIONS":"settings_save_notifications","EVENT_SETTINGS_SAVE_CARD":"settings_save_card","EVENT_DEVICES_DOWNLOAD_IOS":"devices_download_ios","EVENT_DEVICES_DOWNLOAD_APPLETV":"devices_download_appletv","EVENT_DEVICES_DOWNLOAD_ANDROID":"devices_download_android","EVENT_DEVICES_DOWNLOAD_ANDROIDTV":"devices_download_androidtv","EVENT_DEVICES_DOWNLOAD_AMAZON_FIRE":"devices_download_amazon_fire","EVENT_DEVICES_DOWNLOAD_XBOX":"devices_download_xbox","EVENT_DEVICES_DOWNLOAD_ROKU":"devices_download_roku","EVENT_DEVICES_DOWNLOAD_TIZEN":"devices_download_tizen","EVENT_DEVICES_DOWNLOAD_CHROMECAST":"devices_download_chromecast","EVENT_NEWSLETTER_SIGNUP":"event_newsletter_signup","EVENT_METADATA_ADVISORY":"metadata_click_advisory","EVENT_METADATA_CAST":"metadata_click_cast","EVENT_METADATA_CREW":"metadata_click_crew","EVENT_METADATA_GENRES":"metadata_click_genres","EVENT_METADATA_RATING":"metadata_click_rating","EVENT_METADATA_TAGS":"metadata_click_tags","EVENT_SEGMENT_AUTHENTICATION_CONVERSION":"Order Completed","EVENT_SEGMENT_AUTHENTICATION_PURCHASE_STARTED":"Checkout Started","EVENT_SEGMENT_AUTHENTICATION_REGISTRATION":"Registration Completed","EVENT_SEGMENT_AUTHENTICATION_SIGNIN_COMPLETE":"Sign In Complete","EVENT_SEGMENT_AUTHENTICATION_SIGNIN_VIEW":"Sign In Viewed","EVENT_SEGMENT_AUTHENTICATION_SIGNUP_STARTED":"Signed Up","EVENT_SEGMENT_AUTHENTICATION_SIGNUP_VIEW":"Sign Up Viewed","EVENT_SEGMENT_EXTRA_VIEWED":"Extra Content Viewed","EVENT_SEGMENT_SEARCH_EXECUTED":"Search Executed","EVENT_SEGMENT_SEARCH_ITEM_SELECTED":"Search Result Selected","EVENT_SEGMENT_VIDEO_AD_START":"Video Ad Started","EVENT_SEGMENT_VIDEO_AD_CLICKED":"Video Ad Clicked","EVENT_SEGMENT_VIDEO_AD_FINISH":"Video Ad Completed","EVENT_SEGMENT_VIDEO_AD_FAIL":"Video Ad Failed","EVENT_SEGMENT_VIDEO_AD_SKIP":"Video Ad Skipped","EVENT_SEGMENT_VIDEO_COMMENT_ADDED":"Video Comment Added","EVENT_SEGMENT_VIDEO_COMPLETE_PROGRESS_PERCENTILE":"Video Content Completed","EVENT_SEGMENT_VIDEO_ENDED":"Video Content Ended","EVENT_SEGMENT_VIDEO_END_BUFFER":"Video Playback Buffer Completed","EVENT_SEGMENT_VIDEO_HALF_PROGRESS_PERCENTILE":"50% Content Completed","EVENT_SEGMENT_VIDEO_PAUSE":"Video Playback Paused","EVENT_SEGMENT_VIDEO_PLAY":"Video Playback Started","EVENT_SEGMENT_VIDEO_QUARTER_PROGRESS_PERCENTILE":"25% Content Completed","EVENT_SEGMENT_VIDEO_RESUME":"Video Playback Resumed","EVENT_SEGMENT_VIDEO_SEEKED":"Video Playback Seek Completed","EVENT_SEGMENT_VIDEO_SEEKING":"Video Playback Seek Started","EVENT_SEGMENT_VIDEO_START_BUFFER":"Video Playback Buffer Started","EVENT_SEGMENT_VIDEO_TIMEUPDATE":"Video Content Playing","EVENT_SEGMENT_VIDEO_THIRD_PROGRESS_PERCENTILE":"75% Content Completed","EVENT_SEGMENT_VIDEO_WATCHLIST_ADD":"Added to Watch List","EVENT_SEGMENT_VIDEO_WATCHLIST_REMOVE":"Removed from Watch List","EVENT_SEARCH":"search","EVENT_SEARCH_ITEM_SELECTED":"searchitemselected","EVENT_EXTRA_VIEWED":"extraviewed"},\n    PROPERTIES: {"PROPERTY_COLLECTOR_URL":"https://collector.vhx.tv/pixel.gif","PROPERTY_COLLECTION_ID":"collection_id","PROPERTY_DEVICE":"device","PROPERTY_DEVICE_ID":"device_id","PROPERTY_NAME":"name","PROPERTY_PLATFORM":"platform","PROPERTY_PLATFORM_ID":"platform_id","PROPERTY_PLATFORM_VERSION":"platform_version","PROPERTY_PRODUCT_ID":"product_id","PROPERTY_REFERRER":"referrer","PROPERTY_SESSION_ID":"session_id","PROPERTY_SITE_ID":"site_id","PROPERTY_TIMESTAMP":"timestamp","PROPERTY_TYPE":"type","PROPERTY_URL":"url","PROPERTY_USER_AGENT":"user_agent","PROPERTY_USER_EMAIL":"user_email","PROPERTY_USER_ID":"user_id","PROPERTY_VIDEO_ID":"video_id","PROPERTY_VIEW":"view"},\n    DEPRECATED_EVENTS: ["admin_connect_payout","admin_create_coupon","admin_create_screener","admin_link_dev_docs","admin_link_faq","admin_link_help","admin_publish_theme","metadata_click_cast","metadata_click_crew","metadata_click_genres","metadata_click_rating","metadata_click_tags","metadata_click_advisory"],\n    UPDATED_EVENTS: {"search_video":"searchitemselected","search_collection":"searchitemselected","site_search_item_selected":"searchitemselected","site_extras":"extraviewed"},\n    SITE_GANALYTICS_ENABLED: false,\n    SESSION: \'bbc97018fe30aafa062927836a63d0c3\'\n  };\n</script>\n  <script>\n     window.Page = {"PROPERTIES":{"VIEW_TYPE":"collection","PRODUCT_ID":39621,"COLLECTION_ID":97298,"COLLECTION_TITLE":"La Jet\xc3\xa9e"}}\n  </script>\n\n\n\n  <script>\n  var _gaq = _gaq || [];\n\n  (function(i,s,o,g,r,a,m){i[\'GoogleAnalyticsObject\']=r;i[r]=i[r]||function(){\n  (i[r].q=i[r].q||[]).push(arguments)},i[r].l=1*new Date();a=s.createElement(o),\n  m=s.getElementsByTagName(o)[0];a.async=1;a.src=g;m.parentNode.insertBefore(a,m)\n  })(window,document,\'script\',\'//www.google-analytics.com/analytics.js\',\'ga\');\n\n      ga(\'create\', \'UA-19540423-6\', \'auto\');\n    ga(\'require\', \'displayfeatures\');\n\n      ga(\'set\', \'dimension1\', \'criterionchannelchartersu\');\n\n    ga(\'send\', \'pageview\');\n\n\n      (function() {\n        var vhx = document.createElement(\'script\'); vhx.type = \'text/javascript\'; vhx.async = true;\n        vhx.src = \'https://cdn.vhx.tv/assets/tracker-be253e44a9312b41e422e9acec66d157e9c650341ea1701c01c2c8b31b26b9e6.js\';\n        var s = document.getElementsByTagName(\'script\')[0]; s.parentNode.insertBefore(vhx, s);\n      })();\n\n\n    Tracking.SITE_GANALYTICS_ENABLED = true;\n\n      ga(\'create\', \'UA-1852531-11\', \'auto\', { \'name\': \'site\', \'allowLinker\': true});\n\n    ga(\'site.require\', \'displayfeatures\');\n      ga(\'site.require\', \'linker\');\n      ga(\'site.linker:autoLink\', [\'www.criterionchannel.com\',\'criterionchannel.vhx.tv\'] );\n    ga(\'site.set\', \'dimension1\', \'criterionchannelchartersu\');\n    ga(\'site.send\', \'pageview\');\n\n    // <!-- Google Tag Manager -->\n    (function(w,d,s,l,i){w[l]=w[l]||[];w[l].push({\'gtm.start\':\n    new Date().getTime(),event:\'gtm.js\'});var f=d.getElementsByTagName(s)[0],\n    j=d.createElement(s),dl=l!=\'dataLayer\'?\'&l=\'+l:\'\';j.async=true;j.src=\n    \'https://www.googletagmanager.com/gtm.js?id=\'+i+dl;f.parentNode.insertBefore(j,f);\n    })(window,document,\'script\',\'dataLayer\',\'GTM-PKL3NF6\');\n    // <!-- End Google Tag Manager -->\n\n  if (window._current_site && window._current_site.facebook_pixel_codes.length > 0) {\n    // <!-- Facebook Pixel Code -->\n    !function(f,b,e,v,n,t,s)\n    {if(f.fbq)return;n=f.fbq=function(){n.callMethod?\n    n.callMethod.apply(n,arguments):n.queue.push(arguments)};\n    if(!f._fbq)f._fbq=n;n.push=n;n.loaded=!0;n.version=\'2.0\';\n    n.queue=[];t=b.createElement(e);t.async=!0;\n    t.src=v;s=b.getElementsByTagName(e)[0];\n    s.parentNode.insertBefore(t,s)}(window, document,\'script\',\n    \'https://connect.facebook.net/en_US/fbevents.js\');\n    //<!-- End Facebook Pixel Code -->\n  }\n\n  if (window._current_site && window._current_site.twitter_pixel_codes.length > 0) {\n    // <!-- Twitter Pixel Code -->\n    !function(e,t,n,s,u,a){e.twq||(s=e.twq=function(){s.exe?s.exe.apply(s,arguments):s.queue.push(arguments);\n    },s.version=\'1.1\',s.queue=[],u=t.createElement(n),u.async=!0,u.src=\'//static.ads-twitter.com/uwt.js\',\n    a=t.getElementsByTagName(n)[0],a.parentNode.insertBefore(u,a))}(window,document,\'script\');\n    //<!-- End Twitter Pixel Code -->\n  }\n\n      Segment_ID = \'D22V0IgjPYTwUKXVEtSzbmfcnolk4CNB\';\n      !function(){var analytics=window.analytics=window.analytics||[];if(!analytics.initialize)if(analytics.invoked)window.console&&console.error&&console.error("Segment snippet included twice.");else{analytics.invoked=!0;analytics.methods=["trackSubmit","trackClick","trackLink","trackForm","pageview","identify","reset","group","track","ready","alias","debug","page","once","off","on"];analytics.factory=function(t){return function(){var e=Array.prototype.slice.call(arguments);e.unshift(t);analytics.push(e);return analytics}};for(var t=0;t<analytics.methods.length;t++){var e=analytics.methods[t];analytics[e]=analytics.factory(e)}analytics.load=function(t,e){var n=document.createElement("script");n.type="text/javascript";n.async=!0;n.src="https://cdn.segment.com/analytics.js/v1/"+t+"/analytics.min.js";var a=document.getElementsByTagName("script")[0];a.parentNode.insertBefore(n,a);analytics._loadOptions=e};analytics.SNIPPET_VERSION="4.1.0";\n      analytics.load("D22V0IgjPYTwUKXVEtSzbmfcnolk4CNB");\n      }}();\n  </script>\n\n\n\n    \n\n  <script>\n  if (/ticket=/.test(location.search) && window.history.replaceState) {\n    var url = location.href.replace(/\\?([^#]*)/, function(_, search) {\n      search = search.split(\'&\').map(function(v) {\n        return !/^ticket=/.test(v) && !/^support=/.test(v) && !/^sso=/.test(v) && v;\n      }).filter(Boolean).join(\'&\');\n      return search ? \'?\' + search : \'\';\n    });\n\n    if (url != location.href) {\n      window.history.replaceState({}, \'\', url);\n    }\n  }\n\n  if (/return_to=/.test(location.search) && window.history.replaceState) {\n    var url = location.href.replace(/\\?([^#]*)/, function(_, search) {\n      search = search.split(\'&\').map(function(v) {\n        return !/^return_to=/.test(v) && v;\n      }).filter(Boolean).join(\'&\');\n      return search ? \'?\' + search : \'\';\n    });\n\n    if (url != location.href) {\n      window.history.replaceState({}, \'\', url);\n    }\n  }\n\n  if (/logout=/.test(location.search) && window.history.replaceState) {\n    var url = location.href.replace(/\\?([^#]*)/, function(_, search) {\n      search = search.split(\'&\').map(function(v) {\n        return !/^logout=/.test(v) && v;\n      }).filter(Boolean).join(\'&\');\n      return search ? \'?\' + search : \'\';\n    });\n\n    if (url != location.href) {\n      window.history.replaceState({}, \'\', url);\n    }\n  }\n</script>\n\n\n  <script>\n    window.TOKEN = "eyJhbGciOiJSUzI1NiIsImtpZCI6ImQ2YmZlZmMzNGIyNTdhYTE4Y2E2NDUzNDE2ZTlmZmRjNjk4MDAxMDdhZTQ2ZWJhODg0YTU2ZDBjOGQ4NTYzMzgifQ.eyJhcHBfaWQiOjY4NTExLCJleHAiOjE2NzIwMDU3NzgsIm5vbmNlIjoiOWMwM2RhYzRjOGY2OTkzZCIsInNjb3BlcyI6WyJwdWJsaWMiXSwic2Vzc2lvbl9pZCI6InNBTHY0Z3FrdU5CaFhyYWJPNzA4dVE9PSIsInNpdGVfaWQiOjU5MDU0fQ.hDkqY_8qhH87V-E5TwQukxPGiISiNKVecs8_foJsThFAUEhwCR_8bTdiecMIN9-Oj0HyaFXLh1bFX-uVXPZOTU7PhDLbN4BBdSYqTz9m7WhpahkTJ_--NHfHMSAUSZplQ6OCNIBkQI4r0cEh-1Egf5xoUa4eY80hxCvrQfvWXQa1c6zjmxGBKZsu4WxRgquVDEoYtfnSC6IHaN8koqzzt9fvBXuVT8OBzjg7-GjMUTqzFBCb8u0wujrxwg7E1Rzk7JD6uzwNzQk5WNoTnwD3kUSsQkk5FbGmLcJcUULwwv1sh8TPXrVqlNMsbz7KzhxcnRPOPZ5_0pGi7U6I0qCfGQVc0nR0BLHr68r9II8NH8sb8WQsSeFoC5KKLVLr4NRqUBeJPi4uFNkI-jO7us93dbrleuWu7IRziXR8y47N5nNLxZRszMlRixy3TFpf8uU1iyzV5QvSrw4CPDPoL5SrtYnjFTojp7fkq_UEYgrZObjLovqmAkUkFKhrVJfWy4KdWc_S1-lBe-pgVvGM-q1Ec3XMayIrBb0HZgdpJw1ypDorUTuWwhnwMjsRtKNAD37eOE1FCWlB_JH9A-hqR_iGsXfEL9tcJUqrNdQPqkcMG9p9XeX-7C786rzM7QcnIhhMUSW4LzAml67ZSK8P6hPPgK9-vQcksZWjcfXYFiGfvLw";\n\n    (function() {\n      var config = { kitId: \'kyo4vqy\', scriptTimeout: 3000 };\n      var h=document.getElementsByTagName("html")[0];h.className+=" wf-loading";var t=setTimeout(function(){h.className=h.className.replace(/(\\s|^)wf-loading(\\s|$)/g," ");h.className+=" wf-inactive"},config.scriptTimeout);var tk=document.createElement("script"),d=false;tk.src=\'//use.typekit.net/\'+config.kitId+\'.js\';tk.type="text/javascript";tk.async="true";tk.onload=tk.onreadystatechange=function(){var a=this.readyState;if(d||a&&a!="complete"&&a!="loaded")return;d=true;clearTimeout(t);try{Typekit.load(config)}catch(b){}};var s=document.getElementsByTagName("script")[0];s.parentNode.insertBefore(tk,s);\n    })();\n  </script>\n\n  \n  \n  \n</head>\n\n\n<body class="site-background-color  site-color-is-light logged-out collections group-a collection-la-jetee" data-site-color="#b9bcc7" data-track-event="site_movie">\n\n    <!-- Google Tag Manager (noscript) -->\n  <noscript><iframe src="https://www.googletagmanager.com/ns.html?id=GTM-PKL3NF6"\n  height="0" width="0" style="display:none;visibility:hidden"></iframe></noscript>\n  <!-- End Google Tag Manager (noscript) -->\n\n  \n\n  <a href="#maincontent" class="skipto">Skip to main content</a>\n    <div style="display: none !important;" rel="description" class="is-hidden">\n      Classics and discoveries from around the world, thematically programmed with special features, on a streaming service brought to you by the Criterion Collection.\n    </div>\n\n  <div data-flash="{&quot;auto_hide&quot;:true,&quot;style&quot;:&quot;overlay&quot;,&quot;container&quot;:null,&quot;type&quot;:&quot;notice&quot;,&quot;message&quot;:{&quot;body&quot;:null},&quot;show_on_load&quot;:false}" class="is-error is-hidden">\n    <div class="flash-icon">\n      <i class="icon icon-check-inverse icon-check-white"></i>\n      <i class="icon icon-alert-inverse icon-alert-white"></i>\n      <i class="icon icon-check-alt icon-check-blue"></i>\n      <i class="icon medium icon-check"></i>\n      <i class="icon medium icon-alert"></i>\n    </div>\n\n    <div class="flash-msg"></div>\n\n    <div class="flash-close">\n      <i class="icon icon-remove-inverse icon--xsmall"></i>\n      <i class="icon icon-remove-mono icon-x-black icon--xsmall"></i>\n    </div>\n</div>\n\n\n\n\n    \n\n\n\n\n\n\n\n\n\n<header class="nav-background-color border-bottom site-border-color site-header has-logo ">\n  <div class="row full-width padding-top-medium padding-bottom-medium primary-row-container flex-container">\n    <div class="small-16 medium-11 columns identity-container flex-container has-image">\n      <span class="show-for-small-only flex-item padding-right-small show-menu-container">\n        <label for="show-menu" class="show-menu-trigger">\n          <span class="site-link-header-color inverse"><i class="icon site-icon-menu"></i></span>\n        </label>\n      </span>\n\n      <h1 class="identity site-font-primary-family head primary flex-item padding-horizontal-medium">\n            <a href="/">\n              <img class="logo" src="https://vhx.imgix.net/criterionchannel/assets/1c5933b9-19a1-4ea8-9761-726ca79faf44.png?fit=max&h=84&q=92&w=400" alt="The Criterion Channel" />\n</a>      </h1>\n\n      <nav class="navigation-container inline flex-item" role="navigation" aria-label="Primary">\n        <input type="checkbox" id="show-menu" name="show-menu" role="button">\n\n        <div class="navigation-inner site-header-background-color-small-only" id="primary-navigation">\n\n          <a href="/browse" class="browse-link site-font-primary-family head site-link-header-color">\n            Browse\n          </a>\n\n\n              <a href="#"\n                id="search-nav-toggle"\n                class="search-link site-font-primary-family head site-link-header-color">\n                Search\n              </a>\n\n                <a href="https://films.criterionchannel.com/" rel="noopener" class="custom-nav-all-films site-font-primary-family head site-link-header-color" target="_blank">ALL FILMS</a>\n                <a href="https://www.criterion.com/" rel="noopener" class="custom-nav-criterion-com site-font-primary-family head site-link-header-color" target="_blank">Criterion.com</a>\n\n            <a class="show-for-small-only flex-item site-link-header-color site-font-primary-family head subscribe-link" data-track-event-properties="{&quot;type&quot;:&quot;subscription&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/la-jetee&quot;}" data-track-event="site_buy" href="https://www.criterionchannel.com/checkout/subscribe">\n              <strong style="font-weight: bold;">\n                Start Free Trial\n              </strong>\n</a>\n\n              <a href="/login" class="show-for-small-only flex-item user-action-link head site-link-header-color site-font-primary-family">\n                Sign in\n              </a>\n\n        </div>\n      </nav>\n    </div>\n\n    <div class="hide-for-small-only text-right flex-right small-16 medium-5 columns small-only-text-center user-activity site-border-color">\n      <div class="user-actions has-image padding-right-medium flex-container" role="navigation" aria-label="Secondary">\n            <a class="flex-item site-link-header-color site-font-primary-family head subscribe-link user-action-link" data-track-event-properties="{&quot;type&quot;:&quot;subscription&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/la-jetee&quot;}" data-track-event="site_buy" href="https://www.criterionchannel.com/checkout/subscribe">\n              <strong style="font-weight: bold;">\n                Start Free Trial\n              </strong>\n</a>\n\n              <a href="/login" class="flex-item user-action-link head site-link-header-color  site-font-primary-family">\n                Sign In\n              </a>\n      </div>\n    </div>\n  </div>\n</header>\n\n  <div id="search-react-app"\n    data-props="{&quot;overlay&quot;:true,&quot;site_id&quot;:59054,&quot;api_v1_url&quot;:&quot;https://api.vhx.tv&quot;,&quot;api_v2_url&quot;:&quot;https://api.vhx.com&quot;,&quot;user_subscribed&quot;:false,&quot;purchased_product_ids&quot;:[],&quot;theme&quot;:&quot;dark&quot;,&quot;show_filters&quot;:false}">\n  </div>\n\n\n\n  \n\n\n\n<section class="site-background-color site-font-primary-color margin-bottom-large collection-movie top-container">\n  <div class="blurred-background-image"  style="background-image: url(\'https://vhx.imgix.net/criterionchannelchartersu/assets/9dfaaba6-1c22-473c-afe2-0ef43c00c663-663d30eb.jpg?blur=180\')">\n    <div class="gradient-blend" data-gradient-blend></div>\n  </div>\n  <section class="product-feature padding-horizontal-medium padding-top-large padding-bottom-xlarge border-bottom border-bottom-light">\n\n    <div class="row relative">\n      <div class="small-16 medium-8 large-8 collection-details-container columns end">\n\n        <div class="collection-details grid-padding-left">\n            <div class="show-for-small-only cover-art padding-bottom-medium">\n              <div class="embed">\n                <img src="https://vhx.imgix.net/criterionchannelchartersu/assets/9dfaaba6-1c22-473c-afe2-0ef43c00c663-663d30eb.jpg?auto=format%2Ccompress&fit=crop&h=360&q=70&w=640" alt="La Jet\xc3\xa9e" data-image-primary>\n              </div>\n            </div>\n\n          <h1 class="head primary site-font-primary-color collection-title">La Jet\xc3\xa9e</h1>\n\n            <div class="contain padding-top-medium collection-description word-break">\n    <div class="site-font-secondary-color"\n         data-text-show-less="Show less"\n         data-text-show-more="Show more"\n         data-truncate=\'{"max_height": 60, "speed": 300}\'>\n      <p>Directed by Chris Marker \xe2\x80\xa2 1963 \xe2\x80\xa2 France</p>\n\n<p>Chris Marker, filmmaker, poet, novelist, photographer, editor, and now videographer and digital multimedia artist, has been challenging moviegoers, philosophers, and himself for years with his complex queries about time, memory, and the rapid advancement of life on this planet. Marker\xe2\x80\x99s LA JET\xc3\x89E is one of the most influential, radical science-fiction films ever made, a tale of time travel told in still images.</p>\n    </div>\n  </div>\n\n\n          <div class="collection-actions padding-bottom-small padding-top-large">\n                  <a href="https://www.criterionchannel.com/checkout/subscribe?return_to=https%3A%2F%2Fcriterionchannel.vhx.tv%2Fla-jetee"\n      class="btn btn-site-primary custom-btn-action-transact site-font-primary-family uppercase margin-bottom-small margin-right-small"\n      data-track-event="site_collection_buy">\n      Subscribe\n    </a>\n\n              <a href="#" class="btn btn-black custom-btn-action-share margin-bottom-small margin-right-medium " data-reveal-id="modal-la-jetee">\n                Share\n              </a>\n              \n\n\n\n<div class="reveal-modal site-background-color text-center share-modal-container" id="modal-la-jetee" data-reveal>\n  <a class="close-reveal-modal"><i class="icon icon-remove-inverse"></i><i class="icon icon-remove-gray"></i></a>\n  <div class="share-title-container padding-bottom-medium">\n    <h5 class="head secondary site-font-secondary-color site-font-primary-family">\n      Share with your friends\n    </h5>\n  </div>\n  <div class="share-link-container padding-bottom-medium">\n    <ul class="small-block-grid-3">\n      <li>\n        <a href="https://www.facebook.com/sharer/sharer.php?u=https%3A%2F%2Fwww.criterionchannel.com%2Fla-jetee"\n           data-social-popup\n           data-share="facebook"\n           class="btn btn-facebook block border site-border-color"\n           title="Facebook">\n          <i class="vertical-align icon site-icon-facebook icon-center"></i>\n        </a>\n      </li>\n      <li>\n        <a href="https://twitter.com/share?url=https%3A%2F%2Fwww.criterionchannel.com%2Fla-jetee&text=Watch+La+Jet%C3%A9e+from+%40criterionchannl&related=criterionchannl"\n           data-social-popup\n           data-share="twitter"\n           class="btn-twitter btn block border site-border-color"\n           title="Twitter">\n          <i class="vertical-align icon site-icon-twitter icon-center"></i>\n        </a>\n      </li>\n      <li>\n        <a target="_blank" href="/cdn-cgi/l/email-protection#f3cc80869199969087cea49287909bd3bf92d3b99687305a96d395819c9ed3a79b96d3b0819a8796819a9c9dd3b09b929d9d969fd5929e83c8919c978ace9b87878380d6c0b2d6c1b5d6c1b5848484dd90819a8796819a9c9d909b929d9d969fdd909c9ed6c1b59f92de9996879696" class="btn btn-black block border site-border-color" title="Email">\n          <i class="vertical-align icon site-icon-envelope icon-center"></i>\n        </a>\n      </li>\n    </ul>\n  </div>\n    <div class="share-url-container">\n      <form class="form"><input value="https://www.criterionchannel.com/la-jetee" type="text" class="text text-center border-none site-background-color site-font-secondary-color" /></form>\n    </div>\n</div>\n\n          </div>\n        </div>\n      </div>\n\n        <div class="hide-for-small-only medium-8 large-offset-2 large-6 end cover-art columns">\n          <div class="embed-container grid-padding-right">\n            <div class="embed">\n              <img src="https://vhx.imgix.net/criterionchannelchartersu/assets/9dfaaba6-1c22-473c-afe2-0ef43c00c663-663d30eb.jpg?auto=format%2Ccompress&fit=crop&h=720&q=75&w=1280" alt="La Jet\xc3\xa9e" data-image-primary>\n            </div>\n          </div>\n        </div>\n\n    </div>\n  </section>\n\n  <section class="movie-container video-container padding-top-large padding-horizontal-medium">\n    <div class="row relative">\n      <div class="small-16 columns item-grid flush-grid flush-row">\n        \n  <div class="row full-width padding-horizontal-medium">\n    <ul class="small-block-grid-1 medium-block-grid-3 large-block-grid-3 ">\n\n\n\n  \n<li\n  class="js-collection-item\n    collection-item-465713 item-type-video\n    \n    "\n  data-item-id="465713"\n  data-item-type="video"\n  \n    data-tooltip="id: collection-tooltip-465713;\n    position: right;\n    offset: 0;\n    animate: true;\n    timer: 650;\n    delay: false;\n    is_centered: true;"\n    \n>\n  <div class="browse-item-card">\n    <div class="grid-item-padding">\n      <a href="https://www.criterionchannel.com/la-jetee/videos/la-jetee" class="browse-item-link" data-track-event="site_video" data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/la-jetee&quot;,&quot;id&quot;:465713,&quot;context&quot;:&quot;site_context_movie&quot;,&quot;label&quot;:&quot;La Jet\xc3\xa9e&quot;,&quot;canonical_collection&quot;:null}">\n        <div class="browse-image-container">\n            <span class="freemium-lock">\n              <i class="icon icon--xsmall icon-lock-inverse"></i>\n            </span>\n\n\n          <img alt="La Jet\xc3\xa9e" src="https://vhx.imgix.net/criterionchannelchartersu/assets/936f4a01-6c6a-48e5-ae7f-f3b6883fbc12.jpg?auto=format%2Ccompress&fit=crop&h=360&q=70&w=640" />\n\n\n          <div class="browse-info-container">\n              <div class="duration-container\n                is-locked\n                "\n              >\n                28:02\n              </div>\n          </div>\n        </div> \n      </a>\n\n      <div class="padding-small">\n        <div class="site-font-primary-color browse-item-title">\n           <a\n            href="https://www.criterionchannel.com/la-jetee/videos/la-jetee"\n            data-track-event="site_video"\n            data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/la-jetee&quot;,&quot;id&quot;:465713,&quot;context&quot;:&quot;site_context_movie&quot;,&quot;label&quot;:&quot;La Jet\xc3\xa9e&quot;,&quot;canonical_collection&quot;:null}"\n          >\n            <strong title="La Jet\xc3\xa9e">\n              La Jet\xc3\xa9e\n            </strong>\n          </a>\n        </div>\n\n      </div>\n    </div>\n\n\n    <div \n         class="tooltip background-white" id="collection-tooltip-465713">\n      <h3 class="tooltip-item-title site-font-primary-family"><strong>La Jet\xc3\xa9e</strong></h3>\n\n        <div class="transparent padding-top-medium">\n          <p>Directed by Chris Marker \xe2\x80\xa2 1963 \xe2\x80\xa2 France</p>\n\n<p>Chris Marker, filmmaker, poet, novelist, photographer, editor, and now videographer and digital multimedia artist, has been challenging moviegoers, philosophers, and himself for years with his complex queries about time, memory, and the rapid advancement ...</p>\n        </div>\n    </div>\n  </div>\n</li>\n\n\n\n\n    </ul>\n  </div>\n\n      </div>\n    </div>\n  </section>\n    <section class="extras-container video-container padding-horizontal-medium padding-bottom-medium">\n      <div class="row relative border-top border-top-light padding-top-large">\n        <div class="small-16 columns item-grid flush-grid flush-row">\n          <h2 class="site-font-primary-color site-font-primary-family collection-title extras-header grid-padding-left padding-bottom-medium">\n            Extras\n          </h2>\n          \n  <div class="row full-width padding-horizontal-medium">\n    <ul class="small-block-grid-1 medium-block-grid-3 large-block-grid-4 js-load-more-items-container">\n\n\n\n  \n<li\n  class="js-collection-item\n    collection-item-521071 item-type-video\n    \n    "\n  data-item-id="521071"\n  data-item-type="video"\n  \n    data-tooltip="id: collection-tooltip-521071;\n    position: right;\n    offset: 0;\n    animate: true;\n    timer: 650;\n    delay: false;\n    is_centered: true;"\n    \n>\n  <div class="browse-item-card">\n    <div class="grid-item-padding">\n      <a href="https://www.criterionchannel.com/la-jetee/videos/alternate-french-audio" class="browse-item-link" data-track-event="site_video" data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/la-jetee&quot;,&quot;id&quot;:521071,&quot;context&quot;:&quot;site_context_movie&quot;,&quot;label&quot;:&quot;LA JET\xc3\x89E Alternate French Audio&quot;,&quot;canonical_collection&quot;:null}">\n        <div class="browse-image-container">\n            <span class="freemium-lock">\n              <i class="icon icon--xsmall icon-lock-inverse"></i>\n            </span>\n\n\n          <img alt="LA JET\xc3\x89E Alternate French Audio" src="https://vhx.imgix.net/criterionchannelchartersu/assets/c2e02764-7d4d-4faf-b0da-b0ae8978622f-96f43ab0.jpg?auto=format%2Ccompress&fit=crop&h=360&q=70&w=640" />\n\n\n          <div class="browse-info-container">\n              <div class="duration-container\n                is-locked\n                "\n              >\n                28:06\n              </div>\n          </div>\n        </div> \n      </a>\n\n      <div class="padding-small">\n        <div class="site-font-primary-color browse-item-title">\n           <a\n            href="https://www.criterionchannel.com/la-jetee/videos/alternate-french-audio"\n            data-track-event="site_video"\n            data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/la-jetee&quot;,&quot;id&quot;:521071,&quot;context&quot;:&quot;site_context_movie&quot;,&quot;label&quot;:&quot;LA JET\xc3\x89E Alternate French Audio&quot;,&quot;canonical_collection&quot;:null}"\n          >\n            <strong title="LA JET\xc3\x89E Alternate French Audio">\n              LA JET\xc3\x89E Alternate French Audio\n            </strong>\n          </a>\n        </div>\n\n      </div>\n    </div>\n\n\n    <div \n         class="tooltip background-white" id="collection-tooltip-521071">\n      <h3 class="tooltip-item-title site-font-primary-family"><strong>LA JET\xc3\x89E Alternate French Audio</strong></h3>\n\n        <div class="transparent padding-top-medium">\n          <p>Chris Marker created unique English and French narration tracks for LA JET\xc3\x89E and prefers that the film be experienced in the language the viewer is more familiar with. The different tracks are not direct translations of each other or synched to the images in exactly the same way. The French audio...</p>\n        </div>\n    </div>\n  </div>\n</li>\n\n\n\n\n  \n<li\n  class="js-collection-item\n    collection-item-464118 item-type-video\n    \n    "\n  data-item-id="464118"\n  data-item-type="video"\n  \n    data-tooltip="id: collection-tooltip-464118;\n    position: right;\n    offset: 0;\n    animate: true;\n    timer: 650;\n    delay: false;\n    is_centered: true;"\n    \n>\n  <div class="browse-item-card">\n    <div class="grid-item-padding">\n      <a href="https://www.criterionchannel.com/la-jetee/videos/jean-pierre-gorin-on-la-jetee" class="browse-item-link" data-track-event="site_video" data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/la-jetee&quot;,&quot;id&quot;:464118,&quot;context&quot;:&quot;site_context_movie&quot;,&quot;label&quot;:&quot;Jean-Pierre Gorin on LA JET\xc3\x89E&quot;,&quot;canonical_collection&quot;:null}">\n        <div class="browse-image-container">\n            <span class="freemium-lock">\n              <i class="icon icon--xsmall icon-lock-inverse"></i>\n            </span>\n\n\n          <img alt="Jean-Pierre Gorin on LA JET\xc3\x89E" src="https://vhx.imgix.net/criterionchannelchartersu/assets/0f37cf65-075b-4ab8-a05d-2755e73bc57f-6c8410aa.jpg?auto=format%2Ccompress&fit=crop&h=360&q=70&w=640" />\n\n\n          <div class="browse-info-container">\n              <div class="duration-container\n                is-locked\n                "\n              >\n                23:48\n              </div>\n          </div>\n        </div> \n      </a>\n\n      <div class="padding-small">\n        <div class="site-font-primary-color browse-item-title">\n           <a\n            href="https://www.criterionchannel.com/la-jetee/videos/jean-pierre-gorin-on-la-jetee"\n            data-track-event="site_video"\n            data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/la-jetee&quot;,&quot;id&quot;:464118,&quot;context&quot;:&quot;site_context_movie&quot;,&quot;label&quot;:&quot;Jean-Pierre Gorin on LA JET\xc3\x89E&quot;,&quot;canonical_collection&quot;:null}"\n          >\n            <strong title="Jean-Pierre Gorin on LA JET\xc3\x89E">\n              Jean-Pierre Gorin on LA JET\xc3\x89E\n            </strong>\n          </a>\n        </div>\n\n      </div>\n    </div>\n\n\n    <div \n         class="tooltip background-white" id="collection-tooltip-464118">\n      <h3 class="tooltip-item-title site-font-primary-family"><strong>Jean-Pierre Gorin on LA JET\xc3\x89E</strong></h3>\n\n        <div class="transparent padding-top-medium">\n          <p>In 2005, filmmaker, writer, and educator Jean-Pierre Gorin spoke with the Criterion Collection about LA JET\xc3\x89E and the work of director Chris Marker. His observations are presented here.</p>\n        </div>\n    </div>\n  </div>\n</li>\n\n\n\n\n  \n<li\n  class="js-collection-item\n    collection-item-464119 item-type-video\n    \n    "\n  data-item-id="464119"\n  data-item-type="video"\n  \n    data-tooltip="id: collection-tooltip-464119;\n    position: right;\n    offset: 0;\n    animate: true;\n    timer: 650;\n    delay: false;\n    is_centered: true;"\n    \n>\n  <div class="browse-item-card">\n    <div class="grid-item-padding">\n      <a href="https://www.criterionchannel.com/la-jetee/videos/chris-on-chris" class="browse-item-link" data-track-event="site_video" data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/la-jetee&quot;,&quot;id&quot;:464119,&quot;context&quot;:&quot;site_context_movie&quot;,&quot;label&quot;:&quot;Chris on Chris&quot;,&quot;canonical_collection&quot;:null}">\n        <div class="browse-image-container">\n            <span class="freemium-lock">\n              <i class="icon icon--xsmall icon-lock-inverse"></i>\n            </span>\n\n\n          <img alt="Chris on Chris" src="https://vhx.imgix.net/criterionchannelchartersu/assets/e1031be6-c221-4a2e-9dbb-0ed4acb7e77b-0966b827.jpg?auto=format%2Ccompress&fit=crop&h=360&q=70&w=640" />\n\n\n          <div class="browse-info-container">\n              <div class="duration-container\n                is-locked\n                "\n              >\n                09:40\n              </div>\n          </div>\n        </div> \n      </a>\n\n      <div class="padding-small">\n        <div class="site-font-primary-color browse-item-title">\n           <a\n            href="https://www.criterionchannel.com/la-jetee/videos/chris-on-chris"\n            data-track-event="site_video"\n            data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/la-jetee&quot;,&quot;id&quot;:464119,&quot;context&quot;:&quot;site_context_movie&quot;,&quot;label&quot;:&quot;Chris on Chris&quot;,&quot;canonical_collection&quot;:null}"\n          >\n            <strong title="Chris on Chris">\n              Chris on Chris\n            </strong>\n          </a>\n        </div>\n\n      </div>\n    </div>\n\n\n    <div \n         class="tooltip background-white" id="collection-tooltip-464119">\n      <h3 class="tooltip-item-title site-font-primary-family"><strong>Chris on Chris</strong></h3>\n\n        <div class="transparent padding-top-medium">\n          <p>This short piece on director Chris Marker from 2007 was made by the London-based film critic Chris Darke, who has written on all aspects of moving-image culture.</p>\n        </div>\n    </div>\n  </div>\n</li>\n\n\n\n\n  \n<li\n  class="js-collection-item\n    collection-item-464120 item-type-video\n    \n    "\n  data-item-id="464120"\n  data-item-type="video"\n  \n    data-tooltip="id: collection-tooltip-464120;\n    position: right;\n    offset: 0;\n    animate: true;\n    timer: 650;\n    delay: false;\n    is_centered: true;"\n    \n>\n  <div class="browse-item-card">\n    <div class="grid-item-padding">\n      <a href="https://www.criterionchannel.com/la-jetee/videos/on-vertigo" class="browse-item-link" data-track-event="site_video" data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/la-jetee&quot;,&quot;id&quot;:464120,&quot;context&quot;:&quot;site_context_movie&quot;,&quot;label&quot;:&quot;On VERTIGO&quot;,&quot;canonical_collection&quot;:null}">\n        <div class="browse-image-container">\n            <span class="freemium-lock">\n              <i class="icon icon--xsmall icon-lock-inverse"></i>\n            </span>\n\n\n          <img alt="On VERTIGO" src="https://vhx.imgix.net/criterionchannelchartersu/assets/ffbf5599-83ad-4abc-906c-54972cfff7f0-f8694f85.jpg?auto=format%2Ccompress&fit=crop&h=360&q=70&w=640" />\n\n\n          <div class="browse-info-container">\n              <div class="duration-container\n                is-locked\n                "\n              >\n                09:24\n              </div>\n          </div>\n        </div> \n      </a>\n\n      <div class="padding-small">\n        <div class="site-font-primary-color browse-item-title">\n           <a\n            href="https://www.criterionchannel.com/la-jetee/videos/on-vertigo"\n            data-track-event="site_video"\n            data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/la-jetee&quot;,&quot;id&quot;:464120,&quot;context&quot;:&quot;site_context_movie&quot;,&quot;label&quot;:&quot;On VERTIGO&quot;,&quot;canonical_collection&quot;:null}"\n          >\n            <strong title="On VERTIGO">\n              On VERTIGO\n            </strong>\n          </a>\n        </div>\n\n      </div>\n    </div>\n\n\n    <div \n         class="tooltip background-white" id="collection-tooltip-464120">\n      <h3 class="tooltip-item-title site-font-primary-family"><strong>On VERTIGO</strong></h3>\n\n        <div class="transparent padding-top-medium">\n          <p>VERTIGO features prominently in director Chris Marker\xe2\x80\x99s 1983 SANS SOLEIL, but Alfred Hitchcock\xe2\x80\x99s film also exerts an influence on LA JET\xc3\x89E, which Marker made twenty years earlier. This excerpt from the contemporary French television series \xe2\x80\x9cCourt-circuit (le magazine)\xe2\x80\x9d explores Marker\xe2\x80\x99s ongoing f...</p>\n        </div>\n    </div>\n  </div>\n</li>\n\n\n\n\n  \n<li\n  class="js-collection-item\n    collection-item-465250 item-type-video\n    \n    "\n  data-item-id="465250"\n  data-item-type="video"\n  \n    data-tooltip="id: collection-tooltip-465250;\n    position: right;\n    offset: 0;\n    animate: true;\n    timer: 650;\n    delay: false;\n    is_centered: true;"\n    \n>\n  <div class="browse-item-card">\n    <div class="grid-item-padding">\n      <a href="https://www.criterionchannel.com/la-jetee/videos/mira-nair-on-la-jetee" class="browse-item-link" data-track-event="site_video" data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/la-jetee&quot;,&quot;id&quot;:465250,&quot;context&quot;:&quot;site_context_movie&quot;,&quot;label&quot;:&quot;Mira Nair on LA JETEE&quot;,&quot;canonical_collection&quot;:null}">\n        <div class="browse-image-container">\n            <span class="freemium-lock">\n              <i class="icon icon--xsmall icon-lock-inverse"></i>\n            </span>\n\n\n          <img alt="Mira Nair on LA JETEE" src="https://vhx.imgix.net/criterionchannelchartersu/assets/b1dd102c-a283-4efc-8325-8f9d89747b03-3c95fc7c.jpg?auto=format%2Ccompress&fit=crop&h=360&q=70&w=640" />\n\n\n          <div class="browse-info-container">\n              <div class="duration-container\n                is-locked\n                "\n              >\n                02:25\n              </div>\n          </div>\n        </div> \n      </a>\n\n      <div class="padding-small">\n        <div class="site-font-primary-color browse-item-title">\n           <a\n            href="https://www.criterionchannel.com/la-jetee/videos/mira-nair-on-la-jetee"\n            data-track-event="site_video"\n            data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/la-jetee&quot;,&quot;id&quot;:465250,&quot;context&quot;:&quot;site_context_movie&quot;,&quot;label&quot;:&quot;Mira Nair on LA JETEE&quot;,&quot;canonical_collection&quot;:null}"\n          >\n            <strong title="Mira Nair on LA JETEE">\n              Mira Nair on LA JETEE\n            </strong>\n          </a>\n        </div>\n\n      </div>\n    </div>\n\n\n    <div \n         class="tooltip background-white" id="collection-tooltip-465250">\n      <h3 class="tooltip-item-title site-font-primary-family"><strong>Mira Nair on LA JETEE</strong></h3>\n\n    </div>\n  </div>\n</li>\n\n\n\n\n  \n<li\n  class="js-collection-item\n    collection-item-605936 item-type-video\n    \n    "\n  data-item-id="605936"\n  data-item-type="video"\n  \n    data-tooltip="id: collection-tooltip-605936;\n    position: right;\n    offset: 0;\n    animate: true;\n    timer: 650;\n    delay: false;\n    is_centered: true;"\n    \n>\n  <div class="browse-item-card">\n    <div class="grid-item-padding">\n      <a href="https://www.criterionchannel.com/la-jetee/videos/rian-johnson-on-la-jetee" class="browse-item-link" data-track-event="site_video" data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/la-jetee&quot;,&quot;id&quot;:605936,&quot;context&quot;:&quot;site_context_movie&quot;,&quot;label&quot;:&quot;Rian Johnson on LA JET\xc3\x89E&quot;,&quot;canonical_collection&quot;:null}">\n        <div class="browse-image-container">\n            <span class="freemium-lock">\n              <i class="icon icon--xsmall icon-lock-inverse"></i>\n            </span>\n\n\n          <img alt="Rian Johnson on LA JET\xc3\x89E" src="https://vhx.imgix.net/criterionchannelchartersu/assets/709f916b-b6e6-4a0e-bdef-ff5632482b90-fd1ec42c.jpg?auto=format%2Ccompress&fit=crop&h=360&q=70&w=640" />\n\n\n          <div class="browse-info-container">\n              <div class="duration-container\n                is-locked\n                "\n              >\n                02:35\n              </div>\n          </div>\n        </div> \n      </a>\n\n      <div class="padding-small">\n        <div class="site-font-primary-color browse-item-title">\n           <a\n            href="https://www.criterionchannel.com/la-jetee/videos/rian-johnson-on-la-jetee"\n            data-track-event="site_video"\n            data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/la-jetee&quot;,&quot;id&quot;:605936,&quot;context&quot;:&quot;site_context_movie&quot;,&quot;label&quot;:&quot;Rian Johnson on LA JET\xc3\x89E&quot;,&quot;canonical_collection&quot;:null}"\n          >\n            <strong title="Rian Johnson on LA JET\xc3\x89E">\n              Rian Johnson on LA JET\xc3\x89E\n            </strong>\n          </a>\n        </div>\n\n      </div>\n    </div>\n\n\n    <div \n         class="tooltip background-white" id="collection-tooltip-605936">\n      <h3 class="tooltip-item-title site-font-primary-family"><strong>Rian Johnson on LA JET\xc3\x89E</strong></h3>\n\n    </div>\n  </div>\n</li>\n\n\n\n\n  \n<li\n  class="js-collection-item\n    collection-item-464121 item-type-video\n    \n    "\n  data-item-id="464121"\n  data-item-type="video"\n  \n    data-tooltip="id: collection-tooltip-464121;\n    position: right;\n    offset: 0;\n    animate: true;\n    timer: 650;\n    delay: false;\n    is_centered: true;"\n    \n>\n  <div class="browse-item-card">\n    <div class="grid-item-padding">\n      <a href="https://www.criterionchannel.com/la-jetee/videos/david-bowie-s-jump-they-say" class="browse-item-link" data-track-event="site_video" data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/la-jetee&quot;,&quot;id&quot;:464121,&quot;context&quot;:&quot;site_context_movie&quot;,&quot;label&quot;:&quot;David Bowie\xe2\x80\x99s \xe2\x80\x9cJump They Say\xe2\x80\x9d&quot;,&quot;canonical_collection&quot;:null}">\n        <div class="browse-image-container">\n            <span class="freemium-lock">\n              <i class="icon icon--xsmall icon-lock-inverse"></i>\n            </span>\n\n\n          <img alt="David Bowie\xe2\x80\x99s \xe2\x80\x9cJump They Say\xe2\x80\x9d" src="https://vhx.imgix.net/criterionchannelchartersu/assets/7e20262e-2434-44e2-914c-6cb61e1e596c-98a4af79.jpg?auto=format%2Ccompress&fit=crop&h=360&q=70&w=640" />\n\n\n          <div class="browse-info-container">\n              <div class="duration-container\n                is-locked\n                "\n              >\n                01:52\n              </div>\n          </div>\n        </div> \n      </a>\n\n      <div class="padding-small">\n        <div class="site-font-primary-color browse-item-title">\n           <a\n            href="https://www.criterionchannel.com/la-jetee/videos/david-bowie-s-jump-they-say"\n            data-track-event="site_video"\n            data-track-event-properties="{&quot;type&quot;:&quot;video&quot;,&quot;location&quot;:&quot;https://criterionchannel.vhx.tv/la-jetee&quot;,&quot;id&quot;:464121,&quot;context&quot;:&quot;site_context_movie&quot;,&quot;label&quot;:&quot;David Bowie\xe2\x80\x99s \xe2\x80\x9cJump They Say\xe2\x80\x9d&quot;,&quot;canonical_collection&quot;:null}"\n          >\n            <strong title="David Bowie\xe2\x80\x99s \xe2\x80\x9cJump They Say\xe2\x80\x9d">\n              David Bowie\xe2\x80\x99s \xe2\x80\x9cJump They Say\xe2\x80\x9d\n            </strong>\n          </a>\n        </div>\n\n      </div>\n    </div>\n\n\n    <div \n         class="tooltip background-white" id="collection-tooltip-464121">\n      <h3 class="tooltip-item-title site-font-primary-family"><strong>David Bowie\xe2\x80\x99s \xe2\x80\x9cJump They Say\xe2\x80\x9d</strong></h3>\n\n        <div class="transparent padding-top-medium">\n          <p>This brief excerpt from the contemporary French television series \xe2\x80\x9cCourt-circuit (le magazine)\xe2\x80\x9d looks at David Bowie\xe2\x80\x99s 1993 music video \xe2\x80\x9cJump They Say\xe2\x80\x9d\xe2\x80\x94directed by Mark Romanek and shot by Harris Savides\xe2\x80\x94and the ways it pays homage to LA JET\xc3\x89E.</p>\n        </div>\n    </div>\n  </div>\n</li>\n\n\n\n\n    </ul>\n  </div>\n\n          \n        </div>\n      </div>\n    </section>\n</section>\n\n\n    <style>\n  a.site-link-footer-color,\n  a.site-link-footer-color:visited {\n    color: #ffffff !important;\n  }\n  .btn-site-footer,\n  .btn-site-footer:link,\n  .btn-site-footer:visited {\n    color: #b9bcc7;\n    background-color: #000;\n  }\n  .btn-site-footer:hover,\n  .btn-site-footer.is-hover,\n  .btn-site-footer:focus,\n  .btn-site-footer:focus:hover,\n  .btn-site-footer:active,\n  .btn-site-footer.is-active {\n    color: #000;\n    border-color: #000;\n    background-color: #b9bcc7;\n  }\n\n  .btn-logout-form {\n    display: inline;\n    line-height: 0;\n  }\n\n  .btn-logout {\n    background-color: transparent;\n    border: none;\n    cursor: pointer;\n    font-size: 12px;\n    padding: 0;\n  }\n\n  .btn-logout:hover {\n    opacity: .5;\n  }\n</style>\n\n\n<footer class="footer--site border-top site-border-color site-background-color padding-top-medium padding-bottom-medium ">\n  <div class="row full-width">\n    <div class="small-16 medium-6 large-10 columns helper-links small-only-text-center">\n      <div class="padding-left-medium display-flex-container flex-start force-min-height">\n              <a href="https://www.criterion.com/" class="site-font-secondary-color" target="_blank" rel="noopener">THE CRITERION COLLECTION</a>\n\n\n        <a href="/help" class="site-font-secondary-color legal-link help-link" rel="noopener">\n          Help\n        </a>\n        <a href="/tos" class="site-font-secondary-color legal-link terms-link" rel="noopener">\n          Terms\n        </a>\n        <a href="/privacy" class="site-font-secondary-color legal-link privacy-link" rel="noopener">\n          Privacy\n        </a>\n        <a href="/cookies" class="site-font-secondary-color legal-link cookies-link" rel="noopener">\n          Cookies\n        </a>\n\n\n            <a href="/login" class="site-font-secondary-color signin-link">\n              Sign in\n            </a>\n      </div>\n    </div> <!-- columns -->\n    <div class="small-16 medium-10 large-6 small-only-text-center text-right columns vhx-link">\n      <div class="padding-right-small display-flex-container flex-end force-min-height">\n          &nbsp;\n        <div id="language-switcher-react-app"\n             class="padding-right-small"\n             data-supported-locales="[]"\n        </div>\n        <script data-cfasync="false" src="/cdn-cgi/scripts/5c5dd728/cloudflare-static/email-decode.min.js"></script><script src="https://cdn.vhx.tv/assets/vite/assets/language-switcher.c19b2f61.js" crossorigin="anonymous" type="module"></script><link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/jsx-runtime.75dbc984.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/LanguageSwitcher.2b83283a.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/index.71ea8908.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/intl.helpers.2a168b49.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/index.61149844.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/deviceSizes.9973a5fd.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/message.bd28a54d.js" as="script" crossorigin="anonymous" />\n      </div> <!-- display-flex-container -->\n    </div>\n  </div> <!-- row -->\n</footer>\n\n<div class="modal">\n  <div class="close"><a href="#close" alt="Close Window"><strong class="is-inverse">&times;</strong></a></div>\n  <div class="content"></div>\n</div>\n\n<div id="blanket"></div>\n\n\n  <div class="modals">\n    \n  </div>\n\n  <script src="https://cdn.vhx.tv/assets/webpack/vendor-ottbundle-5b2989a58aa1f2ae17a5.js"></script>\n\n  <script>\n    window.VHX = window.VHX || {};\n    window.VHX.config = {\n      api_url: "https://api.vhx.tv",\n      token: "eyJhbGciOiJSUzI1NiIsImtpZCI6ImQ2YmZlZmMzNGIyNTdhYTE4Y2E2NDUzNDE2ZTlmZmRjNjk4MDAxMDdhZTQ2ZWJhODg0YTU2ZDBjOGQ4NTYzMzgifQ.eyJhcHBfaWQiOjY4NTExLCJleHAiOjE2NzIwMDU3NzgsIm5vbmNlIjoiOWMwM2RhYzRjOGY2OTkzZCIsInNjb3BlcyI6WyJwdWJsaWMiXSwic2Vzc2lvbl9pZCI6InNBTHY0Z3FrdU5CaFhyYWJPNzA4dVE9PSIsInNpdGVfaWQiOjU5MDU0fQ.hDkqY_8qhH87V-E5TwQukxPGiISiNKVecs8_foJsThFAUEhwCR_8bTdiecMIN9-Oj0HyaFXLh1bFX-uVXPZOTU7PhDLbN4BBdSYqTz9m7WhpahkTJ_--NHfHMSAUSZplQ6OCNIBkQI4r0cEh-1Egf5xoUa4eY80hxCvrQfvWXQa1c6zjmxGBKZsu4WxRgquVDEoYtfnSC6IHaN8koqzzt9fvBXuVT8OBzjg7-GjMUTqzFBCb8u0wujrxwg7E1Rzk7JD6uzwNzQk5WNoTnwD3kUSsQkk5FbGmLcJcUULwwv1sh8TPXrVqlNMsbz7KzhxcnRPOPZ5_0pGi7U6I0qCfGQVc0nR0BLHr68r9II8NH8sb8WQsSeFoC5KKLVLr4NRqUBeJPi4uFNkI-jO7us93dbrleuWu7IRziXR8y47N5nNLxZRszMlRixy3TFpf8uU1iyzV5QvSrw4CPDPoL5SrtYnjFTojp7fkq_UEYgrZObjLovqmAkUkFKhrVJfWy4KdWc_S1-lBe-pgVvGM-q1Ec3XMayIrBb0HZgdpJw1ypDorUTuWwhnwMjsRtKNAD37eOE1FCWlB_JH9A-hqR_iGsXfEL9tcJUqrNdQPqkcMG9p9XeX-7C786rzM7QcnIhhMUSW4LzAml67ZSK8P6hPPgK9-vQcksZWjcfXYFiGfvLw",\n      token_expires_in: "7200000",\n      user_has_subscription: "false"\n    };\n  </script>\n\n  <script src="https://cdn.vhx.tv/assets/webpack/sites-ottbundle-1e2d12c471814f99fb7e.js"></script>\n  \n\n      <script>\n    window.VHX = window.VHX || {};\n    window.VHX.config = {\n      api_url: "https://api.vhx.tv",\n      token: "eyJhbGciOiJSUzI1NiIsImtpZCI6ImQ2YmZlZmMzNGIyNTdhYTE4Y2E2NDUzNDE2ZTlmZmRjNjk4MDAxMDdhZTQ2ZWJhODg0YTU2ZDBjOGQ4NTYzMzgifQ.eyJhcHBfaWQiOjY4NTExLCJleHAiOjE2NzIwMDU3NzgsIm5vbmNlIjoiOWMwM2RhYzRjOGY2OTkzZCIsInNjb3BlcyI6WyJwdWJsaWMiXSwic2Vzc2lvbl9pZCI6InNBTHY0Z3FrdU5CaFhyYWJPNzA4dVE9PSIsInNpdGVfaWQiOjU5MDU0fQ.hDkqY_8qhH87V-E5TwQukxPGiISiNKVecs8_foJsThFAUEhwCR_8bTdiecMIN9-Oj0HyaFXLh1bFX-uVXPZOTU7PhDLbN4BBdSYqTz9m7WhpahkTJ_--NHfHMSAUSZplQ6OCNIBkQI4r0cEh-1Egf5xoUa4eY80hxCvrQfvWXQa1c6zjmxGBKZsu4WxRgquVDEoYtfnSC6IHaN8koqzzt9fvBXuVT8OBzjg7-GjMUTqzFBCb8u0wujrxwg7E1Rzk7JD6uzwNzQk5WNoTnwD3kUSsQkk5FbGmLcJcUULwwv1sh8TPXrVqlNMsbz7KzhxcnRPOPZ5_0pGi7U6I0qCfGQVc0nR0BLHr68r9II8NH8sb8WQsSeFoC5KKLVLr4NRqUBeJPi4uFNkI-jO7us93dbrleuWu7IRziXR8y47N5nNLxZRszMlRixy3TFpf8uU1iyzV5QvSrw4CPDPoL5SrtYnjFTojp7fkq_UEYgrZObjLovqmAkUkFKhrVJfWy4KdWc_S1-lBe-pgVvGM-q1Ec3XMayIrBb0HZgdpJw1ypDorUTuWwhnwMjsRtKNAD37eOE1FCWlB_JH9A-hqR_iGsXfEL9tcJUqrNdQPqkcMG9p9XeX-7C786rzM7QcnIhhMUSW4LzAml67ZSK8P6hPPgK9-vQcksZWjcfXYFiGfvLw",\n      token_expires_in: "7200000",\n      user_has_subscription: "false"\n    };\n  </script>\n\n  <script src="https://cdn.vhx.tv/assets/webpack/products-ottbundle-12994ed95e958f7c2e65.js"></script>\n    <script src="https://cdn.vhx.tv/assets/vite/assets/search.f6a28d3c.js" crossorigin="anonymous" type="module"></script><link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/jsx-runtime.75dbc984.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/index.esm.f84f874f.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/index.0e8c968f.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/intl.helpers.2a168b49.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/api.utils.c8000605.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/Menu.esm.959a8f77.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/Button.esm.61e4a4cb.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/Text.esm.c0f99a84.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/message.bd28a54d.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/Checkmark.esm.fb551793.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/Link.esm.a6ddba18.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/Search.esm.a44cb748.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/index.71ea8908.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/text-helpers.5cfd261d.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/time-helpers.4df27d81.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/_includes.3534e71e.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/_curry2.3e499763.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/path.235ca80c.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/Header.esm.2a01f61a.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/DismissX.esm.54e5d2b0.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/colors.esm.e8046dae.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/inheritsLoose.a8857277.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/index.61149844.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/ChevronDown.esm.405e6674.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/_isArray.68eb59b2.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/index.096cf963.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/equals.d9a066f9.js" as="script" crossorigin="anonymous" />\n<link rel="modulepreload" href="https://cdn.vhx.tv/assets/vite/_isString.019dcf3c.js" as="script" crossorigin="anonymous" /><link rel="stylesheet" media="screen" href="https://cdn.vhx.tv/assets/vite/assets/search.c6f13361.css" crossorigin="anonymous" />\n\n\n<script>\n  setTimeout(function() {\n    document.documentElement.className += \' is-loaded\';\n    var selects = document.getElementsByClassName(\'select\'),\n        checkboxes = document.getElementsByClassName(\'checkbox\'),\n        i;\n\n    if (selects.length) {\n      for (i = 0; i < selects.length; i++) {\n        selects[i].className += \' ready\';\n      }\n    }\n\n    if (checkboxes.length) {\n      for (i = 0; i < checkboxes.length; i++) {\n        for (var p=0; p < checkboxes[i].childNodes.length; p++) {\n          if (checkboxes[i].childNodes[p].tagName == \'LABEL\') {\n            checkboxes[i].childNodes[p].className += \' ready\';\n            break;\n          }\n        }\n      }\n    }\n  }, 500);\n</script>\n\n  <script>\n  window.addEventListener(\'load\', function() {\n    jQuery(\'a:contains("Sign In")\').click(function(){\n      ga(\'site.send\',\'event\',\'button\',\'click\',\'sign in\');\n    });\n\n    jQuery(\'a:contains("Sign Up")\').click(function(){\n      ga(\'site.send\',\'event\',\'button\',\'click\',\'sign up\');\n    });\n\n    jQuery(\'a:contains("Watch now")\').click(function(){\n      ga(\'site.send\',\'event\',\'button\',\'click\',\'watch now\');\n    });\n\n    jQuery(\'a:contains("Share")\').click(function(){\n      ga(\'site.send\',\'event\',\'button\',\'click\',\'share\');\n    });\n\n    jQuery(\'[href="https://www.criterionchannel.com/the-deviant-and-the-divine/season:1/videos/freaks"],[href="https://www.criterionchannel.com/the-deviant-and-the-divine/season:1/videos/multiple-maniacs"]\').click(function(){\n      ga(\'site.send\',\'event\',\'button\',\'click\',\'video\');\n    });\n  });\n</script>\n\n\n\n    <script>\n      const javascript = \'\\$(document).ready(function(){\\n\\$(\\\'header .logo\\\').attr(\\\'data-swiftype-index\\\', false);\\n\\n\\$(\\"a[href*=\\\'/checkout\\\']\\").attr(\\"data html(\\"Sign Up\\");\\n  \\$(\\"a[href*=\\\'/checkout/subscribe/gift\\\']\\").html(\\"Buy a gift\\");\\n  \\$(\\"a[href*=\\\'/buy/subscription\\\']\\").attr(\\"data-trigger-disabled\\",true);\\n  \\$(\\"a[href*=\\\'/buy/subscription\\\']\\").html(\\"Sign Up\\");\\n  \\$(\\"a.browse-link\\").html(\\"Now Playing\\");\\n  landingPageChanges();\\n  collectionChanges();\\n  helpChanges();\\n  \\$(\\"body.browse .slide-subtitle\\").hide();\\n  \\$(\\"body.logged-in h1.identity a\\").attr(\\"href\\",\\"/\\");\\n  \\$(\\".navigation-inner\\").addClass(\\"is-ready\\");\\n  \\$(\\"nav .custom-nav-all-films\\").attr(\\"target\\", \\"_self\\");\\n\\n  amazon_fire_tv_supported = \\$(\\\'.amazon-fire-tv\\\').parent().parent().children()[1]\\n  \\$(amazon_fire_tv_supported).text(\\"2nd Gen +\\")\\n\\n  android_tv_supported_text = \\$(\\\'.android-tv\\\').parent().parent().children()[1]\\n  \\$(android_tv_supported_text).text(\\"Selected models\\")\\n  android_tv_4k_text = \\$(\\\'.android-tv\\\').parent().parent().children()[2]\\n  \\$(android_tv_4k_text).hide()\\n\\n  new_xbox = \\"<li>Xbox Series X|S<\\/li>\\";\\n  \\$(new_xbox).insertAfter(\\$(\\\'.platforms-modal-content-tv .platforms-modal-icon-container .platform-modal-devices-list > li:nth-of-type(5) li:nth-of-type(2)\\\'));\\n\\n  \\$(\\\'.platforms-modal-content-tv .platforms-modal-icon-container .platform-modal-devices-list > li:nth-of-type(6) li:nth-of-type(2)\\\').text(\\\'Select 2018, 2019 & 2020 Models\\\');\\n\\n  gift_link = \\\'<a href=\\"https://www.criterionchannel.com/checkout/subscribe/gift?gift=1&duration=6\\" class=\\"is-block text dark block padding-medium\\">Give a Gift<\\/a>\\\';\\n  \\$(gift_link).insertAfter(\\$(\\\'.user-dropdown a[href*=\\"/purchases\\"]\\\'));\\n\\n  if (\\$(\\\'meta[name*=\\"twitter:image\\"]\\\').length > 0) {\\n    twitter_content = \\$(\\\'meta[name*=\\"twitter:image\\"]\\\').attr(\\\'content\\\')\\n    new_meta = \\\'<meta class=\\"swiftype\\" name=\\"image\\" data-type=\\"enum\\" content=\\"\\\' + twitter_content + \\\'\\">\\\';\\n    \\$(\\\'head\\\').append(new_meta);\\n  }\\n});\\n\\nfunction collectionChanges(){\\n  if (\\$(\\"body\\").hasClass(\\"collections\\")){\\n    \\$(\\\'.cover-art\\\').removeClass(\\\'large-offset-2 large-6\\\').addClass(\\\'large-8 large-offset-1\\\');\\n\\n    \\$(\\\'section.product-feature\\\').removeClass(\\\'padding-horizontal-medium padding-top-large padding-bottom-xlarge\\\').addClass(\\\'padding-bottom-large\\\');\\n    \\$(\\\'section.product-feature .row.relative\\\').addClass(\\\'full-width\\\');\\n\\n    \\$(\\\'section.product-feature .collection-details-container\\\').removeClass(\\"large-8 end\\").addClass(\\"large-7\\");\\n      \\$(\\\'section.product-feature .collection-details\\\').addClass(\\"padding-left-large padding-right-large padding-top-xlarge\\");\\n\\n      \\$(\\".embed-container\\").removeClass(\\"grid-padding-right\\");\\n  }\\n  if (\\$(\\\'.collection-playlist\\\').length > 0 || \\$(\\\'.collection-series\\\').length > 0) {\\n    \\$(\\\'.video-controls\\\').removeClass(\\\'padding-top-large\\\').addClass(\\\'padding-top-small\\\');\\n  }\\n\\n  if (\\$(\\\'.collection-movie\\\').length > 0) {\\n    \\$(\\\'.collection-actions a:contains(\\"Trailer\\")\\\').html(\\\'<span class=\\"icon-adjustment relative inverse-icon-bg dark-bg margin-bottom-small\\"><i class=\\"icon small icon-play-inverse\\"><\\/i><\\/span> Trailer\\\');\\n    \\$(\\\'.collection-actions a:contains(\\"Movie\\")\\\').html(\\\'<span class=\\"icon-adjustment relative inverse-icon-bg dark-bg margin-bottom-small\\"><i class=\\"icon small icon-play-inverse\\"><\\/i><\\/span> Watch\\\');\\n  }\\n\\n\\n}\\n\\nfunction landingPageChanges(){\\n  \\$(\\\'a:contains(\\"Start Your Free Month\\")\\\').text(\\"Sign Up Now\\");\\n}\\nfunction helpChanges(){\\n  // Custom Help Center Updates\\n  setTimeout(function() {\\n    \\$(\\\'article li:contains(\\"You can plug your laptop or mobile device into your TV using an HDMI cable\\")\\\').each(function() {\\n      \\$(this).html(\\\'<li>It may be possible to connect your laptop or mobile device to your TV using an HDMI cable if all components in the chain are <a href=\\"https://www.criterionchannel.com/help/articles/drm-and-hdcp#HDCP\\">HDCP/DRM compliant.<\\/a><\\/li>\\\');\\n    });\\n\\n    \\$(\\\'article li:contains(\\" and select the language of choice\\")\\\').each(function() {\\n      \\$(this).text(\\$(this).text().replace(\\" and select the language of choice\\", \\"\\"));\\n    });\\n    \\n    a = \\\'<li>If you have a supported device connected to your TV, Criterion also has an app for Apple TV, Amazon Fire, Roku, and Android TV.<\\/li>\\\'\\n    b = \\$(\\\'li:contains(\\"You can mirror or Airplay from your iOS device\xe2\x80\x99s browser.\\")\\\')\\n    b.after(a);\\n\\n    \\$(\\\'li a[href*=\\"670-how-do-i-login-on-the-xbox-app\\"]\\\').parent().hide();\\n  }, 500);\\n\\n  // hide mentions of the VHX articles\\n  // we have to settimeout because the help pages are ajax\\\'ing in content.\\n  // this is best guess on timing\\n  if (\\$(\\"body\\").hasClass(\\"help-articles\\") || \\$(\\"body\\").hasClass(\\"sites-help\\") || \\$(\\"body\\").hasClass(\\"help-categories\\")){\\n   \\$(\\\'#mithril-help\\\').bind(\\"DOMSubtreeModified\\",function(){\\n    var waittoremove = setTimeout(function(){ removeHelpArticles(); }, 500);\\n  });\\n\\n    var waittoremove = setTimeout(function(){ removeHelpArticles(); }, 700);\\n  }\\n}\\nfunction removeHelpArticles(){\\n  var vhxhelp = Array(\\"589e1e742c7d3a784630ccd5\\",\\"581396999033604deb0ec504\\",\\"57fe5ceb9033600277a6980b\\",\\"54b59261e4b0934e822d2132\\",\\"54dbc064e4b086c0c0966908\\",\\"5453fe87e4b02ce342671537\\",\\"555119ebe4b027e1978dfc04\\",\\"549206b9e4b07d03cb251286\\", \\"5c7eee4f04286350d088ab37\\");\\n  for (i=0;i<=vhxhelp.length-1;i++){\\n    \\$(\\".article-\\"+vhxhelp[i]).remove();                                                        \\n  }\\n}\';\n      try {\n        // Using eval so we can catch syntax errors\n        eval(javascript);\n      } catch (err) {\n        console.error(err);\n      }\n    </script>\n</body>\n</html>\n'
    return BeautifulSoup(content, 'html5lib')
