import os
import argparse
import json
import typing as t
from abc import ABC, abstractmethod

from icortex.defaults import DEFAULT_CACHE_PATH
from icortex.helper import prompt_input


def is_str_repr(s: str):
    quotes = ["'", '"']
    return len(s) >= 2 and s[0] in quotes and s[-1] in quotes


class ServiceVariable:
    """A variable for a code generation service

    Args:
        type_ (Any): Variable type.
        default (Any, optional): Default value, should match :data:`type_`.
        help (str, optional): Help string for the variable. Defaults to "".
        secret (bool, optional): When set to
            True, the variable is omitted from caches and the context. Defaults to False.
        argparse_args (List, optional): Args to
            be given to :func:`ArgumentParser.add_argument`. Defaults to [].
        argparse_kwargs (Dict, optional): Keywords args to
            be given to :func:`ArgumentParser.add_argument`. Defaults to {}.
        require_arg (bool, optional): When set to true,
            the prompt parser will raise an error if the variable is not specified.
            Defaults to False.
    """
    def __init__(
        self,
        type_: type,
        default: t.Any = None,
        help: str = "",
        secret: bool = False,
        argparse_args: t.List = [],
        argparse_kwargs: t.Dict = {},
        require_arg: bool = False,
    ):
        self.argparse_args = [*argparse_args]
        self.argparse_kwargs = {**argparse_kwargs}
        self.type = type_
        self.help = help
        self.set_default(default)
        self.set_help(help)
        self.secret = secret
        self.require_arg = require_arg
        if require_arg:
            self.argparse_kwargs["required"] = True
        if self.type is not None:
            self.argparse_kwargs["type"] = self.type

    def set_default(self, val):
        if val is None:
            self.default = None
            return
        assert isinstance(val, self.type)
        self.default = val
        self.argparse_kwargs["default"] = val
        # Update the help string
        self.set_help(self.help)

    def set_help(self, help: str):
        if help is not None:
            help_str = help
            if self.default is not None:
                help_str += f" Default: {repr(self.default)}"
            self.argparse_kwargs["help"] = help_str
        else:
            self.help = None


class ServiceBase(ABC):
    """Abstract base class for interfacing a code generation service.
    Its main purpose is to provide a flexible API for connecting user
    prompts with whatever logic the service
    provider might choose to implement. User prompts adhere to
    POSIX argument syntax and are parsed with
    `argparse <https://docs.python.org/3/library/argparse.html>`__.

    To create a new service:

    - Assign a unique name to :attr:`name`
    - Add your class to the dict :data:`icortex.services.AVAILABLE_SERVICES`.
      Use :attr:`name` as the key and don't forget to include module information.
    - Determine the parameters that the service will use for code generation and add
      them to :attr:`variables`.
    - Implement :func:`generate`.

    Check out :class:`icortex.services.textcortex.TextCortexService` as a
    reference implementation.

    Attributes
    ----------
    variables: Dict[str, ServiceVariable]
        A dict that maps variable names to :class:`ServiceVariable` s.
    name: str
        A unique name.
    description: str
        Description string.
    prompt_parser: argparse.ArgumentParser
        Parser to parse the prompts.
    """

    name: str = "base"
    description: str = "Base class for a code generation service"
    # Each child class will need to add their specific arguments
    # by extending `variables`
    variables: t.Dict[str, ServiceVariable] = {}
    # This has stopped working, fix
    hidden: bool = False

    def __init__(self, **kwargs: t.Dict[str, t.Any]):
        """Classes that derive from ServiceBase are always initialized with
        keyword arguments that contain values for the service variables.
        The values can come
        """
        # Create the prompt parser and add default arguments
        self.prompt_parser = argparse.ArgumentParser(
            add_help=False,
        )
        self.prompt_parser.add_argument(
            "prompt",
            nargs="*",
            type=str,
            help="The prompt that describes what the generated Python code should perform.",
        )
        self.prompt_parser.add_argument(
            "-e",
            "--execute",
            action="store_true",
            required=False,
            help="Execute the Python code returned by TextCortex API directly.",
        )
        self.prompt_parser.add_argument(
            "-r",
            "--regenerate",
            action="store_true",
            required=False,
            help="Make the kernel ignore cached responses and make a new request to TextCortex API.",
        )
        self.prompt_parser.add_argument(
            "-i",
            "--include-history",
            action="store_true",
            required=False,
            help="Submit notebook history along with the prompt.",
        )
        self.prompt_parser.add_argument(
            "-p",
            "--auto-install-packages",
            action="store_true",
            required=False,
            help="Auto-install packages that are imported in the generated code but missing in the active Python environment.",
        )
        self.prompt_parser.add_argument(
            "-o",
            "--nonint",
            action="store_true",
            required=False,
            help=f"Non-interactive, do not ask any questions.",
        )
        self.prompt_parser.add_argument(
            "-q",
            "--quiet",
            action="store_true",
            required=False,
            help="Do not print the generated code.",
        )
        self.prompt_parser.usage = (
            "%%prompt your prompt goes here [-e] [-r] [-i] [-p] ..."
        )

        self.prompt_parser.description = self.description

        # Add service-specific variables
        for key, var in self.variables.items():
            # If user has specified a value for the variable, use that
            # Otherwise, the default value will be used
            if key in kwargs:
                var.set_default(kwargs[key])

            # Omit secret arguments from the parser, but still read them
            if var.secret == False and len(var.argparse_args) > 0:
                self.prompt_parser.add_argument(
                    *var.argparse_args,
                    **var.argparse_kwargs,
                )

    def find_cached_response(
        self,
        request_dict: t.Dict,
        cache_path: str = DEFAULT_CACHE_PATH,
    ):
        cache = self._read_cache(cache_path)
        # If the the same request is found in the cache, return the cached response
        # Return the latest found response by default
        for dict_ in reversed(cache):
            if dict_["request"] == request_dict:
                return dict_["response"]
        return None

    def cache_response(
        self,
        request_dict: t.Dict,
        response_dict: t.Dict,
        cache_path: str = DEFAULT_CACHE_PATH,
    ):
        cache = self._read_cache(DEFAULT_CACHE_PATH)
        cache.append({"request": request_dict, "response": response_dict})
        return self._write_cache(cache, cache_path)

    @abstractmethod
    def generate(
        self,
        prompt: str,
        context: t.Dict[str, t.Any] = {},
    ) -> t.List[t.Dict[t.Any, t.Any]]:
        """Implement the logic that generates code from user prompts here.

        Args:
            prompt (str): The prompt that describes what the generated code should perform
            context (Dict[str, Any], optional): A dict containing the current notebook
                context, that is in the Jupyter notebook format.
                See :class:`icortex.context.ICortexHistory` for more details.

        Returns:
            List[Dict[Any, Any]]: A list that contains code generation results. Should ideally be valid Python code.
        """
        raise NotImplementedError

    def config_dialog(self, skip_defaults=False):
        return_dict = {}
        for key, var in self.variables.items():
            if isinstance(var, ServiceVariable):
                if skip_defaults and var.default is not None:
                    user_val = var.default
                else:
                    kwargs = {"type": var.type}
                    if var.default is not None:
                        kwargs["default"] = repr(var.default)
                    if var.help is not None:
                        prompt = f"{key} ({var.help})"
                    else:
                        prompt = f"{key}"
                    user_val = prompt_input(prompt, **kwargs)
                    # If the input is a string representation, evaluate it
                    # This is for when the user wants to type in a string with escape characters
                    if var.type == str and is_str_repr(user_val):
                        user_val = eval(user_val)
            else:
                raise ValueError(f"Dict entry is not ServiceVariable: {var}")

            return_dict[key] = user_val
        return return_dict

    def get_variable(self, var_name: str) -> ServiceVariable:
        """Get a variable by its name

        Args:
            var_name (str): Name of the variable

        Returns:
            ServiceVariable: Requested variable
        """
        for key, var in self.variables.items():
            if key == var_name:
                return var
        return None

    def get_variable_names(self) -> t.List[str]:
        """Get a list of variable names.

        Returns:
            List[str]: List of variable names
        """
        return [var.name for var in self.variables]

    def _read_cache(self, cache_path):
        # Check whether the cache file already exists
        if os.path.exists(cache_path):
            try:
                with open(cache_path, "r") as f:
                    cache = json.load(f)
            except json.decoder.JSONDecodeError:
                cache = []
        else:
            cache = []
        return cache

    def _write_cache(self, cache: t.List[t.Dict], cache_path):
        with open(cache_path, "w") as f:
            json.dump(cache, f, indent=2)
        return True
