# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pyctuator',
 'pyctuator.environment',
 'pyctuator.health',
 'pyctuator.httptrace',
 'pyctuator.impl',
 'pyctuator.logging',
 'pyctuator.metrics',
 'pyctuator.threads']

package_data = \
{'': ['*'], 'pyctuator': ['logfile/*']}

extras_require = \
{'db': ['sqlalchemy>=1.3,<2.0',
        'PyMySQL>=0.9.3,<0.10.0',
        'cryptography>=2.8,<3.0'],
 'fastapi': ['fastapi>=0.41.0,<0.42.0', 'uvicorn>=0.9.0,<0.10.0'],
 'flask': ['flask>=1.1,<2.0'],
 'psutil': ['psutil>=5.6,<6.0'],
 'redis': ['redis>=3.3,<4.0']}

setup_kwargs = {
    'name': 'pyctuator',
    'version': '0.9',
    'description': 'A Python implementation of the Spring Actuator API for popular web frameworks',
    'long_description': '[![PyPI](https://img.shields.io/pypi/v/pyctuator?color=green&style=plastic)](https://pypi.org/project/pyctuator/)\n[![build](https://github.com/SolarEdgeTech/pyctuator/workflows/build/badge.svg)](https://github.com/SolarEdgeTech/pyctuator/)\n[![Codecov](https://img.shields.io/codecov/c/github/SolarEdgeTech/pyctuator?style=plastic)](https://codecov.io/gh/SolarEdgeTech/pyctuator)\n\n# Pyctuator\n\nMonitoring your Python microservices as if they were Spring\nBoot applications using \n[Spring Boot Admin](https://github.com/codecentric/spring-boot-admin). \n\nThe supported web frameworks are **Flask** and **FastAPI**.\n\nSupport for **Django** is planned as well.\n\nThe following video shows Spring Boot Admin to monitoring and controlling an instance of the [Advanced example](examples/Advanced/README.md):\n![Pyctuator Example](examples/images/Pyctuator_Screencast.gif)\n\n## Requirements\nPython 3.7+\n\nPyctuator has zero hard dependencies.\n\n## Installing\nInstall Pyctuator using pip: `pip3 install pyctuator`\n\n## Why?\nMany Java shops use Spring Boot as their main web framework for developing\nmicroservices. \nThese organizations often use Spring Actuator together with Spring Boot Admin\nto monitor their microservices\' status, gain access to applications\'\n state and configuration, manipulate log levels, etc.\n \nThese organizations often have the occasional Python microservice, especially as\nPython Machine Learning and Data Science packages gain popularity. Setting up\na proper monitoring tool for these microservices is a complex task, and might\nnot be justified for a few Python microservices in a sea of Java microservices.\n\nThis is where Pyctuator comes in. It allows you to easily integrate your Python\nmicroservices into your existing Spring Boot Admin deployment.\n\n## Main Features\nPyctuator is a partial Python implementation of the \n[Spring Actuator API](https://docs.spring.io/spring-boot/docs/2.1.8.RELEASE/actuator-api/html/)  . \n\nIt currently supports the following Actuator features:\n\n* **Application details**\n* **Metrics**\n    * Memory usage\n    * Disk usage \n    * Easily add custom metrics\n* **Health monitors**\n    * Built in MySQL health monitor\n    * Built in Redis health monitor\n    * Easily add custom health monitors\n* **Environment**\n* **Loggers** - Easily change log levels during runtime\n* **Log file** - Tail the application\'s log file\n* **Thread dump** - See which threads are running\n* **HTTP traces** - Tail recent HTTP requests, including status codes and latency\n\n## Quickstart\nThe examples below show a minimal integration of **FastAPI** and **Flask** applications with **Pyctuator**.\n\nAfter installing Flask/FastAPI and Pyctuator, start by launching a local Spring Boot Admin instance:\n\n```sh\ndocker run --rm --name spring-boot-admin -p 8082:8082 michayaak/spring-boot-admin:2.2.2\n```\n\nThen go to `http://localhost:8082` to get to the web UI.\n\n### Flask\nThe following is a complete example and should run as is:\n\n```python\nfrom flask import Flask\nfrom pyctuator.pyctuator import Pyctuator\n\napp_name = "Flask App with Pyctuator"\napp = Flask(app_name)\n\n\n@app.route("/")\ndef hello():\n    return "Hello World!"\n\n\nPyctuator(\n    app,\n    app_name,\n    "http://host.docker.internal:5000",\n    "http://host.docker.internal:5000/pyctuator",\n    "http://localhost:8082/instances"\n)\n\napp.run(debug=False, port=5000)\n```\n\nOnce you run the application, it should automatically register with Spring Boot Admin and should be available in the UI at `http://localhost:8082`\n\n### FastAPI\nThe following is a complete example and should run as is:\n\n```python\nfrom fastapi import FastAPI\nfrom uvicorn import Server\n\nfrom uvicorn.config import Config\nfrom pyctuator.pyctuator import Pyctuator\n\n\napp_name = "FastAPI App with Pyctuator"\napp = FastAPI(title=app_name)\n\n\n@app.get("/")\ndef hello():\n    return "Hello World!"\n\n\nPyctuator(\n    app,\n    "FastAPI Pyctuator",\n    "http://host.docker.internal:8000",\n    "http://host.docker.internal:8000/pyctuator",\n    "http://localhost:8080/instances"\n)\n\nServer(config=(Config(app=app, loop="asyncio"))).run()\n```\n\n\nOnce you run the application, it should automatically register with Spring Boot Admin and should be available in the UI at `http://localhost:8082`\n\n## Advanced Configuration\nThe following sections are intended for advanced users who want to configure advanced Pyctuator features.\n\n### Application Info\nWhile Pyctuator only needs to know the application\'s name, it is recommended that applications monitored by Spring \nBoot Admin will show additional build and git details - this becomes handy when a service is scaled out to multiple instances by showing the version of each instance.\nTo do so, you can provide additional build and git info using methods of the Pyctuator object:\n\n```python\npyctuator = Pyctuator(...)  # arguments removed for brevity\n\npyctuator.set_build_info(\n    name="app",\n    version="1.3.1",\n    time=datetime.fromisoformat("2019-12-21T10:09:54.876091"),\n)\n\npyctuator.set_git_info(\n    commit="7d4fef3",\n    time=datetime.fromisoformat("2019-12-24T14:18:32.123432"),\n    branch="origin/master",\n)\n```\n\nOnce you configure build and git info, you should see them in the Details tab of Spring Boot Admin:\n\n![Detailed Build Info](examples/images/Main_Details_BuildInfo.png)\n\n### DB Health\nFor services that use SQL database via SQLAlchemy, Pyctuator can easily monitor and expose the connection\'s health \nusing the DbHealthProvider class as demonstrated below:\n\n```python\nengine = create_engine("mysql+pymysql://root:root@localhost:3306")\npyctuator = Pyctuator(...)  # arguments removed for brevity\npyctuator.register_health_provider(DbHealthProvider(engine))\n```\n\nOnce you configure the health provider, you should see DB health info in the Details tab of Spring Boot Admin:\n\n![DB Health](examples/images/Main_DB_Health.png)\n\n### Redis health\nIf your service is using Redis, Pyctuator can monitor the connection to Redis by simply initializing a `RedisHealthProvider`:\n\n```python\nr = redis.Redis()\npyctuator = Pyctuator(...)  # arguments removed for brevity\npyctuator.register_health_provider(RedisHealthProvider(r))\n```\n\n### Custom Environment\nOut of the box, Pyctuator exposes Python\'s environment variables to Spring Boot Admin.\n\nIn addition, an application may register an environment provider to provide additional configuration that should be exposed via Spring Boot Admin. \n\nWhen the environment provider is called it should return a dictionary describing the environment. The returned dictionary is exposed to Spring Boot Admin.\n\nSince Spring Boot Admin doesn\'t support hierarchical environment (only a flat key/value mapping), the provided environment is flattened as dot-delimited keys.\n\nPyctuator tries to hide secrets from being exposed to Spring Boot Admin by replacing the values of "suspicious" keys with ***.\n\nSuspicious keys are keys that contain the words "secret", "password" and some forms of "key".\n\nFor example, if an application\'s configuration looks like this:\n\n```python\nconfig = {\n    "a": "s1",\n    "b": {\n        "secret": "ha ha",\n        "c": 625,\n    },\n    "d": {\n        "e": True,\n        "f": "hello",\n        "g": {\n            "h": 123,\n            "i": "abcde"\n        }\n    }\n}\n```\n\nAn environment provider can be registered like so:\n\n```python\npyctuator.register_environment_provider("config", lambda: config)\n```\n\n### Filesystem and Memory Metrics\nPyctuator can provide filesystem and memory metrics.\n\nTo enable these metrics, install [psutil](https://github.com/giampaolo/psutil)\n\nNote that the `psutil` dependency is **optional** and is only required if you want to enable filesystem and memory monitoring.\n\n### Loggers\nPyctuator leverages Python\'s builtin `logging` framework and allows controlling log levels at runtime.\n \nNote that in order to control uvicorn\'s log level, you need to provide a logger object when instantiating it. For example:\n\n\n```python\nmyFastAPIServer = Server(\n    config=Config(\n        logger=logging.getLogger("uvi"), \n        app=app, \n        loop="asyncio"\n    )\n)\n```\n\n## Full blown examples\nThe `examples` folder contains full blown Python projects that are built using [Poetry](https://python-poetry.org/).\n\nTo run these examples, you\'ll need to have Spring Boot Admin running in a local docker container. A Spring Boot Admin Docker image is available [here](https://hub.docker.com/r/michayaak/spring-boot-admin).\n\nUnless the example includes a docker-compose file, you\'ll need to start Spring Boot Admin using docker directly:\n```sh\ndocker run -p 8082:8082 michayaak/spring-boot-admin:2.2.2\n```\n(the docker image\'s tag represents the version of Spring Boot Admin, so if you need to use version `2.0.0`, use `michayaak/spring-boot-admin:2.0.0` instead).\n\nThe examples include\n* [FastAPI Example](examples/FastAPI/README.md) - demonstrates integrating Pyctuator with the FastAPI web framework.\n* [Flask Example](examples/Flask/README.md) - demonstrates integrating Pyctuator with the Flask web framework.\n* [Advanced Example](examples/Advanced/README.md) - demonstrates configuring and using all the advanced features of Pyctuator.\n\n## Contributing\nTo set up a development environment, make sure you have Python 3.7 or newer installed, and run `make bootstrap`.\n\nUse `make check` to run static analysis tools.\n\nUse `make test` to run tests.\n',
    'author': 'Michael Yakobi',
    'author_email': 'michael.yakobi@solaredge.com',
    'maintainer': 'Matan Rubin',
    'maintainer_email': 'matan.rubin@solaredge.com',
    'url': 'https://github.com/SolarEdgeTech/pyctuator',
    'packages': packages,
    'package_data': package_data,
    'extras_require': extras_require,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
