# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pandas_alive']

package_data = \
{'': ['*']}

install_requires = \
['matplotlib>=3.2.1,<4.0.0', 'pandas>=1.0.3,<2.0.0']

setup_kwargs = {
    'name': 'pandas-alive',
    'version': '0.1.13',
    'description': 'Animated plotting extension for Pandas with Matplotlib',
    'long_description': '# Pandas_Alive\n\nAnimated plotting extension for Pandas with Matplotlib\n\n**Pandas_Alive** is intended to provide a plotting backend for animated [matplotlib](https://matplotlib.org/) charts for [Pandas](https://pandas.pydata.org/) DataFrames, similar to the already [existing Visualization feature of Pandas](https://pandas.pydata.org/pandas-docs/stable/visualization.html).\n\nWith **Pandas_Alive**, creating stunning, animated visualisations is as easy as calling:\n\n`df.plot_animated()`\n\n![Example Bar Chart](examples/example-barh-chart.gif)\n\n## Table of Contents\n<!-- START doctoc generated TOC please keep comment here to allow auto update -->\n<!-- DON\'T EDIT THIS SECTION, INSTEAD RE-RUN doctoc TO UPDATE -->\n\n\n- [Installation](#installation)\n- [Usage](#usage)\n  - [Currently Supported Chart Types](#currently-supported-chart-types)\n    - [Horizontal Bar Charts](#horizontal-bar-charts)\n    - [Vertical Bar Charts](#vertical-bar-charts)\n    - [Line Charts](#line-charts)\n    - [Scatter Charts](#scatter-charts)\n    - [Pie Charts](#pie-charts)\n  - [Multiple Charts](#multiple-charts)\n    - [Urban Population](#urban-population)\n    - [Life Expectancy in G7 Countries](#life-expectancy-in-g7-countries)\n- [Future Features](#future-features)\n- [Inspiration](#inspiration)\n- [Requirements](#requirements)\n- [Documentation](#documentation)\n- [Contributing](#contributing)\n  - [Changelog](#changelog)\n\n<!-- END doctoc generated TOC please keep comment here to allow auto update -->\n\n## Installation\n\nInstall with `pip install pandas_alive`\n\n## Usage\n\nAs this package builds upon [`bar_chart_race`](https://github.com/dexplo/bar_chart_race), the example data set is sourced from there.\n\nMust begin with a pandas DataFrame containing \'wide\' data where:\n\n- Every row represents a single period of time\n- Each column holds the value for a particular category\n- The index contains the time component (optional)\n\nThe data below is an example of properly formatted data. It shows total deaths from COVID-19 for the highest 20 countries by date.\n\n![Example Data Table](https://raw.githubusercontent.com/dexplo/bar_chart_race/master/images/wide_data.png)\n[Example Table](examples/example_dataset_table.md)\n\nTo produce the above visualisation:\n\n- Check [Requirements](#requirements) first to ensure you have the tooling installed!\n- Call `plot_animated()` on the DataFrame\n    - Either specify a file name to write to with `df.plot_animated(filename=\'example.mp4\')` or use `df.plot_animated().get_html5_video` to return a HTML5 video\n- Done!\n\n``` python\nimport pandas_alive\n\ncovid_df = pandas_alive.load_dataset()\n\ncovid_df.plot_animated(filename=\'examples/example-barh-chart.gif\')\n```\n\n### Currently Supported Chart Types\n\n`pandas_alive` current supports:\n\n- [Horizontal Bar Charts](#horizontal-bar-charts)\n- [Vertical Bar Charts](#vertical-bar-charts)\n- [Line Charts](#line-charts)\n- [Scatter Charts](#scatter-charts)\n- [Pie Charts](#pie-charts)\n\n#### Horizontal Bar Charts\n\n``` python\nimport pandas_alive\n\ncovid_df = pandas_alive.load_dataset()\n\ncovid_df.plot_animated(filename=\'example-barh-chart.gif\')\n```\n\n![Example Barh Chart](examples/example-barh-chart.gif)\n\n``` python\nimport pandas as pd\nimport pandas_alive\n\nelec_df = pd.read_csv("data/Aus_Elec_Gen_1980_2018.csv",index_col=0,parse_dates=[0],thousands=\',\')\n\nelec_df.fillna(0).plot_animated(\'examples/example-electricity-generated-australia.gif\',period_fmt="%Y",title=\'Australian Electricity Generation Sources 1980-2018\')\n```\n\n![Electricity Example Line Chart](examples/example-electricity-generated-australia.gif)\n\n``` python\nimport pandas_alive\n\ncovid_df = pandas_alive.load_dataset()\n\ndef current_total(values):\n    total = values.sum()\n    s = f\'Total : {int(total)}\'\n    return {\'x\': .85, \'y\': .2, \'s\': s, \'ha\': \'right\', \'size\': 11}\n\ncovid_df.plot_animated(filename=\'examples/summary-func-example.gif\',period_summary_func=current_total)\n```\n\n![Summary Func Example](examples/summary-func-example.gif)\n\n``` python\nimport pandas as pd\nimport pandas_alive\n\nelec_df = pd.read_csv("data/Aus_Elec_Gen_1980_2018.csv",index_col=0,parse_dates=[0],thousands=\',\')\n\nelec_df.fillna(0).plot_animated(\'examples/fixed-example.gif\',period_fmt="%Y",title=\'Australian Electricity Generation Sources 1980-2018\',fixed_max=True,fixed_order=True)\n```\n\n![Fixed Example](examples/fixed-example.gif)\n\n``` python\nimport pandas_alive\n\ncovid_df = pandas_alive.load_dataset()\n\ncovid_df.plot_animated(filename=\'examples/perpendicular-example.gif\',perpendicular_bar_func=\'mean\')\n```\n\n![Perpendicular Example](examples/perpendicular-example.gif)\n\n#### Vertical Bar Charts\n\n``` python\nimport pandas_alive\n\ncovid_df = pandas_alive.load_dataset()\n\ncovid_df.plot_animated(filename=\'examples/example-barv-chart.gif\',orientation=\'v\')\n```\n\n![Example Barv Chart](examples/example-barv-chart.gif)\n\n#### Line Charts\n\nWith as many lines as data columns in the DataFrame.\n\n``` python\nimport pandas_alive\n\ncovid_df = pandas_alive.load_dataset()\n\ncovid_df.diff().fillna(0).plot_animated(filename=\'examples/example-line-chart.gif\',kind=\'line\',period_label={\'x\':0.1,\'y\':0.9})\n```\n\n![Example Line Chart](examples/example-line-chart.gif)\n\n\n#### Scatter Charts\n\n``` python\nimport pandas as pd\nimport pandas_alive\n\nmax_temp_df = pd.read_csv(\n    "data/Newcastle_Australia_Max_Temps.csv",\n    parse_dates={"Timestamp": ["Year", "Month", "Day"]},\n)\nmin_temp_df = pd.read_csv(\n    "data/Newcastle_Australia_Min_Temps.csv",\n    parse_dates={"Timestamp": ["Year", "Month", "Day"]},\n)\n\nmerged_temp_df = pd.merge_asof(max_temp_df, min_temp_df, on="Timestamp")\n\nmerged_temp_df.index = pd.to_datetime(merged_temp_df["Timestamp"].dt.strftime(\'%Y/%m/%d\'))\n\nkeep_columns = ["Minimum temperature (Degree C)", "Maximum temperature (Degree C)"]\n\nmerged_temp_df[keep_columns].resample("Y").mean().plot_animated(filename=\'examples/example-scatter-chart.gif\',kind="scatter",title=\'Max & Min Temperature Newcastle, Australia\')\n\n```\n\n![Example Scatter Chart](examples/example-scatter-chart.gif)\n\n#### Pie Charts\n\n``` python\nimport pandas_alive\n\ncovid_df = pandas_alive.load_dataset()\n\ncovid_df.plot_animated(filename=\'examples/example-pie-chart.gif\',kind="pie",rotatelabels=True)\n```\n\n![Example Pie Chart](examples/example-pie-chart.gif)\n\n### Multiple Charts\n\n`pandas_alive` supports multiple animated charts in a single visualisation.\n\n- Create a list of all charts to include in animation\n- Use `animate_multiple_plots` with a `filename` and the list of charts (this will use `matplotlib.subplots`)\n- Done!\n\n``` python\nimport pandas_alive\n\ncovid_df = pandas_alive.load_dataset()\n\nanimated_line_chart = covid_df.diff().fillna(0).plot_animated(kind=\'line\',period_label=False)\n\nanimated_bar_chart = covid_df.plot_animated(n_visible=10)\n\npandas_alive.animate_multiple_plots(\'examples/example-bar-and-line-chart.gif\',[animated_bar_chart,animated_line_chart])\n```\n\n![Example Bar & Line Chart](examples/example-bar-and-line-chart.gif)\n\n#### Urban Population\n\n``` python\nimport pandas_alive\n\nurban_df = pandas_alive.load_dataset("urban_pop")\n\nanimated_line_chart = (\n    urban_df.sum(axis=1)\n    .pct_change()\n    .dropna()\n    .mul(100)\n    .plot_animated(kind="line", title="Total % Change in Population",period_label=False)\n)\n\nanimated_bar_chart = urban_df.plot_animated(n_visible=10,title=\'Top 10 Populous Countries\',period_fmt="%Y")\n\npandas_alive.animate_multiple_plots(\'examples/example-bar-and-line-urban-chart.gif\',[animated_bar_chart,animated_line_chart],title=\'Urban Population 1977 - 2018\',adjust_subplot_top=0.85)\n```\n\n![Urban Population Bar & Line Chart](examples/example-bar-and-line-urban-chart.gif)\n\n\n#### Life Expectancy in G7 Countries\n\n``` python\nimport pandas_alive\nimport pandas as pd\n\ndata_raw = pd.read_csv(\n    "https://raw.githubusercontent.com/owid/owid-datasets/master/datasets/Long%20run%20life%20expectancy%20-%20Gapminder%2C%20UN/Long%20run%20life%20expectancy%20-%20Gapminder%2C%20UN.csv"\n)\n\nlist_G7 = [\n    "Canada",\n    "France",\n    "Germany",\n    "Italy",\n    "Japan",\n    "United Kingdom",\n    "United States",\n]\n\ndata_raw = data_raw.pivot(\n    index="Year", columns="Entity", values="Life expectancy (Gapminder, UN)"\n)\n\ndata = pd.DataFrame()\ndata["Year"] = data_raw.reset_index()["Year"]\nfor country in list_G7:\n    data[country] = data_raw[country].values\n\ndata = data.fillna(method="pad")\ndata = data.fillna(0)\ndata = data.set_index("Year").loc[1900:].reset_index()\n\ndata["Year"] = pd.to_datetime(data.reset_index()["Year"].astype(str))\n\ndata = data.set_index("Year")\n\nanimated_bar_chart = data.plot_animated(\n    period_fmt="%Y",perpendicular_bar_func="mean", period_length=200,fixed_max=True\n)\n\nanimated_line_chart = data.plot_animated(\n    kind="line", period_fmt="%Y", period_length=200,fixed_max=True\n)\n\npandas_alive.animate_multiple_plots(\n    "examples/life-expectancy.gif",\n    plots=[animated_bar_chart, animated_line_chart],\n    title="Life expectancy in G7 countries up to 2015",\n    adjust_subplot_left=0.2,\n)\n\n```\n\n![Life Expectancy Chart](examples/life-expectancy.gif)\n\n## Future Features\n\nA list of future features that may/may not be developed is:\n\n- Multiple dimension plots (with multi indexed dataframes)\n- Bubble charts\n- Geographic charts (currently using OSM export image, potential [cartopy](https://github.com/SciTools/cartopy))\n\nA chart that was built using a development branch of Pandas_Alive is:\n\n[![NSW COVID19 Cases](examples/nsw-covid.gif)](https://www.youtube.com/watch?v=qyqiYrtpxRE)\n\n## Inspiration\n\nThe inspiration for this project comes from:\n\n- [bar_chart_race](https://github.com/dexplo/bar_chart_race) by [Ted Petrou](https://github.com/tdpetrou)\n- [Pandas-Bokeh](https://github.com/PatrikHlobil/Pandas-Bokeh) by [Patrik Hlobil](https://github.com/PatrikHlobil)\n\n## Requirements\n\nIf you get an error such as `TypeError: \'MovieWriterRegistry\' object is not an iterator`, this signals there isn\'t a writer library installed on your machine.\n\nThis package utilises the [matplotlib.animation function](https://matplotlib.org/3.2.1/api/animation_api.html), thus requiring a writer library.\n\nEnsure to have one of the supported tooling software installed prior to use!\n\n- [ffmpeg](https://ffmpeg.org/)\n- [ImageMagick](https://imagemagick.org/index.php)\n- [Pillow](https://pillow.readthedocs.io/en/stable/)\n- See more at <https://matplotlib.org/3.2.1/api/animation_api.html#writer-classes>\n\n## Documentation\n\nDocumentation is provided at <https://jackmckew.github.io/pandas_alive/>\n\n## Contributing\n\nPull requests are welcome! Please help to cover more and more chart types!\n\n### [Changelog](CHANGELOG.md)\n\n[Changelog](CHANGELOG.md)\n\n``` python\n\n```\n',
    'author': 'JackMcKew',
    'author_email': 'jackmckew2@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/JackMcKew/pandas_alive',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6.1,<4.0.0',
}


setup(**setup_kwargs)
