"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Canary = exports.Test = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const ec2 = require("aws-cdk-lib/aws-ec2");
const iam = require("aws-cdk-lib/aws-iam");
const s3 = require("aws-cdk-lib/aws-s3");
const cdk = require("aws-cdk-lib");
const synthetics_canned_metrics_generated_1 = require("./synthetics-canned-metrics.generated");
const aws_synthetics_1 = require("aws-cdk-lib/aws-synthetics");
/**
 * Specify a test that the canary should run
 */
class Test {
    /**
     * Construct a Test property
     *
     * @param code The code that the canary should run
     * @param handler The handler of the canary
     */
    constructor(code, handler) {
        this.code = code;
        this.handler = handler;
    }
    /**
     * Specify a custom test with your own code
     *
     * @returns `Test` associated with the specified Code object
     * @param options The configuration options
     */
    static custom(options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_synthetics_alpha_CustomTestOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.custom);
            }
            throw error;
        }
        Test.validateHandler(options.handler);
        return new Test(options.code, options.handler);
    }
    /**
     * Verifies that the given handler ends in '.handler'. Returns the handler if successful and
     * throws an error if not.
     *
     * @param handler - the handler given by the user
     */
    static validateHandler(handler) {
        if (!handler.endsWith('.handler')) {
            throw new Error(`Canary Handler must end in '.handler' (${handler})`);
        }
        if (handler.length > 21) {
            throw new Error(`Canary Handler must be less than 21 characters (${handler})`);
        }
    }
}
exports.Test = Test;
_a = JSII_RTTI_SYMBOL_1;
Test[_a] = { fqn: "@aws-cdk/aws-synthetics-alpha.Test", version: "2.27.0-alpha.0" };
/**
 * Define a new Canary
 */
class Canary extends cdk.Resource {
    constructor(scope, id, props) {
        var _c, _d, _e, _f, _g, _h, _j;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_synthetics_alpha_CanaryProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        if (props.canaryName && !cdk.Token.isUnresolved(props.canaryName)) {
            validateName(props.canaryName);
        }
        super(scope, id, {
            physicalName: props.canaryName || cdk.Lazy.string({
                produce: () => this.generateUniqueName(),
            }),
        });
        this.artifactsBucket = (_d = (_c = props.artifactsBucketLocation) === null || _c === void 0 ? void 0 : _c.bucket) !== null && _d !== void 0 ? _d : new s3.Bucket(this, 'ArtifactsBucket', {
            encryption: s3.BucketEncryption.KMS_MANAGED,
            enforceSSL: true,
        });
        this.role = (_e = props.role) !== null && _e !== void 0 ? _e : this.createDefaultRole(props);
        if (props.vpc) {
            // Security Groups are created and/or appended in `createVpcConfig`.
            this._connections = new ec2.Connections({});
        }
        const resource = new aws_synthetics_1.CfnCanary(this, 'Resource', {
            artifactS3Location: this.artifactsBucket.s3UrlForObject((_f = props.artifactsBucketLocation) === null || _f === void 0 ? void 0 : _f.prefix),
            executionRoleArn: this.role.roleArn,
            startCanaryAfterCreation: (_g = props.startAfterCreation) !== null && _g !== void 0 ? _g : true,
            runtimeVersion: props.runtime.name,
            name: this.physicalName,
            schedule: this.createSchedule(props),
            failureRetentionPeriod: (_h = props.failureRetentionPeriod) === null || _h === void 0 ? void 0 : _h.toDays(),
            successRetentionPeriod: (_j = props.successRetentionPeriod) === null || _j === void 0 ? void 0 : _j.toDays(),
            code: this.createCode(props),
            runConfig: this.createRunConfig(props),
            vpcConfig: this.createVpcConfig(props),
        });
        this.canaryId = resource.attrId;
        this.canaryState = resource.attrState;
        this.canaryName = this.getResourceNameAttribute(resource.ref);
    }
    /**
     * Access the Connections object
     *
     * Will fail if not a VPC-enabled Canary
     */
    get connections() {
        if (!this._connections) {
            // eslint-disable-next-line max-len
            throw new Error('Only VPC-associated Canaries have security groups to manage. Supply the "vpc" parameter when creating the Canary.');
        }
        return this._connections;
    }
    /**
     * Measure the Duration of a single canary run, in seconds.
     *
     * @param options - configuration options for the metric
     *
     * @default avg over 5 minutes
     */
    metricDuration(options) {
        return new aws_cloudwatch_1.Metric({
            ...synthetics_canned_metrics_generated_1.CloudWatchSyntheticsMetrics.durationMaximum({ CanaryName: this.canaryName }),
            ...{ statistic: 'Average' },
            ...options,
        }).attachTo(this);
    }
    /**
     * Measure the percentage of successful canary runs.
     *
     * @param options - configuration options for the metric
     *
     * @default avg over 5 minutes
     */
    metricSuccessPercent(options) {
        return this.cannedMetric(synthetics_canned_metrics_generated_1.CloudWatchSyntheticsMetrics.successPercentAverage, options);
    }
    /**
     * Measure the number of failed canary runs over a given time period.
     *
     * Default: sum over 5 minutes
     *
     * @param options - configuration options for the metric
     */
    metricFailed(options) {
        return this.cannedMetric(synthetics_canned_metrics_generated_1.CloudWatchSyntheticsMetrics.failedSum, options);
    }
    /**
     * Returns a default role for the canary
     */
    createDefaultRole(props) {
        var _c;
        const prefix = (_c = props.artifactsBucketLocation) === null || _c === void 0 ? void 0 : _c.prefix;
        // Created role will need these policies to run the Canary.
        // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-synthetics-canary.html#cfn-synthetics-canary-executionrolearn
        const policy = new iam.PolicyDocument({
            statements: [
                new iam.PolicyStatement({
                    resources: ['*'],
                    actions: ['s3:ListAllMyBuckets'],
                }),
                new iam.PolicyStatement({
                    resources: [this.artifactsBucket.bucketArn],
                    actions: ['s3:GetBucketLocation'],
                }),
                new iam.PolicyStatement({
                    resources: [this.artifactsBucket.arnForObjects(`${prefix ? prefix + '/*' : '*'}`)],
                    actions: ['s3:PutObject'],
                }),
                new iam.PolicyStatement({
                    resources: ['*'],
                    actions: ['cloudwatch:PutMetricData'],
                    conditions: { StringEquals: { 'cloudwatch:namespace': 'CloudWatchSynthetics' } },
                }),
                new iam.PolicyStatement({
                    resources: [this.logGroupArn()],
                    actions: ['logs:CreateLogStream', 'logs:CreateLogGroup', 'logs:PutLogEvents'],
                }),
            ],
        });
        const managedPolicies = [];
        if (props.vpc) {
            // Policy that will have ENI creation permissions
            managedPolicies.push(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaVPCAccessExecutionRole'));
        }
        return new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            inlinePolicies: {
                canaryPolicy: policy,
            },
            managedPolicies,
        });
    }
    logGroupArn() {
        return cdk.Stack.of(this).formatArn({
            service: 'logs',
            resource: 'log-group',
            arnFormat: cdk.ArnFormat.COLON_RESOURCE_NAME,
            resourceName: '/aws/lambda/cwsyn-*',
        });
    }
    /**
     * Returns the code object taken in by the canary resource.
     */
    createCode(props) {
        var _c, _d, _e;
        const codeConfig = {
            handler: props.test.handler,
            ...props.test.code.bind(this, props.test.handler, props.runtime.family),
        };
        return {
            handler: codeConfig.handler,
            script: codeConfig.inlineCode,
            s3Bucket: (_c = codeConfig.s3Location) === null || _c === void 0 ? void 0 : _c.bucketName,
            s3Key: (_d = codeConfig.s3Location) === null || _d === void 0 ? void 0 : _d.objectKey,
            s3ObjectVersion: (_e = codeConfig.s3Location) === null || _e === void 0 ? void 0 : _e.objectVersion,
        };
    }
    createRunConfig(props) {
        if (!props.environmentVariables) {
            return undefined;
        }
        return {
            environmentVariables: props.environmentVariables,
        };
    }
    /**
     * Returns a canary schedule object
     */
    createSchedule(props) {
        var _c, _d, _e, _f;
        return {
            durationInSeconds: String(`${(_d = (_c = props.timeToLive) === null || _c === void 0 ? void 0 : _c.toSeconds()) !== null && _d !== void 0 ? _d : 0}`),
            expression: (_f = (_e = props.schedule) === null || _e === void 0 ? void 0 : _e.expressionString) !== null && _f !== void 0 ? _f : 'rate(5 minutes)',
        };
    }
    createVpcConfig(props) {
        if (!props.vpc) {
            if (props.vpcSubnets != null || props.securityGroups != null) {
                throw new Error("You must provide the 'vpc' prop when using VPC-related properties.");
            }
            return undefined;
        }
        const { subnetIds } = props.vpc.selectSubnets(props.vpcSubnets);
        if (subnetIds.length < 1) {
            throw new Error('No matching subnets found in the VPC.');
        }
        let securityGroups;
        if (props.securityGroups && props.securityGroups.length > 0) {
            securityGroups = props.securityGroups;
        }
        else {
            const securityGroup = new ec2.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc,
                description: 'Automatic security group for Canary ' + cdk.Names.uniqueId(this),
            });
            securityGroups = [securityGroup];
        }
        this._connections.addSecurityGroup(...securityGroups);
        return {
            vpcId: props.vpc.vpcId,
            subnetIds,
            securityGroupIds: cdk.Lazy.list({ produce: () => this.connections.securityGroups.map(sg => sg.securityGroupId) }),
        };
    }
    /**
     * Creates a unique name for the canary. The generated name is the physical ID of the canary.
     */
    generateUniqueName() {
        const name = cdk.Names.uniqueId(this).toLowerCase().replace(' ', '-');
        if (name.length <= 21) {
            return name;
        }
        else {
            return name.substring(0, 15) + nameHash(name);
        }
    }
    cannedMetric(fn, props) {
        return new aws_cloudwatch_1.Metric({
            ...fn({ CanaryName: this.canaryName }),
            ...props,
        }).attachTo(this);
    }
}
exports.Canary = Canary;
_b = JSII_RTTI_SYMBOL_1;
Canary[_b] = { fqn: "@aws-cdk/aws-synthetics-alpha.Canary", version: "2.27.0-alpha.0" };
/**
 * Take a hash of the given name.
 *
 * @param name the name to be hashed
 */
function nameHash(name) {
    const md5 = crypto.createHash('sha256').update(name).digest('hex');
    return md5.slice(0, 6);
}
const nameRegex = /^[0-9a-z_\-]+$/;
/**
 * Verifies that the name fits the regex expression: ^[0-9a-z_\-]+$.
 *
 * @param name - the given name of the canary
 */
function validateName(name) {
    if (name.length > 21) {
        throw new Error(`Canary name is too large, must be between 1 and 21 characters, but is ${name.length} (got "${name}")`);
    }
    if (!nameRegex.test(name)) {
        throw new Error(`Canary name must be lowercase, numbers, hyphens, or underscores (got "${name}")`);
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2FuYXJ5LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiY2FuYXJ5LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7OztBQUFBLGlDQUFpQztBQUNqQywrREFBZ0Y7QUFDaEYsMkNBQTJDO0FBQzNDLDJDQUEyQztBQUMzQyx5Q0FBeUM7QUFDekMsbUNBQW1DO0FBS25DLCtGQUFvRjtBQUNwRiwrREFBdUQ7QUFFdkQ7O0dBRUc7QUFDSCxNQUFhLElBQUk7SUEyQmY7Ozs7O09BS0c7SUFDSCxZQUFvQyxJQUFVLEVBQWtCLE9BQWU7UUFBM0MsU0FBSSxHQUFKLElBQUksQ0FBTTtRQUFrQixZQUFPLEdBQVAsT0FBTyxDQUFRO0tBQzlFO0lBakNEOzs7OztPQUtHO0lBQ0ksTUFBTSxDQUFDLE1BQU0sQ0FBQyxPQUEwQjs7Ozs7Ozs7OztRQUM3QyxJQUFJLENBQUMsZUFBZSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUN0QyxPQUFPLElBQUksSUFBSSxDQUFDLE9BQU8sQ0FBQyxJQUFJLEVBQUUsT0FBTyxDQUFDLE9BQU8sQ0FBQyxDQUFDO0tBQ2hEO0lBRUQ7Ozs7O09BS0c7SUFDSyxNQUFNLENBQUMsZUFBZSxDQUFDLE9BQWU7UUFDNUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLEVBQUU7WUFDakMsTUFBTSxJQUFJLEtBQUssQ0FBQywwQ0FBMEMsT0FBTyxHQUFHLENBQUMsQ0FBQztTQUN2RTtRQUNELElBQUksT0FBTyxDQUFDLE1BQU0sR0FBRyxFQUFFLEVBQUU7WUFDdkIsTUFBTSxJQUFJLEtBQUssQ0FBQyxtREFBbUQsT0FBTyxHQUFHLENBQUMsQ0FBQztTQUNoRjtLQUNGOztBQXpCSCxvQkFtQ0M7OztBQTZKRDs7R0FFRztBQUNILE1BQWEsTUFBTyxTQUFRLEdBQUcsQ0FBQyxRQUFRO0lBcUN0QyxZQUFtQixLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFrQjs7Ozs7Ozs7Ozs7UUFDakUsSUFBSSxLQUFLLENBQUMsVUFBVSxJQUFJLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxFQUFFO1lBQ2pFLFlBQVksQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLENBQUM7U0FDaEM7UUFFRCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRTtZQUNmLFlBQVksRUFBRSxLQUFLLENBQUMsVUFBVSxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDO2dCQUNoRCxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLGtCQUFrQixFQUFFO2FBQ3pDLENBQUM7U0FDSCxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsZUFBZSxlQUFHLEtBQUssQ0FBQyx1QkFBdUIsMENBQUUsTUFBTSxtQ0FBSSxJQUFJLEVBQUUsQ0FBQyxNQUFNLENBQUMsSUFBSSxFQUFFLGlCQUFpQixFQUFFO1lBQ3JHLFVBQVUsRUFBRSxFQUFFLENBQUMsZ0JBQWdCLENBQUMsV0FBVztZQUMzQyxVQUFVLEVBQUUsSUFBSTtTQUNqQixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsSUFBSSxTQUFHLEtBQUssQ0FBQyxJQUFJLG1DQUFJLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUV4RCxJQUFJLEtBQUssQ0FBQyxHQUFHLEVBQUU7WUFDYixvRUFBb0U7WUFDcEUsSUFBSSxDQUFDLFlBQVksR0FBRyxJQUFJLEdBQUcsQ0FBQyxXQUFXLENBQUMsRUFBRSxDQUFDLENBQUM7U0FDN0M7UUFFRCxNQUFNLFFBQVEsR0FBYyxJQUFJLDBCQUFTLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUMxRCxrQkFBa0IsRUFBRSxJQUFJLENBQUMsZUFBZSxDQUFDLGNBQWMsT0FBQyxLQUFLLENBQUMsdUJBQXVCLDBDQUFFLE1BQU0sQ0FBQztZQUM5RixnQkFBZ0IsRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLE9BQU87WUFDbkMsd0JBQXdCLFFBQUUsS0FBSyxDQUFDLGtCQUFrQixtQ0FBSSxJQUFJO1lBQzFELGNBQWMsRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUk7WUFDbEMsSUFBSSxFQUFFLElBQUksQ0FBQyxZQUFZO1lBQ3ZCLFFBQVEsRUFBRSxJQUFJLENBQUMsY0FBYyxDQUFDLEtBQUssQ0FBQztZQUNwQyxzQkFBc0IsUUFBRSxLQUFLLENBQUMsc0JBQXNCLDBDQUFFLE1BQU0sRUFBRTtZQUM5RCxzQkFBc0IsUUFBRSxLQUFLLENBQUMsc0JBQXNCLDBDQUFFLE1BQU0sRUFBRTtZQUM5RCxJQUFJLEVBQUUsSUFBSSxDQUFDLFVBQVUsQ0FBQyxLQUFLLENBQUM7WUFDNUIsU0FBUyxFQUFFLElBQUksQ0FBQyxlQUFlLENBQUMsS0FBSyxDQUFDO1lBQ3RDLFNBQVMsRUFBRSxJQUFJLENBQUMsZUFBZSxDQUFDLEtBQUssQ0FBQztTQUN2QyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsUUFBUSxHQUFHLFFBQVEsQ0FBQyxNQUFNLENBQUM7UUFDaEMsSUFBSSxDQUFDLFdBQVcsR0FBRyxRQUFRLENBQUMsU0FBUyxDQUFDO1FBQ3RDLElBQUksQ0FBQyxVQUFVLEdBQUcsSUFBSSxDQUFDLHdCQUF3QixDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUMsQ0FBQztLQUMvRDtJQUVEOzs7O09BSUc7SUFDSCxJQUFXLFdBQVc7UUFDcEIsSUFBSSxDQUFDLElBQUksQ0FBQyxZQUFZLEVBQUU7WUFDdEIsbUNBQW1DO1lBQ25DLE1BQU0sSUFBSSxLQUFLLENBQUMsbUhBQW1ILENBQUMsQ0FBQztTQUN0STtRQUNELE9BQU8sSUFBSSxDQUFDLFlBQVksQ0FBQztLQUMxQjtJQUVEOzs7Ozs7T0FNRztJQUNJLGNBQWMsQ0FBQyxPQUF1QjtRQUMzQyxPQUFPLElBQUksdUJBQU0sQ0FBQztZQUNoQixHQUFHLGlFQUEyQixDQUFDLGVBQWUsQ0FBQyxFQUFFLFVBQVUsRUFBRSxJQUFJLENBQUMsVUFBVSxFQUFFLENBQUM7WUFDL0UsR0FBRyxFQUFFLFNBQVMsRUFBRSxTQUFTLEVBQUU7WUFDM0IsR0FBRyxPQUFPO1NBQ1gsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQztLQUNuQjtJQUVEOzs7Ozs7T0FNRztJQUNJLG9CQUFvQixDQUFDLE9BQXVCO1FBQ2pELE9BQU8sSUFBSSxDQUFDLFlBQVksQ0FBQyxpRUFBMkIsQ0FBQyxxQkFBcUIsRUFBRSxPQUFPLENBQUMsQ0FBQztLQUN0RjtJQUVEOzs7Ozs7T0FNRztJQUNJLFlBQVksQ0FBQyxPQUF1QjtRQUN6QyxPQUFPLElBQUksQ0FBQyxZQUFZLENBQUMsaUVBQTJCLENBQUMsU0FBUyxFQUFFLE9BQU8sQ0FBQyxDQUFDO0tBQzFFO0lBRUQ7O09BRUc7SUFDSyxpQkFBaUIsQ0FBQyxLQUFrQjs7UUFDMUMsTUFBTSxNQUFNLFNBQUcsS0FBSyxDQUFDLHVCQUF1QiwwQ0FBRSxNQUFNLENBQUM7UUFFckQsMkRBQTJEO1FBQzNELDRJQUE0STtRQUM1SSxNQUFNLE1BQU0sR0FBRyxJQUFJLEdBQUcsQ0FBQyxjQUFjLENBQUM7WUFDcEMsVUFBVSxFQUFFO2dCQUNWLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztvQkFDdEIsU0FBUyxFQUFFLENBQUMsR0FBRyxDQUFDO29CQUNoQixPQUFPLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQztpQkFDakMsQ0FBQztnQkFDRixJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7b0JBQ3RCLFNBQVMsRUFBRSxDQUFDLElBQUksQ0FBQyxlQUFlLENBQUMsU0FBUyxDQUFDO29CQUMzQyxPQUFPLEVBQUUsQ0FBQyxzQkFBc0IsQ0FBQztpQkFDbEMsQ0FBQztnQkFDRixJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7b0JBQ3RCLFNBQVMsRUFBRSxDQUFDLElBQUksQ0FBQyxlQUFlLENBQUMsYUFBYSxDQUFDLEdBQUcsTUFBTSxDQUFDLENBQUMsQ0FBQyxNQUFNLEdBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxHQUFHLEVBQUUsQ0FBQyxDQUFDO29CQUNoRixPQUFPLEVBQUUsQ0FBQyxjQUFjLENBQUM7aUJBQzFCLENBQUM7Z0JBQ0YsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO29CQUN0QixTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUM7b0JBQ2hCLE9BQU8sRUFBRSxDQUFDLDBCQUEwQixDQUFDO29CQUNyQyxVQUFVLEVBQUUsRUFBRSxZQUFZLEVBQUUsRUFBRSxzQkFBc0IsRUFBRSxzQkFBc0IsRUFBRSxFQUFFO2lCQUNqRixDQUFDO2dCQUNGLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztvQkFDdEIsU0FBUyxFQUFFLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxDQUFDO29CQUMvQixPQUFPLEVBQUUsQ0FBQyxzQkFBc0IsRUFBRSxxQkFBcUIsRUFBRSxtQkFBbUIsQ0FBQztpQkFDOUUsQ0FBQzthQUNIO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsTUFBTSxlQUFlLEdBQXlCLEVBQUUsQ0FBQztRQUVqRCxJQUFJLEtBQUssQ0FBQyxHQUFHLEVBQUU7WUFDYixpREFBaUQ7WUFDakQsZUFBZSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDLHdCQUF3QixDQUFDLDhDQUE4QyxDQUFDLENBQUMsQ0FBQztTQUNsSDtRQUVELE9BQU8sSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxhQUFhLEVBQUU7WUFDdkMsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLHNCQUFzQixDQUFDO1lBQzNELGNBQWMsRUFBRTtnQkFDZCxZQUFZLEVBQUUsTUFBTTthQUNyQjtZQUNELGVBQWU7U0FDaEIsQ0FBQyxDQUFDO0tBQ0o7SUFFTyxXQUFXO1FBQ2pCLE9BQU8sR0FBRyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsU0FBUyxDQUFDO1lBQ2xDLE9BQU8sRUFBRSxNQUFNO1lBQ2YsUUFBUSxFQUFFLFdBQVc7WUFDckIsU0FBUyxFQUFFLEdBQUcsQ0FBQyxTQUFTLENBQUMsbUJBQW1CO1lBQzVDLFlBQVksRUFBRSxxQkFBcUI7U0FDcEMsQ0FBQyxDQUFDO0tBQ0o7SUFFRDs7T0FFRztJQUNLLFVBQVUsQ0FBQyxLQUFrQjs7UUFDbkMsTUFBTSxVQUFVLEdBQUc7WUFDakIsT0FBTyxFQUFFLEtBQUssQ0FBQyxJQUFJLENBQUMsT0FBTztZQUMzQixHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsS0FBSyxDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUM7U0FDeEUsQ0FBQztRQUNGLE9BQU87WUFDTCxPQUFPLEVBQUUsVUFBVSxDQUFDLE9BQU87WUFDM0IsTUFBTSxFQUFFLFVBQVUsQ0FBQyxVQUFVO1lBQzdCLFFBQVEsUUFBRSxVQUFVLENBQUMsVUFBVSwwQ0FBRSxVQUFVO1lBQzNDLEtBQUssUUFBRSxVQUFVLENBQUMsVUFBVSwwQ0FBRSxTQUFTO1lBQ3ZDLGVBQWUsUUFBRSxVQUFVLENBQUMsVUFBVSwwQ0FBRSxhQUFhO1NBQ3RELENBQUM7S0FDSDtJQUVPLGVBQWUsQ0FBQyxLQUFrQjtRQUN4QyxJQUFJLENBQUMsS0FBSyxDQUFDLG9CQUFvQixFQUFFO1lBQy9CLE9BQU8sU0FBUyxDQUFDO1NBQ2xCO1FBQ0QsT0FBTztZQUNMLG9CQUFvQixFQUFFLEtBQUssQ0FBQyxvQkFBb0I7U0FDakQsQ0FBQztLQUNIO0lBRUQ7O09BRUc7SUFDSyxjQUFjLENBQUMsS0FBa0I7O1FBQ3ZDLE9BQU87WUFDTCxpQkFBaUIsRUFBRSxNQUFNLENBQUMsR0FBRyxZQUFBLEtBQUssQ0FBQyxVQUFVLDBDQUFFLFNBQVMscUNBQU0sQ0FBQyxFQUFFLENBQUM7WUFDbEUsVUFBVSxjQUFFLEtBQUssQ0FBQyxRQUFRLDBDQUFFLGdCQUFnQixtQ0FBSSxpQkFBaUI7U0FDbEUsQ0FBQztLQUNIO0lBRU8sZUFBZSxDQUFDLEtBQWtCO1FBQ3hDLElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRyxFQUFFO1lBQ2QsSUFBSSxLQUFLLENBQUMsVUFBVSxJQUFJLElBQUksSUFBSSxLQUFLLENBQUMsY0FBYyxJQUFJLElBQUksRUFBRTtnQkFDNUQsTUFBTSxJQUFJLEtBQUssQ0FBQyxvRUFBb0UsQ0FBQyxDQUFDO2FBQ3ZGO1lBRUQsT0FBTyxTQUFTLENBQUM7U0FDbEI7UUFFRCxNQUFNLEVBQUUsU0FBUyxFQUFFLEdBQUcsS0FBSyxDQUFDLEdBQUcsQ0FBQyxhQUFhLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBQ2hFLElBQUksU0FBUyxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7WUFDeEIsTUFBTSxJQUFJLEtBQUssQ0FBQyx1Q0FBdUMsQ0FBQyxDQUFDO1NBQzFEO1FBRUQsSUFBSSxjQUFvQyxDQUFDO1FBQ3pDLElBQUksS0FBSyxDQUFDLGNBQWMsSUFBSSxLQUFLLENBQUMsY0FBYyxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7WUFDM0QsY0FBYyxHQUFHLEtBQUssQ0FBQyxjQUFjLENBQUM7U0FDdkM7YUFBTTtZQUNMLE1BQU0sYUFBYSxHQUFHLElBQUksR0FBRyxDQUFDLGFBQWEsQ0FBQyxJQUFJLEVBQUUsZUFBZSxFQUFFO2dCQUNqRSxHQUFHLEVBQUUsS0FBSyxDQUFDLEdBQUc7Z0JBQ2QsV0FBVyxFQUFFLHNDQUFzQyxHQUFHLEdBQUcsQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQzthQUMvRSxDQUFDLENBQUM7WUFDSCxjQUFjLEdBQUcsQ0FBQyxhQUFhLENBQUMsQ0FBQztTQUNsQztRQUNELElBQUksQ0FBQyxZQUFhLENBQUMsZ0JBQWdCLENBQUMsR0FBRyxjQUFjLENBQUMsQ0FBQztRQUV2RCxPQUFPO1lBQ0wsS0FBSyxFQUFFLEtBQUssQ0FBQyxHQUFHLENBQUMsS0FBSztZQUN0QixTQUFTO1lBQ1QsZ0JBQWdCLEVBQUUsR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxjQUFjLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQyxFQUFFLENBQUM7U0FDbEgsQ0FBQztLQUNIO0lBRUQ7O09BRUc7SUFDSyxrQkFBa0I7UUFDeEIsTUFBTSxJQUFJLEdBQUcsR0FBRyxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLENBQUMsV0FBVyxFQUFFLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxHQUFHLENBQUMsQ0FBQztRQUN0RSxJQUFJLElBQUksQ0FBQyxNQUFNLElBQUksRUFBRSxFQUFFO1lBQ3JCLE9BQU8sSUFBSSxDQUFDO1NBQ2I7YUFBTTtZQUNMLE9BQU8sSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLEdBQUcsUUFBUSxDQUFDLElBQUksQ0FBQyxDQUFDO1NBQy9DO0tBQ0Y7SUFFTyxZQUFZLENBQ2xCLEVBQWlELEVBQ2pELEtBQXFCO1FBQ3JCLE9BQU8sSUFBSSx1QkFBTSxDQUFDO1lBQ2hCLEdBQUcsRUFBRSxDQUFDLEVBQUUsVUFBVSxFQUFFLElBQUksQ0FBQyxVQUFVLEVBQUUsQ0FBQztZQUN0QyxHQUFHLEtBQUs7U0FDVCxDQUFDLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxDQUFDO0tBQ25COztBQXBSSCx3QkFxUkM7OztBQUVEOzs7O0dBSUc7QUFDSCxTQUFTLFFBQVEsQ0FBQyxJQUFZO0lBQzVCLE1BQU0sR0FBRyxHQUFHLE1BQU0sQ0FBQyxVQUFVLENBQUMsUUFBUSxDQUFDLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUNuRSxPQUFPLEdBQUcsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDO0FBQ3pCLENBQUM7QUFFRCxNQUFNLFNBQVMsR0FBVyxnQkFBZ0IsQ0FBQztBQUUzQzs7OztHQUlHO0FBQ0gsU0FBUyxZQUFZLENBQUMsSUFBWTtJQUNoQyxJQUFJLElBQUksQ0FBQyxNQUFNLEdBQUcsRUFBRSxFQUFFO1FBQ3BCLE1BQU0sSUFBSSxLQUFLLENBQUMseUVBQXlFLElBQUksQ0FBQyxNQUFNLFVBQVUsSUFBSSxJQUFJLENBQUMsQ0FBQztLQUN6SDtJQUNELElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxFQUFFO1FBQ3pCLE1BQU0sSUFBSSxLQUFLLENBQUMseUVBQXlFLElBQUksSUFBSSxDQUFDLENBQUM7S0FDcEc7QUFDSCxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgY3J5cHRvIGZyb20gJ2NyeXB0byc7XG5pbXBvcnQgeyBNZXRyaWMsIE1ldHJpY09wdGlvbnMsIE1ldHJpY1Byb3BzIH0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWNsb3Vkd2F0Y2gnO1xuaW1wb3J0ICogYXMgZWMyIGZyb20gJ2F3cy1jZGstbGliL2F3cy1lYzInO1xuaW1wb3J0ICogYXMgaWFtIGZyb20gJ2F3cy1jZGstbGliL2F3cy1pYW0nO1xuaW1wb3J0ICogYXMgczMgZnJvbSAnYXdzLWNkay1saWIvYXdzLXMzJztcbmltcG9ydCAqIGFzIGNkayBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IENvZGUgfSBmcm9tICcuL2NvZGUnO1xuaW1wb3J0IHsgUnVudGltZSB9IGZyb20gJy4vcnVudGltZSc7XG5pbXBvcnQgeyBTY2hlZHVsZSB9IGZyb20gJy4vc2NoZWR1bGUnO1xuaW1wb3J0IHsgQ2xvdWRXYXRjaFN5bnRoZXRpY3NNZXRyaWNzIH0gZnJvbSAnLi9zeW50aGV0aWNzLWNhbm5lZC1tZXRyaWNzLmdlbmVyYXRlZCc7XG5pbXBvcnQgeyBDZm5DYW5hcnkgfSBmcm9tICdhd3MtY2RrLWxpYi9hd3Mtc3ludGhldGljcyc7XG5cbi8qKlxuICogU3BlY2lmeSBhIHRlc3QgdGhhdCB0aGUgY2FuYXJ5IHNob3VsZCBydW5cbiAqL1xuZXhwb3J0IGNsYXNzIFRlc3Qge1xuICAvKipcbiAgICogU3BlY2lmeSBhIGN1c3RvbSB0ZXN0IHdpdGggeW91ciBvd24gY29kZVxuICAgKlxuICAgKiBAcmV0dXJucyBgVGVzdGAgYXNzb2NpYXRlZCB3aXRoIHRoZSBzcGVjaWZpZWQgQ29kZSBvYmplY3RcbiAgICogQHBhcmFtIG9wdGlvbnMgVGhlIGNvbmZpZ3VyYXRpb24gb3B0aW9uc1xuICAgKi9cbiAgcHVibGljIHN0YXRpYyBjdXN0b20ob3B0aW9uczogQ3VzdG9tVGVzdE9wdGlvbnMpOiBUZXN0IHtcbiAgICBUZXN0LnZhbGlkYXRlSGFuZGxlcihvcHRpb25zLmhhbmRsZXIpO1xuICAgIHJldHVybiBuZXcgVGVzdChvcHRpb25zLmNvZGUsIG9wdGlvbnMuaGFuZGxlcik7XG4gIH1cblxuICAvKipcbiAgICogVmVyaWZpZXMgdGhhdCB0aGUgZ2l2ZW4gaGFuZGxlciBlbmRzIGluICcuaGFuZGxlcicuIFJldHVybnMgdGhlIGhhbmRsZXIgaWYgc3VjY2Vzc2Z1bCBhbmRcbiAgICogdGhyb3dzIGFuIGVycm9yIGlmIG5vdC5cbiAgICpcbiAgICogQHBhcmFtIGhhbmRsZXIgLSB0aGUgaGFuZGxlciBnaXZlbiBieSB0aGUgdXNlclxuICAgKi9cbiAgcHJpdmF0ZSBzdGF0aWMgdmFsaWRhdGVIYW5kbGVyKGhhbmRsZXI6IHN0cmluZykge1xuICAgIGlmICghaGFuZGxlci5lbmRzV2l0aCgnLmhhbmRsZXInKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBDYW5hcnkgSGFuZGxlciBtdXN0IGVuZCBpbiAnLmhhbmRsZXInICgke2hhbmRsZXJ9KWApO1xuICAgIH1cbiAgICBpZiAoaGFuZGxlci5sZW5ndGggPiAyMSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBDYW5hcnkgSGFuZGxlciBtdXN0IGJlIGxlc3MgdGhhbiAyMSBjaGFyYWN0ZXJzICgke2hhbmRsZXJ9KWApO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3QgYSBUZXN0IHByb3BlcnR5XG4gICAqXG4gICAqIEBwYXJhbSBjb2RlIFRoZSBjb2RlIHRoYXQgdGhlIGNhbmFyeSBzaG91bGQgcnVuXG4gICAqIEBwYXJhbSBoYW5kbGVyIFRoZSBoYW5kbGVyIG9mIHRoZSBjYW5hcnlcbiAgICovXG4gIHByaXZhdGUgY29uc3RydWN0b3IocHVibGljIHJlYWRvbmx5IGNvZGU6IENvZGUsIHB1YmxpYyByZWFkb25seSBoYW5kbGVyOiBzdHJpbmcpIHtcbiAgfVxufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIHNwZWNpZnlpbmcgYSB0ZXN0XG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQ3VzdG9tVGVzdE9wdGlvbnMge1xuICAvKipcbiAgICogVGhlIGNvZGUgb2YgdGhlIGNhbmFyeSBzY3JpcHRcbiAgICovXG4gIHJlYWRvbmx5IGNvZGU6IENvZGUsXG5cbiAgLyoqXG4gICAqIFRoZSBoYW5kbGVyIGZvciB0aGUgY29kZS4gTXVzdCBlbmQgd2l0aCBgLmhhbmRsZXJgLlxuICAgKi9cbiAgcmVhZG9ubHkgaGFuZGxlcjogc3RyaW5nLFxufVxuXG4vKipcbiAqIE9wdGlvbnMgZm9yIHNwZWNpZnlpbmcgdGhlIHMzIGxvY2F0aW9uIHRoYXQgc3RvcmVzIHRoZSBkYXRhIG9mIGVhY2ggY2FuYXJ5IHJ1bi4gVGhlIGFydGlmYWN0cyBidWNrZXQgbG9jYXRpb24gKipjYW5ub3QqKlxuICogYmUgdXBkYXRlZCBvbmNlIHRoZSBjYW5hcnkgaXMgY3JlYXRlZC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBBcnRpZmFjdHNCdWNrZXRMb2NhdGlvbiB7XG4gIC8qKlxuICAgKiBUaGUgczMgbG9jYXRpb24gdGhhdCBzdG9yZXMgdGhlIGRhdGEgb2YgZWFjaCBydW4uXG4gICAqL1xuICByZWFkb25seSBidWNrZXQ6IHMzLklCdWNrZXQ7XG5cbiAgLyoqXG4gICAqIFRoZSBTMyBidWNrZXQgcHJlZml4LiBTcGVjaWZ5IHRoaXMgaWYgeW91IHdhbnQgYSBtb3JlIHNwZWNpZmljIHBhdGggd2l0aGluIHRoZSBhcnRpZmFjdHMgYnVja2V0LlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIHByZWZpeFxuICAgKi9cbiAgcmVhZG9ubHkgcHJlZml4Pzogc3RyaW5nO1xufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGEgY2FuYXJ5XG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQ2FuYXJ5UHJvcHMge1xuICAvKipcbiAgICogVGhlIHMzIGxvY2F0aW9uIHRoYXQgc3RvcmVzIHRoZSBkYXRhIG9mIHRoZSBjYW5hcnkgcnVucy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBBIG5ldyBzMyBidWNrZXQgd2lsbCBiZSBjcmVhdGVkIHdpdGhvdXQgYSBwcmVmaXguXG4gICAqL1xuICByZWFkb25seSBhcnRpZmFjdHNCdWNrZXRMb2NhdGlvbj86IEFydGlmYWN0c0J1Y2tldExvY2F0aW9uO1xuXG4gIC8qKlxuICAgKiBDYW5hcnkgZXhlY3V0aW9uIHJvbGUuXG4gICAqXG4gICAqIFRoaXMgaXMgdGhlIHJvbGUgdGhhdCB3aWxsIGJlIGFzc3VtZWQgYnkgdGhlIGNhbmFyeSB1cG9uIGV4ZWN1dGlvbi5cbiAgICogSXQgY29udHJvbHMgdGhlIHBlcm1pc3Npb25zIHRoYXQgdGhlIGNhbmFyeSB3aWxsIGhhdmUuIFRoZSByb2xlIG11c3RcbiAgICogYmUgYXNzdW1hYmxlIGJ5IHRoZSBBV1MgTGFtYmRhIHNlcnZpY2UgcHJpbmNpcGFsLlxuICAgKlxuICAgKiBJZiBub3Qgc3VwcGxpZWQsIGEgcm9sZSB3aWxsIGJlIGNyZWF0ZWQgd2l0aCBhbGwgdGhlIHJlcXVpcmVkIHBlcm1pc3Npb25zLlxuICAgKiBJZiB5b3UgcHJvdmlkZSBhIFJvbGUsIHlvdSBtdXN0IGFkZCB0aGUgcmVxdWlyZWQgcGVybWlzc2lvbnMuXG4gICAqXG4gICAqIEBzZWUgcmVxdWlyZWQgcGVybWlzc2lvbnM6IGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BV1NDbG91ZEZvcm1hdGlvbi9sYXRlc3QvVXNlckd1aWRlL2F3cy1yZXNvdXJjZS1zeW50aGV0aWNzLWNhbmFyeS5odG1sI2Nmbi1zeW50aGV0aWNzLWNhbmFyeS1leGVjdXRpb25yb2xlYXJuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gQSB1bmlxdWUgcm9sZSB3aWxsIGJlIGdlbmVyYXRlZCBmb3IgdGhpcyBjYW5hcnkuXG4gICAqIFlvdSBjYW4gYWRkIHBlcm1pc3Npb25zIHRvIHJvbGVzIGJ5IGNhbGxpbmcgJ2FkZFRvUm9sZVBvbGljeScuXG4gICAqL1xuICByZWFkb25seSByb2xlPzogaWFtLklSb2xlO1xuXG4gIC8qKlxuICAgKiBIb3cgbG9uZyB0aGUgY2FuYXJ5IHdpbGwgYmUgaW4gYSAnUlVOTklORycgc3RhdGUuIEZvciBleGFtcGxlLCBpZiB5b3Ugc2V0IGB0aW1lVG9MaXZlYCB0byBiZSAxIGhvdXIgYW5kIGBzY2hlZHVsZWAgdG8gYmUgYHJhdGUoMTAgbWludXRlcylgLFxuICAgKiB5b3VyIGNhbmFyeSB3aWxsIHJ1biBhdCAxMCBtaW51dGUgaW50ZXJ2YWxzIGZvciBhbiBob3VyLCBmb3IgYSB0b3RhbCBvZiA2IHRpbWVzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIGxpbWl0XG4gICAqL1xuICByZWFkb25seSB0aW1lVG9MaXZlPzogY2RrLkR1cmF0aW9uO1xuXG4gIC8qKlxuICAgKiBTcGVjaWZ5IHRoZSBzY2hlZHVsZSBmb3IgaG93IG9mdGVuIHRoZSBjYW5hcnkgcnVucy4gRm9yIGV4YW1wbGUsIGlmIHlvdSBzZXQgYHNjaGVkdWxlYCB0byBgcmF0ZSgxMCBtaW51dGVzKWAsIHRoZW4gdGhlIGNhbmFyeSB3aWxsIHJ1biBldmVyeSAxMCBtaW51dGVzLlxuICAgKiBZb3UgY2FuIHNldCB0aGUgc2NoZWR1bGUgd2l0aCBgU2NoZWR1bGUucmF0ZShEdXJhdGlvbilgIChyZWNvbW1lbmRlZCkgb3IgeW91IGNhbiBzcGVjaWZ5IGFuIGV4cHJlc3Npb24gdXNpbmcgYFNjaGVkdWxlLmV4cHJlc3Npb24oKWAuXG4gICAqIEBkZWZhdWx0ICdyYXRlKDUgbWludXRlcyknXG4gICAqL1xuICByZWFkb25seSBzY2hlZHVsZT86IFNjaGVkdWxlO1xuXG4gIC8qKlxuICAgKiBXaGV0aGVyIG9yIG5vdCB0aGUgY2FuYXJ5IHNob3VsZCBzdGFydCBhZnRlciBjcmVhdGlvbi5cbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgc3RhcnRBZnRlckNyZWF0aW9uPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogSG93IG1hbnkgZGF5cyBzaG91bGQgc3VjY2Vzc2Z1bCBydW5zIGJlIHJldGFpbmVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCBEdXJhdGlvbi5kYXlzKDMxKVxuICAgKi9cbiAgcmVhZG9ubHkgc3VjY2Vzc1JldGVudGlvblBlcmlvZD86IGNkay5EdXJhdGlvbjtcblxuICAvKipcbiAgICogSG93IG1hbnkgZGF5cyBzaG91bGQgZmFpbGVkIHJ1bnMgYmUgcmV0YWluZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IER1cmF0aW9uLmRheXMoMzEpXG4gICAqL1xuICByZWFkb25seSBmYWlsdXJlUmV0ZW50aW9uUGVyaW9kPzogY2RrLkR1cmF0aW9uO1xuXG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgY2FuYXJ5LiBCZSBzdXJlIHRvIGdpdmUgaXQgYSBkZXNjcmlwdGl2ZSBuYW1lIHRoYXQgZGlzdGluZ3Vpc2hlcyBpdCBmcm9tXG4gICAqIG90aGVyIGNhbmFyaWVzIGluIHlvdXIgYWNjb3VudC5cbiAgICpcbiAgICogRG8gbm90IGluY2x1ZGUgc2VjcmV0cyBvciBwcm9wcmlldGFyeSBpbmZvcm1hdGlvbiBpbiB5b3VyIGNhbmFyeSBuYW1lLiBUaGUgY2FuYXJ5IG5hbWVcbiAgICogbWFrZXMgdXAgcGFydCBvZiB0aGUgY2FuYXJ5IEFSTiwgd2hpY2ggaXMgaW5jbHVkZWQgaW4gb3V0Ym91bmQgY2FsbHMgb3ZlciB0aGUgaW50ZXJuZXQuXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkV2F0Y2gvbGF0ZXN0L21vbml0b3Jpbmcvc2VydmljZWxlbnNfY2FuYXJpZXNfc2VjdXJpdHkuaHRtbFxuICAgKlxuICAgKiBAZGVmYXVsdCAtIEEgdW5pcXVlIG5hbWUgd2lsbCBiZSBnZW5lcmF0ZWQgZnJvbSB0aGUgY29uc3RydWN0IElEXG4gICAqL1xuICByZWFkb25seSBjYW5hcnlOYW1lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBTcGVjaWZ5IHRoZSBydW50aW1lIHZlcnNpb24gdG8gdXNlIGZvciB0aGUgY2FuYXJ5LlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoL2xhdGVzdC9tb25pdG9yaW5nL0Nsb3VkV2F0Y2hfU3ludGhldGljc19DYW5hcmllc19MaWJyYXJ5Lmh0bWxcbiAgICovXG4gIHJlYWRvbmx5IHJ1bnRpbWU6IFJ1bnRpbWU7XG5cbiAgLyoqXG4gICAqIFRoZSB0eXBlIG9mIHRlc3QgdGhhdCB5b3Ugd2FudCB5b3VyIGNhbmFyeSB0byBydW4uIFVzZSBgVGVzdC5jdXN0b20oKWAgdG8gc3BlY2lmeSB0aGUgdGVzdCB0byBydW4uXG4gICAqL1xuICByZWFkb25seSB0ZXN0OiBUZXN0O1xuXG4gIC8qKlxuICAgKiBLZXktdmFsdWUgcGFpcnMgdGhhdCB0aGUgU3ludGhldGljcyBjYWNoZXMgYW5kIG1ha2VzIGF2YWlsYWJsZSBmb3IgeW91ciBjYW5hcnkgc2NyaXB0cy4gVXNlIGVudmlyb25tZW50IHZhcmlhYmxlc1xuICAgKiB0byBhcHBseSBjb25maWd1cmF0aW9uIGNoYW5nZXMsIHN1Y2ggYXMgdGVzdCBhbmQgcHJvZHVjdGlvbiBlbnZpcm9ubWVudCBjb25maWd1cmF0aW9ucywgd2l0aG91dCBjaGFuZ2luZyB5b3VyXG4gICAqIENhbmFyeSBzY3JpcHQgc291cmNlIGNvZGUuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gZW52aXJvbm1lbnQgdmFyaWFibGVzLlxuICAgKi9cbiAgcmVhZG9ubHkgZW52aXJvbm1lbnRWYXJpYWJsZXM/OiB7IFtrZXk6IHN0cmluZ106IHN0cmluZyB9O1xuXG4gIC8qKlxuICAgKiBUaGUgVlBDIHdoZXJlIHRoaXMgY2FuYXJ5IGlzIHJ1bi5cbiAgICpcbiAgICogU3BlY2lmeSB0aGlzIGlmIHRoZSBjYW5hcnkgbmVlZHMgdG8gYWNjZXNzIHJlc291cmNlcyBpbiBhIFZQQy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBOb3QgaW4gVlBDXG4gICAqL1xuICByZWFkb25seSB2cGM/OiBlYzIuSVZwYztcblxuICAvKipcbiAgICogV2hlcmUgdG8gcGxhY2UgdGhlIG5ldHdvcmsgaW50ZXJmYWNlcyB3aXRoaW4gdGhlIFZQQy4gWW91IG11c3QgcHJvdmlkZSBgdnBjYCB3aGVuIHVzaW5nIHRoaXMgcHJvcC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSB0aGUgVnBjIGRlZmF1bHQgc3RyYXRlZ3kgaWYgbm90IHNwZWNpZmllZFxuICAgKi9cbiAgcmVhZG9ubHkgdnBjU3VibmV0cz86IGVjMi5TdWJuZXRTZWxlY3Rpb247XG5cbiAgLyoqXG4gICAqIFRoZSBsaXN0IG9mIHNlY3VyaXR5IGdyb3VwcyB0byBhc3NvY2lhdGUgd2l0aCB0aGUgY2FuYXJ5J3MgbmV0d29yayBpbnRlcmZhY2VzLiBZb3UgbXVzdCBwcm92aWRlIGB2cGNgIHdoZW4gdXNpbmcgdGhpcyBwcm9wLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIElmIHRoZSBjYW5hcnkgaXMgcGxhY2VkIHdpdGhpbiBhIFZQQyBhbmQgYSBzZWN1cml0eSBncm91cCBpc1xuICAgKiBub3Qgc3BlY2lmaWVkIGEgZGVkaWNhdGVkIHNlY3VyaXR5IGdyb3VwIHdpbGwgYmUgY3JlYXRlZCBmb3IgdGhpcyBjYW5hcnkuXG4gICAqL1xuICByZWFkb25seSBzZWN1cml0eUdyb3Vwcz86IGVjMi5JU2VjdXJpdHlHcm91cFtdO1xufVxuXG4vKipcbiAqIERlZmluZSBhIG5ldyBDYW5hcnlcbiAqL1xuZXhwb3J0IGNsYXNzIENhbmFyeSBleHRlbmRzIGNkay5SZXNvdXJjZSBpbXBsZW1lbnRzIGVjMi5JQ29ubmVjdGFibGUge1xuICAvKipcbiAgICogRXhlY3V0aW9uIHJvbGUgYXNzb2NpYXRlZCB3aXRoIHRoaXMgQ2FuYXJ5LlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHJvbGU6IGlhbS5JUm9sZTtcblxuICAvKipcbiAgICogVGhlIGNhbmFyeSBJRFxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgY2FuYXJ5SWQ6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIHN0YXRlIG9mIHRoZSBjYW5hcnkuIEZvciBleGFtcGxlLCAnUlVOTklORycsICdTVE9QUEVEJywgJ05PVCBTVEFSVEVEJywgb3IgJ0VSUk9SJy5cbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGNhbmFyeVN0YXRlOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBjYW5hcnkgTmFtZVxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgY2FuYXJ5TmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBCdWNrZXQgd2hlcmUgZGF0YSBmcm9tIGVhY2ggY2FuYXJ5IHJ1biBpcyBzdG9yZWQuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgYXJ0aWZhY3RzQnVja2V0OiBzMy5JQnVja2V0O1xuXG4gIC8qKlxuICAgKiBBY3R1YWwgY29ubmVjdGlvbnMgb2JqZWN0IGZvciB0aGUgdW5kZXJseWluZyBMYW1iZGFcbiAgICpcbiAgICogTWF5IGJlIHVuc2V0LCBpbiB3aGljaCBjYXNlIHRoZSBjYW5hcnkgTGFtYmRhIGlzIG5vdCBjb25maWd1cmVkIGZvciB1c2UgaW4gYSBWUEMuXG4gICAqIEBpbnRlcm5hbFxuICAgKi9cbiAgcHJpdmF0ZSByZWFkb25seSBfY29ubmVjdGlvbnM/OiBlYzIuQ29ubmVjdGlvbnM7XG5cbiAgcHVibGljIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBDYW5hcnlQcm9wcykge1xuICAgIGlmIChwcm9wcy5jYW5hcnlOYW1lICYmICFjZGsuVG9rZW4uaXNVbnJlc29sdmVkKHByb3BzLmNhbmFyeU5hbWUpKSB7XG4gICAgICB2YWxpZGF0ZU5hbWUocHJvcHMuY2FuYXJ5TmFtZSk7XG4gICAgfVxuXG4gICAgc3VwZXIoc2NvcGUsIGlkLCB7XG4gICAgICBwaHlzaWNhbE5hbWU6IHByb3BzLmNhbmFyeU5hbWUgfHwgY2RrLkxhenkuc3RyaW5nKHtcbiAgICAgICAgcHJvZHVjZTogKCkgPT4gdGhpcy5nZW5lcmF0ZVVuaXF1ZU5hbWUoKSxcbiAgICAgIH0pLFxuICAgIH0pO1xuXG4gICAgdGhpcy5hcnRpZmFjdHNCdWNrZXQgPSBwcm9wcy5hcnRpZmFjdHNCdWNrZXRMb2NhdGlvbj8uYnVja2V0ID8/IG5ldyBzMy5CdWNrZXQodGhpcywgJ0FydGlmYWN0c0J1Y2tldCcsIHtcbiAgICAgIGVuY3J5cHRpb246IHMzLkJ1Y2tldEVuY3J5cHRpb24uS01TX01BTkFHRUQsXG4gICAgICBlbmZvcmNlU1NMOiB0cnVlLFxuICAgIH0pO1xuXG4gICAgdGhpcy5yb2xlID0gcHJvcHMucm9sZSA/PyB0aGlzLmNyZWF0ZURlZmF1bHRSb2xlKHByb3BzKTtcblxuICAgIGlmIChwcm9wcy52cGMpIHtcbiAgICAgIC8vIFNlY3VyaXR5IEdyb3VwcyBhcmUgY3JlYXRlZCBhbmQvb3IgYXBwZW5kZWQgaW4gYGNyZWF0ZVZwY0NvbmZpZ2AuXG4gICAgICB0aGlzLl9jb25uZWN0aW9ucyA9IG5ldyBlYzIuQ29ubmVjdGlvbnMoe30pO1xuICAgIH1cblxuICAgIGNvbnN0IHJlc291cmNlOiBDZm5DYW5hcnkgPSBuZXcgQ2ZuQ2FuYXJ5KHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgIGFydGlmYWN0UzNMb2NhdGlvbjogdGhpcy5hcnRpZmFjdHNCdWNrZXQuczNVcmxGb3JPYmplY3QocHJvcHMuYXJ0aWZhY3RzQnVja2V0TG9jYXRpb24/LnByZWZpeCksXG4gICAgICBleGVjdXRpb25Sb2xlQXJuOiB0aGlzLnJvbGUucm9sZUFybixcbiAgICAgIHN0YXJ0Q2FuYXJ5QWZ0ZXJDcmVhdGlvbjogcHJvcHMuc3RhcnRBZnRlckNyZWF0aW9uID8/IHRydWUsXG4gICAgICBydW50aW1lVmVyc2lvbjogcHJvcHMucnVudGltZS5uYW1lLFxuICAgICAgbmFtZTogdGhpcy5waHlzaWNhbE5hbWUsXG4gICAgICBzY2hlZHVsZTogdGhpcy5jcmVhdGVTY2hlZHVsZShwcm9wcyksXG4gICAgICBmYWlsdXJlUmV0ZW50aW9uUGVyaW9kOiBwcm9wcy5mYWlsdXJlUmV0ZW50aW9uUGVyaW9kPy50b0RheXMoKSxcbiAgICAgIHN1Y2Nlc3NSZXRlbnRpb25QZXJpb2Q6IHByb3BzLnN1Y2Nlc3NSZXRlbnRpb25QZXJpb2Q/LnRvRGF5cygpLFxuICAgICAgY29kZTogdGhpcy5jcmVhdGVDb2RlKHByb3BzKSxcbiAgICAgIHJ1bkNvbmZpZzogdGhpcy5jcmVhdGVSdW5Db25maWcocHJvcHMpLFxuICAgICAgdnBjQ29uZmlnOiB0aGlzLmNyZWF0ZVZwY0NvbmZpZyhwcm9wcyksXG4gICAgfSk7XG5cbiAgICB0aGlzLmNhbmFyeUlkID0gcmVzb3VyY2UuYXR0cklkO1xuICAgIHRoaXMuY2FuYXJ5U3RhdGUgPSByZXNvdXJjZS5hdHRyU3RhdGU7XG4gICAgdGhpcy5jYW5hcnlOYW1lID0gdGhpcy5nZXRSZXNvdXJjZU5hbWVBdHRyaWJ1dGUocmVzb3VyY2UucmVmKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBY2Nlc3MgdGhlIENvbm5lY3Rpb25zIG9iamVjdFxuICAgKlxuICAgKiBXaWxsIGZhaWwgaWYgbm90IGEgVlBDLWVuYWJsZWQgQ2FuYXJ5XG4gICAqL1xuICBwdWJsaWMgZ2V0IGNvbm5lY3Rpb25zKCk6IGVjMi5Db25uZWN0aW9ucyB7XG4gICAgaWYgKCF0aGlzLl9jb25uZWN0aW9ucykge1xuICAgICAgLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIG1heC1sZW5cbiAgICAgIHRocm93IG5ldyBFcnJvcignT25seSBWUEMtYXNzb2NpYXRlZCBDYW5hcmllcyBoYXZlIHNlY3VyaXR5IGdyb3VwcyB0byBtYW5hZ2UuIFN1cHBseSB0aGUgXCJ2cGNcIiBwYXJhbWV0ZXIgd2hlbiBjcmVhdGluZyB0aGUgQ2FuYXJ5LicpO1xuICAgIH1cbiAgICByZXR1cm4gdGhpcy5fY29ubmVjdGlvbnM7XG4gIH1cblxuICAvKipcbiAgICogTWVhc3VyZSB0aGUgRHVyYXRpb24gb2YgYSBzaW5nbGUgY2FuYXJ5IHJ1biwgaW4gc2Vjb25kcy5cbiAgICpcbiAgICogQHBhcmFtIG9wdGlvbnMgLSBjb25maWd1cmF0aW9uIG9wdGlvbnMgZm9yIHRoZSBtZXRyaWNcbiAgICpcbiAgICogQGRlZmF1bHQgYXZnIG92ZXIgNSBtaW51dGVzXG4gICAqL1xuICBwdWJsaWMgbWV0cmljRHVyYXRpb24ob3B0aW9ucz86IE1ldHJpY09wdGlvbnMpOiBNZXRyaWMge1xuICAgIHJldHVybiBuZXcgTWV0cmljKHtcbiAgICAgIC4uLkNsb3VkV2F0Y2hTeW50aGV0aWNzTWV0cmljcy5kdXJhdGlvbk1heGltdW0oeyBDYW5hcnlOYW1lOiB0aGlzLmNhbmFyeU5hbWUgfSksXG4gICAgICAuLi57IHN0YXRpc3RpYzogJ0F2ZXJhZ2UnIH0sXG4gICAgICAuLi5vcHRpb25zLFxuICAgIH0pLmF0dGFjaFRvKHRoaXMpO1xuICB9XG5cbiAgLyoqXG4gICAqIE1lYXN1cmUgdGhlIHBlcmNlbnRhZ2Ugb2Ygc3VjY2Vzc2Z1bCBjYW5hcnkgcnVucy5cbiAgICpcbiAgICogQHBhcmFtIG9wdGlvbnMgLSBjb25maWd1cmF0aW9uIG9wdGlvbnMgZm9yIHRoZSBtZXRyaWNcbiAgICpcbiAgICogQGRlZmF1bHQgYXZnIG92ZXIgNSBtaW51dGVzXG4gICAqL1xuICBwdWJsaWMgbWV0cmljU3VjY2Vzc1BlcmNlbnQob3B0aW9ucz86IE1ldHJpY09wdGlvbnMpOiBNZXRyaWMge1xuICAgIHJldHVybiB0aGlzLmNhbm5lZE1ldHJpYyhDbG91ZFdhdGNoU3ludGhldGljc01ldHJpY3Muc3VjY2Vzc1BlcmNlbnRBdmVyYWdlLCBvcHRpb25zKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBNZWFzdXJlIHRoZSBudW1iZXIgb2YgZmFpbGVkIGNhbmFyeSBydW5zIG92ZXIgYSBnaXZlbiB0aW1lIHBlcmlvZC5cbiAgICpcbiAgICogRGVmYXVsdDogc3VtIG92ZXIgNSBtaW51dGVzXG4gICAqXG4gICAqIEBwYXJhbSBvcHRpb25zIC0gY29uZmlndXJhdGlvbiBvcHRpb25zIGZvciB0aGUgbWV0cmljXG4gICAqL1xuICBwdWJsaWMgbWV0cmljRmFpbGVkKG9wdGlvbnM/OiBNZXRyaWNPcHRpb25zKTogTWV0cmljIHtcbiAgICByZXR1cm4gdGhpcy5jYW5uZWRNZXRyaWMoQ2xvdWRXYXRjaFN5bnRoZXRpY3NNZXRyaWNzLmZhaWxlZFN1bSwgb3B0aW9ucyk7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBhIGRlZmF1bHQgcm9sZSBmb3IgdGhlIGNhbmFyeVxuICAgKi9cbiAgcHJpdmF0ZSBjcmVhdGVEZWZhdWx0Um9sZShwcm9wczogQ2FuYXJ5UHJvcHMpOiBpYW0uSVJvbGUge1xuICAgIGNvbnN0IHByZWZpeCA9IHByb3BzLmFydGlmYWN0c0J1Y2tldExvY2F0aW9uPy5wcmVmaXg7XG5cbiAgICAvLyBDcmVhdGVkIHJvbGUgd2lsbCBuZWVkIHRoZXNlIHBvbGljaWVzIHRvIHJ1biB0aGUgQ2FuYXJ5LlxuICAgIC8vIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BV1NDbG91ZEZvcm1hdGlvbi9sYXRlc3QvVXNlckd1aWRlL2F3cy1yZXNvdXJjZS1zeW50aGV0aWNzLWNhbmFyeS5odG1sI2Nmbi1zeW50aGV0aWNzLWNhbmFyeS1leGVjdXRpb25yb2xlYXJuXG4gICAgY29uc3QgcG9saWN5ID0gbmV3IGlhbS5Qb2xpY3lEb2N1bWVudCh7XG4gICAgICBzdGF0ZW1lbnRzOiBbXG4gICAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICByZXNvdXJjZXM6IFsnKiddLFxuICAgICAgICAgIGFjdGlvbnM6IFsnczM6TGlzdEFsbE15QnVja2V0cyddLFxuICAgICAgICB9KSxcbiAgICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgIHJlc291cmNlczogW3RoaXMuYXJ0aWZhY3RzQnVja2V0LmJ1Y2tldEFybl0sXG4gICAgICAgICAgYWN0aW9uczogWydzMzpHZXRCdWNrZXRMb2NhdGlvbiddLFxuICAgICAgICB9KSxcbiAgICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgIHJlc291cmNlczogW3RoaXMuYXJ0aWZhY3RzQnVja2V0LmFybkZvck9iamVjdHMoYCR7cHJlZml4ID8gcHJlZml4KycvKicgOiAnKid9YCldLFxuICAgICAgICAgIGFjdGlvbnM6IFsnczM6UHV0T2JqZWN0J10sXG4gICAgICAgIH0pLFxuICAgICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICAgICAgICBhY3Rpb25zOiBbJ2Nsb3Vkd2F0Y2g6UHV0TWV0cmljRGF0YSddLFxuICAgICAgICAgIGNvbmRpdGlvbnM6IHsgU3RyaW5nRXF1YWxzOiB7ICdjbG91ZHdhdGNoOm5hbWVzcGFjZSc6ICdDbG91ZFdhdGNoU3ludGhldGljcycgfSB9LFxuICAgICAgICB9KSxcbiAgICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgIHJlc291cmNlczogW3RoaXMubG9nR3JvdXBBcm4oKV0sXG4gICAgICAgICAgYWN0aW9uczogWydsb2dzOkNyZWF0ZUxvZ1N0cmVhbScsICdsb2dzOkNyZWF0ZUxvZ0dyb3VwJywgJ2xvZ3M6UHV0TG9nRXZlbnRzJ10sXG4gICAgICAgIH0pLFxuICAgICAgXSxcbiAgICB9KTtcblxuICAgIGNvbnN0IG1hbmFnZWRQb2xpY2llczogaWFtLklNYW5hZ2VkUG9saWN5W10gPSBbXTtcblxuICAgIGlmIChwcm9wcy52cGMpIHtcbiAgICAgIC8vIFBvbGljeSB0aGF0IHdpbGwgaGF2ZSBFTkkgY3JlYXRpb24gcGVybWlzc2lvbnNcbiAgICAgIG1hbmFnZWRQb2xpY2llcy5wdXNoKGlhbS5NYW5hZ2VkUG9saWN5LmZyb21Bd3NNYW5hZ2VkUG9saWN5TmFtZSgnc2VydmljZS1yb2xlL0FXU0xhbWJkYVZQQ0FjY2Vzc0V4ZWN1dGlvblJvbGUnKSk7XG4gICAgfVxuXG4gICAgcmV0dXJuIG5ldyBpYW0uUm9sZSh0aGlzLCAnU2VydmljZVJvbGUnLCB7XG4gICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnbGFtYmRhLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgIGlubGluZVBvbGljaWVzOiB7XG4gICAgICAgIGNhbmFyeVBvbGljeTogcG9saWN5LFxuICAgICAgfSxcbiAgICAgIG1hbmFnZWRQb2xpY2llcyxcbiAgICB9KTtcbiAgfVxuXG4gIHByaXZhdGUgbG9nR3JvdXBBcm4oKSB7XG4gICAgcmV0dXJuIGNkay5TdGFjay5vZih0aGlzKS5mb3JtYXRBcm4oe1xuICAgICAgc2VydmljZTogJ2xvZ3MnLFxuICAgICAgcmVzb3VyY2U6ICdsb2ctZ3JvdXAnLFxuICAgICAgYXJuRm9ybWF0OiBjZGsuQXJuRm9ybWF0LkNPTE9OX1JFU09VUkNFX05BTUUsXG4gICAgICByZXNvdXJjZU5hbWU6ICcvYXdzL2xhbWJkYS9jd3N5bi0qJyxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIHRoZSBjb2RlIG9iamVjdCB0YWtlbiBpbiBieSB0aGUgY2FuYXJ5IHJlc291cmNlLlxuICAgKi9cbiAgcHJpdmF0ZSBjcmVhdGVDb2RlKHByb3BzOiBDYW5hcnlQcm9wcyk6IENmbkNhbmFyeS5Db2RlUHJvcGVydHkge1xuICAgIGNvbnN0IGNvZGVDb25maWcgPSB7XG4gICAgICBoYW5kbGVyOiBwcm9wcy50ZXN0LmhhbmRsZXIsXG4gICAgICAuLi5wcm9wcy50ZXN0LmNvZGUuYmluZCh0aGlzLCBwcm9wcy50ZXN0LmhhbmRsZXIsIHByb3BzLnJ1bnRpbWUuZmFtaWx5KSxcbiAgICB9O1xuICAgIHJldHVybiB7XG4gICAgICBoYW5kbGVyOiBjb2RlQ29uZmlnLmhhbmRsZXIsXG4gICAgICBzY3JpcHQ6IGNvZGVDb25maWcuaW5saW5lQ29kZSxcbiAgICAgIHMzQnVja2V0OiBjb2RlQ29uZmlnLnMzTG9jYXRpb24/LmJ1Y2tldE5hbWUsXG4gICAgICBzM0tleTogY29kZUNvbmZpZy5zM0xvY2F0aW9uPy5vYmplY3RLZXksXG4gICAgICBzM09iamVjdFZlcnNpb246IGNvZGVDb25maWcuczNMb2NhdGlvbj8ub2JqZWN0VmVyc2lvbixcbiAgICB9O1xuICB9XG5cbiAgcHJpdmF0ZSBjcmVhdGVSdW5Db25maWcocHJvcHM6IENhbmFyeVByb3BzKTogQ2ZuQ2FuYXJ5LlJ1bkNvbmZpZ1Byb3BlcnR5IHwgdW5kZWZpbmVkIHtcbiAgICBpZiAoIXByb3BzLmVudmlyb25tZW50VmFyaWFibGVzKSB7XG4gICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgIH1cbiAgICByZXR1cm4ge1xuICAgICAgZW52aXJvbm1lbnRWYXJpYWJsZXM6IHByb3BzLmVudmlyb25tZW50VmFyaWFibGVzLFxuICAgIH07XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBhIGNhbmFyeSBzY2hlZHVsZSBvYmplY3RcbiAgICovXG4gIHByaXZhdGUgY3JlYXRlU2NoZWR1bGUocHJvcHM6IENhbmFyeVByb3BzKTogQ2ZuQ2FuYXJ5LlNjaGVkdWxlUHJvcGVydHkge1xuICAgIHJldHVybiB7XG4gICAgICBkdXJhdGlvbkluU2Vjb25kczogU3RyaW5nKGAke3Byb3BzLnRpbWVUb0xpdmU/LnRvU2Vjb25kcygpID8/IDB9YCksXG4gICAgICBleHByZXNzaW9uOiBwcm9wcy5zY2hlZHVsZT8uZXhwcmVzc2lvblN0cmluZyA/PyAncmF0ZSg1IG1pbnV0ZXMpJyxcbiAgICB9O1xuICB9XG5cbiAgcHJpdmF0ZSBjcmVhdGVWcGNDb25maWcocHJvcHM6IENhbmFyeVByb3BzKTogQ2ZuQ2FuYXJ5LlZQQ0NvbmZpZ1Byb3BlcnR5IHwgdW5kZWZpbmVkIHtcbiAgICBpZiAoIXByb3BzLnZwYykge1xuICAgICAgaWYgKHByb3BzLnZwY1N1Ym5ldHMgIT0gbnVsbCB8fCBwcm9wcy5zZWN1cml0eUdyb3VwcyAhPSBudWxsKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihcIllvdSBtdXN0IHByb3ZpZGUgdGhlICd2cGMnIHByb3Agd2hlbiB1c2luZyBWUEMtcmVsYXRlZCBwcm9wZXJ0aWVzLlwiKTtcbiAgICAgIH1cblxuICAgICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgICB9XG5cbiAgICBjb25zdCB7IHN1Ym5ldElkcyB9ID0gcHJvcHMudnBjLnNlbGVjdFN1Ym5ldHMocHJvcHMudnBjU3VibmV0cyk7XG4gICAgaWYgKHN1Ym5ldElkcy5sZW5ndGggPCAxKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ05vIG1hdGNoaW5nIHN1Ym5ldHMgZm91bmQgaW4gdGhlIFZQQy4nKTtcbiAgICB9XG5cbiAgICBsZXQgc2VjdXJpdHlHcm91cHM6IGVjMi5JU2VjdXJpdHlHcm91cFtdO1xuICAgIGlmIChwcm9wcy5zZWN1cml0eUdyb3VwcyAmJiBwcm9wcy5zZWN1cml0eUdyb3Vwcy5sZW5ndGggPiAwKSB7XG4gICAgICBzZWN1cml0eUdyb3VwcyA9IHByb3BzLnNlY3VyaXR5R3JvdXBzO1xuICAgIH0gZWxzZSB7XG4gICAgICBjb25zdCBzZWN1cml0eUdyb3VwID0gbmV3IGVjMi5TZWN1cml0eUdyb3VwKHRoaXMsICdTZWN1cml0eUdyb3VwJywge1xuICAgICAgICB2cGM6IHByb3BzLnZwYyxcbiAgICAgICAgZGVzY3JpcHRpb246ICdBdXRvbWF0aWMgc2VjdXJpdHkgZ3JvdXAgZm9yIENhbmFyeSAnICsgY2RrLk5hbWVzLnVuaXF1ZUlkKHRoaXMpLFxuICAgICAgfSk7XG4gICAgICBzZWN1cml0eUdyb3VwcyA9IFtzZWN1cml0eUdyb3VwXTtcbiAgICB9XG4gICAgdGhpcy5fY29ubmVjdGlvbnMhLmFkZFNlY3VyaXR5R3JvdXAoLi4uc2VjdXJpdHlHcm91cHMpO1xuXG4gICAgcmV0dXJuIHtcbiAgICAgIHZwY0lkOiBwcm9wcy52cGMudnBjSWQsXG4gICAgICBzdWJuZXRJZHMsXG4gICAgICBzZWN1cml0eUdyb3VwSWRzOiBjZGsuTGF6eS5saXN0KHsgcHJvZHVjZTogKCkgPT4gdGhpcy5jb25uZWN0aW9ucy5zZWN1cml0eUdyb3Vwcy5tYXAoc2cgPT4gc2cuc2VjdXJpdHlHcm91cElkKSB9KSxcbiAgICB9O1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYSB1bmlxdWUgbmFtZSBmb3IgdGhlIGNhbmFyeS4gVGhlIGdlbmVyYXRlZCBuYW1lIGlzIHRoZSBwaHlzaWNhbCBJRCBvZiB0aGUgY2FuYXJ5LlxuICAgKi9cbiAgcHJpdmF0ZSBnZW5lcmF0ZVVuaXF1ZU5hbWUoKTogc3RyaW5nIHtcbiAgICBjb25zdCBuYW1lID0gY2RrLk5hbWVzLnVuaXF1ZUlkKHRoaXMpLnRvTG93ZXJDYXNlKCkucmVwbGFjZSgnICcsICctJyk7XG4gICAgaWYgKG5hbWUubGVuZ3RoIDw9IDIxKSB7XG4gICAgICByZXR1cm4gbmFtZTtcbiAgICB9IGVsc2Uge1xuICAgICAgcmV0dXJuIG5hbWUuc3Vic3RyaW5nKDAsIDE1KSArIG5hbWVIYXNoKG5hbWUpO1xuICAgIH1cbiAgfVxuXG4gIHByaXZhdGUgY2FubmVkTWV0cmljKFxuICAgIGZuOiAoZGltczogeyBDYW5hcnlOYW1lOiBzdHJpbmcgfSkgPT4gTWV0cmljUHJvcHMsXG4gICAgcHJvcHM/OiBNZXRyaWNPcHRpb25zKTogTWV0cmljIHtcbiAgICByZXR1cm4gbmV3IE1ldHJpYyh7XG4gICAgICAuLi5mbih7IENhbmFyeU5hbWU6IHRoaXMuY2FuYXJ5TmFtZSB9KSxcbiAgICAgIC4uLnByb3BzLFxuICAgIH0pLmF0dGFjaFRvKHRoaXMpO1xuICB9XG59XG5cbi8qKlxuICogVGFrZSBhIGhhc2ggb2YgdGhlIGdpdmVuIG5hbWUuXG4gKlxuICogQHBhcmFtIG5hbWUgdGhlIG5hbWUgdG8gYmUgaGFzaGVkXG4gKi9cbmZ1bmN0aW9uIG5hbWVIYXNoKG5hbWU6IHN0cmluZyk6IHN0cmluZyB7XG4gIGNvbnN0IG1kNSA9IGNyeXB0by5jcmVhdGVIYXNoKCdzaGEyNTYnKS51cGRhdGUobmFtZSkuZGlnZXN0KCdoZXgnKTtcbiAgcmV0dXJuIG1kNS5zbGljZSgwLCA2KTtcbn1cblxuY29uc3QgbmFtZVJlZ2V4OiBSZWdFeHAgPSAvXlswLTlhLXpfXFwtXSskLztcblxuLyoqXG4gKiBWZXJpZmllcyB0aGF0IHRoZSBuYW1lIGZpdHMgdGhlIHJlZ2V4IGV4cHJlc3Npb246IF5bMC05YS16X1xcLV0rJC5cbiAqXG4gKiBAcGFyYW0gbmFtZSAtIHRoZSBnaXZlbiBuYW1lIG9mIHRoZSBjYW5hcnlcbiAqL1xuZnVuY3Rpb24gdmFsaWRhdGVOYW1lKG5hbWU6IHN0cmluZykge1xuICBpZiAobmFtZS5sZW5ndGggPiAyMSkge1xuICAgIHRocm93IG5ldyBFcnJvcihgQ2FuYXJ5IG5hbWUgaXMgdG9vIGxhcmdlLCBtdXN0IGJlIGJldHdlZW4gMSBhbmQgMjEgY2hhcmFjdGVycywgYnV0IGlzICR7bmFtZS5sZW5ndGh9IChnb3QgXCIke25hbWV9XCIpYCk7XG4gIH1cbiAgaWYgKCFuYW1lUmVnZXgudGVzdChuYW1lKSkge1xuICAgIHRocm93IG5ldyBFcnJvcihgQ2FuYXJ5IG5hbWUgbXVzdCBiZSBsb3dlcmNhc2UsIG51bWJlcnMsIGh5cGhlbnMsIG9yIHVuZGVyc2NvcmVzIChnb3QgXCIke25hbWV9XCIpYCk7XG4gIH1cbn1cbiJdfQ==